/*
 * jclassinfo
 * Copyright (C) 2003  Nicos Panayides
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Nicos Panayides
 * anarxia@gmx.net
 *
 * $Id: attributes_xml.c,v 1.7 2004/03/13 04:28:40 anarxia Exp $
 */

#include "config.h"
#include <stdlib.h>
#include <string.h>
#include <jclass/jclass.h>
#include "common.h"
#include "attributes_xml.h"

void attribute_container_print_xml(AttributeContainer* attribute, 
	IntType int_type, int do_flag, ConstantPool* constant_pool)
{
	int j;
	char* attr_name;
		
	if(attribute == NULL)
		return;
		
	attr_name = jclass_cp_get_constant_value(constant_pool, attribute->name_index, INT_IS_INT);
	
	if(attr_name == NULL)
		return;

	/* Code prints the attribute tag */	
	if(strcmp("Code", attr_name))
	{
		printf("<attribute name=\"%s\"", attr_name);
	
		/* Covers Synthetic and deprecated */
		if(attribute->length == 0)
		{
			puts("/>");
			free(attr_name);
			return;
		}
		puts(">");
	}
		
	if(!strcmp("SourceFile", attr_name))
	{
		SourceFileAttribute* sourcefile;
		sourcefile = jclass_sourcefile_attribute_new(attribute);
		sourcefile_attribute_print_xml(sourcefile, constant_pool);
		jclass_sourcefile_attribute_free(sourcefile);
	}
	else if(!strcmp("InnerClasses", attr_name))
	{
		InnerClassesAttribute* innerclasses;
		innerclasses = jclass_innerclasses_attribute_new(attribute);
		innerclasses_attribute_print_xml(innerclasses, constant_pool);
		jclass_innerclasses_attribute_free(innerclasses);
	}
	else if(!strcmp("ConstantValue", attr_name))
	{
		ConstantValueAttribute* constantvalue;
		constantvalue = jclass_constantvalue_attribute_new(attribute);
		constantvalue_attribute_print_xml(constantvalue, int_type, constant_pool);
		jclass_constantvalue_attribute_free(constantvalue);
	}
	else if(!strcmp("Exceptions", attr_name))
	{
		ExceptionsAttribute* exceptions;
		exceptions = jclass_exceptions_attribute_new(attribute);
		exceptions_attribute_print_xml(exceptions, constant_pool);
		jclass_exceptions_attribute_free(exceptions);
	}
	else if(!strcmp("Code", attr_name))
	{
		if(do_flag & (DISASM|VERBOSE|METHOD_DEBUG_INFO))
		{
			CodeAttribute* code;

			puts("<attribute name=\"Code\">");
			code = jclass_code_attribute_new(attribute);
			code_attribute_print_xml(code, do_flag, constant_pool);
			jclass_code_attribute_free(code);
			puts("</attribute>");
		}
	}
	else
	{
		printf("<bytecode>");
		for(j=0; j < attribute->length; j++)
			printf(" %02X", attribute->contents[j]);
		printf("</bytecode>");
	}
	
	if(strcmp("Code", attr_name))
		puts("</attribute>");

	free(attr_name);
}

void constantvalue_attribute_print_xml(ConstantValueAttribute* attrib, 
	IntType int_type, ConstantPool* constant_pool)
{
	char* cp_string;
		
	cp_string = jclass_cp_get_constant_value(constant_pool, 
				attrib->cp_index, int_type);
		
	printf("<value>%s</value>\n", cp_string);
	free(cp_string);
}

void sourcefile_attribute_print_xml(SourceFileAttribute* attrib, ConstantPool* constant_pool)
{
	char* cp_string;
	
	cp_string = jclass_cp_get_constant_value(constant_pool,
		attrib->filename_index, INT_IS_INT);
		
	printf("<sourcefile>%s</sourcefile>\n", cp_string);
	free(cp_string);
}

void exceptions_attribute_print_xml(ExceptionsAttribute* exceptions, 
	ConstantPool* constant_pool)
{
	char* exception_name;
	int j;
	
	for(j = 0; j < exceptions->no_exceptions; j++)
	{
		exception_name = jclass_cp_get_class_name(constant_pool, exceptions->exception_index[j], 0);
		printf("<exception name=\"%s\"/>\n", exception_name);
		free(exception_name);
	}
}

void innerclasses_attribute_print_xml(InnerClassesAttribute* attrib, 
	ConstantPool* constant_pool)
{	
	uint16_t j;
	uint16_t name_index;
	
	char* innerclass_name;
	char* outerclass_name;
	char* innerclass_type;
	char* innerclass_access;
			
	for(j = 0; j < attrib->no_innerclasses; j++)
	{
		if(attrib->classes[j].type_index)
		{
			innerclass_access = jclass_access_flag_to_string(attrib->classes[j].access_flags, 1);
			
			name_index = attrib->classes[j].name_index;
			
			if(name_index)
			{
				innerclass_name = jclass_utf8_to_string(constant_pool->entries[name_index].info.utf8->contents,
														constant_pool->entries[name_index].info.utf8->length);
			}
			else
				innerclass_name = NULL;
			
				innerclass_type = jclass_cp_get_class_name(constant_pool, 
														attrib->classes[j].type_index, 0);
			
				
			printf("<innerclass access=\"%s\"", innerclass_access);
			free(innerclass_access);	
				
			if(innerclass_type != NULL)
			{
				printf(" name=\"%s\"", innerclass_type);
				free(innerclass_type);	
			}
				
			if (innerclass_name != NULL)
			{
				printf(" shortname=\"%s\"", innerclass_name);
				free(innerclass_name);
			}
				
			if(attrib->classes[j].outer_class_type_index)
			{
				outerclass_name = jclass_cp_get_class_name(constant_pool, 
											attrib->classes[j].outer_class_type_index, 0);
				if (outerclass_name != NULL)
				{
					printf(" member-of=\"%s\"", outerclass_name);
					free(outerclass_name);
				}
			}
			printf("/>\n");
		}
	}
}

void code_attribute_print_xml(CodeAttribute* code, int do_flag, 
	ConstantPool* constant_pool)
{
	uint32_t pc;
	uint32_t instruction_pc;
	int is_wide;
	int i;
	
	int32_t s4op;
	uint16_t u2op;
	int32_t count;
	uint16_t exception_counter;
	uint16_t attribute_counter;
	char* class_name;
	LineNumberAttribute* linenumbertable;
	LocalVariableAttribute* localvariabletable;
	TableSwitchOperand* tableswitch;
	LookupSwitchOperand* lookupswitch;
	
	printf("<code");
	if(do_flag & VERBOSE)
	{
		printf(" max_stack=\"%d\" max_locals=\"%d\"", 
										code->max_stack, code->max_locals);
	}
	printf(">\n");
	pc = 0;
	is_wide = 0;
	
	if(do_flag & DISASM)
	{
		while(pc < code->code_length)
		{
			/* skip illegal opcodes */
			do{
				instruction_pc = pc;
				pc++;
			}while(code->code[instruction_pc] > MAX_LEGAL_OPCODE);
			
			printf("<instruction pc=\"%u\"", instruction_pc);
			if(jclass_code_instruction_op_type(code->code[instruction_pc], 0, is_wide) == OP_TYPE_INSTRUCTION)
			{
				printf("wide=\"yes\"");
				is_wide = 1;
				instruction_pc = pc;
				pc++;
			}
			else
			{
				is_wide = 0;
			}
			
			printf(" name=\"%s\"", jclass_code_instruction_name(code->code[instruction_pc]));
			if(jclass_code_instruction_ops(code->code[instruction_pc]) == 0)
			{
				printf("/>\n");
				continue;
			}
			
			printf(">\n");
			
			for(i = 0; i < jclass_code_instruction_ops(code->code[instruction_pc]); i++)
			{
				switch(jclass_code_instruction_op_type(code->code[instruction_pc], i, is_wide))
				{
					case OP_TYPE_INSTRUCTION:
						fputs("Wide cannot follow another wide.\n", stderr);
						pc++;
						break;
				
					case OP_TYPE_UNSIGNED_BYTE_USELESS:
					case OP_TYPE_ERROR:
						pc++;
					case OP_TYPE_NONE:
						break;
					
					case OP_TYPE_BYTE_ARRAY_TYPE:
						printf("<operand type=\"ArrayType\" value=\"%s\"/>\n", jclass_code_array_name(jclass_code_read_ubyte(code->code, &pc)));
						break;
					
					case OP_TYPE_BYTE_CONSTANT_INDEX:
						class_name = jclass_cp_get_constant_value(constant_pool, code->code[pc], INT_IS_INT);
						printf("<operand type=\"Constant\" value=\"%s\"/>\n", class_name);
						free(class_name);
						pc++;
						break;
					
					case OP_TYPE_BYTE:
						printf("<operand type=\"byte\" value=\"%d\"/>\n", jclass_code_read_byte(code->code, &pc));
						break;
					
					case OP_TYPE_UNSIGNED_BYTE:
						printf("<operand type=\"ubyte\" value=\"%u\"/>\n", jclass_code_read_ubyte(code->code, &pc));
						break;
					
					case OP_TYPE_SHORT:
						printf("<operand type=\"short\" value=\"%d\"/>\n", jclass_code_read_short(code->code, &pc));
						break;
				
					case OP_TYPE_UNSIGNED_SHORT:
						printf("<operand type=\"ushort\" value=\"%u\"/>\n", jclass_code_read_ushort(code->code, &pc));
						break;
				
					case OP_TYPE_SHORT_OFFSET:
						printf("<operand type=\"short\" value=\"%u\"/>\n", (instruction_pc + jclass_code_read_short(code->code, &pc)));
						break;				
				
					case OP_TYPE_INT:
					case OP_TYPE_INT_OFFSET:
						s4op = jclass_code_read_int(code->code, &pc);
						if(jclass_code_instruction_op_type(code->code[instruction_pc], i, is_wide) == OP_TYPE_INT_OFFSET)
							s4op += instruction_pc;
										
						printf("<operand type=\"int\" value=\"%d\"/>\n", s4op);
						break;
						
					case OP_TYPE_SHORT_METHOD_INDEX:
						u2op = jclass_code_read_ushort(code->code, &pc);
						class_name = jclass_cp_get_method_signature(constant_pool, u2op, 0);
						printf("<operand type=\"Method\" value=\"%s\"/>\n", class_name);
						free(class_name);
						break;
					
					case OP_TYPE_SHORT_FIELD_INDEX:
						u2op = jclass_code_read_ushort(code->code, &pc);
						class_name = jclass_cp_get_method_signature(constant_pool, u2op, 0);
						printf("<operand type=\"Field\" value=\"%s\"/>\n", class_name);
						free(class_name);
						break;
					
					case OP_TYPE_SHORT_CLASS_INDEX:
						u2op = jclass_code_read_ushort(code->code, &pc);
						class_name = jclass_cp_get_class_name(constant_pool, u2op, 0);
						printf("<operand type=\"Class\" value=\"%s\"/>\n", class_name);
						free(class_name);
						break;
				
					case OP_TYPE_SHORT_CONSTANT_INDEX:
						u2op = jclass_code_read_ushort(code->code, &pc);
						class_name = jclass_cp_get_constant_value(constant_pool, u2op, INT_IS_SHORT);
						printf("<operand type=\"Constant\" value=\"%s\"/>\n", class_name);
						free(class_name);
						break;
				
					case OP_TYPE_TABLESWITCH:
						tableswitch = jclass_code_read_tableswitch(code->code, &pc);
						
						printf("<operand type=\"switch\">\n");
						printf("<case target=\"%u\"/>\n", tableswitch->default_target);
						for(count = 0; count < tableswitch->num_pairs; count++)
						{
							printf("<case value=\"%d\" target=\"%u\"/>\n", 
								count + tableswitch->low_value, tableswitch->target[count]);
						}
						
						
						printf("</operand>");
						jclass_code_tableswitch_operand_free(tableswitch);
						break;
				
					case OP_TYPE_LOOKUPSWITCH:
						lookupswitch= jclass_code_read_lookupswitch(code->code, &pc);

						printf("<operand type=\"switch\">\n");
						printf("<case target=\"%u\"/>\n", lookupswitch->default_target);	
						for(count = 0; count < lookupswitch->num_pairs; count ++)
						{
							printf("<case value=\"%d\" target=\"%u\"/>\n", 
								lookupswitch->value[count], lookupswitch->target[count]);
						}
						
						
						printf("</operand>");
						jclass_code_lookupswitch_operand_free(lookupswitch);
						break;
				}
			}
			puts("</instruction>");
			
		}		
	}
	
	if((code->exception_table != NULL) && (do_flag & VERBOSE))
	{
		puts("<exception_table>\n");
		for(exception_counter = 0; exception_counter < code->exception_table_length; exception_counter++)
		{
			printf("<exception from=\"%d\"", code->exception_table[exception_counter].start_pc);
			printf(" to=\"%d\"", code->exception_table[exception_counter].end_pc);
			printf(" handler=\"%d\"", code->exception_table[exception_counter].handler_pc);
			
			if(code->exception_table[exception_counter].catch_type)
			{
				class_name = jclass_cp_get_class_name(constant_pool, 
					code->exception_table[exception_counter].catch_type, 0);
				
				printf(" catch=\"%s\"/>", class_name);
				free(class_name);
			}
			else
				puts(" catch=\"any\"/>");
		}
		puts("</exception_table>\n");
	}
	
	if(code->attributes_count && (do_flag & METHOD_DEBUG_INFO))
	{
		for(attribute_counter = 0; attribute_counter < code->attributes_count; attribute_counter++)
		{
			if(jclass_attribute_container_has_attribute(&code->attributes[attribute_counter], "LineNumberTable", constant_pool))
			{
				linenumbertable = jclass_linenumber_attribute_new(&code->attributes[attribute_counter]);
				linenumber_attribute_print_xml(linenumbertable);
				jclass_linenumber_attribute_free(linenumbertable);
			}
			else if(jclass_attribute_container_has_attribute(&code->attributes[attribute_counter], "LocalVariableTable", constant_pool ))
			{
				localvariabletable = jclass_localvariable_attribute_new(&code->attributes[attribute_counter]);
				localvariable_attribute_print_xml(localvariabletable, constant_pool);
				jclass_localvariable_attribute_free(localvariabletable);
			}
		}
	}
	printf("</code>");
}

void localvariable_attribute_print_xml(LocalVariableAttribute* localvariabletable, 
	ConstantPool* constant_pool)
{
	uint16_t counter;
	char* short_name;
	char* type;
	char* type_string;
	
	for(counter = 0; counter < localvariabletable->length; counter++)
	{
		type_string = jclass_utf8_to_string(
			constant_pool->entries[localvariabletable->localvariable[counter].descriptor_index].info.utf8->contents,
			constant_pool->entries[localvariabletable->localvariable[counter].descriptor_index].info.utf8->length);
		
		type = jclass_descriptor_get_type(type_string);
		free(type_string);
		
		short_name = jclass_utf8_to_string(
			constant_pool->entries[localvariabletable->localvariable[counter].name_index].info.utf8->contents,
			constant_pool->entries[localvariabletable->localvariable[counter].name_index].info.utf8->length);
		
		printf("<localvariable index=\"%d\" type=\"%s\" shortname=\"%s\" pc=\"%d\" length=\"%d\"/>\n",
			localvariabletable->localvariable[counter].index,
			type,
			short_name,
			localvariabletable->localvariable[counter].start_pc, 
			localvariabletable->localvariable[counter].length);
		
		free(type);
		free(short_name);
	}
}

void linenumber_attribute_print_xml(LineNumberAttribute* linenumbertable)
{
	uint16_t counter;
	
	for(counter = 0; counter < linenumbertable->length; counter++)
	{
		printf("<linenumber line=\"%d\" pc=\"%d\"/>\n", 
			linenumbertable->line_number[counter].line_number, 
			linenumbertable->line_number[counter].start_pc);
	}
}
