/*
  ==============================================================================

   This file is part of the JUCE library.
   Copyright (c) 2015 - ROLI Ltd.

   Permission is granted to use this software under the terms of either:
   a) the GPL v2 (or any later version)
   b) the Affero GPL v3

   Details of these licenses can be found at: www.gnu.org/licenses

   JUCE is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
   A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

   ------------------------------------------------------------------------------

   To release a closed-source product which uses JUCE, commercial licenses are
   available: visit www.juce.com for more information.

  ==============================================================================
*/

#ifndef JUCE_AUDIOVISUALISER_H_INCLUDED
#define JUCE_AUDIOVISUALISER_H_INCLUDED


//==============================================================================
/**
    A simple component that can be used to show a scrolling waveform of audio data.

    This is a handy way to get a quick visualisation of some audio data. Just create
    one of these, set its size and oversampling rate, and then feed it with incoming
    data by calling one of its pushBuffer() or pushSample() methods.

    You can override its paint method for more customised views, but it's only designed
    as a quick-and-dirty class for simple tasks, so please don't send us feature requests
    for fancy additional features that you'd like it to support! If you're building a
    real-world app that requires more powerful waveform display, you'll probably want to
    create your own component instead.
*/
class JUCE_API AudioVisualiserComponent  : public Component,
                                           private Timer
{
public:
    /** Creates a visualiser with the given number of channels. */
    AudioVisualiserComponent (int initialNumChannels);

    /** Destructor. */
    ~AudioVisualiserComponent();

    /** Changes the number of channels that the visualiser stores. */
    void setNumChannels (int numChannels);

    /** Changes the number of samples that the visualiser keeps in its history.
        Note that this value refers to the number of averaged sample blocks, and each
        block is calculated as the peak of a number of incoming audio samples. To set
        the number of incoming samples per block, use setSamplesPerBlock().
     */
    void setBufferSize (int bufferSize);

    /** */
    void setSamplesPerBlock (int newNumInputSamplesPerBlock) noexcept;

    /** */
    int getSamplesPerBlock() const noexcept                         { return inputSamplesPerBlock; }

    /** Clears the contents of the buffers. */
    void clear();

    /** Pushes a buffer of channels data.
        The number of channels provided here is expected to match the number of channels
        that this AudioVisualiserComponent has been told to use.
    */
    void pushBuffer (const AudioSampleBuffer& bufferToPush);

    /** Pushes a buffer of channels data.
        The number of channels provided here is expected to match the number of channels
        that this AudioVisualiserComponent has been told to use.
    */
    void pushBuffer (const AudioSourceChannelInfo& bufferToPush);

    /** Pushes a buffer of channels data.
        The number of channels provided here is expected to match the number of channels
        that this AudioVisualiserComponent has been told to use.
    */
    void pushBuffer (const float** channelData, int numChannels, int numSamples);

    /** Pushes a single sample (per channel).
        The number of channels provided here is expected to match the number of channels
        that this AudioVisualiserComponent has been told to use.
    */
    void pushSample (const float* samplesForEachChannel, int numChannels);

    /** Sets the colours used to paint the */
    void setColours (Colour backgroundColour, Colour waveformColour) noexcept;

    /** Sets the frequency at which the component repaints itself. */
    void setRepaintRate (int frequencyInHz);

    /** Draws a channel of audio data in the given bounds.
        The default implementation just calls getChannelAsPath() and fits this into the given
        area. You may want to override this to draw things differently.
    */
    virtual void paintChannel (Graphics&, Rectangle<float> bounds,
                               const Range<float>* levels, int numLevels, int nextSample);

    /** Creates a path which contains the waveform shape of a given set of range data.
        The path is normalised so that -1 and +1 are its upper and lower bounds, and it
        goes from 0 to numLevels on the X axis.
    */
    void getChannelAsPath (Path& result, const Range<float>* levels, int numLevels, int nextSample);

    //==============================================================================
    /** @internal */
    void paint (Graphics&) override;

private:
    struct ChannelInfo;
    friend struct ChannelInfo;
    friend struct ContainerDeletePolicy<ChannelInfo>;

    OwnedArray<ChannelInfo> channels;
    int numSamples, inputSamplesPerBlock;
    Colour backgroundColour, waveformColour;

    void timerCallback() override;

    JUCE_DECLARE_NON_COPYABLE_WITH_LEAK_DETECTOR (AudioVisualiserComponent)
};


#endif  // JUCE_AUDIOVISUALISER_H_INCLUDED
