!
!  An initial construction of an individual growth-curve of a star.
!
!  Copyright © 2016 F.Hroch (hroch@physics.muni.cz)
!
!  This file is part of Munipack.
!
!  Munipack is free software: you can redistribute it and/or modify
!  it under the terms of the GNU General Public License as published by
!  the Free Software Foundation, either version 3 of the License, or
!  (at your option) any later version.
!
!  Munipack is distributed in the hope that it will be useful,
!  but WITHOUT ANY WARRANTY; without even the implied warranty of
!  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!  GNU General Public License for more details.
!
!  You should have received a copy of the GNU General Public License
!  along with Munipack.  If not, see <http://www.gnu.org/licenses/>.
!
!

module grow_init

  implicit none

  ! numerical precision of real numbers
  integer, parameter, private :: dbl = selected_real_kind(15)
  real(dbl), parameter, private :: pi = 3.14159265358979312_dbl

  ! print debug informations ?
  logical, parameter, private :: verbose = .false.

  real(dbl), dimension(:), allocatable, private :: cts,dcts,aper,raper
  real(dbl), private :: maxback, hwhm
  integer, private :: nhwhm, naper

  private :: minall, the_constructor

contains


  subroutine growinit(xraper,xcts,xdcts,skyerr,rhwhm,mhwhm,grow,dgrow,valid)

    use NelderMead

    real, dimension(:), intent(in) :: xraper
    real(dbl), dimension(:), intent(in) :: xcts,xdcts
    real, intent(in) :: rhwhm
    real(dbl), intent(in) :: skyerr
    integer, intent(in) :: mhwhm
    real(dbl), dimension(:), intent(out) :: grow,dgrow
    logical, intent(out) :: valid
    real(dbl), dimension(2) :: p,dp
    real(dbl) :: pmin
    integer :: ifault

    naper = size(xraper)
    allocate(raper(naper),aper(naper),cts(naper),dcts(naper))

    raper = xraper
    aper = pi*raper**2
    cts = xcts
    dcts = xdcts
    hwhm = rhwhm
    nhwhm = mhwhm

    ! estimate parameters
    p(1) = 1
    p(2) = 0
    dp(1) = 0.01
    dp(2) = 0.1*skyerr
    maxback = 10*skyerr
    call nelmin1(minall,p,dp,pmin,ifault)
    valid = ifault == 0

    ! update grow curve
    if( valid ) call the_constructor(p(1),p(2),grow,dgrow,valid)

    deallocate(raper,aper,cts,dcts)

  end subroutine growinit


  function minall(p)

    use robustmean

    real(dbl) :: minall
    real(dbl), dimension(:), intent(in) :: p

    real(dbl), dimension(:), allocatable :: grow,dgrow
    real(dbl) :: s,t,b,d,a,fmin,sig
    integer :: i
    logical :: valid

    t = p(1)
    b = p(2)

    ! range check of parameters
    if( .not. (0.5 < t .and. t < 1.0 ) .or.  abs(b) > maxback) then
       minall = 1e5
       return
    end if

    ! determine grow curve with actual parameters
    allocate(grow(naper),dgrow(naper))
    call the_constructor(t,b,grow,dgrow,valid)
    if( .not. valid ) then
       minall = 1e6
       return
    end if

    ! median of absolute deviations
    call rmean(dgrow,d,s)
    sig = max(d,1e-6)

    ! asymptotic parameters
    a = 2*(raper(nhwhm) / hwhm)**2 * (1 - grow(nhwhm))

    ! asympotic estimate for larger radiuses
    s = 0
    do i = nhwhm,naper
       fmin = 1 - a / 2 / (raper(i) / hwhm)**2
       if( grow(i) < fmin ) then
          s = s + abs(grow(i) - fmin) / sig
       else if( grow(i) > 1 ) then
          s = s + abs(grow(i) - 1) / sig
       else
          s = s + 0
       end if
    end do

    minall = abs(b) + max(0.0,1-t) + s

    deallocate(grow,dgrow)

  end function minall

  subroutine the_constructor(t,b,grow,dgrow,valid)

    use qmeans
    use robratio
    use robustmean

    real(dbl), intent(in) :: t,b
    real(dbl), dimension(:), intent(out) :: grow,dgrow
    logical, intent(out) :: valid

    real(dbl), dimension(size(grow)) :: flux,dflux
    real(dbl) :: f0,s
    integer :: i,n

    valid = .false.

    flux = cts - b*aper
    dflux = dcts

    if( any(flux < epsilon(flux)) ) return

    ! set up initial point of grow curve
    grow(nhwhm) = t
    dgrow(nhwhm) = t * dcts(nhwhm)/cts(nhwhm)

    ! compute grow curve at points with smaller radius
    do i = nhwhm-1,1,-1
       grow(i) = grow(i+1)*(flux(i) / flux(i+1))
       dgrow(i) = grow(i)*sqrt((dcts(i)/cts(i))**2 + (dcts(i+1)/cts(i+1))**2)/1.41
    end do

    ! estimate f0 parameter
    n = nhwhm
    call rcal(flux(1:n),dflux(1:n),grow(1:n),dgrow(1:n),f0,s,.false.)

    ! compute grow for larger radiuses
    n = nhwhm + 1
    grow(n:) = flux(n:) / f0
    dgrow(n:) = abs(grow(n:)) * sqrt((dcts(n:)/cts(n:))**2 + s**2)

    valid = .true.

  end subroutine the_constructor

end module grow_init
