///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// compute the element "grad_grad" matrix condition number
// for non-regression test purpose
//
// author: Pierre.Saramito@imag.fr
//
// date: 27 september 2017
//
#include "rheolef/skit.h"
#include "rheolef/quadrature.h"
#include "rheolef/basis.h"
#include "form_element.h"
using namespace rheolef;
using namespace std;
#include "arma2csr.icc"

template <class Basis>
void 
show_grad_grad_base (
  const Basis&      b,
  reference_element hat_K,
  arma::Mat<Float>& grad_grad,
  bool              dump)
{
  build_grad_grad (b, hat_K, grad_grad);
  csr<Float,sequential> grad_grad_csr = arma2csr(grad_grad);
  cout << setprecision(numeric_limits<Float>::digits10)
       << "approx   " << b.name() << endl
       << "element  " << hat_K.name() << endl
       << "det      " << det(grad_grad) << endl
       << "cond     " << cond(grad_grad) << endl
       << "size     " << grad_grad.n_rows << endl
       << "nnz      " << grad_grad_csr.dis_nnz()<< endl
       << "fill     " << 1.*grad_grad_csr.dis_nnz()/sqr(grad_grad.n_rows)<< endl
    ;
  if (!dump) return;
  odiststream out ("grad_grad.mm");
  out << grad_grad_csr;
}
void 
show_grad_grad (
  const basis&   b,
  reference_element hat_K,
  bool              dump)
{
  arma::Mat<Float> grad_grad;
  show_grad_grad_base (b, hat_K, grad_grad, dump);
  cout << setprecision(numeric_limits<Float>::digits10)
       << "node     " << b.option().get_node_name() << endl
       << "raw_poly " << b.option().get_raw_polynomial_name() << endl
    ;
}
int main(int argc, char**argv) {
  environment rheolef (argc, argv);
  basis_option sopt;
  string approx= (argc > 1) ? argv[1]    : "P3";
  char   t     = (argc > 2) ? argv[2][0] : 't';
  string node  = (argc > 3) ? argv[3]    : sopt.get_node_name();
  string poly  = (argc > 4) ? argv[4]    : sopt.get_raw_polynomial_name();
  bool   dump  = (argc > 5);
  sopt.set_node (node);
  sopt.set_raw_polynomial (poly);
  reference_element hat_K;
  hat_K.set_name(t);
  if (approx[0] == 'D') {
    basis_raw b (approx);
    arma::Mat<Float> grad_grad;
    show_grad_grad_base (b, hat_K, grad_grad, dump);
  } else {
    string name = approx + sopt.stamp();
    basis b (name);
    show_grad_grad (b, hat_K, dump);
 }
}
