///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/space.h"
#include "rheolef/space_mult.h"
#include "rheolef/piola.h"

namespace rheolef {

// ======================================================================
// space_base_rep
// ======================================================================
template <class T, class M>
space_base_rep<T,M>::space_base_rep (
    const geo_basic<T,M>&    omega_in,
    std::string              approx,
    std::string              valued)
  : _constit(omega_in,approx,valued),
    _xdof(),
    _have_freezed(false),
    _idof2blk_iub(),
    _iu_ownership(),
    _ib_ownership()
{
  // omega_in is compressed by space_scalar_constitution() allocator
  // when it is a domain: then it becomes a geo_domain, with compressed numbering
  // so, omega_in can be different from omega:
  if (approx == "") return; // empty element => default cstor
  _idof2blk_iub.resize (_constit.ownership());
  init_xdof();
}
template <class T, class M>
space_base_rep<T,M>::space_base_rep (const space_constitution<T,M>& constit)
  : _constit(constit),
    _xdof(),
    _have_freezed(false),
    _idof2blk_iub(),
    _iu_ownership(),
    _ib_ownership()
{
  distributor idof_ownership = _constit.ownership();
  _idof2blk_iub.resize (idof_ownership);
  init_xdof();
}
template <class T, class M>
space_base_rep<T,M>::space_base_rep (const space_mult_list<T,M>& expr)
  : _constit(expr),
    _xdof(),
    _have_freezed(false),
    _idof2blk_iub(),
    _iu_ownership(),
    _ib_ownership()
{
  _idof2blk_iub.resize (_constit.ownership());
  init_xdof();
}
template <class T, class M>
void
space_base_rep<T,M>::init_xdof()
{
  if (_constit.valued_tag() == space_constant::mixed) {
    trace_macro ("init_xdof and mixed valued space: not yet");
    return;
  }
  // precompute one time for all all dof nodes, for an easy interpolation
  // apply only to nodal fem basis (eg. Lagrange) only
  // TODO: in a lasy way: at first call to xdof(idof), call this initialization
  const geo_basic<T,M>& omega = get_geo();
  const numbering<T,M>& fem   = get_numbering();
  bool is_isoparametric = (fem.name() == omega.get_piola_basis().name());
  if (is_isoparametric) {
    // 1.a) geometric nodes coincides to fem dof nodes:  
    _xdof = omega.get_nodes();
    return;
  }
  // 1.b) here is the non-isoparametric case: eg. P0 fem space on a P1 geo
  distributor xdof_ownership;
  if (_constit.valued_tag() == space_constant::scalar) {
    xdof_ownership = ownership();
  } else {
    // get xdof sizes from any component:
    xdof_ownership = _constit[0].ownership();
  }
  _xdof.resize (xdof_ownership);
  std::vector<bool> marked (ndof(), false);
  size_type count_mark = 0;
  std::vector<size_type> dis_idof;
  std::vector<size_type> dis_inod;
  size_type first_dis_idof = xdof_ownership.first_index();
  basis_on_pointset<T> b (fem.get_basis(), omega.get_piola_basis());
  for (typename geo_basic<T,M>::const_iterator iter = omega.begin(), last = omega.end(); iter != last; iter++) {
    const geo_element& K = *iter;
    fem.dis_idof (omega.sizes(), K, dis_idof);
    omega.dis_inod (K, dis_inod);
    for (size_type loc_idof = 0, loc_ndof = dis_idof.size(); loc_idof < loc_ndof; loc_idof++) {
      assert_macro (dis_idof[loc_idof] < xdof_ownership.dis_size(), 
	"dis_idof " << dis_idof[loc_idof] << " out of range [0:" << xdof_ownership.dis_size() << "[");
      if (! xdof_ownership.is_owned (dis_idof[loc_idof])) continue;
      size_type idof = dis_idof[loc_idof] - first_dis_idof;
      if (marked [idof]) continue;
      marked [idof] = true;
      count_mark++;
      _xdof[idof] = piola_transformation (omega, b, K.variant(), dis_inod, loc_idof);
    }
  }
  size_type dis_n_missing = marked.size() - count_mark;
#ifdef _RHEOLEF_HAVE_MPI
  dis_n_missing = mpi::all_reduce (comm(), dis_n_missing, std::plus<size_type>());
#endif // _RHEOLEF_HAVE_MPI
  if (dis_n_missing == 0) {
    return;
  }
  //
  // 2) on some rare 3d bdry geo_domains with distributed nproc > 1, there are missing xdofs...
  // => second pass: assembly only external xdofs, an performs some comms
  for (typename geo_basic<T,M>::const_iterator iter = omega.begin(), last = omega.end(); iter != last; iter++) {
    const geo_element& K = *iter;
    fem.dis_idof (omega.sizes(), K, dis_idof);
    omega.dis_inod (K, dis_inod);
    for (size_type loc_idof = 0, loc_ndof = dis_idof.size(); loc_idof < loc_ndof; loc_idof++) {
      if (xdof_ownership.is_owned (dis_idof[loc_idof])) continue;
      _xdof.dis_entry (dis_idof[loc_idof]) = piola_transformation (omega, b, K.variant(), dis_inod, loc_idof);
    }
  }
  _xdof.dis_entry_assembly();
}
template <class T, class M>
void
space_base_rep<T,M>::base_freeze_body () const
{
  // -----------------------------------------------------------------------
  // 1) loop on domains: mark blocked dofs
  // -----------------------------------------------------------------------
  disarray<size_type,M> blocked_flag = _constit.build_blocked_flag();

  // copy the blocked_flag into the iub disarray, as the "blocked" bit:
  for (size_type idof = 0, ndof = blocked_flag.size(); idof < ndof; idof++) {
    _idof2blk_iub [idof].set_blocked (blocked_flag[idof]);
  }
  // -----------------------------------------------------------------------
  // 2) init numbering
  // -----------------------------------------------------------------------
  size_type n_unknown = 0;
  size_type n_blocked = 0;
  for (size_type idof = 0, ndof = _idof2blk_iub.size(); idof < ndof; idof++) {
    bool blk = _idof2blk_iub [idof].is_blocked();
    if (! blk) {
      _idof2blk_iub[idof].set_iub (n_unknown);
      n_unknown++;
    } else {
      _idof2blk_iub[idof].set_iub (n_blocked);
      n_blocked++;
    }
  }
  size_type dis_n_unknown = n_unknown;
  size_type dis_n_blocked = n_blocked;
#ifdef _RHEOLEF_HAVE_MPI
  if (is_distributed<M>::value) {
    dis_n_unknown = mpi::all_reduce (comm(), dis_n_unknown, std::plus<T>());
    dis_n_blocked = mpi::all_reduce (comm(), dis_n_blocked, std::plus<T>());
  }
#endif // // _RHEOLEF_HAVE_MPI
  _iu_ownership = distributor (dis_n_unknown, comm(), n_unknown);
  _ib_ownership = distributor (dis_n_blocked, comm(), n_blocked);
}
// ----------------------------------------------------------------------------
// used by field & form_assembly
// ----------------------------------------------------------------------------
template <class T, class M>
void
space_base_rep<T,M>::dis_idof (const geo_element& K, std::vector<size_type>& dis_idof) const
{
  freeze_guard();
  _constit.dis_idof (K, dis_idof);
}
// ----------------------------------------------------------------------------
// stamp : e.g. "P1(square)", for field_expr<Expr> checks
// ----------------------------------------------------------------------------
template <class T, class M>
std::string
space_base_rep<T,M>::stamp() const
{
  // TODO: compte stamp one time for all in a space::_stamp member
  std::stringstream os;
  os << _constit;
  return os.str();
}
// ----------------------------------------------------------------------------
// u["left"] 
// => build here the requiresd temporary indirect disarray
// ----------------------------------------------------------------------------
/**
 Implementation note: there is two numbering styles

   bgd_idof : from the current space
   dom_idof : from a space compacted to the domain "dom", as geo_domain does

  This function returns the renumbering disarray "dom_idof2bgd_idof".
  It is a temporary, used at the fly when using the u[dom] syntax.
 */
template <class T, class M>
disarray<typename space_base_rep<T,M>::size_type, M>
space_base_rep<T,M>::build_indirect_array (
    const space_base_rep<T,M>& Wh,	// Wh = space on domain
    const std::string&       dom_name   // redundant: contained in Wh.get_geo()
  ) const
{
  const geo_basic<T,M>& bgd_gamma = get_geo()[dom_name];
  return build_indirect_array (Wh, bgd_gamma);
}
template <class T, class M>
disarray<typename space_base_rep<T,M>::size_type, M>
space_base_rep<T,M>::build_indirect_array (
    const space_base_rep<T,M>& Wh,	// Wh = space on domain
    const geo_basic<T,M>&      bgd_gamma2
  ) const
{
  // TODO: move it to the call ?
  const geo_basic<T,M>& bgd_gamma = bgd_gamma2.get_background_domain();

  // TODO: verifier que le domaine bgd_gamma est compatible:
  // => il doit y avoir un meme maillage de base a bgd_gamma & Wh.get_geo
  const space_base_rep<T,M>& Vh = *this;  // Vh = space on background mesh
  Vh.freeze_guard();
  Wh.freeze_guard();
  const geo_basic<T,M>& dom_gamma = Wh.get_geo();
  size_type map_dim = dom_gamma.map_dimension();
  check_macro (dom_gamma.size() == bgd_gamma.size(), "incompatible domains");
  distributor dom_ownership = Wh.ownership();
  distributor bgd_ownership = Vh.ownership();
  size_type first_dom_dis_idof = dom_ownership.first_index();
  size_type first_bgd_dis_idof = bgd_ownership.first_index();
  std::vector<size_type> dom_dis_idofs, bgd_dis_idofs;
  disarray<size_type, M> dom_idof2bgd_idof (dom_ownership, std::numeric_limits<size_type>::max());
  for (size_type ige = 0, nge = dom_gamma.size(); ige < nge; ige++) {
    const geo_element& dom_S = dom_gamma[ige];
    const geo_element& bgd_S = bgd_gamma[ige];
    Wh.dis_idof (dom_S, dom_dis_idofs);
    Vh.dis_idof (bgd_S, bgd_dis_idofs);
    for (size_type loc_idof = 0, loc_ndof = dom_dis_idofs.size(); loc_idof < loc_ndof; loc_idof++) {
      size_type dom_dis_idof = dom_dis_idofs [loc_idof];
      size_type bgd_dis_idof = bgd_dis_idofs [loc_idof];
      dom_idof2bgd_idof.dis_entry (dom_dis_idof) = bgd_dis_idof;
    }
  }
  dom_idof2bgd_idof.dis_entry_assembly();
  // move to local numbering:
  for (size_type dom_idof = 0, dom_ndof = dom_idof2bgd_idof.size(); dom_idof < dom_ndof; dom_idof++) {
      size_type bgd_dis_idof = dom_idof2bgd_idof [dom_idof];
      size_type dom_dis_idof = dom_idof + first_dom_dis_idof;
      check_macro (bgd_ownership.is_owned (bgd_dis_idof), "bgd_dis_idof="<<bgd_dis_idof<<" is out of range ["<<first_bgd_dis_idof<<":"<< bgd_ownership.last_index()<<"[");
      size_type bgd_idof = bgd_dis_idof - first_bgd_dis_idof;
      dom_idof2bgd_idof [dom_idof] = bgd_idof;
  }
  return dom_idof2bgd_idof;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
template class space_base_rep<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class space_base_rep<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
