/*************************************************************************
** AGLTable.h                                                           **
**                                                                      **
** This file is part of dvisvgm -- a fast DVI to SVG converter          **
** Copyright (C) 2005-2016 Martin Gieseking <martin.gieseking@uos.de>   **
**                                                                      **
** This program is free software; you can redistribute it and/or        **
** modify it under the terms of the GNU General Public License as       **
** published by the Free Software Foundation; either version 3 of       **
** the License, or (at your option) any later version.                  **
**                                                                      **
** This program is distributed in the hope that it will be useful, but  **
** WITHOUT ANY WARRANTY; without even the implied warranty of           **
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         **
** GNU General Public License for more details.                         **
**                                                                      **
** You should have received a copy of the GNU General Public License    **
** along with this program; if not, see <http://www.gnu.org/licenses/>. **
*************************************************************************/

/* The following table provides a compact mapping from PostScript character names
 * to Unicode points. Instead of using the character names directly it maps the
 * hash values (xxhash32) of the names to the corresponding code points.
 * The character mapping is derived from the Adobe Glyph List (AGL):
 * https://github.com/adobe-type-tools/agl-aglfn
 * http://tug.ctan.org/macros/latex/contrib/pdfx/glyphtounicode-cmr.tex */
static struct Hash2Unicode {
	UInt32 hash;
	UInt32 codepoint;
} hash2unicode[] = {
	{0x001cf4a9, 0x0118}, // Eogonek
	{0x003b11fb, 0x055f}, // abbreviationmarkarmenian
	{0x0050a316, 0x0444}, // afii10086
	{0x0066ddfe, 0x03a5}, // Upsilon
	{0x0067a87d, 0x0026}, // ampersand
	{0x007399ea, 0x2118}, // weierstrass
	{0x009a8cbc, 0x30c8}, // tokatakana
	{0x00a8ecc1, 0xfb93}, // gaffinalarabic
	{0x00b0f8d6, 0x05b2}, // hatafpatahwidehebrew
	{0x00b1b890, 0x2486}, // nineteenparen
	{0x00b92975, 0x05a7}, // dargalefthebrew
	{0x00c458a3, 0x2154}, // twothirds
	{0x00e73988, 0x09b8}, // sabengali
	{0x00f475c4, 0x03ee}, // Deicoptic
	{0x00fa9974, 0x0303}, // tildecmb
	{0x01012c9a, 0x05b4}, // hiriq
	{0x0107d6e3, 0x0933}, // lladeva
	{0x01127238, 0x3237}, // ideographiccongratulationparen
	{0x011abc50, 0x0547}, // Shaarmenian
	{0x01297545, 0x25aa}, // H18543
	{0x012d44bc, 0x05d6}, // afii57670
	{0x013044fb, 0x2205}, // emptyset
	{0x01422ce1, 0x0486}, // psilipneumatacyrilliccmb
	{0x01532d0f, 0x002f}, // slashBig
	{0x01571757, 0xfe6b}, // atsmall
	{0x01678eeb, 0x066b}, // decimalseparatorpersian
	{0x018643fe, 0x091a}, // cadeva
	{0x019e6772, 0x3148}, // cieuckorean
	{0x019fa822, 0x05de}, // mem
	{0x01a69c0c, 0x221a}, // radicalbigg
	{0x01cfa7b3, 0x0a3e}, // aamatragurmukhi
	{0x01d7c979, 0x3007}, // ideographiczero
	{0x02057d9e, 0x05e8}, // reshhatafsegol
	{0x0209914d, 0xf7fe}, // Thornsmall
	{0x020d70ad, 0x278b}, // twocircleinversesansserif
	{0x0239eab1, 0x3131}, // kiyeokkorean
	{0x023b0c79, 0xff87}, // nukatakanahalfwidth
	{0x023deac8, 0x3183}, // yesieungpansioskorean
	{0x02409db2, 0x1ed1}, // ocircumflexacute
	{0x02574652, 0x0e1b}, // poplathai
	{0x025b8acc, 0x0a47}, // eematragurmukhi
	{0x02610d57, 0x2668}, // hotsprings
	{0x0292f83b, 0x0e11}, // thonangmonthothai
	{0x0293c9db, 0xff49}, // imonospace
	{0x029e63cc, 0x21e7}, // arrowupwhite
	{0x02a32a9d, 0x0275}, // obarred
	{0x02a97908, 0x09f7}, // fournumeratorbengali
	{0x02abe0be, 0x2556}, // SF210000
	{0x02acd87a, 0x2461}, // twocircle
	{0x02bb8927, 0x0154}, // Racute
	{0x02bd043f, 0x3392}, // mhzsquare
	{0x02bdc138, 0x304a}, // ohiragana
	{0x02c40de7, 0xffe6}, // wonmonospace
	{0x02dab625, 0x310c}, // lbopomofo
	{0x02df8edf, 0x30e7}, // yosmallkatakana
	{0x02e3dfc8, 0x3062}, // dihiragana
	{0x02f13fd7, 0x0029}, // parenrightbig
	{0x02f32b9b, 0x002f}, // slashbigg
	{0x0306380d, 0xff35}, // Umonospace
	{0x03170204, 0x0495}, // ghemiddlehookcyrillic
	{0x03182c9f, 0x25b4}, // blackuppointingsmalltriangle
	{0x0319d343, 0x329e}, // ideographicprintcircle
	{0x031ad266, 0x331e}, // kooposquare
	{0x03274b72, 0x0018}, // controlCAN
	{0x032ef9aa, 0xf7f6}, // Odieresissmall
	{0x0334c4d5, 0x0aa7}, // dhagujarati
	{0x03396436, 0x1ed0}, // Ocircumflexacute
	{0x033b64c4, 0x047f}, // otcyrillic
	{0x033e74dc, 0x1e44}, // Ndotaccent
	{0x034ec8b1, 0x30b0}, // gukatakana
	{0x037b5e30, 0x00b9}, // onesuperior
	{0x03993e60, 0xff57}, // wmonospace
	{0x03a136c3, 0x0e0b}, // sosothai
	{0x03b0103c, 0x2270}, // notlessnorequal
	{0x03c2a8d8, 0x1eb3}, // abrevehookabove
	{0x03d234c7, 0xfea7}, // khahinitialarabic
	{0x03f4da1a, 0x05b8}, // qamatsnarrowhebrew
	{0x040b8001, 0x230a}, // floorleftBig
	{0x0410b525, 0x0138}, // kgreenlandic
	{0x0433eb22, 0x014e}, // Obreve
	{0x043f3fc5, 0x040a}, // afii10059
	{0x044589fe, 0x0646}, // afii57446
	{0x044ba421, 0x0585}, // oharmenian
	{0x0454dddc, 0x0432}, // afii10067
	{0x048b164e, 0x01c3}, // clickretroflex
	{0x048e8b97, 0x0161}, // scaron
	{0x0491732e, 0x3186}, // yeorinhieuhkorean
	{0x049c65ac, 0x2121}, // telephone
	{0x04a45907, 0x041a}, // afii10028
	{0x04c4d94b, 0x05b5}, // tsere1e
	{0x04c8ee7d, 0x042d}, // Ereversedcyrillic
	{0x04de1db0, 0x0950}, // omdeva
	{0x04f4d676, 0x0027}, // quotesingle
	{0x04fb1584, 0xf6c4}, // afii10063
	{0x0500f909, 0x0407}, // Yicyrillic
	{0x0503fcb5, 0x00b1}, // plusminus
	{0x05116c6a, 0x30fc}, // prolongedkana
	{0x05302abd, 0x2025}, // twodotleader
	{0x053ece0c, 0x3050}, // guhiragana
	{0x05574c05, 0x09a3}, // nnabengali
	{0x056bac6c, 0x30a3}, // ismallkatakana
	{0x058218bb, 0x0386}, // Alphatonos
	{0x058691a9, 0x33d2}, // squarelog
	{0x059c61cd, 0x0436}, // zhecyrillic
	{0x059eb4a3, 0x2085}, // fiveinferior
	{0x05a47299, 0x320d}, // hieuhparenkorean
	{0x05a53e96, 0x0282}, // shook
	{0x05b0f8c3, 0x02b5}, // rhookturnedsuperior
	{0x05c5a128, 0xf76b}, // Ksmall
	{0x05cee53c, 0x201d}, // quotedblright
	{0x05de47fd, 0x1e7f}, // vdotbelow
	{0x05e340f3, 0x1e70}, // Tcircumflexbelow
	{0x05e8321d, 0x0325}, // ringbelowcmb
	{0x05ec5d36, 0x2471}, // eighteencircle
	{0x05f03fff, 0x0ae0}, // rrvocalicgujarati
	{0x060beb03, 0x0175}, // wcircumflex
	{0x0618af48, 0x005c}, // backslashBig
	{0x061ad8fc, 0x24ca}, // Ucircle
	{0x061f0bd8, 0x2a00}, // circledotdisplay
	{0x062d146d, 0xf88b}, // maieklowrightthai
	{0x0642035b, 0x0010}, // controlDLE
	{0x0646584a, 0x003d}, // equal
	{0x064874b1, 0x05d3}, // afii57667
	{0x066433cf, 0x20a4}, // lira
	{0x06823c6b, 0xfb4d}, // kafrafehebrew
	{0x06894954, 0xff5e}, // asciitildemonospace
	{0x069a405d, 0x00e6}, // ae
	{0x06b00ffc, 0x0101}, // amacron
	{0x06b72f51, 0x27e8}, // angbracketleftbig
	{0x06be8647, 0x0442}, // afii10084
	{0x06e56a17, 0x05dc}, // lamedholamdagesh
	{0x06ec3366, 0xfe42}, // cornerbracketrightvertical
	{0x0700a693, 0x0475}, // izhitsacyrillic
	{0x07019244, 0xfb02}, // fl
	{0x07072da3, 0x2299}, // circleot
	{0x07099ef9, 0xfeae}, // rehfinalarabic
	{0x0710dd39, 0x02de}, // rhotichookmod
	{0x074aba74, 0x09af}, // yabengali
	{0x07562010, 0x09bc}, // nuktabengali
	{0x075a830a, 0x21e6}, // arrowleftwhite
	{0x076312db, 0x2497}, // sixteenperiod
	{0x0767cf10, 0x1ea8}, // Acircumflexhookabove
	{0x076c3b34, 0x1ec3}, // ecircumflexhookabove
	{0x076dbf41, 0x05b7}, // patah11
	{0x07726745, 0x0e25}, // lolingthai
	{0x078184fa, 0x00f7}, // divide
	{0x0790751c, 0x2466}, // sevencircle
	{0x0793d50d, 0x30bb}, // sekatakana
	{0x07a1ce35, 0x0906}, // aadeva
	{0x07ab20a8, 0x0ab3}, // llagujarati
	{0x07b2b22c, 0x02c6}, // hatwidest
	{0x07e20c30, 0x017b}, // Zdot
	{0x07e38c67, 0x33bb}, // nwsquare
	{0x081dd122, 0x0a38}, // sagurmukhi
	{0x082543e5, 0x33a0}, // cmsquaredsquare
	{0x083d0b54, 0x3227}, // eightideographicparen
	{0x08429fa7, 0x2591}, // ltshade
	{0x084b888b, 0x311a}, // abopomofo
	{0x085499c4, 0x0925}, // thadeva
	{0x086a99d9, 0x01af}, // Uhorn
	{0x087038eb, 0xfb20}, // ayinaltonehebrew
	{0x08729ac0, 0xed18}, // bracehtipdownright
	{0x08905fd6, 0x230b}, // floorrightbigg
	{0x089d739a, 0x005a}, // Z
	{0x08a131c8, 0x096d}, // sevendeva
	{0x08a6b099, 0x02a6}, // ts
	{0x08b5de5a, 0x038a}, // Iotatonos
	{0x08b78f6b, 0xff86}, // nikatakanahalfwidth
	{0x08d57b6a, 0x0019}, // controlEM
	{0x08ddb521, 0x3226}, // sevenideographicparen
	{0x092aa224, 0x0a90}, // aigujarati
	{0x092cd86d, 0x03d6}, // omega1
	{0x09310ab8, 0x027f}, // rfishhookreversed
	{0x094ceadc, 0x0047}, // G
	{0x09751504, 0x038c}, // Omicrontonos
	{0x09790f28, 0x33be}, // kwsquare
	{0x09853aa3, 0x01c1}, // clicklateral
	{0x099430b2, 0xf7f5}, // Otildesmall
	{0x09a03740, 0xfe5b}, // braceleftsmall
	{0x09a4b050, 0x0ae8}, // twogujarati
	{0x09adf253, 0xf721}, // exclamsmall
	{0x09d4b5eb, 0x3388}, // calsquare
	{0x09f2217d, 0x00a9}, // copyright
	{0x09f9df24, 0x1e0c}, // Ddotbelow
	{0x0a040d76, 0x098a}, // uubengali
	{0x0a1d800c, 0x0291}, // zcurl
	{0x0a3a2809, 0xf767}, // Gsmall
	{0x0a3b8eb5, 0x044b}, // yericyrillic
	{0x0a46f2f1, 0x0284}, // dotlessjstrokehook
	{0x0a5cb3b1, 0x30d6}, // bukatakana
	{0x0a5ff1a8, 0xff6b}, // osmallkatakanahalfwidth
	{0x0a67f8fb, 0x24b2}, // wparen
	{0x0a704676, 0xfccc}, // lammeeminitialarabic
	{0x0a8ba8e8, 0x0112}, // Emacron
	{0x0a9b47dd, 0x306f}, // hahiragana
	{0x0aa2156d, 0xfc0c}, // tehhahisolatedarabic
	{0x0abb4ec1, 0x0441}, // afii10083
	{0x0ac66fc0, 0x005b}, // bracketleftBigg
	{0x0adbba15, 0x21c0}, // harpoonrightbarbup
	{0x0ae79191, 0x01e1}, // adotmacron
	{0x0aecd30e, 0x05e8}, // reshpatahhebrew
	{0x0af77d49, 0x09ea}, // fourbengali
	{0x0b1d2d0d, 0xf6f3}, // tsuperior
	{0x0b367d7a, 0x0421}, // Escyrillic
	{0x0b4b7082, 0xff62}, // cornerbracketlefthalfwidth
	{0x0b6abf22, 0x20aa}, // sheqel
	{0x0b7f2b2d, 0x0a5a}, // ghhagurmukhi
	{0x0b92d660, 0x32a6}, // ideographiclowcircle
	{0x0b9e2621, 0x2665}, // heartsuitblack
	{0x0ba5f00c, 0x03cc}, // omicrontonos
	{0x0bae12ff, 0xff2a}, // Jmonospace
	{0x0bd4abb3, 0x0254}, // oopen
	{0x0bd8d304, 0x3215}, // ieungaparenkorean
	{0x0bdad647, 0x0970}, // abbreviationsigndeva
	{0x0bdb550e, 0x0669}, // ninehackarabic
	{0x0be3cda3, 0x1e0b}, // ddotaccent
	{0x0bf8ed4a, 0x09f2}, // rupeemarkbengali
	{0x0bfa9d4e, 0x05b6}, // afii57795
	{0x0c138c8e, 0x308f}, // wahiragana
	{0x0c17017e, 0x02a5}, // dzcurl
	{0x0c19fd92, 0x037a}, // ypogegrammeni
	{0x0c255ae5, 0x0553}, // Piwrarmenian
	{0x0c356707, 0x0625}, // afii57413
	{0x0c678de3, 0x032a}, // bridgebelowcmb
	{0x0c810887, 0x0a88}, // iigujarati
	{0x0c83c594, 0x1e63}, // sdotbelow
	{0x0c8f5261, 0x0164}, // Tcaron
	{0x0cacee48, 0xfba8}, // hehinitialaltonearabic
	{0x0cbb507c, 0x3036}, // circlepostalmark
	{0x0cd99820, 0x05c0}, // paseqhebrew
	{0x0cdb81c4, 0x24a2}, // gparen
	{0x0ce8bb7e, 0x30d5}, // hukatakana
	{0x0cf04968, 0x02be}, // ringhalfright
	{0x0d0eb2f0, 0x315d}, // weokorean
	{0x0d21bb72, 0x2550}, // SF430000
	{0x0d3a66b8, 0x2309}, // ceilingrightbigg
	{0x0d47308f, 0x05d4}, // he
	{0x0d747cfe, 0x04c3}, // Kahookcyrillic
	{0x0d932b5b, 0x30d2}, // hikatakana
	{0x0da4d862, 0x05b6}, // segol13
	{0x0db7d6e4, 0x05d4}, // hehebrew
	{0x0dc03ecb, 0x0a9c}, // jagujarati
	{0x0dd6f75d, 0x09f6}, // threenumeratorbengali
	{0x0de664af, 0x01fa}, // Aringacute
	{0x0deddd7b, 0x017a}, // zacute
	{0x0df6966e, 0x1e4f}, // otildedieresis
	{0x0e0870a7, 0x2713}, // checkmark
	{0x0e0aefc5, 0x05af}, // masoracirclehebrew
	{0x0e15512a, 0xff43}, // cmonospace
	{0x0e157c7d, 0x0166}, // Tbar
	{0x0e34eac4, 0x06ba}, // afii57514
	{0x0e359de3, 0x332b}, // paasentosquare
	{0x0e35e57d, 0x01f4}, // Gacute
	{0x0e6ec8aa, 0x0a40}, // iimatragurmukhi
	{0x0e8140cb, 0x2318}, // propellor
	{0x0e8e8ac7, 0x25aa}, // blacksmallsquare
	{0x0e8ed92c, 0x05b3}, // hatafqamatsquarterhebrew
	{0x0e9c1a93, 0x0149}, // quoterightn
	{0x0eb0ce00, 0xff30}, // Pmonospace
	{0x0ec7e019, 0xfc4b}, // noonjeemisolatedarabic
	{0x0ed8b040, 0x33b5}, // nvsquare
	{0x0edd0c59, 0x0e35}, // saraiithai
	{0x0ee06289, 0x05b0}, // shevaquarterhebrew
	{0x0efc1459, 0x09b6}, // shabengali
	{0x0f02712c, 0x00ea}, // ecircumflex
	{0x0f066e82, 0x33b3}, // mssquare
	{0x0f13ed93, 0xf7f8}, // Oslashsmall
	{0x0f1a6991, 0x24c4}, // Ocircle
	{0x0f2768b1, 0x05b1}, // hatafsegolwidehebrew
	{0x0f332d5e, 0x002e}, // period
	{0x0f35dd15, 0x04b0}, // Ustraightstrokecyrillic
	{0x0f432338, 0x0020}, // space
	{0x0f433f21, 0xff7d}, // sukatakanahalfwidth
	{0x0f610d68, 0x1e53}, // omacronacute
	{0x0f61debf, 0x33b9}, // mvmegasquare
	{0x0f8de5e5, 0xff4b}, // kmonospace
	{0x0f984d6e, 0xfec0}, // dadmedialarabic
	{0x0fcfb12d, 0x016c}, // Ubreve
	{0x0fdfc487, 0x02a8}, // tccurl
	{0x0ff64b0b, 0x0665}, // fivehackarabic
	{0x1001b5d7, 0xfc48}, // meemmeemisolatedarabic
	{0x100f790a, 0x20ac}, // Euro
	{0x101477b7, 0x216a}, // Elevenroman
	{0x1030f103, 0x0918}, // ghadeva
	{0x103612b8, 0x2287}, // supersetorequal
	{0x10459048, 0x2217}, // asteriskmath
	{0x10529f46, 0x0e59}, // ninethai
	{0x10656b29, 0x042a}, // afii10044
	{0x10659a4d, 0x0041}, // A
	{0x106ab99c, 0x0943}, // rvocalicvowelsigndeva
	{0x10827219, 0x01e4}, // Gstroke
	{0x1087cbb6, 0x318d}, // araeakorean
	{0x10c49213, 0x094d}, // viramadeva
	{0x10c6b04c, 0xfee8}, // noonmedialarabic
	{0x10e1204e, 0x224c}, // allequal
	{0x1112335e, 0x0e24}, // ruthai
	{0x1113e9a7, 0x0a87}, // igujarati
	{0x11140e23, 0xff50}, // pmonospace
	{0x112533be, 0x0446}, // tsecyrillic
	{0x1128968b, 0x03dc}, // Digammagreek
	{0x113f9725, 0x1ea1}, // adotbelow
	{0x11572f90, 0x3240}, // ideographicfestivalparen
	{0x1159e298, 0x2484}, // seventeenparen
	{0x115b5935, 0xff78}, // kukatakanahalfwidth
	{0x115c3cb2, 0x3179}, // kapyeounssangpieupkorean
	{0x11672a1e, 0x3051}, // kehiragana
	{0x1179ac42, 0x00a0}, // nbspace
	{0x1198b8ba, 0x05f1}, // vavyodhebrew
	{0x11adf5a7, 0x24a9}, // nparen
	{0x11aeb63a, 0x2272}, // lessorequivalent
	{0x11ba40f5, 0x05ea}, // tavhebrew
	{0x11c11092, 0x1e2c}, // Itildebelow
	{0x11c8e4b3, 0x2660}, // spadesuitblack
	{0x11d3eac3, 0x3060}, // dahiragana
	{0x11d67798, 0xfb33}, // daletdagesh
	{0x11e401c1, 0x05d6}, // zayin
	{0x1216fef8, 0x328e}, // ideographmetalcircle
	{0x1219f723, 0xff6e}, // yosmallkatakanahalfwidth
	{0x1224d569, 0xfee4}, // meemmedialarabic
	{0x1228548c, 0x05d0}, // alefhebrew
	{0x1233afe9, 0x30df}, // mikatakana
	{0x123a07fe, 0x05e6}, // tsadihebrew
	{0x1242de49, 0x042b}, // Yericyrillic
	{0x1246d709, 0x315f}, // wikorean
	{0x126c42a3, 0x21d0}, // arrowleftdbl
	{0x1278a8d5, 0x3087}, // yosmallhiragana
	{0x129b0140, 0x0445}, // afii10087
	{0x12a29be8, 0x03c5}, // upsilon
	{0x12bdd9c7, 0x21a6}, // mapsto
	{0x12d3e8f7, 0x30c0}, // dakatakana
	{0x12def4bb, 0xfeba}, // sadfinalarabic
	{0x12e14144, 0xf726}, // ampersandsmall
	{0x12e22bb8, 0x33c2}, // amsquare
	{0x1329e467, 0x0311}, // breveinvertedcmb
	{0x132b7bd0, 0x2a04}, // unionmultitext
	{0x132e0697, 0x01b3}, // Yhook
	{0x13402162, 0xf6be}, // dotlessj
	{0x134d9c31, 0x304c}, // gahiragana
	{0x13597fb3, 0x04d9}, // afii10846
	{0x135d6341, 0x33c8}, // dbsquare
	{0x13635045, 0x0300}, // gravecmb
	{0x1367bcdf, 0xf88a}, // maiekupperleftthai
	{0x1369554a, 0x207f}, // nsuperior
	{0x136e8d95, 0xf885}, // saraileftthai
	{0x1376f153, 0x0403}, // afii10052
	{0x1384d3da, 0x05b4}, // hiriqnarrowhebrew
	{0x13868087, 0x0636}, // afii57430
	{0x138b1601, 0x0621}, // hamzafathatanarabic
	{0x138fca68, 0x0a30}, // ragurmukhi
	{0x13a69297, 0x0917}, // gadeva
	{0x13bc5cc6, 0xf88d}, // maithoupperleftthai
	{0x13bcc567, 0x25d1}, // circlewithrighthalfblack
	{0x13ccaf5f, 0x3061}, // tihiragana
	{0x13dc1f9e, 0x05b1}, // hatafsegol
	{0x13e2dbb5, 0x1e3a}, // Llinebelow
	{0x140e7a7e, 0x0551}, // Coarmenian
	{0x14117f5a, 0x1e77}, // ucircumflexbelow
	{0x141d63ad, 0x1eda}, // Ohornacute
	{0x142c024d, 0x013a}, // lacute
	{0x142c46b3, 0x2a06}, // unionsqdisplay
	{0x143d707c, 0x05a6}, // merkhakefulalefthebrew
	{0x146c75cf, 0x3275}, // ieungacirclekorean
	{0x146f18bb, 0xfb36}, // zayindagesh
	{0x147a2240, 0x1e81}, // wgrave
	{0x1486cc9b, 0x054b}, // Jheharmenian
	{0x14ac715c, 0x2473}, // twentycircle
	{0x14b421a5, 0x0638}, // zaharabic
	{0x14c795cf, 0x0051}, // Q
	{0x14d1cd4b, 0x2284}, // notsubset
	{0x14d2cd8a, 0x012f}, // iogonek
	{0x14eb6d31, 0x02dc}, // tildewider
	{0x14f2bc91, 0x1eab}, // acircumflextilde
	{0x15045632, 0x05d1}, // bet
	{0x1505dc02, 0x3083}, // yasmallhiragana
	{0x152785c0, 0x30a1}, // asmallkatakana
	{0x1532a7b6, 0x05e7}, // qofsegolhebrew
	{0x15642935, 0x23a9}, // braceleftbt
	{0x1564e3f1, 0x05b9}, // holam26
	{0x15771621, 0x042c}, // Softsigncyrillic
	{0x159ac6ab, 0x0591}, // etnahtafoukhlefthebrew
	{0x159b9dc9, 0x24c6}, // Qcircle
	{0x15d1c25e, 0x04bf}, // chedescenderabkhasiancyrillic
	{0x15dd6b0c, 0x25d9}, // whitecircleinverse
	{0x15f64606, 0x33c5}, // cdsquare
	{0x15f8ec13, 0x037e}, // questiongreek
	{0x1602acd3, 0xff38}, // Xmonospace
	{0x1610c2ad, 0x3181}, // yesieungkorean
	{0x161db4d0, 0xf76c}, // Lsmall
	{0x16393f6d, 0xfef6}, // lamalefmaddaabovefinalarabic
	{0x164a5cd1, 0x0058}, // X
	{0x164b6531, 0xf7eb}, // Edieresissmall
	{0x165341dd, 0x02e0}, // gammasuperior
	{0x167b1fac, 0xfebc}, // sadmedialarabic
	{0x168b9d05, 0x04d5}, // aiecyrillic
	{0x169cceb3, 0x005b}, // bracketleft
	{0x16ad260d, 0x0253}, // bhook
	{0x16b23c67, 0x3135}, // nieuncieuckorean
	{0x16ba0a7a, 0x30a9}, // osmallkatakana
	{0x16d5ac91, 0x3018}, // whitetortoiseshellbracketleft
	{0x16f9045d, 0x0a16}, // khagurmukhi
	{0x17093caa, 0x01e3}, // aemacron
	{0x170fc75c, 0x306c}, // nuhiragana
	{0x171b516a, 0x0a39}, // hagurmukhi
	{0x17301afd, 0x3023}, // threehangzhou
	{0x17314b3c, 0xf7a8}, // Dieresissmall
	{0x173c8456, 0x338b}, // nfsquare
	{0x178d45ae, 0x337e}, // meizierasquare
	{0x17a9e49b, 0x322a}, // ideographicmoonparen
	{0x17ad5313, 0x09ab}, // phabengali
	{0x17da2afa, 0x0335}, // strokeshortoverlaycmb
	{0x17deda0e, 0x00ed}, // iacute
	{0x17e36acb, 0x05b8}, // qamats
	{0x17ef9c62, 0x0a05}, // agurmukhi
	{0x180419cd, 0xfc62}, // shaddakasraarabic
	{0x18120be8, 0x200b}, // zerowidthspace
	{0x18175789, 0x05b3}, // hatafqamatswidehebrew
	{0x181f5c91, 0x091f}, // ttadeva
	{0x182bd096, 0x0148}, // ncaron
	{0x183028bb, 0x04da}, // Schwadieresiscyrillic
	{0x1840ed9a, 0xfea0}, // jeemmedialarabic
	{0x18492236, 0x0e5a}, // angkhankhuthai
	{0x1855a5e3, 0xfe5f}, // numbersignsmall
	{0x185b816d, 0x0146}, // ncommaaccent
	{0x187ab455, 0x0e42}, // saraothai
	{0x1885706c, 0x3157}, // okorean
	{0x18905b85, 0x05e8}, // reshsegol
	{0x18923bff, 0x2135}, // aleph
	{0x1892a13e, 0x005d}, // bracketrightbigg
	{0x18a1a8e1, 0x02c2}, // a40
	{0x18a5c10a, 0x2584}, // dnblock
	{0x18a72a69, 0xfb3a}, // finalkafdageshhebrew
	{0x18b82c53, 0x045c}, // afii10109
	{0x18c46fec, 0x033c}, // seagullbelowcmb
	{0x18d47bfd, 0x3298}, // ideographiclaborcircle
	{0x18dbddd2, 0x0993}, // obengali
	{0x18df8652, 0x2286}, // subsetorequal
	{0x18ea53c7, 0x0663}, // afii57395
	{0x18ec3f78, 0x05e8}, // afii57688
	{0x18f0d507, 0xfe9f}, // jeeminitialarabic
	{0x18fb8128, 0x25c1}, // a1
	{0x190a56aa, 0xfe5a}, // parenrightsmall
	{0x1920c2c2, 0xff65}, // middledotkatakanahalfwidth
	{0x1925bd4b, 0xf6f9}, // Lslashsmall
	{0x192c8826, 0x2283}, // superset
	{0x1940b4fd, 0x2251}, // geometricallyequal
	{0x19427103, 0x2a01}, // circleplustext
	{0x1946a31f, 0x05ae}, // zinorhebrew
	{0x194ec3dd, 0xfb38}, // tetdageshhebrew
	{0x195fdeed, 0x3013}, // getamark
	{0x197f461f, 0x1e03}, // bdotaccent
	{0x1991ce27, 0x007b}, // braceleftBigg
	{0x1994a487, 0x0481}, // koppacyrillic
	{0x19985117, 0x339a}, // nmsquare
	{0x19a1a98b, 0x00ad}, // sfthyphen
	{0x19ad4aac, 0xfef2}, // yehfinalarabic
	{0x19b22272, 0x3269}, // chieuchcirclekorean
	{0x19b344e4, 0x0123}, // gcedilla
	{0x19b3f208, 0x05b8}, // qamats27
	{0x19d833fe, 0x02ba}, // dblprimemod
	{0x19dea593, 0x0428}, // afii10042
	{0x19f7c320, 0x04d1}, // abrevecyrillic
	{0x19fa29b1, 0x327f}, // koreanstandardsymbol
	{0x1a00d3da, 0x032b}, // dblarchinvertedbelowcmb
	{0x1a260946, 0x0447}, // checyrillic
	{0x1a287ed9, 0x0922}, // ddhadeva
	{0x1a28dcc1, 0x005b}, // bracketleftBig
	{0x1a3b33cb, 0xfed7}, // qafinitialarabic
	{0x1a3bf649, 0x05b2}, // hatafpatahhebrew
	{0x1a4f3484, 0x0913}, // odeva
	{0x1a533d39, 0x0306}, // brevecmb
	{0x1ac76244, 0x0425}, // Khacyrillic
	{0x1ac90440, 0x010f}, // dcaron
	{0x1b240a74, 0x00d8}, // Oslash
	{0x1b2b4015, 0xfe65}, // greatersmall
	{0x1b33167a, 0xfe3e}, // dblanglebracketrightvertical
	{0x1b342691, 0x017c}, // zdot
	{0x1b39339b, 0x253c}, // SF050000
	{0x1b3f9c21, 0x1e72}, // Udieresisbelow
	{0x1b4399b2, 0x04c7}, // Enhookcyrillic
	{0x1b5874ea, 0x091b}, // chadeva
	{0x1b8c8992, 0x2122}, // trademark
	{0x1ba72293, 0xff03}, // numbersignmonospace
	{0x1bbb8fb3, 0x0477}, // izhitsadblgravecyrillic
	{0x1bcb2bfd, 0x337d}, // taisyouerasquare
	{0x1be98812, 0x24d4}, // ecircle
	{0x1c079308, 0x09cc}, // auvowelsignbengali
	{0x1c16ebae, 0x0126}, // Hbar
	{0x1c2631dd, 0x3385}, // KBsquare
	{0x1c56e166, 0xfd3f}, // parenrightaltonearabic
	{0x1c70f0f3, 0xf7e7}, // Ccedillasmall
	{0x1c94529b, 0x21b5}, // carriagereturn
	{0x1cb10674, 0x0a27}, // dhagurmukhi
	{0x1cb5367a, 0x0532}, // Benarmenian
	{0x1cdb3e05, 0x2514}, // SF020000
	{0x1ce17c08, 0xfea6}, // khahfinalarabic
	{0x1ce29209, 0x02cc}, // verticallinelowmod
	{0x1ce40996, 0xff3b}, // bracketleftmonospace
	{0x1cf31612, 0x0598}, // zarqahebrew
	{0x1d2559c8, 0x03b9}, // iota
	{0x1d25e3ef, 0x043c}, // emcyrillic
	{0x1d420ccb, 0x0430}, // afii10065
	{0x1d91efe5, 0xf6d8}, // dieresisgrave
	{0x1d954d85, 0x09a4}, // tabengali
	{0x1da7ae8c, 0x00d7}, // multiply
	{0x1db92094, 0x053d}, // Xeharmenian
	{0x1dbbea92, 0x02e6}, // tonebarhighmod
	{0x1dbc2a14, 0x2245}, // congruent
	{0x1dbe86c5, 0x246d}, // fourteencircle
	{0x1dc0644e, 0x2012}, // figuredash
	{0x1ded853e, 0x01cc}, // nj
	{0x1df7638a, 0x00e4}, // adieresis
	{0x1dfc2837, 0x217a}, // elevenroman
	{0x1e0312fa, 0x0290}, // zretroflexhook
	{0x1e065c3a, 0xf7f1}, // Ntildesmall
	{0x1e1332fd, 0x0287}, // tturned
	{0x1e25355a, 0x0461}, // omegacyrillic
	{0x1e38c8d8, 0x012a}, // Imacron
	{0x1e56afb4, 0xfc61}, // shaddadammaarabic
	{0x1e645fd0, 0x00b2}, // twosuperior
	{0x1e6cddcb, 0x0563}, // gimarmenian
	{0x1e7c9862, 0x338f}, // squarekg
	{0x1e82a7c3, 0x2302}, // house
	{0x1e860782, 0x0983}, // visargabengali
	{0x1e917582, 0x09a2}, // ddhabengali
	{0x1ea7d113, 0x031c}, // ringhalfleftbelowcmb
	{0x1eae034a, 0x3122}, // anbopomofo
	{0x1eae1716, 0x0434}, // afii10069
	{0x1edf9913, 0x1e16}, // Emacronacute
	{0x1ef123d5, 0x0413}, // Gecyrillic
	{0x1f06ae25, 0xfb31}, // betdageshhebrew
	{0x1f0c6a2a, 0x0651}, // shaddafathatanarabic
	{0x1f1df6d1, 0x3381}, // nasquare
	{0x1f2383d4, 0x0004}, // controlEOT
	{0x1f2a4214, 0xfdfa}, // sallallahoualayhewasallamarabic
	{0x1f5314ee, 0xf6fc}, // Ringsmall
	{0x1f5dc79d, 0xf7b4}, // Acutesmall
	{0x1f60dad4, 0x05e8}, // reshhebrew
	{0x1f686fbe, 0xff2c}, // Lmonospace
	{0x1f90aeb4, 0xff7b}, // sakatakanahalfwidth
	{0x1f93ce2b, 0xff33}, // Smonospace
	{0x1f9b52d4, 0x0323}, // dotbelowcmb
	{0x1faa2fdf, 0x203e}, // overline
	{0x1faf20cc, 0x02b9}, // primemod
	{0x1fba9d24, 0x1e4a}, // Ncircumflexbelow
	{0x1fc00e7e, 0x0050}, // P
	{0x1fc99492, 0x3126}, // erbopomofo
	{0x1feea4ab, 0x0623}, // afii57411
	{0x20061138, 0x1e02}, // Bdotaccent
	{0x200e9513, 0x030f}, // dblgravecmb
	{0x2012c16a, 0xf6d4}, // cyrbreve
	{0x201a6676, 0x2310}, // logicalnotreversed
	{0x201e3ee9, 0x0310}, // candrabinducmb
	{0x2024cfd1, 0x0475}, // afii10196
	{0x2038e6d5, 0x307a}, // pehiragana
	{0x205d0ec7, 0x03a0}, // Pi
	{0x206a255a, 0xf775}, // Usmall
	{0x207bf81e, 0xff84}, // tokatakanahalfwidth
	{0x2086c569, 0x00d2}, // Ograve
	{0x2086fb87, 0xf772}, // Rsmall
	{0x208b3b40, 0xfed0}, // ghainmedialarabic
	{0x209bfca6, 0x24d5}, // fcircle
	{0x20a31ebd, 0x2489}, // twoperiod
	{0x20abefc1, 0x0385}, // dialytikatonos
	{0x20b8cfa5, 0x0443}, // afii10085
	{0x20bc528f, 0x2042}, // asterism
	{0x20ca67b9, 0x09a1}, // ddabengali
	{0x20ce09b9, 0xfb7b}, // tchehfinalarabic
	{0x20dd36ea, 0x207e}, // parenrightsuperior
	{0x20e840a0, 0x30d4}, // pikatakana
	{0x2111e869, 0x092a}, // padeva
	{0x21185c0e, 0x028e}, // yturned
	{0x2130f98f, 0x0aa0}, // tthagujarati
	{0x215fc096, 0x000d}, // controlCR
	{0x21622652, 0x249e}, // cparen
	{0x21669982, 0x0181}, // Bhook
	{0x217143de, 0x05b3}, // hatafqamats34
	{0x2173a28b, 0x0295}, // glottalstopreversed
	{0x218d5b62, 0x3232}, // ideographichaveparen
	{0x21b37808, 0x2477}, // fourparen
	{0x21bef9d7, 0x0e06}, // khorakhangthai
	{0x21c0f31f, 0x013b}, // Lcommaaccent
	{0x21ce0071, 0x2179}, // tenroman
	{0x21d28f9e, 0x05e9}, // shin
	{0x21f74f30, 0x043e}, // afii10080
	{0x2203033d, 0x019d}, // Nhookleft
	{0x22037801, 0x044b}, // afii10093
	{0x220f1331, 0x05db}, // kafhebrew
	{0x2221c36b, 0x06f6}, // sixpersian
	{0x22351581, 0x059f}, // qarneyparahebrew
	{0x223cb30f, 0xfc9f}, // behmeeminitialarabic
	{0x225099c3, 0x3149}, // ssangcieuckorean
	{0x22542ce4, 0x2207}, // nabla
	{0x22702420, 0x3153}, // eokorean
	{0x2274db8f, 0xff08}, // parenleftmonospace
	{0x2282e86d, 0x00fa}, // uacute
	{0x22a883d2, 0x25c7}, // whitediamond
	{0x22b082b8, 0x005c}, // backslashBigg
	{0x22c93a2b, 0x338d}, // mugsquare
	{0x22e3ac9e, 0xff4c}, // lmonospace
	{0x2320c842, 0x0277}, // omegalatinclosed
	{0x2324e69c, 0x2665}, // heart
	{0x2351e945, 0x0a8a}, // uugujarati
	{0x2352b38c, 0xf6d7}, // dieresisacute
	{0x2356706f, 0x0541}, // Jaarmenian
	{0x23679572, 0x1ef1}, // uhorndotbelow
	{0x236d79e4, 0x301e}, // quotedblprime
	{0x237979f3, 0xff73}, // ukatakanahalfwidth
	{0x23947e9f, 0x0266}, // hhook
	{0x23b6327c, 0x304f}, // kuhiragana
	{0x23b9408d, 0x2208}, // element
	{0x23d291f5, 0x0639}, // ainarabic
	{0x23d42e51, 0xff0b}, // plusmonospace
	{0x23e15f1c, 0xff90}, // mikatakanahalfwidth
	{0x23e70b3f, 0x0664}, // afii57396
	{0x23e9f46b, 0xff80}, // takatakanahalfwidth
	{0x23ea33c4, 0x30f1}, // wekatakana
	{0x23f0f542, 0xff2f}, // Omonospace
	{0x2415f58f, 0x3069}, // dohiragana
	{0x2419b09a, 0x038f}, // Omegatonos
	{0x241b0583, 0x0a9e}, // nyagujarati
	{0x241cc39b, 0x007b}, // braceleftbigg
	{0x241f6494, 0x3105}, // bbopomofo
	{0x2459d6f7, 0x0aea}, // fourgujarati
	{0x24688db0, 0x02c1}, // glottalstopreversedmod
	{0x24776c38, 0x0343}, // koroniscmb
	{0x2479e0d2, 0x3021}, // onehangzhou
	{0x249dd6ee, 0x0665}, // afii57397
	{0x24c0efcc, 0x316d}, // rieulyeorinhieuhkorean
	{0x24ccbd1b, 0x09e6}, // zerobengali
	{0x24d11d48, 0x02c2}, // arrowheadleftmod
	{0x24d6b19b, 0xfee7}, // nooninitialarabic
	{0x24deab24, 0x25a4}, // squarehorizontalfill
	{0x24e9b4e5, 0x21e4}, // arrowtableft
	{0x24f5be18, 0x0308}, // dieresiscmb
	{0x2527d2da, 0x300c}, // cornerbracketleft
	{0x252ef6ac, 0x01de}, // Adieresismacron
	{0x2533ec5c, 0x05e8}, // reshtsere
	{0x253f33be, 0x006e}, // n
	{0x2543ce81, 0x05c1}, // shindothebrew
	{0x254b8857, 0x0929}, // nnnadeva
	{0x2556a01a, 0x005c}, // backslashbigg
	{0x25725d01, 0x0496}, // Zhedescendercyrillic
	{0x2592eb8c, 0x1e39}, // ldotbelowmacron
	{0x25a29a9e, 0x000a}, // controlLF
	{0x25bfabf1, 0x056c}, // liwnarmenian
	{0x25c22a8d, 0x0927}, // dhadeva
	{0x25c3f8ae, 0xff3e}, // asciicircummonospace
	{0x25ca4638, 0x0576}, // nowarmenian
	{0x25d83051, 0x0e19}, // nonuthai
	{0x25e64388, 0x1ee5}, // udotbelow
	{0x25eb8755, 0xff34}, // Tmonospace
	{0x25ef68be, 0x0103}, // abreve
	{0x25f0117e, 0x095f}, // yyadeva
	{0x2614c40e, 0x0aaf}, // yagujarati
	{0x262be20f, 0x0471}, // psicyrillic
	{0x2637caf5, 0x0932}, // ladeva
	{0x263e0b92, 0x09ac}, // babengali
	{0x2652690c, 0x3137}, // tikeutkorean
	{0x265be8ad, 0x01cf}, // Icaron
	{0x265fdad4, 0x05b8}, // qamatshebrew
	{0x26621b4d, 0x0072}, // r
	{0x2669a6cb, 0x211c}, // Rfraktur
	{0x26837556, 0x0048}, // H
	{0x26a72a71, 0x3188}, // yoyaekorean
	{0x26ab6d3a, 0x1ed9}, // ocircumflexdotbelow
	{0x26af8089, 0x304e}, // gihiragana
	{0x26b9c1d9, 0x0e30}, // saraathai
	{0x26c43ea8, 0x0911}, // ocandradeva
	{0x26caa803, 0x21d4}, // arrowdblboth
	{0x26cb382e, 0xf8fc}, // bracerighttp
	{0x26f6f992, 0x04e4}, // Idieresiscyrillic
	{0x26fb496b, 0x3351}, // rittorusquare
	{0x27075678, 0x012e}, // Iogonek
	{0x271cd3fa, 0x27e8}, // angbracketleftBig
	{0x2731451f, 0x0408}, // afii10057
	{0x273977ae, 0x0435}, // afii10070
	{0x27487450, 0x007e}, // asciitilde
	{0x275654ea, 0x099b}, // chabengali
	{0x275a3e46, 0x09a0}, // tthabengali
	{0x275a5b3b, 0x1ee1}, // ohorntilde
	{0x27607db6, 0x05b4}, // hiriq2d
	{0x27a2d18f, 0x05b9}, // holam32
	{0x27b7bf17, 0x031f}, // plusbelowcmb
	{0x27b7cc70, 0xfb2d}, // shindageshsindothebrew
	{0x27d62d65, 0x200d}, // afii301
	{0x27d8dd28, 0x00e8}, // egrave
	{0x27e6d9df, 0x0271}, // mhook
	{0x27f31ec3, 0x3271}, // rieulacirclekorean
	{0x283f3216, 0x2075}, // fivesuperior
	{0x28898020, 0x0281}, // Rsmallinverted
	{0x288edd59, 0x24cc}, // Wcircle
	{0x2892f3ea, 0x2a02}, // circlemultiplydisplay
	{0x28d9fe2f, 0x0e41}, // saraaethai
	{0x28dd4931, 0x3234}, // ideographicnameparen
	{0x28e31924, 0x0a6c}, // sixgurmukhi
	{0x28e4057d, 0xfc58}, // yehmeemisolatedarabic
	{0x28f71acd, 0x0633}, // afii57427
	{0x290ed42e, 0x1e18}, // Ecircumflexbelow
	{0x2916bf1f, 0xf7ef}, // Idieresissmall
	{0x29287ce1, 0x03d2}, // Upsilon1
	{0x29346ac3, 0x2021}, // daggerdbl
	{0x29371ded, 0x1e8f}, // ydotaccent
	{0x2951ba53, 0xf6e2}, // commasuperior
	{0x29734fd7, 0x092d}, // bhadeva
	{0x298e1b46, 0x01ea}, // Oogonek
	{0x29958ce9, 0x0005}, // controlENQ
	{0x29a8f130, 0x04d4}, // Aiecyrillic
	{0x29ab0eda, 0x0146}, // ncedilla
	{0x29b0158e, 0x05e7}, // qofshevahebrew
	{0x29be1625, 0x0028}, // parenleftbig
	{0x29c07bd0, 0x305e}, // zohiragana
	{0x29d0597d, 0x262f}, // yinyang
	{0x29d51dd2, 0x06d1}, // yehthreedotsbelowarabic
	{0x29d5f944, 0x03e4}, // Feicoptic
	{0x29e31b8f, 0x05b1}, // hatafsegolhebrew
	{0x29e6e487, 0x30cc}, // nukatakana
	{0x29e7f6bb, 0x0035}, // five
	{0x2a0fca4c, 0x3085}, // yusmallhiragana
	{0x2a239937, 0x0375}, // numeralsignlowergreek
	{0x2a28d54e, 0x30aa}, // okatakana
	{0x2a619f09, 0x2660}, // spade
	{0x2a645de3, 0x3091}, // wehiragana
	{0x2a64e363, 0x226e}, // notless
	{0x2a6a5dc5, 0x3088}, // yohiragana
	{0x2a72414c, 0x0e44}, // saraaimaimalaithai
	{0x2a8e7779, 0x0ad0}, // omgujarati
	{0x2ac0e180, 0x33a6}, // kmcubedsquare
	{0x2ac17d0f, 0x00f3}, // oacute
	{0x2acd450a, 0x002f}, // slashbig
	{0x2ad04347, 0xed12}, // arrowvertexdbl
	{0x2ae749c7, 0x00e1}, // aacute
	{0x2ae8215f, 0x001d}, // controlGS
	{0x2ae8c5a6, 0x21c4}, // arrowrightoverleft
	{0x2aedcd2c, 0x09fa}, // issharbengali
	{0x2afb134c, 0xfb03}, // ffi
	{0x2b2385f1, 0xf760}, // Gravesmall
	{0x2b27655e, 0x261c}, // pointingindexleftwhite
	{0x2b3ff353, 0x2557}, // SF250000
	{0x2b42ad49, 0xfe55}, // colonsmall
	{0x2b5ac1a6, 0x0045}, // E
	{0x2b84f841, 0x30a6}, // ukatakana
	{0x2b91651d, 0x0319}, // righttackbelowcmb
	{0x2b93dd53, 0x0661}, // afii57393
	{0x2b9750e3, 0x2200}, // universal
	{0x2b9c968a, 0x0315}, // commaaboverightcmb
	{0x2bb5189c, 0x25e3}, // blacklowerlefttriangle
	{0x2bb9b9bf, 0x0621}, // hamzafathaarabic
	{0x2bd253b1, 0x09bf}, // ivowelsignbengali
	{0x2be6415c, 0x05b3}, // hatafqamatshebrew
	{0x2beb7c62, 0x013c}, // lcommaaccent
	{0x2c0f9c0b, 0x0912}, // oshortdeva
	{0x2c15e9f4, 0x091c}, // jadeva
	{0x2c1b74f9, 0x2262}, // notidentical
	{0x2c4944e4, 0x0393}, // Gamma
	{0x2c91b61a, 0x0640}, // kashidaautonosidebearingarabic
	{0x2c964b66, 0x334a}, // miribaarusquare
	{0x2cb17e35, 0xfec6}, // zahfinalarabic
	{0x2cd9d9ec, 0xfc6d}, // behnoonfinalarabic
	{0x2cdfcd2a, 0x30cb}, // nikatakana
	{0x2cf9daf5, 0x3228}, // nineideographicparen
	{0x2d2e3883, 0x2190}, // arrowleft
	{0x2d317780, 0xf778}, // Xsmall
	{0x2d329c65, 0xfb3c}, // lameddageshhebrew
	{0x2d39ea69, 0x019e}, // nlegrightlong
	{0x2d3b565d, 0x05ea}, // afii57690
	{0x2d456f79, 0x25e2}, // blacklowerrighttriangle
	{0x2d52bd2b, 0x3326}, // dorusquare
	{0x2d84140a, 0x0293}, // ezhcurl
	{0x2d8d5b1d, 0x33d0}, // lmsquare
	{0x2d95d169, 0x3155}, // yeokorean
	{0x2d975eca, 0xfb04}, // ffl
	{0x2d9ae85a, 0x0661}, // onehackarabic
	{0x2d9b14ff, 0xfb6d}, // vehmedialarabic
	{0x2da2ea79, 0x313d}, // rieulsioskorean
	{0x2daa1b6f, 0x090a}, // uudeva
	{0x2db9bc28, 0xf7e3}, // Atildesmall
	{0x2e04353d, 0x1ea9}, // acircumflexhookabove
	{0x2e043b05, 0xff0a}, // asteriskmonospace
	{0x2e10a2b1, 0x0374}, // numeralsigngreek
	{0x2e1b300e, 0x05e8}, // reshpatah
	{0x2e1eaa7d, 0xfb49}, // shindagesh
	{0x2e2c25c0, 0x0392}, // Beta
	{0x2e356485, 0x09c2}, // uuvowelsignbengali
	{0x2e4224af, 0x05e4}, // afii57684
	{0x2e5e0023, 0x0a20}, // tthagurmukhi
	{0x2e647759, 0x33d6}, // molsquare
	{0x2e7c6436, 0x3048}, // ehiragana
	{0x2e915a9d, 0xfe3b}, // blacklenticularbracketleftvertical
	{0x2e9d6cac, 0xf7f0}, // Ethsmall
	{0x2e9e14a6, 0x0454}, // ecyrillic
	{0x2e9ef541, 0x0192}, // florin
	{0x2ea8b970, 0x327a}, // phieuphacirclekorean
	{0x2eb22aa1, 0x01dc}, // udieresisgrave
	{0x2eb3bdc7, 0x027e}, // rfishhook
	{0x2eb455b0, 0x25b6}, // blackrightpointingtriangle
	{0x2ec430ea, 0x0078}, // x
	{0x2ec8352a, 0x0143}, // Nacute
	{0x2eea1838, 0x0170}, // Uhungarumlaut
	{0x2efaa14d, 0x3015}, // tortoiseshellbracketright
	{0x2f06c380, 0x0a91}, // ocandragujarati
	{0x2f0e722f, 0x0021}, // exclam
	{0x2f14e4ad, 0x0183}, // btopbar
	{0x2f1fd59c, 0xf6d1}, // cyrBreve
	{0x2f22b335, 0x305f}, // tahiragana
	{0x2f247a45, 0x00b5}, // mu
	{0x2f42e9c9, 0x0405}, // afii10054
	{0x2f4b01e9, 0x03e0}, // Sampigreek
	{0x2f51a2a5, 0x0591}, // etnahtahebrew
	{0x2f5af1cc, 0xfec4}, // tahmedialarabic
	{0x2f5c74c9, 0x30a4}, // ikatakana
	{0x2f5e2692, 0x033f}, // dbloverlinecmb
	{0x2f7e3ce0, 0xfb35}, // vavdagesh65
	{0x2f7fe7da, 0x04ea}, // Obarreddieresiscyrillic
	{0x2f8f84ed, 0x3019}, // whitetortoiseshellbracketright
	{0x2f9c7ff4, 0x221f}, // orthogonal
	{0x2fa13b0c, 0x0309}, // hookcmb
	{0x2fa6d960, 0x0e05}, // khokhonthai
	{0x2fb88e89, 0x0492}, // Ghestrokecyrillic
	{0x2fe066dc, 0x24c8}, // Scircle
	{0x2ff9eee3, 0x3231}, // ideographicstockparen
	{0x30132e73, 0x2474}, // oneparen
	{0x302d72c2, 0xfb2b}, // shinsindothebrew
	{0x3033e257, 0x0416}, // Zhecyrillic
	{0x3034a6d0, 0x0213}, // rinvertedbreve
	{0x3043436d, 0x0208}, // Idblgrave
	{0x3046485b, 0x0a41}, // umatragurmukhi
	{0x30585e10, 0x0631}, // reharabic
	{0x305b2089, 0x006d}, // m
	{0x305dc9d7, 0x2480}, // thirteenparen
	{0x3097f64c, 0x010c}, // Ccaron
	{0x30b4b18c, 0x0e28}, // sosalathai
	{0x30c0161e, 0x01d3}, // Ucaron
	{0x30c9bc6e, 0x0a1e}, // nyagurmukhi
	{0x30cd38cb, 0x33d8}, // pmsquare
	{0x30d25d42, 0x2089}, // nineinferior
	{0x30e6287a, 0x05e7}, // qofhatafpatah
	{0x30eada85, 0x2219}, // bulletoperator
	{0x30f64fef, 0x20aa}, // sheqelhebrew
	{0x310a4774, 0x337f}, // corporationsquare
	{0x31158bc3, 0x339f}, // mmsquaredsquare
	{0x3116a838, 0x0647}, // heharabic
	{0x312db4ff, 0x2040}, // tie
	{0x313237dd, 0xff32}, // Rmonospace
	{0x313f3c20, 0x09df}, // yyabengali
	{0x3154a912, 0x04ab}, // esdescendercyrillic
	{0x31f1489a, 0x0643}, // afii57443
	{0x31f928de, 0x05df}, // finalnun
	{0x31fab77d, 0x1eea}, // Uhorngrave
	{0x3207407e, 0x046b}, // yusbigcyrillic
	{0x3225e9b9, 0x05dc}, // lamedholamhebrew
	{0x3238fa28, 0x3004}, // jis
	{0x323ea229, 0x33ac}, // gpasquare
	{0x324496b3, 0x33bf}, // mwmegasquare
	{0x3248fa12, 0xfe5e}, // tortoiseshellbracketrightsmall
	{0x324bc39e, 0x22da}, // lessequalorgreater
	{0x3266451e, 0x30f2}, // wokatakana
	{0x326ca1fd, 0x013f}, // Ldot
	{0x328fa9de, 0x230b}, // floorrightBigg
	{0x32930f95, 0x046f}, // ksicyrillic
	{0x329a975f, 0x0a0a}, // uugurmukhi
	{0x329ed55c, 0x04e1}, // dzeabkhasiancyrillic
	{0x32d855b8, 0x0ae9}, // threegujarati
	{0x32e77f07, 0x22c2}, // intersectiontext
	{0x32fd46a7, 0x041e}, // afii10032
	{0x32fe41c6, 0x0667}, // afii57399
	{0x330263f2, 0x3136}, // nieunhieuhkorean
	{0x3303dbcb, 0x04b2}, // Hadescendercyrillic
	{0x33042de7, 0x1e90}, // Zcircumflex
	{0x33231bf5, 0x27e8}, // angbracketleftbigg
	{0x335a816e, 0xf739}, // nineoldstyle
	{0x335e3259, 0x3022}, // twohangzhou
	{0x336106b8, 0x05d7}, // afii57671
	{0x33821f87, 0x05e2}, // ayinhebrew
	{0x33849fcd, 0x304d}, // kihiragana
	{0x339cb29c, 0x03cb}, // upsilondieresis
	{0x339dfd30, 0x2200}, // forall
	{0x33a4598f, 0x0e10}, // thothanthai
	{0x33dd39ff, 0x0437}, // zecyrillic
	{0x33ea63c8, 0x24df}, // pcircle
	{0x33f59002, 0x0285}, // eshsquatreversed
	{0x340746a2, 0xff4a}, // jmonospace
	{0x340dcbd1, 0x21d1}, // arrowdbltp
	{0x3425ba3a, 0x056d}, // xeharmenian
	{0x342d52b5, 0x25cb}, // whitecircle
	{0x34399add, 0x0034}, // four
	{0x343b0ff5, 0x1e8b}, // xdotaccent
	{0x343f59f3, 0x0437}, // afii10073
	{0x344b950b, 0x031d}, // uptackbelowcmb
	{0x34515ec0, 0x01ca}, // NJ
	{0x345791f2, 0x027c}, // rlongleg
	{0x346086da, 0x2286}, // reflexsubset
	{0x3463147c, 0x1e69}, // sdotbelowdotaccent
	{0x3471790f, 0xf6f7}, // Dotaccentsmall
	{0x3490ad97, 0x047a}, // Omegaroundcyrillic
	{0x34a7b989, 0x006f}, // o
	{0x34a88183, 0x2640}, // female
	{0x34b5f401, 0x016a}, // Umacron
	{0x34bab99c, 0x24e0}, // qcircle
	{0x34c88fb4, 0x2312}, // arc
	{0x34c908a5, 0x1eb4}, // Abrevetilde
	{0x34d1f962, 0x05e8}, // reshholamhebrew
	{0x34df3d1c, 0x00a1}, // exclamdown
	{0x34e969fb, 0x05a0}, // telishagedolahebrew
	{0x34f9cd37, 0x010b}, // cdot
	{0x35188ac4, 0xfeac}, // thalfinalarabic
	{0x351bf85e, 0x0257}, // dhook
	{0x351e7136, 0x0482}, // thousandcyrillic
	{0x352bce90, 0x00a3}, // sterling
	{0x35378756, 0x2591}, // shadelight
	{0x35440d94, 0x040c}, // Kjecyrillic
	{0x354dad21, 0x0129}, // itilde
	{0x35514624, 0x00d6}, // Odieresis
	{0x357c478e, 0xf736}, // sixoldstyle
	{0x3596098d, 0xfb9f}, // noonghunnafinalarabic
	{0x359e9c03, 0x0418}, // Iicyrillic
	{0x359f6846, 0x1ebc}, // Etilde
	{0x35a9ba78, 0x0a95}, // kagujarati
	{0x35b65af3, 0x33af}, // radoverssquaredsquare
	{0x35ce2a2d, 0x066a}, // afii57381
	{0x35ddec6f, 0x00eb}, // edieresis
	{0x35e5fe3b, 0x3118}, // cbopomofo
	{0x3613bad2, 0x011b}, // ecaron
	{0x361cb4c9, 0xfb6b}, // vehfinalarabic
	{0x361e70b8, 0x0043}, // C
	{0x362459fb, 0xfe97}, // tehinitialarabic
	{0x362bfa3a, 0xf732}, // twooldstyle
	{0x362c9d3a, 0x02b0}, // hsuperior
	{0x36678fdf, 0x2326}, // deleteright
	{0x3673a47b, 0x05e7}, // qofqamatshebrew
	{0x3676afc0, 0x0ac9}, // ocandravowelsigngujarati
	{0x368bf72e, 0x0038}, // eight
	{0x36aaad0e, 0x307b}, // hohiragana
	{0x36d3bc14, 0x0491}, // afii10098
	{0x36d54fb7, 0x015a}, // Sacute
	{0x37117bac, 0x059b}, // tevirlefthebrew
	{0x372986de, 0x05f3}, // gereshhebrew
	{0x373e5e3f, 0xff27}, // Gmonospace
	{0x37497fde, 0x25b7}, // a3
	{0x3781d925, 0xf6ed}, // isuperior
	{0x37866f5c, 0x00df}, // germandbls
	{0x37a73b5a, 0xfee0}, // lammedialarabic
	{0x37a77cc1, 0x030b}, // hungarumlautcmb
	{0x37b36429, 0xff68}, // ismallkatakanahalfwidth
	{0x37b7557e, 0x001c}, // controlFS
	{0x37c2175b, 0xfcca}, // lamhahinitialarabic
	{0x37c23820, 0x24c5}, // Pcircle
	{0x37d669b4, 0x090e}, // eshortdeva
	{0x37e460db, 0x308d}, // rohiragana
	{0x37e5061e, 0x04a3}, // endescendercyrillic
	{0x37f67ca7, 0x029a}, // eopenclosed
	{0x380974a8, 0xf735}, // fiveoldstyle
	{0x381040c4, 0x310d}, // gbopomofo
	{0x381640e0, 0xffe3}, // macronmonospace
	{0x381c7e4d, 0x1ece}, // Ohookabove
	{0x38201bde, 0x0961}, // llvocalicdeva
	{0x38291591, 0x0e2e}, // honokhukthai
	{0x3839681b, 0x3055}, // sahiragana
	{0x3863c9a6, 0x208e}, // parenrightinferior
	{0x386cda71, 0x01f0}, // jcaron
	{0x38746563, 0x01b0}, // uhorn
	{0x38880d1d, 0x0198}, // Khook
	{0x38885f68, 0x091d}, // jhadeva
	{0x3889a61f, 0x005b}, // bracketleftbigg
	{0x388ccfdd, 0x30e3}, // yasmallkatakana
	{0x3896be1c, 0x0025}, // percent
	{0x38a80af1, 0x306d}, // nehiragana
	{0x38bcbef5, 0x05b7}, // afii57798
	{0x38bebcf5, 0x2552}, // SF510000
	{0x38ce8c39, 0x059a}, // yetivhebrew
	{0x38cea50a, 0x24d6}, // gcircle
	{0x38d9b559, 0x3003}, // dittomark
	{0x38de4662, 0x1e7c}, // Vtilde
	{0x38e73ed2, 0x01c0}, // clickdental
	{0x38efc9e4, 0x04a6}, // Pemiddlehookcyrillic
	{0x391bc4d9, 0xf6d3}, // dblGrave
	{0x391e728b, 0x099a}, // cabengali
	{0x392c00af, 0x3305}, // intisquare
	{0x3934b5de, 0x0e43}, // saraaimaimuanthai
	{0x3946429a, 0x3010}, // blacklenticularbracketleft
	{0x396642a3, 0x322d}, // ideographicwoodparen
	{0x396b44dd, 0x0a3c}, // nuktagurmukhi
	{0x396d8b52, 0x0930}, // radeva
	{0x3989511a, 0xfc5e}, // shaddadammatanarabic
	{0x39918d40, 0x30b6}, // zakatakana
	{0x399ac15a, 0x0283}, // esh
	{0x39aa6d90, 0x0a83}, // visargagujarati
	{0x39b06752, 0x004a}, // J
	{0x39d0b19b, 0x00ae}, // registered
	{0x39d5a1b1, 0x019c}, // Mturned
	{0x39e226d5, 0x05e3}, // afii57683
	{0x39f864d5, 0x2462}, // threecircle
	{0x39fd88e8, 0x05a8}, // qadmahebrew
	{0x3a029acd, 0x338e}, // squaremg
	{0x3a0e66b8, 0x1ee6}, // Uhookabove
	{0x3a1089b2, 0x0073}, // s
	{0x3a3b69ae, 0x317c}, // siostikeutkorean
	{0x3a3c0e00, 0xff9d}, // nkatakanahalfwidth
	{0x3a40856d, 0x25a0}, // blacksquare
	{0x3a7498f3, 0x05e2}, // ayin
	{0x3a777405, 0x0945}, // ecandravowelsigndeva
	{0x3a846086, 0xff9c}, // wakatakanahalfwidth
	{0x3aa3cfcb, 0x0960}, // rrvocalicdeva
	{0x3ab26d21, 0x062f}, // dalarabic
	{0x3ade6670, 0x0440}, // afii10082
	{0x3afe4407, 0x25a9}, // squarediagonalcrosshatchfill
	{0x3b0367b4, 0x0a6b}, // fivegurmukhi
	{0x3b43910b, 0x0e36}, // sarauethai
	{0x3b4774fb, 0xfb35}, // vavdageshhebrew
	{0x3b6394a8, 0x30e0}, // mukatakana
	{0x3b722aff, 0x02bf}, // ringhalfleft
	{0x3b86faf7, 0x1e31}, // kacute
	{0x3b8c9510, 0x0926}, // dadeva
	{0x3b8f536a, 0x0564}, // daarmenian
	{0x3b919910, 0x02b4}, // rturnedsuperior
	{0x3b92b9ea, 0x1e4b}, // ncircumflexbelow
	{0x3b9a0136, 0x2468}, // ninecircle
	{0x3b9a26e8, 0x0637}, // taharabic
	{0x3bf4dcc7, 0xfea3}, // hahinitialarabic
	{0x3bf5a3f3, 0x301d}, // quotedblprimereversed
	{0x3c2679f3, 0x25c3}, // whiteleftpointingsmalltriangle
	{0x3c4101c7, 0x0a2b}, // phagurmukhi
	{0x3c47c401, 0x0196}, // Iotaafrican
	{0x3c5c7654, 0x0910}, // aideva
	{0x3c6261b4, 0x322e}, // ideographicmetalparen
	{0x3c6e58f4, 0x05d5}, // vavhebrew
	{0x3c89c6b7, 0x2192}, // arrowright
	{0x3c9425ca, 0x0666}, // sixhackarabic
	{0x3c9b6897, 0xff0e}, // periodmonospace
	{0x3c9bce6c, 0xf7ee}, // Icircumflexsmall
	{0x3ca31461, 0x01d1}, // Ocaron
	{0x3ca4227b, 0x0120}, // Gdot
	{0x3ca9c5ab, 0x0345}, // ypogegrammenigreekcmb
	{0x3cb5e1bf, 0x3146}, // ssangsioskorean
	{0x3cb6098f, 0x25b9}, // whiterightpointingsmalltriangle
	{0x3ccd3832, 0xfc60}, // shaddafathaarabic
	{0x3cfd6c60, 0x305a}, // zuhiragana
	{0x3cfe2d05, 0x018f}, // Schwa
	{0x3d0581a8, 0x313c}, // rieulpieupkorean
	{0x3d200141, 0x30c9}, // dokatakana
	{0x3d25bc32, 0x2116}, // numero
	{0x3d292466, 0x05b1}, // hatafsegol17
	{0x3d30abad, 0x0279}, // rturned
	{0x3d3a35a3, 0x0252}, // ascriptturned
	{0x3d3c2f4b, 0x309d}, // iterationhiragana
	{0x3d44ad12, 0x3210}, // tikeutaparenkorean
	{0x3d4a2f11, 0x0187}, // Chook
	{0x3d50ceda, 0x21a8}, // arrowupdnbse
	{0x3d59a63c, 0x30e2}, // mokatakana
	{0x3d64f67d, 0xf6cd}, // DieresisGrave
	{0x3d6d45f8, 0x0688}, // ddalarabic
	{0x3d794ead, 0x04c8}, // enhookcyrillic
	{0x3d888246, 0x300d}, // cornerbracketright
	{0x3d93ad95, 0x0171}, // uhungarumlaut
	{0x3d987773, 0x03c6}, // phi
	{0x3da1e076, 0xfb2a}, // shinshindot
	{0x3da24bf2, 0x004d}, // M
	{0x3da45134, 0xff3d}, // bracketrightmonospace
	{0x3dac6cb7, 0x212e}, // estimated
	{0x3db24f7f, 0x04eb}, // obarreddieresiscyrillic
	{0x3dc500c0, 0xf6ca}, // Caron
	{0x3ddbfa17, 0xfe64}, // lesssmall
	{0x3de34107, 0xf6d2}, // cyrFlex
	{0x3de84dee, 0x3387}, // GBsquare
	{0x3de8cd4d, 0x02c9}, // firsttonechinese
	{0x3deb36dc, 0x0570}, // hoarmenian
	{0x3dee4810, 0xf7a2}, // centoldstyle
	{0x3e08b864, 0x1e1c}, // Ecedillabreve
	{0x3e2966c3, 0x25a7}, // squareupperlefttolowerrightfill
	{0x3e2ad069, 0x0125}, // hcircumflex
	{0x3e2b18e7, 0x05d2}, // gimelhebrew
	{0x3e2ddf1a, 0x00fb}, // ucircumflex
	{0x3e4b0d7f, 0x2478}, // fiveparen
	{0x3e5c6b2e, 0x2086}, // sixinferior
	{0x3e78c213, 0x03f3}, // yotgreek
	{0x3e7bef16, 0x063a}, // ghainarabic
	{0x3e9c0cab, 0x0e0d}, // yoyingthai
	{0x3ea9e6c5, 0x044e}, // afii10096
	{0x3eb04aa9, 0xf7f2}, // Ogravesmall
	{0x3ec30c4d, 0x0168}, // Utilde
	{0x3ed009a2, 0x05e8}, // reshhiriq
	{0x3ed6d505, 0x3108}, // fbopomofo
	{0x3edc9801, 0x031e}, // downtackbelowcmb
	{0x3edf2653, 0x04ba}, // Shhacyrillic
	{0x3edfbd48, 0x0959}, // khhadeva
	{0x3ee5a28f, 0x0342}, // perispomenigreekcmb
	{0x3ef03dd0, 0xf7e5}, // Aringsmall
	{0x3ef2c5ca, 0x05df}, // finalnunhebrew
	{0x3efcba10, 0x05d3}, // daletpatahhebrew
	{0x3efcef55, 0x09a7}, // dhabengali
	{0x3f19d4d4, 0x30e5}, // yusmallkatakana
	{0x3f35ecb4, 0x0a0f}, // eegurmukhi
	{0x3f36dce9, 0x061b}, // afii57403
	{0x3f3b739c, 0x05e7}, // qofpatahhebrew
	{0x3f4e4960, 0xf76a}, // Jsmall
	{0x3f5ead94, 0x0456}, // icyrillic
	{0x3f61f37a, 0x1ea3}, // ahookabove
	{0x3f76f3d1, 0x0214}, // Udblgrave
	{0x3f77d74f, 0x05e8}, // reshqubuts
	{0x3f817391, 0x314b}, // khieukhkorean
	{0x3f8b0c34, 0x30c7}, // dekatakana
	{0x3fa4349b, 0x0e1d}, // fofathai
	{0x3fa5f151, 0x03de}, // Koppagreek
	{0x3fbf7ccb, 0x01ae}, // Tretroflexhook
	{0x3fddfa91, 0x22ce}, // curlyor
	{0x3fe060fb, 0x0147}, // Ncaron
	{0x3fe534eb, 0x0e12}, // thophuthaothai
	{0x4016947c, 0x246b}, // twelvecircle
	{0x401a74d3, 0x06f3}, // threepersian
	{0x402ddc95, 0xed1a}, // bracehtipupright
	{0x40663d0c, 0xfeb2}, // seenfinalarabic
	{0x4069b3b1, 0x24ab}, // pparen
	{0x407a7b83, 0x0531}, // Aybarmenian
	{0x407de2ef, 0x06d5}, // afii57534
	{0x40882350, 0x0305}, // overlinecmb
	{0x4096d7d2, 0x006a}, // j
	{0x409d1b5a, 0xff97}, // rakatakanahalfwidth
	{0x40b0365e, 0x092f}, // yadeva
	{0x40e21552, 0x30de}, // makatakana
	{0x40eb54f4, 0x046d}, // yusbigiotifiedcyrillic
	{0x413f6e7c, 0x05b4}, // hiriq21
	{0x4144d56c, 0x016d}, // ubreve
	{0x414507c4, 0x05b9}, // holamnarrowhebrew
	{0x414a0074, 0x3336}, // hekutaarusquare
	{0x4156eb7c, 0x3123}, // enbopomofo
	{0x415d9061, 0xff42}, // bmonospace
	{0x415fae27, 0x0398}, // Theta
	{0x4161c806, 0x09e1}, // llvocalicbengali
	{0x416a2ede, 0x0e04}, // khokhwaithai
	{0x4178dd24, 0x2243}, // asymptoticallyequal
	{0x41a22b2c, 0x007c}, // bar
	{0x41a40813, 0xff2d}, // Mmonospace
	{0x41a4780c, 0x0472}, // afii10147
	{0x41c544c2, 0x1ee7}, // uhookabove
	{0x41c76cd5, 0x05d7}, // het
	{0x41cb5b30, 0x02db}, // ogonek
	{0x41da6e39, 0x098c}, // lvocalicbengali
	{0x41e154b7, 0x1edd}, // ohorngrave
	{0x41e44ef7, 0x0aaa}, // pagujarati
	{0x41efdfc2, 0x33c1}, // mohmsquare
	{0x42320627, 0x2495}, // fourteenperiod
	{0x4235d221, 0xfecf}, // ghaininitialarabic
	{0x423f9221, 0x05a7}, // dargahebrew
	{0x4247685c, 0x01ff}, // oslashacute
	{0x4252dd77, 0x02c7}, // caron
	{0x42716524, 0x0069}, // i
	{0x42737aaf, 0x3081}, // mehiragana
	{0x427d3f50, 0x04a9}, // haabkhasiancyrillic
	{0x42803db2, 0x0030}, // zero
	{0x42813ae4, 0x1e24}, // Hdotbelow
	{0x428fedda, 0xfb44}, // pedagesh
	{0x42bba9f5, 0x30f0}, // wikatakana
	{0x42bc1b07, 0x25c4}, // triaglf
	{0x42bed72c, 0x33c0}, // kohmsquare
	{0x42bf360e, 0x1e6e}, // Tlinebelow
	{0x42d74152, 0x0578}, // voarmenian
	{0x42e554b2, 0x013b}, // Lcedilla
	{0x42ed7ca6, 0x0698}, // jeharabic
	{0x42f02b62, 0xfb95}, // gafmedialarabic
	{0x42f35290, 0x0064}, // d
	{0x42fb2842, 0x3120}, // aubopomofo
	{0x42fc57be, 0x3110}, // jbopomofo
	{0x42fd43ba, 0x044c}, // afii10094
	{0x42fdb31a, 0x0070}, // p
	{0x4305bc9e, 0x0920}, // tthadeva
	{0x4306eed3, 0x015d}, // scircumflex
	{0x430c20fb, 0x215e}, // seveneighths
	{0x430ddad4, 0xff20}, // atmonospace
	{0x431e0706, 0x00dc}, // Udieresis
	{0x43221a39, 0x1e91}, // zcircumflex
	{0x4328cb01, 0x339d}, // squarecm
	{0x432e75ab, 0x055d}, // commaarmenian
	{0x43399322, 0x026c}, // lbelt
	{0x434b10a7, 0x0a86}, // aagujarati
	{0x435f906e, 0xfb3e}, // memdageshhebrew
	{0x436008b6, 0x3064}, // tuhiragana
	{0x436f4b68, 0xfeaa}, // dalfinalarabic
	{0x439bf74d, 0xff8b}, // hikatakanahalfwidth
	{0x43a7e4d1, 0x0635}, // afii57429
	{0x43d0d1da, 0x0627}, // afii57415
	{0x43d651b4, 0x2663}, // clubsuitblack
	{0x43d755d6, 0x2212}, // minus
	{0x43dfb761, 0x0592}, // segoltahebrew
	{0x4405f04b, 0x05be}, // afii57645
	{0x4425746a, 0x0a18}, // ghagurmukhi
	{0x44317cf4, 0x2662}, // diamondsuitwhite
	{0x443620cd, 0x00fe}, // thorn
	{0x444b0abf, 0x0467}, // yuslittlecyrillic
	{0x444f60e7, 0x007f}, // controlDEL
	{0x445a3b6e, 0x0447}, // afii10089
	{0x4461957b, 0x041c}, // Emcyrillic
	{0x4464a19f, 0x0acb}, // ovowelsigngujarati
	{0x4483f355, 0x300f}, // whitecornerbracketright
	{0x4487c491, 0x3342}, // hoonsquare
	{0x4492e703, 0x0624}, // afii57412
	{0x44979567, 0x0a25}, // thagurmukhi
	{0x449b4678, 0xfec3}, // tahinitialarabic
	{0x44a78c72, 0x1eef}, // uhorntilde
	{0x44b414b4, 0xff0c}, // commamonospace
	{0x44d687fa, 0x1e06}, // Blinebelow
	{0x450fad6c, 0xf6fd}, // Scaronsmall
	{0x45116064, 0x2592}, // shade
	{0x453aa0af, 0x09f0}, // ramiddlediagonalbengali
	{0x4546a724, 0x05d0}, // afii57664
	{0x454cef44, 0x2642}, // mars
	{0x455bcfc1, 0x041d}, // afii10031
	{0x457de97e, 0x3184}, // kapyeounphieuphkorean
	{0x457e9e97, 0x1ee8}, // Uhornacute
	{0x459f0c78, 0x23a7}, // bracelefttp
	{0x45b3d9fd, 0x0a22}, // ddhagurmukhi
	{0x45b6c88f, 0x029b}, // Gsmallhook
	{0x45b98e95, 0x2163}, // Fourroman
	{0x45c29649, 0x1ef4}, // Ydotbelow
	{0x45c5ae6b, 0x0165}, // tcaron
	{0x45d73e08, 0x0a09}, // ugurmukhi
	{0x45e50e5b, 0x0384}, // tonos
	{0x45f6e82c, 0x057f}, // tiwnarmenian
	{0x45f7d5e0, 0x2084}, // fourinferior
	{0x46038ece, 0x00b6}, // paragraph
	{0x460c35ae, 0x05b8}, // qamats10
	{0x460ca9f0, 0x30e8}, // yokatakana
	{0x4611c6d0, 0x000b}, // controlVT
	{0x461bc854, 0x0194}, // Gammaafrican
	{0x46271982, 0x23ac}, // bracerightmid
	{0x46375ba2, 0x23d0}, // arrowvertex
	{0x4652be4f, 0x1e6d}, // tdotbelow
	{0x46541398, 0x0111}, // dmacron
	{0x46577172, 0x22c3}, // uniondisplay
	{0x465ea2f8, 0x3202}, // tikeutparenkorean
	{0x46611d40, 0x05bb}, // qubuts31
	{0x466452b7, 0x321b}, // hieuhaparenkorean
	{0x467a9a55, 0x055b}, // emphasismarkarmenian
	{0x46ab407a, 0x322b}, // ideographicfireparen
	{0x46ab921d, 0x3076}, // buhiragana
	{0x46ba3911, 0x1e45}, // ndotaccent
	{0x46bce40e, 0xfb4f}, // aleflamedhebrew
	{0x46c4dd5b, 0x0433}, // afii10068
	{0x46ca3a49, 0x2791}, // eightcircleinversesansserif
	{0x46e3006c, 0x055c}, // exclamarmenian
	{0x46f10ed2, 0x222a}, // union
	{0x46f3948a, 0x05d1}, // bethebrew
	{0x46f5f918, 0x05c3}, // afii57658
	{0x46f9c8ca, 0x2195}, // arrowupdn
	{0x470d662e, 0x042e}, // IUcyrillic
	{0x470da4b8, 0x0453}, // gjecyrillic
	{0x471d219d, 0x00f4}, // ocircumflex
	{0x47246853, 0xf890}, // maitriupperleftthai
	{0x472c971d, 0x1e4d}, // otildeacute
	{0x47542f2d, 0x0107}, // cacute
	{0x47849b51, 0x05e5}, // finaltsadi
	{0x478eb915, 0x014f}, // obreve
	{0x47a8409c, 0x05e8}, // reshqubutshebrew
	{0x47b12f1d, 0x3009}, // anglebracketright
	{0x47b78334, 0x09a5}, // thabengali
	{0x47dfd2f2, 0xf7a1}, // exclamdownsmall
	{0x47ee62a0, 0x04f1}, // udieresiscyrillic
	{0x480265ce, 0x06f2}, // twopersian
	{0x48175191, 0x0130}, // Idotaccent
	{0x481e50de, 0xfb32}, // gimeldagesh
	{0x4825c60d, 0x02c3}, // arrowheadrightmod
	{0x482626d7, 0x09ee}, // eightbengali
	{0x4826d3e4, 0x3391}, // khzsquare
	{0x48270352, 0x1e42}, // Mdotbelow
	{0x4848966d, 0x05b2}, // hatafpatah
	{0x485d5052, 0x057d}, // seharmenian
	{0x48908e05, 0xff28}, // Hmonospace
	{0x48a3aad1, 0x1ec8}, // Ihookabove
	{0x48adcc47, 0x0160}, // Scaron
	{0x48b11825, 0x1e3f}, // macute
	{0x48b31eb3, 0x02a3}, // dzaltone
	{0x48bbab6b, 0x1e68}, // Sdotbelowdotaccent
	{0x48cf810a, 0x27e8}, // angbracketleftBigg
	{0x48d0cd97, 0x0460}, // Omegacyrillic
	{0x48ed1289, 0x3180}, // ssangieungkorean
	{0x48f1ea86, 0x007c}, // verticalbar
	{0x49188fb6, 0x261d}, // pointingindexupwhite
	{0x491e8c30, 0x2562}, // SF200000
	{0x4928f75b, 0x1e3d}, // lcircumflexbelow
	{0x49299271, 0x1eee}, // Uhorntilde
	{0x49314f7c, 0x1e96}, // hlinebelow
	{0x493c6957, 0x01ec}, // Oogonekmacron
	{0x497b2a29, 0x03c8}, // psi
	{0x49877605, 0x1ecf}, // ohookabove
	{0x49a6c904, 0x00d5}, // Otilde
	{0x49a8fbe4, 0x227a}, // precedes
	{0x49b28bf6, 0x011f}, // gbreve
	{0x49c941c8, 0x0630}, // thalarabic
	{0x49cf949f, 0x0011}, // controlDC1
	{0x49d53679, 0x053e}, // Caarmenian
	{0x49de9093, 0x0029}, // parenrightBig
	{0x49e41b40, 0x0117}, // edotaccent
	{0x4a059748, 0x2329}, // angleleft
	{0x4a0a939e, 0x02dc}, // ilde
	{0x4a0dc7cd, 0x0273}, // nhookretroflex
	{0x4a1b8688, 0x0a8f}, // egujarati
	{0x4a26a1e2, 0xf88c}, // maieklowleftthai
	{0x4a67a4cd, 0x33c6}, // coverkgsquare
	{0x4a6dc3e0, 0x017c}, // zdotaccent
	{0x4a8f25d9, 0x0134}, // Jcircumflex
	{0x4a911686, 0x042a}, // Hardsigncyrillic
	{0x4aa9a643, 0x220c}, // notcontains
	{0x4ab184ff, 0x221a}, // radicalbt
	{0x4ab871b2, 0x05a9}, // telishaqetanahebrew
	{0x4ad4b644, 0x066c}, // thousandsseparatorpersian
	{0x4ad593e8, 0x05e7}, // qofpatah
	{0x4adaae02, 0x30bc}, // zekatakana
	{0x4b13bfc7, 0x261f}, // pointingindexdownwhite
	{0x4b146e46, 0x0076}, // v
	{0x4b1cfc1b, 0x0052}, // R
	{0x4b214add, 0x05e8}, // reshhatafsegolhebrew
	{0x4b69c8eb, 0x3200}, // kiyeokparenkorean
	{0x4b7a4380, 0x326b}, // thieuthcirclekorean
	{0x4b8cb1d3, 0x315a}, // oekorean
	{0x4b904ad7, 0x09ec}, // sixbengali
	{0x4bc3db0d, 0xfe8e}, // aleffinalarabic
	{0x4bcb1484, 0x0997}, // gabengali
	{0x4bf3941c, 0x04e7}, // odieresiscyrillic
	{0x4c1231bd, 0xfebe}, // dadfinalarabic
	{0x4c224e3d, 0x0216}, // Uinvertedbreve
	{0x4c31d446, 0x05b6}, // segolwidehebrew
	{0x4c330dc7, 0x2303}, // control
	{0x4c539c26, 0xff45}, // emonospace
	{0x4c550d84, 0x0286}, // eshcurl
	{0x4c63022b, 0x00ac}, // logicalnot
	{0x4c636f96, 0xfe4c}, // overlinedblwavy
	{0x4ca2293a, 0x0549}, // Chaarmenian
	{0x4ca721bb, 0x24d9}, // jcircle
	{0x4cbb6976, 0x054c}, // Raarmenian
	{0x4cbfbcf5, 0xf6cb}, // Dieresis
	{0x4cc2766b, 0x0474}, // Izhitsacyrillic
	{0x4ccaa98f, 0x0292}, // ezh
	{0x4cda32dd, 0x2592}, // shademedium
	{0x4cf1d7c1, 0xfb2c}, // shindageshshindot
	{0x4d08f8de, 0x278c}, // threecircleinversesansserif
	{0x4d18f1c3, 0x01e5}, // gstroke
	{0x4d3fee14, 0x2207}, // gradient
	{0x4d5e2eea, 0x09c7}, // evowelsignbengali
	{0x4d66ad61, 0x0259}, // schwa
	{0x4d6f0f44, 0x03b6}, // zeta
	{0x4d76cbca, 0x0939}, // hadeva
	{0x4d773822, 0x007d}, // bracerightbig
	{0x4da2ea17, 0x30f7}, // vakatakana
	{0x4dad3b1f, 0xff47}, // gmonospace
	{0x4db4092d, 0x05b2}, // hatafpatahquarterhebrew
	{0x4dc635ef, 0x03c2}, // sigmafinal
	{0x4dccadbd, 0xf6ea}, // bsuperior
	{0x4dd49001, 0x04ae}, // Ustraightcyrillic
	{0x4dd4e51e, 0x05e3}, // finalpe
	{0x4ddb0ff8, 0x3014}, // tortoiseshellbracketleft
	{0x4ddbe970, 0x0054}, // T
	{0x4ddd9ef4, 0x3296}, // ideographicfinancialcircle
	{0x4dedf33d, 0xff8d}, // hekatakanahalfwidth
	{0x4def9c7c, 0x2327}, // clear
	{0x4dfb4b2a, 0x247b}, // eightparen
	{0x4e03617a, 0x0ac1}, // uvowelsigngujarati
	{0x4e0fdced, 0x0a2d}, // bhagurmukhi
	{0x4e1cdd9c, 0x247d}, // tenparen
	{0x4e63a83b, 0x0642}, // qafarabic
	{0x4e7d8096, 0xff53}, // smonospace
	{0x4e8356bc, 0x0108}, // Ccircumflex
	{0x4eb853e7, 0x3384}, // kasquare
	{0x4ec3d2dc, 0xfccb}, // lamkhahinitialarabic
	{0x4ec752cf, 0x2022}, // bullet
	{0x4ef94777, 0x323b}, // ideographicstudyparen
	{0x4efcaf3c, 0x01ff}, // ostrokeacute
	{0x4f1d81af, 0x05bb}, // qubuts
	{0x4f1d9a74, 0x1e15}, // emacrongrave
	{0x4f1fbad2, 0x33b2}, // mussquare
	{0x4f238367, 0x0270}, // mlonglegturned
	{0x4f23d8fd, 0xfeda}, // kaffinalarabic
	{0x4f2d09de, 0x0662}, // twohackarabic
	{0x4f2efda5, 0xff07}, // quotesinglemonospace
	{0x4f30c414, 0x01c6}, // dzcaron
	{0x4f4be9c8, 0x3129}, // iubopomofo
	{0x4f6c2078, 0x00e5}, // aring
	{0x4f9b207b, 0x06c1}, // hehaltonearabic
	{0x4f9deafc, 0x03c4}, // tau
	{0x4fb92256, 0x216b}, // Twelveroman
	{0x4fdd1a2b, 0x300a}, // dblanglebracketleft
	{0x4feaecfe, 0x22c3}, // uniontext
	{0x4fef28fa, 0xf6c3}, // commaaccent
	{0x5004c9ab, 0x09ed}, // sevenbengali
	{0x50166be8, 0x049a}, // Kadescendercyrillic
	{0x501dd48e, 0x2229}, // intersection
	{0x5024fa7b, 0x260f}, // whitetelephone
	{0x5026482c, 0x228f}, // a60
	{0x502bdceb, 0x3290}, // ideographsuncircle
	{0x503133b5, 0x0009}, // controlHT
	{0x50604a35, 0x05e7}, // qofqubuts
	{0x507713d7, 0x2a02}, // circlemultiplytext
	{0x50794cf3, 0x255d}, // SF260000
	{0x508090a0, 0xff82}, // tukatakanahalfwidth
	{0x50993bc3, 0x05dd}, // finalmemhebrew
	{0x509ec6af, 0x0490}, // Gheupturncyrillic
	{0x50a87245, 0x05b9}, // holamwidehebrew
	{0x50be3a5b, 0x3189}, // yoikorean
	{0x50cc5524, 0x00be}, // threequarters
	{0x50cc8cef, 0x24a3}, // hparen
	{0x50fb6106, 0x30c1}, // tikatakana
	{0x510f444c, 0x0662}, // afii57394
	{0x511118c0, 0x05b8}, // qamatsqatanquarterhebrew
	{0x51250a43, 0x0ab9}, // hagujarati
	{0x5125d1fa, 0x25a1}, // whitesquare
	{0x513a52c9, 0x2510}, // SF030000
	{0x51439af3, 0x064e}, // afii57454
	{0x5147986a, 0xff5b}, // braceleftmonospace
	{0x514d7298, 0x3238}, // ideographiclaborparen
	{0x5153d63f, 0xf895}, // maichattawalowleftthai
	{0x515692ea, 0x0414}, // Decyrillic
	{0x51616742, 0x328c}, // ideographwatercircle
	{0x51817d65, 0xfca1}, // tehjeeminitialarabic
	{0x51969939, 0x03ba}, // kappa
	{0x51ca7ab6, 0x00f6}, // odieresis
	{0x51cbc424, 0x014d}, // omacron
	{0x51d34569, 0x00e9}, // eacute
	{0x51e4f41a, 0xf6db}, // trademarkserif
	{0x51e6847c, 0x05b2}, // hatafpatah16
	{0x51ed3cb2, 0x00af}, // macron
	{0x51f006ea, 0x24af}, // tparen
	{0x51f3c5bb, 0x2082}, // twoinferior
	{0x51fee10e, 0x2210}, // coproducttext
	{0x52099e7d, 0x3267}, // ieungcirclekorean
	{0x5241ded3, 0x0428}, // Shacyrillic
	{0x5247cafc, 0x0691}, // afii57513
	{0x524c924c, 0x02d6}, // plusmod
	{0x525a3324, 0x2606}, // whitestar
	{0x5282fafa, 0x0e09}, // chochingthai
	{0x528afecc, 0xfb89}, // ddalfinalarabic
	{0x52beac4f, 0x25bf}, // whitedownpointingsmalltriangle
	{0x52e15cc8, 0x24b6}, // Acircle
	{0x52f72574, 0x3322}, // sentisquare
	{0x530e1856, 0x05dd}, // finalmem
	{0x531472bf, 0x05bb}, // qubutswidehebrew
	{0x531963a3, 0x0a73}, // uragurmukhi
	{0x531c6e1f, 0x315b}, // yokorean
	{0x532f469f, 0x093d}, // avagrahadeva
	{0x533e9388, 0x02bc}, // afii57929
	{0x537b0d36, 0x0466}, // Yuslittlecyrillic
	{0x5382e913, 0x002f}, // slashBigg
	{0x5396a4ab, 0x2a04}, // unionmultidisplay
	{0x53b3b784, 0xfb4a}, // tavdagesh
	{0x53ca8524, 0x099d}, // jhabengali
	{0x53d60270, 0x05d3}, // dalethiriqhebrew
	{0x53d8dfb9, 0x3115}, // shbopomofo
	{0x53e66e1a, 0x0476}, // Izhitsadblgravecyrillic
	{0x53f49c2a, 0x066c}, // thousandsseparatorarabic
	{0x53f951b5, 0x056f}, // kenarmenian
	{0x53febc17, 0x05e8}, // reshqamatshebrew
	{0x540493c8, 0x03b1}, // alpha
	{0x540c5f40, 0x0552}, // Yiwnarmenian
	{0x542576f9, 0x0a07}, // igurmukhi
	{0x542f1e7a, 0x0151}, // ohungarumlaut
	{0x54761f15, 0x0631}, // afii57425
	{0x54820079, 0x04df}, // zedieresiscyrillic
	{0x548a6dde, 0x0451}, // iocyrillic
	{0x5494ff15, 0x053c}, // Liwnarmenian
	{0x54cc0e6b, 0x0212}, // Rinvertedbreve
	{0x54d729fd, 0xf6bf}, // LL
	{0x54dfda54, 0xfe44}, // whitecornerbracketrightvertical
	{0x54f8c0f2, 0x1e78}, // Utildeacute
	{0x55003750, 0x05b0}, // shevanarrowhebrew
	{0x55021a5a, 0x0644}, // lamarabic
	{0x5509dd21, 0xf7bf}, // questiondownsmall
	{0x550a9f23, 0x0452}, // djecyrillic
	{0x550d7456, 0x0061}, // a
	{0x5512ec97, 0x0067}, // g
	{0x55164cbd, 0xf7fb}, // Ucircumflexsmall
	{0x552705b9, 0xf7e1}, // Aacutesmall
	{0x552e72ea, 0x1e8d}, // xdieresis
	{0x5532b75e, 0x3044}, // ihiragana
	{0x554f67fa, 0x1e4e}, // Otildedieresis
	{0x556bcf7c, 0xff1f}, // questionmonospace
	{0x5575c7a8, 0x09a8}, // nabengali
	{0x557f8e27, 0x1ee0}, // Ohorntilde
	{0x558d2385, 0x0318}, // lefttackbelowcmb
	{0x55aa99d5, 0xfe3f}, // anglebracketleftvertical
	{0x55b44317, 0x1ef2}, // Ygrave
	{0x55b8ceec, 0xed6b}, // radicalvertex
	{0x55c6e8cd, 0x0aac}, // bagujarati
	{0x55e74cbe, 0xf8fb}, // bracketrightbt
	{0x55e81ebb, 0x3124}, // angbopomofo
	{0x56000715, 0x33b0}, // pssquare
	{0x560f90ad, 0x25e4}, // blackupperlefttriangle
	{0x56200891, 0x092b}, // phadeva
	{0x56217879, 0x0596}, // tipehalefthebrew
	{0x56362764, 0x24d0}, // acircle
	{0x565aa859, 0x30a8}, // ekatakana
	{0x565d95fc, 0x3109}, // dbopomofo
	{0x566af414, 0xed6a}, // radicaltp
	{0x567e8709, 0xf6f4}, // Brevesmall
	{0x569f0bdc, 0x05e7}, // qofqamats
	{0x56a0101f, 0x04ac}, // Tedescendercyrillic
	{0x56d036b9, 0x000f}, // controlSI
	{0x56deae12, 0x0102}, // Abreve
	{0x57050efe, 0x0145}, // Ncommaaccent
	{0x5708e98e, 0x3154}, // ekorean
	{0x570da3d3, 0xf8e9}, // copyrightsans
	{0x5713d355, 0x2790}, // sevencircleinversesansserif
	{0x575226bc, 0xf8f7}, // parenrightex
	{0x57687403, 0x0e08}, // chochanthai
	{0x576959da, 0x04b7}, // chedescendercyrillic
	{0x577fdcc5, 0x033a}, // bridgeinvertedbelowcmb
	{0x578594f4, 0x0e47}, // maitaikhuthai
	{0x579e8de2, 0x2074}, // foursuperior
	{0x57a10bfe, 0x0435}, // iecyrillic
	{0x57a26403, 0xf6eb}, // dsuperior
	{0x57c4d153, 0xfb1f}, // yodyodpatahhebrew
	{0x57c8c90f, 0x313b}, // rieulmieumkorean
	{0x57cea503, 0xff17}, // sevenmonospace
	{0x57fc3d5e, 0x255b}, // SF280000
	{0x58064efc, 0x0448}, // shacyrillic
	{0x580bc6b8, 0x2320}, // integraltop
	{0x580e0aa6, 0x005e}, // asciicircum
	{0x5817c838, 0x3141}, // mieumkorean
	{0x581d6ffc, 0x1ef8}, // Ytilde
	{0x58246165, 0x0152}, // OE
	{0x582e4f2f, 0x02a1}, // glottalstopstroke
	{0x58408a1e, 0x2321}, // integralbottom
	{0x584e8397, 0x263a}, // whitesmilingface
	{0x5855c496, 0x0465}, // eiotifiedcyrillic
	{0x587d22eb, 0x208d}, // parenleftinferior
	{0x58a61c85, 0x1e9a}, // arighthalfring
	{0x58ab0a67, 0x0544}, // Menarmenian
	{0x58ae8d36, 0x05e8}, // reshhatafpatah
	{0x58ba15a9, 0x24c9}, // Tcircle
	{0x58c52193, 0x221a}, // radical
	{0x58df0572, 0x0301}, // acutecmb
	{0x58dfd388, 0x3187}, // yoyakorean
	{0x58e61a1f, 0x05e8}, // resh
	{0x58ea1dd9, 0x0650}, // afii57456
	{0x59322213, 0xf7e8}, // Egravesmall
	{0x593b3f38, 0x0aa1}, // ddagujarati
	{0x5951351c, 0x0109}, // ccircumflex
	{0x59664498, 0xff21}, // Amonospace
	{0x597231b2, 0x1e00}, // Aringbelow
	{0x598309ec, 0x1edc}, // Ohorngrave
	{0x598631c5, 0x24c1}, // Lcircle
	{0x59b5003e, 0x3274}, // siosacirclekorean
	{0x59b9b187, 0x0075}, // u
	{0x59be0f3a, 0xff7f}, // sokatakanahalfwidth
	{0x59c46f70, 0x3299}, // ideographicsecretcircle
	{0x59c80d40, 0x2482}, // fifteenparen
	{0x59d6d87b, 0xff74}, // ekatakanahalfwidth
	{0x59d7f689, 0x3393}, // ghzsquare
	{0x59eccfd0, 0x05a3}, // munahlefthebrew
	{0x5a1aca3c, 0xfe34}, // wavyunderscorevertical
	{0x5a24f67c, 0x03b4}, // delta
	{0x5a3b6461, 0xfeb8}, // sheenmedialarabic
	{0x5a532aa6, 0x0586}, // feharmenian
	{0x5a575dc4, 0x0582}, // yiwnarmenian
	{0x5a62c8e8, 0x005c}, // backslash
	{0x5a8d2a4a, 0x09c3}, // rvocalicvowelsignbengali
	{0x5a963c7c, 0x0e02}, // khokhaithai
	{0x5acd345c, 0x339c}, // squaremm
	{0x5ad6e1c9, 0x05e7}, // qofsegol
	{0x5af634c5, 0x059e}, // gershayimaccenthebrew
	{0x5b041347, 0x0581}, // coarmenian
	{0x5b0fd985, 0x0360}, // tildedoublecmb
	{0x5b1907c0, 0x30a5}, // usmallkatakana
	{0x5b1da33a, 0x2206}, // increment
	{0x5b1dbca0, 0x0aa6}, // dagujarati
	{0x5b426364, 0x30cf}, // hakatakana
	{0x5b426591, 0xfeb7}, // sheeninitialarabic
	{0x5b46f9e7, 0x2080}, // zeroinferior
	{0x5b54a5c3, 0x05e1}, // afii57681
	{0x5b5f52e7, 0x0421}, // afii10035
	{0x5b68b5ed, 0xf892}, // maitrilowleftthai
	{0x5b6ab184, 0x25ef}, // largecircle
	{0x5b6f6c30, 0x0003}, // controlETX
	{0x5b720455, 0x014c}, // Omacron
	{0x5b72ad21, 0x03d5}, // phi1
	{0x5b785975, 0x02a7}, // tesh
	{0x5b7a64f2, 0x0597}, // reviahebrew
	{0x5bbfa15f, 0x25c4}, // blackleftpointingpointer
	{0x5bc1fa37, 0xfb47}, // qofdageshhebrew
	{0x5be94211, 0xff24}, // Dmonospace
	{0x5beb1e0f, 0x24cf}, // Zcircle
	{0x5c0e8b47, 0x0426}, // afii10040
	{0x5c185e06, 0x0250}, // aturned
	{0x5c299659, 0x250c}, // SF010000
	{0x5c337c81, 0xff8a}, // hakatakanahalfwidth
	{0x5c3478b9, 0xff3a}, // Zmonospace
	{0x5c349e9d, 0x30c5}, // dukatakana
	{0x5c352033, 0x092c}, // badeva
	{0x5c3b9279, 0x307e}, // mahiragana
	{0x5c3eecc5, 0x2561}, // SF190000
	{0x5c4a060e, 0x0e29}, // sorusithai
	{0x5c6b0ced, 0x05b8}, // qamats33
	{0x5c71c76c, 0x24b8}, // Ccircle
	{0x5c738b36, 0xf776}, // Vsmall
	{0x5c7f4966, 0x01b8}, // Ezhreversed
	{0x5c97be88, 0x0a2a}, // pagurmukhi
	{0x5c9a7487, 0x30ec}, // rekatakana
	{0x5c9f86aa, 0x05e8}, // reshshevahebrew
	{0x5ca0edc2, 0x01e7}, // gcaron
	{0x5cad2e17, 0x0679}, // tteharabic
	{0x5cb64e9e, 0x0150}, // Ohungarumlaut
	{0x5cb98a11, 0x05e9}, // afii57689
	{0x5cc203b0, 0x09eb}, // fivebengali
	{0x5cde4fa9, 0xff39}, // Ymonospace
	{0x5ce216d6, 0x02bb}, // commaturnedmod
	{0x5ce89c18, 0x028c}, // vturned
	{0x5cee9de2, 0x333b}, // peezisquare
	{0x5d053ab9, 0x334d}, // meetorusquare
	{0x5d06b34a, 0x01f5}, // gacute
	{0x5d185c29, 0x0aec}, // sixgujarati
	{0x5d23e967, 0x019a}, // lbar
	{0x5d24fed7, 0x047d}, // omegatitlocyrillic
	{0x5d26ca2e, 0x0996}, // khabengali
	{0x5d32256c, 0x25c6}, // blackdiamond
	{0x5d333915, 0x2234}, // therefore
	{0x5d42ce05, 0x30c2}, // dikatakana
	{0x5d4fa82e, 0xf724}, // dollaroldstyle
	{0x5d590cb1, 0x05bb}, // qubutsnarrowhebrew
	{0x5d71a05b, 0x0017}, // controlETB
	{0x5d85b369, 0x05b8}, // qamats29
	{0x5d8c507f, 0xfb47}, // qofdagesh
	{0x5da58253, 0x30d0}, // bakatakana
	{0x5dba07ed, 0x22a5}, // perpendicular
	{0x5dbeec87, 0x01cd}, // Acaron
	{0x5de3b63c, 0x09cb}, // ovowelsignbengali
	{0x5df717ca, 0x05e8}, // reshtserehebrew
	{0x5e0aac56, 0x0200}, // Adblgrave
	{0x5e1c8dfa, 0x05b0}, // sheva2e
	{0x5e27fa57, 0xff6d}, // yusmallkatakanahalfwidth
	{0x5e36a670, 0xfe33}, // underscorevertical
	{0x5e483ddc, 0x3277}, // chieuchacirclekorean
	{0x5e4f2fbb, 0x1e65}, // sacutedotaccent
	{0x5ea7176f, 0x02d9}, // dotaccent
	{0x5edd1e19, 0xff25}, // Emonospace
	{0x5edd9086, 0x003a}, // colon
	{0x5eddf92b, 0x0044}, // D
	{0x5ee2af9c, 0x0e58}, // eightthai
	{0x5f03252d, 0x0136}, // Kcedilla
	{0x5f0ec9e9, 0x039a}, // Kappa
	{0x5f1bf33a, 0x1e17}, // emacronacute
	{0x5f4a5f07, 0x0abe}, // aavowelsigngujarati
	{0x5f5c5d5f, 0x041b}, // Elcyrillic
	{0x5f63748c, 0x03e6}, // Kheicoptic
	{0x5f6ca553, 0x230a}, // floorleftbigg
	{0x5f7dc76d, 0x033b}, // squarebelowcmb
	{0x5f881d5c, 0x3160}, // yukorean
	{0x5f8f63e2, 0x24a0}, // eparen
	{0x5f99c0ac, 0x01ad}, // thook
	{0x5fa5f5cc, 0xfefa}, // lamalefhamzabelowfinalarabic
	{0x5fb56903, 0x095d}, // rhadeva
	{0x5fbeac33, 0x028b}, // vhook
	{0x5fd46519, 0xf6ef}, // msuperior
	{0x5fe9065e, 0x042e}, // afii10048
	{0x6037ae88, 0x1ef3}, // ygrave
	{0x603b5882, 0x2287}, // reflexsuperset
	{0x603b9d93, 0x00d3}, // Oacute
	{0x603ff393, 0x1e6c}, // Tdotbelow
	{0x60753fe5, 0x066d}, // afii63167
	{0x607c93ed, 0x01b2}, // Vhook
	{0x60a46930, 0x0197}, // Istroke
	{0x60a4c80a, 0x30ee}, // wasmallkatakana
	{0x60ac2314, 0x25cc}, // dottedcircle
	{0x60ba7236, 0x000e}, // controlSO
	{0x60d57bed, 0x0aa4}, // tagujarati
	{0x60e1bf57, 0x1e0d}, // ddotbelow
	{0x60ebe651, 0x1ed8}, // Ocircumflexdotbelow
	{0x60f5a9fa, 0x03b7}, // eta
	{0x6106119c, 0xff37}, // Wmonospace
	{0x610b31e6, 0x01f1}, // DZ
	{0x6114c811, 0x1ef0}, // Uhorndotbelow
	{0x6121a3f6, 0x0e4d}, // nikhahitthai
	{0x61239a5a, 0x01c2}, // clickalveolar
	{0x6143b142, 0x1e67}, // scarondotaccent
	{0x61503571, 0x027b}, // rhookturned
	{0x6154bc05, 0x043b}, // afii10077
	{0x617c687c, 0x01a2}, // Oi
	{0x618467d6, 0xf7f3}, // Oacutesmall
	{0x6198a771, 0xfef3}, // alefmaksurainitialarabic
	{0x61a21109, 0x01c9}, // lj
	{0x61a2d0df, 0x258c}, // lfblock
	{0x61a6f1a4, 0x0264}, // ramshorn
	{0x61b4ed39, 0x25ba}, // triagrt
	{0x61b7afbf, 0x05b5}, // tserenarrowhebrew
	{0x61b9f022, 0x04c4}, // kahookcyrillic
	{0x61ce131a, 0x0647}, // afii57470
	{0x61d46fc2, 0x1ed6}, // Ocircumflextilde
	{0x61d7bcdb, 0x1e57}, // pdotaccent
	{0x61e75298, 0x0417}, // afii10025
	{0x61f36361, 0x06f8}, // eightpersian
	{0x61fe712f, 0x20a4}, // afii08941
	{0x621057dd, 0x1ef5}, // ydotbelow
	{0x62106755, 0xf6e5}, // hypheninferior
	{0x62127977, 0x266a}, // musicalnote
	{0x62161c15, 0x05da}, // finalkafhebrew
	{0x6229838d, 0x05d0}, // alef
	{0x62447ae3, 0x0314}, // commareversedabovecmb
	{0x6248b5e6, 0x33dd}, // wbsquare
	{0x625320aa, 0x30b2}, // gekatakana
	{0x6259e0bf, 0xf6f1}, // rsuperior
	{0x6260c6fc, 0x0589}, // periodarmenian
	{0x6265e881, 0x2266}, // lessoverequal
	{0x626cbaa4, 0x01bf}, // wynn
	{0x62724d89, 0x09ad}, // bhabengali
	{0x62896f4a, 0x05f2}, // yodyodhebrew
	{0x628a5951, 0xff56}, // vmonospace
	{0x62a11b25, 0x1e87}, // wdotaccent
	{0x62accaf5, 0xf8ed}, // parenleftbt
	{0x62b2cd2d, 0x00c2}, // Acircumflex
	{0x62b6e7ac, 0x01a5}, // phook
	{0x62c361d5, 0x05e2}, // afii57682
	{0x62c52689, 0x05d5}, // afii57669
	{0x62ca59ed, 0x03aa}, // Iotadieresis
	{0x62cfccee, 0x017d}, // Zcaron
	{0x62d27ffc, 0x0567}, // eharmenian
	{0x62f7161d, 0x3020}, // postalmarkface
	{0x630680b1, 0x1e20}, // Gmacron
	{0x63070542, 0x0401}, // afii10023
	{0x630758ff, 0x317e}, // sioscieuckorean
	{0x632ae410, 0x0acc}, // auvowelsigngujarati
	{0x634ac34b, 0x2297}, // timescircle
	{0x634e42e5, 0xff2b}, // Kmonospace
	{0x635a9554, 0x04e5}, // idieresiscyrillic
	{0x636fabd5, 0x093c}, // nuktadeva
	{0x637ae7ca, 0x0344}, // dialytikatonoscmb
	{0x637d3539, 0x00bc}, // onequarter
	{0x6389d9bb, 0x0641}, // afii57441
	{0x63a82931, 0x0297}, // cstretched
	{0x63ab7e3b, 0xf6dd}, // rupiah
	{0x63d84bb6, 0x2281}, // notsucceeds
	{0x63dfed74, 0x311f}, // eibopomofo
	{0x63e274d2, 0xfc0b}, // tehjeemisolatedarabic
	{0x63e60b13, 0x0149}, // napostrophe
	{0x63f6cd8f, 0x049d}, // kaverticalstrokecyrillic
	{0x642b78c5, 0x1ed5}, // ocircumflexhookabove
	{0x642e193e, 0x0469}, // yuslittleiotifiedcyrillic
	{0x6434f04a, 0xfd88}, // lammeemhahinitialarabic
	{0x64468e36, 0x0e18}, // thothongthai
	{0x64517fe8, 0x004b}, // K
	{0x6453c78e, 0x064e}, // fathalowarabic
	{0x6454154f, 0xf6ec}, // esuperior
	{0x6460d798, 0x3025}, // fivehangzhou
	{0x64699e37, 0x2309}, // ceilingrightBigg
	{0x646b157e, 0x3242}, // ideographicselfparen
	{0x646c5c19, 0x0ab5}, // vagujarati
	{0x64755597, 0x05e7}, // qofholamhebrew
	{0x647eca4f, 0x09ae}, // mabengali
	{0x64a1b76b, 0x30fa}, // vokatakana
	{0x64d00b32, 0x0114}, // Ebreve
	{0x64d7a402, 0x0e40}, // saraethai
	{0x64d94f4a, 0x0e01}, // kokaithai
	{0x64e7dff2, 0x09c4}, // rrvocalicvowelsignbengali
	{0x64eb016b, 0x2245}, // approximatelyequal
	{0x64fd7a48, 0x2078}, // eightsuperior
	{0x650678be, 0x02cb}, // fourthtonechinese
	{0x65070663, 0x0459}, // ljecyrillic
	{0x6510d99a, 0x0938}, // sadeva
	{0x651d5722, 0x04c2}, // zhebrevecyrillic
	{0x6525cdb1, 0x00ad}, // softhyphen
	{0x6543f12c, 0x01d5}, // Udieresismacron
	{0x654d08a0, 0x0e3a}, // phinthuthai
	{0x657ffabd, 0x22ee}, // ellipsisvertical
	{0x6594aba1, 0x02c3}, // a41
	{0x6598fbfe, 0x3112}, // xbopomofo
	{0x65a5bd1b, 0x24e1}, // rcircle
	{0x65be15d1, 0x0a08}, // iigurmukhi
	{0x65c095c5, 0x200e}, // afii299
	{0x65cc1f56, 0x0545}, // Yiarmenian
	{0x662a6586, 0xf8f8}, // parenrightbt
	{0x662a831c, 0x1eeb}, // uhorngrave
	{0x663a2c1a, 0x3300}, // apaatosquare
	{0x6642f834, 0x0361}, // breveinverteddoublecmb
	{0x665bba62, 0x2017}, // dbllowline
	{0x665e930f, 0x3264}, // mieumcirclekorean
	{0x666ae75e, 0x0159}, // rcaron
	{0x666e8927, 0xfc4e}, // noonmeemisolatedarabic
	{0x666faf51, 0xfba5}, // hehhamzaabovefinalarabic
	{0x6689dbc7, 0x33cb}, // HPsquare
	{0x668e9764, 0x30a2}, // akatakana
	{0x669881eb, 0xf8e5}, // radicalex
	{0x66b92e17, 0x05e8}, // reshholam
	{0x66bb979c, 0x246c}, // thirteencircle
	{0x66c590a5, 0x2236}, // ratio
	{0x66e750ef, 0xf7ec}, // Igravesmall
	{0x66e9c5c7, 0x3084}, // yahiragana
	{0x66ef8fdf, 0x040f}, // afii10145
	{0x66fb9b49, 0x33c9}, // gysquare
	{0x67188e74, 0x0066}, // f
	{0x6740a4ac, 0x25cf}, // H18533
	{0x67427e4a, 0x2079}, // ninesuperior
	{0x674a0210, 0x02b6}, // Rsmallinvertedsuperior
	{0x674c80b7, 0x03da}, // Stigmagreek
	{0x67569fa6, 0x05b5}, // tsere
	{0x6758c83c, 0x2588}, // block
	{0x6776a85a, 0x0ac7}, // evowelsigngujarati
	{0x6779be95, 0x3113}, // zhbopomofo
	{0x6785194b, 0x221f}, // rightangle
	{0x67949ab4, 0xf891}, // maitrilowrightthai
	{0x679d9205, 0x02c5}, // arrowheaddownmod
	{0x679dcadd, 0x0e54}, // fourthai
	{0x679ffc49, 0x2024}, // onedotenleader
	{0x67ea250f, 0x1e2d}, // itildebelow
	{0x67ebcea4, 0x3209}, // chieuchparenkorean
	{0x67f3db47, 0x3073}, // bihiragana
	{0x67f3ecac, 0xfb4b}, // afii57700
	{0x67f7f220, 0x309c}, // semivoicedmarkkana
	{0x67fa1db6, 0x203a}, // guilsinglright
	{0x68070609, 0x0916}, // khadeva
	{0x6808ec86, 0x02d8}, // breve
	{0x6811e3e1, 0x0404}, // Ecyrillic
	{0x6814026d, 0x02c8}, // verticallinemod
	{0x682b08dd, 0x01ac}, // Thook
	{0x683090d6, 0xf6f0}, // osuperior
	{0x6853e235, 0x0652}, // sukunarabic
	{0x685932be, 0xfe36}, // parenrightvertical
	{0x68744fba, 0x220f}, // producttext
	{0x68a76955, 0x00c7}, // Ccedilla
	{0x68b3215e, 0x0339}, // ringhalfrightbelowcmb
	{0x68b3ae0f, 0x222e}, // contintegraldisplay
	{0x68b98a65, 0x0679}, // afii57511
	{0x68ccd1d8, 0x3212}, // mieumaparenkorean
	{0x68eb418e, 0x3011}, // blacklenticularbracketright
	{0x690f8118, 0x320b}, // thieuthparenkorean
	{0x692bc1f7, 0x0a42}, // uumatragurmukhi
	{0x693ccd9c, 0x0432}, // vecyrillic
	{0x6962e430, 0x0322}, // hookretroflexbelowcmb
	{0x6977f4b7, 0xff05}, // percentmonospace
	{0x697925af, 0x05d2}, // gimel
	{0x697feb80, 0x030d}, // verticallineabovecmb
	{0x6991a62a, 0x0935}, // vadeva
	{0x699d525a, 0x0111}, // dcroat
	{0x69aabc4d, 0x06af}, // gafarabic
	{0x69d31152, 0x0e51}, // onethai
	{0x69faaae4, 0x33a7}, // moverssquare
	{0x6a104703, 0x2569}, // SF400000
	{0x6a16e743, 0xfef0}, // alefmaksurafinalarabic
	{0x6a3d8bdc, 0x0a99}, // ngagujarati
	{0x6a511868, 0x03a7}, // Chi
	{0x6a5f0a3f, 0x334e}, // yaadosquare
	{0x6a721907, 0xfecc}, // ainmedialarabic
	{0x6a8d644d, 0x1ead}, // acircumflexdotbelow
	{0x6a972227, 0x0401}, // Iocyrillic
	{0x6a988d53, 0x2035}, // primereversed
	{0x6a9b2b92, 0x0414}, // afii10021
	{0x6aa62850, 0x0662}, // twoarabic
	{0x6aa82ca6, 0x3218}, // khieukhaparenkorean
	{0x6abb1490, 0x005d}, // bracketright
	{0x6ad19d29, 0xfb1f}, // doubleyodpatahhebrew
	{0x6b214948, 0xff6f}, // tusmallkatakanahalfwidth
	{0x6b2d374c, 0xfcdd}, // yehmeeminitialarabic
	{0x6b3f02ab, 0x211e}, // prescription
	{0x6b58bab9, 0x0963}, // llvocalicvowelsigndeva
	{0x6b83db74, 0x2295}, // circleplus
	{0x6b85cd6e, 0x00a4}, // currency
	{0x6b8b924f, 0x30d8}, // hekatakana
	{0x6b8f0a54, 0xf6df}, // centinferior
	{0x6b908b1b, 0x09e7}, // onebengali
	{0x6b98f18c, 0x212b}, // angstrom
	{0x6b9adcc4, 0x25cf}, // blackcircle
	{0x6b9f4fbe, 0x0940}, // iivowelsigndeva
	{0x6ba8c1e2, 0xf6ce}, // Grave
	{0x6baa5ba4, 0x03ce}, // omegatonos
	{0x6bb62dc9, 0xf77a}, // Zsmall
	{0x6bbd76c8, 0x0936}, // shadeva
	{0x6bca6ff5, 0xfe4a}, // overlinecenterline
	{0x6bd59d12, 0x0915}, // kadeva
	{0x6bd609d3, 0x00f0}, // eth
	{0x6be13af7, 0x2199}, // arrowdownleft
	{0x6bf10a81, 0x2015}, // horizontalbar
	{0x6c0e37b1, 0x30ce}, // nokatakana
	{0x6c0f6861, 0x05e7}, // qofhatafsegolhebrew
	{0x6c11086e, 0x0a1c}, // jagurmukhi
	{0x6c180b9f, 0x0462}, // Yatcyrillic
	{0x6c23928a, 0x222e}, // contintegraltext
	{0x6c429460, 0xfb40}, // nundageshhebrew
	{0x6c5c9da6, 0x0457}, // yicyrillic
	{0x6c781e8a, 0x25bd}, // whitedownpointingtriangle
	{0x6c8584d5, 0x3065}, // duhiragana
	{0x6c95b865, 0x0941}, // uvowelsigndeva
	{0x6c98a67f, 0x0479}, // ukcyrillic
	{0x6cbb9b7a, 0xfe5d}, // tortoiseshellbracketleftsmall
	{0x6cce2d3d, 0x0aa5}, // thagujarati
	{0x6cce7f58, 0x043b}, // elcyrillic
	{0x6d02864a, 0x049c}, // Kaverticalstrokecyrillic
	{0x6d1b554b, 0x0982}, // anusvarabengali
	{0x6d3fc0d7, 0x040f}, // Dzhecyrillic
	{0x6d4379b8, 0x220f}, // productdisplay
	{0x6d519305, 0x2169}, // Tenroman
	{0x6d63a915, 0x02bc}, // apostrophemod
	{0x6d6c6ece, 0x307f}, // mihiragana
	{0x6d794da3, 0x0057}, // W
	{0x6d7a8d87, 0x00fd}, // yacute
	{0x6d88930b, 0x044a}, // afii10092
	{0x6d89653d, 0x24e8}, // ycircle
	{0x6d8f5cd4, 0xff95}, // yukatakanahalfwidth
	{0x6d931b7f, 0x255c}, // SF270000
	{0x6d9b9c3e, 0x05dc}, // afii57676
	{0x6dd186b9, 0x0575}, // yiarmenian
	{0x6ddcf118, 0x22a4}, // tackdown
	{0x6ddd69da, 0x05b1}, // hatafsegolquarterhebrew
	{0x6df68266, 0x0388}, // Epsilontonos
	{0x6df8e451, 0x2203}, // existential
	{0x6dfa6cea, 0x0a66}, // zerogurmukhi
	{0x6dff449c, 0xff79}, // kekatakanahalfwidth
	{0x6e298b2c, 0x3380}, // paampssquare
	{0x6e2b4011, 0x2213}, // minusplus
	{0x6e2e0a97, 0xfe66}, // equalsmall
	{0x6e53b9b3, 0xfb00}, // ff
	{0x6e5f0868, 0x30ed}, // rokatakana
	{0x6e7cf223, 0x3071}, // pahiragana
	{0x6e81816d, 0x1e5d}, // rdotbelowmacron
	{0x6e897228, 0x0324}, // dieresisbelowcmb
	{0x6e8ccbbb, 0xfb48}, // reshdageshhebrew
	{0x6eb16a59, 0x2166}, // Sevenroman
	{0x6eb40b0d, 0x04db}, // schwadieresiscyrillic
	{0x6eba69ba, 0xfe9e}, // jeemfinalarabic
	{0x6ebb53ea, 0x2261}, // equivalence
	{0x6ecb4ba4, 0x0449}, // afii10091
	{0x6ecc1f68, 0x25bc}, // triagdn
	{0x6ed918ed, 0xfcd1}, // meemmeeminitialarabic
	{0x6ee27b35, 0x0321}, // hookpalatalizedbelowcmb
	{0x6f028e08, 0x0571}, // jaarmenian
	{0x6f1f2583, 0x25ce}, // bullseye
	{0x6f2573bb, 0x30eb}, // rukatakana
	{0x6f4833d3, 0x0a5c}, // rragurmukhi
	{0x6f57587a, 0x227b}, // succeeds
	{0x6f64edc9, 0x054f}, // Tiwnarmenian
	{0x6f6b6e30, 0x0e0e}, // dochadathai
	{0x6f75692f, 0x0ac8}, // aivowelsigngujarati
	{0x6f7744aa, 0x2196}, // arrowupleft
	{0x6fae2151, 0x0aef}, // ninegujarati
	{0x6fbeef2a, 0x044d}, // ereversedcyrillic
	{0x6fc01d9f, 0x00f8}, // oslash
	{0x6fdfa2dc, 0xf6e7}, // periodinferior
	{0x70068cb8, 0xf7ea}, // Ecircumflexsmall
	{0x702ecc19, 0x3054}, // gohiragana
	{0x70300d95, 0xf761}, // Asmall
	{0x70534c83, 0x05a4}, // mahapakhlefthebrew
	{0x7061d5ad, 0x255a}, // SF380000
	{0x707057b4, 0xfb2e}, // alefpatahhebrew
	{0x7070c1c4, 0x0333}, // dbllowlinecmb
	{0x7077f8df, 0x0163}, // tcommaaccent
	{0x708cf64a, 0x0944}, // rrvocalicvowelsigndeva
	{0x709b2bbe, 0x0056}, // V
	{0x709b801f, 0x05da}, // finalkafshevahebrew
	{0x70a0d365, 0xfeee}, // wawfinalarabic
	{0x70a7b34b, 0x05b5}, // tsere12
	{0x70c1bcae, 0x1ede}, // Ohornhookabove
	{0x70d74f05, 0x04b1}, // ustraightstrokecyrillic
	{0x70d8442c, 0xff3c}, // backslashmonospace
	{0x70e9d43c, 0x2202}, // partialdiff
	{0x70f78dac, 0x24b5}, // zparen
	{0x70f8a40d, 0x05b8}, // afii57797
	{0x71062567, 0xfeea}, // hehfinalalttwoarabic
	{0x710fbfc8, 0x0179}, // Zacute
	{0x712afb91, 0x310b}, // nbopomofo
	{0x7138408a, 0x04cc}, // chekhakassiancyrillic
	{0x71411a8f, 0x3156}, // yekorean
	{0x7154079a, 0x1e50}, // Omacrongrave
	{0x715a0e07, 0xfef4}, // alefmaksuramedialarabic
	{0x715aaa7e, 0x0122}, // Gcommaaccent
	{0x71693ad6, 0x0919}, // ngadeva
	{0x718d2f07, 0xff16}, // sixmonospace
	{0x71a4a95d, 0x1eaa}, // Acircumflextilde
	{0x71b12b2f, 0x01e0}, // Adotmacron
	{0x71b2c74d, 0x00b7}, // periodcentered
	{0x71b2e09c, 0x0e32}, // saraaathai
	{0x71e104c6, 0x3041}, // asmallhiragana
	{0x71ebf117, 0x0640}, // afii57440
	{0x7222af0c, 0x25cb}, // circle
	{0x722d9aa3, 0x2153}, // onethird
	{0x72446324, 0x3164}, // hangulfiller
	{0x7256dcb9, 0x0ac2}, // uuvowelsigngujarati
	{0x725da2a5, 0x0548}, // Voarmenian
	{0x7274464b, 0x0e31}, // maihanakatthai
	{0x7276cdf0, 0x02d7}, // minusmod
	{0x7278753d, 0x04d7}, // iebrevecyrillic
	{0x728cf068, 0x020f}, // oinvertedbreve
	{0x728f761b, 0x3111}, // qbopomofo
	{0x72ad941b, 0xf733}, // threeoldstyle
	{0x72b3c6a8, 0x00b4}, // acute
	{0x72ba7ff6, 0x3093}, // nhiragana
	{0x72c6f1c4, 0xff5c}, // barmonospace
	{0x72c8f209, 0x0136}, // Kcommaaccent
	{0x72e33f41, 0x0419}, // afii10027
	{0x72f1f5cf, 0x0438}, // iicyrillic
	{0x72fc7974, 0x2111}, // Ifraktur
	{0x72fd959a, 0x002a}, // asterisk
	{0x73032cb1, 0x3268}, // cieuccirclekorean
	{0x731486cf, 0xf6e1}, // commainferior
	{0x731a7d35, 0xf6fb}, // Ogoneksmall
	{0x731be3b3, 0x3140}, // rieulhieuhkorean
	{0x73200dbd, 0x0e20}, // phosamphaothai
	{0x7322d42e, 0x24b9}, // Dcircle
	{0x73282fcd, 0x1ee3}, // ohorndotbelow
	{0x732cf2ca, 0x2485}, // eighteenparen
	{0x7338db3a, 0x066b}, // decimalseparatorarabic
	{0x73392eb6, 0x1ed4}, // Ocircumflexhookabove
	{0x733b0480, 0xfe94}, // tehmarbutafinalarabic
	{0x734f6c6b, 0x3090}, // wihiragana
	{0x7375cb46, 0x0686}, // tcheharabic
	{0x73843708, 0x0434}, // decyrillic
	{0x7387f092, 0x2088}, // eightinferior
	{0x73b2bbd4, 0xfe63}, // hyphensmall
	{0x73b70a88, 0x05d8}, // afii57672
	{0x73b7c451, 0x095a}, // ghhadeva
	{0x73c3073f, 0xf6ee}, // lsuperior
	{0x73c5e60b, 0x004e}, // N
	{0x73e1033a, 0x22b4}, // a2
	{0x73e5498e, 0xf7fd}, // Yacutesmall
	{0x73ee6902, 0x0965}, // dbldanda
	{0x73f5578c, 0x093f}, // ivowelsigndeva
	{0x74030714, 0x311c}, // ebopomofo
	{0x7411ab47, 0x00ba}, // ordmasculine
	{0x74144417, 0x311e}, // aibopomofo
	{0x74157df0, 0x0667}, // sevenhackarabic
	{0x741e8130, 0x00cf}, // Idieresis
	{0x7435abfe, 0x057c}, // raarmenian
	{0x743d1e25, 0x0456}, // afii10103
	{0x745c44bf, 0x02e3}, // xsuperior
	{0x746e784f, 0xff48}, // hmonospace
	{0x7476bb3d, 0x21d2}, // dblarrowright
	{0x74814589, 0x05b9}, // holam19
	{0x748db30c, 0x33b1}, // nssquare
	{0x7495de6c, 0x0008}, // controlBS
	{0x74a2fb64, 0x2280}, // notprecedes
	{0x74abeab2, 0x05ac}, // iluyhebrew
	{0x74b1cbfb, 0x22c5}, // dotmath
	{0x74caf1a5, 0x01b4}, // yhook
	{0x74d74b92, 0x2209}, // notelement
	{0x74f1fca9, 0x01db}, // Udieresisgrave
	{0x7506e8c8, 0x24d1}, // bcircle
	{0x751253a5, 0x2168}, // Nineroman
	{0x7542c0c4, 0xf8e8}, // registersans
	{0x7544678b, 0x307d}, // pohiragana
	{0x754982de, 0x06a4}, // veharabic
	{0x755a250b, 0x09dd}, // rhabengali
	{0x755f707c, 0x1ebf}, // ecircumflexacute
	{0x75811646, 0xf6e9}, // asuperior
	{0x75841530, 0x2002}, // enspace
	{0x75907eb0, 0x322c}, // ideographicwaterparen
	{0x759ddc3d, 0x3068}, // tohiragana
	{0x759f9daf, 0x00c4}, // Adieresis
	{0x75aa325f, 0x3047}, // esmallhiragana
	{0x75b73269, 0x27e9}, // angbracketrightbig
	{0x75b9e64d, 0xff46}, // fmonospace
	{0x75bf74c1, 0x0a59}, // khhagurmukhi
	{0x75f4d85d, 0x0a2c}, // bagurmukhi
	{0x7603459c, 0x094c}, // auvowelsigndeva
	{0x760d83de, 0x222b}, // integral
	{0x7612320f, 0x3273}, // pieupacirclekorean
	{0x76186d8e, 0x01f3}, // dz
	{0x7619aedc, 0xfb4c}, // betrafehebrew
	{0x762dd6ce, 0x2260}, // notequal
	{0x763e1836, 0x30c4}, // tukatakana
	{0x7656c681, 0x01c4}, // DZcaron
	{0x767695a3, 0xfed4}, // fehmedialarabic
	{0x76811ae6, 0x0169}, // utilde
	{0x7681e1d4, 0xfe9b}, // thehinitialarabic
	{0x76a40aac, 0x001e}, // controlRS
	{0x76a57fc1, 0x0117}, // edot
	{0x76aa35a2, 0x1e27}, // hdieresis
	{0x76abdb93, 0xfb41}, // samekhdageshhebrew
	{0x76af2566, 0x05c1}, // afii57804
	{0x76b279b9, 0x1e76}, // Ucircumflexbelow
	{0x76be5e67, 0x0389}, // Etatonos
	{0x76e8986c, 0x27e9}, // angbracketrightBigg
	{0x76ec167e, 0x3128}, // ubopomofo
	{0x76ee8ef9, 0x326a}, // khieukhcirclekorean
	{0x76fa92ba, 0x2225}, // parallel
	{0x7722080a, 0x21e2}, // arrowdashright
	{0x772d6b37, 0x3294}, // ideographnamecircle
	{0x774eb078, 0x0328}, // ogonekcmb
	{0x774fe771, 0x064d}, // kasratanarabic
	{0x7752bcc4, 0xf8f6}, // parenrighttp
	{0x775a46a9, 0xffe0}, // centmonospace
	{0x77617e4f, 0xf762}, // Bsmall
	{0x77651620, 0x007b}, // braceleft
	{0x776629fd, 0x044a}, // hardsigncyrillic
	{0x7766a0eb, 0x041e}, // Ocyrillic
	{0x776cd4d6, 0x1e89}, // wdotbelow
	{0x77741a8d, 0x0473}, // fitacyrillic
	{0x7782c966, 0x1e46}, // Ndotbelow
	{0x7786e0a6, 0x246f}, // sixteencircle
	{0x779ce17f, 0x0390}, // iotadieresistonos
	{0x77a6f721, 0x2165}, // Sixroman
	{0x77aefb3f, 0x3027}, // sevenhangzhou
	{0x77b3e914, 0x3106}, // pbopomofo
	{0x77d27346, 0x0110}, // Dcroat
	{0x77e1d263, 0x0442}, // tecyrillic
	{0x77ee824a, 0xfba7}, // hehfinalaltonearabic
	{0x77f415ae, 0x03eb}, // gangiacoptic
	{0x77f608a5, 0x00ec}, // igrave
	{0x78075c57, 0x007d}, // bracerightBig
	{0x78151b28, 0x04f8}, // Yerudieresiscyrillic
	{0x781b1710, 0x3175}, // pieupsiostikeutkorean
	{0x782f5323, 0x2297}, // circlemultiply
	{0x783bb2b4, 0xfeea}, // hehfinalarabic
	{0x7849decd, 0x05f0}, // vavvavhebrew
	{0x7865471d, 0x30f8}, // vikatakana
	{0x7868522a, 0x0172}, // Uogonek
	{0x78728968, 0x0e3f}, // bahtthai
	{0x7877a9ea, 0x05d9}, // yodhebrew
	{0x7878b2f0, 0x323c}, // ideographicsuperviseparen
	{0x788e7d3e, 0x05b3}, // hatafqamats
	{0x78965447, 0x0593}, // shalshelethebrew
	{0x78a67a97, 0xf8ea}, // trademarksans
	{0x78d4db12, 0x032e}, // brevebelowcmb
	{0x78da6d23, 0x046e}, // Ksicyrillic
	{0x78f4df25, 0x04d9}, // schwacyrillic
	{0x79042ee6, 0x30e9}, // rakatakana
	{0x79116479, 0x0039}, // nine
	{0x7919f419, 0x0416}, // afii10024
	{0x792d6db9, 0x0272}, // nhookleft
	{0x793801c3, 0x3158}, // wakorean
	{0x79577113, 0x2565}, // SF480000
	{0x795917d1, 0x0583}, // piwrarmenian
	{0x79638654, 0x1ecd}, // odotbelow
	{0x796eb2f8, 0x3389}, // kcalsquare
	{0x796edb96, 0x0387}, // anoteleia
	{0x797aa124, 0x0445}, // khacyrillic
	{0x7980bc25, 0x00c5}, // Aring
	{0x7987b173, 0x03e8}, // Horicoptic
	{0x79961e28, 0x20a7}, // peseta
	{0x79c6a044, 0xfe3a}, // tortoiseshellbracketrightvertical
	{0x79d35b17, 0x3229}, // tenideographicparen
	{0x79dfa961, 0xfb2d}, // shindageshsindot
	{0x79e5e3c1, 0x0e2f}, // paiyannoithai
	{0x79fc5691, 0x24ce}, // Ycircle
	{0x7a196523, 0x0451}, // afii10071
	{0x7a29214e, 0x006c}, // l
	{0x7a32a546, 0x202d}, // afii61574
	{0x7a3f45e9, 0x0ab8}, // sagujarati
	{0x7a49bde9, 0xfca4}, // tehmeeminitialarabic
	{0x7a49dc21, 0xfee2}, // meemfinalarabic
	{0x7a56cace, 0x33a1}, // squaremsquared
	{0x7a6cfa6a, 0x0a3f}, // imatragurmukhi
	{0x7a75be4d, 0x2559}, // SF490000
	{0x7a8736f8, 0x320c}, // phieuphparenkorean
	{0x7aa071da, 0x0217}, // uinvertedbreve
	{0x7aa95d24, 0xfb34}, // hedageshhebrew
	{0x7aac463a, 0x0669}, // afii57401
	{0x7ab8ba14, 0x1ebe}, // Ecircumflexacute
	{0x7ad78099, 0x05b5}, // afii57794
	{0x7b074f4f, 0x062c}, // jeemarabic
	{0x7b0aca00, 0x3315}, // kiroguramusquare
	{0x7b3db32a, 0x221a}, // radicalBig
	{0x7b443626, 0x3331}, // birusquare
	{0x7b5e0418, 0xff10}, // zeromonospace
	{0x7b7b9d4d, 0x0424}, // afii10038
	{0x7b7fbb65, 0xf7b8}, // Cedillasmall
	{0x7b80aee5, 0x05bb}, // qubutsquarterhebrew
	{0x7b8c3862, 0x02b8}, // ysuperior
	{0x7bac8f33, 0x0016}, // controlSYN
	{0x7bc269dd, 0x0040}, // at
	{0x7bd5ca88, 0x33ca}, // hasquare
	{0x7bd8b82b, 0x3005}, // ideographiciterationmark
	{0x7c301325, 0x0472}, // Fitacyrillic
	{0x7c35d223, 0x05dc}, // lamedholam
	{0x7c3e1570, 0x0028}, // parenleftbigg
	{0x7c3f4566, 0x00e7}, // ccedilla
	{0x7c4777ca, 0xfcc9}, // lamjeeminitialarabic
	{0x7c4f7ece, 0x0ab2}, // lagujarati
	{0x7c53c130, 0x0a13}, // oogurmukhi
	{0x7c613499, 0x2566}, // SF410000
	{0x7c8ffdc8, 0x24e7}, // xcircle
	{0x7c903e9b, 0x1ec7}, // ecircumflexdotbelow
	{0x7cb539e8, 0x096e}, // eightdeva
	{0x7cc006ef, 0x222c}, // dblintegral
	{0x7cdf3e4a, 0x0a2f}, // yagurmukhi
	{0x7cf2e5f4, 0x0029}, // parenright
	{0x7cf2ebbd, 0xff96}, // yokatakanahalfwidth
	{0x7cf8ce2e, 0x3394}, // thzsquare
	{0x7d1ff070, 0xf896}, // thanthakhatupperleftthai
	{0x7d32ca29, 0x0127}, // hbar
	{0x7d44a576, 0x24b1}, // vparen
	{0x7d4ddd33, 0x2554}, // SF390000
	{0x7d86a046, 0xff93}, // mokatakanahalfwidth
	{0x7d8bc7c0, 0x255f}, // SF370000
	{0x7d981506, 0x0a9f}, // ttagujarati
	{0x7dc79678, 0x062a}, // afii57418
	{0x7dc79708, 0x223c}, // tildeoperator
	{0x7dc7cd45, 0x33b6}, // muvsquare
	{0x7dcf75fb, 0x223c}, // similar
	{0x7dd21a95, 0x30b5}, // sakatakana
	{0x7dd63b6c, 0x1ec1}, // ecircumflexgrave
	{0x7de5edf4, 0x05b9}, // afii57806
	{0x7df90cb9, 0x249c}, // aparen
	{0x7e46df3d, 0x30d7}, // pukatakana
	{0x7e4f805e, 0x3067}, // dehiragana
	{0x7e5c9fda, 0x042f}, // IAcyrillic
	{0x7e60a480, 0x0029}, // parenrightbigg
	{0x7e701b6d, 0xff0d}, // hyphenmonospace
	{0x7e739224, 0x3046}, // uhiragana
	{0x7e7d09ba, 0xf6e4}, // dollarsuperior
	{0x7e947543, 0xf773}, // Ssmall
	{0x7ebf5260, 0x0031}, // one
	{0x7eddac19, 0x09f4}, // onenumeratorbengali
	{0x7eefb1cf, 0x05d3}, // daletshevahebrew
	{0x7f0a4f83, 0x0631}, // rehyehaleflamarabic
	{0x7f43cdd7, 0x0ab0}, // ragujarati
	{0x7f56c599, 0x0141}, // Lslash
	{0x7f5b3579, 0x1e74}, // Utildebelow
	{0x7f630a55, 0x1ef9}, // ytilde
	{0x7f7aa31e, 0x0648}, // afii57448
	{0x7f82e2ff, 0x098f}, // ebengali
	{0x7f851974, 0x0568}, // etarmenian
	{0x7f92fe37, 0x066d}, // asteriskarabic
	{0x7f97f320, 0x301c}, // wavedash
	{0x7f99701f, 0xfb2a}, // shinshindothebrew
	{0x7fad7e30, 0xf737}, // sevenoldstyle
	{0x7fbd4335, 0x2309}, // ceilingrightbig
	{0x7ff2087d, 0x01df}, // adieresismacron
	{0x7ffdad4c, 0x326e}, // kiyeokacirclekorean
	{0x801f311e, 0x1ebb}, // ehookabove
	{0x8020003f, 0x05e9}, // shinhebrew
	{0x8027a085, 0x03d2}, // Upsilonhooksymbol
	{0x8029e67d, 0x049b}, // kadescendercyrillic
	{0x803a882f, 0x0a8b}, // rvocalicgujarati
	{0x80417827, 0x33a9}, // pasquare
	{0x80461d8b, 0xf6c7}, // afii10831
	{0x8048e51c, 0x24e6}, // wcircle
	{0x80690312, 0x0155}, // racute
	{0x8076b638, 0xfedf}, // laminitialarabic
	{0x80792dfa, 0x0478}, // Ukcyrillic
	{0x807bd424, 0xf886}, // saraiileftthai
	{0x8082c40e, 0x251c}, // SF080000
	{0x8086d5c5, 0x0490}, // afii10050
	{0x80890e42, 0x0316}, // gravebelowcmb
	{0x808de596, 0x2492}, // elevenperiod
	{0x808eef1b, 0x0068}, // h
	{0x80b21ab0, 0x00f5}, // otilde
	{0x80b54bce, 0x3170}, // mieumpansioskorean
	{0x80c36a57, 0x0480}, // Koppacyrillic
	{0x80f8e4b8, 0x017f}, // slong
	{0x80fb86d5, 0x00ff}, // ydieresis
	{0x810ea19f, 0x02bd}, // afii64937
	{0x810f1d5a, 0x30c6}, // tekatakana
	{0x812767fa, 0x0555}, // Oharmenian
	{0x8140027d, 0x24d3}, // dcircle
	{0x815134d6, 0x00cd}, // Iacute
	{0x81589a47, 0x2220}, // angle
	{0x816211ee, 0x05dc}, // lamedholamdageshhebrew
	{0x81658581, 0x0207}, // einvertedbreve
	{0x816871c9, 0x1ec2}, // Ecircumflexhookabove
	{0x8169da43, 0x0483}, // titlocyrilliccmb
	{0x8170d563, 0x256b}, // SF530000
	{0x81796466, 0xf6f2}, // ssuperior
	{0x817f112a, 0x1e37}, // ldotbelow
	{0x817f9471, 0x1e92}, // Zdotbelow
	{0x81806da2, 0x328d}, // ideographwoodcircle
	{0x8184073f, 0x053b}, // Iniarmenian
	{0x818aac3d, 0x3165}, // ssangnieunkorean
	{0x81b94a51, 0x3233}, // ideographicsocietyparen
	{0x81e18c26, 0x30c3}, // tusmallkatakana
	{0x81eec5b0, 0x320e}, // kiyeokaparenkorean
	{0x8215ea5e, 0x02b2}, // jsuperior
	{0x82173176, 0x05d9}, // afii57673
	{0x8218cc3e, 0x039c}, // Mu
	{0x822cf82c, 0x3070}, // bahiragana
	{0x8230e7a0, 0x32a7}, // ideographicleftcircle
	{0x823da9d2, 0x0413}, // afii10020
	{0x825a2038, 0x3045}, // usmallhiragana
	{0x8260fa87, 0x1ea5}, // acircumflexacute
	{0x8272e0b5, 0x249b}, // twentyperiod
	{0x827d6960, 0xff40}, // gravemonospace
	{0x828072a0, 0x2666}, // diamond
	{0x8281c05a, 0x00b5}, // mu1
	{0x8291bd7d, 0x248e}, // sevenperiod
	{0x82ab3b91, 0x03a3}, // Sigma
	{0x82b182a5, 0x057e}, // vewarmenian
	{0x82b58e6a, 0x1e94}, // Zlinebelow
	{0x82b5e1a0, 0x266b}, // eighthnotebeamed
	{0x82b5fc2f, 0x05bc}, // afii57807
	{0x82c39f59, 0x1e62}, // Sdotbelow
	{0x82c3da09, 0x0a1a}, // cagurmukhi
	{0x82d03f46, 0x056e}, // caarmenian
	{0x82d4758a, 0x25e6}, // whitebullet
	{0x82db51c1, 0x05b6}, // segolnarrowhebrew
	{0x82dffbf3, 0xfe9c}, // thehmedialarabic
	{0x8322270c, 0x1e49}, // nlinebelow
	{0x832d9a03, 0x263b}, // invsmileface
	{0x83393d74, 0xfef7}, // lamalefhamzaaboveisolatedarabic
	{0x83589346, 0x1eb0}, // Abrevegrave
	{0x836fd101, 0x25c7}, // a51
	{0x83736e7a, 0x2308}, // ceilingleftBig
	{0x837601e2, 0x0ae7}, // onegujarati
	{0x8377c3f2, 0x026b}, // lmiddletilde
	{0x837a2830, 0x2113}, // lsquare
	{0x837c1552, 0x2172}, // threeroman
	{0x8388f2be, 0x00c9}, // Eacute
	{0x83a89a03, 0x045f}, // dzhecyrillic
	{0x83aeaef2, 0x1e23}, // hdotaccent
	{0x83c5c486, 0x05e7}, // afii57687
	{0x83cd4320, 0x0565}, // echarmenian
	{0x83e3c3a4, 0x3075}, // huhiragana
	{0x83f86a79, 0x339e}, // squarekm
	{0x83fc4077, 0x0037}, // seven
	{0x83fe9093, 0x0441}, // escyrillic
	{0x840c8507, 0x0313}, // commaabovecmb
	{0x8411d0c3, 0x1e3e}, // Macute
	{0x84132896, 0x0140}, // ldot
	{0x841d72ff, 0x02e4}, // glottalstopreversedsuperior
	{0x841d9ceb, 0x0572}, // ghadarmenian
	{0x8424a64d, 0x05b5}, // tserewidehebrew
	{0x843c7aca, 0x323f}, // ideographicallianceparen
	{0x8445bb28, 0x0905}, // adeva
	{0x8459e5f4, 0x2248}, // approxequal
	{0x84776c26, 0x329d}, // ideographicexcellentcircle
	{0x847d9b8a, 0x1ec5}, // ecircumflextilde
	{0x847ed2d4, 0x0540}, // Hoarmenian
	{0x8487ed37, 0x0621}, // hamzaarabic
	{0x848baa07, 0xfc5f}, // shaddakasratanarabic
	{0x8494dfda, 0x05b1}, // hatafsegol24
	{0x8498dbae, 0xfe86}, // wawhamzaabovefinalarabic
	{0x849b0297, 0x1e2e}, // Idieresisacute
	{0x84a67fee, 0x323a}, // ideographiccallparen
	{0x84b1d089, 0x0430}, // acyrillic
	{0x84de35f4, 0x0210}, // Rdblgrave
	{0x84e0b580, 0x04ef}, // umacroncyrillic
	{0x8501599a, 0x30f6}, // kesmallkatakana
	{0x85103d59, 0xfb39}, // yoddagesh
	{0x851a3968, 0xf6da}, // registerserif
	{0x853a4f53, 0x217b}, // twelveroman
	{0x855c805d, 0xff1a}, // colonmonospace
	{0x856013c8, 0x248a}, // threeperiod
	{0x8567c154, 0x01bc}, // Tonefive
	{0x856bfeb5, 0x0029}, // parenrightBigg
	{0x85a6c8bc, 0x310a}, // tbopomofo
	{0x85b41716, 0x201a}, // quotesinglbase
	{0x85b6b018, 0x2228}, // logicalor
	{0x85c83eb6, 0xff77}, // kikatakanahalfwidth
	{0x85cde6eb, 0xf8e7}, // arrowhorizex
	{0x85ce16f4, 0x05da}, // afii57674
	{0x85e92466, 0x2282}, // propersubset
	{0x85ee23d7, 0x311b}, // obopomofo
	{0x85f325ac, 0x064c}, // dammatanaltonearabic
	{0x85f47ad1, 0x0660}, // zerohackarabic
	{0x85f89b18, 0x3049}, // osmallhiragana
	{0x86129c18, 0x2209}, // notelementof
	{0x861e8048, 0xff54}, // tmonospace
	{0x861ef200, 0x1e75}, // utildebelow
	{0x864008ae, 0x05d3}, // daletqubutshebrew
	{0x8654695b, 0x0020}, // spacehackarabic
	{0x865c7659, 0x0312}, // commaturnedabovecmb
	{0x8665f7ab, 0x201b}, // quotereversed
	{0x8686768e, 0x093e}, // aavowelsigndeva
	{0x8694e2ae, 0x3132}, // ssangkiyeokkorean
	{0x8699e716, 0x09ef}, // ninebengali
	{0x869fef4c, 0x064b}, // afii57451
	{0x86a662d1, 0x05e7}, // qofhebrew
	{0x86a80595, 0x0535}, // Echarmenian
	{0x86c2d4a4, 0x01ef}, // ezhcaron
	{0x86d0b880, 0x01c7}, // LJ
	{0x86d66230, 0x04ad}, // tedescendercyrillic
	{0x86e222a0, 0x05d9}, // yod
	{0x86eea6f9, 0x03a4}, // Tau
	{0x870315b6, 0x30e1}, // mekatakana
	{0x8713ac2f, 0xfb4b}, // vavholam
	{0x87634c8c, 0x05bd}, // siluqhebrew
	{0x877afbbb, 0xfb35}, // afii57723
	{0x877f7c7a, 0xfeb4}, // seenmedialarabic
	{0x8787b92b, 0x02c0}, // glottalstopmod
	{0x87a041c7, 0x0446}, // afii10088
	{0x87a4b30d, 0x263c}, // compass
	{0x87b6c8ba, 0x00c3}, // Atilde
	{0x87bb7ac5, 0x020b}, // iinvertedbreve
	{0x87e03b4f, 0x0642}, // afii57442
	{0x87e97d46, 0x04dd}, // zhedieresiscyrillic
	{0x87ef58ab, 0xfe30}, // twodotleadervertical
	{0x87fc47e5, 0x24d2}, // ccircle
	{0x88044bf5, 0x2167}, // Eightroman
	{0x880a9911, 0x1e34}, // Klinebelow
	{0x881905a5, 0x0e22}, // yoyakthai
	{0x881a4ba0, 0x0931}, // rradeva
	{0x88264250, 0x1e71}, // tcircumflexbelow
	{0x8826a561, 0x03cd}, // upsilontonos
	{0x883bae04, 0x248c}, // fiveperiod
	{0x8841d986, 0x30b8}, // zikatakana
	{0x8853f322, 0x339b}, // mumsquare
	{0x885dcb80, 0x00a7}, // section
	{0x886c5d13, 0x0636}, // dadarabic
	{0x8876a700, 0x1e98}, // wring
	{0x888833ba, 0x02e7}, // tonebarmidmod
	{0x888e1142, 0x026d}, // lhookretroflex
	{0x88a12621, 0x05d4}, // afii57668
	{0x88b6884b, 0x06f7}, // sevenpersian
	{0x88c2fc70, 0x0a6a}, // fourgurmukhi
	{0x88c96d26, 0xfb4a}, // tavdages
	{0x88cc32a3, 0x1ef6}, // Yhookabove
	{0x88da326f, 0x0546}, // Nowarmenian
	{0x88e4df33, 0x1e1b}, // etildebelow
	{0x88e6eee8, 0x3119}, // sbopomofo
	{0x88ea0124, 0x0949}, // ocandravowelsigndeva
	{0x88ea9631, 0xf7af}, // Macronsmall
	{0x88f38eed, 0x21d0}, // arrowdblleft
	{0x88f62270, 0x314f}, // akorean
	{0x88fdc1bb, 0x308e}, // wasmallhiragana
	{0x88fdcf2e, 0x039e}, // Xi
	{0x89098ea4, 0x2211}, // summationdisplay
	{0x893a8f13, 0x09b9}, // habengali
	{0x89468742, 0x0e1f}, // fofanthai
	{0x897340ea, 0x3207}, // ieungparenkorean
	{0x897854cd, 0x060c}, // afii57388
	{0x897adc4f, 0x0128}, // Itilde
	{0x89aa67b7, 0x2161}, // Tworoman
	{0x89ace505, 0xff06}, // ampersandmonospace
	{0x89d8daae, 0x25c9}, // fisheye
	{0x89e05206, 0x0022}, // quotedbl
	{0x89e2c74b, 0x062d}, // afii57421
	{0x89ed1e17, 0x3171}, // kapyeounmieumkorean
	{0x89f4c981, 0x02dc}, // tilde
	{0x89f7042b, 0xfe38}, // bracerightvertical
	{0x89fc7dc4, 0x05bf}, // rafehebrew
	{0x8a1958e1, 0x03d1}, // theta1
	{0x8a27f623, 0x030e}, // dblverticallineabovecmb
	{0x8a36e0e0, 0x0a89}, // ugujarati
	{0x8a64ee32, 0x23a2}, // bracketleftex
	{0x8a743e71, 0x3279}, // thieuthacirclekorean
	{0x8a7ff438, 0x0632}, // zainarabic
	{0x8ab11ede, 0x0907}, // ideva
	{0x8ab50af4, 0xfedb}, // kafinitialarabic
	{0x8ac9c3df, 0x00cb}, // Edieresis
	{0x8ad52e55, 0x05d3}, // dalethatafsegolhebrew
	{0x8ae82e18, 0x05c2}, // sindothebrew
	{0x8aebef01, 0x064f}, // afii57455
	{0x8af00e69, 0x3396}, // mlsquare
	{0x8af0299d, 0x2465}, // sixcircle
	{0x8af3a8ba, 0x1e55}, // pacute
	{0x8afdb385, 0x05b2}, // hatafpatahnarrowhebrew
	{0x8b04e879, 0x0573}, // cheharmenian
	{0x8b2ea84b, 0x045f}, // afii10193
	{0x8b336b03, 0x33c3}, // bqsquare
	{0x8b3d9ff3, 0x313a}, // rieulkiyeokkorean
	{0x8b46ba2c, 0x1e10}, // Dcedilla
	{0x8b5fbe71, 0x0209}, // idblgrave
	{0x8b6e36e5, 0x0001}, // controlSTX
	{0x8b91eefb, 0x0135}, // jcircumflex
	{0x8ba07e30, 0x3236}, // ideographicfinancialparen
	{0x8ba61ca6, 0x0170}, // Udblacute
	{0x8bb0c7c5, 0x04a4}, // Enghecyrillic
	{0x8bb9fe95, 0x1e35}, // klinebelow
	{0x8bc6e552, 0x027a}, // rlonglegturned
	{0x8bd60707, 0x33ce}, // squarekmcapital
	{0x8bdd4dfa, 0x05a1}, // pazerhebrew
	{0x8be0aeeb, 0x1e28}, // Hcedilla
	{0x8bf3f9ec, 0x0aed}, // sevengujarati
	{0x8bfc7ed2, 0x02e9}, // tonebarextralowmod
	{0x8c09cf1f, 0x0251}, // ascript
	{0x8c169f16, 0x0a4c}, // aumatragurmukhi
	{0x8c1a91ec, 0x1ed2}, // Ocircumflexgrave
	{0x8c21575a, 0x2175}, // sixroman
	{0x8c31d770, 0x007d}, // bracerightBigg
	{0x8c382c45, 0xf6c5}, // afii10064
	{0x8c3dc32f, 0x0632}, // afii57426
	{0x8c5fd3c7, 0x05dd}, // afii57677
	{0x8c6ceb51, 0xeb61}, // suppress
	{0x8c7614fd, 0x0046}, // F
	{0x8c789c98, 0x0255}, // ccurl
	{0x8c7eba77, 0x3114}, // chbopomofo
	{0x8c941f03, 0x0639}, // afii57433
	{0x8c9c3f66, 0x0e37}, // saraueethai
	{0x8c9d6579, 0x2308}, // ceilingleftBigg
	{0x8cb6cb59, 0xfb6c}, // vehinitialarabic
	{0x8cbe2a76, 0x0951}, // udattadeva
	{0x8cc838c1, 0x099f}, // ttabengali
	{0x8ccab94e, 0x311d}, // ehbopomofo
	{0x8cd2ceac, 0x322f}, // ideographicearthparen
	{0x8cd87862, 0x2a01}, // circleplusdisplay
	{0x8cdd7a5e, 0xfefb}, // lamalefisolatedarabic
	{0x8cfeaaad, 0x04b4}, // Tetsecyrillic
	{0x8d0e3bcc, 0x0162}, // Tcommaaccent
	{0x8d15f0f9, 0x01d8}, // udieresisacute
	{0x8d17055f, 0xff23}, // Cmonospace
	{0x8d2183aa, 0x00ee}, // icircumflex
	{0x8d24b8c2, 0x0457}, // afii10104
	{0x8d262f6a, 0x33ad}, // radsquare
	{0x8d3bc051, 0x1eb1}, // abrevegrave
	{0x8d430411, 0x0288}, // tretroflexhook
	{0x8d4c7b99, 0x0473}, // afii10195
	{0x8d557bba, 0xfe98}, // tehmedialarabic
	{0x8d5ae6d4, 0x0440}, // ercyrillic
	{0x8d62c566, 0x21e0}, // arrowdashleft
	{0x8d7f2614, 0x2164}, // Fiveroman
	{0x8d983a6e, 0x22b5}, // a4
	{0x8db3e76c, 0x1e97}, // tdieresis
	{0x8dbd6f57, 0xffe1}, // sterlingmonospace
	{0x8dc41abc, 0x1e33}, // kdotbelow
	{0x8de51633, 0x1ed7}, // ocircumflextilde
	{0x8df7afeb, 0x0968}, // twodeva
	{0x8e0d9fbd, 0x0337}, // solidusshortoverlaycmb
	{0x8e2538f6, 0x0420}, // Ercyrillic
	{0x8e325301, 0x0133}, // ij
	{0x8e3386c6, 0x0439}, // iishortcyrillic
	{0x8e360b54, 0x027d}, // rhook
	{0x8e38f2c0, 0x005d}, // bracketrightbig
	{0x8e40b292, 0x01fb}, // aringacute
	{0x8e467ab5, 0x1ecc}, // Odotbelow
	{0x8e474342, 0x1e2a}, // Hbrevebelow
	{0x8e5ae93f, 0x0909}, // udeva
	{0x8e5f575a, 0x0458}, // jecyrillic
	{0x8e5f7ac5, 0x01d2}, // ocaron
	{0x8e6f5aa1, 0x3074}, // pihiragana
	{0x8e757f60, 0x308a}, // rihiragana
	{0x8e7aa1b5, 0x207c}, // equalsuperior
	{0x8e8e0711, 0x042f}, // afii10049
	{0x8e91af6c, 0x3000}, // ideographicspace
	{0x8ec625f1, 0x2304}, // a43
	{0x8ec9ea4f, 0xfb40}, // nundagesh
	{0x8eca9ee8, 0x045c}, // kjecyrillic
	{0x8ed1765d, 0xff36}, // Vmonospace
	{0x8ed539a2, 0x278a}, // onecircleinversesansserif
	{0x8f084bdd, 0x3204}, // mieumparenkorean
	{0x8f19bc84, 0x0e4a}, // maitrithai
	{0x8f289d06, 0x0444}, // efcyrillic
	{0x8f28eae1, 0x013f}, // Ldotaccent
	{0x8f3d926c, 0xf7e2}, // Acircumflexsmall
	{0x8f4b9c51, 0x0404}, // afii10053
	{0x8f505863, 0x04f2}, // Uhungarumlautcyrillic
	{0x8f52df2f, 0x0256}, // dtail
	{0x8f5e284f, 0xfed6}, // qaffinalarabic
	{0x8f760fbe, 0x25e5}, // blackupperrighttriangle
	{0x8f7f3f67, 0x0645}, // afii57445
	{0x8f89b56f, 0x05b8}, // qamatsqatanhebrew
	{0x8f9b61ad, 0x22c0}, // logicalanddisplay
	{0x8fa69b6c, 0x0331}, // macronbelowcmb
	{0x8fbba331, 0x0205}, // edblgrave
	{0x8fc968d8, 0x01b9}, // ezhreversed
	{0x8fce94ba, 0x0395}, // Epsilon
	{0x8fd18473, 0x2593}, // shadedark
	{0x8fe2c390, 0x0e53}, // threethai
	{0x8fe329b9, 0x266c}, // beamedsixteenthnotes
	{0x8fe85541, 0x0637}, // afii57431
	{0x8ff897b6, 0x0042}, // B
	{0x900e8281, 0x1e3b}, // llinebelow
	{0x900fb5c0, 0x0144}, // nacute
	{0x902443c2, 0xfe52}, // periodsmall
	{0x9024a760, 0x029d}, // jcrossedtail
	{0x90307534, 0x3059}, // suhiragana
	{0x9059f738, 0x00b7}, // middot
	{0x906746a4, 0xff75}, // okatakanahalfwidth
	{0x907d968c, 0x0a9b}, // chagujarati
	{0x90872973, 0x0538}, // Etarmenian
	{0x9098fbd4, 0x0002}, // controlSOT
	{0x90995fc1, 0x1e09}, // ccedillaacute
	{0x90a162b6, 0x05b4}, // hiriqwidehebrew
	{0x90b86ad8, 0x30dd}, // pokatakana
	{0x90b9c076, 0xff0f}, // slashmonospace
	{0x90c2be85, 0x0268}, // istroke
	{0x90d8e15f, 0xf6e8}, // periodsuperior
	{0x91032be8, 0x02c6}, // hatwide
	{0x910a1b16, 0x03f1}, // rhosymbolgreek
	{0x91306ea5, 0x2127}, // a48
	{0x9132f814, 0xff22}, // Bmonospace
	{0x9134ebbc, 0x01cb}, // Nj
	{0x913ff5ff, 0x3125}, // engbopomofo
	{0x9141d43c, 0x2126}, // Ohm
	{0x914548fb, 0xf7f9}, // Ugravesmall
	{0x914ce494, 0x05e6}, // afii57686
	{0x9166eec8, 0x33ba}, // pwsquare
	{0x916cdeb8, 0xfed2}, // fehfinalarabic
	{0x917f2f3f, 0x0438}, // afii10074
	{0x9181b388, 0x01eb}, // oogonek
	{0x9184e24f, 0x30ab}, // kakatakana
	{0x919c9ad4, 0xf898}, // thanthakhatlowleftthai
	{0x919f5679, 0x00ca}, // Ecircumflex
	{0x91acc220, 0xff91}, // mukatakanahalfwidth
	{0x91accd4b, 0x0937}, // ssadeva
	{0x91c3e17e, 0x05f2}, // afii57718
	{0x91d99037, 0x1eb9}, // edotbelow
	{0x91de3939, 0x064f}, // dammalowarabic
	{0x91e65480, 0x0abf}, // ivowelsigngujarati
	{0x91ea8b93, 0x2303}, // a42
	{0x91eaac20, 0x3214}, // siosaparenkorean
	{0x920233a7, 0xf6de}, // threequartersemdash
	{0x920dae79, 0x039f}, // Omicron
	{0x9215b042, 0x3147}, // ieungkorean
	{0x9220d7f0, 0xff69}, // usmallkatakanahalfwidth
	{0x923767e3, 0x3133}, // kiyeoksioskorean
	{0x9239e7fb, 0x062b}, // theharabic
	{0x923bf3d0, 0x0330}, // tildebelowcmb
	{0x926b691e, 0x0100}, // Amacron
	{0x92aa52d3, 0x30fb}, // dotkatakana
	{0x92e2ffd9, 0x21e8}, // arrowrightwhite
	{0x92e50e35, 0xf88e}, // maitholowrightthai
	{0x92f283dc, 0x25d8}, // bulletinverse
	{0x92f96dbe, 0x1e56}, // Pdotaccent
	{0x930724f6, 0x06ba}, // noonghunnaarabic
	{0x930c1a0b, 0x001b}, // controlESC
	{0x932512ee, 0x03ea}, // Gangiacoptic
	{0x9330a2fc, 0x0336}, // strokelongoverlaycmb
	{0x934b1595, 0x062d}, // haharabic
	{0x93959445, 0x263a}, // smileface
	{0x939a56c4, 0x03ad}, // epsilontonos
	{0x939b5eb8, 0xfe92}, // behmedialarabic
	{0x93bca3b6, 0x099e}, // nyabengali
	{0x93e00dc4, 0x2193}, // arrowdown
	{0x93eef318, 0x0263}, // gammalatinsmall
	{0x9404d5fc, 0x33d5}, // squaremil
	{0x941a6b5f, 0x0a23}, // nnagurmukhi
	{0x941b20fa, 0xfe4b}, // overlinewavy
	{0x942ad1c7, 0x09be}, // aavowelsignbengali
	{0x9453959c, 0x24bd}, // Hcircle
	{0x9464bc2e, 0x1e66}, // Scarondotaccent
	{0x94724b66, 0x21bc}, // harpoonleftbarbup
	{0x94803386, 0x09d7}, // aulengthmarkbengali
	{0x948a9ce4, 0x05de}, // afii57678
	{0x949bc805, 0x01da}, // udieresiscaron
	{0x94ae0441, 0x0410}, // Acyrillic
	{0x94b7f6ea, 0x0463}, // yatcyrillic
	{0x94c36e74, 0x3261}, // nieuncirclekorean
	{0x94c9571f, 0x25ac}, // blackrectangle
	{0x94ca16e5, 0xf893}, // maichattawaupperleftthai
	{0x94d13d1c, 0xfe39}, // tortoiseshellbracketleftvertical
	{0x94d44c33, 0x007b}, // braceleftbig
	{0x94d74b96, 0x1e64}, // Sacutedotaccent
	{0x94e6f584, 0x3058}, // zihiragana
	{0x94ee5ae7, 0x2792}, // ninecircleinversesansserif
	{0x94f9a508, 0x3265}, // pieupcirclekorean
	{0x9518a20d, 0x30d9}, // bekatakana
	{0x951a0238, 0xfdf2}, // lamlamhehisolatedarabic
	{0x951ae869, 0x09dc}, // rrabengali
	{0x952cce64, 0x04ee}, // Umacroncyrillic
	{0x952ec009, 0x0988}, // iibengali
	{0x95394a64, 0x05da}, // finalkafsheva
	{0x953a0a51, 0x2211}, // summation
	{0x954920d5, 0xf769}, // Ismall
	{0x954a8776, 0x03d1}, // thetasymbolgreek
	{0x95526ac8, 0x2500}, // SF100000
	{0x9559e176, 0x05b7}, // patah2a
	{0x955dbbe7, 0x23aa}, // braceex
	{0x957765bc, 0xfe8a}, // yehhamzaabovefinalarabic
	{0x958830cb, 0x2669}, // quarternote
	{0x9588e4f1, 0xff99}, // rukatakanahalfwidth
	{0x959cf6c1, 0x203b}, // referencemark
	{0x95af6475, 0x05e3}, // finalpehebrew
	{0x95aff05f, 0x03ca}, // iotadieresis
	{0x95b3bc07, 0xfb46}, // tsadidagesh
	{0x95bed968, 0x0e15}, // totaothai
	{0x95cabf3f, 0x21cf}, // arrowrightdblstroke
	{0x95d7e2f4, 0x0032}, // two
	{0x95ed768c, 0x05e0}, // nun
	{0x960140f0, 0x2496}, // fifteenperiod
	{0x961b2e15, 0x1e6f}, // tlinebelow
	{0x96220dd7, 0x318a}, // yuyeokorean
	{0x962b0c72, 0x3323}, // sentosquare
	{0x9638605a, 0x0669}, // ninearabic
	{0x967b01ac, 0x05b9}, // holamhebrew
	{0x967d0326, 0x3134}, // nieunkorean
	{0x968e4cb7, 0xf899}, // nikhahitleftthai
	{0x96a5e022, 0x25d9}, // invcircle
	{0x96b677d5, 0x0153}, // oe
	{0x96c05d98, 0x01f2}, // Dz
	{0x96c1ab16, 0x247f}, // twelveparen
	{0x96d9cc68, 0x0427}, // Checyrillic
	{0x96fd8ec6, 0x1e7a}, // Umacrondieresis
	{0x9711eb31, 0x21aa}, // arrowhookleft
	{0x9741ad45, 0x05bf}, // rafe
	{0x975dc1dc, 0x32a5}, // ideographiccentrecircle
	{0x9776a4ba, 0x0007}, // controlBEL
	{0x977737b3, 0x0265}, // hturned
	{0x9778a35b, 0x0562}, // benarmenian
	{0x977e0dfa, 0x25d0}, // circlewithlefthalfblack
	{0x97843a2e, 0x09f8}, // denominatorminusonenumeratorbengali
	{0x978c8c89, 0x03ac}, // alphatonos
	{0x97ae16ea, 0x23a1}, // bracketlefttp
	{0x97b3e7db, 0x24c3}, // Ncircle
	{0x97e45478, 0x05b6}, // segol2c
	{0x97ea0cb5, 0x04d8}, // Schwacyrillic
	{0x97ebb44e, 0x015f}, // scedilla
	{0x97f03f9c, 0x0419}, // Iishortcyrillic
	{0x97f6721e, 0x05bf}, // afii57841
	{0x980e76a2, 0x1e32}, // Kdotbelow
	{0x98148d7b, 0xff14}, // fourmonospace
	{0x981fc90b, 0x0a1b}, // chagurmukhi
	{0x982585a7, 0x260e}, // telephoneblack
	{0x982718e0, 0x003c}, // less
	{0x982eb09a, 0x0e5b}, // khomutthai
	{0x9853033e, 0x04b5}, // tetsecyrillic
	{0x987e6d13, 0x0411}, // Becyrillic
	{0x9896e370, 0x0402}, // afii10051
	{0x98b02dc0, 0x0a48}, // aimatragurmukhi
	{0x98bf4a1b, 0xf6d0}, // Macron
	{0x98c60f17, 0xf6d9}, // copyrightserif
	{0x98d74b1c, 0x01c8}, // Lj
	{0x98d9aba5, 0x03ae}, // etatonos
	{0x98eba766, 0x018b}, // Dtopbar
	{0x98f4783f, 0x24ac}, // qparen
	{0x98f4b751, 0x0651}, // afii57457
	{0x98ffb065, 0x02c6}, // hatwider
	{0x99104281, 0x02e5}, // tonebarextrahighmod
	{0x99235205, 0x05dc}, // lamedhebrew
	{0x994ebac3, 0x05c0}, // afii57842
	{0x99725844, 0x0320}, // minusbelowcmb
	{0x9982855c, 0x0686}, // afii57507
	{0x99830dc7, 0x062c}, // afii57420
	{0x99863852, 0x03ed}, // shimacoptic
	{0x99997c4f, 0x3314}, // kirosquare
	{0x999c619c, 0x3078}, // hehiragana
	{0x999f4db4, 0x05b4}, // afii57793
	{0x99cca883, 0xff04}, // dollarmonospace
	{0x99e63f81, 0x0962}, // lvocalicvowelsigndeva
	{0x9a069ea3, 0x2267}, // greateroverequal
	{0x9a098276, 0xfea2}, // hahfinalarabic
	{0x9a157ece, 0x246a}, // elevencircle
	{0x9a1c929d, 0x043c}, // afii10078
	{0x9a310f17, 0xff51}, // qmonospace
	{0x9a3391f5, 0x0190}, // Eopen
	{0x9a464a33, 0x0174}, // Wcircumflex
	{0x9a50ec2e, 0x05b6}, // segolquarterhebrew
	{0x9a7aab21, 0x05c3}, // sofpasuqhebrew
	{0x9ac6c137, 0x0e0f}, // topatakthai
	{0x9ae2a69d, 0xf6c0}, // ll
	{0x9aea680b, 0x3029}, // ninehangzhou
	{0x9af6d63b, 0x054a}, // Peharmenian
	{0x9b064cf1, 0xfedf}, // lammeemjeeminitialarabic
	{0x9b09b61d, 0x0a10}, // aigurmukhi
	{0x9b0db21d, 0x0402}, // Djecyrillic
	{0x9b100042, 0x0e21}, // momathai
	{0x9b29e68e, 0x0278}, // philatin
	{0x9b3ff954, 0x0e56}, // sixthai
	{0x9b5a3eb3, 0x0626}, // afii57414
	{0x9b712e01, 0x1eba}, // Ehookabove
	{0x9b73811a, 0x2227}, // logicaland
	{0x9b76648b, 0x041f}, // afii10033
	{0x9b7712b3, 0x1e5a}, // Rdotbelow
	{0x9b8591a5, 0x30ca}, // nakatakana
	{0x9b950b60, 0x095e}, // fadeva
	{0x9ba02025, 0x01e9}, // kcaron
	{0x9bca0720, 0x1e93}, // zdotbelow
	{0x9bcccde6, 0x0e4e}, // yamakkanthai
	{0x9bd59a36, 0x300e}, // whitecornerbracketleft
	{0x9bdb98a4, 0x1e5e}, // Rlinebelow
	{0x9bdfdedf, 0x05d3}, // daletsheva
	{0x9be54046, 0x0e0a}, // chochangthai
	{0x9bfe067d, 0x0405}, // Dzecyrillic
	{0x9c14c866, 0x0484}, // palatalizationcyrilliccmb
	{0x9c1ff986, 0x05f1}, // afii57717
	{0x9c30e64e, 0x0121}, // gdot
	{0x9c3d076c, 0x002d}, // hyphen
	{0x9c5df589, 0x03b2}, // beta
	{0x9c5e488c, 0x05d3}, // dalethebrew
	{0x9c743ddb, 0x3239}, // ideographicrepresentparen
	{0x9cc9b890, 0x2a00}, // circledottext
	{0x9cd2074a, 0x0a36}, // shagurmukhi
	{0x9ce0dacf, 0xfebb}, // sadinitialarabic
	{0x9ce3d2fe, 0x06d2}, // afii57519
	{0x9ce9cdfc, 0x0408}, // Jecyrillic
	{0x9ce9f027, 0x0426}, // Tsecyrillic
	{0x9cf54095, 0x20aa}, // newsheqelsign
	{0x9d1b1141, 0x25d8}, // invbullet
	{0x9d1ed2c0, 0x0120}, // Gdotaccent
	{0x9d25f804, 0x0294}, // glottalstop
	{0x9d3a5187, 0x03bb}, // lambda
	{0x9d4507ca, 0x00a0}, // nonbreakingspace
	{0x9d4ea24d, 0x045a}, // njecyrillic
	{0x9d5ba323, 0x3145}, // sioskorean
	{0x9d5eb9a4, 0x001f}, // controlUS
	{0x9d662219, 0x332a}, // haitusquare
	{0x9d760ad7, 0x3318}, // guramusquare
	{0x9d770652, 0x1e2b}, // hbrevebelow
	{0x9db9ebc8, 0x30ba}, // zukatakana
	{0x9dee7277, 0x2015}, // afii00208
	{0x9df531bb, 0x059b}, // tevirhebrew
	{0x9e021469, 0x22cf}, // curlyand
	{0x9e062707, 0x02a2}, // glottalstopstrokereversed
	{0x9e0bf218, 0x02b1}, // hhooksuperior
	{0x9e0d1458, 0x02c4}, // arrowheadupmod
	{0x9e1247f8, 0x0033}, // three
	{0x9e248728, 0x3053}, // kohiragana
	{0x9e2d5a68, 0x053f}, // Kenarmenian
	{0x9e37413a, 0x22c2}, // intersectiondisplay
	{0x9e4de0cc, 0x221a}, // radicalBigg
	{0x9e5de325, 0x222e}, // contourintegral
	{0x9e65e800, 0x248b}, // fourperiod
	{0x9e98d52c, 0x2483}, // sixteenparen
	{0x9ea14168, 0x05b7}, // patahquarterhebrew
	{0x9ea23fe1, 0x00f1}, // ntilde
	{0x9eac193b, 0x00ef}, // idieresis
	{0x9eb5aea3, 0x3142}, // pieupkorean
	{0x9ebea1a0, 0x3150}, // aekorean
	{0x9ee7bbd1, 0x094b}, // ovowelsigndeva
	{0x9eeac84b, 0xfee7}, // noonhehinitialarabic
	{0x9eedaba9, 0x0113}, // emacron
	{0x9ef0c911, 0xf765}, // Esmall
	{0x9f023815, 0x20ac}, // euro
	{0x9f30fc87, 0xfec2}, // tahfinalarabic
	{0x9f37894c, 0x040e}, // afii10062
	{0x9f53036c, 0x0a19}, // ngagurmukhi
	{0x9f65cf71, 0x1e25}, // hdotbelow
	{0x9f69147e, 0x1e61}, // sdotaccent
	{0x9f6f9105, 0x0433}, // gecyrillic
	{0x9f739695, 0x04f9}, // yerudieresiscyrillic
	{0x9f79f6eb, 0xfb33}, // daletdageshhebrew
	{0x9f7f5e1f, 0xf897}, // thanthakhatlowrightthai
	{0x9f8cff14, 0x003e}, // greater
	{0x9f94b2e4, 0x04a1}, // kabashkircyrillic
	{0x9fa5f7ad, 0x0e49}, // maithothai
	{0x9fa872ec, 0x02dc}, // tildewidest
	{0x9fc7ffac, 0x05b4}, // hiriqhebrew
	{0x9fd406b1, 0xfed8}, // qafmedialarabic
	{0x9fd7c50e, 0x05bb}, // afii57796
	{0x9fdfc7a1, 0x00b0}, // degree
	{0x9ffeaad9, 0x01ed}, // oogonekmacron
	{0xa0144bc6, 0xfc8d}, // noonnoonfinalarabic
	{0xa0166e3d, 0x3159}, // waekorean
	{0xa016fb2d, 0x016b}, // umacron
	{0xa0286aa8, 0x1e04}, // Bdotbelow
	{0xa03db58b, 0x02d2}, // ringhalfrightcentered
	{0xa05ccf71, 0x05e1}, // samekhhebrew
	{0xa069fd2d, 0x012b}, // imacron
	{0xa08ca5a7, 0x2491}, // tenperiod
	{0xa09c7d02, 0x05e8}, // reshhatafpatahhebrew
	{0xa0a317f9, 0x0574}, // menarmenian
	{0xa0adde45, 0xf884}, // maihanakatleftthai
	{0xa0c2ffe3, 0x247a}, // sevenparen
	{0xa0e40fac, 0x337b}, // heiseierasquare
	{0xa0e487b8, 0x33cd}, // KKsquare
	{0xa0ee672b, 0x3143}, // ssangpieupkorean
	{0xa100bc11, 0x043e}, // ocyrillic
	{0xa10462a8, 0x0394}, // Deltagreek
	{0xa11f6f39, 0x0071}, // q
	{0xa12507ea, 0x2177}, // eightroman
	{0xa12d2230, 0x05b5}, // tsere2b
	{0xa134a191, 0xff11}, // onemonospace
	{0xa14f5367, 0x05b0}, // sheva15
	{0xa14fd78e, 0x2193}, // arrowbt
	{0xa157c7c6, 0x0643}, // kafarabic
	{0xa15811a3, 0x061f}, // questionarabic
	{0xa1697005, 0x0015}, // controlNAK
	{0xa16fa8a4, 0x22a3}, // tackleft
	{0xa1703e0a, 0x3219}, // thieuthaparenkorean
	{0xa1850262, 0x05c2}, // afii57803
	{0xa1a14a63, 0x1eb8}, // Edotbelow
	{0xa1ed89db, 0x0202}, // Ainvertedbreve
	{0xa20cadbf, 0x0062}, // b
	{0xa20ea9da, 0x24aa}, // oparen
	{0xa212ed2d, 0x0e38}, // sarauthai
	{0xa23bb3ad, 0x222b}, // integraltext
	{0xa2448aa1, 0x01e6}, // Gcaron
	{0xa2543878, 0x326f}, // nieunacirclekorean
	{0xa254ebdd, 0xfb3e}, // memdagesh
	{0xa259bfe7, 0xff19}, // ninemonospace
	{0xa262edc1, 0xfe49}, // overlinedashed
	{0xa26bc10f, 0x061b}, // semicolonarabic
	{0xa27876ee, 0xfe35}, // parenleftvertical
	{0xa28a5f58, 0xfec8}, // zahmedialarabic
	{0xa28ba8ac, 0x317d}, // siospieupkorean
	{0xa2972ad9, 0x2305}, // projective
	{0xa2be0dd9, 0x062b}, // afii57419
	{0xa2c2120e, 0x0e52}, // twothai
	{0xa2d967e3, 0x2083}, // threeinferior
	{0xa2e1fb7a, 0x221e}, // infinity
	{0xa2f62d95, 0x306e}, // nohiragana
	{0xa3004c6f, 0x092e}, // madeva
	{0xa316ccc6, 0x2044}, // fraction
	{0xa32a4538, 0xfb1f}, // afii57705
	{0xa334b2d1, 0x043d}, // afii10079
	{0xa34b5d2f, 0x0e57}, // seventhai
	{0xa36dbdee, 0x30db}, // hokatakana
	{0xa386d6fe, 0x3327}, // tonsquare
	{0xa3903917, 0x3270}, // tikeutacirclekorean
	{0xa39b2570, 0xfefc}, // lamaleffinalarabic
	{0xa3b51a89, 0x0597}, // reviamugrashhebrew
	{0xa3cc74fa, 0x3056}, // zahiragana
	{0xa3d9a90d, 0x0499}, // zedescendercyrillic
	{0xa3e95215, 0xfef9}, // lamalefhamzabelowisolatedarabic
	{0xa3ec709c, 0x0569}, // toarmenian
	{0xa3fbf1d9, 0x0497}, // zhedescendercyrillic
	{0xa3fe88d1, 0x0104}, // Aogonek
	{0xa40369ed, 0x32a9}, // ideographicmedicinecircle
	{0xa4259ac9, 0xff76}, // kakatakanahalfwidth
	{0xa43a91b7, 0x005b}, // bracketleftbig
	{0xa43cdc2e, 0xfb44}, // pedageshhebrew
	{0xa446d45f, 0x3224}, // fiveideographicparen
	{0xa450f946, 0x0454}, // afii10101
	{0xa456f11e, 0x09e8}, // twobengali
	{0xa457c062, 0x30ad}, // kikatakana
	{0xa45b3183, 0x0aeb}, // fivegujarati
	{0xa45b7f21, 0x010e}, // Dcaron
	{0xa4627c0f, 0x0204}, // Edblgrave
	{0xa478f921, 0x09e0}, // rrvocalicbengali
	{0xa4863185, 0x30f4}, // vukatakana
	{0xa4922e7c, 0x0341}, // acutetonecmb
	{0xa4aa1092, 0x05b1}, // hatafsegol30
	{0xa4aa8935, 0xff58}, // xmonospace
	{0xa4af8f73, 0x2282}, // subset
	{0xa4decb10, 0x0a72}, // irigurmukhi
	{0xa4ebd5d8, 0x05e7}, // qofhiriqhebrew
	{0xa50a3a99, 0x0630}, // afii57424
	{0xa50cf621, 0xff7e}, // sekatakanahalfwidth
	{0xa512e58d, 0xff83}, // tekatakanahalfwidth
	{0xa517b724, 0xfee6}, // noonfinalarabic
	{0xa52168e0, 0x2325}, // option
	{0xa54253fb, 0x05e7}, // qof
	{0xa545c2a6, 0xfb2a}, // afii57694
	{0xa553cf3e, 0x003b}, // semicolon
	{0xa56dfce7, 0x0e03}, // khokhuatthai
	{0xa58382dd, 0x0137}, // kcedilla
	{0xa5b93826, 0x2555}, // SF220000
	{0xa5ecbdaa, 0x03b5}, // epsilon
	{0xa6056425, 0x05b8}, // qamatswidehebrew
	{0xa60745ee, 0x2250}, // approaches
	{0xa6281f81, 0x0142}, // lslash
	{0xa62afc92, 0x0534}, // Daarmenian
	{0xa63a8cce, 0xf7f4}, // Ocircumflexsmall
	{0xa6454b66, 0x2498}, // seventeenperiod
	{0xa64723d6, 0x30af}, // kukatakana
	{0xa6522894, 0x24da}, // kcircle
	{0xa652cff2, 0x24a8}, // mparen
	{0xa65ca284, 0x314d}, // phieuphkorean
	{0xa65ea7b9, 0x00ab}, // guillemotleft
	{0xa6644796, 0xfc94}, // yehnoonfinalarabic
	{0xa66b3ab3, 0x00b8}, // cedilla
	{0xa675e0d6, 0x0156}, // Rcommaaccent
	{0xa683217b, 0x0ab7}, // ssagujarati
	{0xa6a017fd, 0x0415}, // Iecyrillic
	{0xa6ae34a8, 0x3028}, // eighthangzhou
	{0xa6b19efc, 0x0591}, // etnahtafoukhhebrew
	{0xa6b46028, 0xff72}, // ikatakanahalfwidth
	{0xa6bd2b95, 0x1e6b}, // tdotaccent
	{0xa6c10839, 0x0431}, // becyrillic
	{0xa6e68e9f, 0x0966}, // zerodeva
	{0xa6f16c03, 0x03c2}, // sigma1
	{0xa6f2df0d, 0x0a32}, // lagurmukhi
	{0xa6f3cb6a, 0x21d3}, // arrowdblbt
	{0xa716a470, 0x2464}, // fivecircle
	{0xa71dfe13, 0x1ea2}, // Ahookabove
	{0xa73026ce, 0x007a}, // z
	{0xa73199c0, 0x05bb}, // qubutshebrew
	{0xa731e944, 0x0411}, // afii10018
	{0xa7320cb3, 0x3139}, // rieulkorean
	{0xa74014fc, 0x003f}, // question
	{0xa745be27, 0x2308}, // ceilingleftbig
	{0xa74cd67a, 0x025f}, // jdotlessstroke
	{0xa756caf5, 0x04d2}, // Adieresiscyrillic
	{0xa7769b7a, 0x22c1}, // logicalortext
	{0xa7895d88, 0x252c}, // SF060000
	{0xa796d5bb, 0x0267}, // henghook
	{0xa7bff3d5, 0x032c}, // caronbelowcmb
	{0xa7dcd836, 0xfb68}, // ttehinitialarabic
	{0xa7e01c26, 0x01a7}, // Tonetwo
	{0xa7e7d702, 0x05b8}, // qamats1a
	{0xa7edca33, 0x25b7}, // whiterightpointingtriangle
	{0xa7fb6ee6, 0x226f}, // notgreater
	{0xa7fe97d8, 0x3166}, // nieuntikeutkorean
	{0xa815fa8a, 0x309e}, // voicediterationhiragana
	{0xa819fe3e, 0x315e}, // wekorean
	{0xa81ee743, 0x0542}, // Ghadarmenian
	{0xa826b713, 0xfeb6}, // sheenfinalarabic
	{0xa842618b, 0x025b}, // eopen
	{0xa84fdde5, 0x0340}, // gravetonecmb
	{0xa851c76f, 0x05a5}, // merkhahebrew
	{0xa8547bdd, 0x0455}, // afii10102
	{0xa8665e8d, 0x05b1}, // afii57801
	{0xa877e561, 0x2642}, // male
	{0xa87899cd, 0xf6e6}, // hyphensuperior
	{0xa8982f3b, 0x09b2}, // labengali
	{0xa8a24959, 0x00d0}, // Eth
	{0xa8b6c7f5, 0x0a2e}, // magurmukhi
	{0xa8b7f35e, 0x06c1}, // haaltonearabic
	{0xa8bb13d5, 0x0a70}, // tippigurmukhi
	{0xa8ccc65c, 0x3316}, // kiromeetorusquare
	{0xa8db2b93, 0x017e}, // zcaron
	{0xa93204a5, 0x03a6}, // Phi
	{0xa9350b3f, 0x20a1}, // colonmonetary
	{0xa93a2a4f, 0x1e47}, // ndotbelow
	{0xa94e0303, 0x33ab}, // mpasquare
	{0xa94eeaee, 0x01aa}, // eshreversedloop
	{0xa95e2711, 0x1e21}, // gmacron
	{0xa96cef91, 0x3169}, // rieulkiyeoksioskorean
	{0xa981562d, 0x0668}, // eightarabic
	{0xa98e771c, 0x0634}, // sheenarabic
	{0xa9985803, 0x30dc}, // bokatakana
	{0xa99c94c2, 0x01be}, // glottalinvertedstroke
	{0xa9a0932f, 0x2077}, // sevensuperior
	{0xa9af18f2, 0x30b4}, // gokatakana
	{0xa9c0b182, 0x2264}, // lessequal
	{0xa9dc390a, 0x05da}, // finalkafqamatshebrew
	{0xa9e88297, 0x2053}, // a58
	{0xaa13efde, 0x007d}, // braceright
	{0xaa245bb8, 0x0028}, // parenleftBig
	{0xaa4e278b, 0x0967}, // onedeva
	{0xaa51d75f, 0x2476}, // threeparen
	{0xaa69d0f1, 0x256c}, // SF440000
	{0xaa863ce3, 0x0629}, // tehmarbutaarabic
	{0xaa8c5eeb, 0x03e5}, // feicoptic
	{0xaa96b9dc, 0x0665}, // fivearabic
	{0xaaabcf5c, 0x04b9}, // cheverticalstrokecyrillic
	{0xaab6b9a5, 0x0211}, // rdblgrave
	{0xaabfed05, 0xff85}, // nakatakanahalfwidth
	{0xaac3a76a, 0x0559}, // ringhalfleftarmenian
	{0xaac3adf0, 0x3121}, // oubopomofo
	{0xaacaffc4, 0x2162}, // Threeroman
	{0xaad54f7c, 0x3002}, // ideographicperiod
	{0xaaf6eb21, 0x01ce}, // acaron
	{0xab0b499a, 0x099c}, // jabengali
	{0xab126f69, 0x5344}, // twentyhangzhou
	{0xab1f1bb7, 0x05df}, // afii57679
	{0xab24577f, 0x0667}, // sevenarabic
	{0xab2b4200, 0x2105}, // careof
	{0xab3b4b27, 0x24e2}, // scircle
	{0xab52e61f, 0x05d3}, // dalethatafpatah
	{0xab808d1e, 0x2087}, // seveninferior
	{0xab8a6656, 0x01d0}, // icaron
	{0xaba55a59, 0xfb32}, // gimeldageshhebrew
	{0xabb7eb8f, 0x2081}, // oneinferior
	{0xabd373e8, 0x055e}, // questionarmenian
	{0xac05d874, 0x1e84}, // Wdieresis
	{0xac259f58, 0x0448}, // afii10090
	{0xac2c323e, 0x0191}, // Fhook
	{0xac32a034, 0x316f}, // mieumsioskorean
	{0xac483cb3, 0x338c}, // mufsquare
	{0xac4f1094, 0x059c}, // gereshaccenthebrew
	{0xac50a082, 0x33a3}, // mmcubedsquare
	{0xac5faca7, 0x230b}, // floorrightbig
	{0xac67aca2, 0x045b}, // tshecyrillic
	{0xacac7818, 0x219d}, // a59
	{0xacb92bab, 0x0691}, // rreharabic
	{0xacd00f05, 0x21c6}, // arrowleftoverright
	{0xacd11e18, 0x2010}, // hyphentwo
	{0xace7d07a, 0x0921}, // ddadeva
	{0xacfcbdb9, 0x3042}, // ahiragana
	{0xad01f787, 0xff1d}, // equalmonospace
	{0xad1b58f9, 0x0595}, // zaqefgadolhebrew
	{0xad29738f, 0x05f4}, // gershayimhebrew
	{0xad33f4b3, 0x04c1}, // Zhebrevecyrillic
	{0xad37f8e0, 0x0a24}, // tagurmukhi
	{0xad38bc31, 0x2194}, // arrowboth
	{0xad45a65c, 0xf6cc}, // DieresisAcute
	{0xad5fe438, 0x0924}, // tadeva
	{0xad781e89, 0x200c}, // afii61664
	{0xad8ff38a, 0x0a21}, // ddagurmukhi
	{0xadb1b19e, 0x0206}, // Einvertedbreve
	{0xadb53f6b, 0xfbaf}, // yehbarreefinalarabic
	{0xadbea3e4, 0x20aa}, // afii57636
	{0xadc3ff5e, 0x0649}, // afii57449
	{0xadf402a9, 0x0989}, // ubengali
	{0xadf4d422, 0x0625}, // alefhamzabelowarabic
	{0xae04e5ad, 0x0a15}, // kagurmukhi
	{0xae06976c, 0xf6d5}, // cyrflex
	{0xae1f7b0a, 0x2076}, // sixsuperior
	{0xae23dd7b, 0x1eaf}, // abreveacute
	{0xae30147f, 0x0908}, // iideva
	{0xae346d0d, 0x05e8}, // reshsheva
	{0xae56317c, 0x24ae}, // sparen
	{0xae6f7e74, 0x031b}, // horncmb
	{0xae7c975f, 0x249a}, // nineteenperiod
	{0xae8f4e4c, 0x0406}, // afii10055
	{0xaeac4f55, 0xf7e9}, // Eacutesmall
	{0xaeb06274, 0x0423}, // Ucyrillic
	{0xaec173e8, 0x03b8}, // theta
	{0xaee16fb6, 0xfe4f}, // underscorewavy
	{0xaef4b475, 0x1edf}, // ohornhookabove
	{0xaef8393d, 0x0664}, // fourarabic
	{0xaf01f370, 0x278e}, // fivecircleinversesansserif
	{0xaf0ebb84, 0xf6d6}, // dblgrave
	{0xaf2073fd, 0x2103}, // centigrade
	{0xaf3552ce, 0x3072}, // hihiragana
	{0xaf36c6b1, 0x30cd}, // nekatakana
	{0xaf499180, 0x2309}, // ceilingrightBig
	{0xaf4df0df, 0xfc0e}, // tehmeemisolatedarabic
	{0xaf5710c9, 0x00c0}, // Agrave
	{0xaf5b123d, 0x032d}, // circumflexbelowcmb
	{0xaf5ce08f, 0x00e0}, // agrave
	{0xaf788850, 0x00fc}, // udieresis
	{0xaf7abcb1, 0x05b9}, // holam
	{0xaf8a8524, 0x0455}, // dzecyrillic
	{0xafa14924, 0x33d3}, // lxsquare
	{0xafaa365c, 0x30be}, // zokatakana
	{0xafb28009, 0x03e2}, // Sheicoptic
	{0xafb8e89a, 0x0a68}, // twogurmukhi
	{0xafbbfcac, 0xff71}, // akatakanahalfwidth
	{0xafbd0738, 0x305d}, // sohiragana
	{0xafc9b657, 0x0e17}, // thothahanthai
	{0xaff892ca, 0x05d3}, // dalettserehebrew
	{0xb000150a, 0xfe96}, // tehfinalarabic
	{0xb01f8020, 0x0666}, // sixarabic
	{0xb026a3ef, 0x30b3}, // kokatakana
	{0xb0309f24, 0xfee3}, // meeminitialarabic
	{0xb032be97, 0x047b}, // omegaroundcyrillic
	{0xb033a837, 0x0079}, // y
	{0xb03640f2, 0x2252}, // approxequalorimage
	{0xb0522c01, 0x278f}, // sixcircleinversesansserif
	{0xb0791aaf, 0x09f3}, // rupeesignbengali
	{0xb0a20aff, 0x22db}, // greaterequalorless
	{0xb0c33454, 0x028a}, // upsilonlatin
	{0xb0efaba6, 0x0a97}, // gagujarati
	{0xb107bdf6, 0x304b}, // kahiragana
	{0xb1240d86, 0x0474}, // afii10148
	{0xb145d406, 0x327b}, // hieuhacirclekorean
	{0xb169c9ac, 0x1eed}, // uhornhookabove
	{0xb1722e49, 0x3006}, // ideographicclose
	{0xb199f9f3, 0x314a}, // chieuchkorean
	{0xb1a83745, 0x246e}, // fifteencircle
	{0xb1a9eaa4, 0x1eb7}, // abrevedotbelow
	{0xb1b08c26, 0x2020}, // dagger
	{0xb1b2e578, 0xff13}, // threemonospace
	{0xb1c3eac2, 0x1e8c}, // Xdieresis
	{0xb1e18633, 0x0e2d}, // oangthai
	{0xb1eda93c, 0x09e2}, // lvocalicvowelsignbengali
	{0xb1fa6226, 0x0a9a}, // cagujarati
	{0xb225a8bc, 0x0947}, // evowelsigndeva
	{0xb243894e, 0x21d1}, // arrowdblup
	{0xb25639c1, 0x2534}, // SF070000
	{0xb2566e08, 0x064d}, // afii57453
	{0xb256786e, 0x03f2}, // sigmalunatesymbolgreek
	{0xb25b34dc, 0x0e1a}, // bobaimaithai
	{0xb26943db, 0x0420}, // afii10034
	{0xb27e91f4, 0x01bb}, // twostroke
	{0xb290d64a, 0xfe88}, // alefhamzabelowfinalarabic
	{0xb2bbd0d4, 0xfeb0}, // zainfinalarabic
	{0xb2cc02c2, 0x1e38}, // Ldotbelowmacron
	{0xb2d79f3e, 0x00f9}, // ugrave
	{0xb2d7f27b, 0x0652}, // afii57458
	{0xb2d95c63, 0x2518}, // SF040000
	{0xb2f3aff0, 0x0946}, // eshortvowelsigndeva
	{0xb311c284, 0x05e7}, // qoftsere
	{0xb3178333, 0x00da}, // Uacute
	{0xb321fe9c, 0x21e3}, // arrowdashdown
	{0xb327a481, 0x2290}, // a61
	{0xb32daf91, 0xf6c6}, // afii10192
	{0xb32e268f, 0xf6fe}, // Tildesmall
	{0xb3329e90, 0x313e}, // rieulthieuthkorean
	{0xb33c41bc, 0x018c}, // dtopbar
	{0xb340e2fe, 0x04a5}, // enghecyrillic
	{0xb341da2f, 0x1e36}, // Ldotbelow
	{0xb345c512, 0xfea4}, // hahmedialarabic
	{0xb369c9bd, 0x2423}, // blank
	{0xb36f3f4e, 0x054d}, // Seharmenian
	{0xb3880287, 0x05d6}, // zayinhebrew
	{0xb38b59bc, 0x0ac0}, // iivowelsigngujarati
	{0xb3a3592e, 0x21e5}, // arrowtabright
	{0xb3dbcf55, 0x323d}, // ideographicenterpriseparen
	{0xb3e6b497, 0x03c9}, // omega
	{0xb3ed41be, 0x328f}, // ideographearthcircle
	{0xb40169ac, 0x05b0}, // sheva22
	{0xb405e3e3, 0x0439}, // afii10075
	{0xb40aff3d, 0xf730}, // zerooldstyle
	{0xb41baecc, 0x2487}, // twentyparen
	{0xb427632e, 0x05b3}, // hatafqamats1b
	{0xb4344c30, 0x090f}, // edeva
	{0xb43bb55a, 0x3263}, // rieulcirclekorean
	{0xb448d464, 0x0391}, // Alpha
	{0xb45a5763, 0x201b}, // quoteleftreversed
	{0xb45ef9b7, 0x03c0}, // pi
	{0xb47a6410, 0x230a}, // floorleftBigg
	{0xb497903a, 0x01dd}, // eturned
	{0xb4a55071, 0x1e7d}, // vtilde
	{0xb4a6b289, 0x2235}, // because
	{0xb4a7f99d, 0x0954}, // acutedeva
	{0xb4a9d27d, 0x05d2}, // afii57666
	{0xb4c0dc86, 0x0e48}, // maiekthai
	{0xb4c2484c, 0x05dc}, // lamed
	{0xb4c667bc, 0xff61}, // periodhalfwidth
	{0xb4c72b2a, 0x0215}, // udblgrave
	{0xb4e21f31, 0x2198}, // arrowdownright
	{0xb4fef2ed, 0x05d3}, // dalettsere
	{0xb500deca, 0x1e41}, // mdotaccent
	{0xb510d684, 0x1e80}, // Wgrave
	{0xb526b685, 0x22bf}, // righttriangle
	{0xb52e7c1d, 0x091e}, // nyadeva
	{0xb53ca7e2, 0x326c}, // phieuphcirclekorean
	{0xb5866d85, 0x005c}, // backslashbig
	{0xb58e59d7, 0x24ad}, // rparen
	{0xb5af274f, 0x064a}, // afii57450
	{0xb5b94593, 0x010d}, // ccaron
	{0xb5c458a3, 0x05b8}, // qamatsqatanwidehebrew
	{0xb5f24e31, 0xff5a}, // zmonospace
	{0xb600bed1, 0x3024}, // fourhangzhou
	{0xb6052cdb, 0x010b}, // cdotaccent
	{0xb61c54b4, 0x24e4}, // ucircle
	{0xb64312f2, 0x00a2}, // cent
	{0xb6443d26, 0x0622}, // afii57410
	{0xb647ed9e, 0x01d7}, // Udieresisacute
	{0xb652184e, 0x0e4b}, // maichattawathai
	{0xb6588f1c, 0x247c}, // nineparen
	{0xb66bf9b9, 0x33dc}, // svsquare
	{0xb673fbb5, 0x038e}, // Upsilontonos
	{0xb67e35c8, 0x0121}, // gdotaccent
	{0xb6951f83, 0x2210}, // coproductdisplay
	{0xb6bb2a6b, 0x0116}, // Edot
	{0xb6d45c54, 0xff7a}, // kokatakanahalfwidth
	{0xb6f322b3, 0x05e5}, // afii57685
	{0xb6f9554e, 0x3205}, // pieupparenkorean
	{0xb6f9c67c, 0x0464}, // Eiotifiedcyrillic
	{0xb6fea9e7, 0xff3f}, // underscoremonospace
	{0xb70f3f60, 0xff4e}, // nmonospace
	{0xb711b601, 0x039d}, // Nu
	{0xb7124c93, 0xf770}, // Psmall
	{0xb719922a, 0x067e}, // afii57506
	{0xb71d84e2, 0x3167}, // nieunsioskorean
	{0xb726c42e, 0xfb49}, // shindageshhebrew
	{0xb72e5846, 0x1edb}, // ohornacute
	{0xb73606f5, 0x0463}, // afii10194
	{0xb7400632, 0x33aa}, // kpasquare
	{0xb747ebc8, 0x2310}, // revlogicalnot
	{0xb755a24e, 0x310e}, // kbopomofo
	{0xb7934eea, 0x01c5}, // Dzcaron
	{0xb7affc1f, 0x0418}, // afii10026
	{0xb7b78fdd, 0x2463}, // fourcircle
	{0xb7bd89d3, 0x0491}, // gheupturncyrillic
	{0xb7d20c6c, 0x215c}, // threeeighths
	{0xb7e9bb2b, 0x30fe}, // voicediterationkatakana
	{0xb80991a9, 0x04d6}, // Iebrevecyrillic
	{0xb81d8e8c, 0xfb31}, // betdagesh
	{0xb855cda8, 0x3397}, // dlsquare
	{0xb85a6427, 0x0650}, // kasraarabic
	{0xb8632720, 0x0e07}, // ngonguthai
	{0xb879d78f, 0xfe32}, // endashvertical
	{0xb894f4d6, 0x01a0}, // Ohorn
	{0xb8972176, 0x04bb}, // shhacyrillic
	{0xb89948ac, 0x0a93}, // ogujarati
	{0xb8be7e03, 0x2039}, // guilsinglleft
	{0xb8fa96e6, 0x0110}, // Dslash
	{0xb8ff412c, 0xfef8}, // lamalefhamzaabovefinalarabic
	{0xb902d285, 0x053a}, // Zhearmenian
	{0xb90dcf8c, 0xff6a}, // esmallkatakanahalfwidth
	{0xb910864a, 0x2032}, // minute
	{0xb921c241, 0xfeca}, // ainfinalarabic
	{0xb92ccc5d, 0x2469}, // tencircle
	{0xb9305b2b, 0x3174}, // pieupsioskiyeokkorean
	{0xb96268cb, 0x00de}, // Thorn
	{0xb9808b18, 0x09c1}, // uvowelsignbengali
	{0xb9927e88, 0x3382}, // muasquare
	{0xb99f8f9e, 0xfeeb}, // hehinitialarabic
	{0xb9b2e314, 0x0498}, // Zedescendercyrillic
	{0xb9b4563d, 0x24b3}, // xparen
	{0xb9c5eece, 0x05bc}, // dageshhebrew
	{0xb9d8b5d9, 0xfba4}, // hehhamzaaboveisolatedarabic
	{0xb9dbedd2, 0x317b}, // siosnieunkorean
	{0xb9e5ea71, 0x314c}, // thieuthkorean
	{0xb9e8b13e, 0x25a0}, // filledbox
	{0xb9ebf396, 0xf764}, // Dsmall
	{0xb9f42560, 0x0964}, // danda
	{0xb9f5b462, 0x0990}, // aibengali
	{0xba1bcecd, 0x0176}, // Ycircumflex
	{0xba1f80d6, 0x21e1}, // arrowdashup
	{0xba21ad27, 0xfb3a}, // finalkafdagesh
	{0xba3aaf1e, 0x27e9}, // angbracketrightBig
	{0xba4eb5f9, 0x0055}, // U
	{0xba544632, 0x05db}, // afii57675
	{0xba5871eb, 0x033e}, // tildeverticalcmb
	{0xba60a3b8, 0x0902}, // anusvaradeva
	{0xba7e1049, 0x01d9}, // Udieresiscaron
	{0xba8d69d2, 0x232a}, // angleright
	{0xbaa24d97, 0x05e0}, // afii57680
	{0xbaa7aa4c, 0xfedc}, // kafmedialarabic
	{0xbab8d5ec, 0xf771}, // Qsmall
	{0xbabbf0c0, 0x05e8}, // reshhiriqhebrew
	{0xbac7de75, 0x1ed3}, // ocircumflexgrave
	{0xbad44ddc, 0x004f}, // O
	{0xbad7d685, 0x04b8}, // Cheverticalstrokecyrillic
	{0xbad97612, 0x03af}, // iotatonos
	{0xbadf80e2, 0x06f5}, // fivepersian
	{0xbae37657, 0x0621}, // hamzalowkasratanarabic
	{0xbafeb301, 0x0422}, // Tecyrillic
	{0xbb28e2ea, 0x1eb2}, // Abrevehookabove
	{0xbb30f37e, 0x0218}, // Scommaaccent
	{0xbb6353b2, 0x0332}, // lowlinecmb
	{0xbb66e953, 0x22c8}, // a49
	{0xbb72d76e, 0x0139}, // Lacute
	{0xbb89235d, 0xf6e0}, // centsuperior
	{0xbb906e01, 0x3266}, // sioscirclekorean
	{0xbb970588, 0x266f}, // musicsharpsign
	{0xbba252f7, 0x0106}, // Cacute
	{0xbba2c6f4, 0x27e9}, // angbracketrightbigg
	{0xbbae7b40, 0x3172}, // pieupkiyeokkorean
	{0xbbb9d0f4, 0x0934}, // llladeva
	{0xbbd8677b, 0x330d}, // karoriisquare
	{0xbbdbcb55, 0x0a6f}, // ninegurmukhi
	{0xbbdc86be, 0xfb2b}, // shinsindot
	{0xbc041d93, 0x05da}, // finalkafqamats
	{0xbc0d2781, 0x0a1f}, // ttagurmukhi
	{0xbc1b1166, 0x2558}, // SF500000
	{0xbc280da2, 0x1e79}, // utildeacute
	{0xbc3510eb, 0x317f}, // pansioskorean
	{0xbc45cf9a, 0x02a4}, // dezh
	{0xbc75336c, 0x21a9}, // arrowhookright
	{0xbc78e14f, 0x24b0}, // uparen
	{0xbcd2c61b, 0x064c}, // dammatanarabic
	{0xbcf16b16, 0x0621}, // hamzadammatanarabic
	{0xbd0f6f71, 0x004c}, // L
	{0xbd1a9441, 0x3235}, // ideographicspecialparen
	{0xbd1abdb6, 0x0184}, // Tonesix
	{0xbd2d2e5f, 0x054e}, // Vewarmenian
	{0xbd30ce0c, 0x2641}, // earth
	{0xbd4d0860, 0x320a}, // khieukhparenkorean
	{0xbd54bd2d, 0x3208}, // cieucparenkorean
	{0xbd569183, 0x0182}, // Btopbar
	{0xbd5dbcc1, 0x0151}, // odblacute
	{0xbd5ee257, 0x01b7}, // Ezh
	{0xbd85b57b, 0x21a8}, // arrowupdownbase
	{0xbd8f8d24, 0x064b}, // fathatanarabic
	{0xbd906fab, 0x041f}, // Pecyrillic
	{0xbd98b80f, 0x25c8}, // whitediamondcontainingblacksmalldiamond
	{0xbdc1280a, 0x24be}, // Icircle
	{0xbdd372da, 0x318b}, // yuyekorean
	{0xbde8281d, 0x3276}, // cieucacirclekorean
	{0xbe1bc796, 0xfb7c}, // tchehmeeminitialarabic
	{0xbe3e45cf, 0xfb58}, // pehinitialarabic
	{0xbe663ca6, 0x2470}, // seventeencircle
	{0xbe7a58ae, 0x3203}, // rieulparenkorean
	{0xbea937fd, 0x25bc}, // blackdownpointingtriangle
	{0xbec6916e, 0x011c}, // Gcircumflex
	{0xbedd6640, 0x00d9}, // Ugrave
	{0xbee1ad99, 0xed79}, // vextenddouble
	{0xbf12496a, 0x0985}, // abengali
	{0xbf156070, 0x04e2}, // Imacroncyrillic
	{0xbf26dc61, 0x041d}, // Encyrillic
	{0xbf2dca30, 0x3082}, // mohiragana
	{0xbf3cff90, 0x0425}, // afii10039
	{0xbf5cef43, 0x01ee}, // Ezhcaron
	{0xbf673175, 0x01fd}, // aeacute
	{0xbf87a284, 0x00bf}, // questiondown
	{0xbf897387, 0x2273}, // greaterorequivalent
	{0xbf8f3598, 0x25a6}, // squareorthogonalcrosshatchfill
	{0xbf934ed3, 0x04e0}, // Dzeabkhasiancyrillic
	{0xbf97194e, 0x010a}, // Cdot
	{0xbf9bc7bd, 0xfb59}, // pehmedialarabic
	{0xbf9c1926, 0x0627}, // alefarabic
	{0xbfc69ab7, 0xff18}, // eightmonospace
	{0xbfd3ede4, 0x0396}, // Zeta
	{0xbfe44580, 0x05d3}, // dalethiriq
	{0xbffa52a3, 0x33c7}, // cosquare
	{0xc006a810, 0x062a}, // teharabic
	{0xc008508a, 0xff1c}, // lessmonospace
	{0xc00a3b07, 0x24bc}, // Gcircle
	{0xc0126352, 0x0661}, // onearabic
	{0xc03e102c, 0x3012}, // postalmark
	{0xc059a094, 0x0626}, // yehhamzaabovearabic
	{0xc0668ba8, 0x202e}, // afii61575
	{0xc07e7e42, 0x24a6}, // kparen
	{0xc084bd84, 0x1e29}, // hcedilla
	{0xc092fb91, 0x1e95}, // zlinebelow
	{0xc09823db, 0x016f}, // uring
	{0xc09889a1, 0x1e11}, // dcedilla
	{0xc09e394d, 0x2211}, // summationtext
	{0xc0a2bc69, 0x2564}, // SF470000
	{0xc0a93f4f, 0x0e2c}, // lochulathai
	{0xc0bd9f90, 0x094a}, // oshortvowelsigndeva
	{0xc0c043bd, 0x3008}, // anglebracketleft
	{0xc0c1496c, 0x029e}, // kturned
	{0xc0cbe66a, 0x33b7}, // mvsquare
	{0xc0d444a4, 0x3220}, // oneideographicparen
	{0xc0dcb90f, 0x062e}, // khaharabic
	{0xc0efe98c, 0x2191}, // arrowup
	{0xc0f7b81d, 0xf888}, // saraueeleftthai
	{0xc0fb3832, 0x25ab}, // H18551
	{0xc131664b, 0x30e4}, // yakatakana
	{0xc1425417, 0x317a}, // sioskiyeokkorean
	{0xc1641f79, 0xfe4d}, // lowlinedashed
	{0xc1835ca2, 0x02d3}, // ringhalfleftcentered
	{0xc191ae65, 0x0470}, // Psicyrillic
	{0xc1aa3399, 0x2a06}, // unionsqtext
	{0xc1c1cd87, 0xff2e}, // Nmonospace
	{0xc1d325d6, 0x05d8}, // tet
	{0xc1e02cdd, 0x0594}, // zaqefqatanhebrew
	{0xc1e3fd2d, 0x05d5}, // vav
	{0xc1e4c743, 0x3057}, // sihiragana
	{0xc1e7aebd, 0xf779}, // Ysmall
	{0xc1ec1451, 0x03c3}, // sigma
	{0xc1f9ede7, 0x05a3}, // munahhebrew
	{0xc2089a04, 0x05b7}, // patah1d
	{0xc21675c5, 0x0317}, // acutebelowcmb
	{0xc21e92b6, 0x2265}, // greaterequal
	{0xc2352e98, 0x0554}, // Keharmenian
	{0xc25217c2, 0xfeff}, // zerowidthjoiner
	{0xc25a58a0, 0x0a28}, // nagurmukhi
	{0xc25a7d39, 0x2663}, // club
	{0xc2658d7c, 0x3395}, // mulsquare
	{0xc26d6fce, 0x20a9}, // won
	{0xc283f2c3, 0x03bc}, // mugreek
	{0xc29c269a, 0x326d}, // hieuhcirclekorean
	{0xc2c391b3, 0x25b2}, // triagup
	{0xc2d0d165, 0x0137}, // kcommaaccent
	{0xc2db7e0f, 0x04a0}, // Kabashkircyrillic
	{0xc327f9ff, 0xf8eb}, // parenlefttp
	{0xc32ea721, 0x061f}, // afii57407
	{0xc347a5d5, 0x22c1}, // logicalordisplay
	{0xc36634d0, 0x09f9}, // sixteencurrencydenominatorbengali
	{0xc3722d19, 0x23ad}, // bracerightbt
	{0xc3751bae, 0x25ab}, // whitesmallsquare
	{0xc375a046, 0x05b0}, // shevahebrew
	{0xc391e9a8, 0xfe50}, // commasmall
	{0xc3a63833, 0x0621}, // hamzasukunarabic
	{0xc3c94bba, 0x306a}, // nahiragana
	{0xc3d9dd70, 0xfe43}, // whitecornerbracketleftvertical
	{0xc3df4586, 0x0903}, // visargadeva
	{0xc3eb9abd, 0x02c6}, // circumflex
	{0xc3edb597, 0x24bb}, // Fcircle
	{0xc3f1c1ff, 0x00e3}, // atilde
	{0xc3f76044, 0xff7c}, // sikatakanahalfwidth
	{0xc3f8f5bb, 0x04e3}, // imacroncyrillic
	{0xc3fbb2ce, 0x025e}, // eopenreversedclosed
	{0xc4004762, 0x328a}, // ideographmooncircle
	{0xc4289b41, 0x25a5}, // squareverticalfill
	{0xc43e9d0c, 0x1e26}, // Hdieresis
	{0xc4494fb1, 0x098b}, // rvocalicbengali
	{0xc459127a, 0x2174}, // fiveroman
	{0xc460226d, 0xfb34}, // hedagesh
	{0xc4607843, 0x25ca}, // lozenge
	{0xc4991f6f, 0x03d0}, // betasymbolgreek
	{0xc49bedba, 0x1e22}, // Hdotaccent
	{0xc4bd396f, 0xff98}, // rikatakanahalfwidth
	{0xc4cf0400, 0x0ac3}, // rvocalicvowelsigngujarati
	{0xc4d48fb7, 0x0a06}, // aagurmukhi
	{0xc510ccfb, 0x0327}, // cedillacmb
	{0xc51e03d1, 0x0e2a}, // sosuathai
	{0xc5233a99, 0xff41}, // amonospace
	{0xc52d9608, 0x0409}, // afii10058
	{0xc541abe3, 0x0409}, // Ljecyrillic
	{0xc560e83e, 0x1e4c}, // Otildeacute
	{0xc575fbd8, 0x3303}, // aarusquare
	{0xc577ce75, 0xff5d}, // bracerightmonospace
	{0xc579feb1, 0x03bf}, // omicron
	{0xc5904240, 0x1ee9}, // uhornacute
	{0xc594a84a, 0x0a85}, // agujarati
	{0xc5a57fbd, 0x0178}, // Ydieresis
	{0xc5b52809, 0x25ac}, // filledrect
	{0xc5c558c0, 0x2593}, // dkshade
	{0xc5df0b88, 0x0aae}, // magujarati
	{0xc5e2935b, 0x30bf}, // takatakana
	{0xc616ce34, 0x1e83}, // wacute
	{0xc618c356, 0x005f}, // underscore
	{0xc6231f67, 0x3117}, // zbopomofo
	{0xc62d494f, 0x042b}, // afii10045
	{0xc65b9029, 0x0012}, // controlDC2
	{0xc65e4473, 0x0300}, // gravecomb
	{0xc6649ea6, 0x263b}, // blacksmilingface
	{0xc6730a47, 0x3230}, // ideographicsunparen
	{0xc6a18663, 0x2173}, // fourroman
	{0xc6a33bcf, 0xf8f9}, // bracketrighttp
	{0xc6bb9cde, 0x03c1}, // rho
	{0xc6c2ca67, 0x0537}, // Eharmenian
	{0xc6db1db1, 0x0a94}, // augujarati
	{0xc6ddc5a6, 0x0334}, // tildeoverlaycmb
	{0xc6e9fa39, 0x064a}, // yeharabic
	{0xc6fbfdac, 0x025a}, // schwahook
	{0xc6fedf58, 0xf731}, // oneoldstyle
	{0xc703d8e7, 0x33b8}, // kvsquare
	{0xc7199c26, 0xf763}, // Csmall
	{0xc71e49b3, 0xf8fa}, // bracketrightex
	{0xc74688aa, 0x24dc}, // mcircle
	{0xc74a8cc7, 0x05b5}, // tserequarterhebrew
	{0xc74f4c99, 0x0298}, // bilabialclick
	{0xc7515b37, 0x0132}, // IJ
	{0xc755c80f, 0xff67}, // asmallkatakanahalfwidth
	{0xc75bc6a7, 0x01e8}, // Kcaron
	{0xc7618f62, 0x05b3}, // hatafqamats28
	{0xc76816e4, 0x0566}, // zaarmenian
	{0xc76c439e, 0x0115}, // ebreve
	{0xc76ec8b2, 0x0e1c}, // phophungthai
	{0xc77c2828, 0x2206}, // Delta
	{0xc7855795, 0xf894}, // maichattawalowrightthai
	{0xc786ef31, 0x095b}, // zadeva
	{0xc78887c7, 0x1e08}, // Ccedillaacute
	{0xc78fe323, 0x0698}, // afii57508
	{0xc798936b, 0xfc73}, // tehnoonfinalarabic
	{0xc7b3573a, 0x1e2f}, // idieresisacute
	{0xc7daee30, 0x09b7}, // ssabengali
	{0xc7e20869, 0x0aee}, // eightgujarati
	{0xc7ed01af, 0x33a5}, // mcubedsquare
	{0xc7f07bcf, 0x248f}, // eightperiod
	{0xc7f0b4c8, 0x0406}, // Icyrillic
	{0xc7fb5fe5, 0x05b2}, // afii57800
	{0xc808d8ed, 0xf8f5}, // integralex
	{0xc812d1b7, 0x041b}, // afii10029
	{0xc82c0a32, 0x026f}, // mturned
	{0xc834804d, 0x2475}, // twoparen
	{0xc8488aa4, 0x05b0}, // shevawidehebrew
	{0xc84db89c, 0x2013}, // endash
	{0xc85fce80, 0x0901}, // candrabindudeva
	{0xc86d3a57, 0xff31}, // Qmonospace
	{0xc875e6fd, 0x040c}, // afii10061
	{0xc88f7d57, 0x05b3}, // hatafqamatsnarrowhebrew
	{0xc89d4f1f, 0x0006}, // controlACK
	{0xc8ae6fb2, 0x0e26}, // luthai
	{0xc8b50d48, 0x21d4}, // dblarrowleft
	{0xc8c2c42c, 0x0162}, // Tcedilla
	{0xc8c687a4, 0xfb4b}, // vavholamhebrew
	{0xc8cf912e, 0xfe59}, // parenleftsmall
	{0xc8fcce82, 0x0424}, // Efcyrillic
	{0xc8fdfe4b, 0x0914}, // audeva
	{0xc9033cb3, 0x05e7}, // qofhatafpatahhebrew
	{0xc905dac2, 0xf7e0}, // Agravesmall
	{0xc9315b90, 0x0338}, // soliduslongoverlaycmb
	{0xc93304ec, 0x2176}, // sevenroman
	{0xc9366418, 0x1e7e}, // Vdotbelow
	{0xc948d9f2, 0x01fe}, // Oslashacute
	{0xc948dd49, 0x0417}, // Zecyrillic
	{0xc94ac55a, 0xfb4e}, // perafehebrew
	{0xc94ceb7c, 0x1eca}, // Idotbelow
	{0xc956ff7e, 0x1e7b}, // umacrondieresis
	{0xc997284d, 0x03d3}, // Upsilonacutehooksymbolgreek
	{0xc997d626, 0xfe3d}, // dblanglebracketleftvertical
	{0xc9d02325, 0x2283}, // propersuperset
	{0xc9d7e9b6, 0x064c}, // afii57452
	{0xc9d96803, 0x3278}, // khieukhacirclekorean
	{0xc9e8cf5f, 0xfebf}, // dadinitialarabic
	{0xc9ea8b89, 0x305b}, // sehiragana
	{0xc9eb1227, 0x321c}, // cieucuparenkorean
	{0xca07e9ca, 0x02dd}, // hungarumlaut
	{0xca0b3331, 0x1e5f}, // rlinebelow
	{0xca0f9f38, 0xf887}, // saraueleftthai
	{0xca2dd9fa, 0x2590}, // rtblock
	{0xca5ed753, 0x0269}, // iotalatin
	{0xca65e972, 0x0646}, // noonarabic
	{0xcab40374, 0x1e1f}, // fdotaccent
	{0xcabe62a6, 0x0e16}, // thothungthai
	{0xcac3bcb6, 0x33a8}, // moverssquaredsquare
	{0xcac4b1a9, 0x32a8}, // ideographicrightcircle
	{0xcad1f345, 0x01bd}, // tonefive
	{0xcae44cee, 0x0013}, // controlDC3
	{0xcaf66d1c, 0x33b4}, // pvsquare
	{0xcb1a38a0, 0x226b}, // muchgreater
	{0xcb1b6c03, 0x0a5e}, // fagurmukhi
	{0xcb1e3324, 0x042d}, // afii10047
	{0xcb267db3, 0x3221}, // twoideographicparen
	{0xcb281438, 0x33ae}, // radoverssquare
	{0xcb2edf2d, 0x220f}, // product
	{0xcb4fc444, 0x03f0}, // kappasymbolgreek
	{0xcb56efb7, 0x015c}, // Scircumflex
	{0xcb5b537a, 0x0412}, // afii10019
	{0xcb669b9c, 0x1ee2}, // Ohorndotbelow
	{0xcb72660a, 0xff1e}, // greatermonospace
	{0xcb75245d, 0x1e48}, // Nlinebelow
	{0xcb88e590, 0x013c}, // lcedilla
	{0xcb8bc7f1, 0x002b}, // plus
	{0xcbada403, 0x0429}, // Shchacyrillic
	{0xcc11d7b0, 0x000c}, // controlFF
	{0xcc261604, 0x04a2}, // Endescendercyrillic
	{0xcc2b60a5, 0x05e7}, // qofhiriq
	{0xcc3139de, 0x33d1}, // squareln
	{0xcc334bff, 0xf766}, // Fsmall
	{0xcc3accb9, 0x05d8}, // tethebrew
	{0xcc3b7b5c, 0x0649}, // alefmaksuraarabic
	{0xcc447b1d, 0x0296}, // glottalstopinverted
	{0xcc8c13e9, 0x25c0}, // blackleftpointingtriangle
	{0xcc993e5c, 0x025c}, // eopenreversed
	{0xcca8fd16, 0x09e3}, // llvocalicvowelsignbengali
	{0xccaa74e1, 0xfb7d}, // tchehmedialarabic
	{0xccac0ec6, 0x300b}, // dblanglebracketright
	{0xccc389ea, 0xfb30}, // alefdageshhebrew
	{0xccc85a27, 0x0a81}, // candrabindugujarati
	{0xcd07b41d, 0x05bc}, // dagesh
	{0xcd30953c, 0x203c}, // exclamdbl
	{0xcd37d58c, 0x24e3}, // tcircle
	{0xcd415c99, 0x0e55}, // fivethai
	{0xcd499038, 0x33a2}, // kmsquaredsquare
	{0xcd54eec2, 0x06f4}, // fourpersian
	{0xcd5fb77d, 0x0185}, // tonesix
	{0xcd64e087, 0x266b}, // musicalnotedbl
	{0xcd75d5eb, 0x2551}, // SF240000
	{0xcd7ce3d0, 0x24a7}, // lparen
	{0xcd85d846, 0x1eec}, // Uhornhookabove
	{0xcd9d27ad, 0x24c7}, // Rcircle
	{0xcd9f5a2c, 0x0028}, // parenleft
	{0xcda0c667, 0x2018}, // quoteleft
	{0xcdab3631, 0xf7ff}, // Ydieresissmall
	{0xcdae6ea2, 0xff59}, // ymonospace
	{0xcdbb64f4, 0x2105}, // afii61248
	{0xcdd2cfab, 0x24b4}, // yparen
	{0xcdd44c7f, 0x33a4}, // cmcubedsquare
	{0xcde56fe0, 0x05b6}, // segol
	{0xcdea4ff0, 0x03d4}, // Upsilondieresishooksymbolgreek
	{0xcdf0bcd8, 0x0e1e}, // phophanthai
	{0xcdfd40ec, 0x0180}, // bstroke
	{0xce074882, 0x0668}, // afii57400
	{0xce4d2d1e, 0x2563}, // SF230000
	{0xce63250d, 0x005d}, // bracketrightBigg
	{0xce6d06c6, 0x0aab}, // phagujarati
	{0xce6f0d59, 0x3272}, // mieumacirclekorean
	{0xce79a056, 0x3339}, // herutusquare
	{0xce8dac39, 0x25a8}, // squareupperrighttolowerleftfill
	{0xce942ad9, 0x09a6}, // dabengali
	{0xcea39f20, 0x02d5}, // downtackmod
	{0xced82ba6, 0xed19}, // bracehtipupleft
	{0xcedf2a68, 0x0157}, // rcedilla
	{0xcee9759d, 0x043a}, // afii10076
	{0xcef01870, 0x0171}, // udblacute
	{0xcef10e83, 0x046a}, // Yusbigcyrillic
	{0xcef67881, 0x200c}, // zerowidthnonjoiner
	{0xcf235874, 0x030c}, // caroncmb
	{0xcf4c6b71, 0x02d4}, // uptackmod
	{0xcf6550b0, 0xfb4a}, // tavdageshhebrew
	{0xcf6a5e7c, 0x323e}, // ideographicresourceparen
	{0xcf738908, 0x011d}, // gcircumflex
	{0xcf848334, 0x0453}, // afii10100
	{0xcf8cbf6b, 0x0195}, // hv
	{0xcf9b96b5, 0x03bd}, // nu
	{0xcfb1da5d, 0xff89}, // nokatakanahalfwidth
	{0xcfc386a6, 0x00db}, // Ucircumflex
	{0xcfc400d6, 0x06d2}, // yehbarreearabic
	{0xcfd8a703, 0xfe61}, // asterisksmall
	{0xcfe519cb, 0x309b}, // voicedmarkkana
	{0xcfe64e44, 0x0668}, // eighthackarabic
	{0xcff88b7d, 0x05b8}, // qamatsqatannarrowhebrew
	{0xd009507e, 0x05d3}, // daletqamatshebrew
	{0xd0096386, 0x25b3}, // whiteuppointingtriangle
	{0xd0227bd1, 0x0199}, // khook
	{0xd02a9cfe, 0x0105}, // aogonek
	{0xd031b297, 0x24e9}, // zcircle
	{0xd03ef2b0, 0x316b}, // rieulpieupsioskorean
	{0xd0417b9a, 0x0494}, // Ghemiddlehookcyrillic
	{0xd044dddd, 0x0a9d}, // jhagujarati
	{0xd04ad0f0, 0x263c}, // sun
	{0xd04cc01e, 0x0561}, // aybarmenian
	{0xd0511b87, 0x0e4c}, // thanthakhatthai
	{0xd056aca3, 0xff9a}, // rekatakanahalfwidth
	{0xd06b6bb7, 0x096a}, // fourdeva
	{0xd07a803b, 0x09aa}, // pabengali
	{0xd07f3aad, 0xf768}, // Hsmall
	{0xd0847e20, 0x05c4}, // upperdothebrew
	{0xd087e60f, 0x0158}, // Rcaron
	{0xd0897bb6, 0x0579}, // chaarmenian
	{0xd0c5df61, 0x05b4}, // hiriq14
	{0xd0ce4edc, 0x09e9}, // threebengali
	{0xd0d6e6c0, 0xff6c}, // yasmallkatakanahalfwidth
	{0xd0e3648a, 0xfd3e}, // parenleftaltonearabic
	{0xd133ff70, 0x01a1}, // ohorn
	{0xd1373ca2, 0x0648}, // wawarabic
	{0xd13d9bf5, 0xfe37}, // braceleftvertical
	{0xd13faec5, 0x05e1}, // samekh
	{0xd14fc185, 0x045b}, // afii10108
	{0xd17987dd, 0x0543}, // Cheharmenian
	{0xd180d934, 0x0621}, // hamzalowkasraarabic
	{0xd18447d8, 0x0663}, // threearabic
	{0xd18961af, 0xf7fa}, // Uacutesmall
	{0xd18966b1, 0x0a82}, // anusvaragujarati
	{0xd18d83de, 0x0aa3}, // nnagujarati
	{0xd190d310, 0x0a35}, // vagurmukhi
	{0xd191827c, 0x0e39}, // sarauuthai
	{0xd1acdf44, 0xfb94}, // gafinitialarabic
	{0xd1d7231f, 0x04bc}, // Cheabkhasiancyrillic
	{0xd1d9da71, 0x30ea}, // rikatakana
	{0xd2337241, 0x05b6}, // segol1f
	{0xd24297bf, 0xf889}, // maitaikhuleftthai
	{0xd247ef8b, 0x3243}, // ideographicreachparen
	{0xd254c368, 0x0443}, // ucyrillic
	{0xd2658bcb, 0x05b9}, // holamquarterhebrew
	{0xd26b0e16, 0x2321}, // integralbt
	{0xd26ef570, 0x310f}, // hbopomofo
	{0xd293868c, 0x019b}, // lambdastroke
	{0xd297cb79, 0x30b7}, // sikatakana
	{0xd2b4c516, 0x30da}, // pekatakana
	{0xd2c0e1bc, 0x0150}, // Odblacute
	{0xd2cbfc99, 0x05b7}, // patahhebrew
	{0xd2e2a716, 0x03ec}, // Shimacoptic
	{0xd2eaddf5, 0x0633}, // seenarabic
	{0xd2f253f5, 0xfe40}, // anglebracketrightvertical
	{0xd2f3cdf3, 0x1e0e}, // Dlinebelow
	{0xd304784a, 0x05d3}, // dalet
	{0xd308b167, 0x1e5b}, // rdotbelow
	{0xd310b973, 0x25a1}, // a50
	{0xd310fcc1, 0x04e9}, // obarredcyrillic
	{0xd3145153, 0xfeec}, // hehmedialarabic
	{0xd326ec6d, 0x2237}, // proportion
	{0xd3342503, 0x30bd}, // sokatakana
	{0xd33cb244, 0x02cd}, // macronlowmod
	{0xd35ba6fb, 0x04be}, // Chedescenderabkhasiancyrillic
	{0xd3730282, 0xff64}, // ideographiccommaleft
	{0xd3777d75, 0x05d3}, // daletsegolhebrew
	{0xd3797e0f, 0x24db}, // lcircle
	{0xd37b4bcb, 0x041c}, // afii10030
	{0xd3a5ba29, 0x062e}, // afii57422
	{0xd3c22da1, 0x01b5}, // Zstroke
	{0xd3c6a66e, 0x05e4}, // pe
	{0xd3cfef4c, 0x0427}, // afii10041
	{0xd3e84c23, 0x201c}, // quotedblleft
	{0xd421361f, 0x067e}, // peharabic
	{0xd4273f62, 0x30fd}, // iterationkatakana
	{0xd44f2d4c, 0xff9b}, // rokatakanahalfwidth
	{0xd45c6c89, 0x3079}, // behiragana
	{0xd467b0a3, 0x0131}, // dotlessi
	{0xd46989dc, 0x05e7}, // qofholam
	{0xd48c064b, 0x0599}, // pashtahebrew
	{0xd48dad4d, 0x01e2}, // AEmacron
	{0xd4b2bb68, 0x0145}, // Ncedilla
	{0xd4ce7b9e, 0x0621}, // hamzadammaarabic
	{0xd4d3eb56, 0x255e}, // SF360000
	{0xd4e86e58, 0x1ef7}, // yhookabove
	{0xd4e92fa8, 0x0623}, // alefhamzaabovearabic
	{0xd514a0e0, 0x318c}, // yuikorean
	{0xd5189135, 0x1e73}, // udieresisbelow
	{0xd52f1d8f, 0x24cb}, // Vcircle
	{0xd54b71bd, 0x0a6e}, // eightgurmukhi
	{0xd574af4c, 0x318e}, // araeaekorean
	{0xd57a206a, 0x00f2}, // ograve
	{0xd58ee561, 0x04f3}, // uhungarumlautcyrillic
	{0xd5b6f4bf, 0x3163}, // ikorean
	{0xd5b7a706, 0x1e13}, // dcircumflexbelow
	{0xd5da03ab, 0x0123}, // gcommaaccent
	{0xd5da183e, 0xf76e}, // Nsmall
	{0xd5dc1f0e, 0x24a5}, // jparen
	{0xd5e6a2b4, 0x200f}, // afii300
	{0xd6067104, 0x05bd}, // siluqlefthebrew
	{0xd6180af1, 0x0023}, // numbersign
	{0xd6191adc, 0x02ce}, // gravelowmod
	{0xd6480a61, 0x0e46}, // maiyamokthai
	{0xd65815d1, 0x04f5}, // chedieresiscyrillic
	{0xd6674587, 0xfea8}, // khahmedialarabic
	{0xd66f3b98, 0x1ebd}, // etilde
	{0xd67d357f, 0x0e23}, // roruathai
	{0xd67dc19d, 0x1e05}, // bdotbelow
	{0xd682be7e, 0xfe54}, // semicolonsmall
	{0xd689f58d, 0x0024}, // dollar
	{0xd68be98a, 0xff8f}, // makatakanahalfwidth
	{0xd6a99b0e, 0x05aa}, // yerahbenyomolefthebrew
	{0xd6c4c66e, 0x3262}, // tikeutcirclekorean
	{0xd6c7e5a6, 0x03b0}, // upsilondieresistonos
	{0xd6df6252, 0xfef4}, // yehmedialarabic
	{0xd6e234de, 0x044d}, // afii10095
	{0xd7151c8a, 0x040b}, // afii10060
	{0xd71970f6, 0x05d3}, // daletsegol
	{0xd71af0cf, 0x00af}, // overscore
	{0xd72c6112, 0x0036}, // six
	{0xd7363d15, 0x05d3}, // daletholamhebrew
	{0xd73b3901, 0x02da}, // ring
	{0xd7425de1, 0x0a1d}, // jhagurmukhi
	{0xd771b953, 0x3107}, // mbopomofo
	{0xd7a40cc3, 0x315c}, // ukorean
	{0xd7b7f8a3, 0x3094}, // vuhiragana
	{0xd7b8c7af, 0x05b8}, // qamatsquarterhebrew
	{0xd7bc737d, 0x230b}, // floorrightBig
	{0xd7bf0d2a, 0x308b}, // ruhiragana
	{0xd7cebade, 0x01b1}, // Upsilonafrican
	{0xd7d268b5, 0x314e}, // hieuhkorean
	{0xd7ece605, 0x2553}, // SF520000
	{0xd813ab1a, 0x3176}, // pieupcieuckorean
	{0xd816387d, 0x2178}, // nineroman
	{0xd8171429, 0x013e}, // lcaron
	{0xd817c39d, 0x0664}, // fourhackarabic
	{0xd824acfb, 0x05de}, // memhebrew
	{0xd82c0976, 0x05e8}, // reshsegolhebrew
	{0xd849e14d, 0x3052}, // gehiragana
	{0xd84afb0a, 0x00d1}, // Ntilde
	{0xd85534fc, 0x02ca}, // secondtonechinese
	{0xd8708805, 0x01ba}, // ezhtail
	{0xd890928b, 0x0053}, // S
	{0xd893adf8, 0x0580}, // reharmenian
	{0xd8964f73, 0x043d}, // encyrillic
	{0xd89879e8, 0x2116}, // afii61352
	{0xd8a1ab6d, 0x03ab}, // Upsilondieresis
	{0xd8b1d2bf, 0x0177}, // ycircumflex
	{0xd8faed0d, 0x30ef}, // wakatakana
	{0xd9038cdb, 0x32a4}, // ideographichighcircle
	{0xd90a7039, 0x06af}, // afii57509
	{0xd91584cd, 0xfe31}, // emdashvertical
	{0xd92072b9, 0x0493}, // ghestrokecyrillic
	{0xd92d9608, 0x0acd}, // viramagujarati
	{0xd932c15c, 0x30ac}, // gakatakana
	{0xd93c2940, 0x33cf}, // ktsquare
	{0xd94d846e, 0x321a}, // phieuphaparenkorean
	{0xd94faf13, 0x0188}, // chook
	{0xd95c2f59, 0x04bd}, // cheabkhasiancyrillic
	{0xd9697a13, 0x328b}, // ideographfirecircle
	{0xd98cc91f, 0x0307}, // dotaccentcmb
	{0xd98dcef9, 0xf8ff}, // apple
	{0xd991004f, 0x0e4f}, // fongmanthai
	{0xd99e3976, 0x249f}, // dparen
	{0xd9ba695c, 0xff70}, // katahiraprolongmarkhalfwidth
	{0xd9d63664, 0x0189}, // Dafrican
	{0xd9e83df4, 0x096b}, // fivedeva
	{0xd9eba56d, 0x03b3}, // gamma
	{0xda15411c, 0x0304}, // macroncmb
	{0xda2037e1, 0x1e01}, // aringbelow
	{0xda3670ae, 0x3347}, // mansyonsquare
	{0xda38804a, 0xf738}, // eightoldstyle
	{0xda39b9df, 0x013d}, // Lcaron
	{0xda48ff7e, 0x00bb}, // guillemotright
	{0xda4e1891, 0x228b}, // supersetnotequal
	{0xda55d0f3, 0x0a69}, // threegurmukhi
	{0xda620e6e, 0xfca2}, // tehhahinitialarabic
	{0xda734cc8, 0x232b}, // deleteleft
	{0xda85eaa3, 0x2070}, // zerosuperior
	{0xda94576a, 0x0999}, // ngabengali
	{0xda94a677, 0x0634}, // afii57428
	{0xda99b3d7, 0x30d1}, // pakatakana
	{0xda9d5f69, 0xf6c8}, // afii10832
	{0xdab46527, 0x21e9}, // arrowdownwhite
	{0xdac4a95a, 0x002f}, // slash
	{0xdac8670b, 0x011e}, // Gbreve
	{0xdace8a4c, 0xed17}, // bracehtipdownleft
	{0xdad5813e, 0x24bf}, // Jcircle
	{0xdaf9ae21, 0x03e7}, // kheicoptic
	{0xdb00acb2, 0x04dc}, // Zhedieresiscyrillic
	{0xdb07430c, 0x1eac}, // Acircumflexdotbelow
	{0xdb15243d, 0xfb38}, // tetdagesh
	{0xdb180684, 0x059d}, // gereshmuqdamhebrew
	{0xdb19f222, 0x1e30}, // Kacute
	{0xdb215045, 0xf76f}, // Osmall
	{0xdb36c0cb, 0x00c1}, // Aacute
	{0xdb46a061, 0x044f}, // afii10097
	{0xdb4843d8, 0x0140}, // ldotaccent
	{0xdb491e12, 0x3201}, // nieunparenkorean
	{0xdb4ecb82, 0x06a4}, // afii57505
	{0xdb5fdfb2, 0x09cd}, // viramabengali
	{0xdb7c2cdb, 0xf88f}, // maitholowleftthai
	{0xdb8ff30c, 0xf6f5}, // Caronsmall
	{0xdb9c2f74, 0x3063}, // tusmallhiragana
	{0xdb9dda85, 0xfb3b}, // kafdageshhebrew
	{0xdba170e8, 0x0998}, // ghabengali
	{0xdbae2c8c, 0x2277}, // greaterorless
	{0xdbc3c473, 0x001a}, // controlSUB
	{0xdbc6ef9a, 0x05b2}, // hatafpatah23
	{0xdbc71338, 0xff01}, // exclammonospace
	{0xdbcb0069, 0x338a}, // pfsquare
	{0xdbf12380, 0x2460}, // onecircle
	{0xdc0071a3, 0xfb46}, // tsadidageshhebrew
	{0xdc05ec50, 0x30f5}, // kasmallkatakana
	{0xdc0ac9c6, 0x0028}, // parenleftBigg
	{0xdc0ad3ae, 0x012d}, // ibreve
	{0xdc0c240d, 0xff9e}, // voicedmarkkanahalfwidth
	{0xdc0c9e85, 0xff9f}, // semivoicedmarkkanahalfwidth
	{0xdc3d7ac8, 0x04a7}, // pemiddlehookcyrillic
	{0xdc41d3b3, 0x05ab}, // olehebrew
	{0xdc54447c, 0x307c}, // bohiragana
	{0xdc6ca9b3, 0x0584}, // keharmenian
	{0xdc7650d9, 0xf73f}, // questionsmall
	{0xdc7756d1, 0x0077}, // w
	{0xdc7d1de8, 0xfba9}, // hehmedialaltonearabic
	{0xdc7f6ca5, 0x0624}, // wawhamzaabovearabic
	{0xdcc5c006, 0x055a}, // apostrophearmenian
	{0xdce03f6b, 0x261e}, // pointingindexrightwhite
	{0xdcefaeeb, 0x228a}, // subsetnotequal
	{0xdd07775c, 0x3223}, // fourideographicparen
	{0xdd07a474, 0x00b3}, // threesuperior
	{0xdd21d4c1, 0x039b}, // Lambda
	{0xdd2fee63, 0x0aad}, // bhagujarati
	{0xdd4e62a4, 0x0a96}, // khagujarati
	{0xdd55f861, 0x2560}, // SF420000
	{0xdd64bab7, 0x3080}, // muhiragana
	{0xdd68d3ef, 0x00a5}, // yen
	{0xdd8a8538, 0x0a26}, // dagurmukhi
	{0xdd9a009b, 0x2016}, // dblverticalbar
	{0xdda2fef7, 0x33db}, // srsquare
	{0xdda8f1e0, 0x33d4}, // mbsquare
	{0xddd89deb, 0xff4d}, // mmonospace
	{0xdde406ed, 0x23a8}, // braceleftmid
	{0xddfdd08a, 0x30a7}, // esmallkatakana
	{0xddfea657, 0x0049}, // I
	{0xddffcb32, 0x1e60}, // Sdotaccent
	{0xde111430, 0x2271}, // notgreaternorequal
	{0xde159412, 0x2605}, // blackstar
	{0xde3de1bb, 0xfede}, // lamfinalarabic
	{0xde4643cf, 0x0ac5}, // ecandravowelsigngujarati
	{0xde5450d1, 0xff88}, // nekatakanahalfwidth
	{0xde6c8dd1, 0x2223}, // divides
	{0xde91c7ac, 0xfb35}, // vavdagesh
	{0xdea63325, 0x0e0c}, // chochoethai
	{0xdea93241, 0x0385}, // dieresistonos
	{0xdeab4b8b, 0x05ad}, // dehihebrew
	{0xdebc4010, 0xfcd5}, // noonmeeminitialarabic
	{0xdebf0df4, 0x05f0}, // afii57716
	{0xdecde878, 0x1e52}, // Omacronacute
	{0xdee80462, 0xffe5}, // yenmonospace
	{0xdee969b3, 0x3001}, // ideographiccomma
	{0xdef14eee, 0x020e}, // Oinvertedbreve
	{0xdef351c2, 0x020c}, // Odblgrave
	{0xdef92b6a, 0x3349}, // mirisquare
	{0xdf08e8b4, 0xff15}, // fivemonospace
	{0xdf09c757, 0x0186}, // Oopen
	{0xdf1ee74b, 0xff81}, // tikatakanahalfwidth
	{0xdf243dad, 0x044c}, // softsigncyrillic
	{0xdf46fba9, 0x01a4}, // Phook
	{0xdf5e1052, 0x221a}, // radicalbig
	{0xdf80589a, 0x3217}, // chieuchaparenkorean
	{0xdf8c6402, 0x0e45}, // lakkhangyaothai
	{0xdf8fbdeb, 0x05e7}, // qofsheva
	{0xdf9eaf7a, 0x33bc}, // muwsquare
	{0xdfad5d93, 0x22c0}, // logicalandtext
	{0xdfaf476d, 0x0167}, // tbar
	{0xdfb9632c, 0x3144}, // pieupsioskorean
	{0xdfe8c3dc, 0xf7fc}, // Udieresissmall
	{0xdff819d0, 0xfe8c}, // yehhamzaabovemedialarabic
	{0xdffe3761, 0x1e14}, // Emacrongrave
	{0xe0061dae, 0x05d3}, // daletpatah
	{0xe0130535, 0x316c}, // rieulpansioskorean
	{0xe019189f, 0x0994}, // aubengali
	{0xe0197d92, 0x0a4b}, // oomatragurmukhi
	{0xe02aebf6, 0xf6e3}, // dollarinferior
	{0xe0343a59, 0x0323}, // dotbelowcomb
	{0xe03a2368, 0x019f}, // Ocenteredtilde
	{0xe0489c79, 0x017f}, // longs
	{0xe0513bea, 0xf6ff}, // Zcaronsmall
	{0xe0560cdf, 0x1e19}, // ecircumflexbelow
	{0xe065671a, 0x1e0a}, // Ddotaccent
	{0xe07dfee5, 0x04f4}, // Chedieresiscyrillic
	{0xe0800244, 0xfb69}, // ttehmedialarabic
	{0xe0831234, 0x007b}, // braceleftBig
	{0xe0987417, 0x24dd}, // ncircle
	{0xe0ab68f9, 0xf6dc}, // onefitted
	{0xe0ac4869, 0x3225}, // sixideographicparen
	{0xe0c42e1f, 0x3357}, // wattosquare
	{0xe0cf3aca, 0x1e1d}, // ecedillabreve
	{0xe0d114bf, 0x04b6}, // Chedescendercyrillic
	{0xe0dbd3b5, 0x2494}, // thirteenperiod
	{0xe0ec0106, 0x05a6}, // merkhakefulahebrew
	{0xe0ec7a9f, 0x01a6}, // yr
	{0xe0f957bb, 0x0644}, // afii57444
	{0xe10a53c6, 0x0aa8}, // nagujarati
	{0xe13f2d93, 0x3026}, // sixhangzhou
	{0xe173c1f2, 0x1e12}, // Dcircumflexbelow
	{0xe180ca73, 0x1e43}, // mdotbelow
	{0xe1b37094, 0x0629}, // afii57417
	{0xe1bbda55, 0x040e}, // Ushortcyrillic
	{0xe1bf1035, 0x0536}, // Zaarmenian
	{0xe20234a2, 0x012c}, // Ibreve
	{0xe20c937f, 0x041a}, // Kacyrillic
	{0xe20deadd, 0x047c}, // Omegatitlocyrillic
	{0xe222a727, 0x02dc}, // tildewide
	{0xe2234dec, 0xff52}, // rmonospace
	{0xe2256c16, 0x05d3}, // daletqamats
	{0xe22682ea, 0x3260}, // kiyeokcirclekorean
	{0xe22a6510, 0x1e8e}, // Ydotaccent
	{0xe23968a4, 0x04b3}, // hadescendercyrillic
	{0xe25773d9, 0x04d3}, // adieresiscyrillic
	{0xe259edda, 0x0628}, // afii57416
	{0xe25f57e9, 0xfe9a}, // thehfinalarabic
	{0xe26168f6, 0x1e82}, // Wacute
	{0xe28a564c, 0x044f}, // iacyrillic
	{0xe28eea2e, 0x1e51}, // omacrongrave
	{0xe2924f7e, 0x0a67}, // onegurmukhi
	{0xe2948e05, 0xf6f6}, // Circumflexsmall
	{0xe2a7b092, 0x3213}, // pieupaparenkorean
	{0xe2b99909, 0x0e14}, // dodekthai
	{0xe2cfeeb5, 0x0301}, // acutecomb
	{0xe2eccaa5, 0x0059}, // Y
	{0xe2ef1bbf, 0x2197}, // arrowupright
	{0xe2fc74df, 0x0533}, // Gimarmenian
	{0xe2ff3ec5, 0xfb8d}, // rrehfinalarabic
	{0xe2ffc4d4, 0x2661}, // heartsuitwhite
	{0xe3356dd7, 0x1e54}, // Pacute
	{0xe35f1369, 0xfe5c}, // bracerightsmall
	{0xe3708e14, 0xfed3}, // fehinitialarabic
	{0xe37c75f9, 0x03c7}, // chi
	{0xe38423f1, 0x01ab}, // tpalatalhook
	{0xe387ebf8, 0xff94}, // yakatakanahalfwidth
	{0xe39adf52, 0x05b7}, // patahnarrowhebrew
	{0xe39bb5ba, 0x0258}, // ereversed
	{0xe3a00fb0, 0x0aa2}, // ddhagujarati
	{0xe3a0394c, 0x05b3}, // afii57802
	{0xe3bf40b5, 0x21c5}, // arrowupleftofdown
	{0xe3c68591, 0x005d}, // bracketrightBig
	{0xe3cbb73f, 0x0459}, // afii10106
	{0xe3cddac1, 0x0621}, // hamzalowarabic
	{0xe3d5ad06, 0x1ec0}, // Ecircumflexgrave
	{0xe3dd99f7, 0x0596}, // tipehahebrew
	{0xe3fae787, 0x0587}, // echyiwnarmenian
	{0xe3fe00d6, 0x2170}, // oneroman
	{0xe401701f, 0x3016}, // whitelenticularbracketleft
	{0xe41d9109, 0x3089}, // rahiragana
	{0xe424f4cb, 0x0969}, // threedeva
	{0xe42a588e, 0x0e27}, // wowaenthai
	{0xe42a6647, 0x0a71}, // addakgurmukhi
	{0xe433c6bf, 0x042c}, // afii10046
	{0xe44aea0b, 0x0219}, // scommaaccent
	{0xe44ed7aa, 0x2481}, // fourteenparen
	{0xe468e60e, 0x33bd}, // mwsquare
	{0xe47b4b2e, 0xfecb}, // aininitialarabic
	{0xe4896ee4, 0x09c0}, // iivowelsignbengali
	{0xe4924345, 0xff63}, // cornerbracketrighthalfwidth
	{0xe49ba568, 0x090c}, // lvocalicdeva
	{0xe4a8920f, 0x0014}, // controlDC4
	{0xe4c6b94a, 0x0550}, // Reharmenian
	{0xe4c91eec, 0x1ea7}, // acircumflexgrave
	{0xe4ce70c5, 0xfedf}, // lammeemkhahinitialarabic
	{0xe4d0a5c5, 0x2171}, // tworoman
	{0xe4de0824, 0x090b}, // rvocalicdeva
	{0xe4e78bbf, 0x0e33}, // saraamthai
	{0xe4e90251, 0xfb41}, // samekhdagesh
	{0xe4f1acea, 0x24d7}, // hcircle
	{0xe5034999, 0xff55}, // umonospace
	{0xe504c22f, 0x01a9}, // Esh
	{0xe506e1eb, 0x2493}, // twelveperiod
	{0xe5107e85, 0xff92}, // mekatakanahalfwidth
	{0xe5116fc8, 0x2472}, // nineteencircle
	{0xe514f37f, 0x03be}, // xi
	{0xe51bd3a3, 0x0163}, // tcedilla
	{0xe5287de7, 0x3173}, // pieuptikeutkorean
	{0xe5322bbf, 0x25b2}, // blackuppointingtriangle
	{0xe5431590, 0x2296}, // minuscircle
	{0xe54a2cdd, 0x201e}, // quotedblbase
	{0xe5694805, 0x3161}, // eukorean
	{0xe56f3e08, 0x1e9b}, // slongdotaccent
	{0xe5779de7, 0x0666}, // afii57398
	{0xe578bf9d, 0x066d}, // asteriskaltonearabic
	{0xe5850206, 0x2226}, // notparallel
	{0xe59889e5, 0x014b}, // eng
	{0xe59f1c1d, 0x2664}, // spadesuitwhite
	{0xe59f1f71, 0x2203}, // thereexists
	{0xe5dca65a, 0x04aa}, // Esdescendercyrillic
	{0xe5e0ac19, 0x0a74}, // ekonkargurmukhi
	{0xe5eb1828, 0x25e6}, // openbullet
	{0xe5f403ac, 0x0981}, // candrabindubengali
	{0xe5f51e2d, 0x0156}, // Rcedilla
	{0xe61a323e, 0x21d2}, // arrowdblright
	{0xe61dad29, 0x017b}, // Zdotaccent
	{0xe62fb889, 0x2524}, // SF090000
	{0xe65602bc, 0x05d3}, // daletqubuts
	{0xe6699f03, 0x025d}, // eopenreversedhook
	{0xe6743f55, 0x011a}, // Ecaron
	{0xe6861695, 0x018a}, // Dhook
	{0xe68d35b4, 0x0436}, // afii10072
	{0xe68e2cfd, 0x306b}, // nihiragana
	{0xe6910141, 0x06f1}, // onepersian
	{0xe6936418, 0xfb2f}, // alefqamatshebrew
	{0xe6a0587d, 0x3222}, // threeideographicparen
	{0xe6a445c8, 0xfcd2}, // noonjeeminitialarabic
	{0xe6b00dbd, 0x060c}, // commaarabic
	{0xe6bd7d55, 0x04f0}, // Udieresiscyrillic
	{0xe6beeaab, 0x2667}, // clubsuitwhite
	{0xe6d26878, 0x0663}, // threehackarabic
	{0xe6eee43a, 0x031a}, // leftangleabovecmb
	{0xe70bdf5d, 0x1e85}, // wdieresis
	{0xe72762a6, 0x3127}, // ibopomofo
	{0xe74e8d5b, 0x02d0}, // colontriangularmod
	{0xe75de72d, 0x0a4d}, // halantgurmukhi
	{0xe79030f9, 0xf6cf}, // Hungarumlaut
	{0xe796e6c1, 0x032f}, // breveinvertedbelowcmb
	{0xe7a9201c, 0x24e5}, // vcircle
	{0xe7abb869, 0x05b0}, // sheva115
	{0xe7f3b395, 0x24d8}, // icircle
	{0xe80a2426, 0x3116}, // rbopomofo
	{0xe80eec90, 0x0942}, // uuvowelsigndeva
	{0xe832e25c, 0x03ef}, // deicoptic
	{0xe836846a, 0x20a1}, // colonsign
	{0xe8445eb9, 0x05e5}, // finaltsadihebrew
	{0xe84e29b1, 0xfef5}, // lamalefmaddaaboveisolatedarabic
	{0xe853fc35, 0x3177}, // pieupthieuthkorean
	{0xe871a9d4, 0x03e3}, // sheicoptic
	{0xe872f83d, 0x02a0}, // qhook
	{0xe87409fd, 0xfe8b}, // yehhamzaaboveinitialarabic
	{0xe885d69b, 0x1e58}, // Rdotaccent
	{0xe893423b, 0x202c}, // afii61573
	{0xe894f25c, 0x03a1}, // Rho
	{0xe89d944f, 0x043a}, // kacyrillic
	{0xe8aff9ae, 0x24b7}, // Bcircle
	{0xe8d4db79, 0x2320}, // integraltp
	{0xe8d6bd29, 0x0621}, // afii57409
	{0xe8ec3154, 0x2490}, // nineperiod
	{0xe8ef5782, 0x3092}, // wohiragana
	{0xe8f0c584, 0x05d1}, // afii57665
	{0xe8f5ca9b, 0x05b1}, // hatafsegolnarrowhebrew
	{0xe8f7e9bf, 0x320f}, // nieunaparenkorean
	{0xe903bc23, 0xfb36}, // zayindageshhebrew
	{0xe90424fe, 0x2479}, // sixparen
	{0xe927829f, 0x0953}, // gravedeva
	{0xe92a9522, 0x0060}, // grave
	{0xe93b2c93, 0x09c8}, // aivowelsignbengali
	{0xe9544ae6, 0x2308}, // ceilingleftbigg
	{0xe9598c36, 0x018e}, // Ereversed
	{0xe96c43a5, 0x00d4}, // Ocircumflex
	{0xe977a63b, 0x0591}, // etnahtalefthebrew
	{0xe9860674, 0x2467}, // eightcircle
	{0xe994ec07, 0x1e88}, // Wdotbelow
	{0xe9955c1b, 0x2109}, // fahrenheit
	{0xe997ce80, 0x03d6}, // pisymbolgreek
	{0xe9b8fb39, 0x256a}, // SF540000
	{0xe9c5c9e3, 0x0928}, // nadeva
	{0xe9dc1b2b, 0x3162}, // yikorean
	{0xe9f5bb0b, 0x1e40}, // Mdotaccent
	{0xe9fd5bd7, 0x02d1}, // colontriangularhalfmod
	{0xe9fe0986, 0xfe91}, // behinitialarabic
	{0xea0e9b0d, 0x3077}, // puhiragana
	{0xea11b5f2, 0x05d7}, // hethebrew
	{0xea281c50, 0xfb3b}, // kafdagesh
	{0xea281f0f, 0xfb43}, // pefinaldageshhebrew
	{0xea4521ba, 0x1e99}, // yring
	{0xea469ad4, 0x0641}, // feharabic
	{0xea4bf6ca, 0x25b5}, // whiteuppointingsmalltriangle
	{0xea505c24, 0x006b}, // k
	{0xea723c32, 0x05b7}, // patahwidehebrew
	{0xea828d24, 0x0ac4}, // rrvocalicvowelsigngujarati
	{0xea8df8fe, 0xff09}, // parenrightmonospace
	{0xeaaa8586, 0x05b6}, // segolhebrew
	{0xeabb8dad, 0xfeb3}, // seeninitialarabic
	{0xeabdd2cd, 0xff02}, // quotedblmonospace
	{0xeacd0b4a, 0xfb1f}, // doubleyodpatah
	{0xeade9ba5, 0x2113}, // afii61289
	{0xeb0445d5, 0x043f}, // afii10081
	{0xeb095515, 0x0452}, // afii10099
	{0xeb4a0523, 0x056a}, // zhearmenian
	{0xeb4fdea3, 0xfc08}, // behmeemisolatedarabic
	{0xeb576d94, 0x21de}, // pageup
	{0xeb6b92be, 0x2191}, // arrowtp
	{0xeb701704, 0x2014}, // emdash
	{0xeb71d801, 0x30f3}, // nkatakana
	{0xebeccb02, 0x00dd}, // Yacute
	{0xec20a331, 0x096c}, // sixdeva
	{0xec450aad, 0x3211}, // rieulaparenkorean
	{0xec4fc0c5, 0x0173}, // uogonek
	{0xec78ec45, 0x05a4}, // mahapakhhebrew
	{0xec820a21, 0x05b4}, // hiriqquarterhebrew
	{0xec8ae366, 0x0987}, // ibengali
	{0xec967081, 0x040a}, // Njecyrillic
	{0xecad584f, 0x01a8}, // tonetwo
	{0xecc21039, 0x0a98}, // ghagujarati
	{0xecce5cae, 0x056b}, // iniarmenian
	{0xecd385c5, 0x03a8}, // Psi
	{0xecda4c6b, 0x0622}, // alefmaddaabovearabic
	{0xecddb27b, 0x23d0}, // vextendsingle
	{0xecddb519, 0x05db}, // kaf
	{0xecef01e0, 0x1e59}, // rdotaccent
	{0xed102125, 0x24c0}, // Kcircle
	{0xed1afc7c, 0x05b2}, // hatafpatah2f
	{0xed2b4a43, 0x00aa}, // ordfeminine
	{0xed4bb321, 0x313f}, // rieulphieuphkorean
	{0xed65d1e6, 0x05e7}, // qofqubutshebrew
	{0xed65e3bb, 0x057a}, // peharmenian
	{0xed729d82, 0x049e}, // Kastrokecyrillic
	{0xed796a9e, 0x0407}, // afii10056
	{0xed8b66db, 0x040b}, // Tshecyrillic
	{0xed9bf511, 0x063a}, // afii57434
	{0xedfaec74, 0xfe69}, // dollarsmall
	{0xee22e47f, 0x00bd}, // onehalf
	{0xee3352d3, 0x05b0}, // afii57799
	{0xee339d2e, 0x2025}, // twodotenleader
	{0xee421e32, 0x0660}, // zeroarabic
	{0xee6c8858, 0x3168}, // nieunpansioskorean
	{0xee7a31f8, 0x220b}, // suchthat
	{0xee8d09d6, 0xff12}, // twomonospace
	{0xee900f0f, 0x0462}, // afii10146
	{0xee9f1e99, 0x02b7}, // wsuperior
	{0xeeb00f1b, 0x0063}, // c
	{0xeeb5ef47, 0x24c2}, // Mcircle
	{0xeeda5b48, 0x00a6}, // brokenbar
	{0xeeeef128, 0x062f}, // afii57423
	{0xeeef7f7c, 0x01b6}, // zstroke
	{0xef03e03f, 0x32a3}, // ideographiccorrectcircle
	{0xef22f61a, 0x06f0}, // zeropersian
	{0xef24cf3c, 0x00a8}, // dieresis
	{0xef2800a0, 0x00cc}, // Igrave
	{0xef33d78e, 0x25a1}, // H22073
	{0xef3a179c, 0x020a}, // Iinvertedbreve
	{0xef636ee0, 0x05da}, // finalkaf
	{0xef7afe15, 0x007d}, // bracerightbigg
	{0xefa03eab, 0x00c6}, // AE
	{0xefc57067, 0x0260}, // ghook
	{0xefd65ddd, 0x2033}, // second
	{0xefd69119, 0x044e}, // iucyrillic
	{0xefd88572, 0x1ecb}, // idotbelow
	{0xefe0e3fd, 0xfb01}, // fi
	{0xeff59b38, 0x0397}, // Eta
	{0xeff843d0, 0x25a3}, // squarewhitewithsmallblack
	{0xf00181f3, 0x305c}, // zehiragana
	{0xf00455e8, 0x1e86}, // Wdotaccent
	{0xf0045976, 0x3178}, // kapyeounpieupkorean
	{0xf0213847, 0x3383}, // masquare
	{0xf0248bca, 0x0640}, // tatweelarabic
	{0xf029041c, 0x3017}, // whitelenticularbracketright
	{0xf0412bb8, 0x04d0}, // Abrevecyrillic
	{0xf04a093b, 0xff66}, // wokatakanahalfwidth
	{0xf04c3677, 0xfe62}, // plussmall
	{0xf04d9cf4, 0x308c}, // rehiragana
	{0xf064a013, 0x047e}, // Otcyrillic
	{0xf0673e49, 0x096f}, // ninedeva
	{0xf0757f39, 0x030a}, // ringcmb
	{0xf08334d5, 0x05d3}, // dalethatafpatahhebrew
	{0xf08a28a9, 0xfe4e}, // lowlinecenterline
	{0xf09092b0, 0x3185}, // ssanghieuhkorean
	{0xf091911a, 0x223d}, // reversedtilde
	{0xf098620b, 0x066a}, // percentarabic
	{0xf0a5507a, 0x03d5}, // phisymbolgreek
	{0xf0ac995b, 0x337c}, // syouwaerasquare
	{0xf0b806fd, 0xfb7c}, // tchehinitialarabic
	{0xf0c10455, 0x0645}, // meemarabic
	{0xf0d79471, 0x0201}, // adblgrave
	{0xf0e2f076, 0x04cb}, // Chekhakassiancyrillic
	{0xf0ec6a42, 0x0302}, // circumflexcmb
	{0xf0f2a82b, 0x1e8a}, // Xdotaccent
	{0xf0fcc511, 0x3398}, // klsquare
	{0xf1116d2e, 0xfe41}, // cornerbracketleftvertical
	{0xf1200f87, 0x05e6}, // tsadi
	{0xf13a2d0d, 0x0119}, // eogonek
	{0xf1410096, 0x1e1a}, // Etildebelow
	{0xf144c7a3, 0x249d}, // bparen
	{0xf15ab600, 0x05bb}, // qubuts18
	{0xf16238a6, 0x04de}, // Zedieresiscyrillic
	{0xf1aad12e, 0x0638}, // afii57432
	{0xf1b08e52, 0x20ab}, // dong
	{0xf1b0be56, 0x20a2}, // cruzeiro
	{0xf1b5f5d1, 0x05b5}, // tserehebrew
	{0xf1dd7830, 0x033d}, // xabovecmb
	{0xf1ddaa7d, 0x0995}, // kabengali
	{0xf1e94d64, 0x064e}, // fathaarabic
	{0xf1eb4f66, 0x0116}, // Edotaccent
	{0xf1f2ec50, 0x05e7}, // qofhatafsegol
	{0xf1f78ce7, 0x226a}, // muchless
	{0xf2118c6c, 0x0635}, // sadarabic
	{0xf232181a, 0x2640}, // venus
	{0xf237f0f1, 0xff29}, // Imonospace
	{0xf23a5b68, 0x0130}, // Idot
	{0xf24a3a6b, 0x05b8}, // qamats1c
	{0xf2558e7a, 0x230a}, // floorleftbig
	{0xf2620ee8, 0x1e07}, // blinebelow
	{0xf26e5910, 0x045e}, // ushortcyrillic
	{0xf2b4963c, 0x09b0}, // rabengali
	{0xf2b826ec, 0x0449}, // shchacyrillic
	{0xf2c1d44e, 0x05be}, // maqafhebrew
	{0xf2c69081, 0x016e}, // Uring
	{0xf2dd8deb, 0x248d}, // sixperiod
	{0xf2e23a0c, 0xfb57}, // pehfinalarabic
	{0xf2e7f536, 0x0329}, // verticallinebelowcmb
	{0xf2efdad2, 0x05e8}, // reshqamats
	{0xf2f52e6c, 0x00ce}, // Icircumflex
	{0xf2f6e905, 0x24cd}, // Xcircle
	{0xf3086f4b, 0x2030}, // perthousand
	{0xf311fe21, 0xfece}, // ghainfinalarabic
	{0xf315dbae, 0xfe3c}, // blacklenticularbracketrightvertical
	{0xf31fc2c2, 0xf76d}, // Msmall
	{0xf336d994, 0xfef3}, // yehinitialarabic
	{0xf3527249, 0xfb67}, // ttehfinalarabic
	{0xf365ee1e, 0xf6f8}, // Hungarumlautsmall
	{0xf3935843, 0xff26}, // Fmonospace
	{0xf399cd14, 0x0074}, // t
	{0xf3ab1b0a, 0x05d3}, // dalethatafsegol
	{0xf3c08521, 0x0431}, // afii10066
	{0xf3ce4ef0, 0x014a}, // Eng
	{0xf3cfe996, 0x05b8}, // qamatsde
	{0xf3ef0654, 0x2160}, // Oneroman
	{0xf41c3e87, 0x3182}, // yesieungsioskorean
	{0xf4266df0, 0x2253}, // imageorapproximatelyequal
	{0xf4584280, 0x05ea}, // tav
	{0xf4637345, 0x207a}, // plussuperior
	{0xf467a09a, 0x3066}, // tehiragana
	{0xf4728f62, 0x06f9}, // ninepersian
	{0xf47778a3, 0x0958}, // qadeva
	{0xf47c7f06, 0x05a5}, // merkhalefthebrew
	{0xf4970a5b, 0x222b}, // integraldisplay
	{0xf498c20b, 0xfb2c}, // shindageshshindothebrew
	{0xf4a0d900, 0xf7e6}, // AEsmall
	{0xf4a2e6c9, 0xff8e}, // hokatakanahalfwidth
	{0xf4c721dd, 0x0415}, // afii10022
	{0xf4d1afd1, 0x015b}, // sacute
	{0xf4d731e8, 0x0e2b}, // hohipthai
	{0xf4d7dcfe, 0x05b0}, // sheva
	{0xf4ea5918, 0x0423}, // afii10037
	{0xf4f5b85f, 0x05bd}, // afii57839
	{0xf4fec4c5, 0x2026}, // ellipsis
	{0xf4fecbee, 0x3152}, // yaekorean
	{0xf521dc0d, 0xf777}, // Wsmall
	{0xf526b2bc, 0x09f5}, // twonumeratorbengali
	{0xf53d898f, 0x0122}, // Gcedilla
	{0xf54df907, 0x02e8}, // tonebarlowmod
	{0xf573def2, 0x25ba}, // blackrightpointingpointer
	{0xf573f1ec, 0x24a4}, // iparen
	{0xf59704d9, 0xff4f}, // omonospace
	{0xf59943f5, 0x01d4}, // ucaron
	{0xf59f95da, 0x0399}, // Iota
	{0xf5a6729d, 0x247e}, // elevenparen
	{0xf5ab4f6d, 0x0458}, // afii10105
	{0xf5c2a87b, 0x0a14}, // augurmukhi
	{0xf5c40812, 0x2019}, // quoteright
	{0xf5cad972, 0x1ea6}, // Acircumflexgrave
	{0xf5e83826, 0x0986}, // aabengali
	{0xf5f606a8, 0x316e}, // mieumpieupkorean
	{0xf5f79af6, 0x23a3}, // bracketleftbt
	{0xf5fe99ee, 0x2017}, // underscoredbl
	{0xf61328eb, 0x2580}, // upblock
	{0xf61a2336, 0x0157}, // rcommaaccent
	{0xf6228c58, 0x20a3}, // franc
	{0xf6271ec7, 0x0429}, // afii10043
	{0xf630815e, 0x0577}, // shaarmenian
	{0xf643d64b, 0xf774}, // Tsmall
	{0xf64f0a5d, 0x30f9}, // vekatakana
	{0xf64f6666, 0xff44}, // dmonospace
	{0xf6545660, 0x3333}, // huiitosquare
	{0xf655e1cb, 0x316a}, // rieultikeutkorean
	{0xf66aa028, 0x215d}, // fiveeighths
	{0xf67e1ed1, 0x01a3}, // oi
	{0xf6886180, 0x3216}, // cieucaparenkorean
	{0xf68c8679, 0x3043}, // ismallhiragana
	{0xf68fb68d, 0x215b}, // oneeighth
	{0xf6909b76, 0x24a1}, // fparen
	{0xf69fb673, 0xf7e4}, // Adieresissmall
	{0xf6b386e5, 0x1e3c}, // Lcircumflexbelow
	{0xf6c0ec85, 0x1e5c}, // Rdotbelowmacron
	{0xf6ea45f8, 0xf6fa}, // OEsmall
	{0xf6f2a8b6, 0x0289}, // ubar
	{0xf7114d7b, 0x2502}, // SF110000
	{0xf7440454, 0x045e}, // afii10110
	{0xf757213f, 0x01d6}, // udieresismacron
	{0xf779fd74, 0x1ea0}, // Adotbelow
	{0xf7887f64, 0x24ba}, // Ecircle
	{0xf7994ed0, 0x0a6d}, // sevengurmukhi
	{0xf7c65164, 0x266d}, // musicflatsign
	{0xf7d4f2e3, 0x1e1e}, // Fdotaccent
	{0xf7ddf3cd, 0x0a17}, // gagurmukhi
	{0xf7de3a36, 0x3086}, // yuhiragana
	{0xf7edf1a3, 0x0403}, // Gjecyrillic
	{0xf7fe7207, 0x279e}, // arrowrightheavy
	{0xf7fec616, 0x1ec4}, // Ecircumflextilde
	{0xf81e4626, 0x0a02}, // bindigurmukhi
	{0xf8245f14, 0x30e6}, // yukatakana
	{0xf82ad190, 0x05e4}, // pehebrew
	{0xf8376f18, 0x0410}, // afii10017
	{0xf885c738, 0x30d3}, // bikatakana
	{0xf8868f94, 0xff8c}, // hukatakanahalfwidth
	{0xf8892150, 0x04a8}, // Haabkhasiancyrillic
	{0xf89a4fca, 0x05d3}, // daletholam
	{0xf89be814, 0x0abc}, // nuktagujarati
	{0xf8e483f7, 0x1eb5}, // abrevetilde
	{0xf8ef289b, 0x21df}, // pagedown
	{0xf90377b2, 0x3151}, // yakorean
	{0xf90f516b, 0x0a5b}, // zagurmukhi
	{0xf9190810, 0x0203}, // ainvertedbreve
	{0xf929be43, 0x0ab6}, // shagujarati
	{0xf93a01ea, 0xfe6a}, // percentsmall
	{0xf952cde5, 0x1eae}, // Abreveacute
	{0xf95ad1c7, 0x0065}, // e
	{0xf95b34b0, 0x0660}, // afii57392
	{0xf976011b, 0x3399}, // fmsquare
	{0xf99ebcf4, 0x25c1}, // whiteleftpointingtriangle
	{0xf9bdabb3, 0x00e2}, // acircumflex
	{0xf9d67642, 0x00c8}, // Egrave
	{0xf9e5170b, 0x1eb6}, // Abrevedotbelow
	{0xf9e8161d, 0x3206}, // siosparenkorean
	{0xf9eeaebc, 0x002c}, // comma
	{0xf9f4a348, 0x2279}, // notgreaternorless
	{0xf9f6f2fe, 0xfe84}, // alefhamzaabovefinalarabic
	{0xf9f909db, 0x09f1}, // ralowerdiagonalbengali
	{0xfa1f37f7, 0x207d}, // parenleftsuperior
	{0xfa3ebdeb, 0xfb8b}, // jehfinalarabic
	{0xfa46e08a, 0x010a}, // Cdotaccent
	{0xfa6e953f, 0x0e13}, // nonenthai
	{0xfa7aad8b, 0x0e34}, // saraithai
	{0xfa85b29d, 0x0ae6}, // zerogujarati
	{0xfa8f771c, 0xfe82}, // alefmaddaabovefinalarabic
	{0xfa932832, 0x1ec9}, // ihookabove
	{0xfa9cd43f, 0x2488}, // oneperiod
	{0xfa9f7510, 0x1e6a}, // Tdotaccent
	{0xfaa7f693, 0xff1b}, // semicolonmonospace
	{0xfab3dba6, 0x3138}, // ssangtikeutkorean
	{0xfac03db8, 0x015e}, // Scedilla
	{0xfac092ef, 0x24de}, // ocircle
	{0xfad44b21, 0x278d}, // fourcircleinversesansserif
	{0xfadde282, 0x0539}, // Toarmenian
	{0xfaf8abd4, 0x05bb}, // qubuts25
	{0xfb0a35fb, 0xfb2b}, // afii57695
	{0xfb0e1bad, 0x2567}, // SF450000
	{0xfb1373b2, 0x30b9}, // sukatakana
	{0xfb161300, 0x049f}, // kastrokecyrillic
	{0xfb206015, 0x30b1}, // kekatakana
	{0xfb2465d5, 0x0468}, // Yuslittleiotifiedcyrillic
	{0xfb3e0b48, 0x01fc}, // AEacute
	{0xfb4678bc, 0xfe90}, // behfinalarabic
	{0xfb5bf4b4, 0x02cf}, // acutelowmod
	{0xfb5cfdc8, 0x095c}, // dddhadeva
	{0xfb6edad4, 0xf6c9}, // Acute
	{0xfb764dd2, 0x21d3}, // arrowdbldown
	{0xfb7c1fd7, 0x0485}, // dasiapneumatacyrilliccmb
	{0xfbcf44c4, 0x018d}, // deltaturned
	{0xfbd1b93f, 0x33c4}, // squarecc
	{0xfbd50511, 0x04e8}, // Obarredcyrillic
	{0xfbf1fcde, 0x03e9}, // horicoptic
	{0xfc161b2f, 0x2499}, // eighteenperiod
	{0xfc18556b, 0x02bd}, // commareversedmod
	{0xfc1a2c97, 0x046c}, // Yusbigiotifiedcyrillic
	{0xfc2caf5c, 0x2285}, // notsuperset
	{0xfc3393bc, 0x0124}, // Hcircumflex
	{0xfc3a32c2, 0x0a8d}, // ecandragujarati
	{0xfc75d31c, 0x1ec6}, // Ecircumflexdotbelow
	{0xfc7e1ef8, 0x0952}, // anudattadeva
	{0xfc7ea01a, 0x04e6}, // Odieresiscyrillic
	{0xfc8020b6, 0xfb39}, // yoddageshhebrew
	{0xfc828b2d, 0x1ee4}, // Udotbelow
	{0xfc9cf271, 0x020d}, // odblgrave
	{0xfcd52169, 0xf7ed}, // Iacutesmall
	{0xfce47bc6, 0x05b7}, // patah
	{0xfce8ddc1, 0x1e0f}, // dlinebelow
	{0xfce9ddb5, 0x05aa}, // yerahbenyomohebrew
	{0xfcf6e2a9, 0x21ea}, // capslock
	{0xfd00e31a, 0x0303}, // tildecomb
	{0xfd0eac29, 0x0261}, // gscript
	{0xfd1397ce, 0x0412}, // Vecyrillic
	{0xfd166ead, 0x05e0}, // nunhebrew
	{0xfd2c8feb, 0x1ea4}, // Acircumflexacute
	{0xfd5ace9a, 0x057b}, // jheharmenian
	{0xfd6ac237, 0x221d}, // proportional
	{0xfd77296d, 0x04af}, // ustraightcyrillic
	{0xfd891a4c, 0x0948}, // aivowelsigndeva
	{0xfd8944f5, 0x0309}, // hookabovecomb
	{0xfd89977d, 0x0e50}, // zerothai
	{0xfd93a170, 0xf8ec}, // parenleftex
	{0xfd99bb06, 0xfb3c}, // lameddagesh
	{0xfdb6c57c, 0x2215}, // divisionslash
	{0xfdbbbec8, 0x090d}, // ecandradeva
	{0xfdc83f1f, 0x028d}, // wturned
	{0xfdd37935, 0x0422}, // afii10036
	{0xfdec640d, 0x0640}, // kashidaautoarabic
	{0xfdf32442, 0x3390}, // Hzsquare
	{0xfdf4c83e, 0x026e}, // lezh
	{0xfe3d55df, 0x064f}, // dammaarabic
	{0xfe407199, 0x2276}, // lessorgreater
	{0xfe7515f3, 0x03a9}, // Omegagreek
	{0xfe779a6a, 0x045a}, // afii10107
	{0xfea7088a, 0x0628}, // beharabic
	{0xfeb66fd9, 0xfec7}, // zahinitialarabic
	{0xfeb7f263, 0x0556}, // Feharmenian
	{0xfec7bc3b, 0x0651}, // shaddaarabic
	{0xfee2004d, 0x01fe}, // Ostrokeacute
	{0xfee5f25e, 0x2126}, // Omega
	{0xfee9d86c, 0x2295}, // pluscircle
	{0xfef651f8, 0x0688}, // afii57512
	{0xff5dadf4, 0x0193}, // Ghook
	{0xff7d5e86, 0x05e7}, // qoftserehebrew
	{0xff81c116, 0x21cd}, // arrowleftdblstroke
	{0xff8c00d8, 0x3386}, // MBsquare
	{0xff8f2931, 0x30ae}, // gikatakana
	{0xff90fc92, 0x0923}, // nnadeva
	{0xff94689d, 0x04c0}, // palochkacyrillic
	{0xffce1162, 0xf734}, // fouroldstyle
	{0xffe38169, 0x043f}, // pecyrillic
	{0xfffadc30, 0x2568}, // SF460000
};

