/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.testcase;

import java.io.InputStream;

import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.logger.LogKitLogger;
import org.apache.log.Logger;

import org.apache.avalon.excalibur.component.DefaultRoleManager;
import org.apache.avalon.excalibur.component.ExcaliburComponentManager;
import org.apache.avalon.excalibur.logger.DefaultLogKitManager;

import org.apache.log.Hierarchy;
import org.apache.log.LogTarget;
import org.apache.log.Priority;
import org.apache.log.format.PatternFormatter;
import org.apache.log.output.io.StreamTarget;

import junit.framework.TestCase;
import junit.framework.TestResult;
import junit.framework.AssertionFailedError;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.HashMap;
import java.net.URL;

/**
 * JUnit TestCase for Avalon Components
 * <p>
 *   This class will extends the JUnit TestCase class to setup an environment
 *   to easily test Avalon Components. The following methods and instance variables
 *   are exposed for convenience testing:
 * </p>
 * <dl>
 *   <dt>m_logPriority</dt>
 *   <dd>
 *     This variable contains the log priority of the default logger setup
 *     at startup of the test. Overwrite the initialize() method to set a
 *     different Priority.
 *   </dd>
 *   <dt>manager</dt>
 *   <dd>
 *     This instance variable contains a initialized ComponentManager
 *     according to the test case configuration file (see below)
 *   </dd>
 *   <dt>getLogger()</dt>
 *   <dd>
 *     This method returns the default logger for this test case
 *   </dd>
 * </dl>
 * <p>
 *   The test case configuration file has the following structure:
 * </p>
 * <pre>
 *   &lt;testcase&gt;
 *     &lt;annotation&gt;
 *       &lt;![CDATA[
 *         &lt;title&gt;DataSourceComponent tests&lt;/title&gt;
 *         &lt;para&gt;
 *           This is some &lt;emphasis&gt;text&lt;/emphasis&gt; to
 *           describethe tests made herein. Unfortunately the
 *           ConfigurationBuilder used to read the test case configuration
 *           file doesn't allow to have mixed context. This should be fixed
 *           in a future version .
 *         &lt;/para&gt;
 *       ]]&gt;
 *     &lt;/annotation&gt;
 *
 *     &lt;logkit&gt;
 *       &lt;factories&gt;
 *         &lt;factory type="file" class="org.apache.avalon.excalibur.logger.facatory.FileTargetFactory"/&gt;
 *         &lt;factory type="property-filter" class="org.apache.avalon.excalibur.logger.facatory.FilterTargetFactory"/&gt;
 *         &lt;factory type="servlet" class="org.apache.avalon.excalibur.logger.facatory.ServletTargetFactory"/&gt;
 *         ...
 *       &lt;/factories&gt;
 *
 *       &lt;targets&gt;
 *         &lt;file id="root"&gt;
 *           &lt;filename&gt;dev/logs/main.log&lt;/filename&gt;
 *         &lt;/file&gt;
 *         &lt;file id="classloader"&gt;
 *           &lt;filename&gt;dev/logs/classloader.log&lt;/filename&gt;
 *         &lt;/file&gt;
 *         &lt;priority-filter id="servlet" log-level="ERROR"&gt;
 *           &lt;servlet/&gt;
 *         &lt;/priority-filter&gt;
 *       &lt;/targets&gt;
 *
 *       &lt;categories&gt;
 *         &lt;category name="cocoon" log-level="INFO"&gt;
 *           &lt;log-target id-ref="root"/&gt;
 *           &lt;log-target id-ref="servlet"/&gt;
 *
 *           &lt;category name="classloader" log-level="DEBUG"&gt;
 *             &lt;log-target id-ref="classloader"/&gt;
 *           &lt;/category&gt;
 *         &lt;/category&gt;
 *       &lt;/categories&gt;
 *     &lt;/logkit&gt;
 *
 *     &lt;context&gt;
 *       &lt;entry name="foo" value="bar"/&gt;
 *       &lt;entry name="baz" class="my.context.Class"/&gt;
 *     &lt;/context&gt;
 *
 *     &lt;roles&gt;
 *       &lt;role name="org.apache.avalon.excalibur.datasource.DataSourceComponentSelector"
 *             shorthand="datasources"
 *             default-class="org.apache.avalon.excalibur.component.ExcaliburComponentSelector"&gt;
 *          &lt;hint shorthand="jdbc" class="org.apache.avalon.excalibur.datasource.JdbcDataSource"/&gt;
 *       &lt;/role&gt;
 *     &lt;/roles&gt;
 *
 *     &lt;components&gt;
 *       &lt;datasources&gt;
 *         &lt;jdbc name="personell"&gt;
 *           &lt;pool-controller min="5" max="10"/&gt;
 *           &lt;jdbc name="personnel"/&gt;
 *           &lt;dburl&gt;jdbc:odbc:test&lt;/dburl&gt;
 *           &lt;user&gt;test&lt;/user&gt;
 *           &lt;password&gt;test&lt;/password&gt;
 *           &lt;driver&gt;sun.jdbc.odbc.JdbcOdbcDriver&lt;/driver&gt;
 *         &lt;/jdbc&gt;
 *       &lt;/datasources&gt;
 *     &lt;/components&gt;
 *   &lt;/testcase&gt;
 * </pre>
 *
 * @author <a href="mailto:giacomo@apache.org">Giacomo Pati</a>
 * @version $Id: ExcaliburTestCase.java,v 1.15 2001/12/11 09:53:32 jefft Exp $
 */
public class ExcaliburTestCase
    extends TestCase
{
    ///Format of default formatter
    private static final String  FORMAT =
        "%7.7{priority} %5.5{time}   [%8.8{category}] (%{context}): %{message}\\n%{throwable}";

    //The default logger
    private Logger                      m_logger;
    private LogKitLogger                m_logEnabledLogger;
    private ExcaliburComponentManager   m_manager;
    private static HashMap              m_tests = new HashMap();

    protected Priority                  m_logPriority = Priority.DEBUG;
    protected ComponentManager          manager;

    public ExcaliburTestCase( final String name )
    {
        super( name );

        ArrayList methodList = (ArrayList) ExcaliburTestCase.m_tests.get(this.getClass());

        Method[] methods = this.getClass().getDeclaredMethods();

        if ( null == methodList )
        {
            methodList = new ArrayList( methods.length );

            for ( int i = 0; i < methods.length; i++ )
            {
                String methodName = methods[i].getName();
                if (methodName.startsWith( "test" ) &&
                    ( Modifier.isPublic( methods[i].getModifiers() ) ) &&
                    ( methods[i].getReturnType().equals( Void.TYPE ) ) &&
                    ( methods[i].getParameterTypes().length == 0 ) )
                {
                    methodList.add(methodName);
                }
            }

            ExcaliburTestCase.m_tests.put( this.getClass(), methodList );
        }
    }

    /** Return the logger */
    protected Logger getLogger()
    {
        if ( null == m_logger )
        {
            m_logger = this.setupLogger();
        }

        return m_logger;
    }

    /** Return the logger */
    protected LogKitLogger getLogEnabledLogger()
    {
        if ( null == m_logEnabledLogger )
        {
            m_logEnabledLogger = new LogKitLogger( this.getLogger() );
        }

        return m_logEnabledLogger;
    }

    /**
     * Initializes the ComponentManager
     *
     * The configuration file is determined by the class name plus .xtest appended,
     * all '.' replaced by '/' and loaded as a resource via classpath
     */
    protected void prepare()
        throws Exception
    {
        final String resourceName = this.getClass().getName().replace( '.', '/' ) + ".xtest";
        URL resource = this.getClass().getClassLoader().getResource( resourceName );
        if ( resource != null ) {
            getLogger().debug("Loading resource " + resourceName);
            prepare( resource.openStream() );
        }
        else
            getLogger().debug("Resource not found " + resourceName);
    }

    /**
     * Initializes the ComponentManager
     *
     * @param testconf The configuration file is passed as a <code>InputStream</code>
     *
     * A common way to supply a InputStream is to overwrite the initialize() method
     * in the sub class, do there whatever is needed to get the right InputStream object
     * supplying a conformant xtest configuartion and pass it to this initialize method.
     * the mentioned initialize method is also the place to set a different logging priority
     * to the member variable m_logPriority.
     */
    protected final void prepare( final InputStream testconf )
        throws Exception
    {
        getLogger().debug( "ExcaliburTestCase.initialize" );

        final DefaultConfigurationBuilder builder = new DefaultConfigurationBuilder();
        final Configuration conf = builder.build( testconf );

        String annotation = conf.getChild( "annotation" ).getValue( null );

        if ( ( null != annotation ) || !( "".equals( annotation ) ) )
        {
            m_logger.info( annotation );
        }

        Context context = setupContext( conf.getChild( "context" ) );

        m_manager = setupComponentManager( conf.getChild( "components" ),
                                           conf.getChild( "roles" ),
                                           conf.getChild( "logkit" ),
                                           context );
        manager = m_manager;
    }

    /**
     * Disposes the <code>ComponentManager</code>
     */
    final private void done()
    {
        if ( null != m_manager )
        {
            m_manager.dispose();
        }

        m_manager = null;
    }

    /**
     * Override <code>run</code> so that we can have code that is run once.
     */
    final public void run(TestResult result)
    {
        ArrayList methodList = (ArrayList) ExcaliburTestCase.m_tests.get( this.getClass() );

        if ( null == methodList || methodList.isEmpty() )
        {
            return; // The test was already run!  NOTE: this is a hack.
        }

        try
        {
            if (this instanceof Initializable)
            {
                ((Initializable)this).initialize();
            }

            this.prepare();

            Iterator tests = methodList.iterator();

            while ( tests.hasNext() )
            {
                String methodName = (String) tests.next();
                this.setName( methodName );
                m_logger = null;

                if ( this.getLogger().isDebugEnabled() )
                {
                    this.getLogger().debug("Now running the following test: " + methodName);
                }

                super.run(result);
            }

        }
        catch ( Exception e )
        {
            result.addFailure(this, new AssertionFailedError());
        }
        finally
        {
            this.done();

            if (this instanceof Disposable)
            {
                try
                {
                    ((Disposable)this).dispose();
                }
                catch( Exception e )
                {
                    result.addFailure(this, new AssertionFailedError("Disposal Error"));
                }
            }
        }

        methodList.clear();
        ExcaliburTestCase.m_tests.put( this.getClass(), methodList );
    }

    /**
     * Set up logger configuration
     */
    final private Logger setupLogger()
    {
        //FIXME(GP): This method should setup a LogConfigurator and LogManager
        //           according to the configuration spec. not yet completed/implemented
        //           It will return a default logger for now.
        final org.apache.log.Logger logger = Hierarchy.getDefaultHierarchy().getLoggerFor( getName() );
        logger.setPriority( m_logPriority );

        final PatternFormatter formatter = new PatternFormatter( FORMAT );
        final StreamTarget target = new StreamTarget( System.out, formatter );
        logger.setLogTargets( new LogTarget[] { target } );

        return logger;
    }

    /**
     * set up a context according to the xtest configuration specifications context
     * element.
     *
     * A method addContext(DefaultContext context) is called here to enable subclasses
     * to put additional objects into the context programmatically.
     */
    final private Context setupContext( final Configuration conf )
        throws Exception
    {
        //FIXME(GP): This method should setup the Context object according to the
        //           configuration spec. not yet completed
        final DefaultContext context = new DefaultContext();
        final Configuration [] confs = conf.getChildren( "entry" );
        for (int i = 0; i < confs.length; i++)
        {
            final String key = confs[i].getAttribute( "name" );
            final String value = confs[i].getAttribute( "value", null );
            if (value == null) {
                String clazz = confs[i].getAttribute( "class" );
                Object obj = this.getClass().getClassLoader().loadClass( clazz ).newInstance();
                context.put( key, obj );
                if (getLogger().isInfoEnabled())
                    getLogger().info( "ExcaliburTestCase: added an instance of class " + clazz + " to context entry " + key);
            } else {
                context.put( key, value );
                if (getLogger().isInfoEnabled())
                    getLogger().info( "ExcaliburTestCase: added value \""  + value + "\" to context entry " + key);
            }
        }
        addContext( context );
        return( context );
    }

    /**
     * This method may be overwritten by subclasses to put additional objects
     * into the context programmatically.
     */
    protected void addContext( DefaultContext context )
    {
    }

    final private ExcaliburComponentManager setupComponentManager ( final Configuration confCM,
                                                                    final Configuration confRM,
                                                                    final Configuration confLM,
                                                                    final Context context )
        throws Exception
    {
        final DefaultRoleManager roleManager = new DefaultRoleManager();
        roleManager.setLogger( getLogger() );
        roleManager.configure( confRM );

        final DefaultLogKitManager logKitManager = new DefaultLogKitManager();
        logKitManager.enableLogging( getLogEnabledLogger() );
        logKitManager.contextualize( context );
        logKitManager.configure( confLM );

        final ExcaliburComponentManager manager = new ExcaliburComponentManager();
        manager.setLogger( getLogger() );
        manager.setRoleManager( roleManager );
        manager.contextualize( context );
        manager.setLogKitManager( logKitManager );
        manager.configure( confCM );
        manager.initialize();
        return manager;
    }
}
