/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.extension;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.jar.Attributes.Name;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import org.apache.avalon.excalibur.util.DeweyDecimal;
import org.apache.avalon.excalibur.util.StringUtil;

/**
 * <p>Utility class that represents either an available "Optional Package"
 * (formerly known as "Standard Extension") as described in the manifest
 * of a JAR file, or the requirement for such an optional package.</p>
 *
 * <p>For more information about optional packages, see the document
 * <em>Optional Package Versioning</em> in the documentation bundle for your
 * Java2 Standard Edition package, in file
 * <code>guide/extensions/versioning.html</code>.</p>
 *
 * @author <a href="mailto:craigmcc@apache.org">Craig R. McClanahan</a>
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 * @version $Revision: 1.8 $ $Date: 2001/12/11 09:53:34 $
 */
public final class Extension
{
    public static final Compatability COMPATIBLE = 
        new Compatability( "COMPATIBLE" );
    public static final Compatability REQUIRE_SPECIFICATION_UPGRADE = 
        new Compatability( "REQUIRE_SPECIFICATION_UPGRADE" );
    public static final Compatability REQUIRE_VENDOR_SWITCH = 
        new Compatability( "REQUIRE_VENDOR_SWITCH" );
    public static final Compatability REQUIRE_IMPLEMENTATION_UPGRADE = 
        new Compatability( "REQUIRE_IMPLEMENTATION_UPGRADE" );
    public static final Compatability INCOMPATIBLE = 
        new Compatability( "INCOMPATIBLE" );

    /**
     * The name of the optional package being made available, or required.
     */
    private String        m_extensionName;

    /**
     * The version number (dotted decimal notation) of the specification
     * to which this optional package conforms.
     */
    private DeweyDecimal  m_specificationVersion;

    /**
     * The name of the company or organization that originated the
     * specification to which this optional package conforms.
     */
    private String        m_specificationVendor;

    /**
     * The URL from which the most recent version of this optional package
     * can be obtained if it is not already installed.
     */
    private String        m_implementationURL;

    /**
     * The name of the company or organization that produced this
     * implementation of this optional package.
     */
    private String        m_implementationVendor;

    /**
     * The unique identifier of the company that produced the optional
     * package contained in this JAR file.
     */
    private String        m_implementationVendorId;

    /**
     * The version number (dotted decimal notation) for this implementation
     * of the optional package.
     */
    private DeweyDecimal  m_implementationVersion;


    /**
     * Return an array of <code>Extension</code> objects representing optional
     * packages that are available in the JAR file associated with the
     * specified <code>Manifest</code>.  If there are no such optional
     * packages, a zero-length array is returned.
     *
     * @param manifest Manifest to be parsed
     */
    public static Extension[] getAvailable( final Manifest manifest )
    {
        if( null == manifest ) return new Extension[ 0 ];

        final ArrayList results = new ArrayList();

        final Attributes mainAttributes = manifest.getMainAttributes();
        if( null != mainAttributes )
        {
            final Extension extension = getExtension( "", mainAttributes );
            if( null != extension ) results.add( extension );
        }

        final Map entries = manifest.getEntries();
        final Iterator keys = entries.keySet().iterator();
        while( keys.hasNext() )
        {
            final String key = (String)keys.next();
            final Attributes attributes = (Attributes)entries.get( key );
            final Extension extension = getExtension( "", attributes );
            if( null != extension ) results.add( extension );
        }

        return (Extension[])results.toArray( new Extension[ 0 ] );
    }

    /**
     * Return the set of <code>Extension</code> objects representing optional
     * packages that are required by the application contained in the JAR
     * file associated with the specified <code>Manifest</code>.  If there
     * are no such optional packages, a zero-length list is returned.
     *
     * @param manifest Manifest to be parsed
     */
    public static Extension[] getRequired( final Manifest manifest )
    {
        final ArrayList results = new ArrayList();
        final Attributes mainAttributes = manifest.getMainAttributes();

        if( null != mainAttributes )
        {
            getRequired( mainAttributes, results );
        }

        final Map entries = manifest.getEntries();
        final Iterator keys = entries.keySet().iterator();
        while( keys.hasNext() )
        {
            final String key = (String)keys.next();
            final Attributes attributes = (Attributes)entries.get( key );
            getRequired( mainAttributes, results );
        }

        return (Extension[])results.toArray( new Extension[ 0 ] );
    }

    /**
     * Add required optional packages defined in the specified attributes entry, if any.
     *
     * @param attributes Attributes to be parsed
     * @param required list to add required optional packages to
     */
    private static void getRequired( final Attributes attributes,
                                     final ArrayList required )
    {
        final String names = attributes.getValue( Name.EXTENSION_LIST );
        if( null == names ) return;

        final String[] extentions = StringUtil.split( names, " " );
        for( int i = 0; i < extentions.length; i++ )
        {
            final String prefix = extentions[ i ] + "-";
            final Extension extension = getExtension( prefix, attributes );

            if( null != extension )
            {
                required.add( extension );
            }
        }
    }

    /**
     * Extract an Extension from Attributes.
     * Prefix indicates the prefix checked for each string.
     * Usually the prefix is <em>"&lt;extension&gt;-"</em> if looking for a
     * <b>Required</b> extension. If you are looking for an <b>Available</b> extension
     * then the prefix is <em>""</em>.
     *
     * @param prefix the prefix for each attribute name
     * @param attributes Attributes to searched
     * @return the new Extension object, or null
     */
    private static Extension getExtension( final String prefix, final Attributes attributes )
    {
        //WARNING: We trim the values of all the attributes because
        //Some extension declarations are badly defined (ie have spaces 
        //after version or vendorID)
        final String name = getTrimmedString( attributes.getValue( prefix + Name.EXTENSION_NAME ) );
        if( null == name ) return null;

        final String specVendor = 
            getTrimmedString( attributes.getValue( prefix + Name.SPECIFICATION_VENDOR ) );
        final String specVersion = 
            getTrimmedString( attributes.getValue( prefix + Name.SPECIFICATION_VERSION ) );

        final String impVersion = 
            getTrimmedString( attributes.getValue( prefix + Name.IMPLEMENTATION_VERSION ) );
        final String impVendor = 
            getTrimmedString( attributes.getValue( prefix + Name.IMPLEMENTATION_VENDOR ) );
        final String impVendorId = 
            getTrimmedString( attributes.getValue( prefix + Name.IMPLEMENTATION_VENDOR_ID ) );
        final String impURL = 
            getTrimmedString( attributes.getValue( prefix + Name.IMPLEMENTATION_URL ) );

        return new Extension( name, specVersion, specVendor, impVersion,
                              impVendor, impVendorId, impURL );
    }

    private static String getTrimmedString( final String value )
    {
        if( null == value ) return null;
        else
        {
            return value.trim();
        }
    }

    /**
     * The constructor to create Extension object.
     * Note that every component is allowed to be specified
     * but only the extensionName is mandatory.
     *
     * @param extensionName the name of extension.
     * @param specificationVersion the specification Version of extension.
     * @param specificationVendor the specification Vendor of extension.
     * @param implementationVersion the implementation Version of extension.
     * @param implementationVendor the implementation Vendor of extension.
     * @param implementationVendorId the implementation VendorId of extension.
     * @param implementationURL the implementation URL of extension.
     */
    public Extension( final String extensionName,
                      final String specificationVersion,
                      final String specificationVendor,
                      final String implementationVersion,
                      final String implementationVendor,
                      final String implementationVendorId,
                      final String implementationURL )
    {
        m_extensionName = extensionName;
        m_specificationVendor = specificationVendor;

        if( null != specificationVersion )
        {
            try
            {
                m_specificationVersion = new DeweyDecimal( specificationVersion );
            }
            catch( NumberFormatException nfe )
            {
                final String error = "Bad specification version format '" + specificationVersion + 
                    "' in '" + extensionName + "'. (Reason: " + nfe + ")";
                throw new IllegalArgumentException( error );
            }
        }

        m_implementationURL = implementationURL;
        m_implementationVendor = implementationVendor;
        m_implementationVendorId = implementationVendorId;

        if( null != implementationVersion )
        {
            try
            {
                m_implementationVersion = new DeweyDecimal( implementationVersion );
            }
            catch( NumberFormatException nfe )
            {
                final String error = "Bad implementation version format '" + implementationVersion +
                    "' in '" + extensionName + "'. (Reason: " + nfe + ")";
                throw new IllegalArgumentException( error );
            }
        }

        if( null == m_extensionName )
        {
            throw new NullPointerException( "extensionName property is null" );
        }
    }

    public String getExtensionName()
    {
        return m_extensionName;
    }

    public String getSpecificationVendor()
    {
        return m_specificationVendor;
    }

    public DeweyDecimal getSpecificationVersion()
    {
        return m_specificationVersion;
    }

    public String getImplementationURL()
    {
        return m_implementationURL;
    }

    public String getImplementationVendor()
    {
        return m_implementationVendor;
    }

    public String getImplementationVendorId()
    {
        return m_implementationVendorId;
    }

    public DeweyDecimal getImplementationVersion()
    {
        return m_implementationVersion;
    }


    /**
     * Return a Compatibility enum indicating the relationship of this
     * <code>Extension</code> with the specified <code>Extension</code>.
     *
     * @param required Description of the required optional package
     */
    public Compatability getCompatibilityWith( final Extension required )
    {
        // Extension Name must match
        if( false == m_extensionName.equals( required.getExtensionName() ) )
        {
            return INCOMPATIBLE;
        }

        // Available specification version must be >= required
        final DeweyDecimal specificationVersion = required.getSpecificationVersion();
        if( null != specificationVersion )
        {
            if( null == m_specificationVersion ||
                false == isCompatible( m_specificationVersion, specificationVersion ) )
            {
                return REQUIRE_SPECIFICATION_UPGRADE;
            }
        }

        // Implementation Vendor ID must match
        final String implementationVendorId = required.getImplementationVendorId();
        if( null != implementationVendorId )
        {
            if( null == m_implementationVendorId ||
                false == m_implementationVendorId.equals( implementationVendorId ) )
            {
                return REQUIRE_VENDOR_SWITCH;
            }
        }

        // Implementation version must be >= required
        final DeweyDecimal implementationVersion = required.getImplementationVersion();
        if( null != implementationVersion )
        {
            if( null == m_implementationVersion ||
                false == isCompatible( m_implementationVersion, implementationVersion ) )
            {
                return REQUIRE_IMPLEMENTATION_UPGRADE;
            }
        }

        // This available optional package satisfies the requirements
        return COMPATIBLE;
    }

    /**
     * Return <code>true</code> if the specified <code>Extension</code>
     * (which represents an optional package required by an application)
     * is satisfied by this <code>Extension</code> (which represents an
     * optional package that is already installed.  Otherwise, return
     * <code>false</code>.
     *
     * @param required Description of the required optional package
     */
    public boolean isCompatibleWith( final Extension required )
    {
        return ( COMPATIBLE == getCompatibilityWith( required ) );
    }

    /**
     * Return a String representation of this object.
     */
    public String toString()
    {
        final StringBuffer sb = new StringBuffer( "Extension[" );
        sb.append( m_extensionName );

        if( null != m_implementationURL )
        {
            sb.append( ", implementationURL=" );
            sb.append( m_implementationURL );
        }

        if( null != m_implementationVendor )
        {
            sb.append( ", implementationVendor=" );
            sb.append( m_implementationVendor );
        }

        if( null != m_implementationVendorId )
        {
            sb.append( ", implementationVendorId=" );
            sb.append( m_implementationVendorId );
        }

        if( null != m_implementationVersion )
        {
            sb.append( ", implementationVersion=" );
            sb.append( m_implementationVersion );
        }

        if( null != m_specificationVendor )
        {
            sb.append( ", specificationVendor=" );
            sb.append( m_specificationVendor );
        }

        if( null != m_specificationVersion )
        {
            sb.append( ", specificationVersion=" );
            sb.append( m_specificationVersion );
        }

        sb.append( "]" );

        return sb.toString();
    }

    /**
     * Return <code>true</code> if the first version number is greater than
     * or equal to the second; otherwise return <code>false</code>.
     *
     * @param first First version number (dotted decimal)
     * @param second Second version number (dotted decimal)
     *
     * @exception NumberFormatException on a malformed version number
     */
    private boolean isCompatible( final DeweyDecimal first, final DeweyDecimal second )
    {
        return first.isGreaterThanOrEqual( second );
    }

    public static final class Compatability
    {
        private final String        m_name;

        protected Compatability( final String name )
        {
            m_name = name;
        }

        public String toString()
        {
            return m_name;
        }
    }
}
