/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.extension.test;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.jar.Manifest;
import junit.framework.TestCase;
import org.apache.avalon.excalibur.extension.DefaultPackageRepository;
import org.apache.avalon.excalibur.extension.Extension;
import org.apache.avalon.excalibur.extension.OptionalPackage;
import org.apache.avalon.excalibur.extension.PackageManager;
import org.apache.avalon.excalibur.extension.PackageRepository;

/**
 * TestCases for PackageRepository.
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 * @version $Revision: 1.9 $ $Date: 2001/12/11 09:53:35 $
 */
public class PackageRepositoryTestCase
    extends TestCase
{
    private File m_baseDirectory;
    private File m_pathElement1;
    private File m_pathElement2;
    private File[] m_path;

    public PackageRepositoryTestCase( String name )
        throws IOException
    {
        super( name );

        m_baseDirectory = 
            (new File( "../../src/scratchpad/org/apache/avalon/excalibur/extension/test/" )).getCanonicalFile();
        
        m_pathElement1 = new File( m_baseDirectory, "path1" );
        m_pathElement2 = new File( m_baseDirectory, "path2" );
        m_path = new File[] { m_pathElement1, m_pathElement2 };
    }

    public void testGoodPath()
        throws Exception
    {
        final DefaultPackageRepository repository = new DefaultPackageRepository( m_path );
    }

    public void testBadPath()
        throws Exception
    {
        try
        {
            final File pathElement3 = new File( m_baseDirectory, "path3" );
            final File[] path = new File[] { m_pathElement1, m_pathElement2, pathElement3 };
            final DefaultPackageRepository repository = 
                new DefaultPackageRepository( path );
        }
        catch( final IllegalArgumentException iae )
        {
            return;
        }
        
        assertTrue( "Exceptected to fail with bad path element", false );
    }

    public void testBasicScanDependencies()
        throws Exception
    {
        final PackageRepository repository = newPackagerepository();
        doRepositoryTest( repository );
    }

    public void testFSScanDependencies()
        throws Exception
    {
        final PackageRepository repository = new DefaultPackageRepository( m_path );
        doRepositoryTest( repository );
    }
    
    private void doRepositoryTest( final PackageRepository repository )
        throws Exception
    {
        final PackageManager manager = new PackageManager( repository );

        final Manifest manifest2 = getManifest( "manifest-2.mf" );
        final Extension extension1 = Extension.getRequired( manifest2 )[ 0 ];

        final ArrayList dependencies = new ArrayList();
        final ArrayList unsatisfied = new ArrayList();

        manager.scanDependencies( extension1, new Extension[ 0 ], dependencies, unsatisfied );
        System.out.println( "dependencies: " + dependencies );
        
        assertEquals( "dependencies Count", 2, dependencies.size() );
        assertEquals( "unsatisfied Count", 0, unsatisfied.size() );

        final int size = dependencies.size();
        for( int i = 0; i < size; i++ )
        {
            final OptionalPackage optionalPackage = (OptionalPackage)dependencies.get( i );
            final Extension[] extensions = optionalPackage.getAvailableExtensions();
            for( int j = 0; j < extensions.length; j++ )
            {
                final String name = extensions[ j ].getExtensionName();
                if( !name.equals( "excalibur.required1" ) &&
                    !name.equals( "excalibur.required2" ) )
                {
                    assertTrue( "Unexpected extension: " + name, false );
                }                
            }
        }
    }

    private Manifest getManifest( final String name )
        throws Exception
    {
        final InputStream inputStream = getClass().getResourceAsStream( name );
        return new Manifest( inputStream );
    }

    private PackageRepository newPackagerepository()
        throws Exception
    {
        final TestPackageRepository repository = new TestPackageRepository();
        repository.addEntry( "manifest-1.mf" );
        repository.addEntry( "manifest-2.mf" );
        repository.addEntry( "manifest-3.mf" );
        repository.addEntry( "manifest-4.mf" );
        repository.addEntry( "manifest-5.mf" );
        repository.addEntry( "manifest-6.mf" );
        repository.addEntry( "manifest-7.mf" );
        repository.addEntry( "manifest-8.mf" );
        return repository;
    }
}

class TestPackageRepository
    extends DefaultPackageRepository
{
    TestPackageRepository()
        throws Exception
    {
        super( new File[ 0 ] );
    }

    void addEntry( final String manifestLocation )
        throws Exception
    {
        final InputStream inputStream = getClass().getResourceAsStream( manifestLocation );
        final Manifest manifest = new Manifest( inputStream );
        final File file = new File( manifestLocation );
        final Extension[] available = Extension.getAvailable( manifest );
        final Extension[] required = Extension.getRequired( manifest );

        cacheOptionalPackage( new OptionalPackage( file, available, required ) );
    }
}
