/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.vfs;

import java.io.InputStream;
import java.io.IOException;

/**
 * File on virtual file system.
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 */
public final class VFile
{
    ///Complete Path to file
    private final String           m_path;

    ///Underlying resource that if VFileAccessor specific
    private Object                 m_resource;

    ///VFileAccessor via which file is accessed
    private VFileAccessor          m_accessor;

    ///Virtual file system in which this VFile is mounted
    private VFileSystem            m_filesystem;

    /**
     * Constructor for creating a directory entry.
     *
     * @param path the path to directory
     * @param filesystem the filesystem VFile belongs to
     */
    protected VFile( final String path, final VFileSystem filesystem )
    {
        this( path, filesystem, null, null );
    }

    /**
     * Constructor for creating a vfile.
     *
     * @param path the path to directory
     * @param filesystem the filesystem VFile belongs to
     * @param resource resource represented by vfile. Must be non-null if accessor non-null.
     * @param accessor accessor to access vfile. Must be non-null if resource non-null.
     * @exception IllegalArgumentException if an error occurs
     */
    protected VFile( final String path, 
                     final VFileSystem filesystem,                     
                     final Object resource,
                     final VFileAccessor accessor )
        throws IllegalArgumentException
    {
        m_path = path;
        m_filesystem = filesystem;
        m_resource = resource;
        m_accessor = accessor;

        if( null == resource || null == accessor )
        {
            if( null != resource && null != accessor )
            {
                throw new IllegalArgumentException( "Resource and accessor must " +
                                                    "be both null or non-null" );
            }
        }
    }

    /**
     * Get path to file.
     *
     * @return the path to file
     */
    public String getPath()
    {
        return m_path;
    }

    /**
     * Get name of file.
     *
     * @return the name of file
     */
    public String getName()
    { 
        final int index = m_path.lastIndexOf( '/' );
        if( -1 == index )
        {
            return m_path;
        }
        else
        {
            return m_path.substring( index );
        }
    }

    public VFile getParent()
        throws IOException
    { 
        checkValid();
        return m_filesystem.get( getParentsName() ); 
    }

    public VFile[] list()
        throws IOException
    { 
        checkValid();
        return m_filesystem.list( this ); 
    }

    public VFile[] list( final VFileFilter filter )
        throws IOException
    { 
        checkValid();
        return m_filesystem.list( this, filter ); 
    }

    public boolean isDirectory()
    {
        return ( null == m_resource );
    }

    public boolean isFile()
    {
        return ( null != m_resource );
    }

    public long getSize()
        throws IOException
    {
        checkValid();
        if( isDirectory() ) return 0;
        else
        {
            return m_accessor.getSize( this, m_resource );
        }
    }

    public InputStream getInputStream()
        throws IOException
    {
        checkValid();
        if( isDirectory() ) return null;
        else
        {
            return m_accessor.getInputStream( this, m_resource );
        }
    }

    public boolean isValid()
    {
        return ( null != m_filesystem );
    }

    /**
     * Invalidate VFile and free all resources.
     */
    protected void invalidate()
    {
        m_filesystem = null;
        m_resource = null;
        m_accessor = null;
    }

    /**
     * Helper to make sure VFile is valid before proceeding.
     * It will throw an IOException if VFile invalid.
     *
     * @exception IOException if an error occurs
     */
    private void checkValid()
        throws IOException
    {
        if( !isValid() )
        {
            throw new IOException( "Invalid VFile" );
        }
    }

    /**
     * Helper method to get parents filename.
     *
     * @return the parents filename.
     */
    private String getParentsName()
    { 
        final int index = m_path.lastIndexOf( '/' );
        if( -1 == index )
        {
            return m_path;
        }
        else
        {
            return m_path.substring( 0, index );
        }
    }
}
