/*
 * $Id: libgnuvd.h,v 1.3 2002/01/09 23:21:35 binnema Exp $
 */

/*
** Copyright (C) 2001 Dirk-Jan C. Binnema <djcb@djcbsoftware.nl>
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#ifndef _LIBGNUVD_H_
#define _LIBGNUVD_H_

#include <buffer.h>


/*
 * enumeration of VD errors
 */
enum VDError {
	VD_ERR_OK =  0,
	VD_ERR_NOT_FOUND,
        VD_ERR_CONNECT,
	VD_ERR_SOCKET,
	VD_ERR_PROXY,
        VD_ERR_DNS,
	VD_ERR_READ,
	VD_ERR_HTTP,
	VD_ERR_MEM,
	VD_ERR_POINTER,
        VD_ERR_PARAM
};

typedef unsigned int VDErrorType;


/*
 * VD exception structure
 */
struct _VDException {
	unsigned int _err;
	char*        _msg;
	int 	     _owner;
};


/*
 * VD query structure
 */
struct _VDQuery {
	char*	    _search_term;
	char*	    _error;
	Buffer*     _result_buffer;
};


typedef struct _VDQuery VDQuery;
typedef struct _VDException VDException;


/* VDQuery functions */

/*
 * vd_query_new:
 * create a new vd query
 *
 * word:      the word you are looking for (\0 terminated)
 * exception: out parameter receiving exception info
 *
 * returns: ptr to a VDQuery, or NULL if it fails
 *
 * preconditions: word != NULL &&
 *                ex   != NULL 
 */
VDQuery*       vd_query_new     (const char* word, VDException *ex);



/*
 * vd_query_destroy:
 * destroy a vd_query (freeing allocated memory)
 *
 * precondition: vd_query != NULL
 */
void           vd_query_destroy (VDQuery *vd_query);

/*
 * vd_query_perform:
 * perform the vd query over the internet
 * 
 * vd_query: a vd query
 * ex      : a vd exception, receiving error information
 *
 * return an error code (VDError)
 * 
 * preconditions: vd_query != NULL &&
 *                ex       != NULL
 */
VDErrorType        vd_query_perform (const VDQuery *vd_query, VDException *ex);



/* vd_query_results:
 * return the results of the query
 *
 * vd_query: a query object
 * ex:       an exception object receiving error information
 *
 * returns: a Buffer object with the results
 *
 * preconditions: vd_query != NULL &&
 *                vd_exception != NULL
 */
const Buffer*  vd_query_results (const VDQuery *vd_query, VDException *ex);



/* vd_query_search_term:
 * return the searchterm for the query
 *
 * vd_query: a query object
 *
 * returns: a char ptr with the search term
 *
 * preconditions: vd_query != NULL &&
 */
const char *vd_query_search_term (const VDQuery *vd_query);




/* VDException functions */

/*
 * vd_exception_new:
 * create a new vd_exception object
 *
 * returns: a ptr to a new VDException if succeeded, NULL otherwise
 */
VDException*   vd_exception_new     (void);


/*
 * vd_exception_destroy:
 * destroy a vd_exception object, free allocated memory
 *
 * ex: a vd exception
 * 
 * precondition: ex != NULL
 */
void vd_exception_destroy (VDException *ex);


#endif /*_LIBGNUVD_H_*/
