#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/Xm.h>
#include <Xm/Transfer.h>
#include <Xm/TransferP.h>
#include <Xm/DragIcon.h>

#include "ida.h"
#include "loader.h"
#include "viewer.h"
#include "xwd.h"
#include "xdnd.h"
#include "selections.h"

Atom TARGETS;
Atom _MOTIF_EXPORT_TARGETS;
Atom _MOTIF_CLIPBOARD_TARGETS;
Atom _MOTIF_LOSE_SELECTION;
Atom XA_FILE_NAME, XA_FILE, XA_BACKGROUND, XA_FOREGROUND, XA_PIXEL;
Atom _NETSCAPE_URL;
Atom MIME_TEXT_URI_LIST;
Atom MIME_IMAGE_PPM;
Atom MIME_IMAGE_PGM;
Atom MIME_IMAGE_XPM;
Atom MIME_IMAGE_BMP;
Atom MIME_IMAGE_JPEG;
Atom MIME_IMAGE_GIF;
Atom MIME_IMAGE_PNG;
Atom MIME_IMAGE_TIFF;

/* ---------------------------------------------------------------------- */
/* send data (drags, copy)                                                */

struct sel_data {
    struct ida_image img;
    Pixmap pixmap;
    char *filename;
    Pixmap icon_pixmap;
    Widget icon_widget;
};

static struct sel_data primary;
static struct sel_data dragndrop;

static void
iconify(Widget widget, struct sel_data *data)
{
    struct ida_image small;
    int scale,x,y,depth;
    char *src,*dst;
    Arg args[4];
    Cardinal n=0;

    /* calc size */
    memset(&small,0,sizeof(small));
    for (scale = 1;; scale++) {
	small.i.width  = data->img.i.width  / scale;
	small.i.height = data->img.i.height / scale;
	if (small.i.width < 128 && small.i.height < 128)
	    break;
    }

    /* scale down & create pixmap */
    dst = small.data = malloc(small.i.width * small.i.height * 3);
    for (y = 0; y < small.i.height; y++) {
	src = data->img.data + 3 * y * scale * data->img.i.width;
	for (x = 0; x < small.i.width; x++) {
	    dst[0] = src[0];
	    dst[1] = src[1];
	    dst[2] = src[2];
	    dst += 3;
	    src += 3*scale;
	}
    }
    data->icon_pixmap = image_to_pixmap(&small);

    /* build DnD icon */
    n = 0;
    depth = DefaultDepthOfScreen(XtScreen(widget));
    XtSetArg(args[n], XmNpixmap, data->icon_pixmap); n++;
    XtSetArg(args[n], XmNwidth,  small.i.width); n++;
    XtSetArg(args[n], XmNheight, small.i.height); n++;
    XtSetArg(args[n], XmNdepth,  depth); n++;
    data->icon_widget = XmCreateDragIcon(widget,"dragicon",args,n);
    
    free(small.data);
}

static void
sel_free(struct sel_data *data)
{
    if (data->filename) {
	unlink(data->filename);
	free(data->filename);
    }
    if (data->icon_widget)
	XtDestroyWidget(data->icon_widget);
    if (data->icon_pixmap)
	XFreePixmap(dpy,data->icon_pixmap);
    if (data->pixmap)
	XFreePixmap(dpy,data->pixmap);
    if (data->img.data)
	free(data->img.data);
    memset(data,0,sizeof(*data));
}

static void
sel_copy(struct sel_data *data)
{
    sel_free(data);
    data->img = ida->img;
    data->img.data = malloc(ida->img.i.width * ida->img.i.height * 3);
    memcpy(data->img.data, ida->img.data,
	   ida->img.i.width * ida->img.i.height * 3);
}

static void
sel_tmpfile(struct sel_data *data, struct ida_writer *wr)
{
    static char *base = "ida";
    char *tmpdir;
    FILE *fp;
    int fd;

    tmpdir = getenv("TMPDIR");
    if (NULL == tmpdir)
	tmpdir="/tmp";
    data->filename = malloc(strlen(tmpdir)+strlen(base)+16);
    sprintf(data->filename,"%s/%s-XXXXXX",tmpdir,base);
    fd = mkstemp(data->filename);
    fp = fdopen(fd,"w");
    wr->write(fp,&data->img);
    fclose(fp);
}

void
selection_convert(Widget widget, XtPointer ignore, XtPointer call_data)
{
    XmConvertCallbackStruct *ccs = call_data;
    unsigned long *ldata;
    unsigned char *cdata;
    struct sel_data *data;
    char *filename;
    int n;

    if (debug) {
	char *t = !ccs->target    ? NULL : XGetAtomName(dpy,ccs->target);
	char *s = !ccs->selection ? NULL : XGetAtomName(dpy,ccs->selection);
	fprintf(stderr,"conv: target=%s selection=%s\n",t,s);
	if (t) XFree(t);
	if (s) XFree(s);
    }

    data = (ccs->selection == XA_PRIMARY) ? &primary : &dragndrop;
    if (NULL == data->img.data) {
	/* should not happen */
	fprintf(stderr,"Oops: no image data in "
		"selection convert callback ???\n");
	ccs->status = XmCONVERT_REFUSE;
	return;
    }
    
    if ((ccs->target == TARGETS) ||
	(ccs->target == _MOTIF_CLIPBOARD_TARGETS) ||
	(ccs->target == _MOTIF_EXPORT_TARGETS)) {
	/* tell which formats we can handle */
	n = 0;
	ldata = (Atom*)XtMalloc(sizeof(Atom)*12);
	ldata[n++] = TARGETS;
	ldata[n++] = MIME_IMAGE_PPM;
	ldata[n++] = XA_PIXMAP;
	ldata[n++] = XA_FOREGROUND;
	ldata[n++] = XA_BACKGROUND;
	ldata[n++] = XA_COLORMAP;
	ldata[n++] = XA_FILE_NAME;
	ldata[n++] = XA_FILE;
	ldata[n++] = MIME_TEXT_URI_LIST;
	ldata[n++] = _NETSCAPE_URL;
	ccs->value  = ldata;
	ccs->length = n;
	ccs->type   = XA_ATOM;
	ccs->format = 32;
	ccs->status = XmCONVERT_DONE;

    } else if (ccs->target == XA_BACKGROUND ||
	       ccs->target == XA_FOREGROUND ||
	       ccs->target == XA_COLORMAP) {
	n = 0;
	ldata = (Atom*)XtMalloc(sizeof(Atom)*8);
	if (ccs->target == XA_BACKGROUND) {
	    ldata[n++] = WhitePixelOfScreen(XtScreen(widget));
	    ccs->type  = XA_PIXEL;
	}
	if (ccs->target == XA_FOREGROUND) {
	    ldata[n++] = BlackPixelOfScreen(XtScreen(widget));
	    ccs->type  = XA_PIXEL;
	}
	if (ccs->target == XA_COLORMAP) {
	    ldata[n++] = DefaultColormapOfScreen(XtScreen(widget));
	    ccs->type  = XA_COLORMAP;
	}
	ccs->value  = ldata;
	ccs->length = n;
	ccs->format = 32;
	ccs->status = XmCONVERT_DONE;

    } else if (ccs->target == XA_PIXMAP) {
	/* xfer pixmap id */
	if (!data->pixmap)
	    data->pixmap = image_to_pixmap(&data->img);
	ldata = (Pixmap*)XtMalloc(sizeof(Pixmap));
	ldata[0] = data->pixmap;
	if (debug)
	    fprintf(stderr,"conv: pixmap id is 0x%lx\n",ldata[0]);
	ccs->value  = ldata;
	ccs->length = 1;
	ccs->type   = XA_DRAWABLE;
	ccs->format = 32;
	ccs->status = XmCONVERT_DONE;

    } else if (ccs->target == MIME_IMAGE_PPM) {
	/* xfer image data directly */
	cdata = XtMalloc(data->img.i.width * data->img.i.height * 3 + 32);
	n = sprintf(cdata,"P6\n%d %d\n255\n",
		    data->img.i.width, data->img.i.height);
	memcpy(cdata+n, data->img.data, data->img.i.width*data->img.i.height*3);
	ccs->value  = cdata;
	ccs->length = n + data->img.i.width * data->img.i.height * 3;
	ccs->type   = MIME_IMAGE_PPM;
	ccs->format = 8;
	ccs->status = XmCONVERT_DONE;

    } else if (ccs->target == XA_FILE_NAME       ||
	       ccs->target == XA_FILE            ||
	       ccs->target == XA_STRING          ||
	       ccs->target == MIME_TEXT_URI_LIST ||
	       ccs->target == _NETSCAPE_URL) {
	/* xfer image via tmp file */
	if (NULL == data->filename)
	    sel_tmpfile(data,&jpeg_writer);
	if (ccs->target == MIME_TEXT_URI_LIST ||
	    ccs->target == _NETSCAPE_URL) {
	    /* filename => url */
	    filename = XtMalloc(strlen(data->filename)+8);
	    sprintf(filename,"file:%s\r\n",data->filename);
	    ccs->type = ccs->target;
	    if (debug)
		fprintf(stderr,"conv: tmp url is %s\n",filename);
	} else {
	    filename = XtMalloc(strlen(data->filename));
	    strcpy(filename,data->filename);
	    ccs->type = XA_STRING;
	    if (debug)
		fprintf(stderr,"conv: tmp file is %s\n",filename);
	}
	ccs->value  = filename;
	ccs->length = strlen(filename);
	ccs->format = 8;
	ccs->status = XmCONVERT_DONE;

    } else if (ccs->target == _MOTIF_LOSE_SELECTION) {
	/* lost (primary) selection */
	sel_free(data);
	
    } else {
	ccs->status = XmCONVERT_REFUSE;
    }
}

static void
dnd_done(Widget widget, XtPointer ignore, XtPointer call_data)
{
#if 0
    XmDragDropFinishCallbackStruct *fin = call_data;
#endif

    if (debug)
	fprintf(stderr,"conv: transfer finished\n");
    sel_free(&dragndrop);
}

/* ---------------------------------------------------------------------- */
/* receive data (drops, paste)                                            */

static Atom targets[16];
static Cardinal ntargets;

static void
selection_xfer(Widget widget, XtPointer ignore, XtPointer call_data)
{
    XmSelectionCallbackStruct *scs = call_data;
    unsigned char *cdata = scs->value;
    unsigned long *ldata = scs->value;
    Atom target = 0;
    int i,j,pending;
    char *file,*tmp;

    if (debug) {
	char *y = !scs->type      ? NULL : XGetAtomName(dpy,scs->type);
	char *t = !scs->target    ? NULL : XGetAtomName(dpy,scs->target);
	char *s = !scs->selection ? NULL : XGetAtomName(dpy,scs->selection);
	fprintf(stderr,"xfer: id=%p target=%s type=%s selection=%s\n",
		scs->transfer_id,t,y,s);
	if (y) XFree(y);
	if (t) XFree(t);
	if (s) XFree(s);
    }

    pending = scs->remaining;
    if (scs->target == TARGETS) {
	/* look if we find a target we can deal with ... */
	for (i = 0; !target && i < scs->length; i++) {
	    for (j = 0; j < ntargets; j++) {
		if (ldata[i] == targets[j]) {
		    target = ldata[i];
		    break;
		}
	    }
	}
	if (target) {
	    XmTransferValue(scs->transfer_id, target, selection_xfer,
			    NULL, XtLastTimestampProcessed(dpy));
	    pending++;
	}
	if (debug) {
	    fprintf(stderr,"xfer: available targets: ");
	    for (i = 0; i < scs->length; i++) {
		char *name = !ldata[i] ? NULL : XGetAtomName(dpy,ldata[i]);
		fprintf(stderr,"%s%s", i != 0 ? ", " : "", name);
		XFree(name);
	    }
	    fprintf(stderr,"\n");
	    if (0 == scs->length)
		fprintf(stderr,"xfer: Huh? no TARGETS available?\n");
	}
    }

    if (scs->target == XA_FILE_NAME ||
	scs->target == XA_FILE) {
	/* load file */
	if (debug)
	    fprintf(stderr,"xfer: => \"%s\"\n",cdata);
	new_file(cdata,1);
    }

    if (scs->target == _NETSCAPE_URL) {
	/* load file */
	if (NULL != (tmp = strchr(cdata,'\n')))
	    *tmp = 0;
	if (NULL != (tmp = strchr(cdata,'\r')))
	    *tmp = 0;
	if (debug)
	    fprintf(stderr,"xfer: => \"%s\"\n",cdata);
	if (0 == strncasecmp("file:",cdata,5))
	    new_file(cdata+5,1);
    }

    if (scs->target == MIME_TEXT_URI_LIST) {
	/* load file(s) */
	for (file = strtok(cdata,"\r\n");
	     NULL != file;
	     file = strtok(NULL,"\r\n")) {
	    if (debug)
		fprintf(stderr,"xfer: => \"%s\"\n",file);
	    if (0 != strncasecmp("file:",file,5))
		continue;
	    new_file(file+5,1);
	}
    }

    if (scs->target == XA_STRING) {
	/* might be a file name too, but don't complain if not */
	if (debug)
	    fprintf(stderr,"xfer: => \"%s\"\n",cdata);
	new_file(cdata,0);
    }

    if (scs->target == MIME_IMAGE_PPM   ||
	scs->target == MIME_IMAGE_PGM   ||
	scs->target == MIME_IMAGE_JPEG  ||
	scs->target == MIME_IMAGE_GIF   ||
	scs->target == MIME_IMAGE_PNG   ||
	scs->target == MIME_IMAGE_TIFF  ||
	scs->target == MIME_IMAGE_XPM   ||
	scs->target == MIME_IMAGE_BMP) {
	/* xfer image data directly */
	char *filename = load_tmpfile("ida");
	int fd;
	fd = mkstemp(filename);
	write(fd,scs->value,scs->length);
	close(fd);
	if (0 == viewer_loadimage(ida,filename)) {
	    ida->file = scs->selection == XA_PRIMARY ?
		"clipboard image" : "drag'n'drop image";
	    resize_shell();
	}
	unlink(filename);
	free(filename);
    }

    if (((scs->target == XA_PIXMAP) && (scs->type == XA_DRAWABLE))) {
	/* beaming pixmaps between apps */
	Screen *scr;
	Window root;
	Pixmap pix;
	int x,y,w,h,bw,depth;
	XImage *ximage;
	struct ida_image img;
	
	pix = ldata[0];
	if (debug)
	    fprintf(stderr,"xfer: => id=0x%lx\n",pix);
	scr = XtScreen(widget);
	XGetGeometry(dpy,pix,&root,&x,&y,&w,&h,&bw,&depth);
	ximage = XGetImage(dpy,pix,0,0,w,h,-1,ZPixmap);
	parse_ximage(&img, ximage);
	XDestroyImage(ximage);
	viewer_setimage(ida,&img,scs->selection == XA_PRIMARY ?
			"clipboard pixmap" : "drag'n'drop pixmap");
	resize_shell();
    }

    XFree(scs->value);
    if (1 == pending) {
	/* all done -- clean up */
	if (debug)
	    fprintf(stderr,"xfer: all done\n");
	XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_SUCCEED);
	XdndDropFinished(widget,scs);
    }
}

void
selection_dest(Widget  w, XtPointer ignore, XtPointer call_data)
{
    XmDestinationCallbackStruct *dcs = call_data;

    if (NULL != dragndrop.img.data) {
	if (debug)
	    fprintf(stderr,"dest: ignore self drop\n");
	XmTransferDone(dcs->transfer_id, XmTRANSFER_DONE_FAIL);
	return;
    }
    if (debug)
	fprintf(stderr,"dest: xfer id=%p\n",dcs->transfer_id);
    XmTransferValue(dcs->transfer_id, TARGETS, selection_xfer,
		    NULL, XtLastTimestampProcessed(dpy));
}

/* ---------------------------------------------------------------------- */

void ipc_ac(Widget widget, XEvent *event, String *argv, Cardinal *argc)
{
    Widget    drag;
    Arg       args[4];
    Cardinal  n=0;

    if (0 == *argc)
	return;

    if (debug)
	fprintf(stderr,"ipc: %s\n",argv[0]);
    if (0 == strcmp(argv[0],"paste")) {
	XmePrimarySink(ida->widget,XmCOPY,NULL,XtLastTimestampProcessed(dpy));
    }
    if (0 == strcmp(argv[0],"copy")) {
	sel_copy(&primary);
	XmePrimarySource(ida->widget,XtLastTimestampProcessed(dpy));
    }
    if (0 == strcmp(argv[0],"drag")) {
	sel_copy(&dragndrop);
	iconify(widget,&dragndrop);
	XtSetArg(args[n], XmNdragOperations, XmDROP_COPY); n++;
	XtSetArg(args[n], XmNsourcePixmapIcon, dragndrop.icon_widget); n++;
	drag = XmeDragSource(ida->widget, NULL, event, args, n);
	XtAddCallback(drag, XmNdragDropFinishCallback, dnd_done, NULL);
    }
}

void dnd_add(Widget widget)
{
    Arg       args[4];
    Cardinal  n=0;

    XtSetArg(args[n], XmNimportTargets, targets); n++;
    XtSetArg(args[n], XmNnumImportTargets, ntargets); n++;
    XmeDropSink(widget,args,n);
    XdndDropSink(widget);
}

void ipc_init()
{
    TARGETS
	= XInternAtom(dpy, "TARGETS", False);
    _MOTIF_EXPORT_TARGETS =
	XInternAtom(dpy, "_MOTIF_EXPORT_TARGETS", False);
    _MOTIF_CLIPBOARD_TARGETS =
	XInternAtom(dpy, "_MOTIF_CLIPBOARD_TARGETS", False);
    _MOTIF_LOSE_SELECTION =
	XInternAtom(dpy, "_MOTIF_LOSE_SELECTION", False);

    XA_FILE_NAME       = XInternAtom(dpy, "FILE_NAME",     False);
    XA_FILE            = XInternAtom(dpy, "FILE",          False);
    XA_FOREGROUND      = XInternAtom(dpy, "FOREGROUND",    False);
    XA_BACKGROUND      = XInternAtom(dpy, "BACKGROUND",    False);
    XA_PIXEL           = XInternAtom(dpy, "PIXEL",         False);
    _NETSCAPE_URL      = XInternAtom(dpy, "_NETSCAPE_URL", False);

    MIME_TEXT_URI_LIST = XInternAtom(dpy, "text/uri-list", False);
    MIME_IMAGE_PPM     = XInternAtom(dpy, "image/ppm",     False);
    MIME_IMAGE_PGM     = XInternAtom(dpy, "image/pgm",     False);
    MIME_IMAGE_XPM     = XInternAtom(dpy, "image/xpm",     False);
    MIME_IMAGE_BMP     = XInternAtom(dpy, "image/bmp",     False);
    MIME_IMAGE_JPEG    = XInternAtom(dpy, "image/jpeg",    False);
    MIME_IMAGE_GIF     = XInternAtom(dpy, "image/gif",     False);
    MIME_IMAGE_PNG     = XInternAtom(dpy, "image/png",     False);
    MIME_IMAGE_TIFF    = XInternAtom(dpy, "image/tiff",    False);

    targets[ntargets++] = XA_FILE_NAME;
    targets[ntargets++] = XA_FILE;
    targets[ntargets++] = _NETSCAPE_URL;
    targets[ntargets++] = MIME_TEXT_URI_LIST;
    targets[ntargets++] = MIME_IMAGE_PPM;
    targets[ntargets++] = MIME_IMAGE_PGM;
    targets[ntargets++] = MIME_IMAGE_XPM;
    targets[ntargets++] = MIME_IMAGE_BMP;
#ifdef HAVE_LIBJPEG
    targets[ntargets++] = MIME_IMAGE_JPEG;
#endif
#ifdef HAVE_LIBUNGIF
    targets[ntargets++] = MIME_IMAGE_GIF;
#endif
#ifdef HAVE_LIBPNG
    targets[ntargets++] = MIME_IMAGE_PNG;
#endif
#ifdef HAVE_LIBTIFF
    targets[ntargets++] = MIME_IMAGE_TIFF;
#endif
    targets[ntargets++] = XA_PIXMAP;
    targets[ntargets++] = XA_STRING;
}
