      SUBROUTINE slAMPQ (RA, DA, AMPRMS, RM, DM)
*+
*     - - - - - -
*      A M P Q
*     - - - - - -
*
*  Convert star RA,Dec from geocentric apparent to mean place
*
*  The mean coordinate system is the post IAU 1976 system,
*  loosely called FK5.
*
*  Use of this routine is appropriate when efficiency is important
*  and where many star positions are all to be transformed for
*  one epoch and equinox.  The star-independent parameters can be
*  obtained by calling the slMAPA routine.
*
*  Given:
*     RA       d      apparent RA (radians)
*     DA       d      apparent Dec (radians)
*
*     AMPRMS   d(21)  star-independent mean-to-apparent parameters:
*
*       (1)      time interval for proper motion (Julian years)
*       (2-4)    barycentric position of the Earth (AU)
*       (5-7)    heliocentric direction of the Earth (unit vector)
*       (8)      (grav rad Sun)*2/(Sun-Earth distance)
*       (9-11)   ABV: barycentric Earth velocity in units of c
*       (12)     sqrt(1-v**2) where v=modulus(ABV)
*       (13-21)  precession/nutation (3,3) matrix
*
*  Returned:
*     RM       d      mean RA (radians)
*     DM       d      mean Dec (radians)
*
*  References:
*     1984 Astronomical Almanac, pp B39-B41.
*     (also Lederle & Schwan, Astron. Astrophys. 134,
*      1-6, 1984)
*
*  Notes:
*
*  1)  The accuracy is limited by the routine slEVP, called
*      by slMAPA, which computes the Earth position and
*      velocity using the methods of Stumpff.  The maximum
*      error is about 0.3 milliarcsecond.
*
*  2)  Iterative techniques are used for the aberration and
*      light deflection corrections so that the routines
*      slAMP (or slAMPQ) and slMAP (or slMAPQ) are
*      accurate inverses;  even at the edge of the Sun's disc
*      the discrepancy is only about 1 nanoarcsecond.
*
*  Called:  slDS2C, slDIMV, slDVDV, slDVN, slDC2S,
*           slDA2P
*
*  P.T.Wallace   Starlink   21 June 1993
*
*  Copyright (C) 1995 Rutherford Appleton Laboratory
*  Copyright (C) 1995 Association of Universities for Research in Astronomy Inc.
*-

      IMPLICIT NONE

      DOUBLE PRECISION RA,DA,AMPRMS(21),RM,DM

      INTEGER I,J

      DOUBLE PRECISION GR2E,AB1,EHN(3),ABV(3),P3(3),P2(3),
     :                 AB1P1,P1DV,P1DVP1,P1(3),W,PDE,PDEP1,P(3)

      DOUBLE PRECISION slDVDV,slDA2P



*  Unpack scalar and vector parameters
      GR2E = AMPRMS(8)
      AB1 = AMPRMS(12)
      DO I=1,3
         EHN(I) = AMPRMS(I+4)
         ABV(I) = AMPRMS(I+8)
      END DO

*  Apparent RA,Dec to Cartesian
      CALL slDS2C(RA,DA,P3)

*  Precession and nutation
      CALL slDIMV(AMPRMS(13),P3,P2)

*  Aberration
      AB1P1 = AB1+1D0
      DO I=1,3
         P1(I) = P2(I)
      END DO
      DO J=1,2
         P1DV = slDVDV(P1,ABV)
         P1DVP1 = 1D0+P1DV
         W = 1D0+P1DV/AB1P1
         DO I=1,3
            P1(I) = (P1DVP1*P2(I)-W*ABV(I))/AB1
         END DO
         CALL slDVN(P1,P3,W)
         DO I=1,3
            P1(I) = P3(I)
         END DO
      END DO

*  Light deflection
      DO I=1,3
         P(I) = P1(I)
      END DO
      DO J=1,5
         PDE = slDVDV(P,EHN)
         PDEP1 = 1D0+PDE
         W = PDEP1-GR2E*PDE
         DO I=1,3
            P(I) = (PDEP1*P1(I)-GR2E*EHN(I))/W
         END DO
         CALL slDVN(P,P2,W)
         DO I=1,3
            P(I) = P2(I)
         END DO
      END DO

*  Mean RA,Dec
      CALL slDC2S(P,RM,DM)
      RM = slDA2P(RM)

      END
