/***************************************************************************
                          scenarios.c -  description
                             -------------------
    begin                : Tue Mar 12 2002
    copyright            : (C) 2001 by Michael Speck
    email                : kulkanie@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "units.h"
#include "scenarios.h"
#include "parser.h"

/*
====================================================================
Externals
====================================================================
*/
extern char *source_path;
extern char *dest_path;
extern char *custom_name;
extern int  nation_count;
extern char *nations[];
extern char *weather_types[];

int unit_entry_used[UDB_LIMIT];

/*
====================================================================
The scenario result is determined by a list of conditions.
If any of the conditions turn out to be true the scenario ends,
the message is displayed and result is returned (for campaign).
If there is no result after the LAST turn result and message
of the else struct are used.
A condition has two test fields: one is linked with an 
logical AND the other is linkes with an logical OR. Both
fields must return True if the condition is supposed to be
fullfilled. An empty field does always return True.
Struct:
    result {
        check = every_turn | last_turn
        cond {
               and { test .. testX }
               or  { test .. testY }
               result = ...
               message = ... }
        cond { ... }
        else { result = ... message = ... }
    }
Tests:
  control_hex { player x = ... y = ... } 
      :: control a special victory hex
  control_any_hex { player count = ... } 
      :: control this number of victory
         hexes
  control_all_hexes { player }    
      :: conquer everything important
         on the map
  turns_left { count = ... }      
      :: at least so many turns are left
====================================================================
*/
/*
====================================================================
Scenario file names
====================================================================
*/
char *fnames[] = {
    "Poland",
    "Warsaw",
    "Norway",
    "LowCountries",
    "France",
    "Sealion40",
    "NorthAfrica",
    "MiddleEast",
    "ElAlamein",
    "Caucasus",
    "Sealion43",
    "Torch",
    "Husky",
    "Anzio",
    "D-Day",
    "Anvil",
    "Ardennes",
    "Cobra",
    "MarketGarden",
    "BerlinWest",
    "Balkans",
    "Crete",
    "Barbarossa",
    "Kiev",
    "Moscow41",
    "Sevastapol",
    "Moscow42",
    "Stalingrad",
    "Kharkov",
    "Kursk",
    "Moscow43",
    "Byelorussia",
    "Budapest",
    "BerlinEast",
    "Berlin",
    "Washington",
    "EarlyMoscow",
    "SealionPlus"
};

/*
====================================================================
AI modules names
====================================================================
*/
char *ai_modules[] = {
    /* axis, allied -- scenarios as listed above */
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default",
    "default", "default"
};

/*
====================================================================
Locals
====================================================================
*/

/*
====================================================================
Random generator
====================================================================
*/
int seed;
void random_seed( int _seed )
{
    seed = _seed;
}
int random_get( int low, int high )
{
    int p1 = 1103515245;
    int p2 = 12345;
    seed = ( seed * p1 + p2 ) % 2147483647;
    return ( ( abs( seed ) / 3 ) % ( high - low + 1 ) ) + low;
}

/*
====================================================================
Read all flags from MAP%i.SET and add them to dest_file.
We need the scenario file as well as the victory hex positions
are saved there.
====================================================================
*/
int scen_add_flags( FILE *dest_file, FILE *scen_file, int id )
{
    FILE *map_file;
    char path[512];
    int width, height, ibuf;
    int x, y, i, obj;
    int vic_hexes[40]; /* maximum if 20 hexes in PG - terminated with -1 */
    int obj_count = 0;
    /* read victory hexes from scen_file */
    memset( vic_hexes, 0, sizeof(int) * 40 );
    fseek( scen_file, 37, SEEK_SET );
    for ( i = 0; i < 20; i++ ) {
        fread( &vic_hexes[i * 2], 2, 1, scen_file );
        fread( &vic_hexes[i * 2 + 1], 2, 1, scen_file );
        if ( vic_hexes[i * 2] >= 1000 || vic_hexes[i * 2] < 0 )
            break;
        obj_count++;
    }
    /* open set file */
    sprintf( path, "%s/MAP%02i.SET", source_path, id );
    if ( ( map_file = fopen( path, "r" ) ) == 0 ) {
        sprintf( path, "%s/map%02i.set", source_path, id );
        if ( ( map_file = fopen( path, "r" ) ) == 0 ) {
            fprintf( stderr, "%s: file not found\n", path );
            return 0;
        }
    }
    /* read/write map size */
    width = height = 0;
    fseek( map_file, 101, SEEK_SET );
    fread( &width, 2, 1, map_file );
    fseek( map_file, 103, SEEK_SET );
    fread( &height, 2, 1, map_file );
    width++; height++;
    /* owner info */
    fseek( map_file, 123 + 3 * width * height, SEEK_SET );
    for ( y = 0; y < height; y++ ) {
        for ( x = 0; x < width; x++ ) {
            ibuf = 0; fread( &ibuf, 1, 1, map_file );
            if ( ibuf > 0 ) {
                obj = 0;
                for ( i = 0; i < obj_count; i++ )
                    if ( vic_hexes[i * 2] == x && vic_hexes[i * 2 + 1] == y ) {
                        obj = 1; break;
                    }
                fprintf( dest_file, "<flag\nx%i\ny%i\nnation%s\nobj%i\n>\n", x, y, nations[(ibuf - 1) * 3], obj );
            }
        }
    }
    return 1;
}

/*
====================================================================
Panzer General offers two values: weather condition and weather
region which are used to determine the weather throughout the
scenario. As the historical battle did only occur once the weather
may not change from game to game so we compute the weather of a 
scenario depending on three values:
inital condition, weather region, month
Initial Condition:
    1  clear
    0  rain/snow
Regions:
    0  Desert
    1  Mediterranean
    2  Northern Europe
    3  Eastern Europe
====================================================================
*/
void scen_create_weather( FILE *dest_file, FILE *scen_file, int month, int turns )
{
    float month_mod[13] = { 0, 1.7, 1.6, 1.0, 2.0, 1.2, 0.7, 0.5, 0.6, 1.4, 1.7, 2.2, 1.7 };
    int med_weather[4] = { 0, 16, 24, 36 };
    int bad_weather[4] = { 0, 8, 12, 18 };
    int i, result;
    int init_cond = 0, region = 0;
    int weather[turns];
    memset( weather, 0, sizeof( int ) * turns );
    /* get condition and region */
    fseek( scen_file, 16, SEEK_SET );
    fread( &init_cond, 1, 1, scen_file );
    fread( &region, 1, 1, scen_file );
    /* compute the weather */
    random_seed( month * turns + ( region + 1 ) * ( init_cond + 1 ) );
    for ( i = 0; i < turns; i++ ) {
        result = random_get( 1, 100 );
        if ( result <= (int)( month_mod[month] * bad_weather[region] ) ) 
            weather[i] = 2;
        else
            if ( result <= (int)( month_mod[month] * med_weather[region] ) )
                weather[i] = 1;
    }
    /* initial condition */
    weather[0] = (init_cond==1)?0:2;
    /* from december to february turn 2 (rain) into 3 (snow) */
    if ( month < 3 || month == 12 ) {
        for ( i = 0; i < turns; i++ )
            if ( weather[i] == 2 )
                weather[i]++;
    }
    /* write weather */
    fprintf( dest_file, "weather" );
    i = 0;
    while ( i < turns ) {
        fprintf( dest_file, "%s", weather_types[weather[i] * 3] );
        if ( i < turns - 1 )
            fprintf( dest_file, "" );
        i++;
    }
    fprintf( dest_file, "\n" );
}

/*
====================================================================
Read unit data from scen_file, convert and write it to dest_file.
====================================================================
*/
void scen_create_unit( FILE *dest_file, FILE *scen_file )
{
    int id = 0, nation = 0, x = 0, y = 0, str = 0, entr = 0, exp = 0, trsp_id = 0, org_trsp_id = 0;
    /* read unit -- 14 bytes */
    /* icon id */
    fread( &id, 2, 1, scen_file ); /* icon id */
    fread( &org_trsp_id, 2, 1, scen_file ); /* transporter of organic unit */
    fread( &nation, 1, 1, scen_file ); nation--; /* nation + 1 */
    fread( &trsp_id, 2, 1, scen_file ); /* sea/air transport */
    fread( &x, 2, 1, scen_file ); /* x */
    fread( &y, 2, 1, scen_file ); /* y */
    fread( &str, 1, 1, scen_file ); /* strength */
    fread( &entr, 1, 1, scen_file ); /* entrenchment */
    fread( &exp, 1, 1, scen_file ); /* experience */
    /* mark id */
    unit_entry_used[id - 1] = 1;
    if ( trsp_id ) 
        unit_entry_used[trsp_id - 1] = 1;
    else
        if ( org_trsp_id ) 
            unit_entry_used[org_trsp_id - 1] = 1;
    /* write unit */
    fprintf( dest_file, "<unit\n" );
    fprintf( dest_file, "id%i\nnation%s\n", id - 1, nations[nation * 3] );
    fprintf( dest_file, "x%i\ny%i\n", x, y );
    fprintf( dest_file, "str%i\nentr%i\nexp%i\n", str, entr, exp );
    if ( trsp_id == 0 && org_trsp_id == 0 )
        fprintf( dest_file, "trspnone\n" );
    else {
        if ( trsp_id ) 
            fprintf( dest_file, "trsp%i\n", trsp_id - 1 );
        else
            fprintf( dest_file, "trsp%i\n", org_trsp_id - 1 );
    }
    fprintf( dest_file, ">\n" );
}

/*
====================================================================
Add the victory conditions
====================================================================
*/
int major_limits[] = {
    /* if an entry is not -1 it's a default axis offensive 
       and this is the turn number that must remain for a major
       victory when all flags where captured */
    -1, /* UNUSED */
    3, /* POLAND */
    7, /* WARSAW */
    5, /* NORWAY */
    6, /* LOWCOUNRTIES */
    13, /* FRANCE */
    3, /* SEALION 40 */
    4, /* NORTH AFRICA */
    5, /* MIDDLE EAST */
    3, /* EL ALAMEIN */
    12, /* CAUCASUS */
    3, /* SEALION 43 */
    -1, /* TORCH */
    -1, /* HUSKY */
    -1, /* ANZIO */
    -1, /* D-DAY */
    -1, /* ANVIL */
    -1, /* ARDENNES */
    -1, /* COBRA */
    -1, /* MARKETGARDEN */
    -1, /* BERLIN WEST */
    3, /* BALKANS */
    2, /* CRETE */
    10, /* BARBAROSSA */
    8, /* KIEV */
    4, /* MOSCOW 41 */
    3, /* SEVASTAPOL */
    6, /* MOSCOW 42 */
    13, /*STALINGRAD */
    4, /* KHARKOV */
    -1, /*KURSK */
    5, /* MOSCOW 43*/
    -1, /* BYELORUSSIA */
    5, /* BUDAPEST */
    -1, /* BERLIN EAST */
    -1, /* BERLIN */
    7, /* WASHINGTON */
    5, /* EARLY MOSCOW */
    3, /* SEALION PLUS */
};
#define COND_BEGIN fprintf( file, "<cond\n" )
#define COND_END   fprintf( file, ">\n" )
#define COND_RESULT( str ) fprintf( file, "result%s\n", str )
#define COND_MESSAGE( str ) fprintf( file, "message%s\n", str )
void scen_add_vic_conds( FILE *file, int id )
{
    /* for panzer general the check is run every turn with only one
       exception Market-Garden where Arnhem must be captured and 
       hold until end */
    if ( id == 19 )
        fprintf( file, "<result\nchecklast_turn\n" );
    else
        fprintf( file, "<result\ncheckevery_turn\n" );
    /* add conditions */
    if ( major_limits[id] != -1 ) {
        COND_BEGIN;
        fprintf( file, "<and\n<control_all_hexes\nplayeraxis\n>\n<turns_left\ncount%i\n>\n>\n", major_limits[id] );
        COND_RESULT( "major" ); 
        COND_MESSAGE( "Axis Major Victory" );
        COND_END;
        COND_BEGIN;
        fprintf( file, "<and\n<control_all_hexes\nplayeraxis\n>\n>\n" );
        COND_RESULT( "minor" ); 
        COND_MESSAGE( "Axis Minor Victory" );
        COND_END;
        fprintf( file, "<else\n" );
        COND_RESULT( "defeat" ); 
        COND_MESSAGE( "Axis Defeat" );
        COND_END;
    }
    else
    if ( id == 17 || id == 19 ) {
        /* special axis offensives */
        COND_BEGIN;
        switch ( id ) {
            case 17: /* ARDENNES */
                fprintf( file, "<and\n"\
                               "<control_hex\nplayeraxis\nx16\ny16\n>\n"\
                               "<control_hex\nplayeraxis\nx26\ny4\n>\n"\
                               "<control_hex\nplayeraxis\nx27\ny21\n>\n"\
                               "<control_hex\nplayeraxis\nx39\ny21\n>\n"\
                               "<control_hex\nplayeraxis\nx48\ny8\n>\n"\
                               "<control_hex\nplayeraxis\nx54\ny14\n>\n"\
                               "<control_hex\nplayeraxis\nx59\ny18\n>\n"\
                               ">\n" );
                break;
            case 19: /* MARKET-GARDEN */
                fprintf( file, "<and\n<control_hex\nplayeraxis\nx37\ny10>\n>\n" );
                break;
        }
        COND_RESULT( "axis" );
        COND_MESSAGE( "Axis Victory" );        COND_END;
        fprintf( file, "<else\n" );
        COND_RESULT( "allies" ); 
        COND_MESSAGE( "Allied Victory" );
        COND_END;
    }
    else {
        /* allied offensives */
        COND_BEGIN;
        switch ( id ) {
            case 12: /* TORCH */
                fprintf( file, "<or\n<control_hex\nplayerallies\nx27\ny5\n>\n<control_hex_num\nplayerallies\ncount6>\n>\n" );
                break;
            case 13: /* HUSKY */
                fprintf( file, "<or\n<control_hex_num\nplayerallies\ncount14\n>\n>\n" );
                break;
            case 14: /* ANZIO */
                fprintf( file, "<or\n<control_hex\nplayerallies\nx13\ny17\n>\n<control_hex_num\nplayerallies\ncount5\n>\n>\n" );
                break;
            case 15: /* D-DAY */
                fprintf( file, "<or\n<control_hex_num\nplayerallies\ncount4\n>\n>\n" );
                break;
            case 16: /* ANVIL */
                fprintf( file, "<or\n<control_hex_num\nplayerallies\ncount5\n>\n>\n" );
                break;
            case 18: /* COBRA */
                fprintf( file, "<or\n<control_hex_num\nplayerallies\ncount5\n>\n>\n" );
                break;
            case 20: /* BERLIN WEST */
                fprintf( file, "<or\n<control_hex\nplayerallies\nx36\ny14\n>\n<control_hex_num\nplayerallies\ncount5\n>\n>\n" );
                break;
            case 30: /* KURSK */
                fprintf( file, "<or\n<control_all_hexes\nplayerallies\n>\n>\n" );
                break;
            case 32: /* BYELORUSSIA */
                fprintf( file, "<or\n<control_hex\nplayerallies\nx3\ny12\n>\n>\n" );
                break;
            case 34: /* BERLIN EAST */
                fprintf( file, "<or\n<control_hex\nplayerallies\nx36\ny14\n>\n<control_hex_num\nplayerallies\ncount8\n>\n>\n" );
                break;
            case 35: /* BERLIN */
                fprintf( file, "<or\n<control_hex\nplayerallies\nx36\ny14\n>\n>\n" );
                break;
        }
        COND_RESULT( "allies" );
        COND_MESSAGE( "Allied Victory" );
        COND_END;
        fprintf( file, "<else\n" );
        COND_RESULT( "axis" ); 
        COND_MESSAGE( "Axis Victory" );
        COND_END;
    }
    /* end result struct */
    fprintf( file, ">\n" );
}

/*
====================================================================
Publics
====================================================================
*/

/*
====================================================================
If scen_id == -1 convert all scenarios found in 'source_path'.
If scen_id >= 0 convert single scenario from current working 
directory.
====================================================================
*/
int scenarios_convert( int scen_id )
{
    int i, j;
    char dummy[256];
    int  day, month, year, turns, turns_per_day, days_per_turn, ibuf;
    int  unit_offset, unit_count;
    int  axis_orient, axis_strat, allied_strat;
    char path[512];
    FILE *dest_file = 0, *scen_file = 0, *aux_file = 0;
    PData *pd = 0, *reinf, *nation, *unit;
    int def_str, def_exp, def_entr;
    char *str;
    printf( "  scenarios...\n" );
    sprintf( path, "%s/scenarios/pg", dest_path );
    mkdir( path, S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH );
    /* get the reinforcements which are used later */
    if ( scen_id == -1 ) {
        sprintf( path, "%s/data/reinf", SRC_DIR );
        if ( ( pd = parser_read_file( "reinforcements", path ) ) == 0 ) {
            fprintf( stderr, "%s\n", parser_get_error() );
            goto failure;
        }
    }
    /* go */
    for ( i = (( scen_id == -1 ) ? 1 : scen_id); 
          i < (( scen_id == -1 ) ? 38 : scen_id) + 1; i++ ) {
        /* open dest file */
        if ( scen_id == -1 )
            sprintf( path, "%s/scenarios/pg/%s", dest_path, fnames[i - 1] );
        else
            sprintf( path, "%s/scenarios/pg/%s", dest_path, custom_name );
        if ( ( dest_file = fopen( path, "w" ) ) == 0 ) {
            fprintf( stderr, "%s: access denied\n", path );
            goto failure;
        }
        /* scenario name and description */
        fprintf( dest_file, "@\n" );
        if ( scen_id == -1 ) {
            sprintf( path, "%s/SCENSTAT.BIN", source_path );
            if ( ( aux_file = fopen( path, "r" ) ) == 0 ) {
                fprintf( stderr, "%s: file not found\n", path );
                goto failure;
            }
            fseek( aux_file, 40 + (i - 1) * 14, SEEK_SET );
            fread( dummy, 14, 1, aux_file );
            fprintf( dest_file, "name%s\n", dummy );
            fseek( aux_file, 600 + (i - 1) * 160 , SEEK_SET );
            fread( dummy, 160, 1, aux_file );
            fprintf( dest_file, "desc%s\n", dummy );
            fprintf( dest_file, "authorsStrategic Simulation Inc.\n" );
            fclose( aux_file );
        }
        else {
            fprintf( dest_file, "name%s\n", custom_name );
            fprintf( dest_file, "descnone\n" );
            fprintf( dest_file, "authorsnobody\n" );
        }
        /* open scenario file */
        sprintf( path, "%s/GAME%03i.SCN", source_path, i );
        if ( ( scen_file = fopen( path, "r" ) ) == 0 ) {
            sprintf( path, "%s/game%03i.scn", source_path, i );
            if ( ( scen_file = fopen( path, "r" ) ) == 0 ) {
                fprintf( stderr, "%s: file not found\n", path );
                goto failure;
            }
        }
        /* date */
        fseek( scen_file, 22, SEEK_SET );
        day = 0; fread( &day, 1, 1, scen_file );
        month = 0; fread( &month, 1, 1, scen_file );
        year = 0; fread( &year, 1, 1, scen_file );
        fprintf( dest_file, "date%02i.%02i.19%i\n", day, month, year );
        /* turn limit */
        fseek( scen_file, 21, SEEK_SET );
        turns = 0; fread( &turns, 1, 1, scen_file );
        fprintf( dest_file, "turns%i\n", turns );
        fseek( scen_file, 25, SEEK_SET );
        turns_per_day = 0; fread( &turns_per_day, 1, 1, scen_file );
        fprintf( dest_file, "turns_per_day%i\n", turns_per_day );
        days_per_turn = 0; fread( &days_per_turn, 1, 1, scen_file );
        if ( turns_per_day == 0 && days_per_turn == 0 )
            days_per_turn = 1;
        fprintf( dest_file, "days_per_turn%i\n", days_per_turn );
        /* nations */
        fprintf( dest_file, "nation_dbpg.ndb\n" );
        /* units */
        if ( scen_id == -1 || !units_find_panzequp() )
            fprintf( dest_file, "<unit_db\nmainpg.udb\n>\n" );
        /* if there modified units they are added to the 
           scenario file. lgeneral loads from the scenario file
           if no unit_db was specified. */
        /* map:
           a custom scenario will have the map added to the same file which
           will be checked when no map was specified.
           */
        if ( scen_id == -1 )
            fprintf( dest_file, "mappg/map%02i\n", i );
        /* weather */
        scen_create_weather( dest_file, scen_file, month, turns );
        /* flags */
        fprintf( dest_file, "<flags\n" );
        if ( !scen_add_flags( dest_file, scen_file, i ) ) goto failure;
        fprintf( dest_file, ">\n" );
        /* get unit offset */
        fseek( scen_file, 117, SEEK_SET );
        ibuf = 0; fread( &ibuf, 1, 1, scen_file );
        unit_offset = ibuf * 4 + 135;
        /* players */
        fprintf( dest_file, "<players\n" );
        /* axis */
        fseek( scen_file, 12, SEEK_SET );
        /* orientation */
        axis_orient = 0; fread( &axis_orient, 1, 1, scen_file );
        if ( axis_orient == 1 ) 
            sprintf( dummy, "right" );
        else
            sprintf( dummy, "left" );
        /* strategy: -2 (very defensive) to 2 (very aggressive) */
        fseek( scen_file, 15, SEEK_SET );
        axis_strat = 0; fread( &axis_strat, 1, 1, scen_file );
        if ( axis_strat == 0 ) 
            axis_strat = 1;
        else
            axis_strat = -1;
        /* definition */
        fprintf( dest_file, "<axis\nnameAxis\n" );
        fprintf( dest_file, "nationsgerausithunbulrumfinesp\n" );
        fprintf( dest_file, "allied_players\n" );
        fprintf( dest_file, "orientation%s\ncontrolhuman\nstrategy%i\n", dummy, axis_strat );
        if ( scen_id == -1 )
            fprintf( dest_file, "ai_module%s\n", ai_modules[i*2] );
        else
            fprintf( dest_file, "ai_moduledefault\n" );
        /* transporter */
        fprintf( dest_file, "<transporters\n" );
        /* air */
        fseek( scen_file, unit_offset - 8, SEEK_SET );
        ibuf = 0; fread( &ibuf, 2, 1, scen_file );
        if ( ibuf )
            fprintf( dest_file, "<air\nunit%i\ncount50\n>\n", ibuf - 1 );
        /* sea */
        fseek( scen_file, unit_offset - 4, SEEK_SET );
        ibuf = 0; fread( &ibuf, 2, 1, scen_file );
        if ( ibuf )
            fprintf( dest_file, "<sea\nunit%i\ncount50\n>\n", ibuf - 1 );
        fprintf( dest_file, ">\n" );
        fprintf( dest_file, ">\n" );
        /* allies */
        if ( axis_orient == 1 )
            sprintf( dummy, "left" );
        else
            sprintf( dummy, "right" );
        if ( axis_strat == 1 )
            allied_strat = -1;
        else
            allied_strat = 1;
        fprintf( dest_file, "<allies\nnameAllies\n" );
        fprintf( dest_file, "nationsbelluxdenfragreusaturnetnorpolporsosweswiengyug\n" );
        fprintf( dest_file, "allied_players\n" );
        fprintf( dest_file, "orientation%s\ncontrolcpu\nstrategy%i\n", dummy, allied_strat );
        if ( scen_id == -1 )
            fprintf( dest_file, "ai_module%s\n", ai_modules[i*2 + 1] );
        else
            fprintf( dest_file, "ai_moduledefault\n" );
        /* transporter */
        fprintf( dest_file, "<transporters\n" );
        /* air */
        fseek( scen_file, unit_offset - 6, SEEK_SET );
        ibuf = 0; fread( &ibuf, 2, 1, scen_file );
        if ( ibuf )
            fprintf( dest_file, "<air\nunit%i\ncount50\n>\n", ibuf - 1 );
        /* sea */
        fseek( scen_file, unit_offset - 2, SEEK_SET );
        ibuf = 0; fread( &ibuf, 2, 1, scen_file );
        if ( ibuf )
            fprintf( dest_file, "<sea\nunit%i\ncount50\n>\n", ibuf - 1 );
        fprintf( dest_file, ">\n" );
        fprintf( dest_file, ">\n" );
        fprintf( dest_file, ">\n" );
        /* victory conditions */
        if ( scen_id == -1 )
            scen_add_vic_conds( dest_file, i );
        else {
            /* and the default is that the attacker must capture
               all targets */
            fprintf( dest_file, "<result\ncheckevery_turn\n" );
            fprintf( dest_file, "<cond\n" );
            fprintf( dest_file, "<and\n<control_all_hexes\nplayer%s\n>\n>\n",
                     (axis_strat > 0) ? "axis" : "allies" );
            fprintf( dest_file, "resultvictory\n" );
            fprintf( dest_file, "message%s\n", 
                     (axis_strat > 0) ? "Axis Victory" : "Allied Victory" );
            fprintf( dest_file, ">\n" );
            fprintf( dest_file, "<else\n" );
            fprintf( dest_file, "resultdefeat\n" );
            fprintf( dest_file, "message%s\n", 
                     (axis_strat > 0) ? "Axis Defeat" : "Allied Defeat" );
            fprintf( dest_file, ">\n" );
            fprintf( dest_file, ">\n" );
        }
        /* units */
        /* mark all id's that will be used from PANZEQUP.EQP
           for modified unit database */
        memset( unit_entry_used, 0, sizeof( unit_entry_used ) );
        /* count them */
        fseek( scen_file, 33, SEEK_SET );
        ibuf = 0; fread( &ibuf, 1, 1, scen_file );
        unit_count = ibuf; /* core */
        ibuf = 0; fread( &ibuf, 1, 1, scen_file );
        unit_count += ibuf; /* allies */
        ibuf = 0; fread( &ibuf, 1, 1, scen_file );
        unit_count += ibuf; /* auxiliary */
        /* build them */
        fseek( scen_file, unit_offset, SEEK_SET );
        fprintf( dest_file, "<units\n" );
        for ( j = 0; j < unit_count; j++ )
            scen_create_unit( dest_file, scen_file );
        /* reinforcements -- only for original PG scenarios */
        if ( scen_id == -1 ) {
            if ( parser_get_pdata( pd, fnames[i - 1], &reinf ) ) {
                /* there are units stored for this scenario grouped
                   for each nation */
                list_reset( reinf->entries );
                while ( ( nation = list_next( reinf->entries ) ) ) {
                    /* get defaults */
                    def_str = 10; def_exp = 0; def_entr = 0;
                    parser_get_int( nation, "str", &def_str );
                    parser_get_int( nation, "exp", &def_exp );
                    parser_get_int( nation, "entr", &def_entr );
                    /* get units */
                    list_reset( nation->entries );
                    while ( ( unit = list_next( nation->entries ) ) )
                        if ( !strcmp( "unit", unit->name ) ) {
                            /* add unit */
                            fprintf( dest_file, "<unit\n" );
                            fprintf( dest_file, "nation%s\n", nation->name );
                            if ( !parser_get_int( unit, "id", &ibuf ) )
                                goto failure;
                            fprintf( dest_file, "id%i\n", ibuf );
                            ibuf = def_str;  parser_get_int( unit, "str", &ibuf );
                            fprintf( dest_file, "str%i\n", ibuf );
                            ibuf = def_exp;  parser_get_int( unit, "exp", &ibuf );
                            fprintf( dest_file, "exp%i\n", ibuf );
                            ibuf = def_entr; parser_get_int( unit, "entr", &ibuf );
                            fprintf( dest_file, "entr%i\n", ibuf );
                            if ( parser_get_value( unit, "trsp", &str, 0 ) )
                                fprintf( dest_file, "trsp%s\n", str );
                            if ( !parser_get_int( unit, "delay", &ibuf ) )
                                goto failure;
                            fprintf( dest_file, "delay%i\n", ibuf );
                            fprintf( dest_file, ">\n" );
                        }
                }
            }
        }
        fprintf( dest_file, ">\n" );
        fclose( scen_file );
        fclose( dest_file );
    }
    parser_free( &pd );
    return 1;
failure:
    parser_free( &pd );
    if ( aux_file ) fclose( aux_file );
    if ( scen_file ) fclose( scen_file );
    if ( dest_file ) fclose( dest_file );
    return 0;
}
