/* 
 *   Creation Date: <1999/03/13 23:09:47 samuel>
 *   Time-stamp: <2001/01/31 22:20:21 samuel>
 *   
 *	<console_video.c>
 *	
 *	Linux part of MacOS based video driver
 *   
 *   Copyright (C) 1999, 2000 Benjamin Herrenschmidt <bh40@calva.net>
 *
 *   Gamma implemented by Takashi Oe <toe@unlserve.unl.edu>
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

/* #define VERBOSE */

#include <sys/mman.h>
#include <sys/ioctl.h>
#include <linux/fb.h>
#include <pthread.h>
#include <math.h>

#include "verbose.h"
#include "console.h"
#include "memory.h"
#include "wrapper.h"
#include "res_manager.h"
#include "mouse_sh.h"
#include "video_module.h"
#include "os_interface.h"
#include "booter.h"

SET_VERBOSE_NAME("console_video")

typedef struct console_video_data
{
	int		fd;
	int		has_console;

	int		is_open;
	video_desc_t	vmode;		/* if open */

	short		*color_table;
	struct fb_cmap	cmap;

	struct fb_var_screeninfo startup_var;	
	struct fb_fix_screeninfo startup_fix;
	int		startup_mode_ok;

	double		rgamma, ggamma, bgamma;

	int		use_cache;
	int		blank;		/* display power state */
} console_video_data_t;

static console_video_data_t cv[1];

/* prototypes */
static void 	handle_switch(int console_fd, int activate);

static int 	console_video_init( video_module_t *mod );
static void 	console_video_cleanup( video_module_t *mod );
static void 	setcmap(char *pal);
static int	vopen( video_desc_t *mode );
static void	vclose( void );
static int	setup_video_modes( void );
static void	setup_gamma( void );

static int	set_powermode( int linux_powermode );
static int	osip_set_powermode( int, int * );

static video_desc_t video_modes[1];	/* only 1 mode for now */


/* Video module interface */
video_module_t console_video_module  = {
	"console_video",
	console_video_init,
	console_video_cleanup,
	vopen,
	vclose,
	NULL,			/* vrefresh */
	setcmap,

	NULL,			/* supported video modes */
	NULL			/* next */
};


/**************************************************************
*  	Implementation  
**************************************************************/

static int
console_video_init( video_module_t *m )
{
	char	*name;

	memset( cv, 0, sizeof(console_video_data_t) );

	if( !get_bool_res("enable_console_video") )
		return 1;

	if( (name = getenv("FRAMEBUFFER")) == NULL )
		name = "/dev/fb0";
	if( (cv->fd = open( name, O_RDWR )) < 0 ) {
		printm("Error opening frame buffer device '%s'\n",name );
		return 1;
	}
	cv->startup_mode_ok = !ioctl( cv->fd, FBIOGET_VSCREENINFO, &cv->startup_var )
		&& !ioctl( cv->fd, FBIOGET_FSCREENINFO, &cv->startup_fix );
	if( !cv->startup_mode_ok )
		printm("Warning: Startup video mode query failed\n");
	else
		VPRINT("Startup vmode %d x %d\n", cv->startup_var.xres, cv->startup_var.yres );

	if( !console_init() ) {
		close( cv->fd );
		return 1;
	}
	if( setup_video_modes() ){
		LOG("No usable console video modes were found\n");
		close( cv->fd );
		return 1;
	}
	
	cv->cmap.start = 0;
	cv->cmap.len = 256;
	cv->color_table = (short*)malloc( cv->cmap.len * sizeof(short) * 3 );
	cv->cmap.red = cv->color_table;
	cv->cmap.green = cv->color_table + 256;
	cv->cmap.blue = cv->color_table + 512;
	cv->cmap.transp = NULL;

	/* Read gamma correction values from prefs */
	setup_gamma();

	/* Install VESA DPMS handling routine */
	cv->blank = 0; /* assume the power is initially on */
	os_interface_add_proc( OSI_SET_VIDEO_POWER, osip_set_powermode );

	if( (cv->use_cache = (get_bool_res("use_fb_cache") != 0)) )
		printm("Cache enabled for console-video\n");

	/* Install the VC switch hook */
	console_switch_hook = handle_switch;
	return 0;
}


static void
console_video_cleanup( video_module_t *m ) 
{
	if (console_switch_hook == handle_switch)
		console_switch_hook = NULL;

	if( cv->is_open )
		vclose();
	if( cv->fd != -1 )
		close( cv->fd );
	if( cv->color_table )
		free( cv->color_table );

	if( video_modes[0].module_data )
		free( video_modes[0].module_data );

	console_cleanup();
}


static void
setup_gamma( void )
{
	double          val;
	char            *buf;

	cv->rgamma = 1.0;
	cv->bgamma = 1.0;
	cv->ggamma = 1.0;
	if ((buf = get_str_res("gamma")) != NULL) {
		val = atof(buf);
		if (val >= 0.1 && val <= 10)
			cv->rgamma = cv->ggamma = cv->bgamma = 1/val;
		else
			goto gamma_error;
		if ((buf = get_str_res_ind("gamma",0,2)) != NULL) {
			val = atof(buf);
			if (val >= 0.1 && val <= 10)
				cv->ggamma = 1/val;
			else
				goto gamma_error;
			if ((buf = get_str_res_ind("gamma",0,1)) != NULL) {
				val = atof(buf);
				if (val >= 0.1 && val <= 10) {
					cv->bgamma = 1/val;
					return;
				}
			}
			cv->ggamma = cv->rgamma;
		} else
			return;
gamma_error:
		printm("Bad resource value: gamma is expected to be one or"
		       " three values with >= 0.1 and <= 10.0\n");
	}
}


static void
add_mode( struct fb_var_screeninfo *var, int depth, int rowbytes, int offs, ulong refresh )
{
	video_desc_t *vm;

	vm = calloc( sizeof(video_desc_t)+sizeof(struct fb_var_screeninfo), 1 );
	vm->w = var->xres;
	vm->h = var->yres;
	vm->rowbytes = rowbytes;
	vm->offs = offs;
	vm->depth = std_depth( depth );
	var->bits_per_pixel = vm->depth;
	vm->refresh = refresh;
	vm->module_data = (void*)&vm[1];
	*(struct fb_var_screeninfo*)vm->module_data = *var;

	/* add mode to linked list */
	vm->next = console_video_module.modes;			
	console_video_module.modes = vm;
}

static int
setup_video_modes( void )
{
	char buf[160];
	char *vmodes_file = get_str_res("video_modes");
	struct fb_var_screeninfo v;
	int modes_added = 0;
	FILE *f=NULL;

	if( !vmodes_file )
		printm("---> molrc resource 'video_modes' is missing!\n");
	if( vmodes_file && (f=fopen( vmodes_file, "r")) ){
		int line, done=0, err, i;		
		for( line=0; !done; ){
			while( !(done=!fgets( buf, sizeof(buf), f )) ) {
				line++;
				if( buf[0] != '#' )
					break;
			}
			if( done )
				break;
			memset( &v, 0, sizeof(v) );
			err = sscanf(buf, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d", 
				     &v.xres, &v.yres, &v.xres_virtual, &v.yres_virtual, &v.xoffset, &v.yoffset,
				     &v.pixclock, &v.left_margin, &v.right_margin, &v.upper_margin, &v.lower_margin,
 				     &v.hsync_len, &v.vsync_len, &v.sync, &v.vmode ) != 15;
			v.xres_virtual = v.xres;
			v.yres_virtual = v.yres;
			v.xoffset = v.yoffset = 0;
			if( !err ) {
				int ndepths, depth, rowbytes, page_offs, refresh;
				line++;
				err = fscanf(f, "%d %d", &ndepths, &refresh ) != 2;
				for( i=0; i<3; i++ ) {
					if( !err )
						err = fscanf( f, "%d %d %d", &depth, &rowbytes, &page_offs ) != 3;
					if( !err && i < ndepths ) {
						modes_added++;
						add_mode( &v, depth, rowbytes, page_offs, refresh );
					}
				}
				fgets( buf, sizeof(buf), f );
			}
			if( err )
				printm("Line %d in '%s' is malformed.\n", line, vmodes_file );
		}
		fclose( f );
	}
	if( vmodes_file && !f ){
		printm("---> Could not open the video modes configuration file (%s)\n", vmodes_file );
	}
	
	if( !modes_added ) {
		if( cv->startup_mode_ok ){
			struct fb_fix_screeninfo *fix = &cv->startup_fix;
			v = cv->startup_var;
			add_mode( &v, v.bits_per_pixel, fix->line_length, (int)fix->smem_start & 0xfff, (75<<16) );
		}
	}
	
	return !console_video_module.modes;
}


static int
vopen( video_desc_t *vm )
{
	char buf[100];
	struct fb_var_screeninfo var;
	struct fb_fix_screeninfo fix;

	VPRINT("vopen\n");
	VPRINT("var = %p\n", vm->module_data );

	if( cv->is_open )
		return 1;
	var = *(struct fb_var_screeninfo*)vm->module_data;

	/* set resolution */
	var.activate = FB_ACTIVATE_NOW;
	if( !cv->has_console ) {
		/* we set correct resolution on the console to avoid
		 * an unnecessary mode switch when the user activates us
		 */
		if( ioctl( cv->fd, FBIOPUT_VSCREENINFO, &var ) < 0 )
			perrorm("FBIOPUT_VSCREENINO");
		/* always "fail" */
		return 1;
	} else if( !cv->has_console ){
		return 1;
	}

	console_printf("Opening console video %s\n", get_vmode_str(vm,buf,sizeof(buf)) );
	
	/* Put console in graphics mode and open display */
	console_set_gfx_mode(1);
	if( ioctl( cv->fd, FBIOPUT_VSCREENINFO, &var ) < 0 ){
		LOG("Could not set VSCREENINFO\n");
		goto bail;
	}

	/* is this panning necessary? */
        var.xoffset = 0;
        var.yoffset = 0;
        ioctl(cv->fd, FBIOPAN_DISPLAY, &var);

	if (ioctl(cv->fd, FBIOGET_FSCREENINFO, &fix) < 0) {
		LOG("Could not get FSCREENINFO!\n");
		goto bail;
	}

	if( vm->offs != ((ulong)fix.smem_start & 0xfff) ){
		LOG("Framebuffer offset has changed!\n");
		goto bail;
	}
	if( vm->rowbytes != fix.line_length ){
		LOG("Rowbytes has changed (was %d, now %d)!\n", vm->rowbytes, fix.line_length);
		goto bail;
	}

	/* runtime fields */
	vm->map_base = (ulong)fix.smem_start & ~0xfff;
        vm->lvbase = map_phys_mem( NULL, vm->map_base, FBBUF_SIZE(vm), PROT_READ | PROT_WRITE );
	vm->mmu_flags = MAPPING_PHYSICAL;

	if( is_newworld_boot() || is_oldworld_boot() )
		vm->mmu_flags |= MAPPING_DBAT;

	/* Use cache with write through bit set */
	vm->mmu_flags |= cv->use_cache? MAPPING_FORCE_CACHE : MAPPING_MACOS_CONTROLS_CACHE;
	use_hw_cursor(0);	/* no hw-cursor for now */

	cv->is_open = 1;
	cv->vmode = *vm;

	VPRINT("vopen - out\n");
	return 0;

bail:
	console_set_gfx_mode(0);
	console_printf("Unexpectedly failed to enable console video\n");
	return 1;
}

/* This function is sometimes called (indirectly) from the console thread */
static void
vclose( void )
{
	VPRINT("vclose\n");

	if( !cv->is_open )
		return;

	/* Make sure the display power is on when we leave */
	if ( cv->blank )
		set_powermode( 0 ); /* 0 -> power on */

	cv->is_open = 0;

	if( cv->vmode.lvbase ) {
		/* XXX: memset on the framebuffer causes the process to die ?!? */
		size_t size = (cv->vmode.h * cv->vmode.rowbytes)/sizeof(long);
		long *p = (long*)(cv->vmode.lvbase + cv->vmode.offs);
		while( size-- )
			*p++ = 0;
	}
	unmap_mem( cv->vmode.lvbase, FBBUF_SIZE( &cv->vmode ) );
	console_set_gfx_mode(0);

	console_printf("Closing console video.\n");
}

/* format is [R,G,B] x 256 */
static void 
setcmap(char *pal)
{
	int i;

	if( !cv->is_open ) {
		printm("setcmap: Console_video not open!\n");
		return;
	}
	for(i=0; i<256; i++ ){
#if 0
		cv->cmap.red[i] = (*pal++)<<8;
		cv->cmap.green[i] = (*pal++)<<8;
		cv->cmap.blue[i] = (*pal++)<<8;
#else
		cv->cmap.red[i] = (ulong)(65535.0 * pow( (double)(*pal++)/255.0, cv->rgamma ));
		cv->cmap.green[i] = (ulong)(65535.0 * pow( (double)(*pal++)/255.0, cv->ggamma ));
		cv->cmap.blue[i] = (ulong)(65535.0 * pow( (double)(*pal++)/255.0, cv->bgamma ));
#endif
	}
	if( cv->has_console )
		ioctl(cv->fd, FBIOPUTCMAP, &cv->cmap );
}

/* This routine is run by the console thread */
static void
handle_switch( int console_fd, int activate )
{
	VPRINT("switch hook, active: %d\n", activate);

	if( activate ) {
		VPRINT("ACTIVATE\n");
		cv->has_console = 1;
		if( cv->is_open )
			return;
		video_module_become( &console_video_module );
	} else {
		VPRINT("DE-ACTIVATE\n");
		cv->has_console = 0;
		if( !cv->is_open )
			return;
		video_module_yield( &console_video_module, 0 /* may not fail */ );
		if( cv->is_open )
			printm("Internal error: console video is still open!\n");
	}
}


static int
set_powermode( int linuxmode )
{
	VPRINT("set_powermode\n");

	if( !cv->is_open )
		return 1;

	if( cv->has_console )
		ioctl( cv->fd, FBIOBLANK, (unsigned long)linuxmode );
	return 0;
}

/************************************************************************/
/*	OSI Interface							*/
/************************************************************************/

/* from Mac OS's video.h */
enum {
	kAVPowerOff	= 0,
	kAVPowerStandby	= 1,	/* hsync off */
	kAVPowerSuspend	= 2,	/* vsync off */
	kAVPowerOn	= 3
};

/*
   Ideally, we'd want to include a kernel header like linux/console.h
   with proper defines and all and be done with it, but it's not so easy....
*/

/* VESA Blanking Levels */
#define VESA_NO_BLANKING        0
#define VESA_VSYNC_SUSPEND      1
#define VESA_HSYNC_SUSPEND      2
#define VESA_POWERDOWN          3

static int
osip_set_powermode( int sel, int *params )
{
	printm("osip_set_powermode\n");

	switch ( params[0] ) {
	case kAVPowerOff:
		cv->blank = VESA_POWERDOWN + 1;
		break;
	case kAVPowerStandby:
		cv->blank = VESA_HSYNC_SUSPEND + 1;
		break;
	case kAVPowerSuspend:
		cv->blank = VESA_VSYNC_SUSPEND + 1;
		break;
	case kAVPowerOn:
		cv->blank = 0;
		break;
	}

	set_powermode( cv->blank );

	return 0;
}
