/**************************************************************
*   
*   Creation Date: <1999-02-01 04:41:19 samuel>
*   Time-stamp: <2000/10/16 22:22:19 samuel>
*   
*	<thread.c>
*	
*	Thread manager
*   
*   Copyright (C) 1999, 2000 Samuel Rydh (samuel@ibrium.se)
*   
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation
*   
**************************************************************/

#include "mol_config.h"
#include <signal.h>
#include <pthread.h>

#include "debugger.h"
#include "thread.h"

/* GLOBALS */
pthread_t __main_th=0;

/* There seem to be a problem in glibc/the kernel,
 * which sometimes causes two threads to sleep waiting
 * on the same (unlocked) mutex. The workaround signals
 * each thread periodically to make sure no thread sleeps
 * forever.
 */
#define BUG_WORKAROUND

struct pool_thread
{
	pthread_t	thread_id;
	const char	*thread_name;

	volatile int	active;
	pthread_cond_t	active_cond;
	pthread_mutex_t	activate_mutex;
	
	volatile int	cancel;

	void 		(*func)(void*);
	void		*data;

	struct pool_thread 	*next;
	struct pool_thread 	*all_next;
};

static pthread_mutex_t pool_mutex;
static struct pool_thread *all_threads;
static struct pool_thread *free_threads;
static int inited = 0;

#define LOCK 	pthread_mutex_lock( &pool_mutex );
#define UNLOCK	pthread_mutex_unlock( &pool_mutex );

static void *thread_entry( struct pool_thread *th );

#ifdef BUG_WORKAROUND
static void *bug_workaround_entry( void *dummy );
static void bug_workaround_signal( int sig_num );

static pthread_t workaround_th_id;
#endif

void 
threadpool_init( void )
{
#ifdef BUG_WORKAROUND
	sigset_t set;
#endif
	pthread_mutex_init( &pool_mutex, NULL );

	all_threads = NULL;
	free_threads = NULL;

	__main_th = pthread_self();
	inited = 1;

	/* SIGALRM is used to wakeup a sleeping thread */
	signal( SIGALRM, (void*)bug_workaround_signal );
	sigemptyset( &set );
	sigaddset( &set, SIGALRM );
       	pthread_sigmask( SIG_UNBLOCK, &set, NULL );

#ifdef BUG_WORKAROUND
	pthread_create( &workaround_th_id, NULL, (void*)bug_workaround_entry, NULL );
#endif
}

/* It's important that schedule_cleanup is called before
 * the drivers are cleaned up.
 */
void
threadpool_cleanup( void )
{
	struct pool_thread *th, *next;

#ifdef BUG_WORKAROUND
	pthread_cancel( workaround_th_id );
	pthread_join( workaround_th_id, NULL );
#endif

	printm("Terminating threads...\n");
	for( th = all_threads; th; th=next ) {
		if( th->active )
			printm("thread '%s' is active!\n", th->thread_name ? th->thread_name : "<noname>");
		if( !th->cancel ) {
			/* sometimes pthread_cancel doesn't seem to work when 
			 * the thread is in the pthread_cond_wait() loop. Thus
			 * we wake it instead.
			 */
			th->cancel = 1;
			pthread_mutex_lock( &th->activate_mutex );
			pthread_cond_signal( &th->active_cond );
			pthread_mutex_unlock( &th->activate_mutex );

			pthread_cancel( th->thread_id );
			pthread_join( th->thread_id, NULL );
		}
		pthread_cond_destroy( &th->active_cond );
		pthread_mutex_destroy( &th->activate_mutex );
		next = th->all_next;
		free( th );
	}
	/* printm("Done\n"); */
	inited = 0;
}

int
kill_thread( pthread_t th_id )
{
	struct pool_thread *th;
	int i;
	
	LOCK;
	for( th=all_threads; th && th_id != th->thread_id; th=th->all_next )
		;
	if( !th ){
		printm("kill_thread called with unknown thread id %ld!\n", th_id);
		UNLOCK;
		return -1;
	}
	th->cancel = 1;

	if( th->active )
		pthread_cancel( th_id );
	for( i=0; i<15 && th->active; i++ )
		usleep( 100000 );
	if( th->active ){
		printm("Cancelation of thread '%s' failed. Using the big sledge\n", 
		       th->thread_name );
		pthread_kill( th_id, 9 );
		th->active = 0;
	}
	UNLOCK;

	pthread_join( th_id, NULL );
	return 0;
}

static void
thread_exit_func( void *_th )
{
	struct pool_thread *th = _th;
	th->active=0;

	/* this thread might still be in the active list */
}

pthread_t
create_thread( void (*func)(void*), void *data, const char *thread_name )
{
	struct pool_thread *th;

	if( !inited )
		return 0;

	LOCK;
	if( !free_threads ){
		th = calloc( sizeof( struct pool_thread ),1);

		/* insert in lists */
		th->all_next = all_threads;
		all_threads = th;
		
		/* build */
		pthread_cond_init( &th->active_cond, NULL );
		
		if( pthread_create( &th->thread_id, NULL, (void*)thread_entry, (void*)th )){
			printm("Failed creating thread...exiting\n");
			UNLOCK;
			exit(1);
		}
	} else {
		th = free_threads;
		free_threads = th->next;
	}

	/* wake thread */
	pthread_mutex_init( &th->activate_mutex, NULL );
	th->data = data;
	th->func = func;
	th->thread_name = thread_name;
	th->active = 1;
	pthread_mutex_lock( &th->activate_mutex );
	pthread_cond_signal( &th->active_cond );
	pthread_mutex_unlock( &th->activate_mutex );

	UNLOCK;
	return th->thread_id;
}

static void *
thread_entry( struct pool_thread *th )
{
	set_thread_sigmask();

	pthread_cleanup_push( thread_exit_func, th );
	for( ;; ) {
		pthread_mutex_lock( &th->activate_mutex );
		while( !th->active && !th->cancel )
			pthread_cond_wait( &th->active_cond, &th->activate_mutex );
		pthread_mutex_unlock( &th->activate_mutex );

		if( th->cancel )
			break;
		
		th->func( th->data );
		th->active = 0;

		/* Add thread to free list */
		LOCK;
		if( !th->cancel ) {
			th->next = free_threads;
			free_threads = th;
		}
		UNLOCK;
	}
	pthread_cleanup_pop( 1 /*execute */ );
	return NULL;
}

const char *
get_thread_name( void )
{
	struct pool_thread *th;
	pthread_t self = pthread_self();

	for( th = all_threads; th; th=th->all_next ){
		if( th->thread_id == self )
			return th->thread_name ? th->thread_name : "<noname thread>";
	}
	if( self == __main_th )
		return "main-thread";

	return "<non-registered thread>";
}

int
is_main_thread( void )
{
	return (pthread_self() == __main_th) || !inited;
}

/* 
 * This function should be used to setup the default signal mask for all 
 * threads ever created.
 */
void
set_thread_sigmask( void )
{
	sigset_t set;

	/* Best to block SIGINT & SIGPROF */
        sigemptyset( &set );
        sigaddset( &set, SIGINT );
        sigaddset( &set, SIGPROF );
	sigaddset( &set, SIGIO );
	
	pthread_sigmask( SIG_BLOCK, &set, NULL );

	sigemptyset( &set );
	sigaddset( &set, SIGALRM );
       	pthread_sigmask( SIG_UNBLOCK, &set, NULL );
}



#ifdef BUG_WORKAROUND
static void 
bug_workaround_signal( int sig_num )
{
/*	printm("ALRM %ld\n", pthread_self() );*/
}


static void *
bug_workaround_entry( void *dummy ) 
{
	struct pool_thread *th;
	int odd=1;

	set_thread_sigmask();

	for( ;; ){
		odd = !odd;
		sleep(1);

		pthread_testcancel();
		
		pthread_kill( __main_th, SIGALRM );
		for( th=all_threads; odd && th; th=th->all_next )
			pthread_kill( th->thread_id, SIGALRM );
	}
	return NULL;
}
#endif
