/* 
 *   Creation Date: <1999/02/18 22:36:07 samuel>
 *   Time-stamp: <2001/09/29 22:31:10 samuel>
 *   
 *	<timer.c>
 *	
 *	Time and timer handling
 *   
 *   Copyright (C) 1999, 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"
#include <pthread.h>
#include <time.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <signal.h>

#include "timer.h"
#include "thread.h"
#include "mac_registers.h"
#include "molcpu.h"
#include "debugger.h"
#include "promif.h"
#include "../drivers/include/mouse_sh.h"
#include "video.h"
#include "res_manager.h"

//#define TEST_TIMER

typedef struct timer_entry {
	int			is_periodic;

	struct timer_entry 	*next;

	ullong			absticks;

	void			*usr;
	timer_fp 		handler_proc;
	ulong			id;

	/* for periodic timers */
	ullong			mperiod;
	int			allow_skip;
} mtimer_t;

static mtimer_t  		*active_root;
static mtimer_t  		*inactive_root;
static mtimer_t			*free_root;

ulong				__ticks_per_usec_num;
ulong				__ticks_per_usec_den;

volatile int			__recalc_timer;
volatile int 	 		__abort_doze;
static volatile int		is_dozing;

#define MAX_NUM_TIMERS	10
static mtimer_t			timer_slots[MAX_NUM_TIMERS];

static ulong 			next_id;
static ulong			dec_freq;
static pthread_mutex_t 		timer_lock_mutex = PTHREAD_MUTEX_INITIALIZER;

#define DEC_FREQ_601		1000000000

#define MAX_TIMER		0x7fffffff

#define LOCK			pthread_mutex_lock( &timer_lock_mutex );
#define UNLOCK			pthread_mutex_unlock( &timer_lock_mutex );

#define SIGNAL_TIMER_RECALC()	{ __recalc_timer=1; interrupt_emulation(); }

#ifdef TEST_TIMER
static void start_test_timer( void );
#endif

static ulong get_dec_frequency( void );


/************************************************************************/
/*	Initialization							*/
/************************************************************************/

void
timer_init( void )
{
	int i;

	memset( timer_slots, 0, sizeof(struct timer_entry)*MAX_NUM_TIMERS );
	for(i=1; i<MAX_NUM_TIMERS; i++ )
		timer_slots[i-1].next = &timer_slots[i];
	
	free_root = &timer_slots[0];
	active_root = NULL;
	inactive_root = NULL;
	next_id = 0;

	mregs->timer_stamp = get_tbl() + MAX_TIMER;

	dec_freq = get_dec_frequency();
	printm("Timebase frequency: %ld.%02ld MHz\n", 
	       dec_freq/1000000, dec_freq/10000 % 100 );

	__ticks_per_usec_num = dec_freq;
	__ticks_per_usec_den = 1000000;

#ifdef TEST_TIMER
	start_test_timer();
#endif
}


void
timer_cleanup( void )
{
	LOCK;
	active_root = NULL;
	UNLOCK;
}

static ulong
fast_calibrate_dec( void )
{
	struct timeval tv, tv2;
	ulong a,b;

	gettimeofday( &tv, NULL );
	a = get_tbl();
	for( ;; ) {
		gettimeofday( &tv2, NULL );
		b = get_tbl();
		while( tv2.tv_sec-- - tv.tv_sec )
			tv2.tv_usec += 1000000;
		if( tv2.tv_usec - tv.tv_usec > 1000000/10 )
			break;
	}
	return (b-a)*10;
}

static ulong
calibrate_dec( char *err )
{
	struct timeval tv, tv2;
	ulong a,b, val;
	int i;
	
	if( err )
		printm("%s", err );

	printm( "*******************************************************\n"
		"* The timebase frequency for this machine is unknown.\n"
		"* Calibrating - this will take 5 seconds.\n"
		"*******************************************************\n");

	/* this calibration is not perfect (but quite good) */
	val = -1;
	for( i=0; i<5; i++ ) {
		gettimeofday( &tv, NULL );
		a = get_tbl();
		for( ;; ) {
			gettimeofday( &tv2, NULL );
			b = get_tbl();
			if( tv2.tv_sec - tv.tv_sec == 1 && tv2.tv_usec - tv.tv_usec > 0 )
				break;
			if( tv2.tv_sec - tv.tv_sec > 1 )
				break;
		}
		if( val > b-a )
			val = b-a;
		printm("    %08lX\n", b-a);
	}
	printm( "*******************************************************\n"
		"* Adding 'timebase_frequency: 0x%08lX' to\n"
		"* /etc/molrc/ will prevent this calibration delay\n"
		"*******************************************************\n", val );
	for(i=0; i<3; i++ )
		sleep(1);
	return val;
}


static ulong
get_dec_frequency( void )
{
	ulong	val, *ptr;
	mol_device_node_t *cpu;
	
	if( (val=get_numeric_res("timebase_frequency")) != -1 ) {
		printm("WARNING: Using timebasefrequency 0x%08lX from /etc/molrc\n", val);
		return val;
	}

	/* We use the timebase property of the OF-tree to find out
	 * how fast the DEC register is ticking. On the 601,
	 * DEC always counts down by 128 every 128 ns. OF returns the 
	 * frequency of the least significant bit for the 601.
	 *
	 * DEC and TB uses the same timbase (the 601 has a 
	 * no-so-fun RTC register instead of TB)
	 */

	/* Use the kernel value (if available) */
	if( (val=_get_tb_frequency()) )
		return val;

	if( !(cpu = prom_find_type_hw( "cpu" )) )
		return calibrate_dec("No type 'cpu' in device tree\n");

	if( !(ptr=(ulong*)prom_get_property( cpu, "cpu-version", NULL )) )
		return calibrate_dec("No property 'cpu-version'\n");

	if( *((short*)ptr) == 1 )
		return DEC_FREQ_601;

	if( !(ptr = (ulong*)prom_get_property( cpu, "timebase-frequency", NULL )) )
		return calibrate_dec("No 'timebase-frequency' property.\n");
	
	val = fast_calibrate_dec();
	printm("Timebase frequency estimate: %08lX\n", val );

	/* If the approximation is off by more than 5 percent, make a full calibration */ 
	if( abs(val - *ptr) > *ptr/20 )
		return calibrate_dec("The timebase-frequency property seems to be bogus\n");
	
	return *ptr;
}


/************************************************************************/
/*	Timer and doze stuff						*/
/************************************************************************/

void
doze( void )
{
	ullong now;
	ulong usec;
	ulong sl;
	
	now = get_mticks_();
	if( console_video_active() && now - get_mouse_moved_stamp() < usecs_to_mticks_(1000000/15) ) {
		/* perhaps we should sleep _sometimes_ here (10% of the times or so) ? */
		return;
	}

	LOCK;
	usec = 1000;
	if( active_root ) {
		ulong dec = mregs->dec_stamp - (ulong)now;
		if( (sl=active_root->absticks - now) > dec )
			sl = dec;
		if( dec <= 0 )
			__abort_doze=1;
		usec = mticks_to_usecs_( sl );
	}

	/* There is a non-serious racing condition here which
	 * could be fixed by using pthread_cond_wait. However,
	 * that routine seems to be buggy (signals are lost).
	 */
	is_dozing = 1;
	UNLOCK;
	if( !__abort_doze ) {
		/* printm("dozing in %ld usecs...\n", usec); */
		usleep(usec);
	}
	__abort_doze = 0;
	is_dozing = 0;

	/* not strictly necessary, but it might save us a context switch */
	_timer_expired();
}

/* abort_doze ensures that the main-loop is run at least once */
void
abort_doze( void )
{
	__abort_doze = 1;
	if( is_dozing )
		pthread_kill( get_main_th(), SIGALRM );
}


/* main thread */
void
_timer_expired( void )
{
	ullong mt;

	__recalc_timer = 0;
	mregs->flag_bits &= ~fb_TimerINT;

	LOCK;
	mt = get_mticks_();

	while( active_root && active_root->absticks <= mt ) {
		mtimer_t 	*r = active_root;
		timer_fp 	tproc = r->handler_proc;
		void 		*usr = r->usr;
		ulong 		id = r->id;

		/* dequeue */
		active_root = active_root->next;

		if( r->is_periodic ) {
			r->next = inactive_root;
			inactive_root = r;
		} else {
			r->next = free_root;
			free_root = r;
		}
		UNLOCK;
		tproc( id, usr );
		LOCK;

		/* best to update mt (we use realtime...) */
		mt = get_mticks_();
	}

	/* calculate next timer event */
	mregs->timer_stamp = (active_root)?
		active_root->absticks : (ulong)mt + MAX_TIMER;
	UNLOCK;

	/* Flag a kernel recalculation of mol-DEC */
	mregs->flag_bits |= (fb_CheckFlags | fb_RecalcDecInt);
}


/* any thread */
void
cancel_timer( ulong id )
{
	struct timer_entry **tc;

	LOCK;

	for( tc=&active_root; *tc; tc = &(*tc)->next ){
		if( (**tc).id == id ) {
			mtimer_t *t = *tc;
			*tc = (**tc).next;
			t->next = free_root;
			free_root = t;
			break;
		}
	}
	for( tc=&inactive_root; *tc; tc = &(*tc)->next ){
		if( (**tc).id == id ) {
			mtimer_t *t = *tc;
			*tc = (**tc).next;
			t->next = free_root;
			free_root = t;
			break;
		}
	}
	UNLOCK;

	SIGNAL_TIMER_RECALC();
}

/* any thread, LOCK held. The timer t must already be dequeued. */
static int
activate_timer( mtimer_t *t, ullong mt )
{
	mtimer_t **tc;

	/* we keep an ordered chain */
	for( tc = &active_root; *tc && (**tc).absticks < mt ; tc = &(*tc)->next )
		;

	t->next = *tc;
	*tc = t;
	t->absticks = mt;

	/* caller must do 'SIGNAL_TIMER_RECALC();' when the lock is released */
	return 0;
}


int
settimer_abs_mticks( ullong mt, timer_fp tproc, void *usr, int activate )
{
	ulong ret_id;
	mtimer_t *t;

	LOCK;
	if( (t=free_root) == NULL ){
		printm("No free timer slots!\n");
		UNLOCK;
		return -1;
	}
	free_root = free_root->next;

	memset( t, 0, sizeof(mtimer_t) );
	t->usr = usr;
	t->handler_proc = tproc;
	ret_id = t->id = ++next_id;

	if( activate )
		activate_timer( t, mt );
	else {
		t->next = inactive_root;
		inactive_root = t;
	}

	UNLOCK;

	SIGNAL_TIMER_RECALC();
	return ret_id;
}


/************************************************************************/
/*	Periodic timers (ticking realtime)				*/
/************************************************************************/

int
new_ptimer( timer_fp tproc, void *usr )
{
	return settimer_abs_mticks( 0, tproc, usr, 0 );
}

void
free_ptimer( int id )
{
	cancel_timer( id );
}

int
restart_ptimer( int id, ulong uperiod, int allow_skip )
{
	mtimer_t **tc, *tm;
	
	LOCK;
	for( tc=&inactive_root; *tc && (*tc)->id != id ; tc = &(**tc).next )
		;
	if( !*tc ){
		printm("restart_ptimer: No such inactive timer\n");
		UNLOCK;
		return -1;
	}
	tm = *tc;
	*tc = tm->next;
	
	tm->is_periodic = 1;
	tm->allow_skip = allow_skip;
	tm->mperiod = usecs_to_mticks_( uperiod );

	activate_timer( tm, tm->mperiod + get_mticks_() );
	UNLOCK;

	SIGNAL_TIMER_RECALC();
	return 0;
}

int
resume_ptimer( int id )
{
	ullong t;
	mtimer_t **tc, *tm;
	
	LOCK;
	for( tc=&inactive_root; *tc && (*tc)->id != id ; tc = &(**tc).next )
		;
	if( !*tc ){
		printm("resume_ptimer: No such inactive timer\n");
		UNLOCK;
		return -1;
	}
	tm = *tc;
	*tc = tm->next;

	t = tm->absticks + tm->mperiod;
	if( tm->allow_skip ) {
		ullong mt = get_mticks_();
		while( t < mt )
			t += tm->mperiod;
	}
	activate_timer( tm, t );
	UNLOCK;

	SIGNAL_TIMER_RECALC();
	return 0;
}


/************************************************************************/
/*	TESTING								*/
/************************************************************************/

#ifdef TEST_TIMER

static void 
timer_test2( ulong id, void *usr )
{
	static int i=0;
	printm("Timer-%ld [%d]\n", id, ++i );
	settimer_usecs( 1000000, timer_test2, NULL );
}

static void
start_test_timer( void )	
{
	settimer_usecs( 1000000, timer_test2, NULL );
}
#endif
