// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/vtune.cpp,v 1.2 2001/08/13 10:00:42 xhshi Exp $
//


//#define _TEST_java_sourcefile

#include "defines.h"
#include <iostream.h>
#include <sys/types.h>
#include "vtune.h"


////////////////////////////////////////////////////////////////////////////
// data
////////////////////////////////////////////////////////////////////////////
iJIT_ModeFlags VTuneModeFlags = (iJIT_ModeFlags)-1;
////////////////////////////////////////////////////////////////////////////
// functions
////////////////////////////////////////////////////////////////////////////
void set_VTuneModeFlags(iJIT_ModeFlags flags) {
  //
  // set_VTuneModeFlags() is a call-back function for VTune
  //
    VTuneModeFlags = flags;
//    VTuneModeFlags = (iJIT_ModeFlags)-1;
}

void pError(char *fmt, ...) {
  //
  // pError: prints error message and terminate the program
  //
    char buf[4096];
    va_list ap;

    va_start(ap, fmt);
    vsprintf(buf, fmt, ap);
    fprintf(stderr,"%s\n",buf);
    va_end(ap);

    exit(-1);
}

class classpath_t {
    char **classpath;
    int max_classpath_entries;
    int n_entries;

  public:
    classpath_t(int max_classpath_entries=1000) {
        classpath = new char *[max_classpath_entries];
        this->max_classpath_entries = max_classpath_entries;
        this->n_entries = 0;
    }
    ~classpath_t() {
        for(int i=0; i<n_entries; i++)
            if (classpath[i]) {
                free(classpath[i]);
                classpath[i] = NULL;
            }
        delete classpath;
    }
    void add_a_classpath(char *path1, char *path2="") {
      //
      // add_a_classpath
      //
        if (!*path1)
            return;
        char path[1024];
        if (*path2=='\\')
            path2++;
        if (*path1 && path1[strlen(path1)-1]!='\\') {
            sprintf(path,"%s\\%s",path1,path2);
        } else
            sprintf(path,"%s%s",path1,path2);

        for(int i=0; i<n_entries; i++)
            if (strcmpi(classpath[i],path)==0)
                break;

        if (!(i<n_entries)) {
            if (n_entries>=max_classpath_entries)
                pError("*** classpath_t::add_classpath: too many classpath\n");
            classpath[n_entries++] = strdup(path);
        }
    }
    void add_classpath(char *path, bool from_jvc=false) {
      //
      // add_classpath()
      //
        if (!path || !*path)
            return;

        char *q=path,*p;
        do {
            #define CLASSPATH_WHITE_SPACES    "; \t\n\r"
            p = q + strspn(q,CLASSPATH_WHITE_SPACES);
            q = p + strcspn(p,CLASSPATH_WHITE_SPACES);
            if (*q)
                *q++ = 0x00;
          //
          // remove *.zip
          //
            char *r;
            if ( ((r=strstr(p,".zip"))!=NULL || (r=strstr(p,".ZIP"))!=NULL) &&
                 r[4]==0x00 ) {
                if ((r=strrchr(p,'\\'))!=NULL) {
                    *r++ = 0x00;
                    add_a_classpath(p);
                    if (from_jvc && strcmpi(r,"tclasses.zip")==0) {
                    } else if (from_jvc && strcmpi(r,"classes.zip")==0) {
                    }
                } else if (((toupper(p[0])-'A') <= 'Z'-'A') && p[1]==':') {
                    p[2] = 0x00;
                } else
                    *p = 0x00;
            } else
                add_a_classpath(p);
        } while (*(p=q)!=0x00);
    }
    bool fullpath(char *path, int path_len, char *java_classfile) {
      //
      // fullpath()
      //
        for(int i=-1; i<n_entries; i++) {
            if (i==-1) {
                sprintf(path,"%s",java_classfile);
            } else
                sprintf(path,"%s\\%s",classpath[i],java_classfile);
            char *p;
            if (!(((p=strstr(path,".class"))!=NULL) && p[6]==0x00))
                strcat(path,".class");
            struct _stat st;
            if (_stat(path,&st)!=0)
                continue;
            if (st.st_mode & _S_IFREG)
                return true;
        }
        return false;
    }
};
classpath_t classpath;

void settle_classpath() {
  //
  // settle_classpath() looks at several places to hint about classpath
  //                    Zip file isn't supported yet.
  //
    char buf[8192];
  //
  // Hint I: env var. IJCLASSPATH
  //
    char *env = getenv("IJCLASSPATH");
    if (env) {
        env = strdup(env);
        classpath.add_classpath(env);
        delete env;
    }
  //
  // Hint III: hard-wired common path
  //
    char sysdir[512];
    char *default_path = buf;
    GetWindowsDirectory(sysdir,sizeof(sysdir));
    sprintf(default_path,
        "%s\\java\\trustlib\\tclasses.zip;"
        "%s\\java\\trustlib;"
        "%s\\java\\classes\\classes.zip;"
        "%s\\java\\classes;"
        ".;"
        "%s\\java\\lib;",sysdir,sysdir,sysdir,sysdir,sysdir);
    classpath.add_classpath(default_path,true);
}

char *sprint_cp_Utf8(char *buf, int sizeof_buf, char *utf8) {
  //
  // sprint_cp_Utf8() assume bit-7 of all Utf8 char is zero, meaning it's
  //                  ASCII char
  // NOTE:
  //    utf8 :  length(2-byte) + length-bytes
  //
    u2 length = *(u2*)utf8;
    if (length >= sizeof_buf)
        pError("*** sprinf_cp_Utf8: utf8 string too length\n -> %s\n",utf8+2);
    memcpy(buf,utf8+2,length);
    buf[length] = 0x00;
    return buf;
}

char *cp_str[13] = {
    "(nono)",
    "Utf8",
    "Unicode",
    "Integer",
    "Float",
    "Long",
    "Double",
    "Class",
    "String",
    "FieldRef",
    "MethodRef",
    "InterfaceMethodRef",
    "NameAndType"
};

char *java_sourcefile(char *filename_buf, int sizeof_filename_buf,
                    char *java_classfile,
                    char *method_name, size_t code_len,
                    int *lineNo, lineInfo_t **lineInfo) {
  //
  // java_sourcefile() return the .java file name compiled to java_classfile
  //
    int i, fpos = 0;
    file_t vf;
    char path[512];

    if (!classpath.fullpath(path,sizeof(path),java_classfile))
        return NULL;

    if (!vf.openfile(path))
        return NULL;

    if (method_name) {
        assert(lineNo);
        assert(lineInfo);
        *lineNo = 0;
        *lineInfo = NULL;
    }

    vf.rewind();
  //
  // magic, minor/major version
  //
    u4 magic = vf.read_u4(&fpos);    assert(magic==0xCAFEBABE);
    u2 minor_version = vf.read_u2(&fpos);
    u2 major_version = vf.read_u2(&fpos);
  //
  // constant pool
  //
    u2 constant_pool_count = vf.read_u2(&fpos);
    int *const_pool_ofs = NULL;
    void **const_pool_info = NULL;
    if (constant_pool_count>1) {
        const_pool_ofs = new int[constant_pool_count];
        const_pool_info = new void *[constant_pool_count];
        const_pool_ofs[0] = -1;     // 0th is not user-defined
        const_pool_info[0] = NULL;
        for(i=1; i<constant_pool_count;i++) {
            const_pool_info[i] = NULL;
            const_pool_ofs[i] = fpos;
            u1 type = vf.read_u1(&fpos);
            //printf(" i = %d, cp type = %s\n",i,cp_str[type]);
            switch(type) {
              case  1:  // Utf8
              case  2:{ // Unicode
                    u2 length = vf.read_u2(&fpos);
                    char **ptr = (char **)&(const_pool_info[i]);
                    *ptr = new char[length+2];
                    *(u2*)*ptr = length;
                    vf.read((*ptr)+2,length,&fpos);

                 }  break;
              case  5:  // Long
              case  6:  // Double
                    vf.skip(8,&fpos);
                    i++;    // also take the next slot
                    const_pool_info[i] = NULL;
                    const_pool_ofs[i] = -1;

                    break;
              case  7:  // Class
              case  8:  // String
                    vf.skip(2,&fpos);

                    break;
              case  3:  // Integer
              case  4:  // Float
              case  9:  // FieldRef
              case 10:  // MethodRef
              case 11:  // InterfaceMethodRef
              case 12:  // NameAndType
                    vf.skip(4,&fpos);

                    break;
                default:
                    assert(0);
            } // switch
        } // for(int i=1...)
    }
  //
  // access, class, superclass
  //
    u2 access_flags = vf.read_u2(&fpos);
    u2 this_class  = vf.read_u2(&fpos);
    u2 super_class = vf.read_u2(&fpos);
  //
  // implemented interfaces
  //
    u2 implemented_interface_count = vf.read_u2(&fpos);
    vf.skip(2*implemented_interface_count,&fpos);

  //
  // fields
  //
    u2 field_count = vf.read_u2(&fpos);
    for(i=0; i<field_count; i++) {
        u2 access_flags = vf.read_u2(&fpos);
        u2 name_index = vf.read_u2(&fpos);
        u2 type_index = vf.read_u2(&fpos);
        u2 attrib_count = vf.read_u2(&fpos);
        for(int j=0; j<attrib_count; j++) {
            u2 attr_name_index = vf.read_u2(&fpos);
            u4 attr_length = vf.read_u4(&fpos);
            vf.skip(attr_length,&fpos);
        }
    }

  //
  // methods
  //
    char buf[512];
    u2 method_count = vf.read_u2(&fpos);
    for(i=0; i<method_count; i++) {
        u2 access_flags = vf.read_u2(&fpos);
        u2 name_index = vf.read_u2(&fpos);
        u2 type_index = vf.read_u2(&fpos);
        u2 attrib_count = vf.read_u2(&fpos);

        bool method_name_matched = false;
        if (method_name) {
            sprint_cp_Utf8(buf,sizeof(buf),(char*)const_pool_info[name_index]);
            if (strcmp(buf,method_name)==0)
                method_name_matched = true;
        }
        for(int j=0; j<attrib_count; j++) {
            u2 attr_name_index = vf.read_u2(&fpos);
            u4 attr_length = vf.read_u4(&fpos);
            sprint_cp_Utf8(buf,sizeof(buf),(char*)const_pool_info[attr_name_index]);
            if (method_name_matched && strcmp(buf,"Code")==0) {
                u2 max_stack = vf.read_u2(&fpos);
                u2 max_locals = vf.read_u2(&fpos);
                u4 code_length = vf.read_u4(&fpos);
                if (code_len==code_length) {
                    vf.skip(code_length,&fpos);
                    u2 code_exception_count = vf.read_u2(&fpos);
                    vf.skip(sizeof(codeException_t)*code_exception_count,&fpos);
                    u2 code_attrib_count = vf.read_u2(&fpos);
                    for(int a=0; a<code_attrib_count; a++) {
                        u2 code_attr_name_index = vf.read_u2(&fpos);
                        u4 code_attr_length = vf.read_u4(&fpos);
                        sprint_cp_Utf8(buf,sizeof(buf),(char*)const_pool_info[code_attr_name_index]);
                        if (strcmp(buf,"LineNumberTable")==0) {
                         //
                         // read in linno
                         //
                            *lineNo = vf.read_u2(&fpos);
                            *lineInfo = (lineInfo_t *)malloc(sizeof(lineInfo_t) * *lineNo);
                            for(int l=0; l<*lineNo; l++) {
                                (*lineInfo)[l].start_pc = vf.read_u2(&fpos);
                                (*lineInfo)[l].line_number = vf.read_u2(&fpos);
                            }
                        } else
                            vf.skip(code_attr_length,&fpos);
                    }
                } else
                    vf.skip(attr_length-sizeof(max_stack)-sizeof(max_locals)-sizeof(code_length),&fpos);
            } else
                vf.skip(attr_length,&fpos);
        }
    }
  //
  // attributes
  //
    u2 sourcefile_index = 0;
    u2 attrib_count = vf.read_u2(&fpos);
    for(i=0; i<attrib_count; i++) {
        u2 attr_name_index = vf.read_u2(&fpos);
        u4 attr_length = vf.read_u4(&fpos);
        sprint_cp_Utf8(buf, sizeof(buf), (char*)const_pool_info[attr_name_index]);
        if (strcmp(buf,"SourceFile")==0) {
            assert(sourcefile_index==0);
            sourcefile_index = vf.read_u2(&fpos);
            sprint_cp_Utf8(filename_buf,sizeof_filename_buf,
                                    (char*)const_pool_info[sourcefile_index]);
        } else
            vf.skip(attr_length,&fpos);
    }
    if (sourcefile_index==0)
        pError("*** java_sourcefile: no \"SourceFile\" attrib\n");
  //
  // done
  //
    assert(fpos==vf.size());
    if (const_pool_ofs)
        delete const_pool_ofs;
    if (const_pool_info) {
        for(i=0; i<constant_pool_count; i++)
            if (const_pool_info[i])
                delete const_pool_info[i];
        delete const_pool_info;
    }

    return filename_buf;
}


#ifdef _TEST_java_sourcefile
void main(int argc, char *argv[]) {
    if (argc<4) {
        printf("Usage: %s classfilename method_name code_len\n",argv[0]);
        exit(1);
    }
    settle_classpath();
    char *classfile = argv[1];
    char *method_name = argv[2];
    size_t code_len = atoi(argv[3]);
    char buf[512];
    int lineNo;
    lineInfo_t *lineInfo;
    char *sourcefile = java_sourcefile(buf,sizeof(buf),classfile,
                                    method_name, code_len, &lineNo,&lineInfo);
    printf("classfile   : \"%s\"\n"
           "sourcefile  : \"%s\"\n"
           "method_name : \"%s\"\n"
           "code_len    : %d\n", classfile, sourcefile,method_name,code_len);
    if (lineNo)
        for(int i=0; i<lineNo; i++)
            printf("   line %3d: %d\n",
                        lineInfo[i].line_number,
                        lineInfo[i].start_pc);
}
#endif // _TEST_java_sourcefile
