// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/flow_graph_exceptions.cpp,v 1.2 2001/08/13 09:54:55 xhshi Exp $
//

#include "defines.h"
#include <stdlib.h>
#include <string.h>
#include "flow_graph.h"
#include "jit_intf.h"
#include "is_subclass_of.h"  // will go away when the ORP exports the required functionality

/******************************************************************/
/* VM-specific routines below. */
/******************************************************************/


static unsigned vm_num_exception_handlers(Method_Handle m_handle, Eh_Node *caller_eh)
{
    unsigned result = method_get_num_handlers(m_handle);
    if (caller_eh != NULL)
        result += caller_eh->out_edge_size();
    return result;
}

static Class_Handle vm_get_class_handle(Class_Handle c_handle, unsigned idx)
{
    if (idx == 0)
        return NULL;
    return resolve_class_from_constant_pool(c_handle, idx);
}

Eh_Node *Flow_Graph::create_eh_node() {
    return new(mem_manager) Eh_Node(_next_eh_label++);
}

void Flow_Graph::vm_get_nth_eh_info(Eh_Node *caller_eh, unsigned nth,
                                    unsigned &begin, unsigned &end,
                                    Cfg_Node *&handler, Class_Handle &type,
                                    unsigned &cp_index,
                                    Return_Address_Tracking *stk)
{
    unsigned num_caller_eh_handlers = (caller_eh == NULL ? 0 : caller_eh->out_edge_size());
    unsigned num_method_handlers = method_get_num_handlers(_m_handle);
    if (nth < num_method_handlers)
    {
        unsigned tmp_handler;
        method_get_handler_info(_m_handle, nth, &begin, &end, &tmp_handler, &cp_index);
        handler = create_flow_graph(tmp_handler, stk);
        type = vm_get_class_handle(_c_handle, cp_index);
    }
    else
    {
        begin = 0;
        end = bc_length();
        handler = caller_eh->out_edges(nth-num_method_handlers)->handler;
        type = caller_eh->out_edges(nth-num_method_handlers)->class_handle;
        cp_index = caller_eh->out_edges(nth-num_method_handlers)->cp_index;
    }
}

void Flow_Graph::create_flow_graph_handlers(Return_Address_Tracking *stk)
{
    unsigned num_eh = vm_num_exception_handlers(_m_handle, NULL);
    unsigned i;
    unsigned begin, end;
    Cfg_Node *handler;
    Class_Handle type;
    unsigned cp_index;
    for (i=0; i<num_eh; i++)
    {
        stk->reset();
        stk->push();  // exception handler begins with one object on stack
        vm_get_nth_eh_info(NULL, i, begin, end, handler, type, cp_index, stk);
    }
}

/******************************************************************/
/* VM-specific routines above. */
/******************************************************************/

static int compare_unsigned(const void *e1, const void *e2)
{
    return *(unsigned*)e1 - *(unsigned*)e2;
}

// Returns nonzero if ch1 overrides ch2; that is, ch1 is more general
// than ch2, making ch2 redundant.
static int overrides(Class_Handle ch1, Class_Handle ch2)
{
    if (ch1 == NULL || ch1 == ch2)
        return 1;
    if (ch2 == NULL)
        return 0;
    return O3_is_subclass_of(ch2,ch1);
}

struct eh_data
{
    Class_Handle class_handle;
    unsigned cp_index;
    Cfg_Node *handler;
};
struct endpoints
{
    unsigned endpoint;
    int capacity;
    int size;
    struct eh_data *entries;
};

// caller_eh is the Eh_Node of the method doing the inlining.
// To deal with this, we pretend that the exception handler
// information returned from the VM contains this information.
void Flow_Graph::add_exception_info(Eh_Node *caller_eh)
{
    unsigned num_eh = vm_num_exception_handlers(_m_handle, caller_eh);
    if (num_eh == 0)
        return;
    
    // Create the "endpoint array".  A conservative size estimate is
    // double the number of exception handlers.
    unsigned ea_size = 2 * num_eh;
    unsigned *endpoint_array = (unsigned int *)mem_manager.alloc(ea_size*sizeof(*endpoint_array));
    unsigned i;
    
    // Fill and sort the endpoint array.
    for (i=0; i<num_eh; i++)
    {
        unsigned begin, end;
        Cfg_Node *handler;
        Class_Handle type;
        unsigned cp_index;
        vm_get_nth_eh_info(caller_eh, i, begin, end, handler, type, cp_index, NULL);
        endpoint_array[2*i] = begin;
        endpoint_array[2*i+1] = end;
    }
    qsort(endpoint_array, ea_size, sizeof(*endpoint_array), compare_unsigned);
    
    // Remove duplicate endpoints.
    unsigned src, target;
    for (src=target=0; src<ea_size; src++)
    {
        if (endpoint_array[src] != endpoint_array[target])
        {
            target ++;
            endpoint_array[target] = endpoint_array[src];
        }
    }
    ea_size = target + 1;
    
    // Create and initialize the array for the endpoint data.
    struct endpoints *endpoint_data = (struct endpoints *)mem_manager.alloc(ea_size*sizeof(*endpoint_data));
    for (i=0; i<ea_size;i++)
    {
        endpoint_data[i].endpoint = endpoint_array[i];
        endpoint_data[i].capacity = 0;
        endpoint_data[i].size = 0;
        endpoint_data[i].entries = NULL;
    }
    
    // Add each EH entry to the array, splitting where necessary and
    // removing redundants where necessary.
    for (i=0; i<num_eh; i++)
    {
        unsigned begin, end;
        Cfg_Node *handler;
        unsigned j;
        Class_Handle class_handle;
        unsigned cp_index;
        vm_get_nth_eh_info(caller_eh, i, begin, end, handler, class_handle, cp_index, NULL);
        // Add (class_handle,handler) to every segment in [begin..end-1].
        for (j=0; j<ea_size-1; j++)
        {
            // Check whether segment j is contained in the span of
            // exception handler i.
            if (endpoint_data[j].endpoint >= begin && endpoint_data[j+1].endpoint <= end)
            {
                int k;
                for (k=0; k<endpoint_data[j].size; k++)
                {
                    if (overrides(endpoint_data[j].entries[k].class_handle,class_handle))
                        break;
                }
                if (k < endpoint_data[j].size)
                    continue;
                // It's not redundant, so add it to the end of the
                // array.
                
                RESIZE_ARRAY(struct eh_data, endpoint_data[j].entries,
                    endpoint_data[j].capacity, endpoint_data[j].size,
                    2, mem_manager);
                
                unsigned newidx = endpoint_data[j].size;
                endpoint_data[j].entries[newidx].class_handle = class_handle;
                endpoint_data[j].entries[newidx].cp_index = cp_index;
                endpoint_data[j].entries[newidx].handler = handler;
                endpoint_data[j].size ++;
            }
        }
    }
    
    // For each unique entry in the endpoint_data array, create an
    // exception handler node.
    for (i=0; i<ea_size-1; i++)
    {
        if (endpoint_data[i].size == 0)
        {
            continue;
        }
        // Create new Eh_Node, and add it to the flow graph's master list.
        Eh_Node *ehnode = create_eh_node();
        ehnode->insert_before(&_handlers);
        // Add edges from the basic blocks included in this
        // segment to this EH node.
        unsigned k;
        for (k=endpoint_data[i].endpoint; k<endpoint_data[i+1].endpoint; k++)
        {
            assert(k < _prepass->code_length);
            if (_prepass->bytecode_info[k].is_block_entry)
            {
#if 0
                if (_prepass->bytecode_info[k].fgnode == NULL)
                    _prepass->bytecode_info[k].fgnode = create_flow_graph(k);
                _prepass->bytecode_info[k].fgnode->add_eh_edge(mem_manager, ehnode);
#else
                if (_prepass->bytecode_info[k].fgnode != NULL)
                    _prepass->bytecode_info[k].fgnode->add_eh_edge(mem_manager, ehnode);
#endif // 0
            }
        }
        // Find other segments with an identical set of handlers.
        unsigned l;
        for (l=i+1; l<ea_size-1; l++)
        {
            if (endpoint_data[i].size != endpoint_data[l].size)
                continue;
            int m;
            for (m=0; m<endpoint_data[l].size; m++)
                if (endpoint_data[i].entries[m].class_handle != endpoint_data[l].entries[m].class_handle ||
                    endpoint_data[i].entries[m].handler != endpoint_data[l].entries[m].handler)
                    break;
                if (m < endpoint_data[l].size)
                    continue;
                // A match was found.  Add EH edges from those basic
                // blocks, and clear out this entry.
                unsigned k;
                for (k=endpoint_data[l].endpoint; k<endpoint_data[l+1].endpoint; k++)
                {
                    assert(k < _prepass->code_length);
                    if (_prepass->bytecode_info[k].is_block_entry)
                    {
#if 0
                        if (_prepass->bytecode_info[k].fgnode == NULL)
                            _prepass->bytecode_info[k].fgnode = create_flow_graph(k);
                        _prepass->bytecode_info[k].fgnode->add_eh_edge(mem_manager, ehnode);
#else
                        if (_prepass->bytecode_info[k].fgnode != NULL)
                            _prepass->bytecode_info[k].fgnode->add_eh_edge(mem_manager, ehnode);
#endif // 0
                    }
                }
                endpoint_data[l].size = 0;
        }
        int j;
        for (j=0; j<endpoint_data[i].size; j++)
        {
            // Create the flow graph for the handler.
            Cfg_Node *handler = endpoint_data[i].entries[j].handler;
            // Add a new edge from this EH node to the handler.
            ehnode->add_edge(mem_manager,handler,
                endpoint_data[i].entries[j].class_handle,
                endpoint_data[i].entries[j].cp_index);
        }
    }
}

// Move the Eh_Nodes from "merged_fg" into "this".  If that
// would cause a duplicate element, merge them into one.
void Flow_Graph::merge_eh_info(Flow_Graph *merged_fg)
{
    if (merged_fg->need_linearization())
        set_need_linearization();

    Eh_Node *first_inner = merged_fg->_handlers.next();
    Eh_Node *last_inner = &merged_fg->_handlers;
    Mem_Manager mm(1000);
    // Loop through the callee's Eh_Nodes.
    while (first_inner != last_inner)
    {
        Eh_Node *current_inner = first_inner;
        Eh_Node *current_outer;
        first_inner = first_inner->next();
        
        // Loop through the caller's Eh_Nodes, looking for a match.
        Eh_Node *first_outer = _handlers.next();
        Eh_Node *last_outer = &_handlers;
        int found_match = 0;
        while (!found_match && first_outer != last_outer)
        {
            current_outer = first_outer;
            first_outer = first_outer->next();
            if (current_inner->is_identical(current_outer))
            {
                found_match = 1;
                break;
            }
        }
        if (found_match)
        {
            // Remove edges into callee, and point them to the caller instead.
            // This is tricky, because deleting an edge can cause the
            // looping to get messed up.  Solution: copy the edges into a
            // temporary space first.
            Cfg_Int i;
            Cfg_Int size = current_inner->in_edge_size();
            Cfg_Node **node_array =
                (Cfg_Node **)mm.alloc(size * sizeof(Cfg_Node *));
            for (i=0; i<size; i++)
                node_array[i] = current_inner->in_edges(i);
            for (i=0; i<size; i++)
            {
                Cfg_Node *node = node_array[i];
                node->delete_eh_edge(current_inner);
                node->add_eh_edge(mem_manager, current_outer);
            }
        }
        else
        {
            // Move the callee's Eh_Node to the caller.
            current_inner->unlink();
            current_inner->insert_before(&_handlers);
        }
        
    }
}
