// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/card_table.cpp,v 1.3 2001/08/03 03:13:42 xli18 Exp $
//


#include "platform.h"
 
#include "gc_for_orp.h"
 
#include "card_table.h"
#include "gc_header.h"
#include "remembered_set.h"
#include "descendents.h"
#include "generation.h"
#include "Block_Store.h"
#include "gc_globals.h"
#include "gc_hooks.h"
#include "gc_plan.h"
#include "gc_consts.h"
#include "car.h"
#include "gc_debug.h"
#include "orp_types.h"
#ifdef GC_PT_WB
#include <windows.h>
#endif
#include <string.h>

//
// Needs to be global so that the JITs can get to it:
// This is the array of bytes that the JITs mark.
//
extern byte *p_global_card_table;
extern byte *p_virtual_global_card_table;

// Create a Card Manager for a specified heap and card size.
// This is done during GC initialization. The card and heap
// sizes can be determined at ORP startup time.
Card_Table::Card_Table(Gc_Fast_Hooks  *p_gc_hooks,
                       Gc_Plan        *p_gc_plan,
                       unsigned long  final_heap_size,
                       unsigned long  initial_heap_size,
                       unsigned long  card_size_bytes,
                       Block_Store    *p_block_store,
					   POINTER_SIZE_INT  heap_base)
{
#ifdef GC_COPY_V2
//    assert (0) ; // WHY AM I HERE WHEN WE DON"T NEED A WB.
#endif

    assert(card_size_bytes <= 8192);  // We do 4096 for IA32 and 8192 for IA64
    assert(card_size_bytes > 0);
   
	_heap_base = heap_base;
    _max_number_of_cards = final_heap_size / card_size_bytes;
    _number_of_cards = initial_heap_size / card_size_bytes;
    _card_size_bytes = card_size_bytes;
    //
    // Give the write barrier code access to this value:
    //
    _p_block_store = p_block_store;

    p_global_card_table = (byte *)malloc(_max_number_of_cards);

    //
    // Fold the card table base into the heap_base simplify the
    // marking of the card.
    //

    p_virtual_global_card_table = 
        (Byte *)((Byte *)p_global_card_table - ((POINTER_SIZE_INT)heap_base >> GC_CARD_SHIFT_COUNT));

    // <ake sure heap falls on card boundary.

    assert (( ((POINTER_SIZE_INT) heap_base >> GC_CARD_SHIFT_COUNT) << GC_CARD_SHIFT_COUNT) 
            == ((POINTER_SIZE_INT) heap_base));

    _p_card_last_object_table = 
        (Object_Gc_Header **)malloc(_max_number_of_cards * sizeof(void *));

    if ((p_global_card_table == NULL) || 
        (_p_card_last_object_table == NULL)) {
        cerr << "Malloc failed in Card_Table creation" << endl;
        orp_exit(1);
    }

    // Clear the card and last object tables. Only clear what is initially used.
    // extend heap must clear any additional cards.
    // **** If the OS ensures that malloced code is zero this can be removed.
    // NT seems to place cd in malloced code instead of 0.

    memset(p_global_card_table, 0, _max_number_of_cards);
    memset(_p_card_last_object_table, 0, _max_number_of_cards * sizeof(void *));

#ifdef GC_PT_WB    
    _p_page_base_buffer = (PVOID *) malloc (_max_number_of_cards * sizeof(void *)); // Alloc reuse.
    assert (_p_page_base_buffer);
    memset(_p_page_base_buffer, 0, _max_number_of_cards * sizeof(void *));
#endif // GC_PT_WB
}

void 
Card_Table::extend_card_table (unsigned long heap_extension_size_bytes)
{
#if (GC_DEBUG>1)
    orp_cout << "_card_size_bytes is " << _card_size_bytes << endl;
    orp_cout << "_number_of_cards is " << _number_of_cards << endl;
    orp_cout << "Remove the memset here." << endl;
#endif
    POINTER_SIZE_INT number_of_new_cards = heap_extension_size_bytes / _card_size_bytes;

#ifdef _DEBUG
    // This should have beed done when the tables were allocated.
    for (unsigned int i=_number_of_cards; i<(number_of_new_cards+_number_of_cards); i++) {
        assert(p_global_card_table[i] == 0); // if not OS didn't clear or we have random write.
        assert(_p_card_last_object_table [i] == 0);
    }
#endif

//    memset (&p_global_card_table[_number_of_cards], 0, number_of_new_cards);
//    memset (&_p_card_last_object_table [_number_of_cards],
//            0,
//            (number_of_new_cards * sizeof (void *)));
    _number_of_cards += number_of_new_cards;
}

inline POINTER_SIZE_INT
Card_Table::_address_to_card(void *address)
{ 
    assert ((Byte *)address >= (Byte *)_heap_base); 
    
#ifdef GC_WB_FOR_WILLIE

#include "gc_wb_for_willie.h"

#endif // GC_WB_FOR_WILLIE -- Check that ecx is not used.

    return (POINTER_SIZE_INT)(((POINTER_SIZE_INT) address - (POINTER_SIZE_INT)_heap_base) >> GC_CARD_SHIFT_COUNT);
}

inline void *
Card_Table::_p_card_to_start_address(POINTER_SIZE_INT card)
{
    return (void *)((card << GC_CARD_SHIFT_COUNT) + _heap_base);
}

inline void 
Card_Table::_clear_card(POINTER_SIZE_INT card)
{
    p_global_card_table[card] = 0;
}

inline void 
Card_Table::_clear_card(void *address)
{
    POINTER_SIZE_INT card = _address_to_card(address);
    _clear_card(card);
}

inline void 
Card_Table::mark_card(POINTER_SIZE_INT card)
{
    p_global_card_table[card] = 0xFF;
}

//inline 
void 
Card_Table::mark_card(void *address)
{
    POINTER_SIZE_INT card = _address_to_card(address);

    mark_card(card);
}

inline bool 
Card_Table::_is_card_marked(POINTER_SIZE_INT card)
{
    if (p_global_card_table[card] == 0xFF)
        return TRUE;

    return FALSE;
}

inline bool 
Card_Table::_is_card_marked(void *address)
{
    POINTER_SIZE_INT card = _address_to_card(address);

    return _is_card_marked(card);
}

#if (GC_DEBUG>0)    

//
// This is inlined in card_table.h when GC_DEBUG=0
//
// This should be inlined since card scanning looks at it a lot.
//
bool
Card_Table::_is_mature_card(POINTER_SIZE_INT card_index)
{
    bool result;

    void *p_card_start_address = _p_card_to_start_address(card_index);

    int the_gen_number = p_global_bs->get_address_generation (p_card_start_address);

    if ((the_gen_number == YOS_GEN_NUMBER) || (the_gen_number == POINTER_SIZE_MINUS_ONE)) {
        result = false;
//        return false;
    } else {
        if (p_global_bs->p_get_address_container(p_card_start_address)->is_in_allocated_area(p_card_start_address)) {
            result = true;
        } else {
            // We are between the end of the legal objects in MOS but still in the blocks. This
            // means that this card is not in mature object space and should not be scanned.
            result = false;
        }
//        return true;
    }
    // Double check using a different much slower path.

    Generation *p_gen = 
        p_global_bs->p_get_address_generation(p_card_start_address);
 
	assert(p_gen); 

    if (p_gen->is_train_generation()) {
        return result;
    }

    assert (result == false);
    return result;

}

#endif // (GC_DEBUG>0) ??????????????

//
// This is inlined in card_table.h when GC_DEBUG=0
//
// This should be inlined since card scanning looks at it a lot.
//
bool
Card_Table::_is_focus_car_card(POINTER_SIZE_INT card_index)
{
//	if (card_index == 0x40) {
//		orp_cout << "Card Index is 40" << endl;
//	}
    bool result;

    void *p_card_start_address = _p_card_to_start_address(card_index);

    int the_gen_number = p_global_bs->get_address_generation (p_card_start_address);

    if (the_gen_number == YOS_GEN_NUMBER) {
        result = false;
//        return false;
    } else {
    	Car *p_focus_car = ((Mrl_Gc_V1 *)p_gc)->p_get_focus_car();
        if (p_focus_car->is_address_in_this_car(p_card_start_address)) {
            result = false;
        } else {
            result = true;
        }
//        return true;
    }
    return result;
}

//
//  Clears all the cards in a block that is about to allocated.
//
void
Card_Table::clear_cards_in_super_block(void *p_block_start,
								       unsigned long super_block_size_bytes)
{
    POINTER_SIZE_INT start_card = _address_to_card(p_block_start);
	POINTER_SIZE_INT cards_in_super_block = 
		super_block_size_bytes / _card_size_bytes;

    // memset this for efficiency...*******
    for (POINTER_SIZE_INT card_idx = 0; 
         card_idx < cards_in_super_block; 
         card_idx++) {

        _clear_card(start_card + card_idx);

    }
}

void 
Card_Table::set_last_object_address(Object_Gc_Header *address)
{
    POINTER_SIZE_INT card= _address_to_card(address);
#if (GC_DEBUG>1)
    assert(p_global_bs->is_address_in_heap((void *)address));
    assert (p_global_bs->is_address_in_mature_generation((void *)address));
#endif
    _p_card_last_object_table[card] = address;

    return;
}

//
// A block is just being added to a block-list (either a car
// or a step) and the Card Table Manager is being alerted, so
// it can clear out the corresponding "last object" entries.
// USE A MEMSET HERE
void 
Card_Table::clear_last_object_addresses(void *p_block_start,
										int super_block_size_bytes)
{
    POINTER_SIZE_INT start_card = _address_to_card(p_block_start);
	POINTER_SIZE_INT cards_in_super_block = 
		super_block_size_bytes / _card_size_bytes;

    for (POINTER_SIZE_INT card_idx = 0; 
         card_idx < cards_in_super_block; 
         card_idx++) {
//             if ((card_idx + start_card) == 72786) {
//                 orp_cout << "Clearing _p_card_last_object_table 72786" << endl;
//                 start_card = start_card;
//             }
        _p_card_last_object_table[start_card + card_idx] = 
            NULL;
    }
}

bool
Card_Table::_is_card_in_use (POINTER_SIZE_INT card_index)
{
    void *p_card_start_address = _p_card_to_start_address(card_index);

    Gc_Space *p_container = p_global_bs->p_get_address_container(p_card_start_address);

    if (p_container) {
        Generation *p_gen = p_container->p_container;
        if (p_gen) {
            return true;
        } else {
            return false;
        }
    } else {
        return false;
    }
}
// Scan all cards in MOS and put all references into YOS that you find into
// the remembered set.

// I we end up looking at this routine because it is a hot spot we can do the 
// following:
//
// o.   Only look at cards in MOS, this means that instead of scanning
//      all the cards we only scan the MOS cards.
// o.   We can scan 4 cards with a single cmp using a word instead
//      of doing byte compares. This is a win if 1 set of 4 cards out of 4 
//      sets of 4 cards (16 cards) is not marked.
//
// This activity may either be done synchronously, just prior to a barrier GC.
// Or it may be done concurrently with mutator activity by a background GC thread.
// Except, of course, scan_all_cards, which needs to be done just prior to the
// barrier collection.
//
void
Card_Table::scan_cards(Generation *p_mature_gen,
                       Generation *p_young_gen)
{
#if (defined(GC_FIXED_V1)||defined(GC_COPY_V2))
    // There are no cards to scan in the simple fixed collector.
    return;
#endif

#ifdef GC_PT_WB
#define PAGE_SIZE_BYTES (1<<GC_CARD_SHIFT_COUNT)
#define WRITE_WATCH_FLAG_RESET 0x01
    PVOID heap_base = p_global_bs->get_heap_base();
    SIZE_T region_size = p_global_bs->get_heap_size_bytes(); // (DWORD)used_heap_limit - (DWORD)heap_base;
    ULONG total_pages = region_size / PAGE_SIZE_BYTES;
    ULONG buff_size = _number_of_cards; //  total_pages;
    ULONG page_size_temp = PAGE_SIZE_BYTES;
    // Heap base can change, just so it plus size is less than the MEM_COMMIT part of memory.

    if (dll_GetWriteWatch(WRITE_WATCH_FLAG_RESET, heap_base, region_size, _p_page_base_buffer, &buff_size, &page_size_temp) != 0) {
        orp_cout << " GetWriteWatch failure bug card table 392. Perhaps it isn't supported on this platform." << endl;
    }
    //    orp_cout  << endl;
    for (ULONG index = 0; index < buff_size; index++) {
        if ( p_global_bs->is_address_in_mature_generation ((void *)_p_page_base_buffer[index]) ) {
            _scan_page (_p_page_base_buffer[index], p_mature_gen, p_young_gen);
        }
    }
    return;
    // For gathering statistics.
    ULONG number_of_marked_cards = 0;
    for (POINTER_SIZE_INT the_card_index = 0; the_card_index < _number_of_cards; the_card_index++) {
        if (_is_card_marked(the_card_index)) {
            number_of_marked_cards++;
        }
    }

    assert (_number_of_cards >= total_pages); // The card table includes overheaad data that is never marked.
    // assert (number_of_marked_cards <= buff_size);
    // Calculate the total number of cards scanned and the number of dirty pages and compare them.
#endif // GC_PT_WB


    for (POINTER_SIZE_INT card_index = 0; 
                      card_index < _number_of_cards; 
                      card_index++) {
        //
        // Iterate through all the cards.
        // Can't we just iterate through the MOS cards, it will be faster.
        // Why can't I scan these things 4 or 8 bytes at a time it might be faster.....
                          
        if ((_is_card_marked(card_index)) &&
            (_is_mature_card(card_index))) {
            // Since cars are small we have seperated collection into two pieces.
            // One does the YOS areas and this needs to know about all pointers from
            // MOS to YOS, including the pointer in the focus car. Once the YOS is
            // collected we will have all the pointers into the focus car, including 
            // pointers from YOS as well as younger cars in the focus train and the
            // other trains.
// #ifdef GC_SAPPHIRE
            
#if (GC_DEBUG>0)
            // Check to see if we have an unmarked card if the _p_card_last_object_table
            // holds the header to an object whose object starts at the start of this card.
            // Since the _p_card_last_object_table is based on object header and the
            // cards are based on object pointers we can have a marked card without
            // a _p_card_last_object_table. If this is the case then the previous
            // _p_card_last_object__table should hold such an object.
            if (_p_card_last_object_table[card_index] == 0) {
                Object_Gc_Header *p_object_header = _p_card_last_object_table[card_index-1];
                Java_java_lang_Object *p_object = get_object_from_gc_header(p_object_header);
                POINTER_SIZE_INT card_to_check = _address_to_card((void *)p_object);
                assert (card_to_check == card_index);
                assert(is_object_gc_header(p_object_header));
            } else {
                // make sure we have a header in _p_card_last_object_table
                assert(is_object_gc_header((Object_Gc_Header *)_p_card_last_object_table[card_index]));
            }
            // NOTE NOTE NOTE
            // Someday when we only mark object headers these assert will be true.
            // Until then they could well be false.............
            // Note that if mark array locations and the last_object_table is zero then
            // we need to scan the entire card but we know that it has to be all 
            // legal addresses except for possible that the dope vector for the
            // array as well as the vtable might be in the first couple of slots.
            // In addition the array could end before the end of the block.
#endif // (GC_DEBUG>0)
            // We need to clear the card before we scan it, see sapphire code for details.
            _clear_card(card_index);

            _scan_card(card_index,
                       p_mature_gen,
                       p_young_gen);

        } else {
            //
            // If it isn't in mature space, clear it anyway.
            //
            _clear_card(card_index);
        }
    }
    return;
}

// If this is the first card in a block return true.
bool 
Card_Table::_is_first_card_in_super_block (POINTER_SIZE_INT card)
{
#if (GC_DEBUG>2)
    assert (_is_mature_card(card));
#endif // (GC_DEBUG>2)

	void *p_card  = _p_card_to_start_address(card);
	void *p_sb    = 
		p_global_bs->p_get_address_super_block_start_address(p_card);

	if (p_card == p_sb) {
		return true;
	}

	return false;
}


#ifdef GC_PT_WB
//
// Scan all the objects in a single card. Since the first object
// that occupies this card may not be aligned with the start of 
// the card, we might start scanning from the last object in the
// previous card.
//
void
Card_Table::_scan_page(PVOID page_base,
                       Generation *p_mature_gen,
                       Generation *p_young_gen)
{
    
    Object_Gc_Header *p_object_gc_header;

    POINTER_SIZE_INT card = _address_to_card(page_base);
    void *p_card_start_address = _p_card_to_start_address(card);
    assert (page_base == p_card_start_address);

    assert(p_global_bs->is_address_in_heap((void *)p_card_start_address));

    //
    // Start scanning from the last object in *some* previous card.
    //
    // In MOS Objects can span several cards in fact up to the number of cards
    // before you start moving the objects into large object space.
    // This means that you must loop backward through the cards until you
    // find a _p_card_last_object_table[card] that is not null.
    assert ((_p_card_last_object_table[card] == NULL) || 
            (p_card_start_address <= _p_card_last_object_table[card]));

#if 1 // GC_BLOCK_IS_CARD
    if (true) {
        bool result = _is_first_card_in_super_block(card);
        if (!result) {
            assert (0);
            result = true;
        }
#else
    if ((p_card_start_address == _p_card_last_object_table[card]) ||
        (_is_first_card_in_super_block(card) )) {
#endif
        // This card starts with an object so we don't need to scan
        // the previous card. This check also picks up the first card in a block
        // problem of not having a previous card. Since an object will
        // always start at the start of a block.
        // If a card falls on an block boundary then we start at the
        // beginning of the card.
        //
        // How this works with LOS I don't know yet, currently we always collect LOS.....
        //

        assert (_p_card_last_object_table[card]);

        p_object_gc_header = (Object_Gc_Header *)p_card_start_address;
        assert (p_object_gc_header == page_base);
        assert (p_global_bs->is_object_in_heap (get_object_from_gc_header (p_object_gc_header)));
        assert (is_valid_java_object (get_object_from_gc_header (p_object_gc_header)));
    }
    else {
        //
        // The _p_card_last_object_table holds object headers.
        // Cards are marked based on the object which is a word beyond
        // the header. This can lead to an object header lying in the 
        // previous card marked with the object.
        //
        // We need to loop through previous cards until we find one with 
        // and object in the _p_card_last_object_table[card] field.
        //
        POINTER_SIZE_INT start_card = card - 1;
        while (_p_card_last_object_table[start_card] == NULL)
        {
            assert (start_card >= 1); // 0 should have been caught above.
            // At least the first card should have an object associated with it.
            assert (!(_is_first_card_in_super_block(start_card))); 
            start_card--;
        }

        p_object_gc_header = (Object_Gc_Header *)_p_card_last_object_table[start_card];

        assert (p_global_bs->is_object_in_heap (get_object_from_gc_header (p_object_gc_header)));
        assert (is_valid_java_object (get_object_from_gc_header (p_object_gc_header)));
    }

    assert(p_object_gc_header);
    assert(is_object_gc_header(p_object_gc_header));

    void *p_last_address_to_scan = _p_card_last_object_table[card];
    if (p_last_address_to_scan == NULL) { 
        // Case 1. 
        // the object in this card spans the entire card possible into 
        //  the next card(s)
        //
        // The object ends in this card. Since p_last_object_table[card] is NULL
        //  we know that no additional objects have been allocated in this card.
        //
        // In either case
        //
        // -- Set the last address to scan to the start of the card 
        //    so that you only scan the object that overlaps into this card, the
        //    only object in this card.

        p_last_address_to_scan = (void *)((Byte *)p_card_start_address);
    }

    //
    // Scan all the objects that are in this card:
    // This will include scanning the object that could
    // overlap onto the next card.
    //

    while (p_object_gc_header <= p_last_address_to_scan) {

#if (GC_DEBUG>0)
        // We have a header and we want an object to do the asserts with.

        Java_java_lang_Object *p_the_object_for_asserts = 
            get_object_from_gc_header (p_object_gc_header); 
        assert (p_global_bs->is_object_in_heap (p_the_object_for_asserts));
        assert (is_valid_java_object (p_the_object_for_asserts));
        gc_trace (p_the_object_for_asserts, "Scanning a card holding this object."); 
#endif

        //
        // Scan the current object.
        //
        _scan_object(p_object_gc_header,
                     p_mature_gen,
                     p_young_gen);
        //
        // Done with that object - go to the next one:
        //
        p_object_gc_header = 
            p_scan_forward_over_object(p_object_gc_header);
        // We could have moved into uninitialized territory here at the end of a card
        // but if that is the case then it will be > p_last_address_to_scan and
        // all we be OK.
    }

    return;
} // _scan_page
#endif // GC_PT_WB

//
// Scan all the objects in a single card. The first object
// that occupies this card will be aligned with the start of 
// the card block and page size are all the same.
//
void
Card_Table::_scan_card(POINTER_SIZE_INT card,
                       Generation *p_mature_gen,
                       Generation *p_young_gen)
{
    Object_Gc_Header *p_object_gc_header;
 
    assert (card < _number_of_cards); 

    void *p_card_start_address = _p_card_to_start_address(card);

    assert(p_global_bs->is_address_in_heap((void *)p_card_start_address));

    //
    // Start scanning from the last object in *some* previous card.
    //
    // In MOS Objects can span several cards in fact up to the number of cards
    // before you start moving the objects into large object space.
    // This means that you must loop backward through the cards until you
    // find a _p_card_last_object_table[card] that is not null.
    assert ((_p_card_last_object_table[card] == NULL) || 
            (p_card_start_address <= _p_card_last_object_table[card]));


        
    // All cards start with an object since cards are page sized so we don't need to scan
    // the previous card.


    p_object_gc_header = (Object_Gc_Header *)p_card_start_address;


#if (GC_DEBUG>0)
        Java_java_lang_Object *p_the_object_for_asserts = 
                get_object_from_gc_header (p_object_gc_header); 

        assert (p_global_bs->is_object_in_heap (p_the_object_for_asserts));
        assert (is_valid_java_object (p_the_object_for_asserts));
#endif //(GC_DEBUG>2)

#if (GC_DEBUG>2)
    assert(p_object_gc_header);
    assert(is_object_gc_header(p_object_gc_header));
#endif // _DEBUG

    void *p_last_address_to_scan = _p_card_last_object_table[card];
    assert (p_last_address_to_scan != NULL); 
    //
    // Scan all the objects that are in this card:
    // This will include scanning the object that could
    // overlap onto the next card.
    //
    while (p_object_gc_header <= p_last_address_to_scan) {

#if (GC_DEBUG>0)
        // We have a header and we want an object to do the asserts with.

        Java_java_lang_Object *p_the_object_for_asserts = 
            get_object_from_gc_header (p_object_gc_header); 
        assert (p_global_bs->is_object_in_heap (p_the_object_for_asserts));
        assert (is_valid_java_object (p_the_object_for_asserts));
        gc_trace (p_the_object_for_asserts, "Scanning a card holding this object."); 
#endif

        //
        // Scan the current object.
        //
        _scan_object(p_object_gc_header,
                     p_mature_gen,
                     p_young_gen);
        //
        // Done with that object - go to the next one:
        //
        p_object_gc_header = 
            p_scan_forward_over_object(p_object_gc_header);
        // We could have moved into uninitialized territory here at the end of a card
        // but if that is the case then it will be > p_last_address_to_scan and
        // all we be OK.
    }

    return;
}

//#ifdef GC_OFFSET_LOGIC

void
Card_Table::_scan_object(Object_Gc_Header *p_gc_hdr,
                         Generation *p_older_gen,
                         Generation *p_younger_gen)
{   

    assert(p_global_bs->is_address_in_heap((void *)p_gc_hdr));
    assert(is_object_gc_header(p_gc_hdr));

    Java_java_lang_Object *p_object =
        get_object_from_gc_header(p_gc_hdr);
 
    Partial_Reveal_Class *p_class = (*((Partial_Reveal_VTable **)p_object))->clss;  // get_object_class(p_object);

    bool array_p = is_array(p_class);
    
    if (array_p) {
        _scan_array_object (p_gc_hdr, p_older_gen, p_younger_gen);
        return;
    }

    unsigned int *offset_scanner = init_object_scanner (p_object);

    Java_java_lang_Object **pp_target_object;

	while ((pp_target_object = p_get_ref(offset_scanner, p_object)) != NULL) {

        // Move the scanner to the next reference.
        offset_scanner = p_next_ref (offset_scanner);

        // ignore NULL
        if (*pp_target_object==NULL) {
            continue;
        }

#if (GC_DEBUG>0)
        assert(p_global_bs->is_address_in_heap((void *)*pp_target_object));

        Generation *p_source_gen = 
            p_global_bs->p_get_object_generation(p_object);

        if (p_source_gen->is_step_generation()) {
            assert (0); // Why am I here?
            //
            // References which originate in young space are uninteresting.
            //
            continue;
        }
#endif
        //
        // At this point we know that the source is mature space.
        // If it wasn't we would not be scanning this card.
        //
        Generation *p_target_gen =
            p_global_bs->p_get_object_generation(*pp_target_object);

        // Slot does not hold NULL. Slot holds pointer to target_gen. 
        // Add slot to appropriate remset if any.
        p_target_gen->add_entry_to_generation_write_barriers(pp_target_object,
                                                             *pp_target_object);
    }

}

void
Card_Table::_scan_array_object(Object_Gc_Header *p_gc_hdr,
                               Generation *p_older_gen,
                               Generation *p_younger_gen)
{
    assert(p_global_bs->is_address_in_heap((void *)p_gc_hdr));
    assert(is_object_gc_header(p_gc_hdr));

    Java_java_lang_Object *p_object =
        get_object_from_gc_header(p_gc_hdr);

    Partial_Reveal_Class *p_class = (*((Partial_Reveal_VTable **)p_object))->clss;
    // delete get_object_class(get_object_from_gc_header(p_gc_hdr));
    // If array is an array of primitives, then there are no references, so return.
    if (is_array_of_primitives(p_class))
        return;

    // Initialize the array scanner which will scan the array from the
    // top to the bottom. IE from the last element to the first element.

    unsigned int offset = init_array_scanner (p_object);
    
	//
	// Cycle through all the descendents.
	//
	
    Java_java_lang_Object **pp_target_object;
	while ((pp_target_object = p_get_array_ref(p_object, offset)) != NULL) {

        offset = next_array_ref (offset);
        
        // Skip null entries.
        if (*pp_target_object == NULL) {
            continue;
        }


#if (GC_DEBUG>0)
        assert(p_global_bs->is_address_in_heap((void *)*pp_target_object));

        Generation *p_source_gen = 
            p_global_bs->p_get_object_generation(p_object);

        if (p_source_gen->is_step_generation()) {
            assert (0); // Why am I here?
            //
            // References which originate in young space are uninteresting.
            //
            continue;
        }
#endif
        //
        // At this point we know that the source is mature space.
        //
        Generation *p_target_gen =
            p_global_bs->p_get_object_generation(*pp_target_object);
        //
        // The right thing will be done, for young or mature space:
        //

        p_target_gen->
            add_entry_to_generation_write_barriers(pp_target_object,
                                                   *pp_target_object);
    }
}

#if (GC_DEBUG>2)
//
// Verify that this card has a valid last_object_table entry associated
// with it.

void Card_Table::verify_last_object_table (POINTER_SIZE_INT card) 
{
    void *p_card_start_address = _p_card_to_start_address (card);
    if ((p_card_start_address == _p_card_last_object_table[card]) ||
        (_is_first_card_in_super_block(card) )) {
        // This card starts with an object so we don't need to scan
        // the previous card. This check also picks up the first card in a block
        // problem of not having a previous card. Since an object will
        // always start at the start of a block.
        // If a card falls on an block boundary then we start at the
        // beginning of the card.
        //
        // How this works with LOS I don't know yet.....
        //
        
        assert (_p_card_last_object_table[card]);
        
        Object_Gc_Header *p_object_gc_header = (Object_Gc_Header *)p_card_start_address;

        Java_java_lang_Object *p_the_object_for_asserts = 
            get_object_from_gc_header (p_object_gc_header); 
        assert (p_global_bs->is_object_in_heap (p_the_object_for_asserts));
        assert (is_valid_java_object (p_the_object_for_asserts));

    } else {
        // We need to loop through previous cards until we find one with 
        // and object in the _p_card_last_object_table[card] field.
        POINTER_SIZE_INT start_card = card - 1;
        while (_p_card_last_object_table[start_card] == NULL) {
            assert (start_card >= 1); // 0 should have been caught above.
            // At least the first card should have an object associated with it.
            assert (!(_is_first_card_in_super_block(start_card)));                    
            start_card--;
        }
        
        Object_Gc_Header *p_object_gc_header = 
            (Object_Gc_Header *)_p_card_last_object_table[start_card];

        Java_java_lang_Object *p_the_object_for_asserts = 
            get_object_from_gc_header (p_object_gc_header); 
        assert (p_global_bs->is_object_in_heap (p_the_object_for_asserts));
        assert (is_valid_java_object (p_the_object_for_asserts));

    }
}

//
// Verify that the card index in question is that of a
// card that has some objects resident. (i.e. it is not
// a card representing unallocated/unused memory).
//
// Since the card last object table is only valid for cards in
// mature space OK all cards not in mature space.
// 
//

void Card_Table::verify_is_populated_card(POINTER_SIZE_INT card) 
{
    //
    if (_is_mature_card (card)) {
        // We have a mature card, make sure we have a
        // valid last_object_table entry to go along with it.
        verify_last_object_table (card);
    }
    // otherwise we should never be scanning this card anyway so the
    // status is not important.
}

#endif // GC_DEBUG>2

#ifdef GC_PT_WB

#pragma optimize( "", off )

typedef UINT (CALLBACK* FN1)(DWORD, PVOID, SIZE_T, PVOID *, PULONG, PULONG);
typedef UINT (CALLBACK* FN2)(PVOID, SIZE_T);
static FN1 func_GetWriteWatch=NULL;
static FN2 func_ResetWriteWatch=NULL;

#ifdef ORP_POSIX
#define DLL_HANDLE void *
#else
#define DLL_HANDLE HINSTANCE
#endif

static DLL_HANDLE kernel32_handle;

void load_WriteWatch_funcs()
{
            orp_cout << "finding GetWriteWatch and ResetWriteWatch" << endl;
			const char *kernel32_lib_name = "kernel32";
			const char *GetWriteWatchName = "GetWriteWatch";
			const char *ResetWriteWatchName = "ResetWriteWatch";
			kernel32_handle = LoadLibrary(kernel32_lib_name);
            if (kernel32_handle == 0) {
                orp_cout << "Can not find kernel32.dll" << endl;
                exit(99);
            }
			FN1 fn1;
			FN2 fn2;
			fn1 = (FN1)GetProcAddress(kernel32_handle, GetWriteWatchName);
			func_GetWriteWatch = (FN1) fn1;
			orp_cout << "loaded func_GETWriteWatch is " << func_GetWriteWatch << endl;
            fn2 = (FN2)GetProcAddress(kernel32_handle, ResetWriteWatchName);
			func_ResetWriteWatch = (FN2) fn2;
            orp_cout << "loaded func_ResetWriteWatch is " << func_ResetWriteWatch << endl;
}

UINT WINAPI dll_GetWriteWatch (IN DWORD dwFlags,          // write-tracking state
							  IN PVOID lpBaseAddress,    // base address of region
							  IN SIZE_T dwRegionSize,    // size of region
							  IN OUT PVOID  *lpAddresses,    // array of page addresses
							  IN OUT PULONG lpdwCount,   //  IN OUT PULONG_PTR lpdwCount,   // number of addresses returned
							  OUT PULONG lpdwGranularity  // page size
							  )
{
    if (!func_GetWriteWatch) {
        load_WriteWatch_funcs();
    }
    UINT result = (UINT) func_GetWriteWatch (dwFlags, lpBaseAddress, dwRegionSize, lpAddresses, lpdwCount, lpdwGranularity);
    
    return result;
}

UINT WINAPI dll_ResetWriteWatch (PVOID lpBaseAddress,
                          SIZE_T dwRegionSize
                          )
{
    
    if (!func_ResetWriteWatch) {
        load_WriteWatch_funcs();
    }
    return (UINT) func_ResetWriteWatch (lpBaseAddress, dwRegionSize);

}
#pragma optimize( "", on )
#endif GC_PT_WB

// end file card_table.cpp
