/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "stars.h"
#include "catalog.h"
#include "iconlist.h"
#include "rgbdraw.h"
#include "gui.h"

struct catalog *nextcatalog(struct catalog *cat, int flags)
{
	GList *lis = 0;

	if(cat)
	{
		lis = g_list_find(catalogs, cat);
		if(!lis) return 0;
	}
	for(;;)
	{
		if(!lis) lis = catalogs;
		else lis = lis->next;
		if(!lis) return 0;
		cat = (struct catalog *)(lis->data);
		if(!cat) return 0;
		if(flags & SEARCHABLE && cat->searchable == 0) continue;
		if(flags & HASDATA && cat->objects == 0) continue;
		if(flags & LOADDATA && cat->objects == 0)
		{
			catalogloaddata(cat);
			if(cat->objects == 0) continue;
		}
		return cat;
	}
}

int catalogloaddata(struct catalog *cat) 
{
	FILE *in;
	gchar *fullfilename; 
	int i;

	if(cat->ignore) return -1;

	cat->objects = g_new(struct object, cat->numobjects);
	if(cat->objects == 0)
	{
		fprintf(stderr, "catalogloaddata: cannot allocate memory\n");
		return -1;
	}
	if(cat->filename[0] == '/') fullfilename = g_strdup(cat->filename);
	else fullfilename = g_strconcat(PREFIXDIR, "/share/stars/data/", cat->filename, 0);
	in = fopen(fullfilename, "r");
	if(in == 0)
	{
		g_free(cat->objects);
		cat->objects = 0;
		cat->ignore = 1;
		fprintf(stderr, "catalogloaddata: cannot open %s\n", 
			fullfilename);
		g_free(fullfilename);
		return -1;
	}
	g_free(fullfilename);
	if(fread(cat->objects, sizeof(struct object), cat->numobjects, in) != 
			cat->numobjects)
	{
		g_free(cat->objects);
		cat->objects = 0;
		cat->ignore = 1;
		fclose(in);
		fprintf(stderr, "catalogloaddata: file %s too short\n", 
			cat->filename);
		return -1;
	}
	if(cat->numextensions > 0)
	{
		cat->extension = g_new(float, cat->numobjects*cat->numextensions);
		if(cat->extension == 0)
		{
			fclose(in);
			g_free(cat->objects);
			cat->objects = 0;
			cat->ignore = 1;
			return -1;
		}
		if(fread(cat->extension, sizeof(float)*cat->numextensions, cat->numobjects, in) != 
				cat->numobjects)
		{
			fprintf(stderr, "loading of extension data failed\n");
			g_free(cat->objects);
			g_free(cat->extension);
			cat->extension = 0;
			cat->objects = 0;
			cat->ignore = 1;
			fclose(in);
			return -1;
		}
	}
	fclose(in);

	if(fabs(cat->epoch-epoch) > 0.0000001)
	{
//		printf("Precessing %s catalog\n", filebase);
		for(i = 0; i < cat->numobjects; i++) 
			precess(cat->objects + i, epoch-cat->epoch);
	}

	applyclasslimitsall(cat);

	return 0;
}

int catalogunloaddata(struct catalog *cat)
{
	g_free(cat->objects);
	g_free(cat->extension);
	cat->objects = 0;
	cat->extension = 0;

	return 0;
}

void addcatalog(struct catalog *cat)
{
	if(cat == 0)
	{
		printf("BOGUS!  trying to add null catalog\n");
		return;
	}

	catalogs = g_list_append(catalogs, cat);
	refillselectorcatinfo(cat);
}

struct catalog *newcatalog()
{
	struct catalog *cat;

	cat = g_new(struct catalog, 1);

	cat->objects = 0;
	cat->extension = 0;
	cat->searchable = 0;
	cat->catmask = 0x0F;
	cat->ch = 0;
	cat->description = 0;
	cat->name = 0;
	cat->numobjects = 0;
	cat->numextensions = 0;
	cat->epoch = 2000.0;
	cat->type[0] = 0;
	cat->filename[0] = 0;
	cat->info = 0;
	cat->lut_name[0] = 0;
	cat->lookupfile = 0;
	cat->catalogindexoffset = 0;
	cat->ignore = 0;

	return cat;
}

/* loadcatalog no longer explicitly loads data.  It is now loaded on demand. 4-14-2000 */

struct catalog *loadcatalog(char *str)
{
	int i, descstart=0;
	struct catalog *cat;
	char filebase[100], t[20];
	int v;

	cat = newcatalog();
	
	for(i = 0; str[i] != 0; i++) if(str[i] == '\"')
	{
		if(descstart == 0) descstart = i+1; 
		str[i] = 0;
	}
	if(sscanf(str, "%s%s%lf%d%d%d", filebase, t, &(cat->epoch), 
		&(cat->numobjects),
		&(cat->numextensions), &v) != 6)
	
	{
		fprintf(stderr, "loadcatalog: file format error: \n");
		fprintf(stderr, "Line was %s\n", str);
		g_free(cat);
		return 0;
	}
	sprintf(cat->filename, "%s.cat", filebase);
	cat->searchable = 1;

	if(v) cat->catmask = 0x0F;	// Set default to visible for all vids
	else cat->catmask = 0x00;
	strcpy(cat->type, t);
	cat->ch = getclassheader(t);
//	if(cat->ch) printf("Catalog has dedicated extension header\n");
	cat->name = g_strdup(filebase);
	if(descstart > 0) cat->description = g_strdup(str+descstart);
	else cat->description = g_strdup(cat->name);
	cat->info = getcatinfo(filebase);
	return cat;
}

int loadallcatalogs(const char *filename)
{
	struct catalog *cat;
	FILE *in;
	char line[1000];
	int i;
	
	in = fopen(filename, "r");
	if(in == 0) 
	{
		fprintf(stderr, "loadallcatalogs : file not found : %s\n",
			filename);
		return 0;
	}

	for(;;)
	{
		fgets(line, 999, in);
		if(feof(in)) break;
		for(i = 0; line[i] != 0; i++)
			if(line[i] < ' ' || line[i] == '#') line[i] = 0;
		if(line[0] == 0) continue;

		cat = loadcatalog(line);
		if(cat == 0) printf("Error loading catalog : %s\n", line);
		else addcatalog(cat);
	}

	fclose(in);

	return 1;
}

void destroycatalog(struct catalog *cat)
{
	if(cat == 0) return;
	if(cat->objects != 0) g_free(cat->objects);
	if(cat->extension != 0) g_free(cat->extension);
	g_free(cat);
}

void destroyallcatalogs(GList *cats)
{
	struct catalog *c;
	while(cats != 0)
	{
		c = (struct catalog *)cats->data;
		cats = g_list_remove(cats, cats->data);
		destroycatalog(c);		
	}
}

void catalog_render_pixmap(struct catalog *cat, struct iconlist *il, struct viewer *v)
{
	struct object *obj;
	unsigned char amask = v->andmask;
	unsigned char smask = v->showmask;
	long i, xr, yr, px, py, pz, N;
	long long xx, xy, xz, yx, yy, yz, zx, zy, zz;
	long long x, y, z;
	int rs;
	guchar *pixels, *base;
	long lastx=0, lasty=0, lastz = -1;

	xx = (long long)v->rot[0][0];
	xy = (long long)v->rot[0][1];
	xz = (long long)v->rot[0][2];
	yx = (long long)v->rot[1][0];
	yy = (long long)v->rot[1][1];
	yz = (long long)v->rot[1][2];
	zx = (long long)v->rot[2][0];
	zy = (long long)v->rot[2][1];
	zz = (long long)v->rot[2][2];
	xr = (long long)gdk_pixbuf_get_width(v->image);
	yr = (long long)gdk_pixbuf_get_height(v->image);
	rs = gdk_pixbuf_get_rowstride(v->image);
	pixels = gdk_pixbuf_get_pixels(v->image);
	N = v->N;

	// Look, its all integer arrithmetic!
	for(i = 0; i < cat->numobjects; i++)
	{
		obj = cat->objects + i;
		if((obj->mask & amask) != smask) continue;
		if(obj->type >= 192 && obj->type < 224)	// Its a line type.  Draw it.
		{
			x = yx*(long long)obj->x +
                            yy*(long long)obj->y + 
                            yz*(long long)obj->z + v->offx;
			px = (long)(x >> N);
			y = v->offy - zx*(long long)obj->x
                                    - zy*(long long)obj->y  
                                    - zz*(long long)obj->z;
			py = (long)(y >> N);
			z = xx*(long long)obj->x +
                   	    xy*(long long)obj->y + 
                   	    xz*(long long)obj->z;
			pz = (long)(z >> N);

			if(obj->icon != 255 && pz >= 0 && lastz >= 0)
			{
				drawline(v->image, px, py, lastx, lasty, 
				    il->line_thick[obj->type-192][obj->icon],
				    il->line_r[obj->type-192][obj->icon],
				    il->line_g[obj->type-192][obj->icon],
				    il->line_b[obj->type-192][obj->icon]);
			}
			lastx = px;
			lasty = py;
			lastz = pz;
			continue;
		}

		/* Compute and check data here! */

		if((x = yx*(long long)obj->x + 
		    	yy*(long long)obj->y + 
		    	yz*(long long)obj->z + v->offx) < 0) continue;
		px = (long)(x >> N);
		if(px >= xr) continue;
		if((y = v->offy - zx*(long long)obj->x  
		    		- zy*(long long)obj->y  
		    		- zz*(long long)obj->z) < 0) continue;
		py = (long)(y >> N);
		if(py >= yr) continue;
		if(xx*(long long)obj->x + 
		   xy*(long long)obj->y + 
		   xz*(long long)obj->z < 0) continue;

		/* drawing goes here */

		if(obj->type < 64)
		{
			base = pixels+3*px+py*rs;
			base[0] = il->rgb_r[obj->type][obj->icon];
			base[1] = il->rgb_g[obj->type][obj->icon];
			base[2] = il->rgb_b[obj->type][obj->icon];
		}
		else if(obj->type < 192)
		{
			if(il->icon[obj->type-64] == 0) continue;
			drawicon(v->image, px, py, 
				&(il->icon[obj->type-64][obj->icon]));
		}
		else if(obj->type < 255)
		{
			if(il->drawfuncs[obj->type-224] == 0)
			{
				printf("Can't execute plugin %d:%d\n", 
					obj->type, obj->icon);
				continue;
			}
			il->drawfuncs[obj->type-224][obj->icon](v, 
				cat->extension+(i*cat->numextensions), px, py, 
				il->func_r[obj->type-224][obj->icon],
				il->func_g[obj->type-224][obj->icon],
				il->func_b[obj->type-224][obj->icon]);
		}
	}
}

void render_pixmap(struct iconlist *il, struct viewer *v)
{
	struct catalog *cat;
	GList *l;

	for(l = catalogs; l != 0; l = l->next)
	{
		cat = (struct catalog *)(l->data);
		if((cat->catmask & v->andmask) != v->showmask) continue;
		
		if(cat->objects == 0) catalogloaddata(cat);
		if(cat->objects == 0) continue;
		catalog_render_pixmap(cat, icons, v);
	}
}

void render_ps(struct catalog *cat, struct viewer *v, FILE *out, double *scale)
{
	struct object *obj;
	unsigned char amask = v->andmask;
	unsigned char smask = v->showmask;
	long i, xr, yr, px, py, pz, N;
	long long xx, xy, xz, yx, yy, yz, zx, zy, zz;
	long long x, y, z;
	int rs;
	long lastz = -1;
	int needstroke = 0, needmove = 0, lasti = -1, lastt = -1;

	double qx, qy, rx, ry, mqx = 0, mqy = 0;
	double sx, sy;
	float pscale;

	if((cat->catmask & amask) != smask) return;

	if(cat->objects == 0) catalogloaddata(cat);

	if(cat->objects == 0)
	{
		printf("Problem loading data, Skipping for now\n");
		return;
	}

	if(*scale == 0)
	{
		sx = 540.0/(double)v->xres;
		sy = 720.0/(double)v->yres;

		if(sx > sy) *scale = sy; else *scale = sx;

		sx = (*scale)*(double)v->xres;
		sy = (*scale)*(double)v->yres;


		fprintf(out, "newpath 36 756 moveto\n");
		fprintf(out, "%6.2f 0 rlineto\n", sx);
		fprintf(out, "0 %6.2f rlineto\n", -sy);
		fprintf(out, "%6.2f 0 rlineto\n", -sx);
		fprintf(out, "closepath clip stroke\n");
	}

	rx = 36.0;
	ry = 756.0;
	pscale = 0.5*540.0*v->view->zoom;

	xx = (long long)v->rot[0][0];
	xy = (long long)v->rot[0][1];
	xz = (long long)v->rot[0][2];
	yx = (long long)v->rot[1][0];
	yy = (long long)v->rot[1][1];
	yz = (long long)v->rot[1][2];
	zx = (long long)v->rot[2][0];
	zy = (long long)v->rot[2][1];
	zz = (long long)v->rot[2][2];
	xr = (long long)gdk_pixbuf_get_width(v->image);
	yr = (long long)gdk_pixbuf_get_height(v->image);
	rs = gdk_pixbuf_get_rowstride(v->image);
	N = v->N;

	for(i = 0; i < cat->numobjects; i++)
	{
		obj = cat->objects + i;
		if((obj->mask & amask) != smask) continue;
		if(obj->type >= 192 && obj->type < 224)	// Its a line type.  Draw it.
		{
			x = yx*(long long)obj->x +
                            yy*(long long)obj->y + 
                            yz*(long long)obj->z + v->offx;
			px = (long)(x >> N);
			y = v->offy - zx*(long long)obj->x
                                    - zy*(long long)obj->y  
                                    - zz*(long long)obj->z;
			py = (long)(y >> N);
			z = xx*(long long)obj->x +
                   	    xy*(long long)obj->y + 
                   	    xz*(long long)obj->z;
			pz = (long)(z >> N);

			qx = rx + (*scale) * (double)px;
			qy = ry - (*scale) * (double)py;

			if(pz >= 0)
			{

				if(obj->icon == 255 || lastz < 0)
				{
					if(needstroke)
					{	
						needstroke = 0;
						fprintf(out, "l%di%d\n",
							lastt,
							lasti);
						fprintf(out, "stroke\n");
					}
					mqx = qx;
					mqy = qy;
					needmove = 1;
				}
				else
				{
					if(needmove)
					{
					  fprintf(out, "%6.2f %6.2f moveto\n",
						mqx, mqy);
					  needmove = 0;
					}
					needstroke = 1;
					fprintf(out, "%6.2f %6.2f lineto\n",
						qx, qy);
					lastt = obj->type;
					lasti = obj->icon;
				}
			}
			else
			{
				if(needstroke)
				{
					needstroke = 0;
					fprintf(out, "l%di%d\n", lastt, lasti);
					fprintf(out, "stroke\n");
				}
			}
			lastz = pz;
			continue;
		}

		/* Compute and check data here! */

		if((x = yx*(long long)obj->x + 
		    	yy*(long long)obj->y + 
		    	yz*(long long)obj->z + v->offx) < 0) continue;
		px = (long)(x >> N);
		if(px >= xr) continue;
		if((y = v->offy - zx*(long long)obj->x  
		    		- zy*(long long)obj->y  
		    		- zz*(long long)obj->z) < 0) continue;
		py = (long)(y >> N);
		if(py >= yr) continue;
		if(xx*(long long)obj->x + 
		   xy*(long long)obj->y + 
		   xz*(long long)obj->z < 0) continue;

		/* drawing goes here */

		qx = rx + (*scale) * (double)px;
		qy = ry - (*scale) * (double)py;

// Point type and icontype
		if(obj->type < 192) fprintf(out, "%6.2f %6.2f t%di%d\n", 
			qx, qy, obj->type, obj->icon);

		else if(obj->type < 255)
		{
			if(icons->printfuncs[obj->type-224] != 0) 
			{
				fprintf(out, "%6.2f %6.2f ", qx, qy);
				icons->printfuncs[obj->type-224][obj->icon](v, 
					cat->extension+(i*cat->numextensions), 
					pscale, out);
			}
		}
	}
	if(needstroke)
	{	
		needstroke = 0;
		fprintf(out, "l%di%d\n",
			lastt,
			lasti);
		fprintf(out, "stroke\n");
	}
}

int search(struct catalog *cat, int x, int y, struct viewer *v, double *dist)
{
	return -1;
}

void selectall(struct catalog *cat, struct viewer *v)
{
}

void unselectall(struct catalog *cat, struct viewer *v)
{
}

void selectrange(struct catalog *cat, struct viewer *v, int type, 
	int mine, int maxe)
{
}

void unselectrange(struct catalog *cat, struct viewer *v, int type, 
	int mine, int maxe)
{
}

double catclosest(struct catalog *cat, struct viewer *v, long X, long Y, 
	long Z, long *n)
{
	struct object *obj;
	unsigned char amask = v->andmask;
	unsigned char smask = v->showmask;
	long i;
	long long dX, dY, dZ;
	long long min, d;


	min = ((long long)(1))<<61;
	*n = 0;

	if(!cat->objects) return min;
	if((cat->catmask & amask) != smask) return min;

	for(i = 0; i < cat->numobjects; i++)
	{
		obj = cat->objects + i;
		// Only accept those being displayed
		if((obj->mask & amask) != smask) continue;
		// Only accept true objects
		if(obj->type >= 192 && obj->type < 224) continue;
		dX = X - obj->x;
		dY = Y - obj->y;
		dZ = Z - obj->z;
		d = (long long)dX*(long long)dX 
		  + (long long)dY*(long long)dY 
		  + (long long)dZ*(long long)dZ;
		if(d < min) 
		{
			min = d;
			*n = i;
		}
	}
	return sqrt((double)min/(double)(((long long)(1))<<60));
}

double catdist(struct catalog *cat1, int index1, 
               struct catalog *cat2, int index2)
{
	struct object *obj1, *obj2;
	double x1, x2, y1, y2, z1, z2, r1, r2;

	obj1 = cat1->objects + index1;
	obj2 = cat2->objects + index2;

	x1 = obj1->x;
	y1 = obj1->y;
	z1 = obj1->z;
	x2 = obj2->x;
	y2 = obj2->y;
	z2 = obj2->z;
	
	r1 = sqrt(x1*x1 + y1*y1 + z1*z1);
	r2 = sqrt(x2*x2 + y2*y2 + z2*z2);

	return acos((x1*x2 + y1*y2 + z1*z2)/(r1*r2));
}

