/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "object.h"

GList *classlist = 0;

#define PI 3.1415926535898

/* From astronomical almanac */

void precess(struct object *obj, double deltat)
{
	static long long xx, xy, xz, yx, yy, yz, zx, zy, zz;
	static double lastdeltat = 0.0;

	double zetaA, zA, thetaA;
	double T; 
	double c1, c2, c3, s1, s2, s3;
	long long x, y, z;

	if(deltat == 0.0) return;
	if(deltat != lastdeltat)

	{
//		printf("Changing precession to %f years\n", deltat);
		T = deltat*(-0.01);

		zetaA  = (PI/180.0)*(0.6406161*T+0.0000839*T*T+0.0000050*T*T*T);
		zA     = (PI/180.0)*(0.6406161*T+0.0003041*T*T+0.0000051*T*T*T);
		thetaA = (PI/180.0)*(0.5567530*T+0.0001185*T*T+0.0000116*T*T*T);

		c1 = cos(zetaA);  s1 = sin(zetaA);
		c2 = cos(zA);     s2 = sin(zA);
		c3 = cos(thetaA); s3 = sin(thetaA);

		xx = (long long)((double)(1<<30)*(c1*c3*c2-s1*s2));
		xy = (long long)((double)(1<<30)*(-c1*c3*s2-s1*c2));
		xz = (long long)((double)(1<<30)*(c1*s3));
		yx = (long long)((double)(1<<30)*(s1*c3*c2+c1*s2));
		yy = (long long)((double)(1<<30)*(-s1*c3*s2+c1*c2));
		yz = (long long)((double)(1<<30)*(s1*s3));
		zx = (long long)((double)(1<<30)*(-s3*c2));
		zy = (long long)((double)(1<<30)*(s3*s2));
		zz = (long long)((double)(1<<30)*(c3));

		lastdeltat = deltat;
	}

	x = xx*(long long)obj->x + xy*(long long)obj->y + xz*(long long)obj->z;
	y = yx*(long long)obj->x + yy*(long long)obj->y + yz*(long long)obj->z;
	z = zx*(long long)obj->x + zy*(long long)obj->y + zz*(long long)obj->z;

	obj->x = (long)(x >> 30);
	obj->y = (long)(y >> 30);
	obj->z = (long)(z >> 30);
}

void getobjradec(struct object *obj, double *ra, double *dec)
{
	double x, y, z;
	x = obj->x;
	y = obj->y;
	z = obj->z;
	*ra = -atan2(y, x);
	*dec = atan2(z, sqrt(x*x+y*y));
}

int maxclassnumext()
{
	GList *l;
	int n, max = 0;
	for(l = classlist; l != 0; l = l->next)
	{
		n = ((struct classheader *)(l->data))->numext;
		if(n > max) max = n;
	}
	
	return max;
}

void initextension(struct classextension *e, const char *name,
	const char *units, int offset, float defmin, float defmax)
{
	int i;

	e->name = g_strdup(name);
	e->units = g_strdup(units);
	e->offset = offset;
	e->defaultmin = defmin;
	e->defaultmax = defmax;
	for(i = 0; i < 4; i++)
	{
		e->min[i] = defmin;
		e->max[i] = defmax;
	}
}

void setclassextensiondefaults(int viewerid, struct classheader *ch)
{
	int i, j;

	for(i = 0; i < ch->numext; i++) for(j = 0; j < 4; j++)
	{
		ch->extensions[i].min[j] = ch->extensions[i].defaultmin;
		ch->extensions[i].max[j] = ch->extensions[i].defaultmax;
	}
}

void setclassdefaults(int viewerid)
{
	GList *l;

	for(l = classlist; l != 0; l = l->next)
		setclassextensiondefaults(viewerid,
			(struct classheader *)(l->data));
}

int loadclassextension(struct classextension *e, const char *line)
{
	char name[100], units[100];
	int o;
	float min, max;

	if(sscanf(line, "%d %s %s %f %f", &o, name, units, &min, &max) != 5)
	{
		fprintf(stderr, "loadextension: error reading line!\n");
		return -1;
	}
	initextension(e, name, units, o, min, max);

	return 0;
}

struct classheader *new_classheader(int numext)
{
	struct classheader *ch;
	int i;

	ch = g_new(struct classheader, 1);

	if(numext < 0)
	{
		printf("new_classheader problem\n");
		numext = 0;
	}

	if(numext > 0) 
	{
		ch->extensions = g_new(struct classextension, numext);
		for(i = 0; i < numext; i++) 
		{
			ch->extensions[i].name = 0;
			ch->extensions[i].units = 0;
		}
	}
	else ch->extensions = 0;
	ch->class = 0;
	ch->fullname = 0;
	ch->numext = numext;
	ch->magoffset = -1;
	ch->classmask = 0x0F;	/* default: visible to all browsers */
	
	return ch;
}

void delete_classheader(struct classheader *ch)
{
	int i;

	if(!ch) return;
	g_free(ch->class);
	g_free(ch->fullname);
	if(ch->extensions)
	{
		for(i = 0; i < ch->numext; i++)
		{
			g_free(ch->extensions[i].name);
			g_free(ch->extensions[i].units);
		}
		g_free(ch->extensions);
	}
	g_free(ch);
}

void delete_all_classheaders()
{
	while(classlist)
	{
		if(classlist->data == 0) break;
		delete_classheader((struct classheader *)(classlist->data));
		classlist = g_list_remove(classlist, classlist->data);
	}
}

int loadclassheaders(const char *filename)
{
	FILE *in;
	char line[1000];
	int i, j;
	char class[100], fullname[500];
	int numext, hm;
	float magmin, magmax;
	struct classheader *ch;
	char *c;

	in = fopen(filename, "r");
	if(!in)
	{
		fprintf(stderr, "Cannot open extension descriptor file : %s\n",
			filename);
		return -1;
	}

	for(;;)
	{
		fgets(line, 999, in);
		if(feof(in)) break;
		for(i = 0; line[i] != 0; i++)
                        if(line[i] == '#' || line[i] < ' ') line[i] = 0;
		if(line[0] == 0) continue;

		if(sscanf(line, "%s %d %d %f %f %s", class, &numext, &hm,
			&magmin, &magmax, fullname) != 6)
		{
			fprintf(stderr, "Error in extension descriptor "
				"file : %s\n", filename);
			fclose(in);
			return -1;
		}

		for(c = fullname; *c != 0; c++) if(*c == '_') *c = ' ';

		ch = new_classheader(numext);
		
		if(numext > 0) for(i = 0; i < numext;) 
		{
			fgets(line, 999, in);
			if(feof(in))
			{
				fprintf(stderr, "File too short: %s\n",
					filename);
				fclose(in);
				return -1;
			}
			for(j = 0; line[j] != 0; j++)
				if(line[j] == '#' || line[j] < ' ') line[j] = 0;
			if(line[0] == 0) continue;
			if(loadclassextension(ch->extensions+i, line) < 0)
			{
				fclose(in);
				delete_all_classheaders();
				delete_classheader(ch);
				return -1;
			}

			i++;
		}
		for(i = 0; i < numext; i++) if(strcmp(ch->extensions[i].name,
			"Mag") == 0) ch->magoffset = ch->extensions[i].offset;

		ch->hasmag = hm;
		ch->defmagmin = magmin;
		ch->defmagmax = magmax;
		ch->class = g_strdup(class);
		ch->fullname = g_strdup(fullname);
		for(i = 0; i < 4; i++) 
		{
			ch->magmin[i] = magmin;
			ch->magmax[i] = magmax;
		}	

		classlist = g_list_append(classlist, ch);
	}

	fclose(in);

	return numext;
}

struct classheader *getclassheader(const char *key)
{
	GList *l;
	struct classheader *ch;

	for(l = classlist; l != 0; l = l->next)
	{
		ch = (struct classheader *)(l->data);
		if(strcmp(ch->class, key) == 0) return ch;
	}
//	fprintf(stderr, "Class not found in extension list!\n");
	return 0;
}
