/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "viewer.h"
#include "stars.h"
#include "gui.h"
#include "rgbdraw.h"
#include "catinfo.h"
#include "coordutils.h"
#include "print.h"
#include "objectwindow.h"
#include "configure.h"
#include "coordsys.h"

int numviewers = 0;

int viewerIDs[4] = {0, 0, 0, 0};
struct viewer *viewers[4] = {0, 0, 0, 0};

int getID()
{
	int i;
	for(i = 0; i < 4; i++)
	{
		if(viewerIDs[i] == 0)
		{
			viewerIDs[i] = 1;
			return i;
		}
	}
	fprintf(stderr, "CRITICAL ERROR!  ID Clobbered!\n");
	return 0;
}

void freeID(int id)
{
	if(viewerIDs[id] == 0) fprintf(stderr, "Weird! freeID\n");
	viewerIDs[id] = 0;
}

int isvalidviewer(struct viewer *v)
{
	int i;
	
	for(i = 0; i < 4; i++) if(viewerIDs[i] && viewers[i] == v) return 1;

	return 0;
}

static gint viewerexpose(GtkWidget *widget, GdkEventExpose *event, 
	struct viewer *v)
{
	if(v->image != 0) gdk_pixbuf_render_to_drawable(v->image, 
		widget->window, 
		widget->style->fg_gc[GTK_STATE_NORMAL], 
		event->area.x, event->area.y, 
		event->area.x, event->area.y, 
		event->area.width, event->area.height,
		GDK_RGB_DITHER_NORMAL, 0, 0);

	return FALSE;
}

static gint viewerconfig(GtkWidget *widget, GdkEventConfigure *event,
	struct viewer *v)
{
	resize(v, widget->allocation.width, widget->allocation.height);
	
	paint(v);

	return TRUE;
}

static gint viewerclick(GtkWidget *widget, GdkEventButton *event,
	struct viewer *v)
{
	double ra, dec;
	double sx, sy, dx, dy;

	static double lastx = 0, lasty = 0;

	switch(event->button)
	{
		case 1:
		{
			if(event->type == GDK_BUTTON_PRESS)
			{
				lastx = event->x;
				lasty = event->y;
			}
			else if(event->type == GDK_BUTTON_RELEASE)
			{
				dx = fabs(lastx - event->x);
				dy = fabs(lasty - event->y);

				if(dx + dy < 5)
				{
					getradec(v, event->x, event->y, 
						&ra, &dec);
				}
				else
				{
					getradec(v, 0.5*(event->x + lastx),
						    0.5*(event->y + lasty),
						    &ra, &dec);
					sx = (double)v->xres/dx;
					sy = (double)v->yres/dy;
					if(sx < sy) v->view->zoom *= sx;
					else v->view->zoom *= sy;
				}
				if(ra < 0.0) return TRUE;

				setcenter(v, ra, dec);
				paint(v);
			}
			break;
		}
		case 2:
		{
			if(event->type != GDK_BUTTON_PRESS) break;
			setzoom(v, v->view->zoom/2.0);
			break;
		}
		case 3:
		{
			if(event->type != GDK_BUTTON_PRESS) break;
			getradec(v, event->x, event->y, &ra, &dec);
			findclosest(v, ra, dec);
			break;
		}
	}

	return TRUE;
}

gint closeviewer(GtkWidget *window, GdkEvent *event, struct viewer *v)
{
	hideviewerselector(v);

	freeID(v->viewerID);
	numviewers--;
	if(numviewers == 0) gtk_main_quit();
	if(v->image != 0) gdk_pixbuf_unref(v->image);

	gtk_widget_hide(v->viewerwindow);
	gtk_widget_destroy(v->viewerwindow);
	
	deletelocation(v->loc);
	g_free(v);
	
	return TRUE;
}

struct viewer *newviewer()
{
	struct viewer *v;
	GtkWidget *button, *startable, *buttonbar, *viewerbox;
	
	if(numviewers >= 4)
	{
		fprintf(stderr, "Only 4 viewers allowed now!\n");
		return 0;
	}

	v = g_new(struct viewer, 1);

	numviewers++;
	v->viewerID = getID();
	viewers[v->viewerID] = v;
	v->image = 0;
	v->N = 30;
	
	v->coordinateproperties = newcoordpropslist();
	v->view = newvista();
	v->showmask = 1<<(v->viewerID);
	v->andmask = (v->showmask) | 0xF0;
	v->viewerwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);

	applyclasslimits(0, v->viewerID);

	viewerbox = gtk_vbox_new(FALSE, 0);
	buttonbar = newbuttonbar(v);
	startable = gtk_table_new(2, 2, FALSE);
	v->stararea = gtk_drawing_area_new();
	gtk_drawing_area_size(GTK_DRAWING_AREA(v->stararea), 640, 480);
	v->raadjust = gtk_adjustment_new(0.0, 0.0, 24.0, 1.0, 6.0, 1.0);
	v->decadjust = gtk_adjustment_new(0.0, 0.0, 180.0, 15.0, 45.0, 15.0);
	v->rascroll = gtk_hscrollbar_new(GTK_ADJUSTMENT(v->raadjust));
	v->decscroll = gtk_vscrollbar_new(GTK_ADJUSTMENT(v->decadjust));
	v->statusbar = gtk_statusbar_new();
	gtk_statusbar_push(GTK_STATUSBAR(v->statusbar), 
		gtk_statusbar_get_context_id(GTK_STATUSBAR(v->statusbar),
		"newviewer"), "Welcome to Stars!");
	gtk_widget_show(v->statusbar);
	gtk_range_set_update_policy(GTK_RANGE(v->rascroll), GTK_UPDATE_DELAYED);
	gtk_range_set_update_policy(GTK_RANGE(v->decscroll),GTK_UPDATE_DELAYED);
	gtk_widget_show(v->rascroll);
	gtk_widget_show(v->decscroll);
	button = gtk_button_new();
	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
                GTK_SIGNAL_FUNC(zoom_1), (GtkObject *)v);
	gtk_widget_show(button);
	gtk_table_attach(GTK_TABLE(startable), v->stararea, 0, 1, 0, 1, 
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);
	gtk_table_attach(GTK_TABLE(startable), v->rascroll, 0, 1, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_SHRINK, 0, 0);
	gtk_table_attach(GTK_TABLE(startable), v->decscroll, 1, 2, 0, 1, 
		GTK_SHRINK, GTK_FILL | GTK_EXPAND, 0, 0);
	gtk_table_attach(GTK_TABLE(startable), button, 1, 2, 1, 2, 
		GTK_FILL, GTK_FILL, 0, 0);

	gtk_box_pack_start(GTK_BOX(viewerbox), buttonbar, FALSE,FALSE,0);
	gtk_box_pack_start(GTK_BOX(viewerbox), startable, TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(viewerbox), v->statusbar, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(v->viewerwindow), viewerbox);
	gtk_widget_show(viewerbox); 
	gtk_widget_show(v->stararea);
	gtk_window_set_title(GTK_WINDOW(v->viewerwindow), "Stars");
	gtk_widget_show(startable);

	gtk_signal_connect(GTK_OBJECT(v->raadjust), "value_changed",
		GTK_SIGNAL_FUNC(scrollwin), v);
	gtk_signal_connect(GTK_OBJECT(v->decadjust), "value_changed",
		GTK_SIGNAL_FUNC(scrollwin), v);
	gtk_signal_connect(GTK_OBJECT(v->stararea), 
		"expose_event", (GtkSignalFunc)viewerexpose, v);
	gtk_signal_connect(GTK_OBJECT(v->stararea),
		"configure_event", (GtkSignalFunc)viewerconfig, v);
	gtk_signal_connect(GTK_OBJECT(v->stararea),
		"button_press_event", (GtkSignalFunc)viewerclick, v);
	gtk_signal_connect(GTK_OBJECT(v->stararea),
		"button_release_event", (GtkSignalFunc)viewerclick, v);
	gtk_widget_set_events (v->stararea, GDK_EXPOSURE_MASK
                             | GDK_LEAVE_NOTIFY_MASK
                             | GDK_BUTTON_PRESS_MASK
			     | GDK_BUTTON_RELEASE_MASK
                             | GDK_POINTER_MOTION_MASK
                             | GDK_POINTER_MOTION_HINT_MASK);

	gtk_signal_connect(GTK_OBJECT(v->viewerwindow), "delete_event",
                            GTK_SIGNAL_FUNC(closeviewer), (gpointer)v);
	gtk_widget_set_events(v->stararea, GDK_EXPOSURE_MASK
					 | GDK_LEAVE_NOTIFY_MASK
					 | GDK_BUTTON_PRESS_MASK
					 | GDK_BUTTON_RELEASE_MASK
					 | GDK_BUTTON_MOTION_MASK
					 | GDK_POINTER_MOTION_MASK
					 | GDK_POINTER_MOTION_HINT_MASK);

	resize(v, v->stararea->allocation.width,
		  v->stararea->allocation.height);

	computematrix(v);

	gtk_widget_show(v->viewerwindow);

	v->loc = newlocation();

	return v;
}

void matrixrotz(double M[3][3], double angle)
{
	double M0[3][3], R[3][3];
	int i, j, k;
	double s, c;

	for(i = 0; i < 3; i++) for(j = 0; j < 3; j++)
	{
		M0[i][j] = M[i][j];
		M[i][j] = 0.0;
	}
	s = sin(angle);
	c = cos(angle);
	R[0][0] = 1.0;
	R[1][0] = R[0][1] = R[0][2] = R[2][0] = 0.0;
	R[1][1] = c;
	R[2][2] = c;
	R[1][2] = s;
	R[2][1] = -s;
	for(i = 0; i < 3; i++) for(j = 0; j < 3; j++)
		for(k = 0; k < 3; k++) M[i][j] += R[i][k]*M0[k][j];
}

void computematrix(struct viewer *v)
{
	int M;
	double S, r;
	double cRA, cDec, sRA, sDec;
	int i, j;
	double flip = 1.0;
	double *viewrefpos;
	double viewrefx, viewrefy, angle;

	cRA = cos(v->view->ra_center);
	cDec = cos(v->view->dec_center);
	sRA = sin(v->view->ra_center);
	sDec = sin(v->view->dec_center);	
	if(v->view->fliplr) flip = -1.0;
	viewrefpos = getvistarefcoords(v->view->uprefid);

	v->drot[0][0] = cRA*cDec;
	v->drot[0][1] = -sRA*cDec;
	v->drot[0][2] = sDec;
	v->drot[1][0] = flip*sRA;
	v->drot[1][1] = flip*cRA;
	v->drot[1][2] = 0.0;
	v->drot[2][0] = -cRA*sDec;
	v->drot[2][1] = sRA*sDec;
	v->drot[2][2] = cDec;

	if(viewrefpos)
	{
		viewrefx = v->drot[1][0]*viewrefpos[0] + 
			   v->drot[1][1]*viewrefpos[1] +
			   v->drot[1][2]*viewrefpos[2];
		viewrefy = v->drot[2][0]*viewrefpos[0] +
			   v->drot[2][1]*viewrefpos[1] +
			   v->drot[2][2]*viewrefpos[2];
		angle = atan2(viewrefy, viewrefx) -
				(90.0 + v->view->uprotation)*PI/180.0;
		matrixrotz(v->drot, angle);
	}
	if(v->xres < v->yres) r = v->xres;
	else r = v->yres;
	r /= 2.0;

	v->scale = v->view->zoom*r;
	for(M = 0; M < 30; M++) if((1<<M) >= v->scale) break;
	v->offx = (long long)v->xres << (59-M);
	v->offy = (long long)v->yres << (59-M);
	S = v->scale*(double)(1<<(30-M));

	for(j = 0; j < 3; j++) for(i = 0; i < 3; i++)
		v->rot[j][i] = (long)(S*v->drot[j][i]);

	v->N = 60-M;
	configmatrixchanged(v);
}

void setcenter(struct viewer *v, double ra, double dec)
{
	static guint messid = 0;
	static guint contid = 0;
	gfloat ps;
	int u;
	gchar rastr[40], decstr[40], statusstr[100];

	if(ra < 100000.0) v->view->ra_center = ra;
	else ra = v->view->ra_center;
	if(dec < 100000.0) v->view->dec_center = dec;
	else dec = v->view->dec_center;
	computematrix(v);
	GTK_ADJUSTMENT(v->raadjust)->value = ra*12/PI;
	GTK_ADJUSTMENT(v->decadjust)->value = 90.0-dec*180/PI;
	ps = 6.0*v->xres/v->scale;
	if(ps > 12.0) ps = 12.0;
	GTK_ADJUSTMENT(v->raadjust)->upper = 24.0+ps;
	GTK_ADJUSTMENT(v->raadjust)->page_size = ps;
	ps = 45.0*v->yres/v->scale;
	if(ps > 90.0) ps = 90.0;
	GTK_ADJUSTMENT(v->decadjust)->upper = 180.0+ps;
	GTK_ADJUSTMENT(v->decadjust)->page_size = ps;
	
//	gtk_adjustment_value_changed(GTK_ADJUSTMENT(v->raadjust));
//	gtk_adjustment_value_changed(GTK_ADJUSTMENT(v->decadjust));

	if(!contid) contid = gtk_statusbar_get_context_id(
		GTK_STATUSBAR(v->statusbar), "setcenter");

	if(messid) gtk_statusbar_remove(GTK_STATUSBAR(v->statusbar),
		contid, messid);

	double2text(ra*12/PI, rastr, ':');
	rastr[12] = 0;
	double2text(dec*180/PI, decstr, ':');
	if(decstr[0] == '-') decstr[12] = 0;
	else decstr[11] = 0;
	u = uranopage(ra*12/PI, dec*180/PI);
	sprintf(statusstr, "RA = %s    Dec = %s   U = %d", rastr, decstr, u);
	messid = gtk_statusbar_push(GTK_STATUSBAR(v->statusbar),
		contid, statusstr);
}

void viewersetvista(struct viewer *v, const struct vista *view)
{
	if(v->view) deletevista(v->view);
	else fprintf(stderr, "viewersetvista : Weird, null v->view\n");

	v->view = copyvista(view);
	setcenter(v, 999999.9, 999999.9);
	paint(v);
}

void setzoom(struct viewer *v, double zoom)
{
	gfloat ps;

	if(zoom < 1.0)
	{
		printf("Warning, attempting to zoom < 1.0\n");
		zoom = 1.0;
	}
	v->view->zoom = zoom;
	computematrix(v);
	ps = 6.0*v->xres/v->scale;
	if(ps > 12.0) ps = 12.0;
	GTK_ADJUSTMENT(v->raadjust)->upper = 24.0+ps;
	GTK_ADJUSTMENT(v->raadjust)->page_size = ps;
	GTK_ADJUSTMENT(v->raadjust)->page_increment = ps/2.0;
	GTK_ADJUSTMENT(v->raadjust)->step_increment = ps/10.0;
	ps = 45.0*v->yres/v->scale;
	if(ps > 90.0) ps = 90.0;
	GTK_ADJUSTMENT(v->decadjust)->upper = 180.0+ps;
	GTK_ADJUSTMENT(v->decadjust)->page_size = ps;
	GTK_ADJUSTMENT(v->decadjust)->page_increment = ps/2.0;
	GTK_ADJUSTMENT(v->decadjust)->step_increment = ps/10.0;
	
//	gtk_adjustment_value_changed(GTK_ADJUSTMENT(v->raadjust));
//	gtk_adjustment_value_changed(GTK_ADJUSTMENT(v->decadjust));

	paint(v);
}

void getradec(struct viewer *v, double x, double y, double *ra, double *dec)
{
	double x0, y0, z0;
	double x1, y1, z1;

	y0 = (x - v->xres/2.0)/v->scale;
	z0 = (v->yres/2.0 - y)/v->scale;
	if(y0*y0 + z0*z0 > 1.0) 
	{
		*ra = -1.0;
		*dec = 0.0;
		return;
	}
	x0 = sqrt(1.0 - y0*y0 - z0*z0);

	x1 = v->drot[0][0]*x0 + v->drot[1][0]*y0 + v->drot[2][0]*z0;
	y1 = v->drot[0][1]*x0 + v->drot[1][1]*y0 + v->drot[2][1]*z0;
	z1 = v->drot[0][2]*x0 + v->drot[1][2]*y0 + v->drot[2][2]*z0;

	*ra = -atan2(y1, x1);
	while(*ra < 0.0) *ra += 2.0*PI;
	*dec = asin(z1);
}

void resize(struct viewer *v, int newxres, int newyres)
{
	if(v->xres != newxres || v->yres != newyres)
	{
		if(v->image != 0) gdk_pixbuf_unref(v->image);
		if(newxres == 0 || newyres == 0)
			v->image = 0;
		else v->image = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8, 
			newxres, newyres);

		v->xres = newxres;
		v->yres = newyres;
	}
	computematrix(v);
}

void paint(struct viewer *v)
{
	int w, h, rs;
	GtkWidget *widget;

	if(catalogs == 0) 
	{
		fprintf(stderr, "paint: catalogs = 0\n");
		return;
	}

	if(v->image == 0)
	{
		fprintf(stderr, "paint: image = 0\n");
		return;
	}

	w = gdk_pixbuf_get_width(v->image);
	h = gdk_pixbuf_get_height(v->image);
	rs = gdk_pixbuf_get_rowstride(v->image);

	memset(gdk_pixbuf_get_pixels(v->image), 0, rs*h);

	widget = v->stararea;

	render_pixmap(icons, v);

	gdk_pixbuf_render_to_drawable(v->image, widget->window, 
		widget->style->fg_gc[GTK_STATE_NORMAL], 0, 0, 0, 0, w, h,
		GDK_RGB_DITHER_NORMAL, 0, 0);
}

void refreshimage(struct viewer *v)
{
	int w, h;
	
	w = gdk_pixbuf_get_width(v->image);
	h = gdk_pixbuf_get_height(v->image);

	gdk_pixbuf_render_to_drawable(v->image, v->stararea->window, 
		v->stararea->style->fg_gc[GTK_STATE_NORMAL], 0, 0, 0, 0, w, h,
		GDK_RGB_DITHER_NORMAL, 0, 0);
}

void paintall()
{
	int i;
	
	for(i = 0; i < 4; i++) if(viewerIDs[i]) paint(viewers[i]);
}

void findclosest(struct viewer *v, double ra, double dec)
{
	static guint contid;
	guint messid;
	GList *l, *bestcat=0;
	long n, bestnum=0;
	double x, y, z;
	long X, Y, Z;
	double dist, distance = -1.0;

	if(!contid) contid = gtk_statusbar_get_context_id(
		GTK_STATUSBAR(v->statusbar), "setcenter");

	messid = gtk_statusbar_push(GTK_STATUSBAR(v->statusbar), 
		contid, "Looking up object.  Please wait...");

	if(catalogs == 0)
        {
                fprintf(stderr, "findclosest: catalogs = 0\n");
                return;
        }

	x = cos(ra)*cos(dec);
	y = -sin(ra)*cos(dec);
	z = sin(dec);

	X = (long)(x*(double)(1<<30));
	Y = (long)(y*(double)(1<<30));
	Z = (long)(z*(double)(1<<30));

	for(l = catalogs; l != 0; l=l->next) if(l->data != 0)
	{
		if(((struct catalog *)(l->data))->searchable == 0) continue;
		dist = catclosest((struct catalog *)(l->data), v, X, Y, Z, &n);
		if(dist < 0.0) continue;
		if(distance < 0.0 || dist < distance)
		{
			bestcat = l;
			distance = dist;
			bestnum = n;
		}
	}

	if(bestcat) addselectedobject((struct catalog *)(bestcat->data), bestnum, v);

	gtk_statusbar_remove(GTK_STATUSBAR(v->statusbar),
		contid, messid);
}

void applyclasslimits(struct catalog *cat0, int viewerid)
{
	GList *l;
	int amask;
	int omask;
	int i, n;
	struct catalog *cat;
	int magmin, magmax, m;
	float min, max, val;
	int offset;
	float *base;
	int numext;


	if(viewerIDs[viewerid] == 0) return;
	amask = 0xFF & (~(1<<viewerid));
	omask = 1<<viewerid;

	for(l = catalogs; l != 0; l = l->next)	// Replace with iterator
	{
		cat = (struct catalog *)(l->data);
		if(cat0 != 0 && cat0 != cat) continue;
		if(cat->objects == 0) continue;
		if(cat->ch == 0)
		{
//			printf("Multi-class catalogs not yet supported\n");
		}
		else
		{
			numext = cat->ch->numext;
			if(cat->ch->hasmag == 1)
			{
				magmin = (cat->ch->magmin[viewerid]+2.0)*10.0;
				magmax = (cat->ch->magmax[viewerid]+2.0)*10.0;
				for(i = 0; i < cat->numobjects; i++)
				{
					m = cat->objects[i].mag;
					if(m >= magmin && m < magmax)
						cat->objects[i].mask |= omask;
					else cat->objects[i].mask &= amask;
				}
			}
			else for(i = 0; i < cat->numobjects; i++)
				cat->objects[i].mask |= omask;

			for(n = 0; n < numext; n++)
			{
				offset = cat->ch->extensions[n].offset;
				min    = cat->ch->extensions[n].min[viewerid];
				max    = cat->ch->extensions[n].max[viewerid];
				base = cat->extension+offset;
				for(i = 0; i < cat->numobjects; i++)
				{
					val = base[i*numext];
					if(val < min || val >= max)
						cat->objects[i].mask &= amask;
				}
			}
		}
	}
}

void applyclasslimitsall(struct catalog *cat)
{
	int i;

	for(i = 0; i < 4; i++) if(viewerIDs[i]) applyclasslimits(cat, i); //paint(viewers[i]);
}

void viewersetlocation(struct viewer *v, const struct location *loc)
{
	if(v->loc) deletelocation(v->loc);
	v->loc = copylocation(loc);
}
