/* ------------------------------------------------------------------------
 * $Id: RendererImpl.hh,v 1.19 2001/08/27 15:10:26 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-10-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _RendererImpl_hh_
#define _RendererImpl_hh_

// -- System Includes
#include <string>
#include <stack>

// -- 3Dwm Includes
#include "Nobel/Types.hh"
#include "Nobel/Image.hh"
#include "Nobel/Renderer.hh"
#include "Nobel/Appearance.hh"
#include "Polhem/VisitorImpl.hh"

// -- Forward Declarations
//class Matrix3D;

// -- Class Declarations

class RendererImpl : public virtual POA_Nobel::Renderer,
		     public VisitorImpl {
    
public:
    
    /**
     * Default constructor.
     **/
    RendererImpl(int width, int height);
    
    /**
     * Destructor.
     **/
    virtual ~RendererImpl();

    // IDL operations
    virtual void visit(Nobel::Node_ptr n);
    virtual void renderPoints(const Nobel::PointMesh &mesh);
    virtual void renderLines(const Nobel::LineMesh &mesh);
    virtual void renderTriangles(const Nobel::TriangleMesh &mesh);
    virtual void identityTransform();

    // Return the maximum texture size supported by the current hardware
    virtual CORBA::Long getMaxTexSize();
    virtual void disableTexture();

    virtual CORBA::Boolean genTexture(CORBA::Long width, CORBA::Long height, 
				      Nobel::PixelType texFormat, 
				      const Nobel::BinaryData &texData, 
				      CORBA::Long_out texName);

    virtual void updateTexture(const Nobel::Rectangle & rect,
			       Nobel::PixelType texFormat, 
			       const Nobel::BinaryData &texData,
			       CORBA::Long texName);
    
    virtual void applyTexture(CORBA::Long texName);
    virtual void applyMaterial(const Nobel::MaterialAttributes &ma);
    /*
    virtual void setProjection(CORBA::Float fov, CORBA::Float aspect);
    virtual void getProjection(CORBA::Float &fov, CORBA::Float &aspect);
    */
    
    /**
     * Save the rendering state. This will push the changes to the
     * current rendering state to a state stack and prepare for a
     * new state save.
     **/
    virtual void save();
    
    /**
     * Restore the rendering state. The changes to the rendering state
     * will be popped off the state stack and reinstated in the
     * renderer.
     **/
    virtual void restore();
    
    /**
     * Premultiply a given transform with the current transform,
     * yielding a new transform. This means, in effect, applying the
     * local transform of a node child to the current global
     * transform.
     * 
     * @param t the new transform to premultiply to the current.
     **/    
    virtual void multiplyTransform(Nobel::Transform_ptr t);

    /**
     * Set the view transformation matrix into the renderer. This
     * matrix will be applied to all matrix operations in the
     * renderer.
     *
     * @param view the new view transformation matrix.
     **/
    void setViewTransform(const Matrix3D &view);

    /**
     * Retrieve the name of the renderer.
     *
     * @return name of the renderer.
     **/
    std::string getName() const;

    /**
     * Retrieve vendor of the renderer. 
     *
     * @return renderer vendor.
     **/
    std::string getVendor() const;

    /**
     * Retrieve version of the renderer.
     *
     * @return renderer version.
     **/
    std::string getVersion() const;

    /**
     * Retrieve current renderer viewport dimensions.
     *
     * @param w viewport width.
     * @param h viewport height.
     */
    void getViewportDimensions(int &w, int &h) const {
	w = _width;
	h = _height; 
    }

    /**
     * Retrieve current projection parameters.
     *
     * @param fov new field-of-view setting (in degrees).
     * @param aspect aspect ratio (width to height).
     * @param near distance to near clipping plane.
     * @param far distance to far clipping plane.
     **/
    void getProjection(float &fov, float &aspect,
		       float &near, float &far) const;
    
    /**
     * Set new projection parameters into the renderer. 
     *
     * @param fov new field-of-view setting (in degrees).
     * @param aspect aspect ratio (width to height).
     * @param near distance to near clipping plane.
     * @param far distance to far clipping plane.
     **/
    void RendererImpl::setProjection(float fov, float aspect,
				     float near, float far);

private:

    /**
     * Initialize and load the texture into OpenGL
     **/    
    bool genTexture(Nobel::Texture_ptr);

    enum StateFlags { st_transform, st_material };
    
    /// Drawing state struct
    struct state_t {
	unsigned long flags;			// Which states are affected?
    };
    
    /// Current state
    state_t st_current;
    
    /// Drawing state stack
    std::stack<state_t> st_stack;

    /// View transformation matrix
    Matrix3D _view;

    // Viewport dimensions
    int _width;
    int _height;

    // Project parameters
    float _fov;
    float _aspect;
    float _near;
    float _far;
};

// -- Inlines 

inline void RendererImpl::setViewTransform(const Matrix3D &view) {
    _view = view;
}

inline void RendererImpl::getProjection(float &fov, float &aspect,
					float &near, float &far) const {
    fov = _fov; aspect = _aspect; near = _near; far = _far;
}

#endif /* RendererImpl.hh */
