/* ------------------------------------------------------------------------
 * $Id: ExpatParser.cc,v 1.1 2001/08/16 11:32:25 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * This C++ SAX implementation (Simple API for XML) is derived from
 * David Megginson's original Java SAX that can be found on 
 * http://www.megginson.com/SAX/.
 * ------------------------------------------------------------------------
 * File created 2001-08-14 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------ 
 */

// -- System Includes
#include <expat.h>
#include <fstream>

// -- Local Includes
#include "Celsius/SAX/Exception.hh"
#include "Celsius/SAX/AttributeListImpl.hh"
#include "Celsius/SAX/ErrorHandler.hh"
#include "Celsius/SAX/DTDHandler.hh"
#include "Celsius/SAX/EntityResolver.hh"
#include "Celsius/SAX/DocumentHandler.hh"
#include "Celsius/SAX/InputSource.hh"
#include "Celsius/SAX/ExpatParser.hh"

using namespace SAX;

// -- Local Function Prototypes
static void startElementHandler(void *, const XML_Char *, const XML_Char **);
static void endElementHandler(void *, const XML_Char *);
static void characterDataHandler(void *, const XML_Char *, int);

// -- Code Segment

ExpatParser::ExpatParser()
    : _parser(XML_ParserCreate(0)), _resolver(0), _dtdHandler(0),
      _docHandler(0), _errorHandler(0)
{
    // Associate the parser with a pointer to this instance
    XML_SetUserData(_parser, this);
}

ExpatParser::~ExpatParser()
{
    // Delete the parser
    XML_ParserFree(_parser);
}

void ExpatParser::setEntityResolver(EntityResolver *resolver)
{
    std::cerr << "Entity resolver not supported!" << std::endl;
}

void ExpatParser::setDTDHandler(DTDHandler *handler)
{
    std::cerr << "DTD handler not supported!" << std::endl;
}

void ExpatParser::setDocumentHandler(DocumentHandler *handler)
{
    // Update document handler
    _docHandler = handler;
    
    // If this is a valid handler, set event handler functions
    if (_docHandler) {
	XML_SetElementHandler(_parser, startElementHandler, endElementHandler);
	XML_SetCharacterDataHandler(_parser, characterDataHandler);
    }
    // It is not, clear out handlers to stop handling events
    else {
	XML_SetElementHandler(_parser, 0, 0);
	XML_SetCharacterDataHandler(_parser, 0);
    }
}
	
void ExpatParser::setErrorHandler(ErrorHandler *handler)
{
    std::cerr << "Error handler not supported!" << std::endl;
}

void ExpatParser::parseStream(std::istream &is,
			      const std::string &publicId,
			      const std::string &systemId)
{
    // Can we use the stream?
    if (!is) throw SAX::Exception("invalid input source");
    
    // Send a start document event to any handler
    if (_docHandler) _docHandler->startDocument();
    
    // Now read from the input stream
    while (!is.eof()) {
	
	const int BUFFER_SIZE = 2048;
	char buffer[BUFFER_SIZE];

	// Read data from the file
	is.read(buffer, BUFFER_SIZE);
	
	// Parse it and report any errors
	if (!XML_Parse(_parser, buffer, is.gcount(), false))
	    throw ParseException(XML_ErrorString(XML_GetErrorCode(_parser)),
				 publicId, systemId, 
				 XML_GetCurrentLineNumber(_parser),
				 XML_GetCurrentColumnNumber(_parser));
    }
    
    // We need to call expat with a final flag set to true
    if (!XML_Parse(_parser, 0, 0, true))
	throw ParseException(XML_ErrorString(XML_GetErrorCode(_parser)),
			     publicId, systemId, 
			     XML_GetCurrentLineNumber(_parser),
			     XML_GetCurrentColumnNumber(_parser));
    
    // Finally, if we have a document handler, send an end event to it
    if (_docHandler) _docHandler->endDocument();
}

void ExpatParser::parse(InputSource &source)
{
    // If the input source holds a file, then use it
    if (source.getStream() != 0)
	parseStream(*source.getStream(), source.getPublicId(),
		    source.getSystemId());
    // Otherwise open a new file stream using the system id (file name)
    else {
	std::ifstream ifs(source.getSystemId().c_str());
	parseStream(ifs, source.getPublicId(), source.getSystemId());
    }
}
	
void ExpatParser::parse(const std::string &systemId)
{
    // Call the default parse function with a new input source
    InputSource source(systemId);
    parse(source);
}
	
void startElementHandler(void *userData, const XML_Char *name,
			 const XML_Char **atts)
{
    AttributeListImpl attributes;
    
    // Reinterpret the data pointer
    ExpatParser *parser = reinterpret_cast<ExpatParser *>(userData);

    // Build an attribute list
    if (atts != 0)
	for (int i = 0; atts[i] != 0; i += 2)
	    attributes.addAttribute(atts[i], "", atts[i + 1]);
    
    // Now call the document handler with this event
    parser->getDocumentHandler()->startElement(name, attributes);
}

void endElementHandler(void *userData, const XML_Char *name)
{
    // Reinterpret the data pointer
    ExpatParser *parser = reinterpret_cast<ExpatParser *>(userData);
    
    // Send the end element event to the handler
    parser->getDocumentHandler()->endElement(name);
}

void characterDataHandler(void *userData, const XML_Char *s, int len)
{
    // Reinterpret the data pointer
    ExpatParser *parser = reinterpret_cast<ExpatParser *>(userData);
    
    // Build string and send the characters event to the handler
    std::string data(s, len);
    parser->getDocumentHandler()->characters(data);
}
