/* ------------------------------------------------------------------------
 * $Id: BSP.cc,v 1.3 2001/08/02 14:22:11 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-16 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Solid/BSP.hh"
#include "Solid/BSPTree.hh"

// -- Code Segment

BSPNode::BSPNode(const Plane3D &bp)
    : _bp(bp), _front(0), _back(0)
{
    // empty
}

BSPNode::~BSPNode()
{
    // Deallocate the BSP hiearchy
    if (_front != 0) delete _front;
    if (_back != 0) delete _back;
}

void BSPNode::setFront(BSP *front)
{
    // Update pointer
    _front = front;
}

void BSPNode::setBack(BSP *back)
{
    // Update pointer
    _back = back;
}

BSP *BSPNode::clone() const
{
    // Create an exact copy of the current node
    BSPNode *node = new BSPNode(_bp);

    // Copy faces as well
    node->_faces = _faces;

    // Clone any existing subtrees
    if (_front != 0) node->setFront(_front->clone());
    if (_back != 0) node->setBack(_back->clone());

    return node;
}

int BSPNode::getFaceCount() const
{
    // Sum this node's face count and the face count of any subtrees
    return _faces.size() + 
	(_front != 0 ? _front->getFaceCount() : 0) +
	(_back != 0 ? _back->getFaceCount() : 0);
}

void BSPNode::complement()
{
    // Flip the boundary partitioning plane
    _bp.flip();

    // Flip all triangle faces
    for (std::vector<TriangleFace>::iterator i = _faces.begin();
	 i != _faces.end(); i++) {
       	i->v.flip();
	i->n.flip();
	i->tc.flip();
    }

    // Now, complement any existing subtrees
    if (_front != 0) _front->complement();
    if (_back != 0) _back->complement();
}

void BSPNode::buildFaceList(std::vector<TriangleFace> &faces) const
{
    // Insert the faces of this node into the face list
    faces.insert(faces.end(), _faces.begin(), _faces.end());

    // Now, insert faces for any existing subtrees
    if (_front != 0) _front->buildFaceList(faces);
    if (_back != 0) _back->buildFaceList(faces);
}

bool BSPNode::isContained(const Vector3D &p) const
{
    // Classify the point against the boundary partition
    switch (BSPTreeBuilder::classifyPoint(_bp, p)) {
    case BSPTreeBuilder::Coincident: 
	// If it is coincident, we need to check both subtrees
	return (_front != 0 ? _front->isContained(p) : false) ||
	    (_back != 0 ? _back->isContained(p) : false);
    case BSPTreeBuilder::InFrontOf:
	// Check front subtree
	return _front != 0 ? _front->isContained(p) : false;
    case BSPTreeBuilder::InBackOf:
	// Check back subtree
	return _back != 0 ? _back->isContained(p) : false;
    default:
	return false;
    }
}

void BSPNode::pruneFaces(BSPTree &t, BSP *bsp, BSPTreeBuilder::SetOperation op)
{
    // All right, so what we do now is that we decide on which
    // of this node's existing faces to prune away.
    
    // First, sort faces into three different types
    std::vector<TriangleFace> on_list, in_list, out_list;
    
    // Use the BSP tree as a partitioner
    bsp->partitionFaceList(t, _faces, on_list, in_list, out_list);
    
    // All faces on boundaries should be included (this will
    // also clear the existing faces)
    _faces = on_list;
    
    // Now, if this is an intersection operation...
    if (op == BSPTreeBuilder::Intersection) {
	
	// ...we want to save those faces that are outside.
	_faces.insert(_faces.end(), out_list.begin(), out_list.end());
    }
    // If, on the other hand, we're dealing a union...
    else {
	
	// ...it's the inside faces we want to retain.
	_faces.insert(_faces.end(), in_list.begin(), in_list.end());
    }
}

BSP *BSPNode::combine(BSPTree &t, const std::vector<TriangleFace> &faces, 
		      BSPTreeBuilder::SetOperation op)
{
    std::vector<TriangleFace> front_list, back_list;
    
    // Now, partition all faces into the front and back halfspaces.
    // Add all coincident faces to the current face list for the node.
    BSPTreeBuilder::partition(t, _bp, faces, _faces, front_list, back_list);
    
    // Is the front face list empty, or do we have to recurse?
    if (front_list.empty() == false && _front != 0) {

	// Combine the front faces with the front subtree
	BSP *ptr = _front->combine(t, front_list, op);

	// Did we get a new immediate child?
	if (ptr != _front) {
	    
	    // Then delete the old leaf and use the new BSP instead
	    delete _front;
	    _front = ptr;
	}

	// Prune aginst front subtree if this is a union operation
	if (op == BSPTreeBuilder::Union) pruneFaces(t, _front, op);
    }

    // All right, what about the back list, do we need to recurse? 
    if (back_list.empty() == false && _back != 0) {

	// Combine the back faces with the back subtree
	BSP *ptr = _back->combine(t, back_list, op);

	// Did we get a new immediate child?
	if (ptr != _back) {

	    // Then delete the old leaf and use the new BSP instead
	    delete _back;
	    _back = ptr;
	}

	// Prune aginst back subtree if this is an intersection operation
	if (op == BSPTreeBuilder::Intersection) pruneFaces(t, _back, op);
    }
    
    return this;
}

void BSPNode::partitionFaceList(BSPTree &t,
				const std::vector<TriangleFace> &faces,
				std::vector<TriangleFace> &on_list,
				std::vector<TriangleFace> &in_list,
				std::vector<TriangleFace> &out_list) const
{
    std::vector<TriangleFace> front_list, back_list;

    // Partition the faces against the current binary partitioning
    BSPTreeBuilder::partition(t, _bp, faces, on_list, front_list, back_list);
    
    // If there are any faces in the front list...
    if (front_list.empty() == false && _front != 0) {
	
	// ...pass them on recursively to the front subtree
	_front->partitionFaceList(t, front_list, on_list, in_list, out_list);
    }

    // Likewise, if the back list is not empty...
    if (back_list.empty() == false && _back != 0) {

	// ...we want to check them recursively against the back subtree
	_back->partitionFaceList(t, back_list, on_list, in_list, out_list);
    }
}

void BSPNode::transform(const Matrix3D &inv_trafo)
{
    // Transform the partitioning plane
    _bp.transform(inv_trafo);

    // Now, complement any existing subtrees
    if (_front != 0) _front->transform(inv_trafo);
    if (_back != 0) _back->transform(inv_trafo);
}

BSP *BSPLeaf::combine(BSPTree &t, const std::vector<TriangleFace> &faces, 
		      BSPTreeBuilder::SetOperation op)
{
    // Avoid unnecessary set operations if possible
    if ((op == BSPTreeBuilder::Union && _inside == false) ||
	(op == BSPTreeBuilder::Intersection && _inside == true))
	return this;

    // Build a new BSP tree for the remaining faces
    return BSPTreeBuilder::buildTree(t, faces, true);
}

void BSPLeaf::partitionFaceList(BSPTree &t,
				const std::vector<TriangleFace> &faces,
				std::vector<TriangleFace> &on_list,
				std::vector<TriangleFace> &in_list,
				std::vector<TriangleFace> &out_list) const
{
    // Is this cell inside?
    if (_inside == true) {
	// Yes, it is, add faces to the inside list
	in_list.insert(in_list.end(), faces.begin(), faces.end());
    }
    // Outside, then? 
    else {
	// Oh yes, add faces to outside list
	out_list.insert(out_list.end(), faces.begin(), faces.end());
    }
}
