/* ------------------------------------------------------------------------
 * $Id: grammar.yy,v 1.4 2001/06/29 15:00:43 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-12 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

%{
/* -- System Includes */
#include <string>

/* -- Local Includes */
#include "Defines.hh"
#include "AST.hh"
#include "RuleSet.hh"

using namespace Zorn;

/* -- Function Prototypes */
void yyerror(const char *s);
int yylex();

/* -- Globals */
static void (*zorn_error_output)(const char *) = 0;
static const char *zorn_filename = 0;
int zorn_current_line = 0;

%}

%union {
	double double_val;	/* Double-precision floating point values */
	int int_val;		/* Integer (and boolean) value */
	char *string_val;	/* Character string value */
	Zorn::AST *node;	/* Abstract syntax tree node */
}

%token IF THEN ELSE INIT
%token <double_val>	DOUBLE_LIT
%token <int_val>	INT_LIT BOOL_LIT
%token <string_val>	STR_LIT IDENTIFIER

%type <node>		rule action_list action expr param_list fun_call
%type <node>		rule_list init_stmt

/* Operator precedence and associativity */
%nonassoc ':'
%nonassoc IF THEN ELSE
%right '='
%nonassoc EQ NEQ LEQ GEQ '<' '>'
%left '-' '+'
%left '*' '/'
%left NEG		/* Negation (unary minus) */
%left OR
%left AND
%left NOT EXISTS

/* Grammar begins here */
%%

program	: /* empty */
	| program stmt
	;

stmt	: rule				{ RuleSet::instance()->addRule($1); }
	| init_stmt 			{ RuleSet::instance()->addInit($1); }
	;

init_stmt
	: INIT '(' action_list ')'	{ $$ = $3; }
	| error ')'			{ $$ = 0; }
	;
	

rule	: expr ':' action_list ';'	{ $$ = new ASTRule($1, $3); }
	| expr ':' rule_list ';'	{ $$ = new ASTRule($1, $3); }
	| error ';'			{ $$ = 0; }
	;

rule_list
	: rule				{ $$ = $1; }
	| rule rule_list		{
					  /* Need to check for errors */
					  if ($1 == 0)
					      $$ = $2;
					  else {
					    /* All is well, no problem */
					    $1->setNextSibling($2);
					    $$ = $1;
					  }
					}
	;

action_list
	: action			{ $$ = $1; }
	| action ',' action_list	{
					  $1->setNextSibling($3);
					  $$ = $1;
					}
	;

action	: IDENTIFIER '=' expr		{
					  $$ = new ASTAssign($1, $3); 
					  free($1);
					}
	| fun_call			{ $$ = new ASTActionCall($1); }
	;

expr	: '(' expr ')'			{ $$ = $2; }
	| expr '+' expr			{ $$ = new ASTBinOp($1, "+", $3); }
	| expr '-' expr			{ $$ = new ASTBinOp($1, "-", $3); }
	| expr '*' expr			{ $$ = new ASTBinOp($1, "*", $3); }
	| expr '/' expr			{ $$ = new ASTBinOp($1, "/", $3); }
	| '-' expr %prec NEG		{ $$ = new ASTUnOp("-", $2); }
	| expr EQ expr			{ $$ = new ASTBinOp($1, "==", $3); }
	| expr NEQ expr			{ $$ = new ASTBinOp($1, "!=", $3); }
	| expr LEQ expr			{ $$ = new ASTBinOp($1, "<=", $3); }
	| expr GEQ expr			{ $$ = new ASTBinOp($1, ">=", $3); }
	| expr '>' expr			{ $$ = new ASTBinOp($1, ">", $3); }
	| expr '<' expr			{ $$ = new ASTBinOp($1, "<", $3); }
	| expr AND expr 		{ $$ = new ASTAnd($1, "&&", $3); }
	| expr OR expr 			{ $$ = new ASTBinOp($1, "||", $3); }
	| '!' expr %prec NOT		{ $$ = new ASTUnOp("!", $2); }
	| fun_call			{ $$ = $1; }
	| IF expr THEN expr ELSE expr	{ $$ = new ASTIf($2, $4, $6); }
	| DOUBLE_LIT			{ $$ = new ASTDouble($1); }
	| INT_LIT			{ $$ = new ASTInt($1); }
	| BOOL_LIT 			{ $$ = new ASTBool($1); }
	| STR_LIT			{
					  $$ = new ASTString($1);
					  free($1);
					}
	| IDENTIFIER			{ 
					  $$ = new ASTId($1); 
					  free($1);
					}
	| '?' IDENTIFIER %prec EXISTS	{
					  $$ = new ASTExists($2); 
					  free($2);
					}
	;

fun_call
	: IDENTIFIER '(' param_list ')'	{ 
					  $$ = new ASTCall($1, $3); 
					  free($1);
					}
	;

param_list
	: /* empty */			{ $$ = 0; }
	| expr				{ $$ = new ASTParam($1); }
	| expr ',' param_list		{
					  $$ = new ASTParam($1);
					  $$->setNextSibling($3);
					}
	;

/* End of grammar */
%%

void yyerror(const char *s)
{
	#define BUFFER_LEN 1024
	char buffer[BUFFER_LEN];
	
	/* Build the message buffer (according to GNU coding standards) */
	sprintf(buffer, "%s:%d: %s.", zorn_filename, zorn_current_line, s);

	/* Call the output function to emit the message */
	if (zorn_error_output != 0)
		(*zorn_error_output)(buffer);
}

void zorn_init(const char *filename, void (*error_output)(const char *))
{
	/* Initialize the parser */
	zorn_filename = filename;
	zorn_error_output = error_output;
	zorn_current_line = 1;
}
