// Copyright Nick Thompson, 2017
// Use, modification and distribution are subject to the
// Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt
// or copy at http://www.boost.org/LICENSE_1_0.txt)

#ifndef BOOST_MATH_QUADRATURE_DETAIL_EXP_SINH_DETAIL_HPP
#define BOOST_MATH_QUADRATURE_DETAIL_EXP_SINH_DETAIL_HPP

#include <cmath>
#include <vector>
#include <typeinfo>
#include <boost/math/constants/constants.hpp>
#include <boost/math/special_functions/next.hpp>
#include <boost/math/tools/atomic.hpp>
#include <boost/detail/lightweight_mutex.hpp>

namespace boost{ namespace math{ namespace quadrature { namespace detail{


// Returns the exp-sinh quadrature of a function f over the open interval (0, infinity)

template<class Real, class Policy>
class exp_sinh_detail
{
   static const int initializer_selector =
      !std::numeric_limits<Real>::is_specialized || (std::numeric_limits<Real>::radix != 2) ?
      0 :
      (std::numeric_limits<Real>::digits < 30) && (std::numeric_limits<Real>::max_exponent <= 128) ?
      1 :
      (std::numeric_limits<Real>::digits <= std::numeric_limits<double>::digits) && (std::numeric_limits<Real>::max_exponent <= std::numeric_limits<double>::max_exponent) ?
      2 :
      (std::numeric_limits<Real>::digits <= std::numeric_limits<long double>::digits) && (std::numeric_limits<Real>::max_exponent <= 16384) ?
      3 :
#ifdef BOOST_HAS_FLOAT128
      (std::numeric_limits<Real>::digits <= 113) && (std::numeric_limits<Real>::max_exponent <= 16384) ?
      4 :
#endif
      0;
public:
    exp_sinh_detail(size_t max_refinements);

    template<class F>
    Real integrate(const F& f, Real* error, Real* L1, const char* function, Real tolerance, std::size_t* levels) const;

private:
   const std::vector<Real>& get_abscissa_row(std::size_t n)const
   {
#ifndef BOOST_MATH_NO_ATOMIC_INT
      if (m_committed_refinements.load() < n)
         extend_refinements();
      BOOST_ASSERT(m_committed_refinements.load() >= n);
#else
      if (m_committed_refinements < n)
         extend_refinements();
      BOOST_ASSERT(m_committed_refinements >= n);
#endif
      return m_abscissas[n];
   }
   const std::vector<Real>& get_weight_row(std::size_t n)const
   {
#ifndef BOOST_MATH_NO_ATOMIC_INT
      if (m_committed_refinements.load() < n)
         extend_refinements();
      BOOST_ASSERT(m_committed_refinements.load() >= n);
#else
      if (m_committed_refinements < n)
         extend_refinements();
      BOOST_ASSERT(m_committed_refinements >= n);
#endif
      return m_weights[n];
   }
   void init(const mpl::int_<0>&);
   void init(const mpl::int_<1>&);
   void init(const mpl::int_<2>&);
   void init(const mpl::int_<3>&);
#ifdef BOOST_HAS_FLOAT128
   void init(const mpl::int_<4>&);
#endif

   void extend_refinements()const
   {
#ifndef BOOST_MATH_NO_ATOMIC_INT
      boost::detail::lightweight_mutex::scoped_lock guard(m_mutex);
#endif
      //
      // Check some other thread hasn't got here after we read the atomic variable, but before we got here:
      //
#ifndef BOOST_MATH_NO_ATOMIC_INT
      if (m_committed_refinements.load() >= m_max_refinements)
         return;
#else
      if (m_committed_refinements >= m_max_refinements)
         return;
#endif

      using std::ldexp;
      using std::ceil;
      using std::sinh;
      using std::cosh;
      using std::exp;
      std::size_t row  = ++m_committed_refinements;

      Real h = ldexp(Real(1), -static_cast<int>(row));
      const Real t_max = m_t_min + m_abscissas[0].size() - 1;

      size_t k = (size_t)boost::math::lltrunc(ceil((t_max - m_t_min) / (2 * h)));
      m_abscissas[row].reserve(k);
      m_weights[row].reserve(k);
      Real arg = m_t_min;
      size_t j = 0;
      size_t l = 2;
      while (arg + l*h < t_max)
      {
         arg = m_t_min + (2 * j + 1)*h;
         Real x = exp(constants::half_pi<Real>()*sinh(arg));
         m_abscissas[row].emplace_back(x);
         Real w = cosh(arg)*constants::half_pi<Real>()*x;
         m_weights[row].emplace_back(w);
         ++j;
      }
   }

    Real m_tol, m_t_min;

    mutable std::vector<std::vector<Real>> m_abscissas;
    mutable std::vector<std::vector<Real>> m_weights;
    std::size_t                       m_max_refinements;
#ifndef BOOST_MATH_NO_ATOMIC_INT
    mutable boost::math::detail::atomic_unsigned_type      m_committed_refinements;
    mutable boost::detail::lightweight_mutex m_mutex;
#else
    mutable unsigned                  m_committed_refinements;
#endif
};

template<class Real, class Policy>
exp_sinh_detail<Real, Policy>::exp_sinh_detail(size_t max_refinements) 
   : m_abscissas(max_refinements), m_weights(max_refinements),
   m_max_refinements(max_refinements)
{
   init(mpl::int_<initializer_selector>());
}
template<class Real, class Policy>
template<class F>
Real exp_sinh_detail<Real, Policy>::integrate(const F& f, Real* error, Real* L1, const char* function, Real tolerance, std::size_t* levels) const
{
    using std::abs;
    using std::floor;
    using std::tanh;
    using std::sinh;
    using std::sqrt;
    using boost::math::constants::half;
    using boost::math::constants::half_pi;

    Real y_max = f(tools::max_value<Real>());
    if(abs(y_max) > tools::epsilon<Real>() || !(boost::math::isfinite)(y_max))
    {
       return policies::raise_domain_error(function, "The function you are trying to integrate does not go to zero at infinity, and instead evaluates to %1%", y_max, Policy());
    }

    //std::cout << std::setprecision(5*std::numeric_limits<Real>::digits10);

    // Get the party started with two estimates of the integral:
    Real I0 = 0;
    Real L1_I0 = 0;
    for(size_t i = 0; i < m_abscissas[0].size(); ++i)
    {
        Real y = f(m_abscissas[0][i]);
        I0 += y*m_weights[0][i];
        L1_I0 += abs(y)*m_weights[0][i];
    }

    //std::cout << "First estimate : " << I0 << std::endl;
    Real I1 = I0;
    Real L1_I1 = L1_I0;
    for (size_t i = 0; i < m_abscissas[1].size(); ++i)
    {
        Real y = f(m_abscissas[1][i]);
        I1 += y*m_weights[1][i];
        L1_I1 += abs(y)*m_weights[1][i];
    }

    I1 *= half<Real>();
    L1_I1 *= half<Real>();
    Real err = abs(I0 - I1);
    //std::cout << "Second estimate: " << I1 << " Error estimate at level " << 1 << " = " << err << std::endl;

    size_t i = 2;
    for(; i < m_abscissas.size(); ++i)
    {
        I0 = I1;
        L1_I0 = L1_I1;

        I1 = half<Real>()*I0;
        L1_I1 = half<Real>()*L1_I0;
        Real h = (Real) 1/ (Real) (1 << i);
        Real sum = 0;
        Real absum = 0;

        auto abscissas_row = get_abscissa_row(i);
        auto weight_row = get_weight_row(i);

        Real abterm1 = 1;
        Real eps = tools::epsilon<Real>()*L1_I1;
        for(size_t j = 0; j < m_weights[i].size(); ++j)
        {
            Real x = abscissas_row[j];
            Real y = f(x);
            sum += y*weight_row[j];
            Real abterm0 = abs(y)*weight_row[j];
            absum += abterm0;

            // We require two consecutive terms to be < eps in case we hit a zero of f.
            // Numerical experiments indicate that we should start this check at ~30 for floats,
            // ~60 for doubles, and ~100 for long doubles.
            // However, starting the check at x = 10 rather than x = 100 will only save two function evaluations.
            if (x > (Real) 100 && abterm0 < eps && abterm1 < eps)
            {
                break;
            }
            abterm1 = abterm0;
        }

        I1 += sum*h;
        L1_I1 += absum*h;
        err = abs(I0 - I1);
        //std::cout << "Estimate:        " << I1 << " Error estimate at level " << i  << " = " << err << std::endl;
        if (!isfinite(I1))
        {
            return policies::raise_evaluation_error(function, "The exp_sinh quadrature evaluated your function at a singular point and returned %1%. Please ensure your function evaluates to a finite number over its entire domain.", I1, Policy());
        }
        if (err <= tolerance*L1_I1)
        {
            break;
        }
    }

    if (error)
    {
        *error = err;
    }

    if(L1)
    {
        *L1 = L1_I1;
    }

    if (levels)
    {
       *levels = i;
    }

    return I1;
}


template<class Real, class Policy>
void exp_sinh_detail<Real, Policy>::init(const mpl::int_<0>&)
{
   using std::exp;
   using std::log;
   using std::sqrt;
   using std::cosh;
   using std::sinh;
   using std::asinh;
   using std::ceil;
   using boost::math::constants::two_div_pi;
   using boost::math::constants::half_pi;
   using boost::math::constants::half;

   m_committed_refinements = 4;
   // m_t_min is chosen such that x = exp(pi/2 sinh(-t_max)) = very small, but not too small.
   // This is a compromise; we wish to approach a singularity at zero without hitting it through roundoff error.
   // If we choose the small number as epsilon, we do not get close enough to the singularity to achieve high accuracy.
   // If we choose the small number as the min(), then we round off to zero and hit the singularity.
   // The logarithmic average of the min() and the epsilon() has been found to be a reasonable compromise, which achieves high accuracy
   // but does not evaluate the function at the singularity.
   Real tmp = (boost::math::tools::log_min_value<Real>() + log(boost::math::tools::epsilon<Real>()))*half<Real>();
   m_t_min = asinh(two_div_pi<Real>()*tmp);

   // t_max is chosen to make g'(t_max) ~ sqrt(max) (g' grows faster than g).
   // This will allow some flexibility on the users part; they can at least square a number function without overflow.
   // But there is no unique choice; the further out we can evaluate the function, the better we can do on slowly decaying integrands.
   const Real t_max = log(2 * two_div_pi<Real>()*log(2 * two_div_pi<Real>()*sqrt(tools::max_value<Real>())));

   for (size_t i = 0; i <= 4; ++i)
   {
      Real h = (Real)1 / (Real)(1 << i);
      size_t k = (size_t)boost::math::lltrunc(ceil((t_max - m_t_min) / (2 * h)));
      m_abscissas[i].reserve(k);
      m_weights[i].reserve(k);
      Real arg = m_t_min;
      size_t j = 0;
      size_t l = 2;
      if (i == 0)
      {
         l = 1;
      }
      while (arg + l*h < t_max)
      {
         if (i != 0)
         {
            arg = m_t_min + (2 * j + 1)*h;
         }
         else
         {
            arg = m_t_min + j*h;
         }
         Real x = exp(half_pi<Real>()*sinh(arg));
         m_abscissas[i].emplace_back(x);
         Real w = cosh(arg)*half_pi<Real>()*x;
         m_weights[i].emplace_back(w);
         ++j;
      }
   }
/*
   std::cout << std::setprecision(18) << m_t_min << std::endl;
   for (unsigned i = 0; i < m_abscissas[0].size(); ++i)
      std::cout << m_abscissas[0][i] << ", ";
   std::cout << std::endl;
   */
}

template<class Real, class Policy>
void exp_sinh_detail<Real, Policy>::init(const mpl::int_<1>&)
{
   m_abscissas = {
      { 3.47876573e-23f, 5.62503650e-09f, 9.95706124e-04f, 9.67438487e-02f, 7.43599217e-01f, 4.14293205e+00f, 1.08086768e+02f, 4.56291316e+05f, 2.70123007e+15f, },
      { 2.41870864e-14f, 1.02534662e-05f, 1.65637566e-02f, 3.11290799e-01f, 1.64691269e+00f, 1.49800773e+01f, 2.57724301e+03f, 2.24833766e+09f, },
      { 3.24983286e-18f, 2.51095186e-11f, 3.82035773e-07f, 1.33717837e-04f, 4.80260650e-03f, 4.41526928e-02f, 1.83045938e-01f, 4.91960276e-01f, 1.10322609e+00f, 2.53681744e+00f, 7.39791792e+00f, 3.59560256e+01f, 4.36061333e+02f, 2.49501460e+04f, 1.89216933e+07f, 1.03348694e+12f, },
      { 1.51941172e-20f, 3.70201714e-16f, 9.67598102e-13f, 4.44773051e-10f, 5.28493928e-08f, 2.19158236e-06f, 4.00799258e-05f, 3.88011529e-04f, 2.29325538e-03f, 9.25182629e-03f, 2.78117501e-02f, 6.67553298e-02f, 1.35173168e-01f, 2.41374946e-01f, 3.94194704e-01f, 6.07196731e-01f, 9.06432514e-01f, 1.34481045e+00f, 2.03268444e+00f, 3.21243032e+00f, 5.46310949e+00f, 1.03365745e+01f, 2.26486752e+01f, 6.03727778e+01f, 2.08220266e+02f, 1.00431239e+03f, 7.47843388e+03f, 9.75279951e+04f, 2.61755592e+06f, 1.77776624e+08f, 3.98255346e+10f, 4.13443763e+13f, 3.07708133e+17f, },
      { 7.99409438e-22f, 2.41624595e-19f, 3.73461321e-17f, 3.19397902e-15f, 1.62042378e-13f, 5.18579386e-12f, 1.10520072e-10f, 1.64548212e-09f, 1.78534009e-08f, 1.46529196e-07f, 9.40168786e-07f, 4.85507733e-06f, 2.07038029e-05f, 7.45799409e-05f, 2.31536599e-04f, 6.30580368e-04f, 1.53035449e-03f, 3.35582040e-03f, 6.73124842e-03f, 1.24856832e-02f, 2.16245309e-02f, 3.52720523e-02f, 5.45995171e-02f, 8.07587788e-02f, 1.14840025e-01f, 1.57867103e-01f, 2.10837078e-01f, 2.74805391e-01f, 3.51015955e-01f, 4.41077540e-01f, 5.47194016e-01f, 6.72466825e-01f, 8.21304567e-01f, 1.00000000e+00f, 1.21757511e+00f, 1.48706221e+00f, 1.82750536e+00f, 2.26717507e+00f, 2.84887335e+00f, 3.63893880e+00f, 4.74299876e+00f, 6.33444194e+00f, 8.70776542e+00f, 1.23825548e+01f, 1.83151803e+01f, 2.83510579e+01f, 4.62437776e+01f, 8.00917327e+01f, 1.48560852e+02f, 2.97989725e+02f, 6.53443372e+02f, 1.58584068e+03f, 4.31897162e+03f, 1.34084311e+04f, 4.83003053e+04f, 2.05969943e+05f, 1.06363880e+06f, 6.82457850e+06f, 5.60117371e+07f, 6.07724622e+08f, 9.04813016e+09f, 1.92834507e+11f, 6.17122515e+12f, 3.13089095e+14f, 2.67765347e+16f, 4.13865153e+18f, },
      { 1.70893932e-22f, 3.56621447e-21f, 6.19138882e-20f, 9.04299298e-19f, 1.12287188e-17f, 1.19706303e-16f, 1.10583090e-15f, 8.92931857e-15f, 6.35404710e-14f, 4.01527389e-13f, 2.26955738e-12f, 1.15522811e-11f, 5.32913181e-11f, 2.24130967e-10f, 8.64254491e-10f, 3.07161058e-09f, 1.01117742e-08f, 3.09775637e-08f, 8.87004371e-08f, 2.38368096e-07f, 6.03520392e-07f, 1.44488635e-06f, 3.28212299e-06f, 7.09655821e-06f, 1.46494407e-05f, 2.89537394e-05f, 5.49357161e-05f, 1.00313252e-04f, 1.76700203e-04f, 3.00920507e-04f, 4.96484845e-04f, 7.95150594e-04f, 1.23845781e-03f, 1.87911525e-03f, 2.78210510e-03f, 4.02538552e-03f, 5.70009588e-03f, 7.91020800e-03f, 1.07716137e-02f, 1.44106884e-02f, 1.89624177e-02f, 2.45682104e-02f, 3.13735515e-02f, 3.95256605e-02f, 4.91713196e-02f, 6.04550279e-02f, 7.35176150e-02f, 8.84954195e-02f, 1.05520113e-01f, 1.24719213e-01f, 1.46217318e-01f, 1.70138063e-01f, 1.96606781e-01f, 2.25753880e-01f, 2.57718900e-01f, 2.92655274e-01f, 3.30735809e-01f, 3.72158929e-01f, 4.17155794e-01f, 4.65998399e-01f, 5.19008863e-01f, 5.76570161e-01f, 6.39138643e-01f, 7.07258781e-01f, 7.81580731e-01f, 8.62881450e-01f, 9.52090320e-01f, 1.05032052e+00f, 1.15890775e+00f, 1.27945836e+00f, 1.41390963e+00f, 1.56460576e+00f, 1.73439430e+00f, 1.92674937e+00f, 2.14593012e+00f, 2.39718593e+00f, 2.68702407e+00f, 3.02356133e+00f, 3.41698950e+00f, 3.88019661e+00f, 4.42960272e+00f, 5.08629455e+00f, 5.87757956e+00f, 6.83913514e+00f, 8.01801085e+00f, 9.47686632e+00f, 1.13000199e+01f, 1.36021823e+01f, 1.65412214e+01f, 2.03370584e+01f, 2.53000199e+01f, 3.18739815e+01f, 4.07030054e+01f, 5.27358913e+01f, 6.93929374e+01f, 9.28366010e+01f, 1.26418926e+02f, 1.75435645e+02f, 2.48423411e+02f, 3.59440052e+02f, 5.32165336e+02f, 8.07455844e+02f, 1.25762341e+03f, 2.01416017e+03f, 3.32313676e+03f, 5.65930306e+03f, 9.96877263e+03f, 1.82030939e+04f, 3.45378531e+04f, 6.82619916e+04f, 1.40913380e+05f, 3.04680844e+05f, 6.92095957e+05f, 1.65694484e+06f, 4.19519229e+06f, 1.12739016e+07f, 3.22814282e+07f, 9.88946136e+07f, 3.25562103e+08f, 1.15706659e+09f, 4.46167708e+09f, 1.87647826e+10f, 8.65629909e+10f, 4.40614549e+11f, 2.49049013e+12f, 1.57380011e+13f, 1.11990629e+14f, 9.04297390e+14f, 8.35377903e+15f, 8.90573552e+16f, 1.10582857e+18f, 1.61514650e+19f, },
      { 7.75845008e-23f, 3.71846701e-22f, 1.69833677e-21f, 7.40284853e-21f, 3.08399399e-20f, 1.22962599e-19f, 4.69855182e-19f, 1.72288020e-18f, 6.07012059e-18f, 2.05742924e-17f, 6.71669437e-17f, 2.11441966e-16f, 6.42566550e-16f, 1.88715605e-15f, 5.36188198e-15f, 1.47533056e-14f, 3.93507835e-14f, 1.01841667e-13f, 2.55981752e-13f, 6.25453236e-13f, 1.48683211e-12f, 3.44173601e-12f, 7.76421789e-12f, 1.70831312e-11f, 3.66877698e-11f, 7.69632540e-11f, 1.57822184e-10f, 3.16577320e-10f, 6.21604166e-10f, 1.19551931e-09f, 2.25364361e-09f, 4.16647469e-09f, 7.55905964e-09f, 1.34658870e-08f, 2.35675936e-08f, 4.05458117e-08f, 6.86052525e-08f, 1.14227960e-07f, 1.87243781e-07f, 3.02323521e-07f, 4.81026747e-07f, 7.54564302e-07f, 1.16746531e-06f, 1.78236867e-06f, 2.68618781e-06f, 3.99792342e-06f, 5.87841837e-06f, 8.54236163e-06f, 1.22728487e-05f, 1.74387947e-05f, 2.45154696e-05f, 3.41083807e-05f, 4.69806683e-05f, 6.40841007e-05f, 8.65936597e-05f, 1.15945600e-04f, 1.53878746e-04f, 2.02478652e-04f, 2.64224143e-04f, 3.42035594e-04f, 4.39324211e-04f, 5.60041454e-04f, 7.08727668e-04f, 8.90558896e-04f, 1.11139085e-03f, 1.37779898e-03f, 1.69711358e-03f, 2.07744903e-03f, 2.52772622e-03f, 3.05768742e-03f, 3.67790298e-03f, 4.39976940e-03f, 5.23549846e-03f, 6.19809738e-03f, 7.30134015e-03f, 8.55973022e-03f, 9.98845520e-03f, 1.16033342e-02f, 1.34207587e-02f, 1.54576276e-02f, 1.77312787e-02f, 2.02594158e-02f, 2.30600348e-02f, 2.61513493e-02f, 2.95517158e-02f, 3.32795626e-02f, 3.73533204e-02f, 4.17913590e-02f, 4.66119283e-02f, 5.18331072e-02f, 5.74727595e-02f, 6.35484986e-02f, 7.00776615e-02f, 7.70772927e-02f, 8.45641386e-02f, 9.25546518e-02f, 1.01065008e-01f, 1.10111132e-01f, 1.19708739e-01f, 1.29873379e-01f, 1.40620505e-01f, 1.51965539e-01f, 1.63923958e-01f, 1.76511391e-01f, 1.89743720e-01f, 2.03637197e-01f, 2.18208574e-01f, 2.33475238e-01f, 2.49455360e-01f, 2.66168055e-01f, 2.83633553e-01f, 3.01873381e-01f, 3.20910560e-01f, 3.40769809e-01f, 3.61477772e-01f, 3.83063247e-01f, 4.05557445e-01f, 4.28994258e-01f, 4.53410546e-01f, 4.78846448e-01f, 5.05345717e-01f, 5.32956079e-01f, 5.61729623e-01f, 5.91723220e-01f, 6.22998983e-01f, 6.55624768e-01f, 6.89674714e-01f, 7.25229845e-01f, 7.62378724e-01f, 8.01218171e-01f, 8.41854062e-01f, 8.84402205e-01f, 9.28989312e-01f, 9.75754080e-01f, 1.02484839e+00f, 1.07643865e+00f, 1.13070727e+00f, 1.18785434e+00f, 1.24809950e+00f, 1.31168403e+00f, 1.37887320e+00f, 1.44995892e+00f, 1.52526270e+00f, 1.60513906e+00f, 1.68997931e+00f, 1.78021589e+00f, 1.87632722e+00f, 1.97884333e+00f, 2.08835213e+00f, 2.20550671e+00f, 2.33103353e+00f, 2.46574193e+00f, 2.61053497e+00f, 2.76642183e+00f, 2.93453226e+00f, 3.11613304e+00f, 3.31264716e+00f, 3.52567596e+00f, 3.75702486e+00f, 4.00873326e+00f, 4.28310945e+00f, 4.58277134e+00f, 4.91069419e+00f, 5.27026666e+00f, 5.66535674e+00f, 6.10038953e+00f, 6.58043928e+00f, 7.11133842e+00f, 7.69980735e+00f, 8.35360902e+00f, 9.08173387e+00f, 9.89462150e+00f, 1.08044272e+01f, 1.18253437e+01f, 1.29739897e+01f, 1.42698826e+01f, 1.57360130e+01f, 1.73995473e+01f, 1.92926887e+01f, 2.14537359e+01f, 2.39283915e+01f, 2.67713817e+01f, 3.00484719e+01f, 3.38389827e+01f, 3.82389447e+01f, 4.33650689e+01f, 4.93597649e+01f, 5.63975118e+01f, 6.46929803e+01f, 7.45114359e+01f, 8.61821250e+01f, 1.00115581e+02f, 1.16826112e+02f, 1.36961158e+02f, 1.61339834e+02f, 1.91003781e+02f, 2.27284639e+02f, 2.71894067e+02f, 3.27044548e+02f, 3.95612465e+02f, 4.81359585e+02f, 5.89235756e+02f, 7.25795284e+02f, 8.99773468e+02f, 1.12289036e+03f, 1.41097920e+03f, 1.78558211e+03f, 2.27622329e+03f, 2.92367233e+03f, 3.78466551e+03f, 4.93879227e+03f, 6.49862329e+03f, 8.62473434e+03f, 1.15481896e+04f, 1.56044945e+04f, 2.12853507e+04f, 2.93183077e+04f, 4.07905708e+04f, 5.73434125e+04f, 8.14806753e+04f, 1.17063646e+05f, 1.70113785e+05f, 2.50129854e+05f, 3.72274789e+05f, 5.61051155e+05f, 8.56556497e+05f, 1.32526810e+06f, 2.07888648e+06f, 3.30771485e+06f, 5.34063130e+06f, 8.75442405e+06f, 1.45761434e+07f, 2.46634599e+07f, 4.24311457e+07f, 7.42617251e+07f, 1.32291588e+08f, 2.40011058e+08f, 4.43725882e+08f, 8.36456588e+08f, 1.60874083e+09f, 3.15878598e+09f, 6.33624483e+09f, 1.29932136e+10f, 2.72570398e+10f, 5.85372779e+10f, 1.28795973e+11f, 2.90551047e+11f, 6.72570892e+11f, 1.59884056e+12f, 3.90652847e+12f, 9.81916374e+12f, 2.54124546e+13f, 6.77814197e+13f, 1.86501681e+14f, 5.29897885e+14f, 1.55625904e+15f, 4.72943011e+15f, 1.48882761e+16f, 4.86043448e+16f, 1.64741373e+17f, 5.80423410e+17f, 2.12831536e+18f, 8.13255421e+18f, },
      { 5.20331508e-23f, 1.15324162e-22f, 2.52466875e-22f, 5.46028730e-22f, 1.16690465e-21f, 2.46458927e-21f, 5.14543768e-21f, 1.06205431e-20f, 2.16767715e-20f, 4.37564009e-20f, 8.73699691e-20f, 1.72595588e-19f, 3.37377643e-19f, 6.52669145e-19f, 1.24976973e-18f, 2.36916845e-18f, 4.44691383e-18f, 8.26580373e-18f, 1.52174118e-17f, 2.77517606e-17f, 5.01415830e-17f, 8.97689232e-17f, 1.59270821e-16f, 2.80084735e-16f, 4.88253693e-16f, 8.43846463e-16f, 1.44610939e-15f, 2.45762595e-15f, 4.14251017e-15f, 6.92627770e-15f, 1.14889208e-14f, 1.89084205e-14f, 3.08802476e-14f, 5.00504297e-14f, 8.05169965e-14f, 1.28579121e-13f, 2.03847833e-13f, 3.20880532e-13f, 5.01568631e-13f, 7.78600100e-13f, 1.20044498e-12f, 1.83848331e-12f, 2.79712543e-12f, 4.22808302e-12f, 6.35035779e-12f, 9.47805307e-12f, 1.40588174e-11f, 2.07266430e-11f, 3.03739182e-11f, 4.42491437e-11f, 6.40886341e-11f, 9.22929507e-11f, 1.32161843e-10f, 1.88205259e-10f, 2.66552657e-10f, 3.75488615e-10f, 5.26149742e-10f, 7.33426418e-10f, 1.01712318e-09f, 1.40344387e-09f, 1.92688222e-09f, 2.63261606e-09f, 3.57952343e-09f, 4.84396276e-09f, 6.52448685e-09f, 8.74769197e-09f, 1.16754399e-08f, 1.55137320e-08f, 2.05235608e-08f, 2.70341184e-08f, 3.54587968e-08f, 4.63144836e-08f, 6.02447248e-08f, 7.80474059e-08f, 1.00707687e-07f, 1.29437018e-07f, 1.65719157e-07f, 2.11364220e-07f, 2.68571894e-07f, 3.40005066e-07f, 4.28875221e-07f, 5.39041105e-07f, 6.75122241e-07f, 8.42629031e-07f, 1.04811127e-06f, 1.29932703e-06f, 1.60543396e-06f, 1.97720518e-06f, 2.42727196e-06f, 2.97039558e-06f, 3.62377065e-06f, 4.40736236e-06f, 5.34428013e-06f, 6.46118994e-06f, 7.78876789e-06f, 9.36219733e-06f, 1.12217116e-05f, 1.34131848e-05f, 1.59887725e-05f, 1.90076038e-05f, 2.25365270e-05f, 2.66509096e-05f, 3.14354940e-05f, 3.69853096e-05f, 4.34066412e-05f, 5.08180543e-05f, 5.93514765e-05f, 6.91533342e-05f, 8.03857429e-05f, 9.32277499e-05f, 1.07876627e-04f, 1.24549208e-04f, 1.43483273e-04f, 1.64938971e-04f, 1.89200275e-04f, 2.16576471e-04f, 2.47403671e-04f, 2.82046341e-04f, 3.20898851e-04f, 3.64387021e-04f, 4.12969671e-04f, 4.67140163e-04f, 5.27427922e-04f, 5.94399942e-04f, 6.68662248e-04f, 7.50861330e-04f, 8.41685517e-04f, 9.41866302e-04f, 1.05217960e-03f, 1.17344692e-03f, 1.30653650e-03f, 1.45236427e-03f, 1.61189482e-03f, 1.78614219e-03f, 1.97617055e-03f, 2.18309485e-03f, 2.40808123e-03f, 2.65234740e-03f, 2.91716284e-03f, 3.20384886e-03f, 3.51377855e-03f, 3.84837661e-03f, 4.20911898e-03f, 4.59753235e-03f, 5.01519359e-03f, 5.46372894e-03f, 5.94481312e-03f, 6.46016832e-03f, 7.01156301e-03f, 7.60081065e-03f, 8.22976829e-03f, 8.90033499e-03f, 9.61445021e-03f, 1.03740920e-02f, 1.11812753e-02f, 1.20380497e-02f, 1.29464978e-02f, 1.39087327e-02f, 1.49268962e-02f, 1.60031562e-02f, 1.71397050e-02f, 1.83387564e-02f, 1.96025436e-02f, 2.09333170e-02f, 2.23333419e-02f, 2.38048956e-02f, 2.53502659e-02f, 2.69717481e-02f, 2.86716433e-02f, 3.04522558e-02f, 3.23158911e-02f, 3.42648538e-02f, 3.63014456e-02f, 3.84279634e-02f, 4.06466974e-02f, 4.29599296e-02f, 4.53699317e-02f, 4.78789641e-02f, 5.04892744e-02f, 5.32030959e-02f, 5.60226468e-02f, 5.89501290e-02f, 6.19877276e-02f, 6.51376099e-02f, 6.84019251e-02f, 7.17828036e-02f, 7.52823576e-02f, 7.89026802e-02f, 8.26458461e-02f, 8.65139116e-02f, 9.05089155e-02f, 9.46328794e-02f, 9.88878087e-02f, 1.03275694e-01f, 1.07798510e-01f, 1.12458223e-01f, 1.17256783e-01f, 1.22196135e-01f, 1.27278214e-01f, 1.32504950e-01f, 1.37878272e-01f, 1.43400107e-01f, 1.49072382e-01f, 1.54897032e-01f, 1.60875997e-01f, 1.67011231e-01f, 1.73304700e-01f, 1.79758387e-01f, 1.86374297e-01f, 1.93154462e-01f, 2.00100939e-01f, 2.07215821e-01f, 2.14501238e-01f, 2.21959362e-01f, 2.29592410e-01f, 2.37402653e-01f, 2.45392415e-01f, 2.53564085e-01f, 2.61920117e-01f, 2.70463037e-01f, 2.79195450e-01f, 2.88120044e-01f, 2.97239599e-01f, 3.06556989e-01f, 3.16075193e-01f, 3.25797297e-01f, 3.35726506e-01f, 3.45866147e-01f, 3.56219679e-01f, 3.66790698e-01f, 3.77582948e-01f, 3.88600328e-01f, 3.99846898e-01f, 4.11326892e-01f, 4.23044723e-01f, 4.35004995e-01f, 4.47212512e-01f, 4.59672288e-01f, 4.72389556e-01f, 4.85369781e-01f, 4.98618671e-01f, 5.12142186e-01f, 5.25946554e-01f, 5.40038281e-01f, 5.54424165e-01f, 5.69111309e-01f, 5.84107138e-01f, 5.99419409e-01f, 6.15056232e-01f, 6.31026081e-01f, 6.47337815e-01f, 6.64000696e-01f, 6.81024405e-01f, 6.98419060e-01f, 7.16195243e-01f, 7.34364016e-01f, 7.52936944e-01f, 7.71926120e-01f, 7.91344191e-01f, 8.11204381e-01f, 8.31520518e-01f, 8.52307069e-01f, 8.73579162e-01f, 8.95352625e-01f, 9.17644013e-01f, 9.40470650e-01f, 9.63850664e-01f, 9.87803022e-01f, 1.01234758e+00f, 1.03750512e+00f, 1.06329740e+00f, 1.08974721e+00f, 1.11687839e+00f, 1.14471595e+00f, 1.17328606e+00f, 1.20261614e+00f, 1.23273496e+00f, 1.26367264e+00f, 1.29546076e+00f, 1.32813247e+00f, 1.36172249e+00f, 1.39626730e+00f, 1.43180514e+00f, 1.46837616e+00f, 1.50602252e+00f, 1.54478848e+00f, 1.58472055e+00f, 1.62586760e+00f, 1.66828098e+00f, 1.71201469e+00f, 1.75712551e+00f, 1.80367319e+00f, 1.85172058e+00f, 1.90133388e+00f, 1.95258276e+00f, 2.00554062e+00f, 2.06028484e+00f, 2.11689693e+00f, 2.17546288e+00f, 2.23607339e+00f, 2.29882418e+00f, 2.36381627e+00f, 2.43115639e+00f, 2.50095725e+00f, 2.57333803e+00f, 2.64842468e+00f, 2.72635049e+00f, 2.80725648e+00f, 2.89129193e+00f, 2.97861498e+00f, 3.06939317e+00f, 3.16380413e+00f, 3.26203621e+00f, 3.36428929e+00f, 3.47077553e+00f, 3.58172026e+00f, 3.69736291e+00f, 3.81795798e+00f, 3.94377618e+00f, 4.07510558e+00f, 4.21225285e+00f, 4.35554468e+00f, 4.50532923e+00f, 4.66197775e+00f, 4.82588634e+00f, 4.99747780e+00f, 5.17720373e+00f, 5.36554672e+00f, 5.56302277e+00f, 5.77018396e+00f, 5.98762126e+00f, 6.21596768e+00f, 6.45590164e+00f, 6.70815069e+00f, 6.97349551e+00f, 7.25277437e+00f, 7.54688785e+00f, 7.85680417e+00f, 8.18356491e+00f, 8.52829128e+00f, 8.89219104e+00f, 9.27656603e+00f, 9.68282047e+00f, 1.01124700e+01f, 1.05671518e+01f, 1.10486353e+01f, 1.15588347e+01f, 1.20998217e+01f, 1.26738407e+01f, 1.32833247e+01f, 1.39309131e+01f, 1.46194716e+01f, 1.53521138e+01f, 1.61322255e+01f, 1.69634913e+01f, 1.78499242e+01f, 1.87958987e+01f, 1.98061868e+01f, 2.08859991e+01f, 2.20410294e+01f, 2.32775056e+01f, 2.46022448e+01f, 2.60227166e+01f, 2.75471124e+01f, 2.91844234e+01f, 3.09445281e+01f, 3.28382897e+01f, 3.48776660e+01f, 3.70758319e+01f, 3.94473180e+01f, 4.20081658e+01f, 4.47761023e+01f, 4.77707378e+01f, 5.10137879e+01f, 5.45293247e+01f, 5.83440613e+01f, 6.24876734e+01f, 6.69931639e+01f, 7.18972765e+01f, 7.72409663e+01f, 8.30699343e+01f, 8.94352364e+01f, 9.63939781e+01f, 1.04010108e+02f, 1.12355322e+02f, 1.21510104e+02f, 1.31564914e+02f, 1.42621552e+02f, 1.54794728e+02f, 1.68213867e+02f, 1.83025185e+02f, 1.99394097e+02f, 2.17507985e+02f, 2.37579409e+02f, 2.59849828e+02f, 2.84593917e+02f, 3.12124587e+02f, 3.42798827e+02f, 3.77024517e+02f, 4.15268384e+02f, 4.58065302e+02f, 5.06029199e+02f, 5.59865843e+02f, 6.20387872e+02f, 6.88532497e+02f, 7.65382367e+02f, 8.52190227e+02f, 9.50408087e+02f, 1.06172182e+03f, 1.18809220e+03f, 1.33180384e+03f, 1.49552334e+03f, 1.68236894e+03f, 1.89599367e+03f, 2.14068513e+03f, 2.42148533e+03f, 2.74433485e+03f, 3.11624675e+03f, 3.54551666e+03f, 4.04197722e+03f, 4.61730674e+03f, 5.28540457e+03f, 6.06284853e+03f, 6.96945350e+03f, 8.02895513e+03f, 9.26984864e+03f, 1.07264200e+04f, 1.24400169e+04f, 1.44606187e+04f, 1.68487805e+04f, 1.96780458e+04f, 2.30379493e+04f, 2.70377620e+04f, 3.18111749e+04f, 3.75221715e+04f, 4.43724093e+04f, 5.26105241e+04f, 6.25438881e+04f, 7.45535092e+04f, 8.91129656e+04f, 1.06812532e+05f, 1.28390012e+05f, 1.54770253e+05f, 1.87115940e+05f, 2.26893075e+05f, 2.75955654e+05f, 3.36655497e+05f, 4.11985149e+05f, 5.05764405e+05f, 6.22884544e+05f, 7.69629183e+05f, 9.54097173e+05f, 1.18676186e+06f, 1.48121324e+06f, 1.85514609e+06f, 2.33168052e+06f, 2.94113264e+06f, 3.72339780e+06f, 4.73116974e+06f, 6.03430539e+06f, 7.72576515e+06f, 9.92972861e+06f, 1.28127257e+07f, 1.65989637e+07f, 2.15915179e+07f, 2.82017465e+07f, 3.69902945e+07f, 4.87244884e+07f, 6.44590226e+07f, 8.56498776e+07f, 1.14315868e+08f, 1.53268759e+08f, 2.06442545e+08f, 2.79366798e+08f, 3.79850300e+08f, 5.18973079e+08f, 7.12532948e+08f, 9.83165083e+08f, 1.36346329e+09f, 1.90059962e+09f, 2.66319659e+09f, 3.75160395e+09f, 5.31334782e+09f, 7.56648043e+09f, 1.08350637e+10f, 1.56033907e+10f, 2.25993074e+10f, 3.29229832e+10f, 4.82470799e+10f, 7.11297379e+10f, 1.05506900e+11f, 1.57471442e+11f, 2.36513804e+11f, 3.57509889e+11f, 5.43926613e+11f, 8.33024431e+11f, 1.28435637e+12f, 1.99374510e+12f, 3.11642465e+12f, 4.90561997e+12f, 7.77731247e+12f, 1.24197380e+13f, 1.99798484e+13f, 3.23831600e+13f, 5.28864904e+13f, 8.70403770e+13f, 1.44377694e+14f, 2.41399528e+14f, 4.06896744e+14f, 6.91510621e+14f, 1.18504970e+15f, 2.04811559e+15f, 3.57034809e+15f, 6.27861398e+15f, 1.11397125e+16f, 1.99435267e+16f, 3.60337498e+16f, 6.57141972e+16f, 1.20980371e+17f, 2.24875057e+17f, 4.22089025e+17f, 8.00147402e+17f, 1.53216987e+18f, 2.96403754e+18f, 5.79389087e+18f, 1.14455803e+19f, 2.28537992e+19f, },
   };
   m_weights = {
      { 1.79979618e-21f, 1.07218106e-07f, 7.05786060e-03f, 2.72310168e-01f, 1.18863515e+00f, 8.77655464e+00f, 5.33879432e+02f, 5.98892409e+06f, 9.60751551e+16f, },
      { 7.59287827e-13f, 1.18886775e-04f, 7.27332179e-02f, 6.09156795e-01f, 2.71431234e+00f, 4.68800805e+01f, 2.06437304e+04f, 4.85431236e+10f, },
      { 1.30963564e-16f, 6.14135316e-10f, 5.67743391e-06f, 1.21108690e-03f, 2.67259824e-02f, 1.54234107e-01f, 4.23412860e-01f, 8.47913037e-01f, 1.73632925e+00f, 4.63203354e+00f, 1.88206826e+01f, 1.40643917e+02f, 2.73736946e+03f, 2.55633252e+05f, 3.18438602e+08f, 2.86363931e+13f, },
      { 6.93769555e-19f, 1.31670336e-14f, 2.68107110e-11f, 9.60294960e-09f, 8.89417585e-07f, 2.87650015e-05f, 4.10649371e-04f, 3.10797444e-03f, 1.43958814e-02f, 4.56980985e-02f, 1.08787148e-01f, 2.08910486e-01f, 3.43887471e-01f, 5.11338439e-01f, 7.19769211e-01f, 1.00073403e+00f, 1.42660267e+00f, 2.14966467e+00f, 3.50341221e+00f, 6.28632057e+00f, 1.26369961e+01f, 2.90949180e+01f, 7.91163114e+01f, 2.65103292e+02f, 1.15872311e+03f, 7.11886439e+03f, 6.77324248e+04f, 1.13081650e+06f, 3.88995005e+07f, 3.38857764e+09f, 9.74063570e+11f, 1.29789430e+15f, 1.24001927e+19f, },
      { 3.88541434e-20f, 1.03646493e-17f, 1.41388360e-15f, 1.06725054e-13f, 4.77908002e-12f, 1.34999345e-10f, 2.53970414e-09f, 3.33804787e-08f, 3.19755978e-07f, 2.31724882e-06f, 1.31302324e-05f, 5.98917639e-05f, 2.25650360e-04f, 7.18397083e-04f, 1.97196929e-03f, 4.75106406e-03f, 1.02072514e-02f, 1.98317011e-02f, 3.52844239e-02f, 5.81350403e-02f, 8.95955146e-02f, 1.30335749e-01f, 1.80445384e-01f, 2.39557131e-01f, 3.07102681e-01f, 3.82648608e-01f, 4.66260909e-01f, 5.58867257e-01f, 6.62616429e-01f, 7.81267733e-01f, 9.20677638e-01f, 1.08949034e+00f, 1.30019425e+00f, 1.57079633e+00f, 1.92752387e+00f, 2.40924883e+00f, 3.07485695e+00f, 4.01578082e+00f, 5.37784753e+00f, 7.40045071e+00f, 1.04890228e+01f, 1.53538346e+01f, 2.32861156e+01f, 3.67307348e+01f, 6.05296516e+01f, 1.04761593e+02f, 1.91598840e+02f, 3.72918009e+02f, 7.78738763e+02f, 1.76101294e+03f, 4.35837629e+03f, 1.19484066e+04f, 3.67841605e+04f, 1.29157756e+05f, 5.26424122e+05f, 2.54082527e+06f, 1.48545930e+07f, 1.07925566e+08f, 1.00317513e+09f, 1.23283860e+10f, 2.07922173e+11f, 5.01997049e+12f, 1.82006578e+14f, 1.04617001e+16f, 1.01373023e+18f, 1.77530238e+20f, },
      { 8.56958007e-21f, 1.68000718e-19f, 2.74008750e-18f, 3.75978801e-17f, 4.38589881e-16f, 4.39263787e-15f, 3.81223973e-14f, 2.89198757e-13f, 1.93338859e-12f, 1.14783389e-11f, 6.09544349e-11f, 2.91499607e-10f, 1.26339559e-09f, 4.99234840e-09f, 1.80872790e-08f, 6.03998541e-08f, 1.86829770e-07f, 5.37807971e-07f, 1.44704121e-06f, 3.65421571e-06f, 8.69454276e-06f, 1.95621880e-05f, 4.17628758e-05f, 8.48713297e-05f, 1.64680159e-04f, 3.05960283e-04f, 5.45748909e-04f, 9.36950301e-04f, 1.55189915e-03f, 2.48542560e-03f, 3.85690505e-03f, 5.81079770e-03f, 8.51529070e-03f, 1.21588421e-02f, 1.69446644e-02f, 2.30834400e-02f, 3.07847946e-02f, 4.02482241e-02f, 5.16542634e-02f, 6.51566792e-02f, 8.08763802e-02f, 9.88975757e-02f, 1.19266512e-01f, 1.41992893e-01f, 1.67053901e-01f, 1.94400532e-01f, 2.23965873e-01f, 2.55674859e-01f, 2.89455038e-01f, 3.25247905e-01f, 3.63020457e-01f, 4.02776696e-01f, 4.44568958e-01f, 4.88509042e-01f, 5.34779290e-01f, 5.83643845e-01f, 6.35460497e-01f, 6.90693630e-01f, 7.49928915e-01f, 8.13890578e-01f, 8.83462209e-01f, 9.59712352e-01f, 1.04392634e+00f, 1.13764623e+00f, 1.24272128e+00f, 1.36137177e+00f, 1.49627028e+00f, 1.65064527e+00f, 1.82841374e+00f, 2.03435175e+00f, 2.27431458e+00f, 2.55552245e+00f, 2.88693336e+00f, 3.27973254e+00f, 3.74797919e+00f, 4.30946679e+00f, 4.98687594e+00f, 5.80933099e+00f, 6.81451887e+00f, 8.05159726e+00f, 9.58522167e+00f, 1.15011733e+01f, 1.39143002e+01f, 1.69798351e+01f, 2.09096993e+01f, 2.59962450e+01f, 3.26472377e+01f, 4.14380231e+01f, 5.31903193e+01f, 6.90928164e+01f, 9.08883744e+01f, 1.21168895e+02f, 1.63847041e+02f, 2.24923217e+02f, 3.13754154e+02f, 4.45189215e+02f, 6.43236850e+02f, 9.47484116e+02f, 1.42457583e+03f, 2.18920236e+03f, 3.44338342e+03f, 5.55184130e+03f, 9.19045432e+03f, 1.56468513e+04f, 2.74471462e+04f, 4.97037777e+04f, 9.31107740e+04f, 1.80835335e+05f, 3.64968793e+05f, 7.67360053e+05f, 1.68525439e+06f, 3.87686515e+06f, 9.37022570e+06f, 2.38705733e+07f, 6.43128750e+07f, 1.83920179e+08f, 5.60444636e+08f, 1.82722217e+09f, 6.40182180e+09f, 2.42153053e+10f, 9.93804949e+10f, 4.44863150e+11f, 2.18425069e+12f, 1.18337660e+13f, 7.11948688e+13f, 4.78870731e+14f, 3.62710215e+15f, 3.11747341e+16f, 3.06542975e+17f, 3.47854955e+18f, 4.59768243e+19f, 7.14806140e+20f, },
      { 3.95175890e-21f, 1.83575349e-20f, 8.12661397e-20f, 3.43336935e-19f, 1.38634563e-18f, 5.35757029e-18f, 1.98424944e-17f, 7.05221126e-17f, 2.40827550e-16f, 7.91175869e-16f, 2.50347754e-15f, 7.63871031e-15f, 2.25003103e-14f, 6.40502166e-14f, 1.76389749e-13f, 4.70424252e-13f, 1.21618334e-12f, 3.05082685e-12f, 7.43273471e-12f, 1.76028616e-11f, 4.05602375e-11f, 9.10055013e-11f, 1.98994391e-10f, 4.24390078e-10f, 8.83436580e-10f, 1.79636925e-09f, 3.57059250e-09f, 6.94247187e-09f, 1.32133371e-08f, 2.46332536e-08f, 4.50110843e-08f, 8.06630537e-08f, 1.41856144e-07f, 2.44958654e-07f, 4.15579069e-07f, 6.93056106e-07f, 1.13675616e-06f, 1.83473665e-06f, 2.91544023e-06f, 4.56318858e-06f, 7.03833675e-06f, 1.07030190e-05f, 1.60534529e-05f, 2.37597559e-05f, 3.47141604e-05f, 5.00883685e-05f, 7.14005734e-05f, 1.00592372e-04f, 1.40115414e-04f, 1.93027181e-04f, 2.63094779e-04f, 3.54905080e-04f, 4.73978972e-04f, 6.26886955e-04f, 8.21362793e-04f, 1.06641153e-03f, 1.37240787e-03f, 1.75118071e-03f, 2.21607971e-03f, 2.78201983e-03f, 3.46550010e-03f, 4.28459361e-03f, 5.25890609e-03f, 6.40950150e-03f, 7.75879384e-03f, 9.33040551e-03f, 1.11489935e-02f, 1.32400455e-02f, 1.56296499e-02f, 1.83442433e-02f, 2.14103400e-02f, 2.48542509e-02f, 2.87017958e-02f, 3.29780164e-02f, 3.77068968e-02f, 4.29110964e-02f, 4.86117029e-02f, 5.48280093e-02f, 6.15773214e-02f, 6.88747982e-02f, 7.67333308e-02f, 8.51634602e-02f, 9.41733378e-02f, 1.03768728e-01f, 1.13953051e-01f, 1.24727473e-01f, 1.36091031e-01f, 1.48040798e-01f, 1.60572082e-01f, 1.73678660e-01f, 1.87353038e-01f, 2.01586736e-01f, 2.16370598e-01f, 2.31695113e-01f, 2.47550758e-01f, 2.63928342e-01f, 2.80819365e-01f, 2.98216379e-01f, 3.16113348e-01f, 3.34506011e-01f, 3.53392244e-01f, 3.72772414e-01f, 3.92649735e-01f, 4.13030618e-01f, 4.33925021e-01f, 4.55346789e-01f, 4.77314001e-01f, 4.99849320e-01f, 5.22980337e-01f, 5.46739932e-01f, 5.71166640e-01f, 5.96305036e-01f, 6.22206131e-01f, 6.48927802e-01f, 6.76535247e-01f, 7.05101473e-01f, 7.34707835e-01f, 7.65444619e-01f, 7.97411688e-01f, 8.30719192e-01f, 8.65488366e-01f, 9.01852407e-01f, 9.39957463e-01f, 9.79963735e-01f, 1.02204672e+00f, 1.06639858e+00f, 1.11322974e+00f, 1.16277062e+00f, 1.21527359e+00f, 1.27101525e+00f, 1.33029891e+00f, 1.39345744e+00f, 1.46085648e+00f, 1.53289803e+00f, 1.61002461e+00f, 1.69272386e+00f, 1.78153384e+00f, 1.87704900e+00f, 1.97992701e+00f, 2.09089644e+00f, 2.21076567e+00f, 2.34043290e+00f, 2.48089770e+00f, 2.63327413e+00f, 2.79880590e+00f, 2.97888368e+00f, 3.17506505e+00f, 3.38909744e+00f, 3.62294469e+00f, 3.87881764e+00f, 4.15920968e+00f, 4.46693789e+00f, 4.80519096e+00f, 5.17758497e+00f, 5.58822853e+00f, 6.04179895e+00f, 6.54363157e+00f, 7.09982467e+00f, 7.71736306e+00f, 8.40426388e+00f, 9.16974906e+00f, 1.00244499e+01f, 1.09806502e+01f, 1.20525758e+01f, 1.32567410e+01f, 1.46123627e+01f, 1.61418586e+01f, 1.78714466e+01f, 1.98318690e+01f, 2.20592694e+01f, 2.45962577e+01f, 2.74932084e+01f, 3.08098460e+01f, 3.46171893e+01f, 3.89999428e+01f, 4.40594471e+01f, 4.99173320e+01f, 5.67200545e+01f, 6.46445583e+01f, 7.39053537e+01f, 8.47634121e+01f, 9.75373786e+01f, 1.12617765e+02f, 1.30484989e+02f, 1.51732386e+02f, 1.77095712e+02f, 2.07491096e+02f, 2.44064119e+02f, 2.88253545e+02f, 3.41874461e+02f, 4.07227291e+02f, 4.87241400e+02f, 5.85665251e+02f, 7.07319497e+02f, 8.58435639e+02f, 1.04711167e+03f, 1.28392853e+03f, 1.58278901e+03f, 1.96206607e+03f, 2.44618436e+03f, 3.06781187e+03f, 3.87091688e+03f, 4.91505977e+03f, 6.28145970e+03f, 8.08162997e+03f, 1.04697579e+04f, 1.36605846e+04f, 1.79554230e+04f, 2.37803156e+04f, 3.17424455e+04f, 4.27142204e+04f, 5.79596727e+04f, 7.93261335e+04f, 1.09537503e+05f, 1.52647130e+05f, 2.14743829e+05f, 3.05063335e+05f, 4.37755687e+05f, 6.34724899e+05f, 9.30240305e+05f, 1.37850753e+06f, 2.06623977e+06f, 3.13377596e+06f, 4.81098405e+06f, 7.47905793e+06f, 1.17782423e+07f, 1.87980927e+07f, 3.04180655e+07f, 4.99257437e+07f, 8.31551852e+07f, 1.40614107e+08f, 2.41519712e+08f, 4.21576502e+08f, 7.48209440e+08f, 1.35089892e+09f, 2.48263348e+09f, 4.64662007e+09f, 8.86235204e+09f, 1.72348930e+10f, 3.41967381e+10f, 6.92714904e+10f, 1.43352142e+11f, 3.03269524e+11f, 6.56345865e+11f, 1.45422052e+12f, 3.30099910e+12f, 7.68267630e+12f, 1.83474885e+13f, 4.49980389e+13f, 1.13430702e+14f, 2.94148450e+14f, 7.85402504e+14f, 2.16127995e+15f, 6.13534293e+15f, 1.79847736e+16f, 5.44944507e+16f, 1.70858922e+17f, 5.54922744e+17f, 1.86905990e+18f, 6.53599225e+18f, 2.37582887e+19f, 8.98810682e+19f, 3.54341330e+20f, },
      { 2.67108015e-21f, 5.82833463e-21f, 1.25616316e-20f, 2.67469785e-20f, 5.62745845e-20f, 1.17014394e-19f, 2.40511019e-19f, 4.88739481e-19f, 9.82072303e-19f, 1.95168062e-18f, 3.83661097e-18f, 7.46163208e-18f, 1.43594942e-17f, 2.73485792e-17f, 5.15573612e-17f, 9.62223075e-17f, 1.77810682e-16f, 3.25389618e-16f, 5.89765054e-16f, 1.05888451e-15f, 1.88354538e-15f, 3.31989417e-15f, 5.79902273e-15f, 1.00398818e-14f, 1.72308010e-14f, 2.93186753e-14f, 4.94655967e-14f, 8.27635884e-14f, 1.37343706e-13f, 2.26082511e-13f, 3.69205736e-13f, 5.98228147e-13f, 9.61866975e-13f, 1.53484658e-12f, 2.43090464e-12f, 3.82185577e-12f, 5.96531965e-12f, 9.24474797e-12f, 1.42267754e-11f, 2.17427910e-11f, 3.30041201e-11f, 4.97635091e-11f, 7.45399354e-11f, 1.10929412e-10f, 1.64031748e-10f, 2.41032586e-10f, 3.51991946e-10f, 5.10905560e-10f, 7.37124150e-10f, 1.05723929e-09f, 1.50757352e-09f, 2.13744796e-09f, 3.01344401e-09f, 4.22492806e-09f, 5.89117093e-09f, 8.17046854e-09f, 1.12717587e-08f, 1.54693324e-08f, 2.11213594e-08f, 2.86930859e-08f, 3.87857241e-08f, 5.21722335e-08f, 6.98414017e-08f, 9.30518593e-08f, 1.23397923e-07f, 1.62889442e-07f, 2.14048123e-07f, 2.80023159e-07f, 3.64729321e-07f, 4.73011070e-07f, 6.10836627e-07f, 7.85526363e-07f, 1.00602028e-06f, 1.28318979e-06f, 1.63019938e-06f, 2.06292424e-06f, 2.60043021e-06f, 3.26552286e-06f, 4.08537275e-06f, 5.09222413e-06f, 6.32419483e-06f, 7.82617466e-06f, 9.65083023e-06f, 1.18597236e-05f, 1.45245521e-05f, 1.77285168e-05f, 2.15678251e-05f, 2.61533347e-05f, 3.16123436e-05f, 3.80905295e-05f, 4.57540432e-05f, 5.47917575e-05f, 6.54176707e-05f, 7.78734661e-05f, 9.24312223e-05f, 1.09396271e-04f, 1.29110197e-04f, 1.51953965e-04f, 1.78351176e-04f, 2.08771424e-04f, 2.43733750e-04f, 2.83810168e-04f, 3.29629253e-04f, 3.81879756e-04f, 4.41314233e-04f, 5.08752659e-04f, 5.85085996e-04f, 6.71279692e-04f, 7.68377076e-04f, 8.77502620e-04f, 9.99865030e-04f, 1.13676015e-03f, 1.28957360e-03f, 1.45978322e-03f, 1.64896113e-03f, 1.85877551e-03f, 2.09099200e-03f, 2.34747474e-03f, 2.63018699e-03f, 2.94119122e-03f, 3.28264890e-03f, 3.65681963e-03f, 4.06605991e-03f, 4.51282135e-03f, 4.99964828e-03f, 5.52917497e-03f, 6.10412222e-03f, 6.72729343e-03f, 7.40157020e-03f, 8.12990738e-03f, 8.91532760e-03f, 9.76091537e-03f, 1.06698107e-02f, 1.16452023e-02f, 1.26903202e-02f, 1.38084285e-02f, 1.50028172e-02f, 1.62767940e-02f, 1.76336759e-02f, 1.90767806e-02f, 2.06094173e-02f, 2.22348784e-02f, 2.39564300e-02f, 2.57773028e-02f, 2.77006834e-02f, 2.97297055e-02f, 3.18674406e-02f, 3.41168899e-02f, 3.64809756e-02f, 3.89625331e-02f, 4.15643030e-02f, 4.42889240e-02f, 4.71389254e-02f, 5.01167213e-02f, 5.32246039e-02f, 5.64647382e-02f, 5.98391571e-02f, 6.33497571e-02f, 6.69982939e-02f, 7.07863800e-02f, 7.47154815e-02f, 7.87869165e-02f, 8.30018539e-02f, 8.73613125e-02f, 9.18661613e-02f, 9.65171203e-02f, 1.01314762e-01f, 1.06259513e-01f, 1.11351656e-01f, 1.16591337e-01f, 1.21978563e-01f, 1.27513213e-01f, 1.33195039e-01f, 1.39023671e-01f, 1.44998628e-01f, 1.51119321e-01f, 1.57385061e-01f, 1.63795066e-01f, 1.70348473e-01f, 1.77044340e-01f, 1.83881662e-01f, 1.90859375e-01f, 1.97976367e-01f, 2.05231492e-01f, 2.12623572e-01f, 2.20151415e-01f, 2.27813822e-01f, 2.35609599e-01f, 2.43537565e-01f, 2.51596569e-01f, 2.59785494e-01f, 2.68103274e-01f, 2.76548903e-01f, 2.85121445e-01f, 2.93820047e-01f, 3.02643950e-01f, 3.11592502e-01f, 3.20665165e-01f, 3.29861530e-01f, 3.39181328e-01f, 3.48624439e-01f, 3.58190905e-01f, 3.67880941e-01f, 3.77694943e-01f, 3.87633504e-01f, 3.97697421e-01f, 4.07887708e-01f, 4.18205605e-01f, 4.28652591e-01f, 4.39230391e-01f, 4.49940993e-01f, 4.60786652e-01f, 4.71769905e-01f, 4.82893580e-01f, 4.94160809e-01f, 5.05575036e-01f, 5.17140031e-01f, 5.28859900e-01f, 5.40739096e-01f, 5.52782432e-01f, 5.64995090e-01f, 5.77382639e-01f, 5.89951040e-01f, 6.02706666e-01f, 6.15656310e-01f, 6.28807202e-01f, 6.42167019e-01f, 6.55743908e-01f, 6.69546490e-01f, 6.83583887e-01f, 6.97865729e-01f, 7.12402181e-01f, 7.27203953e-01f, 7.42282322e-01f, 7.57649155e-01f, 7.73316926e-01f, 7.89298740e-01f, 8.05608358e-01f, 8.22260217e-01f, 8.39269463e-01f, 8.56651970e-01f, 8.74424378e-01f, 8.92604116e-01f, 9.11209442e-01f, 9.30259469e-01f, 9.49774208e-01f, 9.69774604e-01f, 9.90282579e-01f, 1.01132107e+00f, 1.03291408e+00f, 1.05508673e+00f, 1.07786529e+00f, 1.10127728e+00f, 1.12535146e+00f, 1.15011796e+00f, 1.17560829e+00f, 1.20185546e+00f, 1.22889400e+00f, 1.25676010e+00f, 1.28549162e+00f, 1.31512826e+00f, 1.34571158e+00f, 1.37728514e+00f, 1.40989460e+00f, 1.44358784e+00f, 1.47841507e+00f, 1.51442894e+00f, 1.55168471e+00f, 1.59024039e+00f, 1.63015687e+00f, 1.67149810e+00f, 1.71433126e+00f, 1.75872698e+00f, 1.80475947e+00f, 1.85250679e+00f, 1.90205105e+00f, 1.95347869e+00f, 2.00688065e+00f, 2.06235275e+00f, 2.11999592e+00f, 2.17991652e+00f, 2.24222670e+00f, 2.30704472e+00f, 2.37449538e+00f, 2.44471039e+00f, 2.51782884e+00f, 2.59399766e+00f, 2.67337209e+00f, 2.75611628e+00f, 2.84240383e+00f, 2.93241843e+00f, 3.02635449e+00f, 3.12441791e+00f, 3.22682682e+00f, 3.33381238e+00f, 3.44561973e+00f, 3.56250887e+00f, 3.68475574e+00f, 3.81265333e+00f, 3.94651282e+00f, 4.08666490e+00f, 4.23346116e+00f, 4.38727553e+00f, 4.54850596e+00f, 4.71757611e+00f, 4.89493722e+00f, 5.08107015e+00f, 5.27648761e+00f, 5.48173646e+00f, 5.69740032e+00f, 5.92410235e+00f, 6.16250823e+00f, 6.41332946e+00f, 6.67732689e+00f, 6.95531455e+00f, 7.24816384e+00f, 7.55680807e+00f, 7.88224735e+00f, 8.22555401e+00f, 8.58787841e+00f, 8.97045530e+00f, 9.37461076e+00f, 9.80176975e+00f, 1.02534643e+01f, 1.07313428e+01f, 1.12371793e+01f, 1.17728848e+01f, 1.23405187e+01f, 1.29423019e+01f, 1.35806306e+01f, 1.42580922e+01f, 1.49774818e+01f, 1.57418213e+01f, 1.65543795e+01f, 1.74186947e+01f, 1.83385994e+01f, 1.93182476e+01f, 2.03621450e+01f, 2.14751816e+01f, 2.26626686e+01f, 2.39303784e+01f, 2.52845893e+01f, 2.67321348e+01f, 2.82804577e+01f, 2.99376708e+01f, 3.17126238e+01f, 3.36149769e+01f, 3.56552840e+01f, 3.78450835e+01f, 4.01970005e+01f, 4.27248599e+01f, 4.54438126e+01f, 4.83704762e+01f, 5.15230921e+01f, 5.49217006e+01f, 5.85883374e+01f, 6.25472527e+01f, 6.68251567e+01f, 7.14514957e+01f, 7.64587609e+01f, 8.18828353e+01f, 8.77633847e+01f, 9.41442967e+01f, 1.01074176e+02f, 1.08606902e+02f, 1.16802259e+02f, 1.25726650e+02f, 1.35453899e+02f, 1.46066166e+02f, 1.57654979e+02f, 1.70322410e+02f, 1.84182406e+02f, 1.99362306e+02f, 2.16004568e+02f, 2.34268740e+02f, 2.54333703e+02f, 2.76400239e+02f, 3.00693971e+02f, 3.27468728e+02f, 3.57010397e+02f, 3.89641362e+02f, 4.25725590e+02f, 4.65674502e+02f, 5.09953726e+02f, 5.59090900e+02f, 6.13684688e+02f, 6.74415211e+02f, 7.42056139e+02f, 8.17488717e+02f, 9.01718069e+02f, 9.95892168e+02f, 1.10132394e+03f, 1.21951707e+03f, 1.35219615e+03f, 1.50134197e+03f, 1.66923291e+03f, 1.85849349e+03f, 2.07215152e+03f, 2.31370536e+03f, 2.58720328e+03f, 2.89733724e+03f, 3.24955383e+03f, 3.65018587e+03f, 4.10660860e+03f, 4.62742547e+03f, 5.22268956e+03f, 5.90416786e+03f, 6.68565726e+03f, 7.58336313e+03f, 8.61635357e+03f, 9.80710572e+03f, 1.11821637e+04f, 1.27729327e+04f, 1.46166396e+04f, 1.67574960e+04f, 1.92481112e+04f, 2.21512104e+04f, 2.55417295e+04f, 2.95093735e+04f, 3.41617487e+04f, 3.96282043e+04f, 4.60645561e+04f, 5.36589049e+04f, 6.26388223e+04f, 7.32802431e+04f, 8.59184957e+04f, 1.00962017e+05f, 1.18909442e+05f, 1.40370957e+05f, 1.66095034e+05f, 1.97001996e+05f, 2.34226253e+05f, 2.79169596e+05f, 3.33568603e+05f, 3.99580125e+05f, 4.79889989e+05f, 5.77851588e+05f, 6.97663062e+05f, 8.44594440e+05f, 1.02527965e+06f, 1.24809298e+06f, 1.52363581e+06f, 1.86536786e+06f, 2.29042802e+06f, 2.82070529e+06f, 3.48424008e+06f, 4.31706343e+06f, 5.36561882e+06f, 6.68996113e+06f, 8.36799594e+06f, 1.05011160e+07f, 1.32217203e+07f, 1.67032788e+07f, 2.11738506e+07f, 2.69343047e+07f, 3.43829654e+07f, 4.40490690e+07f, 5.66383460e+07f, 7.30953564e+07f, 9.46890531e+07f, 1.23130681e+08f, 1.60736861e+08f, 2.10656057e+08f, 2.77184338e+08f, 3.66207397e+08f, 4.85821891e+08f, 6.47212479e+08f, 8.65895044e+08f, 1.16348659e+09f, 1.57023596e+09f, 2.12865840e+09f, 2.89877917e+09f, 3.96573294e+09f, 5.45082863e+09f, 7.52773593e+09f, 1.04462776e+10f, 1.45675716e+10f, 2.04161928e+10f, 2.87579864e+10f, 4.07167363e+10f, 5.79499965e+10f, 8.29154750e+10f, 1.19276754e+11f, 1.72524570e+11f, 2.50933409e+11f, 3.67042596e+11f, 5.39962441e+11f, 7.98985690e+11f, 1.18927611e+12f, 1.78088199e+12f, 2.68310388e+12f, 4.06753710e+12f, 6.20525592e+12f, 9.52719664e+12f, 1.47228407e+13f, 2.29025392e+13f, 3.58662837e+13f, 5.65517100e+13f, 8.97859411e+13f, 1.43556057e+14f, 2.31171020e+14f, 3.74966777e+14f, 6.12702071e+14f, 1.00868013e+15f, 1.67323268e+15f, 2.79711270e+15f, 4.71267150e+15f, 8.00353033e+15f, 1.37027503e+16f, 2.36538022e+16f, 4.11734705e+16f, 7.22793757e+16f, 1.27982244e+17f, 2.28603237e+17f, 4.11976277e+17f, 7.49169358e+17f, 1.37488861e+18f, 2.54681529e+18f, 4.76248383e+18f, 8.99167123e+18f, 1.71428840e+19f, 3.30088717e+19f, 6.42020070e+19f, 1.26155602e+20f, 2.50480806e+20f, 5.02601059e+20f, 1.01935525e+21f, },
   };
   m_committed_refinements = static_cast<boost::math::detail::atomic_unsigned_integer_type>(m_weights.size() - 1);
   m_t_min = -4.18750000f;
   if (m_max_refinements >= m_abscissas.size())
   {
      m_abscissas.resize(m_max_refinements + 1);
      m_weights.resize(m_max_refinements + 1);
   }
   else
   {
      m_max_refinements = m_abscissas.size() - 1;
   }
}

template<class Real, class Policy>
void exp_sinh_detail<Real, Policy>::init(const mpl::int_<2>&)
{
   m_abscissas = {
      { 7.241670621354483269e-163, 2.257639733856759198e-60, 1.153241619257215165e-22, 8.747691973876861825e-09, 1.173446923800022477e-03, 1.032756936219208144e-01, 7.719261204224504866e-01, 4.355544675823585545e+00, 1.215101039066652656e+02, 6.228845436711506169e+05, 6.278613977336989392e+15, 9.127414935180233465e+42, 6.091127771174027909e+116, },
      { 4.547459836328942014e-99, 6.678756542928857080e-37, 5.005042973041566360e-14, 1.341318484151208960e-05, 1.833875636365939263e-02, 3.257972971286326131e-01, 1.712014688483495078e+00, 1.613222549264089627e+01, 3.116246745274236447e+03, 3.751603952020919663e+09, 1.132259067258797346e+26, 6.799257464097374238e+70, },
      { 5.314690663257815465e-127, 2.579830034615362946e-77, 3.534801062399966878e-47, 6.733941646704537777e-29, 8.265803726974829043e-18, 4.424914371157762285e-11, 5.390411046738629465e-07, 1.649389713333761449e-04, 5.463728936866216652e-03, 4.787896410534771955e-02, 1.931544616590306846e-01, 5.121421856617965197e-01, 1.144715949265016019e+00, 2.648424684387670480e+00, 7.856804169938798917e+00, 3.944731803343517708e+01, 5.060291993016831194e+02, 3.181117494063683297e+04, 2.820174654949211729e+07, 1.993745099515255184e+12, 1.943469269499068563e+20, 2.858803732300638372e+33, 1.457292199029008637e+55, 8.943565831706355607e+90, 9.016198369791554655e+149, },
      { 8.165631636299519857e-144, 3.658949309353149331e-112, 1.635242513882908826e-87, 2.578381184977746454e-68, 2.305546416275824199e-53, 1.016725540031465162e-41, 1.191823622917539774e-32, 1.379018088205016509e-25, 4.375640088826073184e-20, 8.438464631330991606e-16, 1.838483310261119782e-12, 7.334264181393092650e-10, 7.804740587931068021e-08, 2.970395577741681504e-06, 5.081805431666579484e-05, 4.671401627620431498e-04, 2.652347404231090523e-03, 1.037409202661683856e-02, 3.045225582205323946e-02, 7.178280364982721201e-02, 1.434001065841990688e-01, 2.535640852949085796e-01, 4.113268917643175920e-01, 6.310260805648534613e-01, 9.404706503455087817e-01, 1.396267301972783068e+00, 2.116896928689963277e+00, 3.364289290471596568e+00, 5.770183960005836987e+00, 1.104863531218761752e+01, 2.460224479439805859e+01, 6.699316387888639988e+01, 2.375794092475844708e+02, 1.188092202760116066e+03, 9.269848635975416108e+03, 1.283900116155671304e+05, 3.723397798030112514e+06, 2.793667983952389721e+08, 7.112973790863854188e+10, 8.704037695808749572e+13, 8.001474015782459984e+17, 9.804091819390540578e+22, 3.342777673392873288e+29, 8.160092668471508447e+37, 4.798775331663586528e+48, 3.228614320248853938e+62, 1.836986041572136151e+80, 1.153145986877483804e+103, 2.160972586723647751e+132, },
      { 4.825077401709435655e-153, 3.813781211050297560e-135, 2.377824349780240844e-119, 2.065817295388293122e-105, 4.132105770181358886e-93, 2.963965169989404311e-82, 1.127296662046635391e-72, 3.210346399945695041e-64, 9.282992368222161062e-57, 3.565977853916619677e-50, 2.306962519220473637e-44, 3.098751038516535098e-39, 1.039558064722960891e-34, 1.025256027381235200e-30, 3.432612000569885403e-27, 4.429681881379089961e-24, 2.464589267395236846e-21, 6.526691446363344923e-19, 8.976892324445928684e-17, 6.926277695183452225e-15, 3.208805316815751272e-13, 9.478053068835988899e-12, 1.882052586691155400e-10, 2.632616062773909009e-09, 2.703411837703917665e-08, 2.113642195965330965e-07, 1.299327029813074013e-06, 6.461189935136030673e-06, 2.665090959570723827e-05, 9.322774986189288194e-05, 2.820463407940068813e-04, 7.508613300035051413e-04, 1.786142185986551786e-03, 3.848376610765768211e-03, 7.600810651854199771e-03, 1.390873269178271700e-02, 2.380489559528694982e-02, 3.842796337748997654e-02, 5.895012901671883992e-02, 8.651391160689367948e-02, 1.221961347398101671e-01, 1.670112314557845555e-01, 2.219593619059930701e-01, 2.881200442770917241e-01, 3.667906976948184315e-01, 4.596722879563388211e-01, 5.691113093602836208e-01, 6.984190600916228379e-01, 8.523070690462583711e-01, 1.037505121571600249e+00, 1.263672635742961915e+00, 1.544788480334120896e+00, 1.901333876886441433e+00, 2.363816272813317635e+00, 2.978614980117902904e+00, 3.817957977526709364e+00, 4.997477803461245639e+00, 6.708150685706236545e+00, 9.276566033183386532e+00, 1.328332469239125539e+01, 1.980618680552458639e+01, 3.094452809319702849e+01, 5.101378787119006225e+01, 8.943523638413590523e+01, 1.682138665185088325e+02, 3.427988270281270587e+02, 7.653823671943767281e+02, 1.895993667030670343e+03, 5.285404568827643942e+03, 1.684878049282191210e+04, 6.254388805482299369e+04, 2.759556544455721132e+05, 1.481213238071008345e+06, 9.929728611179601424e+06, 8.564987764771851841e+07, 9.831650826344826952e+08, 1.560339073978569502e+10, 3.575098885016726922e+11, 1.241973798101884982e+13, 6.915106205748805839e+14, 6.571419716645131084e+16, 1.144558033138694099e+19, 3.960915669532823553e+21, 2.984410558028297842e+24, 5.430494850258846715e+27, 2.683747612498502676e+31, 4.114885708325522701e+35, 2.276004816861421600e+40, 5.387544917595833246e+45, 6.623575732955432303e+51, 5.266881304835239338e+58, 3.473234812654772210e+66, 2.517492645985977377e+75, 2.759797646289240629e+85, 6.569603829502412077e+96, 5.116181648220647995e+109, 2.073901892339407423e+124, 7.406462446666255838e+140, },
      { 7.053618140948655098e-158, 2.343354218558056628e-148, 2.062509087689351439e-139, 5.212388628332260488e-131, 4.079380320868843387e-123, 1.061481285006738214e-115, 9.816727607793017691e-109, 3.435400719609722581e-102, 4.825198574681495574e-96, 2.874760995089533358e-90, 7.652499977338879996e-85, 9.556944498127119032e-80, 5.862241023038227937e-75, 1.843934000129616663e-70, 3.096983980846232911e-66, 2.885057452402340330e-62, 1.544904681826443837e-58, 4.917572705671511534e-55, 9.602608566391652866e-52, 1.184882375237471009e-48, 9.499223316355714793e-46, 5.078965858882528461e-43, 1.856080838373584123e-40, 4.744245560917271585e-38, 8.667497891102658240e-36, 1.155086178652063612e-33, 1.144541329818836153e-31, 8.585083084065812874e-30, 4.957702933032408922e-28, 2.239353794616277882e-26, 8.030405447708765492e-25, 2.318459271131684362e-23, 5.460287296679086677e-22, 1.062054307071706375e-20, 1.725955878033239909e-19, 2.369168446274347137e-18, 2.775176063916613602e-17, 2.800847352316621903e-16, 2.457625954357892245e-15, 1.890842052364646528e-14, 1.285791209258834942e-13, 7.786001004707878219e-13, 4.228083024410741194e-12, 2.072664297543567489e-11, 9.229295073519997559e-11, 3.754886152592311575e-10, 1.403443871774813834e-09, 4.843962757371872495e-09, 1.551373196623161433e-08, 4.631448362339623514e-08, 1.294370176865168120e-07, 3.400050664017164356e-07, 8.426290307581447654e-07, 1.977205177561996033e-06, 4.407362363338667830e-06, 9.362197325373404563e-06, 1.900760383449277992e-05, 3.698530963711860636e-05, 6.915333419235766653e-05, 1.245492076251852927e-04, 2.165764713808099093e-04, 3.643870211078977292e-04, 5.943999416122372516e-04, 9.418663022314558591e-04, 1.452364274261880083e-03, 2.183094846035196562e-03, 3.203848855069215278e-03, 4.597532353031862490e-03, 6.460168315117479792e-03, 8.900334989802041559e-03, 1.203804973137064275e-02, 1.600315622064554965e-02, 2.093331703849583304e-02, 2.697174812170771748e-02, 3.426485378063329473e-02, 4.295992956149806344e-02, 5.320309587203163231e-02, 6.513760993479510261e-02, 7.890268021756337834e-02, 9.463287940877026649e-02, 1.124582226719385153e-01, 1.325049504086213973e-01, 1.548970316076579260e-01, 1.797583869192584860e-01, 2.072158210677632145e-01, 2.374026527414815016e-01, 2.704630368855767324e-01, 3.065569893452247137e-01, 3.458661469783558388e-01, 3.886003277325320632e-01, 4.350049951304795319e-01, 4.853697810067132707e-01, 5.400382807495678589e-01, 5.994194092045578293e-01, 6.640006964388650918e-01, 7.343640159321037167e-01, 8.112043806284638130e-01, 8.953526245122194172e-01, 9.878030224123093447e-01, 1.089747207002141516e+00, 1.202616144679226559e+00, 1.328132465995424226e+00, 1.468376159872979355e+00, 1.625867601500928277e+00, 1.803673186618691186e+00, 2.005540624723209206e+00, 2.236073393446881709e+00, 2.500957254018255004e+00, 2.807256477663534857e+00, 3.163804128101147487e+00, 3.581720263742550029e+00, 4.075105576391566303e+00, 4.661977749936137761e+00, 5.365546718714963091e+00, 6.215967676434536043e+00, 7.252774367330402583e+00, 8.528291278204291331e+00, 1.011247001122720391e+01, 1.209982167952718578e+01, 1.461947158782994207e+01, 1.784992423404041042e+01, 2.204102944968352178e+01, 2.754711235628932374e+01, 3.487766600641650640e+01, 4.477610230214251576e+01, 5.834406132739843834e+01, 7.724096630394042216e+01, 1.040101075374387191e+02, 1.426215523101601730e+02, 1.993940974645466479e+02, 2.845939167898235356e+02, 4.152683836292551147e+02, 6.203878718481709769e+02, 9.504080873581791535e+02, 1.495523342124078853e+03, 2.421485328006836634e+03, 4.041977218227396500e+03, 6.969453497454785202e+03, 1.244001690461442846e+04, 2.303794930506892099e+04, 4.437240927040385250e+04, 8.911296561746717657e+04, 1.871159398849787994e+05, 4.119851492265743330e+05, 9.540971729944126398e+05, 2.331680521880789706e+06, 6.034305391011695472e+06, 1.659896369452266448e+07, 4.872448839341613053e+07, 1.532687586549090392e+08, 5.189730792935011722e+08, 1.900599621040508288e+09, 7.566480431232731818e+09, 3.292298322781643849e+10, 1.574714421665075635e+11, 8.330244306239795892e+11, 4.905619969814187571e+12, 3.238316002757222702e+13, 2.413995281454699076e+14, 2.048115587426077343e+15, 1.994352670766892066e+16, 2.248750566422739144e+17, 2.964037541992353401e+18, 4.613233119968213445e+19, 8.569680508342001161e+20, 1.921851711942844799e+22, 5.266829246099861758e+23, 1.786779952992288976e+25, 7.607919705736976491e+26, 4.125721424346450007e+28, 2.894340142292214313e+30, 2.670720269656428272e+32, 3.299248229135205151e+34, 5.560105583582310103e+36, 1.304167266599523020e+39, 4.349382146382717353e+41, 2.109720387774341509e+44, 1.524825352702403324e+47, 1.684941265105084589e+50, 2.925572737558413426e+53, 8.217834961057481281e+56, 3.852117991896536784e+60, 3.114452310394384063e+64, 4.498555465873245751e+68, 1.205113215232800796e+73, 6.230864727145221322e+77, 6.487131248948465269e+82, 1.422810109167834249e+88, 6.897656089181724717e+93, 7.779163462756485195e+99, 2.155213251859555072e+106, 1.554347160152705281e+113, 3.103875072425192272e+120, 1.832673821557018634e+128, 3.431285951865278376e+136, 2.194542081542393530e+145, },
      { 2.363803632659058081e-160, 1.926835442612677686e-155, 1.109114905180506786e-150, 4.556759282087534164e-146, 1.350172241067816232e-141, 2.914359263635229435e-137, 4.627545976953585825e-133, 5.456508344460398758e-129, 4.821828861306345485e-125, 3.221779152402086241e-121, 1.641732102111619421e-117, 6.433569189921227126e-114, 1.954582672700428961e-110, 4.639912078942456372e-107, 8.671928891742699827e-104, 1.285485264305858782e-100, 1.522161801460927566e-97, 1.449767844425295085e-94, 1.118122255504445235e-91, 7.028344777398825069e-89, 3.623454064991238081e-86, 1.541513438874996543e-83, 5.443699502170284982e-81, 1.604913673768949456e-78, 3.972206240977317536e-76, 8.297975554162539562e-74, 1.470748835855054032e-71, 2.222935801472624670e-69, 2.879160361851977720e-67, 3.210837413250902178e-65, 3.097303984958235490e-63, 2.595974479763180595e-61, 1.898656799199089593e-59, 1.216865518398435626e-57, 6.862041810601184397e-56, 3.418134121780773218e-54, 1.509758535747580387e-52, 5.934924977563731784e-51, 2.083865009061241099e-49, 6.558128104492290092e-48, 1.856133016606468181e-46, 4.739964621828176249e-45, 1.095600459825324697e-43, 2.299177139060262518e-42, 4.393663812095906869e-41, 7.667728102142858487e-40, 1.225476279042445010e-38, 1.798526997315960782e-37, 2.430201154741018716e-36, 3.030993518975438712e-35, 3.497966609954172613e-34, 3.744308272796551045e-33, 3.726132797819332658e-32, 3.455018936399215381e-31, 2.991524108706319604e-30, 2.423818520801870809e-29, 1.841452809687011486e-28, 1.314419760826235421e-27, 8.831901010260867670e-27, 5.596660060604091621e-26, 3.350745417080507841e-25, 1.898675566025820409e-24, 1.019982287418197376e-23, 5.203315082978366918e-23, 2.524668746906057148e-22, 1.166904646009344233e-21, 5.145437675264868732e-21, 2.167677145279166596e-20, 8.736996911006110678e-20, 3.373776431076593266e-19, 1.249769727462160008e-18, 4.446913832647864892e-18, 1.521741180930875343e-17, 5.014158301377399707e-17, 1.592708205361177316e-16, 4.882536933653862982e-16, 1.446109387544416586e-15, 4.142510168443201880e-15, 1.148892083132325407e-14, 3.088024760858924214e-14, 8.051699653634442236e-14, 2.038478329249539199e-13, 5.015686309363884049e-13, 1.200444984849900298e-12, 2.797125428309156462e-12, 6.350357793399881333e-12, 1.405881744263466936e-11, 3.037391821635123795e-11, 6.408863411016101449e-11, 1.321618431565916164e-10, 2.665526566207284474e-10, 5.261497418654313068e-10, 1.017123184766088896e-09, 1.926882221639203388e-09, 3.579523428497157488e-09, 6.524486847652635035e-09, 1.167543991262942921e-08, 2.052356080018121741e-08, 3.545879678923676129e-08, 6.024472481556065885e-08, 1.007076869023518125e-07, 1.657191565891799652e-07, 2.685718943404479677e-07, 4.288752213761154116e-07, 6.751222405372943925e-07, 1.048111270324302884e-06, 1.605433960692314060e-06, 2.427271958412371013e-06, 3.623770645356477660e-06, 5.344280132492750309e-06, 7.788767891027678939e-06, 1.122171160022519082e-05, 1.598877254198599908e-05, 2.253652700952153115e-05, 3.143549403208496646e-05, 4.340664122305257288e-05, 5.935147653125578529e-05, 8.038574285450253209e-05, 1.078766266062957565e-04, 1.434832731669987826e-04, 1.892002753957224677e-04, 2.474036705329449166e-04, 3.208988510028906069e-04, 4.129696713145546995e-04, 5.274279220384250390e-04, 6.686622480794640482e-04, 8.416855170641220285e-04, 1.052179598744440400e-03, 1.306536501050643762e-03, 1.611894824798787196e-03, 1.976170547826080496e-03, 2.408081229927640721e-03, 2.917162840577481875e-03, 3.513778549028205519e-03, 4.209118976964403112e-03, 5.015193592567630665e-03, 5.944813116164644191e-03, 7.011563005746090924e-03, 8.229768289624073049e-03, 9.614450207543986041e-03, 1.118127530523730813e-02, 1.294649779580742160e-02, 1.492689615029751590e-02, 1.713970500593860526e-02, 1.960254358145296755e-02, 2.233334186285684056e-02, 2.535026586984720664e-02, 2.867164333232700310e-02, 3.231589109997912964e-02, 3.630144557680610965e-02, 4.064669741956638109e-02, 4.536993166688766414e-02, 5.048927437769432941e-02, 5.602264675683979161e-02, 6.198772763597769678e-02, 6.840192506222012774e-02, 7.528235762939712171e-02, 8.264584606994605986e-02, 9.050891551257121825e-02, 9.888780870447738360e-02, 1.077985103995250356e-01, 1.172567830270636607e-01, 1.272782136821146663e-01, 1.378782724173011162e-01, 1.490723817714478840e-01, 1.608759974398061173e-01, 1.733046999768424060e-01, 1.863742974247175786e-01, 2.001009387790379976e-01, 2.145012382381487190e-01, 2.295924102330349785e-01, 2.453924153016625057e-01, 2.619201169541956490e-01, 2.791954497739298773e-01, 2.972395991130188526e-01, 3.160751928723792943e-01, 3.357265060019327741e-01, 3.562196785212496373e-01, 3.775829480426418792e-01, 3.998468979800887046e-01, 4.230447228497335035e-01, 4.472125123131631074e-01, 4.723895558858634018e-01, 4.986186705332947608e-01, 5.259465537097384485e-01, 5.544241647649479754e-01, 5.841071380560416511e-01, 6.150562315632864018e-01, 6.473378153258308278e-01, 6.810244045956889952e-01, 7.161952432654565143e-01, 7.529369438691556459e-01, 7.913441913000366617e-01, 8.315205183502086596e-01, 8.735791622734589226e-01, 9.176440128265773576e-01, 9.638506636817484398e-01, 1.012347580753402101e+00, 1.063297402882930381e+00, 1.116878392515788506e+00, 1.173286056537125469e+00, 1.232734960362603918e+00, 1.295460761779549539e+00, 1.361722494981910846e+00, 1.431805139837984876e+00, 1.506022516788234345e+00, 1.584720554029819354e+00, 1.668280980969603645e+00, 1.757125510515793421e+00, 1.851720582866847453e+00, 1.952582755329533200e+00, 2.060284836698905963e+00, 2.175462881275503983e+00, 2.298824177179966629e+00, 2.431156386859774759e+00, 2.573338025304717222e+00, 2.726350494395667363e+00, 2.891291931102408784e+00, 3.069393174263124520e+00, 3.262036211067640944e+00, 3.470775532153801919e+00, 3.697362905908153155e+00, 3.943776181224350319e+00, 4.212252847439515687e+00, 4.505329225191826639e+00, 4.825886338442190807e+00, 5.177203733275742875e+00, 5.563022772612923373e+00, 5.987621259260909859e+00, 6.455901637501497370e+00, 6.973495514195020291e+00, 7.546887847708181032e+00, 8.183564906772872855e+00, 8.892191039842283431e+00, 9.682820467523296204e+00, 1.056715177903931837e+01, 1.155883465937652851e+01, 1.267384070151528947e+01, 1.393091310389918289e+01, 1.535211379418177923e+01, 1.696349128797309510e+01, 1.879589868990482198e+01, 2.088599907466058846e+01, 2.327750557804054323e+01, 2.602271658731131093e+01, 2.918442338619305962e+01, 3.283828974258811174e+01, 3.707583192189045823e+01, 4.200816575721451990e+01, 4.777073782243997224e+01, 5.452932468101429049e+01, 6.248767344468634478e+01, 7.189727649240108469e+01, 8.306993427631743111e+01, 9.639397813652482031e+01, 1.123553215857374919e+02, 1.315649140340119335e+02, 1.547947284376312334e+02, 1.830251850988715552e+02, 2.175079854175568113e+02, 2.598498278995140400e+02, 3.121245867818556035e+02, 3.770245173783702458e+02, 4.580653020257635092e+02, 5.598658426219653689e+02, 6.885324967857802403e+02, 8.521902266884453403e+02, 1.061721815114114004e+03, 1.331803836529085656e+03, 1.682368940494210217e+03, 2.140685129891926443e+03, 2.744334847491432747e+03, 3.545516659371773357e+03, 4.617306735234797694e+03, 6.062848530677391758e+03, 8.028955134017154634e+03, 1.072641999277462936e+04, 1.446061873485939411e+04, 1.967804579389513789e+04, 2.703776201447279367e+04, 3.752217148194723312e+04, 5.261052412010591097e+04, 7.455350923854624329e+04, 1.068125318497402759e+05, 1.547702528541975911e+05, 2.268930751685412563e+05, 3.366554971645478061e+05, 5.057644049026088560e+05, 7.696291826884134742e+05, 1.186761864945790800e+06, 1.855146094294667715e+06, 2.941132644236832276e+06, 4.731169740596920355e+06, 7.725765147199987935e+06, 1.281272565991955126e+07, 2.159151785284808339e+07, 3.699029448836502904e+07, 6.445902263727884020e+07, 1.143158678867853615e+08, 2.064425450996979446e+08, 3.798502995329785506e+08, 7.125329484929003007e+08, 1.363463294023391629e+09, 2.663196590686555077e+09, 5.313347815419462975e+09, 1.083506369700027396e+10, 2.259930737910197667e+10, 4.824707991473375387e+10, 1.055069002818752104e+11, 2.365138040635727209e+11, 5.439266129959972285e+11, 1.284356371641026839e+12, 3.116424654245920797e+12, 7.777312465656280419e+12, 1.997984843259596733e+13, 5.288649037339853118e+13, 1.443776937640548342e+14, 4.068967444890414804e+14, 1.185049702391501141e+15, 3.570348091883284324e+15, 1.113971254034978026e+16, 3.603374982229766184e+16, 1.209803708182151942e+17, 4.220890251904870611e+17, 1.532169872312865862e+18, 5.793890867821715890e+18, 2.285379920879842924e+19, 9.415714369232187727e+19, 4.057471211245170887e+20, 1.831405465804324767e+21, 8.671209773404504008e+21, 4.313209261217173994e+22, 2.257498454242656934e+23, 1.245267136898199709e+24, 7.251536499435180219e+24, 4.465573963364524765e+25, 2.913233420596266283e+26, 2.017063171206072979e+27, 1.485014353353330393e+28, 1.164811091759882662e+29, 9.753661264047912784e+29, 8.737124417851167566e+30, 8.390503265508677363e+31, 8.657362701430272680e+32, 9.619472292454361392e+33, 1.153735498483960294e+35, 1.497284701983562213e+36, 2.107816695320163748e+37, 3.227106623185610745e+38, 5.387696372515021985e+39, 9.835496017627849225e+40, 1.968904749086105300e+42, 4.334704147416758275e+43, 1.052717645113369473e+45, 2.829013521120326147e+46, 8.439656297525588822e+47, 2.804279894508234869e+49, 1.041383695988523864e+51, 4.337366591019718310e+52, 2.033523569151676725e+54, 1.077238847489773081e+56, 6.472891251891105455e+57, 4.429404678715878536e+59, 3.466135480828349864e+61, 3.114928656972704276e+63, 3.228947925415990689e+65, 3.878402486902381042e+67, 5.423187597439531197e+69, 8.870779393460412583e+71, 1.705832285076755970e+74, 3.876224350373120420e+76, 1.046359534886878004e+79, 3.373858809560757544e+81, 1.306762499786044015e+84, 6.115300889685679832e+86, 3.478550048884517349e+89, 2.420073578988056289e+92, 2.072453567501123129e+95, 2.199029867204449277e+98, 2.910868575802139983e+101, 4.840699137490951163e+104, 1.018669397739170369e+108, 2.733025017438095928e+111, 9.420797277586029837e+114, 4.205525105722885986e+118, 2.451352708852151939e+122, 1.881577053794165543e+126, 1.918506219134233785e+130, 2.622069659115564900e+134, 4.848463485415763756e+138, 1.224645005481997780e+143, 4.267387286482591954e+147, 2.072505613372582377e+152, },
      { 1.323228129684237783e-161, 4.129002973520822791e-159, 1.178655462569548882e-156, 3.082189008893206231e-154, 7.393542832199414487e-152, 1.629100644355328639e-149, 3.301545529059822941e-147, 6.162031390854241227e-145, 1.060528194470986309e-142, 1.685225757497235089e-140, 2.475534097582263629e-138, 3.365764749507587192e-136, 4.240562683924022383e-134, 4.956794227885611715e-132, 5.381716367914161520e-130, 5.433507172294988849e-128, 5.107031242794315420e-126, 4.473704932098646394e-124, 3.656376947377888629e-122, 2.791170022694259001e-120, 1.992200238692415032e-118, 1.330894359393789718e-116, 8.330356767359890503e-115, 4.890256639970245146e-113, 2.695128935451165447e-111, 1.395829605415630844e-109, 6.799997527188085942e-108, 3.119037767379032293e-106, 1.348260131419216291e-104, 5.497526018943990804e-103, 2.116384670251198533e-101, 7.699148714858061209e-100, 2.649065347250598345e-98, 8.628189263549727753e-97, 2.662520943248368922e-95, 7.790698623582886341e-94, 2.163354866683077281e-92, 5.705576739797220361e-91, 1.430338193028564913e-89, 3.411040781372328747e-88, 7.744268073516449037e-87, 1.675136564303435813e-85, 3.454795810595704816e-84, 6.798573137099477363e-83, 1.277474708033782661e-81, 2.293702139426309483e-80, 3.938021700015175030e-79, 6.469593934876300124e-78, 1.017725266990912471e-76, 1.534019529793324951e-75, 2.216999886838860916e-74, 3.074100747562803362e-73, 4.092295330837549092e-72, 5.233434175636538471e-71, 6.433506079763357418e-70, 7.607042677901362161e-69, 8.656714387163425357e-68, 9.486746058685489974e-67, 1.001756724248288397e-65, 1.019853943834854330e-64, 1.001591106610665630e-63, 9.494277822444263952e-63, 8.691422918891890649e-62, 7.687977047887448276e-61, 6.574408104196605248e-60, 5.438162502918425191e-59, 4.353340831363003212e-58, 3.374338762181243411e-57, 2.533770921173042330e-56, 1.844048925248616738e-55, 1.301410812308480184e-54, 8.910466744374470063e-54, 5.921538384124132331e-53, 3.821356134297705127e-52, 2.395780657353036891e-51, 1.459882187581820236e-50, 8.650105472076777327e-50, 4.985933550797199316e-49, 2.796911903237435916e-48, 1.527570118993503332e-47, 8.126314048196993302e-47, 4.212436363948578182e-46, 2.128604050242564662e-45, 1.048938356323431072e-44, 5.042753142653687842e-44, 2.365999225494165364e-43, 1.083813462091040325e-42, 4.848963367960316169e-42, 2.119612873737657277e-41, 9.055947139022002648e-41, 3.782987192192666650e-40, 1.545649846917574765e-39, 6.178909752126026357e-39, 2.417597558625940386e-38, 9.261305999966332746e-38, 3.474712971194656115e-37, 1.277215890629181345e-36, 4.600938133935473864e-36, 1.624804314773052044e-35, 5.626808103137929972e-35, 1.911442429947086471e-34, 6.371300415498187125e-34, 2.084444531309441237e-33, 6.695356060065574234e-33, 2.112038435637792931e-32, 6.544802906551512393e-32, 1.992864937623987114e-31, 5.964358817764151755e-31, 1.754973231464949500e-30, 5.078231558861773863e-30, 1.445447866528259475e-29, 4.048099759391660786e-29, 1.115752878927994221e-28, 3.027334168442338592e-28, 8.087868498106224788e-28, 2.128106544151858936e-27, 5.516210113930227985e-27, 1.408890921124863906e-26, 3.546520734326774807e-26, 8.800636481096360494e-26, 2.153319509043984465e-25, 5.196136544731926346e-25, 1.236869058422202190e-24, 2.904891674490918873e-24, 6.732707317563258763e-24, 1.540253603361391055e-23, 3.478765727687221019e-23, 7.758450079933031976e-23, 1.708939324269830276e-22, 3.718467010568811152e-22, 7.994094376769029920e-22, 1.698336774318343123e-21, 3.566214469724002275e-21, 7.402848534866351662e-21, 1.519411719755297549e-20, 3.083993994528608740e-20, 6.191388817974459809e-20, 1.229625987010589227e-19, 2.416245949308411084e-19, 4.698551818749419706e-19, 9.042992978848520439e-19, 1.722880198390020817e-18, 3.249832858354112322e-18, 6.070120594586457562e-18, 1.122871881646098441e-17, 2.057429235664205922e-17, 3.734613207742816399e-17, 6.716694369267842075e-17, 1.197063025055043952e-16, 2.114419661115663617e-16, 3.702017138231021853e-16, 6.425665498746337860e-16, 1.105830903726985419e-15, 1.887156051660563224e-15, 3.193979018679125833e-15, 5.361881977473204459e-15, 8.929318568606692809e-15, 1.475330560958586660e-14, 2.418708636765824964e-14, 3.935078350904051302e-14, 6.354047096308654479e-14, 1.018416666466509442e-13, 1.620423782999307693e-13, 2.559817517056126166e-13, 4.015273886294212810e-13, 6.254532358261761291e-13, 9.675981021394182858e-13, 1.486832112534566186e-12, 2.269557377760486879e-12, 3.441736008766365832e-12, 5.185793859860652413e-12, 7.764217889314004663e-12, 1.155228105746548036e-11, 1.708313121464262097e-11, 2.510951856086201897e-11, 3.668776978510952341e-11, 5.329131813941740314e-11, 7.696325397299480856e-11, 1.105200723643722855e-10, 1.578221843796034825e-10, 2.241309672940976766e-10, 3.165773201144956642e-10, 4.447730510871610704e-10, 6.216041661455164049e-10, 8.642544905395987868e-10, 1.195519306516659349e-09, 1.645482121417189823e-09, 2.253643612941620883e-09, 3.071610576496751310e-09, 4.166474690460445927e-09, 5.625036504185181035e-09, 7.559059638953998396e-09, 1.011177417876491092e-08, 1.346588701906267454e-08, 1.785340092957703350e-08, 2.356759364235337519e-08, 3.097756373337616088e-08, 4.054581171302714730e-08, 5.284939280085554173e-08, 6.860525247854168448e-08, 8.870043714076795346e-08, 1.142279599340281637e-07, 1.465291959965373757e-07, 1.872437814520259903e-07, 2.383680961705324062e-07, 3.023235208219232784e-07, 3.820357732606947876e-07, 4.810267467496160044e-07, 6.035203917139166314e-07, 7.545643021775656875e-07, 9.401687861337141280e-07, 1.167465314019272078e-06, 1.444886349199346242e-06, 1.782368666762205796e-06, 2.191582359683820240e-06, 2.686187812137005286e-06, 3.282122985909738110e-06, 3.997923415034129149e-06, 4.855077333283880469e-06, 5.878418366687560187e-06, 7.096558206229387964e-06, 8.542361632206236097e-06, 1.025346618920209381e-05, 1.227284870748632855e-05, 1.464944073127878202e-05, 1.743879474552002742e-05, 2.070380288967650755e-05, 2.451546960924430874e-05, 2.895373942298085844e-05, 3.410838067694928604e-05, 4.007992581615393488e-05, 4.698066833232878622e-05, 5.493571614427227251e-05, 6.408410073746518169e-05, 7.457994093551813828e-05, 8.659365970069775654e-05, 1.003132518682442285e-04, 1.159456002136906496e-04, 1.337178367385581674e-04, 1.538787455425709779e-04, 1.767002031351005554e-04, 2.024786515302844608e-04, 2.315365989746650402e-04, 2.642241426787982083e-04, 3.009205074706080013e-04, 3.420355938637258307e-04, 3.880115286439000550e-04, 4.393242107257947798e-04, 4.964848447258090522e-04, 5.600414544382562271e-04, 6.305803681962314437e-04, 7.087276679481586600e-04, 7.951505937892094439e-04, 8.905588956558126794e-04, 9.957061239230124343e-04, 1.111390850739538593e-03, 1.238457814094548688e-03, 1.377798976832850428e-03, 1.530354493121150144e-03, 1.697113575214988470e-03, 1.879115253782404405e-03, 2.077449025503311209e-03, 2.293255382179820056e-03, 2.527726216158548279e-03, 2.782105097477072741e-03, 3.057687418798497807e-03, 3.355820404885606963e-03, 3.677902984083964409e-03, 4.025385520026097270e-03, 4.399769402530814407e-03, 4.802606497446985045e-03, 5.235498455973840111e-03, 5.700095884774212336e-03, 6.198097378977308725e-03, 6.731248420937948614e-03, 7.301340148374219834e-03, 7.910207996239952125e-03, 8.559730217397303903e-03, 9.251826287833445298e-03, 9.988455202809488913e-03, 1.077161367093554544e-02, 1.160333421372954856e-02, 1.248568317873621646e-02, 1.342075867475355427e-02, 1.441068843813546585e-02, 1.545762763950860648e-02, 1.656375664055830135e-02, 1.773127871080136402e-02, 1.896241771447260382e-02, 2.025941577780677588e-02, 2.162453094709917839e-02, 2.306003484797691421e-02, 2.456821035631025318e-02, 2.615134929114115217e-02, 2.781175013990572523e-02, 2.955171582608151263e-02, 3.137355152920124081e-02, 3.327956256694509270e-02, 3.527205234875621605e-02, 3.735332041012234938e-02, 3.952566053633324126e-02, 4.179135898416228534e-02, 4.415269280953487221e-02, 4.661192830883879903e-02, 4.917131958110712872e-02, 5.183310721786459418e-02, 5.459951712697841302e-02, 5.747275949639657337e-02, 6.045502790319455825e-02, 6.354849857288828754e-02, 6.675532979350985865e-02, 7.007766148848641979e-02, 7.351761495191403887e-02, 7.707729274938041525e-02, 8.075877878706524317e-02, 8.456413855143733669e-02, 8.849541952147546057e-02, 9.255465175496720496e-02, 9.674384865008904765e-02, 1.010650078831426502e-01, 1.055201125230189472e-01, 1.101111323226840632e-01, 1.148400251877307103e-01, 1.197087388218165293e-01, 1.247192125486176994e-01, 1.298733793097628269e-01, 1.351731678380792159e-01, 1.406205050053816316e-01, 1.462173183439629526e-01, 1.519655387409069424e-01, 1.578671033043359383e-01, 1.639239584007306411e-01, 1.701380628625154331e-01, 1.765113913651907042e-01, 1.830459379734134606e-01, 1.897437198555789051e-01, 1.966067811666385690e-01, 2.036371970991047974e-01, 2.108370781024367852e-01, 2.182085742712797843e-01, 2.257538799033364379e-01, 2.334752382279873511e-01, 2.413749463071469410e-01, 2.494553601102403241e-01, 2.577188997656175820e-01, 2.661680549911833443e-01, 2.748053907075124803e-01, 2.836335528372471376e-01, 2.926552742951268547e-01, 3.018733811735925662e-01, 3.112907991295277084e-01, 3.209105599783561596e-01, 3.307358085024083972e-01, 3.407698094811951648e-01, 3.510159549519934555e-01, 3.614777717099542274e-01, 3.721589290577866932e-01, 3.830632468159621812e-01, 3.941947036053136035e-01, 4.055574454148868711e-01, 4.171557944689308074e-01, 4.289942584079951543e-01, 4.410775398002453309e-01, 4.534105460003012245e-01, 4.659983993741692944e-01, 4.788464479101668631e-01, 4.919602762371392109e-01, 5.053457170727489659e-01, 5.190088631261786795e-01, 5.329560794812372669e-01, 5.471940164876055195e-01, 5.617296231898020413e-01, 5.765701613254061793e-01, 5.917232199261468491e-01, 6.071967305576643327e-01, 6.229989832360855492e-01, 6.391386430620321596e-01, 6.556247676153161584e-01, 6.724668251563812272e-01, 6.896747136835329047e-01, 7.072587808981804764e-01, 7.252298451337033758e-01, 7.435992173071710726e-01, 7.623787239570054101e-01, 7.815807314337971290e-01, 8.012181713158943859e-01, 8.213045671260926392e-01, 8.418540624307963733e-01, 8.628814504084197628e-01, 8.844022049795737430e-01, 9.064325135977815717e-01, 9.289893118061069464e-01, 9.520903196722039764e-01, 9.757540802219457353e-01, 1.000000000000000000e+00, 1.024848391894543008e+00, 1.050320520372784475e+00, 1.076438649284173871e+00, 1.103226092399127978e+00, 1.130707266862927052e+00, 1.158907749757141229e+00, 1.187854337974646084e+00, 1.217575111629048984e+00, 1.248099501235266386e+00, 1.279458358915164500e+00, 1.311684033900709062e+00, 1.344810452627081143e+00, 1.378873203729832710e+00, 1.413909628283517352e+00, 1.449958915644490754e+00, 1.487062205287898607e+00, 1.525262695058439148e+00, 1.564605756286502811e+00, 1.605139056255971231e+00, 1.646912688547541313e+00, 1.689979311822189937e+00, 1.734394297653598793e+00, 1.780215888066332921e+00, 1.827505363488657555e+00, 1.876327221885466881e+00, 1.926749369898304239e+00, 1.978843326886336694e+00, 2.032684442834914613e+00, 2.088352131177556992e+00, 2.145930117663470432e+00, 2.205506706496711366e+00, 2.267175065075584681e+00, 2.331033528772661605e+00, 2.397185927317806037e+00, 2.465741934479827004e+00, 2.536817442887937264e+00, 2.610534965993323711e+00, 2.687024069345184956e+00, 2.766421833546071979e+00, 2.848873351459948781e+00, 2.934532262474922666e+00, 3.023561326873131923e+00, 3.116133043635102211e+00, 3.212430315307524598e+00, 3.312647163894682976e+00, 3.416989502097797957e+00, 3.525675964626843197e+00, 3.638938804749809967e+00, 3.757024861729272487e+00, 3.880196605330264341e+00, 4.008733264172298986e+00, 4.142932045347867609e+00, 4.283109453446644399e+00, 4.429602717916437040e+00, 4.582771338567048147e+00, 4.742998759991079249e+00, 4.910694186746867507e+00, 5.086294552335034437e+00, 5.270266656314831820e+00, 5.463109485364516396e+00, 5.665356735708146927e+00, 5.877579556128345480e+00, 6.100389532781943879e+00, 6.334441939256981670e+00, 6.580439277782222274e+00, 6.839135140254664526e+00, 7.111338420820842566e+00, 7.397917915172903763e+00, 7.699807345544508469e+00, 8.018010854664294474e+00, 8.353609016702406728e+00, 8.707765418592385473e+00, 9.081733871099147484e+00, 9.476866315716376006e+00, 9.894621501007146275e+00, 1.033657451045679019e+01, 1.080442723340841910e+01, 1.130001988133777781e+01, 1.182534366375335115e+01, 1.238255475156052427e+01, 1.297398967101161563e+01, 1.360218228861306245e+01, 1.426988256684760289e+01, 1.498007729260327644e+01, 1.573601300513857081e+01, 1.654122137866316500e+01, 1.739954734664685784e+01, 1.831518029132688981e+01, 1.929268866318984532e+01, 2.033705844217826172e+01, 2.145373590584482942e+01, 2.264867523060898736e+01, 2.392839152177298272e+01, 2.530001994731418268e+01, 2.677138174118011529e+01, 2.835105794560498805e+01, 3.004847188085487195e+01, 3.187398146713610639e+01, 3.383898267989664904e+01, 3.595602559959535672e+01, 3.823894472392493310e+01, 4.070300544879345396e+01, 4.336506889917953679e+01, 4.624377760823269784e+01, 4.935976490967979071e+01, 5.273589133292714765e+01, 5.639751178186770847e+01, 6.037277784867852275e+01, 6.469298027622754351e+01, 6.939293735292118365e+01, 7.451143592061966836e+01, 8.009173272176674066e+01, 8.618212503236856949e+01, 9.283660095406551480e+01, 1.001155814082968890e+02, 1.080867678325352448e+02, 1.168261118752949279e+02, 1.264189260858047240e+02, 1.369611577708331715e+02, 1.485608519349011866e+02, 1.613398336385932743e+02, 1.754356453320629017e+02, 1.910037809024609590e+02, 2.082202655019913565e+02, 2.272846389233001078e+02, 2.484234106336023257e+02, 2.718940668983047258e+02, 2.979897251188232016e+02, 3.270445480633676878e+02, 3.594400516741229885e+02, 3.956124653087335485e+02, 4.360613334959077953e+02, 4.813595846269808355e+02, 5.321653357808338203e+02, 5.892357556996862196e+02, 6.534433717775449045e+02, 7.257952842284018994e+02, 8.074558443729566627e+02, 8.997734679339701200e+02, 1.004312392957944252e+03, 1.122890361185594877e+03, 1.257623408459775530e+03, 1.410979202907522234e+03, 1.585840680166573460e+03, 1.785582106601447262e+03, 2.014160171499825914e+03, 2.276223289283167479e+03, 2.577243010007973485e+03, 2.923672325162804598e+03, 3.323136759290736047e+03, 3.784665511113575050e+03, 4.318971620160236406e+03, 4.938792274850918489e+03, 5.659303058273368331e+03, 6.498623292476395004e+03, 7.478433875318933386e+03, 8.624734342286166238e+03, 9.968772633484590145e+03, 1.154818959559393902e+04, 1.340843110702649390e+04, 1.560449453908580443e+04, 1.820309391023133793e+04, 2.128535066649680777e+04, 2.495014598048375046e+04, 2.931830770482188047e+04, 3.453785313845473397e+04, 4.079057084931056631e+04, 4.830030527863206410e+04, 5.734341246586992004e+04, 6.826199159022146453e+04, 8.148067525594191464e+04, 9.752799507478730867e+04, 1.170636462204808295e+05, 1.409133795481584143e+05, 1.701137853111825512e+05, 2.059699426710509940e+05, 2.501298539735692463e+05, 3.046808435555379486e+05, 3.722747886360361411e+05, 4.562913164460176067e+05, 5.610511554921845541e+05, 6.920959565810343691e+05, 8.565564972181198149e+05, 1.063638800552326000e+06, 1.325268101226286025e+06, 1.656944841847240121e+06, 2.078886479301160156e+06, 2.617555920130068069e+06, 3.307714852226224955e+06, 4.195192293202626259e+06, 5.340631300250745566e+06, 6.824578495767020734e+06, 8.754424053248831818e+06, 1.127390159772263517e+07, 1.457614342739689625e+07, 1.892169326841938100e+07, 2.466345986800667442e+07, 3.228142821711217588e+07, 4.243114571539869754e+07, 5.601173714434088431e+07, 7.426172509723072112e+07, 9.889461357830121731e+07, 1.322915875470427182e+08, 1.777766240727455981e+08, 2.400110583389834263e+08, 3.255621033641982742e+08, 4.437258820593761403e+08, 6.077246218504877165e+08, 8.364565879857375417e+08, 1.157066594326456169e+09, 1.608740826498742961e+09, 2.248337657948688269e+09, 3.158785978851336228e+09, 4.461677081363911380e+09, 6.336244831048209270e+09, 9.048130159588677560e+09, 1.299321362309972265e+10, 1.876478261212947929e+10, 2.725703976712888971e+10, 3.982553459064288940e+10, 5.853727794017415415e+10, 8.656299089553103385e+10, 1.287959733041898747e+11, 1.928345065430099883e+11, 2.905510467545806044e+11, 4.406145488098485809e+11, 6.725708918778493152e+11, 1.033486938212196930e+12, 1.598840557086695854e+12, 2.490490134218272825e+12, 3.906528466724583921e+12, 6.171225147961354244e+12, 9.819163736485109137e+12, 1.573800106991564475e+13, 2.541245461530031221e+13, 4.134437628407981776e+13, 6.778141973485971528e+13, 1.119906286595884492e+14, 1.865016806041768967e+14, 3.130890948724989738e+14, 5.298978847669068280e+14, 9.042973899804181753e+14, 1.556259036818991439e+15, 2.701230066368200812e+15, 4.729430105054711279e+15, 8.353779033096586530e+15, 1.488827606293191651e+16, 2.677653466031614956e+16, 4.860434481369499270e+16, 8.905735519300993312e+16, 1.647413728306871552e+17, 3.077081325673016377e+17, 5.804234101329097680e+17, 1.105828570628099614e+18, 2.128315358808074026e+18, 4.138651532085235581e+18, 8.132554212123920035e+18, 1.615146503312570855e+19, 3.242548467260718193e+19, 6.581494581080701321e+19, 1.350831366183090003e+20, 2.804093832520937396e+20, 5.888113683467563837e+20, 1.250923435312468276e+21, 2.689280279098215635e+21, 5.851582825664479700e+21, 1.288917231788944660e+22, 2.874582763768997631e+22, 6.492437335109217869e+22, 1.485286605867082177e+23, 3.442469159113307066e+23, 8.084930196860438207e+23, 1.924506778048094878e+24, 4.643992662491470729e+24, 1.136281452083591334e+25, 2.819664891060694571e+25, 7.097781559991856367e+25, 1.812838850127688486e+26, 4.699012851344539124e+26, 1.236419707162832951e+27, 3.303236261210411286e+27, 8.962558097638891218e+27, 2.470294852986226117e+28, 6.918270960555942883e+28, 1.969189447958411510e+29, 5.698092609453981289e+29, 1.676626156396922084e+30, 5.017901520171556970e+30, 1.527929892279834489e+31, 4.734762318366711949e+31, 1.493572546446777040e+32, 4.797441164681908184e+32, 1.569538296400998732e+33, 5.231651156910242454e+33, 1.777206511525290941e+34, 6.154587299576916134e+34, 2.173469781356604872e+35, 7.829529896526581616e+35, 2.877935554073076917e+36, 1.079761320923458592e+37, 4.136337730951207042e+37, 1.618408489711185844e+38, 6.469770640447824771e+38, 2.643413654859316358e+39, 1.104246728308525703e+40, 4.717842641881260665e+40, 2.062296462389327711e+41, 9.226680005161257219e+41, 4.226544071632731963e+42, 1.983043729707066518e+43, 9.533448690970155039e+43, 4.697914578740208606e+44, 2.373923101980436574e+45, 1.230570211868531753e+46, 6.546344338411695147e+46, 3.575371819335804914e+47, 2.005642453538335506e+48, 1.156055268028903078e+49, 6.849867807870312958e+49, 4.174004815218951121e+50, 2.616872034052857472e+51, 1.688750346837297725e+52, 1.122275666009684101e+53, 7.683968740248677071e+53, 5.422849612654278583e+54, 3.946686701799533415e+55, 2.963543587288132884e+56, 2.297086395798939516e+57, 1.838856414208555761e+58, 1.521049475711243996e+59, 1.300732291175071112e+60, 1.150559591141716740e+61, 1.053265997373725461e+62, 9.984114209879020836e+62, 9.805325615938694719e+63, 9.982463564199115995e+64, 1.054102211457911410e+66, 1.155172684780782463e+67, 1.314571302334116663e+68, 1.554362407685457310e+69, 1.910791206002645077e+70, 2.443616403890711206e+71, 3.252983822318823232e+72, 4.510600140020139737e+73, 6.518821831001902447e+74, 9.825834460774267633e+75, 1.545692063622722856e+77, 2.539346088408163253e+78, 4.359763993811836117e+79, 7.827943627464404744e+80, 1.470896877674301183e+82, 2.894527071420674290e+83, 5.969662541607915492e+84, 1.291277613981057357e+86, 2.931656535626877923e+87, 6.991353547531463135e+88, 1.752671194525972852e+90, 4.622450137056020715e+91, 1.283581933169566226e+93, 3.755839001138390788e+94, 1.158991729845978702e+96, 3.774916315438862678e+97, 1.298844894462381673e+99, 4.725038949943384889e+100, 1.819000031203286740e+102, 7.416966330876906188e+103, 3.206116996910598204e+105, 1.470588770071975193e+107, 7.164198238238641057e+108, 3.710397624567077270e+110, 2.044882454279709373e+112, 1.200428778654730225e+114, 7.513744370030172114e+115, 5.019575746343410636e+117, 3.582726927665698318e+119, 2.734947775877248560e+121, 2.235283764078944248e+123, 1.958084751118243323e+125, 1.840431913109305657e+127, 1.858143260692831108e+129, 2.017432949655777136e+131, 2.358177615888101494e+133, 2.971092974178603610e+135, 4.039532321435816302e+137, 5.933923069661132195e+139, 9.429263693444953240e+141, 1.622841456932873872e+144, 3.028884476067694180e+146, 6.138356175015339477e+148, 1.352531557191942648e+151, 3.244447362295582945e+153, },
   };
   m_weights = {
      { 2.703640234162693583e-160, 3.100862940179668765e-58, 5.828334625665462970e-21, 1.628894422402653830e-07, 8.129907377394029252e-03, 2.851214447180802931e-01, 1.228894002317118650e+00, 9.374610761705565881e+00, 6.136846875218162167e+02, 8.367995944653844271e+06, 2.286032371256753845e+17, 9.029964022492184559e+44, 1.637973037681055808e+119, },
      { 1.029757744225565290e-96, 5.564174008086804112e-35, 1.534846576427062716e-12, 1.519539651119905182e-04, 7.878691652861874032e-02, 6.288072016384128612e-01, 2.842403831496369386e+00, 5.152309209026500589e+01, 2.554172947873109927e+04, 8.291547503290989754e+10, 6.794911791960761587e+27, 1.108995159102362663e+73, },
      { 1.545310485347377408e-124, 4.549745016271158113e-75, 3.781189989988588481e-45, 4.369440793304363176e-27, 3.253896178006708087e-16, 1.057239289288944987e-09, 7.826174663495492476e-06, 1.459783224353939263e-03, 2.972970552567852420e-02, 1.637950661613330541e-01, 4.392303913269138921e-01, 8.744243777287317807e-01, 1.804759465860974506e+00, 4.894937215283148383e+00, 2.036214502429748943e+01, 1.576549789679037479e+02, 3.249553828744194733e+03, 3.335686029489862584e+05, 4.858218914917275532e+08, 5.655171002571584464e+13, 9.084276291356790926e+21, 2.202757570781655071e+35, 1.851176020895552142e+57, 1.873046373612647920e+93, 3.113183070605141140e+152, },
      { 2.690380169654157101e-141, 9.388760099830475385e-110, 3.267856956418766261e-85, 4.012903562780032075e-66, 2.794595941054873674e-51, 9.598140333687791635e-40, 8.762766371925782803e-31, 7.896919977115783593e-24, 1.951680620313826776e-18, 2.931867534349928041e-14, 4.976350908135118762e-11, 1.546933241860617074e-08, 1.283189791774752963e-06, 3.809052946018782340e-05, 5.087526585392884730e-04, 3.656819625189471368e-03, 1.627679402690602992e-02, 5.011672130624018967e-02, 1.165913368715250324e-01, 2.201514148384271336e-01, 3.581909054968942386e-01, 5.288599003801643436e-01, 7.422823219366348741e-01, 1.032914080772662205e+00, 1.478415067523268199e+00, 2.242226697017918644e+00, 3.684755742578570582e+00, 6.677326887819023056e+00, 1.358063058433697357e+01, 3.171262375809110066e+01, 8.776338468947827779e+01, 3.006939713363920293e+02, 1.352196150715330628e+03, 8.616353573310419356e+03, 8.591849573350877359e+04, 1.523635814554291966e+06, 5.663834603448267056e+07, 5.450828629396188577e+09, 1.780881993484818221e+12, 2.797112703281894578e+15, 3.300887168363313931e+19, 5.192538272313512016e+24, 2.273085973059979872e+31, 7.124498195222272142e+39, 5.379592741425673874e+50, 4.647296508337283075e+64, 3.395147156494395571e+82, 2.736576372417856435e+105, 6.584825756536212781e+134, },
      { 1.692276285171240629e-150, 1.180420021590838281e-132, 6.494931071412232065e-117, 4.979673804239645358e-103, 8.790122245397054202e-91, 5.564311726870413043e-80, 1.867634664877268411e-70, 4.693767384843440310e-62, 1.197772698674604837e-54, 4.060530886983702887e-48, 2.318268710612758367e-42, 2.748088060676949794e-37, 8.136086869664039226e-33, 7.081491999860360593e-29, 2.092407629019781417e-25, 2.383020547076997517e-22, 1.170143938604536054e-19, 2.734857915002515580e-17, 3.319894174569245506e-15, 2.260825106530477104e-13, 9.244747974241858562e-12, 2.410325858091057071e-10, 4.224928060220423782e-09, 5.217223349652829804e-08, 4.730110697329046717e-07, 3.265522864288710545e-06, 1.772851678458610971e-05, 7.787346612077215804e-05, 2.838101678971546354e-04, 8.775026198694109646e-04, 2.347474744139291716e-03, 5.529174974874315725e-03, 1.164520226280038968e-02, 2.223487842904240574e-02, 3.896253311038730452e-02, 6.334975706136386464e-02, 9.651712033300261848e-02, 1.390236708907266445e-01, 1.908593745910709887e-01, 2.515965688234414960e-01, 3.206651646562737595e-01, 3.976974208167367099e-01, 4.828935799767836828e-01, 5.773826389735376677e-01, 6.835838865575605461e-01, 8.056083579298257627e-01, 9.497742078309479997e-01, 1.125351459431134254e+00, 1.345711576612114788e+00, 1.630156867495860456e+00, 2.006880650908830857e+00, 2.517828844916874130e+00, 3.226826819856410846e+00, 4.233461155863004269e+00, 5.697400323487776530e+00, 7.882247346334201378e+00, 1.123717929435969530e+01, 1.655437952523069781e+01, 2.528458931361129124e+01, 4.019700050163276117e+01, 6.682515670231120695e+01, 1.168022589948424530e+02, 2.160045684819153702e+02, 4.257255901158116698e+02, 9.017180693982791021e+02, 2.072151523320542727e+03, 5.222689557952776194e+03, 1.461663959276604441e+04, 4.606455611513396576e+04, 1.660950339384278845e+05, 6.976630616605097333e+05, 3.484240083705972727e+06, 2.117385064786894718e+07, 1.607368605379557548e+08, 1.570235957877638143e+09, 2.041619284762317483e+10, 3.670425964529826371e+11, 9.527196643411724126e+12, 3.749667772735766186e+14, 2.365380223523087981e+16, 2.546815287226970627e+18, 5.026010591299970789e+20, 1.970775914722195502e+23, 1.682531038342715298e+26, 3.469062187981719410e+29, 1.942614547946028081e+33, 3.375034694941022784e+37, 2.115298406181711256e+42, 5.673738540911562268e+47, 7.904099301170483654e+53, 7.121903115084356741e+60, 5.321820777644930491e+68, 4.370977753639010591e+77, 5.429657931755513797e+87, 1.464602226824232950e+99, 1.292445035662836561e+112, 5.936633203060705474e+126, 2.402419924621336913e+143, },
      { 2.552410363565288863e-155, 7.965872719315690060e-146, 6.586401422963018216e-137, 1.563673437419490296e-128, 1.149636272392214573e-120, 2.810189759625314580e-113, 2.441446149780773329e-106, 8.026292508555041710e-100, 1.059034284623927886e-93, 5.927259046205893861e-88, 1.482220909125121967e-82, 1.738946448501809732e-77, 1.002047910184021813e-72, 2.960929073720769637e-68, 4.671749731809402860e-64, 4.088398674807775827e-60, 2.056642628601930023e-56, 6.149878578966749305e-53, 1.128142221531950274e-49, 1.307702777646013040e-46, 9.848757125541659318e-44, 4.946847667192787369e-41, 1.698284656321589089e-38, 4.077947349805764486e-36, 6.998897321243266048e-34, 8.762183229651405846e-32, 8.156281709801700633e-30, 5.747366069381804213e-28, 3.117951907317865517e-26, 1.323052992594482858e-24, 4.457166057119926322e-23, 1.208896132634708032e-21, 2.674697849739340358e-20, 4.887394807742436672e-19, 7.461632083041868391e-18, 9.622230748739818989e-17, 1.058884510032627118e-15, 1.003988180288807180e-14, 8.276358838778374127e-14, 5.982281469656734375e-13, 3.821855766886203088e-12, 2.174279097299082001e-11, 1.109294120074848583e-10, 5.109055596902086022e-10, 2.137447956882816268e-09, 8.170468538364022161e-09, 2.869308592926374871e-08, 9.305185930419436742e-08, 2.800231592227134982e-07, 7.855263634214717091e-07, 2.062924236714395731e-06, 5.092224131071637441e-06, 1.185972357373608535e-05, 2.615333473470835518e-05, 5.479175746096322166e-05, 1.093962713107868416e-04, 2.087714243290528595e-04, 3.818797556417767457e-04, 6.712796918790164790e-04, 1.136760145626956604e-03, 1.858775505765622915e-03, 2.941191222579735746e-03, 4.512821350378020080e-03, 6.727293426938802892e-03, 9.760915371480980900e-03, 1.380842853102550981e-02, 1.907678055354397196e-02, 2.577730275571060412e-02, 3.411688991056810143e-02, 4.428892397843486143e-02, 5.646473816310556552e-02, 7.078637998740884103e-02, 8.736131246718460273e-02, 1.062595125372295046e-01, 1.275132133780278017e-01, 1.511193209351630349e-01, 1.770443400812491404e-01, 2.052314915777496186e-01, 2.356095985715091716e-01, 2.681032744853198083e-01, 3.026439500331752405e-01, 3.391813282438962329e-01, 3.776949427111484449e-01, 4.182056049753837852e-01, 4.607866519948383101e-01, 5.055750360563806155e-01, 5.527824318481410262e-01, 6.027066663808878454e-01, 6.557439076684384801e-01, 7.124021812071310501e-01, 7.733169258916167748e-01, 8.392694625821144443e-01, 9.112094418201526544e-01, 9.902825786957198607e-01, 1.077865293953107863e+00, 1.175608288920191064e+00, 1.285491624542001346e+00, 1.409894601042286311e+00, 1.551684711657329886e+00, 1.714331263928885829e+00, 1.902051053858215699e+00, 2.119995922515087770e+00, 2.374495377438728901e+00, 2.673372087884984440e+00, 3.026354489757871517e+00, 3.445619726158519068e+00, 3.946512819227006419e+00, 4.548505964859933724e+00, 5.276487613615791435e+00, 6.162508226184798743e+00, 7.248163842886806184e+00, 8.587878410768473380e+00, 1.025346434903602082e+01, 1.234051869120733230e+01, 1.497748183201988157e+01, 1.833859935862139637e+01, 2.266266859437541631e+01, 2.828045768298752298e+01, 3.565528397044830339e+01, 4.544381261232990127e+01, 5.858833744254070379e+01, 7.645876087681923606e+01, 1.010741758687003802e+02, 1.354538987141142977e+02, 1.841824059064608872e+02, 2.543337025162468240e+02, 3.570103970895535977e+02, 5.099537256432247190e+02, 7.420561390174965949e+02, 1.101323941193719451e+03, 1.669232910686306616e+03, 2.587203282090385703e+03, 4.106608602134535014e+03, 6.685657263550896700e+03, 1.118216368762133982e+04, 1.924811115485038079e+04, 3.416174865734933127e+04, 6.263882227839496242e+04, 1.189094418952240294e+05, 2.342262528110389793e+05, 4.798899889628646876e+05, 1.025279649144740527e+06, 2.290428015483177407e+06, 5.365618820221241118e+06, 1.322172034826883742e+07, 3.438296542047893623e+07, 9.468905314460992170e+07, 2.771843378168242512e+08, 8.658950437199969679e+08, 2.898779165825890846e+09, 1.044627762990198184e+10, 4.071673625087267154e+10, 1.725245696783106160e+11, 7.989856904303845909e+11, 4.067537100664303783e+12, 2.290253922913114847e+13, 1.435560574531699914e+14, 1.008680130601194048e+15, 8.003530334765274913e+15, 7.227937568629809266e+16, 7.491693576707361828e+17, 8.991671234614216799e+18, 1.261556024888540618e+20, 2.090038400033346091e+21, 4.132773073376509056e+22, 9.865671928781943336e+23, 2.877978132616007671e+25, 1.039303004928044064e+27, 4.710544722984128252e+28, 2.719194692980296464e+30, 2.030608169419634520e+32, 1.994536427964099457e+34, 2.622806931876485852e+36, 4.705142628855489738e+38, 1.174794916996875010e+41, 4.170574236544843559e+43, 2.153441953645800917e+46, 1.656794933445123415e+49, 1.948830907651317326e+52, 3.601980393005358786e+55, 1.077033440153993124e+59, 5.374188883861674378e+62, 4.625267105826449467e+66, 7.111646979020385006e+70, 2.027996051444846521e+75, 1.116168784120367146e+80, 1.237019821283735086e+85, 2.888108172342166477e+90, 1.490426937972460544e+96, 1.789306677271856318e+102, 5.276973875344766848e+108, 4.051217867886536330e+115, 8.611617868168979525e+122, 5.412634353380155695e+130, 1.078756609821147465e+139, 7.344353246966125053e+147, },
      { 8.688318611421924613e-158, 6.864317997043424201e-153, 3.829638174036322920e-148, 1.524985558970066863e-143, 4.379527631402474835e-139, 9.162408388991747001e-135, 1.410086556664696347e-130, 1.611529786006329005e-126, 1.380269212504431613e-122, 8.938739565456142404e-119, 4.414803004265274778e-115, 1.676831992534574674e-111, 4.937648515671545377e-108, 1.136068312653058895e-104, 2.057969760853201132e-101, 2.956779836249922681e-98, 3.393449014375824853e-95, 3.132619285740674842e-92, 2.341677665639346254e-89, 1.426656997926173190e-86, 7.128825597334931865e-84, 2.939485275517928205e-81, 1.006113300119903410e-78, 2.874969402023240560e-76, 6.896713338909433222e-74, 1.396405038640012785e-71, 2.398869799873387326e-69, 3.514180228970525006e-67, 4.411557600438730779e-65, 4.768408435763044172e-63, 4.458287229998440383e-61, 3.621710763086768959e-59, 2.567373174003034094e-57, 1.594829856885795944e-55, 8.716746897177859412e-54, 4.208424534880021226e-52, 1.801637343401221381e-50, 6.864432292330768862e-49, 2.336084584516383243e-47, 7.125716658075193173e-46, 1.954733295862350631e-44, 4.838195020814970471e-43, 1.083903033389729471e-41, 2.204655424309513426e-40, 4.083431629921110537e-39, 6.907095608064865023e-38, 1.069951518082577963e-36, 1.521972185061747284e-35, 1.993254198127980161e-34, 2.409552194902670884e-33, 2.695243589253751811e-32, 2.796309045342585624e-31, 2.697138787161831243e-30, 2.423968619042656074e-29, 2.034233848004972409e-28, 1.597498662808006882e-27, 1.176341105034547043e-26, 8.138404856556384931e-26, 5.300199402716282910e-25, 3.255367628680633536e-24, 1.889060856810273071e-23, 1.037502167741821871e-22, 5.402129194695882094e-22, 2.671080147950250592e-21, 1.256163163817414397e-20, 5.627458451375099018e-20, 2.405110192151924414e-19, 9.820723025892385774e-19, 3.836610965933493002e-18, 1.435949417965440387e-17, 5.155736116435221852e-17, 1.778106820243535736e-16, 5.897650538103448384e-16, 1.883545377386949394e-15, 5.799022727889041128e-15, 1.723080101027408120e-14, 4.946559668895564981e-14, 1.373437058883951037e-13, 3.692057356296675476e-13, 9.618669754374864080e-13, 2.430904641718059201e-12, 5.965319652795549281e-12, 1.422677541958913512e-11, 3.300412010407028696e-11, 7.453993539444124847e-11, 1.640317480539372495e-10, 3.519919455549922227e-10, 7.371241496931924727e-10, 1.507573517782825692e-09, 3.013444008176544118e-09, 5.891170930525923854e-09, 1.127175867596519203e-08, 2.112135943063526334e-08, 3.878572405868819131e-08, 6.984140168311147329e-08, 1.233979234102365865e-07, 2.140481233406505212e-07, 3.647293211756793211e-07, 6.108366265875129839e-07, 1.006020283089617901e-06, 1.630199379920459998e-06, 2.600430208375972125e-06, 4.085372746054298735e-06, 6.324194831966406940e-06, 9.650830226718535837e-06, 1.452455211307694488e-05, 2.156782506321975658e-05, 3.161234361554654466e-05, 4.575404320696170555e-05, 6.541767069965264068e-05, 9.243122234114186712e-05, 1.291101968446571125e-04, 1.783511762821284409e-04, 2.437337497712608884e-04, 3.296292528289701234e-04, 4.413142327104518440e-04, 5.850859955683163216e-04, 7.683770763700705263e-04, 9.998650298180469208e-04, 1.289573601590465490e-03, 1.648961132392222413e-03, 2.090991995585424661e-03, 2.630186988492201910e-03, 3.282648895332118799e-03, 4.066059914467245175e-03, 4.999648283080481820e-03, 6.104122218554241819e-03, 7.401570199659662364e-03, 8.915327597805008451e-03, 1.066981070009509413e-02, 1.269032020049755525e-02, 1.500281723149735994e-02, 1.763367592672867332e-02, 2.060941730962251417e-02, 2.395642996410886880e-02, 2.770068343772389725e-02, 3.186744063963193757e-02, 3.648097561865623097e-02, 4.156430303997019336e-02, 4.713892543167989540e-02, 5.322460385886412684e-02, 5.983915712308283792e-02, 6.699829390463281224e-02, 7.471548149065050122e-02, 8.300185389391494996e-02, 9.186616129460712899e-02, 1.013147618591979452e-01, 1.113516561340355690e-01, 1.219785634003157786e-01, 1.331950386328042665e-01, 1.449986280439946752e-01, 1.573850606313672716e-01, 1.703484726870446791e-01, 1.838816618814874884e-01, 1.979763672973498048e-01, 2.126235716643688402e-01, 2.278138220265254991e-01, 2.435375651517067386e-01, 2.597854941629632707e-01, 2.765489031191654411e-01, 2.938200465906351752e-01, 3.115925016510994851e-01, 3.298615301301230823e-01, 3.486244394295739435e-01, 3.678809406939879716e-01, 3.876335036292959599e-01, 4.078877077798518471e-01, 4.286525905940105684e-01, 4.499409931290513174e-01, 4.717699047639316286e-01, 4.941608088016098926e-01, 5.171400313514193966e-01, 5.407390963876342256e-01, 5.649950903858123945e-01, 5.899510404480374918e-01, 6.156563103475134535e-01, 6.421670194591982411e-01, 6.695464901047961714e-01, 6.978657294374126896e-01, 7.272039526349696447e-01, 7.576491548751669105e-01, 7.892987403432202489e-01, 8.222602173936578230e-01, 8.566519699682320391e-01, 8.926041164852169437e-01, 9.302594686857616145e-01, 9.697746043788558519e-01, 1.011321069700320644e+00, 1.055086728430498711e+00, 1.101277278143300224e+00, 1.150117955536247302e+00, 1.201855456275760449e+00, 1.256760098152647779e+00, 1.315128260359919236e+00, 1.377285136373095709e+00, 1.443587843343442141e+00, 1.514428937238563465e+00, 1.590240390338335337e+00, 1.671498096302065311e+00, 1.758726978084942299e+00, 1.852506785760205887e+00, 1.953478685110838140e+00, 2.062352754065132708e+00, 2.179916523112736371e+00, 2.307044718290330681e+00, 2.444710391817196957e+00, 2.593997656772008968e+00, 2.756116279277535182e+00, 2.932418425642610903e+00, 3.124417914187536020e+00, 3.333812383735923205e+00, 3.562508865047068391e+00, 3.812653330296280988e+00, 4.086664902155689132e+00, 4.387275531849634155e+00, 4.717576109385405085e+00, 5.081070154695596855e+00, 5.481736462718817995e+00, 5.924102347216244340e+00, 6.413329458204850426e+00, 6.955314549766230740e+00, 7.556808065486941215e+00, 8.225554008952760095e+00, 8.970455302965185036e+00, 9.801769746699598466e+00, 1.073134279679936208e+01, 1.177288477943655549e+01, 1.294230185297226511e+01, 1.425809217068106541e+01, 1.574182134943112610e+01, 1.741869467329444792e+01, 1.931824763074534781e+01, 2.147518163232618457e+01, 2.393037838236259586e+01, 2.673213477270754163e+01, 2.993767083537830673e+01, 3.361497689655818107e+01, 3.784508348524495401e+01, 4.272485990900652026e+01, 4.837047622725585887e+01, 5.492170063250241752e+01, 6.254725265973777743e+01, 7.145149574983117631e+01, 8.188283528217430591e+01, 9.414429671899321190e+01, 1.086069017070108772e+02, 1.257266497442910506e+02, 1.460661655727672308e+02, 1.703224100743601641e+02, 1.993623058409479084e+02, 2.342687403011957198e+02, 2.764002385528330658e+02, 3.274687277481591846e+02, 3.896413615832930151e+02, 4.656745019682919178e+02, 5.590908996105107215e+02, 6.744152109571297875e+02, 8.174887172033244140e+02, 9.958921680864290197e+02, 1.219517071629880108e+03, 1.501341972869855447e+03, 1.858493492282554856e+03, 2.313705362529768409e+03, 2.897337235279879262e+03, 3.650185874628374320e+03, 4.627425468074182920e+03, 5.904167858279871204e+03, 7.583363128219763259e+03, 9.807105719965428472e+03, 1.277293273832114230e+04, 1.675749596877978193e+04, 2.215121038263169759e+04, 2.950937349291504490e+04, 3.962820433513419525e+04, 5.365890489878942635e+04, 7.328024305737981431e+04, 1.009620167752942516e+05, 1.403709568321740997e+05, 1.970019955923188504e+05, 2.791695960502382133e+05, 3.995801250202947693e+05, 5.778515877588312220e+05, 8.445944401474017243e+05, 1.248092975135001687e+06, 1.865367859966950385e+06, 2.820705292493674480e+06, 4.317063433830483499e+06, 6.689961127164684387e+06, 1.050111601631327499e+07, 1.670327884792325766e+07, 2.693430470211696200e+07, 4.404906898054894166e+07, 7.309535640536363311e+07, 1.231306812701882145e+08, 2.106560568719367745e+08, 3.662073971851359192e+08, 6.472124787519330196e+08, 1.163486593592585616e+09, 2.128658395254150452e+09, 3.965732938755983605e+09, 7.527735928223242836e+09, 1.456757162128879538e+10, 2.875798636941021041e+10, 5.794999654160054887e+10, 1.192767536774485257e+11, 2.509334090779650360e+11, 5.399624414800303207e+11, 1.189276111740286910e+12, 2.683103883355551677e+12, 6.205255919751506427e+12, 1.472284072112162717e+13, 3.586628373992547853e+13, 8.978594107356889337e+13, 2.311710197091641250e+14, 6.127020712804348908e+14, 1.673232679378485978e+15, 4.712671499032329365e+15, 1.370275025680988289e+16, 4.117347054027612886e+16, 1.279822436878842710e+17, 4.119762767831332886e+17, 1.374888606936629814e+18, 4.762483833659790733e+18, 1.714288404980390540e+19, 6.420200704842635702e+19, 2.504808062315322558e+20, 1.019355251138167687e+21, 4.332952958521756932e+21, 1.926416464889827426e+22, 8.971059571108856501e+22, 4.382317748928748816e+23, 2.249003059943548727e+24, 1.214458587662725100e+25, 6.911683912813140938e+25, 4.152578123301633020e+26, 2.638346388179288086e+27, 1.775811490887700718e+28, 1.268552401544524965e+29, 9.635786341213661742e+29, 7.797939379813000783e+30, 6.736900087983560033e+31, 6.226288752443836475e+32, 6.169035287163451891e+33, 6.567250104576983172e+34, 7.528666735185428595e+35, 9.316271421365627344e+36, 1.247410737003664698e+38, 1.811787648043939987e+39, 2.861918583157116420e+40, 4.929657099622567574e+41, 9.284951278562156071e+42, 1.917687997037326435e+44, 4.355948096683946408e+45, 1.091453486585817118e+47, 3.026206402784023251e+48, 9.314478983991942688e+49, 3.193195693823940775e+51, 1.223447678968662613e+53, 5.257403184148516426e+54, 2.543108925126136766e+56, 1.389947584026783879e+58, 8.616987336205957549e+59, 6.083777056769299984e+61, 4.911841077800001710e+63, 4.554259483169784661e+65, 4.870815185962582259e+67, 6.036211886847067841e+69, 8.708377755587698026e+71, 1.469655296381977267e+74, 2.915822924489215887e+76, 6.836044306573246016e+78, 1.903917300559946782e+81, 6.333813341980360028e+83, 2.531082268773868753e+86, 1.222077360592898816e+89, 7.172167453276776330e+91, 5.148160232410244898e+94, 4.548619807672339638e+97, 4.979632843475864923e+100, 6.800802744782331957e+103, 1.166855497965918386e+107, 2.533457765534279043e+110, 7.012864641215147208e+113, 2.494083354169569414e+117, 1.148722178881219993e+121, 6.908313932158993510e+124, 5.470912484744367184e+128, 5.755359832684120769e+132, 8.115681923907451939e+136, 1.548304780334447081e+141, 4.034912159113614601e+145, 1.450632759611715526e+150, 7.268799665580789770e+154, },
      { 4.901759085947701448e-159, 1.505832423620814399e-156, 4.231872109262999523e-154, 1.089479701785106001e-151, 2.572922387150651649e-149, 5.581311054334156941e-147, 1.113575900126970040e-144, 2.046165051332286084e-142, 3.466994885004770636e-140, 5.423795404073501922e-138, 7.843833272402847010e-136, 1.049922957933194415e-133, 1.302301071957418603e-131, 1.498659737828393008e-129, 1.601906622414286282e-127, 1.592248618401983561e-125, 1.473375345916436274e-123, 1.270651551394009593e-121, 1.022408263525766209e-119, 7.683762602329562781e-118, 5.399268127233373186e-116, 3.551074274853494676e-114, 2.188235409519121010e-112, 1.264667515430816934e-110, 6.861807566737243712e-109, 3.498691686825209963e-107, 1.678016807398375157e-105, 7.577439431441931490e-104, 3.224703770159386809e-102, 1.294487090677705963e-100, 4.906133250963454139e-99, 1.757121317988153326e-97, 5.952042491454320383e-96, 1.908566653286417264e-94, 5.798224459236429212e-93, 1.670293239978334727e-91, 4.566236673398083038e-90, 1.185617342791547945e-88, 2.926160027801296929e-87, 6.870061134126707137e-86, 1.535565783500379945e-84, 3.270036736778401257e-83, 6.639558007206580362e-82, 1.286319750967398593e-80, 2.379566581139022958e-79, 4.206268231398883425e-78, 7.109719237833379433e-77, 1.149915104115372777e-75, 1.780876201255594220e-74, 2.642703796179329883e-73, 3.760085375941719327e-72, 5.132920951124251993e-71, 6.727100274601427696e-70, 8.469585621347697498e-69, 1.025032382672232848e-67, 1.193219127557863348e-66, 1.336816930381306582e-65, 1.442283479679798385e-64, 1.499374555004793991e-63, 1.502797203133501438e-62, 1.453005969318485303e-61, 1.355980448377862540e-60, 1.222072412212552127e-59, 1.064223180270520159e-58, 8.959667396075636845e-58, 7.296288808079294105e-57, 5.750255296190181158e-56, 4.388011664829013518e-55, 3.243852451291832398e-54, 2.324239357665538806e-53, 1.614869776203026446e-52, 1.088524605545274842e-51, 7.121755574192829045e-51, 4.524647662549067074e-50, 2.792730715818793035e-49, 1.675384879603864227e-48, 9.773114328777676091e-48, 5.545910766847627082e-47, 3.062809705627873645e-46, 1.646862118038266234e-45, 8.625108513887155847e-45, 4.401687663868890701e-44, 2.189755778847646746e-43, 1.062345336449265889e-42, 5.028036663485684049e-42, 2.322524635717249223e-41, 1.047406593898341306e-40, 4.613438388449698168e-40, 1.985397445118162005e-39, 8.351027367454628343e-39, 3.434440903484543389e-38, 1.381489131877196646e-37, 5.437051201310225224e-37, 2.094357548080647717e-36, 7.898676618592006902e-36, 2.917536870947471272e-35, 1.055788886022716597e-34, 3.744333812160330812e-34, 1.301801185251957290e-33, 4.438346216893387768e-33, 1.484348268951816542e-32, 4.871001129849836971e-32, 1.568903000742513942e-31, 4.961295315917935235e-31, 1.540773910027990821e-30, 4.700558022172014910e-30, 1.409115230718949596e-29, 4.151913103955692034e-29, 1.202737613715427748e-28, 3.426327374934496736e-28, 9.601405359397026012e-28, 2.647278642033773301e-27, 7.183442220565147103e-27, 1.918850545981494042e-26, 5.046974779455992494e-26, 1.307394799925911700e-25, 3.336342198236957082e-25, 8.389259581136262194e-25, 2.079051813513548608e-24, 5.079178967243765280e-24, 1.223501794357837278e-23, 2.906654911057549530e-23, 6.811668606095015470e-23, 1.574985938238025303e-22, 3.593796788969348326e-22, 8.094185411205212564e-22, 1.799796183237481721e-21, 3.951758901641017285e-21, 8.569580068050865775e-21, 1.835753486517298696e-20, 3.885414339966022317e-20, 8.126613972895021790e-20, 1.680007182889503141e-19, 3.433369351563962828e-19, 6.937695550399427499e-19, 1.386345631008981755e-18, 2.740087497759230881e-18, 5.357570288683386626e-18, 1.036464933022803784e-17, 1.984249442010084992e-17, 3.759788006060003409e-17, 7.052211261821684795e-17, 1.309635641529546221e-16, 2.408275496109180528e-16, 4.385898809611711552e-16, 7.911758686849121285e-16, 1.413883597877183873e-15, 2.503477536644680210e-15, 4.392637866550705827e-15, 7.638710306960574612e-15, 1.316703360377476041e-14, 2.250031027275448919e-14, 3.812239733412214953e-14, 6.405021660191363479e-14, 1.067250538270319484e-13, 1.763897493784721010e-13, 2.891987565334547756e-13, 4.704242520369958085e-13, 7.592878273512691990e-13, 1.216183338372525172e-12, 1.933388593436624879e-12, 3.050826852442290751e-12, 4.779080020017636657e-12, 7.432734713385425098e-12, 1.147833888125873666e-11, 1.760286160372422754e-11, 2.681071101623953168e-11, 4.056023754295965437e-11, 6.095443492241537222e-11, 9.100550129616064211e-11, 1.349993452136967652e-10, 1.989943912395156051e-10, 2.914996073619059788e-10, 4.243900781412219621e-10, 6.141353162671391082e-10, 8.834365795894798511e-10, 1.263395594025933170e-09, 1.796369250051716047e-09, 2.539704143326480862e-09, 3.570592498287890499e-09, 4.992348403150539107e-09, 6.942471870489931483e-09, 9.602949600164561371e-09, 1.321333712761666777e-08, 1.808727901635346390e-08, 2.463325364767791516e-08, 3.338047870136870496e-08, 4.501108426108505069e-08, 6.039985413333259594e-08, 8.066305374526097834e-08, 1.072181059018892614e-07, 1.418561443795353991e-07, 1.868297699836383305e-07, 2.449586539172972009e-07, 3.197559780442760832e-07, 4.155790690867544334e-07, 5.378079713325544678e-07, 6.930561064776686194e-07, 8.894175852502122454e-07, 1.136756157868726006e-06, 1.447041212534730898e-06, 1.834736645332833504e-06, 2.317248822354253644e-06, 2.915440225825303911e-06, 3.654215709863551870e-06, 4.563188576773760151e-06, 5.677433909482232878e-06, 7.038336747307571784e-06, 8.694542758083067228e-06, 1.070301902702759858e-05, 1.313023243937403750e-05, 1.605345286789073897e-05, 1.956218797728780449e-05, 2.375975591555218862e-05, 2.876500146954361208e-05, 3.471416041263076209e-05, 4.176287576185915239e-05, 5.008836848967403773e-05, 5.989176390181730373e-05, 7.140057340280213227e-05, 8.487132973049760036e-05, 1.005923719620999934e-04, 1.188867746885496973e-04, 1.401154137398069279e-04, 1.646801587388731249e-04, 1.930271805904271778e-04, 2.256503597954330556e-04, 2.630947792533707128e-04, 3.059602829980946180e-04, 3.549050801425155303e-04, 4.106493712131842727e-04, 4.739789720708565436e-04, 5.457489087697051069e-04, 6.268869550379884668e-04, 7.183970825975973673e-04, 8.213627933082928901e-04, 9.369503011517966364e-04, 1.066411531385725184e-03, 1.211086903819095417e-03, 1.372407867107646339e-03, 1.551899151252505624e-03, 1.751180706119547318e-03, 1.971969294784470944e-03, 2.216079711850908971e-03, 2.485425598581779636e-03, 2.782019828718993257e-03, 3.107974441230220176e-03, 3.465500098895993776e-03, 3.856905054613959619e-03, 4.284593610523639393e-03, 4.751064058515097225e-03, 5.258906094345618421e-03, 5.810797701414435799e-03, 6.409501504198915943e-03, 7.057860595396970186e-03, 7.758793844909123446e-03, 8.515290702888369372e-03, 9.330405513145299523e-03, 1.020725135717912572e-02, 1.114899345297222760e-02, 1.215884213639836574e-02, 1.324004545661629463e-02, 1.439588142011718850e-02, 1.562964992113485073e-02, 1.694466439888404584e-02, 1.834424326453982033e-02, 1.983170114298836870e-02, 2.141033997615067889e-02, 2.308344003609062690e-02, 2.485425089716015368e-02, 2.672598241710042669e-02, 2.870179577730820310e-02, 3.078479463239356953e-02, 3.297801641870515720e-02, 3.528442387069167064e-02, 3.770689679281728890e-02, 4.024822413326941635e-02, 4.291109640390936770e-02, 4.569809848884132640e-02, 4.861170288163592155e-02, 5.165426338866744454e-02, 5.482800933323496446e-02, 5.813504029216542680e-02, 6.157732139347005467e-02, 6.515667920037330165e-02, 6.887479820368566403e-02, 7.273321794107712090e-02, 7.673333075835566151e-02, 8.087638022439339824e-02, 8.516346020789830747e-02, 8.959551462082867423e-02, 9.417333782991444898e-02, 9.889757573450802477e-02, 1.037687275058577967e-01, 1.087871479799008567e-01, 1.139530506928239996e-01, 1.192665115459606141e-01, 1.247274730840887416e-01, 1.303357493688843496e-01, 1.360910314271734020e-01, 1.419928932517243620e-01, 1.480407983306351483e-01, 1.542341066798992024e-01, 1.605720823524863565e-01, 1.670539013962460335e-01, 1.736786602321317742e-01, 1.804453844236544912e-01, 1.873530378080931153e-01, 1.944005319598201097e-01, 2.015867359561292115e-01, 2.089104864161762672e-01, 2.163705977840528187e-01, 2.239658728275971045e-01, 2.316951133252986765e-01, 2.395571309145607347e-01, 2.475507580756380088e-01, 2.556748592267567912e-01, 2.639283419072366399e-01, 2.723101680268593668e-01, 2.808193651612593497e-01, 2.894550378747292326e-01, 2.982163790535362503e-01, 3.071026812346166036e-01, 3.161133479163487600e-01, 3.252479048399920142e-01, 3.345060112323053140e-01, 3.438874710018250777e-01, 3.533922438832718793e-01, 3.630204565265675291e-01, 3.727724135289699431e-01, 3.826486084108677024e-01, 3.926497345378144818e-01, 4.027766959934214472e-01, 4.130306184097598756e-01, 4.234128597639539906e-01, 4.339250211516634154e-01, 4.445689575501645526e-01, 4.553467885857401860e-01, 4.662609093220769612e-01, 4.773140010883521767e-01, 4.885090423676662636e-01, 4.998493197684479070e-01, 5.113384391034281429e-01, 5.229803366027518117e-01, 5.347792902897740156e-01, 5.467399315500809553e-01, 5.588672569262846167e-01, 5.711666401731758417e-01, 5.836438446098876156e-01, 5.963050358078278898e-01, 6.091567946552975691e-01, 6.222061308419237716e-01, 6.354604968083211637e-01, 6.489278022087558681e-01, 6.626164289370386795e-01, 6.765352467684294227e-01, 6.906936296730053994e-01, 7.051014728587479919e-01, 7.197692106055475377e-01, 7.347078349544334315e-01, 7.499289153196209421e-01, 7.654446190944464391e-01, 7.812677333259577661e-01, 7.974116875368567865e-01, 8.138905777776784362e-01, 8.307191919965581771e-01, 8.479130368187123741e-01, 8.654883658328603475e-01, 8.834622094872810766e-01, 9.018524067040521621e-01, 9.206776383262963142e-01, 9.399574625199963151e-01, 9.597123522591707284e-01, 9.799637350309700387e-01, 1.000734034905599933e+00, 1.022046717124952010e+00, 1.043926335373472893e+00, 1.066398581905185161e+00, 1.089490340711946628e+00, 1.113229743930062164e+00, 1.137646231695313314e+00, 1.162770615670420260e+00, 1.188635146483979071e+00, 1.215273585336112390e+00, 1.242721280043529050e+00, 1.271015245815510799e+00, 1.300194251072644711e+00, 1.330298908642019971e+00, 1.361371772686240192e+00, 1.393457441749111730e+00, 1.426602668328411758e+00, 1.460856475415888358e+00, 1.496270280476785338e+00, 1.532898027375920169e+00, 1.570796326794896619e+00, 1.610024605725646420e+00, 1.650645266669431435e+00, 1.692723857217988332e+00, 1.736329250744977731e+00, 1.781533838991654903e+00, 1.828413737391087381e+00, 1.877049004040720448e+00, 1.927523873304087635e+00, 1.979927005099477087e+00, 2.034351751016940433e+00, 2.090896438495766214e+00, 2.149664674393090421e+00, 2.210765669381402212e+00, 2.274314584729113927e+00, 2.340432903144970240e+00, 2.409248825504827076e+00, 2.480897695429288043e+00, 2.555522453844001656e+00, 2.633274125832370887e+00, 2.714312342284411608e+00, 2.798805899057066353e+00, 2.886933356592141886e+00, 2.978883683190077867e+00, 3.074856945413050211e+00, 3.175065049391765683e+00, 3.279732537139255280e+00, 3.389097442334834102e+00, 3.503412210435275865e+00, 3.622944688401595705e+00, 3.747979189802462585e+00, 3.878817641573403805e+00, 4.015780819279312670e+00, 4.159209678351536168e+00, 4.309466789455788368e+00, 4.466937886899736897e+00, 4.632033539816493591e+00, 4.805190956770360727e+00, 4.986875935432896972e+00, 5.177584970080537688e+00, 5.377847530880629761e+00, 5.588228530273088035e+00, 5.809330993233640059e+00, 6.041798949837089488e+00, 6.286320570342285919e+00, 6.543631565013652661e+00, 6.814518873098582608e+00, 7.099824667819718682e+00, 7.400450706942931008e+00, 7.717363061475788814e+00, 8.051597258371279584e+00, 8.404263876795383951e+00, 8.776554641607500109e+00, 9.169749062247565207e+00, 9.585221670276993889e+00, 1.002444991444300704e+01, 1.048902277839603856e+01, 1.098065019316492606e+01, 1.150117332427169985e+01, 1.205257582204547280e+01, 1.263699613338454324e+01, 1.325674098404332380e+01, 1.391430015262873368e+01, 1.461236267104086712e+01, 1.535383460126837531e+01, 1.614185855545811846e+01, 1.697983514525758524e+01, 1.787144656784601339e+01, 1.882068256013178484e+01, 1.983186897964764985e+01, 2.090969930111845450e+01, 2.205926935196095527e+01, 2.328611564861881683e+01, 2.459625773922860138e+01, 2.599624500732998276e+01, 2.749320844694889238e+01, 2.909491798228195984e+01, 3.080984597641076715e+01, 3.264723765414180400e+01, 3.461718925554321861e+01, 3.673073484057443067e+01, 3.899994278315456980e+01, 4.143802312713618427e+01, 4.405944712930142330e+01, 4.688008048840357439e+01, 4.991733195758662298e+01, 5.319031926387298369e+01, 5.672005451703465811e+01, 6.052965158594831140e+01, 6.464455825915836491e+01, 6.909281639443131774e+01, 7.390535370725211687e+01, 7.911631135942343489e+01, 8.476341209659472308e+01, 9.088837435982152722e+01, 9.753737857533253823e+01, 1.047615927251647361e+02, 1.126177653386554197e+02, 1.211688952437418817e+02, 1.304849888043593828e+02, 1.406439169773708701e+02, 1.517323863863765989e+02, 1.638470407739824279e+02, 1.770957117100033620e+02, 1.915988403612775885e+02, 2.074910955409497265e+02, 2.249232172361061194e+02, 2.440641194630869936e+02, 2.651032917390266964e+02, 2.882535448280364212e+02, 3.137541538897424513e+02, 3.418744609277612322e+02, 3.729180087461214321e+02, 4.072272907593818790e+02, 4.451892153103389878e+02, 4.872414000388630927e+02, 5.338794318098249932e+02, 5.856652513400113117e+02, 6.432368496766822816e+02, 7.073194969336578611e+02, 7.787387632221277236e+02, 8.584356387770406827e+02, 9.474841163944599543e+02, 1.047111666301969297e+03, 1.158723113719277435e+03, 1.283928525349707755e+03, 1.424575826189363437e+03, 1.582789006393775706e+03, 1.761012944445459235e+03, 1.962066073573121788e+03, 2.189202360708354222e+03, 2.446184360349559652e+03, 2.737369460761187093e+03, 3.067811870808767638e+03, 3.443383419509962754e+03, 3.870916878218207705e+03, 4.358376293464465508e+03, 4.915059769420260559e+03, 5.551841303216967404e+03, 6.281459704453426129e+03, 7.118864385205665710e+03, 8.081629967627799596e+03, 9.190454321738597280e+03, 1.046975794051835702e+04, 1.194840663946247320e+04, 1.366058463062104793e+04, 1.564685131637809273e+04, 1.795542299179967539e+04, 2.064373043744082514e+04, 2.378031563732670807e+04, 2.744714621995650953e+04, 3.174244552480722739e+04, 3.678416050731336226e+04, 4.271422037773508051e+04, 4.970377768100323981e+04, 5.795967273138576164e+04, 6.773242484608792593e+04, 7.932613346949942761e+04, 9.311077397156915450e+04, 1.095375030536372224e+05, 1.291577556735669526e+05, 1.526471301608741586e+05, 1.808353350969648289e+05, 2.147438294770164181e+05, 2.556332515573999948e+05, 3.050633345562097502e+05, 3.649687926665853954e+05, 4.377556866857485380e+05, 5.264241222943208736e+05, 6.347248990108319410e+05, 7.673600526542426466e+05, 9.302403050337502786e+05, 1.130816502666451845e+06, 1.378507531155523742e+06, 1.685254393964162275e+06, 2.066239770168639390e+06, 2.540825270229354918e+06, 3.133775962036416630e+06, 3.876865148275802393e+06, 4.810984054018349430e+06, 5.988924089534678664e+06, 7.479057929608060924e+06, 9.370225698693408867e+06, 1.177824230977510661e+07, 1.485459301432580619e+07, 1.879809270383398104e+07, 2.387057334436346400e+07, 3.041806552258603202e+07, 3.889950046843262151e+07, 4.992574374586696017e+07, 6.431287504495613210e+07, 8.315518519925858136e+07, 1.079255664704117961e+08, 1.406141073390035115e+08, 1.839201785677305607e+08, 2.415197116904975365e+08, 3.184386015381112281e+08, 4.215765018929686736e+08, 5.604446356915114550e+08, 7.482094398046911572e+08, 1.003175129668246151e+09, 1.350898918997482870e+09, 1.827222165053491590e+09, 2.482633480831760933e+09, 3.388577637234919719e+09, 4.646620065299105644e+09, 6.401821801566297122e+09, 8.862352038053251473e+09, 1.232838602859196811e+10, 1.723489297480180023e+10, 2.421530528469447376e+10, 3.419673813208063025e+10, 4.854312364622606540e+10, 6.927149043760342676e+10, 9.938049490186203616e+10, 1.433521424759854145e+11, 2.079221734483088227e+11, 3.032695241820108158e+11, 4.448631503727710431e+11, 6.563458646477901051e+11, 9.740635696398910980e+11, 1.454220520059656158e+12, 2.184250688898627320e+12, 3.300999104757560757e+12, 5.019970485022749012e+12, 7.682676299017607834e+12, 1.183376596003983872e+13, 1.834748853557035315e+13, 2.863639312458363586e+13, 4.499803892715039958e+13, 7.119486876989154498e+13, 1.134307017980122346e+14, 1.820065782363618395e+14, 2.941484500615394037e+14, 4.788707305890930382e+14, 7.854025036928623551e+14, 1.297894304619860251e+15, 2.161279954782425640e+15, 3.627102147035003834e+15, 6.135342933440950378e+15, 1.046170006362244506e+16, 1.798477357839665686e+16, 3.117473412332331475e+16, 5.449445073049184222e+16, 9.607515505017978212e+16, 1.708589224452677852e+17, 3.065429751110228665e+17, 5.549227437451149511e+17, 1.013730232778046314e+18, 1.869059895876405824e+18, 3.478549552381578424e+18, 6.535992245975463763e+18, 1.240019272261066308e+19, 2.375828866910936629e+19, 4.597682433604432625e+19, 8.988106816837128428e+19, 1.775302379393632263e+20, 3.543413304390973486e+20, 7.148061397675525327e+20, 1.457620510577186305e+21, 3.005137124879829797e+21, 6.265024861633250697e+21, 1.320979941090283816e+22, 2.817487535902146221e+22, 6.079933041429805231e+22, 1.327658853647212083e+23, 2.934311759183641318e+23, 6.565087216807130026e+23, 1.487212273437937650e+24, 3.411840196076788128e+24, 7.928189928797018762e+24, 1.866451877029704857e+25, 4.452521859886739549e+25, 1.076545435174977662e+26, 2.638685681190697586e+26, 6.557908470244186498e+26, 1.652952243735585721e+27, 4.226383395914916199e+27, 1.096450394268080148e+28, 2.886822082999286080e+28, 7.715480389344015925e+28, 2.093728789309964846e+29, 5.770275789447655037e+29, 1.615463845391781140e+30, 4.595470055795608691e+30, 1.328629392686523255e+31, 3.905079681530784219e+31, 1.167134024271997252e+32, 3.548058538654277403e+32, 1.097378059358046160e+33, 3.454102978064445595e+33, 1.106745393701652323e+34, 3.610899559139069994e+34, 1.199946999283670567e+35, 4.062687014190878792e+35, 1.401835223893224514e+36, 4.931085527333162173e+36, 1.768812393284919500e+37, 6.472148293945199961e+37, 2.416453721739211922e+38, 9.208944720398123862e+38, 3.583297028622126676e+39, 1.424097482596699440e+40, 5.782627833426411524e+40, 2.399862204084363183e+41, 1.018291572042305460e+42, 4.419105414822034531e+42, 1.962126117680499311e+43, 8.916742424061253707e+43, 4.148882478294757720e+44, 1.977256529558276930e+45, 9.655300233875401080e+45, 4.832878898335598922e+46, 2.480575878223098058e+47, 1.306102809757654706e+48, 7.057565717289569232e+48, 3.915276522229618618e+49, 2.230898980943393318e+50, 1.306141334496309306e+51, 7.861021286656392627e+51, 4.865583758538451107e+52, 3.098487425915704674e+53, 2.031037614862563901e+54, 1.370999647608260200e+55, 9.534736274325001528e+55, 6.834959923166415407e+56, 5.052733546324789020e+57, 3.853810997282159979e+58, 3.034183107853208298e+59, 2.467161926009838899e+60, 2.072901039813580593e+61, 1.800563980579615383e+62, 1.617764027895344257e+63, 1.504283028250688329e+64, 1.448393206525427172e+65, 1.444855510980115799e+66, 1.494120428855029243e+67, 1.602566566107015722e+68, 1.783880504153942988e+69, 2.061999240572760738e+70, 2.476521794698572715e+71, 3.092349914153497358e+72, 4.016927238305985810e+73, 5.431607545226497387e+74, 7.650086824042822759e+75, 1.123017984114349288e+77, 1.719382952966052004e+78, 2.747335718690686674e+79, 4.584545010557684123e+80, 7.995082041539250252e+81, 1.458119909365899044e+83, 2.783001178679600175e+84, 5.562812231966194628e+85, 1.165338768982404578e+87, 2.560399126432838224e+88, 5.904549641859098192e+89, 1.430278474749838710e+91, 3.642046122956932563e+92, 9.756698571206402300e+93, 2.751946044275883051e+95, 8.179164793643197279e+96, 2.563704735086825890e+98, 8.481656496128255880e+99, 2.964260254403981007e+101, 1.095342970031208886e+103, 4.283148547584870628e+104, 1.773954352944319744e+106, 7.788991081894224760e+107, 3.628931721056821352e+109, 1.795729272516020592e+111, 9.446685151482835339e+112, 5.288263179614488101e+114, 3.153311236741401362e+116, 2.004807079683827669e+118, 1.360407192665237716e+120, 9.862825609807810517e+121, 7.647551788591128099e+123, 6.348802224871730088e+125, 5.649062361980019098e+127, 5.393248003523784781e+129, 5.530897191915703916e+131, 6.099598644640894333e+133, 7.242098433491964504e+135, 9.268083053637375570e+137, 1.279942702416040582e+140, 1.909796626960621302e+142, 3.082540300669885040e+144, 5.388809732384179657e+146, 1.021610251056626535e+149, 2.103005440072790650e+151, 4.706753990348725570e+153, 1.146834128125248991e+156, },
   };
   m_committed_refinements = static_cast<boost::math::detail::atomic_unsigned_integer_type>(m_weights.size() - 1);
   m_t_min = -6.164062500000000000;
   if (m_max_refinements >= m_abscissas.size())
   {
      m_abscissas.resize(m_max_refinements + 1);
      m_weights.resize(m_max_refinements + 1);
   }
   else
   {
      m_max_refinements = m_abscissas.size() - 1;
   }
}
#if LDBL_MAX_EXP == 16384
template<class Real, class Policy>
void exp_sinh_detail<Real, Policy>::init(const mpl::int_<3>&)
{
   m_abscissas = {
      { 1.02756529896290544244959196973059583e-2497L, 2.56737528671961581475200468317128232e-919L, 1.17417808941462434184174780056564573e-338L, 4.82182886130634548471358754377036370e-125L, 1.85613301660646818149136526457281019e-46L, 1.52174118093087534300657777272732001e-17L, 6.75122240537294392532710530940672267e-07L, 5.94481311616464419075825632567494453e-03L, 2.00100938779037997581424620542543429e-01L, 1.17328605653712546899681147538372171e+00L, 8.18356490677287285512063117929807241e+00L, 5.59865842621965368881982340891928481e+02L, 3.69902944883650290371636082450503730e+07L, 4.05747121124517088709477132072461878e+20L, 1.07723884748977308147226626407207905e+56L, 2.07250561337258237728923403163755392e+152L, 1.09889904624000153879292638133263171e+414L, 3.02463014753652876878705286965250144e+1125L, },
      { 3.16339947894004847541521297710937343e-1515L, 7.02786763812753004271170107158747593e-558L, 1.08465499866859733494552744914276656e-205L, 3.97220624097731753646493300738183748e-76L, 1.84145280968701148636501796762008815e-28L, 6.40886341101610144938011594904313817e-11L, 1.89200275395722467663251234615351291e-04L, 5.04892743776943294143478969573235061e-02L, 5.25946553709738448524487603870477588e-01L, 2.72635049439566736308739858009584752e+00L, 4.20081657572145199006060015525396829e+01L, 3.75221714819472331206969009243353365e+04L, 3.11642465424592079726559332629050577e+12L, 9.61947229245436139236957297153092319e+33L, 2.42007357898805628851519620917402234e+92L, 1.34351470989086331111432001407522043e+251L, 4.34937560839564995558174638985438726e+682L, 4.01820305474077838467315109614612109e+1855L, },
      { 2.20630426387562582340901270736066292e-1945L, 3.20902289525796097195876526710905754e-1180L, 3.99075419312896749376535004306952481e-716L, 1.22680463935505388461578092999320797e-434L, 6.60416836081631149205027836695167940e-264L, 2.36380363265905808053031227647927829e-160L, 1.52216180146092756550894692311532796e-97L, 1.89865679919908959317123677189779258e-59L, 2.43020115474101871630434504433731237e-36L, 2.52466874690605714759066961852641028e-22L, 8.05169965363444223580324196979926112e-14L, 1.16754399126294292055540155961164512e-08L, 1.59887725419859990767684898940272339e-05L, 1.30653650105064376247657363109776765e-03L, 1.96025435814529675467866513121889090e-02L, 1.07798510399525035644857486110150806e-01L, 3.35726506001932774135392168623644982e-01L, 7.91344191300036661743190713036639943e-01L, 1.75712551051579342077493249233109834e+00L, 4.50532922519182663928094313196688277e+00L, 1.69634912879730950991827009814964854e+01L, 1.31564914034011933527149556208430823e+02L, 3.54551665937177335700906931219692079e+03L, 7.69629182688413474235641078406238234e+05L, 5.31334781541946297455601223275400364e+09L, 1.11397125403497802554344014693570895e+16L, 2.91323342059626628322620905024841007e+26L, 4.33470414741675827465254900104887599e+43L, 8.87077939346041258291094819287939414e+71L, 4.20552510572288598564618972079522209e+118L, 3.78998758665619810759092788424597102e+195L, 2.85285821381079277498378752581228807e+322L, 4.35767992687500025738868152293904617e+531L, 3.35101732969916400190150631384628168e+876L, 1.39907636805339624393396738787997778e+1445L, 4.39310839133852761600954729262867435e+2382L, },
      { 2.60416027838200926634868701489352534e-2204L, 7.02786036162213986704884019362612079e-1717L, 2.87472449554862557122883045361480216e-1337L, 1.26052689335617094297502885256721406e-1041L, 2.22203737723769125613395083355625930e-811L, 4.59925691253299141912321586723335574e-632L, 2.06110387430222811031505778840893135e-492L, 1.18783333719310840737835079091916047e-383L, 5.99339963191580129696690680497340117e-299L, 5.55156577871295836646608994333019205e-233L, 1.31650343182343962237291034126486362e-181L, 1.35017224106781623208439493207058713e-141L, 1.95458267270042896122641878745090637e-110L, 3.62345406499123808121116200173051239e-86L, 2.87916036185197771976746396107338099e-67L, 1.50975853574758038650941626753584613e-52L, 4.39366381209590686916061999351406334e-41L, 3.72613279781933265810269222029431206e-32L, 3.35074541708050784063416140930256908e-25L, 8.73699691100611067810340427391712629e-20L, 1.44610938754441658588026566732900630e-15L, 2.79712542830915646153531586097835651e-12L, 1.01712318476608889609455916646557919e-09L, 1.00707686902351812525332969334486408e-07L, 3.62377064535647765957086044101238700e-06L, 5.93514765312557852887858990273347709e-05L, 5.27427922038425038994892508697801291e-04L, 2.91716284057748187538183216214780172e-03L, 1.11812753052373081344852811278990196e-02L, 3.23158910999791296389217134204252891e-02L, 7.52823576293971217140453145701166400e-02L, 1.49072381771447883955117212578891471e-01L, 2.61920116954195648982059938116700281e-01L, 4.23044722849733503488984642521144095e-01L, 6.47337815325830827783598163922905049e-01L, 9.63850663681748439752705317539095691e-01L, 1.43180513983798487645924505146418541e+00L, 2.17546288127550398262362304432746644e+00L, 3.47077553215380191942193887548951574e+00L, 5.98762125926090985873603940556086485e+00L, 1.15588346593765285104362265791852722e+01L, 2.60227165873113109321939526707132515e+01L, 7.18972764924010846908346544069757730e+01L, 2.59849827899514040038730358528910272e+02L, 1.33180383652908565595482681440747426e+03L, 1.07264199927746293597346233164919561e+04L, 1.54770252854197591129695964624494039e+05L, 4.73116974059692035499030617203501902e+06L, 3.79850299532978550624970142683299187e+08L, 1.05506900281875210383901752883017567e+11L, 1.44377693764054834189876938092028936e+14L, 1.53216987231286586179401429003390945e+18L, 2.25749845424265693444972830430363021e+23L, 9.75366126404791278425650801377275643e+29L, 3.22710662318561074542283240061745423e+38L, 2.80427989450823486864584373674252213e+49L, 3.11492865697270427645920832671970699e+63L, 3.37385880956075754428852062324609716e+81L, 4.84069913749095116306098688426537434e+104L, 2.62206965911556490036221731139595538e+134L, 3.95724512624896651389226023882699109e+172L, 4.16658233419696139622872515762144259e+221L, 3.67870997815873300310505495683155714e+284L, 2.45353561048417937411865685638919263e+365L, 1.47890836767899626666102902312593029e+469L, 2.66842264577698782467085710268227542e+602L, 3.39406520823137428862537121090858482e+773L, 1.71072929388916771882465310676448508e+993L, 2.17124413154098944294625847065193945e+1275L, 3.67108839134079465802181460380664133e+1637L, 4.72992162468569560939225829187204387e+2102L, },
      { 1.98007625288857013412122004727810624e-2346L, 8.39652992194258659266745309526472882e-2071L, 1.46030671151231956105710426802408027e-1827L, 6.65727248909208458885310717390128801e-1613L, 1.81581117219394388092797723844111009e-1423L, 2.72619674739250219990613941747932927e-1256L, 9.29590026994488020433315206443278096e-1109L, 1.46377917106366800430984793284730939e-978L, 1.15901956513202291149840314964756250e-863L, 2.89578802021929239619753606490350524e-762L, 8.80858736107764685086031513644364894e-673L, 8.19416690740938676449696497280686335e-594L, 4.00916745706227520865721440095991259e-524L, 1.27020033549323657895766092175564312e-462L, 2.38869695900235565316537719061693975e-408L, 1.88391475973801739487501436111488490e-360L, 3.49894275509125434733587030489899389e-318L, 7.01615886164307390146523314851863544e-281L, 5.82310443050787035609701710903434681e-248L, 6.54836166711216765944679142249433382e-219L, 2.84149263299034989667159925017962142e-193L, 1.19812648186130338854458247959762818e-170L, 1.10911490518050678588403791608480488e-150L, 4.62754597695358582538782054737574180e-133L, 1.64173210211161942076907089038989838e-117L, 8.67192889174269982709959745767681981e-104L, 1.11812225550444523531457194417899189e-91L, 5.44369950217028498211921027217139012e-81L, 1.47074883585505403248443957516006445e-71L, 3.09730398495823549037183157408809652e-63L, 6.86204181060118439675162432029019608e-56L, 2.08386500906124109876681630886775453e-49L, 1.09560045982532469724721792988430545e-43L, 1.22547627904244500951228300724497066e-38L, 3.49796660995417261286766404577301528e-34L, 2.99152410870631960427539566799345990e-30L, 8.83190101026086766963628794276374683e-27L, 1.01998228741819737621697177531967545e-23L, 5.14543767526486873178240241354085638e-21L, 1.24976972746216000788747786633668576e-18L, 1.59270820536117731616569419326771375e-16L, 1.14889208313232540746001768184091876e-14L, 5.01568630936388404881919116817433643e-13L, 1.40588174426346693556351457090253957e-11L, 2.66552656620728447371725338328298653e-10L, 3.57952342849715748848195322266581664e-09L, 3.54587967892367612935335788099896843e-08L, 2.68571894340447967655761437829800783e-07L, 1.60543396069231405992036918112042239e-06L, 7.78876789102767893920047052886009844e-06L, 3.14354940320849664611373818938001854e-05L, 1.07876626606295756489869617036039985e-04L, 3.20898851002890606916875803454736038e-04L, 8.41685517064122028463642895198591848e-04L, 1.97617054782608049591289962378898411e-03L, 4.20911897696440311235672067499450921e-03L, 8.22976828962407304914730770011736102e-03L, 1.49268961502975158975071505963565168e-02L, 2.53502658698472066372315325797964553e-02L, 4.06466974195663810865864410951160977e-02L, 6.19877276359776967830177840166714717e-02L, 9.05089155125712182450563511209635860e-02L, 1.27278213682114666289711516555196299e-01L, 1.73304699976842406023992088378438619e-01L, 2.29592410233034978491448000870038137e-01L, 2.97239599113018852567831136709176961e-01L, 3.77582948042641879160927817217899876e-01L, 4.72389555885863401785327685882008067e-01L, 5.84107138056041651067826639644019633e-01L, 7.16195243265456514286143145222900703e-01L, 8.73579162273458922635006193850777630e-01L, 1.06329740288293038148703202109279326e+00L, 1.29546076177954953883340629341283024e+00L, 1.58472055402981935372999954552668421e+00L, 1.95258275532953319986830507945836181e+00L, 2.43115638685977475925559253395572515e+00L, 3.06939317426312452000366876337463330e+00L, 3.94377618122435031881432297693345182e+00L, 5.17720373327574287465684980172077361e+00L, 6.97349551419502029130075958419090062e+00L, 9.68282046752329620408323930717367162e+00L, 1.39309131038991828857956159913232231e+01L, 2.08859990746605884557374483895633501e+01L, 3.28382897425881117409618357674054401e+01L, 5.45293246810142904942496502744410196e+01L, 9.63939781365248203122153852027923369e+01L, 1.83025185098871555179881958564576181e+02L, 3.77024517378370245776498953575447428e+02L, 8.52190226688445340278025693774183776e+02L, 2.14068512989192644263700461005007744e+03L, 6.06284853067739175777069687971114492e+03L, 1.96780457938951378915777196082473423e+04L, 7.45535092385462432927423608234859698e+04L, 3.36655497164547806089503954112397178e+05L, 1.85514609429466771541276069582993197e+06L, 1.28127256599195512602695379572162760e+07L, 1.14315867886785361488804976072737382e+08L, 1.36346329402339162906880444623714674e+09L, 2.25993073791019766670398383679728111e+10L, 5.43926612995997228479509745730508310e+11L, 1.99798484325959673290782033498623904e+13L, 1.18504970239150114123104110601445669e+15L, 1.20980370818215194207683618902723859e+17L, 2.28537992087984292439473774950290832e+19L, 8.67120977340450400779923403171032446e+21L, 7.25153649943518021867879355500893988e+24L, 1.48501435335333039333736586197814469e+28L, 8.39050326550867736343407650659898774e+31L, 1.49728470198356221256669260283197052e+36L, 9.83549601762784922530731490296260394e+40L, 2.82901352112032614706131527899887976e+46L, 4.33736659101971830974347248793299740e+52L, 4.42940467871587853569566546424045697e+59L, 3.87840248690238104186373399976301795e+67L, 3.87622435037312042010986878431609700e+76L, 6.11530088968567983241864570883470440e+86L, 2.19902986720444927672673080107769693e+98L, 2.73302501743809592753235404592828204e+111L, 1.88157705379416554346750034182897643e+126L, 1.22464500548199778033960287699262627e+143L, 1.38089682931886763280665214047107116e+162L, 5.35861724121451785967201429765672158e+183L, 1.55758090841089261682638940540906561e+208L, 8.18640878839858943416304034752205938e+235L, 2.11193471456685131226350980039759737e+267L, 8.29196100559631320106113784120111420e+302L, 1.78609329205356222554295903766141624e+343L, 9.02504022060557635111807879176244000e+388L, 5.55060710318615757165502584223064764e+440L, 2.68431690505806342057709029465018309e+499L, 8.45412788268345327632392585778624037e+565L, 1.90297947362180093404817251840979577e+641L, 4.62215189323912708971592873336189679e+726L, 2.62533210543270812917133055827838787e+823L, 1.13817822812561602537416032879617414e+933L, 1.95538622136936887385682087493096654e+1057L, 1.16937064578842112561072532780020718e+1198L, 3.88035117517281013054730623266703196e+1357L, 2.23743584154220985565150075331030128e+1538L, 1.50889533002797316376499137963913497e+1743L, 1.90641621776286179122749804749534564e+1975L, 1.93080249271305970633007359089743998e+2238L, },
      { 6.85586909881832886951451464772591878e-2421L, 2.92677267243682665120741515792112744e-2274L, 1.63242588944068781532732923861018809e-2136L, 4.10812064897186984969866413043968829e-2007L, 1.49435786878493820312929790405374745e-1885L, 2.34568268208591413691385316271709680e-1771L, 4.43925157300673395426635808699168955e-1664L, 2.65927633627471464278071651186161190e-1563L, 1.24859182507079754991269604284796564e-1468L, 1.07699173095552638424762549404812646e-1379L, 3.79889818022139862967145673157136925e-1296L, 1.16203657951916401289541501876509754e-1217L, 6.24572482158628957114104388658639197e-1144L, 1.14511680730607058597658427712436799e-1074L, 1.33557521045192075572641149916774170e-1009L, 1.77946124922921650079213398130306155e-948L, 4.69414096094237830969243170096712625e-891L, 4.11003931775404922019148615555751563e-837L, 1.94061218235885716000492708678051285e-786L, 7.79551077888128366830436599420622304e-739L, 4.08861993672760936738017704029701813e-694L, 4.18676490857955209780618740974196173e-652L, 1.22153592554301240722464717581112377e-612L, 1.44833892014264382948310395651753089e-575L, 9.74176764355703592877272264974985856e-541L, 5.08509650174353124708754662233746797e-508L, 2.76502760440391058924306176438285126e-477L, 2.06508450540542778211999232986493505e-448L, 2.74686348165461297466693117700925333e-421L, 8.30585680824802634886342521419101161e-396L, 7.18035163161661527310024383149797653e-372L, 2.20117367274887299835421816737413275e-349L, 2.92938701204713835430829657327314622e-328L, 2.04670856065666661814970623236244653e-308L, 8.97499107305077146284125789528155163e-290L, 2.92115595376553501154165826580135083e-272L, 8.26130614121671756652659688034185753e-256L, 2.35396777618554392325312768878283221e-240L, 7.76604519643439440768615254501273896e-226L, 3.38047616528897968517094694597452046e-212L, 2.19496817732884969215875687943120973e-199L, 2.38570033705332693932010529683394408e-187L, 4.83696042571616203283798802745975902e-176L, 2.02525789452808699182428060822541636e-165L, 1.92683544261267768613934270120773225e-155L, 4.55675928208753416426295293555087690e-146L, 2.91435926363522943509016642164959990e-137L, 5.45650834446039875776911937813876647e-129L, 3.22177915240208624055473617062236208e-121L, 6.43356918992122712617506763079342635e-114L, 4.63991207894245637238928734792612888e-107L, 1.28548526430585878202588260968441525e-100L, 1.44976784442529508497784801483631129e-94L, 7.02834477739882506904444529125607921e-89L, 1.54151343887499654318252629091310298e-83L, 1.60491367376894945627919582278915480e-78L, 8.29797555416253956153582009155421152e-74L, 2.22293580147262466985937857298017334e-69L, 3.21083741325090217758867273653191446e-65L, 2.59597447976318059491815288096285983e-61L, 1.21686551839843562559756380386148224e-57L, 3.41813412178077321753808876760492064e-54L, 5.93492497756373178397326564626159285e-51L, 6.55812810449229009165849681654564859e-48L, 4.73996462182817624913218344729897191e-45L, 2.29917713906026251753218108658989939e-42L, 7.66772810214285848709307613966489982e-40L, 1.79852699731596078211094021208953882e-37L, 3.03099351897543871174212829426305144e-35L, 3.74430827279655104510043134133803179e-33L, 3.45501893639921538125092697393537600e-31L, 2.42381852080187080923760278831274486e-29L, 1.31441976082623542058289755424927250e-27L, 5.59666006060409162050973561377317535e-26L, 1.89867556602582040862526655529336748e-24L, 5.20331508297836691753631952262461629e-23L, 1.16690464600934423260878649807673403e-21L, 2.16767714527916659623320357151600374e-20L, 3.37377643107659326572426331386135943e-19L, 4.44691383264786489190950322799097537e-18L, 5.01415830137739970673756811085508269e-17L, 4.88253693365386298188724591254681515e-16L, 4.14251016844320187955159569890123671e-15L, 3.08802476085892421415530065712870862e-14L, 2.03847832924953919920668690856932749e-13L, 1.20044498484990029827708442066317039e-12L, 6.35035779339988133329469515026722103e-12L, 3.03739182163512379493334464659410947e-11L, 1.32161843156591616383150953292325245e-10L, 5.26149741865431306798513845101472462e-10L, 1.92688222163920338777595529204662553e-09L, 6.52448684765263503483691364064710302e-09L, 2.05235608001812174069568432464532832e-08L, 6.02447248155606588490206026101601127e-08L, 1.65719156589179965185098196474263403e-07L, 4.28875221376115411615543853081272079e-07L, 1.04811127032430288440405169279023586e-06L, 2.42727195841237101271799656914039580e-06L, 5.34428013249275030874689017597357100e-06L, 1.12217116002251908227007056847369186e-05L, 2.25365270095215311537786875281660527e-05L, 4.34066412230525728790156860647261934e-05L, 8.03857428545025320920443714330221510e-05L, 1.43483273166998782632422720673552674e-04L, 2.47403670532944916561679376448099333e-04L, 4.12969671314554699465069528559056198e-04L, 6.68662248079464048176726608426967422e-04L, 1.05217959874444039957269021715673554e-03L, 1.61189482479878719591766861379534126e-03L, 2.40808122992764072147410965445120710e-03L, 3.51377854902820551940988678116833582e-03L, 5.01519359256763066459806052313013320e-03L, 7.01156300574609092374647377614547795e-03L, 9.61445020754398604050247376189537687e-03L, 1.29464977958074216006936641439568225e-02L, 1.71397050059386052582575161031849539e-02L, 2.23333418628568405596661137294311321e-02L, 2.86716433323270031012958633821643459e-02L, 3.63014455768061096476987670693482715e-02L, 4.53699316668876641396409342510559075e-02L, 5.60226467568397916052157799832294784e-02L, 6.84019250622201277408275217514564750e-02L, 8.26458460699460598611997466316916292e-02L, 9.88878087044773835985179298425629188e-02L, 1.17256783027063660721874825783310801e-01L, 1.37878272417301116163817842123251189e-01L, 1.60875997439806117261603052596128705e-01L, 1.86374297424717578638710069246872277e-01L, 2.14501238238148719030968730766268227e-01L, 2.45392415301662505713896222647369518e-01L, 2.79195449773929877272191161144448290e-01L, 3.16075192872379294294649481827620946e-01L, 3.56219678521249637323233310278349444e-01L, 3.99846897980088704601361036413251085e-01L, 4.47212512313163107391085998119621599e-01L, 4.98618670533294760818150011647972671e-01L, 5.54424164764947975414033158173924804e-01L, 6.15056231563286401803829984090534551e-01L, 6.81024404595688995229840589926798114e-01L, 7.52936943869155645889792590166619714e-01L, 8.31520518350208659550187071679434118e-01L, 9.17644012826577357641491486976575868e-01L, 1.01234758075340210126152790231895045e+00L, 1.11687839251578850642753265260268328e+00L, 1.23273496036260391788838218839806886e+00L, 1.36172249498191084612564077792923468e+00L, 1.50602251678823434544288956361434194e+00L, 1.66828098096960364511379075299209308e+00L, 1.85172058286684745279366828495999051e+00L, 2.06028483669890596285992438994232879e+00L, 2.29882417717996662881312374953444179e+00L, 2.57333802530471722160283306099247239e+00L, 2.89129193110240878387703249643229033e+00L, 3.26203621106764094357461881213752464e+00L, 3.69736290590815315504600654149629845e+00L, 4.21225284743951568672550404261498190e+00L, 4.82588633844219080687269426289849605e+00L, 5.56302277261292337309431040275971592e+00L, 6.45590163750149736955774273721315509e+00L, 7.54688784770818103174495320257408374e+00L, 8.89219103984228343085851433153285412e+00L, 1.05671517790393183669301386672470668e+01L, 1.26738407015152894733455767759949041e+01L, 1.53521137941817792315134763759132043e+01L, 1.87958986899048219790124225448671683e+01L, 2.32775055780405432294765542807163514e+01L, 2.91844233861930596156964126095351325e+01L, 3.70758319218904582325116346962442845e+01L, 4.77707378224399722392124081823831658e+01L, 6.24876734446863447811669818713989778e+01L, 8.30699342763174311126855611293197423e+01L, 1.12355321585737491915090874360361047e+02L, 1.54794728437631233369474093491065001e+02L, 2.17507985417556811263254275570301936e+02L, 3.12124586781855603508049115425717265e+02L, 4.58065302025763509177549206574829939e+02L, 6.88532496785780240326739257222869086e+02L, 1.06172181511411400423407908864360706e+03L, 1.68236894049421021674529061389284712e+03L, 2.74433484749143274695935635590635551e+03L, 4.61730673523479769409382001763470205e+03L, 8.02895513401715463422140861600391503e+03L, 1.44606187348593941052176609549649644e+04L, 2.70377620144727936743806059397776390e+04L, 5.26105241201059109738487092327866872e+04L, 1.06812531849740275922510876100737843e+05L, 2.26893075168541256271862004134549591e+05L, 5.05764404902608855969935895677933433e+05L, 1.18676186494579079987238943033223000e+06L, 2.94113264423683227561198247386975731e+06L, 7.72576514719998793452586494715302612e+06L, 2.15915178528480833880909472948132169e+07L, 6.44590226372788402030541979463995492e+07L, 2.06442545099697944602613003602586209e+08L, 7.12532948492900300739824830002676479e+08L, 2.66319659068655507734050368458244708e+09L, 1.08350636970002739580921292680640244e+10L, 4.82470799147337538737281851702596161e+10L, 2.36513804063572720852686335350104312e+11L, 1.28435637164102683934687705650167990e+12L, 7.77731246565628041938451985303773831e+12L, 5.28864903733985311795604320721713645e+13L, 4.06896744489041480369429188279659070e+14L, 3.57034809188328432417142254050169691e+15L, 3.60337498222976618360402297528737217e+16L, 4.22089025190487061069708630018503728e+17L, 5.79389086782171589014524617169039021e+18L, 9.41571436923218772663826142877596428e+19L, 1.83140546580432476725446560201754697e+21L, 4.31320926121717399384742067577025671e+22L, 1.24526713689819970862562132007699682e+24L, 4.46557396336452476527850690333252885e+25L, 2.01706317120607297931541128182425236e+27L, 1.16481109175988266234418168600383172e+29L, 8.73712441785116756599086490733480008e+30L, 8.65736270143027267976956887405444609e+32L, 1.15373549848396029384279199583578574e+35L, 2.10781669532016374773265262680619511e+37L, 5.38769637251502198497400987247791395e+39L, 1.96890474908610530032963280794536401e+42L, 1.05271764511336947295087542840771762e+45L, 8.43965629752558882239469624058575644e+47L, 1.04138369598852386365061187570128647e+51L, 2.03352356915167672502196854739506820e+54L, 6.47289125189110545523503523387311437e+57L, 3.46613548082834986367032599943319188e+61L, 3.22894792541599068853355143560944151e+65L, 5.42318759743953119742239523201238475e+69L, 1.70583228507675596981951702618714711e+74L, 1.04635953488687800354356406620895146e+79L, 1.30676249978604401490579288070545190e+84L, 3.47855004888451734894061507389280036e+89L, 2.07245356750112312931588098931095741e+95L, 2.91086857580213998298216107061151403e+101L, 1.01866939773917036897104863230086276e+108L, 9.42079727758602983661368103980660631e+114L, 2.45135270885215193905442997104007131e+122L, 1.91850621913423378510904034515156001e+130L, 4.84846348541576375648253727961523398e+138L, 4.26738728648259195430121429627940133e+147L, 1.41771213016857192102291892633345650e+157L, 1.93681124067818946208050336019146634e+167L, 1.19196764675833722929285298195873591e+178L, 3.64148488428297947856471515841164734e+189L, 6.12360784372861046853690882962104289e+201L, 6.32737167744109620998113270972893770e+214L, 4.51626866922887039378811138276232919e+228L, 2.52237535488279323839235752907864107e+243L, 1.25874430868722260548214869831621228e+259L, 6.46403890833514736730418683378673138e+275L, 3.97014109807165189460577113122885913e+293L, 3.42256445909004961984480518463414679e+312L, 4.91057220648194827953651882481242793e+332L, 1.40575745439889948649867992022706415e+354L, 9.73933590848593290516589462953654487e+376L, 2.00557619166308496177559022371882184e+401L, 1.52772859099735075822031580257311058e+427L, 5.43355654783039835976501066730115367e+454L, 1.15613726424969615026074606149058824e+484L, 1.91611396577413915548297095470351353e+515L, 3.27574835998524173295312535664414153e+548L, 7.78995993645646383749425447261756829e+583L, 3.54265432942533120234904548837850071e+621L, 4.32354359483087940297488323505450874e+661L, 2.03098786518797659659581103998046513e+704L, 5.39103916237599207670086450736143491e+749L, 1.21683151125684538902217928380735342e+798L, 3.60847574304211696962789044183995856e+849L, 2.23398800758266399217702784712099052e+904L, 4.72717380839903008268507408818545095e+962L, 5.77822208457179232925131147188581785e+1024L, 7.13289562956799416571309267253574621e+1090L, 1.61164858402955200792510210236524128e+1161L, 1.25521313739368945437360950048953572e+1236L, 6.61069623781251502357557332678377193e+1315L, 4.82362056335119095907952363185253207e+1400L, 1.04640496911016000942284610597050358e+1491L, 1.52131212817898241350510416765362487e+1587L, 3.52116090648535956154142297718505774e+1689L, 3.25907088535366768900129651211398670e+1798L, 3.21536840347133363006114569121938775e+1914L, 9.60165908424805663006527683476758652e+2037L, 2.63585872900260288034767508588870333e+2169L, 2.17048793513302111330851937361254536e+2309L, 1.88788443108195661060841368444711468e+2458L, },
      { 5.29693440729788041213142409892436291e-2459L, 2.27629257218329628557497350447858260e-2383L, 4.60725896612142987241524079465629916e-2310L, 5.17919364499500847985920034617781519e-2239L, 3.79383002964804382517930061580447157e-2170L, 2.11419993680434447426181517759297327e-2103L, 1.04148667561061807899012277964691785e-2038L, 5.24544425651958823988931435014729457e-1976L, 3.11006352777614281598465337503705373e-1915L, 2.48867462987012194623347147546568943e-1856L, 3.06835915872226280445223729961288427e-1799L, 6.62736493446143254705278263772087543e-1744L, 2.83997245953224047644578590378179268e-1690L, 2.72398780252406063855606534842103674e-1638L, 6.57327304158814917854156231336457877e-1588L, 4.46940189941144606938097964305720201e-1539L, 9.55652954181188776027715447963783638e-1492L, 7.14757266175076046071660887379600587e-1446L, 2.07313155515958333906901076011055929e-1401L, 2.57708633795358827890276221215294553e-1358L, 1.51269996990649935807792458579987758e-1316L, 4.60565435951356389119181468428252427e-1276L, 7.96677448800758121662225971905999717e-1237L, 8.55160853918795880988712436880140365e-1199L, 6.20482659749393290517542190337401836e-1162L, 3.30618935612498022733666989541183557e-1126L, 1.40195518332652973622801089378870590e-1091L, 5.11407919863367919558746085806786902e-1058L, 1.73063614614270605229830917360912415e-1025L, 5.84546019976429168954160400076820148e-994L, 2.11542888104356332114006186983662541e-963L, 8.78597020474401601699260325757052372e-934L, 4.47629976797445775500570938357786257e-905L, 2.98416839309444731431172458639558207e-877L, 2.77125321384854968458662765004085943e-850L, 3.80904190342493701536240175462301111e-824L, 8.21806462726393678491860107910281889e-799L, 2.94631935053803405188027155144659836e-774L, 1.85492994927395428856242032790235018e-750L, 2.16349445690590806097153640474491227e-727L, 4.92371233300030441739039250387920817e-705L, 2.29918059518632617935997695502963251e-683L, 2.31291758268743951509849171359112934e-662L, 5.25491742744220732604319355531444736e-642L, 2.82274223509188430651059911113688649e-622L, 3.74753227935082714244593814667510926e-603L, 1.28370364951438380618871579660566498e-584L, 1.18285412035024318029749091456416413e-566L, 3.05273754301598829645261576934311186e-549L, 2.29479910590203595225694928967822608e-532L, 5.21889625493118727969519850005753886e-516L, 3.72534255592436984201074827726301397e-500L, 8.64949198440528351465957047285602117e-485L, 6.76174414760667924982941341637985863e-470L, 1.84041518883114740241388396417281333e-455L, 1.80160472793323877219953443455302799e-441L, 6.54566528304067135832221898507534820e-428L, 9.10001699811829840956092860906810074e-415L, 4.98609828016939856686029944862830983e-402L, 1.10802830298400645391731986287163178e-389L, 1.02676405187821370308474571067658508e-377L, 4.07575091116228204031238856923140440e-366L, 7.11360268352693191010849541271753635e-355L, 5.59881168833151588006581688986986730e-344L, 2.03642255515559150564975951817039221e-333L, 3.50525657096789033370540964985470150e-323L, 2.92178573100086478331989127444273654e-313L, 1.20598734041934315977832234578459915e-303L, 2.51880216671823764300316483881856209e-294L, 2.71834421831894383133482530254389304e-285L, 1.54702039109098756507051056035620614e-276L, 4.73499485141563990358940703275480353e-268L, 7.94442519500188390796900278354012024e-260L, 7.44316376023327816606172273381802393e-252L, 3.96451700998508534828226048217863900e-244L, 1.22153684948784264379851755556819940e-236L, 2.21421725154084964507841507830563690e-229L, 2.40004857648572823842789487119416272e-222L, 1.58043505420313498164518700461984906e-215L, 6.42021223167302856767365812271085781e-209L, 1.63302423264111059580848381300534588e-202L, 2.63853106939137106330498547542524566e-196L, 2.74613250302397818664806479526069745e-190L, 1.86615306707995508392565990157579707e-184L, 8.38948945233194510175234920774873796e-179L, 2.52701050376399124391991347402621144e-173L, 5.16312575535157598596140018726440984e-168L, 7.24167062135448326922272784801743108e-163L, 7.05361814094865509813912299190351845e-158L, 4.82507740170943565549056875354725391e-153L, 2.34335421855805662819938218222415100e-148L, 8.16563163629951985655858859925168146e-144L, 2.06250908768935143871108361256064164e-139L, 3.81378121105029755974979453855090030e-135L, 5.21238862833226048821040688112783913e-131L, 5.31469066325781546457465599778122553e-127L, 4.07938032086884338695136986806703742e-123L, 2.37782434978024084372113156401306143e-119L, 1.06148128500673821430527965352560854e-115L, 3.65894930935314933069457000324241470e-112L, 9.81672760779301769095882197030988958e-109L, 2.06581729538829312212171371616525011e-105L, 3.43540071960972258090881112115251093e-102L, 4.54745983632894201410075262868047927e-99L, 4.82519857468149557373657687371350595e-96L, 4.13210577018135888633811621378125178e-93L, 2.87476099508953335771741680746804464e-90L, 1.63524251388290882625725240042223140e-87L, 7.65249997733887999623874860707951771e-85L, 2.96396516998940431058211305683891581e-82L, 9.55694449812711903241890069159516005e-80L, 2.57983003461536294556125432978470487e-77L, 5.86224102303822793749090237239965643e-75L, 1.12729666204663539107360352976065345e-72L, 1.84393400012961666333357387323541892e-70L, 2.57838118497774645351182154219344361e-68L, 3.09698398084623291081115732152399867e-66L, 3.21034639994569504125984658412826677e-64L, 2.88505745240234032993466930110171709e-62L, 2.25763973385675919792268556682148533e-60L, 1.54490468182644383657455478866541800e-58L, 9.28299236822216106223683331290050646e-57L, 4.91757270567151153434520348832876307e-55L, 2.30554641627582419897786323384840309e-53L, 9.60260856639165286607071389462405742e-52L, 3.56597785391661967741537394944785975e-50L, 1.18488237523747100855696221290216872e-48L, 3.53480106239996687790067585870262148e-47L, 9.49922331635571479300328969906674959e-46L, 2.30696251922047363681394459290289348e-44L, 5.07896585888252846067820035747838002e-43L, 1.01672554003146516218461940856820913e-41L, 1.85608083837358412341292426322275694e-40L, 3.09875103851653509802894724036653659e-39L, 4.74424556091727158516159565756659137e-38L, 6.67875654292885708009556323027970307e-37L, 8.66749789110265824007798499627002101e-36L, 1.03955806472296089132115323650013375e-34L, 1.15508617865206361238561969897526423e-33L, 1.19182362291753977427429758189318101e-32L, 1.14454132981883615336894355014782203e-31L, 1.02525602738123519983996836134381910e-30L, 8.58508308406581287405862446386142355e-30L, 6.73394164670453777712343530741256529e-29L, 4.95770293303240892202540668612526963e-28L, 3.43261200056988540268170912603591182e-27L, 2.23935379461627788150957730520597272e-26L, 1.37901808820501650857276186986911719e-25L, 8.03040544770876549232435915566049266e-25L, 4.42968188137908996064235843650469764e-24L, 2.31845927113168436195559586507393831e-23L, 1.15324161925721516540799043047625888e-22L, 5.46028729667908667672130073109459001e-22L, 2.46458926739523684644781998993375925e-21L, 1.06205430707170637462172218092564615e-20L, 4.37564008882607318443071720026076737e-20L, 1.72595587803323990868164488937090111e-19L, 6.52669144636334492287282783537910032e-19L, 2.36916844627434713681577417830597819e-18L, 8.26580372697482904251900693228762290e-18L, 2.77517606391661360182153963773908766e-17L, 8.97689232444592868369906653940499583e-17L, 2.80084735231662190309995026747244972e-16L, 8.43846463133099160645680685207593959e-16L, 2.45762595435789224473736552077118113e-15L, 6.92627769518345222522404766069340383e-15L, 1.89084205236464652796008005489924701e-14L, 5.00504297304156635993817971340426042e-14L, 1.28579120925883494201961228234875929e-13L, 3.20880531681575127155908954236910927e-13L, 7.78600100470787821926138878185058803e-13L, 1.83848331026111978204532135720140678e-12L, 4.22808302441074119355483439549639212e-12L, 9.47805306883598889922905503268261293e-12L, 2.07266429754356748908920713182886782e-11L, 4.42491437115776228499841800855357065e-11L, 9.22929507351999755945043065016679100e-11L, 1.88205258669115539986830146476840153e-10L, 3.75488615259231157512941964482754887e-10L, 7.33426418139309264953064008778652175e-10L, 1.40344387177481383433616336399570262e-09L, 2.63261606277390900911399983420396976e-09L, 4.84396275737187249536387241410506136e-09L, 8.74769197387686182490212973737486333e-09L, 1.55137319662316143322632050708021849e-08L, 2.70341183770391766500462033783635805e-08L, 4.63144836233962351361136455522914469e-08L, 7.80474058793106802102732603067941163e-08L, 1.29437017686516812026215434301661584e-07L, 2.11364219596533096486660492476571897e-07L, 3.40005066401716435610198303924157253e-07L, 5.39041104673862946543865026660608261e-07L, 8.42629030758144765430007132771995573e-07L, 1.29932702981307401261989221953085056e-06L, 1.97720517756199603296504985814152326e-06L, 2.97039557774168150435589567759708661e-06L, 4.40736236333866783037595222091818751e-06L, 6.46118993513603067322889724548764197e-06L, 9.36219732537340456332983881649283088e-06L, 1.34131848415120895981721422476655149e-05L, 1.90076038344927799241882635296823326e-05L, 2.66509095957072382671851701769041584e-05L, 3.69853096371186063639032612868033323e-05L, 5.08180543166657948433862427398171792e-05L, 6.91533341923576665290801314459185869e-05L, 9.32277498618928819410507284882673596e-05L, 1.24549207625185292746788896109936768e-04L, 1.64938971333376144939513069096604597e-04L, 2.16576471380809909291159455336007140e-04L, 2.82046340794006881311592802174739018e-04L, 3.64387021107897729238216699262593180e-04L, 4.67140162762043149791404714119969181e-04L, 5.94399941612237251611663813916878716e-04L, 7.50861330003505141341237132619730559e-04L, 9.41866302231455859084164368912569515e-04L, 1.17344692380002247672609547026883890e-03L, 1.45236427426188008347616119045966424e-03L, 1.78614218598655178612808460803997991e-03L, 2.18309484603519656221136749899809853e-03L, 2.65234740423109052289144172110269167e-03L, 3.20384885506921527761765779516587618e-03L, 3.84837661076576821141977260794172255e-03L, 4.59753235303186248996039969885789422e-03L, 5.46372893686621665171770797799402382e-03L, 6.46016831511747979221062326266402012e-03L, 7.60081065185419977059145624254178297e-03L, 8.90033498980204155946674089749748970e-03L, 1.03740920266168385556590451013464350e-02L, 1.20380497313706427480839156336456773e-02L, 1.39087326917827169960002061321148754e-02L, 1.60031562206455496475142539245814110e-02L, 1.83387563636593926316708603595489832e-02L, 2.09333170384958330449215168774593401e-02L, 2.38048955952869498234626954784270233e-02L, 2.69717481217077174816927863942958635e-02L, 3.04522558220532394622866431653207434e-02L, 3.42648537806332947261888782628446295e-02L, 3.84279633774899765386527022493279515e-02L, 4.29599295614980634381395647366200859e-02L, 4.78789641053477195533528647962166054e-02L, 5.32030958720316323100756360890520466e-02L, 5.89501290167188399227102430485391475e-02L, 6.51376099347951026087494855511522993e-02L, 7.17828036498272120091084808533476670e-02L, 7.89026802175633783409928711856568346e-02L, 8.65139116068936794825248092013403649e-02L, 9.46328794087702664882628072944679318e-02L, 1.03275693621920814402887268719345915e-01L, 1.12458222671938515285380933743314032e-01L, 1.22196134739810167085016527168685084e-01L, 1.32504950408621397326604347561435549e-01L, 1.43400106584199068783023171408440267e-01L, 1.54897031607657925984050586220533856e-01L, 1.67011231455784555526366713839660189e-01L, 1.79758386919258486031495958113778865e-01L, 1.93154461659030684603221234896514194e-01L, 2.07215821067763214460783254641335718e-01L, 2.21959361905993070139869913079334838e-01L, 2.37402652741481501585147220599178905e-01L, 2.53564085294908579555677649044302540e-01L, 2.70463036885576732351902861856168718e-01L, 2.88120044277091724111603906974976099e-01L, 3.06556989345224713687672375979413066e-01L, 3.25797297128632613090751563726807919e-01L, 3.45866146978355838846531059919780118e-01L, 3.66790697694818431455168590767201925e-01L, 3.88600327732532063221423261459050385e-01L, 4.11326891764317591973280272152569947e-01L, 4.35004995130479531888748830535836274e-01L, 4.59672287956338821101823148522366487e-01L, 4.85369781006713270719035870359289569e-01L, 5.12142185661796519744086646686900168e-01L, 5.40038280749567858894795821479236183e-01L, 5.69111309360283620800291297165821293e-01L, 5.99419409204557829271245298375247710e-01L, 6.31026080564853461281465428141943759e-01L, 6.64000696438865091754636398259527208e-01L, 6.98419060091622837940058120097784455e-01L, 7.34364015932103716685673325408458579e-01L, 7.71926120422450486565697123079450111e-01L, 8.11204380628463812996672518165491420e-01L, 8.52307069046258371120734338452652256e-01L, 8.95352624512219417238820469064741936e-01L, 9.40470650345508781665473144971561924e-01L, 9.87803022412309344664090567700404412e-01L, 1.03750512157160024916963789173178052e+00L, 1.08974720700214151578746902594297743e+00L, 1.14471594926501601899574650013187236e+00L, 1.20261614467922655880690394062840816e+00L, 1.26367263574296191533610775626416078e+00L, 1.32813246599542422597073187153380823e+00L, 1.39626730197278306832678497202104711e+00L, 1.46837615987297935465145703905241109e+00L, 1.54478848033412089590682530315276939e+00L, 1.62586760150092827681818702612923363e+00L, 1.71201468848349507792697792068680378e+00L, 1.80367318661869118602748089155898811e+00L, 1.90133387688644143294074965806611335e+00L, 2.00554062472320920559824403445004936e+00L, 2.11689692868996327728244730212683488e+00L, 2.23607339344688170923456651751668676e+00L, 2.36381627281331763524062140552607385e+00L, 2.50095725401825500385431630504625834e+00L, 2.64842468438767047986902883861439562e+00L, 2.80725647766353485711865991725063525e+00L, 2.97861498011790290399988989454287074e+00L, 3.16380412810114748721127771257562134e+00L, 3.36428929047159656773947338449998169e+00L, 3.58172026374255002938249218600614501e+00L, 3.81795797752670936449825238124466741e+00L, 4.07510557639156630320887778958143299e+00L, 4.35554467582358554514458856445852610e+00L, 4.66197774993613776113418481343338776e+00L, 4.99747780346124563910807301981274948e+00L, 5.36554671871496309147449064380266706e+00L, 5.77018396000583698718320188114380523e+00L, 6.21596767643453604333341399022283971e+00L, 6.70815068570623654464776592817234118e+00L, 7.25277436733040258314736650500251589e+00L, 7.85680416993879891697239876362854677e+00L, 8.52829127820429133135398118660851136e+00L, 9.27656603318338653224922387509018222e+00L, 1.01124700112272039114486606209390216e+01L, 1.10486353121876175190769261708382627e+01L, 1.20998216795271857764508697373482606e+01L, 1.32833246923912553882423938260725321e+01L, 1.46194715878299420711013972809323962e+01L, 1.61322254926408962729986594994585730e+01L, 1.78499242340404104179578536110600421e+01L, 1.98061868055245863949655128520558624e+01L, 2.20410294496835217815852679074305586e+01L, 2.46022447943980585927583897681190540e+01L, 2.75471123562893237374077750154320094e+01L, 3.09445280931970284883990775946651869e+01L, 3.48776660064165064038939228627010010e+01L, 3.94473180334351770794475603595764904e+01L, 4.47761023021425157569087608052893710e+01L, 5.10137878711900622533650986521384164e+01L, 5.83440613273984383402755554599355811e+01L, 6.69931638788863998843945524712655391e+01L, 7.72409663039404221632393248526916570e+01L, 8.94352363841359052286058140969269159e+01L, 1.04010107537438719134080196856970415e+02L, 1.21510103906665265646188258350644787e+02L, 1.42621552310160172951445448156836659e+02L, 1.68213866518508832477273582457732136e+02L, 1.99394097464546647941096853182355363e+02L, 2.37579409247584470819614691411159418e+02L, 2.84593916789823535562840880888980902e+02L, 3.42798827028127058717471147784029643e+02L, 4.15268383629255114718046929646589502e+02L, 5.06029199301683119404601716655559045e+02L, 6.20387871848170976858394304975149188e+02L, 7.65382367194376728088096731506729391e+02L, 9.50408087358179153450911943224019430e+02L, 1.18809220276011606595956628737466824e+03L, 1.49552334212407885347227350564427644e+03L, 1.89599366703067034325550411232806679e+03L, 2.42148532800683663374818194050951580e+03L, 3.11624674527423644655997407153581040e+03L, 4.04197721822739650017152639557046462e+03L, 5.28540456882764394198106919117899344e+03L, 6.96945349745478520211739265737692803e+03L, 9.26984863597541610825591719881066842e+03L, 1.24400169046144284609469034917486102e+04L, 1.68487804928219120974770366506141126e+04L, 2.30379493050689209894874317068326889e+04L, 3.18111749406368329678640732920395152e+04L, 4.43724092704038524997055436198884382e+04L, 6.25438880548229936896462817079707202e+04L, 8.91129656174671765721819893640615727e+04L, 1.28390011615567130356349325241374642e+05L, 1.87115939884978799395430373139032810e+05L, 2.75955654445572113231216194578725044e+05L, 4.11985149226574332956827494211875921e+05L, 6.22884543671150616939347176756608244e+05L, 9.54097172994412639791727921401626133e+05L, 1.48121323807100834515424707068495244e+06L, 2.33168052188078970586379639354686916e+06L, 3.72339779803011251388319470323527721e+06L, 6.03430539101169547235371451057106735e+06L, 9.92972861117960142404055815146657353e+06L, 1.65989636945226644757161242143615839e+07L, 2.82017465494921172861525124733558875e+07L, 4.87244883934161305263386031770533647e+07L, 8.56498776477185184074970634710495674e+07L, 1.53268758654909039217478083777958109e+08L, 2.79366798395238972059209786152271040e+08L, 5.18973079293501172198046128702649663e+08L, 9.83165082634482695230583899233702397e+08L, 1.90059962104050828804295224765625411e+09L, 3.75160395202091966289927262175925315e+09L, 7.56648043123273181762292360200563868e+09L, 1.56033907397856950198380769768232419e+10L, 3.29229832278164384876778685138061688e+10L, 7.11297379086385418813540334382875157e+10L, 1.57471442166507563544871664087516645e+11L, 3.57509888501672692207543566803978301e+11L, 8.33024430623979589172560805550927471e+11L, 1.99374509951525518357032637324282993e+12L, 4.90561996981418757143934103035069327e+12L, 1.24197379810188498181468317797399218e+13L, 3.23831600275722270184945604347174663e+13L, 8.70403769580874957196023586424815933e+13L, 2.41399528145469907607734183398259453e+14L, 6.91510620574880583944843529198735451e+14L, 2.04811558742607734304993874133785299e+15L, 6.27861397733698939186730156018167292e+15L, 1.99435267076689206564427715465056511e+16L, 6.57141971664513108413598657255436192e+16L, 2.24875056642273914367880154275296162e+17L, 8.00147401578245998412996372425512095e+17L, 2.96403754199235340052960592736049159e+18L, 1.14455803313869409862823561648962815e+19L, 4.61323311996821344548899222001480398e+19L, 1.94346926949906856258436939614621369e+20L, 8.56968050834200116120462221511753399e+20L, 3.96091566953282355348486677100761553e+21L, 1.92185171194284479900050386014427105e+22L, 9.80409181939054057802535035944011717e+22L, 5.26682924609986175842861032563925229e+23L, 2.98441055802829784200443452444088495e+24L, 1.78677995299228897649048545726173530e+25L, 1.13225906725879734647293567479653181e+26L, 7.60791970573697649080495794599946588e+26L, 5.43049485025884671542842347146701195e+27L, 4.12572142434645000681420526588413556e+28L, 3.34277767339287328803586493191472415e+29L, 2.89434014229221431295360965172883804e+30L, 2.68374761249850267614464943620131017e+31L, 2.67072026965642827232315722365639115e+32L, 2.85880373230063837155252556137517211e+33L, 3.29924822913520515079335032162581841e+34L, 4.11488570832552270061078474206882038e+35L, 5.56010558358231010262223361328066200e+36L, 8.16009266847150844693389287229450581e+37L, 1.30416726659952301962536120869903033e+39L, 2.27600481686142160023333409481863034e+40L, 4.34938214638271735275350443120721790e+41L, 9.12741493518023346499222172162825387e+42L, 2.10972038777434150864210644015160107e+44L, 5.38754491759583324556444810556121661e+45L, 1.52482535270240332398284822085798536e+47L, 4.79877533166358652838909903087275144e+48L, 1.68494126510508458862564987284302025e+50L, 6.62357573295543230261174696442002813e+51L, 2.92557273755841342556692584369563440e+53L, 1.45729219902900863678006319195819802e+55L, 8.21783496105748128055143975755446008e+56L, 5.26688130483523933792268242131450300e+58L, 3.85211799189653678380768112270880793e+60L, 3.22861432024885393800430464301385615e+62L, 3.11445231039438406261213455773274485e+64L, 3.47323481265477220985448691064609690e+66L, 4.49855546587324575105781665091655708e+68L, 6.79925746409737423822172568281133860e+70L, 1.20511321523280079626937861818322431e+73L, 2.51749264598597737693408358183917414e+75L, 6.23086472714522132176542446806460931e+77L, 1.83698604157213615145607828551975305e+80L, 6.48713124894846526923286444858655243e+82L, 2.75979764628924062898267004292796591e+85L, 1.42281010916783424894495664030511266e+88L, 8.94356583170635560730657498907040019e+90L, 6.89765608918172471716339680473639534e+93L, 6.56960382950241207669696237814704407e+96L, 7.77916346275648519460667580565807164e+99L, 1.15314598687748380372708686918514146e+103L, 2.15521325185955507170865064451154694e+106L, 5.11618164822064799451529384134754077e+109L, 1.55434716015270528126967279870385072e+113L, 6.09112777117402790869661219995564436e+116L, 3.10387507242519227183111245931080882e+120L, 2.07390189233940742293442790510784430e+124L, 1.83267382155701863417862737648022763e+128L, 2.16097258672364775058232219739796364e+132L, 3.43128595186527837599814654188046354e+136L, 7.40646244666625583794129098263179729e+140L, 2.19454208154239352969892199569722442e+145L, 9.01619836979155465535468945047889426e+149L, 5.18985678737597694246477524117862598e+154L, 4.23046985030263912409378392183252421e+159L, 4.93764276985086773495452256391816235e+164L, 8.34636422063293684495534346523864357e+169L, 2.06741406169751668845974997770098232e+175L, 7.59587841419400960682004001784651511e+180L, 4.19164127392101407920200382533221559e+186L, 3.51927711650483149002609005453729140e+192L, 4.55587470619707394753414927965798200e+198L, 9.21952142596023196043927688149872532e+204L, 2.95816314360706369484966684111893238e+211L, 1.52709952631709289803811134371908265e+218L, 1.28765668330017907582554622529315931e+225L, 1.80129361672056777447357282869624658e+232L, 4.24814651295030396772649227180907798e+239L, 1.71729703963558073111114105671039088e+247L, 1.21046234446012300929891454835993254e+255L, 1.51419519516367160890801105849993081e+263L, 3.42330233588160029564511697140750743e+271L, 1.42528129667493838632765543616342364e+280L, 1.11420723637564670686340338390175118e+289L, 1.66850212135837193821657370527462903e+298L, 4.88589347415031911318629107633374545e+307L, 2.85800617499362161237161192858659000e+317L, 3.41368346308455763011375953918857639e+327L, 8.51659564264685587821018369835222946e+337L, 4.54303089474615832686354150019925596e+348L, 5.30809578740740306246018975605923943e+359L, 1.39268945492416740550127311639881773e+371L, 8.41868946331338761333876333129653244e+382L, 1.20396970846759916705111332553270569e+395L, 4.18638285711073253556696402015828229e+407L, 3.64051583443686661335653674614037576e+420L, 8.15125707810912875910267388062325131e+433L, 4.84241684726444145079336201568928602e+447L, 7.87277386139003022204421580245435912e+461L, 3.61660042166407854517352456605683269e+476L, 4.85176905670774504900384645174873943e+491L, 1.96653101796028683463909850095858046e+507L, 2.49428344091357019569533852688763620e+523L, 1.02650775155921036180043655136824490e+540L, 1.42290670725613581153914997322741401e+557L, 6.90446128383757111350399979478971290e+574L, 1.22038031602184356762554212370154951e+593L, 8.18641498043102997149775477320992897e+611L, 2.17426427576810603662782427977182733e+631L, 2.38847898517251831173924452396821302e+651L, 1.13525581232092079165478024625033648e+672L, 2.44581305055310563196277332943189236e+693L, 2.50579201726264634501291514398286907e+715L, 1.28278958026597426545698815500879845e+738L, 3.45329144611998202962794722592107008e+761L, 5.15301310117757693898843758843439768e+785L, 4.50037434223155300664502702852727419e+810L, 2.43306674872992411217032796443919144e+836L, 8.62798204693025064663883932656343701e+862L, 2.13031523407691556211779361127399737e+890L, 3.89502853428849118703235846476912848e+918L, 5.61967843038535141183624496528991236e+947L, 6.83163157235897275355324461890504612e+977L, 7.48741060910847824069581635250382582e+1008L, 7.93319051954128205327740081284490051e+1040L, 8.73273358333231337767980404927541807e+1073L, 1.07574303828662898687276828597901159e+1108L, 1.60109519481842931883372467310550530e+1143L, 3.11621335415749303163841998134612826e+1179L, 8.60558107743722927550014656572137529e+1216L, 3.66811383278356514418007625117425117e+1255L, 2.63234220176367112942815907184016908e+1295L, 3.47859421502287448304146808910623408e+1336L, 9.28512235755776936781681374078562947e+1378L, 5.50718056653300293052467960576840696e+1422L, 8.00902248373521173018720630861185040e+1467L, 3.16115623922106492070773163357582495e+1514L, 3.76042153408120168432167355515647590e+1562L, 1.50211667261404150161306978400073837e+1612L, 2.25263196634448337432840054986261153e+1663L, 1.42290818050514650710172541634086053e+1716L, 4.26315122517229508281635793163851999e+1770L, 6.84787649140078417523090257285629098e+1826L, 6.69183748343427005004387693124220507e+1884L, 4.53246642529433183363157301179103704e+1944L, 2.43420309539903059914673692609144617e+2006L, 1.19096818482919681799422553000433899e+2070L, 6.12585236774593416353267428912127386e+2135L, 3.84000941993213246405245986391372004e+2203L, 3.41673273574541565248480628561887880e+2273L, 5.05031055516767284204610907378751985e+2345L, 1.45860427844568831785920214989842347e+2420L, },
      { 2.77668375956040189946826811680384390e-2478L, 7.15287685954868211386033128907477031e-2440L, 4.67658088132803463412188029644470498e-2402L, 7.92687480429971819654882242916691373e-2365L, 3.55705626445534513670485482702241051e-2328L, 4.31363186281767249198550472192046640e-2292L, 1.44267570824010102564476209762038016e-2256L, 1.35750289529472083929076668071298865e-2221L, 3.66520151899825097802289209374933991e-2187L, 2.89498463671553268274047707590333923e-2153L, 6.81807725541668247816231315716744273e-2120L, 4.87856900336985600073857171641568070e-2087L, 1.08033682307899983904972807875490412e-2054L, 7.53978658799448170922270322982561647e-2023L, 1.68836600150778453318616795534542245e-1991L, 1.23462709610862382392789185979644933e-1960L, 2.99986697918589103634446685042416531e-1930L, 2.46367649797140140439024370548377672e-1900L, 6.95479514251903643444688832919306991e-1871L, 6.86110776725256744693425828768272774e-1842L, 2.40431940102692787285034949159539178e-1813L, 3.04121511300162701514429163779115709e-1785L, 1.41065291642406553832784938505501922e-1757L, 2.43705806703985396362343530158481976e-1730L, 1.59233397078886467474093245165420794e-1703L, 3.99458646512746569553991304016076066e-1677L, 3.90502825544924862516415029425091730e-1651L, 1.50951212222311306420066585908781007e-1625L, 2.34075577961021270449149847209864885e-1600L, 1.47683791355252947630371426518876483e-1575L, 3.84433827968748974568829430827411882e-1551L, 4.18584678137447017460210584041562890e-1527L, 1.93235335361456340539919561054923147e-1503L, 3.83273493158460886555652388105220442e-1480L, 3.30931424971841824136873930746170663e-1457L, 1.26001001743526741988366704934436124e-1434L, 2.14254382108136960073063000869254434e-1412L, 1.64752740569760866428499059885288678e-1390L, 5.79997534765464443335232132612089697e-1369L, 9.46173366850636035136459393339314051e-1348L, 7.23844554700889711120024618879438753e-1327L, 2.62753543705293236920590575888507321e-1306L, 4.57825670343966465596419576070363442e-1286L, 3.87294895920776670416027972176834657e-1266L, 1.60856514992623900565886618371043785e-1246L, 3.31650403849378163469754348545840043e-1227L, 3.43149117215084179247503681795855710e-1208L, 1.80089379614922388608625914511562841e-1189L, 4.84469660937728881846012977552474102e-1171L, 6.75021934282781912062862469597524256e-1153L, 4.92120879530819746165015110941962755e-1135L, 1.89622303139680815547453480860411409e-1117L, 3.89999690124553358779083557357143651e-1100L, 4.32337852652349730363515343553956403e-1083L, 2.60812148992852188438485836891607672e-1066L, 8.64320507307309471921069873497971138e-1050L, 1.58817234955465364327662145482119776e-1033L, 1.63292822332387428570727069490856970e-1017L, 9.47965403932768371077126663738787002e-1002L, 3.13489123036525936535865202633924346e-986L, 5.95725324626422679969355089398557051e-971L, 6.56135449619885064132736333035718110e-956L, 4.22412947622465431737740854731813869e-941L, 1.60284936502012533605395043475391797e-926L, 3.61426009581007446214636991266185477e-912L, 4.88226731291038533672727194690523707e-898L, 3.98243859045531426665602764668150557e-884L, 1.97696254579881780215274246825324592e-870L, 6.01884829406232602469720991349044932e-857L, 1.13236358528056083157892195864287071e-843L, 1.32634602112101849951006383738692278e-830L, 9.74354258615429475261032804502093079e-818L, 4.52175713954583830576111594951110560e-805L, 1.33512020097306090853791018618669252e-792L, 2.52581092493362086952359277189983609e-780L, 3.08280454280630114861769351743791880e-768L, 2.44403095665612216421741893390173529e-756L, 1.26703351146649868905932200644776032e-744L, 4.32364746058654184636712625519962546e-733L, 9.77482453681545274501227529587168230e-722L, 1.47345296987795364063826018729590588e-710L, 1.49025749519107052339924056423862408e-699L, 1.01758543742056270682348949473524139e-688L, 4.71965368684349520914049336410150458e-678L, 1.49583331955342167906773277357310899e-667L, 3.25877112957152479296585133409409173e-657L, 4.90846979890405909332298369271733953e-647L, 5.14096775171769223933971053266547990e-637L, 3.76527319345060607553853872315139069e-627L, 1.93914103642113548742773247369930873e-617L, 7.06081267960685321695216552061682039e-608L, 1.82753922477698812695612565052211519e-598L, 3.38020931062714080731736679298776434e-589L, 4.49104974727379533831457784467988803e-580L, 4.30831017202845455490779426657646708e-571L, 2.99925774499663529105464847801180013e-562L, 1.52274525111081863903256031902733405e-553L, 5.66595073105051388454002533447155475e-545L, 1.55254034318111114885985765214004285e-536L, 3.14773091677700441775450360300498865e-528L, 4.74422214535871848463366092395468719e-520L, 5.34000677409012855851843363768904621e-512L, 4.50913961542719498094454101135511353e-504L, 2.86916730093441295609899413107663763e-496L, 1.38176315596895654455580263526098670e-488L, 5.05827668288972660188294275444103716e-481L, 1.41354423391013787396132813520577665e-473L, 3.02811286218415761503816814918528786e-466L, 4.99321587478833584968780076440597278e-459L, 6.36350378784780643105814130227578114e-452L, 6.29295581774659679570380847201393083e-445L, 4.84800776366962427166614895670077970e-438L, 2.92081155862662548270928062619807743e-431L, 1.38143568676341254642463548681208773e-424L, 5.14842432351503290408078410534655737e-418L, 1.51753560889290027005627901927469983e-411L, 3.55061704489050704368494163406172279e-405L, 6.61796042955880547227964893320004528e-399L, 9.86122308014976296151434619809887336e-393L, 1.17877481135241077171765329651013943e-386L, 1.13424833803992349974759003704486622e-380L, 8.81502377306393355822212968439637791e-375L, 5.55155950590484925963715579436840823e-369L, 2.84248729913659406224316552722434334e-363L, 1.18704719186027282879490304441234572e-357L, 4.05597847694543335354139081149132852e-352L, 1.13745052103435347539610354499759843e-346L, 2.62608432267715211136705119144544958e-341L, 5.00649221900861773230008659070798254e-336L, 7.90489841299351390278476450028383484e-331L, 1.03673095771954987567298480275353471e-325L, 1.13264118291164663653569719276217293e-320L, 1.03372554114991613828759256440083793e-315L, 7.90343344978986920403039115230438646e-311L, 5.07593758357543835739913968923721419e-306L, 2.74587297706393183600196990448758755e-301L, 1.25448043297991919976296583184269934e-296L, 4.85293287864597257977432400607091472e-292L, 1.59375532748424261820313240836566102e-287L, 4.45470192712163297279675529941780532e-283L, 1.06238236699788796015140463431038408e-278L, 2.16709025689586849941731617555370390e-274L, 3.79017494148388392555292862630176094e-270L, 5.69722642026139785964263462216321432e-266L, 7.37750423818818484376423187717037465e-262L, 8.24899816234186348122208308733670706e-258L, 7.98228295613681065404000331301407417e-254L, 6.69977627078532550007479609463315785e-250L, 4.88831026337525670264792775582398754e-246L, 3.10717465030331182564914656973440633e-242L, 1.72428695582194286916259157368370433e-238L, 8.37152419508690266695542791416172637e-235L, 3.56328545617913642798317603893755984e-231L, 1.33239745308069123170741398284986139e-227L, 4.38557307471736323357712547586541243e-224L, 1.27317190092247560042567075437905721e-220L, 3.26633944706575228492861971563886982e-217L, 7.41961719235926788479176390701225966e-214L, 1.49508892701036107375575902806187267e-210L, 2.67747004659594255579164687433346539e-207L, 4.26921971680475266131733781176106392e-204L, 6.07184376271449065521795822572065912e-201L, 7.71632746256793762696505755410742573e-198L, 8.77761527570739954686084893442788334e-195L, 8.95291300378341203683641755088969233e-192L, 8.20179011370862828929882403579221646e-189L, 6.75977978662441271172788792155811852e-186L, 5.02051176570061397957717306624573193e-183L, 3.36554863928813190587746679952307259e-180L, 2.03960573952427530884860135637723107e-177L, 1.11917598852847666035205866531172342e-174L, 5.56905039413988338846998023185739233e-172L, 2.51682459319287628531543936384778284e-169L, 1.03457507608575432785228105631849600e-166L, 3.87388309025221395194171192891040522e-164L, 1.32322812968423778281527114220160831e-161L, 4.12900297352082279106292344371557207e-159L, 1.17865546256954888181796214775388038e-156L, 3.08218900889320623074522540652758709e-154L, 7.39354283219941448656677403739458341e-152L, 1.62910064435532863938332952898952690e-149L, 3.30154552905982294134609461204752294e-147L, 6.16203139085424122701291732112343081e-145L, 1.06052819447098630885771547158633382e-142L, 1.68522575749723508863020213808775829e-140L, 2.47553409758226362888052187337950144e-138L, 3.36576474950758719210258297005317849e-136L, 4.24056268392402238294912126586145782e-134L, 4.95679422788561171460024764665358982e-132L, 5.38171636791416151987580045952962687e-130L, 5.43350717229498884938353606822409483e-128L, 5.10703124279431542010303665040364840e-126L, 4.47370493209864639350813176634174273e-124L, 3.65637694737788862889473248524269217e-122L, 2.79117002269425900051588497263773104e-120L, 1.99220023869241503219258818787712929e-118L, 1.33089435939378971755299967553769684e-116L, 8.33035676735989050287962245538521119e-115L, 4.89025663997024514599039877971897448e-113L, 2.69512893545116544669129054438916450e-111L, 1.39582960541563084355183973360376419e-109L, 6.79999752718808594213315829447961814e-108L, 3.11903776737903229329061449549842670e-106L, 1.34826013141921629130407145750369350e-104L, 5.49752601894399080415833353152482494e-103L, 2.11638467025119853304622111522504730e-101L, 7.69914871485806120880436816954565416e-100L, 2.64906534725059834511552809323827260e-98L, 8.62818926354972775334178389390317975e-97L, 2.66252094324836892217545797407595895e-95L, 7.79069862358288634135952568964046431e-94L, 2.16335486668307728100344975621110455e-92L, 5.70557673979722036138381850123515391e-91L, 1.43033819302856491333403076823625576e-89L, 3.41104078137232874664267449214467490e-88L, 7.74426807351644903668383259880558649e-87L, 1.67513656430343581317253830008450619e-85L, 3.45479581059570481569225182340141781e-84L, 6.79857313709947736278888732144820603e-83L, 1.27747470803378266136497077558029564e-81L, 2.29370213942630948295519765524670121e-80L, 3.93802170001517503024981641969166523e-79L, 6.46959393487630012362523872090477425e-78L, 1.01772526699091247125279301878345328e-76L, 1.53401952979332495111639447956817637e-75L, 2.21699988683886091607827589036641130e-74L, 3.07410074756280336185727691291524915e-73L, 4.09229533083754909228682061184080234e-72L, 5.23343417563653847061479912948387465e-71L, 6.43350607976335741808585356397544551e-70L, 7.60704267790136216066341175717356605e-69L, 8.65671438716342535748325493579430488e-68L, 9.48674605868548997400346985395655480e-67L, 1.00175672424828839701939548888991483e-65L, 1.01985394383485432972080334959114676e-64L, 1.00159110661066563005593773053264872e-63L, 9.49427782244426395210838765944495599e-63L, 8.69142291889189064854075883690819022e-62L, 7.68797704788744827616631661309525244e-61L, 6.57440810419660524782417308561229212e-60L, 5.43816250291842519145575247702512746e-59L, 4.35334083136300321175659843267140387e-58L, 3.37433876218124341053665144604385146e-57L, 2.53377092117304232961472206140287465e-56L, 1.84404892524861673828316906508664460e-55L, 1.30141081230848018402542829366785560e-54L, 8.91046674437447006307015773182167698e-54L, 5.92153838412413233143345637093941679e-53L, 3.82135613429770512672309553535220252e-52L, 2.39578065735303689149259651605329745e-51L, 1.45988218758182023631230626004517631e-50L, 8.65010547207677732721948607063784916e-50L, 4.98593355079719931559980617333925476e-49L, 2.79691190323743591605512918548961454e-48L, 1.52757011899350333194123701418929652e-47L, 8.12631404819699330158401537112269973e-47L, 4.21243636394857818219784558381591723e-46L, 2.12860405024256466240923390999711474e-45L, 1.04893835632343107182205383744844780e-44L, 5.04275314265368784163135845797922992e-44L, 2.36599922549416536405636377221603052e-43L, 1.08381346209104032486272733428199249e-42L, 4.84896336796031616924505606902108385e-42L, 2.11961287373765727720156892217888941e-41L, 9.05594713902200264778664561627827336e-41L, 3.78298719219266665046394564391309006e-40L, 1.54564984691757476458462458129485585e-39L, 6.17890975212602635731338783415082506e-39L, 2.41759755862594038624882497707942985e-38L, 9.26130599996633274587378852134765724e-38L, 3.47471297119465611529279266980333841e-37L, 1.27721589062918134522531144355249981e-36L, 4.60093813393547386396189442938777048e-36L, 1.62480431477305204386750726891935535e-35L, 5.62680810313792997206232926753677840e-35L, 1.91144242994708647057868821934329247e-34L, 6.37130041549818712538060098629118750e-34L, 2.08444453130944123746087664312229902e-33L, 6.69535606006557423430835320939136513e-33L, 2.11203843563779293148946583571749480e-32L, 6.54480290655151239305039275670085965e-32L, 1.99286493762398711426295964156611288e-31L, 5.96435881776415175461953820911435643e-31L, 1.75497323146494950021419641534165307e-30L, 5.07823155886177386317817053950871430e-30L, 1.44544786652825947518140437679267760e-29L, 4.04809975939166078604969666043969110e-29L, 1.11575287892799422088984788434321808e-28L, 3.02733416844233859190251989588495462e-28L, 8.08786849810622478767566085355596476e-28L, 2.12810654415185893615787768003277562e-27L, 5.51621011393022798460545237305226697e-27L, 1.40889092112486390607440977505465003e-26L, 3.54652073432677480700798349650597512e-26L, 8.80063648109636049392437478276758980e-26L, 2.15331950904398446525184217737998611e-25L, 5.19613654473192634598878003489200027e-25L, 1.23686905842220219027076147492586511e-24L, 2.90489167449091887312162206507161675e-24L, 6.73270731756325876275561922583341148e-24L, 1.54025360336139105521086193967934171e-23L, 3.47876572768722101871596015971992779e-23L, 7.75845007993303197586197585003227682e-23L, 1.70893932426983027624683633350310787e-22L, 3.71846701056881115177897268795618277e-22L, 7.99409437676902992032878645322876456e-22L, 1.69833677431834312305384300763198925e-21L, 3.56621446972400227479906082231485065e-21L, 7.40284853486635166159906810215851685e-21L, 1.51941171975529754933443555359007838e-20L, 3.08399399452860874024652960034823534e-20L, 6.19138881797445980861805317656851788e-20L, 1.22962598701058922677441904835003634e-19L, 2.41624594930841108357336910436255182e-19L, 4.69855181874941970642420918335722321e-19L, 9.04299297884852043932311013611381641e-19L, 1.72288019839002081694998146770641182e-18L, 3.24983285835411232229770728918949423e-18L, 6.07012059458645756189773861988245818e-18L, 1.12287188164609844099198470626937240e-17L, 2.05742923566420592156922015953543702e-17L, 3.73461320774281639944361675972781502e-17L, 6.71669436926784207473111220730444193e-17L, 1.19706302505504395186947411747310186e-16L, 2.11441966111566361680483324148539475e-16L, 3.70201713823102185303168392234499685e-16L, 6.42566549874633786044656050804228769e-16L, 1.10583090372698541886807277008925137e-15L, 1.88715605166056322365232100161599762e-15L, 3.19397901867912583303235022260119744e-15L, 5.36188197747320445907706025927166858e-15L, 8.92931856860669280901808059459161448e-15L, 1.47533056095858665971377793523064186e-14L, 2.41870863676582496406573146120828803e-14L, 3.93507835090405130222724885910838729e-14L, 6.35404709630865447857089142202307170e-14L, 1.01841666646650944181450881700329392e-13L, 1.62042378299930769318560924447489642e-13L, 2.55981751705612616590139075734652273e-13L, 4.01527388629421281039816262827476156e-13L, 6.25453235826176129082121714446163184e-13L, 9.67598102139418285827447083279565919e-13L, 1.48683211253456618620643979203627482e-12L, 2.26955737776048687888315587634986337e-12L, 3.44173600876636583230784574560493186e-12L, 5.18579385986065241250884484837407551e-12L, 7.76421788931400466314739807932612879e-12L, 1.15522810574654803550770552217628934e-11L, 1.70831312146426209744016210099566072e-11L, 2.51095185608620189710869551132160955e-11L, 3.66877697851095234075891821003426570e-11L, 5.32913181394174031371781017456930198e-11L, 7.69632539729948085558352466099257329e-11L, 1.10520072364372285498114055238330020e-10L, 1.57822184379603482492805668832205965e-10L, 2.24130967294097676646756319917307649e-10L, 3.16577320114495664158493107630920887e-10L, 4.44773051087161070439507137569795222e-10L, 6.21604166145516404864589081218699790e-10L, 8.64254490539598786804804563578215933e-10L, 1.19551930651665934901247256485043253e-09L, 1.64548212141718982285430677462884054e-09L, 2.25364361294162088274171207920306355e-09L, 3.07161057649675130982867546470369013e-09L, 4.16647469046044592666212503776391905e-09L, 5.62503650418518103546622761014447205e-09L, 7.55905963895399839577230243791284335e-09L, 1.01117741787649109183519733871956784e-08L, 1.34658870190626745425612716113453914e-08L, 1.78534009295770335027325069409745001e-08L, 2.35675936423533751929780865973963492e-08L, 3.09775637333761608808718626586210342e-08L, 4.05458117130271472976754716443074815e-08L, 5.28493928008555417314519862662075002e-08L, 6.86052524785416844753739601495847428e-08L, 8.87004371407679534565262439555179668e-08L, 1.14227959934028163656084297163993474e-07L, 1.46529195996537375677477289744322135e-07L, 1.87243781452025990276419175076674860e-07L, 2.38368096170532406236914962927186340e-07L, 3.02323520821923278406510621336929642e-07L, 3.82035773260694787625549171519571981e-07L, 4.81026746749616004418812011672214969e-07L, 6.03520391713916631448186742261762126e-07L, 7.54564302177565687477041747448625764e-07L, 9.40168786133714127970401442799808303e-07L, 1.16746531401927207818183968119629644e-06L, 1.44488634919934624164636989185786720e-06L, 1.78236866676220579646941432400191756e-06L, 2.19158235968382023977800841196782357e-06L, 2.68618781213700528587671678902632939e-06L, 3.28212298590973811003340670234049833e-06L, 3.99792341503412914858164425580814066e-06L, 4.85507733328388046876150511289283628e-06L, 5.87841836668756018708322083288294132e-06L, 7.09655820622938796350022322477138609e-06L, 8.54236163220623609743157219064928822e-06L, 1.02534661892020938105973895830986033e-05L, 1.22728487074863285519290054710280426e-05L, 1.46494407312787820235069142302801116e-05L, 1.74387947455200274155445197262654991e-05L, 2.07038028896765075538771239781158806e-05L, 2.45154696092443087426053206140339263e-05L, 2.89537394229808584415606234281235237e-05L, 3.41083806769492860360664819345743796e-05L, 4.00799258161539348751926239815464572e-05L, 4.69806683323287862163878784706404376e-05L, 5.49357161442722725101184470856795070e-05L, 6.40841007374651816919514989779193660e-05L, 7.45799409355181382811997812036059565e-05L, 8.65936597006977565440049012882777955e-05L, 1.00313251868244228486845169918522419e-04L, 1.15945600213690649557974183508671660e-04L, 1.33717836738558167374082301113276138e-04L, 1.53878745542570977882656611663357910e-04L, 1.76700203135100555424143059190915095e-04L, 2.02478651530284460762520573567302864e-04L, 2.31536598974665040190979221162493510e-04L, 2.64224142678798208257632362853594384e-04L, 3.00920507470608001324163745800874977e-04L, 3.42035593863725830721435182160899216e-04L, 3.88011528643900055031892572065609775e-04L, 4.39324210725794779827016989314436073e-04L, 4.96484844725809052233768108298280237e-04L, 5.60041454438256227059209260385185510e-04L, 6.30580368196231443658525282679749066e-04L, 7.08727667948158660008974296844527949e-04L, 7.95150593789209443880323604776955146e-04L, 8.90558895655812679363260602821822057e-04L, 9.95706123923012434268721813542918011e-04L, 1.11139085073953859292031598702982886e-03L, 1.23845781409454868818512091763619876e-03L, 1.37779897683285042761735605405503089e-03L, 1.53035449312115014435580213674818791e-03L, 1.69711357521498846993912286538919474e-03L, 1.87911525378240440542796943910703787e-03L, 2.07744902550331120940589303741307914e-03L, 2.29325538217982005624763697091003995e-03L, 2.52772621615854827916639591227331912e-03L, 2.78210509747707274118095565348853401e-03L, 3.05768741879849780665835354304288470e-03L, 3.35582040488560696306610399114125004e-03L, 3.67790298408396440888192232818825231e-03L, 4.02538552002609726985152127212337675e-03L, 4.39976940253081440732383904779371772e-03L, 4.80260649744698504497144153794077709e-03L, 5.23549845597384011114251649759318761e-03L, 5.70009588477421233645539752147845015e-03L, 6.19809737897730872540802302573847437e-03L, 6.73124842093794861403414763375789954e-03L, 7.30134014837421983438805473382268060e-03L, 7.91020799623995212524470914365122922e-03L, 8.55973021739730390294556694387596169e-03L, 9.25182628783344529797384941938652911e-03L, 9.98845520280948891334775899197118662e-03L, 1.07716136709355454400393298774895433e-02L, 1.16033342137295485601573254130151715e-02L, 1.24856831787362164573968971563345559e-02L, 1.34207586747535542664119568732591706e-02L, 1.44106884381354658508301066473916456e-02L, 1.54576276395086064771405090469805654e-02L, 1.65637566405583013536386920486874088e-02L, 1.77312787108013640215572254554956350e-02L, 1.89624177144726038238696005618612010e-02L, 2.02594157778067758841208537767709612e-02L, 2.16245309470991783928580857382458621e-02L, 2.30600348479769142051214039575544903e-02L, 2.45682103563102531782599654851478334e-02L, 2.61513492911411521698129655317611847e-02L, 2.78117501399057252267505437298152555e-02L, 2.95517158260815126327011961248970821e-02L, 3.13735515292012408141424535463840985e-02L, 3.32795625669450926962877709780283664e-02L, 3.52720523487562160472278601135978908e-02L, 3.73533204101223493795063079870277644e-02L, 3.95256605363332412560750462340568604e-02L, 4.17913589841622853414796289201462570e-02L, 4.41526928095348722137611194180899765e-02L, 4.66119283088387990349053837529086301e-02L, 4.91713195811071287202299070345370512e-02L, 5.18331072178645941782756741967663201e-02L, 5.45995171269784130156535121811973661e-02L, 5.74727594963965733729390200663376457e-02L, 6.04550279031945582526756917103894992e-02L, 6.35484985728882875351754365961380466e-02L, 6.67553297935098586459427868785125518e-02L, 7.00776614884864197850796119077335695e-02L, 7.35176149519140388651301532758182494e-02L, 7.70772927493804152471705586229649727e-02L, 8.07587787870652431694028348485017861e-02L, 8.45641385514373366852999925423097546e-02L, 8.84954195214754605679505353644207668e-02L, 9.25546517549672049628688665169818439e-02L, 9.67438486500890476543279055502732567e-02L, 1.01065007883142650157999494215339249e-01L, 1.05520112523018947242782060692540634e-01L, 1.10111132322684063185699908950501089e-01L, 1.14840025187730710332047574671488736e-01L, 1.19708738821816529269842068051324872e-01L, 1.24719212548617699381893630778769370e-01L, 1.29873379309762826940592292027774350e-01L, 1.35173167838079215925316604016895877e-01L, 1.40620505005381631636575206950501568e-01L, 1.46217318343962952595269565777364191e-01L, 1.51965538740906942447303428154167210e-01L, 1.57867103304335938296117505727756989e-01L, 1.63923958400730641095309983504562072e-01L, 1.70138062862515433149015694041321523e-01L, 1.76511391365190704205555387802167839e-01L, 1.83045937973413460592324819248629329e-01L, 1.89743719855578905138540205901925041e-01L, 1.96606781166638569017145529907938222e-01L, 2.03637197099104797388960830732662422e-01L, 2.10837078102436785186096324004787346e-01L, 2.18208574271279784343490287100076472e-01L, 2.25753879903336437885194071240289887e-01L, 2.33475238227987351081778389476348222e-01L, 2.41374946307146941036852686683657608e-01L, 2.49455360110240324117132946908862256e-01L, 2.57718899765617581970697551365699803e-01L, 2.66168054991183344330456451077482845e-01L, 2.74805390707512480349974276871498222e-01L, 2.83633552837247137560986852113493320e-01L, 2.92655274295126854671240651245542162e-01L, 3.01873381173592566247581847444990310e-01L, 3.11290799129527708372244037377423071e-01L, 3.20910559978356159579257040371157487e-01L, 3.30735808502408397196150346208941643e-01L, 3.40769809481195164806575214000604739e-01L, 3.51015954951993455453034437018576899e-01L, 3.61477771709954227353934951808660907e-01L, 3.72158929057786693176355712369457666e-01L, 3.83063246815962181179662085907016341e-01L, 3.94194703605313603549231284148895684e-01L, 4.05557445414886871112481166359816563e-01L, 4.17155794468930807411520974095114834e-01L, 4.28994258407995154251335647089270309e-01L, 4.41077539800245330879960121109734359e-01L, 4.53410546000301224452886670053661009e-01L, 4.65998399374169294389180835005701951e-01L, 4.78846447910166863084991969300289662e-01L, 4.91960276237139210883303640625007935e-01L, 5.05345717072748965861316155618037977e-01L, 5.19008863126178679451085258491885813e-01L, 5.32956079481237266942061435303847022e-01L, 5.47194016487605519485083198030599369e-01L, 5.61729623189802041293777914000548030e-01L, 5.76570161325406179304747155047028479e-01L, 5.91723219926146849096660434528676034e-01L, 6.07196730557664332661777342118343230e-01L, 6.22998983236085549221918733111272834e-01L, 6.39138643062032159550125999965534073e-01L, 6.55624767615316158435211030870214346e-01L, 6.72466825156381227156528472165863186e-01L, 6.89674713683532904728192023160169670e-01L, 7.07258780898180476422511449104598714e-01L, 7.25229845133703375815759935882969711e-01L, 7.43599217307171072603779249870503705e-01L, 7.62378723957005410057715275736188342e-01L, 7.81580731433797129025240842779308298e-01L, 8.01218171315894385869574690877526536e-01L, 8.21304567126092639185776756157030964e-01L, 8.41854062430796373280836243276360497e-01L, 8.62881450408419762754617604609421022e-01L, 8.84402204979573743001129336964073261e-01L, 9.06432513597781571704880022546834535e-01L, 9.28989311806106946360784332676954119e-01L, 9.52090319672203976394493741712123982e-01L, 9.75754080221945735296148718924093021e-01L, 1.00000000000000000000000000000000000e+00L, 1.02484839189454300830993201809676807e+00L, 1.05032052037278447499546563674080434e+00L, 1.07643864928417387084533130337621185e+00L, 1.10322609239912797847732993345442148e+00L, 1.13070726686292705163871994441376389e+00L, 1.15890774975714122929598339522250570e+00L, 1.18785433797464608445948646027899515e+00L, 1.21757511162904898441624819295410761e+00L, 1.24809950123526638608045628896113726e+00L, 1.27945835891516450038384515634984238e+00L, 1.31168403390070906172271785633240954e+00L, 1.34481045262708114284870961989593531e+00L, 1.37887320372983270957282413312613361e+00L, 1.41390962828351735195133854106565322e+00L, 1.44995891564449075379247910337230275e+00L, 1.48706220528789860745767637311307254e+00L, 1.52526269505843914812343336632313553e+00L, 1.56460575628650281139860620478932861e+00L, 1.60513905625597123057624111991440169e+00L, 1.64691268854754131266306691391994313e+00L, 1.68997931182218993728102497644674194e+00L, 1.73439429765359879279739907195277543e+00L, 1.78021588806633292063608039095887334e+00L, 1.82750536348865755482665463985323837e+00L, 1.87632722188546688075368971900486300e+00L, 1.92674936989830423931126850366320158e+00L, 1.97884332688633669446426238525349115e+00L, 2.03268444283491461260047065213162138e+00L, 2.08835213117755699197923733656794571e+00L, 2.14593011766347043193461156899167303e+00L, 2.20550670649671136552919447649838551e+00L, 2.26717506507558468055611786100163107e+00L, 2.33103352877266160486316503843066293e+00L, 2.39718592731780603672335804074417192e+00L, 2.46574193447982700411979376412836737e+00L, 2.53681744288793726408585573416385695e+00L, 2.61053496599332371109890452901649940e+00L, 2.68702406934518495604136651525501878e+00L, 2.76642183354607197859959809591833922e+00L, 2.84887335145994878054909245340497664e+00L, 2.93453226247492266642719969961850934e+00L, 3.02356132687313192268979888413460939e+00L, 3.11613304363510221142689168603250629e+00L, 3.21243031530752459841477997957148726e+00L, 3.31264716389468297553121493594219218e+00L, 3.41698950209779795665404010466511538e+00L, 3.52567596462684319748110900008331556e+00L, 3.63893880474980996700862087872826533e+00L, 3.75702486172927248744324880848351753e+00L, 3.88019660533026434144487775885161751e+00L, 4.00873326417229898631367723395410172e+00L, 4.14293204534786760904278361346846386e+00L, 4.28310945344664439881789918389859821e+00L, 4.42960271791643703960803985590050978e+00L, 4.58277133856704814714568853892203278e+00L, 4.74299875999107924880685634942448809e+00L, 4.91069418674686750664370967067014440e+00L, 5.08629455233503443708376995130700228e+00L, 5.27026665631483182008254527988202911e+00L, 5.46310948536451639599392689767712196e+00L, 5.66535673570814692696549699934052267e+00L, 5.87757955612834548026155780285696226e+00L, 6.10038953278194387910748891481975603e+00L, 6.33444193925698167042406747055218470e+00L, 6.58043927778222227422530564217212955e+00L, 6.83913514025466452585283059757581636e+00L, 7.11133842082084256614010724304296860e+00L, 7.39791791517290376256435527130893558e+00L, 7.69980734554450846886947561889535387e+00L, 8.01801085466429447369362846417322143e+00L, 8.35360901670240672837408029297072392e+00L, 8.70776541859238547329480193384508847e+00L, 9.08173387109914748403941266390886970e+00L, 9.47686631571637600605881026375055671e+00L, 9.89462150100714627457776733266750735e+00L, 1.03365745104567901877248523564151283e+01L, 1.08044272334084191011529376038077113e+01L, 1.13000198813377778114411160555843168e+01L, 1.18253436637533511507143176436936036e+01L, 1.23825547515605242739627815508172524e+01L, 1.29739896710116156250639129682527238e+01L, 1.36021822886130624460500072703793332e+01L, 1.42698825668476028925550536958145770e+01L, 1.49800772926032764371676872413981972e+01L, 1.57360130051385708102009640040308534e+01L, 1.65412213786631649993077912938609934e+01L, 1.73995473466468578424134221353716244e+01L, 1.83151802913268898058518900530921820e+01L, 1.92926886631898453200253754020840677e+01L, 2.03370584421782617230641820877704224e+01L, 2.14537359058448294184882229567324528e+01L, 2.26486752306089873609060295600452917e+01L, 2.39283915217729827153789982621787140e+01L, 2.53000199473141826806593837999123741e+01L, 2.67713817411801152928733734963681048e+01L, 2.83510579456049880498903420268116163e+01L, 3.00484718808548719519487088002485537e+01L, 3.18739814671361063943508266371074277e+01L, 3.38389826798966490384466465437407911e+01L, 3.59560255995953567185453470776520193e+01L, 3.82389447239249331007247241110692191e+01L, 4.07030054487934539563244342845687403e+01L, 4.33650688991795367943193597150736221e+01L, 4.62437776082326978417918860128324332e+01L, 4.93597649096797907147762266234322051e+01L, 5.27358913329271476516365756663547356e+01L, 5.63975117818677084656045931084727554e+01L, 6.03727778486785227486990377756623302e+01L, 6.46929802762275435095300960911126288e+01L, 6.93929373529211836464995846228029413e+01L, 7.45114359206196683576202953920800440e+01L, 8.00917327217667406576253702254103983e+01L, 8.61821250323685694884618177226323948e+01L, 9.28366009540655148012445479790749067e+01L, 1.00115581408296888990404128808417666e+02L, 1.08086767832535244787263306357432758e+02L, 1.16826111875294927859522500437909465e+02L, 1.26418926085804724017089848087642554e+02L, 1.36961157770833171464154963091055755e+02L, 1.48560851934901186551056930429681893e+02L, 1.61339833638593274264805190552029918e+02L, 1.75435645332062901686076817427004926e+02L, 1.91003780902460959044883635552383857e+02L, 2.08220265501991356519964932798160160e+02L, 2.27284638923300107785084254230641145e+02L, 2.48423410633602325676190041300652816e+02L, 2.71894066898304725754664440871201054e+02L, 2.97989725118823201634188484455430791e+02L, 3.27044548063367687799351246625873367e+02L, 3.59440051674122988480906333522906174e+02L, 3.95612465308733548542080237354845984e+02L, 4.36061333495907795272034066341690570e+02L, 4.81359584626980835451438312811051118e+02L, 5.32165335780833820345250001609436065e+02L, 5.89235755699686219555872484606593438e+02L, 6.53443371777544904545726289396026874e+02L, 7.25795284228401899442052568196334372e+02L, 8.07455844372956662653935015669559504e+02L, 8.99773467933970120032100931203892228e+02L, 1.00431239295794425220145278602443696e+03L, 1.12289036118559487702348487348886695e+03L, 1.25762340845977552983926843880669836e+03L, 1.41097920290752223402430410334039052e+03L, 1.58584068016657346033724860162595503e+03L, 1.78558210660144726153863412855453166e+03L, 2.01416017149982591387363180452914622e+03L, 2.27622328928316747879110750486628791e+03L, 2.57724301000797348481178480414602978e+03L, 2.92367232516280459791647074886201195e+03L, 3.32313675929073604672388801848045153e+03L, 3.78466551111357505023764578327774171e+03L, 4.31897162016023640570560918832525773e+03L, 4.93879227485091848874762796864624098e+03L, 5.65930305827336833061655557231992410e+03L, 6.49862329247639500380687922515090074e+03L, 7.47843387531893338609063539714339383e+03L, 8.62473434228616623771753210405507751e+03L, 9.96877263348459014531785309952890789e+03L, 1.15481895955939390229775103785535678e+04L, 1.34084311070264939001405447133999516e+04L, 1.56044945390858044276508880692505176e+04L, 1.82030939102313379306065080413190777e+04L, 2.12853506664968077681001285108385025e+04L, 2.49501459804837504640469288880047968e+04L, 2.93183077048218804714595369958899209e+04L, 3.45378531384547339729721217099269346e+04L, 4.07905708493105663079713795820150418e+04L, 4.83003052786320640972382415932704632e+04L, 5.73434124658699200382652517134835469e+04L, 6.82619915902214645340692561470062070e+04L, 8.14806752559419146429293810910586320e+04L, 9.75279950747873086720735652227979263e+04L, 1.17063646220480829492981340691767948e+05L, 1.40913379548158414250243653370073176e+05L, 1.70113785311182551164819299122255310e+05L, 2.05969942671050993972655988038600318e+05L, 2.50129853973569246335608225835338567e+05L, 3.04680843555537948624554921830727349e+05L, 3.72274788636036141079112881438654547e+05L, 4.56291316446017606669036491868612297e+05L, 5.61051155492184554128598135519314736e+05L, 6.92095956581034369089275407099645880e+05L, 8.56556497218119814922783780052161157e+05L, 1.06363880055232600043867227938165842e+06L, 1.32526810122628602501044472769613140e+06L, 1.65694484184724012086108587753064923e+06L, 2.07888647930116015575144959412983113e+06L, 2.61755592013006806905408032351706035e+06L, 3.30771485222622495472503881888885584e+06L, 4.19519229320262625923561042278599896e+06L, 5.34063130025074556576550111959530826e+06L, 6.82457849576702073415053978191461581e+06L, 8.75442405324883181776310371883173248e+06L, 1.12739015977226351727931535875936998e+07L, 1.45761434273968962478096940509724237e+07L, 1.89216932684193810033543675766077150e+07L, 2.46634598680066744212983509402346013e+07L, 3.22814282171121758797754478348568006e+07L, 4.24311457153986975423657329267444940e+07L, 5.60117371443408843149644877281459637e+07L, 7.42617250972307211179134697728146644e+07L, 9.88946135783012173106461237736151683e+07L, 1.32291587547042718239491966847893521e+08L, 1.77776624072745598074170059515038266e+08L, 2.40011058338983426258122648845724399e+08L, 3.25562103364198274177076500687162156e+08L, 4.43725882059376140276746390832793801e+08L, 6.07724621850487716521348005200021505e+08L, 8.36456587985737541719041767923031690e+08L, 1.15706659432645616854021468641216152e+09L, 1.60874082649874296139935339596050162e+09L, 2.24833765794868826938828602052210087e+09L, 3.15878597885133622780330724633488102e+09L, 4.46167708136391137964792211498946984e+09L, 6.33624483104820926953501466923377042e+09L, 9.04813015958867755955007060963155039e+09L, 1.29932136230997226464299124717687605e+10L, 1.87647826121294792855983490567276630e+10L, 2.72570397671288897105662591436709544e+10L, 3.98255345906428893971269386370983030e+10L, 5.85372779401741541532402802418687327e+10L, 8.65629908955310338465751817097315538e+10L, 1.28795973304189874698643530406628864e+11L, 1.92834506543009988305207298557577307e+11L, 2.90551046754580604436065170645660613e+11L, 4.40614548809848580878194871514169088e+11L, 6.72570891877849315226112306772636026e+11L, 1.03348693821219692992886708382137863e+12L, 1.59884055708669585428974649723976740e+12L, 2.49049013421827282502410713540807461e+12L, 3.90652846672458392086943060796482254e+12L, 6.17122514796135424360349149730712054e+12L, 9.81916373648510913725273324679675423e+12L, 1.57380010699156447495769482655375521e+13L, 2.54124546153003122065528612181733881e+13L, 4.13443762840798177552480089332110759e+13L, 6.77814197348597152801493367496981491e+13L, 1.11990628659588449226309821338662554e+14L, 1.86501680604176896658450602560601490e+14L, 3.13089094872498973827438463058389098e+14L, 5.29897884766906828009755247847186245e+14L, 9.04297389980418175348985287558326633e+14L, 1.55625903681899143890977078931637702e+15L, 2.70123006636820081209540843148166787e+15L, 4.72943010505471127882390119991674115e+15L, 8.35377903309658653016536824399814765e+15L, 1.48882760629319165095592517030838223e+16L, 2.67765346603161495578266855417246317e+16L, 4.86043448136949927032370428281054854e+16L, 8.90573551930099331171397006211491165e+16L, 1.64741372830687155232163226493215628e+17L, 3.07708132567301637697547718557883369e+17L, 5.80423410132909767997152626617023549e+17L, 1.10582857062809961361827765317796937e+18L, 2.12831535880807402614537212551250197e+18L, 4.13865153208523558147805437822414497e+18L, 8.13255421212392003487525754214292826e+18L, 1.61514650331257085505245767762340125e+19L, 3.24254846726071819306305572744455009e+19L, 6.58149458108070132105640205170170346e+19L, 1.35083136618309000260400853464983546e+20L, 2.80409383252093739584477712160125327e+20L, 5.88811368346756383731750279395970839e+20L, 1.25092343531246827629449550883330451e+21L, 2.68928027909821563456325160060548474e+21L, 5.85158282566447970019894650702188721e+21L, 1.28891723178894465997393609758461271e+22L, 2.87458276376899763123926200401593262e+22L, 6.49243733510921786856841605747277893e+22L, 1.48528660586708217722481671351080137e+23L, 3.44246915911330706635229848078895385e+23L, 8.08493019686043820711319325430367342e+23L, 1.92450677804809487818429003209217169e+24L, 4.64399266249147072865108838294493471e+24L, 1.13628145208359133428909996372167656e+25L, 2.81966489106069457125481864485455314e+25L, 7.09778155999185636739332722019170506e+25L, 1.81283885012768848640687277115301054e+26L, 4.69901285134453912350857508188703800e+26L, 1.23641970716283295078972799846710910e+27L, 3.30323626121041128601791088120844416e+27L, 8.96255809763889121781104645394017356e+27L, 2.47029485298622611738479511793982193e+28L, 6.91827096055594288255981089299725583e+28L, 1.96918944795841151009270119290336857e+29L, 5.69809260945398128927937198565189507e+29L, 1.67662615639692208356086898776485724e+30L, 5.01790152017155697028174057139776321e+30L, 1.52792989227983448898308499437330165e+31L, 4.73476231836671194909529913837515959e+31L, 1.49357254644677704024642645900767165e+32L, 4.79744116468190818427448048047019508e+32L, 1.56953829640099873205260260372105814e+33L, 5.23165115691024245360777615947271007e+33L, 1.77720651152529094086938101558873583e+34L, 6.15458729957691613373022022121395673e+34L, 2.17346978135660487165909974321640569e+35L, 7.82952989652658161590959525826590591e+35L, 2.87793555407307691688503613867678361e+36L, 1.07976132092345859199417272983750960e+37L, 4.13633773095120704247204289059445522e+37L, 1.61840848971118584374440265082561208e+38L, 6.46977064044782477057521822447781556e+38L, 2.64341365485931635830069155722471865e+39L, 1.10424672830852570286491286394819018e+40L, 4.71784264188126066477442678013103229e+40L, 2.06229646238932771079246843113076062e+41L, 9.22668000516125721911884540275100330e+41L, 4.22654407163273196344404760642122190e+42L, 1.98304372970706651824186968975247346e+43L, 9.53344869097015503904308238124562564e+43L, 4.69791457874020860581396665191487360e+44L, 2.37392310198043657407434802189623818e+45L, 1.23057021186853175270406146048991232e+46L, 6.54634433841169514665337805691594822e+46L, 3.57537181933580491356588477879121785e+47L, 2.00564245353833550563491221890112961e+48L, 1.15605526802890307768061561905701576e+49L, 6.84986780787031295787489135284280640e+49L, 4.17400481521895112076613271616259486e+50L, 2.61687203405285747164128861362318936e+51L, 1.68875034683729772513757434403804400e+52L, 1.12227566600968410062529607944214843e+53L, 7.68396874024867707105114297890426357e+53L, 5.42284961265427858271262866960101842e+54L, 3.94668670179953341522351651523862327e+55L, 2.96354358728813288396379337565903477e+56L, 2.29708639579893951580254665307105346e+57L, 1.83885641420855576075627699143076709e+58L, 1.52104947571124399594694864621286492e+59L, 1.30073229117507111236343155772481130e+60L, 1.15055959114171674001974091149754682e+61L, 1.05326599737372546052564481925643420e+62L, 9.98411420987902083614726789220470599e+62L, 9.80532561593869471877040903652671478e+63L, 9.98246356419911599549241472274732206e+64L, 1.05410221145791141007800297806625299e+66L, 1.15517268478078246328479789979967959e+67L, 1.31457130233411666326872185958326427e+68L, 1.55436240768545731029836466857749970e+69L, 1.91079120600264507732475085854133289e+70L, 2.44361640389071120608105036496582382e+71L, 3.25298382231882323170055543487798304e+72L, 4.51060014002013973693091388809534590e+73L, 6.51882183100190244682597556816883900e+74L, 9.82583446077426763261011103909194169e+75L, 1.54569206362272285597219332482212685e+77L, 2.53934608840816325270469005344260713e+78L, 4.35976399381183611739718946785420143e+79L, 7.82794362746440474440241447663834683e+80L, 1.47089687767430118295343594318257369e+82L, 2.89452707142067428998158759253431372e+83L, 5.96966254160791549191064067088778403e+84L, 1.29127761398105735746114390466172354e+86L, 2.93165653562687792342264563025180130e+87L, 6.99135354753146313546505095265922672e+88L, 1.75267119452597285220967300845626556e+90L, 4.62245013705602071519123099756702996e+91L, 1.28358193316956622609422743295635868e+93L, 3.75583900113839078839602923077406673e+94L, 1.15899172984597870198690174300625365e+96L, 3.77491631543886267838503954431054828e+97L, 1.29884489446238167309781708169537790e+99L, 4.72503894994338488909389270574194659e+100L, 1.81900003120328673954815570724729852e+102L, 7.41696633087690618810014849262359799e+103L, 3.20611699691059820440544154944066069e+105L, 1.47058877007197519318900335321912402e+107L, 7.16419823823864105738627712719272912e+108L, 3.71039762456707726983380527316097819e+110L, 2.04488245427970937306347337109290828e+112L, 1.20042877865473022461473385597117568e+114L, 7.51374437003017211432960701498996352e+115L, 5.01957574634341063561427626174780847e+117L, 3.58272692766569831842277681502801710e+119L, 2.73494777587724855953563379276564479e+121L, 2.23528376407894424768162839028330791e+123L, 1.95808475111824332331314972422066776e+125L, 1.84043191310930565706137359879906765e+127L, 1.85814326069283110784535066904520512e+129L, 2.01743294965577713629069559518257063e+131L, 2.35817761588810149418319514273035811e+133L, 2.97109297417860360975602176456436124e+135L, 4.03953232143581630156016504419988399e+137L, 5.93392306966113219484352196331202218e+139L, 9.42926369344495323976110371737768721e+141L, 1.62284145693287387206611282820134815e+144L, 3.02888447606769418009867639216690630e+146L, 6.13835617501533947725917198648160556e+148L, 1.35253155719194264809220346207577825e+151L, 3.24444736229558294520810049145898860e+153L, 8.48424354492815185029440589500302573e+155L, 2.42189217690801196639615731788461939e+158L, 7.55727585868832926111996462541955391e+160L, 2.58138920742415528768470662656564452e+163L, 9.66580408821980500452620634685578942e+165L, 3.97326060268422220330874023543806264e+168L, 1.79563826725695454978311213353995983e+171L, 8.93514523408268859940137248520982995e+173L, 4.90290834459626229428863642680474223e+176L, 2.97128375542216561281658377940574510e+179L, 1.99182881480698947977101116086872463e+182L, 1.47933813166325570382207702423402678e+185L, 1.21924602572867713843112063103440147e+188L, 1.11695489454371994985095225835122915e+191L, 1.13926159735841086544690160419235263e+194L, 1.29595329494635946555823051599904183e+197L, 1.64694619802426865078234124605721432e+200L, 2.34234840634634342775071057533872554e+203L, 3.73486904651415569743903039850576755e+206L, 6.68856535510325485878978796153529680e+209L, 1.34777842855531871957592541417095863e+213L, 3.06153116112389690978003658017819254e+216L, 7.85439891718825136093879357578783886e+219L, 2.28020371103825909996341632353690860e+223L, 7.50526802410090668256555773098038714e+226L, 2.80639879206390242816779047164860586e+230L, 1.19452560453314111686916354242681346e+234L, 5.79949872394246778903337432628166035e+237L, 3.21835787345388969294352208047085465e+241L, 2.04569666433064103875712173825120783e+245L, 1.49258715453013675730747996991557044e+249L, 1.25277443244629163588528004740184593e+253L, 1.21226842377681326200203984685187270e+257L, 1.35547194242695069046025865235623225e+261L, 1.75524005232377336464776480479935346e+265L, 2.63840064228932918271533450220126376e+269L, 4.61448246937529975889164745570586249e+273L, 9.41280682985947960778964419634581143e+277L, 2.24481910655275049981565977905948653e+282L, 6.27448882996682532882323387040658757e+286L, 2.06060958394918535923518047482752655e+291L, 7.97142764215603564227622164969231434e+295L, 3.64182905892925111147897309560503282e+300L, 1.97007938638916489488880184999451702e+305L, 1.26527287963256941162906241681573766e+310L, 9.67374762635351109418734649102546799e+314L, 8.82892141913231571338896445110187498e+319L, 9.64570405228039780469604907885256630e+324L, 1.26503839487205882730553259131731653e+330L, 1.99740647993660386944495280559643123e+335L, 3.80795083906732148311558637535145760e+340L, 8.79159120777085501704319293452205335e+345L, 2.46549631780369377584312978131661447e+351L, 8.42426490586997547506062762517991909e+356L, 3.51804562259170040261749407236406336e+362L, 1.80129565203500398450083781286924107e+368L, 1.13442688952887433720702641069844446e+374L, 8.81641141946113920681847478154293508e+379L, 8.48338453086470297056624825970987040e+385L, 1.01407299264221993702531525534230006e+392L, 1.51103955764611037180263766796269076e+398L, 2.81641187251957701963969414170240418e+404L, 6.58963120298401361748098859356824253e+410L, 1.94234184512060662929742463444500478e+417L, 7.23884585856411287896767410403126131e+423L, 3.42370598009480301692521696476539260e+430L, 2.06270296737946127496845017607144670e+437L, 1.58907837423540569213464552634511445e+444L, 1.57146131021351822368954401625918115e+451L, 2.00271733703560402612457677241115831e+458L, 3.30238681816736073809946856032462582e+465L, 7.07441603885154545912376402781538965e+472L, 1.97695789038711346120868489115303627e+480L, 7.23713029747673018134344413894888363e+487L, 3.48533178833568221019229567888827162e+495L, 2.21771798011905249239941207476859498e+503L, 1.87265680045956064211817379705948563e+511L, 2.10782709864946331624669282953979957e+519L, 3.17689162904658562291907841456392867e+527L, 6.44105645558316603786838285606330189e+535L, 1.76492886625329297926588753856751673e+544L, 6.56708664348495439553998825816543384e+552L, 3.33415826521812264992257180271069504e+561L, 2.32109565019822213867641474306217000e+570L, 2.22665090852540791532492989207079877e+579L, 2.95839667933009412312396349376762019e+588L, 5.47183877884770709370953901555429409e+597L, 1.41626756773793934414234479620392920e+607L, 5.15692247865370682523431517094128585e+616L, 2.65584978465684945355727442801669402e+626L, 1.94515906011252752522390088574648572e+636L, 2.03729480055734573550624444975610288e+646L, 3.06864139959250122576234946830893261e+656L, 6.68523683038794846390318553081110530e+666L, 2.11879952715090010307995150421897875e+677L, 9.82718465915609660920397095265955939e+687L, 6.71024976037303477285708646634894987e+698L, 6.78677922161865926117102990656018413e+709L, 1.02303626651674990928162545434962304e+721L, 2.31286202012488077377232576408525763e+732L, 7.89245107528823830762373616398932534e+743L, 4.09160120200842896534230185492282146e+755L, 3.24379955366775267129216830328274275e+767L, 3.95912453354472811300573118431549283e+779L, 7.48996232152866120855530906466249236e+791L, 2.21152965349315333956435853596087911e+804L, 1.02632075670404876010930426472294612e+817L, 7.53951068632005171876747821502928323e+829L, 8.83108581906785992138514068848879898e+842L, 1.66144742506056069516088714981068146e+856L, 5.05826477150550571129779691932628665e+869L, 2.51102428144577988853257771418346819e+883L, 2.04822869357369644800523164159313000e+897L, 2.76681803049890115118755008828029448e+911L, 6.23888945414058934548223953019542236e+925L, 2.36735167714072318956060511661796446e+940L, 1.52407555570930344049434626852406921e+955L, 1.67862596848152559342541655995681177e+970L, 3.18990333799710745664618509349355327e+985L, 1.05489081758823555970428507172122908e+1001L, 6.12396788613568122227472934597131896e+1016L, 6.29654583950170379780496166626637403e+1032L, 1.15697821762367329197002183707451842e+1049L, 3.83417721859040383986075244204614475e+1065L, 2.31300589079836393970996707559294268e+1082L, 2.56410460141809897241834340099170971e+1099L, 5.27364125127925215166604640933456487e+1116L, 2.03202107773477151599346823269715856e+1134L, 1.48143334195874803524122054239589157e+1152L, 2.06411274147574456351269054783640404e+1170L, 5.55279829459270893053136130201212587e+1188L, 2.91418497041682586483964699073577480e+1207L, 3.01522322419410790036838054063732694e+1226L, 6.21672053535323172221442203567173233e+1245L, 2.58201182234158741299570291718541207e+1265L, 2.18423750518116545391544630374143187e+1285L, 3.80584781426053416091532316078175593e+1305L, 1.38151208502663182821551093570287278e+1326L, 1.05688876376696945875455031156325640e+1347L, 1.72414525934903045436623350959611871e+1368L, 6.06970176363513871871663291361433633e+1389L, 4.66734911165218107635353767855185961e+1411L, 7.93644483902981870648608149469983212e+1433L, 3.02177407323915414129348065742188333e+1456L, 2.60910294567790325670373269654248928e+1479L, 5.17503694719938298503446121278113303e+1502L, 2.38900287619137048290055040441414741e+1526L, 2.60122790255932170291515511953064757e+1550L, 6.77122378037074355003811222286996596e+1574L, 4.27212445104598645308010594377100889e+1599L, 6.62465696881760614600007097777440709e+1624L, 2.56080093301387999855187759148833996e+1650L, 2.50338804461975863168260136920269977e+1676L, 6.28008959392222170902165204120090502e+1702L, 4.10330805623617785960247879181409818e+1729L, 7.08891597895639626581466081525369401e+1756L, 3.28815937986385053022460003637061243e+1784L, 4.15918117856089374793600707142802802e+1812L, 1.45749058887095094990047184214707565e+1841L, 1.43785687357830444275837543949503093e+1870L, 4.05897446691318032347020269164975282e+1899L, 3.33348114079172166390364845912603682e+1929L, 8.09961164105229487950427444532606992e+1959L, 5.92288638308847934616085992236628536e+1990L, 1.32629748644648493533560071181408206e+2022L, 9.25637244456745511366766862807596139e+2053L, 2.04978139964660371723728495327865680e+2086L, 1.46668915962420498068615471735159780e+2119L, 3.45424976463618489443831684054473314e+2152L, 2.72836294216453749285575709798797945e+2186L, 7.36646679330208627221413509790840381e+2220L, 6.93156469113828331566714060809446921e+2255L, 2.31823213431754954747032929849642953e+2291L, 2.81131341663812436900512315439163404e+2327L, 1.26153121461887745230804891576437417e+2364L, 2.13831434840067269117435765818454673e+2401L, 1.39803888650069112961026818896691495e+2439L, },
   };
   m_weights = {
      { 5.90801545776726515999184015930013435e-2494L, 5.43034121836882538817369981204760444e-916L, 9.13645323997978810837635266947225189e-336L, 1.38026921250443161275979571515063966e-122L, 1.95473329586235063070470724741264116e-44L, 5.89765053810344838395474502424100508e-16L, 9.65083022671853583721482039321815318e-06L, 3.18674406396319375693901804428127016e-02L, 4.49940993129051317388425885244478941e-01L, 1.85250678576020588686470383628874232e+00L, 2.14751816323261845650556006802974073e+01L, 3.65018587462837432006103692095515092e+03L, 6.47212478751933019554592655468556643e+08L, 1.92641646488982742563851849719109881e+22L, 1.38994758402678387868517621844463411e+58L, 7.26879966558078977021348471744247141e+154L, 1.04765280279857897745141662958802655e+417L, 7.83837589244484743057347089777048838e+1128L, },
      { 1.10316126712058491420201654940763610e-1511L, 9.01600613495532057799091391040663772e-555L, 5.11904332497495239643103179971390500e-203L, 6.89671333890943322248923557636362424e-74L, 1.17634110503454704251022002576839267e-26L, 1.50757351778282569234155861289206864e-09L, 1.64896113239222241299157252910603817e-03L, 1.70348472687044679146785418051347199e-01L, 8.92604116485216943737039962090086924e-01L, 5.08107015469559685466563888116585167e+00L, 1.70322410074360164066883311173278799e+02L, 3.99580125020294769314748785515084392e+05L, 8.97859410735688933722745437703783632e+13L, 7.52866673518542859464441779223888455e+35L, 5.14816023241024489763203760058639816e+94L, 7.76882384415261869036597544471920535e+253L, 6.83649581024319385239349778375109740e+685L, 1.71685218299087881635674367971803808e+1859L, },
      { 9.87925015299918398144785423598176370e-1942L, 8.71533681693018261767494097538810486e-1177L, 6.57384046217461429576867660255037671e-713L, 1.22572352048272001483240603010122094e-431L, 4.00210488905354704066065300402901994e-261L, 8.68831861142192461329723188826285819e-158L, 3.39344901437582485256542967096807177e-95L, 2.56737317400303409399563341354938201e-57L, 1.99325419812798016097936249998766075e-34L, 1.25616316381741439662336351160566326e-20L, 2.43090464171805920109733890177176767e-12L, 2.14048123340650521217897512392567060e-07L, 1.78351176282128440930436365965135303e-04L, 8.91532759780500845110690795259481658e-03L, 8.30018538939149499579790988285111688e-02L, 2.93820046590635175209882488313404811e-01L, 6.42167019459198241099164736507367056e-01L, 1.25676009815264777866804196778444950e+00L, 2.93241842564261090318837326991751597e+00L, 9.80176974669959846616864231368773370e+00L, 5.49217006325024175243162209529224299e+01L, 6.74415210957129787494085872629636299e+02L, 2.95093734929150449033134748913572566e+04L, 1.05011160163132749880805450719595372e+07L, 1.19276753677448525725842377258604019e+11L, 4.11976276783133288599595346140932524e+17L, 1.77581149088770071821845041850345600e+28L, 4.35594809668394640823177135833505819e+45L, 1.46965529638197726713059781629071505e+74L, 1.14872217888121999297587395528584026e+121L, 1.70677996026663975507589457654701634e+198L, 2.11819696469088836247505505584030188e+325L, 5.33443484802698043954530255107399872e+534L, 6.76327254647744376095801434158695958e+879L, 4.65552684789936854337398747780741866e+1448L, 2.41016379900581970812596189341460931e+2386L, },
      { 1.32133547680717018679108130408012819e-2200L, 2.77712159414975738481258376000179218e-1713L, 8.84696668578869367896605187021695554e-1334L, 3.02118052104884375490712624043751719e-1038L, 4.14765218378536031502305059306013033e-808L, 6.68597993758855914851785488336018106e-629L, 2.33347816130102648013054297856813910e-489L, 1.04733564854277330060008344877597147e-380L, 4.11557211597596026781639255888321135e-296L, 2.96892493585484037081544967750602740e-230L, 5.48318261457236849627593775240447695e-179L, 4.37952763140247483545256113532792419e-139L, 4.93764851567154537678682785150822008e-108L, 7.12882559733493186476365592485215882e-84L, 4.41155760043873077928273281441139888e-65L, 1.80163734340122138141929999274331671e-50L, 4.08343162992111053691005442960554881e-39L, 2.69713878716183124254075973537323621e-30L, 1.88906085681027307067677306620940234e-23L, 3.83661096593349300206254286709601564e-18L, 4.94655966889556498072463502849060578e-14L, 7.45399353944412484667368844933083088e-11L, 2.11213594306352633431635806968088815e-08L, 1.63019937992045999824523554635197574e-06L, 4.57540432069617055539176306478504351e-05L, 5.85085995568316321646161156635372880e-04L, 4.06605991446724517473851752061784970e-03L, 1.76336759267286733236155605055701792e-02L, 5.32246038588641268382381208383524691e-02L, 1.21978563400315778588175663550724559e-01L, 2.27813822026525499121180105863893077e-01L, 3.67880940693987971581072989565083693e-01L, 5.40739096387634225611080390746760034e-01L, 7.57649154875166910544734982341932295e-01L, 1.05508672843049871100505526038205526e+00L, 1.51442893723856346518379289394390563e+00L, 2.30704471829033068051765367243404246e+00L, 3.81265333029628098807578829102732392e+00L, 6.95531454976623073962406546394064472e+00L, 1.42580921706810654107345317832413199e+01L, 3.36149768965581810654207836598212633e+01L, 9.41442967189932119013509095554254321e+01L, 3.27468727748159184559404098660493044e+02L, 1.50134197286985544688570624625182907e+03L, 9.80710571996542847233767661790386013e+03L, 1.00962016775294251580231030319396563e+05L, 1.86536785996695038528848904417900709e+06L, 7.30953564053636331147337800282574617e+07L, 7.52773592822324283583724791889200862e+09L, 2.68310388335555167663870380176682102e+12L, 4.71267149903232936486329698993101877e+15L, 6.42020070484263570170339986066376457e+19L, 1.21445858766272510022421711137530603e+25L, 6.73690008798356003301023318149224184e+31L, 2.86191858315711641955678894569568040e+40L, 3.19319569382394077525802448772198805e+51L, 4.55425948316978466116193749735949380e+65L, 6.33381334198036002776230152386194807e+83L, 1.16685549796591838612114357506921339e+107L, 8.11568192390745193863805331868574764e+136L, 1.57270137093366013936347867540038862e+175L, 2.12621037868720966375758237292361135e+224L, 2.41043281776111273672384537919910315e+287L, 2.06426392024598778796244702435819107e+368L, 1.59767188945824373238194899871319514e+472L, 3.70147206080013271017541277477470426e+605L, 6.04524081068361649977908949693985827e+776L, 3.91244551907499740549590413818514135e+996L, 6.37601410091330839292544093678958764e+1278L, 1.38423256121282951104859278958281990e+1641L, 2.29003319158044389177689218691755697e+2106L, },
      { 1.06947516637266850888077780365944975e-2342L, 4.00222795925000100954191256688394740e-2067L, 6.14269929806784835508930822317395965e-1824L, 2.47129563578700314331274388349812780e-1609L, 5.94856549925636496124140508697277691e-1420L, 7.88155582901689454838569820447222612e-1253L, 2.37169815411747688234160253328737061e-1105L, 3.29576869517934854144272957025066495e-975L, 2.30295331039392476672974731888405196e-860L, 5.07778539549297920546786995561355024e-759L, 1.36309792114536681315299788580123700e-669L, 1.11902249499811588919802641601624761e-590L, 4.83171643219834730998170916090452646e-521L, 1.35092956692688343220060283908963635e-459L, 2.24199581450623280210827812157346751e-405L, 1.56044428013608715771945922727966521e-357L, 2.55762668699602425668170610426828633e-315L, 4.52598451420085229059456309653543845e-278L, 3.31498525601312348031091223503291312e-245L, 3.28982733272263633196035888826313749e-216L, 1.25979671796942324763554469354410763e-190L, 4.68781195667796137227478889031902638e-168L, 3.82963817403632291960149514795452273e-148L, 1.41008655666469634743601323447962987e-130L, 4.41480300426527477757453266178330825e-115L, 2.05796976085320113157119878898298467e-101L, 2.34167766563934625382241547560196138e-89L, 1.00611330011990340950410787621048434e-78L, 2.39886979987338732623660845760149501e-69L, 4.45828722999844038302160563656015401e-61L, 8.71674689717785941226626165471518297e-54L, 2.33608458451638324303027615256535578e-47L, 1.08390303338972947113853753614820116e-41L, 1.06995151808257796332112367831607337e-36L, 2.69524358925375181073032551986305924e-32L, 2.03423384800497240894806821496090050e-28L, 5.30019940271628291011490416486872002e-25L, 5.40212919469588209381281396175343198e-22L, 2.40511019215192441365347431063245533e-19L, 5.15573611643522185168933155470786681e-17L, 5.79902272788904112815174301897601165e-15L, 3.69205735629667547626167545850806392e-13L, 1.42267754195891351150562998109337966e-11L, 3.51991945554992222676432065045765653e-10L, 5.89117093052592385410771587163177426e-09L, 6.98414016831114732860587596174055273e-08L, 6.10836626587512983932161479872803914e-07L, 4.08537274605429873510095058624689151e-06L, 2.15678250632197565791431535444473324e-05L, 9.24312223411418671168580362381399190e-05L, 3.29629252828970123413244980995397823e-04L, 9.99865029818046920794114062146177746e-04L, 2.63018698849220190957883149658594289e-03L, 6.10412221855424181881652487802655548e-03L, 1.26903202004975552477255319538074005e-02L, 2.39564299641088688046037610330961560e-02L, 4.15643030399701933649627965940349917e-02L, 6.69982939046328122378009075425825174e-02L, 1.01314761859197945164098555651429523e-01L, 1.44998628043994675224702383164439509e-01L, 1.97976367297349804840344474368395293e-01L, 2.59785494162963270697436696871054011e-01L, 3.29861530130123082323166204892104315e-01L, 4.07887707779851847083967462215852975e-01L, 4.94160808801609892592520472039421040e-01L, 5.89951040448037491768065148251882226e-01L, 6.97865729437412689560491745121349024e-01L, 8.22260217393657822959055682985146030e-01L, 9.69774604378855851935565350832453271e-01L, 1.15011795553624730201138456447827419e+00L, 1.37728513637309570900453193107529701e+00L, 1.67149809630206531099733950813918712e+00L, 2.06235275406513270776777022052056344e+00L, 2.59399765677200896819549138553557629e+00L, 3.33381238373592320485907218391630985e+00L, 4.38727553184963415451944389447389070e+00L, 5.92410234721624433989387409845478071e+00L, 8.22555400895276009470529071921123608e+00L, 1.17728847794365554942833299884164414e+01L, 1.74186946732944479219083797984245546e+01L, 2.67321347727075416316453470949013980e+01L, 4.27248599090065202592682231550667436e+01L, 7.14514957498311763066605425195361350e+01L, 1.25726649744291050649669245834238885e+02L, 2.34268740301195719755233273502547873e+02L, 4.65674501968291917847819643280019736e+02L, 9.95892168086429019721630255623709337e+02L, 2.31370536252976840857940169336286210e+03L, 5.90416785827987120394897587933071506e+03L, 1.67574959687797819327588127121232764e+04L, 5.36589048987894263464971617898730248e+04L, 1.97001995592318850388501435517959975e+05L, 8.44594440147401724253246450884183480e+05L, 4.31706343383048349864263584742746589e+06L, 2.69343047021169619975848244859698568e+07L, 2.10656056871936774519731164741615606e+08L, 2.12865839525415045164207708849921266e+09L, 2.87579863694102104115679841861873156e+10L, 5.39962441480030320721171298079837788e+11L, 1.47228407211216271679535227733468003e+13L, 6.12702071280434890818621133461421119e+14L, 4.11734705402761288562221821245591184e+16L, 4.76248383365979073253985584808588348e+18L, 1.01935525113816768714253607040570786e+21L, 4.38231774892874881629281035002570378e+23L, 4.15257812330163302032181919338837514e+26L, 9.63578634121366174238646117049249945e+29L, 6.16903528716345189143890155825118447e+33L, 1.24741073700366469803472774007991980e+38L, 9.28495127856215607061013544719468024e+42L, 3.02620640278402325128627613072607942e+48L, 5.25740318414851642612290423653366751e+54L, 6.08377705676929998408087326671143748e+61L, 6.03621188684706784050917948535663342e+69L, 6.83604430657324601596912909835543184e+78L, 1.22207736059289881632444543839493409e+89L, 4.97963284347586492335598341460012283e+100L, 7.01286464121514720750130754209620950e+113L, 5.47091248474436718447273702781160082e+128L, 4.03491215911361460129480089373280294e+145L, 5.15550681905479930138211789209766809e+164L, 2.26698817812971993667638253759384337e+186L, 7.46678442214100282100191345089453705e+210L, 4.44695748246229233245164503407419774e+238L, 1.29997993210113625215904908209705899e+270L, 5.78362417210637255304620183172670719e+305L, 1.41167148054040165920786770540444441e+346L, 8.08286570061195033556699134866983755e+391L, 5.63304784971987527997364020512160855e+443L, 3.08690662195072763023845781932813249e+502L, 1.10165417416868743288711300669064477e+569L, 2.80994196635775688984524212097397961e+644L, 7.73382325928152223721046490397169922e+729L, 4.97761231747547736253787393388911251e+826L, 2.44530978920516083381477794744261645e+936L, 4.76039467365174741738996705683688855e+1060L, 3.22588883912409917905940907837482370e+1201L, 1.21298392121243707893541014829026239e+1361L, 7.92540396126882607375372502759559361e+1541L, 6.05643048475311304903673843949239315e+1746L, 8.67085957383156175019676264069447501e+1978L, 9.95105409544177778456070528537389733e+2241L, },
      { 3.82052466785182332086738926018596976e-2417L, 1.53216690964008904863800245911834695e-2270L, 8.02799618255145245433991916259679940e-2133L, 1.89790062782857972828467648673787073e-2003L, 6.48547054333991583933133337347105329e-1882L, 9.56340910858469468709935501045338939e-1768L, 1.70023842699605420504066989530814298e-1660L, 9.56797730413407205631782708332785103e-1560L, 4.22020698781412857368210883093245672e-1465L, 3.41965451589094380116779072034053750e-1376L, 1.13314142099471085666864638284317139e-1292L, 3.25613842818879565325184054056113549e-1214L, 1.64407833236659941146136019432680305e-1140L, 2.83169222847758408789499219075138705e-1071L, 3.10256729477751738834780026930224437e-1006L, 3.88327366035249790364432313659702349e-945L, 9.62325937366412073215674752789621450e-888L, 7.91532304589191166500084311712723764e-834L, 3.51089649105332123383908279313304145e-783L, 1.32489192907121558509586493733802170e-735L, 6.52783588426344440105254553576625446e-691L, 6.27953766456679638940801082204333268e-649L, 1.72112316179301192237697323835800001e-609L, 1.91704593421434794371085034440245942e-572L, 1.21131389706331685590249812201530647e-537L, 5.93984004695174374389288417212720344e-505L, 3.03411224655952005849867823710301306e-474L, 2.12875962525689396085894169728448953e-445L, 2.66000529174336615171418420545198600e-418L, 7.55590515004675433995195831838744150e-393L, 6.13626926542527991664012204881781789e-369L, 1.76713467741974813886350949888126351e-346L, 2.20926972666826146585381131914638666e-325L, 1.45005559535857920730397373363607727e-305L, 5.97336892939272666152938174641382891e-287L, 1.82640350668529645297131829899777179e-269L, 4.85229717585781075750778298691866187e-253L, 1.29884069259764069953465811099712413e-237L, 4.02542740473423831131996110176920188e-223L, 1.64606390223494048083381164318527223e-209L, 1.00404640707773475495017963404550319e-196L, 1.02517549226209525963922466256750740e-184L, 1.95259262992798399878532201786889482e-173L, 7.68026595299128847616027073204274409e-163L, 6.86431799704342420135947045710241387e-153L, 1.52498555897006686273111644122247925e-143L, 9.16240838899174700108797638312124662e-135L, 1.61152978600632900525315532785259119e-126L, 8.93873956545614240431663830324423008e-119L, 1.67683199253457467411487735035598912e-111L, 1.13606831265305889540738444180574550e-104L, 2.95677983624992268135074874283992128e-98L, 3.13261928574067484199912376683710488e-92L, 1.42665699792617318967988994714464344e-86L, 2.93948527551792820481373266955543872e-81L, 2.87496940202324055984269030763057181e-76L, 1.39640503864001278539518135367012235e-71L, 3.51418022897052500616248358954553443e-67L, 4.76840843576304417158706032926840904e-63L, 3.62171076308676895946675701059866747e-59L, 1.59482985688579594373676029076259231e-55L, 4.20842453488002122572360942178919587e-52L, 6.86443229233076886165541867675398494e-49L, 7.12571665807519317291983672203309613e-46L, 4.83819502081497047116760866720006226e-43L, 2.20465542430951342610708115485440884e-40L, 6.90709560806486502312714752170868272e-38L, 1.52197218506174728353062647705743598e-35L, 2.40955219490267088425869662208370235e-33L, 2.79630904534258562440727543829590381e-31L, 2.42396861904265607424793591553560695e-29L, 1.59749866280800688175296781099864391e-27L, 8.13840485655638493122774039812902482e-26L, 3.25536762868063353648331484818545825e-24L, 1.03750216774182187079375467840556085e-22L, 2.67108014795025059201469276708831348e-21L, 5.62745845137509901846677188959226735e-20L, 9.82072302589238577426509279192638536e-19L, 1.43594941796544038692542800708134381e-17L, 1.77810682024353573557752337941414159e-16L, 1.88354537738694939365287613122504842e-15L, 1.72308010102740811988992770129939744e-14L, 1.37343705888395103661321882765344100e-13L, 9.61866975437486408023966746429500914e-13L, 5.96531965279554928068743818930854269e-12L, 3.30041201040702869600996143071210790e-11L, 1.64031748053937249488182380342568737e-10L, 7.37124149693192472685858181626175497e-10L, 3.01344400817654411848957462000254023e-09L, 1.12717586759651920251576226824261590e-08L, 3.87857240586881913126338809706852880e-08L, 1.23397923410236586482402725064781148e-07L, 3.64729321175679321144258676923993113e-07L, 1.00602028308961790056200899477538690e-06L, 2.60043020837597212496425152713721292e-06L, 6.32419483196640694034831908133223537e-06L, 1.45245521130769448827445571183003177e-05L, 3.16123436155465446648693209598810124e-05L, 6.54176706996526406822348966226430775e-05L, 1.29110196844657112529538619169285993e-04L, 2.43733749771260888436783011892865212e-04L, 4.41314232710451843972392755372463603e-04L, 7.68377076370070526296257588743541811e-04L, 1.28957360159046548970572233319638690e-03L, 2.09099199558542466132447951593809118e-03L, 3.28264889533211879885382599961585359e-03L, 4.99964828308048181967481389975237633e-03L, 7.40157019965966236427260197490815304e-03L, 1.06698107000950941317328124490909795e-02L, 1.50028172314973599445873815719412949e-02L, 2.06094173096225141746698709598883056e-02L, 2.77006834377238972506888931261097291e-02L, 3.64809756186562309685025492261970114e-02L, 4.71389254316798953989957737554750583e-02L, 5.98391571230828379216832783213494747e-02L, 7.47154814906505012222514984217472816e-02L, 9.18661612946071289937159187843364627e-02L, 1.11351656134035569001793372114249223e-01L, 1.33195038632804266520840073116724199e-01L, 1.57385060631367271607409153973946866e-01L, 1.83881661881487488373471043449811640e-01L, 2.12623571664368840241407884037664753e-01L, 2.43537565151706738569000265542366947e-01L, 2.76548903119165441078310983457425711e-01L, 3.11592501651099485071285279910294521e-01L, 3.48624439429573943493094167043683901e-01L, 3.87633503629295959856799538075546008e-01L, 4.28652590594010568395685064250714546e-01L, 4.71769904763931628640496795425394984e-01L, 5.17140031351419396580841715288928105e-01L, 5.64995090385812394527782017926592446e-01L, 6.15656310347513453519162300561374394e-01L, 6.69546490104796171397630195486014552e-01L, 7.27203952634969644713241799501762825e-01L, 7.89298740343220248874987146571520149e-01L, 8.56651969968232039115451358927023892e-01L, 9.30259468685761614462731685806948992e-01L, 1.01132106970032064396659635282297935e+00L, 1.10127727814330022383089577768540741e+00L, 1.20185545627576044892754956074556492e+00L, 1.31512826035991923618357153643853889e+00L, 1.44358784334344214073911010244139658e+00L, 1.59024039033833533691274024525328135e+00L, 1.75872697808494229887678505228447105e+00L, 1.95347868511083813992081514830258545e+00L, 2.17991652311273637078595706271805585e+00L, 2.44471039181719695746189348680321367e+00L, 2.75611627927753518177812012686624980e+00L, 3.12441791418753602036992277268322970e+00L, 3.56250886504706839116501560040688156e+00L, 4.08666490215568913223544572173213403e+00L, 4.71757610938540508471197077535051869e+00L, 5.48173646271881799502596740262131127e+00L, 6.41332945820485042568468273867868285e+00L, 7.55680806548694121454415120092330575e+00L, 8.97045530296518503588356381213781124e+00L, 1.07313427967993620782556935698599286e+01L, 1.29423018529722651121181556330024768e+01L, 1.57418213494311260952898800974816324e+01L, 1.93182476307453478113061967921720372e+01L, 2.39303783823625958630927317369382100e+01L, 2.99376708353783067254755016259184880e+01L, 3.78450834852449540061461594823397372e+01L, 4.83704762272558588741438572877597086e+01L, 6.25472526597377774327448466299291048e+01L, 8.18828352821743059070051234981039846e+01L, 1.08606901707010877244713106575898588e+02L, 1.46066165572767230787668444759129144e+02L, 1.99362305840947908376533577788614663e+02L, 2.76400238552833065835531188582929358e+02L, 3.89641361583293015149089847924625396e+02L, 5.59090899610510721528958323850657754e+02L, 8.17488717203324414010281686447524861e+02L, 1.21951707162988010780952106506178041e+03L, 1.85849349228255485567535180727353270e+03L, 2.89733723527987926224598061281533634e+03L, 4.62742546807418291953673117361684539e+03L, 7.58336312821976325886140349375631572e+03L, 1.27729327383211423049145373221610484e+04L, 2.21512103826316975884795094360035657e+04L, 3.96282043351341952537133210137390141e+04L, 7.32802430573798143066819092764851057e+04L, 1.40370956832174099704802040273767653e+05L, 2.79169596050238213326100612230710484e+05L, 5.77851587758831222006931720166846758e+05L, 1.24809297513500168708148829224766608e+06L, 2.82070529249367448006501528915158547e+06L, 6.68996112716468438709724989481100128e+06L, 1.67032788479232576626369550400657265e+07L, 4.40490689805489416603930530460183080e+07L, 1.23130681270188214533524272282450985e+08L, 3.66207397185135919222246119544674802e+08L, 1.16348659359258561581772655083768774e+09L, 3.96573293875598360514367483348534276e+09L, 1.45675716212887953801924286655056170e+10L, 5.79499965416005488651946885694232582e+10L, 2.50933409077965035999243371354527615e+11L, 1.18927611174028691004735513584719908e+12L, 6.20525591975150642720259081452168385e+12L, 3.58662837399254785268923887857175768e+13L, 2.31171019709164125040709702173384790e+14L, 1.67323267937848597801576240028274938e+15L, 1.37027502568098828908714302508292741e+16L, 1.27982243687884271019103805104886411e+17L, 1.37488860693662981360604951513000987e+18L, 1.71428840498039054019366079813545633e+19L, 2.50480806231532255799256901000499530e+20L, 4.33295295852175693203040531656862705e+21L, 8.97105957110885650118812876460940777e+22L, 2.24900305994354872692347113786351015e+24L, 6.91168391281314093810281357471213740e+25L, 2.63834638817928808592115398804015843e+27L, 1.26855240154452496477895137567276885e+29L, 7.79793937981300078349411089481829644e+30L, 6.22628875244383647460445596438511369e+32L, 6.56725010457698317225350638943220187e+34L, 9.31627142136562734426709726532855891e+36L, 1.81178764804393998725363596058882739e+39L, 4.92965709962256757369004572197380439e+41L, 1.91768799703732643541855471608092309e+44L, 1.09145348658581711775073448931123050e+47L, 9.31447898399194268806130290915959528e+49L, 1.22344767896866261290498650189275522e+53L, 2.54310892512613676637994972961780173e+56L, 8.61698733620595754888079045068786616e+59L, 4.91184107780000171015313299477683427e+63L, 4.87081518596258225878777239561036595e+67L, 8.70837775558769802630502082888045797e+71L, 2.91582292448921588681558819522688100e+76L, 1.90391730055994678151666350663451590e+81L, 2.53108226877386875309925515429807108e+86L, 7.17216745327677632991407821508792163e+91L, 4.54861980767233963821254711174889633e+97L, 6.80080274478233195724672872122891978e+103L, 2.53345776553427904264312812304869400e+110L, 2.49408335416956941404546574182091219e+117L, 6.90831393215899350999406217643925461e+124L, 5.75535983268412076946168827231484587e+132L, 1.54830478033444708124258729693098368e+141L, 1.45063275961171552593742069490051326e+150L, 5.13010920252030309322157027524520131e+159L, 7.46051857949218271625227134136784373e+169L, 4.88752936708905952864518859469811383e+180L, 1.58944916201304126296752123375100638e+192L, 2.84523891966215233479427916942262824e+204L, 3.12952175960493682046861415263020503e+217L, 2.37781288259852734112773228061886887e+231L, 1.41367934226264567696479785830917722e+246L, 7.50969021632501856297416076380790606e+261L, 4.10517599617698914085666788248374635e+278L, 2.68396648103470580650904518179723689e+296L, 2.46300967872463438914762759549033346e+315L, 3.76174998591379278617806685202641214e+335L, 1.14633505463399943022299924904504898e+357L, 8.45422653379120782973725460577460520e+379L, 1.85322036769366179986950823638998209e+404L, 1.50271794914844405931928670543344206e+430L, 5.68929995100375677797682848265066614e+457L, 1.28862756145762463302669091988538860e+487L, 2.27343599851542201859662096966470860e+518L, 4.13728378647859018626664278751490836e+551L, 1.04732962553987345589103224110962753e+587L, 5.07014442635227334186619020470235649e+624L, 6.58680466182542570069097938912153401e+664L, 3.29371230247950075139843014589562364e+707L, 9.30666782392729638823808617314515754e+752L, 2.23612238313938342458217832467289997e+801L, 7.05882324392654993625063412318352329e+852L, 4.65192550274585790446586136162724180e+907L, 1.04784460655109112066824552095698697e+966L, 1.36343029663707923380723270729436320e+1028L, 1.79162856963805080764174564008589158e+1094L, 4.30919215868732016052648412277675057e+1164L, 3.57261639332919274471338835639300206e+1239L, 2.00290113950626521919894557483624467e+1319L, 1.55571074565195195355910058210877816e+1404L, 3.59251763773431142332284400696144078e+1494L, 5.55982146221013438636549073847422510e+1590L, 1.36984614652427999087777938149635656e+1693L, 1.34965612662661283517635152647662065e+1802L, 1.41743602676194002368496649429520279e+1918L, 4.50570145843130891260063218260174869e+2041L, 1.31668432261399763150332726270146708e+2173L, 1.15414484031205360166874295729734758e+2313L, 1.06861618403043050305720276519487893e+2462L, },
      { 2.99827136671822203139659705254947108e-2455L, 1.24882827026046387418324170521821293e-2379L, 2.44988423001973909940876922057761351e-2306L, 2.66927540888423924256197676379835451e-2235L, 1.89512300782142727523964560129359997e-2166L, 1.02360851047736978252291085443458258e-2099L, 4.88730957021167285814599122540648196e-2035L, 2.38575973739960063984613176811112623e-1972L, 1.37101430897591401677957283544539320e-1911L, 1.06333264218828160158212770967007648e-1852L, 1.27067801926958817891969964688113127e-1795L, 2.66010333678426336904594890897508436e-1740L, 1.10484167956085896672647638913688653e-1686L, 1.02711572445228877679664077673865962e-1634L, 2.40228312498468537472913439313888633e-1584L, 1.58314329557188545251248547330891601e-1535L, 3.28094781159949863603124081342946968e-1488L, 2.37840601013012306745513950331950428e-1442L, 6.68624945754691841776166731458196006e-1398L, 8.05587986859135128455444199209469004e-1355L, 4.58316096287519065423952255855034306e-1313L, 1.35248343166964409448147921555689922e-1272L, 2.26752160966952001157168922897574770e-1233L, 2.35909278572023604645731071940961914e-1195L, 1.65903351294385543451470844717805067e-1158L, 8.56804094411838080986008744972959423e-1123L, 3.52140713988280887124946897137042691e-1088L, 1.24502439015653832112887580419259567e-1054L, 4.08361208615792538039018647193947162e-1022L, 1.33685935274460284904788097419890160e-990L, 4.68914584779066055505548198640894341e-960L, 1.88761464755010821415650640440428715e-930L, 9.32118389977759099804089947706407268e-902L, 6.02287072293155834327191570771930380e-874L, 5.42106639858656175956621834625906078e-847L, 7.22191826970524227339284882890548040e-821L, 1.51020072122639353372149509671977024e-795L, 5.24775126181523166368024963040824842e-771L, 3.20220580014087429159629013136223785e-747L, 3.61997704502523196516304924921584345e-724L, 7.98492800624004092804929885145612391e-702L, 3.61392988815279171930655913638563273e-680L, 3.52366894930304180660437805611575996e-659L, 7.75941790954346674475792064117877807e-639L, 4.03982705605992279012386676527754690e-619L, 5.19834669028776461984659088482903679e-600L, 1.72588931588477933787290173538689408e-581L, 1.54137268468058258004178745033532316e-563L, 3.85562019823508006565291873935904091e-546L, 2.80916830174896024691081755907263354e-529L, 6.19212992045820718698085583700822639e-513L, 4.28406369726641450642830646435371014e-497L, 9.64070107276137860266813577293097959e-482L, 7.30474502980487251348070249880206420e-467L, 1.92703875472217605577664104239882285e-452L, 1.82836317722867439282495454446115074e-438L, 6.43850535226391166629598797094394826e-425L, 8.67564527079512826366041439069686406e-412L, 4.60732378382987117665981750506044471e-399L, 9.92355019653945096209219622811507181e-387L, 8.91282113369599082908019662526444168e-375L, 3.42910268669860807072991706793596262e-363L, 5.80083861220020883214825204353252963e-352L, 4.42512315593917892307328787082985460e-341L, 1.56000404760553377203351190297454435e-330L, 2.60259125151860978088621742839596354e-320L, 2.10262994916038774260354592873630300e-310L, 8.41173460487346663387701675407432002e-301L, 1.70280612724644563266907394311476267e-291L, 1.78116409086610297225917597830385624e-282L, 9.82480313804105048460709403983300504e-274L, 2.91457791899830525764575140294420231e-265L, 4.73965774855838520901565109021450612e-257L, 4.30398176296336788976377718753404885e-249L, 2.22193567988028967419035853512154624e-241L, 6.63553772275767193165934337486647504e-234L, 1.16578412206922730540647811503608703e-226L, 1.22474675486304476176402962758102117e-219L, 7.81684108965031293490823429466674471e-213L, 3.07774321188900147460180897292349446e-206L, 7.58759251414243265302591311385078772e-200L, 1.18823395876906229284835410783359514e-193L, 1.19864235179097517931485807315756884e-187L, 7.89484815058732018316693655376229205e-182L, 3.44001585820470930356951346759790116e-176L, 1.00429282413458278190733306028479555e-170L, 1.98881516262612704197372631141044676e-165L, 2.70364023416269358290386889178454878e-160L, 2.55241036356528886321774073237366662e-155L, 1.69227628517124062919375344585284149e-150L, 7.96587271931569005985245219884752131e-146L, 2.69038016965415710141558028398178580e-141L, 6.58640142296301821553052201959182941e-137L, 1.18042002159083828066729019346791192e-132L, 1.56367343741949029569249898774237153e-128L, 1.54531048534737740803236941099285613e-124L, 1.14963627239221457328345144952502533e-120L, 6.49493107141223206477306791588740146e-117L, 2.81018975962531457950258683552245790e-113L, 9.38876009983047538463236277234608306e-110L, 2.44144614978077332940218030178369234e-106L, 4.97967380423964535764132428105410190e-103L, 8.02629250855504170950085275141889693e-100L, 1.02975774422556528975197839046564511e-96L, 1.05903428462392788554333849690582167e-93L, 8.79012224539705420215807394822091933e-91L, 5.92725904620589386119574720007659266e-88L, 3.26785695641876626125438772016710733e-85L, 1.48222090912512196733576972776187035e-82L, 5.56431172687041304333655416134228735e-80L, 1.73894644850180973193834439393406052e-77L, 4.54974501627115811330277659341709811e-75L, 1.00204791018402181301004224966380115e-72L, 1.86763466487726841127561204857118883e-70L, 2.96092907372076963741238677845263773e-68L, 4.01290356278003207547213477434374557e-66L, 4.67174973180940285995675266523420122e-64L, 4.69376738484344030975102668254954324e-62L, 4.08839867480777582716177965835471551e-60L, 3.10086294017966876521788239138705860e-58L, 2.05664262860193002265928563040484563e-56L, 1.19777269867460483717671801459380654e-54L, 6.14987857896674930479991965502306936e-53L, 2.79459594105487367405294144558243202e-51L, 1.12814222153195027375952194262805170e-49L, 4.06053088698370288660361299969573584e-48L, 1.30770277764601303984357205166621959e-46L, 3.78118998998858848129267831928491324e-45L, 9.84875712554165931773609676827898732e-44L, 2.31826871061275836674260034702071180e-42L, 4.94684766719278736886703293137447413e-41L, 9.59814033368779163526953048426266162e-40L, 1.69828465632158908850726775955673984e-38L, 2.74808806067694979431131601699073400e-37L, 4.07794734980576448596813627579221148e-36L, 5.56417400808680411162969051547065270e-35L, 6.99889732124326604821939972498753725e-34L, 8.13608686966403922600365716655177400e-33L, 8.76218322965140584555660990677761542e-32L, 8.76276637192578280309577248461260176e-31L, 8.15628170980170063345810303188191933e-30L, 7.08149199986036059275640615879247609e-29L, 5.74736606938180421303095720610698091e-28L, 4.36944079330436317602800446372377792e-27L, 3.11795190731786551713306066746724497e-26L, 2.09240762901978141675541589343048983e-25L, 1.32305299259448285811442078319939693e-24L, 7.89691997711578359267081292526106870e-24L, 4.45716605711992632203887297614028920e-23L, 2.38302054707699751676620265710858837e-22L, 1.20889613263470803214753580918237251e-21L, 5.82833462566546297036431040060608322e-21L, 2.67469784973934035752489871481337927e-20L, 1.17014393860453605367131279149462226e-19L, 4.88739480774243667158625429475089610e-19L, 1.95168062031382677617975635339602042e-18L, 7.46163208304186839076058133462415646e-18L, 2.73485791500251557979443670181385630e-17L, 9.62223074873981898938153222962813147e-17L, 3.25389617800670808719193210653774700e-16L, 1.05888451003262711834291997535706834e-15L, 3.31989417456924550647348023471492055e-15L, 1.00398818028880718004903688096398433e-14L, 2.93186753434992804142094819281287207e-14L, 8.27635883877837412701034478711895464e-14L, 2.26082510653047710401828799240495861e-13L, 5.98228146965673437508140635644104873e-13L, 1.53484657642706271558550260287175188e-12L, 3.82185576688620308819449258941130914e-12L, 9.24474797424185856221360345157220406e-12L, 2.17427909729908200066806520833424743e-11L, 4.97635090813511876195380613765617708e-11L, 1.10929412007484858271399815868498117e-10L, 2.41032585809105707126681740366639425e-10L, 5.10905559690208602233885434283852003e-10L, 1.05723928928894498681155298267174319e-09L, 2.13744795688281626795384543565327267e-09L, 4.22492806022042378177884800843897317e-09L, 8.17046853836402216132200964074977087e-09L, 1.54693324186061707441588930749976281e-08L, 2.86930859292637487108510416894642893e-08L, 5.21722334965282980403485300265738444e-08L, 9.30518593041943674159836371480799295e-08L, 1.62889442240265382985512933741863589e-07L, 2.80023159222713498193152226335713707e-07L, 4.73011069732904671695491353591017402e-07L, 7.85526363421471709139200654941264315e-07L, 1.28318979177475296348420727772184965e-06L, 2.06292423671439573062357518920248604e-06L, 3.26552286428871054459476770292453844e-06L, 5.09222413107163744070769490625113325e-06L, 7.82617466349549247600633222001776503e-06L, 1.18597235737360853531010814128200937e-05L, 1.77285167845861097132511756107809923e-05L, 2.61533347347083551795448324767635695e-05L, 3.80905294601878233958593851046717171e-05L, 5.47917574609632216617322114415284106e-05L, 7.78734661207721580410532787755478685e-05L, 1.09396271310786841597436260479638662e-04L, 1.51953965111990518244589142984584154e-04L, 2.08771424329052859462812228725402767e-04L, 2.83810167897154635400111152886276561e-04L, 3.81879755641776745662197699923565149e-04L, 5.08752658539288473040921063606067979e-04L, 6.71279691879016478989727452815073290e-04L, 8.77502619869410964596211481799673429e-04L, 1.13676014562695660388935295197579588e-03L, 1.45978322435393926290231969328136083e-03L, 1.85877550576562291461373383997358853e-03L, 2.34747474413929171623192976971642926e-03L, 2.94119122257973574562047404684570218e-03L, 3.65681962518947136825693675506580632e-03L, 4.51282135037802008049500003706382224e-03L, 5.52917497487431572466643627095469859e-03L, 6.72729342693880289247231232743320046e-03L, 8.12990737739402925191380506695287882e-03L, 9.76091537148098089988167779205704956e-03L, 1.16452022628003896766716791241908478e-02L, 1.38084285310255098081489577923406060e-02L, 1.62767940269060299249627834088541599e-02L, 1.90767805535439719611430291452241110e-02L, 2.22348784290424057418829018676115193e-02L, 2.57773027557106041232220835190089403e-02L, 2.97297055256785242037107379259364943e-02L, 3.41168899105681014303940274877700652e-02L, 3.89625331103873045168982262850032695e-02L, 4.42889239784348614272085648656341809e-02L, 5.01167213062401896725837060595306942e-02L, 5.64647381631055655160905842942135176e-02L, 6.33497570613638646379858081046281019e-02L, 7.07863799874088410337603891340254733e-02L, 7.87869165286187403201277486249691639e-02L, 8.73613124671846027274852311959507760e-02L, 9.65171203330026184766273626679604986e-02L, 1.06259512537229504582581766840088037e-01L, 1.16591336871525032405196867775170242e-01L, 1.27513213378027801743713746335418149e-01L, 1.39023670890726644461647310110564995e-01L, 1.51119320935163034926767802526948718e-01L, 1.63795066161333054101624624583286900e-01L, 1.77044340081249140375155906842322507e-01L, 1.90859374591070988709995096163492901e-01L, 2.05231491577749618625157075644473454e-01L, 2.20151414838427133582099008248376327e-01L, 2.35609598571509171617873703655361587e-01L, 2.51596568823441496049692748064095951e-01L, 2.68103274485319808338763876413696112e-01L, 2.85121444718080293120944366000785895e-01L, 3.02643950033175240463650928525241062e-01L, 3.20665164656273759511495325736015581e-01L, 3.39181328243896232934423095526947639e-01L, 3.58190905496894238632513262989320923e-01L, 3.77694942711148444854921008885136613e-01L, 3.97697420816736709936441112125746846e-01L, 4.18205604975383785180004107895309545e-01L, 4.39230391326913892071217294036947977e-01L, 4.60786651994838310108569244001462874e-01L, 4.82893579976783682783259082280697090e-01L, 5.05575036056380615456191240843198089e-01L, 5.28859900380164343616138823034844308e-01L, 5.52782431848141026163801711629569094e-01L, 5.77382638973537667728623589540847426e-01L, 6.02706666380887845415163251209767664e-01L, 6.28807201638412861233038059400000985e-01L, 6.55743907668438480076422889144751851e-01L, 6.83583886557560546105565188637319809e-01L, 7.12402181207131050122140520074497354e-01L, 7.42282321936634874071712163942668772e-01L, 7.73316925891616774766609626518222875e-01L, 8.05608357929825762678392686079844024e-01L, 8.39269462582114444255866071824225817e-01L, 8.74424377728731780655872302934188327e-01L, 9.11209441820152654384317758539800255e-01L, 9.49774207830947999703385257169606442e-01L, 9.90282578695719860661750084965348602e-01L, 1.03291408077266220499949941381670461e+00L, 1.07786529395310786342015831094913754e+00L, 1.12535145943113425409772316889306152e+00L, 1.17560828892019106387681852443264510e+00L, 1.22889400231711865029290591255755794e+00L, 1.28549162454200134646596760738351906e+00L, 1.34571157661211478758710729669915294e+00L, 1.40989460104228631141154693756330089e+00L, 1.47841506752326819907071392253335661e+00L, 1.55168471165732988584072993312632620e+00L, 1.63015686749586045646523839878119043e+00L, 1.71433126392888582869445858515841385e+00L, 1.80475946586097450583979351681595524e+00L, 1.90205105385821569910651758727035418e+00L, 2.00688065090883085704250790253631297e+00L, 2.11999592251508776978462701922381176e+00L, 2.24222669701791864392841075957133612e+00L, 2.37449537743872890090165532424204567e+00L, 2.51782884491687412950125924770996677e+00L, 2.67337208788498443977901451698681543e+00L, 2.84240383149636938582597352632467033e+00L, 3.02635448975787151687626271616744396e+00L, 3.22682681985641084640836568780538092e+00L, 3.44561972615851906798292101239468486e+00L, 3.68475574257857058166416238846674883e+00L, 3.94651281922700641882215705186324970e+00L, 4.23346115586300426882702466682124288e+00L, 4.54850596485993372358839139642384894e+00L, 4.89493721528314838282250644869812537e+00L, 5.27648761361579143482640951008870741e+00L, 5.69740032348777653019331418706396336e+00L, 6.16250822618479874280298612217822529e+00L, 6.67732688781902305621720870171254858e+00L, 7.24816384288680618425673773908490765e+00L, 7.88224734633420137754997440176783259e+00L, 8.58787841076847338040196250304371454e+00L, 9.37461076170556588083589376387132811e+00L, 1.02534643490360208202755813725095827e+01L, 1.12371792943596952958405098972799626e+01L, 1.23405186912073323038589589843390563e+01L, 1.35806305843369735705552654775963352e+01L, 1.49774818320198815684165550755584428e+01L, 1.65543795252306978072060668000283508e+01L, 1.83385993586213963662082373601202455e+01L, 2.03621450242974894275535474068497776e+01L, 2.26626685943754163096516986406459795e+01L, 2.52845893136112912405786286347442878e+01L, 2.82804576829875229843082739941801350e+01L, 3.17126237580911006574531329780573620e+01L, 3.56552839704483033860162655552985983e+01L, 4.01970005016327611735412627281826315e+01L, 4.54438126123299012690083228854751285e+01L, 5.15230920902650058925899224137885947e+01L, 5.85883374425407037894686827319101058e+01L, 6.68251567023112069515824430636222454e+01L, 7.64587608768192360600680527705062884e+01L, 8.77633846894782777864438965135974549e+01L, 1.01074175868700380244224060820937268e+02L, 1.16802258994842453002226105258170771e+02L, 1.35453898714114297683765730207366657e+02L, 1.57654978967903747896778920825912898e+02L, 1.84182405906460887204718250035971904e+02L, 2.16004568481915370203941996632993398e+02L, 2.54333702516246823953928460722661707e+02L, 3.00693971336392029288524445812313026e+02L, 3.57010397089553597741305454749296236e+02L, 4.25725590115811669827894375931285872e+02L, 5.09953725643224718975881101989299450e+02L, 6.13684687521816216675628323432708789e+02L, 7.42056139017496594867562213902074498e+02L, 9.01718069398279102081695287947885045e+02L, 1.10132394119371945146169423670051948e+03L, 1.35219615071533062822615520086342752e+03L, 1.66923291068630661586092129359067647e+03L, 2.07215152332054272741081984137877196e+03L, 2.58720328209038570262408045410782795e+03L, 3.24955382874419473343636700904469745e+03L, 4.10660860213453501373964026030452875e+03L, 5.22268955795277619388623264624171874e+03L, 6.68565726355089669998707553230747827e+03L, 8.61635357331041935557199135249564277e+03L, 1.11821636876213398227716019227207708e+04L, 1.46166395927660444135310088609126834e+04L, 1.92481111548503807855046302083949422e+04L, 2.55417294787310992653570084425920733e+04L, 3.41617486573493312682855974958999477e+04L, 4.60645561151339657587241032816274431e+04L, 6.26388222783949624156629031841856307e+04L, 8.59184957335087735876745002390386986e+04L, 1.18909441895224029432207119296169161e+05L, 1.66095033938427884484826987675525104e+05L, 2.34226252811038979343276539537003434e+05L, 3.33568602948986258437654580586715228e+05L, 4.79889988962864687601246140266254885e+05L, 6.97663061660509733330079420786394691e+05L, 1.02527964914474052697145671441285507e+06L, 1.52363581455429196593138247075709981e+06L, 2.29042801548317740734746692037333714e+06L, 3.48424008370597272699707163872949391e+06L, 5.36561882022124111785765119998290944e+06L, 8.36799594465384427071057649003523823e+06L, 1.32217203482688374239659605009398927e+07L, 2.11738506478689471767346755297403608e+07L, 3.43829654204789362343598103327212044e+07L, 5.66383460344826705635075490736605924e+07L, 9.46890531446099216978642309212033877e+07L, 1.60736860537955754766368401888105613e+08L, 2.77184337816824251216810756452423932e+08L, 4.85821891491727553231947839400706241e+08L, 8.65895043719996967916785083318862010e+08L, 1.57023595787763814315717820663521931e+09L, 2.89877916582589084595240112564380085e+09L, 5.45082862939618857716273065316347458e+09L, 1.04462776299019818355322697028773431e+10L, 2.04161928476231748290058801874576093e+10L, 4.07167362508726715447348035164126587e+10L, 8.29154750329098975446073110291771593e+10L, 1.72524569678310615987347680670953976e+11L, 3.67042596452982637114653516749325639e+11L, 7.98985690430384590881851382687868095e+11L, 1.78088199348481822130071407591775159e+12L, 4.06753710066430378340194042741373814e+12L, 9.52719664341172412580158176782396629e+12L, 2.29025392291311484677517351429897325e+13L, 5.65517100257158446400477892790807623e+13L, 1.43556057453169991412842468344287817e+14L, 3.74966777273576618576956864828262686e+14L, 1.00868013060119404846072345341681988e+15L, 2.79711270328189457750729359218050034e+15L, 8.00353033476527491290557180677857043e+15L, 2.36538022352308798092653239020367364e+16L, 7.22793756862980926623333746953184515e+16L, 2.28603237125675384523138361921766753e+17L, 7.49169357670736182787260118136384206e+17L, 2.54681528722697062733706069581540403e+18L, 8.99167123461421679854681325569917778e+18L, 3.30088716836331393064955543786333400e+19L, 1.26155602488854061815298573346753948e+20L, 5.02601059129997078887664234449072840e+20L, 2.09003840003334609133024432910535654e+21L, 9.08427629135679092557541630988833736e+21L, 4.13277307337650905633261831455054852e+22L, 1.97077591472219550165099701845426740e+23L, 9.86567192878194333593095251120628263e+23L, 5.19253827231351201624600034890440225e+24L, 2.87797813261600767124658244003026900e+25L, 1.68253103834271529795445964171621893e+26L, 1.03930300492804406447430223922613789e+27L, 6.79491179196076158722692150701779494e+27L, 4.71054472298412825215699352759734253e+28L, 3.46906218798171940969795067049186910e+29L, 2.71919469298029646397308102518695120e+30L, 2.27308597305997987201797057125811269e+31L, 2.03060816941963451979890589753656719e+32L, 1.94261454794602808054773809715560372e+33L, 1.99453642796409945684693816457728868e+34L, 2.20275757078165507105373401305070760e+35L, 2.62280693187648585243154230451397289e+36L, 3.37503469494102278406378324431267258e+37L, 4.70514262885548973786602859197635105e+38L, 7.12449819522227214155183099218285736e+39L, 1.17479491699687500984526531336188586e+41L, 2.11529840618171125644793575486664802e+42L, 4.17057423654484355854313046574264130e+43L, 9.02996402249218455912158145716517353e+44L, 2.15344195364580091650722699472214395e+46L, 5.67373854091156226750041136308702235e+47L, 1.65679493344512341491651081156308385e+49L, 5.37959274142567387437479000715431452e+50L, 1.94883090765131732574822421459033340e+52L, 7.90409930117048365355172083360408902e+53L, 3.60198039300535878643618051284879761e+55L, 1.85117602089555214211892984185617326e+57L, 1.07703344015399312416171835284404207e+59L, 7.12190311508435674119313111798381629e+60L, 5.37418888386167437844586074702336014e+62L, 4.64729650833728307491157079699691435e+64L, 4.62526710582644946748054017004844382e+66L, 5.32182077764493049141810605179622557e+68L, 7.11164697902038500579051989777466680e+70L, 1.10899515910236266304919828966923200e+73L, 2.02799605144484652148788128416483129e+75L, 4.37097775363901059074163853783972656e+77L, 1.11616878412036714566047843287003825e+80L, 3.39514715649439557125494531010505893e+82L, 1.23701982128373508607730444611285161e+85L, 5.42965793175551379743384549971838572e+87L, 2.88810817234216647705316265060164148e+90L, 1.87304637361264791993354548498385904e+93L, 1.49042693797246054384025095285834884e+96L, 1.46460222682423294988692623274219021e+99L, 1.78930667727185631768641895397399502e+102L, 2.73657637241785643461962451320315667e+105L, 5.27697387534476684848681508483720442e+108L, 1.29244503566283656061209233007634805e+112L, 4.05121786788653633019340698712667656e+115L, 1.63797303768105580759099639209855651e+119L, 8.61161786816897952522273022842847467e+122L, 5.93663320306070547376727413186770748e+126L, 5.41263435338015569490052234550484999e+130L, 6.58482575653621278094067557171475624e+134L, 1.07875660982114746476517100137547561e+139L, 2.40241992462133691284825971033750350e+143L, 7.34435324696612505273925787730222850e+147L, 3.11318307060514113999896732347290570e+152L, 1.84887752009365552806348833513339683e+157L, 1.55493753489569904210242587934359215e+162L, 1.87247311840034398087857290386878308e+167L, 3.26561398822344966034897508874934342e+172L, 8.34577318106206521818843721578177275e+177L, 3.16365209967032443701802479913075857e+183L, 1.80121863573431611955949232303314972e+189L, 1.56029746814154042157436181814468250e+195L, 2.08399815409632719398233238905315693e+201L, 4.35116508686695844690078637651674080e+207L, 1.44042594283375698824917457557971547e+214L, 7.67198578340176495038072549202750984e+220L, 6.67439909212164301008970795489093574e+227L, 9.63314810388083227111873828289832687e+234L, 2.34398517046722427869034729305744960e+242L, 9.77625325893464652861772945740140456e+249L, 7.10967843958998030041427991161778060e+257L, 9.17597441867286774623146964891772219e+265L, 2.14036206544827332284805812358315196e+274L, 9.19420612380519250234059644495148895e+282L, 7.41568524413971300723452081581189161e+291L, 1.14573384520235218510170025456065061e+301L, 3.46156619789802557007367290577360329e+310L, 2.08912045049233012342096277237544009e+320L, 2.57451388307699184873778115477509299e+330L, 6.62688893229567984520960366416649465e+340L, 3.64721218233974203748299012650543732e+351L, 4.39668921699312866048896359512324981e+362L, 1.19018091828589034728575506532863164e+374L, 7.42292147631996488193728378356566879e+385L, 1.09526098640141261390128648437627728e+398L, 3.92927734436557751277903528467662194e+410L, 3.52539958718458390412925405124992174e+423L, 8.14407380085561132671651242382158564e+436L, 4.99172864167925991508434677916290384e+450L, 8.37313822543668537823480408319630674e+464L, 3.96855771450749046728173847477197045e+479L, 5.49292894873452993989300213358944271e+494L, 2.29708122354182613470710687416853220e+510L, 3.00602816347641813806476628260504686e+526L, 1.27638345565474398023933739505382139e+543L, 1.82543785151714558753981850731109273e+560L, 9.13886193903951047182967076404934097e+577L, 1.66659166393902754183225415655293639e+596L, 1.15345182570915755941253059732455874e+615L, 3.16074696483848527092291852149349249e+634L, 3.58237065512849118228300807716046934e+654L, 1.75676845140981179589000705343726870e+675L, 3.90495239094757591264933748015965848e+696L, 4.12771024114613858596256966986824636e+718L, 2.18017473107123494465764845328386818e+741L, 6.05537171542030830962660182644895823e+764L, 9.32267445280349135813949292533678488e+788L, 8.40039310752404384099688064836067109e+813L, 4.68572386284109286448290494678984852e+839L, 1.71436621216433181556756205470621080e+866L, 4.36726894564208691170045669145775062e+893L, 8.23850494545149522733960578619372006e+921L, 1.22636833130359811940229369518492693e+951L, 1.53817446772510796133546119401337038e+981L, 1.73934010579080275675622752667739992e+1012L, 1.90139542390828308944919910797818652e+1044L, 2.15946649839694693874583032395369802e+1077L, 2.74458385322614086037916005350795661e+1111L, 4.21460437827726866824631065747020388e+1146L, 8.46327678591246014474308263680903354e+1182L, 2.41136671056709946500959023526115708e+1220L, 1.06046799402590770086828664893536628e+1259L, 7.85179239227989671195411257856609735e+1298L, 1.07053761782523090834371225181245647e+1340L, 2.94820375013014518743449818159813519e+1382L, 1.80414267837505733123417059421507186e+1426L, 2.70702818749460407693598763878577805e+1471L, 1.10237898995725987663697548890315962e+1518L, 1.35298567513553498909522877321159185e+1566L, 5.57611902405430874175886881037679558e+1615L, 8.62760618106732185887902244008862925e+1666L, 5.62274836328861658358184936480919820e+1719L, 1.73809780706704538846353037945373733e+1774L, 2.88052131688282318513745853191288493e+1830L, 2.90423838801873888374451692095277096e+1888L, 2.02951924440811805047282880080209219e+1948L, 1.12457154445459099155545223242406413e+2010L, 5.67678101812548596682420412619518564e+2073L, 3.01259124479996611916817067093268060e+2139L, 1.94839799990442184578594656275869899e+2207L, 1.78866123976568468449541367729975757e+2277L, 2.72776450570640988722564402903756955e+2349L, 8.12826724972980545013533499964424395e+2423L, },
      { 1.58403833597397609272337637131715304e-2474L, 4.01729907956437307269377706375953438e-2436L, 2.58580645701935131155084366864422406e-2398L, 4.31502895023559220093536948165386274e-2361L, 1.90627959218882483095663188951084330e-2324L, 2.27589955667507909301636117518932802e-2288L, 7.49364095876812027654477175001110624e-2253L, 6.94191154236392933887288062436725643e-2218L, 1.84522919671267474043685120983334524e-2183L, 1.43487104640347241520251346864979221e-2149L, 3.32692243646438795745471365264644590e-2116L, 2.34362088307498827836613435573448115e-2083L, 5.10938037262713942693655713913568198e-2051L, 3.51060735754549615330960873223686009e-2019L, 7.73934046950102664330281533900661264e-1988L, 5.57169465236073520583741860201801676e-1957L, 1.33280818206626290524674045358563098e-1926L, 1.07761463878385909143899244627194248e-1896L, 2.99487219016399829947920738111566372e-1867L, 2.90872283894068508271645083240599458e-1838L, 1.00349314519658897817146360011284604e-1809L, 1.24963603019269210233323055704536134e-1781L, 5.70651178926661193581089363715896182e-1754L, 9.70578293043209307574192077696195294e-1727L, 6.24328278846472954410732505740796953e-1700L, 1.54193058035632514209035337344023300e-1673L, 1.48399119945730121895709095048379728e-1647L, 5.64752158032889554103477750567193685e-1622L, 8.62167301557505164690437370018957398e-1597L, 5.35528299764763984143338000418037832e-1572L, 1.37241462764951847155586335894427418e-1547L, 1.47116444020541724843474277134735959e-1523L, 6.68618803692998895772448346916660911e-1500L, 1.30561453192449212967328116730742424e-1476L, 1.10983482751594629837009226116529299e-1453L, 4.16014518377810797736243906294304347e-1431L, 6.96431413050131379947014047903616964e-1409L, 5.27224320695995707920555229340409925e-1387L, 1.82727152085917578757399986780345619e-1365L, 2.93468712490571136931409720092296819e-1344L, 2.21029656493081556265677595112314098e-1323L, 7.89892460822856456312620943162424218e-1303L, 1.35498246922475851477828612647429926e-1282L, 1.12846854910106323887966343965144914e-1262L, 4.61424330893765402182473137993848704e-1243L, 9.36605095127672178405946589653781928e-1224L, 9.54054141854710811663726016692763275e-1205L, 4.92938100391460828476769674596652859e-1186L, 1.30552448174985025504800286655378847e-1167L, 1.79081395504489298673122868237542862e-1149L, 1.28534148947828407343141932678878641e-1131L, 4.87584961151572559819046616829450419e-1114L, 9.87277629967922144952427945151885497e-1097L, 1.07748800964228010206356572218732211e-1079L, 6.39928058051557015650179754550910965e-1063L, 2.08781638937389821580749246439535224e-1046L, 3.77684590339710704264469235045684970e-1030L, 3.82307555361237861173188661605677525e-1014L, 2.18500494285229212661260038960152445e-998L, 7.11371665672211288079530438419393765e-983L, 1.33086590459456967821090511427946680e-967L, 1.44309821421378874919204882407008472e-952L, 9.14647515187060050796469146648366553e-938L, 3.41683020290309632448426799765730462e-923L, 7.58515118221027874504569883083804726e-909L, 1.00874302140725085602226074835782694e-894L, 8.10069400524047548461447004237480370e-881L, 3.95900210017885619699714022194103898e-867L, 1.18662867376446832171113003401601030e-853L, 2.19786739893484213511853337557279548e-840L, 2.53446665772094387684997171428381901e-827L, 1.83299289701163579779687245566728015e-814L, 8.37462324951323786164578010773011160e-802L, 2.43440349985894963165438833576050630e-789L, 4.53405919015746926984345204895280285e-777L, 5.44811780296147933680587250736719556e-765L, 4.25227509910246244055269284525208521e-753L, 2.17028567518979939311612851867355063e-741L, 7.29110293097611082922276966606966112e-730L, 1.62280402409213923291155004693280799e-718L, 2.40828308813354260030814303685712546e-707L, 2.39798644647022706006083856285038692e-696L, 1.61202006993232554271262188260349029e-685L, 7.36078014307223356522361505691981658e-675L, 2.29673595120013740585571409952171287e-664L, 4.92601668045549547120303143586093867e-654L, 7.30469847326532782734592211727680659e-644L, 7.53208483045812497621524949105624661e-634L, 5.43101457218649885434844485451114112e-624L, 2.75364547737132169841544376328974616e-614L, 9.87114363624257039419731807833639141e-605L, 2.51532230246971379761137727536103954e-595L, 4.58020252011364324634687161502454437e-586L, 5.99105224430269595047395967421408836e-577L, 5.65817475729598864582006398104320790e-568L, 3.87790653215945353008449656152626419e-559L, 1.93831767408426944576894121111160849e-550L, 7.10042999124518145073529096418920675e-542L, 1.91544155185588660392537613901890409e-533L, 3.82329460613706944718834058784982843e-525L, 5.67308558344799642959861077485254353e-517L, 6.28651971015047054176370541889652534e-509L, 5.22608309691007325295015732741277330e-501L, 3.27380385796518526996963624688451015e-493L, 1.55218880567110686358214096387337744e-485L, 5.59406715134492552797785887998506288e-478L, 1.53903555561583129370612427064103671e-470L, 3.24582771097869247388161880434215690e-463L, 5.26923908371102477241383694074670819e-456L, 6.61116543964979170247326465662672578e-449L, 6.43651161747399650924315876123367361e-442L, 4.88172530890739153159755873311503383e-435L, 2.89552772291014307121494513337008901e-428L, 1.34824561713159645449381177450374044e-421L, 4.94682830703508395886779775564904786e-415L, 1.43550778574773554814450699757551163e-408L, 3.30662278156545301906789072182270217e-402L, 6.06763076341882342375981845143813941e-396L, 8.90102295078976467641802004915383639e-390L, 1.04750026993369008379812665328207076e-383L, 9.92305977697594285442750520712185343e-378L, 7.59233047709729007268137313555513461e-372L, 4.70739644774061079459196244467224618e-366L, 2.37289471679140773116577190541471766e-360L, 9.75578189470698031776059091702014024e-355L, 3.28173795958592945075990020191630023e-349L, 9.06055778471501819614486253220513415e-344L, 2.05942153439035557149195606898627504e-338L, 3.86530964740245225250791387282901296e-333L, 6.00843270324960642367292431619041056e-328L, 7.75791717613089963209182313649669073e-323L, 8.34421688870272754369905134347321346e-318L, 7.49743392110895125137984446953806518e-313L, 5.64335493091932417079632230402715339e-308L, 3.56822312364224796215751885944182157e-303L, 1.90033575264131712027297219010231489e-298L, 8.54728045366493643486236298588028833e-294L, 3.25523625801393911932111212124054975e-289L, 1.05248047241632062076489482718979111e-284L, 2.89617776712893256185087414745466351e-280L, 6.79988531580997322799099912957691248e-276L, 1.36556351036663137564575797911341140e-271L, 2.35130154856703817933308580165070895e-267L, 3.47957918602140446743207712996488528e-263L, 4.43595246742156897562714682208015764e-259L, 4.88306826050902558584812739817346803e-255L, 4.65192688318894071322199120456680912e-251L, 3.84397217698170104458131412518090261e-247L, 2.76116826189849718252009792299798034e-243L, 1.72788148193695597819524956747708857e-239L, 9.44000021446169518374327174980091189e-236L, 4.51212554821909958969480138145008725e-232L, 1.89078188020015147413152497967774898e-228L, 6.96047357819953940988624584998189879e-225L, 2.25551400000095886908107215813083451e-221L, 6.44644677476230409449870129589571800e-218L, 1.62820413693305623578289570883910133e-214L, 3.64118898140332008867604954881435445e-211L, 7.22342104300488240353501601432620632e-208L, 1.27354621995150979542739771832851778e-204L, 1.99918407175965223164864543465886820e-201L, 2.79923301669228713128194864334595626e-198L, 3.50221902155243103789141550909762023e-195L, 3.92214264392110857739154085058742459e-192L, 3.93845056335503285763060194710496279e-189L, 3.55208932101704820479562248595366349e-186L, 2.88218576501349132033303488025583146e-183L, 2.10742246561673617325867420889217990e-180L, 1.39082883056909297835969428096996258e-177L, 8.29809184041509702782403322046680024e-175L, 4.48275079028324875908925844854947607e-172L, 2.19604664947993546897548946859268970e-169L, 9.77074245531826439290686011595488065e-167L, 3.95412900034836577575737163397063382e-164L, 1.45763752143522946134851229157315125e-161L, 4.90175908594770144816169627656347413e-159L, 1.50583242362081439890282730822964005e-156L, 4.23187210926299952258468916772463829e-154L, 1.08947970178510600143883902241342972e-151L, 2.57292238715065164920970933548863818e-149L, 5.58131105433415694107142145348480095e-147L, 1.11357590012697004029170524556205260e-144L, 2.04616505133228608383036998688279538e-142L, 3.46699488500477063569923373366652510e-140L, 5.42379540407350192206120227447952764e-138L, 7.84383327240284700975116490839187822e-136L, 1.04992295793319441478934809931921214e-133L, 1.30230107195741860304672567525213181e-131L, 1.49865973782839300796909511415792928e-129L, 1.60190662241428628230308870106117967e-127L, 1.59224861840198356125270425396669689e-125L, 1.47337534591643627391632824044256382e-123L, 1.27065155139400959289502878067800521e-121L, 1.02240826352576620907328663762929630e-119L, 7.68376260232956278119400404906698852e-118L, 5.39926812723337318582425048609813182e-116L, 3.55107427485349467569160645183534302e-114L, 2.18823540951912101048500342930263842e-112L, 1.26466751543081693428147968135946762e-110L, 6.86180756673724371219029573988488620e-109L, 3.49869168682520996337772693940273721e-107L, 1.67801680739837515692197356611469711e-105L, 7.57743943144193148982405002124429739e-104L, 3.22470377015938680899710533122019572e-102L, 1.29448709067770596337952427228314258e-100L, 4.90613325096345413914934529363944533e-99L, 1.75712131798815332550332783841135938e-97L, 5.95204249145432038279376449394263726e-96L, 1.90856665328641726356485393831448064e-94L, 5.79822445923642921233565089484060205e-93L, 1.67029323997833472695510332615399040e-91L, 4.56623667339808303761814383193281614e-90L, 1.18561734279154794522720467403607684e-88L, 2.92616002780129692916694825399426567e-87L, 6.87006113412670713749912656047162470e-86L, 1.53556578350037994526862611062995548e-84L, 3.27003673677840125658578850353789865e-83L, 6.63955800720658036202444029681088661e-82L, 1.28631975096739859347518450406626244e-80L, 2.37956658113902295790875474474928153e-79L, 4.20626823139888342521923162463424944e-78L, 7.10971923783337943250783648529172174e-77L, 1.14991510411537277718541841100610582e-75L, 1.78087620125559422021122815240864113e-74L, 2.64270379617932988289762938432353092e-73L, 3.76008537594171932684988546369276565e-72L, 5.13292095112425199264463076393673633e-71L, 6.72710027460142769581137467118289653e-70L, 8.46958562134769749810317782657268756e-69L, 1.02503238267223284787859415412841523e-67L, 1.19321912755786334798650299344282298e-66L, 1.33681693038130658171632078204446788e-65L, 1.44228347967979838541089585366023007e-64L, 1.49937455500479399063651651865716553e-63L, 1.50279720313350143765006477517598013e-62L, 1.45300596931848530329429215240805653e-61L, 1.35598044837786253974814702268132430e-60L, 1.22207241221255212718220407758970979e-59L, 1.06422318027052015901612579353436284e-58L, 8.95966739607563684498250506056071091e-58L, 7.29628880807929410507791279851435148e-57L, 5.75025529619018115782221236649660618e-56L, 4.38801166482901351758297994035262224e-55L, 3.24385245129183239778054564241318804e-54L, 2.32423935766553880551312730791561599e-53L, 1.61486977620302644603098163288002715e-52L, 1.08852460554527484166896264813585652e-51L, 7.12175557419282904469498440847683331e-51L, 4.52464766254906707378983948806266105e-50L, 2.79273071581879303547665803856393705e-49L, 1.67538487960386422664995380528599268e-48L, 9.77311432877767609128920687351148833e-48L, 5.54591076684762708155258243373876422e-47L, 3.06280970562787364526087771025696151e-46L, 1.64686211803826623376300968100335938e-45L, 8.62510851388715584666563483664725226e-45L, 4.40168766386889070104410967553191993e-44L, 2.18975577884764674623829698191429056e-43L, 1.06234533644926588892221744026505121e-42L, 5.02803666348568404922856964957017674e-42L, 2.32252463571724922291588384046952088e-41L, 1.04740659389834130621977678673036650e-40L, 4.61343838844969816849890851044847674e-40L, 1.98539744511816200515536891243051363e-39L, 8.35102736745462834322783293199882340e-39L, 3.43444090348454338873945310910863585e-38L, 1.38148913187719664577281415801326678e-37L, 5.43705120131022522415564030011104834e-37L, 2.09435754808064771674030165617400032e-36L, 7.89867661859200690211003183623252601e-36L, 2.91753687094747127218152557081492503e-35L, 1.05578888602271659701510267800456427e-34L, 3.74433381216033081152946793044225824e-34L, 1.30180118525195729018565371931285108e-33L, 4.43834621689338776826444391843192077e-33L, 1.48434826895181654234585115302567433e-32L, 4.87100112984983697071424260901079389e-32L, 1.56890300074251394167498791840311009e-31L, 4.96129531591793523529868958902224169e-31L, 1.54077391002799082070667599670892752e-30L, 4.70055802217201491012415527477898338e-30L, 1.40911523071894959581827729743208189e-29L, 4.15191310395569203370171887252329295e-29L, 1.20273761371542774812275721780834541e-28L, 3.42632737493449673556641936029297478e-28L, 9.60140535939702601155477307042148556e-28L, 2.64727864203377330130778700812469317e-27L, 7.18344222056514710304108009603342664e-27L, 1.91885054598149404233254994607045127e-26L, 5.04697477945599249407828848720385551e-26L, 1.30739479992591170007499101639277446e-25L, 3.33634219823695708156517864262133521e-25L, 8.38925958113626219399750986165640649e-25L, 2.07905181351354860827899559955891196e-24L, 5.07917896724376528007260946058158296e-24L, 1.22350179435783727784530143589061714e-23L, 2.90665491105754953014975765085018324e-23L, 6.81166860609501546977145188171063268e-23L, 1.57498593823802530267320052674032470e-22L, 3.59379678896934832551131875550134077e-22L, 8.09418541120521256436201968800902213e-22L, 1.79979618323748172103553749785052354e-21L, 3.95175890164101728455789689950582204e-21L, 8.56958006805086577463240518925283894e-21L, 1.83575348651729869648335347322858802e-20L, 3.88541433996602231722399876322981769e-20L, 8.12661397289502178971121801417655470e-20L, 1.68000718288950314093575786904308274e-19L, 3.43336935156396282804714345750464903e-19L, 6.93769555039942749928599832442683686e-19L, 1.38634563100898175510110455954156138e-18L, 2.74008749775923088060873199170519103e-18L, 5.35757028868338662612872498580241047e-18L, 1.03646493302280378406673184005440162e-17L, 1.98424944201008499240310250250687800e-17L, 3.75978800606000340899396301904932318e-17L, 7.05221126182168479511704811938887205e-17L, 1.30963564152954622082996853685780292e-16L, 2.40827549610918052797562865726934570e-16L, 4.38589880961171155211843738902302126e-16L, 7.91175868684912128549216604428547791e-16L, 1.41388359787718387270571947224924145e-15L, 2.50347753664468021020115723250325879e-15L, 4.39263786655070582741822344903139686e-15L, 7.63871030696057461236914444427999640e-15L, 1.31670336037747604093705380830550144e-14L, 2.25003102727544891850217684727399098e-14L, 3.81223973341221495282227985716493129e-14L, 6.40502166019136347918960755019998163e-14L, 1.06725053827031948353711195948826425e-13L, 1.76389749378472101040375240485184047e-13L, 2.89198756533454775599767608849931393e-13L, 4.70424252036995808470327107927435060e-13L, 7.59287827351269198973041796516049688e-13L, 1.21618333837252517178450702971971531e-12L, 1.93338859343662487903047794738683986e-12L, 3.05082685244229075126272877890267121e-12L, 4.77908002001763665704541557372044375e-12L, 7.43273471338542509759247990082208445e-12L, 1.14783388812587366578078714999164060e-11L, 1.76028616037242275427822154657272751e-11L, 2.68107110162395316788540367212924769e-11L, 4.05602375429596543728262134893310443e-11L, 6.09544349224153722159236648112669209e-11L, 9.10055012961606421079182569690846497e-11L, 1.34999345213696765153127846451643325e-10L, 1.98994391239515605127817880646635584e-10L, 2.91499607361905978799020781323598840e-10L, 4.24390078141221962088655373293736779e-10L, 6.14135316267139108184909214050184662e-10L, 8.83436579589479851125156253066831718e-10L, 1.26339559402593316965102282486280599e-09L, 1.79636925005171604673781604814805685e-09L, 2.53970414332648086166213535713251755e-09L, 3.57059249828789049877331667413514712e-09L, 4.99234840315053910682137402914672922e-09L, 6.94247187048993148295924434414634261e-09L, 9.60294960016456137105057353194608309e-09L, 1.32133371276166677672139906194961059e-08L, 1.80872790163534638977965561238944112e-08L, 2.46332536476779151638500763225429049e-08L, 3.33804787013687049574791297244907192e-08L, 4.50110842610850506871431532628603516e-08L, 6.03998541333325959410060827223996830e-08L, 8.06630537452609783391065479881400036e-08L, 1.07218105901889261435737857618298194e-07L, 1.41856144379535399139351236825663488e-07L, 1.86829769983638330491745888879438028e-07L, 2.44958653917297200861483891225150109e-07L, 3.19755978044276083153634882857883309e-07L, 4.15579069086754433405340505039048729e-07L, 5.37807971332554467827633744547482376e-07L, 6.93056106477668619385798002644561466e-07L, 8.89417585250212245375810218669817408e-07L, 1.13675615786872600638029106996132895e-06L, 1.44704121253473089806789252868181132e-06L, 1.83473664533283350419063698449974520e-06L, 2.31724882235425364363184937796552995e-06L, 2.91544022582530391094107183656707013e-06L, 3.65421570986355186992807571629053921e-06L, 4.56318857677376015133362727471956847e-06L, 5.67743390948223287752620793591929613e-06L, 7.03833674730757178417069911600525632e-06L, 8.69454275808306722839276847105496531e-06L, 1.07030190270275985848630609511672890e-05L, 1.31302324393740374972736712543053054e-05L, 1.60534528678907389671161995891497537e-05L, 1.95621879772878044903047444244351033e-05L, 2.37597559155521886172243888058434365e-05L, 2.87650014695436120790473222143336354e-05L, 3.47141604126307620878569429267584729e-05L, 4.17628757618591523872557240511914786e-05L, 5.00883684896740377264076716570754303e-05L, 5.98917639018173037280503906445049523e-05L, 7.14005734028021322676450426011314791e-05L, 8.48713297304976003620620060812799411e-05L, 1.00592371962099993408798498455907607e-04L, 1.18886774688549697317681023800631139e-04L, 1.40115413739806927896468824589692520e-04L, 1.64680158738873124934815777767252928e-04L, 1.93027180590427177790609959505517465e-04L, 2.25650359795433055575151925625868196e-04L, 2.63094779253370712822446127059315509e-04L, 3.05960282998094618038061759727754321e-04L, 3.54905080142515530273890819014315333e-04L, 4.10649371213184272695586730470455583e-04L, 4.73978972070856543596623702324547537e-04L, 5.45748908769705106925744948564871657e-04L, 6.26886955037988466777589811218012335e-04L, 7.18397082597597367255394668144143647e-04L, 8.21362793308292890121036733909194916e-04L, 9.36950301151796636389593916360605611e-04L, 1.06641153138572518429387087881256567e-03L, 1.21108690381909541715522898126065244e-03L, 1.37240786710764633931488483556773354e-03L, 1.55189915125250562428574475103346186e-03L, 1.75118070611954731846143391024982058e-03L, 1.97196929478447094442182021751059029e-03L, 2.21607971185090897097885205424488147e-03L, 2.48542559858177963603431553204765315e-03L, 2.78201982871899325746660736300345366e-03L, 3.10797444123022017600460098808123935e-03L, 3.46550009889599377554207066240202628e-03L, 3.85690505461395961880575720620871256e-03L, 4.28459361052363939313848682793859897e-03L, 4.75106405851509722505038175681427297e-03L, 5.25890609434561842084606413815284753e-03L, 5.81079770141443579933464319447926318e-03L, 6.40950150419891594280210549702450453e-03L, 7.05786059539697018633739598386855559e-03L, 7.75879384490912344645900750924930976e-03L, 8.51529070288836937235862937785994511e-03L, 9.33040551314529952303815524992440065e-03L, 1.02072513571791257196870650265821749e-02L, 1.11489934529722276019901656254303012e-02L, 1.21588421363983657377673415921445027e-02L, 1.32400454566162946339651246079183099e-02L, 1.43958814201171884975897081297216705e-02L, 1.56296499211348507284307663166667625e-02L, 1.69446643988840458394481431193257832e-02L, 1.83442432645398203334614155955590374e-02L, 1.98317011429883687012533532738687781e-02L, 2.14103399761506788853091011534572156e-02L, 2.30834400360906269037251797223442256e-02L, 2.48542508971601536767883257823339293e-02L, 2.67259824171004266947925047187844041e-02L, 2.87017957773082030971446046374914401e-02L, 3.07847946323935695276863138190467659e-02L, 3.29780164187051571955970986393933287e-02L, 3.52844238706916706398930809090405599e-02L, 3.77068967928172889046621767464058045e-02L, 4.02482241332694163505581270228186661e-02L, 4.29110964039093677010619866631701642e-02L, 4.56980984888413264034413616112869595e-02L, 4.86117028816359215545912812117182378e-02L, 5.16542633886674445449455067720287452e-02L, 5.48280093332349644585895096278308003e-02L, 5.81350402921654268021994051667911914e-02L, 6.15773213934700546744487567673172633e-02L, 6.51566792003733016470332458328988615e-02L, 6.88747982036856640340840950741077307e-02L, 7.27332179410771208959941663503595393e-02L, 7.67333307583556615050326531083587939e-02L, 8.08763802243933982425303227000944183e-02L, 8.51634602078983074689360505021182257e-02L, 8.95955146208286742294968393977769930e-02L, 9.41733378299144489750576704343915921e-02L, 9.88975757345080247656478735188449399e-02L, 1.03768727505857796722978821461546404e-01L, 1.08787147979900856678514246787241454e-01L, 1.13953050692823999603416543952272158e-01L, 1.19266511545960614110252460361458099e-01L, 1.24727473084088741556780346524879069e-01L, 1.30335749368884349581460099687500407e-01L, 1.36091031427173401966757170543046798e-01L, 1.41992893251724361991875167825655695e-01L, 1.48040798330635148304426193810368709e-01L, 1.54234106679899202351157467338543629e-01L, 1.60572082352486356465237071613156643e-01L, 1.67053901396246033471214692284653203e-01L, 1.73678660232131774208044201415508905e-01L, 1.80445384423654491197811595430697492e-01L, 1.87353037808093115326776430258837481e-01L, 1.94400531959820109714578533182084599e-01L, 2.01586735956129211498107459964096125e-01L, 2.08910486416176267158329341893267875e-01L, 2.16370597784052818695266702661187610e-01L, 2.23965872827597104489240738884195628e-01L, 2.31695113325298676464455224512869174e-01L, 2.39557130914560734720290079897448420e-01L, 2.47550758075638008786996132954185789e-01L, 2.55674859226756791227887436442350007e-01L, 2.63928341907236639931045094151784107e-01L, 2.72310168026859366788753574972476498e-01L, 2.80819365161259349748148678323641718e-01L, 2.89455037874729232565423711864655899e-01L, 2.98216379053536250342702367445487260e-01L, 3.07102681234616603556790972526892770e-01L, 3.16113347916348760019249138447488231e-01L, 3.25247904839992014226666657864488014e-01L, 3.34506011232305314043635112310304911e-01L, 3.43887471001825077739424147442217492e-01L, 3.53392243883271879333889785572599272e-01L, 3.63020456526567529072712529376719554e-01L, 3.72772413528969943076239230990758849e-01L, 3.82648608410867702369280704118517439e-01L, 3.92649734537814481801171047048544340e-01L, 4.02776695993421447225756159687866121e-01L, 4.13030618409759875566958499890031988e-01L, 4.23412859763953990585030832527434639e-01L, 4.33925021151663415382683548295156320e-01L, 4.44568957550164552554831720660662401e-01L, 4.55346788585740186021655771168112611e-01L, 4.66260909322076961243786064631384911e-01L, 4.77314001088352176735947056292312804e-01L, 4.88509042367666263602229097070513234e-01L, 4.99849319768447906973686731056178972e-01L, 5.11338439103428142948352352152044385e-01L, 5.22980336602751811729499672944460101e-01L, 5.34779290289774015573364455286468490e-01L, 5.46739931550080955287380242064660003e-01L, 5.58867256926284616744877484869191055e-01L, 5.71166640173175841717312971842336611e-01L, 5.83643844609887615578228626088462787e-01L, 5.96305035807827889758567710635182685e-01L, 6.09156794655297569062821736349343099e-01L, 6.22206130841923771587050175584017347e-01L, 6.35460496808321163697317796410781296e-01L, 6.48927802208755868138883757264025330e-01L, 6.62616428937038679525110869491604161e-01L, 6.76535246768429422677267756666823087e-01L, 6.90693629673005399383708050292572666e-01L, 7.05101472858747991895757442966727674e-01L, 7.19769210605547537680146085503476253e-01L, 7.34707834954433431482100014962668745e-01L, 7.49928915319620942104629439591963551e-01L, 7.65444619094446439133009723643838630e-01L, 7.81267733325957766141060146767846541e-01L, 7.97411687536856786535849810135961905e-01L, 8.13890577777678436208431256995493454e-01L, 8.30719191996558177068293937598919209e-01L, 8.47913036818712374055336621201445842e-01L, 8.65488365832860347537655780480228440e-01L, 8.83462209487281076620051655836621909e-01L, 9.01852406704052162112602532863159232e-01L, 9.20677638326296314246112109641174382e-01L, 9.39957462519996315051517977272639457e-01L, 9.59712352259170728365681565566195274e-01L, 9.79963735030970038685207606899372777e-01L, 1.00073403490559993309000835270230984e+00L, 1.02204671712495200986974494164817378e+00L, 1.04392633537347289336664619627296763e+00L, 1.06639858190518516071876464455235265e+00L, 1.08949034071194662751500934725633506e+00L, 1.11322974393006216402411430386892104e+00L, 1.13764623169531331372630723630321340e+00L, 1.16277061567042026013306680494416940e+00L, 1.18863514648397907083703649648730714e+00L, 1.21527358533611238960293477132347556e+00L, 1.24272128004352904999786719958821158e+00L, 1.27101524581551079885511358011134290e+00L, 1.30019425107264471110982901454765895e+00L, 1.33029890864201997125110745686528366e+00L, 1.36137177268624019179654148863686280e+00L, 1.39345744174911173037014751038646328e+00L, 1.42660266832841175766232711333173310e+00L, 1.46085647541588835802048454309412779e+00L, 1.49627028047678533760134362560179034e+00L, 1.53289802737592016874439144914426210e+00L, 1.57079632679489661923132169163975144e+00L, 1.61002460572564642000849608863176067e+00L, 1.65064526666943143532240244567555108e+00L, 1.69272385721798833166204038375277879e+00L, 1.73632925074497773069781144176863796e+00L, 1.78153383899165490255124499430464714e+00L, 1.82841373739108738124596203605065226e+00L, 1.87704900404072044775471792428672248e+00L, 1.92752387330408763463712892863710673e+00L, 1.97992700509947708650365975965257989e+00L, 2.03435175101694043322616088522632134e+00L, 2.09089643849576621383396167595935906e+00L, 2.14966467439309042108153883973067514e+00L, 2.21076566938140221236799358744093376e+00L, 2.27431458472911392711904220601704565e+00L, 2.34043290314497023966214961240421096e+00L, 2.40924882550482707578395780381837657e+00L, 2.48089769542928804270777308901824137e+00L, 2.55552245384400165592583716337259322e+00L, 2.63327412583237088656291310907475315e+00L, 2.71431234228441160768610076838330351e+00L, 2.79880589905706635256578441640292763e+00L, 2.88693335659214188612865513175227518e+00L, 2.97888368319007786718164917079966461e+00L, 3.07485694541305021056656397487876403e+00L, 3.17506504939176568339259231884822943e+00L, 3.27973253713925527986226523992226922e+00L, 3.38909744233483410200034132396842728e+00L, 3.50341221043527586541966899038246748e+00L, 3.62294468840159570502872359754361586e+00L, 3.74797918980246258518386393043115941e+00L, 3.87881764157340380510220562013091253e+00L, 4.01578081927931267029397913978942025e+00L, 4.15920967835153616848634566919310171e+00L, 4.30946678945578836835486232808829311e+00L, 4.46693788689973689723986043138805888e+00L, 4.63203353981649359057533901265089751e+00L, 4.80519095677036072684883899803335247e+00L, 4.98687593543289697160079921623416533e+00L, 5.17758497008053768765855285400689531e+00L, 5.37784753088062976076034128768012631e+00L, 5.58822853027308803485444007033170411e+00L, 5.80933099323364005938466841106588229e+00L, 6.04179894983708948784718604678234941e+00L, 6.28632057034228591922334730387683243e+00L, 6.54363156501365266125399451934329088e+00L, 6.81451887309858260791767579514605499e+00L, 7.09982466781971868191373433617700078e+00L, 7.40045070694293100845796745008949286e+00L, 7.71736306147578881407568826714170705e+00L, 8.05159725837127958401987584151171259e+00L, 8.40426387679538395058384455428283593e+00L, 8.77655464160750010892022612040926275e+00L, 9.16974906224756520705691589269677116e+00L, 9.58522167027699388874228356777329920e+00L, 1.00244499144430070386099274885141050e+01L, 1.04890227783960385614814296197680936e+01L, 1.09806501931649260624015170997236345e+01L, 1.15011733242716998455210622743445835e+01L, 1.20525758220454727952816341992419823e+01L, 1.26369961333845432419384699684768677e+01L, 1.32567409840433238017603739136360460e+01L, 1.39143001526287336814820956823084828e+01L, 1.46123626710408671221171338391349055e+01L, 1.53538346012683753120053442509520620e+01L, 1.61418585554581184591279187098492208e+01L, 1.69798351452575852400510409025135738e+01L, 1.78714465678460133882677098961576727e+01L, 1.88206825601317848442892749807248071e+01L, 1.98318689796476498521560439791262319e+01L, 2.09096993011184544989073368255312499e+01L, 2.20592693519609552724995680207860407e+01L, 2.32861156486188168325833440616926330e+01L, 2.45962577392286013760094208279475314e+01L, 2.59962450073299827601794849672397508e+01L, 2.74932084469488923833030086627802945e+01L, 2.90949179822819598429999086944366938e+01L, 3.08098459764107671506384842333391345e+01L, 3.26472376541418040006352140116179211e+01L, 3.46171892555432186128355184008936311e+01L, 3.67307348405744306653537849794079228e+01L, 3.89999427831545697992227880433189487e+01L, 4.14380231271361842696931290595974334e+01L, 4.40594471293014233018696008435982161e+01L, 4.68800804884035743932352319123055757e+01L, 4.99173319575866229822636583993237381e+01L, 5.31903192638729836914160122380617184e+01L, 5.67200545170346581087003708738282234e+01L, 6.05296515859483113978235132985845589e+01L, 6.46445582591583649124780712968047055e+01L, 6.90928163944313177415529394071665670e+01L, 7.39053537072521168719419511314839173e+01L, 7.91163113594234348927381617490387462e+01L, 8.47634120965947230809397578710282148e+01L, 9.08883743598215272174601093484125344e+01L, 9.75373785753325382322893051705294986e+01L, 1.04761592725164736089639534479074532e+02L, 1.12617765338655419662606911403899284e+02L, 1.21168895243741881709481972315918511e+02L, 1.30484988804359382802013970413388238e+02L, 1.40643916977370870089294791135734370e+02L, 1.51732386386376598946153144686407135e+02L, 1.63847040773982427929709267988083986e+02L, 1.77095711710003361983546853982729852e+02L, 1.91598840361277588527216638598210664e+02L, 2.07491095540949726522855395012839290e+02L, 2.24923217236106119394568010864647153e+02L, 2.44064119463086993590008673999888577e+02L, 2.65103291739026696423162894476095053e+02L, 2.88253544828036421249450783362468048e+02L, 3.13754153889742451327114437895012151e+02L, 3.41874460927761232190324931320648150e+02L, 3.72918008746121432097478949634992228e+02L, 4.07227290759381879033987273956073881e+02L, 4.45189215310338987815842929936219157e+02L, 4.87241400038863092710563840938907565e+02L, 5.33879431809824993154116113611340152e+02L, 5.85665251340011311718653543776253641e+02L, 6.43236849676682281579530292235801252e+02L, 7.07319496933657861116096423723089184e+02L, 7.78738763222127723625961426334336958e+02L, 8.58435638777040682657342340030579928e+02L, 9.47484116394459954294416631122613257e+02L, 1.04711166630196929712820316594777632e+03L, 1.15872311371927743485840786701904690e+03L, 1.28392852534970775486315470390457819e+03L, 1.42457582618936343671625036189352836e+03L, 1.58278900639377570598543689683858411e+03L, 1.76101294444545923486104598108258215e+03L, 1.96206607357312178755673700966218336e+03L, 2.18920236070835422209657734442941355e+03L, 2.44618436034955965177471357752869960e+03L, 2.73736946076118709316717034422700237e+03L, 3.06781187080876763823587584403262657e+03L, 3.44338341950996275353420062305895100e+03L, 3.87091687821820770548262640961099474e+03L, 4.35837629346446550811483742251576785e+03L, 4.91505976942026055884439891778599977e+03L, 5.55184130321696740425893724460092414e+03L, 6.28145970445342612877221303833900020e+03L, 7.11886438520566571043900024525721790e+03L, 8.08162996762779959609851789890726579e+03L, 9.19045432173859727983153055486718404e+03L, 1.04697579405183570171054706781681133e+04L, 1.19484066394624731955974446656465043e+04L, 1.36605846306210479280006137726608390e+04L, 1.56468513163780927299848865296760921e+04L, 1.79554229917996753910020302361418910e+04L, 2.06437304374408251386790990436920728e+04L, 2.37803156373267080705808819875316658e+04L, 2.74471462199565095258711815725505419e+04L, 3.17424455248072273922403695999752070e+04L, 3.67841605073133622632674187774059283e+04L, 4.27142203777350805091608476572382608e+04L, 4.97037776810032398127125994364568099e+04L, 5.79596727313857616360132420453444344e+04L, 6.77324248460879259253530791773167141e+04L, 7.93261334694994276050922426242412735e+04L, 9.31107739715691545023768199956003639e+04L, 1.09537503053637222429417759669716212e+05L, 1.29157755673566952599086463851912376e+05L, 1.52647130160874158565239969760504587e+05L, 1.80835335096964828910095096138515211e+05L, 2.14743829477016418053483721316277484e+05L, 2.55633251557399994754680946686048161e+05L, 3.05063334556209750218324509503461449e+05L, 3.64968792666585395419705030588899790e+05L, 4.37755686685748537981529230100717254e+05L, 5.26424122294320873553623779329790816e+05L, 6.34724899010831940960718793968560727e+05L, 7.67360052654242646579956157559309383e+05L, 9.30240305033750278572545038832724621e+05L, 1.13081650266645184495081585123306969e+06L, 1.37850753115552374245055838415718322e+06L, 1.68525439396416227539607702063253047e+06L, 2.06623977016863939032179814240740936e+06L, 2.54082527022935491763036651163297626e+06L, 3.13377596203641663047341272710642846e+06L, 3.87686514827580239293595658539782774e+06L, 4.81098405401834942973721071475737211e+06L, 5.98892408953467866424875864774796159e+06L, 7.47905792960806092396497010475870265e+06L, 9.37022569869340886743682668142431718e+06L, 1.17782423097751066120254539266241595e+07L, 1.48545930143258061872721569284540776e+07L, 1.87980927038339810411221199415802492e+07L, 2.38705733443634639976678131975655720e+07L, 3.04180655225860320213756009749555241e+07L, 3.88995004684326215069338023547372206e+07L, 4.99257437458669601662091946924574592e+07L, 6.43128750449561321017861557895848463e+07L, 8.31551851992585813643149605060553440e+07L, 1.07925566470411796107305712037505869e+08L, 1.40614107339003511498586442587195365e+08L, 1.83920178567730560673713663331917338e+08L, 2.41519711690497536458569405379734816e+08L, 3.18438601538111228132801183275722771e+08L, 4.21576501892968673592102667497462836e+08L, 5.60444635691511454953230090792878868e+08L, 7.48209439804691157215633476755723997e+08L, 1.00317512966824615144294234414110893e+09L, 1.35089891899748286951692285814894218e+09L, 1.82722216505349159044967968934424570e+09L, 2.48263348083176093300905605774694855e+09L, 3.38857763723491971892999484076543171e+09L, 4.64662006529910564426883234085294096e+09L, 6.40182180156629712154379147661940269e+09L, 8.86235203805325147258573091932997831e+09L, 1.23283860285919681076337828978766266e+10L, 1.72348929748018002343932808836953853e+10L, 2.42153052846944737603768649152219024e+10L, 3.41967381320806302529104945578396652e+10L, 4.85431236462260654032773006433818623e+10L, 6.92714904376034267631798991329824532e+10L, 9.93804949018620361644641133903513098e+10L, 1.43352142475985414523266188918663249e+11L, 2.07922173448308822669953899424734201e+11L, 3.03269524182010815814751345567548609e+11L, 4.44863150372771043146060595057040363e+11L, 6.56345864647790105139454816812312939e+11L, 9.74063569639891097956397109813490533e+11L, 1.45422052005965615789962842879352220e+12L, 2.18425068889862732017431485970855335e+12L, 3.30099910475756075681442440083420253e+12L, 5.01997048502274901150163921982195024e+12L, 7.68267629901760783371009641842522603e+12L, 1.18337659600398387182686793240868246e+13L, 1.83474885355703531496992152648661997e+13L, 2.86363931245836358559523100954017414e+13L, 4.49980389271503995761664922194406313e+13L, 7.11948687698915449751693845544920647e+13L, 1.13430701798012234619458399048221594e+14L, 1.82006578236361839541514524597276231e+14L, 2.94148450061539403705926461420998342e+14L, 4.78870730589093038221910605812495108e+14L, 7.85402503692862355090784993078672085e+14L, 1.29789430461986025113447366376693514e+15L, 2.16127995478242564041665605678872071e+15L, 3.62710214703500383377916061381597850e+15L, 6.13534293344095037788551479051826967e+15L, 1.04617000636224450596060724147477482e+16L, 1.79847735783966568554234927670508824e+16L, 3.11747341233233147536151972501734499e+16L, 5.44944507304918422218082921598846642e+16L, 9.60751550501797821243248841373822216e+16L, 1.70858922445267785163471612163490444e+17L, 3.06542975111022866531582625971692938e+17L, 5.54922743745114951066953816028814728e+17L, 1.01373023277804631428305560194491190e+18L, 1.86905989587640582429259395332686294e+18L, 3.47854955238157842413369108802590798e+18L, 6.53599224597546376326701108631484361e+18L, 1.24001927226106630812645719537154550e+19L, 2.37582886691093662903464587039857427e+19L, 4.59768243360443262528520790447211677e+19L, 8.98810681683712842794124622629411311e+19L, 1.77530237939363226330723906827319437e+20L, 3.54341330439097348610370779591666976e+20L, 7.14806139767552532748817894789979543e+20L, 1.45762051057718630539520752847120279e+21L, 3.00513712487982979744944726506420716e+21L, 6.26502486163325069730377264873611059e+21L, 1.32097994109028381590144585332181800e+22L, 2.81748753590214622077676014001554368e+22L, 6.07993304142980523101356268020373032e+22L, 1.32765885364721208288263553578233241e+23L, 2.93431175918364131820516218129002628e+23L, 6.56508721680713002578214496043109168e+23L, 1.48721227343793765013154051597710086e+24L, 3.41184019607678812819334291291332010e+24L, 7.92818992879701876208212761012395842e+24L, 1.86645187702970485690987148720542546e+25L, 4.45252185988673954885155645168863932e+25L, 1.07654543517497766241465472692681594e+26L, 2.63868568119069758552386276646714408e+26L, 6.55790847024418649820013325410592483e+26L, 1.65295224373558572055534033842757998e+27L, 4.22638339591491619855432610245411904e+27L, 1.09645039426808014848951068590561771e+28L, 2.88682208299928608019375692651426610e+28L, 7.71548038934401592468282155249032740e+28L, 2.09372878930996484632394470400138705e+29L, 5.77027578944765503685743525710220950e+29L, 1.61546384539178114004658391605197219e+30L, 4.59547005579560869055708560274363012e+30L, 1.32862939268652325541488091880021431e+31L, 3.90507968153078421921910800039715306e+31L, 1.16713402427199725201363486127342874e+32L, 3.54805853865427740256789009746696294e+32L, 1.09737805935804615955239963430188272e+33L, 3.45410297806444559459316674852385198e+33L, 1.10674539370165232283645935543266144e+34L, 3.61089955913906999391756229494586894e+34L, 1.19994699928367056689901878605794596e+35L, 4.06268701419087879160332452792716198e+35L, 1.40183522389322451388311239745912475e+36L, 4.93108552733316217324966630695184171e+36L, 1.76881239328491949973951925349674313e+37L, 6.47214829394519996070845864216337266e+37L, 2.41645372173921192231147151489004683e+38L, 9.20894472039812386191245000342688835e+38L, 3.58329702862212667598998865894252598e+39L, 1.42409748259669944005518449916929857e+40L, 5.78262783342641152433806724533524027e+40L, 2.39986220408436318313990705343752378e+41L, 1.01829157204230545960231903208713235e+42L, 4.41910541482203453093956062976933998e+42L, 1.96212611768049931065065590432043620e+43L, 8.91674242406125370712417447773466783e+43L, 4.14888247829475772044491476179936724e+44L, 1.97725652955827692966670149947470575e+45L, 9.65530023387540108044378286001152781e+45L, 4.83287889833559892166961276593618835e+46L, 2.48057587822309805808523773881299449e+47L, 1.30610280975765470571043683527277027e+48L, 7.05756571728956923233902004974509542e+48L, 3.91527652222961861840270351783060925e+49L, 2.23089898094339331762072829615745302e+50L, 1.30614133449630930559423540813368509e+51L, 7.86102128665639262739828585146248224e+51L, 4.86558375853845110678814792824481696e+52L, 3.09848742591570467373735298285106117e+53L, 2.03103761486256390140847277681645261e+54L, 1.37099964760826020030246335833308722e+55L, 9.53473627432500152782873935909970826e+55L, 6.83495992316641540715754654005899263e+56L, 5.05273354632478901960113633106144814e+57L, 3.85381099728215997914445000026608173e+58L, 3.03418310785320829824308157124864982e+59L, 2.46716192600983889917114272951924381e+60L, 2.07290103981358059329932171745274528e+61L, 1.80056398057961538305615630314631614e+62L, 1.61776402789534425695658193709805813e+63L, 1.50428302825068832904492692640561694e+64L, 1.44839320652542717214779500803249160e+65L, 1.44485551098011579855868939330990133e+66L, 1.49412042885502924277326734308483245e+67L, 1.60256656610701572194891345599497651e+68L, 1.78388050415394298763315503064430164e+69L, 2.06199924057276073839995264039101124e+70L, 2.47652179469857271457600275335452230e+71L, 3.09234991415349735847967247784284417e+72L, 4.01692723830598581004567643058591262e+73L, 5.43160754522649738742682321649737815e+74L, 7.65008682404282275932608294522372170e+75L, 1.12301798411434928750045233045701052e+77L, 1.71938295296605200401018803799684912e+78L, 2.74733571869068667385885336550518795e+79L, 4.58454501055768412308971248876333874e+80L, 7.99508204153925025226317876728470146e+81L, 1.45811990936589904392804782726885652e+83L, 2.78300117867960017490532778932474629e+84L, 5.56281223196619462795860275879049153e+85L, 1.16533876898240457849831832137567995e+87L, 2.56039912643283822420817604614012086e+88L, 5.90454964185909819164533268125527979e+89L, 1.43027847474983870988836419822056214e+91L, 3.64204612295693256305339371887263313e+92L, 9.75669857120640229967783787120452211e+93L, 2.75194604427588305129487979774984195e+95L, 8.17916479364319727936284803066665955e+96L, 2.56370473508682589047202009616066313e+98L, 8.48165649612825587957771293185197099e+99L, 2.96426025440398100674466172613133159e+101L, 1.09534297003120888620652471421812455e+103L, 4.28314854758487062838465336068295133e+104L, 1.77395435294431974409395342323139198e+106L, 7.78899108189422475969140390458031321e+107L, 3.62893172105682135159715800277136586e+109L, 1.79572927251602059220958063790243562e+111L, 9.44668515148283533862523011302496150e+112L, 5.28826317961448810089554134850695191e+114L, 3.15331123674140136231493452577546350e+116L, 2.00480707968382766860514611914082879e+118L, 1.36040719266523771611516008331762962e+120L, 9.86282560980781051743602195410395887e+121L, 7.64755178859112809938428139017982021e+123L, 6.34880222487173008841344329862213996e+125L, 5.64906236198001909791959175488227229e+127L, 5.39324800352378478055022039572371179e+129L, 5.53089719191570391565414448432944942e+131L, 6.09959864464089433289271654904908851e+133L, 7.24209843349196450391058906977712497e+135L, 9.26808305363737556950784175210077690e+137L, 1.27994270241604058214925694813276489e+140L, 1.90979662696062130244831484512150929e+142L, 3.08254030066988503957988563764712878e+144L, 5.38880973238417965706179354197373928e+146L, 1.02161025105662653467101460483765248e+149L, 2.10300544007279065014421211807768152e+151L, 4.70675399034872557015773705065650223e+153L, 1.14683412812524899101968167573148610e+156L, 3.04620262376779901134561159740015427e+158L, 8.83255301555740088857655282843830654e+160L, 2.79951315833790052768515917244690507e+163L, 9.71307000884329724595836237775132776e+165L, 3.69425449543187541186652254254454177e+168L, 1.54248755204034481247092284401752383e+171L, 7.08075007659653083674442572183052884e+173L, 3.57888569536270290704774796012385631e+176L, 1.99473765641034107285413476042860568e+179L, 1.22789700989638770500092926180435793e+182L, 8.36095001442791620159284429240141671e+184L, 6.30749438488829733664127973626729614e+187L, 5.28039699582730274851611343025005913e+190L, 4.91356459266260178144685151251583280e+193L, 5.09061556382806163145943155929283695e+196L, 5.88195913486315120755496558081929324e+199L, 7.59272859182451584412681161192886089e+202L, 1.09687154447604565252927157604063023e+206L, 1.77650105264457177914887205035954057e+209L, 3.23153511879574325299022213712900198e+212L, 6.61424415330008555964088497876375007e+215L, 1.52611134962392865764465743843181152e+219L, 3.97691502072254861067816109956158434e+222L, 1.17271592685350180601663411607927347e+226L, 3.92076851079784438154800269803529292e+229L, 1.48915601904650135853122181753421959e+233L, 6.43831323019893638007682069747894997e+236L, 3.17506717851901774760997889290937204e+240L, 1.78970981603629426034165602936202663e+244L, 1.15551435950001094003467361479483663e+248L, 8.56366431033316582912879348191377839e+251L, 7.30093770528170061921940254565023758e+255L, 7.17613138815551609337504025522760569e+259L, 8.15019404042552378087870259624107791e+263L, 1.07201229194901841245742935320285314e+268L, 1.63677814658507005123653360340510968e+272L, 2.90775562270833690431514296460636317e+276L, 6.02476179304325236422526103127046739e+280L, 1.45944561364291559252990517774206937e+285L, 4.14353248208966528107409825739318649e+289L, 1.38220972739972368321976302536673924e+294L, 5.43125471568796436605394023488404770e+298L, 2.52039989595272682474151246524004609e+303L, 1.38490332203279628724542169932072504e+308L, 9.03453415480179250943103381623811201e+312L, 7.01620311300324094081813342355630834e+317L, 6.50430483422567669743617206850738485e+322L, 7.21793562600493199951776097195343971e+327L, 9.61542788471746566451160668498411528e+332L, 1.54211655565340209996546117001243607e+338L, 2.98626205265198316430063943341329432e+343L, 7.00309320577422492544778192980108604e+348L, 1.99486089518136492496460973366880635e+354L, 6.92350663128275446878446973158773950e+359L, 2.93684755383592258491626945473185368e+365L, 1.52739293250863294426010436780725812e+371L, 9.77075509850276765943058812672579852e+376L, 7.71310608088780517993569363011828070e+382L, 7.53863036472313339350939561575995488e+388L, 9.15331385155864163687518059055937567e+394L, 1.38538605697398143363179835422022577e+401L, 2.62287133256068615930266548676255190e+407L, 6.23343982286571042809890547122705804e+413L, 1.86628588041890755929921756251934723e+420L, 7.06492694188791912037300528166794228e+426L, 3.39406886800977749230458579824930803e+433L, 2.07704891812976483907563047536681219e+440L, 1.62532865525590176989706193358042030e+447L, 1.63262112354994112352218911327913750e+454L, 2.11342684365334123128646755974119857e+461L, 3.53982137873383101244091847412793714e+468L, 7.70246700302666062188915359100825615e+475L, 2.18636422788338230254129077405708124e+483L, 8.12975261672049081897166938196495657e+490L, 3.97686557044826916151646962242652853e+498L, 2.57033035969007091507617008526033295e+506L, 2.20458407348036235529376715504150602e+514L, 2.52051509380683369529400145083533353e+522L, 3.85871376898138292734730034828559871e+530L, 7.94663225793828689108907007781152553e+538L, 2.21176541223395827472447113026924253e+547L, 8.35931033048630497134082029598865726e+555L, 4.31091797212556485985662056018893390e+564L, 3.04833317305882254988404372326531988e+573L, 2.97034828681146526635277450855709161e+582L, 4.00864404566992526097179834506253327e+591L, 7.53113144131492631353410204171028663e+600L, 1.97996763586022811207860344830247407e+610L, 7.32300332662100885299893727481183608e+619L, 3.83078680906847773947984890776183687e+629L, 2.84987258373806149506606754536239806e+639L, 3.03186631046306534459943952897407369e+649L, 4.63861318260680959360747724936035922e+659L, 1.02646622253495677529633262335412306e+670L, 3.30448344760519942319105428211388701e+680L, 1.55678514244926540491694986676104766e+691L, 1.07975219247204914347510266773073325e+702L, 1.10926415409608640204268413223928121e+713L, 1.69843188882084389205231670587156933e+724L, 3.90025209215096037836224226547246383e+735L, 1.35188796159061122898267917047667153e+747L, 7.11881895741722013661309086794573765e+758L, 5.73263787962641176699015570741387791e+770L, 7.10698682909851371094390920546530663e+782L, 1.36568905751347705631533710470350932e+795L, 4.09591384091714486453161283605440034e+807L, 1.93075428208975454032168180693618424e+820L, 1.44069783794868490577087909226154603e+833L, 1.71407451380697883211569548277738601e+846L, 3.27557874564463748685767424892341420e+859L, 1.01295195987435253561137255728975319e+873L, 5.10768437019276174029340754351596177e+886L, 4.23191986113702770379280941046346950e+900L, 5.80664714178397794019307157189161329e+914L, 1.32995815980583547946711112525038194e+929L, 5.12600842672542100652183507817331030e+943L, 3.35203762278831767267193293263751763e+958L, 3.75009417204153826893896307870937057e+973L, 7.23855050824188975713471018223024339e+988L, 2.43146178229439003194650931378387467e+1004L, 1.43376736234835679429446577737976919e+1020L, 1.49738681485939897351147361037410278e+1036L, 2.79474808763788045317059787958854766e+1052L, 9.40752795141195895210630116566470676e+1068L, 5.76455681194134988563523968421696548e+1085L, 6.49098750070655215612470119412325218e+1102L, 1.35603677524062211064037225514112165e+1120L, 5.30731596106421482379267500594247426e+1137L, 3.93020043869734370491941322540535807e+1155L, 5.56226722613029149136813618379760709e+1173L, 1.51990408820680241841102181882025951e+1192L, 8.10228003417607034858018290055212984e+1210L, 8.51521180721994744419289716417963237e+1229L, 1.78329495296502824223674285976432097e+1249L, 7.52325725334531880022021234450122858e+1268L, 6.46447702663797449769360764507187142e+1288L, 1.14411796435953528891736478925455547e+1309L, 4.21851818337788742448241659152628519e+1329L, 3.27808619028067308419286053048979085e+1350L, 5.43188779501763188996015031600338462e+1371L, 1.94236185576822902995555282219996135e+1393L, 1.51711647857248503907031482339088565e+1415L, 2.62035716514505778741231466041668281e+1437L, 1.01340333806343444348385264050569250e+1460L, 8.88786410219005270769408353397020687e+1482L, 1.79062851327433476234422894649930707e+1506L, 8.39642792144232596950658702528918677e+1529L, 9.28628794761707451127390451820966789e+1553L, 2.45536895088636462839367313490581387e+1578L, 1.57354562229875356978426356241845302e+1603L, 2.47847583619098850454983740131691503e+1628L, 9.73157119375940175602769413110414643e+1653L, 9.66320447484532719750186034583884499e+1679L, 2.46232109538721189753853963925622165e+1706L, 1.63417596934387293040006746046570718e+1733L, 2.86767794799456586471233013913198039e+1760L, 1.35110548955493375904544893914917995e+1788L, 1.73592152545700077097814839827674920e+1816L, 6.17893831026015926230407301367680801e+1844L, 6.19169576655007716364858612984691562e+1873L, 1.77539961419331109157892833779912252e+1903L, 1.48102931564535770775910220632570312e+1933L, 3.65523833138677287346435504279580939e+1963L, 2.71500576533505130343548298805675281e+1994L, 6.17538661057993294979124427622665421e+2025L, 4.37773911614056336228335278244614535e+2057L, 9.84696637606677900175890481086698860e+2089L, 7.15679932872021171306956409599896671e+2122L, 1.71206538074954819527676482193571839e+2156L, 1.37358203471630226464631105286036694e+2190L, 3.76701670663930120912543386433302245e+2224L, 3.60043887764831236838085599723119708e+2259L, 1.22311399177140229476034440208616729e+2295L, 1.50662485913882109231872682750679813e+2331L, 6.86720031172136555587682118613084257e+2367L, 1.18233110675109495729851537371527994e+2405L, 7.85186218386652904615339091354671418e+2442L, },
   };
   m_committed_refinements = static_cast<boost::math::detail::atomic_unsigned_integer_type>(m_weights.size() - 1);
   m_t_min = -8.8984375000000L;
   if (m_max_refinements >= m_abscissas.size())
   {
      m_abscissas.resize(m_max_refinements + 1);
      m_weights.resize(m_max_refinements + 1);
   }
   else
   {
      m_max_refinements = m_abscissas.size() - 1;
   }
}
#endif
#ifdef BOOST_HAS_FLOAT128
template<class Real, class Policy>
void exp_sinh_detail<Real, Policy>::init(const mpl::int_<4>&)
{
   m_abscissas = {
      { 1.02756529896290544244959196973059583e-2497Q, 2.56737528671961581475200468317128232e-919Q, 1.17417808941462434184174780056564573e-338Q, 4.82182886130634548471358754377036370e-125Q, 1.85613301660646818149136526457281019e-46Q, 1.52174118093087534300657777272732001e-17Q, 6.75122240537294392532710530940672267e-07Q, 5.94481311616464419075825632567494453e-03Q, 2.00100938779037997581424620542543429e-01Q, 1.17328605653712546899681147538372171e+00Q, 8.18356490677287285512063117929807241e+00Q, 5.59865842621965368881982340891928481e+02Q, 3.69902944883650290371636082450503730e+07Q, 4.05747121124517088709477132072461878e+20Q, 1.07723884748977308147226626407207905e+56Q, 2.07250561337258237728923403163755392e+152Q, 1.09889904624000153879292638133263171e+414Q, 3.02463014753652876878705286965250144e+1125Q, },
      { 3.16339947894004847541521297710937343e-1515Q, 7.02786763812753004271170107158747593e-558Q, 1.08465499866859733494552744914276656e-205Q, 3.97220624097731753646493300738183748e-76Q, 1.84145280968701148636501796762008815e-28Q, 6.40886341101610144938011594904313817e-11Q, 1.89200275395722467663251234615351291e-04Q, 5.04892743776943294143478969573235061e-02Q, 5.25946553709738448524487603870477588e-01Q, 2.72635049439566736308739858009584752e+00Q, 4.20081657572145199006060015525396829e+01Q, 3.75221714819472331206969009243353365e+04Q, 3.11642465424592079726559332629050577e+12Q, 9.61947229245436139236957297153092319e+33Q, 2.42007357898805628851519620917402234e+92Q, 1.34351470989086331111432001407522043e+251Q, 4.34937560839564995558174638985438726e+682Q, 4.01820305474077838467315109614612109e+1855Q, },
      { 2.20630426387562582340901270736066292e-1945Q, 3.20902289525796097195876526710905754e-1180Q, 3.99075419312896749376535004306952481e-716Q, 1.22680463935505388461578092999320797e-434Q, 6.60416836081631149205027836695167940e-264Q, 2.36380363265905808053031227647927829e-160Q, 1.52216180146092756550894692311532796e-97Q, 1.89865679919908959317123677189779258e-59Q, 2.43020115474101871630434504433731237e-36Q, 2.52466874690605714759066961852641028e-22Q, 8.05169965363444223580324196979926112e-14Q, 1.16754399126294292055540155961164512e-08Q, 1.59887725419859990767684898940272339e-05Q, 1.30653650105064376247657363109776765e-03Q, 1.96025435814529675467866513121889090e-02Q, 1.07798510399525035644857486110150806e-01Q, 3.35726506001932774135392168623644982e-01Q, 7.91344191300036661743190713036639943e-01Q, 1.75712551051579342077493249233109834e+00Q, 4.50532922519182663928094313196688277e+00Q, 1.69634912879730950991827009814964854e+01Q, 1.31564914034011933527149556208430823e+02Q, 3.54551665937177335700906931219692079e+03Q, 7.69629182688413474235641078406238234e+05Q, 5.31334781541946297455601223275400364e+09Q, 1.11397125403497802554344014693570895e+16Q, 2.91323342059626628322620905024841007e+26Q, 4.33470414741675827465254900104887599e+43Q, 8.87077939346041258291094819287939414e+71Q, 4.20552510572288598564618972079522209e+118Q, 3.78998758665619810759092788424597102e+195Q, 2.85285821381079277498378752581228807e+322Q, 4.35767992687500025738868152293904617e+531Q, 3.35101732969916400190150631384628168e+876Q, 1.39907636805339624393396738787997778e+1445Q, 4.39310839133852761600954729262867435e+2382Q, },
      { 2.60416027838200926634868701489352534e-2204Q, 7.02786036162213986704884019362612079e-1717Q, 2.87472449554862557122883045361480216e-1337Q, 1.26052689335617094297502885256721406e-1041Q, 2.22203737723769125613395083355625930e-811Q, 4.59925691253299141912321586723335574e-632Q, 2.06110387430222811031505778840893135e-492Q, 1.18783333719310840737835079091916047e-383Q, 5.99339963191580129696690680497340117e-299Q, 5.55156577871295836646608994333019205e-233Q, 1.31650343182343962237291034126486362e-181Q, 1.35017224106781623208439493207058713e-141Q, 1.95458267270042896122641878745090637e-110Q, 3.62345406499123808121116200173051239e-86Q, 2.87916036185197771976746396107338099e-67Q, 1.50975853574758038650941626753584613e-52Q, 4.39366381209590686916061999351406334e-41Q, 3.72613279781933265810269222029431206e-32Q, 3.35074541708050784063416140930256908e-25Q, 8.73699691100611067810340427391712629e-20Q, 1.44610938754441658588026566732900630e-15Q, 2.79712542830915646153531586097835651e-12Q, 1.01712318476608889609455916646557919e-09Q, 1.00707686902351812525332969334486408e-07Q, 3.62377064535647765957086044101238700e-06Q, 5.93514765312557852887858990273347709e-05Q, 5.27427922038425038994892508697801291e-04Q, 2.91716284057748187538183216214780172e-03Q, 1.11812753052373081344852811278990196e-02Q, 3.23158910999791296389217134204252891e-02Q, 7.52823576293971217140453145701166400e-02Q, 1.49072381771447883955117212578891471e-01Q, 2.61920116954195648982059938116700281e-01Q, 4.23044722849733503488984642521144095e-01Q, 6.47337815325830827783598163922905049e-01Q, 9.63850663681748439752705317539095691e-01Q, 1.43180513983798487645924505146418541e+00Q, 2.17546288127550398262362304432746644e+00Q, 3.47077553215380191942193887548951574e+00Q, 5.98762125926090985873603940556086485e+00Q, 1.15588346593765285104362265791852722e+01Q, 2.60227165873113109321939526707132515e+01Q, 7.18972764924010846908346544069757730e+01Q, 2.59849827899514040038730358528910272e+02Q, 1.33180383652908565595482681440747426e+03Q, 1.07264199927746293597346233164919561e+04Q, 1.54770252854197591129695964624494039e+05Q, 4.73116974059692035499030617203501902e+06Q, 3.79850299532978550624970142683299187e+08Q, 1.05506900281875210383901752883017567e+11Q, 1.44377693764054834189876938092028936e+14Q, 1.53216987231286586179401429003390945e+18Q, 2.25749845424265693444972830430363021e+23Q, 9.75366126404791278425650801377275643e+29Q, 3.22710662318561074542283240061745423e+38Q, 2.80427989450823486864584373674252213e+49Q, 3.11492865697270427645920832671970699e+63Q, 3.37385880956075754428852062324609716e+81Q, 4.84069913749095116306098688426537434e+104Q, 2.62206965911556490036221731139595538e+134Q, 3.95724512624896651389226023882699109e+172Q, 4.16658233419696139622872515762144259e+221Q, 3.67870997815873300310505495683155714e+284Q, 2.45353561048417937411865685638919263e+365Q, 1.47890836767899626666102902312593029e+469Q, 2.66842264577698782467085710268227542e+602Q, 3.39406520823137428862537121090858482e+773Q, 1.71072929388916771882465310676448508e+993Q, 2.17124413154098944294625847065193945e+1275Q, 3.67108839134079465802181460380664133e+1637Q, 4.72992162468569560939225829187204387e+2102Q, },
      { 1.98007625288857013412122004727810624e-2346Q, 8.39652992194258659266745309526472882e-2071Q, 1.46030671151231956105710426802408027e-1827Q, 6.65727248909208458885310717390128801e-1613Q, 1.81581117219394388092797723844111009e-1423Q, 2.72619674739250219990613941747932927e-1256Q, 9.29590026994488020433315206443278096e-1109Q, 1.46377917106366800430984793284730939e-978Q, 1.15901956513202291149840314964756250e-863Q, 2.89578802021929239619753606490350524e-762Q, 8.80858736107764685086031513644364894e-673Q, 8.19416690740938676449696497280686335e-594Q, 4.00916745706227520865721440095991259e-524Q, 1.27020033549323657895766092175564312e-462Q, 2.38869695900235565316537719061693975e-408Q, 1.88391475973801739487501436111488490e-360Q, 3.49894275509125434733587030489899389e-318Q, 7.01615886164307390146523314851863544e-281Q, 5.82310443050787035609701710903434681e-248Q, 6.54836166711216765944679142249433382e-219Q, 2.84149263299034989667159925017962142e-193Q, 1.19812648186130338854458247959762818e-170Q, 1.10911490518050678588403791608480488e-150Q, 4.62754597695358582538782054737574180e-133Q, 1.64173210211161942076907089038989838e-117Q, 8.67192889174269982709959745767681981e-104Q, 1.11812225550444523531457194417899189e-91Q, 5.44369950217028498211921027217139012e-81Q, 1.47074883585505403248443957516006445e-71Q, 3.09730398495823549037183157408809652e-63Q, 6.86204181060118439675162432029019608e-56Q, 2.08386500906124109876681630886775453e-49Q, 1.09560045982532469724721792988430545e-43Q, 1.22547627904244500951228300724497066e-38Q, 3.49796660995417261286766404577301528e-34Q, 2.99152410870631960427539566799345990e-30Q, 8.83190101026086766963628794276374683e-27Q, 1.01998228741819737621697177531967545e-23Q, 5.14543767526486873178240241354085638e-21Q, 1.24976972746216000788747786633668576e-18Q, 1.59270820536117731616569419326771375e-16Q, 1.14889208313232540746001768184091876e-14Q, 5.01568630936388404881919116817433643e-13Q, 1.40588174426346693556351457090253957e-11Q, 2.66552656620728447371725338328298653e-10Q, 3.57952342849715748848195322266581664e-09Q, 3.54587967892367612935335788099896843e-08Q, 2.68571894340447967655761437829800783e-07Q, 1.60543396069231405992036918112042239e-06Q, 7.78876789102767893920047052886009844e-06Q, 3.14354940320849664611373818938001854e-05Q, 1.07876626606295756489869617036039985e-04Q, 3.20898851002890606916875803454736038e-04Q, 8.41685517064122028463642895198591848e-04Q, 1.97617054782608049591289962378898411e-03Q, 4.20911897696440311235672067499450921e-03Q, 8.22976828962407304914730770011736102e-03Q, 1.49268961502975158975071505963565168e-02Q, 2.53502658698472066372315325797964553e-02Q, 4.06466974195663810865864410951160977e-02Q, 6.19877276359776967830177840166714717e-02Q, 9.05089155125712182450563511209635860e-02Q, 1.27278213682114666289711516555196299e-01Q, 1.73304699976842406023992088378438619e-01Q, 2.29592410233034978491448000870038137e-01Q, 2.97239599113018852567831136709176961e-01Q, 3.77582948042641879160927817217899876e-01Q, 4.72389555885863401785327685882008067e-01Q, 5.84107138056041651067826639644019633e-01Q, 7.16195243265456514286143145222900703e-01Q, 8.73579162273458922635006193850777630e-01Q, 1.06329740288293038148703202109279326e+00Q, 1.29546076177954953883340629341283024e+00Q, 1.58472055402981935372999954552668421e+00Q, 1.95258275532953319986830507945836181e+00Q, 2.43115638685977475925559253395572515e+00Q, 3.06939317426312452000366876337463330e+00Q, 3.94377618122435031881432297693345182e+00Q, 5.17720373327574287465684980172077361e+00Q, 6.97349551419502029130075958419090062e+00Q, 9.68282046752329620408323930717367162e+00Q, 1.39309131038991828857956159913232231e+01Q, 2.08859990746605884557374483895633501e+01Q, 3.28382897425881117409618357674054401e+01Q, 5.45293246810142904942496502744410196e+01Q, 9.63939781365248203122153852027923369e+01Q, 1.83025185098871555179881958564576181e+02Q, 3.77024517378370245776498953575447428e+02Q, 8.52190226688445340278025693774183776e+02Q, 2.14068512989192644263700461005007744e+03Q, 6.06284853067739175777069687971114492e+03Q, 1.96780457938951378915777196082473423e+04Q, 7.45535092385462432927423608234859698e+04Q, 3.36655497164547806089503954112397178e+05Q, 1.85514609429466771541276069582993197e+06Q, 1.28127256599195512602695379572162760e+07Q, 1.14315867886785361488804976072737382e+08Q, 1.36346329402339162906880444623714674e+09Q, 2.25993073791019766670398383679728111e+10Q, 5.43926612995997228479509745730508310e+11Q, 1.99798484325959673290782033498623904e+13Q, 1.18504970239150114123104110601445669e+15Q, 1.20980370818215194207683618902723859e+17Q, 2.28537992087984292439473774950290832e+19Q, 8.67120977340450400779923403171032446e+21Q, 7.25153649943518021867879355500893988e+24Q, 1.48501435335333039333736586197814469e+28Q, 8.39050326550867736343407650659898774e+31Q, 1.49728470198356221256669260283197052e+36Q, 9.83549601762784922530731490296260394e+40Q, 2.82901352112032614706131527899887976e+46Q, 4.33736659101971830974347248793299740e+52Q, 4.42940467871587853569566546424045697e+59Q, 3.87840248690238104186373399976301795e+67Q, 3.87622435037312042010986878431609700e+76Q, 6.11530088968567983241864570883470440e+86Q, 2.19902986720444927672673080107769693e+98Q, 2.73302501743809592753235404592828204e+111Q, 1.88157705379416554346750034182897643e+126Q, 1.22464500548199778033960287699262627e+143Q, 1.38089682931886763280665214047107116e+162Q, 5.35861724121451785967201429765672158e+183Q, 1.55758090841089261682638940540906561e+208Q, 8.18640878839858943416304034752205938e+235Q, 2.11193471456685131226350980039759737e+267Q, 8.29196100559631320106113784120111420e+302Q, 1.78609329205356222554295903766141624e+343Q, 9.02504022060557635111807879176244000e+388Q, 5.55060710318615757165502584223064764e+440Q, 2.68431690505806342057709029465018309e+499Q, 8.45412788268345327632392585778624037e+565Q, 1.90297947362180093404817251840979577e+641Q, 4.62215189323912708971592873336189679e+726Q, 2.62533210543270812917133055827838787e+823Q, 1.13817822812561602537416032879617414e+933Q, 1.95538622136936887385682087493096654e+1057Q, 1.16937064578842112561072532780020718e+1198Q, 3.88035117517281013054730623266703196e+1357Q, 2.23743584154220985565150075331030128e+1538Q, 1.50889533002797316376499137963913497e+1743Q, 1.90641621776286179122749804749534564e+1975Q, 1.93080249271305970633007359089743998e+2238Q, },
      { 6.85586909881832886951451464772591878e-2421Q, 2.92677267243682665120741515792112744e-2274Q, 1.63242588944068781532732923861018809e-2136Q, 4.10812064897186984969866413043968829e-2007Q, 1.49435786878493820312929790405374745e-1885Q, 2.34568268208591413691385316271709680e-1771Q, 4.43925157300673395426635808699168955e-1664Q, 2.65927633627471464278071651186161190e-1563Q, 1.24859182507079754991269604284796564e-1468Q, 1.07699173095552638424762549404812646e-1379Q, 3.79889818022139862967145673157136925e-1296Q, 1.16203657951916401289541501876509754e-1217Q, 6.24572482158628957114104388658639197e-1144Q, 1.14511680730607058597658427712436799e-1074Q, 1.33557521045192075572641149916774170e-1009Q, 1.77946124922921650079213398130306155e-948Q, 4.69414096094237830969243170096712625e-891Q, 4.11003931775404922019148615555751563e-837Q, 1.94061218235885716000492708678051285e-786Q, 7.79551077888128366830436599420622304e-739Q, 4.08861993672760936738017704029701813e-694Q, 4.18676490857955209780618740974196173e-652Q, 1.22153592554301240722464717581112377e-612Q, 1.44833892014264382948310395651753089e-575Q, 9.74176764355703592877272264974985856e-541Q, 5.08509650174353124708754662233746797e-508Q, 2.76502760440391058924306176438285126e-477Q, 2.06508450540542778211999232986493505e-448Q, 2.74686348165461297466693117700925333e-421Q, 8.30585680824802634886342521419101161e-396Q, 7.18035163161661527310024383149797653e-372Q, 2.20117367274887299835421816737413275e-349Q, 2.92938701204713835430829657327314622e-328Q, 2.04670856065666661814970623236244653e-308Q, 8.97499107305077146284125789528155163e-290Q, 2.92115595376553501154165826580135083e-272Q, 8.26130614121671756652659688034185753e-256Q, 2.35396777618554392325312768878283221e-240Q, 7.76604519643439440768615254501273896e-226Q, 3.38047616528897968517094694597452046e-212Q, 2.19496817732884969215875687943120973e-199Q, 2.38570033705332693932010529683394408e-187Q, 4.83696042571616203283798802745975902e-176Q, 2.02525789452808699182428060822541636e-165Q, 1.92683544261267768613934270120773225e-155Q, 4.55675928208753416426295293555087690e-146Q, 2.91435926363522943509016642164959990e-137Q, 5.45650834446039875776911937813876647e-129Q, 3.22177915240208624055473617062236208e-121Q, 6.43356918992122712617506763079342635e-114Q, 4.63991207894245637238928734792612888e-107Q, 1.28548526430585878202588260968441525e-100Q, 1.44976784442529508497784801483631129e-94Q, 7.02834477739882506904444529125607921e-89Q, 1.54151343887499654318252629091310298e-83Q, 1.60491367376894945627919582278915480e-78Q, 8.29797555416253956153582009155421152e-74Q, 2.22293580147262466985937857298017334e-69Q, 3.21083741325090217758867273653191446e-65Q, 2.59597447976318059491815288096285983e-61Q, 1.21686551839843562559756380386148224e-57Q, 3.41813412178077321753808876760492064e-54Q, 5.93492497756373178397326564626159285e-51Q, 6.55812810449229009165849681654564859e-48Q, 4.73996462182817624913218344729897191e-45Q, 2.29917713906026251753218108658989939e-42Q, 7.66772810214285848709307613966489982e-40Q, 1.79852699731596078211094021208953882e-37Q, 3.03099351897543871174212829426305144e-35Q, 3.74430827279655104510043134133803179e-33Q, 3.45501893639921538125092697393537600e-31Q, 2.42381852080187080923760278831274486e-29Q, 1.31441976082623542058289755424927250e-27Q, 5.59666006060409162050973561377317535e-26Q, 1.89867556602582040862526655529336748e-24Q, 5.20331508297836691753631952262461629e-23Q, 1.16690464600934423260878649807673403e-21Q, 2.16767714527916659623320357151600374e-20Q, 3.37377643107659326572426331386135943e-19Q, 4.44691383264786489190950322799097537e-18Q, 5.01415830137739970673756811085508269e-17Q, 4.88253693365386298188724591254681515e-16Q, 4.14251016844320187955159569890123671e-15Q, 3.08802476085892421415530065712870862e-14Q, 2.03847832924953919920668690856932749e-13Q, 1.20044498484990029827708442066317039e-12Q, 6.35035779339988133329469515026722103e-12Q, 3.03739182163512379493334464659410947e-11Q, 1.32161843156591616383150953292325245e-10Q, 5.26149741865431306798513845101472462e-10Q, 1.92688222163920338777595529204662553e-09Q, 6.52448684765263503483691364064710302e-09Q, 2.05235608001812174069568432464532832e-08Q, 6.02447248155606588490206026101601127e-08Q, 1.65719156589179965185098196474263403e-07Q, 4.28875221376115411615543853081272079e-07Q, 1.04811127032430288440405169279023586e-06Q, 2.42727195841237101271799656914039580e-06Q, 5.34428013249275030874689017597357100e-06Q, 1.12217116002251908227007056847369186e-05Q, 2.25365270095215311537786875281660527e-05Q, 4.34066412230525728790156860647261934e-05Q, 8.03857428545025320920443714330221510e-05Q, 1.43483273166998782632422720673552674e-04Q, 2.47403670532944916561679376448099333e-04Q, 4.12969671314554699465069528559056198e-04Q, 6.68662248079464048176726608426967422e-04Q, 1.05217959874444039957269021715673554e-03Q, 1.61189482479878719591766861379534126e-03Q, 2.40808122992764072147410965445120710e-03Q, 3.51377854902820551940988678116833582e-03Q, 5.01519359256763066459806052313013320e-03Q, 7.01156300574609092374647377614547795e-03Q, 9.61445020754398604050247376189537687e-03Q, 1.29464977958074216006936641439568225e-02Q, 1.71397050059386052582575161031849539e-02Q, 2.23333418628568405596661137294311321e-02Q, 2.86716433323270031012958633821643459e-02Q, 3.63014455768061096476987670693482715e-02Q, 4.53699316668876641396409342510559075e-02Q, 5.60226467568397916052157799832294784e-02Q, 6.84019250622201277408275217514564750e-02Q, 8.26458460699460598611997466316916292e-02Q, 9.88878087044773835985179298425629188e-02Q, 1.17256783027063660721874825783310801e-01Q, 1.37878272417301116163817842123251189e-01Q, 1.60875997439806117261603052596128705e-01Q, 1.86374297424717578638710069246872277e-01Q, 2.14501238238148719030968730766268227e-01Q, 2.45392415301662505713896222647369518e-01Q, 2.79195449773929877272191161144448290e-01Q, 3.16075192872379294294649481827620946e-01Q, 3.56219678521249637323233310278349444e-01Q, 3.99846897980088704601361036413251085e-01Q, 4.47212512313163107391085998119621599e-01Q, 4.98618670533294760818150011647972671e-01Q, 5.54424164764947975414033158173924804e-01Q, 6.15056231563286401803829984090534551e-01Q, 6.81024404595688995229840589926798114e-01Q, 7.52936943869155645889792590166619714e-01Q, 8.31520518350208659550187071679434118e-01Q, 9.17644012826577357641491486976575868e-01Q, 1.01234758075340210126152790231895045e+00Q, 1.11687839251578850642753265260268328e+00Q, 1.23273496036260391788838218839806886e+00Q, 1.36172249498191084612564077792923468e+00Q, 1.50602251678823434544288956361434194e+00Q, 1.66828098096960364511379075299209308e+00Q, 1.85172058286684745279366828495999051e+00Q, 2.06028483669890596285992438994232879e+00Q, 2.29882417717996662881312374953444179e+00Q, 2.57333802530471722160283306099247239e+00Q, 2.89129193110240878387703249643229033e+00Q, 3.26203621106764094357461881213752464e+00Q, 3.69736290590815315504600654149629845e+00Q, 4.21225284743951568672550404261498190e+00Q, 4.82588633844219080687269426289849605e+00Q, 5.56302277261292337309431040275971592e+00Q, 6.45590163750149736955774273721315509e+00Q, 7.54688784770818103174495320257408374e+00Q, 8.89219103984228343085851433153285412e+00Q, 1.05671517790393183669301386672470668e+01Q, 1.26738407015152894733455767759949041e+01Q, 1.53521137941817792315134763759132043e+01Q, 1.87958986899048219790124225448671683e+01Q, 2.32775055780405432294765542807163514e+01Q, 2.91844233861930596156964126095351325e+01Q, 3.70758319218904582325116346962442845e+01Q, 4.77707378224399722392124081823831658e+01Q, 6.24876734446863447811669818713989778e+01Q, 8.30699342763174311126855611293197423e+01Q, 1.12355321585737491915090874360361047e+02Q, 1.54794728437631233369474093491065001e+02Q, 2.17507985417556811263254275570301936e+02Q, 3.12124586781855603508049115425717265e+02Q, 4.58065302025763509177549206574829939e+02Q, 6.88532496785780240326739257222869086e+02Q, 1.06172181511411400423407908864360706e+03Q, 1.68236894049421021674529061389284712e+03Q, 2.74433484749143274695935635590635551e+03Q, 4.61730673523479769409382001763470205e+03Q, 8.02895513401715463422140861600391503e+03Q, 1.44606187348593941052176609549649644e+04Q, 2.70377620144727936743806059397776390e+04Q, 5.26105241201059109738487092327866872e+04Q, 1.06812531849740275922510876100737843e+05Q, 2.26893075168541256271862004134549591e+05Q, 5.05764404902608855969935895677933433e+05Q, 1.18676186494579079987238943033223000e+06Q, 2.94113264423683227561198247386975731e+06Q, 7.72576514719998793452586494715302612e+06Q, 2.15915178528480833880909472948132169e+07Q, 6.44590226372788402030541979463995492e+07Q, 2.06442545099697944602613003602586209e+08Q, 7.12532948492900300739824830002676479e+08Q, 2.66319659068655507734050368458244708e+09Q, 1.08350636970002739580921292680640244e+10Q, 4.82470799147337538737281851702596161e+10Q, 2.36513804063572720852686335350104312e+11Q, 1.28435637164102683934687705650167990e+12Q, 7.77731246565628041938451985303773831e+12Q, 5.28864903733985311795604320721713645e+13Q, 4.06896744489041480369429188279659070e+14Q, 3.57034809188328432417142254050169691e+15Q, 3.60337498222976618360402297528737217e+16Q, 4.22089025190487061069708630018503728e+17Q, 5.79389086782171589014524617169039021e+18Q, 9.41571436923218772663826142877596428e+19Q, 1.83140546580432476725446560201754697e+21Q, 4.31320926121717399384742067577025671e+22Q, 1.24526713689819970862562132007699682e+24Q, 4.46557396336452476527850690333252885e+25Q, 2.01706317120607297931541128182425236e+27Q, 1.16481109175988266234418168600383172e+29Q, 8.73712441785116756599086490733480008e+30Q, 8.65736270143027267976956887405444609e+32Q, 1.15373549848396029384279199583578574e+35Q, 2.10781669532016374773265262680619511e+37Q, 5.38769637251502198497400987247791395e+39Q, 1.96890474908610530032963280794536401e+42Q, 1.05271764511336947295087542840771762e+45Q, 8.43965629752558882239469624058575644e+47Q, 1.04138369598852386365061187570128647e+51Q, 2.03352356915167672502196854739506820e+54Q, 6.47289125189110545523503523387311437e+57Q, 3.46613548082834986367032599943319188e+61Q, 3.22894792541599068853355143560944151e+65Q, 5.42318759743953119742239523201238475e+69Q, 1.70583228507675596981951702618714711e+74Q, 1.04635953488687800354356406620895146e+79Q, 1.30676249978604401490579288070545190e+84Q, 3.47855004888451734894061507389280036e+89Q, 2.07245356750112312931588098931095741e+95Q, 2.91086857580213998298216107061151403e+101Q, 1.01866939773917036897104863230086276e+108Q, 9.42079727758602983661368103980660631e+114Q, 2.45135270885215193905442997104007131e+122Q, 1.91850621913423378510904034515156001e+130Q, 4.84846348541576375648253727961523398e+138Q, 4.26738728648259195430121429627940133e+147Q, 1.41771213016857192102291892633345650e+157Q, 1.93681124067818946208050336019146634e+167Q, 1.19196764675833722929285298195873591e+178Q, 3.64148488428297947856471515841164734e+189Q, 6.12360784372861046853690882962104289e+201Q, 6.32737167744109620998113270972893770e+214Q, 4.51626866922887039378811138276232919e+228Q, 2.52237535488279323839235752907864107e+243Q, 1.25874430868722260548214869831621228e+259Q, 6.46403890833514736730418683378673138e+275Q, 3.97014109807165189460577113122885913e+293Q, 3.42256445909004961984480518463414679e+312Q, 4.91057220648194827953651882481242793e+332Q, 1.40575745439889948649867992022706415e+354Q, 9.73933590848593290516589462953654487e+376Q, 2.00557619166308496177559022371882184e+401Q, 1.52772859099735075822031580257311058e+427Q, 5.43355654783039835976501066730115367e+454Q, 1.15613726424969615026074606149058824e+484Q, 1.91611396577413915548297095470351353e+515Q, 3.27574835998524173295312535664414153e+548Q, 7.78995993645646383749425447261756829e+583Q, 3.54265432942533120234904548837850071e+621Q, 4.32354359483087940297488323505450874e+661Q, 2.03098786518797659659581103998046513e+704Q, 5.39103916237599207670086450736143491e+749Q, 1.21683151125684538902217928380735342e+798Q, 3.60847574304211696962789044183995856e+849Q, 2.23398800758266399217702784712099052e+904Q, 4.72717380839903008268507408818545095e+962Q, 5.77822208457179232925131147188581785e+1024Q, 7.13289562956799416571309267253574621e+1090Q, 1.61164858402955200792510210236524128e+1161Q, 1.25521313739368945437360950048953572e+1236Q, 6.61069623781251502357557332678377193e+1315Q, 4.82362056335119095907952363185253207e+1400Q, 1.04640496911016000942284610597050358e+1491Q, 1.52131212817898241350510416765362487e+1587Q, 3.52116090648535956154142297718505774e+1689Q, 3.25907088535366768900129651211398670e+1798Q, 3.21536840347133363006114569121938775e+1914Q, 9.60165908424805663006527683476758652e+2037Q, 2.63585872900260288034767508588870333e+2169Q, 2.17048793513302111330851937361254536e+2309Q, 1.88788443108195661060841368444711468e+2458Q, },
      { 5.29693440729788041213142409892436291e-2459Q, 2.27629257218329628557497350447858260e-2383Q, 4.60725896612142987241524079465629916e-2310Q, 5.17919364499500847985920034617781519e-2239Q, 3.79383002964804382517930061580447157e-2170Q, 2.11419993680434447426181517759297327e-2103Q, 1.04148667561061807899012277964691785e-2038Q, 5.24544425651958823988931435014729457e-1976Q, 3.11006352777614281598465337503705373e-1915Q, 2.48867462987012194623347147546568943e-1856Q, 3.06835915872226280445223729961288427e-1799Q, 6.62736493446143254705278263772087543e-1744Q, 2.83997245953224047644578590378179268e-1690Q, 2.72398780252406063855606534842103674e-1638Q, 6.57327304158814917854156231336457877e-1588Q, 4.46940189941144606938097964305720201e-1539Q, 9.55652954181188776027715447963783638e-1492Q, 7.14757266175076046071660887379600587e-1446Q, 2.07313155515958333906901076011055929e-1401Q, 2.57708633795358827890276221215294553e-1358Q, 1.51269996990649935807792458579987758e-1316Q, 4.60565435951356389119181468428252427e-1276Q, 7.96677448800758121662225971905999717e-1237Q, 8.55160853918795880988712436880140365e-1199Q, 6.20482659749393290517542190337401836e-1162Q, 3.30618935612498022733666989541183557e-1126Q, 1.40195518332652973622801089378870590e-1091Q, 5.11407919863367919558746085806786902e-1058Q, 1.73063614614270605229830917360912415e-1025Q, 5.84546019976429168954160400076820148e-994Q, 2.11542888104356332114006186983662541e-963Q, 8.78597020474401601699260325757052372e-934Q, 4.47629976797445775500570938357786257e-905Q, 2.98416839309444731431172458639558207e-877Q, 2.77125321384854968458662765004085943e-850Q, 3.80904190342493701536240175462301111e-824Q, 8.21806462726393678491860107910281889e-799Q, 2.94631935053803405188027155144659836e-774Q, 1.85492994927395428856242032790235018e-750Q, 2.16349445690590806097153640474491227e-727Q, 4.92371233300030441739039250387920817e-705Q, 2.29918059518632617935997695502963251e-683Q, 2.31291758268743951509849171359112934e-662Q, 5.25491742744220732604319355531444736e-642Q, 2.82274223509188430651059911113688649e-622Q, 3.74753227935082714244593814667510926e-603Q, 1.28370364951438380618871579660566498e-584Q, 1.18285412035024318029749091456416413e-566Q, 3.05273754301598829645261576934311186e-549Q, 2.29479910590203595225694928967822608e-532Q, 5.21889625493118727969519850005753886e-516Q, 3.72534255592436984201074827726301397e-500Q, 8.64949198440528351465957047285602117e-485Q, 6.76174414760667924982941341637985863e-470Q, 1.84041518883114740241388396417281333e-455Q, 1.80160472793323877219953443455302799e-441Q, 6.54566528304067135832221898507534820e-428Q, 9.10001699811829840956092860906810074e-415Q, 4.98609828016939856686029944862830983e-402Q, 1.10802830298400645391731986287163178e-389Q, 1.02676405187821370308474571067658508e-377Q, 4.07575091116228204031238856923140440e-366Q, 7.11360268352693191010849541271753635e-355Q, 5.59881168833151588006581688986986730e-344Q, 2.03642255515559150564975951817039221e-333Q, 3.50525657096789033370540964985470150e-323Q, 2.92178573100086478331989127444273654e-313Q, 1.20598734041934315977832234578459915e-303Q, 2.51880216671823764300316483881856209e-294Q, 2.71834421831894383133482530254389304e-285Q, 1.54702039109098756507051056035620614e-276Q, 4.73499485141563990358940703275480353e-268Q, 7.94442519500188390796900278354012024e-260Q, 7.44316376023327816606172273381802393e-252Q, 3.96451700998508534828226048217863900e-244Q, 1.22153684948784264379851755556819940e-236Q, 2.21421725154084964507841507830563690e-229Q, 2.40004857648572823842789487119416272e-222Q, 1.58043505420313498164518700461984906e-215Q, 6.42021223167302856767365812271085781e-209Q, 1.63302423264111059580848381300534588e-202Q, 2.63853106939137106330498547542524566e-196Q, 2.74613250302397818664806479526069745e-190Q, 1.86615306707995508392565990157579707e-184Q, 8.38948945233194510175234920774873796e-179Q, 2.52701050376399124391991347402621144e-173Q, 5.16312575535157598596140018726440984e-168Q, 7.24167062135448326922272784801743108e-163Q, 7.05361814094865509813912299190351845e-158Q, 4.82507740170943565549056875354725391e-153Q, 2.34335421855805662819938218222415100e-148Q, 8.16563163629951985655858859925168146e-144Q, 2.06250908768935143871108361256064164e-139Q, 3.81378121105029755974979453855090030e-135Q, 5.21238862833226048821040688112783913e-131Q, 5.31469066325781546457465599778122553e-127Q, 4.07938032086884338695136986806703742e-123Q, 2.37782434978024084372113156401306143e-119Q, 1.06148128500673821430527965352560854e-115Q, 3.65894930935314933069457000324241470e-112Q, 9.81672760779301769095882197030988958e-109Q, 2.06581729538829312212171371616525011e-105Q, 3.43540071960972258090881112115251093e-102Q, 4.54745983632894201410075262868047927e-99Q, 4.82519857468149557373657687371350595e-96Q, 4.13210577018135888633811621378125178e-93Q, 2.87476099508953335771741680746804464e-90Q, 1.63524251388290882625725240042223140e-87Q, 7.65249997733887999623874860707951771e-85Q, 2.96396516998940431058211305683891581e-82Q, 9.55694449812711903241890069159516005e-80Q, 2.57983003461536294556125432978470487e-77Q, 5.86224102303822793749090237239965643e-75Q, 1.12729666204663539107360352976065345e-72Q, 1.84393400012961666333357387323541892e-70Q, 2.57838118497774645351182154219344361e-68Q, 3.09698398084623291081115732152399867e-66Q, 3.21034639994569504125984658412826677e-64Q, 2.88505745240234032993466930110171709e-62Q, 2.25763973385675919792268556682148533e-60Q, 1.54490468182644383657455478866541800e-58Q, 9.28299236822216106223683331290050646e-57Q, 4.91757270567151153434520348832876307e-55Q, 2.30554641627582419897786323384840309e-53Q, 9.60260856639165286607071389462405742e-52Q, 3.56597785391661967741537394944785975e-50Q, 1.18488237523747100855696221290216872e-48Q, 3.53480106239996687790067585870262148e-47Q, 9.49922331635571479300328969906674959e-46Q, 2.30696251922047363681394459290289348e-44Q, 5.07896585888252846067820035747838002e-43Q, 1.01672554003146516218461940856820913e-41Q, 1.85608083837358412341292426322275694e-40Q, 3.09875103851653509802894724036653659e-39Q, 4.74424556091727158516159565756659137e-38Q, 6.67875654292885708009556323027970307e-37Q, 8.66749789110265824007798499627002101e-36Q, 1.03955806472296089132115323650013375e-34Q, 1.15508617865206361238561969897526423e-33Q, 1.19182362291753977427429758189318101e-32Q, 1.14454132981883615336894355014782203e-31Q, 1.02525602738123519983996836134381910e-30Q, 8.58508308406581287405862446386142355e-30Q, 6.73394164670453777712343530741256529e-29Q, 4.95770293303240892202540668612526963e-28Q, 3.43261200056988540268170912603591182e-27Q, 2.23935379461627788150957730520597272e-26Q, 1.37901808820501650857276186986911719e-25Q, 8.03040544770876549232435915566049266e-25Q, 4.42968188137908996064235843650469764e-24Q, 2.31845927113168436195559586507393831e-23Q, 1.15324161925721516540799043047625888e-22Q, 5.46028729667908667672130073109459001e-22Q, 2.46458926739523684644781998993375925e-21Q, 1.06205430707170637462172218092564615e-20Q, 4.37564008882607318443071720026076737e-20Q, 1.72595587803323990868164488937090111e-19Q, 6.52669144636334492287282783537910032e-19Q, 2.36916844627434713681577417830597819e-18Q, 8.26580372697482904251900693228762290e-18Q, 2.77517606391661360182153963773908766e-17Q, 8.97689232444592868369906653940499583e-17Q, 2.80084735231662190309995026747244972e-16Q, 8.43846463133099160645680685207593959e-16Q, 2.45762595435789224473736552077118113e-15Q, 6.92627769518345222522404766069340383e-15Q, 1.89084205236464652796008005489924701e-14Q, 5.00504297304156635993817971340426042e-14Q, 1.28579120925883494201961228234875929e-13Q, 3.20880531681575127155908954236910927e-13Q, 7.78600100470787821926138878185058803e-13Q, 1.83848331026111978204532135720140678e-12Q, 4.22808302441074119355483439549639212e-12Q, 9.47805306883598889922905503268261293e-12Q, 2.07266429754356748908920713182886782e-11Q, 4.42491437115776228499841800855357065e-11Q, 9.22929507351999755945043065016679100e-11Q, 1.88205258669115539986830146476840153e-10Q, 3.75488615259231157512941964482754887e-10Q, 7.33426418139309264953064008778652175e-10Q, 1.40344387177481383433616336399570262e-09Q, 2.63261606277390900911399983420396976e-09Q, 4.84396275737187249536387241410506136e-09Q, 8.74769197387686182490212973737486333e-09Q, 1.55137319662316143322632050708021849e-08Q, 2.70341183770391766500462033783635805e-08Q, 4.63144836233962351361136455522914469e-08Q, 7.80474058793106802102732603067941163e-08Q, 1.29437017686516812026215434301661584e-07Q, 2.11364219596533096486660492476571897e-07Q, 3.40005066401716435610198303924157253e-07Q, 5.39041104673862946543865026660608261e-07Q, 8.42629030758144765430007132771995573e-07Q, 1.29932702981307401261989221953085056e-06Q, 1.97720517756199603296504985814152326e-06Q, 2.97039557774168150435589567759708661e-06Q, 4.40736236333866783037595222091818751e-06Q, 6.46118993513603067322889724548764197e-06Q, 9.36219732537340456332983881649283088e-06Q, 1.34131848415120895981721422476655149e-05Q, 1.90076038344927799241882635296823326e-05Q, 2.66509095957072382671851701769041584e-05Q, 3.69853096371186063639032612868033323e-05Q, 5.08180543166657948433862427398171792e-05Q, 6.91533341923576665290801314459185869e-05Q, 9.32277498618928819410507284882673596e-05Q, 1.24549207625185292746788896109936768e-04Q, 1.64938971333376144939513069096604597e-04Q, 2.16576471380809909291159455336007140e-04Q, 2.82046340794006881311592802174739018e-04Q, 3.64387021107897729238216699262593180e-04Q, 4.67140162762043149791404714119969181e-04Q, 5.94399941612237251611663813916878716e-04Q, 7.50861330003505141341237132619730559e-04Q, 9.41866302231455859084164368912569515e-04Q, 1.17344692380002247672609547026883890e-03Q, 1.45236427426188008347616119045966424e-03Q, 1.78614218598655178612808460803997991e-03Q, 2.18309484603519656221136749899809853e-03Q, 2.65234740423109052289144172110269167e-03Q, 3.20384885506921527761765779516587618e-03Q, 3.84837661076576821141977260794172255e-03Q, 4.59753235303186248996039969885789422e-03Q, 5.46372893686621665171770797799402382e-03Q, 6.46016831511747979221062326266402012e-03Q, 7.60081065185419977059145624254178297e-03Q, 8.90033498980204155946674089749748970e-03Q, 1.03740920266168385556590451013464350e-02Q, 1.20380497313706427480839156336456773e-02Q, 1.39087326917827169960002061321148754e-02Q, 1.60031562206455496475142539245814110e-02Q, 1.83387563636593926316708603595489832e-02Q, 2.09333170384958330449215168774593401e-02Q, 2.38048955952869498234626954784270233e-02Q, 2.69717481217077174816927863942958635e-02Q, 3.04522558220532394622866431653207434e-02Q, 3.42648537806332947261888782628446295e-02Q, 3.84279633774899765386527022493279515e-02Q, 4.29599295614980634381395647366200859e-02Q, 4.78789641053477195533528647962166054e-02Q, 5.32030958720316323100756360890520466e-02Q, 5.89501290167188399227102430485391475e-02Q, 6.51376099347951026087494855511522993e-02Q, 7.17828036498272120091084808533476670e-02Q, 7.89026802175633783409928711856568346e-02Q, 8.65139116068936794825248092013403649e-02Q, 9.46328794087702664882628072944679318e-02Q, 1.03275693621920814402887268719345915e-01Q, 1.12458222671938515285380933743314032e-01Q, 1.22196134739810167085016527168685084e-01Q, 1.32504950408621397326604347561435549e-01Q, 1.43400106584199068783023171408440267e-01Q, 1.54897031607657925984050586220533856e-01Q, 1.67011231455784555526366713839660189e-01Q, 1.79758386919258486031495958113778865e-01Q, 1.93154461659030684603221234896514194e-01Q, 2.07215821067763214460783254641335718e-01Q, 2.21959361905993070139869913079334838e-01Q, 2.37402652741481501585147220599178905e-01Q, 2.53564085294908579555677649044302540e-01Q, 2.70463036885576732351902861856168718e-01Q, 2.88120044277091724111603906974976099e-01Q, 3.06556989345224713687672375979413066e-01Q, 3.25797297128632613090751563726807919e-01Q, 3.45866146978355838846531059919780118e-01Q, 3.66790697694818431455168590767201925e-01Q, 3.88600327732532063221423261459050385e-01Q, 4.11326891764317591973280272152569947e-01Q, 4.35004995130479531888748830535836274e-01Q, 4.59672287956338821101823148522366487e-01Q, 4.85369781006713270719035870359289569e-01Q, 5.12142185661796519744086646686900168e-01Q, 5.40038280749567858894795821479236183e-01Q, 5.69111309360283620800291297165821293e-01Q, 5.99419409204557829271245298375247710e-01Q, 6.31026080564853461281465428141943759e-01Q, 6.64000696438865091754636398259527208e-01Q, 6.98419060091622837940058120097784455e-01Q, 7.34364015932103716685673325408458579e-01Q, 7.71926120422450486565697123079450111e-01Q, 8.11204380628463812996672518165491420e-01Q, 8.52307069046258371120734338452652256e-01Q, 8.95352624512219417238820469064741936e-01Q, 9.40470650345508781665473144971561924e-01Q, 9.87803022412309344664090567700404412e-01Q, 1.03750512157160024916963789173178052e+00Q, 1.08974720700214151578746902594297743e+00Q, 1.14471594926501601899574650013187236e+00Q, 1.20261614467922655880690394062840816e+00Q, 1.26367263574296191533610775626416078e+00Q, 1.32813246599542422597073187153380823e+00Q, 1.39626730197278306832678497202104711e+00Q, 1.46837615987297935465145703905241109e+00Q, 1.54478848033412089590682530315276939e+00Q, 1.62586760150092827681818702612923363e+00Q, 1.71201468848349507792697792068680378e+00Q, 1.80367318661869118602748089155898811e+00Q, 1.90133387688644143294074965806611335e+00Q, 2.00554062472320920559824403445004936e+00Q, 2.11689692868996327728244730212683488e+00Q, 2.23607339344688170923456651751668676e+00Q, 2.36381627281331763524062140552607385e+00Q, 2.50095725401825500385431630504625834e+00Q, 2.64842468438767047986902883861439562e+00Q, 2.80725647766353485711865991725063525e+00Q, 2.97861498011790290399988989454287074e+00Q, 3.16380412810114748721127771257562134e+00Q, 3.36428929047159656773947338449998169e+00Q, 3.58172026374255002938249218600614501e+00Q, 3.81795797752670936449825238124466741e+00Q, 4.07510557639156630320887778958143299e+00Q, 4.35554467582358554514458856445852610e+00Q, 4.66197774993613776113418481343338776e+00Q, 4.99747780346124563910807301981274948e+00Q, 5.36554671871496309147449064380266706e+00Q, 5.77018396000583698718320188114380523e+00Q, 6.21596767643453604333341399022283971e+00Q, 6.70815068570623654464776592817234118e+00Q, 7.25277436733040258314736650500251589e+00Q, 7.85680416993879891697239876362854677e+00Q, 8.52829127820429133135398118660851136e+00Q, 9.27656603318338653224922387509018222e+00Q, 1.01124700112272039114486606209390216e+01Q, 1.10486353121876175190769261708382627e+01Q, 1.20998216795271857764508697373482606e+01Q, 1.32833246923912553882423938260725321e+01Q, 1.46194715878299420711013972809323962e+01Q, 1.61322254926408962729986594994585730e+01Q, 1.78499242340404104179578536110600421e+01Q, 1.98061868055245863949655128520558624e+01Q, 2.20410294496835217815852679074305586e+01Q, 2.46022447943980585927583897681190540e+01Q, 2.75471123562893237374077750154320094e+01Q, 3.09445280931970284883990775946651869e+01Q, 3.48776660064165064038939228627010010e+01Q, 3.94473180334351770794475603595764904e+01Q, 4.47761023021425157569087608052893710e+01Q, 5.10137878711900622533650986521384164e+01Q, 5.83440613273984383402755554599355811e+01Q, 6.69931638788863998843945524712655391e+01Q, 7.72409663039404221632393248526916570e+01Q, 8.94352363841359052286058140969269159e+01Q, 1.04010107537438719134080196856970415e+02Q, 1.21510103906665265646188258350644787e+02Q, 1.42621552310160172951445448156836659e+02Q, 1.68213866518508832477273582457732136e+02Q, 1.99394097464546647941096853182355363e+02Q, 2.37579409247584470819614691411159418e+02Q, 2.84593916789823535562840880888980902e+02Q, 3.42798827028127058717471147784029643e+02Q, 4.15268383629255114718046929646589502e+02Q, 5.06029199301683119404601716655559045e+02Q, 6.20387871848170976858394304975149188e+02Q, 7.65382367194376728088096731506729391e+02Q, 9.50408087358179153450911943224019430e+02Q, 1.18809220276011606595956628737466824e+03Q, 1.49552334212407885347227350564427644e+03Q, 1.89599366703067034325550411232806679e+03Q, 2.42148532800683663374818194050951580e+03Q, 3.11624674527423644655997407153581040e+03Q, 4.04197721822739650017152639557046462e+03Q, 5.28540456882764394198106919117899344e+03Q, 6.96945349745478520211739265737692803e+03Q, 9.26984863597541610825591719881066842e+03Q, 1.24400169046144284609469034917486102e+04Q, 1.68487804928219120974770366506141126e+04Q, 2.30379493050689209894874317068326889e+04Q, 3.18111749406368329678640732920395152e+04Q, 4.43724092704038524997055436198884382e+04Q, 6.25438880548229936896462817079707202e+04Q, 8.91129656174671765721819893640615727e+04Q, 1.28390011615567130356349325241374642e+05Q, 1.87115939884978799395430373139032810e+05Q, 2.75955654445572113231216194578725044e+05Q, 4.11985149226574332956827494211875921e+05Q, 6.22884543671150616939347176756608244e+05Q, 9.54097172994412639791727921401626133e+05Q, 1.48121323807100834515424707068495244e+06Q, 2.33168052188078970586379639354686916e+06Q, 3.72339779803011251388319470323527721e+06Q, 6.03430539101169547235371451057106735e+06Q, 9.92972861117960142404055815146657353e+06Q, 1.65989636945226644757161242143615839e+07Q, 2.82017465494921172861525124733558875e+07Q, 4.87244883934161305263386031770533647e+07Q, 8.56498776477185184074970634710495674e+07Q, 1.53268758654909039217478083777958109e+08Q, 2.79366798395238972059209786152271040e+08Q, 5.18973079293501172198046128702649663e+08Q, 9.83165082634482695230583899233702397e+08Q, 1.90059962104050828804295224765625411e+09Q, 3.75160395202091966289927262175925315e+09Q, 7.56648043123273181762292360200563868e+09Q, 1.56033907397856950198380769768232419e+10Q, 3.29229832278164384876778685138061688e+10Q, 7.11297379086385418813540334382875157e+10Q, 1.57471442166507563544871664087516645e+11Q, 3.57509888501672692207543566803978301e+11Q, 8.33024430623979589172560805550927471e+11Q, 1.99374509951525518357032637324282993e+12Q, 4.90561996981418757143934103035069327e+12Q, 1.24197379810188498181468317797399218e+13Q, 3.23831600275722270184945604347174663e+13Q, 8.70403769580874957196023586424815933e+13Q, 2.41399528145469907607734183398259453e+14Q, 6.91510620574880583944843529198735451e+14Q, 2.04811558742607734304993874133785299e+15Q, 6.27861397733698939186730156018167292e+15Q, 1.99435267076689206564427715465056511e+16Q, 6.57141971664513108413598657255436192e+16Q, 2.24875056642273914367880154275296162e+17Q, 8.00147401578245998412996372425512095e+17Q, 2.96403754199235340052960592736049159e+18Q, 1.14455803313869409862823561648962815e+19Q, 4.61323311996821344548899222001480398e+19Q, 1.94346926949906856258436939614621369e+20Q, 8.56968050834200116120462221511753399e+20Q, 3.96091566953282355348486677100761553e+21Q, 1.92185171194284479900050386014427105e+22Q, 9.80409181939054057802535035944011717e+22Q, 5.26682924609986175842861032563925229e+23Q, 2.98441055802829784200443452444088495e+24Q, 1.78677995299228897649048545726173530e+25Q, 1.13225906725879734647293567479653181e+26Q, 7.60791970573697649080495794599946588e+26Q, 5.43049485025884671542842347146701195e+27Q, 4.12572142434645000681420526588413556e+28Q, 3.34277767339287328803586493191472415e+29Q, 2.89434014229221431295360965172883804e+30Q, 2.68374761249850267614464943620131017e+31Q, 2.67072026965642827232315722365639115e+32Q, 2.85880373230063837155252556137517211e+33Q, 3.29924822913520515079335032162581841e+34Q, 4.11488570832552270061078474206882038e+35Q, 5.56010558358231010262223361328066200e+36Q, 8.16009266847150844693389287229450581e+37Q, 1.30416726659952301962536120869903033e+39Q, 2.27600481686142160023333409481863034e+40Q, 4.34938214638271735275350443120721790e+41Q, 9.12741493518023346499222172162825387e+42Q, 2.10972038777434150864210644015160107e+44Q, 5.38754491759583324556444810556121661e+45Q, 1.52482535270240332398284822085798536e+47Q, 4.79877533166358652838909903087275144e+48Q, 1.68494126510508458862564987284302025e+50Q, 6.62357573295543230261174696442002813e+51Q, 2.92557273755841342556692584369563440e+53Q, 1.45729219902900863678006319195819802e+55Q, 8.21783496105748128055143975755446008e+56Q, 5.26688130483523933792268242131450300e+58Q, 3.85211799189653678380768112270880793e+60Q, 3.22861432024885393800430464301385615e+62Q, 3.11445231039438406261213455773274485e+64Q, 3.47323481265477220985448691064609690e+66Q, 4.49855546587324575105781665091655708e+68Q, 6.79925746409737423822172568281133860e+70Q, 1.20511321523280079626937861818322431e+73Q, 2.51749264598597737693408358183917414e+75Q, 6.23086472714522132176542446806460931e+77Q, 1.83698604157213615145607828551975305e+80Q, 6.48713124894846526923286444858655243e+82Q, 2.75979764628924062898267004292796591e+85Q, 1.42281010916783424894495664030511266e+88Q, 8.94356583170635560730657498907040019e+90Q, 6.89765608918172471716339680473639534e+93Q, 6.56960382950241207669696237814704407e+96Q, 7.77916346275648519460667580565807164e+99Q, 1.15314598687748380372708686918514146e+103Q, 2.15521325185955507170865064451154694e+106Q, 5.11618164822064799451529384134754077e+109Q, 1.55434716015270528126967279870385072e+113Q, 6.09112777117402790869661219995564436e+116Q, 3.10387507242519227183111245931080882e+120Q, 2.07390189233940742293442790510784430e+124Q, 1.83267382155701863417862737648022763e+128Q, 2.16097258672364775058232219739796364e+132Q, 3.43128595186527837599814654188046354e+136Q, 7.40646244666625583794129098263179729e+140Q, 2.19454208154239352969892199569722442e+145Q, 9.01619836979155465535468945047889426e+149Q, 5.18985678737597694246477524117862598e+154Q, 4.23046985030263912409378392183252421e+159Q, 4.93764276985086773495452256391816235e+164Q, 8.34636422063293684495534346523864357e+169Q, 2.06741406169751668845974997770098232e+175Q, 7.59587841419400960682004001784651511e+180Q, 4.19164127392101407920200382533221559e+186Q, 3.51927711650483149002609005453729140e+192Q, 4.55587470619707394753414927965798200e+198Q, 9.21952142596023196043927688149872532e+204Q, 2.95816314360706369484966684111893238e+211Q, 1.52709952631709289803811134371908265e+218Q, 1.28765668330017907582554622529315931e+225Q, 1.80129361672056777447357282869624658e+232Q, 4.24814651295030396772649227180907798e+239Q, 1.71729703963558073111114105671039088e+247Q, 1.21046234446012300929891454835993254e+255Q, 1.51419519516367160890801105849993081e+263Q, 3.42330233588160029564511697140750743e+271Q, 1.42528129667493838632765543616342364e+280Q, 1.11420723637564670686340338390175118e+289Q, 1.66850212135837193821657370527462903e+298Q, 4.88589347415031911318629107633374545e+307Q, 2.85800617499362161237161192858659000e+317Q, 3.41368346308455763011375953918857639e+327Q, 8.51659564264685587821018369835222946e+337Q, 4.54303089474615832686354150019925596e+348Q, 5.30809578740740306246018975605923943e+359Q, 1.39268945492416740550127311639881773e+371Q, 8.41868946331338761333876333129653244e+382Q, 1.20396970846759916705111332553270569e+395Q, 4.18638285711073253556696402015828229e+407Q, 3.64051583443686661335653674614037576e+420Q, 8.15125707810912875910267388062325131e+433Q, 4.84241684726444145079336201568928602e+447Q, 7.87277386139003022204421580245435912e+461Q, 3.61660042166407854517352456605683269e+476Q, 4.85176905670774504900384645174873943e+491Q, 1.96653101796028683463909850095858046e+507Q, 2.49428344091357019569533852688763620e+523Q, 1.02650775155921036180043655136824490e+540Q, 1.42290670725613581153914997322741401e+557Q, 6.90446128383757111350399979478971290e+574Q, 1.22038031602184356762554212370154951e+593Q, 8.18641498043102997149775477320992897e+611Q, 2.17426427576810603662782427977182733e+631Q, 2.38847898517251831173924452396821302e+651Q, 1.13525581232092079165478024625033648e+672Q, 2.44581305055310563196277332943189236e+693Q, 2.50579201726264634501291514398286907e+715Q, 1.28278958026597426545698815500879845e+738Q, 3.45329144611998202962794722592107008e+761Q, 5.15301310117757693898843758843439768e+785Q, 4.50037434223155300664502702852727419e+810Q, 2.43306674872992411217032796443919144e+836Q, 8.62798204693025064663883932656343701e+862Q, 2.13031523407691556211779361127399737e+890Q, 3.89502853428849118703235846476912848e+918Q, 5.61967843038535141183624496528991236e+947Q, 6.83163157235897275355324461890504612e+977Q, 7.48741060910847824069581635250382582e+1008Q, 7.93319051954128205327740081284490051e+1040Q, 8.73273358333231337767980404927541807e+1073Q, 1.07574303828662898687276828597901159e+1108Q, 1.60109519481842931883372467310550530e+1143Q, 3.11621335415749303163841998134612826e+1179Q, 8.60558107743722927550014656572137529e+1216Q, 3.66811383278356514418007625117425117e+1255Q, 2.63234220176367112942815907184016908e+1295Q, 3.47859421502287448304146808910623408e+1336Q, 9.28512235755776936781681374078562947e+1378Q, 5.50718056653300293052467960576840696e+1422Q, 8.00902248373521173018720630861185040e+1467Q, 3.16115623922106492070773163357582495e+1514Q, 3.76042153408120168432167355515647590e+1562Q, 1.50211667261404150161306978400073837e+1612Q, 2.25263196634448337432840054986261153e+1663Q, 1.42290818050514650710172541634086053e+1716Q, 4.26315122517229508281635793163851999e+1770Q, 6.84787649140078417523090257285629098e+1826Q, 6.69183748343427005004387693124220507e+1884Q, 4.53246642529433183363157301179103704e+1944Q, 2.43420309539903059914673692609144617e+2006Q, 1.19096818482919681799422553000433899e+2070Q, 6.12585236774593416353267428912127386e+2135Q, 3.84000941993213246405245986391372004e+2203Q, 3.41673273574541565248480628561887880e+2273Q, 5.05031055516767284204610907378751985e+2345Q, 1.45860427844568831785920214989842347e+2420Q, },
      { 2.77668375956040189946826811680384390e-2478Q, 7.15287685954868211386033128907477031e-2440Q, 4.67658088132803463412188029644470498e-2402Q, 7.92687480429971819654882242916691373e-2365Q, 3.55705626445534513670485482702241051e-2328Q, 4.31363186281767249198550472192046640e-2292Q, 1.44267570824010102564476209762038016e-2256Q, 1.35750289529472083929076668071298865e-2221Q, 3.66520151899825097802289209374933991e-2187Q, 2.89498463671553268274047707590333923e-2153Q, 6.81807725541668247816231315716744273e-2120Q, 4.87856900336985600073857171641568070e-2087Q, 1.08033682307899983904972807875490412e-2054Q, 7.53978658799448170922270322982561647e-2023Q, 1.68836600150778453318616795534542245e-1991Q, 1.23462709610862382392789185979644933e-1960Q, 2.99986697918589103634446685042416531e-1930Q, 2.46367649797140140439024370548377672e-1900Q, 6.95479514251903643444688832919306991e-1871Q, 6.86110776725256744693425828768272774e-1842Q, 2.40431940102692787285034949159539178e-1813Q, 3.04121511300162701514429163779115709e-1785Q, 1.41065291642406553832784938505501922e-1757Q, 2.43705806703985396362343530158481976e-1730Q, 1.59233397078886467474093245165420794e-1703Q, 3.99458646512746569553991304016076066e-1677Q, 3.90502825544924862516415029425091730e-1651Q, 1.50951212222311306420066585908781007e-1625Q, 2.34075577961021270449149847209864885e-1600Q, 1.47683791355252947630371426518876483e-1575Q, 3.84433827968748974568829430827411882e-1551Q, 4.18584678137447017460210584041562890e-1527Q, 1.93235335361456340539919561054923147e-1503Q, 3.83273493158460886555652388105220442e-1480Q, 3.30931424971841824136873930746170663e-1457Q, 1.26001001743526741988366704934436124e-1434Q, 2.14254382108136960073063000869254434e-1412Q, 1.64752740569760866428499059885288678e-1390Q, 5.79997534765464443335232132612089697e-1369Q, 9.46173366850636035136459393339314051e-1348Q, 7.23844554700889711120024618879438753e-1327Q, 2.62753543705293236920590575888507321e-1306Q, 4.57825670343966465596419576070363442e-1286Q, 3.87294895920776670416027972176834657e-1266Q, 1.60856514992623900565886618371043785e-1246Q, 3.31650403849378163469754348545840043e-1227Q, 3.43149117215084179247503681795855710e-1208Q, 1.80089379614922388608625914511562841e-1189Q, 4.84469660937728881846012977552474102e-1171Q, 6.75021934282781912062862469597524256e-1153Q, 4.92120879530819746165015110941962755e-1135Q, 1.89622303139680815547453480860411409e-1117Q, 3.89999690124553358779083557357143651e-1100Q, 4.32337852652349730363515343553956403e-1083Q, 2.60812148992852188438485836891607672e-1066Q, 8.64320507307309471921069873497971138e-1050Q, 1.58817234955465364327662145482119776e-1033Q, 1.63292822332387428570727069490856970e-1017Q, 9.47965403932768371077126663738787002e-1002Q, 3.13489123036525936535865202633924346e-986Q, 5.95725324626422679969355089398557051e-971Q, 6.56135449619885064132736333035718110e-956Q, 4.22412947622465431737740854731813869e-941Q, 1.60284936502012533605395043475391797e-926Q, 3.61426009581007446214636991266185477e-912Q, 4.88226731291038533672727194690523707e-898Q, 3.98243859045531426665602764668150557e-884Q, 1.97696254579881780215274246825324592e-870Q, 6.01884829406232602469720991349044932e-857Q, 1.13236358528056083157892195864287071e-843Q, 1.32634602112101849951006383738692278e-830Q, 9.74354258615429475261032804502093079e-818Q, 4.52175713954583830576111594951110560e-805Q, 1.33512020097306090853791018618669252e-792Q, 2.52581092493362086952359277189983609e-780Q, 3.08280454280630114861769351743791880e-768Q, 2.44403095665612216421741893390173529e-756Q, 1.26703351146649868905932200644776032e-744Q, 4.32364746058654184636712625519962546e-733Q, 9.77482453681545274501227529587168230e-722Q, 1.47345296987795364063826018729590588e-710Q, 1.49025749519107052339924056423862408e-699Q, 1.01758543742056270682348949473524139e-688Q, 4.71965368684349520914049336410150458e-678Q, 1.49583331955342167906773277357310899e-667Q, 3.25877112957152479296585133409409173e-657Q, 4.90846979890405909332298369271733953e-647Q, 5.14096775171769223933971053266547990e-637Q, 3.76527319345060607553853872315139069e-627Q, 1.93914103642113548742773247369930873e-617Q, 7.06081267960685321695216552061682039e-608Q, 1.82753922477698812695612565052211519e-598Q, 3.38020931062714080731736679298776434e-589Q, 4.49104974727379533831457784467988803e-580Q, 4.30831017202845455490779426657646708e-571Q, 2.99925774499663529105464847801180013e-562Q, 1.52274525111081863903256031902733405e-553Q, 5.66595073105051388454002533447155475e-545Q, 1.55254034318111114885985765214004285e-536Q, 3.14773091677700441775450360300498865e-528Q, 4.74422214535871848463366092395468719e-520Q, 5.34000677409012855851843363768904621e-512Q, 4.50913961542719498094454101135511353e-504Q, 2.86916730093441295609899413107663763e-496Q, 1.38176315596895654455580263526098670e-488Q, 5.05827668288972660188294275444103716e-481Q, 1.41354423391013787396132813520577665e-473Q, 3.02811286218415761503816814918528786e-466Q, 4.99321587478833584968780076440597278e-459Q, 6.36350378784780643105814130227578114e-452Q, 6.29295581774659679570380847201393083e-445Q, 4.84800776366962427166614895670077970e-438Q, 2.92081155862662548270928062619807743e-431Q, 1.38143568676341254642463548681208773e-424Q, 5.14842432351503290408078410534655737e-418Q, 1.51753560889290027005627901927469983e-411Q, 3.55061704489050704368494163406172279e-405Q, 6.61796042955880547227964893320004528e-399Q, 9.86122308014976296151434619809887336e-393Q, 1.17877481135241077171765329651013943e-386Q, 1.13424833803992349974759003704486622e-380Q, 8.81502377306393355822212968439637791e-375Q, 5.55155950590484925963715579436840823e-369Q, 2.84248729913659406224316552722434334e-363Q, 1.18704719186027282879490304441234572e-357Q, 4.05597847694543335354139081149132852e-352Q, 1.13745052103435347539610354499759843e-346Q, 2.62608432267715211136705119144544958e-341Q, 5.00649221900861773230008659070798254e-336Q, 7.90489841299351390278476450028383484e-331Q, 1.03673095771954987567298480275353471e-325Q, 1.13264118291164663653569719276217293e-320Q, 1.03372554114991613828759256440083793e-315Q, 7.90343344978986920403039115230438646e-311Q, 5.07593758357543835739913968923721419e-306Q, 2.74587297706393183600196990448758755e-301Q, 1.25448043297991919976296583184269934e-296Q, 4.85293287864597257977432400607091472e-292Q, 1.59375532748424261820313240836566102e-287Q, 4.45470192712163297279675529941780532e-283Q, 1.06238236699788796015140463431038408e-278Q, 2.16709025689586849941731617555370390e-274Q, 3.79017494148388392555292862630176094e-270Q, 5.69722642026139785964263462216321432e-266Q, 7.37750423818818484376423187717037465e-262Q, 8.24899816234186348122208308733670706e-258Q, 7.98228295613681065404000331301407417e-254Q, 6.69977627078532550007479609463315785e-250Q, 4.88831026337525670264792775582398754e-246Q, 3.10717465030331182564914656973440633e-242Q, 1.72428695582194286916259157368370433e-238Q, 8.37152419508690266695542791416172637e-235Q, 3.56328545617913642798317603893755984e-231Q, 1.33239745308069123170741398284986139e-227Q, 4.38557307471736323357712547586541243e-224Q, 1.27317190092247560042567075437905721e-220Q, 3.26633944706575228492861971563886982e-217Q, 7.41961719235926788479176390701225966e-214Q, 1.49508892701036107375575902806187267e-210Q, 2.67747004659594255579164687433346539e-207Q, 4.26921971680475266131733781176106392e-204Q, 6.07184376271449065521795822572065912e-201Q, 7.71632746256793762696505755410742573e-198Q, 8.77761527570739954686084893442788334e-195Q, 8.95291300378341203683641755088969233e-192Q, 8.20179011370862828929882403579221646e-189Q, 6.75977978662441271172788792155811852e-186Q, 5.02051176570061397957717306624573193e-183Q, 3.36554863928813190587746679952307259e-180Q, 2.03960573952427530884860135637723107e-177Q, 1.11917598852847666035205866531172342e-174Q, 5.56905039413988338846998023185739233e-172Q, 2.51682459319287628531543936384778284e-169Q, 1.03457507608575432785228105631849600e-166Q, 3.87388309025221395194171192891040522e-164Q, 1.32322812968423778281527114220160831e-161Q, 4.12900297352082279106292344371557207e-159Q, 1.17865546256954888181796214775388038e-156Q, 3.08218900889320623074522540652758709e-154Q, 7.39354283219941448656677403739458341e-152Q, 1.62910064435532863938332952898952690e-149Q, 3.30154552905982294134609461204752294e-147Q, 6.16203139085424122701291732112343081e-145Q, 1.06052819447098630885771547158633382e-142Q, 1.68522575749723508863020213808775829e-140Q, 2.47553409758226362888052187337950144e-138Q, 3.36576474950758719210258297005317849e-136Q, 4.24056268392402238294912126586145782e-134Q, 4.95679422788561171460024764665358982e-132Q, 5.38171636791416151987580045952962687e-130Q, 5.43350717229498884938353606822409483e-128Q, 5.10703124279431542010303665040364840e-126Q, 4.47370493209864639350813176634174273e-124Q, 3.65637694737788862889473248524269217e-122Q, 2.79117002269425900051588497263773104e-120Q, 1.99220023869241503219258818787712929e-118Q, 1.33089435939378971755299967553769684e-116Q, 8.33035676735989050287962245538521119e-115Q, 4.89025663997024514599039877971897448e-113Q, 2.69512893545116544669129054438916450e-111Q, 1.39582960541563084355183973360376419e-109Q, 6.79999752718808594213315829447961814e-108Q, 3.11903776737903229329061449549842670e-106Q, 1.34826013141921629130407145750369350e-104Q, 5.49752601894399080415833353152482494e-103Q, 2.11638467025119853304622111522504730e-101Q, 7.69914871485806120880436816954565416e-100Q, 2.64906534725059834511552809323827260e-98Q, 8.62818926354972775334178389390317975e-97Q, 2.66252094324836892217545797407595895e-95Q, 7.79069862358288634135952568964046431e-94Q, 2.16335486668307728100344975621110455e-92Q, 5.70557673979722036138381850123515391e-91Q, 1.43033819302856491333403076823625576e-89Q, 3.41104078137232874664267449214467490e-88Q, 7.74426807351644903668383259880558649e-87Q, 1.67513656430343581317253830008450619e-85Q, 3.45479581059570481569225182340141781e-84Q, 6.79857313709947736278888732144820603e-83Q, 1.27747470803378266136497077558029564e-81Q, 2.29370213942630948295519765524670121e-80Q, 3.93802170001517503024981641969166523e-79Q, 6.46959393487630012362523872090477425e-78Q, 1.01772526699091247125279301878345328e-76Q, 1.53401952979332495111639447956817637e-75Q, 2.21699988683886091607827589036641130e-74Q, 3.07410074756280336185727691291524915e-73Q, 4.09229533083754909228682061184080234e-72Q, 5.23343417563653847061479912948387465e-71Q, 6.43350607976335741808585356397544551e-70Q, 7.60704267790136216066341175717356605e-69Q, 8.65671438716342535748325493579430488e-68Q, 9.48674605868548997400346985395655480e-67Q, 1.00175672424828839701939548888991483e-65Q, 1.01985394383485432972080334959114676e-64Q, 1.00159110661066563005593773053264872e-63Q, 9.49427782244426395210838765944495599e-63Q, 8.69142291889189064854075883690819022e-62Q, 7.68797704788744827616631661309525244e-61Q, 6.57440810419660524782417308561229212e-60Q, 5.43816250291842519145575247702512746e-59Q, 4.35334083136300321175659843267140387e-58Q, 3.37433876218124341053665144604385146e-57Q, 2.53377092117304232961472206140287465e-56Q, 1.84404892524861673828316906508664460e-55Q, 1.30141081230848018402542829366785560e-54Q, 8.91046674437447006307015773182167698e-54Q, 5.92153838412413233143345637093941679e-53Q, 3.82135613429770512672309553535220252e-52Q, 2.39578065735303689149259651605329745e-51Q, 1.45988218758182023631230626004517631e-50Q, 8.65010547207677732721948607063784916e-50Q, 4.98593355079719931559980617333925476e-49Q, 2.79691190323743591605512918548961454e-48Q, 1.52757011899350333194123701418929652e-47Q, 8.12631404819699330158401537112269973e-47Q, 4.21243636394857818219784558381591723e-46Q, 2.12860405024256466240923390999711474e-45Q, 1.04893835632343107182205383744844780e-44Q, 5.04275314265368784163135845797922992e-44Q, 2.36599922549416536405636377221603052e-43Q, 1.08381346209104032486272733428199249e-42Q, 4.84896336796031616924505606902108385e-42Q, 2.11961287373765727720156892217888941e-41Q, 9.05594713902200264778664561627827336e-41Q, 3.78298719219266665046394564391309006e-40Q, 1.54564984691757476458462458129485585e-39Q, 6.17890975212602635731338783415082506e-39Q, 2.41759755862594038624882497707942985e-38Q, 9.26130599996633274587378852134765724e-38Q, 3.47471297119465611529279266980333841e-37Q, 1.27721589062918134522531144355249981e-36Q, 4.60093813393547386396189442938777048e-36Q, 1.62480431477305204386750726891935535e-35Q, 5.62680810313792997206232926753677840e-35Q, 1.91144242994708647057868821934329247e-34Q, 6.37130041549818712538060098629118750e-34Q, 2.08444453130944123746087664312229902e-33Q, 6.69535606006557423430835320939136513e-33Q, 2.11203843563779293148946583571749480e-32Q, 6.54480290655151239305039275670085965e-32Q, 1.99286493762398711426295964156611288e-31Q, 5.96435881776415175461953820911435643e-31Q, 1.75497323146494950021419641534165307e-30Q, 5.07823155886177386317817053950871430e-30Q, 1.44544786652825947518140437679267760e-29Q, 4.04809975939166078604969666043969110e-29Q, 1.11575287892799422088984788434321808e-28Q, 3.02733416844233859190251989588495462e-28Q, 8.08786849810622478767566085355596476e-28Q, 2.12810654415185893615787768003277562e-27Q, 5.51621011393022798460545237305226697e-27Q, 1.40889092112486390607440977505465003e-26Q, 3.54652073432677480700798349650597512e-26Q, 8.80063648109636049392437478276758980e-26Q, 2.15331950904398446525184217737998611e-25Q, 5.19613654473192634598878003489200027e-25Q, 1.23686905842220219027076147492586511e-24Q, 2.90489167449091887312162206507161675e-24Q, 6.73270731756325876275561922583341148e-24Q, 1.54025360336139105521086193967934171e-23Q, 3.47876572768722101871596015971992779e-23Q, 7.75845007993303197586197585003227682e-23Q, 1.70893932426983027624683633350310787e-22Q, 3.71846701056881115177897268795618277e-22Q, 7.99409437676902992032878645322876456e-22Q, 1.69833677431834312305384300763198925e-21Q, 3.56621446972400227479906082231485065e-21Q, 7.40284853486635166159906810215851685e-21Q, 1.51941171975529754933443555359007838e-20Q, 3.08399399452860874024652960034823534e-20Q, 6.19138881797445980861805317656851788e-20Q, 1.22962598701058922677441904835003634e-19Q, 2.41624594930841108357336910436255182e-19Q, 4.69855181874941970642420918335722321e-19Q, 9.04299297884852043932311013611381641e-19Q, 1.72288019839002081694998146770641182e-18Q, 3.24983285835411232229770728918949423e-18Q, 6.07012059458645756189773861988245818e-18Q, 1.12287188164609844099198470626937240e-17Q, 2.05742923566420592156922015953543702e-17Q, 3.73461320774281639944361675972781502e-17Q, 6.71669436926784207473111220730444193e-17Q, 1.19706302505504395186947411747310186e-16Q, 2.11441966111566361680483324148539475e-16Q, 3.70201713823102185303168392234499685e-16Q, 6.42566549874633786044656050804228769e-16Q, 1.10583090372698541886807277008925137e-15Q, 1.88715605166056322365232100161599762e-15Q, 3.19397901867912583303235022260119744e-15Q, 5.36188197747320445907706025927166858e-15Q, 8.92931856860669280901808059459161448e-15Q, 1.47533056095858665971377793523064186e-14Q, 2.41870863676582496406573146120828803e-14Q, 3.93507835090405130222724885910838729e-14Q, 6.35404709630865447857089142202307170e-14Q, 1.01841666646650944181450881700329392e-13Q, 1.62042378299930769318560924447489642e-13Q, 2.55981751705612616590139075734652273e-13Q, 4.01527388629421281039816262827476156e-13Q, 6.25453235826176129082121714446163184e-13Q, 9.67598102139418285827447083279565919e-13Q, 1.48683211253456618620643979203627482e-12Q, 2.26955737776048687888315587634986337e-12Q, 3.44173600876636583230784574560493186e-12Q, 5.18579385986065241250884484837407551e-12Q, 7.76421788931400466314739807932612879e-12Q, 1.15522810574654803550770552217628934e-11Q, 1.70831312146426209744016210099566072e-11Q, 2.51095185608620189710869551132160955e-11Q, 3.66877697851095234075891821003426570e-11Q, 5.32913181394174031371781017456930198e-11Q, 7.69632539729948085558352466099257329e-11Q, 1.10520072364372285498114055238330020e-10Q, 1.57822184379603482492805668832205965e-10Q, 2.24130967294097676646756319917307649e-10Q, 3.16577320114495664158493107630920887e-10Q, 4.44773051087161070439507137569795222e-10Q, 6.21604166145516404864589081218699790e-10Q, 8.64254490539598786804804563578215933e-10Q, 1.19551930651665934901247256485043253e-09Q, 1.64548212141718982285430677462884054e-09Q, 2.25364361294162088274171207920306355e-09Q, 3.07161057649675130982867546470369013e-09Q, 4.16647469046044592666212503776391905e-09Q, 5.62503650418518103546622761014447205e-09Q, 7.55905963895399839577230243791284335e-09Q, 1.01117741787649109183519733871956784e-08Q, 1.34658870190626745425612716113453914e-08Q, 1.78534009295770335027325069409745001e-08Q, 2.35675936423533751929780865973963492e-08Q, 3.09775637333761608808718626586210342e-08Q, 4.05458117130271472976754716443074815e-08Q, 5.28493928008555417314519862662075002e-08Q, 6.86052524785416844753739601495847428e-08Q, 8.87004371407679534565262439555179668e-08Q, 1.14227959934028163656084297163993474e-07Q, 1.46529195996537375677477289744322135e-07Q, 1.87243781452025990276419175076674860e-07Q, 2.38368096170532406236914962927186340e-07Q, 3.02323520821923278406510621336929642e-07Q, 3.82035773260694787625549171519571981e-07Q, 4.81026746749616004418812011672214969e-07Q, 6.03520391713916631448186742261762126e-07Q, 7.54564302177565687477041747448625764e-07Q, 9.40168786133714127970401442799808303e-07Q, 1.16746531401927207818183968119629644e-06Q, 1.44488634919934624164636989185786720e-06Q, 1.78236866676220579646941432400191756e-06Q, 2.19158235968382023977800841196782357e-06Q, 2.68618781213700528587671678902632939e-06Q, 3.28212298590973811003340670234049833e-06Q, 3.99792341503412914858164425580814066e-06Q, 4.85507733328388046876150511289283628e-06Q, 5.87841836668756018708322083288294132e-06Q, 7.09655820622938796350022322477138609e-06Q, 8.54236163220623609743157219064928822e-06Q, 1.02534661892020938105973895830986033e-05Q, 1.22728487074863285519290054710280426e-05Q, 1.46494407312787820235069142302801116e-05Q, 1.74387947455200274155445197262654991e-05Q, 2.07038028896765075538771239781158806e-05Q, 2.45154696092443087426053206140339263e-05Q, 2.89537394229808584415606234281235237e-05Q, 3.41083806769492860360664819345743796e-05Q, 4.00799258161539348751926239815464572e-05Q, 4.69806683323287862163878784706404376e-05Q, 5.49357161442722725101184470856795070e-05Q, 6.40841007374651816919514989779193660e-05Q, 7.45799409355181382811997812036059565e-05Q, 8.65936597006977565440049012882777955e-05Q, 1.00313251868244228486845169918522419e-04Q, 1.15945600213690649557974183508671660e-04Q, 1.33717836738558167374082301113276138e-04Q, 1.53878745542570977882656611663357910e-04Q, 1.76700203135100555424143059190915095e-04Q, 2.02478651530284460762520573567302864e-04Q, 2.31536598974665040190979221162493510e-04Q, 2.64224142678798208257632362853594384e-04Q, 3.00920507470608001324163745800874977e-04Q, 3.42035593863725830721435182160899216e-04Q, 3.88011528643900055031892572065609775e-04Q, 4.39324210725794779827016989314436073e-04Q, 4.96484844725809052233768108298280237e-04Q, 5.60041454438256227059209260385185510e-04Q, 6.30580368196231443658525282679749066e-04Q, 7.08727667948158660008974296844527949e-04Q, 7.95150593789209443880323604776955146e-04Q, 8.90558895655812679363260602821822057e-04Q, 9.95706123923012434268721813542918011e-04Q, 1.11139085073953859292031598702982886e-03Q, 1.23845781409454868818512091763619876e-03Q, 1.37779897683285042761735605405503089e-03Q, 1.53035449312115014435580213674818791e-03Q, 1.69711357521498846993912286538919474e-03Q, 1.87911525378240440542796943910703787e-03Q, 2.07744902550331120940589303741307914e-03Q, 2.29325538217982005624763697091003995e-03Q, 2.52772621615854827916639591227331912e-03Q, 2.78210509747707274118095565348853401e-03Q, 3.05768741879849780665835354304288470e-03Q, 3.35582040488560696306610399114125004e-03Q, 3.67790298408396440888192232818825231e-03Q, 4.02538552002609726985152127212337675e-03Q, 4.39976940253081440732383904779371772e-03Q, 4.80260649744698504497144153794077709e-03Q, 5.23549845597384011114251649759318761e-03Q, 5.70009588477421233645539752147845015e-03Q, 6.19809737897730872540802302573847437e-03Q, 6.73124842093794861403414763375789954e-03Q, 7.30134014837421983438805473382268060e-03Q, 7.91020799623995212524470914365122922e-03Q, 8.55973021739730390294556694387596169e-03Q, 9.25182628783344529797384941938652911e-03Q, 9.98845520280948891334775899197118662e-03Q, 1.07716136709355454400393298774895433e-02Q, 1.16033342137295485601573254130151715e-02Q, 1.24856831787362164573968971563345559e-02Q, 1.34207586747535542664119568732591706e-02Q, 1.44106884381354658508301066473916456e-02Q, 1.54576276395086064771405090469805654e-02Q, 1.65637566405583013536386920486874088e-02Q, 1.77312787108013640215572254554956350e-02Q, 1.89624177144726038238696005618612010e-02Q, 2.02594157778067758841208537767709612e-02Q, 2.16245309470991783928580857382458621e-02Q, 2.30600348479769142051214039575544903e-02Q, 2.45682103563102531782599654851478334e-02Q, 2.61513492911411521698129655317611847e-02Q, 2.78117501399057252267505437298152555e-02Q, 2.95517158260815126327011961248970821e-02Q, 3.13735515292012408141424535463840985e-02Q, 3.32795625669450926962877709780283664e-02Q, 3.52720523487562160472278601135978908e-02Q, 3.73533204101223493795063079870277644e-02Q, 3.95256605363332412560750462340568604e-02Q, 4.17913589841622853414796289201462570e-02Q, 4.41526928095348722137611194180899765e-02Q, 4.66119283088387990349053837529086301e-02Q, 4.91713195811071287202299070345370512e-02Q, 5.18331072178645941782756741967663201e-02Q, 5.45995171269784130156535121811973661e-02Q, 5.74727594963965733729390200663376457e-02Q, 6.04550279031945582526756917103894992e-02Q, 6.35484985728882875351754365961380466e-02Q, 6.67553297935098586459427868785125518e-02Q, 7.00776614884864197850796119077335695e-02Q, 7.35176149519140388651301532758182494e-02Q, 7.70772927493804152471705586229649727e-02Q, 8.07587787870652431694028348485017861e-02Q, 8.45641385514373366852999925423097546e-02Q, 8.84954195214754605679505353644207668e-02Q, 9.25546517549672049628688665169818439e-02Q, 9.67438486500890476543279055502732567e-02Q, 1.01065007883142650157999494215339249e-01Q, 1.05520112523018947242782060692540634e-01Q, 1.10111132322684063185699908950501089e-01Q, 1.14840025187730710332047574671488736e-01Q, 1.19708738821816529269842068051324872e-01Q, 1.24719212548617699381893630778769370e-01Q, 1.29873379309762826940592292027774350e-01Q, 1.35173167838079215925316604016895877e-01Q, 1.40620505005381631636575206950501568e-01Q, 1.46217318343962952595269565777364191e-01Q, 1.51965538740906942447303428154167210e-01Q, 1.57867103304335938296117505727756989e-01Q, 1.63923958400730641095309983504562072e-01Q, 1.70138062862515433149015694041321523e-01Q, 1.76511391365190704205555387802167839e-01Q, 1.83045937973413460592324819248629329e-01Q, 1.89743719855578905138540205901925041e-01Q, 1.96606781166638569017145529907938222e-01Q, 2.03637197099104797388960830732662422e-01Q, 2.10837078102436785186096324004787346e-01Q, 2.18208574271279784343490287100076472e-01Q, 2.25753879903336437885194071240289887e-01Q, 2.33475238227987351081778389476348222e-01Q, 2.41374946307146941036852686683657608e-01Q, 2.49455360110240324117132946908862256e-01Q, 2.57718899765617581970697551365699803e-01Q, 2.66168054991183344330456451077482845e-01Q, 2.74805390707512480349974276871498222e-01Q, 2.83633552837247137560986852113493320e-01Q, 2.92655274295126854671240651245542162e-01Q, 3.01873381173592566247581847444990310e-01Q, 3.11290799129527708372244037377423071e-01Q, 3.20910559978356159579257040371157487e-01Q, 3.30735808502408397196150346208941643e-01Q, 3.40769809481195164806575214000604739e-01Q, 3.51015954951993455453034437018576899e-01Q, 3.61477771709954227353934951808660907e-01Q, 3.72158929057786693176355712369457666e-01Q, 3.83063246815962181179662085907016341e-01Q, 3.94194703605313603549231284148895684e-01Q, 4.05557445414886871112481166359816563e-01Q, 4.17155794468930807411520974095114834e-01Q, 4.28994258407995154251335647089270309e-01Q, 4.41077539800245330879960121109734359e-01Q, 4.53410546000301224452886670053661009e-01Q, 4.65998399374169294389180835005701951e-01Q, 4.78846447910166863084991969300289662e-01Q, 4.91960276237139210883303640625007935e-01Q, 5.05345717072748965861316155618037977e-01Q, 5.19008863126178679451085258491885813e-01Q, 5.32956079481237266942061435303847022e-01Q, 5.47194016487605519485083198030599369e-01Q, 5.61729623189802041293777914000548030e-01Q, 5.76570161325406179304747155047028479e-01Q, 5.91723219926146849096660434528676034e-01Q, 6.07196730557664332661777342118343230e-01Q, 6.22998983236085549221918733111272834e-01Q, 6.39138643062032159550125999965534073e-01Q, 6.55624767615316158435211030870214346e-01Q, 6.72466825156381227156528472165863186e-01Q, 6.89674713683532904728192023160169670e-01Q, 7.07258780898180476422511449104598714e-01Q, 7.25229845133703375815759935882969711e-01Q, 7.43599217307171072603779249870503705e-01Q, 7.62378723957005410057715275736188342e-01Q, 7.81580731433797129025240842779308298e-01Q, 8.01218171315894385869574690877526536e-01Q, 8.21304567126092639185776756157030964e-01Q, 8.41854062430796373280836243276360497e-01Q, 8.62881450408419762754617604609421022e-01Q, 8.84402204979573743001129336964073261e-01Q, 9.06432513597781571704880022546834535e-01Q, 9.28989311806106946360784332676954119e-01Q, 9.52090319672203976394493741712123982e-01Q, 9.75754080221945735296148718924093021e-01Q, 1.00000000000000000000000000000000000e+00Q, 1.02484839189454300830993201809676807e+00Q, 1.05032052037278447499546563674080434e+00Q, 1.07643864928417387084533130337621185e+00Q, 1.10322609239912797847732993345442148e+00Q, 1.13070726686292705163871994441376389e+00Q, 1.15890774975714122929598339522250570e+00Q, 1.18785433797464608445948646027899515e+00Q, 1.21757511162904898441624819295410761e+00Q, 1.24809950123526638608045628896113726e+00Q, 1.27945835891516450038384515634984238e+00Q, 1.31168403390070906172271785633240954e+00Q, 1.34481045262708114284870961989593531e+00Q, 1.37887320372983270957282413312613361e+00Q, 1.41390962828351735195133854106565322e+00Q, 1.44995891564449075379247910337230275e+00Q, 1.48706220528789860745767637311307254e+00Q, 1.52526269505843914812343336632313553e+00Q, 1.56460575628650281139860620478932861e+00Q, 1.60513905625597123057624111991440169e+00Q, 1.64691268854754131266306691391994313e+00Q, 1.68997931182218993728102497644674194e+00Q, 1.73439429765359879279739907195277543e+00Q, 1.78021588806633292063608039095887334e+00Q, 1.82750536348865755482665463985323837e+00Q, 1.87632722188546688075368971900486300e+00Q, 1.92674936989830423931126850366320158e+00Q, 1.97884332688633669446426238525349115e+00Q, 2.03268444283491461260047065213162138e+00Q, 2.08835213117755699197923733656794571e+00Q, 2.14593011766347043193461156899167303e+00Q, 2.20550670649671136552919447649838551e+00Q, 2.26717506507558468055611786100163107e+00Q, 2.33103352877266160486316503843066293e+00Q, 2.39718592731780603672335804074417192e+00Q, 2.46574193447982700411979376412836737e+00Q, 2.53681744288793726408585573416385695e+00Q, 2.61053496599332371109890452901649940e+00Q, 2.68702406934518495604136651525501878e+00Q, 2.76642183354607197859959809591833922e+00Q, 2.84887335145994878054909245340497664e+00Q, 2.93453226247492266642719969961850934e+00Q, 3.02356132687313192268979888413460939e+00Q, 3.11613304363510221142689168603250629e+00Q, 3.21243031530752459841477997957148726e+00Q, 3.31264716389468297553121493594219218e+00Q, 3.41698950209779795665404010466511538e+00Q, 3.52567596462684319748110900008331556e+00Q, 3.63893880474980996700862087872826533e+00Q, 3.75702486172927248744324880848351753e+00Q, 3.88019660533026434144487775885161751e+00Q, 4.00873326417229898631367723395410172e+00Q, 4.14293204534786760904278361346846386e+00Q, 4.28310945344664439881789918389859821e+00Q, 4.42960271791643703960803985590050978e+00Q, 4.58277133856704814714568853892203278e+00Q, 4.74299875999107924880685634942448809e+00Q, 4.91069418674686750664370967067014440e+00Q, 5.08629455233503443708376995130700228e+00Q, 5.27026665631483182008254527988202911e+00Q, 5.46310948536451639599392689767712196e+00Q, 5.66535673570814692696549699934052267e+00Q, 5.87757955612834548026155780285696226e+00Q, 6.10038953278194387910748891481975603e+00Q, 6.33444193925698167042406747055218470e+00Q, 6.58043927778222227422530564217212955e+00Q, 6.83913514025466452585283059757581636e+00Q, 7.11133842082084256614010724304296860e+00Q, 7.39791791517290376256435527130893558e+00Q, 7.69980734554450846886947561889535387e+00Q, 8.01801085466429447369362846417322143e+00Q, 8.35360901670240672837408029297072392e+00Q, 8.70776541859238547329480193384508847e+00Q, 9.08173387109914748403941266390886970e+00Q, 9.47686631571637600605881026375055671e+00Q, 9.89462150100714627457776733266750735e+00Q, 1.03365745104567901877248523564151283e+01Q, 1.08044272334084191011529376038077113e+01Q, 1.13000198813377778114411160555843168e+01Q, 1.18253436637533511507143176436936036e+01Q, 1.23825547515605242739627815508172524e+01Q, 1.29739896710116156250639129682527238e+01Q, 1.36021822886130624460500072703793332e+01Q, 1.42698825668476028925550536958145770e+01Q, 1.49800772926032764371676872413981972e+01Q, 1.57360130051385708102009640040308534e+01Q, 1.65412213786631649993077912938609934e+01Q, 1.73995473466468578424134221353716244e+01Q, 1.83151802913268898058518900530921820e+01Q, 1.92926886631898453200253754020840677e+01Q, 2.03370584421782617230641820877704224e+01Q, 2.14537359058448294184882229567324528e+01Q, 2.26486752306089873609060295600452917e+01Q, 2.39283915217729827153789982621787140e+01Q, 2.53000199473141826806593837999123741e+01Q, 2.67713817411801152928733734963681048e+01Q, 2.83510579456049880498903420268116163e+01Q, 3.00484718808548719519487088002485537e+01Q, 3.18739814671361063943508266371074277e+01Q, 3.38389826798966490384466465437407911e+01Q, 3.59560255995953567185453470776520193e+01Q, 3.82389447239249331007247241110692191e+01Q, 4.07030054487934539563244342845687403e+01Q, 4.33650688991795367943193597150736221e+01Q, 4.62437776082326978417918860128324332e+01Q, 4.93597649096797907147762266234322051e+01Q, 5.27358913329271476516365756663547356e+01Q, 5.63975117818677084656045931084727554e+01Q, 6.03727778486785227486990377756623302e+01Q, 6.46929802762275435095300960911126288e+01Q, 6.93929373529211836464995846228029413e+01Q, 7.45114359206196683576202953920800440e+01Q, 8.00917327217667406576253702254103983e+01Q, 8.61821250323685694884618177226323948e+01Q, 9.28366009540655148012445479790749067e+01Q, 1.00115581408296888990404128808417666e+02Q, 1.08086767832535244787263306357432758e+02Q, 1.16826111875294927859522500437909465e+02Q, 1.26418926085804724017089848087642554e+02Q, 1.36961157770833171464154963091055755e+02Q, 1.48560851934901186551056930429681893e+02Q, 1.61339833638593274264805190552029918e+02Q, 1.75435645332062901686076817427004926e+02Q, 1.91003780902460959044883635552383857e+02Q, 2.08220265501991356519964932798160160e+02Q, 2.27284638923300107785084254230641145e+02Q, 2.48423410633602325676190041300652816e+02Q, 2.71894066898304725754664440871201054e+02Q, 2.97989725118823201634188484455430791e+02Q, 3.27044548063367687799351246625873367e+02Q, 3.59440051674122988480906333522906174e+02Q, 3.95612465308733548542080237354845984e+02Q, 4.36061333495907795272034066341690570e+02Q, 4.81359584626980835451438312811051118e+02Q, 5.32165335780833820345250001609436065e+02Q, 5.89235755699686219555872484606593438e+02Q, 6.53443371777544904545726289396026874e+02Q, 7.25795284228401899442052568196334372e+02Q, 8.07455844372956662653935015669559504e+02Q, 8.99773467933970120032100931203892228e+02Q, 1.00431239295794425220145278602443696e+03Q, 1.12289036118559487702348487348886695e+03Q, 1.25762340845977552983926843880669836e+03Q, 1.41097920290752223402430410334039052e+03Q, 1.58584068016657346033724860162595503e+03Q, 1.78558210660144726153863412855453166e+03Q, 2.01416017149982591387363180452914622e+03Q, 2.27622328928316747879110750486628791e+03Q, 2.57724301000797348481178480414602978e+03Q, 2.92367232516280459791647074886201195e+03Q, 3.32313675929073604672388801848045153e+03Q, 3.78466551111357505023764578327774171e+03Q, 4.31897162016023640570560918832525773e+03Q, 4.93879227485091848874762796864624098e+03Q, 5.65930305827336833061655557231992410e+03Q, 6.49862329247639500380687922515090074e+03Q, 7.47843387531893338609063539714339383e+03Q, 8.62473434228616623771753210405507751e+03Q, 9.96877263348459014531785309952890789e+03Q, 1.15481895955939390229775103785535678e+04Q, 1.34084311070264939001405447133999516e+04Q, 1.56044945390858044276508880692505176e+04Q, 1.82030939102313379306065080413190777e+04Q, 2.12853506664968077681001285108385025e+04Q, 2.49501459804837504640469288880047968e+04Q, 2.93183077048218804714595369958899209e+04Q, 3.45378531384547339729721217099269346e+04Q, 4.07905708493105663079713795820150418e+04Q, 4.83003052786320640972382415932704632e+04Q, 5.73434124658699200382652517134835469e+04Q, 6.82619915902214645340692561470062070e+04Q, 8.14806752559419146429293810910586320e+04Q, 9.75279950747873086720735652227979263e+04Q, 1.17063646220480829492981340691767948e+05Q, 1.40913379548158414250243653370073176e+05Q, 1.70113785311182551164819299122255310e+05Q, 2.05969942671050993972655988038600318e+05Q, 2.50129853973569246335608225835338567e+05Q, 3.04680843555537948624554921830727349e+05Q, 3.72274788636036141079112881438654547e+05Q, 4.56291316446017606669036491868612297e+05Q, 5.61051155492184554128598135519314736e+05Q, 6.92095956581034369089275407099645880e+05Q, 8.56556497218119814922783780052161157e+05Q, 1.06363880055232600043867227938165842e+06Q, 1.32526810122628602501044472769613140e+06Q, 1.65694484184724012086108587753064923e+06Q, 2.07888647930116015575144959412983113e+06Q, 2.61755592013006806905408032351706035e+06Q, 3.30771485222622495472503881888885584e+06Q, 4.19519229320262625923561042278599896e+06Q, 5.34063130025074556576550111959530826e+06Q, 6.82457849576702073415053978191461581e+06Q, 8.75442405324883181776310371883173248e+06Q, 1.12739015977226351727931535875936998e+07Q, 1.45761434273968962478096940509724237e+07Q, 1.89216932684193810033543675766077150e+07Q, 2.46634598680066744212983509402346013e+07Q, 3.22814282171121758797754478348568006e+07Q, 4.24311457153986975423657329267444940e+07Q, 5.60117371443408843149644877281459637e+07Q, 7.42617250972307211179134697728146644e+07Q, 9.88946135783012173106461237736151683e+07Q, 1.32291587547042718239491966847893521e+08Q, 1.77776624072745598074170059515038266e+08Q, 2.40011058338983426258122648845724399e+08Q, 3.25562103364198274177076500687162156e+08Q, 4.43725882059376140276746390832793801e+08Q, 6.07724621850487716521348005200021505e+08Q, 8.36456587985737541719041767923031690e+08Q, 1.15706659432645616854021468641216152e+09Q, 1.60874082649874296139935339596050162e+09Q, 2.24833765794868826938828602052210087e+09Q, 3.15878597885133622780330724633488102e+09Q, 4.46167708136391137964792211498946984e+09Q, 6.33624483104820926953501466923377042e+09Q, 9.04813015958867755955007060963155039e+09Q, 1.29932136230997226464299124717687605e+10Q, 1.87647826121294792855983490567276630e+10Q, 2.72570397671288897105662591436709544e+10Q, 3.98255345906428893971269386370983030e+10Q, 5.85372779401741541532402802418687327e+10Q, 8.65629908955310338465751817097315538e+10Q, 1.28795973304189874698643530406628864e+11Q, 1.92834506543009988305207298557577307e+11Q, 2.90551046754580604436065170645660613e+11Q, 4.40614548809848580878194871514169088e+11Q, 6.72570891877849315226112306772636026e+11Q, 1.03348693821219692992886708382137863e+12Q, 1.59884055708669585428974649723976740e+12Q, 2.49049013421827282502410713540807461e+12Q, 3.90652846672458392086943060796482254e+12Q, 6.17122514796135424360349149730712054e+12Q, 9.81916373648510913725273324679675423e+12Q, 1.57380010699156447495769482655375521e+13Q, 2.54124546153003122065528612181733881e+13Q, 4.13443762840798177552480089332110759e+13Q, 6.77814197348597152801493367496981491e+13Q, 1.11990628659588449226309821338662554e+14Q, 1.86501680604176896658450602560601490e+14Q, 3.13089094872498973827438463058389098e+14Q, 5.29897884766906828009755247847186245e+14Q, 9.04297389980418175348985287558326633e+14Q, 1.55625903681899143890977078931637702e+15Q, 2.70123006636820081209540843148166787e+15Q, 4.72943010505471127882390119991674115e+15Q, 8.35377903309658653016536824399814765e+15Q, 1.48882760629319165095592517030838223e+16Q, 2.67765346603161495578266855417246317e+16Q, 4.86043448136949927032370428281054854e+16Q, 8.90573551930099331171397006211491165e+16Q, 1.64741372830687155232163226493215628e+17Q, 3.07708132567301637697547718557883369e+17Q, 5.80423410132909767997152626617023549e+17Q, 1.10582857062809961361827765317796937e+18Q, 2.12831535880807402614537212551250197e+18Q, 4.13865153208523558147805437822414497e+18Q, 8.13255421212392003487525754214292826e+18Q, 1.61514650331257085505245767762340125e+19Q, 3.24254846726071819306305572744455009e+19Q, 6.58149458108070132105640205170170346e+19Q, 1.35083136618309000260400853464983546e+20Q, 2.80409383252093739584477712160125327e+20Q, 5.88811368346756383731750279395970839e+20Q, 1.25092343531246827629449550883330451e+21Q, 2.68928027909821563456325160060548474e+21Q, 5.85158282566447970019894650702188721e+21Q, 1.28891723178894465997393609758461271e+22Q, 2.87458276376899763123926200401593262e+22Q, 6.49243733510921786856841605747277893e+22Q, 1.48528660586708217722481671351080137e+23Q, 3.44246915911330706635229848078895385e+23Q, 8.08493019686043820711319325430367342e+23Q, 1.92450677804809487818429003209217169e+24Q, 4.64399266249147072865108838294493471e+24Q, 1.13628145208359133428909996372167656e+25Q, 2.81966489106069457125481864485455314e+25Q, 7.09778155999185636739332722019170506e+25Q, 1.81283885012768848640687277115301054e+26Q, 4.69901285134453912350857508188703800e+26Q, 1.23641970716283295078972799846710910e+27Q, 3.30323626121041128601791088120844416e+27Q, 8.96255809763889121781104645394017356e+27Q, 2.47029485298622611738479511793982193e+28Q, 6.91827096055594288255981089299725583e+28Q, 1.96918944795841151009270119290336857e+29Q, 5.69809260945398128927937198565189507e+29Q, 1.67662615639692208356086898776485724e+30Q, 5.01790152017155697028174057139776321e+30Q, 1.52792989227983448898308499437330165e+31Q, 4.73476231836671194909529913837515959e+31Q, 1.49357254644677704024642645900767165e+32Q, 4.79744116468190818427448048047019508e+32Q, 1.56953829640099873205260260372105814e+33Q, 5.23165115691024245360777615947271007e+33Q, 1.77720651152529094086938101558873583e+34Q, 6.15458729957691613373022022121395673e+34Q, 2.17346978135660487165909974321640569e+35Q, 7.82952989652658161590959525826590591e+35Q, 2.87793555407307691688503613867678361e+36Q, 1.07976132092345859199417272983750960e+37Q, 4.13633773095120704247204289059445522e+37Q, 1.61840848971118584374440265082561208e+38Q, 6.46977064044782477057521822447781556e+38Q, 2.64341365485931635830069155722471865e+39Q, 1.10424672830852570286491286394819018e+40Q, 4.71784264188126066477442678013103229e+40Q, 2.06229646238932771079246843113076062e+41Q, 9.22668000516125721911884540275100330e+41Q, 4.22654407163273196344404760642122190e+42Q, 1.98304372970706651824186968975247346e+43Q, 9.53344869097015503904308238124562564e+43Q, 4.69791457874020860581396665191487360e+44Q, 2.37392310198043657407434802189623818e+45Q, 1.23057021186853175270406146048991232e+46Q, 6.54634433841169514665337805691594822e+46Q, 3.57537181933580491356588477879121785e+47Q, 2.00564245353833550563491221890112961e+48Q, 1.15605526802890307768061561905701576e+49Q, 6.84986780787031295787489135284280640e+49Q, 4.17400481521895112076613271616259486e+50Q, 2.61687203405285747164128861362318936e+51Q, 1.68875034683729772513757434403804400e+52Q, 1.12227566600968410062529607944214843e+53Q, 7.68396874024867707105114297890426357e+53Q, 5.42284961265427858271262866960101842e+54Q, 3.94668670179953341522351651523862327e+55Q, 2.96354358728813288396379337565903477e+56Q, 2.29708639579893951580254665307105346e+57Q, 1.83885641420855576075627699143076709e+58Q, 1.52104947571124399594694864621286492e+59Q, 1.30073229117507111236343155772481130e+60Q, 1.15055959114171674001974091149754682e+61Q, 1.05326599737372546052564481925643420e+62Q, 9.98411420987902083614726789220470599e+62Q, 9.80532561593869471877040903652671478e+63Q, 9.98246356419911599549241472274732206e+64Q, 1.05410221145791141007800297806625299e+66Q, 1.15517268478078246328479789979967959e+67Q, 1.31457130233411666326872185958326427e+68Q, 1.55436240768545731029836466857749970e+69Q, 1.91079120600264507732475085854133289e+70Q, 2.44361640389071120608105036496582382e+71Q, 3.25298382231882323170055543487798304e+72Q, 4.51060014002013973693091388809534590e+73Q, 6.51882183100190244682597556816883900e+74Q, 9.82583446077426763261011103909194169e+75Q, 1.54569206362272285597219332482212685e+77Q, 2.53934608840816325270469005344260713e+78Q, 4.35976399381183611739718946785420143e+79Q, 7.82794362746440474440241447663834683e+80Q, 1.47089687767430118295343594318257369e+82Q, 2.89452707142067428998158759253431372e+83Q, 5.96966254160791549191064067088778403e+84Q, 1.29127761398105735746114390466172354e+86Q, 2.93165653562687792342264563025180130e+87Q, 6.99135354753146313546505095265922672e+88Q, 1.75267119452597285220967300845626556e+90Q, 4.62245013705602071519123099756702996e+91Q, 1.28358193316956622609422743295635868e+93Q, 3.75583900113839078839602923077406673e+94Q, 1.15899172984597870198690174300625365e+96Q, 3.77491631543886267838503954431054828e+97Q, 1.29884489446238167309781708169537790e+99Q, 4.72503894994338488909389270574194659e+100Q, 1.81900003120328673954815570724729852e+102Q, 7.41696633087690618810014849262359799e+103Q, 3.20611699691059820440544154944066069e+105Q, 1.47058877007197519318900335321912402e+107Q, 7.16419823823864105738627712719272912e+108Q, 3.71039762456707726983380527316097819e+110Q, 2.04488245427970937306347337109290828e+112Q, 1.20042877865473022461473385597117568e+114Q, 7.51374437003017211432960701498996352e+115Q, 5.01957574634341063561427626174780847e+117Q, 3.58272692766569831842277681502801710e+119Q, 2.73494777587724855953563379276564479e+121Q, 2.23528376407894424768162839028330791e+123Q, 1.95808475111824332331314972422066776e+125Q, 1.84043191310930565706137359879906765e+127Q, 1.85814326069283110784535066904520512e+129Q, 2.01743294965577713629069559518257063e+131Q, 2.35817761588810149418319514273035811e+133Q, 2.97109297417860360975602176456436124e+135Q, 4.03953232143581630156016504419988399e+137Q, 5.93392306966113219484352196331202218e+139Q, 9.42926369344495323976110371737768721e+141Q, 1.62284145693287387206611282820134815e+144Q, 3.02888447606769418009867639216690630e+146Q, 6.13835617501533947725917198648160556e+148Q, 1.35253155719194264809220346207577825e+151Q, 3.24444736229558294520810049145898860e+153Q, 8.48424354492815185029440589500302573e+155Q, 2.42189217690801196639615731788461939e+158Q, 7.55727585868832926111996462541955391e+160Q, 2.58138920742415528768470662656564452e+163Q, 9.66580408821980500452620634685578942e+165Q, 3.97326060268422220330874023543806264e+168Q, 1.79563826725695454978311213353995983e+171Q, 8.93514523408268859940137248520982995e+173Q, 4.90290834459626229428863642680474223e+176Q, 2.97128375542216561281658377940574510e+179Q, 1.99182881480698947977101116086872463e+182Q, 1.47933813166325570382207702423402678e+185Q, 1.21924602572867713843112063103440147e+188Q, 1.11695489454371994985095225835122915e+191Q, 1.13926159735841086544690160419235263e+194Q, 1.29595329494635946555823051599904183e+197Q, 1.64694619802426865078234124605721432e+200Q, 2.34234840634634342775071057533872554e+203Q, 3.73486904651415569743903039850576755e+206Q, 6.68856535510325485878978796153529680e+209Q, 1.34777842855531871957592541417095863e+213Q, 3.06153116112389690978003658017819254e+216Q, 7.85439891718825136093879357578783886e+219Q, 2.28020371103825909996341632353690860e+223Q, 7.50526802410090668256555773098038714e+226Q, 2.80639879206390242816779047164860586e+230Q, 1.19452560453314111686916354242681346e+234Q, 5.79949872394246778903337432628166035e+237Q, 3.21835787345388969294352208047085465e+241Q, 2.04569666433064103875712173825120783e+245Q, 1.49258715453013675730747996991557044e+249Q, 1.25277443244629163588528004740184593e+253Q, 1.21226842377681326200203984685187270e+257Q, 1.35547194242695069046025865235623225e+261Q, 1.75524005232377336464776480479935346e+265Q, 2.63840064228932918271533450220126376e+269Q, 4.61448246937529975889164745570586249e+273Q, 9.41280682985947960778964419634581143e+277Q, 2.24481910655275049981565977905948653e+282Q, 6.27448882996682532882323387040658757e+286Q, 2.06060958394918535923518047482752655e+291Q, 7.97142764215603564227622164969231434e+295Q, 3.64182905892925111147897309560503282e+300Q, 1.97007938638916489488880184999451702e+305Q, 1.26527287963256941162906241681573766e+310Q, 9.67374762635351109418734649102546799e+314Q, 8.82892141913231571338896445110187498e+319Q, 9.64570405228039780469604907885256630e+324Q, 1.26503839487205882730553259131731653e+330Q, 1.99740647993660386944495280559643123e+335Q, 3.80795083906732148311558637535145760e+340Q, 8.79159120777085501704319293452205335e+345Q, 2.46549631780369377584312978131661447e+351Q, 8.42426490586997547506062762517991909e+356Q, 3.51804562259170040261749407236406336e+362Q, 1.80129565203500398450083781286924107e+368Q, 1.13442688952887433720702641069844446e+374Q, 8.81641141946113920681847478154293508e+379Q, 8.48338453086470297056624825970987040e+385Q, 1.01407299264221993702531525534230006e+392Q, 1.51103955764611037180263766796269076e+398Q, 2.81641187251957701963969414170240418e+404Q, 6.58963120298401361748098859356824253e+410Q, 1.94234184512060662929742463444500478e+417Q, 7.23884585856411287896767410403126131e+423Q, 3.42370598009480301692521696476539260e+430Q, 2.06270296737946127496845017607144670e+437Q, 1.58907837423540569213464552634511445e+444Q, 1.57146131021351822368954401625918115e+451Q, 2.00271733703560402612457677241115831e+458Q, 3.30238681816736073809946856032462582e+465Q, 7.07441603885154545912376402781538965e+472Q, 1.97695789038711346120868489115303627e+480Q, 7.23713029747673018134344413894888363e+487Q, 3.48533178833568221019229567888827162e+495Q, 2.21771798011905249239941207476859498e+503Q, 1.87265680045956064211817379705948563e+511Q, 2.10782709864946331624669282953979957e+519Q, 3.17689162904658562291907841456392867e+527Q, 6.44105645558316603786838285606330189e+535Q, 1.76492886625329297926588753856751673e+544Q, 6.56708664348495439553998825816543384e+552Q, 3.33415826521812264992257180271069504e+561Q, 2.32109565019822213867641474306217000e+570Q, 2.22665090852540791532492989207079877e+579Q, 2.95839667933009412312396349376762019e+588Q, 5.47183877884770709370953901555429409e+597Q, 1.41626756773793934414234479620392920e+607Q, 5.15692247865370682523431517094128585e+616Q, 2.65584978465684945355727442801669402e+626Q, 1.94515906011252752522390088574648572e+636Q, 2.03729480055734573550624444975610288e+646Q, 3.06864139959250122576234946830893261e+656Q, 6.68523683038794846390318553081110530e+666Q, 2.11879952715090010307995150421897875e+677Q, 9.82718465915609660920397095265955939e+687Q, 6.71024976037303477285708646634894987e+698Q, 6.78677922161865926117102990656018413e+709Q, 1.02303626651674990928162545434962304e+721Q, 2.31286202012488077377232576408525763e+732Q, 7.89245107528823830762373616398932534e+743Q, 4.09160120200842896534230185492282146e+755Q, 3.24379955366775267129216830328274275e+767Q, 3.95912453354472811300573118431549283e+779Q, 7.48996232152866120855530906466249236e+791Q, 2.21152965349315333956435853596087911e+804Q, 1.02632075670404876010930426472294612e+817Q, 7.53951068632005171876747821502928323e+829Q, 8.83108581906785992138514068848879898e+842Q, 1.66144742506056069516088714981068146e+856Q, 5.05826477150550571129779691932628665e+869Q, 2.51102428144577988853257771418346819e+883Q, 2.04822869357369644800523164159313000e+897Q, 2.76681803049890115118755008828029448e+911Q, 6.23888945414058934548223953019542236e+925Q, 2.36735167714072318956060511661796446e+940Q, 1.52407555570930344049434626852406921e+955Q, 1.67862596848152559342541655995681177e+970Q, 3.18990333799710745664618509349355327e+985Q, 1.05489081758823555970428507172122908e+1001Q, 6.12396788613568122227472934597131896e+1016Q, 6.29654583950170379780496166626637403e+1032Q, 1.15697821762367329197002183707451842e+1049Q, 3.83417721859040383986075244204614475e+1065Q, 2.31300589079836393970996707559294268e+1082Q, 2.56410460141809897241834340099170971e+1099Q, 5.27364125127925215166604640933456487e+1116Q, 2.03202107773477151599346823269715856e+1134Q, 1.48143334195874803524122054239589157e+1152Q, 2.06411274147574456351269054783640404e+1170Q, 5.55279829459270893053136130201212587e+1188Q, 2.91418497041682586483964699073577480e+1207Q, 3.01522322419410790036838054063732694e+1226Q, 6.21672053535323172221442203567173233e+1245Q, 2.58201182234158741299570291718541207e+1265Q, 2.18423750518116545391544630374143187e+1285Q, 3.80584781426053416091532316078175593e+1305Q, 1.38151208502663182821551093570287278e+1326Q, 1.05688876376696945875455031156325640e+1347Q, 1.72414525934903045436623350959611871e+1368Q, 6.06970176363513871871663291361433633e+1389Q, 4.66734911165218107635353767855185961e+1411Q, 7.93644483902981870648608149469983212e+1433Q, 3.02177407323915414129348065742188333e+1456Q, 2.60910294567790325670373269654248928e+1479Q, 5.17503694719938298503446121278113303e+1502Q, 2.38900287619137048290055040441414741e+1526Q, 2.60122790255932170291515511953064757e+1550Q, 6.77122378037074355003811222286996596e+1574Q, 4.27212445104598645308010594377100889e+1599Q, 6.62465696881760614600007097777440709e+1624Q, 2.56080093301387999855187759148833996e+1650Q, 2.50338804461975863168260136920269977e+1676Q, 6.28008959392222170902165204120090502e+1702Q, 4.10330805623617785960247879181409818e+1729Q, 7.08891597895639626581466081525369401e+1756Q, 3.28815937986385053022460003637061243e+1784Q, 4.15918117856089374793600707142802802e+1812Q, 1.45749058887095094990047184214707565e+1841Q, 1.43785687357830444275837543949503093e+1870Q, 4.05897446691318032347020269164975282e+1899Q, 3.33348114079172166390364845912603682e+1929Q, 8.09961164105229487950427444532606992e+1959Q, 5.92288638308847934616085992236628536e+1990Q, 1.32629748644648493533560071181408206e+2022Q, 9.25637244456745511366766862807596139e+2053Q, 2.04978139964660371723728495327865680e+2086Q, 1.46668915962420498068615471735159780e+2119Q, 3.45424976463618489443831684054473314e+2152Q, 2.72836294216453749285575709798797945e+2186Q, 7.36646679330208627221413509790840381e+2220Q, 6.93156469113828331566714060809446921e+2255Q, 2.31823213431754954747032929849642953e+2291Q, 2.81131341663812436900512315439163404e+2327Q, 1.26153121461887745230804891576437417e+2364Q, 2.13831434840067269117435765818454673e+2401Q, 1.39803888650069112961026818896691495e+2439Q, },
   };
   m_weights = {
      { 5.90801545776726515999184015930013435e-2494Q, 5.43034121836882538817369981204760444e-916Q, 9.13645323997978810837635266947225189e-336Q, 1.38026921250443161275979571515063966e-122Q, 1.95473329586235063070470724741264116e-44Q, 5.89765053810344838395474502424100508e-16Q, 9.65083022671853583721482039321815318e-06Q, 3.18674406396319375693901804428127016e-02Q, 4.49940993129051317388425885244478941e-01Q, 1.85250678576020588686470383628874232e+00Q, 2.14751816323261845650556006802974073e+01Q, 3.65018587462837432006103692095515092e+03Q, 6.47212478751933019554592655468556643e+08Q, 1.92641646488982742563851849719109881e+22Q, 1.38994758402678387868517621844463411e+58Q, 7.26879966558078977021348471744247141e+154Q, 1.04765280279857897745141662958802655e+417Q, 7.83837589244484743057347089777048838e+1128Q, },
      { 1.10316126712058491420201654940763610e-1511Q, 9.01600613495532057799091391040663772e-555Q, 5.11904332497495239643103179971390500e-203Q, 6.89671333890943322248923557636362424e-74Q, 1.17634110503454704251022002576839267e-26Q, 1.50757351778282569234155861289206864e-09Q, 1.64896113239222241299157252910603817e-03Q, 1.70348472687044679146785418051347199e-01Q, 8.92604116485216943737039962090086924e-01Q, 5.08107015469559685466563888116585167e+00Q, 1.70322410074360164066883311173278799e+02Q, 3.99580125020294769314748785515084392e+05Q, 8.97859410735688933722745437703783632e+13Q, 7.52866673518542859464441779223888455e+35Q, 5.14816023241024489763203760058639816e+94Q, 7.76882384415261869036597544471920535e+253Q, 6.83649581024319385239349778375109740e+685Q, 1.71685218299087881635674367971803808e+1859Q, },
      { 9.87925015299918398144785423598176370e-1942Q, 8.71533681693018261767494097538810486e-1177Q, 6.57384046217461429576867660255037671e-713Q, 1.22572352048272001483240603010122094e-431Q, 4.00210488905354704066065300402901994e-261Q, 8.68831861142192461329723188826285819e-158Q, 3.39344901437582485256542967096807177e-95Q, 2.56737317400303409399563341354938201e-57Q, 1.99325419812798016097936249998766075e-34Q, 1.25616316381741439662336351160566326e-20Q, 2.43090464171805920109733890177176767e-12Q, 2.14048123340650521217897512392567060e-07Q, 1.78351176282128440930436365965135303e-04Q, 8.91532759780500845110690795259481658e-03Q, 8.30018538939149499579790988285111688e-02Q, 2.93820046590635175209882488313404811e-01Q, 6.42167019459198241099164736507367056e-01Q, 1.25676009815264777866804196778444950e+00Q, 2.93241842564261090318837326991751597e+00Q, 9.80176974669959846616864231368773370e+00Q, 5.49217006325024175243162209529224299e+01Q, 6.74415210957129787494085872629636299e+02Q, 2.95093734929150449033134748913572566e+04Q, 1.05011160163132749880805450719595372e+07Q, 1.19276753677448525725842377258604019e+11Q, 4.11976276783133288599595346140932524e+17Q, 1.77581149088770071821845041850345600e+28Q, 4.35594809668394640823177135833505819e+45Q, 1.46965529638197726713059781629071505e+74Q, 1.14872217888121999297587395528584026e+121Q, 1.70677996026663975507589457654701634e+198Q, 2.11819696469088836247505505584030188e+325Q, 5.33443484802698043954530255107399872e+534Q, 6.76327254647744376095801434158695958e+879Q, 4.65552684789936854337398747780741866e+1448Q, 2.41016379900581970812596189341460931e+2386Q, },
      { 1.32133547680717018679108130408012819e-2200Q, 2.77712159414975738481258376000179218e-1713Q, 8.84696668578869367896605187021695554e-1334Q, 3.02118052104884375490712624043751719e-1038Q, 4.14765218378536031502305059306013033e-808Q, 6.68597993758855914851785488336018106e-629Q, 2.33347816130102648013054297856813910e-489Q, 1.04733564854277330060008344877597147e-380Q, 4.11557211597596026781639255888321135e-296Q, 2.96892493585484037081544967750602740e-230Q, 5.48318261457236849627593775240447695e-179Q, 4.37952763140247483545256113532792419e-139Q, 4.93764851567154537678682785150822008e-108Q, 7.12882559733493186476365592485215882e-84Q, 4.41155760043873077928273281441139888e-65Q, 1.80163734340122138141929999274331671e-50Q, 4.08343162992111053691005442960554881e-39Q, 2.69713878716183124254075973537323621e-30Q, 1.88906085681027307067677306620940234e-23Q, 3.83661096593349300206254286709601564e-18Q, 4.94655966889556498072463502849060578e-14Q, 7.45399353944412484667368844933083088e-11Q, 2.11213594306352633431635806968088815e-08Q, 1.63019937992045999824523554635197574e-06Q, 4.57540432069617055539176306478504351e-05Q, 5.85085995568316321646161156635372880e-04Q, 4.06605991446724517473851752061784970e-03Q, 1.76336759267286733236155605055701792e-02Q, 5.32246038588641268382381208383524691e-02Q, 1.21978563400315778588175663550724559e-01Q, 2.27813822026525499121180105863893077e-01Q, 3.67880940693987971581072989565083693e-01Q, 5.40739096387634225611080390746760034e-01Q, 7.57649154875166910544734982341932295e-01Q, 1.05508672843049871100505526038205526e+00Q, 1.51442893723856346518379289394390563e+00Q, 2.30704471829033068051765367243404246e+00Q, 3.81265333029628098807578829102732392e+00Q, 6.95531454976623073962406546394064472e+00Q, 1.42580921706810654107345317832413199e+01Q, 3.36149768965581810654207836598212633e+01Q, 9.41442967189932119013509095554254321e+01Q, 3.27468727748159184559404098660493044e+02Q, 1.50134197286985544688570624625182907e+03Q, 9.80710571996542847233767661790386013e+03Q, 1.00962016775294251580231030319396563e+05Q, 1.86536785996695038528848904417900709e+06Q, 7.30953564053636331147337800282574617e+07Q, 7.52773592822324283583724791889200862e+09Q, 2.68310388335555167663870380176682102e+12Q, 4.71267149903232936486329698993101877e+15Q, 6.42020070484263570170339986066376457e+19Q, 1.21445858766272510022421711137530603e+25Q, 6.73690008798356003301023318149224184e+31Q, 2.86191858315711641955678894569568040e+40Q, 3.19319569382394077525802448772198805e+51Q, 4.55425948316978466116193749735949380e+65Q, 6.33381334198036002776230152386194807e+83Q, 1.16685549796591838612114357506921339e+107Q, 8.11568192390745193863805331868574764e+136Q, 1.57270137093366013936347867540038862e+175Q, 2.12621037868720966375758237292361135e+224Q, 2.41043281776111273672384537919910315e+287Q, 2.06426392024598778796244702435819107e+368Q, 1.59767188945824373238194899871319514e+472Q, 3.70147206080013271017541277477470426e+605Q, 6.04524081068361649977908949693985827e+776Q, 3.91244551907499740549590413818514135e+996Q, 6.37601410091330839292544093678958764e+1278Q, 1.38423256121282951104859278958281990e+1641Q, 2.29003319158044389177689218691755697e+2106Q, },
      { 1.06947516637266850888077780365944975e-2342Q, 4.00222795925000100954191256688394740e-2067Q, 6.14269929806784835508930822317395965e-1824Q, 2.47129563578700314331274388349812780e-1609Q, 5.94856549925636496124140508697277691e-1420Q, 7.88155582901689454838569820447222612e-1253Q, 2.37169815411747688234160253328737061e-1105Q, 3.29576869517934854144272957025066495e-975Q, 2.30295331039392476672974731888405196e-860Q, 5.07778539549297920546786995561355024e-759Q, 1.36309792114536681315299788580123700e-669Q, 1.11902249499811588919802641601624761e-590Q, 4.83171643219834730998170916090452646e-521Q, 1.35092956692688343220060283908963635e-459Q, 2.24199581450623280210827812157346751e-405Q, 1.56044428013608715771945922727966521e-357Q, 2.55762668699602425668170610426828633e-315Q, 4.52598451420085229059456309653543845e-278Q, 3.31498525601312348031091223503291312e-245Q, 3.28982733272263633196035888826313749e-216Q, 1.25979671796942324763554469354410763e-190Q, 4.68781195667796137227478889031902638e-168Q, 3.82963817403632291960149514795452273e-148Q, 1.41008655666469634743601323447962987e-130Q, 4.41480300426527477757453266178330825e-115Q, 2.05796976085320113157119878898298467e-101Q, 2.34167766563934625382241547560196138e-89Q, 1.00611330011990340950410787621048434e-78Q, 2.39886979987338732623660845760149501e-69Q, 4.45828722999844038302160563656015401e-61Q, 8.71674689717785941226626165471518297e-54Q, 2.33608458451638324303027615256535578e-47Q, 1.08390303338972947113853753614820116e-41Q, 1.06995151808257796332112367831607337e-36Q, 2.69524358925375181073032551986305924e-32Q, 2.03423384800497240894806821496090050e-28Q, 5.30019940271628291011490416486872002e-25Q, 5.40212919469588209381281396175343198e-22Q, 2.40511019215192441365347431063245533e-19Q, 5.15573611643522185168933155470786681e-17Q, 5.79902272788904112815174301897601165e-15Q, 3.69205735629667547626167545850806392e-13Q, 1.42267754195891351150562998109337966e-11Q, 3.51991945554992222676432065045765653e-10Q, 5.89117093052592385410771587163177426e-09Q, 6.98414016831114732860587596174055273e-08Q, 6.10836626587512983932161479872803914e-07Q, 4.08537274605429873510095058624689151e-06Q, 2.15678250632197565791431535444473324e-05Q, 9.24312223411418671168580362381399190e-05Q, 3.29629252828970123413244980995397823e-04Q, 9.99865029818046920794114062146177746e-04Q, 2.63018698849220190957883149658594289e-03Q, 6.10412221855424181881652487802655548e-03Q, 1.26903202004975552477255319538074005e-02Q, 2.39564299641088688046037610330961560e-02Q, 4.15643030399701933649627965940349917e-02Q, 6.69982939046328122378009075425825174e-02Q, 1.01314761859197945164098555651429523e-01Q, 1.44998628043994675224702383164439509e-01Q, 1.97976367297349804840344474368395293e-01Q, 2.59785494162963270697436696871054011e-01Q, 3.29861530130123082323166204892104315e-01Q, 4.07887707779851847083967462215852975e-01Q, 4.94160808801609892592520472039421040e-01Q, 5.89951040448037491768065148251882226e-01Q, 6.97865729437412689560491745121349024e-01Q, 8.22260217393657822959055682985146030e-01Q, 9.69774604378855851935565350832453271e-01Q, 1.15011795553624730201138456447827419e+00Q, 1.37728513637309570900453193107529701e+00Q, 1.67149809630206531099733950813918712e+00Q, 2.06235275406513270776777022052056344e+00Q, 2.59399765677200896819549138553557629e+00Q, 3.33381238373592320485907218391630985e+00Q, 4.38727553184963415451944389447389070e+00Q, 5.92410234721624433989387409845478071e+00Q, 8.22555400895276009470529071921123608e+00Q, 1.17728847794365554942833299884164414e+01Q, 1.74186946732944479219083797984245546e+01Q, 2.67321347727075416316453470949013980e+01Q, 4.27248599090065202592682231550667436e+01Q, 7.14514957498311763066605425195361350e+01Q, 1.25726649744291050649669245834238885e+02Q, 2.34268740301195719755233273502547873e+02Q, 4.65674501968291917847819643280019736e+02Q, 9.95892168086429019721630255623709337e+02Q, 2.31370536252976840857940169336286210e+03Q, 5.90416785827987120394897587933071506e+03Q, 1.67574959687797819327588127121232764e+04Q, 5.36589048987894263464971617898730248e+04Q, 1.97001995592318850388501435517959975e+05Q, 8.44594440147401724253246450884183480e+05Q, 4.31706343383048349864263584742746589e+06Q, 2.69343047021169619975848244859698568e+07Q, 2.10656056871936774519731164741615606e+08Q, 2.12865839525415045164207708849921266e+09Q, 2.87579863694102104115679841861873156e+10Q, 5.39962441480030320721171298079837788e+11Q, 1.47228407211216271679535227733468003e+13Q, 6.12702071280434890818621133461421119e+14Q, 4.11734705402761288562221821245591184e+16Q, 4.76248383365979073253985584808588348e+18Q, 1.01935525113816768714253607040570786e+21Q, 4.38231774892874881629281035002570378e+23Q, 4.15257812330163302032181919338837514e+26Q, 9.63578634121366174238646117049249945e+29Q, 6.16903528716345189143890155825118447e+33Q, 1.24741073700366469803472774007991980e+38Q, 9.28495127856215607061013544719468024e+42Q, 3.02620640278402325128627613072607942e+48Q, 5.25740318414851642612290423653366751e+54Q, 6.08377705676929998408087326671143748e+61Q, 6.03621188684706784050917948535663342e+69Q, 6.83604430657324601596912909835543184e+78Q, 1.22207736059289881632444543839493409e+89Q, 4.97963284347586492335598341460012283e+100Q, 7.01286464121514720750130754209620950e+113Q, 5.47091248474436718447273702781160082e+128Q, 4.03491215911361460129480089373280294e+145Q, 5.15550681905479930138211789209766809e+164Q, 2.26698817812971993667638253759384337e+186Q, 7.46678442214100282100191345089453705e+210Q, 4.44695748246229233245164503407419774e+238Q, 1.29997993210113625215904908209705899e+270Q, 5.78362417210637255304620183172670719e+305Q, 1.41167148054040165920786770540444441e+346Q, 8.08286570061195033556699134866983755e+391Q, 5.63304784971987527997364020512160855e+443Q, 3.08690662195072763023845781932813249e+502Q, 1.10165417416868743288711300669064477e+569Q, 2.80994196635775688984524212097397961e+644Q, 7.73382325928152223721046490397169922e+729Q, 4.97761231747547736253787393388911251e+826Q, 2.44530978920516083381477794744261645e+936Q, 4.76039467365174741738996705683688855e+1060Q, 3.22588883912409917905940907837482370e+1201Q, 1.21298392121243707893541014829026239e+1361Q, 7.92540396126882607375372502759559361e+1541Q, 6.05643048475311304903673843949239315e+1746Q, 8.67085957383156175019676264069447501e+1978Q, 9.95105409544177778456070528537389733e+2241Q, },
      { 3.82052466785182332086738926018596976e-2417Q, 1.53216690964008904863800245911834695e-2270Q, 8.02799618255145245433991916259679940e-2133Q, 1.89790062782857972828467648673787073e-2003Q, 6.48547054333991583933133337347105329e-1882Q, 9.56340910858469468709935501045338939e-1768Q, 1.70023842699605420504066989530814298e-1660Q, 9.56797730413407205631782708332785103e-1560Q, 4.22020698781412857368210883093245672e-1465Q, 3.41965451589094380116779072034053750e-1376Q, 1.13314142099471085666864638284317139e-1292Q, 3.25613842818879565325184054056113549e-1214Q, 1.64407833236659941146136019432680305e-1140Q, 2.83169222847758408789499219075138705e-1071Q, 3.10256729477751738834780026930224437e-1006Q, 3.88327366035249790364432313659702349e-945Q, 9.62325937366412073215674752789621450e-888Q, 7.91532304589191166500084311712723764e-834Q, 3.51089649105332123383908279313304145e-783Q, 1.32489192907121558509586493733802170e-735Q, 6.52783588426344440105254553576625446e-691Q, 6.27953766456679638940801082204333268e-649Q, 1.72112316179301192237697323835800001e-609Q, 1.91704593421434794371085034440245942e-572Q, 1.21131389706331685590249812201530647e-537Q, 5.93984004695174374389288417212720344e-505Q, 3.03411224655952005849867823710301306e-474Q, 2.12875962525689396085894169728448953e-445Q, 2.66000529174336615171418420545198600e-418Q, 7.55590515004675433995195831838744150e-393Q, 6.13626926542527991664012204881781789e-369Q, 1.76713467741974813886350949888126351e-346Q, 2.20926972666826146585381131914638666e-325Q, 1.45005559535857920730397373363607727e-305Q, 5.97336892939272666152938174641382891e-287Q, 1.82640350668529645297131829899777179e-269Q, 4.85229717585781075750778298691866187e-253Q, 1.29884069259764069953465811099712413e-237Q, 4.02542740473423831131996110176920188e-223Q, 1.64606390223494048083381164318527223e-209Q, 1.00404640707773475495017963404550319e-196Q, 1.02517549226209525963922466256750740e-184Q, 1.95259262992798399878532201786889482e-173Q, 7.68026595299128847616027073204274409e-163Q, 6.86431799704342420135947045710241387e-153Q, 1.52498555897006686273111644122247925e-143Q, 9.16240838899174700108797638312124662e-135Q, 1.61152978600632900525315532785259119e-126Q, 8.93873956545614240431663830324423008e-119Q, 1.67683199253457467411487735035598912e-111Q, 1.13606831265305889540738444180574550e-104Q, 2.95677983624992268135074874283992128e-98Q, 3.13261928574067484199912376683710488e-92Q, 1.42665699792617318967988994714464344e-86Q, 2.93948527551792820481373266955543872e-81Q, 2.87496940202324055984269030763057181e-76Q, 1.39640503864001278539518135367012235e-71Q, 3.51418022897052500616248358954553443e-67Q, 4.76840843576304417158706032926840904e-63Q, 3.62171076308676895946675701059866747e-59Q, 1.59482985688579594373676029076259231e-55Q, 4.20842453488002122572360942178919587e-52Q, 6.86443229233076886165541867675398494e-49Q, 7.12571665807519317291983672203309613e-46Q, 4.83819502081497047116760866720006226e-43Q, 2.20465542430951342610708115485440884e-40Q, 6.90709560806486502312714752170868272e-38Q, 1.52197218506174728353062647705743598e-35Q, 2.40955219490267088425869662208370235e-33Q, 2.79630904534258562440727543829590381e-31Q, 2.42396861904265607424793591553560695e-29Q, 1.59749866280800688175296781099864391e-27Q, 8.13840485655638493122774039812902482e-26Q, 3.25536762868063353648331484818545825e-24Q, 1.03750216774182187079375467840556085e-22Q, 2.67108014795025059201469276708831348e-21Q, 5.62745845137509901846677188959226735e-20Q, 9.82072302589238577426509279192638536e-19Q, 1.43594941796544038692542800708134381e-17Q, 1.77810682024353573557752337941414159e-16Q, 1.88354537738694939365287613122504842e-15Q, 1.72308010102740811988992770129939744e-14Q, 1.37343705888395103661321882765344100e-13Q, 9.61866975437486408023966746429500914e-13Q, 5.96531965279554928068743818930854269e-12Q, 3.30041201040702869600996143071210790e-11Q, 1.64031748053937249488182380342568737e-10Q, 7.37124149693192472685858181626175497e-10Q, 3.01344400817654411848957462000254023e-09Q, 1.12717586759651920251576226824261590e-08Q, 3.87857240586881913126338809706852880e-08Q, 1.23397923410236586482402725064781148e-07Q, 3.64729321175679321144258676923993113e-07Q, 1.00602028308961790056200899477538690e-06Q, 2.60043020837597212496425152713721292e-06Q, 6.32419483196640694034831908133223537e-06Q, 1.45245521130769448827445571183003177e-05Q, 3.16123436155465446648693209598810124e-05Q, 6.54176706996526406822348966226430775e-05Q, 1.29110196844657112529538619169285993e-04Q, 2.43733749771260888436783011892865212e-04Q, 4.41314232710451843972392755372463603e-04Q, 7.68377076370070526296257588743541811e-04Q, 1.28957360159046548970572233319638690e-03Q, 2.09099199558542466132447951593809118e-03Q, 3.28264889533211879885382599961585359e-03Q, 4.99964828308048181967481389975237633e-03Q, 7.40157019965966236427260197490815304e-03Q, 1.06698107000950941317328124490909795e-02Q, 1.50028172314973599445873815719412949e-02Q, 2.06094173096225141746698709598883056e-02Q, 2.77006834377238972506888931261097291e-02Q, 3.64809756186562309685025492261970114e-02Q, 4.71389254316798953989957737554750583e-02Q, 5.98391571230828379216832783213494747e-02Q, 7.47154814906505012222514984217472816e-02Q, 9.18661612946071289937159187843364627e-02Q, 1.11351656134035569001793372114249223e-01Q, 1.33195038632804266520840073116724199e-01Q, 1.57385060631367271607409153973946866e-01Q, 1.83881661881487488373471043449811640e-01Q, 2.12623571664368840241407884037664753e-01Q, 2.43537565151706738569000265542366947e-01Q, 2.76548903119165441078310983457425711e-01Q, 3.11592501651099485071285279910294521e-01Q, 3.48624439429573943493094167043683901e-01Q, 3.87633503629295959856799538075546008e-01Q, 4.28652590594010568395685064250714546e-01Q, 4.71769904763931628640496795425394984e-01Q, 5.17140031351419396580841715288928105e-01Q, 5.64995090385812394527782017926592446e-01Q, 6.15656310347513453519162300561374394e-01Q, 6.69546490104796171397630195486014552e-01Q, 7.27203952634969644713241799501762825e-01Q, 7.89298740343220248874987146571520149e-01Q, 8.56651969968232039115451358927023892e-01Q, 9.30259468685761614462731685806948992e-01Q, 1.01132106970032064396659635282297935e+00Q, 1.10127727814330022383089577768540741e+00Q, 1.20185545627576044892754956074556492e+00Q, 1.31512826035991923618357153643853889e+00Q, 1.44358784334344214073911010244139658e+00Q, 1.59024039033833533691274024525328135e+00Q, 1.75872697808494229887678505228447105e+00Q, 1.95347868511083813992081514830258545e+00Q, 2.17991652311273637078595706271805585e+00Q, 2.44471039181719695746189348680321367e+00Q, 2.75611627927753518177812012686624980e+00Q, 3.12441791418753602036992277268322970e+00Q, 3.56250886504706839116501560040688156e+00Q, 4.08666490215568913223544572173213403e+00Q, 4.71757610938540508471197077535051869e+00Q, 5.48173646271881799502596740262131127e+00Q, 6.41332945820485042568468273867868285e+00Q, 7.55680806548694121454415120092330575e+00Q, 8.97045530296518503588356381213781124e+00Q, 1.07313427967993620782556935698599286e+01Q, 1.29423018529722651121181556330024768e+01Q, 1.57418213494311260952898800974816324e+01Q, 1.93182476307453478113061967921720372e+01Q, 2.39303783823625958630927317369382100e+01Q, 2.99376708353783067254755016259184880e+01Q, 3.78450834852449540061461594823397372e+01Q, 4.83704762272558588741438572877597086e+01Q, 6.25472526597377774327448466299291048e+01Q, 8.18828352821743059070051234981039846e+01Q, 1.08606901707010877244713106575898588e+02Q, 1.46066165572767230787668444759129144e+02Q, 1.99362305840947908376533577788614663e+02Q, 2.76400238552833065835531188582929358e+02Q, 3.89641361583293015149089847924625396e+02Q, 5.59090899610510721528958323850657754e+02Q, 8.17488717203324414010281686447524861e+02Q, 1.21951707162988010780952106506178041e+03Q, 1.85849349228255485567535180727353270e+03Q, 2.89733723527987926224598061281533634e+03Q, 4.62742546807418291953673117361684539e+03Q, 7.58336312821976325886140349375631572e+03Q, 1.27729327383211423049145373221610484e+04Q, 2.21512103826316975884795094360035657e+04Q, 3.96282043351341952537133210137390141e+04Q, 7.32802430573798143066819092764851057e+04Q, 1.40370956832174099704802040273767653e+05Q, 2.79169596050238213326100612230710484e+05Q, 5.77851587758831222006931720166846758e+05Q, 1.24809297513500168708148829224766608e+06Q, 2.82070529249367448006501528915158547e+06Q, 6.68996112716468438709724989481100128e+06Q, 1.67032788479232576626369550400657265e+07Q, 4.40490689805489416603930530460183080e+07Q, 1.23130681270188214533524272282450985e+08Q, 3.66207397185135919222246119544674802e+08Q, 1.16348659359258561581772655083768774e+09Q, 3.96573293875598360514367483348534276e+09Q, 1.45675716212887953801924286655056170e+10Q, 5.79499965416005488651946885694232582e+10Q, 2.50933409077965035999243371354527615e+11Q, 1.18927611174028691004735513584719908e+12Q, 6.20525591975150642720259081452168385e+12Q, 3.58662837399254785268923887857175768e+13Q, 2.31171019709164125040709702173384790e+14Q, 1.67323267937848597801576240028274938e+15Q, 1.37027502568098828908714302508292741e+16Q, 1.27982243687884271019103805104886411e+17Q, 1.37488860693662981360604951513000987e+18Q, 1.71428840498039054019366079813545633e+19Q, 2.50480806231532255799256901000499530e+20Q, 4.33295295852175693203040531656862705e+21Q, 8.97105957110885650118812876460940777e+22Q, 2.24900305994354872692347113786351015e+24Q, 6.91168391281314093810281357471213740e+25Q, 2.63834638817928808592115398804015843e+27Q, 1.26855240154452496477895137567276885e+29Q, 7.79793937981300078349411089481829644e+30Q, 6.22628875244383647460445596438511369e+32Q, 6.56725010457698317225350638943220187e+34Q, 9.31627142136562734426709726532855891e+36Q, 1.81178764804393998725363596058882739e+39Q, 4.92965709962256757369004572197380439e+41Q, 1.91768799703732643541855471608092309e+44Q, 1.09145348658581711775073448931123050e+47Q, 9.31447898399194268806130290915959528e+49Q, 1.22344767896866261290498650189275522e+53Q, 2.54310892512613676637994972961780173e+56Q, 8.61698733620595754888079045068786616e+59Q, 4.91184107780000171015313299477683427e+63Q, 4.87081518596258225878777239561036595e+67Q, 8.70837775558769802630502082888045797e+71Q, 2.91582292448921588681558819522688100e+76Q, 1.90391730055994678151666350663451590e+81Q, 2.53108226877386875309925515429807108e+86Q, 7.17216745327677632991407821508792163e+91Q, 4.54861980767233963821254711174889633e+97Q, 6.80080274478233195724672872122891978e+103Q, 2.53345776553427904264312812304869400e+110Q, 2.49408335416956941404546574182091219e+117Q, 6.90831393215899350999406217643925461e+124Q, 5.75535983268412076946168827231484587e+132Q, 1.54830478033444708124258729693098368e+141Q, 1.45063275961171552593742069490051326e+150Q, 5.13010920252030309322157027524520131e+159Q, 7.46051857949218271625227134136784373e+169Q, 4.88752936708905952864518859469811383e+180Q, 1.58944916201304126296752123375100638e+192Q, 2.84523891966215233479427916942262824e+204Q, 3.12952175960493682046861415263020503e+217Q, 2.37781288259852734112773228061886887e+231Q, 1.41367934226264567696479785830917722e+246Q, 7.50969021632501856297416076380790606e+261Q, 4.10517599617698914085666788248374635e+278Q, 2.68396648103470580650904518179723689e+296Q, 2.46300967872463438914762759549033346e+315Q, 3.76174998591379278617806685202641214e+335Q, 1.14633505463399943022299924904504898e+357Q, 8.45422653379120782973725460577460520e+379Q, 1.85322036769366179986950823638998209e+404Q, 1.50271794914844405931928670543344206e+430Q, 5.68929995100375677797682848265066614e+457Q, 1.28862756145762463302669091988538860e+487Q, 2.27343599851542201859662096966470860e+518Q, 4.13728378647859018626664278751490836e+551Q, 1.04732962553987345589103224110962753e+587Q, 5.07014442635227334186619020470235649e+624Q, 6.58680466182542570069097938912153401e+664Q, 3.29371230247950075139843014589562364e+707Q, 9.30666782392729638823808617314515754e+752Q, 2.23612238313938342458217832467289997e+801Q, 7.05882324392654993625063412318352329e+852Q, 4.65192550274585790446586136162724180e+907Q, 1.04784460655109112066824552095698697e+966Q, 1.36343029663707923380723270729436320e+1028Q, 1.79162856963805080764174564008589158e+1094Q, 4.30919215868732016052648412277675057e+1164Q, 3.57261639332919274471338835639300206e+1239Q, 2.00290113950626521919894557483624467e+1319Q, 1.55571074565195195355910058210877816e+1404Q, 3.59251763773431142332284400696144078e+1494Q, 5.55982146221013438636549073847422510e+1590Q, 1.36984614652427999087777938149635656e+1693Q, 1.34965612662661283517635152647662065e+1802Q, 1.41743602676194002368496649429520279e+1918Q, 4.50570145843130891260063218260174869e+2041Q, 1.31668432261399763150332726270146708e+2173Q, 1.15414484031205360166874295729734758e+2313Q, 1.06861618403043050305720276519487893e+2462Q, },
      { 2.99827136671822203139659705254947108e-2455Q, 1.24882827026046387418324170521821293e-2379Q, 2.44988423001973909940876922057761351e-2306Q, 2.66927540888423924256197676379835451e-2235Q, 1.89512300782142727523964560129359997e-2166Q, 1.02360851047736978252291085443458258e-2099Q, 4.88730957021167285814599122540648196e-2035Q, 2.38575973739960063984613176811112623e-1972Q, 1.37101430897591401677957283544539320e-1911Q, 1.06333264218828160158212770967007648e-1852Q, 1.27067801926958817891969964688113127e-1795Q, 2.66010333678426336904594890897508436e-1740Q, 1.10484167956085896672647638913688653e-1686Q, 1.02711572445228877679664077673865962e-1634Q, 2.40228312498468537472913439313888633e-1584Q, 1.58314329557188545251248547330891601e-1535Q, 3.28094781159949863603124081342946968e-1488Q, 2.37840601013012306745513950331950428e-1442Q, 6.68624945754691841776166731458196006e-1398Q, 8.05587986859135128455444199209469004e-1355Q, 4.58316096287519065423952255855034306e-1313Q, 1.35248343166964409448147921555689922e-1272Q, 2.26752160966952001157168922897574770e-1233Q, 2.35909278572023604645731071940961914e-1195Q, 1.65903351294385543451470844717805067e-1158Q, 8.56804094411838080986008744972959423e-1123Q, 3.52140713988280887124946897137042691e-1088Q, 1.24502439015653832112887580419259567e-1054Q, 4.08361208615792538039018647193947162e-1022Q, 1.33685935274460284904788097419890160e-990Q, 4.68914584779066055505548198640894341e-960Q, 1.88761464755010821415650640440428715e-930Q, 9.32118389977759099804089947706407268e-902Q, 6.02287072293155834327191570771930380e-874Q, 5.42106639858656175956621834625906078e-847Q, 7.22191826970524227339284882890548040e-821Q, 1.51020072122639353372149509671977024e-795Q, 5.24775126181523166368024963040824842e-771Q, 3.20220580014087429159629013136223785e-747Q, 3.61997704502523196516304924921584345e-724Q, 7.98492800624004092804929885145612391e-702Q, 3.61392988815279171930655913638563273e-680Q, 3.52366894930304180660437805611575996e-659Q, 7.75941790954346674475792064117877807e-639Q, 4.03982705605992279012386676527754690e-619Q, 5.19834669028776461984659088482903679e-600Q, 1.72588931588477933787290173538689408e-581Q, 1.54137268468058258004178745033532316e-563Q, 3.85562019823508006565291873935904091e-546Q, 2.80916830174896024691081755907263354e-529Q, 6.19212992045820718698085583700822639e-513Q, 4.28406369726641450642830646435371014e-497Q, 9.64070107276137860266813577293097959e-482Q, 7.30474502980487251348070249880206420e-467Q, 1.92703875472217605577664104239882285e-452Q, 1.82836317722867439282495454446115074e-438Q, 6.43850535226391166629598797094394826e-425Q, 8.67564527079512826366041439069686406e-412Q, 4.60732378382987117665981750506044471e-399Q, 9.92355019653945096209219622811507181e-387Q, 8.91282113369599082908019662526444168e-375Q, 3.42910268669860807072991706793596262e-363Q, 5.80083861220020883214825204353252963e-352Q, 4.42512315593917892307328787082985460e-341Q, 1.56000404760553377203351190297454435e-330Q, 2.60259125151860978088621742839596354e-320Q, 2.10262994916038774260354592873630300e-310Q, 8.41173460487346663387701675407432002e-301Q, 1.70280612724644563266907394311476267e-291Q, 1.78116409086610297225917597830385624e-282Q, 9.82480313804105048460709403983300504e-274Q, 2.91457791899830525764575140294420231e-265Q, 4.73965774855838520901565109021450612e-257Q, 4.30398176296336788976377718753404885e-249Q, 2.22193567988028967419035853512154624e-241Q, 6.63553772275767193165934337486647504e-234Q, 1.16578412206922730540647811503608703e-226Q, 1.22474675486304476176402962758102117e-219Q, 7.81684108965031293490823429466674471e-213Q, 3.07774321188900147460180897292349446e-206Q, 7.58759251414243265302591311385078772e-200Q, 1.18823395876906229284835410783359514e-193Q, 1.19864235179097517931485807315756884e-187Q, 7.89484815058732018316693655376229205e-182Q, 3.44001585820470930356951346759790116e-176Q, 1.00429282413458278190733306028479555e-170Q, 1.98881516262612704197372631141044676e-165Q, 2.70364023416269358290386889178454878e-160Q, 2.55241036356528886321774073237366662e-155Q, 1.69227628517124062919375344585284149e-150Q, 7.96587271931569005985245219884752131e-146Q, 2.69038016965415710141558028398178580e-141Q, 6.58640142296301821553052201959182941e-137Q, 1.18042002159083828066729019346791192e-132Q, 1.56367343741949029569249898774237153e-128Q, 1.54531048534737740803236941099285613e-124Q, 1.14963627239221457328345144952502533e-120Q, 6.49493107141223206477306791588740146e-117Q, 2.81018975962531457950258683552245790e-113Q, 9.38876009983047538463236277234608306e-110Q, 2.44144614978077332940218030178369234e-106Q, 4.97967380423964535764132428105410190e-103Q, 8.02629250855504170950085275141889693e-100Q, 1.02975774422556528975197839046564511e-96Q, 1.05903428462392788554333849690582167e-93Q, 8.79012224539705420215807394822091933e-91Q, 5.92725904620589386119574720007659266e-88Q, 3.26785695641876626125438772016710733e-85Q, 1.48222090912512196733576972776187035e-82Q, 5.56431172687041304333655416134228735e-80Q, 1.73894644850180973193834439393406052e-77Q, 4.54974501627115811330277659341709811e-75Q, 1.00204791018402181301004224966380115e-72Q, 1.86763466487726841127561204857118883e-70Q, 2.96092907372076963741238677845263773e-68Q, 4.01290356278003207547213477434374557e-66Q, 4.67174973180940285995675266523420122e-64Q, 4.69376738484344030975102668254954324e-62Q, 4.08839867480777582716177965835471551e-60Q, 3.10086294017966876521788239138705860e-58Q, 2.05664262860193002265928563040484563e-56Q, 1.19777269867460483717671801459380654e-54Q, 6.14987857896674930479991965502306936e-53Q, 2.79459594105487367405294144558243202e-51Q, 1.12814222153195027375952194262805170e-49Q, 4.06053088698370288660361299969573584e-48Q, 1.30770277764601303984357205166621959e-46Q, 3.78118998998858848129267831928491324e-45Q, 9.84875712554165931773609676827898732e-44Q, 2.31826871061275836674260034702071180e-42Q, 4.94684766719278736886703293137447413e-41Q, 9.59814033368779163526953048426266162e-40Q, 1.69828465632158908850726775955673984e-38Q, 2.74808806067694979431131601699073400e-37Q, 4.07794734980576448596813627579221148e-36Q, 5.56417400808680411162969051547065270e-35Q, 6.99889732124326604821939972498753725e-34Q, 8.13608686966403922600365716655177400e-33Q, 8.76218322965140584555660990677761542e-32Q, 8.76276637192578280309577248461260176e-31Q, 8.15628170980170063345810303188191933e-30Q, 7.08149199986036059275640615879247609e-29Q, 5.74736606938180421303095720610698091e-28Q, 4.36944079330436317602800446372377792e-27Q, 3.11795190731786551713306066746724497e-26Q, 2.09240762901978141675541589343048983e-25Q, 1.32305299259448285811442078319939693e-24Q, 7.89691997711578359267081292526106870e-24Q, 4.45716605711992632203887297614028920e-23Q, 2.38302054707699751676620265710858837e-22Q, 1.20889613263470803214753580918237251e-21Q, 5.82833462566546297036431040060608322e-21Q, 2.67469784973934035752489871481337927e-20Q, 1.17014393860453605367131279149462226e-19Q, 4.88739480774243667158625429475089610e-19Q, 1.95168062031382677617975635339602042e-18Q, 7.46163208304186839076058133462415646e-18Q, 2.73485791500251557979443670181385630e-17Q, 9.62223074873981898938153222962813147e-17Q, 3.25389617800670808719193210653774700e-16Q, 1.05888451003262711834291997535706834e-15Q, 3.31989417456924550647348023471492055e-15Q, 1.00398818028880718004903688096398433e-14Q, 2.93186753434992804142094819281287207e-14Q, 8.27635883877837412701034478711895464e-14Q, 2.26082510653047710401828799240495861e-13Q, 5.98228146965673437508140635644104873e-13Q, 1.53484657642706271558550260287175188e-12Q, 3.82185576688620308819449258941130914e-12Q, 9.24474797424185856221360345157220406e-12Q, 2.17427909729908200066806520833424743e-11Q, 4.97635090813511876195380613765617708e-11Q, 1.10929412007484858271399815868498117e-10Q, 2.41032585809105707126681740366639425e-10Q, 5.10905559690208602233885434283852003e-10Q, 1.05723928928894498681155298267174319e-09Q, 2.13744795688281626795384543565327267e-09Q, 4.22492806022042378177884800843897317e-09Q, 8.17046853836402216132200964074977087e-09Q, 1.54693324186061707441588930749976281e-08Q, 2.86930859292637487108510416894642893e-08Q, 5.21722334965282980403485300265738444e-08Q, 9.30518593041943674159836371480799295e-08Q, 1.62889442240265382985512933741863589e-07Q, 2.80023159222713498193152226335713707e-07Q, 4.73011069732904671695491353591017402e-07Q, 7.85526363421471709139200654941264315e-07Q, 1.28318979177475296348420727772184965e-06Q, 2.06292423671439573062357518920248604e-06Q, 3.26552286428871054459476770292453844e-06Q, 5.09222413107163744070769490625113325e-06Q, 7.82617466349549247600633222001776503e-06Q, 1.18597235737360853531010814128200937e-05Q, 1.77285167845861097132511756107809923e-05Q, 2.61533347347083551795448324767635695e-05Q, 3.80905294601878233958593851046717171e-05Q, 5.47917574609632216617322114415284106e-05Q, 7.78734661207721580410532787755478685e-05Q, 1.09396271310786841597436260479638662e-04Q, 1.51953965111990518244589142984584154e-04Q, 2.08771424329052859462812228725402767e-04Q, 2.83810167897154635400111152886276561e-04Q, 3.81879755641776745662197699923565149e-04Q, 5.08752658539288473040921063606067979e-04Q, 6.71279691879016478989727452815073290e-04Q, 8.77502619869410964596211481799673429e-04Q, 1.13676014562695660388935295197579588e-03Q, 1.45978322435393926290231969328136083e-03Q, 1.85877550576562291461373383997358853e-03Q, 2.34747474413929171623192976971642926e-03Q, 2.94119122257973574562047404684570218e-03Q, 3.65681962518947136825693675506580632e-03Q, 4.51282135037802008049500003706382224e-03Q, 5.52917497487431572466643627095469859e-03Q, 6.72729342693880289247231232743320046e-03Q, 8.12990737739402925191380506695287882e-03Q, 9.76091537148098089988167779205704956e-03Q, 1.16452022628003896766716791241908478e-02Q, 1.38084285310255098081489577923406060e-02Q, 1.62767940269060299249627834088541599e-02Q, 1.90767805535439719611430291452241110e-02Q, 2.22348784290424057418829018676115193e-02Q, 2.57773027557106041232220835190089403e-02Q, 2.97297055256785242037107379259364943e-02Q, 3.41168899105681014303940274877700652e-02Q, 3.89625331103873045168982262850032695e-02Q, 4.42889239784348614272085648656341809e-02Q, 5.01167213062401896725837060595306942e-02Q, 5.64647381631055655160905842942135176e-02Q, 6.33497570613638646379858081046281019e-02Q, 7.07863799874088410337603891340254733e-02Q, 7.87869165286187403201277486249691639e-02Q, 8.73613124671846027274852311959507760e-02Q, 9.65171203330026184766273626679604986e-02Q, 1.06259512537229504582581766840088037e-01Q, 1.16591336871525032405196867775170242e-01Q, 1.27513213378027801743713746335418149e-01Q, 1.39023670890726644461647310110564995e-01Q, 1.51119320935163034926767802526948718e-01Q, 1.63795066161333054101624624583286900e-01Q, 1.77044340081249140375155906842322507e-01Q, 1.90859374591070988709995096163492901e-01Q, 2.05231491577749618625157075644473454e-01Q, 2.20151414838427133582099008248376327e-01Q, 2.35609598571509171617873703655361587e-01Q, 2.51596568823441496049692748064095951e-01Q, 2.68103274485319808338763876413696112e-01Q, 2.85121444718080293120944366000785895e-01Q, 3.02643950033175240463650928525241062e-01Q, 3.20665164656273759511495325736015581e-01Q, 3.39181328243896232934423095526947639e-01Q, 3.58190905496894238632513262989320923e-01Q, 3.77694942711148444854921008885136613e-01Q, 3.97697420816736709936441112125746846e-01Q, 4.18205604975383785180004107895309545e-01Q, 4.39230391326913892071217294036947977e-01Q, 4.60786651994838310108569244001462874e-01Q, 4.82893579976783682783259082280697090e-01Q, 5.05575036056380615456191240843198089e-01Q, 5.28859900380164343616138823034844308e-01Q, 5.52782431848141026163801711629569094e-01Q, 5.77382638973537667728623589540847426e-01Q, 6.02706666380887845415163251209767664e-01Q, 6.28807201638412861233038059400000985e-01Q, 6.55743907668438480076422889144751851e-01Q, 6.83583886557560546105565188637319809e-01Q, 7.12402181207131050122140520074497354e-01Q, 7.42282321936634874071712163942668772e-01Q, 7.73316925891616774766609626518222875e-01Q, 8.05608357929825762678392686079844024e-01Q, 8.39269462582114444255866071824225817e-01Q, 8.74424377728731780655872302934188327e-01Q, 9.11209441820152654384317758539800255e-01Q, 9.49774207830947999703385257169606442e-01Q, 9.90282578695719860661750084965348602e-01Q, 1.03291408077266220499949941381670461e+00Q, 1.07786529395310786342015831094913754e+00Q, 1.12535145943113425409772316889306152e+00Q, 1.17560828892019106387681852443264510e+00Q, 1.22889400231711865029290591255755794e+00Q, 1.28549162454200134646596760738351906e+00Q, 1.34571157661211478758710729669915294e+00Q, 1.40989460104228631141154693756330089e+00Q, 1.47841506752326819907071392253335661e+00Q, 1.55168471165732988584072993312632620e+00Q, 1.63015686749586045646523839878119043e+00Q, 1.71433126392888582869445858515841385e+00Q, 1.80475946586097450583979351681595524e+00Q, 1.90205105385821569910651758727035418e+00Q, 2.00688065090883085704250790253631297e+00Q, 2.11999592251508776978462701922381176e+00Q, 2.24222669701791864392841075957133612e+00Q, 2.37449537743872890090165532424204567e+00Q, 2.51782884491687412950125924770996677e+00Q, 2.67337208788498443977901451698681543e+00Q, 2.84240383149636938582597352632467033e+00Q, 3.02635448975787151687626271616744396e+00Q, 3.22682681985641084640836568780538092e+00Q, 3.44561972615851906798292101239468486e+00Q, 3.68475574257857058166416238846674883e+00Q, 3.94651281922700641882215705186324970e+00Q, 4.23346115586300426882702466682124288e+00Q, 4.54850596485993372358839139642384894e+00Q, 4.89493721528314838282250644869812537e+00Q, 5.27648761361579143482640951008870741e+00Q, 5.69740032348777653019331418706396336e+00Q, 6.16250822618479874280298612217822529e+00Q, 6.67732688781902305621720870171254858e+00Q, 7.24816384288680618425673773908490765e+00Q, 7.88224734633420137754997440176783259e+00Q, 8.58787841076847338040196250304371454e+00Q, 9.37461076170556588083589376387132811e+00Q, 1.02534643490360208202755813725095827e+01Q, 1.12371792943596952958405098972799626e+01Q, 1.23405186912073323038589589843390563e+01Q, 1.35806305843369735705552654775963352e+01Q, 1.49774818320198815684165550755584428e+01Q, 1.65543795252306978072060668000283508e+01Q, 1.83385993586213963662082373601202455e+01Q, 2.03621450242974894275535474068497776e+01Q, 2.26626685943754163096516986406459795e+01Q, 2.52845893136112912405786286347442878e+01Q, 2.82804576829875229843082739941801350e+01Q, 3.17126237580911006574531329780573620e+01Q, 3.56552839704483033860162655552985983e+01Q, 4.01970005016327611735412627281826315e+01Q, 4.54438126123299012690083228854751285e+01Q, 5.15230920902650058925899224137885947e+01Q, 5.85883374425407037894686827319101058e+01Q, 6.68251567023112069515824430636222454e+01Q, 7.64587608768192360600680527705062884e+01Q, 8.77633846894782777864438965135974549e+01Q, 1.01074175868700380244224060820937268e+02Q, 1.16802258994842453002226105258170771e+02Q, 1.35453898714114297683765730207366657e+02Q, 1.57654978967903747896778920825912898e+02Q, 1.84182405906460887204718250035971904e+02Q, 2.16004568481915370203941996632993398e+02Q, 2.54333702516246823953928460722661707e+02Q, 3.00693971336392029288524445812313026e+02Q, 3.57010397089553597741305454749296236e+02Q, 4.25725590115811669827894375931285872e+02Q, 5.09953725643224718975881101989299450e+02Q, 6.13684687521816216675628323432708789e+02Q, 7.42056139017496594867562213902074498e+02Q, 9.01718069398279102081695287947885045e+02Q, 1.10132394119371945146169423670051948e+03Q, 1.35219615071533062822615520086342752e+03Q, 1.66923291068630661586092129359067647e+03Q, 2.07215152332054272741081984137877196e+03Q, 2.58720328209038570262408045410782795e+03Q, 3.24955382874419473343636700904469745e+03Q, 4.10660860213453501373964026030452875e+03Q, 5.22268955795277619388623264624171874e+03Q, 6.68565726355089669998707553230747827e+03Q, 8.61635357331041935557199135249564277e+03Q, 1.11821636876213398227716019227207708e+04Q, 1.46166395927660444135310088609126834e+04Q, 1.92481111548503807855046302083949422e+04Q, 2.55417294787310992653570084425920733e+04Q, 3.41617486573493312682855974958999477e+04Q, 4.60645561151339657587241032816274431e+04Q, 6.26388222783949624156629031841856307e+04Q, 8.59184957335087735876745002390386986e+04Q, 1.18909441895224029432207119296169161e+05Q, 1.66095033938427884484826987675525104e+05Q, 2.34226252811038979343276539537003434e+05Q, 3.33568602948986258437654580586715228e+05Q, 4.79889988962864687601246140266254885e+05Q, 6.97663061660509733330079420786394691e+05Q, 1.02527964914474052697145671441285507e+06Q, 1.52363581455429196593138247075709981e+06Q, 2.29042801548317740734746692037333714e+06Q, 3.48424008370597272699707163872949391e+06Q, 5.36561882022124111785765119998290944e+06Q, 8.36799594465384427071057649003523823e+06Q, 1.32217203482688374239659605009398927e+07Q, 2.11738506478689471767346755297403608e+07Q, 3.43829654204789362343598103327212044e+07Q, 5.66383460344826705635075490736605924e+07Q, 9.46890531446099216978642309212033877e+07Q, 1.60736860537955754766368401888105613e+08Q, 2.77184337816824251216810756452423932e+08Q, 4.85821891491727553231947839400706241e+08Q, 8.65895043719996967916785083318862010e+08Q, 1.57023595787763814315717820663521931e+09Q, 2.89877916582589084595240112564380085e+09Q, 5.45082862939618857716273065316347458e+09Q, 1.04462776299019818355322697028773431e+10Q, 2.04161928476231748290058801874576093e+10Q, 4.07167362508726715447348035164126587e+10Q, 8.29154750329098975446073110291771593e+10Q, 1.72524569678310615987347680670953976e+11Q, 3.67042596452982637114653516749325639e+11Q, 7.98985690430384590881851382687868095e+11Q, 1.78088199348481822130071407591775159e+12Q, 4.06753710066430378340194042741373814e+12Q, 9.52719664341172412580158176782396629e+12Q, 2.29025392291311484677517351429897325e+13Q, 5.65517100257158446400477892790807623e+13Q, 1.43556057453169991412842468344287817e+14Q, 3.74966777273576618576956864828262686e+14Q, 1.00868013060119404846072345341681988e+15Q, 2.79711270328189457750729359218050034e+15Q, 8.00353033476527491290557180677857043e+15Q, 2.36538022352308798092653239020367364e+16Q, 7.22793756862980926623333746953184515e+16Q, 2.28603237125675384523138361921766753e+17Q, 7.49169357670736182787260118136384206e+17Q, 2.54681528722697062733706069581540403e+18Q, 8.99167123461421679854681325569917778e+18Q, 3.30088716836331393064955543786333400e+19Q, 1.26155602488854061815298573346753948e+20Q, 5.02601059129997078887664234449072840e+20Q, 2.09003840003334609133024432910535654e+21Q, 9.08427629135679092557541630988833736e+21Q, 4.13277307337650905633261831455054852e+22Q, 1.97077591472219550165099701845426740e+23Q, 9.86567192878194333593095251120628263e+23Q, 5.19253827231351201624600034890440225e+24Q, 2.87797813261600767124658244003026900e+25Q, 1.68253103834271529795445964171621893e+26Q, 1.03930300492804406447430223922613789e+27Q, 6.79491179196076158722692150701779494e+27Q, 4.71054472298412825215699352759734253e+28Q, 3.46906218798171940969795067049186910e+29Q, 2.71919469298029646397308102518695120e+30Q, 2.27308597305997987201797057125811269e+31Q, 2.03060816941963451979890589753656719e+32Q, 1.94261454794602808054773809715560372e+33Q, 1.99453642796409945684693816457728868e+34Q, 2.20275757078165507105373401305070760e+35Q, 2.62280693187648585243154230451397289e+36Q, 3.37503469494102278406378324431267258e+37Q, 4.70514262885548973786602859197635105e+38Q, 7.12449819522227214155183099218285736e+39Q, 1.17479491699687500984526531336188586e+41Q, 2.11529840618171125644793575486664802e+42Q, 4.17057423654484355854313046574264130e+43Q, 9.02996402249218455912158145716517353e+44Q, 2.15344195364580091650722699472214395e+46Q, 5.67373854091156226750041136308702235e+47Q, 1.65679493344512341491651081156308385e+49Q, 5.37959274142567387437479000715431452e+50Q, 1.94883090765131732574822421459033340e+52Q, 7.90409930117048365355172083360408902e+53Q, 3.60198039300535878643618051284879761e+55Q, 1.85117602089555214211892984185617326e+57Q, 1.07703344015399312416171835284404207e+59Q, 7.12190311508435674119313111798381629e+60Q, 5.37418888386167437844586074702336014e+62Q, 4.64729650833728307491157079699691435e+64Q, 4.62526710582644946748054017004844382e+66Q, 5.32182077764493049141810605179622557e+68Q, 7.11164697902038500579051989777466680e+70Q, 1.10899515910236266304919828966923200e+73Q, 2.02799605144484652148788128416483129e+75Q, 4.37097775363901059074163853783972656e+77Q, 1.11616878412036714566047843287003825e+80Q, 3.39514715649439557125494531010505893e+82Q, 1.23701982128373508607730444611285161e+85Q, 5.42965793175551379743384549971838572e+87Q, 2.88810817234216647705316265060164148e+90Q, 1.87304637361264791993354548498385904e+93Q, 1.49042693797246054384025095285834884e+96Q, 1.46460222682423294988692623274219021e+99Q, 1.78930667727185631768641895397399502e+102Q, 2.73657637241785643461962451320315667e+105Q, 5.27697387534476684848681508483720442e+108Q, 1.29244503566283656061209233007634805e+112Q, 4.05121786788653633019340698712667656e+115Q, 1.63797303768105580759099639209855651e+119Q, 8.61161786816897952522273022842847467e+122Q, 5.93663320306070547376727413186770748e+126Q, 5.41263435338015569490052234550484999e+130Q, 6.58482575653621278094067557171475624e+134Q, 1.07875660982114746476517100137547561e+139Q, 2.40241992462133691284825971033750350e+143Q, 7.34435324696612505273925787730222850e+147Q, 3.11318307060514113999896732347290570e+152Q, 1.84887752009365552806348833513339683e+157Q, 1.55493753489569904210242587934359215e+162Q, 1.87247311840034398087857290386878308e+167Q, 3.26561398822344966034897508874934342e+172Q, 8.34577318106206521818843721578177275e+177Q, 3.16365209967032443701802479913075857e+183Q, 1.80121863573431611955949232303314972e+189Q, 1.56029746814154042157436181814468250e+195Q, 2.08399815409632719398233238905315693e+201Q, 4.35116508686695844690078637651674080e+207Q, 1.44042594283375698824917457557971547e+214Q, 7.67198578340176495038072549202750984e+220Q, 6.67439909212164301008970795489093574e+227Q, 9.63314810388083227111873828289832687e+234Q, 2.34398517046722427869034729305744960e+242Q, 9.77625325893464652861772945740140456e+249Q, 7.10967843958998030041427991161778060e+257Q, 9.17597441867286774623146964891772219e+265Q, 2.14036206544827332284805812358315196e+274Q, 9.19420612380519250234059644495148895e+282Q, 7.41568524413971300723452081581189161e+291Q, 1.14573384520235218510170025456065061e+301Q, 3.46156619789802557007367290577360329e+310Q, 2.08912045049233012342096277237544009e+320Q, 2.57451388307699184873778115477509299e+330Q, 6.62688893229567984520960366416649465e+340Q, 3.64721218233974203748299012650543732e+351Q, 4.39668921699312866048896359512324981e+362Q, 1.19018091828589034728575506532863164e+374Q, 7.42292147631996488193728378356566879e+385Q, 1.09526098640141261390128648437627728e+398Q, 3.92927734436557751277903528467662194e+410Q, 3.52539958718458390412925405124992174e+423Q, 8.14407380085561132671651242382158564e+436Q, 4.99172864167925991508434677916290384e+450Q, 8.37313822543668537823480408319630674e+464Q, 3.96855771450749046728173847477197045e+479Q, 5.49292894873452993989300213358944271e+494Q, 2.29708122354182613470710687416853220e+510Q, 3.00602816347641813806476628260504686e+526Q, 1.27638345565474398023933739505382139e+543Q, 1.82543785151714558753981850731109273e+560Q, 9.13886193903951047182967076404934097e+577Q, 1.66659166393902754183225415655293639e+596Q, 1.15345182570915755941253059732455874e+615Q, 3.16074696483848527092291852149349249e+634Q, 3.58237065512849118228300807716046934e+654Q, 1.75676845140981179589000705343726870e+675Q, 3.90495239094757591264933748015965848e+696Q, 4.12771024114613858596256966986824636e+718Q, 2.18017473107123494465764845328386818e+741Q, 6.05537171542030830962660182644895823e+764Q, 9.32267445280349135813949292533678488e+788Q, 8.40039310752404384099688064836067109e+813Q, 4.68572386284109286448290494678984852e+839Q, 1.71436621216433181556756205470621080e+866Q, 4.36726894564208691170045669145775062e+893Q, 8.23850494545149522733960578619372006e+921Q, 1.22636833130359811940229369518492693e+951Q, 1.53817446772510796133546119401337038e+981Q, 1.73934010579080275675622752667739992e+1012Q, 1.90139542390828308944919910797818652e+1044Q, 2.15946649839694693874583032395369802e+1077Q, 2.74458385322614086037916005350795661e+1111Q, 4.21460437827726866824631065747020388e+1146Q, 8.46327678591246014474308263680903354e+1182Q, 2.41136671056709946500959023526115708e+1220Q, 1.06046799402590770086828664893536628e+1259Q, 7.85179239227989671195411257856609735e+1298Q, 1.07053761782523090834371225181245647e+1340Q, 2.94820375013014518743449818159813519e+1382Q, 1.80414267837505733123417059421507186e+1426Q, 2.70702818749460407693598763878577805e+1471Q, 1.10237898995725987663697548890315962e+1518Q, 1.35298567513553498909522877321159185e+1566Q, 5.57611902405430874175886881037679558e+1615Q, 8.62760618106732185887902244008862925e+1666Q, 5.62274836328861658358184936480919820e+1719Q, 1.73809780706704538846353037945373733e+1774Q, 2.88052131688282318513745853191288493e+1830Q, 2.90423838801873888374451692095277096e+1888Q, 2.02951924440811805047282880080209219e+1948Q, 1.12457154445459099155545223242406413e+2010Q, 5.67678101812548596682420412619518564e+2073Q, 3.01259124479996611916817067093268060e+2139Q, 1.94839799990442184578594656275869899e+2207Q, 1.78866123976568468449541367729975757e+2277Q, 2.72776450570640988722564402903756955e+2349Q, 8.12826724972980545013533499964424395e+2423Q, },
      { 1.58403833597397609272337637131715304e-2474Q, 4.01729907956437307269377706375953438e-2436Q, 2.58580645701935131155084366864422406e-2398Q, 4.31502895023559220093536948165386274e-2361Q, 1.90627959218882483095663188951084330e-2324Q, 2.27589955667507909301636117518932802e-2288Q, 7.49364095876812027654477175001110624e-2253Q, 6.94191154236392933887288062436725643e-2218Q, 1.84522919671267474043685120983334524e-2183Q, 1.43487104640347241520251346864979221e-2149Q, 3.32692243646438795745471365264644590e-2116Q, 2.34362088307498827836613435573448115e-2083Q, 5.10938037262713942693655713913568198e-2051Q, 3.51060735754549615330960873223686009e-2019Q, 7.73934046950102664330281533900661264e-1988Q, 5.57169465236073520583741860201801676e-1957Q, 1.33280818206626290524674045358563098e-1926Q, 1.07761463878385909143899244627194248e-1896Q, 2.99487219016399829947920738111566372e-1867Q, 2.90872283894068508271645083240599458e-1838Q, 1.00349314519658897817146360011284604e-1809Q, 1.24963603019269210233323055704536134e-1781Q, 5.70651178926661193581089363715896182e-1754Q, 9.70578293043209307574192077696195294e-1727Q, 6.24328278846472954410732505740796953e-1700Q, 1.54193058035632514209035337344023300e-1673Q, 1.48399119945730121895709095048379728e-1647Q, 5.64752158032889554103477750567193685e-1622Q, 8.62167301557505164690437370018957398e-1597Q, 5.35528299764763984143338000418037832e-1572Q, 1.37241462764951847155586335894427418e-1547Q, 1.47116444020541724843474277134735959e-1523Q, 6.68618803692998895772448346916660911e-1500Q, 1.30561453192449212967328116730742424e-1476Q, 1.10983482751594629837009226116529299e-1453Q, 4.16014518377810797736243906294304347e-1431Q, 6.96431413050131379947014047903616964e-1409Q, 5.27224320695995707920555229340409925e-1387Q, 1.82727152085917578757399986780345619e-1365Q, 2.93468712490571136931409720092296819e-1344Q, 2.21029656493081556265677595112314098e-1323Q, 7.89892460822856456312620943162424218e-1303Q, 1.35498246922475851477828612647429926e-1282Q, 1.12846854910106323887966343965144914e-1262Q, 4.61424330893765402182473137993848704e-1243Q, 9.36605095127672178405946589653781928e-1224Q, 9.54054141854710811663726016692763275e-1205Q, 4.92938100391460828476769674596652859e-1186Q, 1.30552448174985025504800286655378847e-1167Q, 1.79081395504489298673122868237542862e-1149Q, 1.28534148947828407343141932678878641e-1131Q, 4.87584961151572559819046616829450419e-1114Q, 9.87277629967922144952427945151885497e-1097Q, 1.07748800964228010206356572218732211e-1079Q, 6.39928058051557015650179754550910965e-1063Q, 2.08781638937389821580749246439535224e-1046Q, 3.77684590339710704264469235045684970e-1030Q, 3.82307555361237861173188661605677525e-1014Q, 2.18500494285229212661260038960152445e-998Q, 7.11371665672211288079530438419393765e-983Q, 1.33086590459456967821090511427946680e-967Q, 1.44309821421378874919204882407008472e-952Q, 9.14647515187060050796469146648366553e-938Q, 3.41683020290309632448426799765730462e-923Q, 7.58515118221027874504569883083804726e-909Q, 1.00874302140725085602226074835782694e-894Q, 8.10069400524047548461447004237480370e-881Q, 3.95900210017885619699714022194103898e-867Q, 1.18662867376446832171113003401601030e-853Q, 2.19786739893484213511853337557279548e-840Q, 2.53446665772094387684997171428381901e-827Q, 1.83299289701163579779687245566728015e-814Q, 8.37462324951323786164578010773011160e-802Q, 2.43440349985894963165438833576050630e-789Q, 4.53405919015746926984345204895280285e-777Q, 5.44811780296147933680587250736719556e-765Q, 4.25227509910246244055269284525208521e-753Q, 2.17028567518979939311612851867355063e-741Q, 7.29110293097611082922276966606966112e-730Q, 1.62280402409213923291155004693280799e-718Q, 2.40828308813354260030814303685712546e-707Q, 2.39798644647022706006083856285038692e-696Q, 1.61202006993232554271262188260349029e-685Q, 7.36078014307223356522361505691981658e-675Q, 2.29673595120013740585571409952171287e-664Q, 4.92601668045549547120303143586093867e-654Q, 7.30469847326532782734592211727680659e-644Q, 7.53208483045812497621524949105624661e-634Q, 5.43101457218649885434844485451114112e-624Q, 2.75364547737132169841544376328974616e-614Q, 9.87114363624257039419731807833639141e-605Q, 2.51532230246971379761137727536103954e-595Q, 4.58020252011364324634687161502454437e-586Q, 5.99105224430269595047395967421408836e-577Q, 5.65817475729598864582006398104320790e-568Q, 3.87790653215945353008449656152626419e-559Q, 1.93831767408426944576894121111160849e-550Q, 7.10042999124518145073529096418920675e-542Q, 1.91544155185588660392537613901890409e-533Q, 3.82329460613706944718834058784982843e-525Q, 5.67308558344799642959861077485254353e-517Q, 6.28651971015047054176370541889652534e-509Q, 5.22608309691007325295015732741277330e-501Q, 3.27380385796518526996963624688451015e-493Q, 1.55218880567110686358214096387337744e-485Q, 5.59406715134492552797785887998506288e-478Q, 1.53903555561583129370612427064103671e-470Q, 3.24582771097869247388161880434215690e-463Q, 5.26923908371102477241383694074670819e-456Q, 6.61116543964979170247326465662672578e-449Q, 6.43651161747399650924315876123367361e-442Q, 4.88172530890739153159755873311503383e-435Q, 2.89552772291014307121494513337008901e-428Q, 1.34824561713159645449381177450374044e-421Q, 4.94682830703508395886779775564904786e-415Q, 1.43550778574773554814450699757551163e-408Q, 3.30662278156545301906789072182270217e-402Q, 6.06763076341882342375981845143813941e-396Q, 8.90102295078976467641802004915383639e-390Q, 1.04750026993369008379812665328207076e-383Q, 9.92305977697594285442750520712185343e-378Q, 7.59233047709729007268137313555513461e-372Q, 4.70739644774061079459196244467224618e-366Q, 2.37289471679140773116577190541471766e-360Q, 9.75578189470698031776059091702014024e-355Q, 3.28173795958592945075990020191630023e-349Q, 9.06055778471501819614486253220513415e-344Q, 2.05942153439035557149195606898627504e-338Q, 3.86530964740245225250791387282901296e-333Q, 6.00843270324960642367292431619041056e-328Q, 7.75791717613089963209182313649669073e-323Q, 8.34421688870272754369905134347321346e-318Q, 7.49743392110895125137984446953806518e-313Q, 5.64335493091932417079632230402715339e-308Q, 3.56822312364224796215751885944182157e-303Q, 1.90033575264131712027297219010231489e-298Q, 8.54728045366493643486236298588028833e-294Q, 3.25523625801393911932111212124054975e-289Q, 1.05248047241632062076489482718979111e-284Q, 2.89617776712893256185087414745466351e-280Q, 6.79988531580997322799099912957691248e-276Q, 1.36556351036663137564575797911341140e-271Q, 2.35130154856703817933308580165070895e-267Q, 3.47957918602140446743207712996488528e-263Q, 4.43595246742156897562714682208015764e-259Q, 4.88306826050902558584812739817346803e-255Q, 4.65192688318894071322199120456680912e-251Q, 3.84397217698170104458131412518090261e-247Q, 2.76116826189849718252009792299798034e-243Q, 1.72788148193695597819524956747708857e-239Q, 9.44000021446169518374327174980091189e-236Q, 4.51212554821909958969480138145008725e-232Q, 1.89078188020015147413152497967774898e-228Q, 6.96047357819953940988624584998189879e-225Q, 2.25551400000095886908107215813083451e-221Q, 6.44644677476230409449870129589571800e-218Q, 1.62820413693305623578289570883910133e-214Q, 3.64118898140332008867604954881435445e-211Q, 7.22342104300488240353501601432620632e-208Q, 1.27354621995150979542739771832851778e-204Q, 1.99918407175965223164864543465886820e-201Q, 2.79923301669228713128194864334595626e-198Q, 3.50221902155243103789141550909762023e-195Q, 3.92214264392110857739154085058742459e-192Q, 3.93845056335503285763060194710496279e-189Q, 3.55208932101704820479562248595366349e-186Q, 2.88218576501349132033303488025583146e-183Q, 2.10742246561673617325867420889217990e-180Q, 1.39082883056909297835969428096996258e-177Q, 8.29809184041509702782403322046680024e-175Q, 4.48275079028324875908925844854947607e-172Q, 2.19604664947993546897548946859268970e-169Q, 9.77074245531826439290686011595488065e-167Q, 3.95412900034836577575737163397063382e-164Q, 1.45763752143522946134851229157315125e-161Q, 4.90175908594770144816169627656347413e-159Q, 1.50583242362081439890282730822964005e-156Q, 4.23187210926299952258468916772463829e-154Q, 1.08947970178510600143883902241342972e-151Q, 2.57292238715065164920970933548863818e-149Q, 5.58131105433415694107142145348480095e-147Q, 1.11357590012697004029170524556205260e-144Q, 2.04616505133228608383036998688279538e-142Q, 3.46699488500477063569923373366652510e-140Q, 5.42379540407350192206120227447952764e-138Q, 7.84383327240284700975116490839187822e-136Q, 1.04992295793319441478934809931921214e-133Q, 1.30230107195741860304672567525213181e-131Q, 1.49865973782839300796909511415792928e-129Q, 1.60190662241428628230308870106117967e-127Q, 1.59224861840198356125270425396669689e-125Q, 1.47337534591643627391632824044256382e-123Q, 1.27065155139400959289502878067800521e-121Q, 1.02240826352576620907328663762929630e-119Q, 7.68376260232956278119400404906698852e-118Q, 5.39926812723337318582425048609813182e-116Q, 3.55107427485349467569160645183534302e-114Q, 2.18823540951912101048500342930263842e-112Q, 1.26466751543081693428147968135946762e-110Q, 6.86180756673724371219029573988488620e-109Q, 3.49869168682520996337772693940273721e-107Q, 1.67801680739837515692197356611469711e-105Q, 7.57743943144193148982405002124429739e-104Q, 3.22470377015938680899710533122019572e-102Q, 1.29448709067770596337952427228314258e-100Q, 4.90613325096345413914934529363944533e-99Q, 1.75712131798815332550332783841135938e-97Q, 5.95204249145432038279376449394263726e-96Q, 1.90856665328641726356485393831448064e-94Q, 5.79822445923642921233565089484060205e-93Q, 1.67029323997833472695510332615399040e-91Q, 4.56623667339808303761814383193281614e-90Q, 1.18561734279154794522720467403607684e-88Q, 2.92616002780129692916694825399426567e-87Q, 6.87006113412670713749912656047162470e-86Q, 1.53556578350037994526862611062995548e-84Q, 3.27003673677840125658578850353789865e-83Q, 6.63955800720658036202444029681088661e-82Q, 1.28631975096739859347518450406626244e-80Q, 2.37956658113902295790875474474928153e-79Q, 4.20626823139888342521923162463424944e-78Q, 7.10971923783337943250783648529172174e-77Q, 1.14991510411537277718541841100610582e-75Q, 1.78087620125559422021122815240864113e-74Q, 2.64270379617932988289762938432353092e-73Q, 3.76008537594171932684988546369276565e-72Q, 5.13292095112425199264463076393673633e-71Q, 6.72710027460142769581137467118289653e-70Q, 8.46958562134769749810317782657268756e-69Q, 1.02503238267223284787859415412841523e-67Q, 1.19321912755786334798650299344282298e-66Q, 1.33681693038130658171632078204446788e-65Q, 1.44228347967979838541089585366023007e-64Q, 1.49937455500479399063651651865716553e-63Q, 1.50279720313350143765006477517598013e-62Q, 1.45300596931848530329429215240805653e-61Q, 1.35598044837786253974814702268132430e-60Q, 1.22207241221255212718220407758970979e-59Q, 1.06422318027052015901612579353436284e-58Q, 8.95966739607563684498250506056071091e-58Q, 7.29628880807929410507791279851435148e-57Q, 5.75025529619018115782221236649660618e-56Q, 4.38801166482901351758297994035262224e-55Q, 3.24385245129183239778054564241318804e-54Q, 2.32423935766553880551312730791561599e-53Q, 1.61486977620302644603098163288002715e-52Q, 1.08852460554527484166896264813585652e-51Q, 7.12175557419282904469498440847683331e-51Q, 4.52464766254906707378983948806266105e-50Q, 2.79273071581879303547665803856393705e-49Q, 1.67538487960386422664995380528599268e-48Q, 9.77311432877767609128920687351148833e-48Q, 5.54591076684762708155258243373876422e-47Q, 3.06280970562787364526087771025696151e-46Q, 1.64686211803826623376300968100335938e-45Q, 8.62510851388715584666563483664725226e-45Q, 4.40168766386889070104410967553191993e-44Q, 2.18975577884764674623829698191429056e-43Q, 1.06234533644926588892221744026505121e-42Q, 5.02803666348568404922856964957017674e-42Q, 2.32252463571724922291588384046952088e-41Q, 1.04740659389834130621977678673036650e-40Q, 4.61343838844969816849890851044847674e-40Q, 1.98539744511816200515536891243051363e-39Q, 8.35102736745462834322783293199882340e-39Q, 3.43444090348454338873945310910863585e-38Q, 1.38148913187719664577281415801326678e-37Q, 5.43705120131022522415564030011104834e-37Q, 2.09435754808064771674030165617400032e-36Q, 7.89867661859200690211003183623252601e-36Q, 2.91753687094747127218152557081492503e-35Q, 1.05578888602271659701510267800456427e-34Q, 3.74433381216033081152946793044225824e-34Q, 1.30180118525195729018565371931285108e-33Q, 4.43834621689338776826444391843192077e-33Q, 1.48434826895181654234585115302567433e-32Q, 4.87100112984983697071424260901079389e-32Q, 1.56890300074251394167498791840311009e-31Q, 4.96129531591793523529868958902224169e-31Q, 1.54077391002799082070667599670892752e-30Q, 4.70055802217201491012415527477898338e-30Q, 1.40911523071894959581827729743208189e-29Q, 4.15191310395569203370171887252329295e-29Q, 1.20273761371542774812275721780834541e-28Q, 3.42632737493449673556641936029297478e-28Q, 9.60140535939702601155477307042148556e-28Q, 2.64727864203377330130778700812469317e-27Q, 7.18344222056514710304108009603342664e-27Q, 1.91885054598149404233254994607045127e-26Q, 5.04697477945599249407828848720385551e-26Q, 1.30739479992591170007499101639277446e-25Q, 3.33634219823695708156517864262133521e-25Q, 8.38925958113626219399750986165640649e-25Q, 2.07905181351354860827899559955891196e-24Q, 5.07917896724376528007260946058158296e-24Q, 1.22350179435783727784530143589061714e-23Q, 2.90665491105754953014975765085018324e-23Q, 6.81166860609501546977145188171063268e-23Q, 1.57498593823802530267320052674032470e-22Q, 3.59379678896934832551131875550134077e-22Q, 8.09418541120521256436201968800902213e-22Q, 1.79979618323748172103553749785052354e-21Q, 3.95175890164101728455789689950582204e-21Q, 8.56958006805086577463240518925283894e-21Q, 1.83575348651729869648335347322858802e-20Q, 3.88541433996602231722399876322981769e-20Q, 8.12661397289502178971121801417655470e-20Q, 1.68000718288950314093575786904308274e-19Q, 3.43336935156396282804714345750464903e-19Q, 6.93769555039942749928599832442683686e-19Q, 1.38634563100898175510110455954156138e-18Q, 2.74008749775923088060873199170519103e-18Q, 5.35757028868338662612872498580241047e-18Q, 1.03646493302280378406673184005440162e-17Q, 1.98424944201008499240310250250687800e-17Q, 3.75978800606000340899396301904932318e-17Q, 7.05221126182168479511704811938887205e-17Q, 1.30963564152954622082996853685780292e-16Q, 2.40827549610918052797562865726934570e-16Q, 4.38589880961171155211843738902302126e-16Q, 7.91175868684912128549216604428547791e-16Q, 1.41388359787718387270571947224924145e-15Q, 2.50347753664468021020115723250325879e-15Q, 4.39263786655070582741822344903139686e-15Q, 7.63871030696057461236914444427999640e-15Q, 1.31670336037747604093705380830550144e-14Q, 2.25003102727544891850217684727399098e-14Q, 3.81223973341221495282227985716493129e-14Q, 6.40502166019136347918960755019998163e-14Q, 1.06725053827031948353711195948826425e-13Q, 1.76389749378472101040375240485184047e-13Q, 2.89198756533454775599767608849931393e-13Q, 4.70424252036995808470327107927435060e-13Q, 7.59287827351269198973041796516049688e-13Q, 1.21618333837252517178450702971971531e-12Q, 1.93338859343662487903047794738683986e-12Q, 3.05082685244229075126272877890267121e-12Q, 4.77908002001763665704541557372044375e-12Q, 7.43273471338542509759247990082208445e-12Q, 1.14783388812587366578078714999164060e-11Q, 1.76028616037242275427822154657272751e-11Q, 2.68107110162395316788540367212924769e-11Q, 4.05602375429596543728262134893310443e-11Q, 6.09544349224153722159236648112669209e-11Q, 9.10055012961606421079182569690846497e-11Q, 1.34999345213696765153127846451643325e-10Q, 1.98994391239515605127817880646635584e-10Q, 2.91499607361905978799020781323598840e-10Q, 4.24390078141221962088655373293736779e-10Q, 6.14135316267139108184909214050184662e-10Q, 8.83436579589479851125156253066831718e-10Q, 1.26339559402593316965102282486280599e-09Q, 1.79636925005171604673781604814805685e-09Q, 2.53970414332648086166213535713251755e-09Q, 3.57059249828789049877331667413514712e-09Q, 4.99234840315053910682137402914672922e-09Q, 6.94247187048993148295924434414634261e-09Q, 9.60294960016456137105057353194608309e-09Q, 1.32133371276166677672139906194961059e-08Q, 1.80872790163534638977965561238944112e-08Q, 2.46332536476779151638500763225429049e-08Q, 3.33804787013687049574791297244907192e-08Q, 4.50110842610850506871431532628603516e-08Q, 6.03998541333325959410060827223996830e-08Q, 8.06630537452609783391065479881400036e-08Q, 1.07218105901889261435737857618298194e-07Q, 1.41856144379535399139351236825663488e-07Q, 1.86829769983638330491745888879438028e-07Q, 2.44958653917297200861483891225150109e-07Q, 3.19755978044276083153634882857883309e-07Q, 4.15579069086754433405340505039048729e-07Q, 5.37807971332554467827633744547482376e-07Q, 6.93056106477668619385798002644561466e-07Q, 8.89417585250212245375810218669817408e-07Q, 1.13675615786872600638029106996132895e-06Q, 1.44704121253473089806789252868181132e-06Q, 1.83473664533283350419063698449974520e-06Q, 2.31724882235425364363184937796552995e-06Q, 2.91544022582530391094107183656707013e-06Q, 3.65421570986355186992807571629053921e-06Q, 4.56318857677376015133362727471956847e-06Q, 5.67743390948223287752620793591929613e-06Q, 7.03833674730757178417069911600525632e-06Q, 8.69454275808306722839276847105496531e-06Q, 1.07030190270275985848630609511672890e-05Q, 1.31302324393740374972736712543053054e-05Q, 1.60534528678907389671161995891497537e-05Q, 1.95621879772878044903047444244351033e-05Q, 2.37597559155521886172243888058434365e-05Q, 2.87650014695436120790473222143336354e-05Q, 3.47141604126307620878569429267584729e-05Q, 4.17628757618591523872557240511914786e-05Q, 5.00883684896740377264076716570754303e-05Q, 5.98917639018173037280503906445049523e-05Q, 7.14005734028021322676450426011314791e-05Q, 8.48713297304976003620620060812799411e-05Q, 1.00592371962099993408798498455907607e-04Q, 1.18886774688549697317681023800631139e-04Q, 1.40115413739806927896468824589692520e-04Q, 1.64680158738873124934815777767252928e-04Q, 1.93027180590427177790609959505517465e-04Q, 2.25650359795433055575151925625868196e-04Q, 2.63094779253370712822446127059315509e-04Q, 3.05960282998094618038061759727754321e-04Q, 3.54905080142515530273890819014315333e-04Q, 4.10649371213184272695586730470455583e-04Q, 4.73978972070856543596623702324547537e-04Q, 5.45748908769705106925744948564871657e-04Q, 6.26886955037988466777589811218012335e-04Q, 7.18397082597597367255394668144143647e-04Q, 8.21362793308292890121036733909194916e-04Q, 9.36950301151796636389593916360605611e-04Q, 1.06641153138572518429387087881256567e-03Q, 1.21108690381909541715522898126065244e-03Q, 1.37240786710764633931488483556773354e-03Q, 1.55189915125250562428574475103346186e-03Q, 1.75118070611954731846143391024982058e-03Q, 1.97196929478447094442182021751059029e-03Q, 2.21607971185090897097885205424488147e-03Q, 2.48542559858177963603431553204765315e-03Q, 2.78201982871899325746660736300345366e-03Q, 3.10797444123022017600460098808123935e-03Q, 3.46550009889599377554207066240202628e-03Q, 3.85690505461395961880575720620871256e-03Q, 4.28459361052363939313848682793859897e-03Q, 4.75106405851509722505038175681427297e-03Q, 5.25890609434561842084606413815284753e-03Q, 5.81079770141443579933464319447926318e-03Q, 6.40950150419891594280210549702450453e-03Q, 7.05786059539697018633739598386855559e-03Q, 7.75879384490912344645900750924930976e-03Q, 8.51529070288836937235862937785994511e-03Q, 9.33040551314529952303815524992440065e-03Q, 1.02072513571791257196870650265821749e-02Q, 1.11489934529722276019901656254303012e-02Q, 1.21588421363983657377673415921445027e-02Q, 1.32400454566162946339651246079183099e-02Q, 1.43958814201171884975897081297216705e-02Q, 1.56296499211348507284307663166667625e-02Q, 1.69446643988840458394481431193257832e-02Q, 1.83442432645398203334614155955590374e-02Q, 1.98317011429883687012533532738687781e-02Q, 2.14103399761506788853091011534572156e-02Q, 2.30834400360906269037251797223442256e-02Q, 2.48542508971601536767883257823339293e-02Q, 2.67259824171004266947925047187844041e-02Q, 2.87017957773082030971446046374914401e-02Q, 3.07847946323935695276863138190467659e-02Q, 3.29780164187051571955970986393933287e-02Q, 3.52844238706916706398930809090405599e-02Q, 3.77068967928172889046621767464058045e-02Q, 4.02482241332694163505581270228186661e-02Q, 4.29110964039093677010619866631701642e-02Q, 4.56980984888413264034413616112869595e-02Q, 4.86117028816359215545912812117182378e-02Q, 5.16542633886674445449455067720287452e-02Q, 5.48280093332349644585895096278308003e-02Q, 5.81350402921654268021994051667911914e-02Q, 6.15773213934700546744487567673172633e-02Q, 6.51566792003733016470332458328988615e-02Q, 6.88747982036856640340840950741077307e-02Q, 7.27332179410771208959941663503595393e-02Q, 7.67333307583556615050326531083587939e-02Q, 8.08763802243933982425303227000944183e-02Q, 8.51634602078983074689360505021182257e-02Q, 8.95955146208286742294968393977769930e-02Q, 9.41733378299144489750576704343915921e-02Q, 9.88975757345080247656478735188449399e-02Q, 1.03768727505857796722978821461546404e-01Q, 1.08787147979900856678514246787241454e-01Q, 1.13953050692823999603416543952272158e-01Q, 1.19266511545960614110252460361458099e-01Q, 1.24727473084088741556780346524879069e-01Q, 1.30335749368884349581460099687500407e-01Q, 1.36091031427173401966757170543046798e-01Q, 1.41992893251724361991875167825655695e-01Q, 1.48040798330635148304426193810368709e-01Q, 1.54234106679899202351157467338543629e-01Q, 1.60572082352486356465237071613156643e-01Q, 1.67053901396246033471214692284653203e-01Q, 1.73678660232131774208044201415508905e-01Q, 1.80445384423654491197811595430697492e-01Q, 1.87353037808093115326776430258837481e-01Q, 1.94400531959820109714578533182084599e-01Q, 2.01586735956129211498107459964096125e-01Q, 2.08910486416176267158329341893267875e-01Q, 2.16370597784052818695266702661187610e-01Q, 2.23965872827597104489240738884195628e-01Q, 2.31695113325298676464455224512869174e-01Q, 2.39557130914560734720290079897448420e-01Q, 2.47550758075638008786996132954185789e-01Q, 2.55674859226756791227887436442350007e-01Q, 2.63928341907236639931045094151784107e-01Q, 2.72310168026859366788753574972476498e-01Q, 2.80819365161259349748148678323641718e-01Q, 2.89455037874729232565423711864655899e-01Q, 2.98216379053536250342702367445487260e-01Q, 3.07102681234616603556790972526892770e-01Q, 3.16113347916348760019249138447488231e-01Q, 3.25247904839992014226666657864488014e-01Q, 3.34506011232305314043635112310304911e-01Q, 3.43887471001825077739424147442217492e-01Q, 3.53392243883271879333889785572599272e-01Q, 3.63020456526567529072712529376719554e-01Q, 3.72772413528969943076239230990758849e-01Q, 3.82648608410867702369280704118517439e-01Q, 3.92649734537814481801171047048544340e-01Q, 4.02776695993421447225756159687866121e-01Q, 4.13030618409759875566958499890031988e-01Q, 4.23412859763953990585030832527434639e-01Q, 4.33925021151663415382683548295156320e-01Q, 4.44568957550164552554831720660662401e-01Q, 4.55346788585740186021655771168112611e-01Q, 4.66260909322076961243786064631384911e-01Q, 4.77314001088352176735947056292312804e-01Q, 4.88509042367666263602229097070513234e-01Q, 4.99849319768447906973686731056178972e-01Q, 5.11338439103428142948352352152044385e-01Q, 5.22980336602751811729499672944460101e-01Q, 5.34779290289774015573364455286468490e-01Q, 5.46739931550080955287380242064660003e-01Q, 5.58867256926284616744877484869191055e-01Q, 5.71166640173175841717312971842336611e-01Q, 5.83643844609887615578228626088462787e-01Q, 5.96305035807827889758567710635182685e-01Q, 6.09156794655297569062821736349343099e-01Q, 6.22206130841923771587050175584017347e-01Q, 6.35460496808321163697317796410781296e-01Q, 6.48927802208755868138883757264025330e-01Q, 6.62616428937038679525110869491604161e-01Q, 6.76535246768429422677267756666823087e-01Q, 6.90693629673005399383708050292572666e-01Q, 7.05101472858747991895757442966727674e-01Q, 7.19769210605547537680146085503476253e-01Q, 7.34707834954433431482100014962668745e-01Q, 7.49928915319620942104629439591963551e-01Q, 7.65444619094446439133009723643838630e-01Q, 7.81267733325957766141060146767846541e-01Q, 7.97411687536856786535849810135961905e-01Q, 8.13890577777678436208431256995493454e-01Q, 8.30719191996558177068293937598919209e-01Q, 8.47913036818712374055336621201445842e-01Q, 8.65488365832860347537655780480228440e-01Q, 8.83462209487281076620051655836621909e-01Q, 9.01852406704052162112602532863159232e-01Q, 9.20677638326296314246112109641174382e-01Q, 9.39957462519996315051517977272639457e-01Q, 9.59712352259170728365681565566195274e-01Q, 9.79963735030970038685207606899372777e-01Q, 1.00073403490559993309000835270230984e+00Q, 1.02204671712495200986974494164817378e+00Q, 1.04392633537347289336664619627296763e+00Q, 1.06639858190518516071876464455235265e+00Q, 1.08949034071194662751500934725633506e+00Q, 1.11322974393006216402411430386892104e+00Q, 1.13764623169531331372630723630321340e+00Q, 1.16277061567042026013306680494416940e+00Q, 1.18863514648397907083703649648730714e+00Q, 1.21527358533611238960293477132347556e+00Q, 1.24272128004352904999786719958821158e+00Q, 1.27101524581551079885511358011134290e+00Q, 1.30019425107264471110982901454765895e+00Q, 1.33029890864201997125110745686528366e+00Q, 1.36137177268624019179654148863686280e+00Q, 1.39345744174911173037014751038646328e+00Q, 1.42660266832841175766232711333173310e+00Q, 1.46085647541588835802048454309412779e+00Q, 1.49627028047678533760134362560179034e+00Q, 1.53289802737592016874439144914426210e+00Q, 1.57079632679489661923132169163975144e+00Q, 1.61002460572564642000849608863176067e+00Q, 1.65064526666943143532240244567555108e+00Q, 1.69272385721798833166204038375277879e+00Q, 1.73632925074497773069781144176863796e+00Q, 1.78153383899165490255124499430464714e+00Q, 1.82841373739108738124596203605065226e+00Q, 1.87704900404072044775471792428672248e+00Q, 1.92752387330408763463712892863710673e+00Q, 1.97992700509947708650365975965257989e+00Q, 2.03435175101694043322616088522632134e+00Q, 2.09089643849576621383396167595935906e+00Q, 2.14966467439309042108153883973067514e+00Q, 2.21076566938140221236799358744093376e+00Q, 2.27431458472911392711904220601704565e+00Q, 2.34043290314497023966214961240421096e+00Q, 2.40924882550482707578395780381837657e+00Q, 2.48089769542928804270777308901824137e+00Q, 2.55552245384400165592583716337259322e+00Q, 2.63327412583237088656291310907475315e+00Q, 2.71431234228441160768610076838330351e+00Q, 2.79880589905706635256578441640292763e+00Q, 2.88693335659214188612865513175227518e+00Q, 2.97888368319007786718164917079966461e+00Q, 3.07485694541305021056656397487876403e+00Q, 3.17506504939176568339259231884822943e+00Q, 3.27973253713925527986226523992226922e+00Q, 3.38909744233483410200034132396842728e+00Q, 3.50341221043527586541966899038246748e+00Q, 3.62294468840159570502872359754361586e+00Q, 3.74797918980246258518386393043115941e+00Q, 3.87881764157340380510220562013091253e+00Q, 4.01578081927931267029397913978942025e+00Q, 4.15920967835153616848634566919310171e+00Q, 4.30946678945578836835486232808829311e+00Q, 4.46693788689973689723986043138805888e+00Q, 4.63203353981649359057533901265089751e+00Q, 4.80519095677036072684883899803335247e+00Q, 4.98687593543289697160079921623416533e+00Q, 5.17758497008053768765855285400689531e+00Q, 5.37784753088062976076034128768012631e+00Q, 5.58822853027308803485444007033170411e+00Q, 5.80933099323364005938466841106588229e+00Q, 6.04179894983708948784718604678234941e+00Q, 6.28632057034228591922334730387683243e+00Q, 6.54363156501365266125399451934329088e+00Q, 6.81451887309858260791767579514605499e+00Q, 7.09982466781971868191373433617700078e+00Q, 7.40045070694293100845796745008949286e+00Q, 7.71736306147578881407568826714170705e+00Q, 8.05159725837127958401987584151171259e+00Q, 8.40426387679538395058384455428283593e+00Q, 8.77655464160750010892022612040926275e+00Q, 9.16974906224756520705691589269677116e+00Q, 9.58522167027699388874228356777329920e+00Q, 1.00244499144430070386099274885141050e+01Q, 1.04890227783960385614814296197680936e+01Q, 1.09806501931649260624015170997236345e+01Q, 1.15011733242716998455210622743445835e+01Q, 1.20525758220454727952816341992419823e+01Q, 1.26369961333845432419384699684768677e+01Q, 1.32567409840433238017603739136360460e+01Q, 1.39143001526287336814820956823084828e+01Q, 1.46123626710408671221171338391349055e+01Q, 1.53538346012683753120053442509520620e+01Q, 1.61418585554581184591279187098492208e+01Q, 1.69798351452575852400510409025135738e+01Q, 1.78714465678460133882677098961576727e+01Q, 1.88206825601317848442892749807248071e+01Q, 1.98318689796476498521560439791262319e+01Q, 2.09096993011184544989073368255312499e+01Q, 2.20592693519609552724995680207860407e+01Q, 2.32861156486188168325833440616926330e+01Q, 2.45962577392286013760094208279475314e+01Q, 2.59962450073299827601794849672397508e+01Q, 2.74932084469488923833030086627802945e+01Q, 2.90949179822819598429999086944366938e+01Q, 3.08098459764107671506384842333391345e+01Q, 3.26472376541418040006352140116179211e+01Q, 3.46171892555432186128355184008936311e+01Q, 3.67307348405744306653537849794079228e+01Q, 3.89999427831545697992227880433189487e+01Q, 4.14380231271361842696931290595974334e+01Q, 4.40594471293014233018696008435982161e+01Q, 4.68800804884035743932352319123055757e+01Q, 4.99173319575866229822636583993237381e+01Q, 5.31903192638729836914160122380617184e+01Q, 5.67200545170346581087003708738282234e+01Q, 6.05296515859483113978235132985845589e+01Q, 6.46445582591583649124780712968047055e+01Q, 6.90928163944313177415529394071665670e+01Q, 7.39053537072521168719419511314839173e+01Q, 7.91163113594234348927381617490387462e+01Q, 8.47634120965947230809397578710282148e+01Q, 9.08883743598215272174601093484125344e+01Q, 9.75373785753325382322893051705294986e+01Q, 1.04761592725164736089639534479074532e+02Q, 1.12617765338655419662606911403899284e+02Q, 1.21168895243741881709481972315918511e+02Q, 1.30484988804359382802013970413388238e+02Q, 1.40643916977370870089294791135734370e+02Q, 1.51732386386376598946153144686407135e+02Q, 1.63847040773982427929709267988083986e+02Q, 1.77095711710003361983546853982729852e+02Q, 1.91598840361277588527216638598210664e+02Q, 2.07491095540949726522855395012839290e+02Q, 2.24923217236106119394568010864647153e+02Q, 2.44064119463086993590008673999888577e+02Q, 2.65103291739026696423162894476095053e+02Q, 2.88253544828036421249450783362468048e+02Q, 3.13754153889742451327114437895012151e+02Q, 3.41874460927761232190324931320648150e+02Q, 3.72918008746121432097478949634992228e+02Q, 4.07227290759381879033987273956073881e+02Q, 4.45189215310338987815842929936219157e+02Q, 4.87241400038863092710563840938907565e+02Q, 5.33879431809824993154116113611340152e+02Q, 5.85665251340011311718653543776253641e+02Q, 6.43236849676682281579530292235801252e+02Q, 7.07319496933657861116096423723089184e+02Q, 7.78738763222127723625961426334336958e+02Q, 8.58435638777040682657342340030579928e+02Q, 9.47484116394459954294416631122613257e+02Q, 1.04711166630196929712820316594777632e+03Q, 1.15872311371927743485840786701904690e+03Q, 1.28392852534970775486315470390457819e+03Q, 1.42457582618936343671625036189352836e+03Q, 1.58278900639377570598543689683858411e+03Q, 1.76101294444545923486104598108258215e+03Q, 1.96206607357312178755673700966218336e+03Q, 2.18920236070835422209657734442941355e+03Q, 2.44618436034955965177471357752869960e+03Q, 2.73736946076118709316717034422700237e+03Q, 3.06781187080876763823587584403262657e+03Q, 3.44338341950996275353420062305895100e+03Q, 3.87091687821820770548262640961099474e+03Q, 4.35837629346446550811483742251576785e+03Q, 4.91505976942026055884439891778599977e+03Q, 5.55184130321696740425893724460092414e+03Q, 6.28145970445342612877221303833900020e+03Q, 7.11886438520566571043900024525721790e+03Q, 8.08162996762779959609851789890726579e+03Q, 9.19045432173859727983153055486718404e+03Q, 1.04697579405183570171054706781681133e+04Q, 1.19484066394624731955974446656465043e+04Q, 1.36605846306210479280006137726608390e+04Q, 1.56468513163780927299848865296760921e+04Q, 1.79554229917996753910020302361418910e+04Q, 2.06437304374408251386790990436920728e+04Q, 2.37803156373267080705808819875316658e+04Q, 2.74471462199565095258711815725505419e+04Q, 3.17424455248072273922403695999752070e+04Q, 3.67841605073133622632674187774059283e+04Q, 4.27142203777350805091608476572382608e+04Q, 4.97037776810032398127125994364568099e+04Q, 5.79596727313857616360132420453444344e+04Q, 6.77324248460879259253530791773167141e+04Q, 7.93261334694994276050922426242412735e+04Q, 9.31107739715691545023768199956003639e+04Q, 1.09537503053637222429417759669716212e+05Q, 1.29157755673566952599086463851912376e+05Q, 1.52647130160874158565239969760504587e+05Q, 1.80835335096964828910095096138515211e+05Q, 2.14743829477016418053483721316277484e+05Q, 2.55633251557399994754680946686048161e+05Q, 3.05063334556209750218324509503461449e+05Q, 3.64968792666585395419705030588899790e+05Q, 4.37755686685748537981529230100717254e+05Q, 5.26424122294320873553623779329790816e+05Q, 6.34724899010831940960718793968560727e+05Q, 7.67360052654242646579956157559309383e+05Q, 9.30240305033750278572545038832724621e+05Q, 1.13081650266645184495081585123306969e+06Q, 1.37850753115552374245055838415718322e+06Q, 1.68525439396416227539607702063253047e+06Q, 2.06623977016863939032179814240740936e+06Q, 2.54082527022935491763036651163297626e+06Q, 3.13377596203641663047341272710642846e+06Q, 3.87686514827580239293595658539782774e+06Q, 4.81098405401834942973721071475737211e+06Q, 5.98892408953467866424875864774796159e+06Q, 7.47905792960806092396497010475870265e+06Q, 9.37022569869340886743682668142431718e+06Q, 1.17782423097751066120254539266241595e+07Q, 1.48545930143258061872721569284540776e+07Q, 1.87980927038339810411221199415802492e+07Q, 2.38705733443634639976678131975655720e+07Q, 3.04180655225860320213756009749555241e+07Q, 3.88995004684326215069338023547372206e+07Q, 4.99257437458669601662091946924574592e+07Q, 6.43128750449561321017861557895848463e+07Q, 8.31551851992585813643149605060553440e+07Q, 1.07925566470411796107305712037505869e+08Q, 1.40614107339003511498586442587195365e+08Q, 1.83920178567730560673713663331917338e+08Q, 2.41519711690497536458569405379734816e+08Q, 3.18438601538111228132801183275722771e+08Q, 4.21576501892968673592102667497462836e+08Q, 5.60444635691511454953230090792878868e+08Q, 7.48209439804691157215633476755723997e+08Q, 1.00317512966824615144294234414110893e+09Q, 1.35089891899748286951692285814894218e+09Q, 1.82722216505349159044967968934424570e+09Q, 2.48263348083176093300905605774694855e+09Q, 3.38857763723491971892999484076543171e+09Q, 4.64662006529910564426883234085294096e+09Q, 6.40182180156629712154379147661940269e+09Q, 8.86235203805325147258573091932997831e+09Q, 1.23283860285919681076337828978766266e+10Q, 1.72348929748018002343932808836953853e+10Q, 2.42153052846944737603768649152219024e+10Q, 3.41967381320806302529104945578396652e+10Q, 4.85431236462260654032773006433818623e+10Q, 6.92714904376034267631798991329824532e+10Q, 9.93804949018620361644641133903513098e+10Q, 1.43352142475985414523266188918663249e+11Q, 2.07922173448308822669953899424734201e+11Q, 3.03269524182010815814751345567548609e+11Q, 4.44863150372771043146060595057040363e+11Q, 6.56345864647790105139454816812312939e+11Q, 9.74063569639891097956397109813490533e+11Q, 1.45422052005965615789962842879352220e+12Q, 2.18425068889862732017431485970855335e+12Q, 3.30099910475756075681442440083420253e+12Q, 5.01997048502274901150163921982195024e+12Q, 7.68267629901760783371009641842522603e+12Q, 1.18337659600398387182686793240868246e+13Q, 1.83474885355703531496992152648661997e+13Q, 2.86363931245836358559523100954017414e+13Q, 4.49980389271503995761664922194406313e+13Q, 7.11948687698915449751693845544920647e+13Q, 1.13430701798012234619458399048221594e+14Q, 1.82006578236361839541514524597276231e+14Q, 2.94148450061539403705926461420998342e+14Q, 4.78870730589093038221910605812495108e+14Q, 7.85402503692862355090784993078672085e+14Q, 1.29789430461986025113447366376693514e+15Q, 2.16127995478242564041665605678872071e+15Q, 3.62710214703500383377916061381597850e+15Q, 6.13534293344095037788551479051826967e+15Q, 1.04617000636224450596060724147477482e+16Q, 1.79847735783966568554234927670508824e+16Q, 3.11747341233233147536151972501734499e+16Q, 5.44944507304918422218082921598846642e+16Q, 9.60751550501797821243248841373822216e+16Q, 1.70858922445267785163471612163490444e+17Q, 3.06542975111022866531582625971692938e+17Q, 5.54922743745114951066953816028814728e+17Q, 1.01373023277804631428305560194491190e+18Q, 1.86905989587640582429259395332686294e+18Q, 3.47854955238157842413369108802590798e+18Q, 6.53599224597546376326701108631484361e+18Q, 1.24001927226106630812645719537154550e+19Q, 2.37582886691093662903464587039857427e+19Q, 4.59768243360443262528520790447211677e+19Q, 8.98810681683712842794124622629411311e+19Q, 1.77530237939363226330723906827319437e+20Q, 3.54341330439097348610370779591666976e+20Q, 7.14806139767552532748817894789979543e+20Q, 1.45762051057718630539520752847120279e+21Q, 3.00513712487982979744944726506420716e+21Q, 6.26502486163325069730377264873611059e+21Q, 1.32097994109028381590144585332181800e+22Q, 2.81748753590214622077676014001554368e+22Q, 6.07993304142980523101356268020373032e+22Q, 1.32765885364721208288263553578233241e+23Q, 2.93431175918364131820516218129002628e+23Q, 6.56508721680713002578214496043109168e+23Q, 1.48721227343793765013154051597710086e+24Q, 3.41184019607678812819334291291332010e+24Q, 7.92818992879701876208212761012395842e+24Q, 1.86645187702970485690987148720542546e+25Q, 4.45252185988673954885155645168863932e+25Q, 1.07654543517497766241465472692681594e+26Q, 2.63868568119069758552386276646714408e+26Q, 6.55790847024418649820013325410592483e+26Q, 1.65295224373558572055534033842757998e+27Q, 4.22638339591491619855432610245411904e+27Q, 1.09645039426808014848951068590561771e+28Q, 2.88682208299928608019375692651426610e+28Q, 7.71548038934401592468282155249032740e+28Q, 2.09372878930996484632394470400138705e+29Q, 5.77027578944765503685743525710220950e+29Q, 1.61546384539178114004658391605197219e+30Q, 4.59547005579560869055708560274363012e+30Q, 1.32862939268652325541488091880021431e+31Q, 3.90507968153078421921910800039715306e+31Q, 1.16713402427199725201363486127342874e+32Q, 3.54805853865427740256789009746696294e+32Q, 1.09737805935804615955239963430188272e+33Q, 3.45410297806444559459316674852385198e+33Q, 1.10674539370165232283645935543266144e+34Q, 3.61089955913906999391756229494586894e+34Q, 1.19994699928367056689901878605794596e+35Q, 4.06268701419087879160332452792716198e+35Q, 1.40183522389322451388311239745912475e+36Q, 4.93108552733316217324966630695184171e+36Q, 1.76881239328491949973951925349674313e+37Q, 6.47214829394519996070845864216337266e+37Q, 2.41645372173921192231147151489004683e+38Q, 9.20894472039812386191245000342688835e+38Q, 3.58329702862212667598998865894252598e+39Q, 1.42409748259669944005518449916929857e+40Q, 5.78262783342641152433806724533524027e+40Q, 2.39986220408436318313990705343752378e+41Q, 1.01829157204230545960231903208713235e+42Q, 4.41910541482203453093956062976933998e+42Q, 1.96212611768049931065065590432043620e+43Q, 8.91674242406125370712417447773466783e+43Q, 4.14888247829475772044491476179936724e+44Q, 1.97725652955827692966670149947470575e+45Q, 9.65530023387540108044378286001152781e+45Q, 4.83287889833559892166961276593618835e+46Q, 2.48057587822309805808523773881299449e+47Q, 1.30610280975765470571043683527277027e+48Q, 7.05756571728956923233902004974509542e+48Q, 3.91527652222961861840270351783060925e+49Q, 2.23089898094339331762072829615745302e+50Q, 1.30614133449630930559423540813368509e+51Q, 7.86102128665639262739828585146248224e+51Q, 4.86558375853845110678814792824481696e+52Q, 3.09848742591570467373735298285106117e+53Q, 2.03103761486256390140847277681645261e+54Q, 1.37099964760826020030246335833308722e+55Q, 9.53473627432500152782873935909970826e+55Q, 6.83495992316641540715754654005899263e+56Q, 5.05273354632478901960113633106144814e+57Q, 3.85381099728215997914445000026608173e+58Q, 3.03418310785320829824308157124864982e+59Q, 2.46716192600983889917114272951924381e+60Q, 2.07290103981358059329932171745274528e+61Q, 1.80056398057961538305615630314631614e+62Q, 1.61776402789534425695658193709805813e+63Q, 1.50428302825068832904492692640561694e+64Q, 1.44839320652542717214779500803249160e+65Q, 1.44485551098011579855868939330990133e+66Q, 1.49412042885502924277326734308483245e+67Q, 1.60256656610701572194891345599497651e+68Q, 1.78388050415394298763315503064430164e+69Q, 2.06199924057276073839995264039101124e+70Q, 2.47652179469857271457600275335452230e+71Q, 3.09234991415349735847967247784284417e+72Q, 4.01692723830598581004567643058591262e+73Q, 5.43160754522649738742682321649737815e+74Q, 7.65008682404282275932608294522372170e+75Q, 1.12301798411434928750045233045701052e+77Q, 1.71938295296605200401018803799684912e+78Q, 2.74733571869068667385885336550518795e+79Q, 4.58454501055768412308971248876333874e+80Q, 7.99508204153925025226317876728470146e+81Q, 1.45811990936589904392804782726885652e+83Q, 2.78300117867960017490532778932474629e+84Q, 5.56281223196619462795860275879049153e+85Q, 1.16533876898240457849831832137567995e+87Q, 2.56039912643283822420817604614012086e+88Q, 5.90454964185909819164533268125527979e+89Q, 1.43027847474983870988836419822056214e+91Q, 3.64204612295693256305339371887263313e+92Q, 9.75669857120640229967783787120452211e+93Q, 2.75194604427588305129487979774984195e+95Q, 8.17916479364319727936284803066665955e+96Q, 2.56370473508682589047202009616066313e+98Q, 8.48165649612825587957771293185197099e+99Q, 2.96426025440398100674466172613133159e+101Q, 1.09534297003120888620652471421812455e+103Q, 4.28314854758487062838465336068295133e+104Q, 1.77395435294431974409395342323139198e+106Q, 7.78899108189422475969140390458031321e+107Q, 3.62893172105682135159715800277136586e+109Q, 1.79572927251602059220958063790243562e+111Q, 9.44668515148283533862523011302496150e+112Q, 5.28826317961448810089554134850695191e+114Q, 3.15331123674140136231493452577546350e+116Q, 2.00480707968382766860514611914082879e+118Q, 1.36040719266523771611516008331762962e+120Q, 9.86282560980781051743602195410395887e+121Q, 7.64755178859112809938428139017982021e+123Q, 6.34880222487173008841344329862213996e+125Q, 5.64906236198001909791959175488227229e+127Q, 5.39324800352378478055022039572371179e+129Q, 5.53089719191570391565414448432944942e+131Q, 6.09959864464089433289271654904908851e+133Q, 7.24209843349196450391058906977712497e+135Q, 9.26808305363737556950784175210077690e+137Q, 1.27994270241604058214925694813276489e+140Q, 1.90979662696062130244831484512150929e+142Q, 3.08254030066988503957988563764712878e+144Q, 5.38880973238417965706179354197373928e+146Q, 1.02161025105662653467101460483765248e+149Q, 2.10300544007279065014421211807768152e+151Q, 4.70675399034872557015773705065650223e+153Q, 1.14683412812524899101968167573148610e+156Q, 3.04620262376779901134561159740015427e+158Q, 8.83255301555740088857655282843830654e+160Q, 2.79951315833790052768515917244690507e+163Q, 9.71307000884329724595836237775132776e+165Q, 3.69425449543187541186652254254454177e+168Q, 1.54248755204034481247092284401752383e+171Q, 7.08075007659653083674442572183052884e+173Q, 3.57888569536270290704774796012385631e+176Q, 1.99473765641034107285413476042860568e+179Q, 1.22789700989638770500092926180435793e+182Q, 8.36095001442791620159284429240141671e+184Q, 6.30749438488829733664127973626729614e+187Q, 5.28039699582730274851611343025005913e+190Q, 4.91356459266260178144685151251583280e+193Q, 5.09061556382806163145943155929283695e+196Q, 5.88195913486315120755496558081929324e+199Q, 7.59272859182451584412681161192886089e+202Q, 1.09687154447604565252927157604063023e+206Q, 1.77650105264457177914887205035954057e+209Q, 3.23153511879574325299022213712900198e+212Q, 6.61424415330008555964088497876375007e+215Q, 1.52611134962392865764465743843181152e+219Q, 3.97691502072254861067816109956158434e+222Q, 1.17271592685350180601663411607927347e+226Q, 3.92076851079784438154800269803529292e+229Q, 1.48915601904650135853122181753421959e+233Q, 6.43831323019893638007682069747894997e+236Q, 3.17506717851901774760997889290937204e+240Q, 1.78970981603629426034165602936202663e+244Q, 1.15551435950001094003467361479483663e+248Q, 8.56366431033316582912879348191377839e+251Q, 7.30093770528170061921940254565023758e+255Q, 7.17613138815551609337504025522760569e+259Q, 8.15019404042552378087870259624107791e+263Q, 1.07201229194901841245742935320285314e+268Q, 1.63677814658507005123653360340510968e+272Q, 2.90775562270833690431514296460636317e+276Q, 6.02476179304325236422526103127046739e+280Q, 1.45944561364291559252990517774206937e+285Q, 4.14353248208966528107409825739318649e+289Q, 1.38220972739972368321976302536673924e+294Q, 5.43125471568796436605394023488404770e+298Q, 2.52039989595272682474151246524004609e+303Q, 1.38490332203279628724542169932072504e+308Q, 9.03453415480179250943103381623811201e+312Q, 7.01620311300324094081813342355630834e+317Q, 6.50430483422567669743617206850738485e+322Q, 7.21793562600493199951776097195343971e+327Q, 9.61542788471746566451160668498411528e+332Q, 1.54211655565340209996546117001243607e+338Q, 2.98626205265198316430063943341329432e+343Q, 7.00309320577422492544778192980108604e+348Q, 1.99486089518136492496460973366880635e+354Q, 6.92350663128275446878446973158773950e+359Q, 2.93684755383592258491626945473185368e+365Q, 1.52739293250863294426010436780725812e+371Q, 9.77075509850276765943058812672579852e+376Q, 7.71310608088780517993569363011828070e+382Q, 7.53863036472313339350939561575995488e+388Q, 9.15331385155864163687518059055937567e+394Q, 1.38538605697398143363179835422022577e+401Q, 2.62287133256068615930266548676255190e+407Q, 6.23343982286571042809890547122705804e+413Q, 1.86628588041890755929921756251934723e+420Q, 7.06492694188791912037300528166794228e+426Q, 3.39406886800977749230458579824930803e+433Q, 2.07704891812976483907563047536681219e+440Q, 1.62532865525590176989706193358042030e+447Q, 1.63262112354994112352218911327913750e+454Q, 2.11342684365334123128646755974119857e+461Q, 3.53982137873383101244091847412793714e+468Q, 7.70246700302666062188915359100825615e+475Q, 2.18636422788338230254129077405708124e+483Q, 8.12975261672049081897166938196495657e+490Q, 3.97686557044826916151646962242652853e+498Q, 2.57033035969007091507617008526033295e+506Q, 2.20458407348036235529376715504150602e+514Q, 2.52051509380683369529400145083533353e+522Q, 3.85871376898138292734730034828559871e+530Q, 7.94663225793828689108907007781152553e+538Q, 2.21176541223395827472447113026924253e+547Q, 8.35931033048630497134082029598865726e+555Q, 4.31091797212556485985662056018893390e+564Q, 3.04833317305882254988404372326531988e+573Q, 2.97034828681146526635277450855709161e+582Q, 4.00864404566992526097179834506253327e+591Q, 7.53113144131492631353410204171028663e+600Q, 1.97996763586022811207860344830247407e+610Q, 7.32300332662100885299893727481183608e+619Q, 3.83078680906847773947984890776183687e+629Q, 2.84987258373806149506606754536239806e+639Q, 3.03186631046306534459943952897407369e+649Q, 4.63861318260680959360747724936035922e+659Q, 1.02646622253495677529633262335412306e+670Q, 3.30448344760519942319105428211388701e+680Q, 1.55678514244926540491694986676104766e+691Q, 1.07975219247204914347510266773073325e+702Q, 1.10926415409608640204268413223928121e+713Q, 1.69843188882084389205231670587156933e+724Q, 3.90025209215096037836224226547246383e+735Q, 1.35188796159061122898267917047667153e+747Q, 7.11881895741722013661309086794573765e+758Q, 5.73263787962641176699015570741387791e+770Q, 7.10698682909851371094390920546530663e+782Q, 1.36568905751347705631533710470350932e+795Q, 4.09591384091714486453161283605440034e+807Q, 1.93075428208975454032168180693618424e+820Q, 1.44069783794868490577087909226154603e+833Q, 1.71407451380697883211569548277738601e+846Q, 3.27557874564463748685767424892341420e+859Q, 1.01295195987435253561137255728975319e+873Q, 5.10768437019276174029340754351596177e+886Q, 4.23191986113702770379280941046346950e+900Q, 5.80664714178397794019307157189161329e+914Q, 1.32995815980583547946711112525038194e+929Q, 5.12600842672542100652183507817331030e+943Q, 3.35203762278831767267193293263751763e+958Q, 3.75009417204153826893896307870937057e+973Q, 7.23855050824188975713471018223024339e+988Q, 2.43146178229439003194650931378387467e+1004Q, 1.43376736234835679429446577737976919e+1020Q, 1.49738681485939897351147361037410278e+1036Q, 2.79474808763788045317059787958854766e+1052Q, 9.40752795141195895210630116566470676e+1068Q, 5.76455681194134988563523968421696548e+1085Q, 6.49098750070655215612470119412325218e+1102Q, 1.35603677524062211064037225514112165e+1120Q, 5.30731596106421482379267500594247426e+1137Q, 3.93020043869734370491941322540535807e+1155Q, 5.56226722613029149136813618379760709e+1173Q, 1.51990408820680241841102181882025951e+1192Q, 8.10228003417607034858018290055212984e+1210Q, 8.51521180721994744419289716417963237e+1229Q, 1.78329495296502824223674285976432097e+1249Q, 7.52325725334531880022021234450122858e+1268Q, 6.46447702663797449769360764507187142e+1288Q, 1.14411796435953528891736478925455547e+1309Q, 4.21851818337788742448241659152628519e+1329Q, 3.27808619028067308419286053048979085e+1350Q, 5.43188779501763188996015031600338462e+1371Q, 1.94236185576822902995555282219996135e+1393Q, 1.51711647857248503907031482339088565e+1415Q, 2.62035716514505778741231466041668281e+1437Q, 1.01340333806343444348385264050569250e+1460Q, 8.88786410219005270769408353397020687e+1482Q, 1.79062851327433476234422894649930707e+1506Q, 8.39642792144232596950658702528918677e+1529Q, 9.28628794761707451127390451820966789e+1553Q, 2.45536895088636462839367313490581387e+1578Q, 1.57354562229875356978426356241845302e+1603Q, 2.47847583619098850454983740131691503e+1628Q, 9.73157119375940175602769413110414643e+1653Q, 9.66320447484532719750186034583884499e+1679Q, 2.46232109538721189753853963925622165e+1706Q, 1.63417596934387293040006746046570718e+1733Q, 2.86767794799456586471233013913198039e+1760Q, 1.35110548955493375904544893914917995e+1788Q, 1.73592152545700077097814839827674920e+1816Q, 6.17893831026015926230407301367680801e+1844Q, 6.19169576655007716364858612984691562e+1873Q, 1.77539961419331109157892833779912252e+1903Q, 1.48102931564535770775910220632570312e+1933Q, 3.65523833138677287346435504279580939e+1963Q, 2.71500576533505130343548298805675281e+1994Q, 6.17538661057993294979124427622665421e+2025Q, 4.37773911614056336228335278244614535e+2057Q, 9.84696637606677900175890481086698860e+2089Q, 7.15679932872021171306956409599896671e+2122Q, 1.71206538074954819527676482193571839e+2156Q, 1.37358203471630226464631105286036694e+2190Q, 3.76701670663930120912543386433302245e+2224Q, 3.60043887764831236838085599723119708e+2259Q, 1.22311399177140229476034440208616729e+2295Q, 1.50662485913882109231872682750679813e+2331Q, 6.86720031172136555587682118613084257e+2367Q, 1.18233110675109495729851537371527994e+2405Q, 7.85186218386652904615339091354671418e+2442Q, },
   };
   m_committed_refinements = static_cast<boost::math::detail::atomic_unsigned_integer_type>(m_weights.size() - 1);
   m_t_min = -8.89265635039934944849389055479809884Q;
   if (m_max_refinements >= m_abscissas.size())
   {
      m_abscissas.resize(m_max_refinements + 1);
      m_weights.resize(m_max_refinements + 1);
   }
   else
   {
      m_max_refinements = m_abscissas.size() - 1;
   }
}
#endif

}
}
}
}
#endif
