#include <assert.h>
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <linux/fdreg.h>
#include <linux/fd.h>
#include <syslog.h> 
#include "dbootstrap.h"
#include "lang.h"
#include "util.h"
#include <stdlib.h>
#include <dirent.h>
#include <slang.h>
#include <newt.h>

#if #cpu(sparc)
# define USE_SILO
#elif #cpu(m68k)
# define USE_LILOM68K
#elif #cpu(powerpc)
# define USE_PLILO
#elif #cpu(alpha)
# define USE_ABOOT
#elif #cpu(arm)
/* netwinder doesn't need boot config support (?); unsure about riscpc req'mts */
#elif #cpu(hppa)
# define USE_PALO
#elif #cpu(ia64)
# define USE_ELILO
#elif #cpu(s390)
# define USE_ZIPL
/* for now MIPS port only supports ip22 */
#elif ( #cpu(mips) && !(defined (MIPSEL)) )
# define USE_DVHTOOL
#elif ( #cpu(mips) && defined (MIPSEL) ) 
# define USE_DELO
#else
# define USE_LILO
#endif

#ifdef _TESTING_
#define BC_DIR_PREFIX  ""
#define BC_LILO_CONF   "etc_lilo.conf_test"
#define BC_DELO_CONF   "etc_delo.conf_test"
#define BC_INST_MBR    "echo pretending to install-mbr device=%s"
#define BC_BOOT_DFLT   "echo pretending to activate device=%s, partition=%s"
#define BC_RUN_LILO    "echo pretend running lilo..."
#define BC_RUN_ELILO   "echo pretend running elilo..."
#define BC_RUN_DELO    "echo pretend running delo..."
#define BC_SFORMAT     "/usr/bin/superformat"
#define BC_RUN_SFORMAT "echo pretending to superformat device=%s, size=%s"

#define BC_ABOOT_CONF  "etc_aboot.conf_test"
#define BC_MILO_CONF   "etc_milo.conf_test"
#define BC_APB_CFG     "dos_abp.cfg_test"
#define BC_RUN_SWRITEBOOT "/bin/echo /sbin/swriteboot -v %s /boot/bootlx"
#define BC_RUN_ABOOTCONF "/bin/echo /sbin/abootconf %s %s"
#define BC_RUN_MILOCTL "/bin/echo /sbin/miloctl cmd=\"boot %s:\" /instmnt/milo"
#define BC_RUN_CATSBOOT "/bin/echo chroot /target /usr/sbin/catsboot /vmlinuz /vmlinuz"
#define BC_RUN_ZIPL    "/bin/echo chroot /target /sbin/zipl"
#else

#define BC_DIR_PREFIX  "/target"
#define BC_LILO_CONF   "/target/etc/lilo.conf"
#define BC_DELO_CONF   "/target/etc/delo.conf"
#define BC_INST_MBR    "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/install-mbr %s"
#define BC_BOOT_DFLT   "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/activate %s %s"
#define BC_RUN_LILO    "chroot /target /sbin/lilo >/dev/null"
#define BC_RUN_ELILO   "chroot /target /usr/sbin/elilo --autoconf --boot %s --root %s --efiboot"
#define BC_RUN_DELO    "chroot /target /sbin/delo"
#define BC_SILO_CONF   "/target/etc/silo.conf"
#define BC_RUN_SILO    "chroot /target /sbin/silo -f >/dev/null"

#define BC_PALO_CONF   "/target/etc/palo.conf"
#define BC_RUN_PALO    "chroot /target /sbin/palo"

#define BC_RUN_YBIN   "yabootconfig -q -t /target -b %s -r %s"
#define BC_QUIK_CONF  "/target/etc/quik.conf"
#define BC_RUN_QUIK   "/target/sbin/quik -v -f -r /target"

#define BC_RUN_CATSBOOT "chroot /target /usr/sbin/catsboot /vmlinuz /vmlinuz"

/* Technically, LD_LIBRARY_PATH is not needed here, since we have
   swriteboot and abootconf on the root disk (for rescue use) but oh well */
#define BC_ABOOT_CONF "/target/etc/aboot.conf"
#define BC_RUN_SWRITEBOOT "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/swriteboot -f3 %s /target/boot/bootlx"
#define BC_RUN_ABOOTCONF "export LD_LIBRARY_PATH\"=/target/lib:/target/usr/lib\"; /target/sbin/abootconf %s %s"

#define BC_MILO_CONF  "/target/etc/milo.conf"
#define BC_RUN_MILOCTL "/sbin/miloctl cmd=\"boot %s:\" /instmnt/milo"

#define BC_APB_CFG "/target/dos/apb.cfg"

#define BC_VMELILO_CONF  "/target/etc/vmelilo.conf"
#define BC_RUN_VMELILO    "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; /target/sbin/vmelilo -f -w /target >/dev/null"

#define BC_SFORMAT     "/target/usr/bin/superformat"
#define BC_RUN_SFORMAT "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; "BC_SFORMAT" %s %s"

#define BC_PREPDD      "/target/bin/dd"
#define BC_RUN_PREPDD  "export LD_LIBRARY_PATH=\"/target/lib:/target/usr/lib\"; "BC_PREPDD" if=%s of=%s 2>/dev/null"

#define BC_ZIPL_CONF   "/target/etc/zipl.conf"
#define BC_PARMFILE    "/target/boot/parmfile"
#define BC_RUN_ZIPL    "chroot /target /sbin/zipl"

#endif

static int write_boot_floppy (char *device)
{
  struct stat tmp;
  FILE* filep;
  int size, status;
  const char* fs_type = "auto";

  int fd;
  struct floppy_struct floppy_type;

  snprintf(prtbuf, sizeof(prtbuf), "umount %s", device);
  execlog(prtbuf, LOG_DEBUG);
  execlog("umount /floppy", LOG_DEBUG);

#if #cpu (powerpc)
  if (strcmp(Arch2, "PReP") == 0) {
      pleaseWaitBox(_("Copying the operating system kernel..."));
      snprintf(prtbuf, sizeof(prtbuf), BC_RUN_PREPDD, "/target/vmlinuz", device);
      status = execlog(prtbuf, LOG_INFO);
      boxPopWindow();
      if (status) {
	  eject_floppy(device);
	  return 1;
      }
      else return 0;
  }
#endif

#ifdef SCSI_FLOPPY
  if (strncmp(device, "/dev/sfd", 8)) /* not scsi floppy */
#endif
#if 0 /*PETA*/
    fdflush(device);
#else
    {
       int fd = open(device, 0);

       if (fd) {
	 ioctl(fd, FDFLUSH, 0);
	 close(fd);
       }
    }
#endif

  /*
   * get floppy geometry
   *
   * This code is taken from getfdprm, fdutils-4.3 by Alain Knaff
   * Alain.Knaff@inrialpes.fr, released under GPL.
   */

  fd = open(device, 3);
  if ( fd < 0 )
    fd = open(device, O_RDONLY);
  if ( fd < 0 ) {
    ERRMSG("error opening %s (%d)", device, fd);
    return 1;
  }
  
  if (ioctl(fd, FDGETPRM, &floppy_type) < 0) {
    close(fd);
    ERRMSG("error getting floppy geometry from %s", device);
    return 1; 
  }
  close(fd);

  /*
   * End of the code taken from getfdprm.c in fdutils.
   */

  size = floppy_type.size / 2;
  INFOMSG("write_boot_floppy: found floppy of size %d", size);

  /* We don't have (yet) 2.88 MB bootable floppy images to use with syslinux */
  if (size == 2880) size = 1440;


  if (stat(BC_SFORMAT, &tmp) == 0) {
    boxSuspend();
    printf(CLEAR);
    printf(_("Formatting the floppy...\n"));
    /* We don't have (yet) 2.88 MB bootable floppy ... */
    snprintf(prtbuf, sizeof(prtbuf), BC_RUN_SFORMAT, device,
             (size > 720) ? "hd" : "dd");
    /* DO NOT USE execlog() */
    INFOMSG("formatting floppy with cmd '%s'", prtbuf);
    status = system(prtbuf);
    boxResume();
    if (status) return 1;
  }
    
  pleaseWaitBox(_("Creating a filesystem on the floppy..."));

#if #cpu (m68k)
  if (strcmp(Arch2, "Atari") == 0)
     snprintf(prtbuf, sizeof(prtbuf), "zcat %s/usr/lib/atari-bootstrap/img%d%s.gz > %s",
	     BC_DIR_PREFIX,size,Arch2,device);
#elif #cpu (i386)
  snprintf(prtbuf, sizeof(prtbuf), "zcat %s/usr/lib/syslinux/img%dk.gz > %s",
	  BC_DIR_PREFIX,size,device);
#elif #cpu (sparc)
  /* read it from miniroot fs */
  snprintf(prtbuf, sizeof(prtbuf), "zcat %s/usr/lib/silo/silo%dk-%s.gz > %s",
	  BC_DIR_PREFIX,size,Arch2,device);
#elif #cpu (alpha)
  /* XXX: maybe we should do the same thing as Sparc? (i.e. make a "template" image) */
  snprintf(prtbuf, sizeof(prtbuf), "/sbin/mkfs.ext2 -Onone %s >/dev/null 2>&1 && "
	   "/sbin/e2writeboot %s /target/boot/bootlx >/dev/null",
	   device, device);
  fs_type = "ext2";
#endif
  status = execlog(prtbuf, LOG_INFO);

  boxPopWindow();
  if (status)
    return 1;

  snprintf(prtbuf, sizeof(prtbuf), "mount -t %s %s /floppy", fs_type, device);
  status = execlog(prtbuf, LOG_INFO);
  if (status)
    return 1;

  pleaseWaitBox(_("Copying the operating system kernel..."));

  snprintf(prtbuf, sizeof(prtbuf), "cp %s/vmlinuz /floppy/linux.bin", BC_DIR_PREFIX);
  status = execlog(prtbuf, LOG_INFO);

  if (status) {
    boxPopWindow();
    return 1;
  }

  sync();

  boxPopWindow();
  
  pleaseWaitBox(_("Writing the configuration files...\n"));
#if #cpu (m68k)
  if (strcmp(Arch2, "Atari") == 0) {
    if ((filep = fopen("/floppy/bootargs", "w")) == NULL) {
      execlog("umount /floppy", LOG_INFO);
      boxPopWindow();
      return 1;
    }
    fprintf(filep, "-s -k a:\\linux.bin root=%s %s", Root->name, append_opts);
    fclose(filep);
  }
#elif #cpu (i386)
  if ((filep = fopen("/floppy/syslinux.cfg", "w")) == NULL) {
    execlog("umount /floppy", LOG_INFO);
    boxPopWindow();
    return 1;
  }
  fprintf(filep, "DISPLAY message.txt\n");
  fprintf(filep, "TIMEOUT 40\n");
  fprintf(filep, "PROMPT 1\n");
  fprintf(filep, "DEFAULT linux.bin\n");
  fprintf(filep, "APPEND root=%s ro %s\n", Root->name, append_opts);
  fclose(filep);
#elif #cpu (sparc)
  if ((filep = fopen("/floppy/etc/silo.conf", "w")) == NULL) {
    execlog("umount /floppy", LOG_INFO);
    boxPopWindow();
    eject_floppy(device);
    return 1;
  }
  fprintf(filep, "partition=3\n");
  fprintf(filep, "root=%s\n", device);
  fprintf(filep, "timeout=100\n");
  fprintf(filep, "image=/linux.bin\n");
  fprintf(filep, "label=linux\n");
  fprintf(filep, "read-only\n");
  fprintf(filep, "append=\"root=%s %s\"\n", Root->name, append_opts);
  fclose(filep);
  /* no need to run silo at this time; the floppy disk is already bootable */
#elif #cpu (alpha)
  if (mkdir("/floppy/etc", 0755) == -1) {
    execlog("umount /floppy", LOG_INFO);
    boxPopWindow();
    return 1;
  }
  if ((filep = fopen("/floppy/etc/aboot.conf", "w")) == NULL) {
    execlog("umount /floppy", LOG_INFO);
    boxPopWindow();
    return 1;
  }
  fprintf(filep, "0:linux.bin ro root=%s %s\n", Root->name, append_opts);
  fclose(filep);
#endif
  
  if ((filep = fopen("/floppy/message.txt", "w")) != NULL) {
  /* FIXME: do we really want this file written as utf8? */
  /* TRANS: The next message has to be DOS formatted for syslinux. Please
   * use only ASCII characters.  */
    fprintf(filep, _("\nLinux will be started automatically using the kernel on this floppy disk.\nThe root filesystem will be mounted from %s .\n\nWhen you start to type a command at the \"boot:\" prompt, the automatic\nbootstrap process will stop. You may then type the name of the kernel to\nuse followed by any kernel options in the form option=value.\n\nThe kernel must be on the floppy disk in the first floppy drive. A kernel\nfile called \"linux.bin\" was installed on this floppy when it was created. If you\nwish to use a kernel on the hard disk, remove the floppy and press RESET\n\n"), Root->name);
    fclose(filep);
    status = 0;
  } else 
    status = 1;
  
  boxPopWindow();
  execlog("umount /floppy", LOG_INFO);

  /* eject the floppy disk if avail (see dbootstrap.h) */
  eject_floppy(device);

  return status;
}

int make_boot_floppy (void)
{
  char* device = "/dev/fd0";
  char* designation = _("first");
  char* formatted = "";
  int result;
  struct stat tmp;
  int notCreatedBootFloppy_save;

  /* In case any of the problem boxes below return */
  notCreatedBootFloppy_save = notCreatedBootFloppy;
  notCreatedBootFloppy = 0;
#if #cpu (m68k)
  if (strcmp(Arch2, "Amiga") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for Amiga."),
               _("Problem"));
    return 0;
  }
  if (strcmp(Arch2, "VME") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for VME."),
               _("Problem"));
    return 0;
  }
  if (strcmp(Arch2, "Macintosh") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for the Macintosh."),
               _("Problem"));
    return 0;
  }
#endif

#if #cpu (powerpc)
  if (strcmp(Arch2, "apus") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for PowerPC/apus."),
               _("Problem"));
    return 0;
  }
  if (strcmp(Arch2, "CHRP") == 0) {
    problemBox(_("Creating a boot floppy is still not possible for PowerPC/CHRP."),
               _("Problem"));
    return 0;
  }
  if (strstr(Arch2, "PowerMac")) {
    problemBox(_("Creating a boot floppy is still not possible for PowerPC/PowerMac."), 
               _("Problem"));
    return 0;
  }
#endif

#if #cpu (alpha)
  if (! srm_boot) {
    /* Not SRM, no boot floppy */
    problemBox(_("Creating a boot floppy is still not possible for Alphas that boot from MILO"),
	       _("Problem"));
    return 0;
  }
#endif

  notCreatedBootFloppy = notCreatedBootFloppy_save;

  if (strcmp(InstallationRootDevice, "/dev/fd0") == 0) {
    snprintf(prtbuf, sizeof(prtbuf),
             _("You are not using the RAM-disk, and thus the Installation Root Disk in the first floppy drive cannot be removed.\nIf your second floppy drive is the same size as the first one, you can use it to write a boot floppy which you will place in the first drive when you boot the system. Otherwise, you can skip this step and use the Debian Rescue floppy to boot your hard disk by typing \"linux root=%s\" at the \"boot: \" prompt, or you may be able to boot directly from the hard disk.\nUse the second floppy drive to create a boot floppy?"),
             Root->name);
    if (yesNoBox(prtbuf, _("Use Second Floppy Drive?")) == DLG_NO)
      return 1;
    device = "/dev/fd1";
    designation = _("second");
  }

  if (stat(BC_SFORMAT, &tmp) != 0) {
    formatted = _(", formatted");
  }

  vaproblemBox(_("Change Disk"), 
               _("Please place a blank%s floppy disk in the %s floppy "
                 "disk drive, and press ENTER."),
               formatted, designation);

  result = write_boot_floppy(device);
  if (result != 0) {
      vaproblemBox(_("Problem"), 
                   _("Creation of the boot floppy failed. Please make sure that "
                     "the floppy was not write-protected, and that you put it "
                     "in the %s drive. Try another floppy if the problem persists."),
                   designation);
  } else {
    notCreatedBootFloppy = 0;
  }  
  return result;
}

static int firstnumber(const char* boot)
{
  int idx, ch;
  idx = strlen(boot);
  do {
    idx--;
    ch = boot[idx];
  } while ((ch >= '0') && (ch <= '9'));
  idx++;
  return idx;
}

#if #cpu (i386)
#define DEV_HDA      "/dev/hda"
#define DEV_SDA      "/dev/sda"
#define DEV_RD_C0D0  "/dev/rd/c0d0"
#define DEV_IDA_C0D0 "/dev/ida/c0d0"
#define DEV_CCISS_C0D0 "/dev/cciss/c0d0"
#define DEV_ATARAID_D0 "/dev/ataraid/d0"

/* part_disk_name: return name of partition's disk device, if available */
char *part_disk_name(struct fdisk_partition *part)
{
  return (part->disk ? part->disk->name : NULL);
}

/* is_first_scsi: return true if partition is on the first SCSI disk */
int is_first_scsi(struct fdisk_partition *part)
{
  char *dname = part_disk_name(part);
  return dname && (strcmp(dname, DEV_SDA)      == 0 ||
		   strcmp(dname, DEV_RD_C0D0)  == 0 ||
         strcmp(dname, DEV_CCISS_C0D0)  == 0 ||
		   strcmp(dname, DEV_IDA_C0D0) == 0 ||
         strcmp(dname, DEV_ATARAID_D0) == 0);
}

/* is_first_ide: return true if partition is on the first IDE disk */
int is_first_ide(struct fdisk_partition *part)
{
  char *dname = part_disk_name(part); //FIXME - check how ide vs. scsi is handled
  return dname && strcmp(dname, DEV_HDA) == 0;
}

/* is_logical_part: return true if partition is an MS-DOS logical partition */
int is_logical_part(struct fdisk_partition *part)
{
  if (part->disk && strcmp(part->disk->partition_format, "msdos") == 0) {
    char *s, *t;

    assert(*part->name);
    /* find the first digit in the part->name, set t to the rest of the string */
    for (s = part->name, t = NULL; *s; s++) {
      if (isdigit(*s)) {
	t = s;
	break;
      }
    }
    if (t) {
      assert(strlen(t) <= 5);
      return(atoi(t) > 4);
    }
  }
  return 0;
}

/* !!!! strlen and sizeof are not the same thing. Please don't change this
 * again !!!!!! -randolph
 */
/* is_ide: returns true if partition is on an IDE disk */
int is_ide(const char *name)
{
  return strncmp(name, DEV_HDA, strlen(DEV_HDA) - 1) == 0;
}

/* is_scsi: returns true if partition is on a SCSI disk */
int is_scsi(const char *name)
{
  return strncmp(name, DEV_SDA, strlen(DEV_SDA) - 1) == 0;
}

/* is_scsird: returns true if partition is on a SCSI disk (/dev/rd/c*) */
int is_scsird(const char *name)
{
  return strncmp(name, DEV_RD_C0D0, strlen(DEV_RD_C0D0) - 3) == 0;
}
  
/* is_scsiida: returns true if partition is on a SCSI disk (/dev/rd/c*) */
int is_scsiida(const char *name)
{
  return strncmp(name, DEV_IDA_C0D0, strlen(DEV_IDA_C0D0) - 3) == 0;
}
/* is_scsicciss: returns true if partition is on a CCISS disk (/dev/cciss/c*) */
int is_scsicciss(const char *name)
{
  return strncmp(name, DEV_CCISS_C0D0, strlen(DEV_CCISS_C0D0) - 3) == 0;
}
/* is_ataraid: returns true if partition is on a SCSI disk (/dev/ataraid/d*) */
int is_ataraid(const char *name)
{
  return strncmp(name, DEV_ATARAID_D0, strlen(DEV_ATARAID_D0) - 3) == 0;
}
  
/* Return a string containing the name of the first existing IDE disk.  */
char *first_ide_disk(void)
{
  static char disk[] = "/dev/hda";

  for (; disk[strlen(disk) - 1] < 'l'; (disk[strlen(disk) - 1])++)
    if (fdisk_find_disk(disk) != NULL)
      return disk;

  return NULL;
}

/* Return a string containing the name of the first existing SCSI disk.  */
char *first_scsi_disk(void)
{
  static char disk[] = "/dev/sda";

  for (; disk[strlen(disk) - 1] < 'l'; (disk[strlen(disk) - 1])++)
    if (fdisk_find_disk(disk) != NULL)
      return disk;

  return NULL;
}
#endif /* #cpu (i386) */


#ifdef USE_SILO
/* SILO is a special boot loader. It can actually read ext2, UFS and
 * iso9660 partitions. This means that it uses actual paths at boot time
 * as opposed to sectors/blocks like lilo does. Because of this we have to
 * take special care when "/boot" is not on the same partion as "/". So
 * how do we do this you say? We always set "partition" to Boot->name,
 * then if we find that /boot is a seperate partition, we muck around with
 * the vmlinuz symlinks to make everything happy. */
static int run_silo(void)
{
  FILE* filep;
  int idx;
  char *boot_part, *root_part;
  int retcode;

  if ((Boot = fdisk_find_partition_by_mntpoint ("/target/boot")) == NULL)
      Boot = Root;
  
  idx=firstnumber(Boot->name);
  boot_part=Boot->name+idx;
  idx=firstnumber(Root->name);
  root_part=Root->name+idx;

  pleaseWaitBox(_("Running SILO to make the kernel bootable from the hard disk without a boot floppy..."));
  if((filep=fopen(BC_SILO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }

  fprintf(filep,"partition=%s\n",boot_part);
  fprintf(filep,"root=%s\n",Root->name);
  fprintf(filep,"timeout=100\n");
  /* This is where it gets tricky */
  /* We need to create a symlink on the boot partition to the kernel image. */
  if (Boot != Root) {
      char buf[256];
      char *p;
      int count = 0;

      if ((count = readlink("/target/vmlinuz", buf, sizeof(buf))) != -1) {
	  buf[count] = '\0';

	  /* strip leading directories off target name */
	  if ((p = strrchr(buf, '/')) == NULL) {
	      p = buf; /* no '/' found */
	  } else {
	      p = p + 1; 
	  }

	  unlink("/target/boot/vmlinuz"); /* make sure this is gone */

	  if ((symlink(p, "/target/boot/vmlinuz")) == -1) {
	      ERRMSG("Error creating /target/boot/vmlinuz symlink");
	      return 1;
	  }
      }
      else {
	  ERRMSG("Error resolving /target/vmlinuz symlink");
	  return 1;
      }
  }
  fprintf(filep,"image=%s/vmlinuz\n", boot_part); /* just to make sure the partition is right */
  fprintf(filep,"label=linux\nread-only\n");
  fclose(filep);
  sleep(1);		/* wait some amount of time to see the message box */
  retcode = (execlog(BC_RUN_SILO, LOG_INFO) == 0) ? 0 : 1;
  boxPopWindow();
  return retcode;
}
#endif

#ifdef USE_PALO   /* HPPA boot loader */
static int run_palo(void)
{
  FILE* filep;
  int idx;
  char *boot_part, *root_part;
  char *boot_disk;
  char buf[256];
  char *name;
  int count = 0;

  if ((count = readlink("/target/vmlinux", buf, sizeof(buf))) != -1)
    buf[count] = '\0';
  else {
    ERRMSG("Unable to resolve /target/vmlinux: %s", strerror(errno));
    boxPopWindow();
    problemBox(_("The requested operation cannot be performed because "
		"the /target/vmlinux symlink could not be resolved."), 
		_("Need valid symlink"));
    return 1;
  }

    /* strip leading directories off target name */
    if ((name = strrchr(buf, '/')) == NULL)
      name = buf; /* no '/' found */
    else
      name = name + 1; 

  if ((Boot = fdisk_find_partition_by_mntpoint ("/target/boot")) == NULL)
      Boot = Root;

  boot_disk = strdup( Boot->name ); /* We extract the device */
  idx=firstnumber(boot_disk);
  boot_disk[idx] = '\0'; /* Cut the partition number */ 
  idx=firstnumber(Boot->name);
  boot_part=Boot->name+idx;
  idx=firstnumber(Root->name);
  root_part=Root->name+idx;

  unlink("/target/boot/vmlinux"); /* make sure this is gone */

  if ((symlink(name, "/target/boot/vmlinux")) == -1) {
    ERRMSG("Error creating /target/boot/vmlinux symlink: %s", strerror(errno));
    return 1;
  }
  else {
    unlink("/target/vmlinux"); /* remove this so it won't confuse people */
    if (NULL != (filep = fopen("/target/etc/kernel-img.conf", "w"))) {
      fprintf(filep, "## Generated by dbootstrap palo configuration\n" );
      fprintf(filep, "## Create relative symlinks in /boot incase it's a partition\n");
      fprintf(filep, "## We don't handle floppies yet, and there is no need to rerun\n");
      fprintf(filep, "## palo every time the kernel is updated if using symlinks.\n\n");
      fprintf(filep, "do_symlinks = Yes\n");
      fprintf(filep, "image_in_boot = Yes\n");
      fprintf(filep, "relative_links = Yes\n");
      fprintf(filep, "do_bootfloppy = No\n");
      fprintf(filep, "do_bootloader = No\n");
      fclose(filep);
    } else {
      ERRMSG("Error creating /target/etc/kernel-img.conf: %s", strerror(errno));
      return 1;
    }
  }

  pleaseWaitBox(_("Running PALO to make the kernel bootable from the hard disk..."));
  if((filep=fopen(BC_PALO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }

  if (Boot == Root)
    fprintf(filep,"--commandline=%s/boot/vmlinux root=%s HOME=/\n", boot_part, Root->name);
  else
    fprintf(filep,"--commandline=%s/vmlinux root=%s HOME=/\n", boot_part, Root->name);
  fprintf(filep,"--recoverykernel=/boot/%s\n", name);
  fprintf(filep,"--init-partitioned=%s\n", boot_disk);
  free( boot_disk ); /* Free used memory */
  fclose(filep);
  sleep(1);             /* wait some amount of time to see the message box */
  boxPopWindow();
  if( execlog(BC_RUN_PALO, LOG_INFO) ) return 1;
  return 0;
}
#endif

#ifdef USE_PLILO  /* PowerPC-LILO: for the moment it is Quik for everything except NewWorld PowerMacs and CHRP which use yaboot */

#define DEV_HDA      "/dev/hda"
#define DEV_SDA      "/dev/sda"

/* Quik setup for OldWorld powermacs and some non-apple powerpcs */
static int run_plilo_quik(void)
{
    FILE* filep;
    char vmlinux[256], *partition;
    char ofdev[1024];
    struct stat statbuf;
    int idx;
    int exitcode;
    int oldworld = 0;
    int retcode = 0;

    idx = firstnumber(Root->name);
    partition = Boot->name+idx;

/* 
    Requirements for quik

    First disk (ide or scsi -- probably scsi only)
    quik doesn't like /vmlinux symlink?
    nvsetenv boot-device and perhaps boot-file
    /boot on same partition as root.
*/

  /* this makes things much more readable later */ 
  if ((Arch3 != NULL) && (!strcmp(Arch3, "OldWorld")))
      oldworld = 1;

/* if ATA, then it must be on the first disk. */
  if ((!strncmp("/dev/hd", Root->disk->name, 7)) && (0 != strcmp(DEV_HDA, Root->disk->name))) {
      problemBox(_("The requested action cannot be performed because the root "
		   "partition must be on the first disk."),
		 _("Root Must Be On First Disk"));
      return 1;
  }

/* /boot must be on the same partition as / */
  if ((Boot = fdisk_find_partition_by_mntpoint ("/target/boot")) == NULL)
        Boot = Root;

  if (Boot != Root) {
      problemBox(_("The requested action cannot be performed because /boot "
		   "must not be a separate partition."),
		 _("/boot Cannot Be a Partition"));
      return 1;
  }

  if (!is_fstype("/target", "ext2")) {
      problemBox(_("The requested action cannot be performed because the root "
		   "filesystem is not Ext2."), _("Root filesystem must be Ext2"));
      return 1;
  }

  /* Issue warning */
  if (strstr(Arch2, "PowerMac")) {   /* PowerMac specific warning */
      if (DLG_NO == yesNoBox(_("You have chosen to install the quik bootloader.  "
			       "You will not be able to boot any other "
			       "operating system from this disk. Furthermore, your "
			       "machine may not be bootable in any manner after "
			       "this process completes. If you are left with a blank "
			       "screen, you may need to try a cold boot and hold down "
			       "CMD-OPT-P-R. Finally, this code has not been "
			       "thoroughly tested. Are you sure that you want to do this?"),
			     _("Are You Sure?")))
	  return 1;
  }
  else {                              /* Non-PowerMac warning */
      if (DLG_NO == yesNoBox(_("You have chosen to install the quik bootloader.  "
			       "You will not be able to boot any other "
			       "operating system from this disk. Furthermore, your "
			       "machine may not be bootable in any manner after this "
			       "process completes. Finally, this code has not been "
			       "thoroughly tested. Are you sure that you want to do this?"),
			     _("Are You Sure?")))
	  return 1;
  }

  pleaseWaitBox(_("Running quik to make Debian bootable directly from the hard disk..."));

  /* 
     Translate /vmlinux symbolic link to real kernel 
     since apparently quik doesn't like the link.
  */
  if (NAME_ISLNK("/target/vmlinux", &statbuf)) {
      int count;

      if ((count = readlink("/target/vmlinux", vmlinux, sizeof(vmlinux))) != -1) {
	  vmlinux[count] = '\0';
      } else {
	  ERRMSG("Unable to resolve /target/vmlinux: %s", strerror(errno));
	  boxPopWindow();
	  problemBox(_("The requested operation cannot be performed because "
		       "the /target/vmlinux symlink could not be resolved."), 
		     _("Need valid symlink"));
	  return 1;
      }
  } else
      strcpy(vmlinux, "vmlinux");

  /* get OpenFirmware device path to root disk */ 
  snprintf(prtbuf, sizeof(prtbuf), "ofpath %s", Root->disk->name);
  if (execlog_to_string(prtbuf, &exitcode, ofdev, sizeof(ofdev)-1, 1) || exitcode) {
      ERRMSG("Unable to get OpenFirmware device name");
      boxPopWindow();
      problemBox(_("The requested operation cannot be performed because "
		   "the OpenFirmware device name for the root disk could "
		   "not be determined."), _("Unable to install Quik"));
      return 1;
  } else
      /* strip newline from ofpath output and replace it with 0, this gets us partition 0 */
      ofdev[strlen(ofdev)-1] = '0';

  /* Write /target/etc/quik.conf */
  if((filep = fopen(BC_QUIK_CONF, "w")) == NULL) {
      ERRMSG("Unable to create %s: %s", BC_QUIK_CONF, strerror(errno));
      retcode = 1;
  } else {
      fprintf(filep, "## This file generated during installation.\n");
      fprintf(filep, "## Debian GNU/Linux PowerPC\n\n");
      fprintf(filep, "init-message=\"Debian GNU/Linux PowerPC (woody)\"\n");
      fprintf(filep, "default=Linux\n");
      fprintf(filep, "timeout=100\n");
      fprintf(filep, "root=%s\n", Root->name);
      fprintf(filep, "partition=%s\n\n", partition);
      fprintf(filep, "## Do not point image= to a symlink, quik can't follow symlinks\n");
      fprintf(filep, "image=/%s\n", vmlinux); /* target symlink is relative */
      fprintf(filep, "\tlabel=Linux\n");
      fprintf(filep, "\tread-only\n");
      if (strlen(append_opts))
	  fprintf(filep, "\tappend=\"%s\"\n", append_opts);
      fclose(filep);
  }
  /* Execute quik */
  if (retcode != 1) {
      INFOMSG("executing: %s", BC_RUN_QUIK);
      retcode = (execlog(BC_RUN_QUIK, LOG_INFO) == 0) ? 0 : 1;
  }

  /* OldWorld mac -- setup openfirmware as best we can */
  if ((oldworld) && (retcode == 0)) {
      /* Execute nvsetenv */
      snprintf(prtbuf, sizeof(prtbuf), "nvsetenv boot-device %s", ofdev);
      INFOMSG( "executing: %s", prtbuf); 
      if(execlog(prtbuf, LOG_INFO)) {
	  boxPopWindow();
	  problemBox(_("Unable to set the OpenFirmware boot-device variable.\n\n"
		       "You will have to configure OpenFirmware yourself to boot."),
		     _("Unable to configure OpenFirmware"));
	  return 1;
      }
      INFOMSG( "executing: nvsetenv boot-command \"begin ['] boot catch 1000 ms cr again\""); 
      if(execlog("nvsetenv boot-command \"begin ['] boot catch 1000 ms cr again\"", LOG_INFO)) {
	  boxPopWindow();
	  problemBox(_("Unable to set the OpenFirmware boot-command variable.\n\n"
		       "You may have intermittent boot failures."), _("Notice"));
	  return 1;
      }
  }
  sleep(2);             /* wait some amount of time to see the message box */
  boxPopWindow();

  if (retcode == 1) {
      problemBox(_("Quik installation failed.\n\nYou won't be able to boot without a floppy."),
		 _("Problem"));
  }
  return retcode;
}

/* NewWorld PowerMac and CHRP bootloader setup */
static int run_plilo_yaboot(void)
{
    FILE* filep;
    int retcode = 0;
    struct fdisk_partition *bootstrap;

    /* 
       yaboot requirements

       800K Apple_Bootstrap partition must exist.
       Apple_Bootstrap must be on same disk as root.
    */
    if (strstr(Arch2, "PowerMac"))
        bootstrap = fdisk_find_partition_by_type(PTYPE_MAC_BOOT);
    else
        bootstrap = fdisk_find_partition_by_type(PTYPE_CHRP_BOOT);

    /* so we can deal with cross device symlinks */
    if ((Boot = fdisk_find_partition_by_mntpoint ("/target/boot")) == NULL)
	Boot = Root;

    /* /boot and / has to be on the same disk on non-PowerMac */
    if (!strstr(Arch2, "PowerMac")) {
	if (Boot->disk->name != Root->disk->name) {
	    problemBox (_("The requested operation can't be performed because "
			  "the /boot partition must be on the same disk as the "
			  "/ (root) partition."),
			_("Root and /boot partitions must be on the same disk"));
	    return 1;
	}
    }

    /* Check for an Apple_Bootstrap partition */
    if (bootstrap == NULL) {
        if (strstr(Arch2, "PowerMac")) {
	    problemBox (_("The requested operation can't be performed because "
			  "yaboot requires an Apple_Bootstrap partition.  "
			  "You must repartition your disk"),
			_("An Apple_Bootstrap partition is required"));
	    return 1;
	}
	else {
	    problemBox (_("The requested operation can't be performed because "
			  "yaboot requires a CHRP (type 0x41) bootstrap partition.  "
			  "You must repartition your disk"),
			_("A CHRP bootstrap partition is required"));
	    return 1;
	}
    }

    /* Make sure it's at least 800K */
    else if (800 > bootstrap->size) {
        if (strstr( Arch2, "PowerMac")) {
	    problemBox (_("The requested operation can't be performed because yaboot "
			  "requires an Apple_Bootstrap partition of at least 800K.  "
			  "You must repartition your disk"),
			_("Bootstrap partition must be 800K"));
	    return 1;
	}
	else {
	    problemBox (_("The requested operation can't be performed because yaboot "
			  "requires a CHRP bootstrap partition of at least 800K.  "
			  "You must repartition your disk"),
			_("Bootstrap partition must be 800K"));
	    return 1;
	}
    }

    /* Make sure it's on the same disk as the root disk */
    else if (0 != strcmp(bootstrap->disk->name, Root->disk->name)) {
        if (strstr(Arch2, "PowerMac")) {
	    problemBox (_("The requested operation can't be performed because yaboot "
			  "requires the Apple_Bootstrap and / partitions to be on the "
			  "same disk."),
			_("Bootstrap and root partitions must be on the same disk"));
	    return 1;
	} 
	else {
	    problemBox (_("The requested operation can't be performed because yaboot "
			  "requires the CHRP bootstrap and / partitions to be on "
			  "the same disk."),
			_("Bootstrap and root partitions must be on the same disk"));
	    return 1;
	}
    }

    /* Execute yabootconfig */
    else {

	pleaseWaitBox(_("Running yabootconfig to make Debian bootable directly from the hard disk..."));

	/* We need to create a symlink on the /boot partition to the
	   kernel image. cross device symlinks break yaboot */
	if (Boot != Root) {
	    char buf[256];
	    char *p;
	    int count = 0;

	    if ((count = readlink("/target/vmlinux", buf, sizeof(buf))) != -1) {
		buf[count] = '\0';

		/* strip leading directories off target name */
		if ((p = strrchr(buf, '/')) == NULL) {
		    p = buf; /* no '/' found */
		} else {
		    p = p + 1; 
		}

		unlink("/target/boot/vmlinux"); /* make sure this is gone */

		if ((symlink(p, "/target/boot/vmlinux")) == -1) {
		    ERRMSG("Error creating /target/boot/vmlinux symlink: %s", strerror(errno));
		    retcode = 1;
		} else {
		    unlink("/target/vmlinux"); /* remove this so it won't annoy yabootconfig */
		    if (NULL != (filep = fopen("/target/etc/kernel-img.conf", "w"))) {
			fprintf(filep, "## Generated by dbootstrap yaboot configuration\n" );
			fprintf(filep, "## Create relative symlinks in /boot since it's a partition\n\n");
			fprintf(filep, "do_symlinks = true\n");
			fprintf(filep, "image_in_boot = true\n");
			fprintf(filep, "relative_links = true\n");
			fclose(filep);
		    } else
			ERRMSG("Error creating /target/etc/kernel-img.conf: %s, ignoring", strerror(errno));
		}
	    }
	    else
		ERRMSG("Error resolving /target/vmlinux symlink: %s.  "
		       "Letting yabootconfig deal with it", strerror(errno));
	}

	if (retcode != 1) {
	    if (strlen(append_opts))
		snprintf(prtbuf, sizeof(prtbuf), BC_RUN_YBIN " --kernel-args \"%s\"",
			 bootstrap->name, Root->name, append_opts);
	    else
		snprintf(prtbuf, sizeof(prtbuf), BC_RUN_YBIN, bootstrap->name, Root->name);
	    INFOMSG( "executing: %s", prtbuf );
	    retcode = (execlog(prtbuf, LOG_INFO) == 0) ? 0 : 1;
	}

	sleep(2);             /* wait some amount of time to see the message box */
	boxPopWindow();
    }

    if (retcode == 1) {
	problemBox(_("Yaboot installation failed.\n\nYou won't be able to boot without "
		     "a CDROM or network boot server."), _("Problem"));
    }
    return retcode;
}

/* run right bootloader setup function for this subarch */
static int run_plilo(void)
{
    int ireturn = 0;

    if (((!strcmp(Arch2, "chrp")) ||
	 ((strstr(Arch2, "PowerMac")) &&
	  (!strcmp(Arch3, "NewWorld")))))
	ireturn = run_plilo_yaboot();
    else
	ireturn = run_plilo_quik();

    return ireturn;
}
#endif

#ifdef USE_LILOM68K
static int run_vmelilo(const char *boot)
{
  FILE* filep;

  pleaseWaitBox(_("Running VMELILO to make the kernel able to boot from the hard disk without a boot floppy..."));
  if((filep=fopen(BC_VMELILO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"default   = Linux\n");
  fprintf(filep,"boot      = %s\n", boot);
  fprintf(filep,"delay     = 2\n");
  fprintf(filep,"\n");
  fprintf(filep,"[boot]\n");
  fprintf(filep,"label     = Linux\n");
  fprintf(filep,"image     = /vmlinuz\n");
  fprintf(filep,"root      = %s\n", Root->name);
  fprintf(filep,"read-only\n");
  fclose(filep);
  sleep(1);		/* wait some amount of time to see the message box */
  boxPopWindow();
  if(strlen(boot)==0) return 1;
  if(execlog(BC_RUN_VMELILO, LOG_INFO)) return 1;
  return 0;
}
#endif

#ifdef USE_ABOOT /* Alpha equivalent to the above (for SRM only though) */
static int run_aboot(void)
{
  FILE *filep;
  int  idx;
  char *boot_part, *root_part, *device;

  if ((Boot = fdisk_find_partition_by_mntpoint ("/target/boot")) == NULL)
    Boot = Root;

  idx=firstnumber(Boot->name);
  boot_part=Boot->name+idx;
  idx=firstnumber(Root->name);
  root_part=Root->name+idx;

  /* write aboot.conf anyway since it's basically harmless */
  if ((filep = fopen(BC_ABOOT_CONF, "w")) == NULL) {
    problemBox(_("Could not write to /etc/aboot.conf"), _("Problem"));
    return 1;
  }

  /* /vmlinuz links to boot/vmlinuz-x.x.xx. Our aboot handles symbolic
     links so this is just fine. */
  if (serialConsole >= 0) {
    fprintf(filep, "0:%s/vmlinuz ro root=%s console=ttyS%d\n",
	    boot_part, Root->name, serialConsole);
  } else {
    fprintf(filep, "0:%s/vmlinuz ro root=%s\n",
	    boot_part, Root->name);
  }
  fclose(filep);

  /* Copied from SILO code */
  /* We need to create a symlink on the boot partition to the kernel image. */
  if (Boot != Root) {
      char buf[256];
      char *p;
      int count = 0;

      if ((count = readlink("/target/vmlinuz", buf, sizeof(buf))) != -1) {
	  buf[count] = '\0';

	  /* strip leading directories off target name */
	  if ((p = strrchr(buf, '/')) == NULL) {
	      p = buf; /* no '/' found */
	  } else {
	      p = p + 1; 
	  }

	  unlink("/target/boot/vmlinuz"); /* make sure this is gone */

	  if ((symlink(p, "/target/boot/vmlinuz")) == -1) {
	      ERRMSG("Error creaing /target/boot/vmlinuz symlink");
	      return 1;
	  }
      }
      else {
	  ERRMSG("Error resolving /target/vmlinuz symlink");
	  return 1;
      }
  }
  /* End copied from SILO code */

  device = strdup(Root->name);	/* "Memory leaks are quite acceptable in
				    many applications." -- Bjarne Stroustrup */
  device[idx] = '\0';
  /* Since running aboot can be harmful, ask first */
  if (! bootargs.isquiet ) {
    sprintf(prtbuf,_("You have chosen to write the aboot bootloader to the boot sectors of %s.  You will not be able to boot any other operating systems on this disk. Are you sure you want to do this?"),
	    device);
    if (yesNoBox(prtbuf, _("Are You Sure?")) == DLG_NO)
      return 0;
  }
  pleaseWaitBox(_("Writing aboot to the first sector of the hard disk..."));
  sleep(1);
  boxPopWindow();
  {
    snprintf(prtbuf, sizeof(prtbuf), BC_RUN_SWRITEBOOT, device);
    if (execlog(prtbuf, LOG_INFO)) return 1;
    snprintf(prtbuf, sizeof(prtbuf), BC_RUN_ABOOTCONF, device, root_part);
    if (execlog(prtbuf, LOG_INFO)) return 1;
  }
  return 0;
}

static int mount_dos_fs(const char* dev, const char* mount)
{
  int status;

  do {
    unmount_dir(dev);
    unmount_dir(mount);
    /* use vfat instead of msdos if available. Reason: fat32 compatibility */
    sprintf(prtbuf, "mount -t %s %s %s", !is_filesystem_supported("vfat") ? "vfat" : "msdos", dev, mount);
    status = execlog(prtbuf, LOG_INFO);
    if (status) {
      int answer;
      /* Offer to initialize it if the mount fails */
      sprintf(prtbuf, _("Failed to mount %s as a FAT partition.  "
			"This could simply be because it is not yet "
			"formatted as one.  Would you like to format "
			"it now?"), dev);
      answer = twoButtonBox(prtbuf, _("Initialize FAT partition?"),
			    _("Format"), _("Cancel"), 1);
      if (answer == DLG_NO)
	return -1;
      sprintf(prtbuf, "mkdosfs %s", dev);
      if (execlog(prtbuf, LOG_INFO)) {
	sprintf(prtbuf, "Failed to make a FAT filesystem on %s", dev);
	problemBox(prtbuf, _("FAT filesystem init failed"));
	return -1;
      }
    }
  } while (status);

  return 0;
}

/* mount_and_check_floppy() imposes too much policy on us */
static int just_mount_the_damn_floppy(char *message)
{
  const char **fs_type;

  if (!system("cat /proc/mounts | grep -q /floppy"))
    execlog("umount /floppy", LOG_DEBUG);
  if (twoButtonBox(message, _("Please insert the disk"),
		   _("Continue"), _("Cancel"), 1) == DLG_NO)
    return -2;
  for (fs_type = fs_type_tab; *fs_type; fs_type++) {
    sprintf(prtbuf, "mount -r -t %s %s /floppy",
	    *fs_type, Archive_Dir);
    if (execlog(prtbuf, LOG_INFO) == 0) /* success */
      return 0;
  }
  return -1;
}

static int install_milo(const char* boot, struct fdisk_partition* p)
{
  FILE *filep;
  const char* bootdev;
  char *milo_path = NULL, *linload_path = NULL;
  struct stat statbuf;
  int status;

  /* skip past the "/dev/" */
  bootdev = strrchr(boot, '/');
  if (bootdev == NULL) {
    /* panic! */
    return 1;
  }
  bootdev++;

  /* write milo.conf */
  if ((filep = fopen(BC_MILO_CONF, "w")) == NULL) {
    problemBox(_("Could not write to /etc/milo.conf"), _("Problem"));
    return 1;
  }
  fprintf(filep,
	  "# MILO configuration, created by the Debian installer\n"
	  "timeout=100\n"
	  "image=%s:vmlinuz\n"
	  "root=%s\n",
	  bootdev, Root->name);
  if (serialConsole >= 0) {
    fprintf(filep, "append=\"console=ttyS%d %s\"\n",
	    serialConsole, append_opts);
  } else if (strlen (append_opts) > 0) {
    fprintf(filep, "append=\"%s\"\n", append_opts);
  }
  fclose(filep);

  /* This is totally fucking broken global variable abuse */
  disqtype = milo;
  if (choose_medium())
    return 1;

  if (strcmp(Archive_Dir, "netfetch") == 0) {
    /* punt for now, because netfetch can't do what we want */
    problemBox(_("Installing MILO from the network is not yet supported"),
	       _("Problem"));
    return 1;
  } else if (strncmp(Archive_Dir, "/dev/fd", 7) == 0) {
    /* again, mount_and_check_floppy() can't do what we want */
    const char **path;
    const char *paths[] = { "/floppy/%s",
			    "/floppy/milo_%s",
			    "/floppy/milo", NULL };

    for (;;) {
      sprintf(prtbuf,
	      _("Place the MILO disk you used to boot, or "
		"a floppy disk containing `linload.exe' and a MILO binary "
		/* XXX bad internationalization (word order) */
		"named `%s', in the %s floppy drive."),
	      Arch2, ((Archive_Dir[7] == '0')
		      ? _("first") : _("second")));
      if (just_mount_the_damn_floppy(prtbuf)) {
	problemBox(_("Unable to mount the floppy.  You may have "
		     "inserted the wrong floppy.  Please try again.")
		   , _("Mount failed"));
	continue;
      }
      /* Try to find the MILO binary */
      for (path = paths; *path; path++) {
	sprintf(prtbuf, *path, Arch2);
	DEBUGMSG("looking for %s", prtbuf);
	if (NAME_ISREG(prtbuf, &statbuf)) {
	  milo_path = strdup(prtbuf);
	  break;
	}
      }
      if (!*path) {
	sprintf(prtbuf, 
		_("Unable to find a MILO binary on the floppy "
		  "(tried `%s', `milo_%s', and `milo').  Please "
		  "try again."), Arch2, Arch2);
	problemBox(prtbuf, _("MILO not found"));
	continue;
      }
      if (!NAME_ISREG("/floppy/linload.exe", &statbuf)) {
	problemBox(_("Unable to find `linload.exe' on the floppy.  "
		     "Please try again."), _("linload.exe not found"));
	continue;
      }
      DEBUGMSG("found milo and linload.exe");
      break;
    }
    linload_path = strdup("/floppy/linload.exe");
  } else {
    /* This is all mostly useless, since choose_medium() takes the
       liberty (GAR GAR GAR) of doing most of it for us in a horrible
       and broken way.  But we still need to do it because the MILO
       could be in ${Arch2} or in MILO/${Arch2}, and we don't know
       which. */
    const char **path;
    const char *paths[] = { "%s/MILO/%s", /* CD or NFS */
			    "%s/%s",
			    "%s/milo_%s",
			    "%s/milo", NULL };
    const char *linload_paths[] = { "%s/MILO/linload.exe",
				    "%s/linload.exe", NULL };

    /* Try to find the MILO binary */
    for (path = paths; *path; path++) {
      sprintf(prtbuf, *path, Archive_Dir, Arch2);
      DEBUGMSG("looking for %s", prtbuf);
      if (NAME_ISREG(prtbuf, &statbuf)) {
	milo_path = strdup(prtbuf);
	break;
      }
    }
    if (!*path) {
      sprintf(prtbuf, 
	      _("Unable to find a MILO binary in %s "
		"(tried `MILO/%s', `%s', `milo_%s', and `milo')."),
	      Archive_Dir, Arch2, Arch2);
	problemBox(prtbuf, _("MILO not found"));
	goto error_out;
    }
    /* and linload.exe */
    for (path = linload_paths; *path; path++) {
      sprintf(prtbuf, *path, Archive_Dir);
      DEBUGMSG("looking for %s", prtbuf);
      if (NAME_ISREG(prtbuf, &statbuf)) {
	linload_path = strdup(prtbuf);
	break;
      }
    }
    if (!*path) {
      sprintf(prtbuf, 
	      _("Unable to find `linload.exe' in %s "
		"(tried `MILO/linload.exe' and `linload.exe')."),
	      Archive_Dir);
	problemBox(prtbuf, _("linload.exe not found"));
	goto error_out;
    }
  }
  /* Mount the filesystem */
  if (mount_dos_fs(p->name, "/instmnt"))
    goto error_out;

  /* Now copy MILO and linload.exe */
  pleaseWaitBox(_("Copying MILO and linload.exe..."));
  sprintf(prtbuf, "cp %s /instmnt/milo", milo_path);
  status = execlog(prtbuf, LOG_INFO);
  if (status) {
    problemBox(_("Couldn't copy MILO to the FAT partition"),
	       _("MILO install failed"));
    goto error_out;
  }
  sprintf(prtbuf, "cp %s /instmnt", linload_path);
  status = execlog(prtbuf, LOG_INFO);
  if (status) {
    problemBox(_("Couldn't copy linload.exe to the FAT partition"),
	       _("MILO install failed"));
    goto error_out;
  }

#if MILOCTL_ON_ROOTDISK /* The MILO utilities are not packaged yet,
			   and I don't want to kludge it in */
  /* Now run miloctl on the MILO binary */
  sprintf(prtbuf, BC_RUN_MILOCTL, bootdev);
  status = execlog(prtbuf, LOG_INFO);
  if (status) {
    problemBox(_("Couldn't run miloctl on the installed MILO.  "
		 "MILO will not boot automatically when started."),
	       _("miloctl failed"));
    goto error_out;
  }
#endif
  
  return 0;

 error_out:
  if (milo_path) free(milo_path);
  return 1;
}

static int install_apb(const char *boot, struct fdisk_partition *p)
{
  struct stat statbuf;
  FILE *f;

  /* This is kind of suboptimal, as we really want to mount the DOS
     partition as /boot or something like that.  Anyhow we will try to
     mount the filesystem first, then write apb.cfg and up1000.pal,
     then copy the kernel to it.  */
  if (mkdir("/target/dos", 0755)) {
    if (errno != EEXIST)
      return 1;
    if (!NAME_ISDIR("/target/dos", &statbuf))
      return 1;
  }
  if (mount_dos_fs(p->name, "/target/dos"))
    return 1;
  if ((f = fopen("/target/etc/fstab", "a")) == NULL)
    return 1;
  fprintf(f, "%s /dos vfat defaults 0 0\n", p->name);
  fclose(f);

  if ((f = fopen(BC_APB_CFG, "w")) == NULL) {
    problemBox(_("Could not write to /dos/apb.cfg"), _("Problem"));
    return 1;
  }
  if (serialConsole >= 0)
    fprintf(f, "debian|bootl up1000.pal vmlinuz root=%s console=ttyS%d\n",
	    Root->name, serialConsole);
  else
    fprintf(f, "debian|bootl up1000.pal vmlinuz root=%s\n", Root->name);
  fclose(f);

  disqtype = apb;
  if (choose_medium())
    return 1;

  if (strcmp(Archive_Dir, "netfetch") == 0) {
    /* punt for now, because netfetch can't do what we want */
    problemBox(_("Installing APB from the network is not yet supported"),
	       _("Problem"));
    return 1;
  } else if (strncmp(Archive_Dir, "/dev/fd", 7) == 0) {
    /* this is easy, thankfully */
    mount_and_check_floppy(Archive_Dir, "rescue", _("Rescue Floppy"));
    strcpy(Archive_Dir, "/floppy");
  } else {
    /* ?? */
  }

  pleaseWaitBox(_("Copying APB and up1000.pal..."));
  sprintf(prtbuf, "cp %s/apb.exe /target/dos", Archive_Dir);
  if (execlog(prtbuf, LOG_INFO)) {
    problemBox(_("Couldn't copy APB to the FAT partition"),
	       _("APB install failed"));
    return 1;
  }
  sprintf(prtbuf, "cp %s/up1000.pal /target/dos", Archive_Dir);
  if (execlog(prtbuf, LOG_INFO)) {
    problemBox(_("Couldn't copy up1000.pal to the FAT partition"),
	       _("APB install failed"));
    return 1;
  }
  if (execlog("cp /target/boot/vmlinuz /target/dos", LOG_INFO)) {
    problemBox(_("Couldn't copy the installed kernel to the FAT partition"),
	       _("APB install failed"));
    return 1;
  }

  return 0;
}
#endif /* USE_ABOOT */

#ifdef USE_ELILO
static int install_elilo(void)
{
  int status;
  struct fdisk_partition *p;
  struct d_choices *choices;
  struct fdisk_partition **plist;
  char *dev;
  int r, i = 0;

  /* Try to find a suitable boot partition */
  choices = malloc(4 * sizeof(struct d_choices));
  plist   = malloc(4 * sizeof(struct fdisk_partition *));

  for (p = fdisk_partitions_by_type[FSTYPE_MSDOS];
       p; p = p->next_by_type) {
    /* add it to the list */
    plist[i]          = p;
    choices[i].string = p->name;
    choices[i].tag    = NULL;
    choices[i].state  = 0;
    i++;
    if ((i % 4) == 0) {
      choices = realloc(choices,
      		  (i+4) * sizeof(struct d_choices));
      plist   = realloc(plist,
      		  (i+4) * sizeof(struct fdisk_partition *));
    }
  }
  if (i == 0) {
    problemBox(_("The ELILO bootloader must reside on a "
      	   "partition containing a FAT (MSDOS) file system "
	   "in order to be read by the BIOS.  "
      	   "You appear not to have any such partitions.  "
      	   "This means you will not be able to boot directly "
      	   "from disk.\n"), _("No FAT partitions detected"));
    goto error_out;
  }
  r = menuBox(_("ELILO, the boot loader, must reside on a partition "
	"containing a FAT filesystem.  "
      	"All such partitions on your system are listed below.  "
      	"Please select the one you wish to install the boot "
      	"loader on to."),
      	_("Select Partition"), choices, i, 1);
  if (r < 0)
    goto error_out;
  dev = plist[r]->name;

  pleaseWaitBox(_("Running ELILO to make the kernel able to boot "
		"from the hard disk..."));

  /* Now run /sbin/elilo */
  sprintf(prtbuf, BC_RUN_ELILO, dev, Root->name);
  status = execlog(prtbuf, LOG_INFO);
  boxPopWindow();
  if (status) {
    sprintf(prtbuf, _("Failed to install the boot loader on %s, status %d.  This might simply be because the partition has not been formatted, or because there is insufficient space.  Would you like me to create a new filesystem on %s and try again? Warning, all data existing on %s will be lost."), dev, status, dev, dev);
    if (yesNoBox(prtbuf, _("ELILO install failed")) == DLG_NO)
      goto error_out;
    pleaseWaitBox(_("Running ELILO to make the kernel able to boot "
		"from the hard disk..."));
    /* XXX this should be --format really, but that needs elilo 3.2 */
    /* XXX elilo 3.2 accepts --format and --force for now */
    sprintf(prtbuf, BC_RUN_ELILO " --force", dev, Root->name);
    status = execlog(prtbuf, LOG_INFO);
    boxPopWindow();
    if (status) {
      problemBox(_("Install of ELILO failed, you will not be able "
			"to boot your system from the harddisk"),
			_("ELILO install failed"));
      goto error_out;
    }
  }
  free(plist);
  free(choices);
  return 0;

 error_out:
  free(plist);
  free(choices);
  return 1;
}
#endif /* USE_ELILO */

#ifdef USE_DELO
static int run_delo(void)
{

  /* DELO support added by Karsten Merker <merker@linuxtag.org> */
  /* 2002/01/18. This was mostly done by heavily peeking at the */
  /* bootloader code for the other archs and piecing together   */
  /* the needed elements.                                       */
  /* DELO has to be installed into the bootsector of the drive  */
  /* containing the root partition                              */

  FILE* filep;
  char *boot_disk;
  int idx;

  boot_disk = strdup(Boot->name);
  idx=firstnumber(boot_disk);
  boot_disk[idx] = '\0'; /* Cut off the partition number */

  INFOMSG(_("writing delo.conf with boot of '%s' and root of '%s'"),
          boot_disk, Root->name);

  pleaseWaitBox(_("Installing /etc/delo.conf and running DELO to make the "
                "system able to boot from the hard disk..."));

  if((filep=fopen(BC_DELO_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }
  fprintf(filep,"label=linux\n");
  fprintf(filep,"	image=/vmlinux\n");
  fprintf(filep,"	append=\"root=%s",Root->name);
  if (serialConsole >= 0) {
    fprintf(filep, " console=ttyS%d", serialConsole);
  }  
  fprintf(filep,"\"\n\n");
  fclose(filep);

  snprintf(prtbuf, sizeof(prtbuf), "chroot /target /sbin/delo %s", boot_disk);
  if( execlog(prtbuf, LOG_DEBUG) ) {
    boxPopWindow();
    problemBox(_("DELO failed."),_("Critical Error"));
    return 1;
  }
  boxPopWindow();
  return 0;
}
#endif

#ifdef USE_DVHTOOL
static int run_dvhtool(void)
{
  int count;
  char buf[256];
  char* disk_name;
  struct fdisk_partition* volhdr;

  if (strcmp(Arch2, "r4k-ip22") != 0) {
	problemBox(_("Making the system bootable from hard disk is currently only supported for the mips ip22 subarchitecture - sorry."),
        	_("Not an ip22 machine"));
	return 0;
  } 
  volhdr = fdisk_find_partition_by_type(PTYPE_SGI_VOLHDR);
  if( !volhdr ) {
    	problemBox(_("I couldn't find any partitions with SGI disklabel. I can't make your system bootable from hard disk."),
               _("No SGI disklabel found"));
    	return 1;
  } 
  disk_name = volhdr->disk->name;
  if ((count=readlink("/target/vmlinux", buf, sizeof(buf))) == -1) {
	ERRMSG(_("Error resolving /target/vmlinux symlink"));
	return 1;
  }
  buf[count] = '\0';
  snprintf(prtbuf, sizeof(prtbuf), 
	"dvhtool -d %s --unix-to-vh /target/%s linux.bin", disk_name, buf );
  if( execlog(prtbuf, LOG_DEBUG) ) {
	problemBox(_("Dvhtool failed - cannot continue."),_("Critical Error"));
	 return 1;
  }
  vaproblemBox(_("Important"),
	_("The kernel has been copied into the volume header of disk %s. "
	  "Please adjust your PROM settings after rebooting as follows.")
	  , disk_name);
  vaproblemBox(_("Prom Settings"),
	_("setenv OSLoader linux\n"
	  "setenv SystemPartition scsi(0)disk(X)rdisk(0)partition(8)\n"
	  "where X is the scsi id of disk %s"), disk_name);
  vaproblemBox(_("Prom Settings"),
	_("Furthermore you can set your root partition with:\n"
	  "setenv OSLoadPartition %s"), Root->name);
  return 0;
}
#endif

#ifdef USE_ZIPL
static int install_zipl(const char * boot)
{
  FILE* filep;
  char line[255];
  FILE * f; 

  /* TRANS: IPL (Initial Program Load) means "boot" in mainframe context and 
     DASD can be viewed as a hard disk. Please don't translate these terms */
  pleaseWaitBox(_("Running ZIPL to make the kernel able to IPL from the DASD..."));

  if(strlen(boot)==0) return 1;

  if((filep=fopen(BC_ZIPL_CONF,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }

  fprintf(filep,"[defaultboot]\n");
  fprintf(filep,"default=debian\n\n");
  fprintf(filep,"[debian]\n");
  fprintf(filep,"target=/boot\n");
  fprintf(filep,"image=/vmlinuz\n");
  fprintf(filep,"parmfile=/boot/parmfile\n");
  fclose(filep);

  if((filep=fopen(BC_PARMFILE,"w"))==NULL) {
    boxPopWindow();
    return 1;
  }

  fprintf(filep, "root=%s ro noinitrd dasd=", boot);

  f = fopen("/proc/subchannels", "r");
  if (f) {
    char device[17];
    char sch[17];
    char devtype[17];
    char devmodel[17];
    char inuse[4];
    int i=0;
    fgets(line, sizeof line, f);
    fgets(line, sizeof line, f);
    while (fgets(line, sizeof line, f)) {
      if (sscanf(line, "%s %s %s %s %s ", device, sch, devtype, devmodel, inuse) == 5) {
        if((!memcmp(devtype, "3390", 4) || !memcmp(devtype, "3380", 4) ||
           !memcmp(devtype, "9345", 4) || !memcmp(devtype, "9336", 4) ||
           !memcmp(devtype, "3370", 4)) && !strcmp(inuse, "yes")) {
            if(i > 0) fprintf(filep, ",");
            fprintf(filep, "%s", device);
            i++;
        }
      }
    }
    fclose(f); 
  }
 
  fprintf(filep, " vmpoff=\"LOGOFF\"\n");

  fclose(filep);
  sleep(1);             /* wait some amount of time to see the message box */
  boxPopWindow();
  if(execlog(BC_RUN_ZIPL, LOG_INFO)) return 1;
  return 0;
}
#endif /* USE_ZIPL */

#ifdef USE_LILO
/*
 * Note:  To modify this template, please use `bootconfig-convert-
 *        lilotemplate', and follow the instructions in the docstring.
 */

/*
 * # /etc/lilo.conf - See: `lilo(8)' and `lilo.conf(5)',
 * # ---------------       `install-mbr(8)', `/usr/share/doc/lilo/',
 * #                       and `/usr/share/doc/mbr/'.
 *
 * # +---------------------------------------------------------------+
 * # |                        !! Reminder !!                         |
 * # |                                                               |
 * # | Don't forget to run `lilo' after you make changes to this     |
 * # | conffile, `/boot/bootmess.txt', or install a new kernel.  The |
 * # | computer will most likely fail to boot if a kernel-image      |
 * # | post-install script or you don't remember to run `lilo'.      |
 * # |                                                               |
 * # +---------------------------------------------------------------+
 *
 * # Support LBA for large hard disks.
 * #
 * lba32
 *
 * # Overrides the default mapping between harddisk names and the BIOS'
 * # harddisk order. Use with caution.
 * %s
 * %s
 *
 * # Specifies the boot device.  This is where Lilo installs its boot
 * # block.  It can be either a partition, or the raw device, in which
 * # case it installs in the MBR, and will overwrite the current MBR.
 * #
 * boot=%s
 *
 * # Specifies the device that should be mounted as root. (`/')
 * #
 * root=%s
 *
 * # Enable map compaction:
 * # Tries to merge read requests for adjacent sectors into a single
 * # read request. This drastically reduces load time and keeps the
 * # map smaller.  Using `compact' is especially recommended when
 * # booting from a floppy disk.  It is disabled here by default
 * # because it doesn't always work.
 * #
 * # compact
 *
 * # Installs the specified file as the new boot sector
 * # You have the choice between: bmp, compat, menu and text
 * # Look in /boot/ and in lilo.conf(5) manpage for details
 * #
 * install=/boot/boot-menu.b
 *
 * # Specifies the location of the map file
 * #
 * map=/boot/map
 *
 * # You can set a password here, and uncomment the `restricted' lines
 * # in the image definitions below to make it so that a password must
 * # be typed to boot anything but a default configuration.  If a
 * # command line is given, other than one specified by an `append'
 * # statement in `lilo.conf', the password will be required, but a
 * # standard default boot will not require one.
 * #
 * # This will, for instance, prevent anyone with access to the
 * # console from booting with something like `Linux init=/bin/sh',
 * # and thus becoming `root' without proper authorization.
 * #
 * # Note that if you really need this type of security, you will
 * # likely also want to use `install-mbr' to reconfigure the MBR
 * # program, as well as set up your BIOS to disallow booting from
 * # removable disk or CD-ROM, then put a password on getting into the
 * # BIOS configuration as well.  Please RTFM `install-mbr(8)'.
 * #
 * # password=tatercounter2000
 *
 * # Specifies the number of deciseconds (0.1 seconds) LILO should
 * # wait before booting the first image.
 * #
 * delay=20
 *
 * # You can put a customized boot message up if you like.  If you use
 * # `prompt', and this computer may need to reboot unattended, you
 * # must specify a `timeout', or it will sit there forever waiting
 * # for a keypress.  `single-key' goes with the `alias' lines in the
 * # `image' configurations below.  eg: You can press `1' to boot
 * # `Linux', `2' to boot `LinuxOLD', if you uncomment the `alias'.
 * #
 * # message=/boot/bootmess.txt
 * #	prompt
 * %s#	single-key
 * #	delay=100
 * #	timeout=100
 *
 * # Specifies the VGA text mode at boot time. (normal, extended, ask, <mode>)
 * #
 * # vga=ask
 * # vga=9
 * #
 * vga=normal
 *
 * # Kernel command line options that apply to all installed images go
 * # here.  See: The `boot-prompt-HOWO' and `kernel-parameters.txt' in
 * # the Linux kernel `Documentation' directory.
 * #
 * # append=""
 * %s
 * # Boot up Linux by default.
 * #
 * default=Linux
 *
 * image=/vmlinuz
 * 	label=Linux
 * 	read-only
 * #	restricted
 * #	alias=1
 *
 * image=/vmlinuz.old
 * 	label=LinuxOLD
 * 	read-only
 * 	optional
 * #	restricted
 * #	alias=2
 *
 * # If you have another OS on this machine to boot, you can uncomment the
 * # following lines, changing the device name on the `other' line to
 * # where your other OS' partition is.
 * #
 * # other=/dev/hda4
 * #	label=HURD
 * #	restricted
 * #	alias=3
 */

#define BC_LILOTEMPLATE1 \
"# /etc/lilo.conf - See: `lilo(8)' and `lilo.conf(5)',\n" \
"# ---------------       `install-mbr(8)', `/usr/share/doc/lilo/',\n" \
"#                       and `/usr/share/doc/mbr/'.\n" \
"\n" \
"# +---------------------------------------------------------------+\n" \
"# |                        !! Reminder !!                         |\n" \
"# |                                                               |\n" \
"# | Don't forget to run `lilo' after you make changes to this     |\n" \
"# | conffile, `/boot/bootmess.txt', or install a new kernel.  The |\n" \
"# | computer will most likely fail to boot if a kernel-image      |\n" \
"# | post-install script or you don't remember to run `lilo'.      |\n" \
"# |                                                               |\n" \
"# +---------------------------------------------------------------+\n" \
"\n" \
"# Support LBA for large hard disks.\n" \
"#\n" \
"lba32\n" \
"\n" \
"# Overrides the default mapping between harddisk names and the BIOS'\n" \
"# harddisk order. Use with caution.\n" \
"%s\n%s\n" \
"# Specifies the boot device.  This is where Lilo installs its boot\n" \
"# block.  It can be either a partition, or the raw device, in which\n" \
"# case it installs in the MBR, and will overwrite the current MBR.\n" \
"#\n" \
"boot=%s\n" \
"\n" \
"# Specifies the device that should be mounted as root. (`/')\n" \
"#\n" \
"root=%s\n" \
"\n" \
"# Enable map compaction:\n" \
"# Tries to merge read requests for adjacent sectors into a single\n" \
"# read request. This drastically reduces load time and keeps the\n" \
"# map smaller.  Using `compact' is especially recommended when\n" \
"# booting from a floppy disk.  It is disabled here by default\n" \
"# because it doesn't always work.\n" \
"#\n" \
"# compact\n" \
"\n" \
"# Installs the specified file as the new boot sector\n" \
"# You have the choice between: bmp, compat, menu and text\n" \
"# Look in /boot/ and in lilo.conf(5) manpage for details\n" \
"#\n" \
"install=/boot/boot-menu.b\n" \
"\n" \
"# Specifies the location of the map file\n" \
"#\n" \
"map=/boot/map\n" \
"\n" \
"# You can set a password here, and uncomment the `restricted' lines\n" \
"# in the image definitions below to make it so that a password must\n" \
"# be typed to boot anything but a default configuration.  If a\n" \
"# command line is given, other than one specified by an `append'\n" \
"# statement in `lilo.conf', the password will be required, but a\n" \
"# standard default boot will not require one.\n" \
"#\n" \
"# This will, for instance, prevent anyone with access to the\n" \
"# console from booting with something like `Linux init=/bin/sh',\n" \
"# and thus becoming `root' without proper authorization.\n" \
"#\n" \
"# Note that if you really need this type of security, you will\n" \
"# likely also want to use `install-mbr' to reconfigure the MBR\n" \
"# program, as well as set up your BIOS to disallow booting from\n" \
"# removable disk or CD-ROM, then put a password on getting into the\n" \
"# BIOS configuration as well.  Please RTFM `install-mbr(8)'.\n" \
"#\n" \
"# password=tatercounter2000\n" \
"\n" \
"# Specifies the number of deciseconds (0.1 seconds) LILO should\n" \
"# wait before booting the first image.\n" \
"#\n" \
"delay=20\n" \
"\n" \
"# You can put a customized boot message up if you like.  If you use\n" \
"# `prompt', and this computer may need to reboot unattended, you\n" \
"# must specify a `timeout', or it will sit there forever waiting\n" \
"# for a keypress.  `single-key' goes with the `alias' lines in the\n" \
"# `image' configurations below.  eg: You can press `1' to boot\n" \
"# `Linux', `2' to boot `LinuxOLD', if you uncomment the `alias'.\n" \
"#\n" \
"# message=/boot/bootmess.txt\n" \
"%s" \
"#	prompt\n" \
"#	single-key\n" \
"#	delay=100\n" \
"#	timeout=100\n" \
"\n" \
"# Specifies the VGA text mode at boot time. (normal, extended, ask, <mode>)\n" \
"#\n" \
"# vga=ask\n" \
"# vga=9\n" \
"#\n" \
"vga=normal\n" \
"\n" \
"# Kernel command line options that apply to all installed images go\n" \
"# here.  See: The `boot-prompt-HOWO' and `kernel-parameters.txt' in\n" \
"# the Linux kernel `Documentation' directory.\n" \
"#\n" \
"# append=\"\"\n" \
"%s\n" \
"# Boot up Linux by default.\n" \
"#\n" \
"default=Linux\n" \
"\n" \
"image=/vmlinuz\n" \
"	label=Linux\n" 

#define BC_LILOTEMPLATE2 \
"   initrd=/boot/initrd.debinstall\n"

#define BC_LILOTEMPLATE3 \
"	read-only\n" \
"#	restricted\n" \
"#	alias=1\n" \
"\n" \
"image=/vmlinuz.old\n" \
"	label=LinuxOLD\n" \
"	read-only\n" \
"	optional\n" \
"#	restricted\n" \
"#	alias=2\n" \
"\n" \
"# If you have another OS on this machine to boot, you can uncomment the\n" \
"# following lines, changing the device name on the `other' line to\n" \
"# where your other OS' partition is.\n" \
"#\n" \
"# other=/dev/hda4\n" \
"#	label=HURD\n" \
"#	restricted\n" \
"#	alias=3\n"

static char * get_ramdisk_size(char *files[25])
{
	DIR * dir;
	struct dirent * file;
	struct stat statbuf;
	unsigned int size = 0;
	int i = 0;
	char * ascii_size;
	char * tmp = NULL;
	
	dir = opendir(MOD_SAVE_DIR);
	/* hopefully only one or two modules in this dir */
	file = readdir(dir);
	while ( file != NULL )
	{
		if ( *(file->d_name) == '.' )
		{
			file = readdir(dir);
			continue;
		}
		tmp = realloc(tmp,sizeof(MOD_SAVE_DIR) + sizeof("/") + sizeof(file->d_name));
		strcpy(tmp,MOD_SAVE_DIR "/");
		strcat(tmp,file->d_name);
		stat(tmp,&statbuf);
		size += statbuf.st_size;
		file = readdir(dir);
	}
	free(tmp);
	closedir(dir);
	size += 4096; /* big buffer for the linuxrc shell script */
	while ( strcmp(files[i],"") != 0 )
	{
		stat(files[i],&statbuf);
		size += statbuf.st_size;
		++i;
	}
	/* for directories and device*/
	size += (4096 * 8);
	size /= 1024;
	size += 30; /* trust no one */
	ascii_size = malloc(5 * sizeof(char));
	snprintf(ascii_size,5,"%u",size);
	return(ascii_size);
}

static char * get_device_number(char * device)
{
	unsigned char major, minor;
	char *number;
#if 1
	unsigned int temp;
	char *command;
	FILE *pipe;

	/* FIXME: Can we do something better ? */
	/* we get the major number of the device */
	command = malloc(sizeof("expr substr \"`ls -l ") + strlen(device) + sizeof("`\" 35 2"));
	if ( command == NULL )
	{
		problemBox(_("Can't allocate memory"),_("Critical Error"));
		exit(1);
	}
	strcpy(command,"expr substr \"`ls -l ");
	strcat(command,device);
	strcat(command,"`\" 35 2");
	pipe = popen(command,"r");
	if ( pipe == NULL )
	{
		problemBox(_("Can't open pipe"),_("Critical Error"));
		exit(1);
	}
	fscanf(pipe,"%u",&temp);
	major = (unsigned char) temp;
	pclose(pipe);
	/* we change the magic 35 into magic 40 
	   to get the minor number */
	command[strlen(command) - 3 ] = '0';
	command[strlen(command) - 4 ] = '4';
	pipe = popen(command,"r");
	if ( pipe == NULL )
	{
		problemBox(_("Can't open pipe"),_("Critical Error"));
		exit(1);
	}
	fscanf(pipe,"%u",&temp);
	minor = (unsigned char) temp;
	pclose(pipe);
	number = malloc(sizeof("0x") + 4);
	if ( number == NULL )
	{
		problemBox(_("Can't allocate memory"),_("Critical Error"));
		exit(1);
	}
    free (command); /* Free what we do not need any more */
#else
    struct stat buf;
	
    if (stat (device, &buf) == 0)
        if (S_ISCHR (buf.st_mode) || S_ISBLK (buf.st_mode))
        {
            major = MAJOR (buf.st_rdev);
            minor = MINOR (buf.st_rdev);
        }
        else
        {
            /* FIXME: should the file name be printed here? */
            problemBox (_("Specified file is not device", _("Critical Error")));

            exit (1);
        }
    else
    {
        problemBox (_("Cannot stat device", _("Critical Error")));

        exit (1);
    }
#endif
	sprintf(number,"0x%02u%02u",(unsigned int)major,(unsigned int)minor);

	return number;
}

#define FLOP_TMP_DIR "/tmp/mnt-flop-ramdisk"
#define FLOP_RAMDISK "/dev/ram8"

int create_ramdisk(char * device_number,char * size)
{
	 DIR *dir;
	 struct dirent * dirent;
	 char * tmp;
	 FILE * fd;
	 FILE * fd2;
	 char smp = 0;
	 char * command;
	 char buffer[512];
	 if ( yesNoBox(_("Do you plan to use SMP on this machine?"),_("Question")) == DLG_YES )
	 {
		 smp = 1;
	 }
	 
	 if ( mkdir(FLOP_TMP_DIR,0777) != 0 )
	 {
		 perrorBox(_("Unable to create a temporary ramdisk directory"));
		 return 1;
	 }
	 /* we "zero" the device to enhance the compression */
	 command = malloc(sizeof("dd < /dev/zero > " FLOP_RAMDISK " bs=1k count=") + strlen(size));
	 if ( command == NULL )
	 {
		 problemBox(_("Can't allocate memory"),_("Critical Error"));
	 	 return 1;
	 }
	 strcpy(command,"dd < /dev/zero > " FLOP_RAMDISK " bs=1k count=");
	 strcat(command,size);
	 if (execlog(command,LOG_INFO) )
	 {
		 problemBox(_("Emptying ramdisk file resulted in an error, can't continue"),_("Critical Error"));
		 return 1;
	 }
	 /* a minix fs would be better. But we don't have the tools available */
	 command = realloc(command,sizeof("/sbin/mkfs.ext2 -F -i 4096 -m 0 " FLOP_RAMDISK " ") + strlen(size));
	 strcpy(command,"/sbin/mkfs.ext2 -F -i 4096 -m 0 " FLOP_RAMDISK " ");
	 strcat(command,size);
	 if ( execlog(command,LOG_INFO) )
	 {
		 problemBox(_("Creating an ext2 partition in ramdisk resulted in an error, can't continue"),_("Critical Error"));
		 return 1;
	 }
	 /* FIXME:
	  	It would be better to do this with a system call
	 if ( mount(FLOP_RAMDISK,FLOP_TMP_DIR,"ext2",MS_MGC_VAL,"loop") ) does not work */
	 if ( execlog("mount -t ext2 -o loop " FLOP_RAMDISK " " FLOP_TMP_DIR,LOG_DEBUG ) )
	 {
		 perrorBox(_("Mounting ramdisk partition failed:"));
		 return 1;
	 }
	 /* we create our tree */
	 rmdir(FLOP_TMP_DIR "/lost+found");
	 mkdir(FLOP_TMP_DIR "/dev",0755);
	 mkdir(FLOP_TMP_DIR "/proc",0755);
	 mkdir(FLOP_TMP_DIR "/lib",0755);
	 mkdir(FLOP_TMP_DIR "/bin",0755);
	 mkdir(FLOP_TMP_DIR "/sbin",0755);
	 mkdir(FLOP_TMP_DIR "/lib/modules",0755);
	 execlog("mknod " FLOP_TMP_DIR "/dev/tty1 c 4 1",LOG_DEBUG);
	 /* chmod 555 /FLOP_TMP_DIR/proc */
	 chmod(FLOP_TMP_DIR "/proc",S_IRUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);
	 /* linuxrc is a file that is directly launched by initrd at boot
	  * so we gonna make a script that simply loads the modules */
	 fd = fopen(FLOP_TMP_DIR "/linuxrc","w");
	 if ( fd == NULL )
	 {
		perrorBox(_("Can't open file for writing:"));
		return 1;
	 }
	 fprintf(fd,"#!/bin/sh\n");

	 
	 dir = opendir(MOD_SAVE_DIR);
	 dirent = readdir(dir);
	 while ( dirent != NULL )
	 {
	 	tmp = strstr(dirent->d_name,".o");
		if ( (tmp) && tmp[2] == '\0' )
		{
			command = realloc(command,sizeof("cp ") + sizeof(MOD_SAVE_DIR) + sizeof("/")+ strlen(dirent->d_name) 
					+ sizeof(" " FLOP_TMP_DIR "/lib/modules"));
			strcpy(command,"cp ");
			strcat(command,MOD_SAVE_DIR);
			strcat(command,"/");
			strcat(command,dirent->d_name);
			strcat(command," " FLOP_TMP_DIR "/lib/modules");
			if ( execlog(command,LOG_DEBUG ) )
			{
				problemBox(_("Unable to copy saved modules to ramdisk, can't continue"),_("Critical Error"));
				return 1;
			}
			/*
			if ( smp )
			{
				fprintf(fd,"/sbin/insmod -f -P smp /lib/modules/%s\n",dirent->d_name);
			} else {
				fprintf(fd,"/sbin/insmod -f /lib/modules/%s\n",dirent->d_name);
			}
			*/
		}
		dirent = readdir(dir);
	 }
	 closedir(dir);
	 
	 fd2 = fopen(MOD_SAVE_FILE,"r");
	 if ( fd2 == NULL )
	 {
		 problemBox(_("Can't open the 'mod-save' file. Please report this bug"),_("Critical Error"));
		 return 1;
	 }
	 while (fgets(buffer,sizeof(buffer),fd2))
	 {
		if ( smp )
		{
				fprintf(fd,"/sbin/insmod -f -P smp /lib/modules/%s\n",buffer);
		} else {
				fprintf(fd,"/sbin/insmod -f /lib/modules/%s\n",buffer);
		}
	 } 
	 fclose(fd2);
	 fprintf(fd,"/bin/mount -t proc proc /proc\n");
	 fprintf(fd,"echo %s > /proc/sys/kernel/real-root-dev\n",device_number);
	 fprintf(fd,"/bin/umount /proc\n");
	 fclose(fd);
	 /* chmod 755 */
	 chmod(FLOP_TMP_DIR "/linuxrc",S_IRUSR | S_IXUSR | S_IWUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH);

	 if (execlog("cp /sbin/insmod " FLOP_TMP_DIR "/sbin",LOG_DEBUG) )
	 {
		problemBox(_("Can't copy needed files"),_("Critical Error"));
		return 1;
	 }
	 if (execlog("cp /bin/mount /bin/umount /bin/sh " FLOP_TMP_DIR "/bin",LOG_DEBUG) )
	 {
		problemBox(_("Can't copy needed files"),_("Critical Error"));
		return 1;
	 }
	 if (execlog("cp /lib/ld-linux.so.2 /lib/libc.so.6 " FLOP_TMP_DIR "/lib",LOG_DEBUG) )
	 {
		problemBox(_("Can't copy needed files"),_("Critical Error"));
		return 1;
	 }
	 execlog("umount " FLOP_TMP_DIR,LOG_DEBUG);
	 rmdir(FLOP_TMP_DIR);
	 command = realloc(command,sizeof("dd < " FLOP_RAMDISK " > /target/boot/initrd.debinstall bs=1k count=") +
			 strlen(size));
	 strcpy(command,"dd < " FLOP_RAMDISK " > /target/boot/initrd.debinstall bs=1k count=");
	 strcat(command,size);
	 if ( execlog(command,LOG_DEBUG) )
	 {
		 problemBox("can't dd",_("Critical Error"));
		 return 1;
	 }
	 execlog("/target/bin/gzip -9f /target/boot/initrd.debinstall",LOG_DEBUG);
	 execlog("mv /target/boot/initrd.debinstall.gz /target/boot/initrd.debinstall",LOG_DEBUG);
	 sync(); /* I do not know if we need this, but this can't harm */
	 free(command);
	 return 0;
}	 
	 

#if #cpu(i386)
static int run_lilo(char *boot, int fixmap, struct d_choices *others, int o_number)
{
  FILE* filep;
  int status = 0, i=0;
#ifndef _TESTING_
  struct stat statbuf;
  char * devnumber;
  char * size;
  /* files that we need */
  char  *files[]={
		"/sbin/insmod",
		"/bin/sh",
		"/bin/mount",
		"/bin/umount",
		"/lib/ld-linux.so.2",
		"/lib/libc.so.6",
		"" /* we use the NULL string to find the end */
  };
#endif
  char *append, *prompt="";
  char * bootpart_map;
  char * boot_map;
  
  pleaseWaitBox(_("Running LILO to make the kernel able to boot "
		  "from the hard disk without a boot floppy..."));

  if ((filep = fopen(BC_LILO_CONF, "w")) == NULL) {
    boxPopWindow();
#ifdef _TESTING_
    fprintf(stderr, "Cannot open %s: %s\n", BC_LILO_CONF, strerror(errno));
    sleep(2);
#endif
    return 1;
  }

  /* shouldn't be possible for name to be blank, but would explain Bug #53807 */
  assert(Root->name && Root->name[0]);
  assert(boot && boot[0]);

  boot_map = calloc(1, 32);
  bootpart_map = calloc(1, 32);
  if(fixmap==1) {
     /* harddisk with Boot's location */
     char *bootpart_dev = strndup(Boot->name, 8);
     /* okay, setting real int13 values for the root and boot devices */

     sprintf(boot_map,"disk=%s\nbios=0x80\n", first_ide_disk());
     /* if /boot is elsewhere, map it too */
     if(strcmp(bootpart_dev, first_ide_disk()) != 0)
        sprintf(bootpart_map,"disk=%s\nbios=0x%i\n", bootpart_dev, *(bootpart_dev+7) - *(boot+7) + 80);
  }
  else {
     bootpart_map = "#disk=/dev/sda\n#    bios=0x80\n";
     boot_map = "#disk=/dev/hde\n#    bios=0x81\n";
  }

  INFOMSG("writing lilo.conf with boot of '%s' and root of '%s'%s",
          boot, Root->name, fixmap ? " and device mapping" : "");

  if(o_number > 0) 
     prompt = "prompt\ntimeout=150\n";
  else 
     prompt = "";

  if (append_opts && strlen (append_opts)) {
      append = (char *)malloc (strlen (append_opts) + 1 + 10); /* "append=\"" + "\"\n"*/
      sprintf (append, "append=\"%s\"\n", append_opts);
      fprintf(filep, BC_LILOTEMPLATE1, boot_map, bootpart_map, boot, Root->name, prompt, append);
      free (append);
  } else {
      fprintf(filep, BC_LILOTEMPLATE1, boot_map, bootpart_map, boot, Root->name, prompt, "");
  }
#ifndef _TESTING_
  if ( NAME_ISDIR(MOD_SAVE_DIR,&statbuf) )
  {
	 devnumber = get_device_number(Root->name);
	 size = get_ramdisk_size(files);
	 if ( create_ramdisk(devnumber,size) )
	 {
		 return 1;
	 }
	 free(size);
	 free(devnumber);
	 fprintf(filep,BC_LILOTEMPLATE2);
  }
#endif
  fprintf(filep, BC_LILOTEMPLATE3);

  for(i=0; i < o_number; i++) {
     fprintf(filep, "other=%s\n  label=\"%s(%s)\"\n\n", others[i].tag, others[i].string, strrchr(others[i].tag, '/')+1);
  }
  fclose(filep);
  chmod(BC_LILO_CONF, 0640);
  boxPopWindow();
#if !defined _TESTING_
  status = execlog(BC_RUN_LILO, LOG_INFO);
#endif
  return status;
}



static int install_mbr(const char* boot, int *installed_mbr)
{
  int status = 0;
  char *device = "";

  /* FIXME:  Check to make sure this is correct */
#if #cpu (i386)
  if (is_scsird(boot))
    device = DEV_RD_C0D0;
  else if (is_scsiida(boot)) 
    device = DEV_IDA_C0D0;
  else if (is_scsicciss(boot))
    device = DEV_CCISS_C0D0;
  else if (is_ataraid(boot))
    device = DEV_ATARAID_D0;
  else 
#endif
  {
    int idx;
    idx=firstnumber(boot);
    device=strdup(boot);
    device[idx]='\0';
  }

  snprintf(prtbuf, sizeof(prtbuf), _("A master boot record is required to boot the system.\nIf you are already using a boot manager, and want to keep it, answer \"No\" to the following question. If you don't know what a boot manager is or whether you have one, answer \"Yes\".\n\nInstall a master boot record on %s?"),device);
  if (bootargs.isquiet || 
      yesNoBox(prtbuf, _("Create Master Boot Record?")) == DLG_YES) {
    snprintf(prtbuf, sizeof(prtbuf), BC_INST_MBR, device);
    INFOMSG("installing mbr on %s", device);
#ifndef _TESTING_
    status = execlog(prtbuf, LOG_INFO);
#else
    status = 0;
#endif
    if (status == 0)
      *installed_mbr = 1;

    if (! bootargs.isquiet)
	problemBox(_("The master boot record program that was just installed has features not found in standard MBR programs.  A prompt can be triggered on startup by holding down the shift key. It shows the active partition, followed by the letters `F', for floppy, and `A', for advanced. You can press a number to boot the active partition, or `F' to boot from the floppy drive. Pressing `A' gets you a new prompt, this time listing all of the primary partitions on the drive, not just the active ones. By default, if no key was pressed, it boots into the first active partition on your hard disk. For more information about this MBR program, and how you can customize its configuration, please read the 'install-mbr' manual page."), _("Important Information about the installed MBR"));
  }
  return status;
}
#endif

static int set_boot_default(const char* boot)
{
  int idx, status = 0;
  int rootnum;
  char *device, *partition;

  idx         = firstnumber(boot);
  device      = strdup(boot);
  rootnum     = atoi(&device[idx]);
  device[idx] = '\0';
  partition   = strdup(boot + idx);
  snprintf(prtbuf, sizeof(prtbuf), _("If you want the Debian system to boot automatically from the hard disk when you turn your system on, answer \"Yes\" to the following question.\nIf you have another operating system that you'd prefer to be the one that boots automatically, answer \"No\".\nBoot the Debian system on %s as the default?"), boot);
  if (yesNoBox(prtbuf,_("Make Linux the Default Boot Partition?")) == DLG_YES) {
    INFOMSG("making linux partition '%s' the default for device '%s'",
            partition, device);
    snprintf(prtbuf, sizeof(prtbuf), BC_BOOT_DFLT, device, partition);
    status = execlog(prtbuf, LOG_INFO);
  } else {
    snprintf(prtbuf, sizeof(prtbuf), _("You have selected not to boot Debian as the default. To boot into Debian, you will need to hold down the SHIFT key on startup. When you see a prompt that says 123F: (the numbers may be different, depending on your configuration) enter %d (the root partition of your Debian installation). This will only work if you have previously chosen to install the master boot record."), rootnum);
    problemBox(prtbuf, _("How to boot into Debian"));
  }
  free(device);
  free(partition);
  return status;
}
#endif /* USE_LILO */


int make_bootable(void) 
{
  int status = 0;
  char *boot = 0;

#if #cpu (i386)
  int installed_mbr = 0, fix_mapping = 0;
  char *device = "";
  struct d_choices opt[2];
  struct stat statbuf;
  int res;
  struct d_choices *bootable = NULL;
  int boot_number=0;

  boot = Boot->name;

  if (! boot) {
    /* problem earlier and user responded "no": lilo.conf will be broken */
    problemBox(_("Please boot the system using the rescue boot method "
                 "and configure LILO manually."), _("Problem"));
    return 1;
  }

  /* set device to what we think is the boot disk */
  if (is_ide(boot)) device = first_ide_disk ();
  else if (is_scsi(boot)) device = DEV_SDA;
  else if (is_scsird(boot)) device = DEV_RD_C0D0;
  else if (is_scsiida(boot)) device = DEV_IDA_C0D0;
  else if (is_scsicciss(boot)) device = DEV_CCISS_C0D0;
  else if (is_ataraid(boot)) device = DEV_ATARAID_D0;
  else {
    INFOMSG("couldn't determine drive type for new boot partition %s", boot);
    device = inputBox(_("The attempt to deduce what your boot disk is failed.  "
                        "Please enter the name of your boot disk, for instance, "
                        "'/dev/hda' or '/dev/sda'."),
                      _("Choose Boot Disk"), DEV_HDA);
    if (! NAME_ISBLK(device, &statbuf) ) {
      vaproblemBox(_("Invalid Boot Disk"), 
                   _("There is no block device '%s'."), device);
      return 1;
    }
  }

  /* XFS stores the superblock in the very first block of the
     filesystem.  This means that installing LILO on a partition
     containing an XFS superblock will destroy the filesystem.  So we
     can only install onto the MBR on ia32 systems when we're using
     XFS. */
  if ((is_xfs)
      && ({
	  struct fdisk_partition *part = fdisk_find_partition_by_mntpoint("/target/boot");
	  is_fstype(part ? "/target/boot" : "/target", "xfs");
      })) {
      if (!yesNoBox(_("The boot file system type is XFS.  In order to make your system "
		      "bootable, LILO can only be installed into the master boot record "
		      "(MBR), which means LILO will take control of the boot process.\nContinue?"),
		    _("Warning")))
	  return 1;

      boot = device;
      installed_mbr = 1;
  } else {
      opt[0].tag = device;
      /* TRANS: keep this under one line (about 70 characters) */
      opt[0].string = _("Install LILO in the MBR (use this if unsure).");
      opt[1].tag = Boot->name;
      /* TRANS: keep this under one line (about 70 characters) */
      opt[1].string = _("Install LILO in the boot partition's boot sector.");
      if (Boot == Root)
	  /* TRANS: keep this under one line (about 70 characters) */
	  opt[1].string = _("Install LILO in the root partition's boot sector.");
      snprintf(prtbuf, sizeof(prtbuf), _("LILO can be installed either into the master boot record (MBR), or into the %s boot block. If installed into the MBR, LILO will take control of the boot process. If you choose not to install LILO into the MBR, you will have the opportunity later on to install an alternative MBR program (for bootstrapping LILO).\n"),
	       Boot->name);
      if ((res = menuBox(prtbuf, _("Where should the LILO boot loader be installed?"), opt, 2, 1)) == -1)
	      return 0;
      
      boot = opt[res].tag;
      installed_mbr = (res == 0);
  }

  /* screen is sometimes not cleared, forcing */
  newtCls();

  if (! NAME_ISBLK(boot, &statbuf) ) {
    vaproblemBox(_("Invalid Boot Disk"),
                 _("There is no block device '%s'.  You need to try this step again "
                   "with different choices, or perhaps make the device by hand."),
                 boot);
    return 1;
  }

  /* if there is no SCSI hd, no /dev/hda disk, but another IDE harddisks,
   * this looks like the typical bios-mapping situation. Workaround... */
  /* But do not do this on ATARAID, there we should trust lilo */
  sprintf(prtbuf, _("Your first IDE harddisk is not connected as the primary master device (/dev/hda). To get around this, most BIOS setups enable emulation so that the first disk is seen as IDE-0 in the boot environment. This emulation cannot be detected by Linux or LILO. However, we can define %s as the first hard disk in the lilo configuration to allow LILO to take advantage of the emulation. If you think your BIOS works this way or if you installed boot management software to enable this emulation, you probably want to say yes. Would you like to enable this mapping?\n\n"), first_ide_disk());
  if( (first_scsi_disk() == NULL) &&
      (first_ide_disk() != NULL) &&
      (strcmp(first_ide_disk(), DEV_HDA) != 0) &&
      (!is_ataraid(boot)) &&
      (  yesNoBox(prtbuf, _("Virtual IDE device mapping")) )
    )
  {
     fix_mapping=1;
     goto idemapped;
  }
  fix_mapping=0;

  /* make sure this is first IDE or SCSI device  */
  if ( ! ( is_first_ide(Boot) || is_first_scsi(Boot) ||
	   yesNoBox(_("With your current configuration, LILO will try to boot "
		      "from a disk that is not your first IDE or SCSI disk. "
		      "It's possible that your BIOS doesn't support that.\n\n"
		      "Do you want to install the MBR anyway?"), _("Problem")) ) ) {
      return 1;
  }

idemapped:
  /* check for booting from SCSI but we also have /dev/hda */
  if ( is_first_scsi(Boot) && fdisk_find_disk(DEV_HDA) &&
	 ! yesNoBox(_("With your current configuration, LILO will try to boot "
		      "from your first SCSI disk, but your system also has a "
		      "master IDE disk at the primary IDE controller. "
		      "It's possible that your BIOS doesn't support that.\n\n"
		      "Do you want to install the MBR anyway?"), _("Problem")) ) {
      return 1;
  }

  /* screen is sometimes not cleared, forcing */
  newtCls();

  /* begin other-detection */
  if(fdisk_partitions != NULL) {
     struct fdisk_partition *testpart = fdisk_partitions;
     char bootmagic[2];
     FILE *fd;

     bootable = (struct d_choices *) malloc(20*sizeof(struct d_choices));
     boot_number=0;
     while(testpart) {
        bzero(bootmagic, 2);
        if( (!strcmp(testpart->name, Root->name)) || testpart->type == 0x05 ||testpart->type == 0x0f)
           goto nextone; // skip Root, skip extended partitions
        fd = fopen(testpart->name, "r");
        if (!fd)
           goto nextone;
        fseek (fd, 510, SEEK_SET);
        fread (bootmagic, 1, 2, fd);
        if(!strncmp(bootmagic, "U\252", 2)) {
           /* we have one */
           char *type;
           switch (testpart->type) {
              case 0x83:
                 type = "Linux";
                 break;
              case 0x41:
              case 0x81:
                 type = "Minix";
                 break;
              case 0x07:
                 type = "WinNT";
                 break;
              case 0x0c:
              case 0x0b:
                 type = "Windows";
                 break;
              case 0x06:
              case 0x04:
                 type = "Win/DOS";
                 break;
              case 0x01:
                 type = "DOS";
                 break;
              default:
                 type = "Other";
           }
           bootable[boot_number].tag = testpart->name;
           bootable[boot_number].state = 0;
           bootable[boot_number++].string = type;  //FIXME
        }
        fclose(fd);
nextone:
        testpart = testpart->next;
     }
     
     if(boot_number) {
        int status;
        struct d_choices options[3];
        options[0].tag = _("Include");
        options[0].state = 0;
        options[0].string = _("Put all into the menu");
        options[1].tag = _("View");
        options[1].state = 0;
        options[1].string = _("Show the list of bootable partitions");
        options[2].tag = _("Ignore");
        options[2].state = 0;
        options[2].string = _("Do not include any other partition");
redisplay_options:
        status = menuBox(_("LILO setup discovered boot signatures on other partitions in your system. If you installed other operating systems on this machine, you may want to include them in the boot menu. Please choose how to handle them: "), _("Other bootable partitions"), options, 3, 0);
        switch (status) {
           case 0:
              break;
           case 1:
              menuBox(_("Following partitions are bootable: "), _("Other bootable partitions"), bootable, boot_number, 1);
              goto redisplay_options;
           case 2:
              boot_number=0;
              break;
           default:
              boot_number=0;
              return 1; /* Cancelled, failed, whatever */
        }
     }
  }

#if defined _TESTING_
  fprintf(stderr, "About to run_lilo(), boot is %s\n", boot);
  sleep(2);
#endif

  status = run_lilo(boot, fix_mapping, bootable, boot_number);

#if defined _TESTING_
  fprintf(stderr, "Just ran_lilo(), status is %d\n", status);
  sleep(2);
#endif

  if (status) {
    problemBox(_("LILO wasn't able to install. In order boot your system, you must create a boot floppy. It won't be bootable without one.\nThe most common reason for LILO installation failure is a kernel disk location extending beyond the 1023rd cylinder. With an older BIOS, LILO is unable to load any block of the kernel from a disk cylinder numbered higher than 1023. This is often a problem with disks larger than 524MB or so. One way to solve the problem is to make a small (5-10Mb) partition at the beginning of the disk and mount it on \"/boot\", so that the entire \"/boot\" partition will be below the 1023rd cylinder. Please refer to the installation manual for further information.\n"), _("Problem"));
    return 1;
  } else {
     problemBox(_("NOTE: The default LILO configuration is not secure enough against local attacks. A user with physical access can modify the boot parameters and (for example) execute every program with root permissions. Please read the LILO documentation after the installation, especially the part about \"password\" and \"restricted\" options."), _("Securing LILO"));
    if (!installed_mbr) {
      status = install_mbr(boot, &installed_mbr);
      if (status == 0 && installed_mbr) {
	  status = set_boot_default(boot);
#if !defined _TESTING_
	  execlog(BC_RUN_LILO, LOG_INFO); /* Must run `lilo' again, otherwise it will not reboot. (MBR prompt is all I get without this line - karlheg) */
#endif
      }
    }
  }
  notInstalledLILO = 0;
  
#elif #cpu (m68k)
  boot = Root->name;
  if (strcmp(Arch2, "VME") == 0) {
    boot=strdup(Root->name);
    boot[firstnumber(boot)] = '\0';
    status=run_vmelilo(boot);
    if(status!=0) {
      problemBox(_("VMELILO wasn't able to install. You'll still be able to boot your system if you create a boot floppy, but it won't be able to boot without a floppy."),_("Problem"));
      free(boot);
      return 1;
    }
    notInstalledLILO=0;
    free(boot);
  } else {
    problemBox(_("Installing a boot loader is not yet possible for Debian/m68k"),
	       _("Problem"));
  }
#elif #cpu (powerpc)
  struct fdisk_partition *boot_part;
  boot = Root->name;

  if (strcmp(Arch2, "PReP") == 0) {
    if ((boot_part = fdisk_find_partition_by_type(PTYPE_PREP_BOOT)) != NULL) {
      /* Copy a kernel to the PReP boot partition */
      pleaseWaitBox(_("Copying the operating system kernel..."));
      snprintf(prtbuf, sizeof(prtbuf), BC_RUN_PREPDD, "/target/vmlinuz", boot_part->name);
      status=execlog(prtbuf, LOG_INFO);
      /* FIXME MDP: Handle the error cases */
      boxPopWindow();
      notInstalledLILO=0;
      return 0;
    }
    else {
        /* FIXME MDP: Add in error message, must learn pointerize */
/*      problemBox(_("No PReP boot partition found"),_("Problem")); */
      return 1;
    }
  }
  else {
    status = run_plilo();
    if(status != 0) {
	return 1;
    }
  }
  notInstalledLILO=0;
#elif #cpu (sparc)
  status=run_silo();
  if(status!=0) {
    problemBox(_("SILO wasn't able to install. You'll still be able to boot your system if you create a boot floppy, but it won't be able to boot without a floppy."),_("Problem"));
    return 1;
  }
  notInstalledLILO=0;
#elif #cpu (alpha)
  if (srm_boot) {
    status = run_aboot();
    if (status) {
      problemBox(_("aboot wasn't able to install.  "
		   "Most likely this is because your first partition overlaps "
		   "the area where aboot was to be written. "
		   "Either repartition your disk and try again, or "
		   "boot the system using the rescue boot method "
		   "and install aboot manually."), _("Problem"));
      return 1;
    }
  } else {
    /* Try to find a FAT partition and copy MILO/APB to it */
    struct d_choices *choices;
    struct fdisk_partition **plist, *p;
    int r, i = 0;
    int is_apb = 0;

    choices = malloc(4 * sizeof(struct d_choices));
    plist   = malloc(4 * sizeof(struct fdisk_partition *));

    for (p = fdisk_partitions_by_type[FSTYPE_MSDOS];
	 p; p = p->next_by_type) {
      /* add it to the list */
      plist[i]          = p;
      choices[i].string = p->name;
      choices[i].tag    = NULL;
      choices[i].state  = 0;
      i++;
      if ((i % 4) == 0) {
	choices = realloc(choices,
			  (i+4) * sizeof(struct d_choices));
	plist   = realloc(plist,
			  (i+4) * sizeof(struct fdisk_partition *));
      }
    }
    if (i == 0) {
      problemBox(_("The MILO and APB bootloaders must reside on a FAT "
		   "partition in order to be read by ARC/AlphaBIOS firmware.  "
		   "You appear not to have any DOS FAT partitions.  "
		   "This means you will not be able to boot directly "
		   "from disk.\n"), _("No FAT partitions detected"));
      return 1;
    }
    r = menuBox(_("Select a FAT partition to install MILO or APB on"),
		_("Select Partition"), choices, i, 1);
    if (r < 0)
      p = NULL;
    else
      p = plist[r];

    free(plist);
    free(choices);
    if (p == NULL) return 1;
    if (strcmp(Arch2, "nautilus") == 0) {
      is_apb = 1;
      status = install_apb(boot, p);
    } else
      status = install_milo(boot, p);
    if (status) {
      sprintf(prtbuf, _("%s wasn't able to install. You'll still be "
			"able to boot your system from the disk you "
			"used to boot the Debian installer."),
	      is_apb ? "APB" : "MILO");
      problemBox(prtbuf, _("Problem"));
      return 1;
    }
  }
  notInstalledLILO=0;
#elif #cpu (arm)
  if (!strcmp(Arch2, "riscpc")) {
      problemBox(_("This option has no effect.  Debian GNU/Linux is "
		   "launched from RISC OS."),
		 _("Problem"));
  }
  else if (!strcmp(Arch2, "netwinder")) {
      extern const char *Arch3;
      if (!strcasecmp(Arch3, "rebel-netwinder")) {
	  problemBox(_("This option has no effect.  You need to configure "
		       "NeTTrom manually."),
		     _("Problem"));
      }
      else if (!strcasecmp(Arch3, "chalice-cats")) {
	  status = execlog(BC_RUN_CATSBOOT, LOG_INFO);
      }
      else {
	  problemBox(_("Don't know how to boot this system.  You must "
		       "configure your loader manually."),
		     _("Problem"));
      }
  }
  notInstalledLILO=0;
#elif #cpu(hppa)
  status=run_palo();
  if(status!=0) {
    problemBox(_("PALO wasn't able to install.  You can still remote boot and point the root device to the installed system."),_("Problem"));
    return 1;
  }
  notInstalledLILO=0;
#elif #cpu (ia64)
  status=install_elilo();
  if(status!=0) {
    problemBox(_("ELILO wasn't able to install.  You can still remote boot and point the root device to the installed system."),_("Problem"));
    return 1;
  }
  notInstalledLILO=0;
#elif #cpu (s390)
  boot = Root->name;
  status=install_zipl(boot);
  if(status!=0) {
    problemBox(_("ZIPL could not be installed. You are not able to IPL this Linux system right now. Please see the log file and manually repair the problem."),_("Problem"));
    return 1;
  }
  notInstalledLILO=0;
#elif ( #cpu(mips) && !(defined (MIPSEL)) )
  status=run_dvhtool();
  if(status!=0) {
    problemBox(_("Dvhtool wasn't able to install the kernel. You can still remote boot and point the root device to the installed system."),_("Problem"));
    return 1; 
  }
  notInstalledLILO=0;
#elif ( #cpu(mips) && defined (MIPSEL) ) 
  status=run_delo();
  if(status!=0) {
    problemBox(_("You need to setup DELO yourself, sorry"),
	       _("Problem"));
    return 1;
   }
   notInstalledLILO=0;
#else
# warning no boot loader to configure
#endif
  return status; 
}

#ifdef _TESTING_
/*
 * To test, compile using: make bootconfig_test
 */

void readcmdline (void);
void setup_image_names(void);
int main(int argc, char** argv)
{
  LOAD_TRMFILE("test.trm");
  readcmdline();
  get_kver();
  Root = calloc(1, sizeof(struct fdisk_partition));
  fdisk_reread();
#if #cpu(alpha)
  srm_boot = get_alpha_boot_method();
  Arch2 = get_alpha_model();
#endif
  setup_image_names();
  Root->name = strdup("/dev/hdc1");
  Root->disk = fdisk_disks;
  InstallationRootDevice = strdup("/dev/hdc");

  append_opts = strdup ("aha152x=0x200");

  if (argc != 2) {
      fprintf(stderr, "Boot Config test program\n%s (floppy|lilo)\n", argv[0]);
      exit(1);
  }
  
  if (strcmp(argv[1], "floppy") == 0) {
    boxInit();
    make_boot_floppy();
    boxFinished();
  }
  else if (strcmp(argv[1], "lilo") == 0) {
    printf("About to begin test number 1.\n");
    sleep(2);

    Boot = Root = calloc(1, sizeof(struct fdisk_partition));

    fdisk_disks = calloc(1, sizeof(struct fdisk_disk));
    fdisk_disks->name = strdup("/dev/hdc");
    fdisk_disks->partition_format = "msdos";

    Root->name = strdup("/dev/hdc1");
    Root->mount_point = strdup("/");
    Root->disk = fdisk_disks;

    InstallationRootDevice = strdup("/dev/hdc");

    boxInit();
    make_bootable();
    boxFinished();
    printf("Test number one complete...\n");
    sleep(2);

    printf("About to begin test number two.\n");
    sleep(2);

    Boot = calloc(1, sizeof(struct fdisk_partition));
    Root = calloc(1, sizeof(struct fdisk_partition));

    fdisk_disks = calloc(1, sizeof(struct fdisk_disk));
    fdisk_disks->name = strdup("/dev/hdc");
    fdisk_disks->partition_format = "msdos";

    Boot->name = strdup("/dev/hdc1");
    Boot->mount_point = strdup("/boot");
    Boot->disk = fdisk_disks;

    Root->name = strdup("/dev/hdc3");
    Root->mount_point = strdup("/");
    Root->disk = fdisk_disks;

    InstallationRootDevice = strdup("/dev/hdc");

    boxInit();
    make_bootable();
    boxFinished();
    printf("Test number two complete.\n");
    sleep(2);
  }
  else {
    fprintf(stderr, "Boot Config test program\n%s (floppy|lilo)\n", argv[0]);
    exit(-1);
  }
  exit (0);
}
#endif

/*
 * Local variables:
 *   c-basic-offset: 4
 *   indent-tabs-mode: t
 * End:
 */
