/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * sema.c: Copyright (C) Eric Prevoteau <www@ac2i.tzo.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <limits.h>
#include <pthread.h>
#include <glib.h>

#include "sema.h"

#if !(defined(BSD) && (BSD >= 199103))
       #if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
       /* union semun is defined by including <sys/sem.h> */
       #else
       /* according to X/OPEN we have to define it ourselves */
       union semun {
               int val;                    /* value for SETVAL */
               struct semid_ds *buf;       /* buffer for IPC_STAT, IPC_SET */
               unsigned short int *array;  /* array for GETALL, SETALL */
               struct seminfo *__buf;      /* buffer for IPC_INFO */
       };
       #endif
#endif

#ifndef IPC_ALLOC
#define IPC_ALLOC 0
#endif

#include "uaddr.h"
#include "status.h"

/***********************************************************************/
/* the following functions manage the bandwidth upload limit           */
/* The system is build on 2 semaphores and 1 file                      */
/* 1) the file contains the semaphore key. This allows all DCTC on the */
/*    same computer to share the bandwidth                             */
/* 2) The second side is the managment function itself. It is the 2    */
/*    semaphore. The first semaphore is used to choose which DCTC (if  */
/*    more than one exists) is the "clock" master. The second semaphore*/
/*    is the bandwidth limitation semaphore (BLS). Despite the fact it */
/*    is a semaphore, it acts more like a set of token.                */
/*    a) Every second, the clock master resets BLS to a given value    */
/*      (for example, 8 for 4KB/s (1=512byte/s).                       */
/*    b) When a DCTC wants to sends something, it must first acquire 1 */
/*       or more token (never acquire more than 1 at the same time,    */
/*       if you need 3 for example, you must acquire 3 times 1 token,  */
/*       this is needed if for example the speed is limited to 4KB/s   */
/*       (max BLS=8) and you want to send 8KB (you must have 16 token).*/
/* Addendum: The current speed limit is stored inside a 3rd semaphore  */
/*       This value must be shared by all DCTC (because we don't know  */
/*       which one is the clock master. A piece of shared memore is    */
/*       sufficient but is too heavy to set up. Thus, to simplify, the */
/*       3rd semaphore value is copied into the 2nd by the clock master*/
/*       every second.                                                 */
/***********************************************************************/

/******************************/
/* initialize semaphore array */
/*************************************************************************************/
/* input: keyfile : if not exists, it is created and the semaphore key is put inside */
/*                  if exists but the semaphore key inside is invalid, same as above */
/*                  if exists and contains a valid key, nothing is done              */
/*        spd_limit is the default speed limit (in number of 512bytes slice)         */
/*************************************************************************************/
/* output: 0=ok, !=0=error                                              */
/*         on success, *cur_semid is the semaphore id to use with semop */
/************************************************************************/
int do_sema_init(char *keyfile, int *cur_semid, int spd_limit, int dl_spd_limit, int gath_spd_limit)
{
	int fd;
	key_t key;

	int semid;

	fd=open(keyfile,O_CREAT|O_WRONLY|O_EXCL,0600);		/* create the file if not exists */
	if(fd==-1)
	{
		if(errno==EEXIST)
		{
			printf("file exists.\n");
			fd=open(keyfile,O_RDWR);
			if(fd==-1)
			{
				perror("open(R).");
				return 1;
			}

			if(read(fd,&key,sizeof(key))!=sizeof(key))
			{
				close(fd);
				/* fail to read current key, create a new file and force generation of new sema */
				create_new_sema:
				printf("creating new sema.\n");
				fd=open(keyfile,O_CREAT|O_WRONLY,0600); 
				if(fd==-1)
				{
					perror("open(W2).");
					return 1;
				}
			
			}
			else
			{
				close(fd);
				/* a key exist */
				semid=semget(key,0,IPC_ALLOC);
				if(semid==-1)
					goto create_new_sema;
				printf("current sema found.\n");
				goto eofunc;
			}
		}
		else
		{
			perror("open(W).");
			return 1;
		}
	}
	
	printf("creating.\n");
	key=rand();
	while((semid=semget(key,SEMA_ARRAY_LEN,IPC_CREAT|IPC_EXCL|0600))==-1)	/* allocate 3 semaphores */
		key++;

	printf("semid=%d\n",semid);
	printf("created %08X.\n",key);

	if(write(fd,&key,sizeof(key))!=sizeof(key))
	{
		close(fd);
		unlink(keyfile);
	}

	/* initialize sema array */
	{
		union semun v;

		v.val=1;
		if(semctl(semid,0,SETVAL,v)==-1)
			perror("semctl0");

		/* upload speed values */
		v.val=0;
		if(semctl(semid,1,SETVAL,v)==-1)
			perror("semctl1");
		v.val=spd_limit;
		if(semctl(semid,2,SETVAL,v)==-1)
			perror("semctl2");

		/* download speed values */
		v.val=0;
		if(semctl(semid,3,SETVAL,v)==-1)
			perror("semctl3");
		v.val=dl_spd_limit;
		if(semctl(semid,4,SETVAL,v)==-1)
			perror("semctl4");

		/* download speed values */
		v.val=0;
		if(semctl(semid,5,SETVAL,v)==-1)
			perror("semctl5");
		v.val=gath_spd_limit;
		if(semctl(semid,6,SETVAL,v)==-1)
			perror("semctl6");
	}
	close(fd);
	eofunc:
	*cur_semid=semid;
	return 0;
}

/******************************************************************/
/* this is the thread acting as clock                             */
/* every second, it resets the 2nd semaphore to its initial value */
/******************************************************************/
static void *sema_master(void *dm_val)
{
	int semid=(int)dm_val;
	FLAG1_STRUCT fs1;

#ifndef BUGGY_LIBS

	if(sizeof(fs1)!=sizeof(unsigned long int))
		fprintf(stderr,"FLAG1_STRUCT has an invalid size (%d), result can be erroneous\n",sizeof(fs1));
	/* set the is_clock_master flag of the gstatus */
	fs1.full=GET_GSTATUS_FLAG1();
	fs1.bf.is_clock_master=1;
	SET_GSTATUS_FLAG1(fs1.full);

	while(1)
	{
		union semun v;
#else
	union semun v;								/* on some old buggy libs, calling semctl(GETVAL) from thread before calling SETVAL generate bus error */

	if(sizeof(fs1)!=sizeof(unsigned long int))
		fprintf(stderr,"FLAG1_STRUCT has an invalid size (%d), result can be erroneous\n",sizeof(fs1));
	/* set the is_clock_master flag of the gstatus */
	fs1.full=GET_GSTATUS_FLAG1();
	fs1.bf.is_clock_master=1;
	SET_GSTATUS_FLAG1(fs1.full);

	v.val=0;										/* we use 0 to temporarily alter the speed without a significant impact */
	if(semctl(semid,1,SETVAL,v)==-1)
		perror("semctl0");
	while(1)
	{
#endif
		int i;

		/* reset upload/download and gather speed */
		for(i=0;i<3;i++)
		{
			/* reset speed */
			v.val=semctl(semid,2+2*i,GETVAL);
			if(v.val==-1)
				v.val=SEMVMX;

			if(semctl(semid,1+2*i,SETVAL,v)==-1)
				perror("semctl1");
		}
		sleep(1);
	}
}

/**********************************************/
/* create clock thread                        */
/* on error, the master semaphore is released */
/**********************************************/
static void create_sema_master(int semid)
{
	pthread_attr_t thread_attr;
	pthread_t thread_id;

   pthread_attr_init (&thread_attr);
   pthread_attr_setdetachstate(&thread_attr, PTHREAD_CREATE_DETACHED);
   if(pthread_create(&thread_id,&thread_attr, (void*)sema_master,(void*)semid)!=0)
	{
		/* if the creation of the clock thread fails, release the master sema */
		/* else nobody will try to create a new clock and all xfers will hang */
		struct sembuf sb={0,+1,SEM_UNDO};			/* master sema */
		semop(semid,&sb,1);
	}
	else
	{
		/* to reduce duplicated code, the DCTC being the clock master is also the one performing UADDR action */
		create_uaddr_thread();
	}
}

/******************************************************************************************/
/* to avoid forever hanging of download, we must regularly check if a clock master exists */
/******************************************************************************************/
void check_sema_master(int semid)
{
	struct sembuf sb={0,-1,IPC_NOWAIT|SEM_UNDO};			/* master sema */

	if(semop(semid,&sb,1)==0)
	{
		/* to get slice, the function checks if the clock thread still runs */
		create_sema_master(semid);
	}
}

/************************/
/* get 1 512Bytes slice */
/*******************************************/
/* the function ends when it has the slice */
/*******************************************/
void get_slice(int semid, SPD_SEMA semnum)
{
	while(1)
	{
		struct sembuf local={0,-1,0};		/* slave sema */

		local.sem_num=semnum;
		if(semop(semid,&local,1)==0)
		{
			/* we have what we want */
			return;
		}
	}
}

/*************************/
/* get nb 512Bytes slice */
/*******************************************/
/* the function ends when it has the slice */
/*******************************************/
void get_ul_slices(int semid,int nb)
{
	while(nb>0)
	{
		get_slice(semid,UL_SEMA);
		nb--;
	}
}

/************************/
/* get nb 1KBytes slice */
/*******************************************/
/* the function ends when it has the slice */
/*******************************************/
void get_dl_slices(int semid,int nb)
{
	while(nb>0)
	{
		get_slice(semid,DL_SEMA);
		nb--;
	}
}

/************************/
/* get nb 8KBytes slice */
/*******************************************/
/* the function ends when it has the slice */
/*******************************************/
void get_gather_slices(int semid,int nb)
{
	while(nb>0)
	{
		get_slice(semid,GATHER_SEMA);
		nb--;
	}
}

