/* DCTC - a Direct Connect text clone for Linux
 * Copyright (C) 2001 Eric Prevoteau
 *
 * dctc_gdl123.c: Copyright (C) Eric Prevoteau <www@a2pb.gotdns.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/*
$Id: dctc_gdl123.c,v 1.3 2003/12/28 08:12:38 uid68112 Exp $
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <netdb.h>
#include <strings.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/mman.h>
#include <glib.h>

typedef struct
{
	const gchar *dot_cmd_file;
	const gchar *data_file;
	const gchar *org_data_file;
	off_t data_file_size;
	const guint8 *mapped_data_file;
} IN_ENTRY;

/* lot of small fragments will slow download */
#define MIN_FRAG_SIZE (8192*10)


static void load_in_entry(IN_ENTRY *dir, char *dir_name)
{
	FILE *f;
	char buf[5120];
	gchar **fields;
	int fd;

	dir->dot_cmd_file=g_strconcat(dir_name,"/.cmd",NULL);
	dir->data_file=NULL;
	dir->mapped_data_file=NULL;

	f=fopen(dir->dot_cmd_file,"rb");
	if(f==NULL)
	{
		fprintf(stderr,"Unable to open file '%s' because %s\n",dir->dot_cmd_file,strerror(errno));
		exit(0);
	}

	if((fgets(buf,sizeof(buf)-1,f)==NULL)||(buf[0]!='G'))
	{
		fprintf(stderr,"Invalid file '%s'. The first line is not a G one.\n",dir->dot_cmd_file);
		exit(0);
	}
	fclose(f);

	fields=g_strsplit(buf,"|",0);
	if((fields[0]==NULL)||(fields[1]==NULL)||(fields[2]==NULL)||(fields[3]==NULL)||(fields[4]!=NULL))
	{
		fprintf(stderr,"Invalid G line '%s'. Must have exactly 4 fields.\n",dir->dot_cmd_file);
		exit(0);
	}

	dir->org_data_file=g_strdup(fields[2]);
	dir->data_file=g_strconcat(dir_name,"/",fields[2],NULL);
	dir->data_file_size=strtoul(fields[3],NULL,10);

	g_strfreev(fields);

	fd=open(dir->data_file,O_RDWR);
	if(fd==-1)
	{
		fprintf(stderr,"Unable to open file '%s' because %s\n",dir->data_file,strerror(errno));
		exit(0);
	}

	dir->mapped_data_file=mmap(NULL,dir->data_file_size,PROT_READ,MAP_SHARED,fd,0);
	if(dir->mapped_data_file==MAP_FAILED)
	{
		fprintf(stderr,"Unable to map file '%s' in memory because %s\n",dir->data_file,strerror(errno));
		exit(0);
	}
	close(fd);
}

static void create_out_entry(IN_ENTRY *dir, const char *out_dir, FILE **cmd_fd)
{
	gchar *cmd_path;
	FILE *d1_dot_cmd;
	char buf[5120];

	cmd_path=g_strconcat(out_dir,"/.cmd",NULL);
	mkdir(out_dir,0777);

	/* create the .cmd file */
	*cmd_fd=fopen(cmd_path,"wb");
	if(*cmd_fd==NULL)
	{
		fprintf(stderr,"Unable to create .cmd file '%s' because %s\n",cmd_path,strerror(errno));
		exit(0);
	}

	d1_dot_cmd=fopen(dir->dot_cmd_file,"rb");
	if(d1_dot_cmd==NULL)
	{
		fprintf(stderr,"Unable to open file '%s' because %s\n",dir->dot_cmd_file,strerror(errno));
		exit(0);
	}

	while(fgets(buf,sizeof(buf)-1,d1_dot_cmd)!=NULL)
	{
		if(fwrite(buf,1,strlen(buf),*cmd_fd)!=strlen(buf))
		{
			fprintf(stderr,"Unable to write in file '%s' because %s\n",cmd_path,strerror(errno));
			exit(0);
		}
	}

	fclose(d1_dot_cmd);

	g_free(cmd_path);
}

static void create_frag(const guint8 *mem, size_t start_pos, size_t end_pos_excluded, const char *out_dir, FILE *out_cmd_fd)
{
	int fd;
	char buf[64];
	gchar *pth;
	size_t len;

	len=end_pos_excluded-start_pos;
	if(len<MIN_FRAG_SIZE)
		return;

	sprintf(buf,"/%08X-%08X",start_pos,end_pos_excluded);
	pth=g_strconcat(out_dir,buf,NULL);

	fd=open(pth,O_WRONLY|O_CREAT|O_TRUNC,0777);
	if(fd==-1)
	{
		fprintf(stderr,"Unable to create in file '%s' because %s\n",pth,strerror(errno));
		exit(0);
	}

	if(write(fd,mem+start_pos,len)!=len)
	{
		fprintf(stderr,"Unable to write in file '%s' because %s\n",pth,strerror(errno));
		exit(0);
	}

	close(fd);

	fprintf(out_cmd_fd,"A|GDL/%s|%d|-\n",pth,start_pos);
	g_free(pth);
}

/********************************************************************/
/* copy each part of dir1 and dir2 data file into a file in out_dir */
/* and add an entry in out_dir .cmd file                            */
/********************************************************************/
static void delta_and_save(IN_ENTRY *dir1,IN_ENTRY *dir2, FILE *out_cmd_fd, const char *out_dir)
{
	size_t beg_frg=0;
	size_t pos;

	pos=0;
	while(pos<dir1->data_file_size)
	{
		if(dir1->mapped_data_file[pos]!=dir2->mapped_data_file[pos])
		{
			if(beg_frg<pos)
			{
				create_frag(dir1->mapped_data_file,beg_frg,pos,out_dir,out_cmd_fd);
			}
			beg_frg=pos+1;
		}
		pos++;
	}

	if(beg_frg<pos)
	{
		create_frag(dir1->mapped_data_file,beg_frg,pos,out_dir,out_cmd_fd);
	}
}

static void create_lock_file(const char *dir)
{
	gchar *cmd_path;
	int cmd_fd;

	cmd_path=g_strconcat(dir,"/.lock",NULL);

	/* create the .cmd file */
	cmd_fd=open(cmd_path,O_WRONLY|O_CREAT|O_TRUNC,0777);
	if(cmd_fd==-1)
	{
		fprintf(stderr,"Unable to create .lock file '%s' because %s\n",cmd_path,strerror(errno));
		exit(0);
	}

	close(cmd_fd);
}

int main(int argc,char **argv)
{
	IN_ENTRY dir1, dir2;
	FILE *out_entry_cmd_fd;

	if(argc!=4)
	{
		printf("Usage: %s first_directory second_directory output_directory\n",argv[0]);
		exit(1);
	}

	load_in_entry(&dir1,argv[1]);
	load_in_entry(&dir2,argv[2]);

	if(dir1.data_file_size!=dir2.data_file_size)
	{
		printf("%s and %s has a different size.\n",argv[1],argv[2]);
		exit(2);
	}

	if(!memcmp(dir1.mapped_data_file,dir2.mapped_data_file,dir1.data_file_size))
	{
		printf("%s and %s has the same content.\n",argv[1],argv[2]);
		exit(0);
	}

	create_out_entry(&dir1,argv[3],&out_entry_cmd_fd);

	delta_and_save(&dir1,&dir2,out_entry_cmd_fd,argv[3]);
	create_lock_file(argv[3]);
	fclose(out_entry_cmd_fd);
	exit(0);
}
