/*
 *   This file is part of Dianara
 *   Copyright 2012-2017  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "contactmanager.h"

ContactManager::ContactManager(PumpController *pumpController,
                               GlobalObject *globalObject,
                               QWidget *parent) : QWidget(parent)
{
    this->pController = pumpController;
    this->globalObj = globalObject;

    this->followingManually = false;


    // After receiving a contact list, update it
    connect(pController, SIGNAL(contactListReceived(QString,QVariantList,int)),
            this, SLOT(setContactListsContents(QString,QVariantList,int)));

    // After receiving the list of lists, update it
    connect(pController, SIGNAL(listsListReceived(QVariantList)),
            this, SLOT(setListsListContents(QVariantList)));

    // After having checked if a webfinger ID is valid, follow it (or not)
    connect(pController, SIGNAL(contactVerified(QString,int,bool,QString)),
            this, SLOT(followContact(QString,int,bool,QString)));

    // If a person can't be followed at the moment, re-enable widgets to try again
    connect(pController, SIGNAL(cannotFollowNow(QString)),
            this, SLOT(onCantFollowNow(QString)));


    mainLayout = new QVBoxLayout();
    mainLayout->setAlignment(Qt::AlignTop);


    QString webfingerHelpMessage = tr("username@server.org or "
                                      "https://server.org/username");

    topLayout = new QHBoxLayout();
    enterAddressLabel = new QLabel(tr("&Enter address to follow:"), this);
    enterAddressLabel->setToolTip("<b></b>"
                                  + webfingerHelpMessage); // HTML tags for wordwrap
    // Ensure it will get focus first, before addressLineEdit
    enterAddressLabel->setFocusPolicy(Qt::StrongFocus);

    addressLineEdit = new QLineEdit(this);
    addressLineEdit->setPlaceholderText(webfingerHelpMessage);
    addressLineEdit->setToolTip("<b></b>"
                                + webfingerHelpMessage);  // HTML tags to get wordwrap
#if QT_VERSION >= QT_VERSION_CHECK(5, 0, 0)
    addressLineEdit->setClearButtonEnabled(true);
#endif
    connect(addressLineEdit, SIGNAL(textChanged(QString)),
            this, SLOT(toggleFollowButton(QString)));
    connect(addressLineEdit, SIGNAL(returnPressed()),
            this, SLOT(validateContactId()));

    enterAddressLabel->setBuddy(addressLineEdit);


    followButton = new QPushButton(QIcon::fromTheme("list-add-user",
                                                    QIcon(":/images/list-add.png")),
                                   tr("&Follow"),
                                   this);
    followButton->setDisabled(true); // Disabled until an address is typed
    connect(followButton, SIGNAL(clicked()),
            this, SLOT(validateContactId()));
    topLayout->addWidget(enterAddressLabel);
    topLayout->addWidget(addressLineEdit);
    topLayout->addWidget(followButton);

    mainLayout->addLayout(topLayout);
    mainLayout->addSpacing(4);


    // Widgets for list of 'following' and 'followers'
    this->followingWidget = new ContactList(this->pController,
                                            this->globalObj,
                                            "following",
                                            this);
    connect(pController, SIGNAL(contactFollowed(ASPerson*)),
            followingWidget, SLOT(addSingleContact(ASPerson*)));
    connect(pController, SIGNAL(contactUnfollowed(ASPerson*)),
            followingWidget, SLOT(removeSingleContact(ASPerson*)));

    connect(followingWidget, SIGNAL(contactCountChanged(int)),
            this, SLOT(changeFollowingCount(int)));


    this->followersWidget = new ContactList(this->pController,
                                            this->globalObj,
                                            "followers",
                                            this);


    // Widget for the list of 'person lists'
    this->listsManager = new ListsManager(this->pController, this);
    listsScrollArea = new QScrollArea(this);
    listsScrollArea->setWidget(this->listsManager);
    listsScrollArea->setWidgetResizable(true);
    listsScrollArea->setFrameStyle(QFrame::NoFrame);


    // Widget for the list of site users; users from your own server
    this->siteUsersList = new SiteUsersList(this->pController,
                                            this->globalObj,
                                            this);

    // Options menu
    optionsMenu = new QMenu("*options-menu*", this);
    optionsMenu->addAction(QIcon::fromTheme("view-refresh",
                                            QIcon(":/images/menu-refresh.png")),
                           tr("Reload Followers"),
                           this,
                           SLOT(refreshFollowers()));
    optionsMenu->addAction(QIcon::fromTheme("view-refresh",
                                            QIcon(":/images/menu-refresh.png")),
                           tr("Reload Following"),
                           this,
                           SLOT(refreshFollowing()));
    optionsMenu->addSeparator();
    optionsMenu->addAction(QIcon::fromTheme("document-export",
                                            QIcon(":/images/button-download.png")),
                           tr("Export Followers"),
                           this,
                           SLOT(exportFollowers()));
    optionsMenu->addAction(QIcon::fromTheme("document-export",
                                            QIcon(":/images/button-download.png")),
                           tr("Export Following"),
                           this,
                           SLOT(exportFollowing()));
    optionsMenu->addSeparator();
    optionsMenu->addAction(QIcon::fromTheme("view-refresh",
                                            QIcon(":/images/menu-refresh.png")),
                           tr("Reload Lists"),
                           this,
                           SLOT(refreshPersonLists()));


    optionsButton = new QPushButton(QIcon::fromTheme("configure",
                                                     QIcon(":/images/button-configure.png")),
                                    "", this);
    optionsButton->setMenu(optionsMenu);


    this->tabWidget = new QTabWidget(this);
    tabWidget->addTab(followersWidget,
                      QIcon::fromTheme("meeting-observer",
                                       QIcon(":/images/no-avatar.png")),
                      "*followers*");
    tabWidget->addTab(followingWidget,
                      QIcon::fromTheme("meeting-participant",
                                       QIcon(":/images/no-avatar.png")),
                      "*following*");
    tabWidget->addTab(listsScrollArea,
                      QIcon::fromTheme("preferences-contact-list",
                                       QIcon(":/images/button-edit.png")),
                      "*lists*");
    tabWidget->addTab(siteUsersList,
                      QIcon::fromTheme("system-users",
                                       QIcon(":/images/no-avatar.png")),
                      tr("&Neighbors"));
    tabWidget->setCornerWidget(this->optionsButton);




    this->followersCount = 0;
    this->followingCount = 0;
    this->listsCount = 0;
    this->setTabLabels();


    mainLayout->addWidget(tabWidget);
    this->setLayout(mainLayout);

    qDebug() << "Contact manager created";
}


ContactManager::~ContactManager()
{
    qDebug() << "Contact manager destroyed";
}




void ContactManager::setTabLabels()
{
    this->tabWidget->setTabText(0, tr("Follo&wers")
                                   + QString(" (%1)")
                                     .arg(QLocale::system()
                                          .toString(this->followersCount)));

    this->tabWidget->setTabText(1, tr("Followin&g")
                                   + QString(" (%1)")
                                     .arg(QLocale::system()
                                          .toString(this->followingCount)));

    this->tabWidget->setTabText(2, tr("&Lists")
                                + QString(" (%1)").arg(this->listsCount)); // Not worth localizing
}




/*
 * Write the list of contacts (following or followers)
 * to a file selected by the user
 *
 */
void ContactManager::exportContactsToFile(QString listType)
{
    QString dialogTitle = listType == "following" ?
                    tr("Export list of 'following' to a file") :
                    tr("Export list of 'followers' to a file");

    QString suggestedFilename = "dianara-"
                                + pController->currentUsername()
                                + "-"
                                + listType;

    QString filename = QFileDialog::getSaveFileName(this, dialogTitle,
                                                    QDir::homePath() + "/"
                                                    + suggestedFilename,
                                                    "").trimmed();

    if (filename.isEmpty()) // If dialog was cancelled, do nothing
    {
        return;
    }


    qDebug() << "Exporting to:"  << filename;

    QFile exportFile(filename);
    exportFile.open(QIODevice::WriteOnly);

    if (listType == "following")
    {
        exportFile.write(this->followingWidget->getContactsStringForExport()
                                               .toLocal8Bit());
    }
    else // "followers"
    {
        exportFile.write(this->followersWidget->getContactsStringForExport()
                                               .toLocal8Bit());
    }

    exportFile.close();
}



void ContactManager::enableManualFollowWidgets()
{
    this->addressLineEdit->setEnabled(true);
    this->followButton->setEnabled(!addressLineEdit->text().isEmpty());
}


/*****************************************************************************/
/*********************************** SLOTS ***********************************/
/*****************************************************************************/



void ContactManager::setContactListsContents(QString listType,
                                             QVariantList contactList,
                                             int totalReceivedCount)
{
    qDebug() << "ContactManager; Setting contact list contents";

    if (listType == "following")
    {
        if (totalReceivedCount <= 200) // Only for the first batch
        {
            this->followingWidget->clearListContents();
            followingCount = 0;
        }
        followingWidget->setListContents(contactList);

        if (totalReceivedCount < pController->currentFollowingCount())
        {
            pController->getContactList(listType, totalReceivedCount);
        }

        this->followingCount += contactList.size();
    }
    else
    {
        if (totalReceivedCount <= 200)
        {
            this->followersWidget->clearListContents();
            followersCount = 0;
        }
        followersWidget->setListContents(contactList);

        if (totalReceivedCount < pController->currentFollowersCount())
        {
            pController->getContactList(listType, totalReceivedCount);
        }

        this->followersCount += contactList.size();
    }


    // Update tab labels with number of following or followers, which were updated before
    this->setTabLabels();
}



/*
 * Fill the list of lists
 *
 */
void ContactManager::setListsListContents(QVariantList listsList)
{
    this->listsCount = listsList.count();
    // Update tab labels with number of following or followers, which were updated before
    this->setTabLabels();


    this->listsManager->setListsList(listsList);
}


void ContactManager::changeFollowingCount(int difference)
{
    this->followingCount += difference; // FIXME: pumpController should be notified

    this->setTabLabels();

    // Set "Following" tab as active, only when the contact manager is not visible
    // (following someone from an AvatarButton menu)
    if (!this->isVisible())
    {
        this->tabWidget->setCurrentIndex(1);
    }
}



/*
 * Ask for the updated list of Following
 *
 */
void ContactManager::refreshFollowing()
{
    qDebug() << "Refreshing list of Following...";
    this->pController->getContactList("following");
}


/*
 * Ask for the updated list of followers
 *
 */
void ContactManager::refreshFollowers()
{
    qDebug() << "Refreshing list of Followers...";
    this->pController->getContactList("followers");
}


/*
 * Export list of "Following" to a text file
 *
 */
void ContactManager::exportFollowing()
{
    qDebug() << "Exporting Following...";
    exportContactsToFile("following");
}


/*
 * Export list of "Followers" to a text file
 *
 */
void ContactManager::exportFollowers()
{
    qDebug() << "Exporting Followers...";
    exportContactsToFile("followers");
}


void ContactManager::refreshPersonLists()
{
    qDebug() << "Refreshing list of person lists...";
    this->pController->getListsList();
}



/*
 * Enable or disable Follow button
 *
 */
void ContactManager::toggleFollowButton(QString currentAddress)
{
    if (currentAddress.isEmpty())
    {
        this->followButton->setDisabled(true);
    }
    else
    {
        this->followButton->setEnabled(true);
    }
}



/*
 * Add the address entered by the user to the /following list.
 *
 * This supports adding webfinger addresses in the form
 * user@hostname or https://host/username.
 *
 * First step is checking, via Webfinger, if the ID exists and its server is up.
 *
 */
void ContactManager::validateContactId()
{
    QString address = this->addressLineEdit->text().trimmed();
    bool validAddress = false;

    qDebug() << "ContactManager::followContact(); Address entered:" << address;

    // First, if the address is in URL form, convert it
    if (address.startsWith("https://") || address.startsWith("http://"))
    {
        address.remove("https://");
        address.remove("http://");

        if (address.contains("/")) // Very basic sanity check
        {
            QStringList addressParts = address.split("/");
            address = addressParts.at(1) + "@" + addressParts.at(0);
            // .at(2) could also have stuff, so don't use .first() and .last()
        }
    }

    // Then, check that the address actually matches something@somewhere.tld
    if (address.contains(QRegExp(".+@.+\\..+")))
    {
        validAddress = true;
    }
    else
    {
        qDebug() << "Invalid webfinger address!";
        this->addressLineEdit->setFocus();
    }


    if (validAddress)
    {
        this->followingManually = true;
        this->followingAddress = address;

        this->tabWidget->setFocus(); // Take focus away from the widgets that will be disabled now

        this->addressLineEdit->setDisabled(true);
        this->followButton->setDisabled(true);

        this->pController->followContact(address);
    }
}


void ContactManager::followContact(QString userId, int httpCode,
                                   bool requestTimedOut, QString serverVersion)
{
    if (httpCode != 200)
    {
        QString hostname = userId.split('@').last();

        // FIXME/TMP: Unreliable, future/alternate implementations might not use this
        bool isPumpServer = serverVersion.contains("pump.io");

        QString specificError;
        if (httpCode == 404)
        {
            if (isPumpServer)
            {
                specificError = tr("The server seems to be a Pump server, "
                                   "but the account does not exist.");
            }
            else
            {
                specificError = tr("%1 doesn't seem to be a Pump server.",
                                   "%1 is a hostname")
                                .arg(hostname);
            }
        }
        else
        {
            // Really only error 0 would mean unavailable server
            // Any other error code probably means the server is not a Pump server
            specificError = tr("Following this account at this time "
                               "will probably not work.");
        }

        // Timeout was reached or general connection error
        if (requestTimedOut || httpCode == 0)
        {
            specificError.append(" "
                                 + tr("The %1 server seems unavailable.",
                                      "%1 is a hostname")
                                   .arg(hostname));
        }

        if (serverVersion.isEmpty())
        {
            serverVersion = tr("Unknown",
                               "Refers to server version");
        }

        int choice = QMessageBox::warning(this,
                      tr("Error"),
                      "<big>"
                      + tr("The user address %1 does not exist, or the "
                           "%2 server is down.")
                        .arg("<b>" + userId + "</b>")
                        .arg("<b>" + hostname + "</b>")
                      + "</big>"
                        "<br><hr><br>"
                      + specificError
                      + "<br><br>"
                      + tr("Server version") + ": <i>"
                      + serverVersion
                      + "</i>"
                        "<br><br>"
                      + tr("Check the address, and keep in mind that "
                           "usernames are case-sensitive.")
                      + "<br><br><br><br>"
                      + tr("Do you want to try following this address anyway?")
                      + "<br>"
                      + tr("(not recommended)")
                      + "<br>"
                        "<br>",
                      tr("Yes, follow anyway"),
                      tr("No, cancel"),
                      "", 1, 1);

        if (choice == 1)
        {
            if (followingManually && userId == this->followingAddress)
            {
                this->enableManualFollowWidgets();
                this->addressLineEdit->setFocus(); // Re-focus to try again, maybe

                this->followingManually = false;
                this->followingAddress.clear();
            }

            return;
        }
    }

    qDebug() << "About to follow this address:" << userId;
    this->globalObj->setStatusMessage(tr("About to follow %1...")
                                      .arg("'" + userId + "'"));
    this->pController->followVerifiedContact(userId);

    if (followingManually && userId == this->followingAddress)
    {
        this->enableManualFollowWidgets();
        this->addressLineEdit->clear(); // Which will disable Follow button again

        this->followingManually = false;
        this->followingAddress.clear();

        // Activate 'Following' tab
        this->tabWidget->setFocus();
        this->tabWidget->setCurrentIndex(1);
    }
}


void ContactManager::onCantFollowNow(QString userId)
{
    if (followingManually && userId == this->followingAddress)
    {
        this->enableManualFollowWidgets();
        this->addressLineEdit->setFocus();
    }
}

