# Copyright (C) 2008-2012 Joachim B. Haga and Fredrik Valdmanis
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Martin Sandve Alnaes, 2008.
# Modified by Anders Logg, 2008-2010.
#
# First added:  2008-03-05
# Last changed: 2013-05-29

import os
import dolfin.cpp as cpp
import ufl

__all__ = ['plot']

# Compatibility with book
def _VTKPlotter_write_ps(self, *args, **kwargs) :
  print "*** Warning: VTKPlotter::write_ps() is not implemented -- use write_pdf instead"

def plot(object, *args, **kwargs):
    """
    Plot given object.

    *Arguments*
        object
            a :py:class:`Mesh <dolfin.cpp.Mesh>`, a :py:class:`MeshFunction
            <dolfin.cpp.MeshFunction>`, a :py:class:`Function
            <dolfin.functions.function.Function>`, a :py:class:`Expression`
            <dolfin.cpp.Expression>, a :py:class:`DirichletBC`
            <dolfin.cpp.DirichletBC> or a :py:class:`FiniteElement
            <ufl.FiniteElement>`.

    *Examples of usage*
        In the simplest case, to plot only e.g. a mesh, simply use

        .. code-block:: python

            mesh = UnitSquare(4,4)
            plot(mesh)

        Use the ``title`` argument to specify title of the plot

        .. code-block:: python

            plot(mesh, tite="Finite element mesh")

        It is also possible to plot an element

        .. code-block:: python

            element = FiniteElement("BDM", tetrahedron, 3)
            plot(element)

        Vector valued functions can be visualized with an alternative mode

        .. code-block:: python

            plot(u, mode = "glyphs")

        A more advanced example

        .. code-block:: python

            plot(u,
                 wireframe = True,              # use wireframe rendering
                 interactive = False,           # do not hold plot on screen
                 scalarbar = False,             # hide the color mapping bar
                 hardcopy_prefix = "myplot",    # default plotfile name
                 scale = 2.0                    # scale the warping/glyphs
                 title = "Fancy plot"           # Set your own title
                 )

    """

    mesh = kwargs.get('mesh')

    p = cpp.Parameters()
    for key in kwargs:
        # If there is a "mesh" kwarg it should not be added to the parameters
        if key != "mesh":
            try:
                p.add(key, kwargs[key])
            except TypeError:
                cpp.warning("Incompatible type for keyword argument \"%s\". Ignoring." % key)

    # Plot element
    if isinstance(object, ufl.FiniteElementBase):
        if os.environ.get("DOLFIN_NOPLOT", "0") != "0": return
        import ffc
        return ffc.plot(object, *args, **kwargs)

    if mesh is None and len(args) == 1 and isinstance(args[0], cpp.Mesh):
        mesh = args[0]

    # Plot expression
    if isinstance(object, cpp.Expression):
        if mesh is None:
            raise TypeError, "expected a mesh when plotting an expression."
        return cpp.plot(object, mesh, p)

    # Try to project if object is not a standard plottable type
    if not isinstance(object, (cpp.Function, cpp.Expression, cpp.Mesh,
        cpp.DirichletBC, cpp.MeshFunction, cpp.MeshFunctionBool,
        cpp.MeshFunctionInt, cpp.MeshFunctionDouble,
        cpp.MeshFunctionSizet, cpp.DirichletBC, cpp.CSGGeometry)):

        from dolfin.fem.projection import project
        try:
            cpp.info("Object cannot be plotted directly, projecting to"\
                    " piecewise linears.")
            object = project(object, mesh=mesh)
        except Exception as e:
            msg = ("Don't know how to plot given object:\n  %s\n"\
                    "and projection failed:\n  %s") % (str(object), str(e))
            #raise RuntimeError(msg)
            raise

    plot_object = cpp.plot(object, p)
    plot_object.write_ps = _VTKPlotter_write_ps

    # Avoid premature deletion of plotted objects if they go out of scope
    # before the plot window is closed. The plotter itself is safe, since it's
    # created in the plot() C++ function, not directly from Python. But the
    # Python plotter proxy may disappear, so we can't store the references
    # there.
    global _objects_referenced_from_plot_windows
    _objects_referenced_from_plot_windows[plot_object.key()] = (object, mesh, p)

    return plot_object

_objects_referenced_from_plot_windows = {}
