#!/usr/bin/perl

# Copyright (C) 2008-2010 Sun Microsystems, Inc. All rights reserved.
# Use is subject to license terms.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301
# USA

use lib 'lib';
use lib "$ENV{RQG_HOME}/lib";
use strict;
use Carp;
use Getopt::Long;

use GenTest;
use GenTest::Properties;
use GenTest::Constants;
use GenTest::App::GenTest;

my $DEFAULT_THREADS = 10;
my $DEFAULT_QUERIES = 1000;
my $DEFAULT_DURATION = 3600;
my $DEFAULT_DSN = 'dbi:mysql:host=localhost:port=9306:user=root:database=test';

my @ARGV_saved = @ARGV;

my $options = {};
my $opt_result = GetOptions($options,
                            'config=s',
                            'dsn=s@',
                            'dsn1=s',
                            'dsn2=s',
                            'dsn3=s',
                            'engine=s',
                            'generator=s',
                            'gendata:s',
                            'grammar=s',
                            'redefine=s',
                            'testname=s',
                            'threads=i',
                            'queries=s',
                            'duration=s',
                            'help',
                            'debug',
                            'rpl_mode=s',
                            'validators:s@',
                            'reporters:s@',
                            'transformers:s@',
                            'report-xml-tt',
                            'report-xml-tt-type=s',
                            'report-xml-tt-dest=s',
                            'seed=s',
                            'mask=i',
                            'mask-level=i',
                            'rows=i',
                            'varchar-length=i',
                            'xml-output=s',
                            'sqltrace',
                            'no-err-filter',
                            'views',
                            'start-dirty',
                            'filter=s',
                            'valgrind',
                            'valgrind-xml',
                            'notnull',
                            'debug',
                            'querytimeout=i');
backwardCompatability($options);
my $config = GenTest::Properties->new(
    options => $options,
    defaults => {dsn=>[$DEFAULT_DSN],
                 seed => 1,
                 queries => $DEFAULT_QUERIES,
                 duration => $DEFAULT_DURATION,
                 threads => $DEFAULT_THREADS},
    legal => ['dsn',
              'engine',
              'gendata',
              'generator',
              'grammar',
              'redefine',
              'testname',
              'threads',
              'queries',
              'duration',
              'help',
              'debug',
              'rpl_mode',
              'validators',
              'reporters',
              'transformers',
              'report-xml-tt',
              'report-xml-tt-type',
              'report-xml-tt-dest',
              'seed',
              'mask',
              'mask-level',
              'rows',
              'varchar-length',
              'xml-output',
              'views',
              'sqltrace',
              'no-err-filter',
              'start-dirty',
              'filter',
              'valgrind',
              'valgrind-xml',
              'sqltrace',
              'notnull',
              'querytimeout'],
    help => \&help);

help() if !$opt_result || $config->help;

say("Starting: $0 ".join(" ", @ARGV_saved));

$ENV{RQG_DEBUG} = 1 if defined $config->debug;
my $gentest = GenTest::App::GenTest->new(config => $config);

my $status = $gentest->run();

safe_exit($status);

sub help {

    print <<EOF
$0 - Testing via random query generation. Options:

        --dsn      : DBI resources to connect to (default $DEFAULT_DSN).
                      Supported databases are MySQL, Drizzle, PostgreSQL, JavaDB
                      first --dsn must be to MySQL or Drizzle
        --gendata   : Execute gendata-old.pl in order to populate tables with simple data (default NO)
        --gendata=s : Execute gendata.pl in order to populate tables with data 
                      using the argument as specification file to gendata.pl
        --engine    : Table engine to use when creating tables with gendata (default: no ENGINE for CREATE TABLE)
        --threads   : Number of threads to spawn (default $DEFAULT_THREADS)
        --queries   : Numer of queries to execute per thread (default $DEFAULT_QUERIES);
        --duration  : Duration of the test in seconds (default $DEFAULT_DURATION seconds);
        --grammar   : Grammar file to use for generating the queries (REQUIRED);
        --redefine  : Grammar file to redefine and/or add rules to the given grammar
        --seed      : PRNG seed (default 1). If --seed=time, the current time will be used.
        --rpl_mode  : Replication mode
        --validator : Validator classes to be used. Defaults
                           ErrorMessageCorruption if one or two MySQL dsns
                           ResultsetComparator3 if 3 dsns
                           ResultsetComparartor if 2 dsns
        --reporter  : ErrorLog, Backtrace if one or two MySQL dsns
        --mask      : A seed to a random mask used to mask (reduce) the grammar.
        --mask-level: How many levels deep the mask is applied (default 1)
        --rows      : Number of rows to generate for each table in gendata.pl, unless specified in the ZZ file
        --varchar-length: maximum length of strings (deault 1) in gendata.pl
        --views     : Pass --views to gendata-old.pl or gendata.pl
        --filter    : ......
        --sqltrace  : Print all generated SQL statements.
        --no-err-filter:  Do not suppress error messages.  Output all error messages encountered.
        --start-dirty: Do not generate data (use existing database(s))
        --xml-output: Name of a file to which an XML report will be written if this option is set.
        --report-xml-tt: Report test results in XML-format to the Test Tool (TT) reporting framework.
        --report-xml-tt-type: Type of TT XML transport to use (e.g. scp)
        --report-xml-tt-dest: Destination of TT XML report (e.g. user\@host:/path/to/location (for type scp))
        --testname  : Name of test, used for reporting purposes.
        --valgrind  : ......
        --filter    : ......
        --help      : This help message
        --debug     : Provide debug output
EOF
	;
	safe_exit(1);
}

sub backwardCompatability {
    my ($options) = @_;
    if (defined $options->{dsn}) {
        croak ("Do not combine --dsn and --dsnX") 
            if defined $options->{dsn1} or
            defined $options->{dsn2} or
            defined $options->{dsn3};
        
    } else {
        my @dsns;
        foreach my $i (1..3) {
            if (defined $options->{'dsn'.$i}) {
                push @dsns, $options->{'dsn'.$i};
                delete $options->{'dsn'.$i};
            }
        }
        $options->{dsn} = \@dsns;
    }
        
    if (grep (/,/,@{$options->{reporters}})) {
        my $newreporters = [];
        map {push(@$newreporters,split(/,/,$_))} @{$options->{reporters}};
        $options->{reporters}=$newreporters ;
    }

    if (grep (/,/,@{$options->{transformers}})) {
        my $newtransformers = [];
        map {push(@$newtransformers,split(/,/,$_))} @{$options->{transformers}};
        $options->{transformers}=$newtransformers ;
    }

    if (grep (/,/,@{$options->{validators}})) {
        my $newvalidators = [];
        map {push(@$newvalidators,split(/,/,$_))} @{$options->{validators}};
        $options->{validators}=$newvalidators ;
    }

    if (not defined $options->{generator}) {
        $options->{generator} = 'FromGrammar';
    }
}

