// -*- tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
// vi: set et ts=4 sw=2 sts=2:
/****************************************************************************/
/*                                                                          */
/* File:      rm.c                                                          */
/*                                                                          */
/* Purpose:   rule manager for 2D and 3D refinement rules                   */
/*                                                                          */
/* Author:    Stefan Lang                                                   */
/*            Institut fuer Computeranwendungen III                         */
/*            Universitaet Stuttgart                                        */
/*            Pfaffenwaldring 27                                            */
/*            70550 Stuttgart                                               */
/*                                                                          */
/* History:   21.11.95 begin, ugp version 3.0                               */
/*                                                                          */
/* Remarks:                                                                 */
/*                                                                          */
/****************************************************************************/

/****************************************************************************/
/*                                                                          */
/* include files                                                            */
/* system include files                                                     */
/* application include files                                                */
/*                                                                          */
/****************************************************************************/

#include <config.h>

/* standard C library */
#include <cassert>
#include <cstdio>
#include <cmath>
#include <cstdlib>

/* low module */
#include "architecture.h"
#include "debug.h"
#include "fileopen.h"
#include "general.h"
#include "misc.h"

/* dev module */
#include <dev/ugdevices.h>

/* gm module */
#include "evm.h"
#include "gm.h"
#include "refine.h"
#include "shapes.h"
#include "rm.h"
#include "cw.h"
#include "elements.h"

#include "defaults.h"
#ifdef ModelP
#include "parallel.h"
using namespace PPIF;
#endif

USING_UG_NAMESPACES

/****************************************************************************/
/*																			*/
/* defines in the following order											*/
/*																			*/
/*		  compile time constants defining static data size (i.e. arrays)	*/
/*		  other constants													*/
/*		  macros															*/
/*																			*/
/****************************************************************************/

/* macros defining best refrule, specify exactly one of them !! */
/*#define __SHORTEST_INTERIOR_EDGE__*/
/*#define __MIDDLE_INTERIOR_EDGE__*/
#define __LONGEST_INTERIOR_EDGE__

#define NOINDEX         -1

/* rule count for element types */
#ifdef __TWODIM__
#define MAX_TRI_RULES   18
#define MAX_QUA_RULES   17
#else
#ifndef TET_RULESET
#define MAX_TET_RULES   6
#endif
#define MAX_PYR_RULES   5
#define MAX_PRI_RULES   15
#define MAX_HEX_RULES   13
#endif

/* shorthand notation */
#define FO                              FATHER_SIDE_OFFSET

/****************************************************************************/
/*																			*/
/* data structures used in this source file (exported data structures are	*/
/*		  in the corresponding include file!)								*/
/*																			*/
/****************************************************************************/

/****************************************************************************/
/*																			*/
/* definition of exported global variables									*/
/*																			*/
/****************************************************************************/

INT NS_DIM_PREFIX MaxRules[TAGS] = {0,0,0,0,0,0,0,0};
INT NS_DIM_PREFIX MaxNewCorners[TAGS] = {0,0,0,0,0,0,0,0};
INT NS_DIM_PREFIX MaxNewEdges[TAGS] = {0,0,0,0,0,0,0,0};
INT NS_DIM_PREFIX CenterNodeIndex[TAGS] = {0,0,0,0,0,0,0,0};
REFRULE * NS_DIM_PREFIX RefRules[TAGS] = {NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL};
std::unique_ptr<SHORT[]> NS_DIM_PREFIX Pattern2Rule[TAGS];

#ifdef __THREEDIM__
/* define the standard regular rules for tetrahedrons */
FULLREFRULEPTR NS_DIM_PREFIX theFullRefRule;
static ElementVectorProcPtr theDirectionElemEval;
#endif


/****************************************************************************/
/*                                                                          */
/* definition of variables global to this source file only (static!)        */
/*                                                                          */
/****************************************************************************/

#ifdef __TWODIM__
static REFRULE Empty_Rule =
{-1,-1,NO_CLASS,-1,{-1,-1,-1,-1},-1,
 {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},
 {{-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}};

/* define Rules for Triangles */
static REFRULE TriangleRules[MAX_TRI_RULES] = {
  /* T_NOREF */
  {TRIANGLE,T_NOREF,NO_CLASS,0,
   {0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},
   {{-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_COPY */
  {TRIANGLE,T_COPY,RED_CLASS|GREEN_CLASS,1,
   {0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,1,2,-1},{FO+0,FO+1,FO+2,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_RED */
  {TRIANGLE,T_RED,RED_CLASS|GREEN_CLASS|SWITCH_CLASS,4,
   {1,1,1,0},(1<<3)-1,
   {{0,1},{1,2},{0,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,5,-1},{FO+0, 3,FO+2,-1},0},
                             {TRIANGLE,{3,1,4,-1},{FO+0,FO+1, 3,-1},0},
                             {TRIANGLE,{4,2,5,-1},{ FO+1,FO+2,3,-1},0},
                             {TRIANGLE,{3,4,5,-1},{ 1, 2, 0,-1},0}}},


  /* T_BISECT_1 edge 0 bisected */
  {TRIANGLE,T_BISECT_1_0,RED_CLASS|GREEN_CLASS,2,
   {1,0,0,0},1,
   {{0,1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,2,-1},{FO+0, 1,FO+2,-1},0},
                             {TRIANGLE,{3,1,2,-1},{FO+0,FO+1, 0,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_1 edge 1 bisected */
  {TRIANGLE,T_BISECT_1_1,RED_CLASS|GREEN_CLASS,2,
   {0,1,0,0},1<<1,
   {{-1,-1},{0,2},{-1,-1},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,1,4,-1},{FO+0,FO+1, 1,-1},0},
                             {TRIANGLE,{0,4,2,-1},{ 0,FO+1,FO+2,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_1 edge 2 bisected */
  {TRIANGLE,T_BISECT_1_2,RED_CLASS|GREEN_CLASS,2,
   {0,0,1,0},1<<2,
   {{-1,-1},{-1,-1},{0,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,1,5,-1},{FO+0, 1,FO+2,-1},0},
                             {TRIANGLE,{5,1,2,-1},{ 0,FO+1,FO+2,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},


  /* T_BISECT_2_T1 edge 2 not bisected */
  {TRIANGLE,T_BISECT_2_T1_2,RED_CLASS|GREEN_CLASS,3,
   {1,1,0,0},(1<<2)-1,
   {{0,1},{1,2},{-1,-1},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,2,-1},{FO+0, 2,FO+2,-1},0},
                             {TRIANGLE,{3,1,4,-1},{FO+0,FO+1, 2,-1},0},
                             {TRIANGLE,{3,4,2,-1},{ 1,FO+1, 0,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_2_T1 edge 0 not bisected */
  {TRIANGLE,T_BISECT_2_T1_0,RED_CLASS|GREEN_CLASS,3,
   {0,1,1,0},6,
   {{-1,-1},{0,2},{1,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,1,4,-1},{FO+0,FO+1, 1,-1},0},
                             {TRIANGLE,{0,4,5,-1},{ 0, 2,FO+2,-1},0},
                             {TRIANGLE,{5,4,2,-1},{ 1,FO+1,FO+2,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_2_T1 edge 1 not bisected */
  {TRIANGLE,T_BISECT_2_T1_1,RED_CLASS|GREEN_CLASS,3,
   {1,0,1,0},5,
   {{0,1},{-1,-1},{1,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,5,-1},{FO+0, 1,FO+2,-1},0},
                             {TRIANGLE,{3,1,5,-1},{FO+0, 2, 0,-1},0},
                             {TRIANGLE,{5,1,2,-1},{ 1,FO+1,FO+2,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_2_T2 edge 1 not bisected */
  {TRIANGLE,T_BISECT_2_T2_1,RED_CLASS|GREEN_CLASS,3,
   {1,0,1,0},5,
   {{0,1},{-1,-1},{0,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,5,-1},{FO+0, 1,FO+2,-1},0},
                             {TRIANGLE,{5,3,2,-1},{ 0, 2,FO+2,-1},0},
                             {TRIANGLE,{3,1,2,-1},{FO+0,FO+1, 1,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_2_T2 edge 2 not bisected */
  {TRIANGLE,T_BISECT_2_T2_2,RED_CLASS|GREEN_CLASS,3,
   {1,1,0,0},3,
   {{0,1},{0,2},{-1,-1},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,4,-1},{FO+0, 1, 2,-1},0},
                             {TRIANGLE,{3,1,4,-1},{FO+0,FO+1, 0,-1},0},
                             {TRIANGLE,{0,4,2,-1},{ 0,FO+1,FO+2,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_2_T2 edge 0 not bisected */
  {TRIANGLE,T_BISECT_2_T2_0,RED_CLASS|GREEN_CLASS,3,
   {0,1,1,0},6,
   {{-1,-1},{1,2},{0,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,1,5,-1},{FO+0, 1,FO+2,-1},0},
                             {TRIANGLE,{5,1,4,-1},{ 0,FO+1, 2,-1},0},
                             {TRIANGLE,{5,4,2,-1},{ 1,FO+1,FO+2,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_2_Q edge 0 not bisected */
  {TRIANGLE,T_BISECT_2_Q_0,RED_CLASS|GREEN_CLASS,2,
   {0,1,1,0},6,
   {{-1,-1},{0,2},{0,3},{-1,-1},{-1,-1}},
   {{QUADRILATERAL,{0,1,4,5},{FO+0,FO+1, 1,FO+2},0},
                             {TRIANGLE,{5,4,2,-1},{ 0,FO+1,FO+2,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_2_Q edge 1 not bisected */
  {TRIANGLE,T_BISECT_2_Q_1,RED_CLASS|GREEN_CLASS,2,
   {1,0,1,0},5,
   {{0,1},{-1,-1},{0,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,5,-1},{FO+0, 1,FO+2,-1},0},
                             {QUADRILATERAL,{3,1,2,5},{FO+0,FO+1,FO+2, 0},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_2_Q edge 2 not bisected */
  {TRIANGLE,T_BISECT_2_Q_2,RED_CLASS|GREEN_CLASS,2,
   {1,1,0,0},3,
   {{0,1},{0,2},{-1,-1},{-1,-1},{-1,-1}},
   {{QUADRILATERAL,{0,3,4,2},{FO+0, 1,FO+1,FO+2},0},
                             {TRIANGLE,{3,1,4,-1},{FO+0,FO+1, 0,-1},0},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* T_BISECT_3 edge 0 */
  {TRIANGLE,T_BISECT_3_0,RED_CLASS|GREEN_CLASS,4,
   {1,1,1,0},7,
   {{0,1},{3,2},{0,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,5,-1},{FO+0, 1,FO+2,-1},0},
                             {TRIANGLE,{5,3,2,-1},{ 0, 2,FO+2,-1},0},
                             {TRIANGLE,{3,4,2,-1},{ 3,FO+1, 1,-1},0},
                             {TRIANGLE,{3,1,4,-1},{FO+0,FO+1, 2,-1},0}}},

  /* T_BISECT_3 edge 1 */
  {TRIANGLE,T_BISECT_3_1,RED_CLASS|GREEN_CLASS,4,
   {1,1,1,0},7,
   {{0,1},{0,2},{1,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,4,-1},{FO+0, 3, 1,-1},0},
                             {TRIANGLE,{0,4,5,-1},{ 0, 2,FO+2,-1},0},
                             {TRIANGLE,{5,4,2,-1},{ 1,FO+1,FO+2,-1},0},
                             {TRIANGLE,{3,1,4,-1},{FO+0,FO+1, 0,-1},0}}},

  /* T_BISECT_3 edge 2 */
  {TRIANGLE,T_BISECT_3_2,RED_CLASS|GREEN_CLASS,4,
   {1,1,1,0},7,
   {{0,1},{2,2},{1,2},{-1,-1},{-1,-1}},
   {{TRIANGLE,{0,3,5,-1},{FO+0, 1,FO+2,-1},0},
                             {TRIANGLE,{3,1,5,-1},{FO+0, 2, 0,-1},0},
                             {TRIANGLE,{5,1,4,-1},{ 1,FO+1, 3,-1},0},
                             {TRIANGLE,{5,4,2,-1},{ 2,FO+1,FO+2,-1},0}}},

};

/* define Rules for Quadrilaterals */
static REFRULE QuadrilateralRules[MAX_QUA_RULES] =
{
  /* Q_NOREF */
  {QUADRILATERAL,Q_NOREF,NO_CLASS,0,
   {0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},
   {{-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_COPY */
  {QUADRILATERAL,Q_COPY,RED_CLASS|GREEN_CLASS,1,
   {0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},
   {{QUADRILATERAL,{0,1,2,3},{FO+0,FO+1,FO+2,FO+3},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_RED */
  {QUADRILATERAL,Q_RED,RED_CLASS|GREEN_CLASS|SWITCH_CLASS,4,
   {1,1,1,1,1},(1<<5)-1,
   {{0,1},{1,2},{2,3},{3,0},{0,2}},
   {{QUADRILATERAL,{0,4,8,7},{FO+0, 1, 3,FO+3},-1},
                             {QUADRILATERAL,{4,1,5,8},{FO+0,FO+1, 2, 0},-1},
                             {QUADRILATERAL,{8,5,2,6},{ 1,FO+1,FO+2, 3},-1},
                             {QUADRILATERAL,{7,8,6,3},{ 0, 2,FO+2,FO+3},-1}}},

  /* Q_CLOSE_1 edge 0 and 1 bisected */
  {QUADRILATERAL,Q_CLOSE_1_0,RED_CLASS|GREEN_CLASS,3,
   {1,1,0,0,1},3+16,
   {{0,1},{1,2},{-1,-1},{-1,-1},{0,2}},
   {{QUADRILATERAL,{0,4,8,3},{FO+0, 1, 2,FO+3},-1},
                             {QUADRILATERAL,{4,1,5,8},{FO+0,FO+1, 2, 0},-1},
                             {QUADRILATERAL,{8,5,2,3},{ 1,FO+1,FO+2, 0},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_CLOSE_1 edge 1 and 2 bisected */
  {QUADRILATERAL,Q_CLOSE_1_1,RED_CLASS|GREEN_CLASS,3,
   {0,1,1,0,1},6+16,
   {{-1,-1},{0,2},{1,3},{-1,-1},{0,3}},
   {{QUADRILATERAL,{0,1,5,8},{FO+0,FO+1, 1, 2},-1},
                             {QUADRILATERAL,{8,5,2,6},{ 0,FO+1,FO+2, 2},-1},
                             {QUADRILATERAL,{0,8,6,3},{ 0, 1,FO+2,FO+3},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_CLOSE_1 edge 2 and 3 bisected */
  {QUADRILATERAL,Q_CLOSE_1_2,RED_CLASS|GREEN_CLASS,3,
   {0,0,1,1,1},12+16,
   {{-1,-1},{-1,-1},{1,3},{0,3},{0,2}},
   {{QUADRILATERAL,{0,1,8,7},{FO+0, 1, 2,FO+3},-1},
                             {QUADRILATERAL,{8,1,2,6},{ 0,FO+1,FO+2, 2},-1},
                             {QUADRILATERAL,{7,8,6,3},{ 0, 1,FO+2,FO+3},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_CLOSE_1 edge 0 and 3 bisected */
  {QUADRILATERAL,Q_CLOSE_1_3,RED_CLASS|GREEN_CLASS,3,
   {1,0,0,1,1},9+16,
   {{0,1},{-1,-1},{-1,-1},{0,3},{0,2}},
   {{QUADRILATERAL,{0,4,8,7},{FO+0, 1, 2,FO+3},-1},
                             {QUADRILATERAL,{4,1,2,8},{FO+0,FO+1, 2, 0},-1},
                             {QUADRILATERAL,{7,8,2,3},{ 0, 1,FO+2,FO+3},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_BLUE edge 0 and 2 bisected */
  {QUADRILATERAL,Q_BLUE_0,RED_CLASS|GREEN_CLASS,2,
   {1,0,1,0,0},5,
   {{0,1},{-1,-1},{0,2},{-1,-1},{-1,-1}},
   {{QUADRILATERAL,{0,4,6,3},{FO+0, 1,FO+2,FO+3},-1},
                             {QUADRILATERAL,{4,1,2,6},{FO+0,FO+1,FO+2, 0},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_BLUE edge 1 and 3 bisected */
  {QUADRILATERAL,Q_BLUE_1,RED_CLASS|GREEN_CLASS,2,
   {0,1,0,1,0},10,
   {{-1,-1},{0,2},{-1,-1},{0,3},{-1,-1}},
   {{QUADRILATERAL,{0,1,5,7},{FO+0,FO+1, 1,FO+3},-1},
                             {QUADRILATERAL,{7,5,2,3},{ 0,FO+1,FO+2,FO+3},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_CLOSE_2 edge 0 bisected */
  {QUADRILATERAL,Q_CLOSE_2_0,RED_CLASS|GREEN_CLASS,3,
   {1,0,0,0,1},1+16,
   {{0,1},{-1,-1},{-1,-1},{-1,-1},{0,2}},
   {{QUADRILATERAL,{0,4,8,3},{FO+0, 1, 2,FO+3},-1},
                             {QUADRILATERAL,{4,1,2,8},{FO+0,FO+1, 2, 0},-1},
                             {TRIANGLE,{8,2,3,-1},{ 1,FO+2, 0,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_CLOSE_2 edge 1 bisected */
  {QUADRILATERAL,Q_CLOSE_2_1,RED_CLASS|GREEN_CLASS,3,
   {0,1,0,0,1},2+16,
   {{-1,-1},{0,2},{-1,-1},{-1,-1},{0,3}},
   {{QUADRILATERAL,{0,1,5,8},{FO+0,FO+1, 1, 2},-1},
                             {QUADRILATERAL,{8,5,2,3},{ 0,FO+1,FO+2, 2},-1},
                             {TRIANGLE,{0,8,3,-1},{ 0, 1,FO+3,-1},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_CLOSE_2 edge 2 bisected */
  {QUADRILATERAL,Q_CLOSE_2_2,RED_CLASS|GREEN_CLASS,3,
   {0,0,1,0,1},4+16,
   {{-1,-1},{-1,-1},{1,3},{-1,-1},{0,2}},
   {{TRIANGLE,{0,1,8,-1},{FO+0, 1, 2,-1},-1},
                             {QUADRILATERAL,{8,1,2,6},{ 0,FO+1,FO+2, 2},-1},
                             {QUADRILATERAL,{0,8,6,3},{ 0, 1,FO+2,FO+3},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_CLOSE_2 edge 3 bisected */
  {QUADRILATERAL,Q_CLOSE_2_3,RED_CLASS|GREEN_CLASS,3,
   {0,0,0,1,1},8+16,
   {{-1,-1},{-1,-1},{-1,-1},{0,3},{0,2}},
   {{QUADRILATERAL,{0,1,8,7},{FO+0, 1, 2,FO+3},-1},
                             {TRIANGLE,{1,2,8,-1},{FO+1, 2, 0,-1},-1},
                             {QUADRILATERAL,{7,8,2,3},{ 0, 1,FO+2,FO+3},-1},
                             {-1,{-1,-1,-1,-1},{-1,-1,-1,-1},-1}}},

  /* Q_CLOSE_3 edge 0 not bisected */
  {QUADRILATERAL,Q_CLOSE_3_0,RED_CLASS|GREEN_CLASS,4,
   {0,1,1,1,0},14,
   {{-1,-1},{0,2},{1,2},{0,3},{-1,-1}},
   {{QUADRILATERAL,{0,1,5,7},{FO+0,FO+1, 1,FO+3},-1},
                             {TRIANGLE,{7,5,6,-1},{ 0, 2, 3,-1},-1},
                             {TRIANGLE,{5,2,6,-1},{FO+1,FO+2, 1,-1},-1},
                             {TRIANGLE,{7,6,3,-1},{ 1,FO+2,FO+3,-1},-1}}},

  /* Q_CLOSE_3 edge 1 not bisected */
  {QUADRILATERAL,Q_CLOSE_3_1,RED_CLASS|GREEN_CLASS,4,
   {1,0,1,1,0},13,
   {{0,1},{-1,-1},{1,2},{0,2},{-1,-1}},
   {{TRIANGLE,{0,4,7,-1},{FO+0, 1,FO+3,-1},-1},
                             {TRIANGLE,{7,4,6,-1},{ 0, 3, 2,-1},-1},
                             {TRIANGLE,{7,6,3,-1},{ 1,FO+2,FO+3,-1},-1},
                             {QUADRILATERAL,{4,1,2,6},{FO+0,FO+1,FO+2, 1},-1}}},

  /* Q_CLOSE_3 edge 2 not bisected */
  {QUADRILATERAL,Q_CLOSE_3_2,RED_CLASS|GREEN_CLASS,4,
   {1,1,0,1,0},11,
   {{0,1},{2,2},{-1,-1},{0,2},{-1,-1}},
   {{TRIANGLE,{0,4,7,-1},{FO+0, 1,FO+3,-1},-1},
                             {TRIANGLE,{4,5,7,-1},{ 2, 3, 0,-1},-1},
                             {TRIANGLE,{4,1,5,-1},{FO+0,FO+1, 1,-1},-1},
                             {QUADRILATERAL,{7,5,2,3},{ 1,FO+1,FO+2,FO+3},-1}}},

  /* Q_CLOSE_3 edge 3 not bisected */
  {QUADRILATERAL,Q_CLOSE_3_3,RED_CLASS|GREEN_CLASS,4,
   {1,1,1,0,0},7,
   {{0,1},{2,2},{0,2},{-1,-1},{-1,-1}},
   {{QUADRILATERAL,{0,4,6,3},{FO+0, 1,FO+2,FO+3},-1},
                             {TRIANGLE,{4,5,6,-1},{ 2, 3, 0,-1},-1},
                             {TRIANGLE,{4,1,5,-1},{FO+0,FO+1, 1,-1},-1},
                             {TRIANGLE,{5,2,6,-1},{FO+1,FO+2, 1,-1},-1}}}

};

#endif

#ifdef __THREEDIM__

static INT theBFRRDirID;      /* env type for BestFullRefRule       */
static INT theBFRRVarID;

static REFRULE Empty_Rule =
{-1,-1,NO_CLASS,-1,
 {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1},-1,
 {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                        {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                        {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                        {-1,-1}},
 {{-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                        {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}};

#ifndef TET_RULESET
/* define the regular rules for tetrahedron */
static REFRULE TetrahedronRules[MAX_TET_RULES] =
{
  /* TET_NO_REF */
  {TETRAHEDRON,0,NO_CLASS,0,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* TET_COPY */
  {TETRAHEDRON,1,YELLOW_CLASS,1,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{TETRAHEDRON,{ 0, 1, 2, 3,-1,-1,-1,-1},{FO+0,FO+1,FO+2,FO+3,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* TET_RED equals TET_RED_2_4 */
  {TETRAHEDRON,2,RED_CLASS|SWITCH_CLASS,8,
   {1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0},(1<<6)-1,

   /* sonandnode */
   {{0,1},{1,1},{0,2},{0,3},{1,2},{2,2},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{TETRAHEDRON,{0,4,6,7,-1,-1,-1,-1},{FO+0,4,FO+2,FO+3,-1,-1},0x0},
                    {TETRAHEDRON,{4,5,8,1,-1,-1,-1,-1},{5,FO+1,FO+3,FO+0,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x1 | (0x3<<3) | (0x3<<(2*3))},
                    {TETRAHEDRON,{5,6,9,2,-1,-1,-1,-1},{6,FO+2,FO+1,FO+0,-1,-1},
                          0x4<<PATHDEPTHSHIFT | 0x1 | (0x3<<3) | (0x1<<(2*3)) | (0x3<<(3*3))},
                    {TETRAHEDRON,{7,8,9,3,-1,-1,-1,-1},{7,FO+1,FO+2,FO+3,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{4,6,7,8,-1,-1,-1,-1},{0,7,FO+3,5,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{4,5,6,8,-1,-1,-1,-1},{FO+0,6,4,1,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{5,6,8,9,-1,-1,-1,-1},{5,7,FO+1,2,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{6,7,8,9,-1,-1,-1,-1},{4,3,6,FO+2,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* TET_RED_0_5 */
  {TETRAHEDRON,3,RED_CLASS|SWITCH_CLASS,8,
   {1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0},(1<<6)-1,

   /* sonandnode */
   {{1,0},{0,0},{0,1},{2,0},{1,2},{0,2},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{TETRAHEDRON,{5,6,9,2,-1,-1,-1,-1},{4,FO+2,FO+1,FO+0,-1,-1},0x0},
                    {TETRAHEDRON,{4,5,8,1,-1,-1,-1,-1},{5,FO+1,FO+3,FO+0,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x1 | (0x3<<3) | (0x3<<(2*3))},
                    {TETRAHEDRON,{7,8,9,3,-1,-1,-1,-1},{6,FO+1,FO+2,FO+3,-1,-1},
                          0x4<<PATHDEPTHSHIFT | 0x1 | (0x3<<3) | (0x1<<(2*3)) | (0x3<<(3*3))},
                    {TETRAHEDRON,{0,4,6,7,-1,-1,-1,-1},{FO+0,7,FO+2,FO+3,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{4,5,6,9,-1,-1,-1,-1},{FO+0,0,7,5,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{5,8,9,4,-1,-1,-1,-1},{FO+1,6,4,1,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{4,7,8,9,-1,-1,-1,-1},{FO+3,2,5,7,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{4,6,7,9,-1,-1,-1,-1},{3,FO+2,6,4,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* TET_RED_1_3 */
  {TETRAHEDRON,4,RED_CLASS|SWITCH_CLASS,8,
   {1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0},(1<<6)-1,

   /* sonandnode */
   {{1,1},{1,1},{2,2},{0,0},{0,1},{0,2},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{TETRAHEDRON,{7,8,9,3,-1,-1,-1,-1},{4,FO+1,FO+2,FO+3,-1,-1},0x0},
                    {TETRAHEDRON,{4,5,8,1,-1,-1,-1,-1},{5,FO+1,FO+3,FO+0,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x1 | (0x3<<3) | (0x3<<(2*3))},
                    {TETRAHEDRON,{0,4,6,7,-1,-1,-1,-1},{FO+0,6,FO+2,FO+3,-1,-1},
                          0x4<<PATHDEPTHSHIFT | 0x1 | (0x3<<3) | (0x1<<(2*3)) | (0x3<<(3*3))},
                    {TETRAHEDRON,{5,6,9,2,-1,-1,-1,-1},{7,FO+2,FO+1,FO+0,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{5,7,8,9,-1,-1,-1,-1},{5,0,FO+1,7,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{4,5,7,8,-1,-1,-1,-1},{6,4,FO+3,1,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{4,5,6,7,-1,-1,-1,-1},{FO+0,7,2,5,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{5,6,7,9,-1,-1,-1,-1},{6,FO+2,4,3,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* TET_RED_HEX */
  {TETRAHEDRON,5,RED_CLASS|SWITCH_CLASS,4,
   {1,1,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0},(1<<11)-1,

   /* sonandnode */
   {{0,2},{1,3},{0,0},{0,5},{3,6},{3,4},
                    {0,3},{0,6},{3,7},{0,4},{0,7},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{HEXAHEDRON,{4,1,5,10,13,8,11,14},{FO+0,FO+3,FO+1,1,2,3},0x0},
                    {HEXAHEDRON,{5,2,6,10,11,9,12,14},{FO+0,FO+1,FO+2,2,0,3},0x0},
                    {HEXAHEDRON,{0,4,10,6,7,13,14,12},{FO+0,FO+3,0,1,FO+2,3},0x0},
                    {HEXAHEDRON,{13,8,11,14,7,3,9,12},{0,FO+3,FO+1,1,2,FO+2},0x0},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}}

};
#endif

/* define the regular rules for pyramids */
static REFRULE PyramidRules[MAX_PYR_RULES] =
{
  /* PYR_NO_REF */
  {PYRAMID,0,NO_CLASS,0,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PYR_COPY */
  {PYRAMID,1,YELLOW_CLASS,1,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{PYRAMID,{ 0, 1, 2, 3, 4,-1,-1,-1},{FO+0,FO+1,FO+2,FO+3,FO+4,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PYR_RED */
  {PYRAMID,2,RED_CLASS|SWITCH_CLASS,10,
   {1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0,0,0},(1<<9)-1,

   /* sonandnode */
   {{0,1},{1,2},{2,3},{0,3},{0,4},{1,4},
                    {2,4},{3,4},{0,2},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PYRAMID,{ 0, 5, 13, 8, 9,-1,-1,-1},{FO+0,FO+1,4,7,FO+4,-1},-1},
                    {PYRAMID,{5,1,6,13,10,-1,-1,-1},{FO+0,FO+1,FO+2,5,4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x1<<3)},
                    {PYRAMID,{13,6,2,7,11,-1,-1,-1},{FO+0,5,FO+2,FO+3,6,-1},
                          0x4<<PATHDEPTHSHIFT | 0x2 | (0x1<<3) | (0x3<<(2*3)) | (0x1<<(3*3))},
                    {PYRAMID,{8,13,7,3,12,-1,-1,-1},{FO+0,7,6,FO+3,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x3 | (0x2<<3)},
                    {TETRAHEDRON,{9,10,5,13,-1,-1,-1,-1},{FO+1,1,0,8,-1,-1},
                          0x1<<PATHDEPTHSHIFT | 0x2},
                    {TETRAHEDRON,{10,11,6,13,-1,-1,-1,-1},{FO+2,2,1,8,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x2 | (0x1<<3) | (0x3<<(2*3))},
                    {TETRAHEDRON,{11,12,7,13,-1,-1,-1,-1},{FO+3,3,2,8,-1,-1},
                          0x3<<PATHDEPTHSHIFT | 0x3 | (0x2<<3) | (0x2<<(2*3))},
                    {TETRAHEDRON,{12,9,8,13,-1,-1,-1,-1},{FO+4,0,3,8,-1,-1},
                          0x1<<PATHDEPTHSHIFT | 0x3},
                    {PYRAMID,{12,11,10,9,13,-1,-1,-1},{9,6,5,4,7,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x3<<3)},
                    {PYRAMID,{9,10,11,12,4,-1,-1,-1},{8,FO+1,FO+2,FO+3,FO+4,-1},
                          0x3<<PATHDEPTHSHIFT | 0x2 | (0x3<<3) | (0x0<<(2*3))},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PYR_bisect_0_1 */
  {PYRAMID,3,RED_CLASS|SWITCH_CLASS,2,
   {1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},5,

   /* sonandnode */
   {{0,1},{1,2},{2,3},{0,3},{0,4},{1,4},
                    {2,4},{3,4},{0,2},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PYRAMID,{ 0, 5, 7, 3, 4,-1,-1,-1},{FO+0,FO+1,1,FO+3,FO+4,-1},-1},
                    {PYRAMID,{5,1,2,7,4,-1,-1,-1},{FO+0,FO+1,FO+2,FO+3,0,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x1<<3)},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PYR_bisect_0_2 */
  {PYRAMID,4,RED_CLASS|SWITCH_CLASS,2,
   {0,1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},10,

   /* sonandnode */
   {{0,1},{1,2},{2,3},{0,3},{0,4},{1,4},
                    {2,4},{3,4},{0,2},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PYRAMID,{ 0, 1, 6, 8, 4,-1,-1,-1},{FO+0,FO+1,FO+2,1,FO+4,-1},-1},
                    {PYRAMID,{8,6,2,3,4,-1,-1,-1},{FO+0,0,FO+2,FO+3,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x1<<3)},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}}

};

/* define the regular rules for prisms */
static REFRULE PrismRules[MAX_PRI_RULES] =
{
  /* PRI_NO_REF */
  {PRISM,0,NO_CLASS,0,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_COPY */
  {PRISM,1,YELLOW_CLASS,1,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{PRISM,{ 0, 1, 2, 3, 4, 5,-1,-1},{FO+0,FO+1,FO+2,FO+3,FO+4,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_RED */
  {PRISM,2,RED_CLASS|SWITCH_CLASS,8,
   {1,1,1,1,1,1,1,1,1,0,1,1,1,0,0,0,0,0,0},
   (1<<9)-1+(1<<10)+(1<<11)+(1<<12),

   /* sonandnode */
   {{0,1},{1,1},{0,2},{0,3},{1,4},{2,5},
                    {4,4},{5,5},{4,5},{-1,-1},{0,4},{1,5},{0,5},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PRISM,{0,6,8,9,16,18,-1,-1},{FO+0,FO+1,2,FO+3,4,-1},-1},
                    {PRISM,{6,1,7,16,10,17,-1,-1},{FO+0,FO+1,FO+2,2,5,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x2<<3)},
                    {PRISM,{8,6,7,18,16,17,-1,-1},{FO+0,0,1,3,6,-1},
                          0x1<<PATHDEPTHSHIFT | 0x2},
                    {PRISM,{8,7,2,18,17,11,-1,-1},{FO+0,2,FO+2,FO+3,7,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x3<<3)},
                    {PRISM,{9,16,18,3,12,14,-1,-1},{0,FO+1,6,FO+3,FO+4,-1},
                          0x1<<PATHDEPTHSHIFT | 0x4},
                    {PRISM,{16,10,17,12,4,13,-1,-1},{1,FO+1,FO+2,6,FO+4,-1},
                          0x3<<PATHDEPTHSHIFT | 0x2 | (0x2<<3) | (0x4<<(2*3))},
                    {PRISM,{18,16,17,14,12,13,-1,-1},{2,4,5,7,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x4<<3)},
                    {PRISM,{18,17,11,14,13,5,-1,-1},{3,6,FO+2,FO+3,FO+4,-1},
                          0x3<<PATHDEPTHSHIFT | 0x2 | (0x3<<3) | (0x4<<(2*3))},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_QUADSECT */
  {PRISM,3,RED_CLASS|SWITCH_CLASS,4,
   {1,1,1,0,0,0,1,1,1,0,0,0,0,0,0,0,0,0,0},
   (1<<3)-1+(1<<6)+(1<<7)+(1<<8),

   /* sonandnode */
   {{0,0},{0,0},{0,0},{-1,-1},{-1,-1},{-1,-1},
                    {0,0},{0,0},{0,0},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PRISM,{0,6,8,3,12,14,-1,-1},{FO+0,FO+1,2,FO+3,FO+4,-1},-1},
                    {PRISM,{6,1,7,12,4,13,-1,-1},{FO+0,FO+1,FO+2,2,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x2<<3)},
                    {PRISM,{8,6,7,14,12,13,-1,-1},{FO+0,0,1,3,FO+4,-1},
                          0x1<<PATHDEPTHSHIFT | 0x2},
                    {PRISM,{8,7,2,14,13,5,-1,-1},{FO+0,2,FO+2,FO+3,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x3<<3)},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_BISECT_0_1 */
  {PRISM,4,RED_CLASS|SWITCH_CLASS,2,
   {1,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0},
   1 + (1<<6),

   /* sonandnode */
   {{0,1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {0,4},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PRISM,{0,6,2,3,12,5,-1,-1},{FO+0,FO+1,1,FO+3,FO+4,-1},-1},
                    {PRISM,{6,1,2,12,4,5,-1,-1},{FO+0,FO+1,FO+2,0,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x2<<3)},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_BISECT_0_2 */
  {PRISM,5,RED_CLASS|SWITCH_CLASS,2,
   {0,1,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0},
   (1<<1) + (1<<7),

   /* sonandnode */
   {{-1,-1},{0,1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{0,4},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PRISM,{0,7,2,3,13,5,-1,-1},{FO+0,1,FO+2,FO+3,FO+4,-1},-1},
                    {PRISM,{0,1,7,3,4,13,-1,-1},{FO+0,FO+1,FO+2,0,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x2<<3)},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_BISECT_0_3 */
  {PRISM,6,RED_CLASS|SWITCH_CLASS,2,
   {0,0,1,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0},
   (1<<2) + (1<<8),

   /* sonandnode */
   {{-1,-1},{-1,-1},{0,2},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{0,5},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PRISM,{0,1,8,3,4,14,-1,-1},{FO+0,FO+1,1,FO+3,FO+4,-1},-1},
                    {PRISM,{8,1,2,14,4,5,-1,-1},{FO+0,0,FO+2,FO+3,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x2<<3)},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_BISECT_1_2 */
  {PRISM,7,RED_CLASS|SWITCH_CLASS,2,
   {0,0,0,1,1,1,0,0,0,0,0,0,0,0,0,0,0,0,0},
   (1<<3) + (1<<4) + (1<<5),

   /* sonandnode */
   {{-1,-1},{-1,-1},{-1,-1},{0,0},{0,0},{0,0},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PRISM,{0,1,2,9,10,11,-1,-1},{FO+0,FO+1,FO+2,FO+3,1,-1},-1},
                    {PRISM,{9,10,11,3,4,5,-1,-1},{0,FO+1,FO+2,FO+3,FO+4,-1},
                          0x2<<PATHDEPTHSHIFT | 0x2 | (0x2<<3)},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_BISECT_HEX0 */
  {PRISM,8,RED_CLASS|SWITCH_CLASS,2,
   {1,0,1,0,0,0,1,0,1,0,0,0,0,0,0,0,0,0,0},
   1+(1<<2)+(1<<6)+(1<<8),

   /* sonandnode */
   {{0,1},{-1,-1},{0,2},{-1,-1},{-1,-1},{-1,-1},
                    {0,4},{-1,-1},{0,5},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{PRISM,{0,6,8,3,12,14,-1,-1},{FO+0,FO+1,1,FO+3,FO+4,-1},-1},
                    {HEXAHEDRON,{6,1,2,8,12,4,5,14},{FO+0,FO+1,FO+2,FO+3,0,FO+4},
                          0x1<<PATHDEPTHSHIFT | 0x2},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_BISECT_HEX1 */
  {PRISM,9,RED_CLASS|SWITCH_CLASS,2,
   {0,1,1,0,0,0,0,1,1,0,0,0,0,0,0,0,0,0,0},
   (1<<1)+(1<<2)+(1<<7)+(1<<8),

   /* sonandnode */
   {{-1,-1},{0,2},{0,3},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{0,6},{0,7},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{HEXAHEDRON,{0,1,7,8,3,4,13,14},{FO+0,FO+1,FO+2,1,FO+3,FO+4},-1},
                    {PRISM,{7,8,2,13,14,5,-1,-1},{FO+0,0,FO+2,FO+3,FO+4,-1},
                          0x1<<PATHDEPTHSHIFT | 0x3},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_BISECT_HEX2 */
  {PRISM,10,RED_CLASS|SWITCH_CLASS,2,
   {1,1,0,0,0,0,1,1,0,0,0,0,0,0,0,0,0,0,0},
   1+(1<<1)+(1<<6)+(1<<7),

   /* sonandnode */
   {{0,1},{0,2},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {0,5},{0,6},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{HEXAHEDRON,{6,7,2,0,12,13,5,3},{FO+0,1,FO+2,FO+3,FO+1,FO+4},-1},
                    {PRISM,{6,1,7,12,4,13,-1,-1},{FO+0,FO+1,FO+2,0,FO+4,-1},
                          0x1<<PATHDEPTHSHIFT | 0x2},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_trisect_0_4 */
  {PRISM,11,RED_CLASS|SWITCH_CLASS,3,
   {1,1,1,0,0,0,1,1,1,1,0,0,0,1,0,0,0,0,0},
   (1<<3)-1 + (1<<6) + (1<<7) + (1<<8) + (1<<9) + (1<<13),

   /* sonandnode */
   {{0,1},{1,1},{0,2},{0,3},{1,4},{2,5},
                    {4,4},{5,5},{4,5},{-1,-1},{0,4},{1,5},{0,5},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},

   /* sons */
   {{HEXAHEDRON,{ 0, 6,15,8,3,12,19,14},{FO+0,FO+1,1,2,FO+3,FO+4},-1},
                    {HEXAHEDRON,{ 6, 1, 7,15,12,4,13,19},{FO+0,FO+1,FO+2,2,0,FO+4},
              0x1<<PATHDEPTHSHIFT | 0x2},
                    {HEXAHEDRON,{ 8,15,7,2,14,19,13,5},{FO+0,0,1,FO+2,FO+3,FO+4},
              0x1<<PATHDEPTHSHIFT | 0x2},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_rotate_l */
  {PRISM,12,RED_CLASS,1,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{PRISM,{ 1, 2, 0, 4, 5, 3,-1,-1},{FO+0,FO+2,FO+3,FO+1,FO+4,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_rotate_r */
  {PRISM,13,RED_CLASS,1,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{PRISM,{ 2, 0, 1, 5, 3, 4,-1,-1},{FO+0,FO+3,FO+1,FO+2,FO+4,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* PRI_QUADSECT_HEXPRI0 */
  {PRISM,14,RED_CLASS,4,
   {1,1,0,1,1,1,1,1,0,0,1,1,0,0,0,0,0,0,0},
   3 + (1<<3) + (1<<4) + (1<<5) + (1<<6) + (1<<7) + (1<<10) + (1<<11),

   {{0,1},{0,2},{-1,-1},{0,4},{1,5},{0,7},
                    {3,5},{3,6},{-1,-1},{-1,-1},{0,5},{0,6},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{HEXAHEDRON,{11, 9, 0, 2,17,16,6, 7},{FO+3,3,FO+1,FO+0,FO+2,1},-1},
                    {PRISM,{ 6, 1, 7,16,10,17,-1,-1},{FO+0,FO+1,FO+2,0,2,-1},-1},
                    {PRISM,{16,10,17,12, 4,13,-1,-1},{1,FO+1,FO+2,3,FO+4,-1},-1},
                    {HEXAHEDRON,{ 5, 3, 9,11,13,12,16,17},{FO+3,FO+4,FO+1,0,FO+2,2},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}}

};

/* define the regular rules for hexahedra */
static REFRULE HexahedronRules[MAX_HEX_RULES] =
{
  /* HEX_NO_REF */
  {HEXAHEDRON,0,NO_CLASS,0,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_COPY */
  {HEXAHEDRON,1,YELLOW_CLASS,1,
   {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},0,
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                    {-1,-1}},
   {{HEXAHEDRON,{0,1,2,3,4,5,6,7},{FO+0,FO+1,FO+2,FO+3,FO+4,FO+5},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                    {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_RED */
  {HEXAHEDRON,2,RED_CLASS|SWITCH_CLASS,8,
   {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1},(1<<19)-1,

   /* sonandnode */
   {{0,1},{1,2},{2,3},{3,0},
                     {0,4},{1,5},{2,6},{3,7},
                     {4,5},{5,6},{6,7},{7,4},
                     {0,2},{0,5},{1,6},{2,7},{0,7},{4,6},{0,6}},

   /* sons */
   {{HEXAHEDRON,{ 0, 8,20,11,12,21,26,24},{FO+0,FO+1, 1, 3,FO+4, 4},-1},
                     {HEXAHEDRON,{ 8, 1, 9,20,21,13,22,26},{FO+0,FO+1,FO+2, 2, 0, 5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {HEXAHEDRON,{20, 9, 2,10,26,22,14,23},{FO+0, 1,FO+2,FO+3, 3, 6},
                     0x2<<PATHDEPTHSHIFT | 0x2 | 0x3<<3},
                     {HEXAHEDRON,{11,20,10, 3,24,26,23,15},{FO+0, 0, 2,FO+3,FO+4, 7},
                     0x1<<PATHDEPTHSHIFT | 0x3},

                     {HEXAHEDRON,{12,21,26,24, 4,16,25,19},{ 0,FO+1, 5, 7,FO+4,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x5},
                     {HEXAHEDRON,{21,13,22,26,16, 5,17,25},{ 1,FO+1,FO+2, 6, 4,FO+5},
                     0x2<<PATHDEPTHSHIFT | 0x5 | 0x2<<3},
                     {HEXAHEDRON,{26,22,14,23,25,17, 6,18},{ 2, 5,FO+2,FO+3, 7,FO+5},
                     0x3<<PATHDEPTHSHIFT | 0x5 | 0x2<<3 | 0x3<<(2*3)},
                     {HEXAHEDRON,{24,26,23,15,19,25,18, 7},{ 3, 4, 6,FO+3,FO+4,FO+5},
                     0x2<<PATHDEPTHSHIFT | 0x5 | 0x3<<3},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_bisect_0_1 */
  {HEXAHEDRON,3,RED_CLASS|SWITCH_CLASS,2,
   {1,0,1,0,0,0,0,0,1,0,1,0,0,0,0,0,0,0,0},
   1 + (1<<2) + (1<<8) + (1<<10),

   /* sonandnode */
   {{0,0},{-1,-1},{0,0},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{0,0},{-1,-1},{0,0},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1}},

   /* sons */
   {{HEXAHEDRON,{ 0, 8,10,3,4,16,18,7},{FO+0,FO+1, 1,FO+3,FO+4,FO+5},-1},
                     {HEXAHEDRON,{ 8, 1, 2,10,16,5,6,18},{FO+0,FO+1,FO+2,FO+3, 0,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_bisect_0_2 */
  {HEXAHEDRON,4,RED_CLASS|SWITCH_CLASS,2,
   {0,1,0,1,0,0,0,0,0,1,0,1,0,0,0,0,0,0,0},
   (1<<1) + (1<<3) + (1<<9) + (1<<11),

   /* sonandnode */
   {{-1,-1},{0,0},{-1,-1},{0,0},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{0,0},{-1,-1},{0,0},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1}},

   /* sons */
   {{HEXAHEDRON,{ 0, 1,9,11,4,5,17,19},{FO+0,FO+1,FO+2,1,FO+4,FO+5},-1},
                     {HEXAHEDRON,{ 11, 9, 2,3,19,17,6,7},{FO+0,0,FO+2,FO+3,FO+4,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_bisect_0_3 */
  {HEXAHEDRON,5,RED_CLASS|SWITCH_CLASS,2,
   {0,0,0,0,1,1,1,1,0,0,0,0,0,0,0,0,0,0,0},
   (1<<4) + (1<<5) + (1<<6) + (1<<7),

   /* sonandnode */
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},{0,0},{0,0},
                     {0,0},{0,0},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1}},

   /* sons */
   {{HEXAHEDRON,{ 0, 1,2,3,12,13,14,15},{FO+0,FO+1,FO+2,FO+3,FO+4,1},-1},
                     {HEXAHEDRON,{ 12,13,14,15,4,5,6,7},{0,FO+1,FO+2,FO+3,FO+4,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_quadsect_0 */
  {HEXAHEDRON,6,RED_CLASS|SWITCH_CLASS,4,
   {1,1,1,1,0,0,0,0,1,1,1,1,1,0,0,0,0,1,0},
   (1<<4)-1 + (1<<8) + (1<<9) + (1<<10) + (1<<11) + (1<<12) + (1<<17),

   /* sonandnode */
   {{0,1},{1,2},{2,3},{0,3},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {0,5},{1,6},{2,7},{0,7},
                     {0,2},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{0,6},{-1,-1}},

   /* sons */
   {{HEXAHEDRON,{ 0, 8,20,11,4,16,25,19},{FO+0,FO+1,1,3,FO+4,FO+5},-1},
                     {HEXAHEDRON,{ 8, 1, 9,20,16,5,17,25},{FO+0,FO+1,FO+2,2,0,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {HEXAHEDRON,{ 20, 9, 2,10,25,17,6,18},{FO+0,1,FO+2,FO+3,3,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {HEXAHEDRON,{ 11, 20, 10,3,19,25,18,7},{FO+0,0,2,FO+3,FO+4,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_quadsect_1 */
  {HEXAHEDRON,7,RED_CLASS|SWITCH_CLASS,4,
   {1,0,1,0,1,1,1,1,1,0,1,0,0,1,0,1,0,0,0},
   1 + (1<<2) + (1<<4) + (1<<5) + (1<<6) + (1<<7) + (1<<8) + (1<<10) + (1<<13) + (1<<15),

   /* sonandnode */
   {{0,1},{-1,-1},{0,2},{-1,-1},
                     {0,4},{1,5},{1,6},{0,7},
                     {3,5},{-1,-1},{3,6},{-1,-1},
                     {-1,-1},{0,5},{-1,-1},{0,6},{-1,-1},{-1,-1},{-1,-1}},

   /* sons */
   {{HEXAHEDRON,{ 0, 8,10,3,12,21,23,15},{FO+0,FO+1,1,FO+3,FO+4,3},-1},
                     {HEXAHEDRON,{ 8, 1, 2,10,21,13,14,23},{FO+0,FO+1,FO+2,FO+3,0,2},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {HEXAHEDRON,{ 21, 13, 14,23,16,5,6,18},{1,FO+1,FO+2,FO+3,3,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {HEXAHEDRON,{ 12, 21, 23,15,4,16,18,7},{0,FO+1,2,FO+3,FO+4,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_quadsect_2 */
  {HEXAHEDRON,8,RED_CLASS|SWITCH_CLASS,4,
   {0,1,0,1,1,1,1,1,0,1,0,1,0,0,1,0,1,0,0},
   (1<<1) + (1<<3) + (1<<4) + (1<<5) + (1<<6) + (1<<7) + (1<<9) + (1<<11) + (1<<14) + (1<<16),

   /* sonandnode */
   {{-1,-1},{0,2},{-1,-1},{0,3},
                     {0,4},{0,5},{1,6},{1,7},
                     {-1,-1},{3,6},{-1,-1},{3,7},
                     {-1,-1},{-1,-1},{0,6},{-1,-1},{0,7},{-1,-1},{-1,-1}},

   /* sons */
   {{HEXAHEDRON,{ 0, 1,9,11,12,13,22,24},{FO+0,FO+1,FO+2,1,FO+4,3},-1},
                     {HEXAHEDRON,{ 11, 9, 2,3,24,22,14,15},{FO+0,0,FO+2,FO+3,FO+4,2},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {HEXAHEDRON,{ 24, 22, 14,15,19,17,6,7},{1,3,FO+2,FO+3,FO+4,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {HEXAHEDRON,{ 12, 13, 22,24,4,5,17,19},{0,FO+1,FO+2,2,FO+4,FO+5},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_trisect_0 */
  {HEXAHEDRON,9,RED_CLASS|SWITCH_CLASS,3,
   {1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0},(1<<2)+1,

   /* sonandnode */
   {{0,0},{-1,-1},{0,0},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1}},

   /* sons */
   {{PRISM,{0,4,8,3,7,10,-1,-1},{FO+1,FO+4,1,FO+0,FO+3,-1},-1},
                     {PRISM,{ 4, 5, 8,7,6,10,-1,-1},{FO+1,FO+5,2,0,FO+3,-1},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {PRISM,{ 5, 1, 8,6,2,10,-1,-1},{FO+1,FO+2,FO+0,1,FO+3,-1},
                     0x1<<PATHDEPTHSHIFT | 0x2},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
                     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}}},

  /* HEX_trisect_5 */
  {HEXAHEDRON,9,RED_CLASS|SWITCH_CLASS,3,
   {0,0,0,0,0,0,0,0,1,0,1,0,0,0,0,0,0,0,0},
   (1<<8) + (1<<10),

   /* sonandnode */
   {{-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {0,0},{-1,-1},{0,0},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},

   /* sons */
   {
     {PRISM,{ 4, 16,0,7,18,3,-1,-1},{FO+1,FO+5,1,FO+4,FO+3,-1},-1},
     {PRISM,{ 16, 1,0,18,2,3,-1,-1},{FO+1,2,FO+0,0,FO+3,-1},-1},
     {PRISM,{ 5, 1,16,6,2,18,-1,-1},{FO+1,FO+2,1,FO+5,FO+3,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}
   }},

  /* HEX_bisect_hexpri0 */
  {HEXAHEDRON,10,RED_CLASS|SWITCH_CLASS,2,
   {0,0,0,1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0},
   (1<<3) + (1<<11),

   /* sonandnode */
   {{-1,-1},{-1,-1},{-1,-1},{0,3},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{0,7},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},

   /* sons */
   {
     {HEXAHEDRON,{ 0, 1, 2,11, 4, 5, 6,19},{FO+0,FO+1,FO+2,1,FO+4,FO+5},-1},
     {PRISM,{11, 2, 3,19, 6, 7,-1,-1},{FO+0,0,FO+2,FO+3,FO+4,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}
   }},

  /* HEX_bisect_hexpri1 */
  {HEXAHEDRON,11,RED_CLASS|SWITCH_CLASS,2,
   {1,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0},
   1 + (1<<8),

   /* sonandnode */
   {{0,0},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},
                     {0,4},{-1,-1},{-1,-1},{-1,-1},
                     {-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1},{-1,-1}},

   /* sons */
   {
     {HEXAHEDRON,{ 8, 1, 2, 3,16, 5, 6, 7},{FO+0,FO+1,FO+2,FO+3,1,FO+5},-1},
     {PRISM,{ 8, 3, 0,16, 7, 4,-1,-1},{FO+0,0,FO+2,FO+3,FO+4,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1},
     {-1,{-1,-1,-1,-1,-1,-1,-1,-1},{-1,-1,-1,-1,-1,-1},-1}
   }}

};
#endif

/****************************************************************************/
/*                                                                          */
/* forward declarations of functions used before they are defined           */
/*                                                                          */
/****************************************************************************/

#ifdef __THREEDIM__

INT NS_DIM_PREFIX GetRule_AnisotropicRed (ELEMENT *theElement, INT *Rule)
{
  DOUBLE area,norm;
  DOUBLE_VECTOR a,b,c;

  switch (TAG(theElement))
  {
  case TETRAHEDRON :
    *Rule=TET_RED;
    return (0);

  case PYRAMID :
    *Rule=PYR_RED;
    return (0);

  case PRISM :
    *Rule=PRI_RED;
    V3_SUBTRACT(CVECT(MYVERTEX(CORNER(theElement,1))),CVECT(MYVERTEX(CORNER(theElement,0))),a);
    V3_SUBTRACT(CVECT(MYVERTEX(CORNER(theElement,2))),CVECT(MYVERTEX(CORNER(theElement,0))),b);
    V3_VECTOR_PRODUCT(a,b,c);
    V3_EUKLIDNORM(c,area);
    area *= 0.5;
    V3_SUBTRACT(CVECT(MYVERTEX(CORNER(theElement,3))),CVECT(MYVERTEX(CORNER(theElement,0))),a);
    V3_EUKLIDNORM(a,norm);
    if (norm < 0.25*SQRT(area))
    {
      *Rule=PRI_QUADSECT;
      return(1);
    }
    break;

  case HEXAHEDRON :
    *Rule=HEXA_RED;
    return (0);

  default :
    assert(0);
  }

  return (0);
}


/****************************************************************************/
/** \brief Compute best full refined refrule for the element

   \param theElement - for that element

   This function computes the best full refined refrule for the element.

   \return
   Mark: number of refrule
 */
/****************************************************************************/

static INT ShortestInteriorEdge (ELEMENT *theElement)
{
  DOUBLE *Corners[MAX_CORNERS_OF_ELEM];
  DOUBLE_VECTOR MidPoints[MAX_EDGES_OF_ELEM];
  INT i, flags;
  DOUBLE Dist_0_5, Dist_1_3, Dist_2_4;

  /* get physical position of the corners */
  for (i=0; i<CORNERS_OF_ELEM(theElement); i++)
    Corners[i] = CVECT(MYVERTEX(CORNER(theElement,i)));

  /* get physical position of the midpoints of the edges */
  for (i=0; i<EDGES_OF_ELEM(theElement); i++)
    V3_LINCOMB(0.5, Corners[CORNER_OF_EDGE(theElement,i,0)], 0.5, Corners[CORNER_OF_EDGE(theElement,i,1)], MidPoints[i]);

  /* compute distances */
  V3_EUKLIDNORM_OF_DIFF(MidPoints[0], MidPoints[5], Dist_0_5)
  V3_EUKLIDNORM_OF_DIFF(MidPoints[1], MidPoints[3], Dist_1_3)
  V3_EUKLIDNORM_OF_DIFF(MidPoints[2], MidPoints[4], Dist_2_4)

  /* return best Refrule (shortest interior edge) */
  flags = (Dist_0_5 < Dist_1_3) ;
  flags |= ((Dist_1_3 < Dist_2_4) <<1) ;
  flags |= ((Dist_2_4 < Dist_0_5) <<2) ;
  assert(flags != 7);

  switch(flags)
  {
  case 0 :                                              /* Dist_0_5 = Dist_2_4 = Dist_1_3 */
    return (FULL_REFRULE_0_5);
  case 1 :                                              /* Dist_0_5 < Dist_2_4 < Dist_1_3 */
    return (FULL_REFRULE_0_5);
  case 2 :                                              /* Dist_1_3 < Dist_0_5 < Dist_2_4 */
    return (FULL_REFRULE_1_3);
  case 3 :                                              /* Dist_0_5 < Dist_1_3 < Dist_2_4 */
    return (FULL_REFRULE_0_5);
  case 4 :                                              /* Dist_2_4 < Dist_1_3 < Dist_0_5 */
    return (FULL_REFRULE_2_4);
  case 5 :                                              /* Dist_2_4 < Dist_0_5 < Dist_1_3 */
    return (FULL_REFRULE_2_4);
  case 6 :                                              /* Dist_1_3 < Dist_2_4 < Dist_0_5 */
    return (FULL_REFRULE_1_3);
  }
  return (-1);
}


#ifdef __ALLRULES__


/****************************************************************************/
/** \brief Compute best full refined refrule for the element

   \param theElement - for that element

   This function computes the best full refined refrule for the element

   \return
   Mark: number of refrule
 */
/****************************************************************************/

static INT MinimalSideAngle (ELEMENT *theElement)
{
  DOUBLE *Corners[MAX_CORNERS_OF_ELEM];
  DOUBLE_VECTOR MidPoints[MAX_EDGES_OF_ELEM];
  INT i,j,k,l,imin;
  DOUBLE MaxAngle,Max,Min;

  /* get physical position of the corners */
  for (i=0; i<CORNERS_OF_ELEM(theElement); i++)
    Corners[i] = CVECT(MYVERTEX(CORNER(theElement,i)));

  /* get physical position of the midpoints of the edges */
  for (i=0; i<EDGES_OF_ELEM(theElement); i++)
    V3_LINCOMB(0.5, Corners[CORNER_OF_EDGE(theElement,i,0)], 0.5, Corners[CORNER_OF_EDGE(theElement,i,1)], MidPoints[i]);

  /* try possebilities */
  Min = 190.0;
  for (i=0; i<3; i++)
  {
    j = OPPOSITE_EDGE(theElement,i);
    Corners[2] = MidPoints[i];
    Corners[3] = MidPoints[j];

    Max = 0.0;
    for (k=0; k<2; k++)
    {
      for (l=0; l<2; l++)
        Corners[l] = MidPoints[SideEdgesOfEdge[i][k][l]];
      if (TetMaxSideAngle(theElement,Corners,&MaxAngle))
        return (FULL_REFRULE_0_5);
      Max = MAX(Max,MaxAngle);
    }
    for (k=0; k<2; k++)
    {
      for (l=0; l<2; l++)
        Corners[l] = MidPoints[SideEdgesOfEdge[j][k][l]];
      if (TetMaxSideAngle(theElement,Corners,&MaxAngle))
        return (FULL_REFRULE_0_5);
      Max = MAX(Max,MaxAngle);
    }
    if (Max<Min)
    {
      Min = Max;
      imin = i;
    }
  }

  switch (imin)
  {
  case 0 :
    return (FULL_REFRULE_0_5);
  case 1 :
    return (FULL_REFRULE_1_3);
  case 2 :
    return (FULL_REFRULE_2_4);
  }
}


/****************************************************************************/
/** \brief Compute best full refined refrule for the element

   \param theElement - for that element

   This function computes the best full refined refrule for the element

   \return
   Mark: number of refrule
 */
/****************************************************************************/

static INT MinimalSideEntry (ELEMENT *theElement)
{
  DOUBLE *Corners[MAX_CORNERS_OF_ELEM];
  DOUBLE_VECTOR MidPoints[MAX_EDGES_OF_ELEM];
  INT i,j,k,l,imin;
  DOUBLE Angle[MAX_EDGES_OF_ELEM],Length[MAX_EDGES_OF_ELEM],Max,Min,help;

  /* get physical position of the corners */
  for (i=0; i<CORNERS_OF_ELEM(theElement); i++)
    Corners[i] = CVECT(MYVERTEX(CORNER(theElement,i)));

  /* get physical position of the midpoints of the edges */
  for (i=0; i<EDGES_OF_ELEM(theElement); i++)
    V3_LINCOMB(0.5, Corners[CORNER_OF_EDGE(theElement,i,0)], 0.5, Corners[CORNER_OF_EDGE(theElement,i,1)], MidPoints[i]);

  /* try possibilities */
  Min = MAX_C;
  for (i=0; i<3; i++)
  {
    j = OPPOSITE_EDGE(theElement,i);
    Corners[2] = MidPoints[i];
    Corners[3] = MidPoints[j];

    Max = 0.0;
    for (k=0; k<2; k++)
    {
      for (l=0; l<2; l++)
        Corners[l] = MidPoints[SideEdgesOfEdge[i][k][l]];
      if (TetAngleAndLength(theElement,Corners,Angle,Length))
        return (FULL_REFRULE_0_5);
      for (l=0; l<EDGES_OF_ELEM(theElement); l++)
        if (Angle[l]>PI/2.0)
        {
          help = ABS(Length[l]*(DOUBLE)(cos((double)Angle[l])/sin((double)Angle[l])));
          Max = MAX(Max,help);
        }
    }
    for (k=0; k<2; k++)
    {
      for (l=0; l<2; l++)
        Corners[l] = MidPoints[SideEdgesOfEdge[j][k][l]];
      if (TetAngleAndLength(theElement,Corners,Angle,Length))
        return (FULL_REFRULE_0_5);
      for (i=0; i<EDGES_OF_ELEM(theElement); i++)
        if (Angle[l]>PI/2.0)
        {
          help = ABS(Length[l]*(DOUBLE)(cos((double)Angle[l])/sin((double)Angle[l])));
          Max = MAX(Max,help);
        }
    }
    if (Max<Min)
    {
      Min = Max;
      imin = i;
    }
  }

  switch (imin)
  {
  case 0 :
    return (FULL_REFRULE_0_5);
  case 1 :
    return (FULL_REFRULE_1_3);
  case 2 :
    return (FULL_REFRULE_2_4);
  }
}


/****************************************************************************/
/** \brief Compute best full refined refrule for the element

   \param theElement - for that element

   This function computes the best full refined refrule for the element: optimal laplace-disc w.r.t. M-Matrix eigenschaft.

   \return
   Mark: number of refrule
 */
/****************************************************************************/

static INT BestLaplaceMMatrix (ELEMENT *theElement)
{
  DOUBLE *Corners[MAX_CORNERS_OF_ELEM];
  DOUBLE_VECTOR MidPoints[MAX_EDGES_OF_ELEM];
  INT i,j,k,l,imin,TBFR,refrule;
  DOUBLE Angle[MAX_EDGES_OF_ELEM],Length[MAX_EDGES_OF_ELEM],sum,Min;

  /* get physical position of the corners */
  for (i=0; i<CORNERS_OF_ELEM(theElement); i++)
    Corners[i] = CVECT(MYVERTEX(CORNER(theElement,i)));

  /* get physical position of the midpoints of the edges */
  for (i=0; i<EDGES_OF_ELEM(theElement); i++)
    V3_LINCOMB(0.5, Corners[CORNER_OF_EDGE(theElement,i,0)], 0.5, Corners[CORNER_OF_EDGE(theElement,i,1)], MidPoints[i]);

  /* try possebilities */
  Min = MAX_C; imin = -1;
  for (i=0; i<3; i++)
  {
    j = OPPOSITE_EDGE(theElement,i);
    Corners[2] = MidPoints[i];
    Corners[3] = MidPoints[j];

    sum = 0.0;
    for (k=0; k<2; k++)
    {
      for (l=0; l<2; l++)
        Corners[l] = MidPoints[SideEdgesOfEdge[i][k][l]];
      if (TetAngleAndLength(theElement,Corners,Angle,Length))
        return (FULL_REFRULE_0_5);
      for (l=0; l<EDGES_OF_ELEM(theElement); l++)
        if (Angle[l]>PI/2.0)
          sum += ABS(Length[l]*(DOUBLE)cos((double)Angle[l])/(DOUBLE)sin((double)Angle[l]));
    }
    for (k=0; k<2; k++)
    {
      for (l=0; l<2; l++)
        Corners[l] = MidPoints[SideEdgesOfEdge[j][k][l]];
      if (TetAngleAndLength(theElement,Corners,Angle,Length))
        return (FULL_REFRULE_0_5);
      for (i=0; i<EDGES_OF_ELEM(theElement); i++)
        if (Angle[l]>PI/2.0)
          sum += ABS(Length[l]*(DOUBLE)cos((double)Angle[l])/(DOUBLE)sin((double)Angle[l]));
    }
    if (sum<Min)
    {
      Min = sum;
      imin = i;
    }
  }

  switch (imin)
  {
  case 0 :
    refrule = FULL_REFRULE_0_5;
    break;
  case 1 :
    refrule = FULL_REFRULE_1_3;
    break;
  case 2 :
    refrule = FULL_REFRULE_2_4;
    break;
  }


  TBFR = ShortestInteriorEdge(theElement);
  if (imin == -1)
  {
    refrule = TBFR;
    UserWrite ("#");
  }

  return (refrule);
}

#endif


/****************************************************************************/
/** \brief Compute best full refined refrule for the element

   \param theElement - for that element

   This function computes the best full refined refrule for the element: optimal laplace-disc w.r.t. M-Matrix eigenschaft

   \return
   .n   Mark: number of refrule
 */
/****************************************************************************/

static INT MaxPerpendicular (ELEMENT *theElement)
{
  DOUBLE *Corners[MAX_CORNERS_OF_ELEM];
  DOUBLE_VECTOR MidPoints[MAX_EDGES_OF_ELEM],a,b,c;
  INT i,j,imin,TBFR,refrule;
  DOUBLE sprd,Max;

  /* get physical position of the corners */
  for (i=0; i<CORNERS_OF_ELEM(theElement); i++)
    Corners[i] = CVECT(MYVERTEX(CORNER(theElement,i)));

  /* get physical position of the midpoints of the edges */
  for (i=0; i<EDGES_OF_ELEM(theElement); i++)
    V3_LINCOMB(0.5, Corners[CORNER_OF_EDGE(theElement,i,0)], 0.5, Corners[CORNER_OF_EDGE(theElement,i,1)], MidPoints[i]);

  /* try possebilities */
  Max = -MAX_C; imin = -1;
  for (i=0; i<3; i++)
  {
    j = OPPOSITE_EDGE(theElement,i);

    V3_SUBTRACT(Corners[CORNER_OF_EDGE(theElement,i,0)],Corners[CORNER_OF_EDGE(theElement,i,1)],a)
    V3_SUBTRACT(Corners[CORNER_OF_EDGE(theElement,j,0)],Corners[CORNER_OF_EDGE(theElement,j,1)],b)
    V3_VECTOR_PRODUCT(a,b,c)
    V3_Normalize(c);
    V3_SUBTRACT(MidPoints[i],MidPoints[j],a)
    V3_Normalize(a);
    V3_SCALAR_PRODUCT(a,c,sprd)
    sprd = ABS(sprd);

    if (sprd>Max)
    {
      Max = sprd;
      imin = i;
    }
  }

  switch (imin)
  {
  case 0 :
    refrule = FULL_REFRULE_0_5;
    break;
  case 1 :
    refrule = FULL_REFRULE_1_3;
    break;
  case 2 :
    refrule = FULL_REFRULE_2_4;
    break;
  }


  TBFR = ShortestInteriorEdge (theElement);
  if (imin == -1)
  {
    refrule = TBFR;
    UserWrite ("#");
  }

  return (refrule);
}


/****************************************************************************/
/** \brief
   MaxRightAngle - compute best full refined refrule for the element

   SYNOPSIS:
   static INT MaxRightAngle (ELEMENT *theElement);

   PARAMETERS:
   \param theElement - for that element

   DESCRIPTION:
   This function computes the best full refined refrule for the element: optimal laplace-disc w.r.t. M-Matrix eigenschaft

   \return
   INT
   .n   Mark: number of refrule
 */
/****************************************************************************/

static INT MaxRightAngle (ELEMENT *theElement)
{
  DOUBLE *Corners[MAX_CORNERS_OF_ELEM];
  DOUBLE_VECTOR a,b;
  INT i,j,imin,TBFR,refrule;
  DOUBLE sprd,Min;

  /* get physical position of the corners */
  for (i=0; i<CORNERS_OF_ELEM(theElement); i++)
    Corners[i] = CVECT(MYVERTEX(CORNER(theElement,i)));

  /* try possebilities */
  Min = MAX_C; imin = -1;
  for (i=0; i<3; i++)
  {
    j = OPPOSITE_EDGE(theElement,i);

    V3_SUBTRACT(Corners[CORNER_OF_EDGE(theElement,i,0)],Corners[CORNER_OF_EDGE(theElement,i,1)],a)
    V3_Normalize(a);
    V3_SUBTRACT(Corners[CORNER_OF_EDGE(theElement,j,0)],Corners[CORNER_OF_EDGE(theElement,j,1)],b)
    V3_Normalize(b);
    V3_SCALAR_PRODUCT(a,b,sprd)
    sprd = ABS(sprd);

    if (sprd<Min)
    {
      Min = sprd;
      imin = i;
    }
  }

  switch (imin)
  {
  case 0 :
    refrule = FULL_REFRULE_0_5;
    break;
  case 1 :
    refrule = FULL_REFRULE_1_3;
    break;
  case 2 :
    refrule = FULL_REFRULE_2_4;
    break;
  }


  TBFR = ShortestInteriorEdge (theElement);
  if (imin == -1)
  {
    refrule = TBFR;
    UserWrite ("#");
  }

  return (refrule);
}


/****************************************************************************/
/** \brief
   MaxArea - compute best full refined refrule for the element

   SYNOPSIS:
   static INT MaxArea (ELEMENT *theElement);

   PARAMETERS:
   \param theElement - for that element

   DESCRIPTION:
   This function computes the best full refined refrule for the element: optimal laplace-disc w.r.t. M-Matrix eigenschaft

   \return
   INT
   .n   Mark: number of refrule
 */
/****************************************************************************/

static INT MaxArea (ELEMENT *theElement)
{
  DOUBLE *Corners[MAX_CORNERS_OF_ELEM];
  DOUBLE_VECTOR a,b,c;
  INT i,j,imin,TBFR,refrule;
  DOUBLE norm,Max;

  /* get physical position of the corners */
  for (i=0; i<CORNERS_OF_ELEM(theElement); i++)
    Corners[i] = CVECT(MYVERTEX(CORNER(theElement,i)));

  /* try possebilities */
  Max = -MAX_C; imin = -1;
  for (i=0; i<3; i++)
  {
    j = OPPOSITE_EDGE(theElement,i);

    V3_SUBTRACT(Corners[CORNER_OF_EDGE(theElement,i,0)],Corners[CORNER_OF_EDGE(theElement,i,1)],a)
    V3_SUBTRACT(Corners[CORNER_OF_EDGE(theElement,j,0)],Corners[CORNER_OF_EDGE(theElement,j,1)],b)
    V3_VECTOR_PRODUCT(a,b,c)
    V3_EUKLIDNORM(c,norm)

    if (norm>Max)
    {
      Max = norm;
      imin = i;
    }
  }

  switch (imin)
  {
  case 0 :
    refrule = FULL_REFRULE_0_5;
    break;
  case 1 :
    refrule = FULL_REFRULE_1_3;
    break;
  case 2 :
    refrule = FULL_REFRULE_2_4;
    break;
  }


  TBFR = ShortestInteriorEdge (theElement);
  if (imin == -1)
  {
    refrule = TBFR;
    UserWrite ("#");
  }

  return (refrule);
}


/****************************************************************************/
/** \brief
   Alignment -  compute best full refined refrule for the element

   SYNOPSIS:
   static INT Alignment (ELEMENT *theElement);

   PARAMETERS:
   \param theElement - for that element

   DESCRIPTION:
   This function computes the best full refined refrule for the element according to velocity

   \return
   INT
   .n   Mark: number of refrule
 */
/****************************************************************************/

static INT Alignment (ELEMENT *theElement)
{
  DOUBLE *Corners[MAX_CORNERS_OF_ELEM];
  DOUBLE_VECTOR MidPoints[MAX_EDGES_OF_ELEM], help, MidPoint;
  DOUBLE_VECTOR Velocity;
  INT i, imax;
  DOUBLE Dist_0_5, Dist_1_3, Dist_2_4, max;

  /* get physical position of the corners */
  V3_CLEAR(MidPoint)
  for (i=0; i<CORNERS_OF_ELEM(theElement); i++)
    Corners[i] = CVECT(MYVERTEX(CORNER(theElement,i)));
  (*theDirectionElemEval)(theElement,(const DOUBLE **)Corners,
                          (DOUBLE *)LMP(CORNERS_OF_ELEM(theElement)),
                          Velocity);

  /* get physical position of the midpoints of the edges */
  for (i=0; i<EDGES_OF_ELEM(theElement); i++)
    V3_LINCOMB(0.5, Corners[CORNER_OF_EDGE(theElement,i,0)], 0.5, Corners[CORNER_OF_EDGE(theElement,i,1)], MidPoints[i]);

  /* compute differences */
  max=-MAX_C;
  for (i=0; i<EDGES_OF_ELEM(theElement); i++)
  {
    V3_SUBTRACT(Corners[CORNER_OF_EDGE(theElement,i,0)], Corners[CORNER_OF_EDGE(theElement,i,1)], help)
    V3_Normalize(help);
    if (ABS(Velocity[0]*help[0]+Velocity[1]*help[1]+Velocity[2]*help[2])>max) {imax=i; max=ABS(Velocity[0]*help[0]+Velocity[1]*help[1]+Velocity[2]*help[2]);}
  }
  V3_EUKLIDNORM_OF_DIFF(MidPoints[0], MidPoints[5], Dist_0_5)
  V3_EUKLIDNORM_OF_DIFF(MidPoints[1], MidPoints[3], Dist_1_3)
  V3_EUKLIDNORM_OF_DIFF(MidPoints[2], MidPoints[4], Dist_2_4)
  switch (imax)
  {
  case 0 : if (Dist_1_3<Dist_2_4) return (FULL_REFRULE_1_3);
    else return (FULL_REFRULE_2_4);
  case 1 : if (Dist_0_5<Dist_2_4) return (FULL_REFRULE_0_5);
    else return (FULL_REFRULE_2_4);
  case 2 : if (Dist_1_3<Dist_0_5) return (FULL_REFRULE_1_3);
    else return (FULL_REFRULE_0_5);
  case 3 : if (Dist_0_5<Dist_2_4) return (FULL_REFRULE_0_5);
    else return (FULL_REFRULE_2_4);
  case 4 : if (Dist_1_3<Dist_0_5) return (FULL_REFRULE_1_3);
    else return (FULL_REFRULE_0_5);
  case 5 : if (Dist_1_3<Dist_2_4) return (FULL_REFRULE_1_3);
    else return (FULL_REFRULE_2_4);
  }
  return (-1);
}
#endif /* __THREEDIM__ */


/****************************************************************************/
/** \brief Mark an element for refinement

   \param theElement - Element to be refined
   \param rule - type of refinement mark

   This function marks an element for refinement

   \return <ul>
   <li> 1 if element has been marked </li>
   <li> 0 if element cannot be marked </li>
   </ul>
 */
/****************************************************************************/

INT NS_DIM_PREFIX MarkForRefinement (ELEMENT *theElement, enum RefinementRule rule, INT side)
{
  if (theElement == NULL) return(0);
        #ifdef ModelP
  if (EGHOST(theElement)) return(0);
        #endif

  SETCOARSEN(theElement,0);

  if (rule != COARSE)
    theElement = ELEMENT_TO_MARK(theElement);
  ASSERT(theElement!=NULL);

  PRINTDEBUG(gm,4,(PFMT "MarkForRefinement() e=" EID_FMTX "rule=%d\n",
                   me,EID_PRTX(theElement),rule))

  /* choose dimension */
  switch (DIM)
  {
                #ifdef __TWODIM__
  /* 2D case */
  case (2) :

    switch (TAG(theElement))
    {
    case (TRIANGLE) :
      switch (rule)
      {
      case COARSE :
        SETCOARSEN(theElement,1);
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        break;

      case NO_REFINEMENT :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        break;

      case COPY :
        SETMARK(theElement,T_COPY);
        SETMARKCLASS(theElement,RED_CLASS);
        break;

      case RED :
        SETMARK(theElement,T_RED);
        SETMARKCLASS(theElement,RED_CLASS);
        break;

      /* TODO: these marks must be introduced first
         case BISECTION_3:
              if (side<0) return (GM_ERROR);
              SETMARK(theElement,TRI_BISECT_3+side%3);
              SETMARKCLASS(theElement,RED_CLASS);
              break;

         case BISECTION_1:
              if (side<0) return (GM_ERROR);
              SETMARK(theElement,TRI_BISECT_1+side%3);
              SETMARKCLASS(theElement,RED_CLASS);
              break;

         case BISECTION_2_Q:
              if (side<0) return (GM_ERROR);
              SETMARK(theElement,TRI_BISECT_2_Q+side%3);
              SETMARKCLASS(theElement,RED_CLASS);
              break;

         case BISECTION_2_T1:
              if (side<0) return (GM_ERROR);
              SETMARK(theElement,TRI_BISECT_2_T1+side%3);
              SETMARKCLASS(theElement,RED_CLASS);
              break;

         case BISECTION_2_T2:
              if (side<0) return (GM_ERROR);
              SETMARK(theElement,TRI_BISECT_2_T2+side%3);
              SETMARKCLASS(theElement,RED_CLASS);
              break;
       */

      default :
        return(GM_ERROR);
      }
      break;

    case (QUADRILATERAL) :
      switch (rule)
      {
      case COARSE :
        SETCOARSEN(theElement,1);
        SETMARKCLASS(theElement,0);
        SETMARK(theElement,NO_REFINEMENT);
        break;

      case NO_REFINEMENT :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        break;

      case COPY :
        SETMARK(theElement,Q_COPY);
        SETMARKCLASS(theElement,RED_CLASS);
        break;

      case RED :
        SETMARK(theElement,Q_RED);
        SETMARKCLASS(theElement,RED_CLASS);
        break;

      /* TODO: these mark must be introduced first */
      case BLUE :
        if (side<0) return (GM_ERROR);
        if (side%2 == 0)
          SETMARK(theElement,Q_BLUE_0);
        else
          SETMARK(theElement,Q_BLUE_1);
        SETMARKCLASS(theElement,RED_CLASS);
        break;

      default :
        return(GM_ERROR);
      }
      break;

    default :
      return(GM_ERROR);
    }
    break;
                        #endif /* __TWODIM__ */


                #ifdef __THREEDIM__
  /* 3D case */
  case (3) :
    switch (TAG(theElement))
    {
    case (TETRAHEDRON) :
      switch(rule)
      {
      case (COARSE) :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        SETCOARSEN(theElement,1);
        break;
      case (NO_REFINEMENT) :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        break;
      case (COPY) :
        SETMARK(theElement,TET_COPY);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (RED) :
        SETMARK(theElement,(*theFullRefRule)(theElement));
        SETMARKCLASS(theElement,RED_CLASS);
        break;
#ifndef TET_RULESET
      case (TETRA_RED_HEX) :
        SETMARK(theElement,TET_RED_HEX);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
#endif
      default :
        return(GM_ERROR);
      }
      break;

    case (PYRAMID) :
      switch(rule)
      {
      case (COARSE) :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        SETCOARSEN(theElement,1);
        break;
      case (NO_REFINEMENT) :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        break;
      case (COPY) :
        SETMARK(theElement,PYR_COPY);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (RED) :
        SETMARK(theElement,PYR_RED);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      default :
        return(GM_ERROR);
      }
      break;

    case (PRISM) :
      switch(rule)
      {
      case (COARSE) :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        SETCOARSEN(theElement,1);
        break;
      case (NO_REFINEMENT) :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        break;
      case (COPY) :
        SETMARK(theElement,PRI_COPY);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (RED) :
        SETMARKCLASS(theElement,RED_CLASS);

                                                        #ifdef __ANISOTROPIC__
        {
          INT newrule;

          if (GetRule_AnisotropicRed(theElement,&newrule))
          {
            SETMARK(theElement,newrule);
            break;
          }
        }
                                                        #endif

        SETMARK(theElement,PRI_RED);
        break;
      case (PRISM_BISECT_1_2) :
        SETMARK(theElement,PRI_BISECT_1_2);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_QUADSECT) :
        SETMARK(theElement,PRI_QUADSECT);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_BISECT_HEX0) :
        SETMARK(theElement,PRI_BISECT_HEX0);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_BISECT_HEX1) :
        SETMARK(theElement,PRI_BISECT_HEX1);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_BISECT_HEX2) :
        SETMARK(theElement,PRI_BISECT_HEX2);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_ROTATE_LEFT) :
        SETMARK(theElement,PRI_ROT_L);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_ROTATE_RGHT) :
        SETMARK(theElement,PRI_ROT_R);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_QUADSECT_HEXPRI0) :
        SETMARK(theElement,PRI_QUADSECT_HEXPRI0);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_BISECT_0_1) :
        SETMARK(theElement,PRI_BISECT_0_1);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_BISECT_0_2) :
        SETMARK(theElement,PRI_BISECT_0_2);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (PRISM_BISECT_0_3) :
        SETMARK(theElement,PRI_BISECT_0_3);
        SETMARKCLASS(theElement,RED_CLASS);
        break;

      default :
        return(GM_ERROR);
      }
      break;

    case (HEXAHEDRON) :
      switch(rule)
      {
      case (COARSE) :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        SETCOARSEN(theElement,1);
        break;
      case (NO_REFINEMENT) :
        SETMARK(theElement,NO_REFINEMENT);
        SETMARKCLASS(theElement,0);
        break;
      case (COPY) :
        SETMARK(theElement,HEXA_COPY);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (RED) :
        SETMARK(theElement,HEXA_RED);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_BISECT_0_1) :
        SETMARK(theElement,HEXA_BISECT_0_1);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_BISECT_0_2) :
        SETMARK(theElement,HEXA_BISECT_0_2);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_BISECT_0_3) :
        SETMARK(theElement,HEXA_BISECT_0_3);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_TRISECT_0) :
        SETMARK(theElement,HEXA_TRISECT_0);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_TRISECT_5) :
        SETMARK(theElement,HEXA_TRISECT_5);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_QUADSECT_0) :
        SETMARK(theElement,HEXA_QUADSECT_0);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_QUADSECT_1) :
        SETMARK(theElement,HEXA_QUADSECT_1);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_QUADSECT_2) :
        SETMARK(theElement,HEXA_QUADSECT_2);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_BISECT_HEXPRI0) :
        SETMARK(theElement,HEXA_BISECT_HEXPRI0);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      case (HEX_BISECT_HEXPRI1) :
        SETMARK(theElement,HEXA_BISECT_HEXPRI1);
        SETMARKCLASS(theElement,RED_CLASS);
        break;
      default :
        return(GM_ERROR);
      }
      break;

    default :
      return(GM_ERROR);
    }
    break;
                        #endif /* __THREEDIM__ */

  default :
    return(GM_ERROR);
  }

  return(GM_OK);
}

/****************************************************************************/
/** \brief Return true when element can be tagged for refinement

   \param theElement - element to refine

   This function returns true (1) when element can be tagged for refinement

   \return <ul>
   <li> false - do not tag element </li>
   <li> true - element can be tagged for refinement </li>
   </ul>
 */
/****************************************************************************/

INT NS_DIM_PREFIX EstimateHere (const ELEMENT *theElement)
{
        #ifdef ModelP
  if (EGHOST(theElement)) return(0);
        #endif
  return(LEAFELEM(theElement));
}


/****************************************************************************/
/** \brief Return mark of rule for a specific pattern

   \param theElement - element rule is searched for
   \param pattern: pattern a rule is searched for

   This function returns mark of rule for a specific pattern

   \return Mark rule; values of the unnamed enums in the file rm.h
   mark of rule
 */
/****************************************************************************/

INT NS_DIM_PREFIX Patterns2Rules(ELEMENT *theElement, INT pattern)
{
        #ifdef __TWODIM__
  switch (TAG(theElement)) {
  case (TRIANGLE) :
    switch (pattern) {
    /** \todo 0 can mean T_COPY OR T_NOREF */
    case (0) : return(T_NOREF);
    case (1) : return(T_BISECT_1_0);
    case (2) : return(T_BISECT_1_1);
    case (3) : return(T_BISECT_2_T1_2);
    case (4) : return(T_BISECT_1_2);
    case (5) : return(T_BISECT_2_T1_1);
    case (6) : return(T_BISECT_2_T1_0);
    case (7) : return(T_RED);
    default :
      assert(0);
      PrintErrorMessage('E',"Patterns2Rules","no mapping for TRIANGLE and this pattern!");
      return(-1);
    }
    break;
  case (QUADRILATERAL) :
    switch (pattern) {
    /** \todo 0 can mean Q_COPY OR Q_NOREF */
    case (0) : return(Q_NOREF);
    case (5) : return(Q_BLUE_0);
    case (7) : return(Q_CLOSE_3_3);
    case (10) : return(Q_BLUE_1);
    case (11) : return(Q_CLOSE_3_2);
    case (13) : return(Q_CLOSE_3_1);
    case (14) : return(Q_CLOSE_3_0);

    case (1) :                                    /* mapping for green closure */
    case (17) : return(Q_CLOSE_2_0);

    case (2) :                                    /* mapping for green closure */
    case (18) : return(Q_CLOSE_2_1);

    case (3) :                                    /* mapping for green closure */
    case (19) : return(Q_CLOSE_1_0);

    case (4) :                                    /* mapping for green closure */
    case (20) : return(Q_CLOSE_2_2);

    case (6) :                                    /* mapping for green closure */
    case (22) : return(Q_CLOSE_1_1);

    case (8) :                                    /* mapping for green closure */
    case (24) : return(Q_CLOSE_2_3);

    case (9) :                                    /* mapping for green closure */
    case (25) : return(Q_CLOSE_1_3);

    case (12) :                                    /* mapping for green closure */
    case (28) : return(Q_CLOSE_1_2);

    case (15) :                                    /* mapping for green closure */
    case (31) : return(Q_RED);
    default :
      assert(0);
      PrintErrorMessage('E',"Patterns2Rules","no mapping for QUADRILATERAL and this pattern!");
      return(-1);
    }
    break;
  default :
    PrintErrorMessage('E',"Patterns2Rules","Elementtype not found!");
    assert(0); return(-1);
  }
        #endif
        #ifdef __THREEDIM__
  switch (TAG(theElement)) {
  case (TETRAHEDRON) :
#ifdef TET_RULESET
    /* convert pattern to old style */
    pattern &= (~(1<<10));

    IFDEBUG(gm,0)
    int tetrarule;
    if (pattern<0 || pattern>1023)
      PRINTDEBUG(gm,0,(PFMT "Pattern2Rule(): ERROR elem=" EID_FMTX
                       " pattern=%d\n",me,EID_PRTX(theElement),pattern))
      assert(pattern>=0 && pattern<=1023);
    tetrarule = Pattern2Rule[TAG(theElement)][pattern];
    if (tetrarule<0 || tetrarule>MaxRules[TETRAHEDRON])
      PRINTDEBUG(gm,0,(PFMT "Pattern2Rule(): ERROR elem=" EID_FMTX
                       " pattern=%d rule=%d\n",me,EID_PRTX(theElement),pattern,tetrarule))
      assert(tetrarule>=0 && tetrarule<=MaxRules[TETRAHEDRON]);
    ENDDEBUG

    return(Pattern2Rule[TAG(theElement)][pattern]);
#else
    if (MARKCLASS(theElement) != RED_CLASS) return(0);
    switch (pattern) {
    /* copy rule */
    case (0) :
      return(0);
    case (63) :
      return(TET_RED);
    case (1023) :
      return(TET_RED_HEX);
    default :
      PrintErrorMessage('E',"Patterns2Rules","no mapping for TETRAHEDRON and this pattern!");
      assert(0); return(-1);
    }
    break;
#endif

  case (PYRAMID) :
    if (MARKCLASS(theElement) != RED_CLASS) return(0);
    switch (pattern) {
    /* copy rule */
    case (0) :
      return(0);
    case (511) :
      return(PYR_RED);
    default :
      PrintErrorMessage('E',"Patterns2Rules","no mapping for PYRAMID and this pattern!");
      assert(0); return(-1);
    }
    break;

  case (PRISM) :
    if (MARKCLASS(theElement) != RED_CLASS) return(0);
    switch (pattern) {
    /* copy rule */
    case (0) :
      return(0);
    case (7679) :
      return(PRI_RED);
    case (455) :
      return(PRI_QUADSECT);
    case 56 :
      return PRI_BISECT_1_2;
    case 65 :
      return PRI_BISECT_0_1;
    case 130 :
      return PRI_BISECT_0_2;
    case 260 :
      return PRI_BISECT_0_3;
    case 325 :
      return PRI_BISECT_HEX0;
    case 195 :
      return PRI_BISECT_HEX1;
    case 390 :
      return PRI_BISECT_HEX2;
    default :
      PrintErrorMessageF('E',"Patterns2Rules","no mapping for PRISM and pattern %d!", pattern);
                                                #ifndef __ANISOTROPIC__
      assert(0);
                                                #endif
      return(-1);
    }
    break;

  case (HEXAHEDRON) :
    if (MARKCLASS(theElement) != RED_CLASS) return(0);
    switch (pattern) {
    /* copy rule */
    case (0) :
      return(0);
    /* full red rule */
    case (262143) :
      return(HEXA_RED);
    case (1285) :
      return HEXA_BISECT_0_1;
    case (2570) :
      return HEXA_BISECT_0_2;
    case (240) :
      return HEXA_BISECT_0_3;
    case (139023) :
      return HEXA_QUADSECT_0;
    case (42485) :
      return HEXA_QUADSECT_1;
    case (84730) :
      return HEXA_QUADSECT_2;
    case (5) :
      return HEXA_TRISECT_0;
    case (1280) :
      return HEXA_TRISECT_5;
    case (2056) :
      return HEXA_BISECT_HEXPRI0;
    case (257) :
      return HEXA_BISECT_HEXPRI1;
    default :
      PrintErrorMessage('E',"Patterns2Rules","no mapping for HEXAHEDRON and this pattern!");
      UserWriteF("pattern=%d\n",pattern);
      assert(0); return(-1);
    }
    break;

  default :
    PrintErrorMessage('E',"Patterns2Rules","Elementtype not found!");
    assert(0); return(-1);
  }
        #endif
  PrintErrorMessage('E',"Patterns2Rules","Elementtype not found!");
  assert(0); return(-1);
}

/****************************************************************************/
/** \brief Gets the element which has to be marked

   \param MarkElement - element to be estimated

   This function gets the element which has to be marked

   \return <ul>
   <li> first element downward with class RED_CLASS </li>
   <li> NULL if MarkElement was no surface element </li>
   </ul>
 */
/****************************************************************************/

ELEMENT * NS_DIM_PREFIX ELEMENT_TO_MARK (ELEMENT *theElement)
{
  if (IS_REFINED(theElement)) return(NULL);

  PRINTDEBUG(gm,4,(PFMT "ELEMENT_TO_MARK() called for e=" EID_FMTX "\n",
                   me,EID_PRTX(theElement)))

  while (ECLASS(theElement) != RED_CLASS)
  {
    theElement = EFATHER(theElement);
    ASSERT(theElement != NULL);
  }

  return(theElement);
}


/****************************************************************************/
/** \brief Gets rule of refinement

   \param theElement - element to refine
   \param rule - filled with current refinement rule
   \param data - filled with side, if rule is oriented

   This function gets rule of refinement

   \return
   .n   0: side information valid
   .n   1: rule without orientation
 */
/****************************************************************************/

INT NS_DIM_PREFIX GetRefinementMark (ELEMENT *theElement, INT *rule, void *data)
{
  INT *side = (INT*)data;
  INT mark;

  if (LEAFELEM(theElement) &&
      ECLASS(theElement) != RED_CLASS)
    theElement = ELEMENT_TO_MARK(theElement);

  ASSERT(theElement != NULL);
  if (!((ECLASS(theElement) == RED_CLASS)
        && (REFINECLASS(theElement) != RED_CLASS)))
  {
    printf("GetRefinementMark: eclass=%d refineclass=%d\n",
           ECLASS(theElement),REFINECLASS(theElement));
    return(-1);
  }

  mark = MARK(theElement);

        #if defined(__THREEDIM__)
  /* tetrahedra have three red rules */
  if (TAG(theElement)==TETRAHEDRON &&
      (mark==TET_RED_2_4 || mark==TET_RED_0_5 || mark==TET_RED_1_3))
  {
    *rule=RED;
    return(GM_RULE_WITHOUT_ORIENTATION);
  }
        #endif

  switch (mark)
  {
  case RED :
    *rule=RED;
    break;
#ifdef __TWODIM__
  case Q_BLUE_0 :
    *rule = BLUE;
    break;
  case Q_BLUE_1 :
    *rule = BLUE;
    break;
#endif
  case NO_REFINEMENT :
    *rule=NO_REFINEMENT;
    if (COARSEN(theElement)) *rule = COARSE;
    break;
  case COPY :
    *rule=COPY; break;
  default :
    *rule=NO_REFINEMENT;  break;
  }
  *side=0;
  return(GM_RULE_WITHOUT_ORIENTATION);
}


/****************************************************************************/
/** \brief
   GetRefinementMarkType - gets type of mark for an element

   SYNOPSIS:
   INT GetRefinementMarkType (ELEMENT *theElement);

   PARAMETERS:
   \param theElement - element to refine

   DESCRIPTION:
   This function gets the type of mark for an element

   \return
   INT
   .n   0 if element is not marked
   .n   1 if element is marked for refinement
   .n   -1 if element is marked for coarsening
 */
/****************************************************************************/

INT NS_DIM_PREFIX GetRefinementMarkType (ELEMENT *theElement)
{
  INT rule;
  INT side;

  if (GetRefinementMark(theElement,&rule,&side) == -1) RETURN(GM_ERROR);

  switch (rule)
  {
  case RED :
#ifdef __TWODIM__
  case BLUE :
#endif
    return(1);
  case COPY :
  case NO_REFINEMENT :     return(0);
  case COARSE :            return(-1);
  default :                        assert(0);
  }

  return(0);
}

/****************************************************************************/
/*
   PrintSonData -

   SYNOPSIS:
   static INT PrintSonData(struct sondata theSonData);

   PARAMETERS:
   \param theSonData

   DESCRIPTION:

   \return
   INT
 */
/****************************************************************************/

static INT PrintSonData (struct sondata theSonData, PrintfProcPtr Printf)
{
  char buffer[128];
  int i,j;
  int path = theSonData.path;
  int pd = PATHDEPTH(path);

  Printf("tag=%d ",(int)theSonData.tag);

  j = 0;
  j = sprintf(buffer," corners=");
  for (i=0; i<element_descriptors[theSonData.tag]->corners_of_elem; i++)
  {
    j += sprintf(buffer+j,"%2d ",(int)theSonData.corners[i]);
  }
  Printf(buffer);

  j = 0;
  j += sprintf(buffer,"  nb=");
  for (i=0; i<element_descriptors[theSonData.tag]->sides_of_elem; i++)
  {
    j += sprintf(buffer+j,"%2d ",(int)theSonData.nb[i]);
  }
  Printf(buffer);

  Printf("  path of depth %d=",pd);
  /*for (i=8*sizeof(INT)-1; i>=0; i--)
     {
          sprintf(buffer,"%d",(int)((theSonData.path>>i) & 0x1));
          if (i%2 == 0 && theSonData.tag == TETRAHEDRON)	sprintf(buffer+1," ");
          if (i%3 == 0 && theSonData.tag == HEXAHEDRON)	sprintf(buffer+1," ");
          Printf(buffer);
     }*/
  if (pd>=MAX_PATH_DEPTH)
    Printf(" ERROR: path depth > MAX_PATH_DEPTH");
  else
    for (j=0; j<pd; j++)
    {
      int ns = NEXTSIDE(path,j);
      Printf("%2d",ns);
    }

  return(0);
}

/****************************************************************************/
/** \brief
   ShowRefRule -

   SYNOPSIS:
   INT ShowRefRule (INT tag, INT nb);

   PARAMETERS:
   \param tag
   \param nb

   DESCRIPTION:


   \return
   INT
 */
/****************************************************************************/

INT NS_DIM_PREFIX ShowRefRuleX (INT tag, INT nb, PrintfProcPtr Printf)
{
  INT i;
  REFRULE *theRule;

  if (MaxRules[tag]<=nb)
  {
    Printf("ShowRefRule(): ERROR: nb=%d but MaxRules[%d]=%d\n",nb,tag,MaxRules[tag]);
    return (1);
  }

  theRule=&(RefRules[tag][nb]);

  /* header */
  Printf("\n");
  Printf("RefRule %3d:\n",nb);

  /* nsons, mark and class */
  Printf("   tag=%d mark=%3d class=%2d, nsons=%d\n",(int)theRule->tag,(int)theRule->mark,(int)theRule->rclass,(int)theRule->nsons);

  /* pattern */
  Printf("   pattern= ");
  for (i=0; i<(element_descriptors[tag]->edges_of_elem+element_descriptors[tag]->sides_of_elem+1); i++)
    Printf("%2d ",(int)theRule->pattern[i]);
  Printf("\n");

  /* pat */
  Printf("   pat    = ");
  for (i=0; i<(element_descriptors[tag]->edges_of_elem+element_descriptors[tag]->sides_of_elem+1); i++)
    Printf("%2d ",(int)((theRule->pat>>i) & 0x1));
  Printf("\n");

  /* sonandnode */
  for (i=0; i<MAX_NEW_CORNERS(tag); i++)
  {
    Printf("   newnode %2d: sonandnode[%2d][0]=%2d",i,i,(int)theRule->sonandnode[i][0]);
    Printf("  [%2d][1]=%2d\n",i,(int)theRule->sonandnode[i][1]);
  }
  Printf("\n");

  /* print edge data
     Printf("   Edge data\n");
     for (i=0; i<MAX_NEW_EDGES(tag); i++)
     {
          Printf("      e %2d: ",i);
          PrintEdgeData(theRule->edges[i]);
          if (i%2 == 1) Printf("\n");
     }
     Printf("\n");*/

  /* print sondata data */
  Printf("   Son data\n");
  for (i=0; i<(int)theRule->nsons; i++)
  {
    Printf("      son %2d: ",i);
    PrintSonData(theRule->sons[i],Printf);
    Printf("\n");
  }

  return (0);
}

INT NS_DIM_PREFIX ShowRefRule (INT tag, INT nb)
{
  return (ShowRefRuleX(tag,nb,UserWriteF));
}

/****************************************************************************/
/** \brief
   FReadRule - Read the rule data set and initialize the rules data

   SYNOPSIS:
   static int FReadRule (FILE *stream, REFRULE *theRule);

   PARAMETERS:
   \param stream - file which stores rules
   \param theRule - pointer to rule structure

   DESCRIPTION:
   This function reads the rule data set and initializes the rules data

   \return
   INT
   .n   0 - ok
   .n   >0 - error
 */
/****************************************************************************/

#define TET_RULE_FATHER_SIDE_OFFSET             20

static int FReadRule (FILE *stream, REFRULE *theRule)
{
  int i,j;
  int ns,p0,p1,p2,p3,p4,p5,pat;
  int type,from,to,side;
  int c0,c1,c2,c3,n0,n1,n2,n3,pa;
  int sn0,sn1;

  /* init tag */
  theRule->tag = TETRAHEDRON;

  if (fscanf(stream,"%d  %d %d %d %d %d %d  %d",&ns,&p0,&p1,&p2,&p3,&p4,&p5,&pat)!=8) return (1);

  theRule->nsons = ns;
  theRule->pattern[0] = p0;
  theRule->pattern[1] = p1;
  theRule->pattern[2] = p2;
  theRule->pattern[3] = p3;
  theRule->pattern[4] = p4;
  theRule->pattern[5] = p5;
  theRule->pat = pat;

  /* MAXEDGES = 16 for tetrahedra */
  for (i=0; i<16; i++)
  {
    if (fscanf(stream," %d %d %d %d",&type,&from,&to,&side)!=4) return (1);
  }

  /* MAX_SONS = 12 for tetrahedra */
  for (i=0; i<12; i++)
  {
    if (fscanf(stream," %d %d %d %d %d %d %d %d %d",&c0,&c1,&c2,&c3,&n0,&n1,&n2,&n3,&pa)!=9) return (1);
    theRule->sons[i].tag = TETRAHEDRON;
    if (c0 == 10) c0 = 14;
    theRule->sons[i].corners[0] = c0;
    if (c1 == 10) c1 = 14;
    theRule->sons[i].corners[1] = c1;
    if (c2 == 10) c2 = 14;
    theRule->sons[i].corners[2] = c2;
    if (c3 == 10) c3 = 14;
    theRule->sons[i].corners[3] = c3;
    theRule->sons[i].nb[0]          = n0;
    theRule->sons[i].nb[1]          = n1;
    theRule->sons[i].nb[2]          = n2;
    theRule->sons[i].nb[3]          = n3;
    theRule->sons[i].path           = pa;
    for (j=0; j<SIDES_OF_TAG(TETRAHEDRON); j++)
      if (theRule->sons[i].nb[j]>=TET_RULE_FATHER_SIDE_OFFSET)
        theRule->sons[i].nb[j] += FATHER_SIDE_OFFSET-TET_RULE_FATHER_SIDE_OFFSET;
  }

  /* NEWCORNERS = 7 for tetrahedra */
  /* read edge node information */
  for (i=0; i<6; i++)
  {
    if (fscanf(stream," %d %d",&sn0,&sn1)!=2) return (1);
    theRule->sonandnode[i][0] = sn0;
    theRule->sonandnode[i][1] = sn1;
  }
  /* read center node information */
  if (fscanf(stream," %d %d",&sn0,&sn1)!=2) return (1);
  theRule->sonandnode[10][0] = sn0;
  theRule->sonandnode[10][1] = sn1;

  /* init Pattern and pat for center node */
  if (theRule->sonandnode[10][0] != NO_CENTER_NODE)
  {
    theRule->pattern[10] = 1;
    theRule->pat |= (1<<10);
  }

  return (0);
}


#ifdef __THREEDIM__

#ifdef TET_RULESET
/****************************************************************************/
/*
   CorrectRuleXX -

   SYNOPSIS:
   static int CorrectRule (REFRULE *theRule);

   PARAMETERS:
   \param theRule

   DESCRIPTION:

   \return
   int
 */
/****************************************************************************/

static int CorrectRule40 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<9; i++)
  {
    if (i==1 || i==4 || i==8) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule41 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<9; i++)
  {
    if (i==1 || i==4 || i==7) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule52 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<9; i++)
  {
    if (i==2 || i==8) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}
static int CorrectRule53 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<9; i++)
  {
    if (i==2 || i==7) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule85 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<9; i++)
  {
    if (i==4) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule86 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<9; i++)
  {
    if (i==4) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule111 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<9; i++)
  {
    if (i==6 || i==8) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule112 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<9; i++)
  {
    if (i==6 || i==7) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule135 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<12; i++)
  {
    if (i==1 || i==3 || i==11) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule136 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<12; i++)
  {
    if (i==2 || i==3 || i==11) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule155 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<10; i++)
  {
    if (i==5 || i==7 || i==9) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule156 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<10; i++)
  {
    if (i==5 || i==8 || i==9) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule183 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<12; i++)
  {
    if (i==3 || i==7 || i==11) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int CorrectRule184 (REFRULE *theRule)
{
  int i;
  int c0,n1;

  for (i=0; i<12; i++)
  {
    if (i==2 || i==8 || i==11) {
      c0 = theRule->sons[i].corners[0];
      theRule->sons[i].corners[0] = theRule->sons[i].corners[1];
      theRule->sons[i].corners[1] = c0;
      n1 = theRule->sons[i].nb[1];
      theRule->sons[i].nb[1] = theRule->sons[i].nb[2];
      theRule->sons[i].nb[2] = n1;
    }
  }

  return (0);
}

static int      CheckVolumes(REFRULE *Rule)
{
  INT i,j,k,tag;
  DOUBLE sp;
  DOUBLE_VECTOR coords [MAX_CORNERS_OF_ELEM+MAX_NEW_CORNERS_DIM];
  DOUBLE_VECTOR coord;
  DOUBLE_VECTOR p[MAX_CORNERS_OF_ELEM];

  /* this function work only for TETRAHEDRA!!! */
  tag = TETRAHEDRON;

  /* reset coords */
  for (i=0; i<MAX_CORNERS_OF_ELEM+MAX_NEW_CORNERS_DIM; i++)
    V3_CLEAR(coords[i])

    /* compute coordinates of corners */
    for (i=0,j=0; i<CORNERS_OF_REF(tag); i++,j++)
      V3_COPY(LOCAL_COORD_OF_REF(tag,i),coords[j])

      /* compute coordinates of midnodes */
      for (i=0; i<EDGES_OF_REF(tag); i++,j++)
      {
        V3_CLEAR(coord)
        for (k=0; k<CORNERS_OF_EDGE; k++)
          V3_ADD(LOCAL_COORD_OF_REF(tag,CORNER_OF_EDGE_REF(tag,i,k)),coord,coord)
          V3_SCALE(1.0/CORNERS_OF_EDGE,coord)
          V3_COPY(coord,coords[j]);
      }

  /* compute coordinates of sidenodes */
  for (i=0; i<SIDES_OF_REF(tag); i++,j++)
  {
    V3_CLEAR(coord)
    for (k=0; k<CORNERS_OF_SIDE_REF(tag,i); k++)
      V3_ADD(LOCAL_COORD_OF_REF(tag,CORNER_OF_SIDE_REF(tag,i,k)),coord,coord)
      V3_SCALE(1.0/CORNERS_OF_SIDE_REF(tag,i),coord)
      V3_COPY(coord,coords[j]);
  }

  /* compute coordinates of center node */
  V3_CLEAR(coord)
  for (i=0; i<CORNERS_OF_REF(tag); i++)
  {
    V3_ADD(LOCAL_COORD_OF_REF(tag,i),coord,coord)
  }
  V3_SCALE(1.0/CORNERS_OF_REF(tag),coord)
  V3_COPY(coord,coords[j]);

  if (0)
    for (i=0; i<MAX_CORNERS_OF_ELEM+MAX_NEW_CORNERS_DIM; i++)
    {
      UserWriteF("CheckVolumes(): i=%d x=%.8f y=%.8f z=%.8f\n",
                 i,coords[i][0],coords[i][1],coords[i][2]);
    }

  /* check the son volumes being really greater than zero */
  for (i=0; i<NSONS_OF_RULE(Rule); i++)
  {
    for (k=1; k<4; k++)
      V3_SUBTRACT(coords[SON_CORNER_OF_RULE(Rule,i,k)],coords[SON_CORNER_OF_RULE(Rule,i,0)],p[k])

      V3_VECTOR_PRODUCT(p[1],p[2],p[0])
      V3_SCALAR_PRODUCT(p[0],p[3],sp)

      if (sp<=0.0)
      {
        UserWriteF("negative volume=%f for son=%d rule=%d\n",sp,i,MARK_OF_RULE(Rule));
        assert(0);
      }
  }

  return(0);
}
#endif

/****************************************************************************/
/** \brief
   InitRuleManager3D - Read the rule data set and initialize the rules

   This function reads the rule data set and initializes the rules data structure for tetrahedrons. Initializes the regular refinement rules (red rules) for hexahedrons. Irregular refinement of green closure is done algorithmically.

   \return
   .n   0 if ok
   .n   >0 if an error occurs
 */
/****************************************************************************/

static INT InitRuleManager3D (void)
{
  FULLREFRULE *newFRR;
  int nRules;
        #ifdef TET_RULESET
  int nPatterns, i, P2R;
  FILE *stream;
  REFRULE *Rules;
  char buffer[256];
        #endif

  /************************************************************************/
  /*																		*/
  /* read refinement rules for tetrahedrons from file 'RefRules.data'		*/
  /*																		*/
  /************************************************************************/
#ifdef TET_RULESET
  /* open file */
        #ifdef ModelP
  if (me == master)
        #endif
  {
    if (GetDefaultValue(DEFAULTSFILENAME,"refrulefile",buffer)==0)
    {
      if (ExpandCShellVars(buffer)==NULL)
      {
        PrintErrorMessageF('W',"InitRuleManager3D","could not expand shell variables in 'refrulefile' of defaults file '%s'",DEFAULTSFILENAME);
        return (1);
      }
      stream = fileopen(buffer,"r");
    }
    else
      stream = fileopen("RefRules.data","r");
    if (stream==NULL)
    {
      UserWrite("ERROR: could not open file 'RefRules.data'\n");
      fclose(stream);
      return (__LINE__);
    }

    /* read Rules and nPatterns from file */
    if (fscanf(stream,"%d %d\n",&nRules,&nPatterns)!=2)
    {
      UserWrite("ERROR: failed to read Rules and nPatterns\n");
      fclose(stream);
      return (__LINE__);
    }
  }

        #ifdef ModelP
  Broadcast(&nRules,sizeof(nRules));
  Broadcast(&nPatterns,sizeof(nPatterns));
        #endif

  /* get storage for Rules */
  Rules = (REFRULE *) malloc(nRules*sizeof(REFRULE));
  if (Rules==NULL)
  {
    UserWrite("ERROR: no storage for Rules\n");
    fclose(stream);
    return (__LINE__);
  }

  /* get storage for Pattern2Rule */
  Pattern2Rule[TETRAHEDRON] = std::make_unique<SHORT[]>(nPatterns);
  if (Pattern2Rule[TETRAHEDRON]==nullptr)
  {
    UserWrite("ERROR: no storage for Pattern2Rule\n");
    fclose(stream);
    return (__LINE__);
  }
  for (i=0; i<nPatterns; i++) Pattern2Rule[TETRAHEDRON][i] = -1;

        #ifdef ModelP
  if (me == master)
        #endif
  {
    /* read Rules */
    for (i=0; i<nRules; i++)
    {
      Rules[i] = Empty_Rule;
      Rules[i].mark = i;
      Rules[i].rclass = RED_CLASS|GREEN_CLASS;
      if (FReadRule(stream,Rules+i)) return (__LINE__);
    }

    /* read Pattern2Rule */
    for (i=0; i<nPatterns; i++)
    {
      if (fscanf(stream,"%d",&P2R)!=1) return (__LINE__);
      Pattern2Rule[TETRAHEDRON][i] = P2R;
      PRINTDEBUG(gm,4,("Pattern2Rules[%4x]=%4d\n",i,P2R))
    }

    fclose(stream);

    /* bug fix */
    CorrectRule40(&Rules[40]);
    CorrectRule41(&Rules[41]);
    CorrectRule52(&Rules[52]);
    CorrectRule53(&Rules[53]);
    CorrectRule85(&Rules[85]);
    CorrectRule86(&Rules[86]);
    CorrectRule111(&Rules[111]);
    CorrectRule112(&Rules[112]);
    CorrectRule135(&Rules[135]);
    CorrectRule136(&Rules[136]);
    CorrectRule155(&Rules[155]);
    CorrectRule156(&Rules[156]);
    CorrectRule183(&Rules[183]);
    CorrectRule184(&Rules[184]);

    for (i=0; i<nRules; i++)
    {
      CheckVolumes(Rules+i);
    }
  }

        #ifdef ModelP
  Broadcast(Rules,nRules*sizeof(REFRULE));
  Broadcast(Pattern2Rule[TETRAHEDRON].get(),nPatterns*sizeof(SHORT));
        #endif
#else
  nRules = MAX_TET_RULES;
#endif

  /* now make rules for tetrahedrons globally available */
  MaxRules[TETRAHEDRON] = nRules;
  MaxNewCorners[TETRAHEDRON] = 11;
  MaxNewEdges[TETRAHEDRON] = 16;
  CenterNodeIndex[TETRAHEDRON] = 10;
#ifdef TET_RULESET
  RefRules[TETRAHEDRON] = Rules;
#else
  RefRules[TETRAHEDRON] = TetrahedronRules;
#endif


  /************************************************************************/
  /*																		*/
  /*  init refinement rules from for pyramids                           */
  /*																		*/
  /************************************************************************/

  nRules = MAX_PYR_RULES;

  /* make rules for pyramids globally available */
  MaxRules[PYRAMID] = nRules;
  MaxNewCorners[PYRAMID] = 19;
  MaxNewEdges[PYRAMID] = 54;
  CenterNodeIndex[PYRAMID] = 18;
  RefRules[PYRAMID] = PyramidRules;

  /************************************************************************/
  /*                                                                      */
  /*  init refinement rules for prisms                                    */
  /*                                                                      */
  /************************************************************************/

  nRules = MAX_PRI_RULES;

  /* make rules for prisms globally available */
  MaxRules[PRISM] = nRules;
  MaxNewCorners[PRISM] = 19;
  MaxNewEdges[PRISM] = 54;
  CenterNodeIndex[PRISM] = 18;
  RefRules[PRISM] = PrismRules;

  /************************************************************************/
  /*																		*/
  /*  init refinement rules from for hexahedrons                        */
  /*																		*/
  /************************************************************************/

  nRules = MAX_HEX_RULES;

  /* make rules for tetrahedrons globally available */
  MaxRules[HEXAHEDRON] = nRules;
  MaxNewCorners[HEXAHEDRON] = 19;
  MaxNewEdges[HEXAHEDRON] = 54;
  CenterNodeIndex[HEXAHEDRON] = 18;
  RefRules[HEXAHEDRON] = HexahedronRules;


  /************************************************************************/
  /*																		*/
  /* install best full refrules for tetrahedrons							*/
  /*																		*/
  /************************************************************************/

  /* install the /Menu directory */
  if (ChangeEnvDir("/")==NULL)
  {
    PrintErrorMessage('F',"InitRuleManager3D","could not changedir to root");
    return(__LINE__);
  }
  theBFRRDirID = GetNewEnvDirID();
  if (MakeEnvItem("best full refrule",theBFRRDirID,sizeof(ENVDIR))==NULL)
  {
    PrintErrorMessage('F',"InitRuleManager3D","could not install '/best full refrule' dir");
    return(__LINE__);
  }
  if (ChangeEnvDir("/best full refrule")==NULL)
    return(__LINE__);

  theBFRRVarID = GetNewEnvVarID();

  newFRR = (FULLREFRULE *) MakeEnvItem("shortestie",theBFRRVarID,sizeof(FULLREFRULE));
  if (newFRR==NULL)
    return(__LINE__);
  newFRR->theFullRefRule = ShortestInteriorEdge;

  newFRR = (FULLREFRULE *) MakeEnvItem("maxper",theBFRRVarID,sizeof(FULLREFRULE));
  if (newFRR==NULL)
    return(__LINE__);
  newFRR->theFullRefRule = MaxPerpendicular;

  newFRR = (FULLREFRULE *) MakeEnvItem("mra",theBFRRVarID,sizeof(FULLREFRULE));
  if (newFRR==NULL)
    return(__LINE__);
  newFRR->theFullRefRule = MaxRightAngle;

  newFRR = (FULLREFRULE *) MakeEnvItem("maxarea",theBFRRVarID,sizeof(FULLREFRULE));
  if (newFRR==NULL)
    return(__LINE__);
  newFRR->theFullRefRule = MaxArea;

#ifdef __ALLRULES__
  newFRR = (FULLREFRULE *) MakeEnvItem("minangle",theBFRRVarID,sizeof(FULLREFRULE));
  if (newFRR==NULL)
    return(__LINE__);
  newFRR->theFullRefRule = MinimalSideAngle;

  newFRR = (FULLREFRULE *) MakeEnvItem("bestm",theBFRRVarID,sizeof(FULLREFRULE));
  if (newFRR==NULL)
    return(__LINE__);
  newFRR->theFullRefRule = BestLaplaceMMatrix;

  newFRR = (FULLREFRULE *) MakeEnvItem("minentry",theBFRRVarID,sizeof(FULLREFRULE));
  if (newFRR==NULL)
    return(__LINE__);
  newFRR->theFullRefRule = MinimalSideEntry;
#endif

  /* default full refrule */
  theFullRefRule = ShortestInteriorEdge;

  UserWrite("3D RefRules installed\n");

  return (GM_OK);
}


/****************************************************************************/
/*
   SetAlignmentPtr -

   SYNOPSIS:
   INT SetAlignmentPtr (MULTIGRID *theMG, EVECTOR *direction);

   PARAMETERS:
   \param theMG
   \param direction

   DESCRIPTION:

   \return
   INT
 */
/****************************************************************************/

INT NS_DIM_PREFIX SetAlignmentPtr (MULTIGRID *theMG, EVECTOR *direction)
{
  if (direction != NULL)
  {
    if ((*(direction->PreprocessProc))(ENVITEM_NAME(direction),theMG)) return(1);
    theDirectionElemEval = direction->EvalProc;
    theFullRefRule = Alignment;
  }
  else
    theFullRefRule = ShortestInteriorEdge;

  return(0);
}

#endif /* __THREEDIM__ */

#ifdef __TWODIM__


/****************************************************************************/
/** \brief
   InitRuleManager2D - Initializes rules for triangles and quadrilaterals

   SYNOPSIS:
   static INT InitRuleManager2D (void);

   PARAMETERS:
   .  void

   DESCRIPTION:
   This function initializes rules for triangles and quadrilaterals

   \return
   INT
   .n   0 - ok
   .n   >0 - error
 */
/****************************************************************************/

static INT InitRuleManager2D (void)
{
  int nPatterns;

  /************************************************************************/
  /*																		*/
  /*  init refinement rules for triangles                                       */
  /*																		*/
  /************************************************************************/

  /* get storage for Pattern2Rule */
  nPatterns = 17;       /* there are 2^3 different patterns */
  /** \todo delete all concerning Pattern2Rule */
  Pattern2Rule[TRIANGLE] = std::make_unique<SHORT[]>(nPatterns);
  if (Pattern2Rule[TRIANGLE] == nullptr)
  {
    UserWrite("ERROR: no storage for Pattern2Rule\n");
    return (__LINE__);
  }

  /* Pattern2Rule gives the starting index for rules with same pattern */
  Pattern2Rule[TRIANGLE][0] = T_COPY;                           /* 0 0 0 */
  Pattern2Rule[TRIANGLE][1] = T_BISECT_1_0;             /* 0 0 1 */
  Pattern2Rule[TRIANGLE][2] = T_BISECT_1_1;             /* 0 1 0 */
  Pattern2Rule[TRIANGLE][3] = T_BISECT_2_T1_0;          /* 0 1 1 */
  Pattern2Rule[TRIANGLE][4] = T_BISECT_1_2;                     /* 1 0 0 */
  Pattern2Rule[TRIANGLE][5] = NOINDEX;                  /* 1 0 1 */
  Pattern2Rule[TRIANGLE][6] = NOINDEX;                  /* 1 1 0 */
  Pattern2Rule[TRIANGLE][7] = T_RED;                            /* 1 1 1 */

  /* now make rules for triangles globally available */
  MaxRules[TRIANGLE] = MAX_TRI_RULES;
  MaxNewCorners[TRIANGLE] = 3;
  MaxNewEdges[TRIANGLE] = 9;
  CenterNodeIndex[TRIANGLE] = 4;
  RefRules[TRIANGLE] = TriangleRules;


  /************************************************************************/
  /*																		*/
  /*  init refinement rules for quadrilaterals                          */
  /*																		*/
  /************************************************************************/

  /* get storage for Pattern2Rule */
  nPatterns = 32;       /* there are 2^5 different patterns */
  /** \todo delete all concerning Pattern2Rule */
  Pattern2Rule[QUADRILATERAL] = std::make_unique<SHORT[]>(nPatterns);
  if (Pattern2Rule[QUADRILATERAL] == nullptr)
  {
    UserWrite("ERROR: no storage for Pattern2Rule\n");
    return (__LINE__);
  }

  /* Pattern2Rule gives the starting index for rules with same pattern */
  Pattern2Rule[QUADRILATERAL][ 0] = NOINDEX;                    /* 0 0 0 0 0 */
  Pattern2Rule[QUADRILATERAL][ 1] = NOINDEX;                    /* 0 0 0 0 1 */
  Pattern2Rule[QUADRILATERAL][ 2] = NOINDEX;                    /* 0 0 0 1 0 */
  Pattern2Rule[QUADRILATERAL][ 3] = NOINDEX;                    /* 0 0 0 1 1 */
  Pattern2Rule[QUADRILATERAL][ 4] = NOINDEX;                    /* 0 0 1 0 0 */
  Pattern2Rule[QUADRILATERAL][ 5] = NOINDEX;                    /* 0 0 1 0 1 */
  Pattern2Rule[QUADRILATERAL][ 6] = NOINDEX;                    /* 0 0 1 1 0 */
  Pattern2Rule[QUADRILATERAL][ 7] = NOINDEX;                    /* 0 0 1 1 1 */
  Pattern2Rule[QUADRILATERAL][ 8] = NOINDEX;                    /* 0 1 0 0 0 */
  Pattern2Rule[QUADRILATERAL][ 9] = NOINDEX;                    /* 0 1 0 0 1 */
  Pattern2Rule[QUADRILATERAL][10] = NOINDEX;                    /* 0 1 0 1 0 */
  Pattern2Rule[QUADRILATERAL][11] = NOINDEX;                    /* 0 1 0 1 1 */
  Pattern2Rule[QUADRILATERAL][12] = NOINDEX;                    /* 0 1 1 0 0 */
  Pattern2Rule[QUADRILATERAL][13] = NOINDEX;                    /* 0 1 1 0 1 */
  Pattern2Rule[QUADRILATERAL][14] = NOINDEX;                    /* 0 1 1 1 0 */
  Pattern2Rule[QUADRILATERAL][15] = NOINDEX;                    /* 0 1 1 1 1 */
  Pattern2Rule[QUADRILATERAL][16] = NOINDEX;                    /* 1 0 0 0 0 */
  Pattern2Rule[QUADRILATERAL][17] = NOINDEX;                    /* 1 0 0 0 1 */
  Pattern2Rule[QUADRILATERAL][18] = NOINDEX;                    /* 1 0 0 1 0 */
  Pattern2Rule[QUADRILATERAL][19] = NOINDEX;                    /* 1 0 0 1 1 */
  Pattern2Rule[QUADRILATERAL][20] = NOINDEX;                    /* 1 0 1 0 0 */
  Pattern2Rule[QUADRILATERAL][21] = NOINDEX;                    /* 1 0 1 0 1 */
  Pattern2Rule[QUADRILATERAL][22] = NOINDEX;                    /* 1 0 1 1 0 */
  Pattern2Rule[QUADRILATERAL][23] = NOINDEX;                    /* 1 0 1 1 1 */
  Pattern2Rule[QUADRILATERAL][24] = NOINDEX;                    /* 1 1 0 0 0 */
  Pattern2Rule[QUADRILATERAL][25] = NOINDEX;                    /* 1 1 0 0 1 */
  Pattern2Rule[QUADRILATERAL][26] = NOINDEX;                    /* 1 1 0 1 0 */
  Pattern2Rule[QUADRILATERAL][27] = NOINDEX;                    /* 1 1 0 1 1 */
  Pattern2Rule[QUADRILATERAL][28] = NOINDEX;                    /* 1 1 1 0 0 */
  Pattern2Rule[QUADRILATERAL][29] = NOINDEX;                    /* 1 1 1 0 1 */
  Pattern2Rule[QUADRILATERAL][30] = NOINDEX;                    /* 1 1 1 1 0 */
  Pattern2Rule[QUADRILATERAL][31] = Q_RED;                      /* 1 1 1 1 1 */

  /* now make rules for quadrilaterals globally available */
  MaxRules[QUADRILATERAL] = MAX_QUA_RULES;
  MaxNewCorners[QUADRILATERAL] = 4;
  MaxNewEdges[QUADRILATERAL] = 12;
  CenterNodeIndex[QUADRILATERAL] = 4;
  RefRules[QUADRILATERAL] = QuadrilateralRules;

  return (GM_OK);
}
#endif /* __TWODIM__ */


/****************************************************************************/
/** \brief InitRuleManager Initialize the 2- or 3D rule set

   This function initialize the 2- or 3D rule set
 */
/****************************************************************************/

INT NS_DIM_PREFIX InitRuleManager (void)
{
  INT err;

  /* init 2- or 3D rulemanager */
  if ((err=CONCAT(InitRuleManager,DIM,D) ()) != GM_OK)
  {
    SetHiWrd(err,__LINE__);
    return(err);
  }
  return (0);
}
