package Emdebian::Tools;
use Carp;
use warnings;
use strict;
use File::HomeDir;
use Cwd;
use Cache::Apt::Config;
use Cache::Apt::Lookup;
use Cache::Apt::Package;
use Debian::Debhelper::Dh_Lib;
use Text::Diff;
use IO::Uncompress::Gunzip qw(gunzip $GunzipError) ;
use Parse::DebControl;
use Exporter;

=pod

=head1 Name

Emdebian::Tools - Support functions for emdebian-tools

=head1 Copyright and Licence

 Copyright (C) 2006-2008  Neil Williams <codehelp@debian.org>

 This package is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

=cut

use vars qw(@ISA @EXPORT);
@ISA=qw(Exporter);
@EXPORT=qw(emdeb_versionstring extract_emdebversion bug_config
check_emdebian_control create_patches get_dpkg_cross_dir
check_toolchains find_latest_gcc find_latest_libc get_targetsuite
check_dist host_arch get_username get_workdir check_workdir
prepare_checklist get_aptagent tools_version prepare_pristine
parse_xcontrol lookup_sourcepkg target_config lookup_dependencies
get_standards_version get_primary write_target_config );

my $OURVERSION = "1.0.0";

my @extradirs=qw/rules.d config xsfbs patches-applied debhelper.in /;

=head1 tools_version

all scripts (even bash ones) call this function to get the current
version string for the tools.

Parses the Debian changelog file and retrieves the most recent
version.

=cut

sub tools_version {
	my $input = "/usr/share/doc/libemdebian-tools-perl/changelog.gz";
	my $z = new IO::Uncompress::Gunzip $input
		or carp "IO::Uncompress::Gunzip failed: $GunzipError\n";
	my @output = <$z>;
	my @list=();
	foreach my $str (@output)
	{
		if ($str =~ /^emdebian-tools\s\((.*)\)\s[a-z]*;.*\n$/)
		{
			push @list, $1;
		}
	}
	return $list[0] if @list;
	return $OURVERSION;
}

=head1 get_config

Reads in the values set by debconf - used particularly by emsource
but available to all tools.

=cut

sub get_config
{
	my $config;
	my $dpkg_cross_dir = &get_dpkg_cross_dir;
	# allow user-specific overrides
	my $cfile = $dpkg_cross_dir . "/emsource";
	# otherwise use the debconf default.
	$cfile = "/etc/emsource.conf" if (! -f $cfile);
	if ( -f $cfile)
	{
		# read emsource config file, if any.
		$config = Config::Auto::parse("$cfile", format => "colon");
	}
	# collect emsandbox config into the same hash
	$cfile = $dpkg_cross_dir . "/emsandbox";
	$cfile = "/etc/emsandbox.conf" if (! -f $cfile);
	if ( -f $cfile)
	{
		my $sandbox = Config::Auto::parse("$cfile", format => "colon");
		$config->{'workingdir'} = $sandbox->{'workingdir'};
		# only override if tools not installed
		if (not defined $config->{'targetsuite'})
		{
			$config->{'targetsuite'} = $sandbox->{'targetsuite'};
		}
		# if tools is installed and suite differs, carp.
		# shouldn't really happen.
		elsif ($config->{'targetsuite'} ne $sandbox->{'targetsuite'})
		{
			carp ("Conflict between targetsuite setting of emdebian-tools".
			" and emdebian-rootfs! Use dpkg-reconfigure to set the same".
			" value in both packages. Using emdebian-tools setting.");
		}
	}
	return $config;
}

=head1 get_username

Returns the Emdebian SVN username configured by debconf or an empty string.

=cut

sub get_username
{
	my $username = "";
	my $config = &get_config();
	$username = $config->{'username'} if ($config->{'username'} ne "");
	return $username;
}

=head1 get_workdir

Returns the Emdebian SVN working directory configured by debconf or
the default working directory /

=cut

sub get_workdir
{
	my $workdir = "";
	my $config = &get_config();
	$workdir = $config->{'workingdir'} if ($config->{'workingdir'} ne "");
	return $workdir;
}

=head1 get_aptagent

Returns the debconf selection of whether to use apt-get or aptitude to
install packages from the Emdebian repository.

=cut

sub get_aptagent
{
	my $aptagent = "apt-get";
	my $config = &get_config();
	croak ("missing aptagent entry in ~/.apt-cross/emsource - see /etc/emsource.conf")
		if (!defined($config->{'aptagent'}));
	$aptagent = "aptitude" if ($config->{'aptagent'} eq "false");
	return $aptagent;
}

=head1 get_primary

emdebian-tools needs to be able to query apt cache data from a Debian mirror
that supports all cross-building architectures - these repositories are called
'primary mirrors' in Debian. If /etc/apt/sources.list does not contain a primary
Debian mirror, a primary mirror will have been configured in /etc/emsource.conf
using debconf.

get_primary returns the name of that mirror or undef if a primary already exists
- to use with 'apt-cross -m', wrap the returned value in 'ftp://$val/debian'

 my $val = &get_primary;
 my $mirror = "";
 $mirror = "-m ftp://$val/debian" if (defined $val);
 system ("apt-cross $mirror -u");

The default value is ftp.fr.debian.org so the default mirror to pass to
apt-cross is:
 apt-cross -v -m ftp://ftp.fr.debian.org/debian -u

=cut

sub get_primary
{
	my $primary;
	my $config = &get_config();
	$primary = $config->{'primary'};
	return undef if ($primary eq "");
	return $primary;
}

=head1 get_targetsuite

Returns the debconf selection of the default target suite. Falls back
to the previous apt-cross method reliant on apt-cache policy if no
value is retrieved from debconf. Also sets this value for all
calls to Cache::Apt:: for this process.

The specific repository codename must be supported by all
sources in your sources.list. debconf only supports the unchanging
codenames: unstable, testing and stable.

=cut

sub get_targetsuite
{
	my $suite  = "";
	my $config = &get_config();
	if (not defined($config->{'targetsuite'}))
	{
		my $msg = "missing target suite setting in ~/.apt-cross/emsource";
		$msg .=  " - see /etc/emsource.conf. Trying apt-cross method";
		carp ($msg);
		$suite = &get_suite;
	}
	else
	{
		$suite = $config->{'targetsuite'};
	}
	&set_suite($suite);
	return $suite;
}

=head1 check_workdir

Ensures the new directory is writable and tries to create it
if it does not already exist. If that fails, return a message
and let the script decide how to handle the error.

=cut

sub check_workdir
{
	my $workdir = $_[0];
	my $msg = "Working directory '$workdir' exists but is not writable! Use\n";
	$msg .= "'sudo dpkg-reconfigure emdebian-tools'\n to specify a writable location.\n";
	$msg .= "See emsource(1) or emsandbox(1) for more information.\n";
	# avoid dying within the module, let the script decide whether to die.
	return $msg if ((! -w $workdir) and (-d $workdir));
	system ("mkdir -p $workdir") if (! -d $workdir);
	$msg = "Working directory has been specified as '$workdir' but it does not exist\n";
	$msg .= "and cannot be created using 'mkdir -p'!: $!\n";
	$msg .= "See emsource(1) or emsandbox(1) for more information.\n";
	# avoid dying within the module, let the script decide whether to die.
	return $msg if (! -d $workdir);
	return "";
}

=head1 emdeb_versionstring($)

Generates the emdebian version string, appended to the dh{VERSION}

Pass "new" for a new upstream package or "next" for another release
of the same upstream package or blank to get the complete version
string.

=cut

sub emdeb_versionstring {
	my $emdebvers = "";
	my $debvers = $dh{VERSION};
	if ($debvers =~ /(.*)(em[0-9]+)$/) {
		$debvers = $1;
		$emdebvers = $2;
		$emdebvers =~ /^em([0-9]+)$/;
		my $emN = $1;
		$emN = 1 if (eval($emN) == 0);
		if ($_[0] eq "new") {
			$emN = 1;
			$emdebvers = "em$emN";
		}
		if ($_[0] eq "next") {
			$emN++;
			$emdebvers = "em$emN";
		}
	}
	else {
		$emdebvers="em1";
	}
	return "${debvers}${emdebvers}";
}

=head1 extract_emdebversion($)

Return just the emN part of the version string.

=cut

sub extract_emdebversion {
	if ($_[0] =~ /^Version: (.*)(em[0-9]+)$/) {
		return $2;
	}
	else { return ""; }
}

=head1 check_emdebian_control

Check the current location is a debian package.
Sets the current working directory as the directory above debian/

=cut

sub check_emdebian_control()
{
	my $pkg;
	# check this is a debian working directory
	# read debian/control
	# parse for locale packages.
	until (-f "debian/control")
	{
		chdir ".." or croak "Cannot change directory ../ $!";
		if (cwd() eq '/')
		{
			croak "Cannot find debian/control anywhere!\nAre you in the source code tree?\n";
		}
	}
}

=head1 create_patches($)

Creates and updates patches for all files in debian/ for $package

 CARE: takes the current version from the current working directory.
 Need to run &prepare_patches first to put pristine originals in ../$package.old/

Creates debian/xcontrol in $package.old as it will not usually
exist otherwise. Diff is only created if xcontrol is needed.

=cut

sub create_patches
{
	my $package = shift;
	my $diff = '';
	my $cwd = cwd;
	opendir (DEBIAN, "debian") or croak ("Cannot open debian directory: $!");
	 my @patchfiles=grep(!/^\.\.?$/, readdir DEBIAN);
	closedir (DEBIAN);
	foreach my $file (@patchfiles)
	{
		# skip directories
		next if ( -d "debian/$file");
		# skip binary files (yes, really! - see sysvinit)
		next if ( -B "debian/$file");
		if (! -f "../$package.old/debian/xcontrol")
		{
			open (XCONTROL, ">../$package.old/debian/xcontrol")
				or carp ("Cannot create empty xcontrol file : $!\n");
			close (XCONTROL);
		}
		# ignore generated files.
		if (( $file =~ /\.orig$/) or ( $file =~ /\.rej$/)
			or ($file =~ /\.debhelper$/) or ($file =~ /\.substvars$/)
			or ($file =~ /\.debhelper\.log$/) or ($file =~ /^files$/))
		{
			unlink ("../emdebian-$file.patch") if (-f "../emdebian-$file.patch");
			next;
		}
		if ((! -f "../$package.old/debian/$file") && (! -d "../$package.old/debian/$file"))
		{
			# create a blank
			open (BLANK, ">../$package.old/debian/$file")
				or carp ("Cannot create empty $file : $!\n");
			close (BLANK);
		}
		chdir ("../");
		$diff = diff "$package.old/debian/$file", "$cwd/debian/$file", { STYLE => "Unified" };
		# cancel the patch file if it is no longer needed
		if (( -f "emdebian-$file.patch") or ($diff ne ''))
		{
			open (PATCHFILE, ">emdebian-$file.patch") or
				carp ("Cannot open emdebian-$file.patch : $!\n");
			print PATCHFILE $diff;
			close (PATCHFILE);
		}
		if (( -f "emdebian-$file.patch") and ($diff eq ''))
		{
			unlink ("emdebian-$file.patch");
		}
		chdir ("$cwd");
	}
	# handle changeovers to new code nicely.
	foreach my $d (@extradirs)
	{
		if (( -d "debian/$d") && ( ! -d "../$package.old/debian/$d"))
		{
			my $carp  = "$OURVERSION update: no pristine source directory ";
			$carp .= "../$package.old/debian/$d\n";
			$carp .= "changes to files in debian/$d cannot be patched automatically.";
			carp ("$carp");
		}
		if (( -d "debian/$d") && ( -d "../$package.old/debian/$d"))
		{
			opendir (DEBIAN, "debian/$d") or
				croak ("Cannot open the existing debian/$d directory: $!");
			@patchfiles=grep(!/^\.\.?$/, readdir DEBIAN);
			closedir (DEBIAN);
			# diff modifications in debian/$d into new patch files for SVN
			foreach my $file (@patchfiles)
			{
				# files must exist in both locations.
				next if (! -f "../$package.old/debian/$d/$file");
				next if (! -f "debian/$d/$file");
				chdir ("../");
				$diff = diff "$package.old/debian/$d/$file", "$cwd/debian/$d/$file", { STYLE => "Unified" };
				# don't create zero length patch files
				if ($diff ne '')
				{
					open (PATCHFILE, ">emdebian-$d-$file.patch") or
						carp ("Cannot open emdebian-$d-$file.patch : $!\n");
					print PATCHFILE $diff;
					close (PATCHFILE);
				}
				# instead remove the patch file if no diff.
				else { unlink ("emdebian-$d-$file.patch"); }
				chdir ("$cwd");
			}
		}
	}
	# copy new patches in debian/patches into new debian-patch files for SVN
	# emsource copies in the opposite direction.
	my $patchdir;
	$patchdir = "debian/patches-applied" if (-d "debian/patches-applied");
	$patchdir = "debian/patches" if (-d "debian/patches");
	return if (not defined $patchdir);
	opendir (DEBIAN, $patchdir) or croak ("Cannot open debian patch directory: $!");
	@patchfiles=grep(!/^\.\.?$/, readdir DEBIAN);
	closedir (DEBIAN);
	foreach my $file (@patchfiles)
	{
		next if (-d "$patchdir/$file");
		# skip unchanged upstream patches
		if (-f "../$package.old/$patchdir/$file")
		{
			chdir ("../");
			$diff = '';
			$diff = diff "$package.old/$patchdir/$file", 
				"$cwd/$patchdir/$file", { STYLE => "Unified" };
			chdir ("$cwd");
			next if ($diff eq '');
		}
		# skip CDBS patch log files
		next if ($file =~ /\.level-[0-9]\.log$/);
		chdir ("../");
		open (DPATCH, "$cwd/$patchdir/$file") or carp ("Cannot read source patch: $!\n");
		my @p=<DPATCH>;
		close (DPATCH);
		open (PATCH, ">debian-patch-$file") or carp ("Cannot open debian-patch-$file : $!\n");
		print PATCH @p;
		close (PATCH);
		chdir ("$cwd");
	}
}

=head1 get_dpkg_cross_dir

chroot-safe method of determining the userspace directory to be used
for the cache files.

=cut

sub get_dpkg_cross_dir()
{
	my $home = File::HomeDir->my_home;
	# safeguard, just in case.
	$home = "/tmp" if (!$home);
	my $path =  "$home/.apt-cross";
	mkdir $path if (! -d $path);
	return $path;
}

=head1 prepare_checklist($)

Prepares an EXACT string for each of all the required toolchain
packages for the specified architecture - returns a reference to
an array of the package names.

 use Text::Wrap;
 my $list = &prepare_checklist($arch, $target_gnu_type);
 print wrap('','', @$list);

=cut

sub prepare_checklist()
{
	my $arch = $_[0];
	my $target_gnu_type = $_[1];
	my $suite = &get_suite;
	my $gcc_latest =  &find_latest_gcc("gcc", $arch, $suite);
	my $gcc_vers = "gcc-" . $gcc_latest;
	my $libc_latest = &find_latest_libc("libc", $arch, $suite);
	my $libc_vers = "libc" . $libc_latest;
	push my @list ,"binutils-${target_gnu_type}";
	push @list, "${gcc_vers}-${target_gnu_type}-base";
	push @list, "${gcc_vers}-${target_gnu_type}";
	push @list, "cpp-${gcc_latest}-${target_gnu_type}";
	push @list, "g++-${gcc_latest}-${target_gnu_type}";
	push @list, "${libc_vers}-${arch}-cross";
	push @list, "${libc_vers}-dev-${arch}-cross";
	push @list, "libstdc++${libc_latest}-${arch}-cross";
	push @list, "libstdc++${libc_latest}-${gcc_latest}-dev-${arch}-cross";
	push @list, "libstdc++${libc_latest}-${gcc_latest}-pic-${arch}-cross";
	push @list, "libgcc1-${arch}-cross";
	push @list, &check_linux($arch);
	my $multilib;
	my $our_arch = `dpkg-architecture -qDEB_BUILD_ARCH`;
	chomp ($our_arch);
	$multilib = "libc6-dev-i386" if ($our_arch =~ /^amd64$/);
	push @list, $multilib if (defined $multilib);
	return \@list;
}

=head1 check_linux($)

linux-kernel-headers has been replaced by linux-libc-dev but older
systems still require linux-kernel-headers.

&check_linux($arch) returns the name of the alternative to use
for the specified architecture.

If linux-kernel-headers is already installed on the build machine,
require linux-kernel-headers-${arch}-cross.
If not, require linux-libc-dev-${arch}-cross

=cut

sub check_linux ()
{
	my $arch = $_[0];
	my $query = `dpkg-query -W -f=' \${Package} \${Status}' linux-kernel-headers 2>/dev/null`;
	return "linux-kernel-headers-${arch}-cross"
		if ($query eq " linux-kernel-headers install ok installed");
	return "linux-libc-dev-${arch}-cross";
}

=head1 check_toolchains ($$)

 arg 1 : architecture. arg 2: target_gnu_type

Creates the list of toolchain packages with &prepare_checklist
and then queries dpkg to see which are actually installed. The check
does NOT depend on specific versions except that gcc-4.1 will fail
the check if gcc-4.2 is the latest version available, etc.

=cut

sub check_toolchains()
{
	my $arch = $_[0];
	my $target = $_[1];
	my $retval = "true";
	my $list = &prepare_checklist($arch, $target);
	my $status = 'install ok installed';
	my $success = "";
	my $string = " ";
	foreach my $res (sort @$list)
	{
		$success = `dpkg-query -W -f='\${Package} \${Status}' $res 2>/dev/null`;
		$string = "$res $status";
		$retval = "false" if ($success ne $string);
	}
	return $retval;
}

=head1 find_latest_gcc($$)

Only use where the package NAME contains a mathematically correct
version, e.g. gcc-3.4 vs gcc-4.1, libc6 vs libc7 etc.
returns 0 if no gcc package can be found in the cache.

Calling binlookup does slow things down a bit but it ensures that a
real binary package exists, not just a metapackage. binlookup is only
called if the iterator specifies a package name that matches the
gcc regular expression.

=cut

sub find_latest_gcc()
{
	my $arch = $_[1];
	my $suite = $_[2];
	my %h = ();
	my $pkg;
	my $dpkg_cross_dir = &get_cachedir;
	&set_suite($suite);
	&check_cache_arch($arch);
	my $config = &init_cache(0);
	my $iter = &get_cache_iter();
	do {
		$pkg = $iter->next;
		$h{$pkg}++ if (($pkg) and ($pkg =~ /gcc-([0-9\.\-]+)$/) and (&binlookup($pkg)));
	} while ($pkg);
	my @list = sort keys (%h);
	my $choice = 0;
	foreach my $line (@list)
	{
		if ($line =~ /gcc-([0-9\.\-]+)$/)
		{
			if ($1 > $choice) { $choice = $1; }
		}
	}
	return $choice;
}

=head1 find_latest_libc($$)

Only use where the package NAME contains a mathematically correct
version, e.g. gcc-3.4 vs gcc-4.1, libc6 vs libc7 etc.

Calling binlookup does slow things down a bit but it ensures that a
real binary package exists, not just a metapackage. binlookup is only
called if the iterator specifies a package name that matches the
libc regular expression.

=cut

sub find_latest_libc()
{
	my $arch = $_[1];
	my $suite = $_[2];
	my %h = ();
	my $pkg;
	my $dpkg_cross_dir = &get_cachedir;
	&set_suite($suite);
	&check_cache_arch($arch);
	my $config = &init_cache(0);
	my $iter = &get_cache_iter();
	do {
		$pkg = $iter->next;
		$h{$pkg}++ if (($pkg) and ($pkg =~ /libc([0-9]+)$/) and (&binlookup($pkg)));
	} while ($pkg);
	my @list = sort keys (%h);
	my $choice = 0;
	foreach my $line (@list)
	{
		if ($line =~ /libc([0-9]+)$/)
		{
			if ($1 > $choice) { $choice = $1; }
		}
	}
	return $choice;
}

=head1 host_arch

Shorthand to get the host architecture from dpkg-architecture
without the terminal newline.

=cut

sub host_arch()
{
	my $result = `dpkg-architecture -qDEB_HOST_ARCH`;
	chomp ($result);
	return $result;
}

=head1 prepare_pristine

Used by emsource and em_make to prepare a safe copy of the debian/
directory so that the patches can be prepared against the originals later.

Once created, ../$package.old is not overwritten. emsource will remove
it if the --clean option is used.

=cut

sub prepare_pristine
{
	my $package = $_[0];
	my @control;
	my $cwd = cwd;
	# keep the originals pristine - need to remove the .old if the upstream is new.
	return if (-d "../${package}.old/");
	# may need to copy all directories, as long as this is a clean debian directory.
	mkdir "../$package.old", 0755;
	mkdir "../$package.old/debian", 0755;
	foreach my $extra (@extradirs)
	{
		mkdir "../$package.old/debian/$extra", 0755 if (-d "debian/$extra");
	}
	mkdir "../$package.old/debian/patches", 0755;
	mkdir "../$package.old/debian/patches-applied", 0755 
		if (-d "debian/patches-applied");
	# read all files from the debian/ directory.
	opendir (DEBIAN, "debian") or croak ("Cannot open $cwd/debian directory: $!");
	my @patchfiles=grep(!/^\.\.?$/, readdir DEBIAN);
	closedir (DEBIAN);
	foreach my $file (@patchfiles)
	{
		# skip directories or other unwanted entries
		next if (! -f "debian/$file");
		open (CPY, "debian/$file") or die "Cannot open $file: $!";
		@control=<CPY>;
		close CPY;
		chdir ("../$package.old/");
		open (CPY, ">debian/$file") or die "Cannot write $file: $!";
		print CPY @control;
		close CPY;
		chdir ("$cwd");
	}
	# back up existing files in debian/rules.d (if any) {gcc}
	# or config {busybox} or xsfbs {libx11}
	foreach my $extra (@extradirs)
	{
		if ( -d "debian/$extra")
		{
			opendir (DEBIAN, "debian/$extra") or 
				die ("Cannot open debian $extra directory: $!");
			@patchfiles=grep(!/^\.\.?$/, readdir DEBIAN);
			closedir (DEBIAN);
			foreach my $file (@patchfiles)
			{
				# skip empty directories
				next if (! -f "debian/$extra/$file");
				open (CPY, "debian/$extra/$file") or die "Cannot open $file: $!";
				@control=<CPY>;
				close CPY;
				chdir ("../$package.old/");
				open (CPY, ">debian/$extra/$file") or die "Cannot write $file: $!";
				print CPY @control;
				close CPY;
				chdir ("$cwd");
			}
		}
	}
	my $patchdir;
	$patchdir = "debian/patches-applied" if (-d "debian/patches-applied");
	$patchdir = "debian/patches" if (-d "debian/patches");
	return if (not defined $patchdir);
	# now back up any existing files in debian/patches
	opendir (DEBIAN, $patchdir) or croak ("Cannot open debian patch directory: $!");
	@patchfiles=grep(!/^\.\.?$/, readdir DEBIAN);
	closedir (DEBIAN);
	foreach my $file (@patchfiles)
	{
		# skip directories
		next if (! -f "$patchdir/$file");
		open (CPY, "$patchdir/$file") or die "Cannot open $file: $!";
		@control=<CPY>;
		close CPY;
		chdir ("../$package.old/");
		open (CPY, ">$patchdir/$file") or die "Cannot write $file: $!";
		print CPY @control;
		close CPY;
		chdir ("$cwd");
	}
}

=head1 bug_config

Read the embug offline data file for details of cross-building bugs
and return a hash, indexed by package name, containing the bug
numbers if a bug has been filed.

=cut

sub bug_config
{
	my $apt_cross_dir = &get_dpkg_cross_dir;
	my $cfile = $apt_cross_dir . "/embug";
	if (not -f $cfile)
	{
		open (CFILE, ">$cfile") or
			croak ("cannot create config file: $!\n");
		close (CFILE);
	}
	my $config = Config::Auto::parse("$cfile", format => "colon");
	return $config;
}

=head1 target_config

Read the embug offline data file for details of cross-building patches
that are out of date (i.e. do not apply successfully to the current
Debian version) and return a hash, indexed by package name.

=cut

sub target_config
{
	my $apt_cross_dir = &get_dpkg_cross_dir;
	my $cfile = $apt_cross_dir . "/emtarget";
	if (not -f $cfile)
	{
		open (CFILE, ">$cfile") or
			croak ("cannot create config file: $!\n");
		close (CFILE);
	}
	my $config = Config::Auto::parse("$cfile", format => "colon");
	return $config;
}

=head1 parse_xcontrol

Returns a hash of the contents of debian/xcontrol using
the Parse::DebControl module. See Parse::DebControl (3)
for more information on the hash contents.

Returns undefined if no debian/xcontrol file exists.

=cut

sub parse_xcontrol
{
	my $parser = new Parse::DebControl;
	my $xcontrol;
	my $options;
	return $xcontrol if (! -f "debian/xcontrol");
	$xcontrol = $parser->parse_file('./debian/xcontrol', $options);
	return $xcontrol;
}

=head1 lookup_sourcepkg

Returns a hash where the key is the submitted package name and the
value is the returned source package for that binary or the package
name if already a source package.

First parameter is the architecture to query (to cope with modified
packages), second parameter is the name of the package.

e.g.

 use Data::Dumper;
 use Emdebian::Tools;
 $var = &lookup_sourcepkg("arm", "libqof1");
 print Dumper($var);
 $VAR1 = {
         'libqof1' => 'qof'
         };

=cut

sub lookup_sourcepkg
{
	&check_cache_arch(shift);
	return undef unless (&get_suite);
	&init_cache(1);
	my $package = shift;
	my $emp = AptCrossPackage->new();
	$emp->Package($package);
	$emp = &srclookup($package);
	my $src = (defined $emp->{SourcePkg}) ? $emp->{SourcePkg} : $emp->{Package};
	return {$package, $src};
}

=head1 lookup_dependencies

Returns a space-separated string containing the names of
dependencies of the requested package. (Pass each to
&lookup_sourcepkg to get the appropriate source package to build
the dependency.)

First parameter is the architecture to query (to cope with modified
packages), second parameter is the name of the package.

e.g.

 use Emdebian::Tools;
 print &lookup_dependencies ("arm", "libqof1");

 libc6 libgda3-3 libglib2.0-0

=cut

sub lookup_dependencies
{
	my $arch = shift;
	&check_cache_arch($arch);
	return undef unless (&get_suite);
	&init_cache(1);
	my $package = shift;
	my $emp = AptCrossPackage->new();
	$emp->Package($package);
	$emp = &lookup_pkg($emp);
	my @list=();
	my $str;
	my $depend = $$emp->Depends;
	foreach my $d (@$depend)
	{
		my $n = $$d->Package;
		my $p = new AptCrossPackage;
		$p->Distribution(&get_suite);
		$p->Architecture($arch);
		$p->Package($n);
		&lookup_pkg($p);
		push @list, $p->Package;
	}
	return join(" ", @list);
}

=head1 get_standards_version

Simple subroutine that returns the current Debian Standards Version
for packages built using emdebian-tools. This will, usually, match the
current Standards Version in Debian unstable - this routine provides
a single location to update the version when necessary.

It could be possible to automate this value using a query against
the debian-policy package in Debian but that could lead to emdebian-tools
building packages that do not actually conform to that version of Policy.
Instead, the version is updated when the code conforms to the updated
Policy.

=cut

sub get_standards_version
{
	return "3.8.0";
}

sub write_target_config
{
	my $config = shift;
	if (not defined $config)
	{
		print "no defined config.\n";
		return;
	}
	my $apt_cross_dir = &get_dpkg_cross_dir;
	my $cfile = $apt_cross_dir . "/emtarget";
	open (CFILE, ">$cfile") or
		carp ("cannot write config file: $!\n");
	foreach my $pkg (sort keys %$config)
	{
		# write $pkg and $b to CFILE
		print CFILE "$pkg: " . $config->{$pkg} . "\n" if (defined $config->{$pkg});
	}
	close (CFILE);
}

1;
