/*
 * $Id: vt102_monitor.c,v 1.21 2009-08-25 06:30:04 sand Exp $
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* video output of vt102 (i.e. a display w. just a video signal) */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "vt102_monitor.h"

struct cpssp {
	struct sig_opt_rgb *port_opt_video;

	uint8_t pixel_buffer[MAX_WIDTH * MAX_HEIGHT * MAX_DEPTH / 8];
	uint8_t *write_pointer;
	int last_width;
	int last_height;
	unsigned int width;
	unsigned int height;
	unsigned int x;
	unsigned int y;
};

static void
vt102_monitor_power_230v_set(void *_cpssp, unsigned int val)
{
	/* FIXME */
}

static void
vt102_monitor_out(
	void *_cpssp,
	uint8_t red,
	uint8_t green,
	uint8_t blue
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->write_pointer[0] != blue
	 || cpssp->write_pointer[1] != green
	 || cpssp->write_pointer[2] != red) {

		cpssp->write_pointer[0] = blue;
		cpssp->write_pointer[1] = green;
		cpssp->write_pointer[2] = red;
		sig_opt_rgb_pixel_set(cpssp->port_opt_video, cpssp,
				cpssp->x, cpssp->y,
				red, green, blue);
	}

	cpssp->write_pointer += 4;
	cpssp->x++;
}

static void
vt102_monitor_vert_retrace(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->height = cpssp->y;

	if (cpssp->width != cpssp->last_width
	 || cpssp->height != cpssp->last_height) {

		if (cpssp->height < cpssp->last_height && cpssp->last_height != -1) {
			/* fewer lines
			 * -> fill the rest with black */
			while (cpssp->y < cpssp->last_height) {
				while (cpssp->x < cpssp->last_width) {
					vt102_monitor_out(cpssp, 0, 0, 0);
				}
				cpssp->x = 0;
				cpssp->y++;
				cpssp->write_pointer = &cpssp->pixel_buffer[cpssp->y * MAX_WIDTH * 4];
			}
		}

		cpssp->last_width = cpssp->width;
		cpssp->last_height = cpssp->height;

		sig_opt_rgb_size_set(cpssp->port_opt_video, cpssp, cpssp->width, cpssp->height);
	}

	cpssp->y = 0;

	cpssp->write_pointer = &cpssp->pixel_buffer[0];

	sig_opt_rgb_sync(cpssp->port_opt_video, cpssp);
}

static void
vt102_monitor_hor_retrace(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->width = cpssp->x;

	if (cpssp->width < cpssp->last_width && cpssp->last_width != -1) {
		/* lines became shorter
		 * -> fill the rest with ,,black'' */
		while (cpssp->x < cpssp->last_width) {
			vt102_monitor_out(cpssp, 0, 0, 0);
		}
	}

	cpssp->x = 0;
	cpssp->y++;
	cpssp->write_pointer = &cpssp->pixel_buffer[cpssp->y * MAX_WIDTH * 4];
}

static void
vt102_monitor_no_sync(void *_cpssp)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	int y;
	int x;

	/* reset monitor state */
	cpssp->y = 0;
	cpssp->x = 0;
	cpssp->width = cpssp->last_width;
	cpssp->height = cpssp->last_height;
	cpssp->write_pointer = &cpssp->pixel_buffer[0];

	for (y = 0; y < 400; y++) {
		for (x = 0; x < 640; x++) {
			vt102_monitor_out(cpssp, 0, 0, 0);
		}
		vt102_monitor_hor_retrace(cpssp);
	}
	vt102_monitor_vert_retrace(cpssp);
}

void *
vt102_monitor_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_boolean *port_power_230v,
	struct sig_video *port_in_video,
	struct sig_opt_rgb *port_opt_video,
	struct sig_boolean *port_screen_shot,
	struct sig_boolean *port_rec
)
{
	static const struct sig_boolean_funcs power_230v_funcs = {
		.set = vt102_monitor_power_230v_set,
	};
	static const struct sig_video_funcs video_funcs = {
		.out = vt102_monitor_out,
		.vert_retrace = vt102_monitor_vert_retrace,
		.hor_retrace = vt102_monitor_hor_retrace,
		.no_sync = vt102_monitor_no_sync,
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	memset(cpssp->pixel_buffer, 0, sizeof(cpssp->pixel_buffer));
	cpssp->write_pointer = &cpssp->pixel_buffer[0];
	cpssp->last_width = -1;
	cpssp->last_height = -1;
	cpssp->width = 0;
	cpssp->height = 0;
	cpssp->x = 0;
	cpssp->y = 0;

	/* Call */
	sig_video_connect(port_in_video, cpssp, &video_funcs);

	/* Out */
	cpssp->port_opt_video = port_opt_video;

	/* In */
	sig_boolean_connect_in(port_power_230v, cpssp, &power_230v_funcs);

	return cpssp;
}

void
vt102_monitor_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	free(cpssp);
}
