/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __SYSTEM_NODE_TREE_H_INCLUDED
#define __SYSTEM_NODE_TREE_H_INCLUDED

#include "lib/tree.h"

#include <assert.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef enum {
	NODE_UNDEFINED = 0,
	NODE_SYSTEM = 1,
	NODE_COMPONENT = 2,
	NODE_CHIP = 3,
	NODE_BUS = 4,
} node_type;

typedef enum {
	DATA_TYPE_UINT8_T = 0,
	DATA_TYPE_UINT16_T = 1,
	DATA_TYPE_UINT32_T = 2,
	DATA_TYPE_UINT64_T = 3,
	DATA_TYPE_INT8_T = 4,
	DATA_TYPE_INT16_T = 5,
	DATA_TYPE_INT32_T = 6,
	DATA_TYPE_INT64_T = 7,
	DATA_TYPE_INT = 8,
	DATA_TYPE_UINT = 9,
} data_type_t;

static inline int is_signed(data_type_t t) {
	switch (t) {
	case DATA_TYPE_UINT8_T:
	case DATA_TYPE_UINT16_T:
	case DATA_TYPE_UINT32_T:
	case DATA_TYPE_UINT64_T:
	case DATA_TYPE_UINT:
		return 0;
	case DATA_TYPE_INT8_T:
	case DATA_TYPE_INT16_T:
	case DATA_TYPE_INT32_T:
	case DATA_TYPE_INT64_T:
	case DATA_TYPE_INT:
		return 1;
	}
	return 0;
}

static inline int get_word_size(data_type_t t) {
	switch (t) {
	case DATA_TYPE_UINT8_T:
		return sizeof(uint8_t);
	case DATA_TYPE_UINT16_T:
		return sizeof(uint16_t);
	case DATA_TYPE_UINT32_T:
		return sizeof(uint32_t);
	case DATA_TYPE_UINT64_T:
		return sizeof(uint64_t);
	case DATA_TYPE_INT8_T:
		return sizeof(int8_t);
	case DATA_TYPE_INT16_T:
		return sizeof(int16_t);
	case DATA_TYPE_INT32_T:
		return sizeof(int32_t);
	case DATA_TYPE_INT64_T:
		return sizeof(int64_t);
	case DATA_TYPE_INT:
		return sizeof(int);
	case DATA_TYPE_UINT:
		return sizeof(unsigned int);
	}
	return 1;
}

typedef struct {
	const char *name;
	size_t offset;
	data_type_t type;
	uint8_t is_used_in_breakpoints;
} system_node_field;

#define MAX_FIELDS 256

typedef void (*set_inspection_cb_t)(void *opaque, int is_under_inspection);
typedef void (*single_step_cb_t)(void *opaque);

typedef struct system_node_t system_node;
struct system_node_t {
	const char *name;
	node_type type;
	void *opaque;
	set_inspection_cb_t set_inspection_cb;
	single_step_cb_t single_step_cb;
	uint8_t has_active_breakpoints;

	system_node_field fields[MAX_FIELDS];
	int num_fields;
};

void system_node_tree_init(void);
tree_node* system_get_node_tree(void);

system_node* system_find_tree_node(const char *name, int len);

system_node*
system_comp_register(const char *name, void *opaque);

void
system_comp_register_inspection_cb(system_node *node, set_inspection_cb_t cb);
void
system_comp_register_step_cb(system_node *node, single_step_cb_t cb);


/**
 * @brief system_node_register_field
 * @param node
 * @param name This pointer will be saved internally and must always be valid.
 * @param offset Offset of this field in the component state structure.
 */
void system_node_register_field(system_node *node,
                                const char *name,
                                size_t offset,
                                data_type_t type);

static inline system_node_field*
system_get_node_field(system_node *node, size_t offset) {
	unsigned i;
	for (i = 0; i<node->num_fields; i++) {
		system_node_field *field = &node->fields[i];
		if (field->offset == offset) {
			return field;
		}
	}
	/* field was not registered before */
	fprintf(stderr, "Registered state variables:\n");
	for (i = 0; i<node->num_fields; i++) {
		system_node_field *field = &node->fields[i];
		fprintf(stderr, "%8d: %s\n", (int) field->offset, field->name);
	}
	fprintf(stderr, "Missing: offset %d\n", (int) offset);
	assert(0);
	return NULL;
}

system_node_field*
system_find_field(system_node *current,
                  const char *name, int len,
                  system_node **out_node);

#endif /* __SYSTEM_NODE_TREE_H_INCLUDED */
