##################################################
# Engines
##################################################
# input properties
# /engines/engine[n]/oil-pressure-psi
# /engines/engine[n]/mp-osi
# output properties
# /instrumentation/annunciator/oil
##################################################
var Engines = {};
Engines.new = func {
  var obj = {};
  obj.parents = [Engines];

  obj.enginesNode = props.globals.getNode( "/engines" );
  obj.engineNodes = obj.enginesNode.getChildren( "engine" );

  obj.annunciatorNode = props.globals.getNode( "/instrumentation/annunciator" );
  obj.oilNode = obj.annunciatorNode.getNode( "oil", 1 );
  obj.oilNode.setBoolValue( 0 );

  for( var i = 0; i < size(obj.engineNodes); i = i+1 ) {
    var s = "overboost[" ~ i ~ "]";
    var n = obj.annunciatorNode.getNode( s, 1 );
    n.setBoolValue( 0 );
  }
  obj.overboostNodes = obj.annunciatorNode.getChildren( "overboost" );

  obj.jsbEngineNodes = props.globals.getNode("/fdm/jsbsim/propulsion").getChildren("engine");
  obj.pressureNode      = props.globals.getNode( "/systems/static/pressure-inhg" );
  obj.totalPressureNode = props.globals.getNode( "/systems/pitot/total-pressure-inhg" ).getValue();

  obj.controlsNodes = props.globals.getNode( "/controls/engines" ).getChildren( "engine" );

  obj.cowlFlapsPosNode = props.globals.getNode( "/surface-positions/speedbrake-pos-norm", 1 );
  return obj;
}

Engines.update = func {
  me.oil = 0;

  for( var i = 0; i < size(me.engineNodes); i = i+1 ) {
    var oil_pressure_psi = me.engineNodes[i].getNode("oil-pressure-psi").getValue();
    if( oil_pressure_psi != nil and oil_pressure_psi < 30 ) {
      me.oil = 1;
      break;
    }

    me.mp_osi = me.engineNodes[i].getNode("mp-osi").getValue();
    if( me.mp_osi != nil and me.mp_osi > 40.0 ) {
      me.overboostNodes[i].setBoolValue( 1 );
    } else {
      me.overboostNodes[i].setBoolValue( 0 );
    }

    # create a rpm-norm-inv property for propdisc transparency
    me.rpm = me.engineNodes[i].getNode( "rpm" ).getValue();
    var rpm_norm = me.rpm/2575;
    if( rpm_norm > 1 ) {
      rpm_norm = 1;
    }
    me.engineNodes[i].getNode( "rpm-norm-inv", 1 ).setDoubleValue( 1-rpm_norm );

    # twiggle with manifold pressure
    var mapAdjustNode = me.jsbEngineNodes[i].getNode( "map-adjust" );
    if( mapAdjustNode != nil ) {

      # map depends on RPM
      var ma = rpm_norm*0.37 + 0.63;
      mapAdjustNode.setDoubleValue( ma );

      # and on ram pressure
     if( me.pressureNode != nil and me.totalPressureNode != nil ) {
       var d = me.pressureNode.getValue();
       if( d != 0 ) {
         ma *= me.totalPressureNode.getValue() / d;
       }
     }

    }

    # adjust cooling area of engine depending of position of cowl flaps
    # range 2 - 2.5
    if( me.cowlFlapsPosNode != nil ) {
      var n = me.jsbEngineNodes[i].getNode("cooling-area");
      if( n != nil ) {
        n.setDoubleValue( me.cowlFlapsPosNode.getValue()*0.5 + 2 );
      }
    }

    # check for prop feather
    var featherN = me.controlsNodes[i].getChild( "propeller-feather" );
    if( me.controlsNodes[i].getChild( "propeller-pitch" ).getValue() < 0.1 ) {
      if( featherN.getBoolValue() == 0 and me.rpm > 800 ) {
        featherN.setBoolValue( 1 );
      }
    } else { 
      if( featherN.getBoolValue() == 1 and me.rpm > 400 ) {
        featherN.setBoolValue( 0 );
      }
    }
  }
  me.oilNode.setBoolValue( me.oil );
}

##################################################
# Gear in Transit 
##################################################
# input properties
# /gear/gear[n]/position-norm
# output properties
# /gear/in-transit if any position-norm not 0 and not 1
##################################################
var GearInTransit = {};
GearInTransit.new = func {
  var obj = {};
  obj.parents = [GearInTransit];
  obj.gearNode = props.globals.getNode( "/gear" );
  obj.gearNodes = obj.gearNode.getChildren( "gear" );
  obj.inTransitNode = obj.gearNode.getNode( "in-transit", 1 );
  obj.inTransitNode.setBoolValue( 0 );
  return obj;
}

GearInTransit.update = func {
  var inTransit = 0;

  for( i = 0; i < size(me.gearNodes); i = i+1 ) {
    var position_norm = me.gearNodes[i].getNode("position-norm").getValue();
    if( position_norm != nil and position_norm > 0.0 and position_norm < 1.0 ) {
      inTransit = 1;
      break;
    }
  }
  me.inTransitNode.setBoolValue( inTransit );
}

##################################################
# Suction 
##################################################
# input properties
# /system/vacuum[n]/suction-inhg
# output properties
# /instrumentation/vacuum/suction-inhg    max of all suction-inhg
# /instrumentation/vacuum/inoperative[n]  true if suction-inhg is less 3.5
# /instrumentation/annunciator/suction    true if any suction is inoperative
##################################################
var Suction= {};

Suction.new = func {
  var obj = {};
  obj.parents = [Suction];
  obj.vacuumNodes = props.globals.getNode("/systems").getChildren("vacuum");

  obj.instrumentationNode = props.globals.getNode( "/instrumentation/vacuum", 1 );
  obj.suctionNode = obj.instrumentationNode.getNode( "suction-inhg", 1 );
  
  for( var i = 0; i < size(obj.vacuumNodes); i = i+1 ) {
    var s = "inoperative[" ~ i ~ "]";
    var n = obj.instrumentationNode.getNode( s, 1 );
    n.setBoolValue( 1 );
  }
  obj.inoperativeNodes = obj.instrumentationNode.getChildren( "inoperative" );

  obj.annunciatorNode = props.globals.getNode( "/instrumentation/annunciator/vacuum", 1 );

  return obj;
}

Suction.update = func {
  me.suction_inhg = 0.0;
  me.annunciator = 0;

  # find max(suction-inhg) of all vacuum systems
  # end set warning-flag if U/S

  for( var i = 0; i < size(me.vacuumNodes); i = i+1 ) {
    me.suction = me.vacuumNodes[i].getNode("suction-inhg").getValue();
    if( me.suction > me.suction_inhg ) {
      me.suction_inhg = me.suction;
    }
    if( me.suction < 3.5 ) {
      me.inoperativeNodes[i].setBoolValue( 1 );
      me.annunciator = 1;
    } else {
      me.inoperativeNodes[i].setBoolValue( 0 );
    }
  }
  me.suctionNode.setDoubleValue( me.suction_inhg );
  me.annunciatorNode.setBoolValue( me.annunciator );
}

##################################################
var heightNode = props.globals.getNode( "/position/altitude-agl-ft", "true" );
heightNode.setDoubleValue( 0.0 );
var dhNode = props.globals.getNode( "/instrumentation/radar-altimeter/decision-height", "true" );
dhNode.setDoubleValue( 0.0 );
var dhFlagNode = props.globals.getNode( "/instrumentation/radar-altimeter/decision-height-flag", "true" );
dhFlagNode.setBoolValue( 0 );

###################################################
# set DH Flag
var dhflag = -1;
var radarAltimeter = func {
  var d = 0;
  if( heightNode.getValue() <= dhNode.getValue() ) {
    d = 1;
  }
  if( d != dhflag ) {
    dhflag = d;
    dhFlagNode.setBoolValue( dhflag );
  }
}
###################################################
# Slaved Gyro
# fast mode: 180deg/min - 3deg/sec
# slow mode: 3deg/min   - 0.05deg/sec
var SlavedGyro = {};
SlavedGyro.new = func {
  var obj = {};
  obj.parents = [SlavedGyro];

  obj.hiNode = props.globals.getNode( "/instrumentation/heading-indicator" );
  obj.errorIndicatorNode = obj.hiNode.getNode( "error-indicator", 1 );
  obj.errorIndicatorNode.setDoubleValue( 0 );

  obj.offsetNode = obj.hiNode.getNode( "offset-deg", 1 );

  obj.modeNode = obj.hiNode.getNode( "mode-auto", 1 );
  obj.modeNode.setBoolValue( 0 );

  obj.rotateNode = obj.hiNode.getNode( "rotate", 1 );
  obj.rotateNode.setIntValue( 0 );

  obj.timeNode = props.globals.getNode( "/sim/time/elapsed-sec" );
 
  obj.fastrate = 3;
  obj.slowrate = 0.05;
  obj.last = obj.timeNode.getValue();
  obj.dt = 0;

  return obj;
}

SlavedGyro.update = func {
  var now = me.timeNode.getValue();
  me.dt = now - me.last;
  me.last = now;

  if( me.modeNode.getBoolValue() ) {
    # slaved
    var offset = me.offsetNode.getValue();
 
    var rate = 0.0;
    if( offset >= 0.0 ) {
      rate = me.slowrate;
    }
    if( offset > 3.0 ) {
      rate = me.fastrate;
    } 
    if( offset < 0.0 ) {
      rate = -me.slowrate;
    } 
    if( offset < -3.0 ) {
      rate = -me.fastrate;
    } 
    me.rotate( rate );

  } else {
    # free
    me.rotate( me.fastrate * me.rotateNode.getValue() );
  }
}

SlavedGyro.rotate = func {
  var rate = arg[0];
  if( rate == 0 ) {
    return;
  }
  me.offsetNode.setDoubleValue( me.offsetNode.getValue() - me.dt * rate );
}



###################################################
var gearInTransit = GearInTransit.new();
var suction = Suction.new();
var engines = Engines.new();

var slavedGyro = SlavedGyro.new();

var seneca_update = func {
  gearInTransit.update();
  suction.update();
  engines.update();
  slavedGyro.update();
  radarAltimeter();
  settimer(seneca_update, 0.1 );
}

setlistener("/sim/signals/fdm-initialized", seneca_update);

var paxDoor = aircraft.door.new( "/sim/model/door-positions/pax-door", 1, 0 );
var baggageDoor = aircraft.door.new( "/sim/model/door-positions/baggage-door", 2, 0 );
var rightDoor = aircraft.door.new( "/sim/model/door-positions/right-door", 1, 0 );

var beacon = aircraft.light.new( "/sim/model/lights/beacon", [0.05, 0.05, 0.05, 0.45 ], "/controls/lighting/beacon" );
var strobe = aircraft.light.new( "/sim/model/lights/strobe", [0.05, 0.05, 0.05, 0.05, 0.05, 0.35 ], "/controls/lighting/strobe" );

setprop( "/instrumentation/nav[0]/ident", 0 );
setprop( "/instrumentation/nav[1]/ident", 0 );

########################################
# Sync the magneto switches with magneto properties
########################################
var setmagneto = func {
  var eng = arg[0];
  var mag = arg[1];
  var on = props.globals.getNode( "/controls/engines/engine[" ~ eng ~ "]/magneto[" ~ mag ~ "]" ).getValue();
  var m = props.globals.getNode( "/controls/engines/engine[" ~ eng ~ "]/magnetos" );

  var v = m.getValue();

  # I wish nasal had binary operators...
  if( on ) {
    if( mag == 0 ) {
      if( v == 0 or v == 2 ) {
        v = v + 1;
      }
    }
    if( mag == 1 ) {
      if( v == 0 or v == 1 ) {
        v = v + 2;
      }
    }
  } else {
    if( mag == 0 ) {
      if( v == 1 or v == 3 ) {
        v = v - 1;
      }
    }
    if( mag == 1 ) {
      if( v ==2 or v == 3 ) {
        v = v - 2;
      }
    }
  }

  m.setIntValue( v );
}

var magnetolistener = func {
  var m = cmdarg();
  var eng = m.getParent();
  var m2 = eng.getChildren("magneto");
  var v = m.getValue();
  if( v == 0 ) {
    m2[0].setBoolValue( 0 );
    m2[1].setBoolValue( 0 );
  }
  if( v == 1 ) {
    m2[0].setBoolValue( 1 );
    m2[1].setBoolValue( 0);
  }
  if( v == 2 ) {
    m2[0].setBoolValue( 0 );
    m2[1].setBoolValue( 1 );
  }
  if( v == 3 ) {
    m2[0].setBoolValue( 1 );
    m2[1].setBoolValue( 1 );
  }
};

setlistener( "/controls/engines/engine[0]/magnetos", magnetolistener );
setlistener( "/controls/engines/engine[1]/magnetos", magnetolistener );

########################################
# Sync the fuel selector controls with the properties
#/controls/fuel/tank[n]/fuel_selector (boolean)
#/controls/fuel/tank[n]/to_engine (int)
########################################
#fuelselectorlistener = func {
#}

#setlistener( "/controls/fuel/tank[0]/fuel_selector", fuelselectorlistener );
#setlistener( "/controls/fuel/tank[1]/fuel_selector", fuelselectorlistener );
#setlistener( "/controls/fuel/tank[0]/to_engine",     fuelselectorlistener );
#setlistener( "/controls/fuel/tank[1]/to_engine",     fuelselectorlistener );

########################################
# Sync the dimmer controls with the according properties
########################################

var instrumentsFactorNode = props.globals.getNode( "/sim/model/material/instruments/factor" );
var dimmerlistener = func {
  var  n = cmdarg();
  instrumentsFactorNode.setValue( n.getValue() );
}

setlistener( "/controls/lighting/instruments-norm", dimmerlistener, 1, 0 );

########################################
# fuel pumps and primer
########################################
var FuelPumpHandler = {};
FuelPumpHandler.new = func {
  var m = {};
  m.parents = [FuelPumpHandler];
  m.engineControlRootN = props.globals.getNode( "/controls/engines/engine[" ~ arg[0] ~ "]" );
  m.annunciatorLightN = props.globals.getNode( "/instrumentation/annunciator/fuelpump[" ~ arg[0] ~ "]", 1 );
  if( m.annunciatorLightN.getValue() == nil ) {
    m.annunciatorLightN.setBoolValue( 0 );
  }

  m.fuelPumpControlN = m.engineControlRootN.getNode( "fuel-pump", 1 );
  if( m.fuelPumpControlN.getValue() == nil ) {
    m.fuelPumpControlN.setBoolValue( 0 );
  }
  m.primerControlN   = m.engineControlRootN.getNode( "primer", 1 );
  if( m.primerControlN.getValue() == nil ) {
    m.primerControlN.setBoolValue( 0 );
  }

  setlistener( m.fuelPumpControlN, func { m.listener() }, 1, 0 );
  setlistener( m.primerControlN, func { m.listener() }, 1, 0 );

  return m;
};

FuelPumpHandler.listener = func {
  var v = me.fuelPumpControlN.getBoolValue();
  if( v == 0 ) {
    v = me.primerControlN.getBoolValue();
  }
  me.annunciatorLightN.setBoolValue( v );
};

FuelPumpHandler.new( 0 );
FuelPumpHandler.new( 1 );
