/*
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *	fssubs.c: fs utility subroutines
 */

#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mntent.h>

#include "jfs_byteorder.h"
#include "jfs_types.h"
#include "jfs_superblock.h"
#include "jfs_filsys.h"
#include "jfs_dinode.h"
#include "jfs_dmap.h"
#include "jfs_imap.h"
#include "jfs_debug.h"

#include "devices.h"
#include "debug.h"

#include "fssubs.h"
#include "utilsubs.h"
#include "message.h"

/* forward references */
static void bAllocate(void);
int Is_Root_Mounted_RO(void);

/* Global Data */

int32_t	Fail_Quietly = 0;	/* If set, don't report errors */

HFILE	LVHandle;
LV_t 	LVMount;	/* logical volume descriptor */
LV_t	*lvMount = &LVMount;

dinode_t	DIBMap;		/* block allocation map inode */
dinode_t	*diBMap = &DIBMap;

dinode_t	DIIMap;		/* inode allocation map inode */
dinode_t	*diIMap = &DIIMap;

int32_t	IAGNumber = 0;

/*
 *	file system under defragfs
 */
FS_t	FSMount;	/* file system descriptor */
FS_t	*fsMount = &FSMount;

/* imap xtree sequential read control */
int64_t	     IMapXtreeLMLeaf; /* imap xtree leftmost leaf */
xtpage_t	 IMapXtreeLeaf;   /* imap xtree leaf buffer */
IMapXtree_t	 IMapXtree = {&IMapXtreeLeaf, XTENTRYSTART, 1};

/* endian routines */
extern uint32_t type_jfs;
extern void ujfs_swap_dbmap_t( dbmap_t * );
extern void ujfs_swap_dmapctl_t( dmapctl_t * );
extern void ujfs_swap_dmap_t( dmap_t * );
extern void ujfs_swap_dinomap_t( dinomap_t * );
extern void ujfs_swap_iag_t( iag_t * );
extern void ujfs_swap_xtpage_t( xtpage_t * );

extern int errno;

/*
 *	openLV()
 */
int32_t openLV(char	 *LVName)
{
	int32_t	 rc = 0;
	int32_t	 pbsize;
	int64_t	 LVSize;
     
    if ( ( rc = Is_Device_Mounted( LVName )) == MSG_JFS_VOLUME_IS_MOUNTED ) {   /* is mounted */
        printf("\n%s is mounted.\n", LVName);
    } else if ( rc == MSG_JFS_VOLUME_IS_MOUNTED_RO ) { /* is mounted read only */
        printf("\n%s is mounted 'read only'.\n", LVName);
    } else if ( rc == MSG_JFS_MNT_LIST_ERROR ) { /* setmntent failed */
        printf("\nCannot detect if %s is mounted.\n", LVName);
        return rc;
    } else if ( rc == MSG_JFS_NOT_JFS ) { /* mounted, but not jfs */
        printf("\nDevice mounted, but not type jfs!\n");
        return rc;
    } else { /* not mounted */
        printf("\n%s is NOT mounted.\n", LVName);
        return -1;
    }

	/* open the device and get its physical block size */
	if (( rc = ujfs_open_device(LVName, &LVHandle, &pbsize, READONLY) ))    {
		if (!Fail_Quietly)
			message_user(MSG_OSO_CANT_OPEN, NULL, 0, STDOUT_CODE,
				     NO_RESPONSE, OSO_MSG);
		return rc;
	}

	LVMount.pbsize = pbsize;
	LVMount.l2pbsize = log2shift(pbsize);
     
	/* get logical volume size */
	if (( rc = ujfs_get_dev_size(LVHandle, &LVSize)) )    {
		message_user(MSG_OSO_ERR_ACCESSING_DISK, NULL, 0, STDOUT_CODE, NO_RESPONSE, OSO_MSG);
		ujfs_close(LVHandle);
		return rc;
	}

	LVMount.LVSize = LVSize >> LVMount.l2pbsize;

	return rc;
}
	

/*
 *	openFS() 
 */ 
int32_t openFS(void)
{
	int32_t	 rc;
	struct superblock  *sb;
	buf_t	 *bp;

	/*
	 *	validate and retrieve fs parameters from superblock
	 */
	/* try to read the primary superblock */
	rc = bRawRead(LVHandle, (int64_t)SUPER1_OFF, (int32_t)PAGESIZE, &bp);
	if (rc != 0)    {
		/* try to read the secondary superblock */
		rc = bRawRead(LVHandle, (int64_t)SUPER2_OFF, (int32_t)PAGESIZE, &bp);
		if (rc != 0)       {
			return rc;
		}
	}

	sb = (struct superblock *)bp->b_data;

	/* check magic/version number */
	if (strncmp(sb->s_magic,JFS_MAGIC,(unsigned)strlen(JFS_MAGIC))
	    || (sb->s_version != JFS_VERSION))    {
		message_user(MSG_JFS_BAD_SUPERBLOCK, NULL, 0, STDOUT_CODE, NO_RESPONSE, JFS_MSG);
		return -1;
	}

	if (sb->s_state & FM_DIRTY)    {
		message_user(MSG_JFS_DIRTY, NULL, 0, STDOUT_CODE, NO_RESPONSE, JFS_MSG);
		return -1;
	}

	/* assert(lvMount->pbsize == sb->s_pbsize); */
	/* assert(lvMount->l2pbsize == sb->s_l2pbsize); */

	fsMount->bsize = sb->s_bsize;
	fsMount->l2bsize = sb->s_l2bsize;
	fsMount->l2bfactor = sb->s_l2bfactor;
	fsMount->nbperpage = PAGESIZE >> fsMount->l2bsize;
	fsMount->l2nbperpage = log2shift(fsMount->nbperpage);
	fsMount->FSSize = sb->s_size >> sb->s_l2bfactor;
	fsMount->AGSize = sb->s_agsize;

	bRelease(bp);

	return rc;
}


/*
 *	closeFS()
 */
void closeFS(void)
{
	ujfs_close(LVHandle);
}


/*
 * NAME:	readBMapGCP()
 *
 * FUNCTION:	read the bmap global control page and 
 *		the initial level L2, L1.0 and L0.0 control pages;
 *
 * note: bmap file structure is satic during defragfs()
 * (defragfs() and extendfs() are mutually exclusive operation);
 */
int32_t readBMapGCP(BMap_t  *bMap)
{
	int32_t	 rc;
	int64_t	 xaddr;
	int32_t	 xlen;
	dbmap_t	 *bcp;

	/* first 4 pages are made contiguous by mkfs() */
	xlen = 4 << fsMount->l2nbperpage;
    rc = xtLookup(diBMap, (int64_t)0, &xaddr, &xlen, 0);
    if ( rc )
        return rc;

	rc = pRead(fsMount, xaddr, xlen, (void *)bMap);
	if ( rc )    {
		return rc;
	}

	bcp = (dbmap_t *)bMap;

    /* swap if on big endian machine */
    if (type_jfs & JFS_SWAP_BYTES) {
        ujfs_swap_dbmap_t( bcp );
    }

	fsMount->nAG = bcp->dn_numag;

	return 0;
}

		
/*
 * NAME:	readBMapLCP()
 *
 * FUNCTION:  	read level control page of given level for given block number;
 */
int32_t readBMapLCP( int64_t	bn,
	                 int32_t	level,
	                 dmapctl_t  *lcp)

{
	int32_t	 rc = 0;
	int64_t	 xoff = 0, xaddr;
	int32_t	 xlen;

	switch (level)    {
	case 0:
		xoff = BLKTOL0(bn, fsMount->l2nbperpage);
		break;
	case 1:
		xoff = BLKTOL1(bn, fsMount->l2nbperpage);
		break;
	case 2:
		xoff = fsMount->nbperpage;
		break;
	}

	xlen = fsMount->nbperpage;
	rc = xtLookup(diBMap, xoff, &xaddr, &xlen, 0);		
	if ( rc != 0 )
		return rc;

	rc = pRead(fsMount, xaddr, xlen, (void *)lcp);
	if ( rc )    {
		return rc;
	}

    /* swap if on big endian machine */
    if (type_jfs & JFS_SWAP_BYTES) {
        ujfs_swap_dmapctl_t( lcp );
    }

	return 0;
}


/*
 * NAME:        readBMap()
 *
 * FUNCTION:	Read ndmaps dmap pages starting from the start_dmap.    
 *
 * note: bmap file structure is satic during defragfs()
 * (defragfs() and extendfs() are mutually exclusive operation);
 */
int32_t readBMap( int64_t  start_dmap,  /* the dmap number to start read */
	              int32_t  ndmaps,      /* number of dmap pages read */
	              dmap_t   *dmap)	    /* buffer pointer */
{
    int32_t	 rc;
	int64_t	 xaddr;	
	int64_t	 xoff;
	int32_t	 xlen0, xlen;

	/* convert dmap number to xoffset */
	xaddr = start_dmap << L2BPERDMAP;
    xoff = BLKTODMAP(xaddr, fsMount->l2nbperpage);

	xlen0 = ndmaps << fsMount->l2nbperpage;
	xlen = fsMount->nbperpage;
	for ( ; xlen0 > 0; xlen0 -= xlen) {
        rc = xtLookup(diBMap, xoff, &xaddr, &xlen, 0);
        if ( rc )
            return rc;

		rc = pRead(fsMount, xaddr, xlen, (void *)dmap);
		if ( rc )       {
			return rc;
		}

        /* swap if on big endian machine */
        if (type_jfs & JFS_SWAP_BYTES) {
            ujfs_swap_dmap_t( dmap );
        }

		xoff += xlen;
		dmap = (dmap_t *)((char *)dmap + PAGESIZE);
	}

	return 0;
}


/*
 * NAME:	readIMapGCPSequential()
 *
 * FUNCTION:	read the imap global control page.
 */
int32_t readIMapGCPSequential( IMap_t  *iMap,
	                           iag_t   *iag  )
{
	int32_t	  rc;
	xtpage_t  *p;
	int64_t	  xaddr;

	/* read in leftmost leaft xtpage */
	rc = xtLMLeaf(diIMap, &IMapXtreeLeaf); 
	if ( rc != 0 )
		return rc;

	p = &IMapXtreeLeaf;
	IMapXtreeLMLeaf = addressPXD(&p->header.self);

	/* read IMap control page */
	xaddr = addressXAD(&p->xad[XTENTRYSTART]);
	rc = pRead(fsMount, xaddr, fsMount->nbperpage, (void *)iMap);

    /* swap if on big endian machine */
    if (type_jfs & JFS_SWAP_BYTES) {
        ujfs_swap_dinomap_t( &(iMap->ctl) );
    }

	/* init for start of bitmap page read */
	IAGNumber = 0;
	IMapXtree.index = XTENTRYSTART;
	IMapXtree.page = 1;	/* skip for global control page */
	iag->iagnum = -1;

	return rc;
}


/*
 * NAME:	readIMapSequential()
 *
 * FUNCTION:	read one iag page at a time. 
 *
 * state variable:
 *	IMapXtree.leaf - current imap leaf xtpage under scan;
 *	IMapXtree.index - current xad entry index in iMapXtreeLeaf; 
 *	IMapXtree.page - iag number to read within current imap extent; 
 *
 * note: IMap pages may grow (but NOT freed) dynamically; 
 */
int32_t readIMapSequential(iag_t  *iag)
{
	int32_t	  rc;
	xtpage_t  *p;
	int64_t	  xaddr;
	int32_t	  xlen, delta;

	p = &IMapXtreeLeaf;

	while(1) {
		/* continue with current leaf xtpage ? */
		if (IMapXtree.index < p->header.nextindex) {
			/* determine iag page extent */
			xaddr = addressXAD(&p->xad[IMapXtree.index]);
			xlen = lengthXAD(&p->xad[IMapXtree.index]);

			if (IMapXtree.page) {
				/* compute offset within current extent */
                delta = IMapXtree.page << fsMount->l2nbperpage;
                xaddr += delta;
                xlen -= delta;
			}

			/* read a iag page */
			rc = pRead(fsMount, xaddr, fsMount->nbperpage,
				       (void *)iag);

            /* swap if on big endian machine */
            if (type_jfs & JFS_SWAP_BYTES) {
                ujfs_swap_iag_t( iag );
            }

			if ( rc != 0 ) {
				return rc;
			}

			/*current extent has more iag */
			if ( xlen > fsMount->nbperpage ) {
				/* continue to read from current extent */
				IMapXtree.page++;
			} else  {
				IMapXtree.index++;
				IMapXtree.page = 0;
			}

			return 0;
		}

		/* read next/right sibling leaf xtpage */
		if (p->header.next != 0) {
			xaddr = p->header.next;

			/* init for next leaf xtpage */
			IMapXtree.index = XTENTRYSTART;
			IMapXtree.page = 0;
		} else {
			/* init for start of bitmap page read */
			IAGNumber = 0;
			IMapXtree.index = XTENTRYSTART;
			IMapXtree.page = 1; /* skip for global control page */

			if (p->header.prev != 0)
				xaddr = IMapXtreeLMLeaf;
 			/* a single leaf xtree */
			else
				continue;
		}

		/* read next xtree leaf page */
		rc = pRead(fsMount, xaddr, fsMount->nbperpage,
			       (void *)&IMapXtreeLeaf);

        /* swap if on big endian machine */
        if (type_jfs & JFS_SWAP_BYTES) {
            ujfs_swap_xtpage_t( &IMapXtreeLeaf );
        }

		if ( rc != 0 ) {
			return rc;
		}
	}
}


/*
 * NAME:        xtLMLeaf()
 *
 * FUNCTION:    read in leftmost leaf page of the xtree
 *              by traversing down leftmost path of xtree;
 */
int32_t xtLMLeaf( dinode_t	*dip,	/* disk inode */
                  xtpage_t  *pp)	/* pointer to leftmost leaf xtpage */
{
        int       rc;
        xtpage_t  *p;
        int64_t	  xaddr;

        /* start from root in dinode */
        p = (xtpage_t *)&dip->di_btroot;
        /* is this page leaf ? */
        if (p->header.flag & BT_LEAF) {
		   p->header.next = p->header.prev = 0;
		   memcpy(pp, p, DISIZE);
           return 0;
    	}

        /*
         * traverse down leftmost child node to the leftmost leaf of xtree
         */
        while (1) {
            /* read in the leftmost child page */
	    	xaddr = addressXAD(&p->xad[XTENTRYSTART]);
		    rc = pRead(fsMount, xaddr, fsMount->nbperpage, (void *)pp);
            if (rc) {
                return rc;
	    	}

            /* swap if on big endian machine */
            if (type_jfs & JFS_SWAP_BYTES) {
                ujfs_swap_xtpage_t( pp );
            }

            /* is this page leaf ? */
            if (pp->header.flag & BT_LEAF)
                return 0;
		    else
               	p = pp;
        }
}


/*
 *	 xtree key/entry comparison: extent offset
 *
 * return:
 *	-1: k < start of extent
 *	 0: start_of_extent <= k <= end_of_extent
 *	 1: k > end_of_extent
 */
#define XT_CMP(CMP, K, X, OFFSET64)\
{\
	OFFSET64 = offsetXAD(X);\
	(CMP) = ((K) >= OFFSET64 + lengthXAD(X)) ? 1 :\
	      ((K) < OFFSET64) ? -1 : 0;\
}


/*
 *	xtLookup()
 *
 * function:	search for the xad entry covering specified offset.
 *
 * parameters:
 *	ip	    - file object;
 *	xoff	- extent offset;
 *	cmpp	- comparison result:
 *	btstack	- traverse stack;
 *	flag	- search process flag (XT_INSERT);
 *
 * returns:
 *	btstack contains (bn, index) of search path traversed to the entry.
 *	*cmpp is set to result of comparison with the entry returned.
 *	the page containing the entry is pinned at exit.
 */
int32_t xtLookup( dinode_t	*dip,
	              int64_t	xoff,	/* offset of extent */
	              int64_t	*xaddr,
	              int32_t	*xlen,
	              uint32_t  flag)
{
	int32_t	  rc = 0;
	buf_t	  *bp = NULL;	/* page buffer */
	xtpage_t  *p;	/* page */
	xad_t	  *xad;
	int64_t	  t64;
	int32_t	  cmp = 1, base, index, lim;
	int32_t	  t32;

	/*
	 *	search down tree from root:
	 *
	 * between two consecutive entries of <Ki, Pi> and <Kj, Pj> of
	 * internal page, child page Pi contains entry with k, Ki <= K < Kj.  
	 *
 	 * if entry with search key K is not found
 	 * internal page search find the entry with largest key Ki 
	 * less than K which point to the child page to search;
 	 * leaf page search find the entry with smallest key Kj 
	 * greater than K so that the returned index is the position of
	 * the entry to be shifted right for insertion of new entry.
	 * for empty tree, search key is greater than any key of the tree.
	 */

	/* start with inline root */
	p = (xtpage_t *)&dip->di_btroot;

xtSearchPage:
	lim = p->header.nextindex - XTENTRYSTART; 

	/*
	 * binary search with search key K on the current page
	 */
	for (base = XTENTRYSTART; lim; lim >>= 1) {
		index = base + (lim >> 1);

		XT_CMP(cmp, xoff, &p->xad[index], t64);
		if (cmp == 0) {
			/*
			 *	search hit
			 */
			/* search hit - leaf page:
			 * return the entry found
			 */
			if (p->header.flag & BT_LEAF) {
				/* save search result */
				xad = &p->xad[index];
				t32 = xoff - offsetXAD(xad);
				*xaddr = addressXAD(xad) + t32; 
				*xlen = MIN(*xlen, lengthXAD(xad) - t32);

				rc = cmp;
				goto out;
			}

			/* search hit - internal page:
			 * descend/search its child page 
			 */
			goto descend;
		}

		if (cmp > 0) {
			base = index + 1;
			--lim;
		}
	}

	/*
	 *	search miss
	 *
	 * base is the smallest index with key (Kj) greater than 
	 * search key (K) and may be zero or maxentry index.
	 */
	/*
	 * search miss - leaf page: 
	 *
	 * return location of entry (base) where new entry with
	 * search key K is to be inserted.
	 */
	if (p->header.flag & BT_LEAF) {
		rc = cmp;
		goto out;
	}

	/*
	 * search miss - non-leaf page:
	 *
	 * if base is non-zero, decrement base by one to get the parent
	 * entry of the child page to search.
	 */
	index = base ? base - 1 : base;

	/* 
	 * go down to child page
	 */
descend:
	/* get the child page block number */
	t64 = addressXAD(&p->xad[index]);

	/* release parent page */
	if (bp)
		bRelease(bp);

	/* read child page */
	bRead(fsMount, t64, fsMount->nbperpage, &bp);

	p = (xtpage_t *)bp->b_data;
	goto xtSearchPage;

out:
	/* release current page */
	if (bp)
		bRelease(bp);

	return rc;
}


/*
 * NAME:	fscntl()
 *
 * FUNCTION:	call fs control service;
 */
int32_t fscntl( uint32_t       cmd,
	            void	       *pList,		/* parameter list */
	            unsigned long  *pListLen,	/* length of pList */
	            void	       *pData,		/* data area */
	            unsigned long  *pDataLen)	/* length of pData */
{
	int32_t	 rc = 0;

	return rc;
}


/*
 *	pRead()
 *
 * read into specified buffer;
 */
int32_t pRead( FS_t	    *fsMount,
	           int64_t  xaddr,	   /* in bsize */
	           int32_t  xlen,	   /* in bsize */
	           void	    *p)	       /* buffer */
{
	int32_t	 rc;
	int64_t	 off;	/* offset in byte */
	int32_t	 len;	/* length in byte */

	/* read in from disk */
	off = xaddr << fsMount->l2bsize;
	len = xlen << fsMount->l2bsize;
	if ( (rc = ujfs_rw_diskblocks(LVHandle, off, len, p, GET)) ) {
		message_user(MSG_OSO_READ_ERROR, NULL, 0, STDOUT_CODE, NO_RESPONSE, OSO_MSG);
	}

	return rc;
}


/*
 *	buffer pool
 *	===========
 */
int32_t	 NBUFFER = 8;
int32_t	 nBuffer = 0;
buf_t	 *bCachelist = NULL;

static void bAllocate(void)
{
	buf_t	 *bp;
	char	 *buffer;
	int32_t	 i;

	/* allocate buffer header */
	if ((bp = malloc(NBUFFER * sizeof(buf_t))) == NULL) {
		message_user(MSG_OSO_NOT_ENOUGH_MEMORY, NULL, 0, STDOUT_CODE, NO_RESPONSE, OSO_MSG);
		exit (ERROR_DISK_FULL);
	}

	/* allocate buffer pages */
	if ((buffer = malloc(NBUFFER * PAGESIZE)) == NULL) {
		message_user(MSG_OSO_NOT_ENOUGH_MEMORY, NULL, 0, STDOUT_CODE, NO_RESPONSE, OSO_MSG);
		exit (ERROR_DISK_FULL);
	}

	/* insert buffer headers in lru/freelist */
	for (i = 0; i < NBUFFER; i++, bp++, buffer += PAGESIZE) {
		bp->b_data = buffer;
		bp->b_next = bCachelist;
		bCachelist = bp;
	}

	nBuffer += NBUFFER;
}

/*
 *	bRead()
 *
 * aloocate, read and return a buffer;
 */
int32_t bRead( FS_t	    *fsMount,
	           int64_t  xaddr,	  /* in bsize */
	           int32_t  xlen,	  /* in bsize */
	           buf_t    **bpp)
{
	int32_t	 rc;
	int64_t	 off;	/* offset in byte */
	int32_t	 len;	/* length in byte */
	buf_t	 *bp;

	/* allocate buffer */
	if (bCachelist == NULL)
		bAllocate();
	bp = bCachelist;
	bCachelist = bp->b_next;

	/* read in from disk */
	off = xaddr << fsMount->l2bsize;
	len = xlen << fsMount->l2bsize;
	rc = ujfs_rw_diskblocks(LVHandle, off, len, (void *)bp->b_data, GET);
	if (rc == 0)
		*bpp = bp;
    else {
		message_user(MSG_OSO_READ_ERROR, NULL, 0, STDOUT_CODE, NO_RESPONSE, OSO_MSG);
		bRelease(bp);
	}

	return rc;
}

int32_t bRawRead( uint32_t  LVHandle,
	              int64_t	off,	/* in byte */
	              int32_t	len,	/* in byte */
	              buf_t     **bpp)
{
	int32_t	 rc;
	buf_t	 *bp;

	/* allocate buffer */
	if (bCachelist == NULL)
		bAllocate();
	bp = bCachelist;
	bCachelist = bp->b_next;

	/* read in from disk */
	rc = ujfs_rw_diskblocks(LVHandle, off, len, (void *)bp->b_data, GET);
	if (rc == 0)
		*bpp = bp;
    else {
		message_user(MSG_OSO_READ_ERROR, NULL, 0, STDOUT_CODE, NO_RESPONSE, OSO_MSG);
		bRelease(bp);
	}

	return rc;
}

void bRelease(buf_t	 *bp)
{
	/* release buffer */
	bp->b_next = bCachelist;
	bCachelist = bp;
}
 
/*--------------------------------------------------------------------
 * NAME: Is_Device_Mounted
 *
 * FUNCTION: Determine if the device specified is mounted.
 *
 * PRE CONDITIONS: Device_Name must be the name of the device.
 *
 * POST CONDITIONS:
 *
 * PARAMETERS: Device_Name - The name of the device.
 *
 * RETURNS:
 *      0 if the device is not mounted.
 *    > 0 if the device is mounted or an error occurs.
 *
 * NOTES: borrowed some of this routine from e2fsck
 */
int Is_Device_Mounted(char  *Device_Name)
{

  FILE           *Mount_Records;  /* Pointer for system's mount records */
  struct mntent  *Mount_Data;     /* Holds data for entry in mount list */
  int            Mounted_RC = 0;  /* Holds return code.             */
  char           Root_FS_Type[] = "0000000";

  #define ROOT_DIR "/"

  /* Attempt to open /proc/mounts for access. */
  if ( (Mount_Records = setmntent("/proc/mounts","r")) == NULL ) {
    /* Unable to access list of mounted devices in /proc/mounts! */
    /* Attempt to open /etc/mtab for access.                     */
    if ( (Mount_Records = setmntent(MOUNTED,"r")) == NULL ) {
      /* Unable to access list of mounted devices in /etc/mtab!  */
      return MSG_JFS_MNT_LIST_ERROR;
    } 
  }

  /* Attempt to find specified device name in mount records */
  while ((Mount_Data = getmntent (Mount_Records)) != NULL) {
    /* save root file system type for later, if needed */
    if (strcmp(ROOT_DIR, Mount_Data->mnt_dir) == 0)
      strcpy(Root_FS_Type, Mount_Data->mnt_type);
    if (strcmp(Device_Name, Mount_Data->mnt_fsname) == 0)
	  break;
  }

  if (Mount_Data == 0) {
#ifndef __GNU__ /* The GNU hurd is broken with respect to stat devices */
    struct stat st_root, st_file;
    /*
	 * If the specified device name was not found in the mount records, 
     * do an extra check to see if this is the root device.  We can't 
     * trust /etc/mtab, and /proc/mounts will only list /dev/root for 
     * the root filesystem.  Argh.  Instead we will check if the given 
     * device has the same major/minor number as the device that the 
     * root directory is on.
	 */
	if (stat("/", &st_root) == 0 && stat(Device_Name, &st_file) == 0) {
	  if (st_root.st_dev == st_file.st_rdev) {
        if ( Is_Root_Mounted_RO() )
          Mounted_RC = MSG_JFS_VOLUME_IS_MOUNTED_RO;
        else
          Mounted_RC = MSG_JFS_VOLUME_IS_MOUNTED;
        /*
         * Make a best effort to ensure that
         * the root file system type is jfs
         */
        if ( (strcmp(Root_FS_Type, "0000000") != '0') &&
             (strcmp(Root_FS_Type, "jfs") != 0 )         )
          /* is mounted, found type, is not type jfs */
          Mounted_RC = MSG_JFS_NOT_JFS;
	  }
	}
#endif

  } else {
    if ( strcmp(Mount_Data->mnt_type, "jfs") == 0 ) {
      /* is mounted, is type jfs */
      Mounted_RC = MSG_JFS_VOLUME_IS_MOUNTED;

      /* See if we're mounted 'ro' (read only)          */
      /* If we are booting on a jfs device, hasmntopt   */
      /* may return 'rw' even though we're mounted 'ro' */
      /* so let's check for that situation              */
      if (!strcmp(Mount_Data->mnt_dir, "/")) {
        if ( Is_Root_Mounted_RO() )
          Mounted_RC = MSG_JFS_VOLUME_IS_MOUNTED_RO;
      } else {
        /* Check to see if the 'ro' option is set */
        if (hasmntopt(Mount_Data, MNTOPT_RO))
          Mounted_RC = MSG_JFS_VOLUME_IS_MOUNTED_RO;
      }
    } else {
        /* is mounted, is not type jfs */
        Mounted_RC = MSG_JFS_NOT_JFS;
    }
  }

  /* Close the stream. */
  endmntent(Mount_Records);

  /* Return the appropriate value. */
  return Mounted_RC;
}


/*--------------------------------------------------------------------
 * NAME: Is_Root_Mounted_RO
 *
 * FUNCTION: Determine if root is mounted read only
 *
 * RETURNS:
 *      0 if root is not mounted READ ONLY
 *    > 0 if root is mounted READ ONLY
 *
 * NOTES: borrowed some of this routine from e2fsck
 */
int Is_Root_Mounted_RO()
{
  int  fd, rc=0;

  #define TEST_FILE "/.ismount-test-file"		
  
  fd = open(TEST_FILE, O_RDWR|O_CREAT);

  if (fd < 0) {
    if (errno == EROFS)
      rc = MSG_JFS_VOLUME_IS_MOUNTED_RO;
  } else {
    close(fd);
    (void) unlink(TEST_FILE);
  }

  return rc;
}


/*--------------------------------------------------------------------
 * NAME: Is_Device_Type_JFS
 *
 * FUNCTION: Determine if the device specified is of type JFS in /etc/fstab
 *
 * PRE CONDITIONS: Device_Name must be the name of the device.
 *
 * POST CONDITIONS:
 *
 * PARAMETERS: Device_Name - The name of the device.
 *
 * RETURNS:
 *      0 if the device is type JFS in /etc/fstab
 *    > 0 if the device is not type JFS or an error occurs.
 *
 */
int Is_Device_Type_JFS(char  *Device_Name)
{

  FILE           *FS_Info_File;   /* Pointer for system's mount records */
  struct mntent  *FS_Data;        /* Holds data for entry in mount list */
  int            Is_JFS_RC = 0;   /* Holds return code.             */

  /* Attempt to open /etc/fstab for access. */
  if ( (FS_Info_File = setmntent(_PATH_MNTTAB,"r")) == NULL ) {
    /* Unable to access mount description file /etc/fstab!  */
    return MSG_JFS_MNT_LIST_ERROR;
  }

  /* Attempt to find specified device name in filesystem records */
  while ((FS_Data = getmntent (FS_Info_File)) != NULL)
    if (strcmp(Device_Name, FS_Data->mnt_fsname) == 0)
	  break;

  if (FS_Data) {
    if ( strcmp(FS_Data->mnt_type, "jfs") != 0 )
      /* is mounted, is type jfs */
      Is_JFS_RC = MSG_JFS_NOT_JFS;
  } else {
    Is_JFS_RC = MSG_JFS_DEV_NOT_IN_TABLE;
  }

  /* Close the stream. */
  endmntent(FS_Info_File);

  /* Return the appropriate value. */
  return Is_JFS_RC;
}
