\ hp54750.4th
\
\ GPIB interface to HP54750 digitizing scope
\
\ Requires the following source files:
\
\	matrix.4th
\	files.4th
\	strings.4th
\	matfiles.4th
\	gpib.4th
\
\ Revisions:
\
\	3-23-1999  ported from UR/FORTH  KM
\	3-26-1999  completed port  KM
\	5-7-1999   added waveform to file storage KM
\

7 constant SCOPE                                                     
variable nwfmpts                                                
fvariable hp_yoffset
fvariable hp_yscale

create hp_waveform 2 4096 * 8 * 8 + allot
                                                                
: get_gpib_fnumber ( n -- f | convert gpib ascii to float )     
\ n is the number of characters in the ascii string             
    dup gpib_in_buf fnumber_buf 1+ rot cmove                   
    dup fnumber_buf c!                                          
    1+ 0 fnumber_buf rot + c!                                   
    fnumber_buf string>f ; 

: reset_scope ( -- )                                            
   " *RST" SCOPE send_command ;                                 
                                                                
: autoscale ( -- )                                              
   " :AUT" SCOPE send_command ;                                 
                                                                
: clear_display ( -- )                                          
   " :CDIS" SCOPE send_command ;    

: set_record_length ( n -- | set the number of points )         
    s>string count                                              
    " ACQ:POIN " count 2swap strcat strpck                      
    SCOPE send_command ;

: set_averaging ( n -- | averaging on/off if n true/false )     
    if " ACQ:AVER ON" else " ACQ:AVER OFF" then                 
    SCOPE send_command ;

: set_num_averages ( n -- | set number of averages )            
    s>string count " :ACQ:COUN " count                          
    2swap strcat strpck SCOPE send_command ;

: get_record_length ( -- n | get number of points )             
    " :WAV:POIN?" SCOPE send_command                            
    6 SCOPE read_bytes                                          
    4 get_gpib_fnumber f>s                                     
    dup nwfmpts ! ;        

: download_waveform ( ch -- | download waveform data )          
    s>string count " :WAV:SOUR CHAN" count                      
    2swap strcat strpck                                         
    SCOPE send_command                                          
    " :WAV:FORM WORD" SCOPE send_command                        
    get_record_length                                           
    2* 10 +                      \ number of bytes in record     
    " :WAV:DATA?" SCOPE send_command                            
    SCOPE read_bytes ;

: get_waveform ( n -- rc | read ch n waveform onto stack )      
    download_waveform                                           
    iberr if cr ." SCOPE GPIB TIMEOUT" cr then                        
    gpib_in_buf 9 +      \ copy waveform onto stack            
    nwfmpts @ 0 do dup w@ swap 2+ loop 
    drop nwfmpts @ ;

: get_time_position ( -- f | return the x offset )              
    " :TIMEBASE:POSITION?" SCOPE send_command                   
    14 SCOPE read_bytes                                         
    11 get_gpib_fnumber ;    

: get_time_range ( -- f | return full scale time in sec )       
    " :TIMEBASE:RANGE?" SCOPE send_command                      
    12 SCOPE read_bytes                                         
    9 get_gpib_fnumber ;

: get_time_increment ( -- f | delta t between points in sec )   
    get_time_range                                              
    get_record_length                                           
    s>f f/ ; 

: get_channel_offset ( n -- f | return ch n vertical offset in volts )
	s>string " CHANNEL" count rot count strcat
	" :OFFSET?" count strcat strpck SCOPE send_command
	12 SCOPE read_bytes
	9 get_gpib_fnumber ;

: get_channel_scale ( n -- f | return ch n vertical scale in volts/div )
	s>string " CHANNEL" count rot count strcat
	" :SCALE?" count strcat strpck SCOPE send_command
	12 SCOPE read_bytes
	9 get_gpib_fnumber ;

: get_yinc ( -- f | return the volts/unit scale factor for the waveform )
	" :WAVEFORM:YINCREMENT?" SCOPE send_command
	16 SCOPE read_bytes
	11 get_gpib_fnumber ;

: get_yorigin ( -- f | return vertical y value for data count of zero )
	gpib_in_buf 14 32 fill
	" :WAVEFORM:YORIGIN?" SCOPE send_command
	16 SCOPE read_bytes
	12 get_gpib_fnumber ;

: record_waveform ( n -- | download ch n waveform into fmatrix hp_waveform )
	download_waveform
	iberr if
	  cr ." SCOPE GPIB TIMEOUT" cr
	  exit
	then
	nwfmpts @ 2 hp_waveform mat_size!

	\ fill first column of hp_waveform with time in seconds

	get_time_position
	get_time_increment
	nwfmpts @ 0 do 
	  fover i 1+ 1 hp_waveform fmat!
	  fdup >r >r f+ r> r>
	loop
	fdrop fdrop

	\ fill second column of hp_waveform with waveform data in volts

	get_yinc hp_yscale f!
	get_yorigin hp_yoffset f!

	gpib_in_buf 9 +
	nwfmpts @ 0 do
	  dup w@
	  s>f hp_yscale f@ f* hp_yoffset f@ f+	   
	  i 1+ 2 hp_waveform fmat!
	  2+	  
	loop
	drop ;
	     

: write_waveform ( ^str -- | write the waveform to specified file )
	hp_waveform swap write_fmat_ascii_file ;



