/*******************************************************************************
 * Copyright (c) 2008-2011 The Khronos Group Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and/or associated documentation files (the
 * "Materials"), to deal in the Materials without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Materials, and to
 * permit persons to whom the Materials are furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Materials.
 *
 * THE MATERIALS ARE PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * MATERIALS OR THE USE OR OTHER DEALINGS IN THE MATERIALS.
 ******************************************************************************/

/*! \file
 *
 *   \brief C++ bindings for OpenCL 1.0 (rev 48) and OpenCL 1.1 (rev 33)    
 *   \author Benedict R. Gaster and Laurent Morichetti
 *   
 *   Additions and fixes from Brian Cole, March 3rd 2010.
 *   
 *   \version 1.1
 *   \date June 2010
 *
 *   Optional extension support
 *
 *         cl
 *         cl_ext_device_fission
 *				#define USE_CL_DEVICE_FISSION
 */

/*! \mainpage
 * \section intro Introduction
 * For many large applications C++ is the language of choice and so it seems
 * reasonable to define C++ bindings for OpenCL.
 *
 *
 * The interface is contained with a single C++ header file \em cl.hpp and all
 * definitions are contained within the namespace \em cl. There is no additional
 * requirement to include \em cl.h and to use either the C++ or original C
 * bindings it is enough to simply include \em cl.hpp.
 *
 * The bindings themselves are lightweight and correspond closely to the
 * underlying C API. Using the C++ bindings introduces no additional execution
 * overhead.
 *
 * For detail documentation on the bindings see:
 *
 * The OpenCL C++ Wrapper API 1.1 (revision 04)
 *  http://www.khronos.org/registry/cl/specs/opencl-cplusplus-1.1.pdf
 *
 * \section example Example
 *
 * The following example shows a general use case for the C++
 * bindings, including support for the optional exception feature and
 * also the supplied vector and string classes, see following sections for
 * decriptions of these features.
 *
 * \code
 * #define __CL_ENABLE_EXCEPTIONS
 * 
 * #if defined(__APPLE__) || defined(__MACOSX)
 * #include <OpenCL/cl.hpp>
 * #else
 * #include <CL/cl.hpp>
 * #endif
 * #include <cstdio>
 * #include <cstdlib>
 * #include <iostream>
 * 
 *  const char * helloStr  = "__kernel void "
 *                           "hello(void) "
 *                           "{ "
 *                           "  "
 *                           "} ";
 * 
 *  int
 *  main(void)
 *  {
 *     cl_int err = CL_SUCCESS;
 *     try {
 *
 *       std::vector<cl::Platform> platforms;
 *       cl::Platform::get(&platforms);
 *       if (platforms.size() == 0) {
 *           std::cout << "Platform size 0\n";
 *           return -1;
 *       }
 *
 *       cl_context_properties properties[] = 
 *          { CL_CONTEXT_PLATFORM, (cl_context_properties)(platforms[0])(), 0};
 *       cl::Context context(CL_DEVICE_TYPE_CPU, properties); 
 * 
 *       std::vector<cl::Device> devices = context.getInfo<CL_CONTEXT_DEVICES>();
 * 
 *       cl::Program::Sources source(1,
 *           std::make_pair(helloStr,strlen(helloStr)));
 *       cl::Program program_ = cl::Program(context, source);
 *       program_.build(devices);
 * 
 *       cl::Kernel kernel(program_, "hello", &err);
 * 
 *       cl::Event event;
 *       cl::CommandQueue queue(context, devices[0], 0, &err);
 *       queue.enqueueNDRangeKernel(
 *           kernel, 
 *           cl::NullRange, 
 *           cl::NDRange(4,4),
 *           cl::NullRange,
 *           NULL,
 *           &event); 
 * 
 *       event.wait();
 *     }
 *     catch (cl::Error err) {
 *        std::cerr 
 *           << "ERROR: "
 *           << err.what()
 *           << "("
 *           << err.err()
 *           << ")"
 *           << std::endl;
 *     }
 * 
 *    return EXIT_SUCCESS;
 *  }
 * 
 * \endcode
 *
 */
#ifndef CL_HPP_
#define CL_HPP_

#ifdef _WIN32
#include <windows.h>
#include <malloc.h>
#pragma push_macro("max")
#undef max
#if defined(USE_DX_INTEROP)
#include <CL/cl_d3d10.h>
#endif
#endif // _WIN32

// 
#if defined(USE_CL_DEVICE_FISSION)
#include <CL/cl_ext.h>
#endif

#if defined(__APPLE__) || defined(__MACOSX)
#include <OpenGL/OpenGL.h>
#include <OpenCL/opencl.h>
#include <libkern/OSAtomic.h>
#else
#include <GL/gl.h>
#include <CL/opencl.h>
#endif // !__APPLE__

#if !defined(CL_CALLBACK)
#define CL_CALLBACK
#endif //CL_CALLBACK

#include <utility>
#include <limits>

#if !defined(__NO_STD_VECTOR)
#include <vector>
#endif

#if !defined(__NO_STD_STRING)
#include <string>
#endif 

#if defined(linux) || defined(__APPLE__) || defined(__MACOSX)
# include <alloca.h>
#endif // linux

#include <cstring>

#if defined(_WIN32)
#include <functional>
#else
#include <tr1/functional>
#endif

/*! \namespace cl
 *
 * \brief The OpenCL C++ bindings are defined within this namespace.
 *
 */
namespace cl {

#define __INIT_CL_EXT_FCN_PTR(name) \
    if(!pfn_##name) { \
        pfn_##name = (PFN_##name) \
            clGetExtensionFunctionAddress(#name); \
        if(!pfn_##name) { \
        } \
    }

class Program;
class Device;
class Context;
class CommandQueue;
class Memory;

#if defined(__CL_ENABLE_EXCEPTIONS)
#include <exception>
/*! \class Error
 * \brief Exception class
 */
class Error : public std::exception
{
private:
    cl_int err_;
    const char * errStr_;
public:
    /*! Create a new CL error exception for a given error code
     *  and corresponding message.
     */
    Error(cl_int err, const char * errStr = NULL) : err_(err), errStr_(errStr)
    {}

    ~Error() throw() {}

    /*! \brief Get error string associated with exception
     *
     * \return A memory pointer to the error message string.
     */
    virtual const char * what() const throw ()
    {
        if (errStr_ == NULL) {
            return "empty";
        }
        else {
            return errStr_;
        }
    }

    /*! \brief Get error code associated with exception
     *
     *  \return The error code.
     */
    cl_int err(void) const { return err_; }
};

#define __ERR_STR(x) #x
#else
#define __ERR_STR(x) NULL
#endif // __CL_ENABLE_EXCEPTIONS

//! \cond DOXYGEN_DETAIL
#if !defined(__CL_USER_OVERRIDE_ERROR_STRINGS)
#define __GET_DEVICE_INFO_ERR               __ERR_STR(clGetDeviceInfo)
#define __GET_PLATFORM_INFO_ERR             __ERR_STR(clGetPlatformInfo)
#define __GET_DEVICE_IDS_ERR                __ERR_STR(clGetDeviceIDs)
#define __GET_PLATFORM_IDS_ERR              __ERR_STR(clGetPlatformIDs)
#define __GET_CONTEXT_INFO_ERR              __ERR_STR(clGetContextInfo)
#define __GET_EVENT_INFO_ERR                __ERR_STR(clGetEventInfo)
#define __GET_EVENT_PROFILE_INFO_ERR        __ERR_STR(clGetEventProfileInfo)
#define __GET_MEM_OBJECT_INFO_ERR           __ERR_STR(clGetMemObjectInfo)
#define __GET_IMAGE_INFO_ERR                __ERR_STR(clGetImageInfo)
#define __GET_SAMPLER_INFO_ERR              __ERR_STR(clGetSamplerInfo)
#define __GET_KERNEL_INFO_ERR               __ERR_STR(clGetKernelInfo)
#define __GET_KERNEL_WORK_GROUP_INFO_ERR    __ERR_STR(clGetKernelWorkGroupInfo)
#define __GET_PROGRAM_INFO_ERR              __ERR_STR(clGetProgramInfo)
#define __GET_PROGRAM_BUILD_INFO_ERR        __ERR_STR(clGetProgramBuildInfo)
#define __GET_COMMAND_QUEUE_INFO_ERR        __ERR_STR(clGetCommandQueueInfo)

#define __CREATE_CONTEXT_ERR                __ERR_STR(clCreateContext)
#define __CREATE_CONTEXT_FROM_TYPE_ERR      __ERR_STR(clCreateContextFromType)
#define __GET_SUPPORTED_IMAGE_FORMATS_ERR   __ERR_STR(clGetSupportedImageFormats)

#define __CREATE_BUFFER_ERR                 __ERR_STR(clCreateBuffer)
#define __CREATE_SUBBUFFER_ERR              __ERR_STR(clCreateSubBuffer)
#define __CREATE_GL_BUFFER_ERR              __ERR_STR(clCreateFromGLBuffer)
#define __CREATE_GL_RENDER_BUFFER_ERR       __ERR_STR(clCreateFromGLBuffer)
#define __CREATE_GL_TEXTURE_2D_ERR          __ERR_STR(clCreateFromGLTexture2D)
#define __CREATE_GL_TEXTURE_3D_ERR          __ERR_STR(clCreateFromGLTexture3D)
#define __GET_GL_OBJECT_INFO_ERR            __ERR_STR(clGetGLObjectInfo)
#define __CREATE_IMAGE2D_ERR                __ERR_STR(clCreateImage2D)
#define __CREATE_IMAGE3D_ERR                __ERR_STR(clCreateImage3D)
#define __CREATE_SAMPLER_ERR                __ERR_STR(clCreateSampler)
#define __SET_MEM_OBJECT_DESTRUCTOR_CALLBACK_ERR __ERR_STR(clSetMemObjectDestructorCallback)

#define __CREATE_USER_EVENT_ERR             __ERR_STR(clCreateUserEvent)
#define __SET_USER_EVENT_STATUS_ERR         __ERR_STR(clSetUserEventStatus)
#define __SET_EVENT_CALLBACK_ERR            __ERR_STR(clSetEventCallback)
#define __WAIT_FOR_EVENTS_ERR               __ERR_STR(clWaitForEvents)

#define __CREATE_KERNEL_ERR                 __ERR_STR(clCreateKernel)
#define __SET_KERNEL_ARGS_ERR               __ERR_STR(clSetKernelArg)
#define __CREATE_PROGRAM_WITH_SOURCE_ERR    __ERR_STR(clCreateProgramWithSource)
#define __CREATE_PROGRAM_WITH_BINARY_ERR    __ERR_STR(clCreateProgramWithBinary)
#define __BUILD_PROGRAM_ERR                 __ERR_STR(clBuildProgram)
#define __CREATE_KERNELS_IN_PROGRAM_ERR     __ERR_STR(clCreateKernelsInProgram)

#define __CREATE_COMMAND_QUEUE_ERR          __ERR_STR(clCreateCommandQueue)
#define __SET_COMMAND_QUEUE_PROPERTY_ERR    __ERR_STR(clSetCommandQueueProperty)
#define __ENQUEUE_READ_BUFFER_ERR           __ERR_STR(clEnqueueReadBuffer)
#define __ENQUEUE_READ_BUFFER_RECT_ERR      __ERR_STR(clEnqueueReadBufferRect)
#define __ENQUEUE_WRITE_BUFFER_ERR          __ERR_STR(clEnqueueWriteBuffer)
#define __ENQUEUE_WRITE_BUFFER_RECT_ERR     __ERR_STR(clEnqueueWriteBufferRect)
#define __ENQEUE_COPY_BUFFER_ERR            __ERR_STR(clEnqueueCopyBuffer)
#define __ENQEUE_COPY_BUFFER_RECT_ERR       __ERR_STR(clEnqueueCopyBufferRect)
#define __ENQUEUE_READ_IMAGE_ERR            __ERR_STR(clEnqueueReadImage)
#define __ENQUEUE_WRITE_IMAGE_ERR           __ERR_STR(clEnqueueWriteImage)
#define __ENQUEUE_COPY_IMAGE_ERR            __ERR_STR(clEnqueueCopyImage)
#define __ENQUEUE_COPY_IMAGE_TO_BUFFER_ERR  __ERR_STR(clEnqueueCopyImageToBuffer)
#define __ENQUEUE_COPY_BUFFER_TO_IMAGE_ERR  __ERR_STR(clEnqueueCopyBufferToImage)
#define __ENQUEUE_MAP_BUFFER_ERR            __ERR_STR(clEnqueueMapBuffer)
#define __ENQUEUE_MAP_IMAGE_ERR             __ERR_STR(clEnqueueMapImage)
#define __ENQUEUE_UNMAP_MEM_OBJECT_ERR      __ERR_STR(clEnqueueUnMapMemObject)
#define __ENQUEUE_NDRANGE_KERNEL_ERR        __ERR_STR(clEnqueueNDRangeKernel)
#define __ENQUEUE_TASK_ERR                  __ERR_STR(clEnqueueTask)
#define __ENQUEUE_NATIVE_KERNEL             __ERR_STR(clEnqueueNativeKernel)
#define __ENQUEUE_MARKER_ERR                __ERR_STR(clEnqueueMarker)
#define __ENQUEUE_WAIT_FOR_EVENTS_ERR       __ERR_STR(clEnqueueWaitForEvents)
#define __ENQUEUE_BARRIER_ERR               __ERR_STR(clEnqueueBarrier)

#define __ENQUEUE_ACQUIRE_GL_ERR            __ERR_STR(clEnqueueAcquireGLObjects)
#define __ENQUEUE_RELEASE_GL_ERR            __ERR_STR(clEnqueueReleaseGLObjects)

#define __UNLOAD_COMPILER_ERR               __ERR_STR(clUnloadCompiler)

#define __FLUSH_ERR                         __ERR_STR(clFlush)
#define __FINISH_ERR                        __ERR_STR(clFinish)

#define __CREATE_SUB_DEVICES                __ERR_STR(clCreateSubDevicesEXT)
#endif // __CL_USER_OVERRIDE_ERROR_STRINGS
//! \endcond

/*! \class string
 * \brief Simple string class, that provides a limited subset of std::string
 * functionality but avoids many of the issues that come with that class.
 */
class string
{
private:
    ::size_t size_;
    char * str_;
public:
    string(void) : size_(0), str_(NULL)
    {
    }

    string(char * str, ::size_t size) :
        size_(size),
        str_(NULL)
    {
        str_ = new char[size_+1];
        if (str_ != NULL) {
            memcpy(str_, str, size_  * sizeof(char));
            str_[size_] = '\0';
        }
        else {
            size_ = 0;
        }
    }

    string(char * str) :
        str_(NULL)
    {
        size_= ::strlen(str);
        str_ = new char[size_ + 1];
        if (str_ != NULL) {
            memcpy(str_, str, (size_ + 1) * sizeof(char));
        }
        else {
            size_ = 0;
        }
    }

    string& operator=(const string& rhs)
    {
        if (this == &rhs) {
            return *this;
        }

        if (rhs.size_ == 0 || rhs.str_ == NULL) {
            size_ = 0;
            str_  = NULL;
        } 
        else {
            size_ = rhs.size_;
            str_ = new char[size_ + 1];
            if (str_ != NULL) {
                memcpy(str_, rhs.str_, (size_ + 1) * sizeof(char));
            }
            else {
                size_ = 0;
            }
        }

        return *this;
    }

    string(const string& rhs)
    {
        *this = rhs;
    }

    ~string()
    {
        if (str_ != NULL) {
            delete[] str_;
        }
    }

    ::size_t size(void) const   { return size_; }
    ::size_t length(void) const { return size(); }

    const char * c_str(void) const { return (str_) ? str_ : "";}
};

#if !defined(__USE_DEV_STRING) && !defined(__NO_STD_STRING)
#include <string>
typedef std::string STRING_CLASS;
#elif !defined(__USE_DEV_STRING) 
typedef cl::string STRING_CLASS;
#endif

#if !defined(__USE_DEV_VECTOR) && !defined(__NO_STD_VECTOR)
#include <vector>
#define VECTOR_CLASS std::vector
#elif !defined(__USE_DEV_VECTOR) 
#define VECTOR_CLASS cl::vector 
#endif

#if !defined(__MAX_DEFAULT_VECTOR_SIZE)
#define __MAX_DEFAULT_VECTOR_SIZE 10
#endif

/*! \class vector
 * \brief Fixed sized vector implementation that mirroring 
 * std::vector functionality.
 */
template <typename T, unsigned int N = __MAX_DEFAULT_VECTOR_SIZE>
class vector
{
private:
    T data_[N];
    unsigned int size_;
    bool empty_;
public:
    vector() : 
        size_(static_cast<unsigned int>(-1)),
        empty_(true)
    {}

    ~vector() {}

    unsigned int size(void) const
    {
        return size_ + 1;
    }

    void clear()
    {
        size_ = -1;
        empty_ = true;
    }

    void push_back (const T& x)
    { 
        if (size() < N) {
            size_++;  
            data_[size_] = x;
            empty_ = false;
        }
    }

    void pop_back(void)
    {
        if (!empty_) {
            data_[size_].~T();
            size_--;
            if (size_ == -1) {
                empty_ = true;
            }
        }
    }
  
    vector(const vector<T, N>& vec) : 
        size_(vec.size_),
        empty_(vec.empty_)
    {
        if (!empty_) {
            memcpy(&data_[0], &vec.data_[0], size() * sizeof(T));
        }
    } 

    vector(unsigned int size, const T& val = T()) :
        size_(-1),
        empty_(true)
    {
        for (unsigned int i = 0; i < size; i++) {
            push_back(val);
        }
    }

    vector<T, N>& operator=(const vector<T, N>& rhs)
    {
        if (this == &rhs) {
            return *this;
        }

        size_  = rhs.size_;
        empty_ = rhs.empty_;

        if (!empty_) {	
            memcpy(&data_[0], &rhs.data_[0], size() * sizeof(T));
        }
    
        return *this;
    }

    bool operator==(vector<T,N> &vec)
    {
        if (empty_ && vec.empty_) {
            return true;
        }

        if (size() != vec.size()) {
            return false;
        }

        return memcmp(&data_[0], &vec.data_[0], size() * sizeof(T)) == 0 ? true : false;
    }
  
    operator T* ()             { return data_; }
    operator const T* () const { return data_; }
   
    bool empty (void) const
    {
        return empty_;
    }
  
    unsigned int max_size (void) const
    {
        return N;
    }

    unsigned int capacity () const
   {
        return sizeof(T) * N;
    }

    T& operator[](int index)
    {
        return data_[index];
    }
  
    T operator[](int index) const
    {
        return data_[index];
    }
  
    template<class I>
    void assign(I start, I end)
    {
        clear();   
        while(start < end) {
            push_back(*start);
            start++;
        }
    }

    /*! \class iterator
     * \brief Iterator class for vectors
     */
    class iterator
    {
    private:
        vector<T,N> vec_;
        int index_;
        bool initialized_;
    public:
        iterator(void) : 
            index_(-1),
            initialized_(false)
        {
            index_ = -1;
            initialized_ = false;
        }

        ~iterator(void) {}

        static iterator begin(vector<T,N> &vec)
        {
            iterator i;

            if (!vec.empty()) {
                i.index_ = 0;
            }

            i.vec_ = vec;
            i.initialized_ = true;
            return i;
        }

        static iterator end(vector<T,N> &vec)
        {
            iterator i;

            if (!vec.empty()) {
                i.index_ = vec.size();
            }
            i.vec_ = vec;
            i.initialized_ = true;
            return i;
        }
    
        bool operator==(iterator i)
        {
            return ((vec_ == i.vec_) && 
                    (index_ == i.index_) && 
                    (initialized_ == i.initialized_));
        }

        bool operator!=(iterator i)
        {
            return (!(*this==i));
        }

        void operator++()
        {
            index_++;
        }

        void operator++(int x)
        {
            index_ += x;
        }

        void operator--()
        {
            index_--;
        }

        void operator--(int x)
        {
            index_ -= x;
        }

        T operator *()
        {
            return vec_[index_];
        }
    };

    iterator begin(void)
    {
        return iterator::begin(*this);
    }

    iterator end(void)
    {
        return iterator::end(*this);
    }

    T& front(void)
    {
        return data_[0];
    }

    T& back(void)
    {
        return data_[size_];
    }

    const T& front(void) const
    {
        return data_[0];
    }

    const T& back(void) const
    {
        return data_[size_];
    }
};  

namespace detail {
#define __DEFAULT_NOT_INITIALIZED 1
#define __DEFAULT_BEING_INITIALIZED 2
#define __DEFAULT_INITIALIZED 4

    /*
     * Compare and exchange primitives are needed for handling of defaults
    */
    inline int compare_exchange(volatile int * dest, int exchange, int comparand)
    {
#ifdef _WIN32
        return (int)(InterlockedCompareExchange(
           (volatile long*)dest, 
           (long)exchange, 
           (long)comparand));
#elif defined(__APPLE__) || defined(__MACOSX)
		return OSAtomicOr32Orig((uint32_t)exchange, (volatile uint32_t*)dest);
#else // !_WIN32 || defined(__APPLE__) || defined(__MACOSX)
        return (__sync_val_compare_and_swap(
            dest, 
            comparand, 
            exchange));
#endif // !_WIN32
    }
}; // namespace detail

    
/*!
 * \brief size_t class used to interface between C++ and
 * OpenCL C calls that require arrays of size_t values, who's
 * size is known statically.
 */
template <int N>
struct size_t : public cl::vector< ::size_t, N> { };

namespace detail {

// GetInfo help struct
template <typename Functor, typename T>
struct GetInfoHelper
{
    static cl_int
    get(Functor f, cl_uint name, T* param)
    {
        return f(name, sizeof(T), param, NULL);
    }
};

// Specialized GetInfoHelper for VECTOR_CLASS params
template <typename Func, typename T>
struct GetInfoHelper<Func, VECTOR_CLASS<T> >
{
    static cl_int get(Func f, cl_uint name, VECTOR_CLASS<T>* param)
    {
        ::size_t required;
        cl_int err = f(name, 0, NULL, &required);
        if (err != CL_SUCCESS) {
            return err;
        }

        T* value = (T*) alloca(required);
        err = f(name, required, value, NULL);
        if (err != CL_SUCCESS) {
            return err;
        }

        param->assign(&value[0], &value[required/sizeof(T)]);
        return CL_SUCCESS;
    }
};

// Specialized for getInfo<CL_PROGRAM_BINARIES>
template <typename Func>
struct GetInfoHelper<Func, VECTOR_CLASS<char *> >
{
    static cl_int
    get(Func f, cl_uint name, VECTOR_CLASS<char *>* param)
    {
      ::size_t nDevices;
      ::size_t * binary_sizes;
      char ** values;

      cl_int err = f(CL_PROGRAM_NUM_DEVICES, sizeof(nDevices), &nDevices, NULL);
      if (err != CL_SUCCESS) {
          return err;
      }

      binary_sizes = (::size_t*)alloca(sizeof(::size_t)*nDevices);
      err = f(CL_PROGRAM_BINARY_SIZES, sizeof(::size_t)*nDevices, binary_sizes, NULL);
      if (err != CL_SUCCESS) {
          return err;
      }

      values = (char **) alloca(sizeof(char*)*nDevices);
      for(cl_uint i = 0; i < nDevices; i++ )
      {
        if( binary_sizes[i] != 0 )
        {     		 
            values[i]= (char *)alloca( sizeof(char)*binary_sizes[i]);
        }
        else
        {
            values[i] = NULL;
        }
      }
      err = f(name, sizeof(char *)*nDevices, values, NULL);
      if (err != CL_SUCCESS) {
        return err;
      }
      
      param->assign(values,values+nDevices);
      return CL_SUCCESS;
    }
};

// Specialized GetInfoHelper for STRING_CLASS params
template <typename Func>
struct GetInfoHelper<Func, STRING_CLASS>
{
    static cl_int get(Func f, cl_uint name, STRING_CLASS* param)
    {
        ::size_t required;
        cl_int err = f(name, 0, NULL, &required);
        if (err != CL_SUCCESS) {
            return err;
        }

        char* value = (char*) alloca(required);
        err = f(name, required, value, NULL);
        if (err != CL_SUCCESS) {
            return err;
        }

        *param = value;
        return CL_SUCCESS;
    }
};

// Specialized GetInfoHelper for cl::size_t params
template <typename Func, ::size_t N>
struct GetInfoHelper<Func, size_t<N> >
{
    static cl_int get(Func f, cl_uint name, size_t<N>* param)
    {
        ::size_t required;
        cl_int err = f(name, 0, NULL, &required);
        if (err != CL_SUCCESS) {
            return err;
        }

        ::size_t* value = (::size_t*) alloca(required);
        err = f(name, required, value, NULL);
        if (err != CL_SUCCESS) {
            return err;
        }

        param->assign(&value[0], &value[required/sizeof(::size_t)]);
        return CL_SUCCESS;
    }
};

#define __GET_INFO_HELPER_WITH_RETAIN(CPP_TYPE) \
namespace detail { \
template <typename Func> \
struct GetInfoHelper<Func, CPP_TYPE> \
{ \
    static cl_int get(Func f, cl_uint name, CPP_TYPE* param) \
    { \
      cl_uint err = f(name, sizeof(CPP_TYPE), param, NULL); \
      if (err != CL_SUCCESS) { \
        return err; \
      } \
      \
      return ReferenceHandler<CPP_TYPE::cl_type>::retain((*param)()); \
    } \
}; \
} 


#define __PARAM_NAME_INFO_1_0(F) \
    F(cl_platform_info, CL_PLATFORM_PROFILE, STRING_CLASS) \
    F(cl_platform_info, CL_PLATFORM_VERSION, STRING_CLASS) \
    F(cl_platform_info, CL_PLATFORM_NAME, STRING_CLASS) \
    F(cl_platform_info, CL_PLATFORM_VENDOR, STRING_CLASS) \
    F(cl_platform_info, CL_PLATFORM_EXTENSIONS, STRING_CLASS) \
    \
    F(cl_device_info, CL_DEVICE_TYPE, cl_device_type) \
    F(cl_device_info, CL_DEVICE_VENDOR_ID, cl_uint) \
    F(cl_device_info, CL_DEVICE_MAX_COMPUTE_UNITS, cl_uint) \
    F(cl_device_info, CL_DEVICE_MAX_WORK_ITEM_DIMENSIONS, cl_uint) \
    F(cl_device_info, CL_DEVICE_MAX_WORK_GROUP_SIZE, ::size_t) \
    F(cl_device_info, CL_DEVICE_MAX_WORK_ITEM_SIZES, VECTOR_CLASS< ::size_t>) \
    F(cl_device_info, CL_DEVICE_PREFERRED_VECTOR_WIDTH_CHAR, cl_uint) \
    F(cl_device_info, CL_DEVICE_PREFERRED_VECTOR_WIDTH_SHORT, cl_uint) \
    F(cl_device_info, CL_DEVICE_PREFERRED_VECTOR_WIDTH_INT, cl_uint) \
    F(cl_device_info, CL_DEVICE_PREFERRED_VECTOR_WIDTH_LONG, cl_uint) \
    F(cl_device_info, CL_DEVICE_PREFERRED_VECTOR_WIDTH_FLOAT, cl_uint) \
    F(cl_device_info, CL_DEVICE_PREFERRED_VECTOR_WIDTH_DOUBLE, cl_uint) \
    F(cl_device_info, CL_DEVICE_MAX_CLOCK_FREQUENCY, cl_uint) \
    F(cl_device_info, CL_DEVICE_ADDRESS_BITS, cl_uint) \
    F(cl_device_info, CL_DEVICE_MAX_READ_IMAGE_ARGS, cl_uint) \
    F(cl_device_info, CL_DEVICE_MAX_WRITE_IMAGE_ARGS, cl_uint) \
    F(cl_device_info, CL_DEVICE_MAX_MEM_ALLOC_SIZE, cl_ulong) \
    F(cl_device_info, CL_DEVICE_IMAGE2D_MAX_WIDTH, ::size_t) \
    F(cl_device_info, CL_DEVICE_IMAGE2D_MAX_HEIGHT, ::size_t) \
    F(cl_device_info, CL_DEVICE_IMAGE3D_MAX_WIDTH, ::size_t) \
    F(cl_device_info, CL_DEVICE_IMAGE3D_MAX_HEIGHT, ::size_t) \
    F(cl_device_info, CL_DEVICE_IMAGE3D_MAX_DEPTH, ::size_t) \
    F(cl_device_info, CL_DEVICE_IMAGE_SUPPORT, cl_bool) \
    F(cl_device_info, CL_DEVICE_MAX_PARAMETER_SIZE, ::size_t) \
    F(cl_device_info, CL_DEVICE_MAX_SAMPLERS, cl_uint) \
    F(cl_device_info, CL_DEVICE_MEM_BASE_ADDR_ALIGN, cl_uint) \
    F(cl_device_info, CL_DEVICE_MIN_DATA_TYPE_ALIGN_SIZE, cl_uint) \
    F(cl_device_info, CL_DEVICE_SINGLE_FP_CONFIG, cl_device_fp_config) \
    F(cl_device_info, CL_DEVICE_GLOBAL_MEM_CACHE_TYPE, cl_device_mem_cache_type) \
    F(cl_device_info, CL_DEVICE_GLOBAL_MEM_CACHELINE_SIZE, cl_uint)\
    F(cl_device_info, CL_DEVICE_GLOBAL_MEM_CACHE_SIZE, cl_ulong) \
    F(cl_device_info, CL_DEVICE_GLOBAL_MEM_SIZE, cl_ulong) \
    F(cl_device_info, CL_DEVICE_MAX_CONSTANT_BUFFER_SIZE, cl_ulong) \
    F(cl_device_info, CL_DEVICE_MAX_CONSTANT_ARGS, cl_uint) \
    F(cl_device_info, CL_DEVICE_LOCAL_MEM_TYPE, cl_device_local_mem_type) \
    F(cl_device_info, CL_DEVICE_LOCAL_MEM_SIZE, cl_ulong) \
    F(cl_device_info, CL_DEVICE_ERROR_CORRECTION_SUPPORT, cl_bool) \
    F(cl_device_info, CL_DEVICE_PROFILING_TIMER_RESOLUTION, ::size_t) \
    F(cl_device_info, CL_DEVICE_ENDIAN_LITTLE, cl_bool) \
    F(cl_device_info, CL_DEVICE_AVAILABLE, cl_bool) \
    F(cl_device_info, CL_DEVICE_COMPILER_AVAILABLE, cl_bool) \
    F(cl_device_info, CL_DEVICE_EXECUTION_CAPABILITIES, cl_device_exec_capabilities) \
    F(cl_device_info, CL_DEVICE_QUEUE_PROPERTIES, cl_command_queue_properties) \
    F(cl_device_info, CL_DEVICE_PLATFORM, cl_platform_id) \
    F(cl_device_info, CL_DEVICE_NAME, STRING_CLASS) \
    F(cl_device_info, CL_DEVICE_VENDOR, STRING_CLASS) \
    F(cl_device_info, CL_DRIVER_VERSION, STRING_CLASS) \
    F(cl_device_info, CL_DEVICE_PROFILE, STRING_CLASS) \
    F(cl_device_info, CL_DEVICE_VERSION, STRING_CLASS) \
    F(cl_device_info, CL_DEVICE_EXTENSIONS, STRING_CLASS) \
    \
    F(cl_context_info, CL_CONTEXT_REFERENCE_COUNT, cl_uint) \
    F(cl_context_info, CL_CONTEXT_DEVICES, VECTOR_CLASS<Device>) \
    F(cl_context_info, CL_CONTEXT_PROPERTIES, VECTOR_CLASS<cl_context_properties>) \
    \
    F(cl_event_info, CL_EVENT_COMMAND_QUEUE, cl::CommandQueue) \
    F(cl_event_info, CL_EVENT_COMMAND_TYPE, cl_command_type) \
    F(cl_event_info, CL_EVENT_REFERENCE_COUNT, cl_uint) \
    F(cl_event_info, CL_EVENT_COMMAND_EXECUTION_STATUS, cl_uint) \
    \
    F(cl_profiling_info, CL_PROFILING_COMMAND_QUEUED, cl_ulong) \
    F(cl_profiling_info, CL_PROFILING_COMMAND_SUBMIT, cl_ulong) \
    F(cl_profiling_info, CL_PROFILING_COMMAND_START, cl_ulong) \
    F(cl_profiling_info, CL_PROFILING_COMMAND_END, cl_ulong) \
    \
    F(cl_mem_info, CL_MEM_TYPE, cl_mem_object_type) \
    F(cl_mem_info, CL_MEM_FLAGS, cl_mem_flags) \
    F(cl_mem_info, CL_MEM_SIZE, ::size_t) \
    F(cl_mem_info, CL_MEM_HOST_PTR, void*) \
    F(cl_mem_info, CL_MEM_MAP_COUNT, cl_uint) \
    F(cl_mem_info, CL_MEM_REFERENCE_COUNT, cl_uint) \
    F(cl_mem_info, CL_MEM_CONTEXT, cl::Context) \
    \
    F(cl_image_info, CL_IMAGE_FORMAT, cl_image_format) \
    F(cl_image_info, CL_IMAGE_ELEMENT_SIZE, ::size_t) \
    F(cl_image_info, CL_IMAGE_ROW_PITCH, ::size_t) \
    F(cl_image_info, CL_IMAGE_SLICE_PITCH, ::size_t) \
    F(cl_image_info, CL_IMAGE_WIDTH, ::size_t) \
    F(cl_image_info, CL_IMAGE_HEIGHT, ::size_t) \
    F(cl_image_info, CL_IMAGE_DEPTH, ::size_t) \
    \
    F(cl_sampler_info, CL_SAMPLER_REFERENCE_COUNT, cl_uint) \
    F(cl_sampler_info, CL_SAMPLER_CONTEXT, cl::Context) \
    F(cl_sampler_info, CL_SAMPLER_NORMALIZED_COORDS, cl_addressing_mode) \
    F(cl_sampler_info, CL_SAMPLER_ADDRESSING_MODE, cl_filter_mode) \
    F(cl_sampler_info, CL_SAMPLER_FILTER_MODE, cl_bool) \
    \
    F(cl_program_info, CL_PROGRAM_REFERENCE_COUNT, cl_uint) \
    F(cl_program_info, CL_PROGRAM_CONTEXT, cl::Context) \
    F(cl_program_info, CL_PROGRAM_NUM_DEVICES, cl_uint) \
    F(cl_program_info, CL_PROGRAM_DEVICES, VECTOR_CLASS<cl_device_id>) \
    F(cl_program_info, CL_PROGRAM_SOURCE, STRING_CLASS) \
    F(cl_program_info, CL_PROGRAM_BINARY_SIZES, VECTOR_CLASS< ::size_t>) \
    F(cl_program_info, CL_PROGRAM_BINARIES, VECTOR_CLASS<char *>) \
    \
    F(cl_program_build_info, CL_PROGRAM_BUILD_STATUS, cl_build_status) \
    F(cl_program_build_info, CL_PROGRAM_BUILD_OPTIONS, STRING_CLASS) \
    F(cl_program_build_info, CL_PROGRAM_BUILD_LOG, STRING_CLASS) \
    \
    F(cl_kernel_info, CL_KERNEL_FUNCTION_NAME, STRING_CLASS) \
    F(cl_kernel_info, CL_KERNEL_NUM_ARGS, cl_uint) \
    F(cl_kernel_info, CL_KERNEL_REFERENCE_COUNT, cl_uint) \
    F(cl_kernel_info, CL_KERNEL_CONTEXT, cl::Context) \
    F(cl_kernel_info, CL_KERNEL_PROGRAM, cl::Program) \
    \
    F(cl_kernel_work_group_info, CL_KERNEL_WORK_GROUP_SIZE, ::size_t) \
    F(cl_kernel_work_group_info, CL_KERNEL_COMPILE_WORK_GROUP_SIZE, cl::size_t<3>) \
    F(cl_kernel_work_group_info, CL_KERNEL_LOCAL_MEM_SIZE, cl_ulong) \
    \
    F(cl_command_queue_info, CL_QUEUE_CONTEXT, cl::Context) \
    F(cl_command_queue_info, CL_QUEUE_DEVICE, cl::Device) \
    F(cl_command_queue_info, CL_QUEUE_REFERENCE_COUNT, cl_uint) \
    F(cl_command_queue_info, CL_QUEUE_PROPERTIES, cl_command_queue_properties)

#if defined(CL_VERSION_1_1)
#define __PARAM_NAME_INFO_1_1(F) \
    F(cl_context_info, CL_CONTEXT_NUM_DEVICES, cl_uint)\
    F(cl_device_info, CL_DEVICE_PREFERRED_VECTOR_WIDTH_HALF, cl_uint) \
    F(cl_device_info, CL_DEVICE_NATIVE_VECTOR_WIDTH_CHAR, cl_uint) \
    F(cl_device_info, CL_DEVICE_NATIVE_VECTOR_WIDTH_SHORT, cl_uint) \
    F(cl_device_info, CL_DEVICE_NATIVE_VECTOR_WIDTH_INT, cl_uint) \
    F(cl_device_info, CL_DEVICE_NATIVE_VECTOR_WIDTH_LONG, cl_uint) \
    F(cl_device_info, CL_DEVICE_NATIVE_VECTOR_WIDTH_FLOAT, cl_uint) \
    F(cl_device_info, CL_DEVICE_NATIVE_VECTOR_WIDTH_DOUBLE, cl_uint) \
    F(cl_device_info, CL_DEVICE_NATIVE_VECTOR_WIDTH_HALF, cl_uint) \
    F(cl_device_info, CL_DEVICE_DOUBLE_FP_CONFIG, cl_device_fp_config) \
    F(cl_device_info, CL_DEVICE_HALF_FP_CONFIG, cl_device_fp_config) \
    F(cl_device_info, CL_DEVICE_HOST_UNIFIED_MEMORY, cl_bool) \
    F(cl_device_info, CL_DEVICE_OPENCL_C_VERSION, STRING_CLASS) \
    \
    F(cl_mem_info, CL_MEM_ASSOCIATED_MEMOBJECT, cl::Memory) \
    F(cl_mem_info, CL_MEM_OFFSET, ::size_t) \
    \
    F(cl_kernel_work_group_info, CL_KERNEL_PREFERRED_WORK_GROUP_SIZE_MULTIPLE, ::size_t) \
    F(cl_kernel_work_group_info, CL_KERNEL_PRIVATE_MEM_SIZE, cl_ulong) \
    \
    F(cl_event_info, CL_EVENT_CONTEXT, cl::Context)
#endif // CL_VERSION_1_1

#if defined(USE_CL_DEVICE_FISSION)
#define __PARAM_NAME_DEVICE_FISSION(F) \
    F(cl_device_info, CL_DEVICE_PARENT_DEVICE_EXT, cl_device_id) \
    F(cl_device_info, CL_DEVICE_PARTITION_TYPES_EXT, VECTOR_CLASS<cl_device_partition_property_ext>) \
    F(cl_device_info, CL_DEVICE_AFFINITY_DOMAINS_EXT, VECTOR_CLASS<cl_device_partition_property_ext>) \
    F(cl_device_info, CL_DEVICE_REFERENCE_COUNT_EXT , cl_uint) \
    F(cl_device_info, CL_DEVICE_PARTITION_STYLE_EXT, VECTOR_CLASS<cl_device_partition_property_ext>)
#endif // USE_CL_DEVICE_FISSION

template <typename enum_type, cl_int Name>
struct param_traits {};

#define __CL_DECLARE_PARAM_TRAITS(token, param_name, T) \
struct token;                                        \
template<>                                           \
struct param_traits<detail:: token,param_name>       \
{                                                    \
    enum { value = param_name };                     \
    typedef T param_type;                            \
};

__PARAM_NAME_INFO_1_0(__CL_DECLARE_PARAM_TRAITS)
#if defined(CL_VERSION_1_1)
__PARAM_NAME_INFO_1_1(__CL_DECLARE_PARAM_TRAITS)
#endif // CL_VERSION_1_1

#if defined(USE_CL_DEVICE_FISSION)
__PARAM_NAME_DEVICE_FISSION(__CL_DECLARE_PARAM_TRAITS);
#endif // USE_CL_DEVICE_FISSION

#ifdef CL_PLATFORM_ICD_SUFFIX_KHR
__CL_DECLARE_PARAM_TRAITS(cl_platform_info, CL_PLATFORM_ICD_SUFFIX_KHR, STRING_CLASS)
#endif

#ifdef CL_DEVICE_PROFILING_TIMER_OFFSET_AMD
__CL_DECLARE_PARAM_TRAITS(cl_device_info, CL_DEVICE_PROFILING_TIMER_OFFSET_AMD, cl_ulong)
#endif

#ifdef CL_DEVICE_COMPUTE_CAPABILITY_MAJOR_NV
__CL_DECLARE_PARAM_TRAITS(cl_device_info, CL_DEVICE_COMPUTE_CAPABILITY_MAJOR_NV, cl_uint)
#endif
#ifdef CL_DEVICE_COMPUTE_CAPABILITY_MINOR_NV
__CL_DECLARE_PARAM_TRAITS(cl_device_info, CL_DEVICE_COMPUTE_CAPABILITY_MINOR_NV, cl_uint)
#endif
#ifdef CL_DEVICE_REGISTERS_PER_BLOCK_NV
__CL_DECLARE_PARAM_TRAITS(cl_device_info, CL_DEVICE_REGISTERS_PER_BLOCK_NV, cl_uint)
#endif
#ifdef CL_DEVICE_WARP_SIZE_NV
__CL_DECLARE_PARAM_TRAITS(cl_device_info, CL_DEVICE_WARP_SIZE_NV, cl_uint)
#endif
#ifdef CL_DEVICE_GPU_OVERLAP_NV
__CL_DECLARE_PARAM_TRAITS(cl_device_info, CL_DEVICE_GPU_OVERLAP_NV, cl_bool)
#endif
#ifdef CL_DEVICE_KERNEL_EXEC_TIMEOUT_NV
__CL_DECLARE_PARAM_TRAITS(cl_device_info, CL_DEVICE_KERNEL_EXEC_TIMEOUT_NV, cl_bool)
#endif
#ifdef CL_DEVICE_INTEGRATED_MEMORY_NV
__CL_DECLARE_PARAM_TRAITS(cl_device_info, CL_DEVICE_INTEGRATED_MEMORY_NV, cl_bool)
#endif

// Convenience functions

template <typename Func, typename T>
inline cl_int
getInfo(Func f, cl_uint name, T* param)
{
    return GetInfoHelper<Func, T>::get(f, name, param);
}

template <typename Func, typename Arg0>
struct GetInfoFunctor0
{
    Func f_; const Arg0& arg0_;
    cl_int operator ()(
        cl_uint param, ::size_t size, void* value, ::size_t* size_ret)
    { return f_(arg0_, param, size, value, size_ret); }
};

template <typename Func, typename Arg0, typename Arg1>
struct GetInfoFunctor1
{
    Func f_; const Arg0& arg0_; const Arg1& arg1_;
    cl_int operator ()(
        cl_uint param, ::size_t size, void* value, ::size_t* size_ret)
    { return f_(arg0_, arg1_, param, size, value, size_ret); }
};

template <typename Func, typename Arg0, typename T>
inline cl_int
getInfo(Func f, const Arg0& arg0, cl_uint name, T* param)
{
    GetInfoFunctor0<Func, Arg0> f0 = { f, arg0 };
    return GetInfoHelper<GetInfoFunctor0<Func, Arg0>, T>
        ::get(f0, name, param);
}

template <typename Func, typename Arg0, typename Arg1, typename T>
inline cl_int
getInfo(Func f, const Arg0& arg0, const Arg1& arg1, cl_uint name, T* param)
{
    GetInfoFunctor1<Func, Arg0, Arg1> f0 = { f, arg0, arg1 };
    return GetInfoHelper<GetInfoFunctor1<Func, Arg0, Arg1>, T>
        ::get(f0, name, param);
}

template<typename T>
struct ReferenceHandler
{ };

template <>
struct ReferenceHandler<cl_device_id>
{
    // cl_device_id does not have retain().
    static cl_int retain(cl_device_id)
    { return CL_INVALID_DEVICE; }
    // cl_device_id does not have release().
    static cl_int release(cl_device_id)
    { return CL_INVALID_DEVICE; }
};

template <>
struct ReferenceHandler<cl_platform_id>
{
    // cl_platform_id does not have retain().
    static cl_int retain(cl_platform_id)
    { return CL_INVALID_PLATFORM; }
    // cl_platform_id does not have release().
    static cl_int release(cl_platform_id)
    { return CL_INVALID_PLATFORM; }
};

template <>
struct ReferenceHandler<cl_context>
{
    static cl_int retain(cl_context context)
    { return ::clRetainContext(context); }
    static cl_int release(cl_context context)
    { return ::clReleaseContext(context); }
};

template <>
struct ReferenceHandler<cl_command_queue>
{
    static cl_int retain(cl_command_queue queue)
    { return ::clRetainCommandQueue(queue); }
    static cl_int release(cl_command_queue queue)
    { return ::clReleaseCommandQueue(queue); }
};

template <>
struct ReferenceHandler<cl_mem>
{
    static cl_int retain(cl_mem memory)
    { return ::clRetainMemObject(memory); }
    static cl_int release(cl_mem memory)
    { return ::clReleaseMemObject(memory); }
};

template <>
struct ReferenceHandler<cl_sampler>
{
    static cl_int retain(cl_sampler sampler)
    { return ::clRetainSampler(sampler); }
    static cl_int release(cl_sampler sampler)
    { return ::clReleaseSampler(sampler); }
};

template <>
struct ReferenceHandler<cl_program>
{
    static cl_int retain(cl_program program)
    { return ::clRetainProgram(program); }
    static cl_int release(cl_program program)
    { return ::clReleaseProgram(program); }
};

template <>
struct ReferenceHandler<cl_kernel>
{
    static cl_int retain(cl_kernel kernel)
    { return ::clRetainKernel(kernel); }
    static cl_int release(cl_kernel kernel)
    { return ::clReleaseKernel(kernel); }
};

template <>
struct ReferenceHandler<cl_event>
{
    static cl_int retain(cl_event event)
    { return ::clRetainEvent(event); }
    static cl_int release(cl_event event)
    { return ::clReleaseEvent(event); }
};

template <typename T>
class Wrapper
{
public:
    typedef T cl_type;

protected:
    cl_type object_;

public:
    Wrapper() : object_(NULL) { }

    Wrapper(const cl_type &obj) : object_(obj) { }

    ~Wrapper()
    {
        if (object_ != NULL) { release(); }
    }

    Wrapper(const Wrapper<cl_type>& rhs)
    {
        object_ = rhs.object_;
        if (object_ != NULL) { retain(); }
    }

    Wrapper<cl_type>& operator = (const Wrapper<cl_type>& rhs)
    {
        if (object_ != NULL) { release(); }
        object_ = rhs.object_;
        if (object_ != NULL) { retain(); }
        return *this;
    }

    Wrapper<cl_type>& operator = (const cl_type &rhs)
    {
        if (object_ != NULL) { release(); }
        object_ = rhs;
        return *this;
    }

    cl_type operator ()() const { return object_; }

    cl_type& operator ()() { return object_; }

protected:

    cl_int retain() const
    {
        return ReferenceHandler<cl_type>::retain(object_);
    }

    cl_int release() const
    {
        return ReferenceHandler<cl_type>::release(object_);
    }
};

#if defined(__CL_ENABLE_EXCEPTIONS)
static inline cl_int errHandler (
    cl_int err,
    const char * errStr = NULL) throw(Error)
{
    if (err != CL_SUCCESS) {
        throw Error(err, errStr);
    }
    return err;
}
#else
static inline cl_int errHandler (cl_int err, const char * errStr = NULL)
{
    return err;
}
#endif // __CL_ENABLE_EXCEPTIONS

} // namespace detail
//! \endcond

/*! \stuct ImageFormat
 * \brief ImageFormat interface fro cl_image_format.
 */
struct ImageFormat : public cl_image_format
{
    ImageFormat(){}

    ImageFormat(cl_channel_order order, cl_channel_type type)
    {
        image_channel_order = order;
        image_channel_data_type = type;
    }

    ImageFormat& operator = (const ImageFormat& rhs)
    {
        if (this != &rhs) {
            this->image_channel_data_type = rhs.image_channel_data_type;
            this->image_channel_order     = rhs.image_channel_order;
        }
        return *this;
    }
};

/*! \class Device
 * \brief Device interface for cl_device_id.
 */
class Device : public detail::Wrapper<cl_device_id>
{
public:
    Device() : detail::Wrapper<cl_type>() { }

    Device(const Device& device) : detail::Wrapper<cl_type>(device) { }

    Device(const cl_device_id &device) : detail::Wrapper<cl_type>(device) { }

    static Device getDefault(cl_int * err = NULL);

    Device& operator = (const Device& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    Device& operator = (const cl_device_id& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    template <typename T>
    cl_int getInfo(cl_device_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetDeviceInfo, object_, name, param),
            __GET_DEVICE_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_device_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_device_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

#if defined(USE_CL_DEVICE_FISSION)
    cl_int createSubDevices(
        const cl_device_partition_property_ext * properties,
        VECTOR_CLASS<Device>* devices)
    {
        typedef CL_API_ENTRY cl_int 
            ( CL_API_CALL * PFN_clCreateSubDevicesEXT)(
                cl_device_id /*in_device*/,
                const cl_device_partition_property_ext * /* properties */,
                cl_uint /*num_entries*/,
                cl_device_id * /*out_devices*/,
                cl_uint * /*num_devices*/ ) CL_EXT_SUFFIX__VERSION_1_1;

        static PFN_clCreateSubDevicesEXT pfn_clCreateSubDevicesEXT = NULL;
        __INIT_CL_EXT_FCN_PTR(clCreateSubDevicesEXT);

        cl_uint n = 0;
        cl_int err = pfn_clCreateSubDevicesEXT(object_, properties, 0, NULL, &n);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __CREATE_SUB_DEVICES);
        }

        cl_device_id* ids = (cl_device_id*) alloca(n * sizeof(cl_device_id));
        err = pfn_clCreateSubDevicesEXT(object_, properties, n, ids, NULL);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __CREATE_SUB_DEVICES);
        }

        devices->assign(&ids[0], &ids[n]);
        return CL_SUCCESS;
    }
#endif
};

/*! \class Platform
 *  \brief Platform interface.
 */
class Platform : public detail::Wrapper<cl_platform_id>
{
public:
    static const Platform null();

    Platform() : detail::Wrapper<cl_type>()  { }

    Platform(const Platform& platform) : detail::Wrapper<cl_type>(platform) { }

    Platform(const cl_platform_id &platform) : detail::Wrapper<cl_type>(platform) { }

    Platform& operator = (const Platform& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    Platform& operator = (const cl_platform_id& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    cl_int getInfo(cl_platform_info name, STRING_CLASS* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetPlatformInfo, object_, name, param),
            __GET_PLATFORM_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_platform_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_platform_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    cl_int getDevices(
        cl_device_type type,
        VECTOR_CLASS<Device>* devices) const
    {
        cl_uint n = 0;
        cl_int err = ::clGetDeviceIDs(object_, type, 0, NULL, &n);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_DEVICE_IDS_ERR);
        }

        cl_device_id* ids = (cl_device_id*) alloca(n * sizeof(cl_device_id));
        err = ::clGetDeviceIDs(object_, type, n, ids, NULL);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_DEVICE_IDS_ERR);
        }

        devices->assign(&ids[0], &ids[n]);
        return CL_SUCCESS;
    }

#if defined(USE_DX_INTEROP)
   /*! \brief Get the list of available D3D10 devices.
     *
     *  \param d3d_device_source.
     *
     *  \param d3d_object.
     *
     *  \param d3d_device_set.
     *
     *  \param devices returns a vector of OpenCL D3D10 devices found. The cl::Device
     *  values returned in devices can be used to identify a specific OpenCL
     *  device. If \a devices argument is NULL, this argument is ignored.
     *
     *  \return One of the following values:
     *    - CL_SUCCESS if the function is executed successfully.
     *
     *  The application can query specific capabilities of the OpenCL device(s)
     *  returned by cl::getDevices. This can be used by the application to
     *  determine which device(s) to use.
     *
     * \note In the case that exceptions are enabled and a return value
     * other than CL_SUCCESS is generated, then cl::Error exception is
     * generated.
     */
    cl_int getDevices(
        cl_d3d10_device_source_khr d3d_device_source,
        void *                     d3d_object,
        cl_d3d10_device_set_khr    d3d_device_set,
        VECTOR_CLASS<Device>* devices) const
    {
        typedef CL_API_ENTRY cl_int (CL_API_CALL *PFN_clGetDeviceIDsFromD3D10KHR)(
            cl_platform_id platform, 
            cl_d3d10_device_source_khr d3d_device_source, 
            void * d3d_object,
            cl_d3d10_device_set_khr d3d_device_set,
            cl_uint num_entries,
            cl_device_id * devices,
            cl_uint* num_devices);

        static PFN_clGetDeviceIDsFromD3D10KHR pfn_clGetDeviceIDsFromD3D10KHR = NULL;
        __INIT_CL_EXT_FCN_PTR(clGetDeviceIDsFromD3D10KHR);

        cl_uint n = 0;
        cl_int err = pfn_clGetDeviceIDsFromD3D10KHR(
            object_, 
            d3d_device_source, 
            d3d_object,
            d3d_device_set, 
            0, 
            NULL, 
            &n);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_DEVICE_IDS_ERR);
        }

        cl_device_id* ids = (cl_device_id*) alloca(n * sizeof(cl_device_id));
        err = pfn_clGetDeviceIDsFromD3D10KHR(
            object_, 
            d3d_device_source, 
            d3d_object,
            d3d_device_set,
            n, 
            ids, 
            NULL);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_DEVICE_IDS_ERR);
        }

        devices->assign(&ids[0], &ids[n]);
        return CL_SUCCESS;
    }
#endif

    static cl_int get(
        VECTOR_CLASS<Platform>* platforms)
    {
        cl_uint n = 0;
        cl_int err = ::clGetPlatformIDs(0, NULL, &n);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_PLATFORM_IDS_ERR);
        }

        cl_platform_id* ids = (cl_platform_id*) alloca(
            n * sizeof(cl_platform_id));
        err = ::clGetPlatformIDs(n, ids, NULL);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_PLATFORM_IDS_ERR);
        }

        platforms->assign(&ids[0], &ids[n]);
        return CL_SUCCESS;
    }

    static cl_int get(
        Platform * platform)
    {
        cl_uint n = 0;
        cl_int err = ::clGetPlatformIDs(0, NULL, &n);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_PLATFORM_IDS_ERR);
        }

        cl_platform_id* ids = (cl_platform_id*) alloca(
            n * sizeof(cl_platform_id));
        err = ::clGetPlatformIDs(n, ids, NULL);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_PLATFORM_IDS_ERR);
        }

        *platform = ids[0];
        return CL_SUCCESS;
    }

    static Platform get(
        cl_int * errResult = NULL)
    {
        Platform platform;
        cl_uint n = 0;
        cl_int err = ::clGetPlatformIDs(0, NULL, &n);
        if (err != CL_SUCCESS) {
            detail::errHandler(err, __GET_PLATFORM_IDS_ERR);
            if (errResult != NULL) {
                *errResult = err;
            }
        }

        cl_platform_id* ids = (cl_platform_id*) alloca(
            n * sizeof(cl_platform_id));
        err = ::clGetPlatformIDs(n, ids, NULL);

        if (err != CL_SUCCESS) {
            detail::errHandler(err, __GET_PLATFORM_IDS_ERR);
        }

        if (errResult != NULL) {
            *errResult = err;
        }
        
        return ids[0];
    }
};

static inline cl_int
UnloadCompiler()
{
    return ::clUnloadCompiler();
}

class Context : public detail::Wrapper<cl_context>
{
private:
    static volatile int default_initialized_;
    static Context default_;
    static volatile cl_int default_error_;
public:
    Context(
        const VECTOR_CLASS<Device>& devices,
        cl_context_properties* properties = NULL,
        void (CL_CALLBACK * notifyFptr)(
            const char *,
            const void *,
            ::size_t,
            void *) = NULL,
        void* data = NULL,
        cl_int* err = NULL)
    {
        cl_int error;
        object_ = ::clCreateContext(
            properties, (cl_uint) devices.size(),
            (cl_device_id*) &devices.front(),
            notifyFptr, data, &error);

        detail::errHandler(error, __CREATE_CONTEXT_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Context(
        cl_device_type type,
        cl_context_properties* properties = NULL,
        void (CL_CALLBACK * notifyFptr)(
            const char *,
            const void *,
            ::size_t,
            void *) = NULL,
        void* data = NULL,
        cl_int* err = NULL)
    {
        cl_int error;

#if !defined(__APPLE__) || !defined(__MACOS)
        cl_context_properties prop[4] = {CL_CONTEXT_PLATFORM, 0, 0, 0 };	
        if (properties == NULL) {
            prop[1] = (cl_context_properties)Platform::get(&error)();
            if (error != CL_SUCCESS) {
                detail::errHandler(error, __CREATE_CONTEXT_FROM_TYPE_ERR);
                if (err != NULL) {
                    *err = error;
                    return;
                }
            }

            properties = &prop[0];
        }
#endif
        object_ = ::clCreateContextFromType(
            properties, type, notifyFptr, data, &error);

        detail::errHandler(error, __CREATE_CONTEXT_FROM_TYPE_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    static Context getDefault(cl_int * err = NULL) 
    {
        int state = detail::compare_exchange(
            &default_initialized_, 
            __DEFAULT_BEING_INITIALIZED, __DEFAULT_NOT_INITIALIZED);
        
        if (state & __DEFAULT_INITIALIZED) {
            if (err != NULL) {
                *err = default_error_;
            }
            return default_;
        }

        if (state & __DEFAULT_BEING_INITIALIZED) {
              // Assume writes will propagate eventually...
              while(default_initialized_ != __DEFAULT_INITIALIZED) {
                                ;
              }

            if (err != NULL) {
                *err = default_error_;
            }
            return default_;
        }

        cl_int error;
        default_ = Context(
            CL_DEVICE_TYPE_DEFAULT,
            NULL,
            NULL,
            NULL,
            &error);

        default_error_ = error;
        // Assume writes will propagate eventually...
        default_initialized_ = __DEFAULT_INITIALIZED;
        if (err != NULL) {
            *err = default_error_;
        }
        return default_;

    }

    Context() : detail::Wrapper<cl_type>() { }

    Context(const Context& context) : detail::Wrapper<cl_type>(context) { }

    Context(const cl_context& context) : detail::Wrapper<cl_type>(context) { }

    Context& operator = (const Context& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    Context& operator = (const cl_context& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    template <typename T>
    cl_int getInfo(cl_context_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetContextInfo, object_, name, param),
            __GET_CONTEXT_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_context_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_context_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    cl_int getSupportedImageFormats(
        cl_mem_flags flags,
        cl_mem_object_type type,
        VECTOR_CLASS<ImageFormat>* formats) const
    {
        cl_uint numEntries;
        cl_int err = ::clGetSupportedImageFormats(
           object_, 
           flags,
           type, 
           0, 
           NULL, 
           &numEntries);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_SUPPORTED_IMAGE_FORMATS_ERR);
        }

        ImageFormat* value = (ImageFormat*)
            alloca(numEntries * sizeof(ImageFormat));
        err = ::clGetSupportedImageFormats(
            object_, 
            flags, 
            type, 
            numEntries,
            (cl_image_format*) value, 
            NULL);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __GET_SUPPORTED_IMAGE_FORMATS_ERR);
        }

        formats->assign(&value[0], &value[numEntries]);
        return CL_SUCCESS;
    }
};

inline Device Device::getDefault(cl_int * err)
{
    cl_int error;
    Device device;

    Context context = Context::getDefault(&error);
    detail::errHandler(error, __CREATE_COMMAND_QUEUE_ERR);

    if (error != CL_SUCCESS) {
        if (err != NULL) {
            *err = error;
        }
    }
    else {
        device = context.getInfo<CL_CONTEXT_DEVICES>()[0];
        if (err != NULL) {
            *err = CL_SUCCESS;
        }
    }

    return device;
}

#ifdef _WIN32
__declspec(selectany) volatile int Context::default_initialized_ = __DEFAULT_NOT_INITIALIZED;
__declspec(selectany) Context Context::default_;
__declspec(selectany) volatile cl_int Context::default_error_ = CL_SUCCESS;
#else
__attribute__((weak)) volatile int Context::default_initialized_ = __DEFAULT_NOT_INITIALIZED;
__attribute__((weak)) Context Context::default_;
__attribute__((weak)) volatile cl_int Context::default_error_ = CL_SUCCESS;
#endif

__GET_INFO_HELPER_WITH_RETAIN(cl::Context)

/*! \class Event
 * \brief Event interface for cl_event.
 */
class Event : public detail::Wrapper<cl_event>
{
public:
    Event() : detail::Wrapper<cl_type>() { }

    Event(const Event& event) : detail::Wrapper<cl_type>(event) { }

    Event(const cl_event& event) : detail::Wrapper<cl_type>(event) { }

    Event& operator = (const Event& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    Event& operator = (const cl_event& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    template <typename T>
    cl_int getInfo(cl_event_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetEventInfo, object_, name, param),
            __GET_EVENT_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_event_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_event_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    template <typename T>
    cl_int getProfilingInfo(cl_profiling_info name, T* param) const
    {
        return detail::errHandler(detail::getInfo(
            &::clGetEventProfilingInfo, object_, name, param),
            __GET_EVENT_PROFILE_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_profiling_info, name>::param_type
    getProfilingInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_profiling_info, name>::param_type param;
        cl_int result = getProfilingInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    cl_int wait() const
    {
        return detail::errHandler(
            ::clWaitForEvents(1, &object_),
            __WAIT_FOR_EVENTS_ERR);
    }

#if defined(CL_VERSION_1_1)
    cl_int setCallback(
        cl_int type,
        void (CL_CALLBACK * pfn_notify)(cl_event, cl_int, void *),		
        void * user_data = NULL)
    {
        return detail::errHandler(
            ::clSetEventCallback(
                object_,
                type,
                pfn_notify,
                user_data), 
            __SET_EVENT_CALLBACK_ERR);
    }
#endif

    static cl_int
    waitForEvents(const VECTOR_CLASS<Event>& events)
    {
        return detail::errHandler(
            ::clWaitForEvents(
                (cl_uint) events.size(), (cl_event*)&events.front()),
            __WAIT_FOR_EVENTS_ERR);
    }
};

__GET_INFO_HELPER_WITH_RETAIN(cl::Event)

#if defined(CL_VERSION_1_1)
/*! \class UserEvent
 * \brief User event interface for cl_event.
 */
class UserEvent : public Event
{
public:
    UserEvent(
        const Context& context,
        cl_int * err = NULL)
    {
        cl_int error;
        object_ = ::clCreateUserEvent(
            context(),
            &error);

        detail::errHandler(error, __CREATE_USER_EVENT_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    UserEvent() : Event() { }

    UserEvent(const UserEvent& event) : Event(event) { }

    UserEvent& operator = (const UserEvent& rhs)
    {
        if (this != &rhs) {
            Event::operator=(rhs);
        }
        return *this;
    }

    cl_int setStatus(cl_int status)
    {
        return detail::errHandler(
            ::clSetUserEventStatus(object_,status), 
            __SET_USER_EVENT_STATUS_ERR);
    }
};
#endif

inline static cl_int
WaitForEvents(const VECTOR_CLASS<Event>& events)
{
    return detail::errHandler(
        ::clWaitForEvents(
            (cl_uint) events.size(), (cl_event*)&events.front()),
        __WAIT_FOR_EVENTS_ERR);
}

/*! \class Memory
 * \brief Memory interface for cl_mem.
 */
class Memory : public detail::Wrapper<cl_mem>
{
public:
    Memory() : detail::Wrapper<cl_type>() { }

    Memory(const Memory& memory) : detail::Wrapper<cl_type>(memory) { }

    Memory(const cl_mem& memory) : detail::Wrapper<cl_type>(memory) { }

    Memory& operator = (const Memory& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    Memory& operator = (const cl_mem& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    template <typename T>
    cl_int getInfo(cl_mem_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetMemObjectInfo, object_, name, param),
            __GET_MEM_OBJECT_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_mem_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_mem_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

#if defined(CL_VERSION_1_1)
    cl_int setDestructorCallback(
        void (CL_CALLBACK * pfn_notify)(cl_mem, void *),		
        void * user_data = NULL)
    {
        return detail::errHandler(
            ::clSetMemObjectDestructorCallback(
                object_,
                pfn_notify,
                user_data), 
            __SET_MEM_OBJECT_DESTRUCTOR_CALLBACK_ERR);
    }
#endif

};

__GET_INFO_HELPER_WITH_RETAIN(cl::Memory)

/*! \class Buffer
 * \brief Memory buffer interface.
 */
class Buffer : public Memory
{
public:
    Buffer(
        const Context& context,
        cl_mem_flags flags,
        ::size_t size,
        void* host_ptr = NULL,
        cl_int* err = NULL)
    {
        cl_int error;
        object_ = ::clCreateBuffer(context(), flags, size, host_ptr, &error);

        detail::errHandler(error, __CREATE_BUFFER_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Buffer(
         cl_mem_flags flags,
        ::size_t size,
        void* host_ptr = NULL,
        cl_int* err = NULL)
    {
        cl_int error;

        Context context = Context::getDefault(err);

        object_ = ::clCreateBuffer(context(), flags, size, host_ptr, &error);

        detail::errHandler(error, __CREATE_BUFFER_ERR);
        if (err != NULL) {
            *err = error;
        }
    }


    Buffer() : Memory() { }

    Buffer(const Buffer& buffer) : Memory(buffer) { }

    Buffer(const cl_mem& buffer) : Memory(buffer) { }

    Buffer& operator = (const Buffer& rhs)
    {
        if (this != &rhs) {
            Memory::operator=(rhs);
        }
        return *this;
    }

    Buffer& operator = (const cl_mem& rhs)
    {
        Memory::operator=(rhs);
        return *this;
    }

#if defined(CL_VERSION_1_1)
    Buffer createSubBuffer(
        cl_mem_flags flags,
        cl_buffer_create_type buffer_create_type,
        const void * buffer_create_info,
        cl_int * err = NULL)
    {
        Buffer result;
        cl_int error;
        result.object_ = ::clCreateSubBuffer(
            object_, 
            flags, 
            buffer_create_type, 
            buffer_create_info, 
            &error);

        detail::errHandler(error, __CREATE_SUBBUFFER_ERR);
        if (err != NULL) {
            *err = error;
        }

        return result;
    }		
#endif
};

#if defined (USE_DX_INTEROP)
class BufferD3D10 : public Buffer
{
public:
    typedef CL_API_ENTRY cl_mem (CL_API_CALL *PFN_clCreateFromD3D10BufferKHR)(
    cl_context context, cl_mem_flags flags, ID3D10Buffer*  buffer,
    cl_int* errcode_ret);

    BufferD3D10(
        const Context& context,
        cl_mem_flags flags,
        ID3D10Buffer* bufobj,
        cl_int * err = NULL)
    {
        static PFN_clCreateFromD3D10BufferKHR pfn_clCreateFromD3D10BufferKHR = NULL;
        __INIT_CL_EXT_FCN_PTR(clCreateFromD3D10BufferKHR);

        cl_int error;
        object_ = pfn_clCreateFromD3D10BufferKHR(
            context(),
            flags,
            bufobj,
            &error);

        detail::errHandler(error, __CREATE_GL_BUFFER_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    BufferD3D10() : Buffer() { }

    BufferD3D10(const BufferD3D10& buffer) : Buffer(buffer) { }

    BufferD3D10(const cl_mem& buffer) : Buffer(buffer) { }

    BufferD3D10& operator = (const BufferD3D10& rhs)
    {
        if (this != &rhs) {
            Buffer::operator=(rhs);
        }
        return *this;
    }

    BufferD3D10& operator = (const cl_mem& rhs)
    {
        Buffer::operator=(rhs);
        return *this;
    }
};
#endif

/*! \class BufferGL
 * \brief Memory buffer interface for GL interop.
 */
class BufferGL : public Buffer
{
public:
    BufferGL(
        const Context& context,
        cl_mem_flags flags,
        GLuint bufobj,
        cl_int * err = NULL)
    {
        cl_int error;
        object_ = ::clCreateFromGLBuffer(
            context(),
            flags,
            bufobj,
            &error);

        detail::errHandler(error, __CREATE_GL_BUFFER_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    BufferGL() : Buffer() { }

    BufferGL(const BufferGL& buffer) : Buffer(buffer) { }

    BufferGL(const cl_mem& buffer) : Buffer(buffer) { }

    BufferGL& operator = (const BufferGL& rhs)
    {
        if (this != &rhs) {
            Buffer::operator=(rhs);
        }
        return *this;
    }

    BufferGL& operator = (const cl_mem& rhs)
    {
        Buffer::operator=(rhs);
        return *this;
    }

    cl_int getObjectInfo(
        cl_gl_object_type *type,
        GLuint * gl_object_name)
    {
        return detail::errHandler(
            ::clGetGLObjectInfo(object_,type,gl_object_name),
            __GET_GL_OBJECT_INFO_ERR);
    }
};

/*! \class BufferRenderGL
 * \brief Memory buffer interface for GL interop with renderbuffer.
 */
class BufferRenderGL : public Buffer
{
public:
    BufferRenderGL(
        const Context& context,
        cl_mem_flags flags,
        GLuint bufobj,
        cl_int * err = NULL)
    {
        cl_int error;
        object_ = ::clCreateFromGLRenderbuffer(
            context(),
            flags,
            bufobj,
            &error);

        detail::errHandler(error, __CREATE_GL_RENDER_BUFFER_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    BufferRenderGL() : Buffer() { }

    BufferRenderGL(const BufferGL& buffer) : Buffer(buffer) { }

    BufferRenderGL(const cl_mem& buffer) : Buffer(buffer) { }

    BufferRenderGL& operator = (const BufferRenderGL& rhs)
    {
        if (this != &rhs) {
            Buffer::operator=(rhs);
        }
        return *this;
    }

    BufferRenderGL& operator = (const cl_mem& rhs)
    {
        Buffer::operator=(rhs);
        return *this;
    }

    cl_int getObjectInfo(
        cl_gl_object_type *type,
        GLuint * gl_object_name)
    {
        return detail::errHandler(
            ::clGetGLObjectInfo(object_,type,gl_object_name),
            __GET_GL_OBJECT_INFO_ERR);
    }
};

/*! \class Image
 * \brief Base class  interface for all images.
 */
class Image : public Memory
{
protected:
    Image() : Memory() { }

    Image(const Image& image) : Memory(image) { }

    Image(const cl_mem& image) : Memory(image) { }

    Image& operator = (const Image& rhs)
    {
        if (this != &rhs) {
            Memory::operator=(rhs);
        }
        return *this;
    }

    Image& operator = (const cl_mem& rhs)
    {
        Memory::operator=(rhs);
        return *this;
    }

public:
    template <typename T>
    cl_int getImageInfo(cl_image_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetImageInfo, object_, name, param),
            __GET_IMAGE_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_image_info, name>::param_type
    getImageInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_image_info, name>::param_type param;
        cl_int result = getImageInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }
};

/*! \class Image2D
 * \brief Image interface for 2D images.
 */
class Image2D : public Image
{
public:
    Image2D(
        const Context& context,
        cl_mem_flags flags,
        ImageFormat format,
        ::size_t width,
        ::size_t height,
        ::size_t row_pitch = 0,
        void* host_ptr = NULL,
        cl_int* err = NULL)
    {
        cl_int error;
        object_ = ::clCreateImage2D(
            context(), flags,&format, width, height, row_pitch, host_ptr, &error);

        detail::errHandler(error, __CREATE_IMAGE2D_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Image2D() { }

    Image2D(const Image2D& image2D) : Image(image2D) { }

    Image2D(const cl_mem& image2D) : Image(image2D) { }

    Image2D& operator = (const Image2D& rhs)
    {
        if (this != &rhs) {
            Image::operator=(rhs);
        }
        return *this;
    }

    Image2D& operator = (const cl_mem& rhs)
    {
        Image::operator=(rhs);
        return *this;
    }
};

/*! \class Image2DGL
 * \brief 2D image interface for GL interop.
 */
class Image2DGL : public Image2D
{
public:
    Image2DGL(
        const Context& context,
        cl_mem_flags flags,
        GLenum target,
        GLint  miplevel,
        GLuint texobj,
        cl_int * err = NULL)
    {
        cl_int error;
        object_ = ::clCreateFromGLTexture2D(
            context(),
            flags,
            target,
            miplevel,
            texobj,
            &error);

        detail::errHandler(error, __CREATE_GL_TEXTURE_2D_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Image2DGL() : Image2D() { }

    Image2DGL(const Image2DGL& image) : Image2D(image) { }

    Image2DGL(const cl_mem& image) : Image2D(image) { }

    Image2DGL& operator = (const Image2DGL& rhs)
    {
        if (this != &rhs) {
            Image2D::operator=(rhs);
        }
        return *this;
    }

    Image2DGL& operator = (const cl_mem& rhs)
    {
        Image2D::operator=(rhs);
        return *this;
    }
};

/*! \class Image3D
 * \brief Image interface for 3D images.
 */
class Image3D : public Image
{
public:
    Image3D(
        const Context& context,
        cl_mem_flags flags,
        ImageFormat format,
        ::size_t width,
        ::size_t height,
        ::size_t depth,
        ::size_t row_pitch = 0,
        ::size_t slice_pitch = 0,
        void* host_ptr = NULL,
        cl_int* err = NULL)
    {
        cl_int error;
        object_ = ::clCreateImage3D(
            context(), flags, &format, width, height, depth, row_pitch,
            slice_pitch, host_ptr, &error);

        detail::errHandler(error, __CREATE_IMAGE3D_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Image3D() { }

    Image3D(const Image3D& image3D) : Image(image3D) { }

    Image3D(const cl_mem& image3D) : Image(image3D) { }

    Image3D& operator = (const Image3D& rhs)
    {
        if (this != &rhs) {
            Image::operator=(rhs);
        }
        return *this;
    }

    Image3D& operator = (const cl_mem& rhs)
    {
        Image::operator=(rhs);
        return *this;
    }
};

/*! \class Image2DGL
 * \brief 2D image interface for GL interop.
 */
class Image3DGL : public Image3D
{
public:
    Image3DGL(
        const Context& context,
        cl_mem_flags flags,
        GLenum target,
        GLint  miplevel,
        GLuint texobj,
        cl_int * err = NULL)
    {
        cl_int error;
        object_ = ::clCreateFromGLTexture3D(
            context(),
            flags,
            target,
            miplevel,
            texobj,
            &error);

        detail::errHandler(error, __CREATE_GL_TEXTURE_3D_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Image3DGL() : Image3D() { }

    Image3DGL(const Image3DGL& image) : Image3D(image) { }

    Image3DGL(const cl_mem& image) : Image3D(image) { }

    Image3DGL& operator = (const Image3DGL& rhs)
    {
        if (this != &rhs) {
            Image3D::operator=(rhs);
        }
        return *this;
    }

    Image3DGL& operator = (const cl_mem& rhs)
    {
        Image3D::operator=(rhs);
        return *this;
    }
};

/*! \class Sampler
 * \brief Sampler interface for cl_sampler.
 */
class Sampler : public detail::Wrapper<cl_sampler>
{
public:
    Sampler() { }

    Sampler(
        const Context& context,
        cl_bool normalized_coords,
        cl_addressing_mode addressing_mode,
        cl_filter_mode filter_mode,
        cl_int* err = NULL)
    {
        cl_int error;
        object_ = ::clCreateSampler(
            context(), 
            normalized_coords,
            addressing_mode,
            filter_mode,
            &error);

        detail::errHandler(error, __CREATE_SAMPLER_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Sampler(const Sampler& sampler) : detail::Wrapper<cl_type>(sampler) { }

    Sampler(const cl_sampler& sampler) : detail::Wrapper<cl_type>(sampler) { }

    Sampler& operator = (const Sampler& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    Sampler& operator = (const cl_sampler& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    template <typename T>
    cl_int getInfo(cl_sampler_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetSamplerInfo, object_, name, param),
            __GET_SAMPLER_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_sampler_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_sampler_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }
};

__GET_INFO_HELPER_WITH_RETAIN(cl::Sampler)

class Program;
class CommandQueue;
class Kernel;

/*! \class NDRange
 * \brief NDRange interface
 */
class NDRange
{
private:
    size_t<3> sizes_;
    cl_uint dimensions_;

public:
    NDRange()
        : dimensions_(0)
    { }

    NDRange(::size_t size0)
        : dimensions_(1)
    {
        sizes_.push_back(size0);
    }

    NDRange(::size_t size0, ::size_t size1)
        : dimensions_(2)
    {
        sizes_.push_back(size0);
        sizes_.push_back(size1);
    }

    NDRange(::size_t size0, ::size_t size1, ::size_t size2)
        : dimensions_(3)
    {
        sizes_.push_back(size0);
        sizes_.push_back(size1);
        sizes_.push_back(size2);
    }

    operator const ::size_t*() const { return (const ::size_t*) sizes_; }
    ::size_t dimensions() const { return dimensions_; }
};

static const NDRange NullRange;

/*!
 * \struct LocalSpaceArg
 * \brief Local address raper for use with Kernel::setArg
 */
struct LocalSpaceArg
{
    ::size_t size_;
};

namespace detail {

template <typename T>
struct KernelArgumentHandler
{
    static ::size_t size(const T&) { return sizeof(T); }
    static T* ptr(T& value) { return &value; }
};

template <>
struct KernelArgumentHandler<LocalSpaceArg>
{
    static ::size_t size(const LocalSpaceArg& value) { return value.size_; }
    static void* ptr(LocalSpaceArg&) { return NULL; }
};

} 
//! \endcond

inline LocalSpaceArg
__local(::size_t size)
{
    LocalSpaceArg ret = { size };
    return ret;
}

//class KernelFunctor;

/*! \class Kernel
 * \brief Kernel interface that implements cl_kernel
 */
class Kernel : public detail::Wrapper<cl_kernel>
{
public:
    inline Kernel(const Program& program, const char* name, cl_int* err = NULL);

    Kernel() { }

    Kernel(const Kernel& kernel) : detail::Wrapper<cl_type>(kernel) { }

    Kernel(const cl_kernel& kernel) : detail::Wrapper<cl_type>(kernel) { }

    Kernel& operator = (const Kernel& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    Kernel& operator = (const cl_kernel& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    template <typename T>
    cl_int getInfo(cl_kernel_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetKernelInfo, object_, name, param),
            __GET_KERNEL_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_kernel_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_kernel_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    template <typename T>
    cl_int getWorkGroupInfo(
        const Device& device, cl_kernel_work_group_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(
                &::clGetKernelWorkGroupInfo, object_, device(), name, param),
                __GET_KERNEL_WORK_GROUP_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_kernel_work_group_info, name>::param_type
        getWorkGroupInfo(const Device& device, cl_int* err = NULL) const
    {
        typename detail::param_traits<
        detail::cl_kernel_work_group_info, name>::param_type param;
        cl_int result = getWorkGroupInfo(device, name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    template <typename T>
    cl_int setArg(cl_uint index, T value)
    {
        return detail::errHandler(
            ::clSetKernelArg(
                object_,
                index,
                detail::KernelArgumentHandler<T>::size(value),
                detail::KernelArgumentHandler<T>::ptr(value)),
            __SET_KERNEL_ARGS_ERR);
    }

    cl_int setArg(cl_uint index, ::size_t size, void* argPtr)
    {
        return detail::errHandler(
            ::clSetKernelArg(object_, index, size, argPtr),
            __SET_KERNEL_ARGS_ERR);
    }
};

__GET_INFO_HELPER_WITH_RETAIN(cl::Kernel)

/*! \class Program
 * \brief Program interface that implements cl_program.
 */
class Program : public detail::Wrapper<cl_program>
{
public:
    typedef VECTOR_CLASS<std::pair<const void*, ::size_t> > Binaries;
    typedef VECTOR_CLASS<std::pair<const char*, ::size_t> > Sources;

    Program(
        const STRING_CLASS& source,
        cl_int* err = NULL)
    {
        cl_int error;

        const char * strings = source.c_str();
        const ::size_t length  = source.size();

        Context context = Context::getDefault(err);

        object_ = ::clCreateProgramWithSource(
            context(), (cl_uint)1, &strings, &length, &error);

        detail::errHandler(error, __CREATE_PROGRAM_WITH_SOURCE_ERR);

        if (error == CL_SUCCESS) {

            error = ::clBuildProgram(
                object_,
                0,
                NULL,
                "",
                NULL,
                NULL);

            detail::errHandler(error, __BUILD_PROGRAM_ERR);
        }

        if (err != NULL) {
            *err = error;
        }
    }

    Program(
        const STRING_CLASS& source,
		bool build,
        cl_int* err = NULL)
    {
        cl_int error;

        const char * strings = source.c_str();
        const ::size_t length  = source.size();

        Context context = Context::getDefault(err);

        object_ = ::clCreateProgramWithSource(
            context(), (cl_uint)1, &strings, &length, &error);

        detail::errHandler(error, __CREATE_PROGRAM_WITH_SOURCE_ERR);

        if (error == CL_SUCCESS && build) {

            error = ::clBuildProgram(
                object_,
                0,
                NULL,
                "",
                NULL,
                NULL);

            detail::errHandler(error, __BUILD_PROGRAM_ERR);
        }

        if (err != NULL) {
            *err = error;
        }
    }

    Program(
        const Context& context,
        const STRING_CLASS& source,
        bool build = false,
        cl_int* err = NULL)
    {
        cl_int error;

        const char * strings = source.c_str();
        const ::size_t length  = source.size();

        object_ = ::clCreateProgramWithSource(
            context(), (cl_uint)1, &strings, &length, &error);

        detail::errHandler(error, __CREATE_PROGRAM_WITH_SOURCE_ERR);

        if (error == CL_SUCCESS && build) {

            error = ::clBuildProgram(
                object_,
                0,
                NULL,
                "",
                NULL,
                NULL);

            detail::errHandler(error, __BUILD_PROGRAM_ERR);
        }

        if (err != NULL) {
            *err = error;
        }
    }

    Program(
        const Context& context,
        const Sources& sources,
        cl_int* err = NULL)
    {
        cl_int error;

        const ::size_t n = (::size_t)sources.size();
        ::size_t* lengths = (::size_t*) alloca(n * sizeof(::size_t));
        const char** strings = (const char**) alloca(n * sizeof(const char*));

        for (::size_t i = 0; i < n; ++i) {
            strings[i] = sources[(int)i].first;
            lengths[i] = sources[(int)i].second;
        }

        object_ = ::clCreateProgramWithSource(
            context(), (cl_uint)n, strings, lengths, &error);

        detail::errHandler(error, __CREATE_PROGRAM_WITH_SOURCE_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Program(
        const Context& context,
        const VECTOR_CLASS<Device>& devices,
        const Binaries& binaries,
        VECTOR_CLASS<cl_int>* binaryStatus = NULL,
        cl_int* err = NULL)
    {
        cl_int error;
        const ::size_t n = binaries.size();
        ::size_t* lengths = (::size_t*) alloca(n * sizeof(::size_t));
        const unsigned char** images = (const unsigned char**) alloca(n * sizeof(const void*));

        for (::size_t i = 0; i < n; ++i) {
            images[i] = (const unsigned char*)binaries[(int)i].first;
            lengths[i] = binaries[(int)i].second;
        }

        object_ = ::clCreateProgramWithBinary(
            context(), (cl_uint) devices.size(),
            (cl_device_id*)&devices.front(),
            lengths, images, binaryStatus != NULL
               ? (cl_int*) &binaryStatus->front()
               : NULL, &error);

        detail::errHandler(error, __CREATE_PROGRAM_WITH_BINARY_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    Program() { }

    Program(const Program& program) : detail::Wrapper<cl_type>(program) { }

    Program(const cl_program& program) : detail::Wrapper<cl_type>(program) { }

    Program& operator = (const Program& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    Program& operator = (const cl_program& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    cl_int build(
        const VECTOR_CLASS<Device>& devices,
        const char* options = NULL,
        void (CL_CALLBACK * notifyFptr)(cl_program, void *) = NULL,
        void* data = NULL) const
    {
        return detail::errHandler(
            ::clBuildProgram(
                object_,
                (cl_uint)
                devices.size(),
                (cl_device_id*)&devices.front(),
                options,
                notifyFptr,
                data),
                __BUILD_PROGRAM_ERR);
    }

    cl_int build(
        const char* options = NULL,
        void (CL_CALLBACK * notifyFptr)(cl_program, void *) = NULL,
        void* data = NULL) const
    {
        return detail::errHandler(
            ::clBuildProgram(
                object_,
                0,
                NULL,
                options,
                notifyFptr,
                data),
                __BUILD_PROGRAM_ERR);
    }

    template <typename T>
    cl_int getInfo(cl_program_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(&::clGetProgramInfo, object_, name, param),
            __GET_PROGRAM_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_program_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_program_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    template <typename T>
    cl_int getBuildInfo(
        const Device& device, cl_program_build_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(
                &::clGetProgramBuildInfo, object_, device(), name, param),
                __GET_PROGRAM_BUILD_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_program_build_info, name>::param_type
    getBuildInfo(const Device& device, cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_program_build_info, name>::param_type param;
        cl_int result = getBuildInfo(device, name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    cl_int createKernels(VECTOR_CLASS<Kernel>* kernels)
    {
        cl_uint numKernels;
        cl_int err = ::clCreateKernelsInProgram(object_, 0, NULL, &numKernels);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __CREATE_KERNELS_IN_PROGRAM_ERR);
        }

        Kernel* value = (Kernel*) alloca(numKernels * sizeof(Kernel));
        err = ::clCreateKernelsInProgram(
            object_, numKernels, (cl_kernel*) value, NULL);
        if (err != CL_SUCCESS) {
            return detail::errHandler(err, __CREATE_KERNELS_IN_PROGRAM_ERR);
        }

        kernels->assign(&value[0], &value[numKernels]);
        return CL_SUCCESS;
    }
};

template<>
inline VECTOR_CLASS<char *> cl::Program::getInfo<CL_PROGRAM_BINARIES>(cl_int* err) const
{
    VECTOR_CLASS< ::size_t> sizes = getInfo<CL_PROGRAM_BINARY_SIZES>();
    VECTOR_CLASS<char *> binaries;
    for (VECTOR_CLASS< ::size_t>::iterator s = sizes.begin(); s != sizes.end(); ++s) 
    {
        char *ptr = NULL;
        if (*s != 0) 
            ptr = new char[*s];
        binaries.push_back(ptr);
    }
    
    cl_int result = getInfo(CL_PROGRAM_BINARIES, &binaries);
    if (err != NULL) {
        *err = result;
    }
    return binaries;
}

__GET_INFO_HELPER_WITH_RETAIN(cl::Program)

inline Kernel::Kernel(const Program& program, const char* name, cl_int* err)
{
    cl_int error;

    object_ = ::clCreateKernel(program(), name, &error);
    detail::errHandler(error, __CREATE_KERNEL_ERR);

    if (err != NULL) {
        *err = error;
    }

}

/*! \class CommandQueue
 * \brief CommandQueue interface for cl_command_queue.
 */
class CommandQueue : public detail::Wrapper<cl_command_queue>
{
private:
    static volatile int default_initialized_;
    static CommandQueue default_;
    static volatile cl_int default_error_;
public:
   CommandQueue(
        cl_command_queue_properties properties,
        cl_int* err = NULL)
    {
        cl_int error;

        Context context = Context::getDefault(&error);
        detail::errHandler(error, __CREATE_COMMAND_QUEUE_ERR);

        if (error != CL_SUCCESS) {
            if (err != NULL) {
                *err = error;
            }
        }
        else {
            Device device = context.getInfo<CL_CONTEXT_DEVICES>()[0];

            object_ = ::clCreateCommandQueue(
                context(), device(), properties, &error);

            detail::errHandler(error, __CREATE_COMMAND_QUEUE_ERR);
            if (err != NULL) {
                *err = error;
            }
        }
    }

    CommandQueue(
        const Context& context,
        const Device& device,
        cl_command_queue_properties properties = 0,
        cl_int* err = NULL)
    {
        cl_int error;
        object_ = ::clCreateCommandQueue(
            context(), device(), properties, &error);

        detail::errHandler(error, __CREATE_COMMAND_QUEUE_ERR);
        if (err != NULL) {
            *err = error;
        }
    }

    static CommandQueue getDefault(cl_int * err = NULL) 
    {
        int state = detail::compare_exchange(
            &default_initialized_, 
            __DEFAULT_BEING_INITIALIZED, __DEFAULT_NOT_INITIALIZED);

        if (state & __DEFAULT_INITIALIZED) {
            if (err != NULL) {
                *err = default_error_;
            }
            return default_;
        }

        if (state & __DEFAULT_BEING_INITIALIZED) {
              // Assume writes will propagate eventually...
              while(default_initialized_ != __DEFAULT_INITIALIZED) {
                                ;
              }

            if (err != NULL) {
                *err = default_error_;
            }
            return default_;
        }

        cl_int error;

        Context context = Context::getDefault(&error);
        detail::errHandler(error, __CREATE_COMMAND_QUEUE_ERR);

        if (error != CL_SUCCESS) {
            if (err != NULL) {
                *err = error;
            }
        }
        else {
            Device device = context.getInfo<CL_CONTEXT_DEVICES>()[0];


            default_ = CommandQueue(context, device, 0, &error);

            detail::errHandler(error, __CREATE_COMMAND_QUEUE_ERR);
            if (err != NULL) {
                *err = error;
            }
        }

        default_error_ = error;
        // Assume writes will propagate eventually...
        default_initialized_ = __DEFAULT_INITIALIZED;
        if (err != NULL) {
            *err = default_error_;
        }
        return default_;

    }

    CommandQueue() { }

    CommandQueue(const CommandQueue& commandQueue) : detail::Wrapper<cl_type>(commandQueue) { }

    CommandQueue(const cl_command_queue& commandQueue) : detail::Wrapper<cl_type>(commandQueue) { }

    CommandQueue& operator = (const CommandQueue& rhs)
    {
        if (this != &rhs) {
            detail::Wrapper<cl_type>::operator=(rhs);
        }
        return *this;
    }

    CommandQueue& operator = (const cl_command_queue& rhs)
    {
        detail::Wrapper<cl_type>::operator=(rhs);
        return *this;
    }

    template <typename T>
    cl_int getInfo(cl_command_queue_info name, T* param) const
    {
        return detail::errHandler(
            detail::getInfo(
                &::clGetCommandQueueInfo, object_, name, param),
                __GET_COMMAND_QUEUE_INFO_ERR);
    }

    template <cl_int name> typename
    detail::param_traits<detail::cl_command_queue_info, name>::param_type
    getInfo(cl_int* err = NULL) const
    {
        typename detail::param_traits<
            detail::cl_command_queue_info, name>::param_type param;
        cl_int result = getInfo(name, &param);
        if (err != NULL) {
            *err = result;
        }
        return param;
    }

    cl_int enqueueReadBuffer(
        const Buffer& buffer,
        cl_bool blocking,
        ::size_t offset,
        ::size_t size,
        void* ptr,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueReadBuffer(
                object_, buffer(), blocking, offset, size,
                ptr,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_READ_BUFFER_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueWriteBuffer(
        const Buffer& buffer,
        cl_bool blocking,
        ::size_t offset,
        ::size_t size,
        const void* ptr,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueWriteBuffer(
                object_, buffer(), blocking, offset, size,
                ptr,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
                __ENQUEUE_WRITE_BUFFER_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueCopyBuffer(
        const Buffer& src,
        const Buffer& dst,
        ::size_t src_offset,
        ::size_t dst_offset,
        ::size_t size,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueCopyBuffer(
                object_, src(), dst(), src_offset, dst_offset, size,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQEUE_COPY_BUFFER_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

#if defined(CL_VERSION_1_1)
    cl_int enqueueReadBufferRect(
        const Buffer& buffer,
        cl_bool blocking,
        const size_t<3>& buffer_offset,
        const size_t<3>& host_offset,
        const size_t<3>& region,
        ::size_t buffer_row_pitch,
        ::size_t buffer_slice_pitch,
        ::size_t host_row_pitch,
        ::size_t host_slice_pitch,
        void *ptr,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueReadBufferRect(
                object_, 
                buffer(), 
                blocking, 
                (const ::size_t *)buffer_offset,
                (const ::size_t *)host_offset,
                (const ::size_t *)region,
                buffer_row_pitch,
                buffer_slice_pitch,
                host_row_pitch,
                host_slice_pitch,
                ptr,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
                __ENQUEUE_READ_BUFFER_RECT_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }


    cl_int enqueueWriteBufferRect(
        const Buffer& buffer,
        cl_bool blocking,
        const size_t<3>& buffer_offset,
        const size_t<3>& host_offset,
        const size_t<3>& region,
        ::size_t buffer_row_pitch,
        ::size_t buffer_slice_pitch,
        ::size_t host_row_pitch,
        ::size_t host_slice_pitch,
        void *ptr,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueWriteBufferRect(
                object_, 
                buffer(), 
                blocking, 
                (const ::size_t *)buffer_offset,
                (const ::size_t *)host_offset,
                (const ::size_t *)region,
                buffer_row_pitch,
                buffer_slice_pitch,
                host_row_pitch,
                host_slice_pitch,
                ptr,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
                __ENQUEUE_WRITE_BUFFER_RECT_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueCopyBufferRect(
        const Buffer& src,
        const Buffer& dst,
        const size_t<3>& src_origin,
        const size_t<3>& dst_origin,
        const size_t<3>& region,
        ::size_t src_row_pitch,
        ::size_t src_slice_pitch,
        ::size_t dst_row_pitch,
        ::size_t dst_slice_pitch,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueCopyBufferRect(
                object_, 
                src(), 
                dst(), 
                (const ::size_t *)src_origin, 
                (const ::size_t *)dst_origin, 
                (const ::size_t *)region,
                src_row_pitch,
                src_slice_pitch,
                dst_row_pitch,
                dst_slice_pitch,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQEUE_COPY_BUFFER_RECT_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }
#endif

    cl_int enqueueReadImage(
        const Image& image,
        cl_bool blocking,
        const size_t<3>& origin,
        const size_t<3>& region,
        ::size_t row_pitch,
        ::size_t slice_pitch,
        void* ptr,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueReadImage(
                object_, image(), blocking, (const ::size_t *) origin,
                (const ::size_t *) region, row_pitch, slice_pitch, ptr,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_READ_IMAGE_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueWriteImage(
        const Image& image,
        cl_bool blocking,
        const size_t<3>& origin,
        const size_t<3>& region,
        ::size_t row_pitch,
        ::size_t slice_pitch,
        void* ptr,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueWriteImage(
                object_, image(), blocking, (const ::size_t *) origin,
                (const ::size_t *) region, row_pitch, slice_pitch, ptr,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_WRITE_IMAGE_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueCopyImage(
        const Image& src,
        const Image& dst,
        const size_t<3>& src_origin,
        const size_t<3>& dst_origin,
        const size_t<3>& region,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueCopyImage(
                object_, src(), dst(), (const ::size_t *) src_origin,
                (const ::size_t *)dst_origin, (const ::size_t *) region,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_COPY_IMAGE_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueCopyImageToBuffer(
        const Image& src,
        const Buffer& dst,
        const size_t<3>& src_origin,
        const size_t<3>& region,
        ::size_t dst_offset,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueCopyImageToBuffer(
                object_, src(), dst(), (const ::size_t *) src_origin,
                (const ::size_t *) region, dst_offset,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_COPY_IMAGE_TO_BUFFER_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueCopyBufferToImage(
        const Buffer& src,
        const Image& dst,
        ::size_t src_offset,
        const size_t<3>& dst_origin,
        const size_t<3>& region,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueCopyBufferToImage(
                object_, src(), dst(), src_offset,
                (const ::size_t *) dst_origin, (const ::size_t *) region,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_COPY_BUFFER_TO_IMAGE_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    void* enqueueMapBuffer(
        const Buffer& buffer,
        cl_bool blocking,
        cl_map_flags flags,
        ::size_t offset,
        ::size_t size,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL,
        cl_int* err = NULL) const
    {
        cl_int error;
        void * result = ::clEnqueueMapBuffer(
            object_, buffer(), blocking, flags, offset, size,
            (events != NULL) ? (cl_uint) events->size() : 0,
            (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
            (cl_event*) event,
            &error);

        detail::errHandler(error, __ENQUEUE_MAP_BUFFER_ERR);
        if (err != NULL) {
            *err = error;
        }
        return result;
    }

    void* enqueueMapImage(
        const Image& buffer,
        cl_bool blocking,
        cl_map_flags flags,
        const size_t<3>& origin,
        const size_t<3>& region,
        ::size_t * row_pitch,
        ::size_t * slice_pitch,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL,
        cl_int* err = NULL) const
    {
        cl_int error;
        void * result = ::clEnqueueMapImage(
            object_, buffer(), blocking, flags,
            (const ::size_t *) origin, (const ::size_t *) region,
            row_pitch, slice_pitch,
            (events != NULL) ? (cl_uint) events->size() : 0,
            (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
            (cl_event*) event,
            &error);

        detail::errHandler(error, __ENQUEUE_MAP_IMAGE_ERR);
        if (err != NULL) {
              *err = error;
        }
        return result;
    }

    cl_int enqueueUnmapMemObject(
        const Memory& memory,
        void* mapped_ptr,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueUnmapMemObject(
                object_, memory(), mapped_ptr,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_UNMAP_MEM_OBJECT_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueNDRangeKernel(
        const Kernel& kernel,
        const NDRange& offset,
        const NDRange& global,
        const NDRange& local = NullRange,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueNDRangeKernel(
                object_, kernel(), (cl_uint) global.dimensions(),
                offset.dimensions() != 0 ? (const ::size_t*) offset : NULL,
                (const ::size_t*) global,
                local.dimensions() != 0 ? (const ::size_t*) local : NULL,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_NDRANGE_KERNEL_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueTask(
        const Kernel& kernel,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueTask(
                object_, kernel(),
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_TASK_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueNativeKernel(
        void (*userFptr)(void *),
        std::pair<void*, ::size_t> args,
        const VECTOR_CLASS<Memory>* mem_objects = NULL,
        const VECTOR_CLASS<const void*>* mem_locs = NULL,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL) const
    {
        cl_mem * mems = (mem_objects != NULL && mem_objects->size() > 0) 
            ? (cl_mem*) alloca(mem_objects->size() * sizeof(cl_mem))
            : NULL;

        if (mems != NULL) {
            for (unsigned int i = 0; i < mem_objects->size(); i++) {
                mems[i] = ((*mem_objects)[i])();
            }
        }

        cl_event tmp;
        cl_int err = detail::errHandler(
            ::clEnqueueNativeKernel(
                object_, userFptr, args.first, args.second,
                (mem_objects != NULL) ? (cl_uint) mem_objects->size() : 0,
                mems,
                (mem_locs != NULL) ? (const void **) &mem_locs->front() : NULL,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_NATIVE_KERNEL);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }

    cl_int enqueueMarker(Event* event = NULL) const
    {
        return detail::errHandler(
            ::clEnqueueMarker(object_, (cl_event*) event),
            __ENQUEUE_MARKER_ERR);
    }

    cl_int enqueueWaitForEvents(const VECTOR_CLASS<Event>& events) const
    {
        return detail::errHandler(
            ::clEnqueueWaitForEvents(
                object_,
                (cl_uint) events.size(),
                (const cl_event*) &events.front()),
            __ENQUEUE_WAIT_FOR_EVENTS_ERR);
    }

    cl_int enqueueAcquireGLObjects(
         const VECTOR_CLASS<Memory>* mem_objects = NULL,
         const VECTOR_CLASS<Event>* events = NULL,
         Event* event = NULL) const
     {
        cl_event tmp;
        cl_int err = detail::errHandler(
             ::clEnqueueAcquireGLObjects(
                 object_,
                 (mem_objects != NULL) ? (cl_uint) mem_objects->size() : 0,
                 (mem_objects != NULL) ? (const cl_mem *) &mem_objects->front(): NULL,
                 (events != NULL) ? (cl_uint) events->size() : 0,
                 (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                 (event != NULL) ? &tmp : NULL),
             __ENQUEUE_ACQUIRE_GL_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
     }

    cl_int enqueueReleaseGLObjects(
         const VECTOR_CLASS<Memory>* mem_objects = NULL,
         const VECTOR_CLASS<Event>* events = NULL,
         Event* event = NULL) const
     {
        cl_event tmp;
        cl_int err = detail::errHandler(
             ::clEnqueueReleaseGLObjects(
                 object_,
                 (mem_objects != NULL) ? (cl_uint) mem_objects->size() : 0,
                 (mem_objects != NULL) ? (const cl_mem *) &mem_objects->front(): NULL,
                 (events != NULL) ? (cl_uint) events->size() : 0,
                 (events != NULL && events->size() > 0) ? (cl_event*) &events->front() : NULL,
                 (event != NULL) ? &tmp : NULL),
             __ENQUEUE_RELEASE_GL_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
     }

#if defined (USE_DX_INTEROP)
typedef CL_API_ENTRY cl_int (CL_API_CALL *PFN_clEnqueueAcquireD3D10ObjectsKHR)(
    cl_command_queue command_queue, cl_uint num_objects,
    const cl_mem* mem_objects, cl_uint num_events_in_wait_list,
    const cl_event* event_wait_list, cl_event* event);
typedef CL_API_ENTRY cl_int (CL_API_CALL *PFN_clEnqueueReleaseD3D10ObjectsKHR)(
    cl_command_queue command_queue, cl_uint num_objects,
    const cl_mem* mem_objects,  cl_uint num_events_in_wait_list,
    const cl_event* event_wait_list, cl_event* event);

    cl_int enqueueAcquireD3D10Objects(
         const VECTOR_CLASS<Memory>* mem_objects = NULL,
         const VECTOR_CLASS<Event>* events = NULL,
         Event* event = NULL) const
     {
         static PFN_clEnqueueAcquireD3D10ObjectsKHR pfn_clEnqueueAcquireD3D10ObjectsKHR = NULL;
         __INIT_CL_EXT_FCN_PTR(clEnqueueAcquireD3D10ObjectsKHR);
        
        cl_event tmp;
        cl_int err = detail::errHandler(
             pfn_clEnqueueAcquireD3D10ObjectsKHR(
                 object_,
                 (mem_objects != NULL) ? (cl_uint) mem_objects->size() : 0,
                 (mem_objects != NULL) ? (const cl_mem *) &mem_objects->front(): NULL,
                 (events != NULL) ? (cl_uint) events->size() : 0,
                 (events != NULL) ? (cl_event*) &events->front() : NULL,
                 (event != NULL) ? &tmp : NULL),
             __ENQUEUE_ACQUIRE_GL_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
     }

    cl_int enqueueReleaseD3D10Objects(
         const VECTOR_CLASS<Memory>* mem_objects = NULL,
         const VECTOR_CLASS<Event>* events = NULL,
         Event* event = NULL) const
    {
        static PFN_clEnqueueReleaseD3D10ObjectsKHR pfn_clEnqueueReleaseD3D10ObjectsKHR = NULL;
        __INIT_CL_EXT_FCN_PTR(clEnqueueReleaseD3D10ObjectsKHR);

        cl_event tmp;
        cl_int err = detail::errHandler(
            pfn_clEnqueueReleaseD3D10ObjectsKHR(
                object_,
                (mem_objects != NULL) ? (cl_uint) mem_objects->size() : 0,
                (mem_objects != NULL) ? (const cl_mem *) &mem_objects->front(): NULL,
                (events != NULL) ? (cl_uint) events->size() : 0,
                (events != NULL) ? (cl_event*) &events->front() : NULL,
                (event != NULL) ? &tmp : NULL),
            __ENQUEUE_RELEASE_GL_ERR);

        if (event != NULL && err == CL_SUCCESS)
            *event = tmp;

        return err;
    }
#endif

    cl_int enqueueBarrier() const
    {
        return detail::errHandler(
            ::clEnqueueBarrier(object_),
            __ENQUEUE_BARRIER_ERR);
    }

    cl_int flush() const
    {
        return detail::errHandler(::clFlush(object_), __FLUSH_ERR);
    }

    cl_int finish() const
    {
        return detail::errHandler(::clFinish(object_), __FINISH_ERR);
    }
};

__GET_INFO_HELPER_WITH_RETAIN(cl::CommandQueue)

#ifdef _WIN32
__declspec(selectany) volatile int CommandQueue::default_initialized_ = __DEFAULT_NOT_INITIALIZED;
__declspec(selectany) CommandQueue CommandQueue::default_;
__declspec(selectany) volatile cl_int CommandQueue::default_error_ = CL_SUCCESS;
#else
__attribute__((weak)) volatile int CommandQueue::default_initialized_ = __DEFAULT_NOT_INITIALIZED;
__attribute__((weak)) CommandQueue CommandQueue::default_;
__attribute__((weak)) volatile cl_int CommandQueue::default_error_ = CL_SUCCESS;
#endif

inline cl_int enqueueReadBuffer(
    const Buffer& buffer,
    cl_bool blocking,
    ::size_t offset,
    ::size_t size,
    void* ptr,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueReadBuffer(buffer, blocking, offset, size, ptr, events, event);
}

inline cl_int enqueueWriteBuffer(
        const Buffer& buffer,
        cl_bool blocking,
        ::size_t offset,
        ::size_t size,
        const void* ptr,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueWriteBuffer(buffer, blocking, offset, size, ptr, events, event);
}

inline cl_int enqueueCopyBuffer(
        const Buffer& src,
        const Buffer& dst,
        ::size_t src_offset,
        ::size_t dst_offset,
        ::size_t size,
        const VECTOR_CLASS<Event>* events = NULL,
        Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueCopyBuffer(src, dst, src_offset, dst_offset, size, events, event);
}

#if defined(CL_VERSION_1_1)
inline cl_int enqueueReadBufferRect(
    const Buffer& buffer,
    cl_bool blocking,
    const size_t<3>& buffer_offset,
    const size_t<3>& host_offset,
    const size_t<3>& region,
    ::size_t buffer_row_pitch,
    ::size_t buffer_slice_pitch,
    ::size_t host_row_pitch,
    ::size_t host_slice_pitch,
    void *ptr,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueReadBufferRect(
        buffer, 
        blocking, 
        buffer_offset, 
        host_offset,
        region,
        buffer_row_pitch,
        host_slice_pitch,
        host_row_pitch,
        host_slice_pitch,
        ptr, 
        events, 
        event);
}

inline cl_int enqueueWriteBufferRect(
    const Buffer& buffer,
    cl_bool blocking,
    const size_t<3>& buffer_offset,
    const size_t<3>& host_offset,
    const size_t<3>& region,
    ::size_t buffer_row_pitch,
    ::size_t buffer_slice_pitch,
    ::size_t host_row_pitch,
    ::size_t host_slice_pitch,
    void *ptr,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueWriteBufferRect(
        buffer, 
        blocking, 
        buffer_offset, 
        host_offset,
        region,
        buffer_row_pitch,
        buffer_slice_pitch,
        host_row_pitch,
        host_slice_pitch,
        ptr, 
        events, 
        event);
}

inline cl_int enqueueCopyBufferRect(
    const Buffer& src,
    const Buffer& dst,
    const size_t<3>& src_origin,
    const size_t<3>& dst_origin,
    const size_t<3>& region,
    ::size_t src_row_pitch,
    ::size_t src_slice_pitch,
    ::size_t dst_row_pitch,
    ::size_t dst_slice_pitch,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueCopyBufferRect(
        src,
        dst,
        src_origin,
        dst_origin,
        region,
        src_row_pitch,
        src_slice_pitch,
        dst_row_pitch,
        dst_slice_pitch,
        events, 
        event);
}
#endif

inline cl_int enqueueReadImage(
    const Image& image,
    cl_bool blocking,
    const size_t<3>& origin,
    const size_t<3>& region,
    ::size_t row_pitch,
    ::size_t slice_pitch,
    void* ptr,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL) 
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueReadImage(
        image,
        blocking,
        origin,
        region,
        row_pitch,
        slice_pitch,
        ptr,
        events, 
        event);
}

inline cl_int enqueueWriteImage(
    const Image& image,
    cl_bool blocking,
    const size_t<3>& origin,
    const size_t<3>& region,
    ::size_t row_pitch,
    ::size_t slice_pitch,
    void* ptr,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueWriteImage(
        image,
        blocking,
        origin,
        region,
        row_pitch,
        slice_pitch,
        ptr,
        events, 
        event);
}

inline cl_int enqueueCopyImage(
    const Image& src,
    const Image& dst,
    const size_t<3>& src_origin,
    const size_t<3>& dst_origin,
    const size_t<3>& region,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueCopyImage(
        src,
        dst,
        src_origin,
        dst_origin,
        region,
        events,
        event);
}

inline cl_int enqueueCopyImageToBuffer(
    const Image& src,
    const Buffer& dst,
    const size_t<3>& src_origin,
    const size_t<3>& region,
    ::size_t dst_offset,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueCopyImageToBuffer(
        src,
        dst,
        src_origin,
        region,
        dst_offset,
        events,
        event);
}

inline cl_int enqueueCopyBufferToImage(
    const Buffer& src,
    const Image& dst,
    ::size_t src_offset,
    const size_t<3>& dst_origin,
    const size_t<3>& region,
    const VECTOR_CLASS<Event>* events = NULL,
    Event* event = NULL)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.enqueueCopyBufferToImage(
        src,
        dst,
        src_offset,
        dst_origin,
        region,
        events,
        event);
}


inline cl_int flush(void)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }

    return queue.flush();
}

inline cl_int finish(void)
{
    cl_int error;
    CommandQueue queue = CommandQueue::getDefault(&error);

    if (error != CL_SUCCESS) {
        return error;
    }


    return queue.finish();
}


// Kernel Functor support
// New interface as of September 2011
// Requires the C++11 std::tr1::function (note do not support TR1)
// Visual Studio 2010 and GCC 4.2

#if (defined(_WIN32) && (_MSC_VER >= 1600)) || defined(linux) || defined(__APPLE__) || defined(__MACOSX)
struct EnqueueArgs
{
    CommandQueue queue_;
    NDRange offset_;
    NDRange global_;
    NDRange local_;

    EnqueueArgs(NDRange global) : 
      queue_(CommandQueue::getDefault()),
      offset_(NullRange), 
      global_(global),
      local_(NullRange)
    {

    }

    EnqueueArgs(NDRange global, NDRange local) : 
      queue_(CommandQueue::getDefault()),
      offset_(NullRange), 
      global_(global),
      local_(local)
    {

    }

    EnqueueArgs(NDRange offset, NDRange global, NDRange local) : 
      queue_(CommandQueue::getDefault()),
      offset_(offset), 
      global_(global),
      local_(local)
    {

    }

    EnqueueArgs(CommandQueue queue, NDRange global) : 
      queue_(queue),
      offset_(NullRange), 
      global_(global),
      local_(NullRange)
    {

    }

    EnqueueArgs(CommandQueue queue, NDRange global, NDRange local) : 
      queue_(queue),
      offset_(NullRange), 
      global_(global),
      local_(local)
    {

    }

    EnqueueArgs(CommandQueue queue, NDRange offset, NDRange global, NDRange local) : 
      queue_(queue),
      offset_(offset), 
      global_(global),
      local_(local)
    {

    }
};

namespace detail {

class NullType {};

template<int index, typename T0>
struct SetArg
{
    static void set (Kernel kernel, T0 arg)
    {
        kernel.setArg(index, arg);
    }
};


template<int index>
struct SetArg<index, NullType>
{
    static void set (Kernel kernel, NullType)
    {
    }
};

template<typename T>
struct MoveToDevice
{
	static void move(CommandQueue, T)
	{
	}
};


template<typename T>
struct MoveFromDevice
{
	static void move(CommandQueue, T)
	{
	}
};


template <
   typename T0,   typename T1,   typename T2,   typename T3,
   typename T4,   typename T5,   typename T6,   typename T7,
   typename T8,   typename T9,   typename T10,   typename T11,
   typename T12,   typename T13,   typename T14,   typename T15,
   typename T16,   typename T17,   typename T18,   typename T19,
   typename T20,   typename T21,   typename T22,   typename T23,
   typename T24,   typename T25,   typename T26,   typename T27,
   typename T28,   typename T29,   typename T30,   typename T31,
   typename T32,   typename T33,   typename T34,   typename T35,
   typename T36,   typename T37,   typename T38,   typename T39,
   typename T40,   typename T41,   typename T42,   typename T43,
   typename T44,   typename T45,   typename T46,   typename T47,
   typename T48,   typename T49,   typename T50,   typename T51,
   typename T52,   typename T53,   typename T54,   typename T55,
   typename T56,   typename T57,   typename T58,   typename T59,
   typename T60,   typename T61,   typename T62,   typename T63,
   typename T64,   typename T65,   typename T66,   typename T67,
   typename T68,   typename T69,   typename T70,   typename T71,
   typename T72,   typename T73,   typename T74,   typename T75,
   typename T76,   typename T77,   typename T78,   typename T79,
   typename T80,   typename T81,   typename T82,   typename T83,
   typename T84,   typename T85,   typename T86,   typename T87,
   typename T88,   typename T89,   typename T90,   typename T91,
   typename T92,   typename T93,   typename T94,   typename T95,
   typename T96,   typename T97,   typename T98,   typename T99,
   typename T100,   typename T101,   typename T102,   typename T103,
   typename T104,   typename T105,   typename T106,   typename T107,
   typename T108,   typename T109,   typename T110,   typename T111,
   typename T112,   typename T113,   typename T114,   typename T115,
   typename T116,   typename T117,   typename T118,   typename T119,
   typename T120,   typename T121,   typename T122,   typename T123,
   typename T124,   typename T125,   typename T126,   typename T127>
class KernelFunctorGlobal
{
private:
    Kernel kernel_;

public:
   KernelFunctorGlobal(
        Kernel kernel,
        cl_int * err = NULL) :
            kernel_(kernel)
    {}

   KernelFunctorGlobal(
        const Program& program,
        const STRING_CLASS name,
        cl_int * err = NULL) :
            kernel_(program, name.c_str(), err)
    {}

    Event operator() (
        const EnqueueArgs& args,
        T0 t0,
        T1 t1 = NullType(),
        T2 t2 = NullType(),
        T3 t3 = NullType(),
        T4 t4 = NullType(),
        T5 t5 = NullType(),
        T6 t6 = NullType(),
        T7 t7 = NullType(),
        T8 t8 = NullType(),
        T9 t9 = NullType(),
        T10 t10 = NullType(),
        T11 t11 = NullType(),
        T12 t12 = NullType(),
        T13 t13 = NullType(),
        T14 t14 = NullType(),
        T15 t15 = NullType(),
        T16 t16 = NullType(),
        T17 t17 = NullType(),
        T18 t18 = NullType(),
        T19 t19 = NullType(),
        T20 t20 = NullType(),
        T21 t21 = NullType(),
        T22 t22 = NullType(),
        T23 t23 = NullType(),
        T24 t24 = NullType(),
        T25 t25 = NullType(),
        T26 t26 = NullType(),
        T27 t27 = NullType(),
        T28 t28 = NullType(),
        T29 t29 = NullType(),
        T30 t30 = NullType(),
        T31 t31 = NullType(),
        T32 t32 = NullType(),
        T33 t33 = NullType(),
        T34 t34 = NullType(),
        T35 t35 = NullType(),
        T36 t36 = NullType(),
        T37 t37 = NullType(),
        T38 t38 = NullType(),
        T39 t39 = NullType(),
        T40 t40 = NullType(),
        T41 t41 = NullType(),
        T42 t42 = NullType(),
        T43 t43 = NullType(),
        T44 t44 = NullType(),
        T45 t45 = NullType(),
        T46 t46 = NullType(),
        T47 t47 = NullType(),
        T48 t48 = NullType(),
        T49 t49 = NullType(),
        T50 t50 = NullType(),
        T51 t51 = NullType(),
        T52 t52 = NullType(),
        T53 t53 = NullType(),
        T54 t54 = NullType(),
        T55 t55 = NullType(),
        T56 t56 = NullType(),
        T57 t57 = NullType(),
        T58 t58 = NullType(),
        T59 t59 = NullType(),
        T60 t60 = NullType(),
        T61 t61 = NullType(),
        T62 t62 = NullType(),
        T63 t63 = NullType(),
        T64 t64 = NullType(),
        T65 t65 = NullType(),
        T66 t66 = NullType(),
        T67 t67 = NullType(),
        T68 t68 = NullType(),
        T69 t69 = NullType(),
        T70 t70 = NullType(),
        T71 t71 = NullType(),
        T72 t72 = NullType(),
        T73 t73 = NullType(),
        T74 t74 = NullType(),
        T75 t75 = NullType(),
        T76 t76 = NullType(),
        T77 t77 = NullType(),
        T78 t78 = NullType(),
        T79 t79 = NullType(),
        T80 t80 = NullType(),
        T81 t81 = NullType(),
        T82 t82 = NullType(),
        T83 t83 = NullType(),
        T84 t84 = NullType(),
        T85 t85 = NullType(),
        T86 t86 = NullType(),
        T87 t87 = NullType(),
        T88 t88 = NullType(),
        T89 t89 = NullType(),
        T90 t90 = NullType(),
        T91 t91 = NullType(),
        T92 t92 = NullType(),
        T93 t93 = NullType(),
        T94 t94 = NullType(),
        T95 t95 = NullType(),
        T96 t96 = NullType(),
        T97 t97 = NullType(),
        T98 t98 = NullType(),
        T99 t99 = NullType(),
        T100 t100 = NullType(),
        T101 t101 = NullType(),
        T102 t102 = NullType(),
        T103 t103 = NullType(),
        T104 t104 = NullType(),
        T105 t105 = NullType(),
        T106 t106 = NullType(),
        T107 t107 = NullType(),
        T108 t108 = NullType(),
        T109 t109 = NullType(),
        T110 t110 = NullType(),
        T111 t111 = NullType(),
        T112 t112 = NullType(),
        T113 t113 = NullType(),
        T114 t114 = NullType(),
        T115 t115 = NullType(),
        T116 t116 = NullType(),
        T117 t117 = NullType(),
        T118 t118 = NullType(),
        T119 t119 = NullType(),
        T120 t120 = NullType(),
        T121 t121 = NullType(),
        T122 t122 = NullType(),
        T123 t123 = NullType(),
        T124 t124 = NullType(),
        T125 t125 = NullType(),
        T126 t126 = NullType(),
        T127 t127 = NullType())
    {
        Event event;
        SetArg<0, T0>::set(kernel_, t0);
        SetArg<1, T1>::set(kernel_, t1);
        SetArg<2, T2>::set(kernel_, t2);
        SetArg<3, T3>::set(kernel_, t3);
        SetArg<4, T4>::set(kernel_, t4);
        SetArg<5, T5>::set(kernel_, t5);
        SetArg<6, T6>::set(kernel_, t6);
        SetArg<7, T7>::set(kernel_, t7);
        SetArg<8, T8>::set(kernel_, t8);
        SetArg<9, T9>::set(kernel_, t9);
        SetArg<10, T10>::set(kernel_, t10);
        SetArg<11, T11>::set(kernel_, t11);
        SetArg<12, T12>::set(kernel_, t12);
        SetArg<13, T13>::set(kernel_, t13);
        SetArg<14, T14>::set(kernel_, t14);
        SetArg<15, T15>::set(kernel_, t15);
        SetArg<16, T16>::set(kernel_, t16);
        SetArg<17, T17>::set(kernel_, t17);
        SetArg<18, T18>::set(kernel_, t18);
        SetArg<19, T19>::set(kernel_, t19);
        SetArg<20, T20>::set(kernel_, t20);
        SetArg<21, T21>::set(kernel_, t21);
        SetArg<22, T22>::set(kernel_, t22);
        SetArg<23, T23>::set(kernel_, t23);
        SetArg<24, T24>::set(kernel_, t24);
        SetArg<25, T25>::set(kernel_, t25);
        SetArg<26, T26>::set(kernel_, t26);
        SetArg<27, T27>::set(kernel_, t27);
        SetArg<28, T28>::set(kernel_, t28);
        SetArg<29, T29>::set(kernel_, t29);
        SetArg<30, T30>::set(kernel_, t30);
        SetArg<31, T31>::set(kernel_, t31);
        SetArg<32, T32>::set(kernel_, t32);
        SetArg<33, T33>::set(kernel_, t33);
        SetArg<34, T34>::set(kernel_, t34);
        SetArg<35, T35>::set(kernel_, t35);
        SetArg<36, T36>::set(kernel_, t36);
        SetArg<37, T37>::set(kernel_, t37);
        SetArg<38, T38>::set(kernel_, t38);
        SetArg<39, T39>::set(kernel_, t39);
        SetArg<40, T40>::set(kernel_, t40);
        SetArg<41, T41>::set(kernel_, t41);
        SetArg<42, T42>::set(kernel_, t42);
        SetArg<43, T43>::set(kernel_, t43);
        SetArg<44, T44>::set(kernel_, t44);
        SetArg<45, T45>::set(kernel_, t45);
        SetArg<46, T46>::set(kernel_, t46);
        SetArg<47, T47>::set(kernel_, t47);
        SetArg<48, T48>::set(kernel_, t48);
        SetArg<49, T49>::set(kernel_, t49);
        SetArg<50, T50>::set(kernel_, t50);
        SetArg<51, T51>::set(kernel_, t51);
        SetArg<52, T52>::set(kernel_, t52);
        SetArg<53, T53>::set(kernel_, t53);
        SetArg<54, T54>::set(kernel_, t54);
        SetArg<55, T55>::set(kernel_, t55);
        SetArg<56, T56>::set(kernel_, t56);
        SetArg<57, T57>::set(kernel_, t57);
        SetArg<58, T58>::set(kernel_, t58);
        SetArg<59, T59>::set(kernel_, t59);
        SetArg<60, T60>::set(kernel_, t60);
        SetArg<61, T61>::set(kernel_, t61);
        SetArg<62, T62>::set(kernel_, t62);
        SetArg<63, T63>::set(kernel_, t63);
        SetArg<64, T64>::set(kernel_, t64);
        SetArg<65, T65>::set(kernel_, t65);
        SetArg<66, T66>::set(kernel_, t66);
        SetArg<67, T67>::set(kernel_, t67);
        SetArg<68, T68>::set(kernel_, t68);
        SetArg<69, T69>::set(kernel_, t69);
        SetArg<70, T70>::set(kernel_, t70);
        SetArg<71, T71>::set(kernel_, t71);
        SetArg<72, T72>::set(kernel_, t72);
        SetArg<73, T73>::set(kernel_, t73);
        SetArg<74, T74>::set(kernel_, t74);
        SetArg<75, T75>::set(kernel_, t75);
        SetArg<76, T76>::set(kernel_, t76);
        SetArg<77, T77>::set(kernel_, t77);
        SetArg<78, T78>::set(kernel_, t78);
        SetArg<79, T79>::set(kernel_, t79);
        SetArg<80, T80>::set(kernel_, t80);
        SetArg<81, T81>::set(kernel_, t81);
        SetArg<82, T82>::set(kernel_, t82);
        SetArg<83, T83>::set(kernel_, t83);
        SetArg<84, T84>::set(kernel_, t84);
        SetArg<85, T85>::set(kernel_, t85);
        SetArg<86, T86>::set(kernel_, t86);
        SetArg<87, T87>::set(kernel_, t87);
        SetArg<88, T88>::set(kernel_, t88);
        SetArg<89, T89>::set(kernel_, t89);
        SetArg<90, T90>::set(kernel_, t90);
        SetArg<91, T91>::set(kernel_, t91);
        SetArg<92, T92>::set(kernel_, t92);
        SetArg<93, T93>::set(kernel_, t93);
        SetArg<94, T94>::set(kernel_, t94);
        SetArg<95, T95>::set(kernel_, t95);
        SetArg<96, T96>::set(kernel_, t96);
        SetArg<97, T97>::set(kernel_, t97);
        SetArg<98, T98>::set(kernel_, t98);
        SetArg<99, T99>::set(kernel_, t99);
        SetArg<100, T100>::set(kernel_, t100);
        SetArg<101, T101>::set(kernel_, t101);
        SetArg<102, T102>::set(kernel_, t102);
        SetArg<103, T103>::set(kernel_, t103);
        SetArg<104, T104>::set(kernel_, t104);
        SetArg<105, T105>::set(kernel_, t105);
        SetArg<106, T106>::set(kernel_, t106);
        SetArg<107, T107>::set(kernel_, t107);
        SetArg<108, T108>::set(kernel_, t108);
        SetArg<109, T109>::set(kernel_, t109);
        SetArg<110, T110>::set(kernel_, t110);
        SetArg<111, T111>::set(kernel_, t111);
        SetArg<112, T112>::set(kernel_, t112);
        SetArg<113, T113>::set(kernel_, t113);
        SetArg<114, T114>::set(kernel_, t114);
        SetArg<115, T115>::set(kernel_, t115);
        SetArg<116, T116>::set(kernel_, t116);
        SetArg<117, T117>::set(kernel_, t117);
        SetArg<118, T118>::set(kernel_, t118);
        SetArg<119, T119>::set(kernel_, t119);
        SetArg<120, T120>::set(kernel_, t120);
        SetArg<121, T121>::set(kernel_, t121);
        SetArg<122, T122>::set(kernel_, t122);
        SetArg<123, T123>::set(kernel_, t123);
        SetArg<124, T124>::set(kernel_, t124);
        SetArg<125, T125>::set(kernel_, t125);
        SetArg<126, T126>::set(kernel_, t126);
        SetArg<127, T127>::set(kernel_, t127);

		// Move any arguments that need moving
        MoveToDevice<T0>::move(args.queue_, t0);
        MoveToDevice<T1>::move(args.queue_, t1);
        MoveToDevice<T2>::move(args.queue_, t2);
        MoveToDevice<T3>::move(args.queue_, t3);
        MoveToDevice<T4>::move(args.queue_, t4);
        MoveToDevice<T5>::move(args.queue_, t5);
        MoveToDevice<T6>::move(args.queue_, t6);
        MoveToDevice<T7>::move(args.queue_, t7);
        MoveToDevice<T8>::move(args.queue_, t8);
        MoveToDevice<T9>::move(args.queue_, t9);
        MoveToDevice<T10>::move(args.queue_, t10);
        MoveToDevice<T11>::move(args.queue_, t11);
        MoveToDevice<T12>::move(args.queue_, t12);
        MoveToDevice<T13>::move(args.queue_, t13);
        MoveToDevice<T14>::move(args.queue_, t14);
        MoveToDevice<T15>::move(args.queue_, t15);
        MoveToDevice<T16>::move(args.queue_, t16);
        MoveToDevice<T17>::move(args.queue_, t17);
        MoveToDevice<T18>::move(args.queue_, t18);
        MoveToDevice<T19>::move(args.queue_, t19);
        MoveToDevice<T20>::move(args.queue_, t20);
        MoveToDevice<T21>::move(args.queue_, t21);
        MoveToDevice<T22>::move(args.queue_, t22);
        MoveToDevice<T23>::move(args.queue_, t23);
        MoveToDevice<T24>::move(args.queue_, t24);
        MoveToDevice<T25>::move(args.queue_, t25);
        MoveToDevice<T26>::move(args.queue_, t26);
        MoveToDevice<T27>::move(args.queue_, t27);
        MoveToDevice<T28>::move(args.queue_, t28);
        MoveToDevice<T29>::move(args.queue_, t29);
        MoveToDevice<T30>::move(args.queue_, t30);
        MoveToDevice<T31>::move(args.queue_, t31);
        MoveToDevice<T32>::move(args.queue_, t32);
        MoveToDevice<T33>::move(args.queue_, t33);
        MoveToDevice<T34>::move(args.queue_, t34);
        MoveToDevice<T35>::move(args.queue_, t35);
        MoveToDevice<T36>::move(args.queue_, t36);
        MoveToDevice<T37>::move(args.queue_, t37);
        MoveToDevice<T38>::move(args.queue_, t38);
        MoveToDevice<T39>::move(args.queue_, t39);
        MoveToDevice<T40>::move(args.queue_, t40);
        MoveToDevice<T41>::move(args.queue_, t41);
        MoveToDevice<T42>::move(args.queue_, t42);
        MoveToDevice<T43>::move(args.queue_, t43);
        MoveToDevice<T44>::move(args.queue_, t44);
        MoveToDevice<T45>::move(args.queue_, t45);
        MoveToDevice<T46>::move(args.queue_, t46);
        MoveToDevice<T47>::move(args.queue_, t47);
        MoveToDevice<T48>::move(args.queue_, t48);
        MoveToDevice<T49>::move(args.queue_, t49);
        MoveToDevice<T50>::move(args.queue_, t50);
        MoveToDevice<T51>::move(args.queue_, t51);
        MoveToDevice<T52>::move(args.queue_, t52);
        MoveToDevice<T53>::move(args.queue_, t53);
        MoveToDevice<T54>::move(args.queue_, t54);
        MoveToDevice<T55>::move(args.queue_, t55);
        MoveToDevice<T56>::move(args.queue_, t56);
        MoveToDevice<T57>::move(args.queue_, t57);
        MoveToDevice<T58>::move(args.queue_, t58);
        MoveToDevice<T59>::move(args.queue_, t59);
        MoveToDevice<T60>::move(args.queue_, t60);
        MoveToDevice<T61>::move(args.queue_, t61);
        MoveToDevice<T62>::move(args.queue_, t62);
        MoveToDevice<T63>::move(args.queue_, t63);
        MoveToDevice<T64>::move(args.queue_, t64);
        MoveToDevice<T65>::move(args.queue_, t65);
        MoveToDevice<T66>::move(args.queue_, t66);
        MoveToDevice<T67>::move(args.queue_, t67);
        MoveToDevice<T68>::move(args.queue_, t68);
        MoveToDevice<T69>::move(args.queue_, t69);
        MoveToDevice<T70>::move(args.queue_, t70);
        MoveToDevice<T71>::move(args.queue_, t71);
        MoveToDevice<T72>::move(args.queue_, t72);
        MoveToDevice<T73>::move(args.queue_, t73);
        MoveToDevice<T74>::move(args.queue_, t74);
        MoveToDevice<T75>::move(args.queue_, t75);
        MoveToDevice<T76>::move(args.queue_, t76);
        MoveToDevice<T77>::move(args.queue_, t77);
        MoveToDevice<T78>::move(args.queue_, t78);
        MoveToDevice<T79>::move(args.queue_, t79);
        MoveToDevice<T80>::move(args.queue_, t80);
        MoveToDevice<T81>::move(args.queue_, t81);
        MoveToDevice<T82>::move(args.queue_, t82);
        MoveToDevice<T83>::move(args.queue_, t83);
        MoveToDevice<T84>::move(args.queue_, t84);
        MoveToDevice<T85>::move(args.queue_, t85);
        MoveToDevice<T86>::move(args.queue_, t86);
        MoveToDevice<T87>::move(args.queue_, t87);
        MoveToDevice<T88>::move(args.queue_, t88);
        MoveToDevice<T89>::move(args.queue_, t89);
        MoveToDevice<T90>::move(args.queue_, t90);
        MoveToDevice<T91>::move(args.queue_, t91);
        MoveToDevice<T92>::move(args.queue_, t92);
        MoveToDevice<T93>::move(args.queue_, t93);
        MoveToDevice<T94>::move(args.queue_, t94);
        MoveToDevice<T95>::move(args.queue_, t95);
        MoveToDevice<T96>::move(args.queue_, t96);
        MoveToDevice<T97>::move(args.queue_, t97);
        MoveToDevice<T98>::move(args.queue_, t98);
        MoveToDevice<T99>::move(args.queue_, t99);
        MoveToDevice<T100>::move(args.queue_, t100);
        MoveToDevice<T101>::move(args.queue_, t101);
        MoveToDevice<T102>::move(args.queue_, t102);
        MoveToDevice<T103>::move(args.queue_, t103);
        MoveToDevice<T104>::move(args.queue_, t104);
        MoveToDevice<T105>::move(args.queue_, t105);
        MoveToDevice<T106>::move(args.queue_, t106);
        MoveToDevice<T107>::move(args.queue_, t107);
        MoveToDevice<T108>::move(args.queue_, t108);
        MoveToDevice<T109>::move(args.queue_, t109);
        MoveToDevice<T110>::move(args.queue_, t110);
        MoveToDevice<T111>::move(args.queue_, t111);
        MoveToDevice<T112>::move(args.queue_, t112);
        MoveToDevice<T113>::move(args.queue_, t113);
        MoveToDevice<T114>::move(args.queue_, t114);
        MoveToDevice<T115>::move(args.queue_, t115);
        MoveToDevice<T116>::move(args.queue_, t116);
        MoveToDevice<T117>::move(args.queue_, t117);
        MoveToDevice<T118>::move(args.queue_, t118);
        MoveToDevice<T119>::move(args.queue_, t119);
        MoveToDevice<T120>::move(args.queue_, t120);
        MoveToDevice<T121>::move(args.queue_, t121);
        MoveToDevice<T122>::move(args.queue_, t122);
        MoveToDevice<T123>::move(args.queue_, t123);
        MoveToDevice<T124>::move(args.queue_, t124);
        MoveToDevice<T125>::move(args.queue_, t125);
        MoveToDevice<T126>::move(args.queue_, t126);
        MoveToDevice<T127>::move(args.queue_, t127);

        args.queue_.enqueueNDRangeKernel(
            kernel_,
            args.offset_,
            args.global_,
            args.local_,
            NULL,
            &event);

		// Move any arguments that need moving
        MoveFromDevice<T0>::move(args.queue_, t0);
        MoveFromDevice<T1>::move(args.queue_, t1);
        MoveFromDevice<T2>::move(args.queue_, t2);
        MoveFromDevice<T3>::move(args.queue_, t3);
        MoveFromDevice<T4>::move(args.queue_, t4);
        MoveFromDevice<T5>::move(args.queue_, t5);
        MoveFromDevice<T6>::move(args.queue_, t6);
        MoveFromDevice<T7>::move(args.queue_, t7);
        MoveFromDevice<T8>::move(args.queue_, t8);
        MoveFromDevice<T9>::move(args.queue_, t9);
        MoveFromDevice<T10>::move(args.queue_, t10);
        MoveFromDevice<T11>::move(args.queue_, t11);
        MoveFromDevice<T12>::move(args.queue_, t12);
        MoveFromDevice<T13>::move(args.queue_, t13);
        MoveFromDevice<T14>::move(args.queue_, t14);
        MoveFromDevice<T15>::move(args.queue_, t15);
        MoveFromDevice<T16>::move(args.queue_, t16);
        MoveFromDevice<T17>::move(args.queue_, t17);
        MoveFromDevice<T18>::move(args.queue_, t18);
        MoveFromDevice<T19>::move(args.queue_, t19);
        MoveFromDevice<T20>::move(args.queue_, t20);
        MoveFromDevice<T21>::move(args.queue_, t21);
        MoveFromDevice<T22>::move(args.queue_, t22);
        MoveFromDevice<T23>::move(args.queue_, t23);
        MoveFromDevice<T24>::move(args.queue_, t24);
        MoveFromDevice<T25>::move(args.queue_, t25);
        MoveFromDevice<T26>::move(args.queue_, t26);
        MoveFromDevice<T27>::move(args.queue_, t27);
        MoveFromDevice<T28>::move(args.queue_, t28);
        MoveFromDevice<T29>::move(args.queue_, t29);
        MoveFromDevice<T30>::move(args.queue_, t30);
        MoveFromDevice<T31>::move(args.queue_, t31);
        MoveFromDevice<T32>::move(args.queue_, t32);
        MoveFromDevice<T33>::move(args.queue_, t33);
        MoveFromDevice<T34>::move(args.queue_, t34);
        MoveFromDevice<T35>::move(args.queue_, t35);
        MoveFromDevice<T36>::move(args.queue_, t36);
        MoveFromDevice<T37>::move(args.queue_, t37);
        MoveFromDevice<T38>::move(args.queue_, t38);
        MoveFromDevice<T39>::move(args.queue_, t39);
        MoveFromDevice<T40>::move(args.queue_, t40);
        MoveFromDevice<T41>::move(args.queue_, t41);
        MoveFromDevice<T42>::move(args.queue_, t42);
        MoveFromDevice<T43>::move(args.queue_, t43);
        MoveFromDevice<T44>::move(args.queue_, t44);
        MoveFromDevice<T45>::move(args.queue_, t45);
        MoveFromDevice<T46>::move(args.queue_, t46);
        MoveFromDevice<T47>::move(args.queue_, t47);
        MoveFromDevice<T48>::move(args.queue_, t48);
        MoveFromDevice<T49>::move(args.queue_, t49);
        MoveFromDevice<T50>::move(args.queue_, t50);
        MoveFromDevice<T51>::move(args.queue_, t51);
        MoveFromDevice<T52>::move(args.queue_, t52);
        MoveFromDevice<T53>::move(args.queue_, t53);
        MoveFromDevice<T54>::move(args.queue_, t54);
        MoveFromDevice<T55>::move(args.queue_, t55);
        MoveFromDevice<T56>::move(args.queue_, t56);
        MoveFromDevice<T57>::move(args.queue_, t57);
        MoveFromDevice<T58>::move(args.queue_, t58);
        MoveFromDevice<T59>::move(args.queue_, t59);
        MoveFromDevice<T60>::move(args.queue_, t60);
        MoveFromDevice<T61>::move(args.queue_, t61);
        MoveFromDevice<T62>::move(args.queue_, t62);
        MoveFromDevice<T63>::move(args.queue_, t63);
        MoveFromDevice<T64>::move(args.queue_, t64);
        MoveFromDevice<T65>::move(args.queue_, t65);
        MoveFromDevice<T66>::move(args.queue_, t66);
        MoveFromDevice<T67>::move(args.queue_, t67);
        MoveFromDevice<T68>::move(args.queue_, t68);
        MoveFromDevice<T69>::move(args.queue_, t69);
        MoveFromDevice<T70>::move(args.queue_, t70);
        MoveFromDevice<T71>::move(args.queue_, t71);
        MoveFromDevice<T72>::move(args.queue_, t72);
        MoveFromDevice<T73>::move(args.queue_, t73);
        MoveFromDevice<T74>::move(args.queue_, t74);
        MoveFromDevice<T75>::move(args.queue_, t75);
        MoveFromDevice<T76>::move(args.queue_, t76);
        MoveFromDevice<T77>::move(args.queue_, t77);
        MoveFromDevice<T78>::move(args.queue_, t78);
        MoveFromDevice<T79>::move(args.queue_, t79);
        MoveFromDevice<T80>::move(args.queue_, t80);
        MoveFromDevice<T81>::move(args.queue_, t81);
        MoveFromDevice<T82>::move(args.queue_, t82);
        MoveFromDevice<T83>::move(args.queue_, t83);
        MoveFromDevice<T84>::move(args.queue_, t84);
        MoveFromDevice<T85>::move(args.queue_, t85);
        MoveFromDevice<T86>::move(args.queue_, t86);
        MoveFromDevice<T87>::move(args.queue_, t87);
        MoveFromDevice<T88>::move(args.queue_, t88);
        MoveFromDevice<T89>::move(args.queue_, t89);
        MoveFromDevice<T90>::move(args.queue_, t90);
        MoveFromDevice<T91>::move(args.queue_, t91);
        MoveFromDevice<T92>::move(args.queue_, t92);
        MoveFromDevice<T93>::move(args.queue_, t93);
        MoveFromDevice<T94>::move(args.queue_, t94);
        MoveFromDevice<T95>::move(args.queue_, t95);
        MoveFromDevice<T96>::move(args.queue_, t96);
        MoveFromDevice<T97>::move(args.queue_, t97);
        MoveFromDevice<T98>::move(args.queue_, t98);
        MoveFromDevice<T99>::move(args.queue_, t99);
        MoveFromDevice<T100>::move(args.queue_, t100);
        MoveFromDevice<T101>::move(args.queue_, t101);
        MoveFromDevice<T102>::move(args.queue_, t102);
        MoveFromDevice<T103>::move(args.queue_, t103);
        MoveFromDevice<T104>::move(args.queue_, t104);
        MoveFromDevice<T105>::move(args.queue_, t105);
        MoveFromDevice<T106>::move(args.queue_, t106);
        MoveFromDevice<T107>::move(args.queue_, t107);
        MoveFromDevice<T108>::move(args.queue_, t108);
        MoveFromDevice<T109>::move(args.queue_, t109);
        MoveFromDevice<T110>::move(args.queue_, t110);
        MoveFromDevice<T111>::move(args.queue_, t111);
        MoveFromDevice<T112>::move(args.queue_, t112);
        MoveFromDevice<T113>::move(args.queue_, t113);
        MoveFromDevice<T114>::move(args.queue_, t114);
        MoveFromDevice<T115>::move(args.queue_, t115);
        MoveFromDevice<T116>::move(args.queue_, t116);
        MoveFromDevice<T117>::move(args.queue_, t117);
        MoveFromDevice<T118>::move(args.queue_, t118);
        MoveFromDevice<T119>::move(args.queue_, t119);
        MoveFromDevice<T120>::move(args.queue_, t120);
        MoveFromDevice<T121>::move(args.queue_, t121);
        MoveFromDevice<T122>::move(args.queue_, t122);
        MoveFromDevice<T123>::move(args.queue_, t123);
        MoveFromDevice<T124>::move(args.queue_, t124);
        MoveFromDevice<T125>::move(args.queue_, t125);
        MoveFromDevice<T126>::move(args.queue_, t126);
        MoveFromDevice<T127>::move(args.queue_, t127);

        return event;
    }

    Event operator() (
        const EnqueueArgs& args,
        const Event& waitEvent,
        T0 t0,
        T1 t1 = NullType(),
        T2 t2 = NullType(),
        T3 t3 = NullType(),
        T4 t4 = NullType(),
        T5 t5 = NullType(),
        T6 t6 = NullType(),
        T7 t7 = NullType(),
        T8 t8 = NullType(),
        T9 t9 = NullType(),
        T10 t10 = NullType(),
        T11 t11 = NullType(),
        T12 t12 = NullType(),
        T13 t13 = NullType(),
        T14 t14 = NullType(),
        T15 t15 = NullType(),
        T16 t16 = NullType(),
        T17 t17 = NullType(),
        T18 t18 = NullType(),
        T19 t19 = NullType(),
        T20 t20 = NullType(),
        T21 t21 = NullType(),
        T22 t22 = NullType(),
        T23 t23 = NullType(),
        T24 t24 = NullType(),
        T25 t25 = NullType(),
        T26 t26 = NullType(),
        T27 t27 = NullType(),
        T28 t28 = NullType(),
        T29 t29 = NullType(),
        T30 t30 = NullType(),
        T31 t31 = NullType(),
        T32 t32 = NullType(),
        T33 t33 = NullType(),
        T34 t34 = NullType(),
        T35 t35 = NullType(),
        T36 t36 = NullType(),
        T37 t37 = NullType(),
        T38 t38 = NullType(),
        T39 t39 = NullType(),
        T40 t40 = NullType(),
        T41 t41 = NullType(),
        T42 t42 = NullType(),
        T43 t43 = NullType(),
        T44 t44 = NullType(),
        T45 t45 = NullType(),
        T46 t46 = NullType(),
        T47 t47 = NullType(),
        T48 t48 = NullType(),
        T49 t49 = NullType(),
        T50 t50 = NullType(),
        T51 t51 = NullType(),
        T52 t52 = NullType(),
        T53 t53 = NullType(),
        T54 t54 = NullType(),
        T55 t55 = NullType(),
        T56 t56 = NullType(),
        T57 t57 = NullType(),
        T58 t58 = NullType(),
        T59 t59 = NullType(),
        T60 t60 = NullType(),
        T61 t61 = NullType(),
        T62 t62 = NullType(),
        T63 t63 = NullType(),
        T64 t64 = NullType(),
        T65 t65 = NullType(),
        T66 t66 = NullType(),
        T67 t67 = NullType(),
        T68 t68 = NullType(),
        T69 t69 = NullType(),
        T70 t70 = NullType(),
        T71 t71 = NullType(),
        T72 t72 = NullType(),
        T73 t73 = NullType(),
        T74 t74 = NullType(),
        T75 t75 = NullType(),
        T76 t76 = NullType(),
        T77 t77 = NullType(),
        T78 t78 = NullType(),
        T79 t79 = NullType(),
        T80 t80 = NullType(),
        T81 t81 = NullType(),
        T82 t82 = NullType(),
        T83 t83 = NullType(),
        T84 t84 = NullType(),
        T85 t85 = NullType(),
        T86 t86 = NullType(),
        T87 t87 = NullType(),
        T88 t88 = NullType(),
        T89 t89 = NullType(),
        T90 t90 = NullType(),
        T91 t91 = NullType(),
        T92 t92 = NullType(),
        T93 t93 = NullType(),
        T94 t94 = NullType(),
        T95 t95 = NullType(),
        T96 t96 = NullType(),
        T97 t97 = NullType(),
        T98 t98 = NullType(),
        T99 t99 = NullType(),
        T100 t100 = NullType(),
        T101 t101 = NullType(),
        T102 t102 = NullType(),
        T103 t103 = NullType(),
        T104 t104 = NullType(),
        T105 t105 = NullType(),
        T106 t106 = NullType(),
        T107 t107 = NullType(),
        T108 t108 = NullType(),
        T109 t109 = NullType(),
        T110 t110 = NullType(),
        T111 t111 = NullType(),
        T112 t112 = NullType(),
        T113 t113 = NullType(),
        T114 t114 = NullType(),
        T115 t115 = NullType(),
        T116 t116 = NullType(),
        T117 t117 = NullType(),
        T118 t118 = NullType(),
        T119 t119 = NullType(),
        T120 t120 = NullType(),
        T121 t121 = NullType(),
        T122 t122 = NullType(),
        T123 t123 = NullType(),
        T124 t124 = NullType(),
        T125 t125 = NullType(),
        T126 t126 = NullType(),
        T127 t127 = NullType())
    {
        Event event;
        SetArg<0, T0>::set(kernel_, t0);
        SetArg<1, T1>::set(kernel_, t1);
        SetArg<2, T2>::set(kernel_, t2);
        SetArg<3, T3>::set(kernel_, t3);
        SetArg<4, T4>::set(kernel_, t4);
        SetArg<5, T5>::set(kernel_, t5);
        SetArg<6, T6>::set(kernel_, t6);
        SetArg<7, T7>::set(kernel_, t7);
        SetArg<8, T8>::set(kernel_, t8);
        SetArg<9, T9>::set(kernel_, t9);
        SetArg<10, T10>::set(kernel_, t10);
        SetArg<11, T11>::set(kernel_, t11);
        SetArg<12, T12>::set(kernel_, t12);
        SetArg<13, T13>::set(kernel_, t13);
        SetArg<14, T14>::set(kernel_, t14);
        SetArg<15, T15>::set(kernel_, t15);
        SetArg<16, T16>::set(kernel_, t16);
        SetArg<17, T17>::set(kernel_, t17);
        SetArg<18, T18>::set(kernel_, t18);
        SetArg<19, T19>::set(kernel_, t19);
        SetArg<20, T20>::set(kernel_, t20);
        SetArg<21, T21>::set(kernel_, t21);
        SetArg<22, T22>::set(kernel_, t22);
        SetArg<23, T23>::set(kernel_, t23);
        SetArg<24, T24>::set(kernel_, t24);
        SetArg<25, T25>::set(kernel_, t25);
        SetArg<26, T26>::set(kernel_, t26);
        SetArg<27, T27>::set(kernel_, t27);
        SetArg<28, T28>::set(kernel_, t28);
        SetArg<29, T29>::set(kernel_, t29);
        SetArg<30, T30>::set(kernel_, t30);
        SetArg<31, T31>::set(kernel_, t31);
        SetArg<32, T32>::set(kernel_, t32);
        SetArg<33, T33>::set(kernel_, t33);
        SetArg<34, T34>::set(kernel_, t34);
        SetArg<35, T35>::set(kernel_, t35);
        SetArg<36, T36>::set(kernel_, t36);
        SetArg<37, T37>::set(kernel_, t37);
        SetArg<38, T38>::set(kernel_, t38);
        SetArg<39, T39>::set(kernel_, t39);
        SetArg<40, T40>::set(kernel_, t40);
        SetArg<41, T41>::set(kernel_, t41);
        SetArg<42, T42>::set(kernel_, t42);
        SetArg<43, T43>::set(kernel_, t43);
        SetArg<44, T44>::set(kernel_, t44);
        SetArg<45, T45>::set(kernel_, t45);
        SetArg<46, T46>::set(kernel_, t46);
        SetArg<47, T47>::set(kernel_, t47);
        SetArg<48, T48>::set(kernel_, t48);
        SetArg<49, T49>::set(kernel_, t49);
        SetArg<50, T50>::set(kernel_, t50);
        SetArg<51, T51>::set(kernel_, t51);
        SetArg<52, T52>::set(kernel_, t52);
        SetArg<53, T53>::set(kernel_, t53);
        SetArg<54, T54>::set(kernel_, t54);
        SetArg<55, T55>::set(kernel_, t55);
        SetArg<56, T56>::set(kernel_, t56);
        SetArg<57, T57>::set(kernel_, t57);
        SetArg<58, T58>::set(kernel_, t58);
        SetArg<59, T59>::set(kernel_, t59);
        SetArg<60, T60>::set(kernel_, t60);
        SetArg<61, T61>::set(kernel_, t61);
        SetArg<62, T62>::set(kernel_, t62);
        SetArg<63, T63>::set(kernel_, t63);
        SetArg<64, T64>::set(kernel_, t64);
        SetArg<65, T65>::set(kernel_, t65);
        SetArg<66, T66>::set(kernel_, t66);
        SetArg<67, T67>::set(kernel_, t67);
        SetArg<68, T68>::set(kernel_, t68);
        SetArg<69, T69>::set(kernel_, t69);
        SetArg<70, T70>::set(kernel_, t70);
        SetArg<71, T71>::set(kernel_, t71);
        SetArg<72, T72>::set(kernel_, t72);
        SetArg<73, T73>::set(kernel_, t73);
        SetArg<74, T74>::set(kernel_, t74);
        SetArg<75, T75>::set(kernel_, t75);
        SetArg<76, T76>::set(kernel_, t76);
        SetArg<77, T77>::set(kernel_, t77);
        SetArg<78, T78>::set(kernel_, t78);
        SetArg<79, T79>::set(kernel_, t79);
        SetArg<80, T80>::set(kernel_, t80);
        SetArg<81, T81>::set(kernel_, t81);
        SetArg<82, T82>::set(kernel_, t82);
        SetArg<83, T83>::set(kernel_, t83);
        SetArg<84, T84>::set(kernel_, t84);
        SetArg<85, T85>::set(kernel_, t85);
        SetArg<86, T86>::set(kernel_, t86);
        SetArg<87, T87>::set(kernel_, t87);
        SetArg<88, T88>::set(kernel_, t88);
        SetArg<89, T89>::set(kernel_, t89);
        SetArg<90, T90>::set(kernel_, t90);
        SetArg<91, T91>::set(kernel_, t91);
        SetArg<92, T92>::set(kernel_, t92);
        SetArg<93, T93>::set(kernel_, t93);
        SetArg<94, T94>::set(kernel_, t94);
        SetArg<95, T95>::set(kernel_, t95);
        SetArg<96, T96>::set(kernel_, t96);
        SetArg<97, T97>::set(kernel_, t97);
        SetArg<98, T98>::set(kernel_, t98);
        SetArg<99, T99>::set(kernel_, t99);
        SetArg<100, T100>::set(kernel_, t100);
        SetArg<101, T101>::set(kernel_, t101);
        SetArg<102, T102>::set(kernel_, t102);
        SetArg<103, T103>::set(kernel_, t103);
        SetArg<104, T104>::set(kernel_, t104);
        SetArg<105, T105>::set(kernel_, t105);
        SetArg<106, T106>::set(kernel_, t106);
        SetArg<107, T107>::set(kernel_, t107);
        SetArg<108, T108>::set(kernel_, t108);
        SetArg<109, T109>::set(kernel_, t109);
        SetArg<110, T110>::set(kernel_, t110);
        SetArg<111, T111>::set(kernel_, t111);
        SetArg<112, T112>::set(kernel_, t112);
        SetArg<113, T113>::set(kernel_, t113);
        SetArg<114, T114>::set(kernel_, t114);
        SetArg<115, T115>::set(kernel_, t115);
        SetArg<116, T116>::set(kernel_, t116);
        SetArg<117, T117>::set(kernel_, t117);
        SetArg<118, T118>::set(kernel_, t118);
        SetArg<119, T119>::set(kernel_, t119);
        SetArg<120, T120>::set(kernel_, t120);
        SetArg<121, T121>::set(kernel_, t121);
        SetArg<122, T122>::set(kernel_, t122);
        SetArg<123, T123>::set(kernel_, t123);
        SetArg<124, T124>::set(kernel_, t124);
        SetArg<125, T125>::set(kernel_, t125);
        SetArg<126, T126>::set(kernel_, t126);
        SetArg<127, T127>::set(kernel_, t127);

		// Move any arguments that need moving
        MoveToDevice<T0>::move(args.queue_, t0);
        MoveToDevice<T1>::move(args.queue_, t1);
        MoveToDevice<T2>::move(args.queue_, t2);
        MoveToDevice<T3>::move(args.queue_, t3);
        MoveToDevice<T4>::move(args.queue_, t4);
        MoveToDevice<T5>::move(args.queue_, t5);
        MoveToDevice<T6>::move(args.queue_, t6);
        MoveToDevice<T7>::move(args.queue_, t7);
        MoveToDevice<T8>::move(args.queue_, t8);
        MoveToDevice<T9>::move(args.queue_, t9);
        MoveToDevice<T10>::move(args.queue_, t10);
        MoveToDevice<T11>::move(args.queue_, t11);
        MoveToDevice<T12>::move(args.queue_, t12);
        MoveToDevice<T13>::move(args.queue_, t13);
        MoveToDevice<T14>::move(args.queue_, t14);
        MoveToDevice<T15>::move(args.queue_, t15);
        MoveToDevice<T16>::move(args.queue_, t16);
        MoveToDevice<T17>::move(args.queue_, t17);
        MoveToDevice<T18>::move(args.queue_, t18);
        MoveToDevice<T19>::move(args.queue_, t19);
        MoveToDevice<T20>::move(args.queue_, t20);
        MoveToDevice<T21>::move(args.queue_, t21);
        MoveToDevice<T22>::move(args.queue_, t22);
        MoveToDevice<T23>::move(args.queue_, t23);
        MoveToDevice<T24>::move(args.queue_, t24);
        MoveToDevice<T25>::move(args.queue_, t25);
        MoveToDevice<T26>::move(args.queue_, t26);
        MoveToDevice<T27>::move(args.queue_, t27);
        MoveToDevice<T28>::move(args.queue_, t28);
        MoveToDevice<T29>::move(args.queue_, t29);
        MoveToDevice<T30>::move(args.queue_, t30);
        MoveToDevice<T31>::move(args.queue_, t31);
        MoveToDevice<T32>::move(args.queue_, t32);
        MoveToDevice<T33>::move(args.queue_, t33);
        MoveToDevice<T34>::move(args.queue_, t34);
        MoveToDevice<T35>::move(args.queue_, t35);
        MoveToDevice<T36>::move(args.queue_, t36);
        MoveToDevice<T37>::move(args.queue_, t37);
        MoveToDevice<T38>::move(args.queue_, t38);
        MoveToDevice<T39>::move(args.queue_, t39);
        MoveToDevice<T40>::move(args.queue_, t40);
        MoveToDevice<T41>::move(args.queue_, t41);
        MoveToDevice<T42>::move(args.queue_, t42);
        MoveToDevice<T43>::move(args.queue_, t43);
        MoveToDevice<T44>::move(args.queue_, t44);
        MoveToDevice<T45>::move(args.queue_, t45);
        MoveToDevice<T46>::move(args.queue_, t46);
        MoveToDevice<T47>::move(args.queue_, t47);
        MoveToDevice<T48>::move(args.queue_, t48);
        MoveToDevice<T49>::move(args.queue_, t49);
        MoveToDevice<T50>::move(args.queue_, t50);
        MoveToDevice<T51>::move(args.queue_, t51);
        MoveToDevice<T52>::move(args.queue_, t52);
        MoveToDevice<T53>::move(args.queue_, t53);
        MoveToDevice<T54>::move(args.queue_, t54);
        MoveToDevice<T55>::move(args.queue_, t55);
        MoveToDevice<T56>::move(args.queue_, t56);
        MoveToDevice<T57>::move(args.queue_, t57);
        MoveToDevice<T58>::move(args.queue_, t58);
        MoveToDevice<T59>::move(args.queue_, t59);
        MoveToDevice<T60>::move(args.queue_, t60);
        MoveToDevice<T61>::move(args.queue_, t61);
        MoveToDevice<T62>::move(args.queue_, t62);
        MoveToDevice<T63>::move(args.queue_, t63);
        MoveToDevice<T64>::move(args.queue_, t64);
        MoveToDevice<T65>::move(args.queue_, t65);
        MoveToDevice<T66>::move(args.queue_, t66);
        MoveToDevice<T67>::move(args.queue_, t67);
        MoveToDevice<T68>::move(args.queue_, t68);
        MoveToDevice<T69>::move(args.queue_, t69);
        MoveToDevice<T70>::move(args.queue_, t70);
        MoveToDevice<T71>::move(args.queue_, t71);
        MoveToDevice<T72>::move(args.queue_, t72);
        MoveToDevice<T73>::move(args.queue_, t73);
        MoveToDevice<T74>::move(args.queue_, t74);
        MoveToDevice<T75>::move(args.queue_, t75);
        MoveToDevice<T76>::move(args.queue_, t76);
        MoveToDevice<T77>::move(args.queue_, t77);
        MoveToDevice<T78>::move(args.queue_, t78);
        MoveToDevice<T79>::move(args.queue_, t79);
        MoveToDevice<T80>::move(args.queue_, t80);
        MoveToDevice<T81>::move(args.queue_, t81);
        MoveToDevice<T82>::move(args.queue_, t82);
        MoveToDevice<T83>::move(args.queue_, t83);
        MoveToDevice<T84>::move(args.queue_, t84);
        MoveToDevice<T85>::move(args.queue_, t85);
        MoveToDevice<T86>::move(args.queue_, t86);
        MoveToDevice<T87>::move(args.queue_, t87);
        MoveToDevice<T88>::move(args.queue_, t88);
        MoveToDevice<T89>::move(args.queue_, t89);
        MoveToDevice<T90>::move(args.queue_, t90);
        MoveToDevice<T91>::move(args.queue_, t91);
        MoveToDevice<T92>::move(args.queue_, t92);
        MoveToDevice<T93>::move(args.queue_, t93);
        MoveToDevice<T94>::move(args.queue_, t94);
        MoveToDevice<T95>::move(args.queue_, t95);
        MoveToDevice<T96>::move(args.queue_, t96);
        MoveToDevice<T97>::move(args.queue_, t97);
        MoveToDevice<T98>::move(args.queue_, t98);
        MoveToDevice<T99>::move(args.queue_, t99);
        MoveToDevice<T100>::move(args.queue_, t100);
        MoveToDevice<T101>::move(args.queue_, t101);
        MoveToDevice<T102>::move(args.queue_, t102);
        MoveToDevice<T103>::move(args.queue_, t103);
        MoveToDevice<T104>::move(args.queue_, t104);
        MoveToDevice<T105>::move(args.queue_, t105);
        MoveToDevice<T106>::move(args.queue_, t106);
        MoveToDevice<T107>::move(args.queue_, t107);
        MoveToDevice<T108>::move(args.queue_, t108);
        MoveToDevice<T109>::move(args.queue_, t109);
        MoveToDevice<T110>::move(args.queue_, t110);
        MoveToDevice<T111>::move(args.queue_, t111);
        MoveToDevice<T112>::move(args.queue_, t112);
        MoveToDevice<T113>::move(args.queue_, t113);
        MoveToDevice<T114>::move(args.queue_, t114);
        MoveToDevice<T115>::move(args.queue_, t115);
        MoveToDevice<T116>::move(args.queue_, t116);
        MoveToDevice<T117>::move(args.queue_, t117);
        MoveToDevice<T118>::move(args.queue_, t118);
        MoveToDevice<T119>::move(args.queue_, t119);
        MoveToDevice<T120>::move(args.queue_, t120);
        MoveToDevice<T121>::move(args.queue_, t121);
        MoveToDevice<T122>::move(args.queue_, t122);
        MoveToDevice<T123>::move(args.queue_, t123);
        MoveToDevice<T124>::move(args.queue_, t124);
        MoveToDevice<T125>::move(args.queue_, t125);
        MoveToDevice<T126>::move(args.queue_, t126);
        MoveToDevice<T127>::move(args.queue_, t127);

		VECTOR_CLASS<Event> events(&waitEvent, &waitEvent);

        args.queue_.enqueueNDRangeKernel(
            kernel_,
            args.offset_,
            args.global_,
            args.local_,
            &events,
            &event);

		// Move any arguments that need moving
       MoveFromDevice<T0>::move(args.queue_, t0);
        MoveFromDevice<T1>::move(args.queue_, t1);
        MoveFromDevice<T2>::move(args.queue_, t2);
        MoveFromDevice<T3>::move(args.queue_, t3);
        MoveFromDevice<T4>::move(args.queue_, t4);
        MoveFromDevice<T5>::move(args.queue_, t5);
        MoveFromDevice<T6>::move(args.queue_, t6);
        MoveFromDevice<T7>::move(args.queue_, t7);
        MoveFromDevice<T8>::move(args.queue_, t8);
        MoveFromDevice<T9>::move(args.queue_, t9);
        MoveFromDevice<T10>::move(args.queue_, t10);
        MoveFromDevice<T11>::move(args.queue_, t11);
        MoveFromDevice<T12>::move(args.queue_, t12);
        MoveFromDevice<T13>::move(args.queue_, t13);
        MoveFromDevice<T14>::move(args.queue_, t14);
        MoveFromDevice<T15>::move(args.queue_, t15);
        MoveFromDevice<T16>::move(args.queue_, t16);
        MoveFromDevice<T17>::move(args.queue_, t17);
        MoveFromDevice<T18>::move(args.queue_, t18);
        MoveFromDevice<T19>::move(args.queue_, t19);
        MoveFromDevice<T20>::move(args.queue_, t20);
        MoveFromDevice<T21>::move(args.queue_, t21);
        MoveFromDevice<T22>::move(args.queue_, t22);
        MoveFromDevice<T23>::move(args.queue_, t23);
        MoveFromDevice<T24>::move(args.queue_, t24);
        MoveFromDevice<T25>::move(args.queue_, t25);
        MoveFromDevice<T26>::move(args.queue_, t26);
        MoveFromDevice<T27>::move(args.queue_, t27);
        MoveFromDevice<T28>::move(args.queue_, t28);
        MoveFromDevice<T29>::move(args.queue_, t29);
        MoveFromDevice<T30>::move(args.queue_, t30);
        MoveFromDevice<T31>::move(args.queue_, t31);
        MoveFromDevice<T32>::move(args.queue_, t32);
        MoveFromDevice<T33>::move(args.queue_, t33);
        MoveFromDevice<T34>::move(args.queue_, t34);
        MoveFromDevice<T35>::move(args.queue_, t35);
        MoveFromDevice<T36>::move(args.queue_, t36);
        MoveFromDevice<T37>::move(args.queue_, t37);
        MoveFromDevice<T38>::move(args.queue_, t38);
        MoveFromDevice<T39>::move(args.queue_, t39);
        MoveFromDevice<T40>::move(args.queue_, t40);
        MoveFromDevice<T41>::move(args.queue_, t41);
        MoveFromDevice<T42>::move(args.queue_, t42);
        MoveFromDevice<T43>::move(args.queue_, t43);
        MoveFromDevice<T44>::move(args.queue_, t44);
        MoveFromDevice<T45>::move(args.queue_, t45);
        MoveFromDevice<T46>::move(args.queue_, t46);
        MoveFromDevice<T47>::move(args.queue_, t47);
        MoveFromDevice<T48>::move(args.queue_, t48);
        MoveFromDevice<T49>::move(args.queue_, t49);
        MoveFromDevice<T50>::move(args.queue_, t50);
        MoveFromDevice<T51>::move(args.queue_, t51);
        MoveFromDevice<T52>::move(args.queue_, t52);
        MoveFromDevice<T53>::move(args.queue_, t53);
        MoveFromDevice<T54>::move(args.queue_, t54);
        MoveFromDevice<T55>::move(args.queue_, t55);
        MoveFromDevice<T56>::move(args.queue_, t56);
        MoveFromDevice<T57>::move(args.queue_, t57);
        MoveFromDevice<T58>::move(args.queue_, t58);
        MoveFromDevice<T59>::move(args.queue_, t59);
        MoveFromDevice<T60>::move(args.queue_, t60);
        MoveFromDevice<T61>::move(args.queue_, t61);
        MoveFromDevice<T62>::move(args.queue_, t62);
        MoveFromDevice<T63>::move(args.queue_, t63);
        MoveFromDevice<T64>::move(args.queue_, t64);
        MoveFromDevice<T65>::move(args.queue_, t65);
        MoveFromDevice<T66>::move(args.queue_, t66);
        MoveFromDevice<T67>::move(args.queue_, t67);
        MoveFromDevice<T68>::move(args.queue_, t68);
        MoveFromDevice<T69>::move(args.queue_, t69);
        MoveFromDevice<T70>::move(args.queue_, t70);
        MoveFromDevice<T71>::move(args.queue_, t71);
        MoveFromDevice<T72>::move(args.queue_, t72);
        MoveFromDevice<T73>::move(args.queue_, t73);
        MoveFromDevice<T74>::move(args.queue_, t74);
        MoveFromDevice<T75>::move(args.queue_, t75);
        MoveFromDevice<T76>::move(args.queue_, t76);
        MoveFromDevice<T77>::move(args.queue_, t77);
        MoveFromDevice<T78>::move(args.queue_, t78);
        MoveFromDevice<T79>::move(args.queue_, t79);
        MoveFromDevice<T80>::move(args.queue_, t80);
        MoveFromDevice<T81>::move(args.queue_, t81);
        MoveFromDevice<T82>::move(args.queue_, t82);
        MoveFromDevice<T83>::move(args.queue_, t83);
        MoveFromDevice<T84>::move(args.queue_, t84);
        MoveFromDevice<T85>::move(args.queue_, t85);
        MoveFromDevice<T86>::move(args.queue_, t86);
        MoveFromDevice<T87>::move(args.queue_, t87);
        MoveFromDevice<T88>::move(args.queue_, t88);
        MoveFromDevice<T89>::move(args.queue_, t89);
        MoveFromDevice<T90>::move(args.queue_, t90);
        MoveFromDevice<T91>::move(args.queue_, t91);
        MoveFromDevice<T92>::move(args.queue_, t92);
        MoveFromDevice<T93>::move(args.queue_, t93);
        MoveFromDevice<T94>::move(args.queue_, t94);
        MoveFromDevice<T95>::move(args.queue_, t95);
        MoveFromDevice<T96>::move(args.queue_, t96);
        MoveFromDevice<T97>::move(args.queue_, t97);
        MoveFromDevice<T98>::move(args.queue_, t98);
        MoveFromDevice<T99>::move(args.queue_, t99);
        MoveFromDevice<T100>::move(args.queue_, t100);
        MoveFromDevice<T101>::move(args.queue_, t101);
        MoveFromDevice<T102>::move(args.queue_, t102);
        MoveFromDevice<T103>::move(args.queue_, t103);
        MoveFromDevice<T104>::move(args.queue_, t104);
        MoveFromDevice<T105>::move(args.queue_, t105);
        MoveFromDevice<T106>::move(args.queue_, t106);
        MoveFromDevice<T107>::move(args.queue_, t107);
        MoveFromDevice<T108>::move(args.queue_, t108);
        MoveFromDevice<T109>::move(args.queue_, t109);
        MoveFromDevice<T110>::move(args.queue_, t110);
        MoveFromDevice<T111>::move(args.queue_, t111);
        MoveFromDevice<T112>::move(args.queue_, t112);
        MoveFromDevice<T113>::move(args.queue_, t113);
        MoveFromDevice<T114>::move(args.queue_, t114);
        MoveFromDevice<T115>::move(args.queue_, t115);
        MoveFromDevice<T116>::move(args.queue_, t116);
        MoveFromDevice<T117>::move(args.queue_, t117);
        MoveFromDevice<T118>::move(args.queue_, t118);
        MoveFromDevice<T119>::move(args.queue_, t119);
        MoveFromDevice<T120>::move(args.queue_, t120);
        MoveFromDevice<T121>::move(args.queue_, t121);
        MoveFromDevice<T122>::move(args.queue_, t122);
        MoveFromDevice<T123>::move(args.queue_, t123);
        MoveFromDevice<T124>::move(args.queue_, t124);
        MoveFromDevice<T125>::move(args.queue_, t125);
        MoveFromDevice<T126>::move(args.queue_, t126);
        MoveFromDevice<T127>::move(args.queue_, t127);
        return event;
    }

    Event operator() (
        const EnqueueArgs& args,
        const VECTOR_CLASS<Event>& waitEvents,
        T0 t0,
        T1 t1 = NullType(),
        T2 t2 = NullType(),
        T3 t3 = NullType(),
        T4 t4 = NullType(),
        T5 t5 = NullType(),
        T6 t6 = NullType(),
        T7 t7 = NullType(),
        T8 t8 = NullType(),
        T9 t9 = NullType(),
        T10 t10 = NullType(),
        T11 t11 = NullType(),
        T12 t12 = NullType(),
        T13 t13 = NullType(),
        T14 t14 = NullType(),
        T15 t15 = NullType(),
        T16 t16 = NullType(),
        T17 t17 = NullType(),
        T18 t18 = NullType(),
        T19 t19 = NullType(),
        T20 t20 = NullType(),
        T21 t21 = NullType(),
        T22 t22 = NullType(),
        T23 t23 = NullType(),
        T24 t24 = NullType(),
        T25 t25 = NullType(),
        T26 t26 = NullType(),
        T27 t27 = NullType(),
        T28 t28 = NullType(),
        T29 t29 = NullType(),
        T30 t30 = NullType(),
        T31 t31 = NullType(),
        T32 t32 = NullType(),
        T33 t33 = NullType(),
        T34 t34 = NullType(),
        T35 t35 = NullType(),
        T36 t36 = NullType(),
        T37 t37 = NullType(),
        T38 t38 = NullType(),
        T39 t39 = NullType(),
        T40 t40 = NullType(),
        T41 t41 = NullType(),
        T42 t42 = NullType(),
        T43 t43 = NullType(),
        T44 t44 = NullType(),
        T45 t45 = NullType(),
        T46 t46 = NullType(),
        T47 t47 = NullType(),
        T48 t48 = NullType(),
        T49 t49 = NullType(),
        T50 t50 = NullType(),
        T51 t51 = NullType(),
        T52 t52 = NullType(),
        T53 t53 = NullType(),
        T54 t54 = NullType(),
        T55 t55 = NullType(),
        T56 t56 = NullType(),
        T57 t57 = NullType(),
        T58 t58 = NullType(),
        T59 t59 = NullType(),
        T60 t60 = NullType(),
        T61 t61 = NullType(),
        T62 t62 = NullType(),
        T63 t63 = NullType(),
        T64 t64 = NullType(),
        T65 t65 = NullType(),
        T66 t66 = NullType(),
        T67 t67 = NullType(),
        T68 t68 = NullType(),
        T69 t69 = NullType(),
        T70 t70 = NullType(),
        T71 t71 = NullType(),
        T72 t72 = NullType(),
        T73 t73 = NullType(),
        T74 t74 = NullType(),
        T75 t75 = NullType(),
        T76 t76 = NullType(),
        T77 t77 = NullType(),
        T78 t78 = NullType(),
        T79 t79 = NullType(),
        T80 t80 = NullType(),
        T81 t81 = NullType(),
        T82 t82 = NullType(),
        T83 t83 = NullType(),
        T84 t84 = NullType(),
        T85 t85 = NullType(),
        T86 t86 = NullType(),
        T87 t87 = NullType(),
        T88 t88 = NullType(),
        T89 t89 = NullType(),
        T90 t90 = NullType(),
        T91 t91 = NullType(),
        T92 t92 = NullType(),
        T93 t93 = NullType(),
        T94 t94 = NullType(),
        T95 t95 = NullType(),
        T96 t96 = NullType(),
        T97 t97 = NullType(),
        T98 t98 = NullType(),
        T99 t99 = NullType(),
        T100 t100 = NullType(),
        T101 t101 = NullType(),
        T102 t102 = NullType(),
        T103 t103 = NullType(),
        T104 t104 = NullType(),
        T105 t105 = NullType(),
        T106 t106 = NullType(),
        T107 t107 = NullType(),
        T108 t108 = NullType(),
        T109 t109 = NullType(),
        T110 t110 = NullType(),
        T111 t111 = NullType(),
        T112 t112 = NullType(),
        T113 t113 = NullType(),
        T114 t114 = NullType(),
        T115 t115 = NullType(),
        T116 t116 = NullType(),
        T117 t117 = NullType(),
        T118 t118 = NullType(),
        T119 t119 = NullType(),
        T120 t120 = NullType(),
        T121 t121 = NullType(),
        T122 t122 = NullType(),
        T123 t123 = NullType(),
        T124 t124 = NullType(),
        T125 t125 = NullType(),
        T126 t126 = NullType(),
        T127 t127 = NullType())
    {
        Event event;
        SetArg<0, T0>::set(kernel_, t0);
        SetArg<1, T1>::set(kernel_, t1);
        SetArg<2, T2>::set(kernel_, t2);
        SetArg<3, T3>::set(kernel_, t3);
        SetArg<4, T4>::set(kernel_, t4);
        SetArg<5, T5>::set(kernel_, t5);
        SetArg<6, T6>::set(kernel_, t6);
        SetArg<7, T7>::set(kernel_, t7);
        SetArg<8, T8>::set(kernel_, t8);
        SetArg<9, T9>::set(kernel_, t9);
        SetArg<10, T10>::set(kernel_, t10);
        SetArg<11, T11>::set(kernel_, t11);
        SetArg<12, T12>::set(kernel_, t12);
        SetArg<13, T13>::set(kernel_, t13);
        SetArg<14, T14>::set(kernel_, t14);
        SetArg<15, T15>::set(kernel_, t15);
        SetArg<16, T16>::set(kernel_, t16);
        SetArg<17, T17>::set(kernel_, t17);
        SetArg<18, T18>::set(kernel_, t18);
        SetArg<19, T19>::set(kernel_, t19);
        SetArg<20, T20>::set(kernel_, t20);
        SetArg<21, T21>::set(kernel_, t21);
        SetArg<22, T22>::set(kernel_, t22);
        SetArg<23, T23>::set(kernel_, t23);
        SetArg<24, T24>::set(kernel_, t24);
        SetArg<25, T25>::set(kernel_, t25);
        SetArg<26, T26>::set(kernel_, t26);
        SetArg<27, T27>::set(kernel_, t27);
        SetArg<28, T28>::set(kernel_, t28);
        SetArg<29, T29>::set(kernel_, t29);
        SetArg<30, T30>::set(kernel_, t30);
        SetArg<31, T31>::set(kernel_, t31);
        SetArg<32, T32>::set(kernel_, t32);
        SetArg<33, T33>::set(kernel_, t33);
        SetArg<34, T34>::set(kernel_, t34);
        SetArg<35, T35>::set(kernel_, t35);
        SetArg<36, T36>::set(kernel_, t36);
        SetArg<37, T37>::set(kernel_, t37);
        SetArg<38, T38>::set(kernel_, t38);
        SetArg<39, T39>::set(kernel_, t39);
        SetArg<40, T40>::set(kernel_, t40);
        SetArg<41, T41>::set(kernel_, t41);
        SetArg<42, T42>::set(kernel_, t42);
        SetArg<43, T43>::set(kernel_, t43);
        SetArg<44, T44>::set(kernel_, t44);
        SetArg<45, T45>::set(kernel_, t45);
        SetArg<46, T46>::set(kernel_, t46);
        SetArg<47, T47>::set(kernel_, t47);
        SetArg<48, T48>::set(kernel_, t48);
        SetArg<49, T49>::set(kernel_, t49);
        SetArg<50, T50>::set(kernel_, t50);
        SetArg<51, T51>::set(kernel_, t51);
        SetArg<52, T52>::set(kernel_, t52);
        SetArg<53, T53>::set(kernel_, t53);
        SetArg<54, T54>::set(kernel_, t54);
        SetArg<55, T55>::set(kernel_, t55);
        SetArg<56, T56>::set(kernel_, t56);
        SetArg<57, T57>::set(kernel_, t57);
        SetArg<58, T58>::set(kernel_, t58);
        SetArg<59, T59>::set(kernel_, t59);
        SetArg<60, T60>::set(kernel_, t60);
        SetArg<61, T61>::set(kernel_, t61);
        SetArg<62, T62>::set(kernel_, t62);
        SetArg<63, T63>::set(kernel_, t63);
        SetArg<64, T64>::set(kernel_, t64);
        SetArg<65, T65>::set(kernel_, t65);
        SetArg<66, T66>::set(kernel_, t66);
        SetArg<67, T67>::set(kernel_, t67);
        SetArg<68, T68>::set(kernel_, t68);
        SetArg<69, T69>::set(kernel_, t69);
        SetArg<70, T70>::set(kernel_, t70);
        SetArg<71, T71>::set(kernel_, t71);
        SetArg<72, T72>::set(kernel_, t72);
        SetArg<73, T73>::set(kernel_, t73);
        SetArg<74, T74>::set(kernel_, t74);
        SetArg<75, T75>::set(kernel_, t75);
        SetArg<76, T76>::set(kernel_, t76);
        SetArg<77, T77>::set(kernel_, t77);
        SetArg<78, T78>::set(kernel_, t78);
        SetArg<79, T79>::set(kernel_, t79);
        SetArg<80, T80>::set(kernel_, t80);
        SetArg<81, T81>::set(kernel_, t81);
        SetArg<82, T82>::set(kernel_, t82);
        SetArg<83, T83>::set(kernel_, t83);
        SetArg<84, T84>::set(kernel_, t84);
        SetArg<85, T85>::set(kernel_, t85);
        SetArg<86, T86>::set(kernel_, t86);
        SetArg<87, T87>::set(kernel_, t87);
        SetArg<88, T88>::set(kernel_, t88);
        SetArg<89, T89>::set(kernel_, t89);
        SetArg<90, T90>::set(kernel_, t90);
        SetArg<91, T91>::set(kernel_, t91);
        SetArg<92, T92>::set(kernel_, t92);
        SetArg<93, T93>::set(kernel_, t93);
        SetArg<94, T94>::set(kernel_, t94);
        SetArg<95, T95>::set(kernel_, t95);
        SetArg<96, T96>::set(kernel_, t96);
        SetArg<97, T97>::set(kernel_, t97);
        SetArg<98, T98>::set(kernel_, t98);
        SetArg<99, T99>::set(kernel_, t99);
        SetArg<100, T100>::set(kernel_, t100);
        SetArg<101, T101>::set(kernel_, t101);
        SetArg<102, T102>::set(kernel_, t102);
        SetArg<103, T103>::set(kernel_, t103);
        SetArg<104, T104>::set(kernel_, t104);
        SetArg<105, T105>::set(kernel_, t105);
        SetArg<106, T106>::set(kernel_, t106);
        SetArg<107, T107>::set(kernel_, t107);
        SetArg<108, T108>::set(kernel_, t108);
        SetArg<109, T109>::set(kernel_, t109);
        SetArg<110, T110>::set(kernel_, t110);
        SetArg<111, T111>::set(kernel_, t111);
        SetArg<112, T112>::set(kernel_, t112);
        SetArg<113, T113>::set(kernel_, t113);
        SetArg<114, T114>::set(kernel_, t114);
        SetArg<115, T115>::set(kernel_, t115);
        SetArg<116, T116>::set(kernel_, t116);
        SetArg<117, T117>::set(kernel_, t117);
        SetArg<118, T118>::set(kernel_, t118);
        SetArg<119, T119>::set(kernel_, t119);
        SetArg<120, T120>::set(kernel_, t120);
        SetArg<121, T121>::set(kernel_, t121);
        SetArg<122, T122>::set(kernel_, t122);
        SetArg<123, T123>::set(kernel_, t123);
        SetArg<124, T124>::set(kernel_, t124);
        SetArg<125, T125>::set(kernel_, t125);
        SetArg<126, T126>::set(kernel_, t126);
        SetArg<127, T127>::set(kernel_, t127);

		// Move any arguments that need moving
        MoveToDevice<T0>::move(args.queue_, t0);
        MoveToDevice<T1>::move(args.queue_, t1);
        MoveToDevice<T2>::move(args.queue_, t2);
        MoveToDevice<T3>::move(args.queue_, t3);
        MoveToDevice<T4>::move(args.queue_, t4);
        MoveToDevice<T5>::move(args.queue_, t5);
        MoveToDevice<T6>::move(args.queue_, t6);
        MoveToDevice<T7>::move(args.queue_, t7);
        MoveToDevice<T8>::move(args.queue_, t8);
        MoveToDevice<T9>::move(args.queue_, t9);
        MoveToDevice<T10>::move(args.queue_, t10);
        MoveToDevice<T11>::move(args.queue_, t11);
        MoveToDevice<T12>::move(args.queue_, t12);
        MoveToDevice<T13>::move(args.queue_, t13);
        MoveToDevice<T14>::move(args.queue_, t14);
        MoveToDevice<T15>::move(args.queue_, t15);
        MoveToDevice<T16>::move(args.queue_, t16);
        MoveToDevice<T17>::move(args.queue_, t17);
        MoveToDevice<T18>::move(args.queue_, t18);
        MoveToDevice<T19>::move(args.queue_, t19);
        MoveToDevice<T20>::move(args.queue_, t20);
        MoveToDevice<T21>::move(args.queue_, t21);
        MoveToDevice<T22>::move(args.queue_, t22);
        MoveToDevice<T23>::move(args.queue_, t23);
        MoveToDevice<T24>::move(args.queue_, t24);
        MoveToDevice<T25>::move(args.queue_, t25);
        MoveToDevice<T26>::move(args.queue_, t26);
        MoveToDevice<T27>::move(args.queue_, t27);
        MoveToDevice<T28>::move(args.queue_, t28);
        MoveToDevice<T29>::move(args.queue_, t29);
        MoveToDevice<T30>::move(args.queue_, t30);
        MoveToDevice<T31>::move(args.queue_, t31);
        MoveToDevice<T32>::move(args.queue_, t32);
        MoveToDevice<T33>::move(args.queue_, t33);
        MoveToDevice<T34>::move(args.queue_, t34);
        MoveToDevice<T35>::move(args.queue_, t35);
        MoveToDevice<T36>::move(args.queue_, t36);
        MoveToDevice<T37>::move(args.queue_, t37);
        MoveToDevice<T38>::move(args.queue_, t38);
        MoveToDevice<T39>::move(args.queue_, t39);
        MoveToDevice<T40>::move(args.queue_, t40);
        MoveToDevice<T41>::move(args.queue_, t41);
        MoveToDevice<T42>::move(args.queue_, t42);
        MoveToDevice<T43>::move(args.queue_, t43);
        MoveToDevice<T44>::move(args.queue_, t44);
        MoveToDevice<T45>::move(args.queue_, t45);
        MoveToDevice<T46>::move(args.queue_, t46);
        MoveToDevice<T47>::move(args.queue_, t47);
        MoveToDevice<T48>::move(args.queue_, t48);
        MoveToDevice<T49>::move(args.queue_, t49);
        MoveToDevice<T50>::move(args.queue_, t50);
        MoveToDevice<T51>::move(args.queue_, t51);
        MoveToDevice<T52>::move(args.queue_, t52);
        MoveToDevice<T53>::move(args.queue_, t53);
        MoveToDevice<T54>::move(args.queue_, t54);
        MoveToDevice<T55>::move(args.queue_, t55);
        MoveToDevice<T56>::move(args.queue_, t56);
        MoveToDevice<T57>::move(args.queue_, t57);
        MoveToDevice<T58>::move(args.queue_, t58);
        MoveToDevice<T59>::move(args.queue_, t59);
        MoveToDevice<T60>::move(args.queue_, t60);
        MoveToDevice<T61>::move(args.queue_, t61);
        MoveToDevice<T62>::move(args.queue_, t62);
        MoveToDevice<T63>::move(args.queue_, t63);
        MoveToDevice<T64>::move(args.queue_, t64);
        MoveToDevice<T65>::move(args.queue_, t65);
        MoveToDevice<T66>::move(args.queue_, t66);
        MoveToDevice<T67>::move(args.queue_, t67);
        MoveToDevice<T68>::move(args.queue_, t68);
        MoveToDevice<T69>::move(args.queue_, t69);
        MoveToDevice<T70>::move(args.queue_, t70);
        MoveToDevice<T71>::move(args.queue_, t71);
        MoveToDevice<T72>::move(args.queue_, t72);
        MoveToDevice<T73>::move(args.queue_, t73);
        MoveToDevice<T74>::move(args.queue_, t74);
        MoveToDevice<T75>::move(args.queue_, t75);
        MoveToDevice<T76>::move(args.queue_, t76);
        MoveToDevice<T77>::move(args.queue_, t77);
        MoveToDevice<T78>::move(args.queue_, t78);
        MoveToDevice<T79>::move(args.queue_, t79);
        MoveToDevice<T80>::move(args.queue_, t80);
        MoveToDevice<T81>::move(args.queue_, t81);
        MoveToDevice<T82>::move(args.queue_, t82);
        MoveToDevice<T83>::move(args.queue_, t83);
        MoveToDevice<T84>::move(args.queue_, t84);
        MoveToDevice<T85>::move(args.queue_, t85);
        MoveToDevice<T86>::move(args.queue_, t86);
        MoveToDevice<T87>::move(args.queue_, t87);
        MoveToDevice<T88>::move(args.queue_, t88);
        MoveToDevice<T89>::move(args.queue_, t89);
        MoveToDevice<T90>::move(args.queue_, t90);
        MoveToDevice<T91>::move(args.queue_, t91);
        MoveToDevice<T92>::move(args.queue_, t92);
        MoveToDevice<T93>::move(args.queue_, t93);
        MoveToDevice<T94>::move(args.queue_, t94);
        MoveToDevice<T95>::move(args.queue_, t95);
        MoveToDevice<T96>::move(args.queue_, t96);
        MoveToDevice<T97>::move(args.queue_, t97);
        MoveToDevice<T98>::move(args.queue_, t98);
        MoveToDevice<T99>::move(args.queue_, t99);
        MoveToDevice<T100>::move(args.queue_, t100);
        MoveToDevice<T101>::move(args.queue_, t101);
        MoveToDevice<T102>::move(args.queue_, t102);
        MoveToDevice<T103>::move(args.queue_, t103);
        MoveToDevice<T104>::move(args.queue_, t104);
        MoveToDevice<T105>::move(args.queue_, t105);
        MoveToDevice<T106>::move(args.queue_, t106);
        MoveToDevice<T107>::move(args.queue_, t107);
        MoveToDevice<T108>::move(args.queue_, t108);
        MoveToDevice<T109>::move(args.queue_, t109);
        MoveToDevice<T110>::move(args.queue_, t110);
        MoveToDevice<T111>::move(args.queue_, t111);
        MoveToDevice<T112>::move(args.queue_, t112);
        MoveToDevice<T113>::move(args.queue_, t113);
        MoveToDevice<T114>::move(args.queue_, t114);
        MoveToDevice<T115>::move(args.queue_, t115);
        MoveToDevice<T116>::move(args.queue_, t116);
        MoveToDevice<T117>::move(args.queue_, t117);
        MoveToDevice<T118>::move(args.queue_, t118);
        MoveToDevice<T119>::move(args.queue_, t119);
        MoveToDevice<T120>::move(args.queue_, t120);
        MoveToDevice<T121>::move(args.queue_, t121);
        MoveToDevice<T122>::move(args.queue_, t122);
        MoveToDevice<T123>::move(args.queue_, t123);
        MoveToDevice<T124>::move(args.queue_, t124);
        MoveToDevice<T125>::move(args.queue_, t125);
        MoveToDevice<T126>::move(args.queue_, t126);
        MoveToDevice<T127>::move(args.queue_, t127);

        args.queue_.enqueueNDRangeKernel(
            kernel_,
            args.offset_,
            args.global_,
            args.local_,
            &waitEvents,
            &event);

		// Move any arguments that need moving
       MoveFromDevice<T0>::move(args.queue_, t0);
        MoveFromDevice<T1>::move(args.queue_, t1);
        MoveFromDevice<T2>::move(args.queue_, t2);
        MoveFromDevice<T3>::move(args.queue_, t3);
        MoveFromDevice<T4>::move(args.queue_, t4);
        MoveFromDevice<T5>::move(args.queue_, t5);
        MoveFromDevice<T6>::move(args.queue_, t6);
        MoveFromDevice<T7>::move(args.queue_, t7);
        MoveFromDevice<T8>::move(args.queue_, t8);
        MoveFromDevice<T9>::move(args.queue_, t9);
        MoveFromDevice<T10>::move(args.queue_, t10);
        MoveFromDevice<T11>::move(args.queue_, t11);
        MoveFromDevice<T12>::move(args.queue_, t12);
        MoveFromDevice<T13>::move(args.queue_, t13);
        MoveFromDevice<T14>::move(args.queue_, t14);
        MoveFromDevice<T15>::move(args.queue_, t15);
        MoveFromDevice<T16>::move(args.queue_, t16);
        MoveFromDevice<T17>::move(args.queue_, t17);
        MoveFromDevice<T18>::move(args.queue_, t18);
        MoveFromDevice<T19>::move(args.queue_, t19);
        MoveFromDevice<T20>::move(args.queue_, t20);
        MoveFromDevice<T21>::move(args.queue_, t21);
        MoveFromDevice<T22>::move(args.queue_, t22);
        MoveFromDevice<T23>::move(args.queue_, t23);
        MoveFromDevice<T24>::move(args.queue_, t24);
        MoveFromDevice<T25>::move(args.queue_, t25);
        MoveFromDevice<T26>::move(args.queue_, t26);
        MoveFromDevice<T27>::move(args.queue_, t27);
        MoveFromDevice<T28>::move(args.queue_, t28);
        MoveFromDevice<T29>::move(args.queue_, t29);
        MoveFromDevice<T30>::move(args.queue_, t30);
        MoveFromDevice<T31>::move(args.queue_, t31);
        MoveFromDevice<T32>::move(args.queue_, t32);
        MoveFromDevice<T33>::move(args.queue_, t33);
        MoveFromDevice<T34>::move(args.queue_, t34);
        MoveFromDevice<T35>::move(args.queue_, t35);
        MoveFromDevice<T36>::move(args.queue_, t36);
        MoveFromDevice<T37>::move(args.queue_, t37);
        MoveFromDevice<T38>::move(args.queue_, t38);
        MoveFromDevice<T39>::move(args.queue_, t39);
        MoveFromDevice<T40>::move(args.queue_, t40);
        MoveFromDevice<T41>::move(args.queue_, t41);
        MoveFromDevice<T42>::move(args.queue_, t42);
        MoveFromDevice<T43>::move(args.queue_, t43);
        MoveFromDevice<T44>::move(args.queue_, t44);
        MoveFromDevice<T45>::move(args.queue_, t45);
        MoveFromDevice<T46>::move(args.queue_, t46);
        MoveFromDevice<T47>::move(args.queue_, t47);
        MoveFromDevice<T48>::move(args.queue_, t48);
        MoveFromDevice<T49>::move(args.queue_, t49);
        MoveFromDevice<T50>::move(args.queue_, t50);
        MoveFromDevice<T51>::move(args.queue_, t51);
        MoveFromDevice<T52>::move(args.queue_, t52);
        MoveFromDevice<T53>::move(args.queue_, t53);
        MoveFromDevice<T54>::move(args.queue_, t54);
        MoveFromDevice<T55>::move(args.queue_, t55);
        MoveFromDevice<T56>::move(args.queue_, t56);
        MoveFromDevice<T57>::move(args.queue_, t57);
        MoveFromDevice<T58>::move(args.queue_, t58);
        MoveFromDevice<T59>::move(args.queue_, t59);
        MoveFromDevice<T60>::move(args.queue_, t60);
        MoveFromDevice<T61>::move(args.queue_, t61);
        MoveFromDevice<T62>::move(args.queue_, t62);
        MoveFromDevice<T63>::move(args.queue_, t63);
        MoveFromDevice<T64>::move(args.queue_, t64);
        MoveFromDevice<T65>::move(args.queue_, t65);
        MoveFromDevice<T66>::move(args.queue_, t66);
        MoveFromDevice<T67>::move(args.queue_, t67);
        MoveFromDevice<T68>::move(args.queue_, t68);
        MoveFromDevice<T69>::move(args.queue_, t69);
        MoveFromDevice<T70>::move(args.queue_, t70);
        MoveFromDevice<T71>::move(args.queue_, t71);
        MoveFromDevice<T72>::move(args.queue_, t72);
        MoveFromDevice<T73>::move(args.queue_, t73);
        MoveFromDevice<T74>::move(args.queue_, t74);
        MoveFromDevice<T75>::move(args.queue_, t75);
        MoveFromDevice<T76>::move(args.queue_, t76);
        MoveFromDevice<T77>::move(args.queue_, t77);
        MoveFromDevice<T78>::move(args.queue_, t78);
        MoveFromDevice<T79>::move(args.queue_, t79);
        MoveFromDevice<T80>::move(args.queue_, t80);
        MoveFromDevice<T81>::move(args.queue_, t81);
        MoveFromDevice<T82>::move(args.queue_, t82);
        MoveFromDevice<T83>::move(args.queue_, t83);
        MoveFromDevice<T84>::move(args.queue_, t84);
        MoveFromDevice<T85>::move(args.queue_, t85);
        MoveFromDevice<T86>::move(args.queue_, t86);
        MoveFromDevice<T87>::move(args.queue_, t87);
        MoveFromDevice<T88>::move(args.queue_, t88);
        MoveFromDevice<T89>::move(args.queue_, t89);
        MoveFromDevice<T90>::move(args.queue_, t90);
        MoveFromDevice<T91>::move(args.queue_, t91);
        MoveFromDevice<T92>::move(args.queue_, t92);
        MoveFromDevice<T93>::move(args.queue_, t93);
        MoveFromDevice<T94>::move(args.queue_, t94);
        MoveFromDevice<T95>::move(args.queue_, t95);
        MoveFromDevice<T96>::move(args.queue_, t96);
        MoveFromDevice<T97>::move(args.queue_, t97);
        MoveFromDevice<T98>::move(args.queue_, t98);
        MoveFromDevice<T99>::move(args.queue_, t99);
        MoveFromDevice<T100>::move(args.queue_, t100);
        MoveFromDevice<T101>::move(args.queue_, t101);
        MoveFromDevice<T102>::move(args.queue_, t102);
        MoveFromDevice<T103>::move(args.queue_, t103);
        MoveFromDevice<T104>::move(args.queue_, t104);
        MoveFromDevice<T105>::move(args.queue_, t105);
        MoveFromDevice<T106>::move(args.queue_, t106);
        MoveFromDevice<T107>::move(args.queue_, t107);
        MoveFromDevice<T108>::move(args.queue_, t108);
        MoveFromDevice<T109>::move(args.queue_, t109);
        MoveFromDevice<T110>::move(args.queue_, t110);
        MoveFromDevice<T111>::move(args.queue_, t111);
        MoveFromDevice<T112>::move(args.queue_, t112);
        MoveFromDevice<T113>::move(args.queue_, t113);
        MoveFromDevice<T114>::move(args.queue_, t114);
        MoveFromDevice<T115>::move(args.queue_, t115);
        MoveFromDevice<T116>::move(args.queue_, t116);
        MoveFromDevice<T117>::move(args.queue_, t117);
        MoveFromDevice<T118>::move(args.queue_, t118);
        MoveFromDevice<T119>::move(args.queue_, t119);
        MoveFromDevice<T120>::move(args.queue_, t120);
        MoveFromDevice<T121>::move(args.queue_, t121);
        MoveFromDevice<T122>::move(args.queue_, t122);
        MoveFromDevice<T123>::move(args.queue_, t123);
        MoveFromDevice<T124>::move(args.queue_, t124);
        MoveFromDevice<T125>::move(args.queue_, t125);
        MoveFromDevice<T126>::move(args.queue_, t126);
        MoveFromDevice<T127>::move(args.queue_, t127);

        return event;
    }
};

//------------------------------------------------------------------------------------------------------


template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119,
	typename T120,
	typename T121,
	typename T122,
	typename T123,
	typename T124,
	typename T125,
	typename T126,
	typename T127>
struct functionImplementation_
{
};

template<
	typename T0>
struct functionImplementation_<
	T0,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0)
	{
		return functor_(
			enqueueArgs,
			arg0);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1>
struct functionImplementation_<
	T0,
	T1,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2>
struct functionImplementation_<
	T0,
	T1,
	T2,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	T119,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118,
		T119 arg119)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118,
			arg119);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119,
	typename T120>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	T119,
	T120,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118,
		T119 arg119,
		T120 arg120)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118,
			arg119,
			arg120);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119,
	typename T120,
	typename T121>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	T119,
	T120,
	T121,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118,
		T119 arg119,
		T120 arg120,
		T121 arg121)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118,
			arg119,
			arg120,
			arg121);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119,
	typename T120,
	typename T121,
	typename T122>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	T119,
	T120,
	T121,
	T122,
	NullType,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		NullType,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118,
		T119 arg119,
		T120 arg120,
		T121 arg121,
		T122 arg122)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118,
			arg119,
			arg120,
			arg121,
			arg122);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119,
	typename T120,
	typename T121,
	typename T122,
	typename T123>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	T119,
	T120,
	T121,
	T122,
	T123,
	NullType,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		T123,
		NullType,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		T123)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118,
		T119 arg119,
		T120 arg120,
		T121 arg121,
		T122 arg122,
		T123 arg123)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118,
			arg119,
			arg120,
			arg121,
			arg122,
			arg123);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119,
	typename T120,
	typename T121,
	typename T122,
	typename T123,
	typename T124>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	T119,
	T120,
	T121,
	T122,
	T123,
	T124,
	NullType,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		T123,
		T124,
		NullType,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		T123,
		T124)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118,
		T119 arg119,
		T120 arg120,
		T121 arg121,
		T122 arg122,
		T123 arg123,
		T124 arg124)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118,
			arg119,
			arg120,
			arg121,
			arg122,
			arg123,
			arg124);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119,
	typename T120,
	typename T121,
	typename T122,
	typename T123,
	typename T124,
	typename T125>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	T119,
	T120,
	T121,
	T122,
	T123,
	T124,
	T125,
	NullType,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		T123,
		T124,
		T125,
		NullType,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		T123,
		T124,
		T125)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118,
		T119 arg119,
		T120 arg120,
		T121 arg121,
		T122 arg122,
		T123 arg123,
		T124 arg124,
		T125 arg125)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118,
			arg119,
			arg120,
			arg121,
			arg122,
			arg123,
			arg124,
			arg125);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};

template<
	typename T0,
	typename T1,
	typename T2,
	typename T3,
	typename T4,
	typename T5,
	typename T6,
	typename T7,
	typename T8,
	typename T9,
	typename T10,
	typename T11,
	typename T12,
	typename T13,
	typename T14,
	typename T15,
	typename T16,
	typename T17,
	typename T18,
	typename T19,
	typename T20,
	typename T21,
	typename T22,
	typename T23,
	typename T24,
	typename T25,
	typename T26,
	typename T27,
	typename T28,
	typename T29,
	typename T30,
	typename T31,
	typename T32,
	typename T33,
	typename T34,
	typename T35,
	typename T36,
	typename T37,
	typename T38,
	typename T39,
	typename T40,
	typename T41,
	typename T42,
	typename T43,
	typename T44,
	typename T45,
	typename T46,
	typename T47,
	typename T48,
	typename T49,
	typename T50,
	typename T51,
	typename T52,
	typename T53,
	typename T54,
	typename T55,
	typename T56,
	typename T57,
	typename T58,
	typename T59,
	typename T60,
	typename T61,
	typename T62,
	typename T63,
	typename T64,
	typename T65,
	typename T66,
	typename T67,
	typename T68,
	typename T69,
	typename T70,
	typename T71,
	typename T72,
	typename T73,
	typename T74,
	typename T75,
	typename T76,
	typename T77,
	typename T78,
	typename T79,
	typename T80,
	typename T81,
	typename T82,
	typename T83,
	typename T84,
	typename T85,
	typename T86,
	typename T87,
	typename T88,
	typename T89,
	typename T90,
	typename T91,
	typename T92,
	typename T93,
	typename T94,
	typename T95,
	typename T96,
	typename T97,
	typename T98,
	typename T99,
	typename T100,
	typename T101,
	typename T102,
	typename T103,
	typename T104,
	typename T105,
	typename T106,
	typename T107,
	typename T108,
	typename T109,
	typename T110,
	typename T111,
	typename T112,
	typename T113,
	typename T114,
	typename T115,
	typename T116,
	typename T117,
	typename T118,
	typename T119,
	typename T120,
	typename T121,
	typename T122,
	typename T123,
	typename T124,
	typename T125,
	typename T126>
struct functionImplementation_<
	T0,
	T1,
	T2,
	T3,
	T4,
	T5,
	T6,
	T7,
	T8,
	T9,
	T10,
	T11,
	T12,
	T13,
	T14,
	T15,
	T16,
	T17,
	T18,
	T19,
	T20,
	T21,
	T22,
	T23,
	T24,
	T25,
	T26,
	T27,
	T28,
	T29,
	T30,
	T31,
	T32,
	T33,
	T34,
	T35,
	T36,
	T37,
	T38,
	T39,
	T40,
	T41,
	T42,
	T43,
	T44,
	T45,
	T46,
	T47,
	T48,
	T49,
	T50,
	T51,
	T52,
	T53,
	T54,
	T55,
	T56,
	T57,
	T58,
	T59,
	T60,
	T61,
	T62,
	T63,
	T64,
	T65,
	T66,
	T67,
	T68,
	T69,
	T70,
	T71,
	T72,
	T73,
	T74,
	T75,
	T76,
	T77,
	T78,
	T79,
	T80,
	T81,
	T82,
	T83,
	T84,
	T85,
	T86,
	T87,
	T88,
	T89,
	T90,
	T91,
	T92,
	T93,
	T94,
	T95,
	T96,
	T97,
	T98,
	T99,
	T100,
	T101,
	T102,
	T103,
	T104,
	T105,
	T106,
	T107,
	T108,
	T109,
	T110,
	T111,
	T112,
	T113,
	T114,
	T115,
	T116,
	T117,
	T118,
	T119,
	T120,
	T121,
	T122,
	T123,
	T124,
	T125,
	T126,
	NullType>
{
	typedef detail::KernelFunctorGlobal<
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		T123,
		T124,
		T125,
		T126,
		NullType> FunctorType;

	FunctorType functor_;

	functionImplementation_(const FunctorType &functor) :
		functor_(functor)
	{
	}
        
	typedef std::tr1::function<Event (
		const EnqueueArgs&,
		T0,
		T1,
		T2,
		T3,
		T4,
		T5,
		T6,
		T7,
		T8,
		T9,
		T10,
		T11,
		T12,
		T13,
		T14,
		T15,
		T16,
		T17,
		T18,
		T19,
		T20,
		T21,
		T22,
		T23,
		T24,
		T25,
		T26,
		T27,
		T28,
		T29,
		T30,
		T31,
		T32,
		T33,
		T34,
		T35,
		T36,
		T37,
		T38,
		T39,
		T40,
		T41,
		T42,
		T43,
		T44,
		T45,
		T46,
		T47,
		T48,
		T49,
		T50,
		T51,
		T52,
		T53,
		T54,
		T55,
		T56,
		T57,
		T58,
		T59,
		T60,
		T61,
		T62,
		T63,
		T64,
		T65,
		T66,
		T67,
		T68,
		T69,
		T70,
		T71,
		T72,
		T73,
		T74,
		T75,
		T76,
		T77,
		T78,
		T79,
		T80,
		T81,
		T82,
		T83,
		T84,
		T85,
		T86,
		T87,
		T88,
		T89,
		T90,
		T91,
		T92,
		T93,
		T94,
		T95,
		T96,
		T97,
		T98,
		T99,
		T100,
		T101,
		T102,
		T103,
		T104,
		T105,
		T106,
		T107,
		T108,
		T109,
		T110,
		T111,
		T112,
		T113,
		T114,
		T115,
		T116,
		T117,
		T118,
		T119,
		T120,
		T121,
		T122,
		T123,
		T124,
		T125,
		T126)> type_;

	Event operator()(const EnqueueArgs& enqueueArgs,
		T0 arg0,
		T1 arg1,
		T2 arg2,
		T3 arg3,
		T4 arg4,
		T5 arg5,
		T6 arg6,
		T7 arg7,
		T8 arg8,
		T9 arg9,
		T10 arg10,
		T11 arg11,
		T12 arg12,
		T13 arg13,
		T14 arg14,
		T15 arg15,
		T16 arg16,
		T17 arg17,
		T18 arg18,
		T19 arg19,
		T20 arg20,
		T21 arg21,
		T22 arg22,
		T23 arg23,
		T24 arg24,
		T25 arg25,
		T26 arg26,
		T27 arg27,
		T28 arg28,
		T29 arg29,
		T30 arg30,
		T31 arg31,
		T32 arg32,
		T33 arg33,
		T34 arg34,
		T35 arg35,
		T36 arg36,
		T37 arg37,
		T38 arg38,
		T39 arg39,
		T40 arg40,
		T41 arg41,
		T42 arg42,
		T43 arg43,
		T44 arg44,
		T45 arg45,
		T46 arg46,
		T47 arg47,
		T48 arg48,
		T49 arg49,
		T50 arg50,
		T51 arg51,
		T52 arg52,
		T53 arg53,
		T54 arg54,
		T55 arg55,
		T56 arg56,
		T57 arg57,
		T58 arg58,
		T59 arg59,
		T60 arg60,
		T61 arg61,
		T62 arg62,
		T63 arg63,
		T64 arg64,
		T65 arg65,
		T66 arg66,
		T67 arg67,
		T68 arg68,
		T69 arg69,
		T70 arg70,
		T71 arg71,
		T72 arg72,
		T73 arg73,
		T74 arg74,
		T75 arg75,
		T76 arg76,
		T77 arg77,
		T78 arg78,
		T79 arg79,
		T80 arg80,
		T81 arg81,
		T82 arg82,
		T83 arg83,
		T84 arg84,
		T85 arg85,
		T86 arg86,
		T87 arg87,
		T88 arg88,
		T89 arg89,
		T90 arg90,
		T91 arg91,
		T92 arg92,
		T93 arg93,
		T94 arg94,
		T95 arg95,
		T96 arg96,
		T97 arg97,
		T98 arg98,
		T99 arg99,
		T100 arg100,
		T101 arg101,
		T102 arg102,
		T103 arg103,
		T104 arg104,
		T105 arg105,
		T106 arg106,
		T107 arg107,
		T108 arg108,
		T109 arg109,
		T110 arg110,
		T111 arg111,
		T112 arg112,
		T113 arg113,
		T114 arg114,
		T115 arg115,
		T116 arg116,
		T117 arg117,
		T118 arg118,
		T119 arg119,
		T120 arg120,
		T121 arg121,
		T122 arg122,
		T123 arg123,
		T124 arg124,
		T125 arg125,
		T126 arg126)
	{
		return functor_(
			enqueueArgs,
			arg0,
			arg1,
			arg2,
			arg3,
			arg4,
			arg5,
			arg6,
			arg7,
			arg8,
			arg9,
			arg10,
			arg11,
			arg12,
			arg13,
			arg14,
			arg15,
			arg16,
			arg17,
			arg18,
			arg19,
			arg20,
			arg21,
			arg22,
			arg23,
			arg24,
			arg25,
			arg26,
			arg27,
			arg28,
			arg29,
			arg30,
			arg31,
			arg32,
			arg33,
			arg34,
			arg35,
			arg36,
			arg37,
			arg38,
			arg39,
			arg40,
			arg41,
			arg42,
			arg43,
			arg44,
			arg45,
			arg46,
			arg47,
			arg48,
			arg49,
			arg50,
			arg51,
			arg52,
			arg53,
			arg54,
			arg55,
			arg56,
			arg57,
			arg58,
			arg59,
			arg60,
			arg61,
			arg62,
			arg63,
			arg64,
			arg65,
			arg66,
			arg67,
			arg68,
			arg69,
			arg70,
			arg71,
			arg72,
			arg73,
			arg74,
			arg75,
			arg76,
			arg77,
			arg78,
			arg79,
			arg80,
			arg81,
			arg82,
			arg83,
			arg84,
			arg85,
			arg86,
			arg87,
			arg88,
			arg89,
			arg90,
			arg91,
			arg92,
			arg93,
			arg94,
			arg95,
			arg96,
			arg97,
			arg98,
			arg99,
			arg100,
			arg101,
			arg102,
			arg103,
			arg104,
			arg105,
			arg106,
			arg107,
			arg108,
			arg109,
			arg110,
			arg111,
			arg112,
			arg113,
			arg114,
			arg115,
			arg116,
			arg117,
			arg118,
			arg119,
			arg120,
			arg121,
			arg122,
			arg123,
			arg124,
			arg125,
			arg126);
	}

	operator type_ ()    
	{
		return type_(*this);
	}

};





} // namespace detail

//----------------------------------------------------------------------------------------------

template <
   typename T0,   typename T1 = detail::NullType,   typename T2 = detail::NullType,
   typename T3 = detail::NullType,   typename T4 = detail::NullType,
   typename T5 = detail::NullType,   typename T6 = detail::NullType,
   typename T7 = detail::NullType,   typename T8 = detail::NullType,
   typename T9 = detail::NullType,   typename T10 = detail::NullType,
   typename T11 = detail::NullType,   typename T12 = detail::NullType,
   typename T13 = detail::NullType,   typename T14 = detail::NullType,
   typename T15 = detail::NullType,   typename T16 = detail::NullType,
   typename T17 = detail::NullType,   typename T18 = detail::NullType,
   typename T19 = detail::NullType,   typename T20 = detail::NullType,
   typename T21 = detail::NullType,   typename T22 = detail::NullType,
   typename T23 = detail::NullType,   typename T24 = detail::NullType,
   typename T25 = detail::NullType,   typename T26 = detail::NullType,
   typename T27 = detail::NullType,   typename T28 = detail::NullType,
   typename T29 = detail::NullType,   typename T30 = detail::NullType,
   typename T31 = detail::NullType,   typename T32 = detail::NullType,
   typename T33 = detail::NullType,   typename T34 = detail::NullType,
   typename T35 = detail::NullType,   typename T36 = detail::NullType,
   typename T37 = detail::NullType,   typename T38 = detail::NullType,
   typename T39 = detail::NullType,   typename T40 = detail::NullType,
   typename T41 = detail::NullType,   typename T42 = detail::NullType,
   typename T43 = detail::NullType,   typename T44 = detail::NullType,
   typename T45 = detail::NullType,   typename T46 = detail::NullType,
   typename T47 = detail::NullType,   typename T48 = detail::NullType,
   typename T49 = detail::NullType,   typename T50 = detail::NullType,
   typename T51 = detail::NullType,   typename T52 = detail::NullType,
   typename T53 = detail::NullType,   typename T54 = detail::NullType,
   typename T55 = detail::NullType,   typename T56 = detail::NullType,
   typename T57 = detail::NullType,   typename T58 = detail::NullType,
   typename T59 = detail::NullType,   typename T60 = detail::NullType,
   typename T61 = detail::NullType,   typename T62 = detail::NullType,
   typename T63 = detail::NullType,   typename T64 = detail::NullType,
   typename T65 = detail::NullType,   typename T66 = detail::NullType,
   typename T67 = detail::NullType,   typename T68 = detail::NullType,
   typename T69 = detail::NullType,   typename T70 = detail::NullType,
   typename T71 = detail::NullType,   typename T72 = detail::NullType,
   typename T73 = detail::NullType,   typename T74 = detail::NullType,
   typename T75 = detail::NullType,   typename T76 = detail::NullType,
   typename T77 = detail::NullType,   typename T78 = detail::NullType,
   typename T79 = detail::NullType,   typename T80 = detail::NullType,
   typename T81 = detail::NullType,   typename T82 = detail::NullType,
   typename T83 = detail::NullType,   typename T84 = detail::NullType,
   typename T85 = detail::NullType,   typename T86 = detail::NullType,
   typename T87 = detail::NullType,   typename T88 = detail::NullType,
   typename T89 = detail::NullType,   typename T90 = detail::NullType,
   typename T91 = detail::NullType,   typename T92 = detail::NullType,
   typename T93 = detail::NullType,   typename T94 = detail::NullType,
   typename T95 = detail::NullType,   typename T96 = detail::NullType,
   typename T97 = detail::NullType,   typename T98 = detail::NullType,
   typename T99 = detail::NullType,   typename T100 = detail::NullType,
   typename T101 = detail::NullType,   typename T102 = detail::NullType,
   typename T103 = detail::NullType,   typename T104 = detail::NullType,
   typename T105 = detail::NullType,   typename T106 = detail::NullType,
   typename T107 = detail::NullType,   typename T108 = detail::NullType,
   typename T109 = detail::NullType,   typename T110 = detail::NullType,
   typename T111 = detail::NullType,   typename T112 = detail::NullType,
   typename T113 = detail::NullType,   typename T114 = detail::NullType,
   typename T115 = detail::NullType,   typename T116 = detail::NullType,
   typename T117 = detail::NullType,   typename T118 = detail::NullType,
   typename T119 = detail::NullType,   typename T120 = detail::NullType,
   typename T121 = detail::NullType,   typename T122 = detail::NullType,
   typename T123 = detail::NullType,   typename T124 = detail::NullType,
   typename T125 = detail::NullType,   typename T126 = detail::NullType,
   typename T127 = detail::NullType>
struct make_kernel :
    public detail::functionImplementation_<
               T0,   T1,   T2,   T3,
               T4,   T5,   T6,   T7,
               T8,   T9,   T10,   T11,
               T12,   T13,   T14,   T15,
               T16,   T17,   T18,   T19,
               T20,   T21,   T22,   T23,
               T24,   T25,   T26,   T27,
               T28,   T29,   T30,   T31,
               T32,   T33,   T34,   T35,
               T36,   T37,   T38,   T39,
               T40,   T41,   T42,   T43,
               T44,   T45,   T46,   T47,
               T48,   T49,   T50,   T51,
               T52,   T53,   T54,   T55,
               T56,   T57,   T58,   T59,
               T60,   T61,   T62,   T63,
               T64,   T65,   T66,   T67,
               T68,   T69,   T70,   T71,
               T72,   T73,   T74,   T75,
               T76,   T77,   T78,   T79,
               T80,   T81,   T82,   T83,
               T84,   T85,   T86,   T87,
               T88,   T89,   T90,   T91,
               T92,   T93,   T94,   T95,
               T96,   T97,   T98,   T99,
               T100,   T101,   T102,   T103,
               T104,   T105,   T106,   T107,
               T108,   T109,   T110,   T111,
               T112,   T113,   T114,   T115,
               T116,   T117,   T118,   T119,
               T120,   T121,   T122,   T123,
               T124,   T125,   T126,   T127>
{
public:
	typedef detail::KernelFunctorGlobal<             
		       T0,   T1,   T2,   T3,
               T4,   T5,   T6,   T7,
               T8,   T9,   T10,   T11,
               T12,   T13,   T14,   T15,
               T16,   T17,   T18,   T19,
               T20,   T21,   T22,   T23,
               T24,   T25,   T26,   T27,
               T28,   T29,   T30,   T31,
               T32,   T33,   T34,   T35,
               T36,   T37,   T38,   T39,
               T40,   T41,   T42,   T43,
               T44,   T45,   T46,   T47,
               T48,   T49,   T50,   T51,
               T52,   T53,   T54,   T55,
               T56,   T57,   T58,   T59,
               T60,   T61,   T62,   T63,
               T64,   T65,   T66,   T67,
               T68,   T69,   T70,   T71,
               T72,   T73,   T74,   T75,
               T76,   T77,   T78,   T79,
               T80,   T81,   T82,   T83,
               T84,   T85,   T86,   T87,
               T88,   T89,   T90,   T91,
               T92,   T93,   T94,   T95,
               T96,   T97,   T98,   T99,
               T100,   T101,   T102,   T103,
               T104,   T105,   T106,   T107,
               T108,   T109,   T110,   T111,
               T112,   T113,   T114,   T115,
               T116,   T117,   T118,   T119,
               T120,   T121,   T122,   T123,
               T124,   T125,   T126,   T127> FunctorType;

    make_kernel(
        const Program& program,
        const STRING_CLASS name,
        cl_int * err = NULL) :
           detail::functionImplementation_<
                    T0,   T1,   T2,   T3,
                       T4,   T5,   T6,   T7,
                       T8,   T9,   T10,   T11,
                       T12,   T13,   T14,   T15,
                       T16,   T17,   T18,   T19,
                       T20,   T21,   T22,   T23,
                       T24,   T25,   T26,   T27,
                       T28,   T29,   T30,   T31,
                       T32,   T33,   T34,   T35,
                       T36,   T37,   T38,   T39,
                       T40,   T41,   T42,   T43,
                       T44,   T45,   T46,   T47,
                       T48,   T49,   T50,   T51,
                       T52,   T53,   T54,   T55,
                       T56,   T57,   T58,   T59,
                       T60,   T61,   T62,   T63,
                       T64,   T65,   T66,   T67,
                       T68,   T69,   T70,   T71,
                       T72,   T73,   T74,   T75,
                       T76,   T77,   T78,   T79,
                       T80,   T81,   T82,   T83,
                       T84,   T85,   T86,   T87,
                       T88,   T89,   T90,   T91,
                       T92,   T93,   T94,   T95,
                       T96,   T97,   T98,   T99,
                       T100,   T101,   T102,   T103,
                       T104,   T105,   T106,   T107,
                       T108,   T109,   T110,   T111,
                       T112,   T113,   T114,   T115,
                       T116,   T117,   T118,   T119,
                       T120,   T121,   T122,   T123,
                       T124,   T125,   T126,   T127>(
            FunctorType(program, name, err)) 
    {}

    make_kernel(
        const Kernel kernel,
        cl_int * err = NULL) :
           detail::functionImplementation_<
                    T0,   T1,   T2,   T3,
                       T4,   T5,   T6,   T7,
                       T8,   T9,   T10,   T11,
                       T12,   T13,   T14,   T15,
                       T16,   T17,   T18,   T19,
                       T20,   T21,   T22,   T23,
                       T24,   T25,   T26,   T27,
                       T28,   T29,   T30,   T31,
                       T32,   T33,   T34,   T35,
                       T36,   T37,   T38,   T39,
                       T40,   T41,   T42,   T43,
                       T44,   T45,   T46,   T47,
                       T48,   T49,   T50,   T51,
                       T52,   T53,   T54,   T55,
                       T56,   T57,   T58,   T59,
                       T60,   T61,   T62,   T63,
                       T64,   T65,   T66,   T67,
                       T68,   T69,   T70,   T71,
                       T72,   T73,   T74,   T75,
                       T76,   T77,   T78,   T79,
                       T80,   T81,   T82,   T83,
                       T84,   T85,   T86,   T87,
                       T88,   T89,   T90,   T91,
                       T92,   T93,   T94,   T95,
                       T96,   T97,   T98,   T99,
                       T100,   T101,   T102,   T103,
                       T104,   T105,   T106,   T107,
                       T108,   T109,   T110,   T111,
                       T112,   T113,   T114,   T115,
                       T116,   T117,   T118,   T119,
                       T120,   T121,   T122,   T123,
                       T124,   T125,   T126,   T127>(
            FunctorType(kernel, err)) 
    {}    
};

#endif

//----------------------------------------------------------------------------------------------------------------------

#undef __ERR_STR
#if !defined(__CL_USER_OVERRIDE_ERROR_STRINGS)
#undef __GET_DEVICE_INFO_ERR
#undef __GET_PLATFORM_INFO_ERR
#undef __GET_DEVICE_IDS_ERR
#undef __GET_CONTEXT_INFO_ERR
#undef __GET_EVENT_INFO_ERR
#undef __GET_EVENT_PROFILE_INFO_ERR
#undef __GET_MEM_OBJECT_INFO_ERR
#undef __GET_IMAGE_INFO_ERR
#undef __GET_SAMPLER_INFO_ERR
#undef __GET_KERNEL_INFO_ERR
#undef __GET_KERNEL_WORK_GROUP_INFO_ERR
#undef __GET_PROGRAM_INFO_ERR
#undef __GET_PROGRAM_BUILD_INFO_ERR
#undef __GET_COMMAND_QUEUE_INFO_ERR

#undef __CREATE_CONTEXT_ERR
#undef __CREATE_CONTEXT_FROM_TYPE_ERR
#undef __GET_SUPPORTED_IMAGE_FORMATS_ERR

#undef __CREATE_BUFFER_ERR
#undef __CREATE_SUBBUFFER_ERR
#undef __CREATE_IMAGE2D_ERR
#undef __CREATE_IMAGE3D_ERR
#undef __CREATE_SAMPLER_ERR
#undef __SET_MEM_OBJECT_DESTRUCTOR_CALLBACK_ERR

#undef __CREATE_USER_EVENT_ERR
#undef __SET_USER_EVENT_STATUS_ERR
#undef __SET_EVENT_CALLBACK_ERR

#undef __WAIT_FOR_EVENTS_ERR

#undef __CREATE_KERNEL_ERR
#undef __SET_KERNEL_ARGS_ERR
#undef __CREATE_PROGRAM_WITH_SOURCE_ERR
#undef __CREATE_PROGRAM_WITH_BINARY_ERR
#undef __BUILD_PROGRAM_ERR
#undef __CREATE_KERNELS_IN_PROGRAM_ERR

#undef __CREATE_COMMAND_QUEUE_ERR
#undef __SET_COMMAND_QUEUE_PROPERTY_ERR
#undef __ENQUEUE_READ_BUFFER_ERR
#undef __ENQUEUE_WRITE_BUFFER_ERR
#undef __ENQUEUE_READ_BUFFER_RECT_ERR
#undef __ENQUEUE_WRITE_BUFFER_RECT_ERR
#undef __ENQEUE_COPY_BUFFER_ERR
#undef __ENQEUE_COPY_BUFFER_RECT_ERR
#undef __ENQUEUE_READ_IMAGE_ERR
#undef __ENQUEUE_WRITE_IMAGE_ERR
#undef __ENQUEUE_COPY_IMAGE_ERR
#undef __ENQUEUE_COPY_IMAGE_TO_BUFFER_ERR
#undef __ENQUEUE_COPY_BUFFER_TO_IMAGE_ERR
#undef __ENQUEUE_MAP_BUFFER_ERR
#undef __ENQUEUE_MAP_IMAGE_ERR
#undef __ENQUEUE_UNMAP_MEM_OBJECT_ERR
#undef __ENQUEUE_NDRANGE_KERNEL_ERR
#undef __ENQUEUE_TASK_ERR
#undef __ENQUEUE_NATIVE_KERNEL

#undef __UNLOAD_COMPILER_ERR
#endif //__CL_USER_OVERRIDE_ERROR_STRINGS

#undef __GET_INFO_HELPER_WITH_RETAIN

// Extensions
#undef __INIT_CL_EXT_FCN_PTR
#undef __CREATE_SUB_DEVICES

#if defined(USE_CL_DEVICE_FISSION)
#undef __PARAM_NAME_DEVICE_FISSION
#endif // USE_CL_DEVICE_FISSION

#undef __DEFAULT_NOT_INITIALIZED 
#undef __DEFAULT_BEING_INITIALIZED 
#undef __DEFAULT_INITIALIZED

} // namespace cl

#ifdef _WIN32
#pragma pop_macro("max")
#endif // _WIN32

#endif // CL_HPP_
