/***************************************************************************
  preferences.cc
  -------------------
  A class to access persistent preferences for an application. Utilizes XML/DOM.
  Basic format is:
    <!DOCTYPE preferences>
    <preferences version="0.1" application="MyApp" >
        <group name="Default" >
            <option key="alpha" value="true" />
            <option key="beta" value="99" />
            <option key="gamma" value="test" />
        </group>
    </preferences>
  -------------------
  begin         Tue Sep 12 2000
  author        David Johnson, david@usermode.org
  -------------------
  Copyright 2000, David Johnson
  Please see the header file for copyright and license information
***************************************************************************/


// TODO: fix up to account for worst case scenarios:
//      keys without values in file, and
//      checking for a key that doesn't exist puts it into the map
//      then it gets written out if dirty, corrupting the file

// TODO: Fix up error reporting

#include <qdom.h>
#include <qfile.h>
#include <qtextstream.h>

#include "preferences.h"

//////////////////////////////////////////////////////////////////////////////
// Construction                                                             //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// Preferences()
// -------------
// Constructor. Takes the preferences file name as an argument.

Preferences::Preferences(const QString& filename,
                         const QString& format,
                         const QString& version)
    : _currentgroup(),
      _file(filename),
      _format(format),
      _version(version),
      _groups()
{
    readData();
    _dirty = false;
    _currentgroup = "Default";
}

//////////////////////////////////////////////////////////////////////////////
//  ~Preferences()
// ---------------
// Destructor

Preferences::~Preferences()
{
    if (_dirty) writeData();
}

//////////////////////////////////////////////////////////////////////////////
// Settings
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// getBoolean()
// ------------
// Get a boolean value

bool Preferences::getBool(const QString& key, bool def)
{
    _buffer = getString(key, def ? "true" : "false");
    if (_buffer.isEmpty()) return def;
    if (_buffer.contains("true"))
        return true;
    else
        return false;
}

//////////////////////////////////////////////////////////////////////////////
// setBoolean()
// ------------
// Set a boolean value

void Preferences::setBool(const QString& key, bool value)
{
    _groups[_currentgroup][key].value = value ? "true" : "false";
    _groups[_currentgroup][key].cdata = false;
    _dirty = true;
}

//////////////////////////////////////////////////////////////////////////////
// getFont()
// -----------
// Get a completely constructed QFont

QFont Preferences::getFont(const QString& key, QFont &def)
{
  QFont ret;
  ret.setFamily( getString( QString("font_") + key + "_family", def.family() ) );
  ret.setWeight( getNumber( QString("font_") + key + "_weight", def.weight() ) );
  ret.setPointSize( getNumber( QString("font_") + key + "_pointsize", def.pointSize() ) );
  ret.setItalic( getBool( QString("font_") + key + "_italic", def.italic() ) );
  ret.setUnderline( getBool( QString("font_") + key + "_underline", def.underline() ) );

  return ret;

}

//////////////////////////////////////////////////////////////////////////////
// setFont()
// -----------
// Set a font.

void Preferences::setFont(const QString& key, QFont &value)
{
	setString( QString("font_") + key + "_family", value.family() );
	setNumber( QString("font_") + key + "_weight", value.weight() );
	setNumber( QString("font_") + key + "_pointsize", value.pointSize() );
	setBool( QString("font_") + key + "_italic", value.italic() );
	setBool( QString("font_") + key + "_underline", value.underline() );
}


//////////////////////////////////////////////////////////////////////////////
// getNumber()
// -----------
// Get a number value

long Preferences::getNumber(const QString& key, long def)
{
    _buffer = getString(key, QString::number(def));
	if (_buffer.isEmpty()) return def;

    bool ok;
    long num = _buffer.toLong(&ok);
    if (ok) return num;
    else return def;
}

//////////////////////////////////////////////////////////////////////////////
// setNumber()
// -----------
// Set a number value

void Preferences::setNumber(const QString& key, long value)
{
    _buffer.setNum(value);

    _groups[_currentgroup][key].value = _buffer;
    _groups[_currentgroup][key].cdata = false;
    _dirty = true;
}

//////////////////////////////////////////////////////////////////////////////
// getCDATA()
// -----------
// Get a CDATA value

QString Preferences::getCDATA(const QString& key, const QString& def)
{
    _buffer = "";
    if (_groups.contains(_currentgroup)) {
        if (_groups[_currentgroup].contains(key)) {
            _buffer = _groups[_currentgroup][key].value;
        }
    }
	//qDebug("::getCDATA \"%s\"", _buffer.latin1());
	if (_buffer.isEmpty()) return def;
    return _buffer;
}

//////////////////////////////////////////////////////////////////////////////
// getAttr1()
// -----------
// Get a string value

QString Preferences::getAttr1(const QString& key, const QString& def)
{
    _buffer = "";
    if (_groups.contains(_currentgroup)) {
        if (_groups[_currentgroup].contains(key)) {
            _buffer = _groups[_currentgroup][key].attr1;
        }
    }
	if (_buffer.isEmpty()) return def;
    return _buffer;
}

//////////////////////////////////////////////////////////////////////////////
// getAttr2()
// -----------
// Get a string value

QString Preferences::getAttr2(const QString& key, const QString& def)
{
    _buffer = "";
    if (_groups.contains(_currentgroup)) {
        if (_groups[_currentgroup].contains(key)) {
            _buffer = _groups[_currentgroup][key].attr2;
        }
    }
	if (_buffer.isEmpty()) return def;
    return _buffer;
}

//////////////////////////////////////////////////////////////////////////////
// getAttr3()
// -----------
// Get a string value

QString Preferences::getAttr3(const QString& key, const QString& def)
{
    _buffer = "";
    if (_groups.contains(_currentgroup)) {
        if (_groups[_currentgroup].contains(key)) {
            _buffer = _groups[_currentgroup][key].attr3;
        }
    }
	if (_buffer.isEmpty()) return def;
    return _buffer;
}


//////////////////////////////////////////////////////////////////////////////
// getString()
// -----------
// Get a string value

QString Preferences::getString(const QString& key, const QString& def)
{
    _buffer = "";
    if (_groups.contains(_currentgroup)) {
        if (_groups[_currentgroup].contains(key)) {
            _buffer = _groups[_currentgroup][key].value;
        }
    }
	if (_buffer.isEmpty()) return def;
    return _buffer;
}

//////////////////////////////////////////////////////////////////////////////
// setCDATA()
// -----------
// Set a string value, and also set the cdata flag to true!

void Preferences::setCDATA(const QString& key, const QString &cdata)
{
		//qDebug("::setCDATA %s %s\n", key.latin1(), cdata.latin1());
    _groups[_currentgroup][key].value = cdata;
    _groups[_currentgroup][key].cdata = true;
    _dirty = true;
}

//////////////////////////////////////////////////////////////////////////////
// setAttr2()
// -----------
// Set a string value

void Preferences::setAttr2(const QString& key, const QString &attr2)
{
    _groups[_currentgroup][key].attr2 = attr2;
    _groups[_currentgroup][key].cdata = false;
    _dirty = true;
}

//////////////////////////////////////////////////////////////////////////////
// setAttr3()
// -----------
// Set a string value

void Preferences::setAttr3(const QString& key, const QString &attr3)
{
    _groups[_currentgroup][key].attr3 = attr3;
    _groups[_currentgroup][key].cdata = false;
    _dirty = true;
}

//////////////////////////////////////////////////////////////////////////////
// setAttr1()
// -----------
// Set a string value

void Preferences::setAttr1(const QString& key, const QString &attr1)
{
    _groups[_currentgroup][key].attr1 = attr1;
    _groups[_currentgroup][key].cdata = false;
    _dirty = true;
}

//////////////////////////////////////////////////////////////////////////////
// setString()
// -----------
// Set a string value

void Preferences::setString(const QString& key, const QString& value, const QString &attr1, const QString &attr2, const QString &attr3)
{
    _groups[_currentgroup][key].value = value;
    _groups[_currentgroup][key].attr1 = attr1;
    _groups[_currentgroup][key].attr2 = attr2;
    _groups[_currentgroup][key].attr3 = attr3;
    _groups[_currentgroup][key].cdata = false;
    _dirty = true;
}
		
//////////////////////////////////////////////////////////////////////////////
// removeValue()
// -------------
// Remove a value from the preferences

void Preferences::removeKey(const QString& key)
{
    _groups[_currentgroup].remove(key);
}

//////////////////////////////////////////////////////////////////////////////
// removeGroup()
// -------------
// Remove a group from the preferences, and all its options

void Preferences::removeGroup()
{
  _groups.remove(_currentgroup);
  _dirty = true;
  flush();
}

//////////////////////////////////////////////////////////////////////////////
// flush()
// -------
// Flush the preferences to file

bool Preferences::flush()
{
    bool b = false;
    if (_dirty) {
        b = writeData();
        _dirty = false;
    }
    return b;
}

//////////////////////////////////////////////////////////////////////////////
// Serialization                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// readData()
// ----------
// Read data from the file

void Preferences::readData()
{
    // open file
    QFile* datafile = new QFile(_file);
    

    if (datafile == NULL) {
      qWarning("datafile is NULL");
      // Do we have read access?
    }
    
    if (!datafile->open(IO_ReadOnly)) {
        // error opening file
        //qWarning("Problem opening preferences file for reading.");
        datafile->close();
        delete (datafile);
        return;
    }
    // open dom document
    QDomDocument doc("preferences");
    doc.setContent(datafile);
    datafile->close();
    delete (datafile);

    // check the doc type and stuff
    if (doc.doctype().name() != "preferences") {
        // wrong file type
        qWarning("Not a valid preferences file");
        return;
    }
    QDomElement root = doc.documentElement();
    if (root.attribute("application") != _format && root.attribute("application") != QString("Kaim")) {
        // right file type, wrong application
        qWarning(QString("Not a preferences file for " + _format).latin1());
//        return;
    }
    // We don't care about application version...

    // get list of groups
    QDomNodeList nodes = root.elementsByTagName("group");

    // iterate through the groups
    QDomNodeList options;
    for (unsigned n=0; n<nodes.count(); ++n) {
        if (nodes.item(n).isElement()) {
            processGroup(nodes.item(n).toElement());
        }
    }
}

void Preferences::processGroup(QDomElement group)
{
    QDomElement elem;
    QDomNodeList options;
    _currentgroup = group.attribute("name", "Default");
    options = group.elementsByTagName("option");
    for (unsigned n=0; n<options.count(); ++n) {
        if (options.item(n).isElement()) {
            elem = options.item(n).toElement();
		        setString(elem.attribute("key"), elem.attribute("value"),
							elem.attribute("attr1"), elem.attribute("attr2"),
							elem.attribute("attr3"));
        }
    }

    options = group.elementsByTagName("cdata");
    for (unsigned n=0; n<options.count(); ++n) {
        if (options.item(n).isElement()) {
            elem = options.item(n).toElement();
						//qDebug("setCDATA %s %s\n", elem.attribute("key").latin1(), (elem.firstChild().toCDATASection()).data().latin1() );
						setCDATA(elem.attribute("key"), (elem.firstChild().toCDATASection()).data());
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// writeData()
// -----------
// Write data out to the file

bool Preferences::writeData()
{
    QDomDocument doc("preferences");

    // create the root element
    QDomElement root = doc.createElement(doc.doctype().name());
    root.setAttribute("version", _version);
    root.setAttribute("application", _format);

    // now do our options group by group
    QMap<QString, PrefMap>::Iterator git;
    PrefMap::Iterator pit;
    QDomElement group, option;
		QDomCDATASection cdata;
    for (git = _groups.begin(); git != _groups.end(); ++git) {
        // create a group element
        group = doc.createElement("group");
        group.setAttribute("name", git.key());
        // add in options
        for (pit = (*git).begin(); pit != (*git).end(); ++pit) {
						if(pit.data().cdata == true){
							option = doc.createElement("cdata");
							cdata = doc.createCDATASection("cdata");
							//cdata.setAttribute("key", pit.key());
							//cdata.setAttribute("cdata", "true");
							//cdata.setAttribute("value", pit.data().value);
							cdata.setData(pit.data().value);
							option.appendChild(cdata);
							option.setAttribute("key", pit.key());
							group.appendChild(option);
						} else {
	            option = doc.createElement("option");
	            option.setAttribute("key", pit.key());
	            option.setAttribute("value", pit.data().value);
							if(!pit.data().attr1.isNull()){
								option.setAttribute("attr1", pit.data().attr1);
							}
							if(!pit.data().attr2.isNull()){
								option.setAttribute("attr2", pit.data().attr2);
							}
							if(!pit.data().attr3.isNull()){
								option.setAttribute("attr3", pit.data().attr3);
							}
	            group.appendChild(option);
						}
        }
        root.appendChild(group);
    }
    doc.appendChild(root);

    // open file
    QFile* datafile = new QFile(_file);
    if (!datafile->open(IO_WriteOnly)) {
        // error opening file
        qWarning("Problems opening preferences file for writing.");
        datafile->close();
        delete (datafile);
        return false;
    }

    // write it out
    QTextStream textstream(datafile);
    doc.save(textstream, 0);
    datafile->close();
    delete (datafile);
		return true;
}
