#include <string.h>
#include <time.h>

#include <gdk/gdkkeysyms.h>
#include <glade/glade.h>
#include <gtk/gtk.h>

#include "kpcalendarentryinfodialog.h"
#include "kpnewsplitworkoutdialog.h"
#include "kppreferencesdialog.h"
#include "kpnewworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpcalendarview.h"
#include "kpmainwindow.h"
#include "kpchartview.h"
#include "kpstatusbar.h"
#include "kpviewmodel.h"
#include "kptreeview.h"
#include "kplistview.h"
#include "kplogstore.h"
#include "kpguiutils.h"
#include "kpchart.h"

#include "kpview.h"
#include "kpstatsview.h"

#include "../kpplugin.h"
#include "../kpcomment.h"
#include "../kpworkout.h"
#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kpsettings.h"
#include "../kputil.h"

/* Static functions */
static void     kp_main_window_init_log         (KPMainWindow *window,
                                                 const gchar *filename);
static void     kp_main_window_deinit_log       (KPMainWindow *window);
static void     kp_main_window_class_init       (KPMainWindowClass *klass);
static void     kp_main_window_init             (KPMainWindow *dialog);
static void     kp_main_window_finalize         (GObject *object);

static void     kp_main_window_update_date_text (KPMainWindow *window);
static void     calendar_view_day_selected      (KPCalendarView *cv,
                                                 KPDate *date,
                                                 KPMainWindow *window);
static void     log_connect_signals             (KPTrainingLog *log,
                                                 KPMainWindow *window);
static void     log_disconnect_signals          (KPTrainingLog *log,
                                                 KPMainWindow *window);
static void     log_entry_removed               (KPTrainingLog *log, guint d,
                                                 guint m, guint y,
                                                 const gchar *mark,
                                                 KPMainWindow *window);
static void     log_entry_added                 (KPTrainingLog *log, guint d,
                                                 guint m, guint y,
                                                 const gchar *mark,
                                                 KPMainWindow *window);
static void     log_changed                     (KPTrainingLog *log,
                                                 KPMainWindow *window);

/* Callbacks for KPMainWindow */
static gboolean main_window_key_press           (GtkWidget *widget,
                                                 GdkEventKey *key,
                                                 KPMainWindow window);
static void     on_main_window_destroy          (GtkWidget *widget,
                                                 KPMainWindow *window);
static void     on_menu_help_about_activate     (GtkMenuItem *item,
                                                 KPMainWindow *window);
static void     on_menu_file_quit_activate      (GtkMenuItem *item,
                                                 KPMainWindow *window);
static void     on_menu_file_new_log_activate   (GtkMenuItem *item,
                                                 KPMainWindow *window);
static void     on_menu_file_open_log_activate  (GtkMenuItem *item,
                                                 KPMainWindow *window);
static void     on_menu_file_save_log_activate  (GtkMenuItem *item,
                                                 KPMainWindow *window);
static void     on_menu_file_save_log_as_activate(GtkMenuItem *item,
                                                 KPMainWindow *window);
static void     on_menu_view_activate           (GtkRadioMenuItem *item,
                                                 KPMainWindow *window);
static void     menu_view_show_toolbar_toggled  (GtkCheckMenuItem *item,
                                                 KPMainWindow *window);
static void     menu_view_show_tree_toggled     (GtkCheckMenuItem *item,
                                                 KPMainWindow *window);
static void     menu_view_show_statusbar_toggled(GtkCheckMenuItem *item,
                                                 KPMainWindow *window);
static void     on_menu_edit_pref_activate      (GtkMenuItem *item,
                                                 KPMainWindow *winddow);
static void     on_menu_cal_navigation_activate (GtkMenuItem *item,
                                                 KPMainWindow *window);
static void     on_toolbar_html_button_clicked  (GtkButton *button,
                                                 KPMainWindow *window);
static void     on_menu_remove_button_activate  (GtkMenuItem *item,
                                                 KPMainWindow *window);
static void     on_toolbar_add_button_clicked   (GtkButton *button,
                                                 KPMainWindow *window);
static void     on_toolbar_add_s_button_clicked (GtkButton *button,
                                                 KPMainWindow *window);
static void     on_toolbar_comment_button_clicked
                                                (GtkButton *button,
                                                 KPMainWindow *window);
static void     on_toolbar_cview_button_clicked (GtkWidget *widget,
                                                 KPMainWindow *window);
static void     workout_tree_selection_changed  (GtkTreeSelection *selection,
                                                 KPMainWindow *window);
static void     on_paned_handle_move            (GObject *pane,
                                                 GParamSpec *spec,
                                                 gpointer data);
static void     kp_view_view_set                (KPViewModel *model,
                                                 KPViewModelType type,
                                                 KPMainWindow *window);
static void     kp_view_date_set                (KPViewModel *model,
                                                 KPDate *date,
                                                 KPMainWindow *window);
static void     kp_view_viewer_set              (KPView *view,
                                                 KPViewModel *model,
                                                 KPMainWindow *window);
static gboolean save_logfile                    (KPMainWindow *window,
                                                 const gchar *filename);
static void     new_log                         (KPMainWindow *window);
static void     open_log                        (KPMainWindow *window);
static gboolean save_log                        (KPMainWindow *window);
static void     quit                            (KPMainWindow *window);
static void     update_toolbar_state            (KPMainWindow *window);

static GuiModuleSignalsData signals_data[] = {
{"menu_file_new_log",   "activate", CB (on_menu_file_new_log_activate),   NULL},
{"menu_file_open_log",  "activate", CB (on_menu_file_open_log_activate),  NULL},
{"menu_file_save_log",  "activate", CB (on_menu_file_save_log_activate),  NULL},
{"menu_file_save_log_as","activate",CB (on_menu_file_save_log_as_activate),NULL},
{"menu_file_quit",      "activate", CB (on_menu_file_quit_activate),      NULL},
{"menu_view_calendar",  "activate", CB (on_menu_view_activate),           NULL},
{"menu_view_list",      "activate", CB (on_menu_view_activate),           NULL},
{"menu_view_statistics","activate", CB (on_menu_view_activate),           NULL},
{"menu_view_chart",     "activate", CB (on_menu_view_activate),           NULL},
{"menu_help_about",     "activate", CB (on_menu_help_about_activate),     NULL},
{"menu_cal_add_mon",    "activate", CB (on_menu_cal_navigation_activate), NULL},
{"menu_cal_add_day",    "activate", CB (on_menu_cal_navigation_activate), NULL},
{"menu_cal_add_week",   "activate", CB (on_menu_cal_navigation_activate), NULL},
{"menu_cal_sub_mon",    "activate", CB (on_menu_cal_navigation_activate), NULL},
{"menu_cal_sub_day",    "activate", CB (on_menu_cal_navigation_activate), NULL},
{"menu_cal_sub_week",   "activate", CB (on_menu_cal_navigation_activate), NULL},
{"menu_cal_today",      "activate", CB (on_menu_cal_navigation_activate), NULL},
{"menu_log_remove",     "activate", CB (on_menu_remove_button_activate),  NULL},
{"menu_log_add_split",  "activate", CB (on_toolbar_add_s_button_clicked), NULL},
{"menu_log_add_workout","activate", CB (on_toolbar_add_button_clicked),   NULL},
{"menu_log_add_comment","activate", CB (on_toolbar_comment_button_clicked),NULL},
{"menu_edit_pref",      "activate", CB (on_menu_edit_pref_activate),      NULL},
{"toolbar_next_button", "clicked",  CB (on_menu_cal_navigation_activate), NULL},
{"toolbar_prev_button", "clicked",  CB (on_menu_cal_navigation_activate), NULL},
{"toolbar_open_button", "clicked",  CB (on_menu_file_open_log_activate),  NULL},
{"toolbar_new_button",  "clicked",  CB (on_menu_file_new_log_activate),   NULL},
{"toolbar_save_button", "clicked",  CB (on_menu_file_save_log_activate),  NULL},
{"menu_view_show_toolbar","activate",CB (menu_view_show_toolbar_toggled), NULL},
{"menu_view_show_tree", "activate", CB (menu_view_show_tree_toggled),     NULL},
{"menu_view_show_sb",   "activate", CB (menu_view_show_statusbar_toggled),NULL},
{"toolbar_html_button", "clicked",  CB (on_toolbar_html_button_clicked),  NULL},
{"toolbar_day_button",  "clicked",  CB (on_toolbar_cview_button_clicked), NULL},
{"toolbar_week_button", "clicked",  CB (on_toolbar_cview_button_clicked), NULL},
{"toolbar_month_button","clicked",  CB (on_toolbar_cview_button_clicked), NULL},
{"toolbar_year_button", "clicked",  CB (on_toolbar_cview_button_clicked), NULL},
{"menu_view_day",       "activate", CB (on_toolbar_cview_button_clicked), NULL},
{"menu_view_week",      "activate", CB (on_toolbar_cview_button_clicked), NULL},
{"menu_view_month",     "activate", CB (on_toolbar_cview_button_clicked), NULL},
{"menu_view_year",      "activate", CB (on_toolbar_cview_button_clicked), NULL},
{"menu_view_all_time",  "activate", CB (on_toolbar_cview_button_clicked), NULL},
{"h_paned",     "notify::position", CB (on_paned_handle_move),            NULL},
{ NULL,                  NULL,      NULL,                                 NULL},
};

typedef struct KPMainWindowPrivateData_
{
  KPTrainingLog *log;
  GString *window_title;
  GString *logfile;

  KPCalendarView *cv;
  KPTreeView *tree;
  GtkWidget *sw;
  GtkWidget *date_label;

  GtkWidget *view;
  GtkWidget *vbox;

  KPStatusbar *sbar;
 
  GtkWidget *toolbar_log;
  GtkWidget *toolbar_new_button;
  GtkWidget *toolbar_open_button;
  GtkWidget *toolbar_save_button;

  GtkWidget *paned_window;

  GtkWidget *menu_save_log_button;
  GtkWidget *menu_log;

  GtkWidget *menu_log_remove_entry;
  GtkWidget *menu_file_separator;
  GtkWidget *menu_import;
  GtkWidget *menu_export;
  GtkWidget *menubar;

  GtkWidget *toolbar_next_button;
  GtkWidget *toolbar_prev_button;

  GtkWidget *view_menu;

  gboolean log_is_changed;
} KPMainWindowPrivateData;

#define KP_MAIN_WINDOW_PRIVATE_DATA(widget) ((KPMainWindowPrivateData*) \
        (KP_MAIN_WINDOW (widget)->private_data))

static GObjectClass *parent_class = NULL;
static KPMainWindow *_window = NULL;

GType
kp_main_window_get_type (void)
{
  static GType kp_main_window_type = 0;

  if (kp_main_window_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPMainWindowClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_main_window_class_init,
      NULL,
      NULL,
      sizeof (KPMainWindow),
      0,
      (GInstanceInitFunc) kp_main_window_init,
      NULL,
    };

    kp_main_window_type = g_type_register_static (GTK_TYPE_WINDOW,
                                                 "KPMainWindow",
                                                  &our_info, 0);
  }

  return kp_main_window_type;
}

static void
kp_main_window_class_init (KPMainWindowClass * klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_main_window_finalize;
}

static void
kp_main_window_init (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkWidget *menuitem;
  GtkWidget *vbox;
  GtkWidget *sw_view;
  GdkPixbuf *icon;
  GladeXML *xml;
  gchar *file;
  gint width = -1;
  gint height = -1;

  xml = kp_gui_load ("main_window", "mainwin_vbox");
  
  window->private_data = g_new0 (KPMainWindowPrivateData, 1);
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  p_data->log_is_changed = FALSE;
  p_data->window_title = g_string_new (NULL);
  p_data->logfile = g_string_new (NULL);
  p_data->log = NULL;
  p_data->paned_window = KP_W (xml, "h_paned");
   
  vbox = KP_W (xml, "mainwin_vbox");
  gtk_container_add (GTK_CONTAINER (window), vbox);
  
  p_data->sbar = KP_STATUSBAR (kp_statusbar_new (NULL));
  gtk_box_pack_end (GTK_BOX (vbox), GTK_WIDGET (p_data->sbar), FALSE, TRUE, 0);
  gtk_widget_show (GTK_WIDGET (p_data->sbar));
 
  p_data->tree = KP_TREE_VIEW (kp_tree_view_new (NULL));
  p_data->view = GTK_WIDGET (kp_view_new (0, 0, 0));

  sw_view = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw_view),
                                  GTK_POLICY_AUTOMATIC, 
                                  GTK_POLICY_AUTOMATIC);
      
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (sw_view),
                                         GTK_WIDGET (p_data->view));
  gtk_widget_show (sw_view);

  p_data->sw = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (p_data->sw),
                                  GTK_POLICY_AUTOMATIC, 
                                  GTK_POLICY_AUTOMATIC);
      
  gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (p_data->sw),
                                         GTK_WIDGET (p_data->tree));
  gtk_widget_show (GTK_WIDGET (p_data->tree));
  gtk_widget_show (p_data->sw);
  
  gtk_paned_add1 (GTK_PANED (p_data->paned_window), GTK_WIDGET (p_data->sw));
  gtk_paned_add2 (GTK_PANED (p_data->paned_window), GTK_WIDGET (sw_view));

  gtk_widget_show (GTK_WIDGET (p_data->paned_window));
  
  p_data->cv = KP_CALENDAR_VIEW (kp_calendar_view_new (KP_VIEW_MODEL_TYPE_DAY, 
                                 0, 0, 0));
  g_object_set (G_OBJECT (p_data->cv), "line-spacing", 2, NULL);
  
  gtk_widget_set_sensitive (GTK_WIDGET (p_data->cv), TRUE);

  kp_view_add_viewer (KP_VIEW (p_data->view),
                      KP_VIEW_MODEL (p_data->cv));
  kp_view_add_viewer (KP_VIEW (p_data->view),
                      KP_VIEW_MODEL (kp_stats_view_new ()));
  kp_view_add_viewer (KP_VIEW (p_data->view),
                      KP_VIEW_MODEL (kp_list_view_new ()));
  kp_view_add_viewer (KP_VIEW (p_data->view),
                      KP_VIEW_MODEL (kp_chart_view_new ()));

  kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view),
                               KP_VIEW_MODEL_TYPE_MONTH);

  g_signal_connect (G_OBJECT (p_data->view), "view-set",
                    G_CALLBACK (kp_view_view_set), window);
  g_signal_connect (G_OBJECT (p_data->view), "date-set",
                    G_CALLBACK (kp_view_date_set), window);
  g_signal_connect (G_OBJECT (p_data->view), "viewer-set",
                    G_CALLBACK (kp_view_viewer_set), window);
  
  kp_statusbar_set_viewer_name (p_data->sbar,
    G_OBJECT_TYPE_NAME (kp_view_get_current_viewer (KP_VIEW (p_data->view))));
  

  if (GTK_WIDGET_CAN_FOCUS (GTK_WIDGET (p_data->cv)))
    gtk_widget_grab_focus (GTK_WIDGET (p_data->cv));

  /* Put widgets to private data */
  p_data->vbox = KP_W (xml, "mainwin_vbox");

  p_data->date_label = KP_W (xml, "date_label");
  p_data->toolbar_open_button = KP_W (xml, "toolbar_open_button");
  p_data->toolbar_save_button = KP_W (xml, "toolbar_save_button");
  p_data->toolbar_new_button = KP_W (xml, "toolbar_new_button");
  p_data->toolbar_log = KP_W (xml, "toolbar_log");
  
  p_data->menu_log_remove_entry = KP_W (xml, "menu_log_remove");
  p_data->menu_file_separator = KP_W (xml, "separator1");
  p_data->menu_save_log_button = KP_W (xml, "menu_file_save_log");
  p_data->menu_log = KP_W (xml, "menu_log");
  p_data->menu_import = KP_W (xml, "menu_file_import");
  p_data->menu_export = KP_W (xml, "menu_file_export");
  p_data->menubar = KP_W (xml, "menubar");
  p_data->view_menu = KP_W (xml, "menu_view_menu");
 
  p_data->toolbar_next_button = KP_W (xml, "toolbar_next_button");
  p_data->toolbar_prev_button = KP_W (xml, "toolbar_prev_button");
  
  kp_gui_module_signals_connect_data (xml, signals_data, window);

  g_return_if_fail (GTK_IS_TREE_VIEW (p_data->tree));
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (p_data->tree));
  gtk_widget_add_events (GTK_WIDGET (window), GDK_KEY_PRESS_MASK);
  g_return_if_fail (selection != NULL);

  /** Toolbar things **/

  /* Toolbar */
  menuitem = KP_W (xml, "menu_view_show_toolbar");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem),
                                  kp_settings_get_bool ("show_toolbar"));
  if (kp_settings_get_bool ("show_toolbar"))
    gtk_widget_show (p_data->toolbar_log);
  else
    gtk_widget_hide (p_data->toolbar_log);


  /* Statusbar */
  menuitem = KP_W (xml, "menu_view_show_sb");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem),
                                  kp_settings_get_bool ("show_statusbar"));
  if (kp_settings_get_bool ("show_statusbar"))
    gtk_widget_show (GTK_WIDGET (p_data->sbar));
  else
    gtk_widget_hide (GTK_WIDGET (p_data->sbar));
 
  /* Treeview */
  menuitem = KP_W (xml, "menu_view_show_tree");
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem),
                                  kp_settings_get_bool ("show_sidebar"));
  
  if (kp_settings_get_bool ("show_sidebar")) {
    width = kp_settings_get_int ("sidebar_width");

    if (kp_settings_get_bool ("save_window_geometry") && width > 0)
      gtk_paned_set_position (GTK_PANED (p_data->paned_window), width);

    gtk_widget_show (GTK_WIDGET (p_data->sw));
  }
  else
    gtk_widget_hide (GTK_WIDGET (p_data->sw));

  file = g_build_filename (KIPINA_PIXMAP_DIR, "runner.xpm", NULL);
  if ((icon = gdk_pixbuf_new_from_file (file, NULL)))
    gtk_window_set_icon (GTK_WINDOW (window), icon);
  g_free (file);
 
  if (kp_settings_get_bool ("save_window_geometry")) {
    width = kp_settings_get_int ("main_window_width");
    height = kp_settings_get_int ("main_window_height");
    
    if (width > 0 && height > 0)
      gtk_window_set_default_size (GTK_WINDOW (window), width, height);
  }
  
  g_signal_connect (G_OBJECT (p_data->cv), "day_selected",
                    G_CALLBACK (calendar_view_day_selected), window);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (workout_tree_selection_changed), window);
  g_signal_connect (G_OBJECT (window), "destroy",
                    G_CALLBACK (on_main_window_destroy), window);
  g_signal_connect (G_OBJECT (window), "key_press_event",
                    G_CALLBACK (main_window_key_press), window);
 
  kp_main_window_update_date_text (window);
  g_object_unref (G_OBJECT (xml));
}



static void
kp_main_window_finalize (GObject *object)
{
  KPMainWindow *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_MAIN_WINDOW (object));

  dialog = KP_MAIN_WINDOW (object);

  g_return_if_fail (dialog->private_data != NULL);
  g_free (dialog->private_data);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


/**
 * kp_main_window_new:
 * @file: Name of the log file to be opened or NULL.
 *
 * Just create a new kipinämain window.
 *
 * Returns: New KPMainWindow widget
 */
GtkWidget *
kp_main_window_new (void)
{
  KPMainWindow *window;

  if (_window)
    return NULL;
  
  window = g_object_new (KP_TYPE_MAIN_WINDOW, NULL);
  _window = window;

  kp_plugin_load_plugins ();
  
  update_toolbar_state (window);
  
  return GTK_WIDGET (window);
}

void
kp_main_window_set_log_file (KPMainWindow *window, const gchar *file)
{
  kp_debug ("File: %p", file);  
  kp_main_window_init_log (window, file);
  update_toolbar_state (window);
}

static gchar *
get_log_name (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  return p_data->logfile->str;
}

static gboolean
log_has_name (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  return (p_data->logfile->len > 0);
}


static void
update_window_title (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gchar buf[256];
  gchar mod[32];
  gchar *str;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  if (log_has_name (window))
    str = get_log_name (window);
  else
    str = _("Untitled");
 
  /* Put modified -tag to title if the log is modified */
  if (p_data->log_is_changed) {
    mod[0] = ' ';
    strncpy (&mod[1], _("[modified]"), sizeof (mod)-1-1);
  } else
    mod[0] = 0;
  
  g_snprintf (buf, sizeof (buf), "%s%s - Kipin\303\244", str, mod);
  gtk_window_set_title (GTK_WINDOW (window), buf);
}


static void
set_log_file (KPMainWindow *window, const gchar *name)
{
  KPMainWindowPrivateData *p_data;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  g_string_assign (p_data->logfile, (name) ? name : "");

  update_window_title (window);
}



static void
show_error_msg (KPMainWindow *window, const gchar *format, ...)
{
  KPMainWindowPrivateData *p_data;
  gchar buf[256];
  va_list args;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
 
  va_start (args, format);
  g_vsnprintf (buf, sizeof (buf), format, args);
  va_end (args);
  
  kp_gui_report_error (GTK_WINDOW (window), p_data->sbar, buf); 
}

/* The radio menu items can come outdated if some other part of the code
 * (like the treeview) sets the view type, but this way we can fix that.
 */
static void
update_menu_bar_view_item (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  KPViewModelType type;
  GList *list;
  GtkWidget *menu;
  gchar *w_name = NULL;
  
  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  type = kp_view_model_get_view_type (KP_VIEW_MODEL (p_data->view));

  switch (type)
  {
    case KP_VIEW_MODEL_TYPE_DAY:
      w_name = "menu_view_day";
      break;
    case KP_VIEW_MODEL_TYPE_WEEK:
      w_name ="menu_view_week";
      break;
    case KP_VIEW_MODEL_TYPE_MONTH:
      w_name ="menu_view_month";
      break;
    case KP_VIEW_MODEL_TYPE_YEAR:
      w_name ="menu_view_year";
      break;
    case KP_VIEW_MODEL_TYPE_ALL_TIME:
      w_name ="menu_view_all_time";
      break;
    default:
      g_assert_not_reached ();
  }
  kp_debug ("Widget name: %s", w_name);
  
  menu = p_data->view_menu;
  list = gtk_container_get_children (GTK_CONTAINER (menu));
  while (list) {
    if (strcmp (gtk_widget_get_name (GTK_WIDGET (list->data)), w_name) == 0) {
      gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (list->data), TRUE);
      break;
    }
    list = list->next;
  }
  g_list_free (list);
}



static void
kp_view_view_set (KPViewModel *model, KPViewModelType type,
                  KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_debug ("View set: %d", type);
  kp_main_window_update_date_text (window);
  kp_statusbar_set_view_type (p_data->sbar, type);

  update_menu_bar_view_item (window);
  update_toolbar_state (window);
}


static void
kp_view_date_set (KPViewModel *model, KPDate *date, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_main_window_update_date_text (window);

  kp_debug ("Date set: %d.%d.%d", date->d, date->m, date->y);
}

static void
kp_view_viewer_set (KPView *view, KPViewModel *model, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_statusbar_set_viewer_name (p_data->sbar, G_OBJECT_TYPE_NAME (model));
 
  kp_debug ("Viewer set.");
}


static void
kp_main_window_update_date_text (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  struct tm tm;
  GDate *date;
  gchar buf[128];
  gchar *format;
  gchar *tmp = NULL;
  gchar *str;
  KPDate d;

  g_return_if_fail (KP_IS_MAIN_WINDOW (window));
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_view_model_get_dmy (KP_VIEW_MODEL (p_data->view), &d.d, &d.m, &d.y);
  date = g_date_new_dmy (d.d, d.m, d.y);

  g_return_if_fail (date != NULL);

  format = kp_view_model_get_date_format (KP_VIEW_MODEL (p_data->view));
  
  g_date_to_struct_tm (date, &tm);
  strftime (buf, sizeof (buf) - 1, format, &tm);

  /*
   * If locale charset is not UTF-8, it can cause some trouble because
   * GTK+ needs valid UTF-8, so we need to check and convert if needed.
   */
  if (!g_utf8_validate (buf, -1, NULL)) {
    kp_debug ("Invalid utf-8 from strftime(), trying to convert..");
    
    if (!(tmp = g_locale_to_utf8 (buf, -1, NULL, NULL, NULL))) {
      kp_debug ("Can't convert, date str is now invalid, returning..", buf);
      return;
    }
    if (tmp) { 
      g_snprintf (buf, sizeof (buf) - 1, "%s", tmp);
      g_free (tmp);
    }
  }
  str = g_strdup_printf ("<span size=\"x-large\"><b>%s</b></span>", buf);

  if (!GTK_IS_LABEL (p_data->date_label))
    return;
  
  gtk_label_set_markup (GTK_LABEL (p_data->date_label), str);
  g_date_free (date);
  g_free (str);
}


static void
kp_main_window_init_log (KPMainWindow *window, const gchar *filename)
{
  KPMainWindowPrivateData *p_data;
  KPTrainingLog *log;
  KPLogStore *store;
  GError *err;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  
  log = kp_training_log_new ();

  /* If the filename was given, try to open the log on 
   * that name */
  if (filename) {
    err = NULL;
    if (kp_training_log_add_from_file (log, filename, &err)) {
      if (KP_IS_TRAINING_LOG (p_data->log))
        kp_main_window_deinit_log (window);
      set_log_file (window, filename);
      p_data->log = log;
      kp_statusbar_set_format_message (p_data->sbar, _("Log opened: \"%s\"."),
                                       filename);
    } else {
      g_object_unref (log);
      show_error_msg (window, (err) ? err->message : "Unknown error!");
      if (err)
        g_error_free (err);
      
      kp_debug ("Couldn't open the log, return!");
      
      return;
    }
  } else { 
    if (KP_IS_TRAINING_LOG (p_data->log))
      kp_main_window_deinit_log (window);
    
    set_log_file (window, NULL);
    p_data->log = log;
    kp_statusbar_set_message (p_data->sbar, _("New log file created."));
  }

  kp_debug ("TrainingLog has %u entries", kp_training_log_get_size (p_data->log));
  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));

  kp_statusbar_set_log (KP_STATUSBAR (p_data->sbar), p_data->log);
  log_connect_signals (p_data->log, window);
  kp_view_model_set_log (KP_VIEW_MODEL (p_data->view), p_data->log);
  
  p_data->log_is_changed = FALSE;

  /* Update views according to the new log */
  store = kp_log_store_new ();
  gtk_tree_view_set_model (GTK_TREE_VIEW (p_data->tree),
                           GTK_TREE_MODEL (store));
  kp_log_store_attach_log (store, p_data->log);
  kp_tree_view_set_log (p_data->tree, p_data->log);
}


static void
kp_main_window_deinit_log (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_view_model_unset_log (KP_VIEW_MODEL (p_data->view));
  kp_tree_view_unset_log (p_data->tree);
  kp_statusbar_unset_log (KP_STATUSBAR (p_data->sbar));

  log_disconnect_signals (p_data->log, window);

  set_log_file (window, NULL);
  g_object_unref (p_data->log);
}


static void
on_paned_handle_move (GObject *pane, GParamSpec *spec, gpointer data)
                    
{
  gint width;
  g_return_if_fail (GTK_IS_PANED (pane));
  
  width = gtk_paned_get_position (GTK_PANED (pane));
  kp_settings_set_int ("sidebar_width", width);
}

static void
on_toolbar_add_button_clicked (GtkButton * button, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  GtkWidget *dialog;
  KPDate date;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
 
  kp_view_model_get_dmy (KP_VIEW_MODEL (p_data->view),
                        &date.d, &date.m, &date.y);
  
  dialog = kp_new_workout_dialog_new (&date, p_data->log);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


static void
on_toolbar_add_s_button_clicked (GtkButton * button, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  GtkWidget *dialog;
  KPDate date;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
 
  kp_calendar_view_get_date (p_data->cv, &date.d, &date.m, &date.y);
  
  dialog = kp_new_split_workout_dialog_new (&date, p_data->log);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
}


static void
on_toolbar_comment_button_clicked (GtkButton *button, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  KPComment *comment; 
  GtkWidget *dialog;
  GDate *date;
  guint d, m, y;
 
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
  
  kp_calendar_view_get_date (p_data->cv, &d, &m, &y);
  date = g_date_new_dmy (d, m, y);
  comment = kp_comment_new ("", "");
  
  dialog = kp_new_comment_dialog_new (date, comment);
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);
  g_date_free (date);
 
  if (strlen (comment->title->str) > 0) 
    kp_training_log_add (p_data->log, KP_CALENDAR_ENTRY (comment));
  else 
    g_object_unref (G_OBJECT (comment));
}


static void
on_menu_remove_button_activate (GtkMenuItem *item, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  KPCalendarEntry *entry;
  const gchar *mark;
  guint d, m, y;
 
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  
  g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
  
  kp_calendar_view_get_date (p_data->cv, &d, &m, &y);
  mark = kp_calendar_view_get_current_mark (p_data->cv);
 
  /* There is no mark in that day */
  if (mark == NULL)
    return;
 
  entry = kp_training_log_get_entry (p_data->log, d, m, y, mark);
  g_return_if_fail (KP_IS_CALENDAR_ENTRY (entry));
    
  kp_training_log_remove (p_data->log, entry);
  
  kp_debug ("Removing: %s", mark);
}


static void
on_toolbar_html_button_clicked (GtkButton * button, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  GError *err;
  gchar *dir = kp_settings_get_str ("html_output_dir");

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  
  if (!p_data->log) {
    kp_gui_report_error (GTK_WINDOW (window),
                         p_data->sbar,
                      _("There is no training log to convert!"));
    return;
  }
  if (!dir) {
    g_warning (_("Setting 'html_output_dir' is missing, "
                 "can't create HTML statistics!"));
    return;
  }
  g_print(_("Generating HTML statistics.. (Output: %s)\n"), dir);
 
  g_assert (KP_IS_TRAINING_LOG (p_data->log));
  err = NULL;
  kp_training_log_create_html_stats (p_data->log, dir, &err);

  if (err) {
    kp_gui_report_error (GTK_WINDOW (window), p_data->sbar, err->message);
    g_error_free (err);
  }
}


static void
on_toolbar_cview_button_clicked (GtkWidget *widget, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  const gchar *name = gtk_widget_get_name (widget);
  KPViewModelType type;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  g_return_if_fail (name != NULL);

  /* If this is called for the previous active radio menu item, return */
  if (strncmp (name, "menu", 4) == 0
   && !gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)))
    return;
   
  if (strcmp (name, "toolbar_day_button") == 0
   || strcmp (name, "menu_view_day") == 0)
    type = KP_VIEW_MODEL_TYPE_DAY;
  else if (strcmp (name, "toolbar_week_button") == 0
        || strcmp (name, "menu_view_week") == 0)
    type = KP_VIEW_MODEL_TYPE_WEEK;
  else if (strcmp (name, "toolbar_month_button") == 0
        || strcmp (name, "menu_view_month") == 0)
    type = KP_VIEW_MODEL_TYPE_MONTH;
  else if (strcmp (name, "toolbar_year_button") == 0
        || strcmp (name, "menu_view_year") == 0)
    type = KP_VIEW_MODEL_TYPE_YEAR;
  else if (strcmp (name, "menu_view_all_time") == 0)
    type = KP_VIEW_MODEL_TYPE_ALL_TIME;
  else {
    type = 0; /* prevent compiler warning.. */
    g_assert_not_reached ();
  }
  
  kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view), type);
}


static void
on_main_window_destroy (GtkWidget *widget, KPMainWindow *window)
{
  gint width, height;

  if (kp_settings_get_bool ("save_window_geometry")) {
    gtk_window_get_size (GTK_WINDOW (widget), &width, &height);
  
    kp_settings_set_int ("main_window_width", width);
    kp_settings_set_int ("main_window_height", height);
  }    
    
  kp_settings_save ();
  gtk_widget_destroy (GTK_WIDGET (window));
  gtk_main_quit ();
}


/* 
 * HANDLERS FOR MENUBAR ITEMS 
 */
static void
on_menu_file_new_log_activate (GtkMenuItem *item, KPMainWindow *window)
{
  new_log (window);
}


static void
on_menu_file_open_log_activate (GtkMenuItem *item, KPMainWindow *window)
{
  open_log (window);
}


static void
on_menu_file_save_log_activate (GtkMenuItem *item, KPMainWindow *window)
{
  save_log (window);
}


static void
on_menu_file_save_log_as_activate (GtkMenuItem *item, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gchar *old;
  gchar *file;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  file = kp_gui_get_file_to_save (GTK_WINDOW (window));

  if (file) {
    old = g_strdup (p_data->logfile->str);
    set_log_file (window, file);
  } else
    return;

  if (!save_log (window)) {
    kp_statusbar_set_message (p_data->sbar, "Can't save the log!");
    set_log_file (window, old);
    g_free (old);
  }
}


static void
on_menu_file_quit_activate (GtkMenuItem *item, KPMainWindow *window)
{
  quit (window);
}


static void
menu_view_show_statusbar_toggled (GtkCheckMenuItem *item, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gboolean visible;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  (visible = gtk_check_menu_item_get_active (item))
    ? gtk_widget_show (GTK_WIDGET (p_data->sbar))
    : gtk_widget_hide (GTK_WIDGET (p_data->sbar));

  kp_settings_set_bool ("show_statusbar", visible);
}


static void
menu_view_show_toolbar_toggled (GtkCheckMenuItem *item, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gboolean visible;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  (visible = gtk_check_menu_item_get_active (item))
    ? gtk_widget_show (p_data->toolbar_log)
    : gtk_widget_hide (p_data->toolbar_log);

  kp_settings_set_bool ("show_toolbar", visible);
}


static void
menu_view_show_tree_toggled (GtkCheckMenuItem *item, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gboolean visible;
  gint width;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
 
  visible = gtk_check_menu_item_get_active (item);
  
  if (visible) {
    gtk_widget_show (GTK_WIDGET (p_data->sw));
    gtk_paned_add1 (GTK_PANED (p_data->paned_window), p_data->sw);
    g_object_unref (p_data->tree);
    g_object_unref (p_data->sw);

    width = kp_settings_get_int ("sidebar_width");
    /* Use 100 as default value */
    width = (width <= 0) ? 100 : width;
    
    gtk_paned_set_position (GTK_PANED (p_data->paned_window), width);
  } else {
    gtk_widget_hide (p_data->sw);
    g_object_ref (G_OBJECT (p_data->sw));
    g_object_ref (G_OBJECT (p_data->tree));
    gtk_container_remove (GTK_CONTAINER (p_data->paned_window), p_data->sw);
  }

  kp_settings_set_bool ("show_sidebar", visible);
}


static void
on_menu_view_activate (GtkRadioMenuItem *item, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  const gchar *name;
  
  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (item)))
    name = gtk_widget_get_name (GTK_WIDGET (item));
  else
    return;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  if (strcmp (name, "menu_view_statistics") == 0)
    kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), "KPStatsView");
  else if (strcmp (name, "menu_view_chart") == 0)
    kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), "KPChartView");
  else if (strcmp (name, "menu_view_calendar") == 0)
    kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), "KPCalendarView");
  else if (strcmp (name, "menu_view_list") == 0)
    kp_view_set_viewer_by_object_name (KP_VIEW (p_data->view), "KPListView");
  else
    g_return_if_reached ();
}


static void
on_menu_edit_pref_activate (GtkMenuItem *item, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  GtkWidget *dialog;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  dialog = kp_preferences_dialog_new ();
  gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  kp_statusbar_update (p_data->sbar);
  kp_statusbar_set_message (p_data->sbar, _("Settings saved."));
}


static void
on_menu_help_about_activate (GtkMenuItem *item, KPMainWindow *window)
{
  GladeXML *about = kp_gui_load ("about", "about_dialog");
  
  g_signal_connect (G_OBJECT (KP_W (about, "ok_button")),
                   "clicked",
                    G_CALLBACK (kp_gui_destroy_widget_passed),
                    KP_W (about, "about_dialog"));

  g_object_unref (about);
}


static void
on_menu_cal_navigation_activate (GtkMenuItem *item, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  KPViewModelType cv_type;
  const gchar *name = gtk_widget_get_name (GTK_WIDGET (item));
  GDate *date;
  guint d, m, y;
  guint direction; /* 0 back, 1 forward */
  guint months;
  guint days;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  g_assert (name != NULL);

  if (strcmp (name, "menu_cal_today") == 0) {
    date = g_date_new ();
    g_date_set_time (date, time(NULL));
    goto set;
  }

  kp_view_model_get_dmy (KP_VIEW_MODEL (p_data->view), &d, &m, &y);
  
  date = g_date_new_dmy (d, m, y);

  g_return_if_fail (date != NULL);
  g_return_if_fail (strlen (name) >= 16);

  direction = (strncmp (&name[9], "add", 3) == 0);
  
  if (strcmp (name, "toolbar_prev_button") == 0
   || strcmp (name, "toolbar_next_button") == 0) {

    direction = (strcmp (name, "toolbar_next_button") == 0);
    cv_type = kp_view_model_get_view_type (KP_VIEW_MODEL (p_data->view));
    
  } else {
    if (strncmp (&name[13], "day", 3) == 0)
      cv_type = KP_VIEW_MODEL_TYPE_DAY;
    else if (strncmp (&name[13], "wee", 3) == 0)
      cv_type = KP_VIEW_MODEL_TYPE_WEEK;
    else if (strncmp (&name[13], "mon", 3) == 0)
      cv_type = KP_VIEW_MODEL_TYPE_MONTH;
    else if (strncmp (&name[13], "yea", 3) == 0)
      cv_type = KP_VIEW_MODEL_TYPE_YEAR;
    else {
      g_print("NAME: %s (%s)\n", name, &name[13]);
      cv_type = 0; /* Prevent a compiler warning */
      g_assert_not_reached ();
    }
  }

  switch (cv_type)
  {
    case KP_VIEW_MODEL_TYPE_DAY:
      days = 1;
      months = 0;
      break;
    case KP_VIEW_MODEL_TYPE_WEEK:
      days = 7;
      months = 0;
      break;
    case KP_VIEW_MODEL_TYPE_MONTH:
      days = 0;
      months = 1;
      break;
    case KP_VIEW_MODEL_TYPE_YEAR:
      days = 0;
      months = 12;
      break;
    case KP_VIEW_MODEL_TYPE_ALL_TIME:
      return;
      
    default:
      days = 0;
      months = 0;
      g_assert_not_reached ();
  }

  if (direction == 1) {
    g_date_add_months (date, months);
    g_date_add_days (date, days);
  } else {
    g_date_subtract_months (date, months);
    g_date_subtract_days (date, days);
  }
    
set:
  kp_view_model_set_dmy (KP_VIEW_MODEL (p_data->view),
                         g_date_get_day (date),
                         g_date_get_month (date),
                         g_date_get_year (date));
  
  g_date_free (date);
}

/*
 * This is called when user selects a node in the sidebar tree.
 */
static void
workout_tree_selection_changed (GtkTreeSelection *selection,
                                KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  KPLogStoreRecordType type;
  KPViewModelType view_type;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint d, m, y;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_debug ("Selection changed.");
  
  if (gtk_tree_selection_get_selected (selection, &model, &iter))
    kp_log_store_get_date (KP_LOG_STORE (model), &iter, &d, &m, &y);
  else {
    kp_debug ("Can't get selected item, return.");
    return;
  }

  type = kp_log_store_get_iter_type (KP_LOG_STORE (model), &iter);

  switch (type)
  {
    case KP_LOG_STORE_REC_YEAR:
      view_type = KP_VIEW_MODEL_TYPE_YEAR;
      m = 1;
      d = 1;
      break;
      
    case KP_LOG_STORE_REC_MONTH:
      view_type = KP_VIEW_MODEL_TYPE_MONTH;
      d = 1;
      break;
      
    case KP_LOG_STORE_REC_DAY:
      view_type = KP_VIEW_MODEL_TYPE_DAY;
      break;

    case KP_LOG_STORE_REC_ROOT:
      view_type = KP_VIEW_MODEL_TYPE_ALL_TIME;
      kp_view_model_get_dmy (KP_VIEW_MODEL (p_data->view), &d, &m, &y);
      break;
      
    case KP_LOG_STORE_REC_ENTRY:
    case KP_LOG_STORE_REC_INVALID:
    default:
      return;
  }

  g_return_if_fail (g_date_valid_dmy (d, m, y));
 
  kp_view_model_set_dmy (KP_VIEW_MODEL (p_data->view), d, m, y);
  kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view), view_type);
}



/**
 * save_logfile:
 * @window: A #KPMainWindow.
 * @filename: The file to save the log into.
 * 
 * Saves the log to the file and does all necessary checks before
 * actually doing it.
 */
static gboolean
save_logfile (KPMainWindow *window, const gchar *filename)
{
  KPMainWindowPrivateData *p_data;
  gboolean save = TRUE; /* Save it ? */
  GError *err;
  gint resp;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_debug ("Saving continues..");
  
  if (!filename) {
    filename = kp_gui_get_file_to_save (GTK_WINDOW (window));
    if (!filename)
      return FALSE;
  }

  kp_debug ("Got filename: %s\n", filename);
  
  /* Give the user a warning if the file already exists */
  if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
   
    /* If the filename is the same old one, don't show the warning. */
    if ((log_has_name (window) == FALSE)
     || (log_has_name (window) == TRUE
       && strcmp (p_data->logfile->str, filename) != 0)) 
    {
      resp = kp_gui_get_yes_no_cancel (GTK_WINDOW (window),
                                     _("File exists! Overwrite?"));
      if (resp != GTK_RESPONSE_YES)
        save = FALSE;
    } 
  }
 
  if (save) {
    err = NULL;
    if (kp_training_log_save (p_data->log, filename, &err) == TRUE) {
      kp_statusbar_set_format_message (p_data->sbar,
                                     _("Log saved: \"%s\"."), filename);
      p_data->log_is_changed = FALSE;
      set_log_file (window, filename);
      return TRUE;
    }
    else {
      show_error_msg (window, _("Couldn't save the log to %s: %s!"),
                      filename,
                     (err) ? err->message : "");
      if (err)
         g_error_free (err);
    }
  }
  return FALSE;
}


static void
new_log (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gint response;
  gchar *file;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  if (KP_IS_TRAINING_LOG (p_data->log) && p_data->log_is_changed == TRUE) {
    response = kp_gui_get_yes_no_cancel (GTK_WINDOW (window),
                                    _("Do you want to save the current log "
                                      "before creating a new one?"));
    if (response == GTK_RESPONSE_YES) {
      /* Save the log before creating new one */
      file = kp_gui_get_file_to_save (GTK_WINDOW (window));

      kp_debug ("Got filename: %s", file);
      if (!file)
        return;
      
      /* Try to save the log. */
      if (!save_logfile (window, file))
        g_free (file);
    }
  }
  /* There is no log, so just create a new one. */
  kp_main_window_init_log (window, NULL);
  update_toolbar_state (window);
}


static void
quit (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gint response;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  if (p_data->log_is_changed) {
    response = kp_gui_get_yes_no_cancel (GTK_WINDOW (window),
                                       _("Save the changes?"));

    /* User wants to save the log */
    if (response == GTK_RESPONSE_YES) {
      /* Don't quit if the saving wasn't succesful */
      if (!save_log (window))
        return;
    }
    else if (response == GTK_RESPONSE_CANCEL) {
      return;
    }
  }
  g_signal_emit_by_name (G_OBJECT (window), "destroy");
}


static void
open_log (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gchar *save_file;
  gint response;
  GError *err;
  gchar *file;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
 
  file = kp_gui_get_file_to_open (GTK_WINDOW (window));
  if (file) {

    if (p_data->log_is_changed == FALSE)
      goto open_log;
    
    response = kp_gui_get_yes_no_cancel (GTK_WINDOW (window),
                                      _("Do you want to save the current log "
                                        "before creating a new one?"));

    /* User wants to save the log */
    if (response == GTK_RESPONSE_YES) {
      /* There is a log and it has a filename. */
      if (KP_IS_TRAINING_LOG (p_data->log) && p_data->logfile->len > 0) {
        /* FIXME: error reporting! */
        kp_training_log_save (p_data->log, p_data->logfile->str, NULL);
        goto open_log;
      }
      /* There is a log but it has no name, so ask the name to save it */
      else if (KP_IS_TRAINING_LOG (p_data->log)
            && log_has_name (window) == FALSE) {
        save_file = kp_gui_get_file_to_save (GTK_WINDOW (window));
        
        /* User selects a file, so save the log and create a new one */
        if (save_file) {
          err = NULL;
          if (!kp_training_log_save (p_data->log, save_file, &err)) {
            show_error_msg (window, _("Couldn't save the log to %s: %s!"),
                            save_file, (err) ? err->message : "");
            if (err)
              g_error_free (err);
            return;
          }
          goto open_log;
        }
        else
          return;
      }
    }
    /* User don't want to save the log, so just create a new one */
    else if (response == GTK_RESPONSE_NO)
      goto open_log;
    else
      return;
  }

  return;
  
open_log:
  kp_main_window_init_log (KP_MAIN_WINDOW (window), file);
  update_toolbar_state (window);
}


static gboolean
save_log (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gchar *filename = NULL;
  gboolean val = FALSE;

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  g_return_val_if_fail (KP_IS_TRAINING_LOG (p_data->log), FALSE);
      
  /* Try to get a filename to save the log into */
  if (log_has_name (window) == FALSE) {
    
    filename = kp_gui_get_file_to_save (GTK_WINDOW (window));
   
    if (filename == NULL) 
      return FALSE;
 
    if (!save_logfile (window, filename)) {
      g_free (filename);
      return FALSE;
    }
  }
  else
    val = save_logfile (window, p_data->logfile->str);

  update_toolbar_state (window);

  return val;
}

static void
log_connect_signals (KPTrainingLog *log, KPMainWindow *window)
{
  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), window);
  g_signal_connect (G_OBJECT (log), "entry-added",
                    G_CALLBACK (log_entry_added), window);
  g_signal_connect (G_OBJECT (log), "entry-removed",
                    G_CALLBACK (log_entry_removed), window);
}

static void
log_disconnect_signals (KPTrainingLog *log, KPMainWindow *window)
{
  g_signal_handlers_disconnect_by_func (log, log_entry_removed, window);
  g_signal_handlers_disconnect_by_func (log, log_entry_added, window);
  g_signal_handlers_disconnect_by_func (log, log_changed, window);
}

static void
log_entry_removed (KPTrainingLog *log, guint d, guint m, guint y,
                   const gchar *mark, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_statusbar_set_message (p_data->sbar, _("Entry removed from the log."));
}

static void
log_entry_added (KPTrainingLog *log, guint d, guint m, guint y,
                 const gchar *mark, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);

  kp_statusbar_set_message (p_data->sbar, _("New entry added to the log."));
}

static void
log_changed (KPTrainingLog *log, KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
 
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  p_data->log_is_changed = TRUE;

  update_window_title (window);
}

static void
calendar_view_day_selected (KPCalendarView *cv, KPDate *date,
                            KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  GList *entries;
  gchar buf[64];
  gchar *str;
  guint len;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  entries = kp_training_log_get_day (p_data->log, date->d, date->m, date->y);

  len = g_list_length (entries);
  
  if (len > 1)
    g_snprintf (buf, sizeof (buf)-1, _("%u Entries"), len);
  else if (len == 1)
    strncpy (buf, _("1 Entry"), sizeof (buf)-1);
  else
    strncpy (buf, _("No Entries"), sizeof (buf)-1);

  str = g_strdup_printf (_("%u.%u.%u (%s) selected."),
                           date->d, date->m, date->y, buf);
  
  kp_statusbar_set_message (p_data->sbar, str);

  update_toolbar_state (window);
  
  kp_date_free (date);
  g_free (str);
}


static void
update_toolbar_state (KPMainWindow *window)
{
  KPMainWindowPrivateData *p_data;
  gboolean all_time_set;
  gboolean day_has_mark;
  GList *buttons;
  gboolean log;

  g_return_if_fail (window != NULL);

  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (window);
  log = (p_data->log != NULL);

  all_time_set = (kp_view_model_get_view_type (KP_VIEW_MODEL (p_data->view))
              ==  KP_VIEW_MODEL_TYPE_ALL_TIME); 
  
  gtk_widget_set_sensitive (p_data->toolbar_next_button, !all_time_set);
  gtk_widget_set_sensitive (p_data->toolbar_prev_button, !all_time_set);
  
  day_has_mark = (kp_calendar_view_get_current_mark (p_data->cv) != NULL);
  gtk_widget_set_sensitive (p_data->menu_log_remove_entry, day_has_mark);
  
  for (buttons = gtk_container_get_children (GTK_CONTAINER (p_data->toolbar_log));
       buttons != NULL;
       buttons = buttons->next)
    gtk_widget_set_sensitive (GTK_WIDGET (buttons->data), log);

  for (buttons = gtk_container_get_children (GTK_CONTAINER (p_data->menu_log));
       buttons != NULL;
       buttons = buttons->next)
    gtk_widget_set_sensitive (GTK_WIDGET (buttons->data), log);
    
  gtk_widget_set_sensitive (GTK_WIDGET (p_data->toolbar_new_button), TRUE);
  gtk_widget_set_sensitive (GTK_WIDGET (p_data->toolbar_open_button), TRUE);

  if (log) 
    gtk_widget_show (p_data->paned_window);
  else 
    gtk_widget_hide (p_data->paned_window);
}


KPTrainingLog *
kp_main_window_get_log (void)
{
  KPMainWindowPrivateData *p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);
  g_return_val_if_fail (p_data != NULL, NULL);
 
  return p_data->log;
}

/*
  will be added to KPMainWindowPrivateData:

  menu_import
  menu_export
  menubar  

   
 */

GtkWidget *
kp_main_window_add_to_import_menu (GtkMenuItem *item)
{
  KPMainWindowPrivateData * p_data;
  GtkWidget *menu;
 
  g_return_val_if_fail (item != NULL, NULL);
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (_window);

  g_return_val_if_fail (p_data->menu_import != NULL, NULL);
  menu = gtk_menu_item_get_submenu (GTK_MENU_ITEM (p_data->menu_import));

  if (!menu) {
    menu = gtk_menu_new ();
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (p_data->menu_import), menu);
    gtk_widget_show (p_data->menu_file_separator);
  }
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), GTK_WIDGET (item));
  gtk_widget_show_all (p_data->menu_import);
  
  return NULL;
}

GtkWidget *
kp_main_window_add_to_export_menu (GtkMenuItem *item)
{
  /* TODO: implement */
  return NULL;
}

GtkWidget *
kp_main_window_add_to_menubar (GtkMenu *menu)
{
  /* TODO: implement */
  return NULL;
}



static gboolean
main_window_key_press (GtkWidget *widget, GdkEventKey *key, KPMainWindow window)
{
  KPMainWindowPrivateData *p_data;
  guint cv_type;
  
  p_data = KP_MAIN_WINDOW_PRIVATE_DATA (KP_MAIN_WINDOW (widget));
  cv_type = kp_view_model_get_view_type (KP_VIEW_MODEL (p_data->view));

  switch (key->keyval)
  {
    case GDK_KP_Left:
    case GDK_Left:
      if (key->state & GDK_CONTROL_MASK) {
        if (cv_type > 0)
          kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view),
                                       cv_type - 1);
        else
          kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view),
                                       KP_VIEW_MODEL_TYPE_N - 1);
      } else {
        return FALSE;
      }
      break;
      
    case GDK_KP_Right:
    case GDK_Right:
      if (key->state & GDK_CONTROL_MASK) {
        if (cv_type < KP_VIEW_MODEL_TYPE_N - 1)
          kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view),
                                       cv_type + 1);
        else
          kp_view_model_set_view_type (KP_VIEW_MODEL (p_data->view), 0);
      } else {
        return FALSE;
      }
      break;

    case GDK_KP_Up:
    case GDK_Up:
      if (key->state & GDK_CONTROL_MASK) {
        kp_view_set_viewer_next (KP_VIEW (p_data->view));
      } else {
        return FALSE;
      } 
      break;

    case GDK_KP_Down:
    case GDK_Down:
      if (key->state & GDK_CONTROL_MASK) {
        kp_view_set_viewer_prev (KP_VIEW (p_data->view));
      } else {
        return FALSE;
      }
      break;

    case GDK_s:
      if (key->state & GDK_CONTROL_MASK)
        save_log (KP_MAIN_WINDOW (widget));
    break;

    case GDK_o:
      if (key->state & GDK_CONTROL_MASK)
        open_log (KP_MAIN_WINDOW (widget));
    break;

    case GDK_n:
      if (key->state & GDK_CONTROL_MASK)
        new_log (KP_MAIN_WINDOW (widget));
    break;
      
    default:
      /* pass the event to some else place */
      return FALSE;
  }
  return TRUE;
}

