/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <cstdio>   // for file operations
#include <string>
#include <sstream>
#include <iostream>
#include <fstream>
#include <gcrypt.h>
#include <cerrno>
#include <cassert>

#include "diary.hpp"
#include "helpers.hpp"
#include "lifeograph.hpp"


using namespace LIFEO;


// STATIC MEMBERS
Diary*                  Diary::d;
#if LIFEOGRAPH_DEBUG_BUILD
bool                    Diary::s_flag_ignore_locks( true );
#else
bool                    Diary::s_flag_ignore_locks( false );
#endif
ElementShower< Diary >* Diary::shower( NULL );

static const char   DEFAULT_CHAPTER_CTG_NAME[]  = N_( "Default" );

// PARSING HELPERS
Date
get_db_line_date( const Glib::ustring& line )
{
    Date date( 0 );

    for( unsigned int i = 2;
         i < line.size() && i < 12 && int ( line[ i ] ) >= '0' && int ( line[ i ] ) <= '9';
         i++ )
    {
        date.m_date = ( date.m_date * 10 ) + int ( line[ i ] ) - '0';
    }

    return( date );
}

Glib::ustring
get_db_line_name( const Glib::ustring& line )
{
    Glib::ustring::size_type begin( line.find( '\t' ) );
    if( begin == std::string::npos )
        begin = 2;
    else
        begin++;

    return( line.substr( begin ) );
}

// DIARY ===========================================================================================
Diary::Diary()
:   DiaryElement( this, ES::VOID ), m_path( "" ), m_passphrase( "" ),
    m_ptr2chapter_ctg_cur( NULL ), m_topics( this, "topics" ), m_todo_groups( this, "todogrps" ),
    m_startup_elem_id( HOME_CURRENT_ELEM ), m_last_elem_id( DEID_MIN ),
    m_option_sorting_criteria( SC_DATE ), m_read_version( 0 ),
    m_flag_read_only( false ), m_search_text( "" )
{
    m_filter_active = new Filter( NULL, _( "Active Filter" ) );
    m_filter_default = new Filter( NULL, "Default Filter" );
}

Diary::~Diary()
{
    remove_lock();
}

Result
Diary::init_new( const std::string& path )
{
    clear();
    Result result( set_path( path, true ) );

    if( result != LIFEO::SUCCESS )
    {
        clear();
        return result;
    }

    // every diary must at least have one chapter category:
    m_ptr2chapter_ctg_cur = create_chapter_ctg( _( DEFAULT_CHAPTER_CTG_NAME ) );

    add_today();
    
    return LIFEO::SUCCESS;
}

void
Diary::clear()
{
    if( remove_lock() )
        m_path.clear();

    m_read_version = 0;
    DiaryBase::clear();         // clear IDs
    create_new_id( this );      // add DEID_MIN back to IDs pool
    create_new_id( &m_topics ); // FIXME: do topics really need a deid?
    m_entries.clear();
    m_tags.clear();
    m_tag_categories.clear();
    m_chapter_categories.clear();
    m_topics.clear();
    m_todo_groups.clear();

    m_ptr2chapter_ctg_cur = NULL;
    m_untagged.get_items()->clear();
    m_untagged.reset_theme();
    m_startup_elem_id = HOME_CURRENT_ELEM;
    m_last_elem_id = DEID_MIN;

    m_passphrase.clear();

    m_search_text.clear();
    m_filter_default->reset();
    m_filter_active->reset();

    // NOTE: only reset body options here:
    m_language.clear();
    m_option_sorting_criteria = SC_DATE;
}

const Glib::RefPtr< Gdk::Pixbuf >&
Diary::get_icon() const
{
    return Lifeograph::icons->diary_16;
}
const Glib::RefPtr< Gdk::Pixbuf >&
Diary::get_icon32() const
{
    return Lifeograph::icons->diary_32;
}

LIFEO::Result
Diary::set_path( const std::string& path, bool new_file, bool read_only )
{
    // CHECK FILE SYSTEM PERMISSIONS
    if( access( path.c_str(), F_OK ) != 0 ) // check existence
    {
        if( errno == ENOENT )
        {
            if( !new_file )
            {
                PRINT_DEBUG( "File is not found" );
                return LIFEO::FILE_NOT_FOUND;
            }
        }
        else
            return LIFEO::FAILURE; // should not be the case
    }
    else if( access( path.c_str(), R_OK ) != 0 ) // check read access
    {
        PRINT_DEBUG( "File is not readable" );
        return LIFEO::FILE_NOT_READABLE;
    }
    else if( access( path.c_str(), W_OK ) != 0 ) // check write access
    {
        // errno == EACCES or errno == EROFS but no need to bother
        PRINT_DEBUG( "File is not writable, opening read-only..." );
        read_only = true;
    }

    // CHECK LOCK
    std::string path_lock( path + LOCK_SUFFIX );
    if( Glib::file_test( path_lock, Glib::FILE_TEST_EXISTS ) )
    {
        if( s_flag_ignore_locks )
            print_info( "Ignored file lock" );
        else
        if( ! read_only )
            return LIFEO::FILE_LOCKED;
    }

    // REMOVE PREVIOUS LOCK IF ANY
    if( ! new_file ) // init_new() clears diary first, so not necessary there
        remove_lock(); // remove existing lock

    // ACCEPT PATH
    m_path = path;
    m_name = Glib::filename_display_basename( path );
    m_flag_read_only = read_only;

    // "TOUCH" THE NEW LOCK
    if( ! read_only && ! new_file )
    {
        FILE *fp = fopen( path_lock.c_str(), "a+" );
        if( fp )
            fclose( fp );
    }

    return LIFEO::SUCCESS;
}

const std::string&
Diary::get_path() const
{
    return m_path;
}

bool
Diary::is_path_set() const
{
    return ( (bool) m_path.size() );
}

bool
Diary::set_passphrase( const std::string& passphrase )
{
    if( passphrase.size() >= PASSPHRASE_MIN_SIZE )
    {
        m_passphrase = passphrase;
        return true;
    }
    else
        return false;
}

void
Diary::clear_passphrase()
{
    m_passphrase.clear();
}

std::string
Diary::get_passphrase() const
{
    return m_passphrase;
}

bool
Diary::compare_passphrase( const std::string& passphrase ) const
{
    return( m_passphrase == passphrase );
}

bool
Diary::is_passphrase_set() const
{
    return (bool) m_passphrase.size();
}

LIFEO::Result
Diary::read_header()
{
    std::ifstream file( m_path.c_str() );

    if( ! file.is_open() )
    {
        print_error( "failed to open diary file " + m_path );
        return LIFEO::COULD_NOT_START;
    }
    std::string line;

    getline( file, line );

    if( line != LIFEO::DB_FILE_HEADER )
    {
        file.close();
        return LIFEO::CORRUPT_FILE;
    }

    while( getline( file, line ) )
    {
        switch( line[ 0 ] )
        {
            case 'V':
                m_read_version = convert_string( line.substr( 2 ) );
                if( m_read_version < LIFEO::DB_FILE_VERSION_INT_MIN )
                {
                    file.close();
                    return LIFEO::INCOMPATIBLE_FILE_OLD;
                }
                else if( m_read_version > LIFEO::DB_FILE_VERSION_INT )
                {
                    file.close();
                    return LIFEO::INCOMPATIBLE_FILE_NEW;
                }
                break;
            case 'E':
                if( line[ 2 ] == 'y' )
                    // passphrase is set to a dummy value to indicate that diary
                    // is an encrypted one until user enters the real passphrase
                    m_passphrase = " ";
                else
                    m_passphrase.clear();
                break;
            case 0: // end of header
                m_body_position = file.tellg();
                file.close();
                return LIFEO::SUCCESS;
            default:
                print_error( "unrecognized header line: " + line );
                break;
        }
    }

    file.close();
    m_path = "";
    return LIFEO::CORRUPT_FILE;
}

LIFEO::Result
Diary::read_body()
{
    if( m_passphrase.empty() )
        return read_plain();
    else
        return read_encrypted();
}

LIFEO::Result
Diary::read_plain()
{
    std::ifstream file( m_path.c_str() );

    if( ! file.is_open() )
    {
        print_error( "failed to open diary file " + m_path );
        return LIFEO::COULD_NOT_START;
    }

    file.seekg( 0, std::ios::end );

    int readsize = file.tellg() - m_body_position - 1;

    if( readsize <= 0 )
    {
        print_info( "diary has no body" );
        file.close();
        return LIFEO::CORRUPT_FILE;
    }

    file.seekg( m_body_position );

    char *readbuffer = new char[ readsize + 1 ];
    // TODO: check validity

    file.read( readbuffer, readsize );

    readbuffer[ readsize ] = 0;   // terminating zero

    file.close();

    std::stringstream stream( readbuffer );

    delete[] readbuffer;

    return parse_db_body_text( stream );
}

LIFEO::Result
Diary::read_encrypted()
{
    std::ifstream file( m_path.c_str(), std::ios::in | std::ios::binary );

    if( ! file.is_open() )
    {
        print_error( "Failed to open diary file" + m_path );
        return LIFEO::COULD_NOT_START;
    }
    file.seekg( m_body_position );

    std::string line, content;
    CipherBuffers buf;

    try
    {
        // Allocate memory for salt
        buf.salt = new unsigned char[ LIFEO::Cipher::cSALT_SIZE ];
        if( ! buf.salt )
            throw LIFEO::Error( "Unable to allocate memory for salt" );
        // Read salt value
        file.read( ( char* ) buf.salt, LIFEO::Cipher::cSALT_SIZE );

        buf.iv = new unsigned char[ LIFEO::Cipher::cIV_SIZE ];
        if( ! buf.iv )
            throw LIFEO::Error( "Unable to allocate memory for IV" );
        // Read IV
        file.read( ( char* ) buf.iv, LIFEO::Cipher::cIV_SIZE );

        LIFEO::Cipher::expand_key( m_passphrase.c_str(), buf.salt, &buf.key );

        // Calulate bytes of data in file
        size_t size = LIFEO::get_file_size( file ) - file.tellg();
        if( size <= 3 )
        {
            file.close();
            buf.clear();
            return LIFEO::CORRUPT_FILE;
        }
        buf.buffer = new unsigned char[ size + 1 ];
        if( ! buf.buffer )
            throw LIFEO::Error( "Unable to allocate memory for buffer" );

        file.read( ( char* ) buf.buffer, size );
        LIFEO::Cipher::decrypt_buffer( buf.buffer, size, buf.key, buf.iv );
        file.close();

        // passphrase check
        if( buf.buffer[ 0 ] != m_passphrase[ 0 ] && buf.buffer[ 1 ] != '\n' )
        {
            buf.clear();
            return LIFEO::WRONG_PASSWORD;
        }

        buf.buffer[ size ] = 0;   // terminating zero
    }
    catch( ... )
    {
        buf.clear();
        return LIFEO::COULD_NOT_START;
    }

    std::stringstream stream;
    buf.buffer += 2;    // ignore first two chars which are for passphrase checking
    stream << buf.buffer;
    buf.buffer -= 2;    // restore pointer to the start of the buffer before deletion

    buf.clear();
    return parse_db_body_text( stream );
}

inline LIFEO::Result
Diary::parse_db_body_text( std::stringstream& stream )
{
    if( m_read_version == 1010 || m_read_version == 1011 )
        return parse_db_body_text_1010( stream );
    else if( m_read_version == 56 )
        return parse_db_body_text_56( stream );
    else
        return parse_db_body_text_110( stream );
}

LIFEO::Result
Diary::parse_db_body_text_1010( std::stringstream& stream )
{
    std::string         line( "" );
    Entry*              entry_new = NULL;
    CategoryChapters*   ptr2chapter_ctg = NULL;
    Chapter*            ptr2chapter = NULL;
    CategoryTags*       ptr2tag_ctg = NULL;
    Tag*                ptr2tag = NULL;
    bool                flag_first_paragraph( false );

    // TAG DEFINITIONS & CHAPTERS
    while( getline( stream, line ) )
    {
        if( line[ 0 ] == 0 )    // end of section
            break;
        else
        if( line.size() < 3 )
            continue;
        else
        {
            switch( line[ 0 ] )
            {
                case 'I':   // id
                    set_force_id( LIFEO::convert_string( line.substr( 2 ) ) );
                break;
                // TAGS
                case 'T':   // tag category
                    ptr2tag_ctg = create_tag_ctg( line.substr( 2 ) );
                    ptr2tag_ctg->set_expanded( line[ 1 ] == 'e' );
                    break;
                case 't':   // tag
                    ptr2tag = create_tag( line.substr( 2 ), ptr2tag_ctg );
                    break;
                case 'u':
                    ptr2tag = &m_untagged;
                    // no break
                case 'm':
                    if( ptr2tag == NULL )
                    {
                        print_error( "No tag declared for theme" );
                        break;
                    }
                    switch( line[ 1 ] )
                    {
                        case 'f':   // font
                            ptr2tag->get_own_theme()->font =
                                    Pango::FontDescription( line.substr( 2 ) );
                            break;
                        case 'b':   // base color
                            ptr2tag->get_own_theme()->color_base.set( line.substr( 2 ) );
                            break;
                        case 't':   // text color
                            ptr2tag->get_own_theme()->color_text.set( line.substr( 2 ) );
                            break;
                        case 'h':   // heading color
                            ptr2tag->get_own_theme()->color_heading.set( line.substr( 2 ) );
                            break;
                        case 's':   // subheading color
                            ptr2tag->get_own_theme()->color_subheading.set( line.substr( 2 ) );
                            break;
                        case 'l':   // highlight color
                            ptr2tag->get_own_theme()->color_highlight.set( line.substr( 2 ) );
                            break;
                    }
                    break;
                case 'f':   // default filter
                    switch( line[ 1 ] )
                    {
                        case 's':   // status
                            if( line.size() < 9 )
                            {
                                print_error( "status filter length error" );
                                continue;
                            }
                            m_filter_default->set_trash( line[ 2 ] == 'T', line[ 3 ] == 't' );
                            m_filter_default->set_favorites( line[ 4 ] == 'F', line[ 5 ] == 'f' );
                            m_filter_default->set_todo(
                                    line[ 6 ] == 'T', line[ 7 ] == 'D', line[ 8 ] == 'C' );
                            break;
                        case 't':   // tag
                        {
                            PoolTags::iterator iter_tag( m_tags.find( line.substr( 2 ) ) );
                            if( iter_tag != m_tags.end() )
                                m_filter_default->set_tag( iter_tag->second );
                            else
                                print_error( "Reference to undefined tag: " + line.substr( 2 ) );
                            break;
                        }
                        case 'b':   // begin date
                            m_filter_default->set_date_begin(
                                    LIFEO::convert_string( line.substr( 2 ) ) );
                            break;
                        case 'e':   // end date
                            m_filter_default->set_date_end(
                                    LIFEO::convert_string( line.substr( 2 ) ) );
                            break;
                    }
                    break;
                case 'o':   // ordinal chapter (topic)
                    ptr2chapter = m_topics.create_chapter(
                            get_db_line_name( line ), get_db_line_date( line ) );
                    ptr2chapter->set_expanded( line[ 1 ] == 'e' );
                    break;
                case 'd':   // to-do group
                    if( line[ 1 ] == ':' ) // declaration
                    {
                        ptr2chapter = m_todo_groups.create_chapter(
                                get_db_line_name( line ), get_db_line_date( line ) );
                    }
                    else // options
                    {
                        ptr2chapter->set_expanded( line[ 2 ] == 'e' );
                        if( line[ 3 ] == 'd' )
                            ptr2chapter->set_todo_status( ES::DONE );
                        else if( line[ 3 ] == 'c' )
                            ptr2chapter->set_todo_status( ES::CANCELED );
                    }
                    break;
                case 'C':   // chapter category
                    ptr2chapter_ctg = create_chapter_ctg( line.substr( 2 ) );
                    if( line[ 1 ] == 'c' )
                        m_ptr2chapter_ctg_cur = ptr2chapter_ctg;
                    break;
                case 'c':   // chapter
                    if( ptr2chapter_ctg == NULL )
                    {
                        print_error( "No chapter category defined" );
                        break;
                    }
                    ptr2chapter = ptr2chapter_ctg->create_chapter(
                            get_db_line_name( line ), get_db_line_date( line ) );
                    ptr2chapter->set_expanded( line[ 1 ] == 'e' );
                    break;
                case 'O':   // options
                    m_option_sorting_criteria = line[ 2 ];
                    break;
                case 'l':   // language
                    m_language = line.substr( 2 );
                    break;
                case 'S':   // startup action
                    m_startup_elem_id = LIFEO::convert_string( line.substr( 2 ) );
                    break;
                case 'L':
                    m_last_elem_id = LIFEO::convert_string( line.substr( 2 ) );
                    break;
                default:
                    print_error( "unrecognized line:\n" + line );
                    return LIFEO::CORRUPT_FILE;
            }
        }
    }

    // ENTRIES
    while( getline( stream, line ) )
    {
        if( line.size() < 2 )
            continue;

        switch( line[ 0 ] )
        {
            case 'I':
                set_force_id( LIFEO::convert_string( line.substr( 2 ) ) );
                break;
            case 'E':   // new entry
            case 'e':   // trashed
                if( line.size() < 5 )
                    continue;

                entry_new = new Entry( this, LIFEO::convert_string( line.substr( 4 ) ),
                                       line[ 1 ] == 'f' );

                m_entries[ entry_new->m_date.m_date ] = entry_new;

                if( line[ 0 ] == 'e' )
                    entry_new->set_trashed( true );
                if( line[ 2 ] == 'h' )
                    m_filter_default->add_entry( entry_new );
                if( line[ 3 ] == 'd' )
                    entry_new->set_todo_status( ES::DONE );
                else if( line[ 3 ] == 'c' )
                    entry_new->set_todo_status( ES::CANCELED );

                flag_first_paragraph = true;
                break;
            case 'D':   // creation & change dates (optional)
                if( entry_new == NULL )
                {
                    print_error( "No entry declared" );
                    break;
                }
                if( line[ 1 ] == 'r' )
                    entry_new->m_date_created = LIFEO::convert_string( line.substr( 2 ) );
                else    // it should be 'h'
                    entry_new->m_date_changed = LIFEO::convert_string( line.substr( 2 ) );
                break;
            case 'T':   // tag
                if( entry_new == NULL )
                    print_error( "No entry declared" );
                else
                {
                    PoolTags::iterator iter_tag( m_tags.find( line.substr( 2 ) ) );
                    if( iter_tag != m_tags.end() )
                    {
                        entry_new->add_tag( iter_tag->second );
                        if( line[1] == 'T' )
                            entry_new->set_theme_tag( iter_tag->second );
                    }
                    else
                        print_error( "Reference to undefined tag: " + line.substr( 2 ) );
                }
                break;
            case 'l':   // language
                if( entry_new == NULL )
                    print_error( "No entry declared" );
                else
                    entry_new->set_lang( line.substr( 2 ) );
                break;
            case 'P':    // paragraph
                if( entry_new == NULL )
                {
                    print_error( "No entry declared" );
                    break;
                }
                if( flag_first_paragraph )
                {
                    if( line.size() > 2 )
                        entry_new->m_text = line.substr( 2 );
                    entry_new->m_name = entry_new->m_text;
                    flag_first_paragraph = false;
                }
                else
                {
                    entry_new->m_text += "\n";
                    entry_new->m_text += line.substr( 2 );
                }
                break;
            default:
                print_error( "unrecognized line:\n" + line );
                return LIFEO::CORRUPT_FILE;
                break;
        }
    }

    // every diary must at least have one chapter category:
    if( m_chapter_categories.empty() )
        m_ptr2chapter_ctg_cur = create_chapter_ctg( _( DEFAULT_CHAPTER_CTG_NAME ) );

    if( m_startup_elem_id > HOME_FIXED_ELEM )
        if( get_element( m_startup_elem_id ) == NULL )
        {
            print_error( "startup element cannot be found in db" );
            m_startup_elem_id = DEID_MIN;
        }

    if( m_entries.size() < 1 )
    {
        print_info( "a dummy entry added to the diary" );
        add_today();
    }

    m_filter_active->set( m_filter_default );

    return LIFEO::SUCCESS;
}

LIFEO::Result
Diary::parse_db_body_text_110( std::stringstream& stream )
{
    std::string         line( "" );
    Entry*              entry_new = NULL;
    CategoryChapters*   ptr2chapter_ctg = NULL;
    Chapter*            ptr2chapter = NULL;
    CategoryTags*       ptr2tag_ctg = NULL;
    Theme*              ptr2theme = NULL;
    Theme*              ptr2default_theme = NULL;
    bool                flag_first_paragraph( false );

    // add tag for system theme
    create_tag( "[ - 0 - ]", NULL )->get_own_theme();

    // TAG DEFINITIONS & CHAPTERS
    while( getline( stream, line ) )
    {
        if( line[ 0 ] == 0 )    // end of section
            break;
        else
        if( line.size() < 3 )
            continue;
        else
        {
            switch( line[ 0 ] )
            {
                case 'I':
                    set_force_id( LIFEO::convert_string( line.substr( 2 ) ) );
                break;
                case 'T':   // tag category
                    ptr2tag_ctg = create_tag_ctg( line.substr( 2 ) );
                    ptr2tag_ctg->set_expanded( line[ 1 ] == 'e' );
                    break;
                case 't':   // tag
                    create_tag( line.substr( 2 ), ptr2tag_ctg );
                    break;
                case 'C':   // chapter category
                    ptr2chapter_ctg = create_chapter_ctg( line.substr( 2 ) );
                    if( line[ 1 ] == 'c' || m_ptr2chapter_ctg_cur == NULL ) // for v73
                        m_ptr2chapter_ctg_cur = ptr2chapter_ctg;
                    break;
                case 'o':   // ordinal chapter (topic)
                    ptr2chapter = m_topics.create_chapter(
                            get_db_line_name( line ), get_db_line_date( line ) );
                    ptr2chapter->set_expanded( line[ 1 ] == 'e' );
                    break;
                case 'c':   // chapter
                    if( ptr2chapter_ctg == NULL )
                    {
                        print_error( "No chapter category defined" );
                        break;
                    }
                        if( m_read_version < 74 )
                            ptr2chapter = new Chapter( this, line.substr( 2 ) );
                        else
                            ptr2chapter = ptr2chapter_ctg->create_chapter(
                                get_db_line_name( line ), get_db_line_date( line ) );
                        ptr2chapter->set_expanded( line[ 1 ] == 'e' );
                    break;
                case 'd':   // chapter begin date (v73)
                    if( ptr2chapter != NULL )   // chapter begin date
                    {
                        Date date( LIFEO::convert_string( line.substr( 2 ) ) );
                        ptr2chapter_ctg->set_chapter_date( ptr2chapter, date.m_date );
                    }
                    break;
                case 'M':
                    {
                        // themes with same name as tags are merged into existing tags
                        ptr2theme = create_tag( line.substr( 2 ) )->get_own_theme();

                        if( line[ 1 ] == 'd' )
                            ptr2default_theme = ptr2theme;
                    }
                    break;
                case 'm':
                    if( ptr2theme == NULL )
                    {
                        print_error( "No theme declared" );
                        break;
                    }
                    switch( line[ 1 ] )
                    {
                        case 'f':   // font
                            ptr2theme->font = Pango::FontDescription( line.substr( 2 ) );
                            break;
                        case 'b':   // base color
                            ptr2theme->color_base.set( line.substr( 2 ) );
                            break;
                        case 't':   // text color
                            ptr2theme->color_text.set( line.substr( 2 ) );
                            break;
                        case 'h':   // heading color
                            ptr2theme->color_heading.set( line.substr( 2 ) );
                            break;
                        case 's':   // subheading color
                            ptr2theme->color_subheading.set( line.substr( 2 ) );
                            break;
                        case 'l':   // highlight color
                            ptr2theme->color_highlight.set( line.substr( 2 ) );
                            break;
                    }
                    break;
                case 'O':   // options
                    if( line.size() < 4 )
                        break;
                    if ( line[ 2 ] == 's' ) // for versions prior to 110
                        m_language = HELPERS::get_env_lang();
                    m_option_sorting_criteria = line[ 3 ];
                    break;
                case 'l':   // language
                    m_language = line.substr( 2 );
                    break;
                case 'S':   // startup action
                    m_startup_elem_id = LIFEO::convert_string( line.substr( 2 ) );
                    break;
                case 'L':
                    m_last_elem_id = LIFEO::convert_string( line.substr( 2 ) );
                    break;
                default:
                    print_error( "unrecognized line:\n" + line );
                    return LIFEO::CORRUPT_FILE;
            }
        }
    }

    // ENTRIES
    while( getline( stream, line ) )
    {
        if( line.size() < 2 )
            continue;

        switch( line[ 0 ] )
        {
            case 'I':
                set_force_id( LIFEO::convert_string( line.substr( 2 ) ) );
                break;
            case 'E':   // new entry
            case 'e':   // trashed
                entry_new = new Entry( this, LIFEO::convert_string( line.substr( 2 ) ),
                                       line[ 1 ] == 'f' );
                if( m_read_version < 74 )
                    entry_new->m_date.m_date++; // entry orders should start from 1
                m_entries[ entry_new->m_date.m_date ] = entry_new;
                if( line[ 0 ] == 'e' )
                {
                    entry_new->set_trashed( true );
                    // trashed entries are always hidden at the login
                    entry_new->set_filtered_out( true );
                }
                flag_first_paragraph = true;
                break;
            case 'D':   // creation & change dates (optional)
                if( entry_new == NULL )
                {
                    print_error( "No entry declared" );
                    break;
                }
                if( line[ 1 ] == 'r' )
                    entry_new->m_date_created = LIFEO::convert_string( line.substr( 2 ) );
                else    // it should be 'h'
                    entry_new->m_date_changed = LIFEO::convert_string( line.substr( 2 ) );
                break;
            case 'M':   // themes are converted into tags
            case 'T':   // tag
                if( entry_new == NULL )
                    print_error( "No entry declared" );
                else
                {
                    PoolTags::iterator iter_tag( m_tags.find( line.substr( 2 ) ) );
                    if( iter_tag != m_tags.end() )
                        entry_new->add_tag( iter_tag->second );
                    else
                        print_error( "Reference to undefined tag: " + line.substr( 2 ) );
                }
                break;
            case 'l':   // language
                if( entry_new == NULL )
                    print_error( "No entry declared" );
                else
                    entry_new->set_lang( line.substr( 2 ) );
                break;
            case 'P':    // paragraph
                if( entry_new == NULL )
                {
                    print_error( "No entry declared" );
                    break;
                }
                if( flag_first_paragraph )
                {
                    if( line.size() > 2 )
                        entry_new->m_text = line.substr( 2 );
                    entry_new->m_name = entry_new->m_text;
                    flag_first_paragraph = false;
                }
                else
                {
                    entry_new->m_text += "\n";
                    entry_new->m_text += line.substr( 2 );
                }
                break;
            default:
                print_error( "unrecognized line (110):\n" + line );
                return LIFEO::CORRUPT_FILE;
                break;
        }
    }

    // every diary must at least have one chapter category:
    if( m_chapter_categories.empty() )
        m_ptr2chapter_ctg_cur = create_chapter_ctg( _( DEFAULT_CHAPTER_CTG_NAME ) );

    if( m_startup_elem_id > HOME_FIXED_ELEM )
        if( get_element( m_startup_elem_id ) == NULL )
        {
            print_error( "startup element cannot be found in db" );
            m_startup_elem_id = DEID_MIN;
        }

    if( m_entries.size() < 1 )
    {
        print_info( "a dummy entry added to the diary" );
        add_today();
    }

    // if default theme is different than the system theme, set the untagged accordingly
    if( ptr2default_theme )
    {
        m_untagged.create_own_theme_duplicating( ptr2default_theme );
    }

    return LIFEO::SUCCESS;
}

LIFEO::Result
Diary::parse_db_body_text_56( std::stringstream& stream )
{
    std::string         line( "" );
    Glib::ustring       content( "" );
    Date::date_t        date( 0 );
    bool                flag_favorite( false );
    Entry               *entry_new = NULL;
    CategoryChapters    *ptr2chapter_ctg = NULL;
    Chapter             *ptr2chapter = NULL;
    std::vector< Tag* > tmp_vector_tags;
    std::string         str_tags;

    // TAG DEFINITIONS & CHAPTERS
    while( getline( stream, line ) )
    {
        if( line[ 0 ] == 0 )
            break;
        else
        if( line.size() < 3 )
            continue;
        else
        {
            switch( line[ 0 ] )
            {
                case 'T':   // tag
                    try
                    {
                        Tag *tmp_tag = create_tag( line.substr( 2 ) );
                        tmp_vector_tags.push_back( tmp_tag );
                        // TODO: this is a temporary solution. will be fixed in new diary version
                    }
                    catch( std::exception &ex )
                    {
                        throw LIFEO::Error( ex.what() );
                    }
                    break;
                case 'L':   // current entryliststyle
                    // not used anymore...
                    break;
                case 'S':   // chapterset
                    try
                    {
                        ptr2chapter_ctg = create_chapter_ctg( line.substr( 2 ) );
                        if( m_ptr2chapter_ctg_cur == NULL )
                            m_ptr2chapter_ctg_cur = ptr2chapter_ctg;
                    }
                    catch( std::exception &ex )
                    {
                        throw LIFEO::Error( ex.what() );
                    }
                    break;
                case 'C':   // chapter
                    if( m_chapter_categories.size() > 0 )
                    {
                        try
                        {
                            ptr2chapter = new Chapter( this, line.substr( 2 ) );
                        }
                        catch( std::exception &ex )
                        {
                            throw LIFEO::Error( ex.what() );
                        }
                    }
                    break;
                case 'd':   // begin date
                    if( ptr2chapter != NULL )   // chapter begin date
                    {
                        Date date( line.substr( 2 ) );
                        ptr2chapter_ctg->set_chapter_date( ptr2chapter, date.m_date );
                    }
                    break;
                case 'O':   // options
                    if ( line[ 2 ] == 's' )
                        m_language = HELPERS::get_env_lang();
                    break;
                case 'M':
                {
                    Theme* untagged_theme( m_untagged.get_own_theme() );

                    switch( line[1] )
                    {
                        case 'f':   // font
                            untagged_theme->font = Pango::FontDescription( line.substr( 2 ) );
                            break;
                        case 'b':   // base color
                            untagged_theme->color_base.set( line.substr( 2 ) );
                            break;
                        case 't':   // text color
                            untagged_theme->color_text.set( line.substr( 2 ) );
                            break;
                        case 'h':   // heading color
                            untagged_theme->color_heading.set( line.substr( 2 ) );
                            break;
                        case 's':   // subheading color
                            untagged_theme->color_subheading.set( line.substr( 2 ) );
                            break;
                        case 'l':   // highlight color
                            untagged_theme->color_highlight.set( line.substr( 2 ) );
                            break;
                    }
                    break;
                }
                default:
                    print_error( "unrecognized line (56):\n" + line );
                    return LIFEO::CORRUPT_FILE;
            }
        }
    }

    // ENTRIES
    while( getline( stream, line ) )
    {
        switch( line[ 0 ] )
        {
            case 'A':   // date
            {   // FIXME: temporary (for compatibility)
                Date::date_t date_new = LIFEO::convert_string( line.substr( 2 ) );
                if( date_new == ( date >> 10 ) )
                    date++;
                else
                    date = ( date_new << 10 ) + 1;
                break;
            }
            case 'C':    // content
                // when writing a diary file by hand, one can easily forget to put
                // a blank space after C, so we have to check
                if( line.size() < 2 )
                    continue;

                content += line.substr( 2 );
                content += "\n";
                break;
            case 'T':    // tags
                str_tags = line;
            break;
            case '}':    // end of Entry
            {
                // erase redundant new line:
                content.erase( content.size() - 1 );

                entry_new = new Entry( this, date, content, flag_favorite );

                // tags
                std::vector< Tag* >::const_iterator iter_tag = tmp_vector_tags.begin();
                for( std::string::size_type i = 2; i < str_tags.size(); i++ )
                {
                    if( iter_tag == tmp_vector_tags.end() )
                        //TODO: issue a warning
                        break;
                    else
                    if( str_tags[ i ] == '#' )
                        entry_new->add_tag( *iter_tag );
                    else
                    if( str_tags[ i ] == ' ' )
                        continue;

                    ++iter_tag;
                }
                // TODO: per entry spellcheck options
                //entry_new->set_spellcheck( opt_spell );
                m_entries[ date ] = entry_new;

                content.clear();
                str_tags.clear();
                break;
            }
            case '{':
                flag_favorite = false;
                break;
            case '[':
                flag_favorite = true;
                break;
            case 0:
            case '\n':
                break;
            default:
                print_error( "unrecognized line:\n" + line );
                return LIFEO::CORRUPT_FILE;
                break;
        }
    }

    // every diary must at least have one chapter category:
    if( m_chapter_categories.empty() )
        m_ptr2chapter_ctg_cur = create_chapter_ctg( _( DEFAULT_CHAPTER_CTG_NAME ) );

    if( m_entries.size() < 1 )
    {
        print_info( "a dummy entry added to the diary" );
        add_today();
    }

    return LIFEO::SUCCESS;
}

LIFEO::Result
Diary::write()
{
    assert( m_flag_read_only == false );

    // UPGRADE BACKUP
    if( m_read_version != DB_FILE_VERSION_INT )
        copy_file_suffix( m_path, ".", m_read_version );

    // BACKUP THE PREVIOUS VERSION
    if( Glib::file_test( m_path, Glib::FILE_TEST_EXISTS ) )
    {
        std::string path_old( m_path + ".~previousversion~" );
        rename( m_path.c_str(), path_old.c_str() );
    }

    // WRITE THE FILE
    Result result( write( m_path ) );

    // DAILY BACKUP SAVES
#if LIFEOGRAPH_DEBUG_BUILD
    if( copy_file_suffix(
            m_path, "." + Date::format_string( Date::get_today(), "%1-%2-%3" ), -1 ) )
        print_info( "daily backup has been written successfully" );
#endif

    return result;
}

LIFEO::Result
Diary::write( const std::string& path )
{
    m_flag_only_save_filtered = false;

    if( m_passphrase.empty() )
        return write_plain( path );
    else
        return write_encrypted( path );
}

LIFEO::Result
Diary::write_copy( const std::string& path, const std::string& passphrase, bool flag_filtered )
{
    m_flag_only_save_filtered = flag_filtered;

    Result result;

    if( passphrase.empty() )
        result = write_plain( path );
    else
    {
        std::string passphrase_actual( m_passphrase );
        m_passphrase = passphrase;
        result = write_encrypted( path );
        m_passphrase = passphrase_actual;
    }

    return result;
}

LIFEO::Result
Diary::write_txt( const std::string& path, bool flag_filtered )
{
    std::ofstream file( path.c_str(), std::ios::out | std::ios::trunc );
    if( ! file.is_open() )
    {
        print_error( "i/o error: " + path );
        return LIFEO::COULD_NOT_START;
    }

    const CategoryChapters  *chapters( Diary::d->get_current_chapter_ctg() );
    CategoryChapters::const_reverse_iterator
                            iter_chapter( chapters->rbegin() );
    Chapter                 *chapter( NULL );
    bool                    flag_chapter_round( true );    // chapter or topic round
    const std::string       separator( "---------------------------------------------\n");

    // ENTRIES
    for( EntryIterReverse itr_entry = m_entries.rbegin();
         itr_entry != m_entries.rend();
         ++itr_entry )
    {
        Entry *entry( itr_entry->second );

        for( ; ; ++iter_chapter)
        {
            if( iter_chapter == chapters->rend() )
            {
                if( flag_chapter_round )
                {
                    chapters = Diary::d->get_topics();
                    iter_chapter = chapters->rbegin();
                    if( iter_chapter == chapters->rend() )
                        break;
                    flag_chapter_round = false;
                    continue;
                }
                else
                    break;
            }

            chapter = iter_chapter->second;

            if( chapter->is_initialized() )
            {
                if( entry->get_date() >= chapter->get_date() )
                    file << "\n\n" << separator << chapter->get_date().format_string() <<
                    " - " << chapter->get_name() << "\n";
                else
                    break;
            }
        }

        // purge empty entries:
        if( ( entry->m_text.size() < 1 && entry->get_tags().size() < 1 ) ||
            ( entry->get_filtered_out() && flag_filtered ) )
            continue;

        file << separator;

        // DATE AND FAVOREDNESS
        file << entry->get_date().format_string();
        if( entry->is_favored() )
            file << "  [" << _( "FAVORITE" ) << "]";

        // CONTENT
        file << "\n" << entry->get_text();

        // TAGS
        const Tagset &tagset = entry->get_tags();
        for( Tagset::const_iterator itr_tags = tagset.begin();
             itr_tags != tagset.end();
             ++itr_tags )
        {
            if( itr_tags == tagset.begin() )
                file << "\n\n" << _( "TAGS" ) << ": ";
            else
                file << ", ";

            file << ( *itr_tags )->get_name();
        }

        file << "\n\n\n";
    }
    file.close();
    return SUCCESS;
}

LIFEO::Result
Diary::write_plain( const std::string& path, bool flag_header_only )
{
    std::ofstream file( path.c_str(), std::ios::out | std::ios::trunc );
    if( ! file.is_open() )
    {
        print_error( "i/o error: " + path );
        return LIFEO::COULD_NOT_START;
    }

    std::stringstream output;
    create_db_header_text( output, flag_header_only );
    // header only mode is for encrypted diaries
    if( ! flag_header_only )
    {
        create_db_body_text( output );
    }

    file << output.str();
    file.close();

    return LIFEO::SUCCESS;
}

LIFEO::Result
Diary::write_encrypted( const std::string& path )
{
    // writing header:
    write_plain( path, true );
    std::ofstream file( path.c_str(), std::ios::out | std::ios::app | std::ios::binary );
    if( ! file.is_open() )
    {
        print_error( "i/o error: " + path );
        return LIFEO::COULD_NOT_START;
    }
    std::stringstream output;
    CipherBuffers buf;

    // first char of passphrase for validity checking
    output << m_passphrase[ 0 ] << '\n';
    create_db_body_text( output );

    // encryption
    try {
        size_t size =  output.str().size() + 1;

        LIFEO::Cipher::create_new_key( m_passphrase.c_str(), &buf.salt, &buf.key );

        LIFEO::Cipher::create_iv( &buf.iv );

        buf.buffer = new unsigned char[ size ];
        memcpy( buf.buffer, output.str().c_str(), size );

        LIFEO::Cipher::encrypt_buffer( buf.buffer, size, buf.key, buf.iv );

        file.write( ( char* ) buf.salt, LIFEO::Cipher::cSALT_SIZE );
        file.write( ( char* ) buf.iv, LIFEO::Cipher::cIV_SIZE );
        file.write( ( char* ) buf.buffer, size );
    }
    catch( ... )
    {
        buf.clear();
        return LIFEO::FAILURE;
    }

    file.close();
    buf.clear();
    return LIFEO::SUCCESS;
}

bool
Diary::create_db_header_text( std::stringstream &output, bool encrypted )
{
    output << LIFEO::DB_FILE_HEADER;
    output << "\nV " << LIFEO::DB_FILE_VERSION_INT;
    output << ( encrypted ? "\nE yes" : "\nE no" );
    output << "\n\n"; // end of header

    return true;
}

inline void
Diary::create_db_tag_text( char type, const Tag* tag, std::stringstream& output )
{
    if( type == 'm' )
        output << "ID" << tag->get_id() << "\nt " << tag->get_name_std() << '\n';

    if( tag->get_has_own_theme() )
    {
        Theme* theme( tag->get_theme() );

        output << type << 'f' << theme->font.to_string() << '\n';
        output << type << 'b' << convert_gdkrgba_to_string( theme->color_base ) << '\n';
        output << type << 't' << convert_gdkrgba_to_string( theme->color_text ) << '\n';
        output << type << 'h' << convert_gdkrgba_to_string( theme->color_heading ) << '\n';
        output << type << 's' << convert_gdkrgba_to_string( theme->color_subheading ) << '\n';
        output << type << 'l' << convert_gdkrgba_to_string( theme->color_highlight ) << '\n';
    }
}

bool
Diary::create_db_body_text( std::stringstream& output )
{
    // OPTIONS
    output << "O " << m_option_sorting_criteria << '\n';
    if( !m_language.empty() )
        output << "l " << m_language << '\n';

    // STARTUP ACTION (HOME ITEM)
    output << "S " << m_startup_elem_id << '\n';
    output << "L " << m_last_elem_id << '\n';

    // ROOT TAGS
    for( PoolTags::const_iterator iter = m_tags.begin();
         iter != m_tags.end();
         ++iter )
    {
        if( iter->second->get_category() == NULL )
            create_db_tag_text( 'm', iter->second, output );
    }
    // CATEGORIZED TAGS
    for( PoolCategoriesTags::const_iterator iter = m_tag_categories.begin();
         iter != m_tag_categories.end();
         ++iter )
    {
        // tag category:
        CategoryTags* ctg( iter->second );
        output << "ID" << ctg->get_id()
               << "\nT" << ( ctg->get_expanded() ? 'e' : '-' )
               << ctg->get_name_std() << '\n';
        // tags in it:
        for( CategoryTags::const_iterator iter_tag = ctg->begin();
             iter_tag != ctg->end();
             ++iter_tag )
        {
            create_db_tag_text( 'm', *iter_tag, output );
        }
    }
    // UNTAGGED THEME
    create_db_tag_text( 'u', &m_untagged, output );

    // TOPICS
    for( CategoryChapters::iterator iter_chapter = m_topics.begin();
         iter_chapter != m_topics.end();
         ++iter_chapter )
    {
        Chapter* chapter( iter_chapter->second );
        output << "ID" << chapter->get_id()
               << "\no" <<  ( chapter->get_expanded() ? 'e' : '-' )
               << iter_chapter->first << '\t' << chapter->get_name_std() // date (order) + name
               << '\n';
    }
    // TO-DO GROUPS
    for( CategoryChapters::iterator iter_chapter = m_todo_groups.begin();
         iter_chapter != m_todo_groups.end();
         ++iter_chapter )
    {
        Chapter* chapter( iter_chapter->second );
        output << "ID" << chapter->get_id()
               << "\nd:"
               << iter_chapter->first << '\t' << chapter->get_name_std() // date (order) + name
               << "\nd " << ( chapter->get_expanded() ? 'e' : '-' )
               << ( ( chapter->get_todo_status() & ES::DONE ) ? 'd' :
                       ( chapter->get_todo_status() & ES::CANCELED ) ? 'c' : '-' )
               << '\n';
    }
    // CHAPTERS
    for( PoolCategoriesChapters::iterator iter = m_chapter_categories.begin();
         iter != m_chapter_categories.end();
         ++iter )
    {
        // chapter category:
        CategoryChapters* ctg( iter->second );
        output << "ID" << ctg->get_id()
               << "\nC" << ( ctg == m_ptr2chapter_ctg_cur ? 'c' : '-' )
               << ctg->get_name_std() << '\n';
        // chapters in it:
        for( CategoryChapters::iterator iter_chapter = ctg->begin();
             iter_chapter != ctg->end();
             ++iter_chapter )
        {
            output << "ID" << iter_chapter->second->get_id()
                   << "\nc" << ( iter_chapter->second->get_expanded() ? 'e' : '-' )
                   << iter_chapter->first   // date
                   << '\t' << iter_chapter->second->get_name_std() << '\n';
        }
    }
    // FILTER
    ElemStatus fs( m_filter_default->get_status() );
    output << "fs" << ( fs & ES::SHOW_TRASHED ? 'T' : '-' )
                   << ( fs & ES::SHOW_NOT_TRASHED ? 't' : '-' )
                   << ( fs & ES::SHOW_FAVORED ? 'F' : '-' )
                   << ( fs & ES::SHOW_NOT_FAVORED ? 'f' : '-' )
                   << ( fs & ES::SHOW_TODO ? 'T' : '-' )
                   << ( fs & ES::SHOW_DONE ? 'D' : '-' )
                   << ( fs & ES::SHOW_CANCELED ? 'C' : '-' )
                   << '\n';
    if( m_filter_default->get_status() & ES::FILTER_TAG )
        output << "ft" << m_filter_default->get_tag()->get_name_std() << '\n';
    if( m_filter_default->get_status() & ES::FILTER_DATE_BEGIN )
        output << "fb" << m_filter_default->get_date_begin() << '\n';
    if( m_filter_default->get_status() & ES::FILTER_DATE_END )
        output << "fe" << m_filter_default->get_date_end() << '\n';

    output << '\n'; // end of section

    // ENTRIES
    for( EntryIterConst iter_entry = m_entries.begin();
         iter_entry != m_entries.end();
         ++iter_entry )
    {
        Entry* entry = ( *iter_entry ).second;

        // purge empty entries:
        if( entry->m_text.size() < 1 && entry->m_tags.empty() ) continue;
        // optionally only save filtered entries:
        else
        if( entry->get_filtered_out() && m_flag_only_save_filtered ) continue;

        // ENTRY DATE
        output << "ID" << entry->get_id() << "\n"
               << ( entry->is_trashed() ? 'e' : 'E' )
               << ( entry->is_favored() ? 'f' : '-' )
               << ( m_filter_default->is_entry_filtered( entry ) ? 'h' : '-' )
               << ( ( entry->get_todo_status() & ES::DONE ) ? 'd' :
                       ( entry->get_todo_status() & ES::CANCELED ) ? 'c' : '-' )
               << entry->m_date.m_date << '\n';
        output << "Dr" << entry->m_date_created << '\n';
        output << "Dh" << entry->m_date_changed << '\n';

        // TAGS
        for( Tagset::const_iterator iter_tag = entry->m_tags.begin();
             iter_tag != entry->m_tags.end();
             ++iter_tag )
        {
            Tag* tag( *iter_tag );
            output << "T" << ( tag == entry->get_theme_tag() ? 'T' : '-' )
                   << ( *iter_tag )->get_name_std() << '\n';
        }

        // LANGUAGE
        if( entry->get_lang() != LANG_INHERIT_DIARY )
            output << "l " << entry->get_lang() << '\n';

        // CONTENT
        if( entry->m_text.empty() )
            output << "\n";
        else
        {
            // NOTE: for some reason, implicit conversion from Glib:ustring...
            // ...fails while substr()ing when LANG=C
            // we might reconsider storing text of entries as std::string.
            // for now we convert entry text to std::string here:
            std::string             content( entry->m_text );
            std::string::size_type  pt_start( 0 ), pt_end( 0 );

            while( true )
            {
                pt_end = content.find( '\n', pt_start );
                if( pt_end == std::string::npos )
                {
                    pt_end = content.size();
                    output << "P " << content.substr( pt_start, content.size() - pt_start )
                           << "\n\n";
                    break; // end of while( true )
                }
                else
                {
                    pt_end++;
                    output << "P " << content.substr( pt_start, pt_end - pt_start );
                    pt_start = pt_end;
                }
            }
        }
    }

    return true;
}

DiaryElement*
Diary::get_element( DEID id ) const
{
    PoolDEIDs::const_iterator iter( m_ids.find( id ) );
    return( iter == m_ids.end() ? NULL : iter->second );
}

DiaryElement*
Diary::get_startup_elem() const
{
    DiaryElement *elem = NULL;

    switch( m_startup_elem_id )
    {
        case HOME_CURRENT_ELEM:
            elem = get_most_current_elem();
            break;
        case HOME_LAST_ELEM:
            elem = get_element( m_last_elem_id );
            break;
        case DEID_UNSET:
            break;
        default:
            elem = get_element( m_startup_elem_id );
            break;
    }

    return ( elem ? elem : d );
}

void
Diary::set_startup_elem( const DEID id )
{
    m_startup_elem_id = id;
}

DiaryElement*
Diary::get_most_current_elem() const
{
    Glib::Date gdate;
    gdate.set_time_current();
    Date date( gdate.get_year(), gdate.get_month(), gdate.get_day() );
    Date::date_t diff1( Date::ORDINAL_FLAG );
    long diff2;
    DiaryElement *elem( NULL );
    bool descending( false );
    for( EntryIterConst iter = m_entries.begin(); iter != m_entries.end(); ++iter )
    {
        if( ! iter->second->get_filtered_out() && ! iter->second->get_date().is_ordinal() )
        {
            diff2 = iter->second->get_date().m_date - date.m_date;
            if( diff2 < 0 ) diff2 *= -1;
            if( static_cast< unsigned long >( diff2 ) < diff1 )
            {
                diff1 = diff2;
                elem = iter->second;
                descending = true;
            }
            else
            if( descending )
                break;
        }
    }

    if( elem )
        return elem;
    else
        return( const_cast< Diary* >( this ) );
}

DiaryElement*
Diary::get_prev_session_elem() const
{
    return get_element( m_last_elem_id );
}

void
Diary::set_last_elem( const DiaryElement* elem )
{
    m_last_elem_id = elem->get_id();
}

// LOCKS ===========================================================================================
inline bool
Diary::remove_lock()
{
    if( m_path.empty() )
        return false;

    std::string path_lock( m_path + LOCK_SUFFIX );
    if( Glib::file_test( path_lock, Glib::FILE_TEST_EXISTS ) )
        remove( path_lock.c_str() );
    return true;
}

// ENTRIES =========================================================================================
Entry*
Diary::add_today()
{
    return create_entry( Date::get_today() );
}

Entry*
Diary::get_entry( const Date::date_t date )
{
    EntryIter iter( m_entries.find( date ) );
    if( iter != m_entries.end() )
        if( iter->second->get_filtered_out() == false )
            return iter->second;

    return NULL;
}

Entry*
Diary::get_entry_today()
{
    // FIXME: handle filtered out case
    return get_entry( Date::get_today( 1 ) );  // 1 is the order
}

EntryVector*
Diary::get_entries( Date::date_t date ) // takes pure date
{
    EntryVector *entries = new EntryVector;

    EntryIter iter = m_entries.find( date + 1 );
    if( iter == m_entries.end() )
        return entries; // return empty vector

    for( ; ; --iter )
    {
        if( iter->second->get_date().get_pure() == date )
        {
            if( iter->second->get_filtered_out() == false )
                entries->push_back( iter->second );
        }
        else
            break;
        if( iter == m_entries.begin() )
            break;
    }
    return entries;
}

bool
Diary::get_day_has_multiple_entries( const Date& date_impure )
{
    Date::date_t date = date_impure.get_pure();
    EntryIterConst iter = m_entries.find( date + 2 );
    if( iter == m_entries.end() )
        return false;

    for( ; iter != m_entries.begin(); --iter )
    {
        if( iter->second->get_date().get_pure() == date )
        {
            if( iter->second->get_filtered_out() == false )
                return true;
        }
        else
            break;
    }

    return false;
}

Entry*
Diary::get_entry_next_in_day( const Date& date )
{
    EntryIter entry_1st( m_entries.find( date.get_pure() + 1 ) );
    if( entry_1st == m_entries.end() )
        return NULL;
    EntryIter entry_next( m_entries.find( date.m_date + 1 ) );
    if( entry_next != m_entries.end() )
        return entry_next->second->get_filtered_out() ? NULL : entry_next->second;
    else
    if( date.get_order() > 1 )
        return entry_1st->second->get_filtered_out() ? NULL : entry_1st->second;
    else
        return NULL;
}

Entry*
Diary::get_entry_first()
{
    // return first unfiltered entry
    for( EntryIter iter = m_entries.begin(); iter != m_entries.end(); ++iter )
    {
        Entry *entry = iter->second;
        if( entry->get_filtered_out() == false )
            return( entry );
    }
    return NULL;
}

bool
Diary::is_first( Entry const* const entry ) const
{
    return( entry == m_entries.begin()->second );
}

bool
Diary::is_last( Entry const* const entry ) const
{
    return( entry == m_entries.rbegin()->second );
}

Entry*
Diary::create_entry( Date::date_t date, const Glib::ustring& content, bool flag_favorite )
{
    // make it the last entry of its day:
    Date::reset_order_1( date );
    while( m_entries.find( date ) != m_entries.end() )
        ++date;

    Entry *entry = new Entry( this, date, content, flag_favorite );

    m_entries[ date ] = entry;

    return( entry );
}

bool
Diary::dismiss_entry( Entry* entry )
{
    Date::date_t date = entry->get_date().m_date;

    // fix startup element:
    if( m_startup_elem_id == entry->get_id() )
        m_startup_elem_id = DEID_MIN;

    // remove from tags:
    for( Tagset::iterator iter = entry->get_tags().begin();
         iter != entry->get_tags().end();
         ++iter )
        ( *iter )->get_items()->erase( entry );

    // remove from filters:
    if( m_filter_active->is_entry_filtered( entry ) )
        m_filter_active->remove_entry( entry );
    if( m_filter_default->is_entry_filtered( entry ) )
        m_filter_default->remove_entry( entry );

    // erase entry from map:
    m_entries.erase( date );

    // fix entry order:
    int i = 1;
    for( EntryIter iter = m_entries.find( date + i );
         iter != m_entries.end();
         iter = m_entries.find( date + i ) )
    {
        Entry *entry2fix = iter->second;
        m_entries.erase( iter );
        entry2fix->m_date.m_date--;
        m_entries[ entry2fix->get_date().m_date ] = entry2fix;
        ++i;
    }

    delete entry;
    return true;
}

/*int
Diary::list_entries()
{
    int number_of_entries( 0 );
    for( Entryiter t1_itr = m_entries.begin(); t1_itr != m_entries.end(); t1_itr++ )
    {
        std::cout << t1_itr->second->get_date().format_string() << std::endl;
        number_of_entries++;
    }
    std::cout << number_of_entries << " entries total." << std::endl;
    return( number_of_entries );
}*/

void
Diary::set_entry_date( Entry* entry, const Date& date )
{
    EntryIter iter;
    Date::date_t d( entry->m_date.m_date );
    m_entries.erase( d );

    for( iter = m_entries.find( ++d );
         iter != m_entries.end();
         iter = m_entries.find( ++d ) )
    {
        Entry *entry_shift( iter->second );
        m_entries.erase( d );
        entry_shift->set_date( d - 1 );
        m_entries[ d - 1 ] = entry_shift;
    }

    // find the last entry in the date/order
    d = date.m_date;
    bool flag_replace( false );
    while( m_entries.find( d ) != m_entries.end() )
    {
        flag_replace = true;
        d++;
    }

    if( flag_replace )
    {
        for( iter = m_entries.find( --d );
             d >= date.m_date;
             iter = m_entries.find( --d ) )
        {
            Entry *entry_shift( iter->second );
            m_entries.erase( d );
            entry_shift->set_date( d + 1 );
            m_entries[ d + 1 ] = entry_shift;
        }
    }

    entry->set_date( date.m_date );
    m_entries[ date.m_date ] = entry;
}

// FILTERING =======================================================================================
void
Diary::set_search_text( const Glib::ustring& filter )
{
    m_search_text = filter;
    m_filter_active->set_status_outstanding();
}

int
Diary::replace_text( const Glib::ustring& newtext )
{
    Glib::ustring::size_type iter_str;
    const int chardiff = newtext.size() - m_search_text.size();

    Entry *entry;

    for( EntryIter iter_entry = m_entries.begin();
         iter_entry != m_entries.end();
         iter_entry++ )
    {
        entry = iter_entry->second;
        if( entry->get_filtered_out() )
            continue;

        Glib::ustring entrytext = entry->get_text().lowercase();
        iter_str = 0;
        int count = 0;
        while( ( iter_str = entrytext.find( m_search_text, iter_str ) ) != std::string::npos )
        {
            entry->get_text().erase(
                iter_str + ( chardiff * count ), m_search_text.size() );
            entry->get_text().insert(
                iter_str + ( chardiff * count ), newtext );
            count++;
            iter_str += m_search_text.size();
        }
    }

    return 0;   // reserved
}

// TAGS ============================================================================================
Tag*
Diary::create_tag( const Glib::ustring& name, CategoryTags* category )
{
    PoolTags::iterator iter = m_tags.find( name );
    if( iter != m_tags.end() )
    {
        PRINT_DEBUG( "Tag already exists: " + name );
        return( iter->second );
    }
    Tag *tag( new Tag( this, name, category ) );
    m_tags.insert( PoolTags::value_type( name, tag ) );
    return tag;
}

void
Diary::dismiss_tag( Tag* tag, bool flag_dismiss_associated )
{
    // fix or dismiss associated entries
    if( flag_dismiss_associated )
    {
        while( tag->get_items()->size() > 0 )
            dismiss_entry( dynamic_cast< Entry* >( * tag->get_items()->begin() ) );
    }
    else
    {
        for( SetDiaryElements::const_iterator iter = tag->get_items()->begin();
             iter != tag->get_items()->end();
             ++iter )
        {
            Entry* entry = dynamic_cast< Entry* >( *iter );
            entry->remove_tag( tag );
        }
    }

    // remove from category if any
    if( tag->get_category() != NULL )
        tag->get_category()->erase( tag );

    // clear filters if necessary
    if( tag == m_filter_active->get_tag() )
        m_filter_active->set_tag( NULL );
    if( tag == m_filter_default->get_tag() )
        m_filter_default->set_tag( NULL );

    m_tags.erase( tag->get_name() );
    delete tag;
}

CategoryTags*
Diary::create_tag_ctg()
{
    Glib::ustring name = create_unique_name_for_map( m_tag_categories, _( "New category" ) );
    CategoryTags *new_category = new CategoryTags( this, name );
    m_tag_categories.insert( PoolCategoriesTags::value_type( name, new_category ) );

    return new_category;
}

CategoryTags*
Diary::create_tag_ctg( const Glib::ustring& name )  // used while reading diary file
{
    try
    {
        CategoryTags *new_category = new CategoryTags( this, name );
        m_tag_categories.insert( PoolCategoriesTags::value_type( name, new_category ) );
        return new_category;
    }
    catch( std::exception &ex )
    {
        throw LIFEO::Error( ex.what() );
    }
}

void
Diary::dismiss_tag_ctg( CategoryTags* ctg, bool flag_dismiss_contained )
{
    // fix or dismiss contained tags
    if( flag_dismiss_contained )
    {
        while( ctg->size() > 0 )
            dismiss_tag( * ctg->begin() );
    }
    else
    {
        for( CategoryTags::const_iterator iter = ctg->begin(); iter != ctg->end(); ++iter )
            ( * iter )->set_category( NULL );
    }

    // remove from the list and delete
    m_tag_categories.erase( ctg->get_name() );
    delete ctg;
}

// CHAPTERS ========================================================================================
CategoryChapters*
Diary::create_chapter_ctg()
{
    Glib::ustring name = create_unique_name_for_map( m_chapter_categories, _( "New category" ) );
    CategoryChapters* category = new CategoryChapters( this, name );
    m_chapter_categories.insert( PoolCategoriesChapters::value_type( name, category ) );

    return category;
}

CategoryChapters*
Diary::create_chapter_ctg( const Glib::ustring& name )
{
    // name's availability must be checked beforehand
    try
    {
        CategoryChapters* category = new CategoryChapters( this, name );
        m_chapter_categories.insert( PoolCategoriesChapters::value_type( name, category ) );
        return category;
    }
    catch( std::exception& ex )
    {
        throw LIFEO::Error( ex.what() );
    }
}

void
Diary::dismiss_chapter_ctg( CategoryChapters* category )
{
    if( category == m_ptr2chapter_ctg_cur )
        m_ptr2chapter_ctg_cur = m_chapter_categories.begin()->second;

    m_chapter_categories.erase( category->get_name() );
    delete category;
}

bool
Diary::rename_chapter_ctg( CategoryChapters* category, const Glib::ustring& name )
{
    if( m_chapter_categories.count( name ) > 0 )
        return false;

    m_chapter_categories.erase( category->get_name() );
    category->set_name( name );
    m_chapter_categories.insert(
            PoolCategoriesChapters::value_type( name, category ) );

    return true;
}

void
Diary::dismiss_chapter( Chapter* chapter, bool flag_dismiss_contained )
{
    if( chapter->is_ordinal() ) // TOPIC OR TODO GROUP
    {
        CategoryChapters* ptr2ctg( chapter->get_date().is_todo() ?
                &m_todo_groups : &m_topics );

        // ORDER SHIFTING TO PRESERVE CONTINUITY
        if( ptr2ctg->size() == 1 )  // must be handled separately
        {
            ptr2ctg->erase( chapter->get_date().m_date );
            delete chapter;
            return;
        }

        // CALCULATE HELPER VALUES
        Date d_chapter( chapter->get_date() );
        Date::date_t d_chapter_next( d_chapter.m_date + Date::ORDINAL_STEP );
        bool flag_last_chapter( ptr2ctg->find( d_chapter_next ) == ptr2ctg->end() );

        int last_order_of_prev_chapter( 0 );
        if( flag_last_chapter )
        {
            Chapter* chapter_prev( ptr2ctg->get_chapter(
                    d_chapter.m_date - Date::ORDINAL_STEP ) );
            Date d_first_free( chapter_prev->get_free_order() );
            last_order_of_prev_chapter = d_first_free.get_order() - 1;
        }
        else
        {
            Date d_first_free( chapter->get_free_order() );
            last_order_of_prev_chapter = d_first_free.get_order() - 1;
        }

        // ACTUALLY DISMISS THE TOPIC
        ptr2ctg->erase( chapter->get_date().m_date );
        delete chapter;

        // SHIFT TOPICS
        for( Date::date_t d = d_chapter.m_date + Date::ORDINAL_STEP; ;
             d += Date::ORDINAL_STEP )
        {
            CategoryChapters::iterator iter( ptr2ctg->find( d ) );
            if( iter == ptr2ctg->end() )
                break;
            Chapter* chpt( iter->second );
            ptr2ctg->erase( d );
            chpt->set_date( d - Date::ORDINAL_STEP );
            ( *ptr2ctg )[ d - Date::ORDINAL_STEP ] = chpt;
        }

        // SHIFT ENTRIES
        if( m_entries.size() > 0 )
        {
            bool flag_contained_not_finished( true );

            for( Date::date_t d = d_chapter.m_date + 1; d <= m_entries.begin()->first; )
            {
                EntryIter iter( m_entries.find( d ) );
                if( iter == m_entries.end() )
                {
                    d += Date::ORDINAL_STEP;
                    Date::reset_order_1( d );
                    flag_contained_not_finished = false;
                    continue;
                }

                Entry *entry( iter->second );

                if( flag_dismiss_contained && flag_contained_not_finished )
                {
                    dismiss_entry( entry );
                    continue;
                }

                int order_diff( entry->get_date().get_ordinal_order() -
                                d_chapter.get_ordinal_order() );

                if( order_diff > 0 || ( order_diff == 0 && flag_last_chapter ) )
                {
                    m_entries.erase( d );
                    Date::date_t d_new( d - Date::ORDINAL_STEP );
                    if( ( order_diff == 1 || ( order_diff == 0 && flag_last_chapter ) ) &&
                          !flag_dismiss_contained )
                        d_new += last_order_of_prev_chapter;
                    entry->set_date( d_new );
                    m_entries[ d_new ] = entry;
                }
                d++;
            }
        }
    }
    else // TEMPORAL CHAPTER
    {
        if( flag_dismiss_contained )
        {
            CategoryChapters::iterator iter_chapter(
                    m_ptr2chapter_ctg_cur->find( chapter->get_date().m_date ) );

            Date::date_t date_end;
            if( iter_chapter == m_ptr2chapter_ctg_cur->begin() )
                date_end = Date::DATE_MAX;
            else
            {
                iter_chapter--;
                date_end = iter_chapter->first;
            }

            EntryVector contained_entries;

            for( EntryIter iter = m_entries.begin(); iter != m_entries.end(); ++iter )
            {
                Entry *entry( iter->second );
                if( entry->m_date > chapter->get_date() && entry->m_date.m_date < date_end )
                    contained_entries.push_back( entry );
            }

            for( EntryVectorIter iter = contained_entries.begin();
                 iter != contained_entries.end();
                 ++iter )
                dismiss_entry( *iter );
        }

        m_ptr2chapter_ctg_cur->dismiss_chapter( chapter );
    }
}

Chapter*
Diary::add_topic( const Glib::ustring& name )
{
    Chapter* c( m_topics.create_chapter( name, m_topics.get_free_order() ) );
    c->set_expanded( true );
    return c;
}

Chapter*
Diary::add_todo_group( const Glib::ustring& name )
{
    Date d( 0 );
    if( m_todo_groups.empty() )
        d.m_date = Date::ORDINAL_FLAG | Date::TODO_FLAG;
    else
        d = m_todo_groups.get_free_order();

    Chapter* c( m_todo_groups.create_chapter( name, d ) );
    c->set_expanded( true );
    return c;
}

void
Diary::set_topic_order( Chapter* chapter, Date::date_t date )
{
    assert( chapter->is_ordinal() );

    CategoryChapters* ptr2ctg( chapter->get_date().is_todo() ? &m_todo_groups : &m_topics );

    long step( chapter->get_date().m_date > date ? -Date::ORDINAL_STEP : Date::ORDINAL_STEP );
    EntryVector ev;

    Date::date_t d_e( chapter->get_date().m_date + 1 );
    EntryIter iter_entry( m_entries.find( d_e ) );

    while( iter_entry != m_entries.end() )
    {
        Entry* entry( iter_entry->second );

        m_entries.erase( d_e );
        ev.push_back( entry );

        iter_entry = m_entries.find( ++d_e );
    }

    ptr2ctg->erase( chapter->get_date().m_date );

    // SHIFT TOPICS
    for( Date::date_t d = chapter->get_date().m_date + step; d != ( date + step ); d += step )
    {
        CategoryChapters::iterator iter( ptr2ctg->find( d ) );
        if( iter == ptr2ctg->end() )
            break;
        Chapter* chpt( iter->second );
        ptr2ctg->erase( d );
        chpt->set_date( d - step );
        ( *ptr2ctg )[ d - step ] = chpt;

        d_e = d + 1;
        iter_entry = m_entries.find( d_e );
        while( iter_entry != m_entries.end() )
        {
            Entry* entry( iter_entry->second );

            m_entries.erase( d_e );
            entry->set_date( d_e - step );
            m_entries[ d_e - step ] = entry;

            iter_entry = m_entries.find( ++d_e );
        }
    }

    chapter->set_date( date );
    ( *ptr2ctg )[ date ] = chapter;
    for( EntryVectorIter i = ev.begin(); i != ev.end(); i++ )
    {
        Entry* entry( *i );

        entry->set_date( ++date );
        m_entries[ date ] = entry;
    }
}

Date
Diary::get_free_chapter_order_temporal()
{
    time_t t = time( NULL );
    struct tm *ti = localtime( &t );
    Date date( ti->tm_year + 1900, ti->tm_mon + 1, ti->tm_mday );

    while( m_ptr2chapter_ctg_cur->get_chapter( date.m_date ) )
    {
        date.forward_day();
    }

    return date;
}

Date
Diary::get_free_chapter_order_ordinal()
{
    return m_ptr2chapter_ctg_cur->get_free_order();
}

bool
Diary::make_free_entry_order( Date& date )
{
    date.reset_order_1();
    while( m_entries.find( date.m_date ) != m_entries.end() )
        ++date.m_date;

    return true;    // reserved for bounds checking
}

// SHOW ============================================================================================
void
Diary::show()
{
    if( shower != NULL )
        shower->show( *this );
}

// IMPORTING =======================================================================================
bool
Diary::import_tag( Tag* tag )
{
    Tag* new_tag( create_tag( tag->get_name() ) );
    if( new_tag == NULL )
        return false;

    if( tag->get_has_own_theme() )
        new_tag->create_own_theme_duplicating( tag->get_theme() );

    return true;
}

bool
Diary::import_entries( const Diary& diary,
                       bool flag_import_tags,
                       const Glib::ustring &tag_all_str )
{
    Entry* entry_new;
    Entry* entry_ext;
    Tag* tag_all = NULL;

    if( ! tag_all_str.empty() )
        tag_all = create_tag( tag_all_str );

    for( EntryIterConstRev iter = diary.m_entries.rbegin();
         iter != diary.m_entries.rend();
         ++iter )
    {
        entry_ext = iter->second;
        entry_new = new Entry( this,
                               entry_ext->m_date.m_date,
                               entry_ext->m_text,
                               entry_ext->is_favored() );

        // fix order:
        entry_new->m_date.reset_order_1();
        while( m_entries.find( entry_new->m_date.m_date ) != m_entries.end() )
            entry_new->m_date.m_date++;

        // copy dates:
        entry_new->m_date_created = entry_ext->m_date_created;
        entry_new->m_date_changed = entry_ext->m_date_changed;

        // insert it into the diary:
        m_entries[ entry_new->m_date.m_date ] = entry_new;

        if( flag_import_tags )
        {
            Tagset &tags = entry_ext->get_tags();
            Tag *tag;

            for( Tagset::const_iterator iter = tags.begin(); iter != tags.end(); ++iter )
            {
                tag = ( m_tags.find( ( *iter )->get_name() ) )->second;
                entry_new->add_tag( tag );
            }

            // preserve the theme:
            if( entry_ext->get_theme_is_set() )
            {
                tag = ( m_tags.find( entry_ext->get_theme_tag()->get_name() ) )->second;
                entry_new->set_theme_tag( tag );
            }
        }
        if( tag_all )
            entry_new->add_tag( tag_all );
    }

    return true;
}

bool
Diary::import_chapters( const Diary& diary )
{
    for( PoolCategoriesChapters::const_iterator i_cc = diary.m_chapter_categories.begin();
         i_cc != diary.m_chapter_categories.end();
         ++i_cc )
    {
        CategoryChapters* cc( i_cc->second );
        CategoryChapters* cc_new = create_chapter_ctg(
                create_unique_name_for_map( m_chapter_categories, cc->get_name() ) );

        for( CategoryChapters::iterator i_c = cc->begin(); i_c != cc->end(); ++i_c )
        {
            Chapter* chapter( i_c->second );
            cc_new->create_chapter( chapter->get_name(), chapter->get_date() );
        }
    }

    return true;
}

