/* 
 * $Id: input.c,v 1.15 2001/07/16 22:59:01 doviende Exp $
 *
 * libarr - a screen management toolkit
 *
 * Copyright (C) 2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Author: Chris Bond
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <sys/types.h>
#include <sys/ioctl.h>

#include <termios.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#if defined(__linux__)
#include <sys/kd.h>
#endif

#include "arr.h"
#include "types.h"
#include "ktree.h"

/*
 * These input routines are used for reading things from the console.  They are
 * unusually complicated because they need to work using both GPM and every
 * single terminal type.
 */
static struct termios orig_term;
static byte_t set_term;

#if defined(__linux__)
static struct kbentry orig_kbe;
static struct kbsentry orig_kbs;
static byte_t set_kb;
#endif

int
input_init(void)
{
	struct termios term;

#if defined(__linux__)
	/* The keyboard remapping code works on Linux only.  Everything else
	 * will have to do without shift-tab for now...
	 */
	struct kbentry kbe;
	struct kbsentry kbs;
#endif

	if (tcgetattr(0, &term) < 0) {
		perror("Unable to get terminal attributes");
		return -2;
	}

	memcpy(&orig_term, &term, sizeof(term));
	cfmakeraw(&term);

	if (tcsetattr(0, TCSANOW, &term) < 0) {
		perror("Unable to set terminal attributes");
		return -2;
	}
	else
		++set_term;

#if defined(__linux__)
	kbe.kb_table = orig_kbe.kb_table = 1;		/* shift */
	kbe.kb_index = orig_kbe.kb_index = 15;		/* tab */
	kbs.kb_func  = orig_kbs.kb_func  = (128 - 1);	/* F128 (arbitrary) */

	if ((ioctl(0, KDGKBENT, &orig_kbe) < 0) ||
	    (ioctl(0, KDGKBSENT, &orig_kbs) < 0))
		return -1;

	kbe.kb_value = (0x100 + 128 - 1);		/* F128 */
	strcpy((char *) kbs.kb_string, "\033$");	/* arbitrary */

	if ((ioctl(0, KDSKBENT, &kbe) < 0) ||
	    (ioctl(0, KDSKBSENT, &kbs) < 0))
		return -1;
	else
		++set_kb;
#endif

	return 0;
}

int
input_reset(void)
{
#if defined(__linux__)
	int r = 0;

	if ((set_kb) && ((ioctl(0, KDSKBENT, &orig_kbe) < 0) ||
			 (ioctl(0, KDSKBSENT, &orig_kbs) < 0)))
		r = -1;

	if ((set_term) && (tcsetattr(0, TCSANOW, &orig_term) < 0))
		r = -1;

	return r;
#else
	return 0;
#endif
}

arr_kqueue *
arr_key(func)
	int (*func)(FILE *);
{
	/* Most of the functionality that used to be in this file has been
	 * moved to ktree.c (where read_queue() is).
	 */
	return read_queue(func);
}
