/* ***************************************************************************
 *
 * Pico Technology USBTC08 Device Driver
 *
 *//**
 * \file      main.cpp
 * \brief     C++ Example Program using the USBTC08 driver
 **//*
 *
 * Copyright (c) 2007, Pico Technology.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  * Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  * Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *  * The name of Pico Technology may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Version $Id: main.cpp,v 1.4 2007/08/02 09:45:00 douglas Exp $
 *
 *************************************************************************** */

#include <unistd.h>

// Determine the OS we are using and include correct header files.
// The rest of this file is platform-independant
#if defined(linux) || defined(__linux) || defined(__linux__)
// Linux
#include <libusbtc08-1.7/TC08Device.h>
#include <libusbtc08-1.7/TC08Enumerator.h>
#include <signal.h>
#include <stdio.h>

#elif defined(__APPLE__)
// Mac OSX
#include <CoreFoundation/CoreFoundation.h>
#include <mach/mach.h>
#include "TC08/TC08.h"

#else 
// An unsupported OS. It's possible that the program may nonetheless work
// but some changes may be needed.
#error Sorry, your OS does not appear to be supported. 

#endif

// Non-interactive test program to demonstrate driver features
int main (int argc, const char * argv[]) {
	
	// We need a TC08Enumerator object to download FW and to list devices
	TC08Enumerator *enumerator;

	int deviceCount=0;                  // Number of attached devices
	TC08Device **tc08_devices=NULL;     // Pointer to list of TC08Devices
	const USBTC08_UNIT_INFO	*unitInfo;      // Struct to hold unit information
	
	// Let the user know we've started
	printf("\nTC08 Driver Example\n");
	printf("-------------------\n\n");
	
	// Create the TC08Enumerator to download FW
	// printf("Creating enumerator...\n");
	enumerator = new TC08Enumerator();
	
	// Check that the enumerator was successfully created
	if (enumerator) {
		// printf("Got enumerator\n");
	} else {
		printf("Failed to create enumerator\n");
		return -1;
	}
	
	// Give the enumerator a chance to find any units already connected
	// This must be done with a timeout, see documentation
	int timeout = 20;
	printf("Looking for devices...");
	while(!(enumerator->Ready()) && timeout-- > 0){
		printf(".");
		fflush(stdout);
		sleep(1);
	}

	
	// Check whether we found all devices
	if (timeout <= 0) {
		printf(" timed out\n");
		return -1;
	} else {
		printf(" done!\n");
	}
	
	// Count the units currently connected
	printf("Counting TC08s: ");	
	deviceCount = enumerator->Count();
	printf("%i found.\n",deviceCount);
	
	// If we didn't find any devices, give up
	if (deviceCount == 0) {
		printf("No TC08 devices were found, exiting.\n");
		return -1;
	}	
	
	// Create a TC08Device array big enough for the connected devices
	// printf("Allocating buffer.\n");	
	tc08_devices = new TC08Device* [deviceCount];
	if (!tc08_devices) {
		printf("Couldn't create TC08Device array - out of memory?\n");
		return -1;
	}
	
	// Get a list of the TC08s
	printf("Enumerating TC08s - ");	
	enumerator->Enumerate(tc08_devices,deviceCount);
	
	// If we didn't find any devices, give up
	if (deviceCount == 0) {
		printf("\nno TC08 devices were found\n");
		return -1;
	}	
	
	// Print the list, with serial numbers
	printf("found these devices on the system:\n");
	for(int i=0;i<deviceCount;i++) {
		printf("[%s]\n",tc08_devices[i]->GetSerialString());
	}
	
	
	// Open all connected devices with TC08Device Open() method
	printf("\nOpening all connected devices...\n");
	for (int i = 0; i < deviceCount; i++) {
		tc08_devices[i]->Open();
		
		// Get unit info and display it
		if((unitInfo=tc08_devices[i]->GetUnitInfo())==NULL) {
			printf("Could not get unit info for Unit %d\n", i);
		} else {
			printf("Unit %d: Hardware Version %i, Serial Number \"%s\",\n        Calibration Date \"%s\", Firmware Version %i.%i.%i\n",
				   i,
				   unitInfo->HardwareVersion,
				   unitInfo->Serial,
				   unitInfo->CalDate,
				   unitInfo->FirmwareVersion.Major,unitInfo->FirmwareVersion.Minor,unitInfo->FirmwareVersion.Revision);
		}
	} 
	
	
	for (int i = 0; i < deviceCount; i++) {
		// Demonstrate the LED call
		// printf("Testing LED on Unit %d\n", i);
		for(int j=0;j<USBTC08_LED_MAX;j++) {
			tc08_devices[i]->SetLED((USBTC08_LED_STATES)(j));
			usleep(100000);
		}
		
		
		// Set up all channels for thermocouple measurement
		// printf("Setting up channels.\n");
		tc08_devices[i]->SetChannelConfig(1,'K');
		tc08_devices[i]->SetChannelConfig(2,'K');
		tc08_devices[i]->SetChannelConfig(3,'K');
		tc08_devices[i]->SetChannelConfig(4,'K');
		tc08_devices[i]->SetChannelConfig(5,'K');
		tc08_devices[i]->SetChannelConfig(6,'K');
		tc08_devices[i]->SetChannelConfig(7,'K');
		tc08_devices[i]->SetChannelConfig(8,'K');
	}
	
	// Demonstrate several conversions with GetSingle()
	printf("\nGetting temperature readings on all units (GetSingle)\n");
	printf("          CJC    Ch1    Ch2    Ch3    Ch4    Ch5    Ch6    Ch7    Ch8\n");
	float temps[9];
	unsigned short flags;
	for (int conversions = 0; conversions < 10; conversions++) {
		for (int i = 0; i < deviceCount; i++) {
			if (PICO_SUCCESS != tc08_devices[i]->GetSingle(temps,&flags,USBTC08_UNITS_CENTIGRADE))
				continue; // Check whether the unit is still connected
			printf("Unit %d: %6.2f",i , temps[0]);
			for (short ch = 1; ch < 9; ch++) {
				printf(" %6.2f",temps[ch]);
			}
			printf ("\n");
			if (flags) printf("An overflow occurred\n");
			usleep(500000);
		}
	}
	
    // Demonstrate streaming on Unit 1
	
	printf("\nGetting temperature readings on first unit (Streaming)\n");
	
	// Find the minimum interval and start running at it
	long minInterval=tc08_devices[0]->GetMinimumInterval();
	printf("Minimal sampling interval is %ld ms\n",minInterval);

	// setting the smapling interval
	long setInterval = 10000;
	tc08_devices[0]->Run(setInterval);
	printf("Manually set sampling interval is %ld ms\n",setInterval);
	
	// Buffers for each channel
#define BUF_SIZE 2048
	float bufCJC[BUF_SIZE];
	float bufCh1[BUF_SIZE];
	float bufCh2[BUF_SIZE];
	float bufCh3[BUF_SIZE];
	float bufCh4[BUF_SIZE];
	float bufCh5[BUF_SIZE];
	float bufCh6[BUF_SIZE];
	float bufCh7[BUF_SIZE];
	float bufCh8[BUF_SIZE];
	// An array of 9 buffers
	float* channels[]={bufCJC,bufCh1,bufCh2,bufCh3,bufCh4,bufCh5,bufCh6,bufCh7,bufCh8};
	unsigned int lengths[9];
	
	printf("          CJC    Ch1    Ch2    Ch3    Ch4    Ch5    Ch6    Ch7    Ch8\n");
	
	// Get readings and display them
	unsigned short overflow=0;
	int printCount = 9;

	while(printCount) {
		for (short ch = 0; ch < 9; ch++) {
			lengths[ch]=BUF_SIZE;
		}
		PICO_RETURNS retVal=tc08_devices[0]->GetValues(channels,lengths,&overflow,USBTC08_UNITS_CENTIGRADE,true);
		if(retVal == PICO_SUCCESS) {
			int maxLength = 0;
			for (short ch = 0; ch < 9; ch++) {
				maxLength = lengths[ch] > (unsigned int)maxLength ? lengths[ch] : maxLength;
			}
			
			for (int sample = 0; sample < maxLength; sample++){
				printf("Unit 0:");
				for (short ch = 0; ch < 9; ch++) {
					if(channels[ch]!=NULL) {
						if (lengths[ch] > (unsigned int)sample){
							printf(" %6.2f",channels[ch][sample]);
						} else {
							printf ("       ");
						}
					} else {
						printf(" <null>");
					}
				}
				if(overflow) {
					printf("\t(An overflow occurred)\n");
				} else {
					printf ("\n");
				}
				printCount--;
			}
			
		} else if(retVal<0) {
			printf("Error getting values\n");
			break;
		}
		usleep(100000);
	}
	// Check that the device is still streaming (e.g. hasn't been unplugged)
	if(tc08_devices[0]->GetDeviceState()!=USBTC08_STATE_STREAMING) {
		switch(tc08_devices[0]->GetDeviceState()) {
			case USBTC08_STATE_DISCONNECTED:
				printf("Device is disconnected.\n");
				break;
			case USBTC08_STATE_LOCKED:
				printf("Device is locked.\n");
				break;
			case USBTC08_STATE_CLOSED:
				printf("Device is closed.\n");
				break;
			case USBTC08_STATE_IDLE:
				printf("Device is idle.\n");
				break;
			default:
				break;
		}
	} else {
		printf("Stopping device.\n\n");
		tc08_devices[0]->Stop();
	}
	
	// Clean up
	delete(enumerator);
	
	if(tc08_devices) {
		for(int i=0;i<deviceCount;i++) {
			printf("Closing Unit %d\n", i);			
			tc08_devices[i]->Close();
			delete tc08_devices[i];
		}
		delete[] tc08_devices;
	}
	
	printf("Exiting.\n");
    return 0;
}
