#GPL
#GPL  libwhisker copyright 2000,2001,2002 by rfp.labs
#GPL
#GPL  This program is free software; you can redistribute it and/or
#GPL  modify it under the terms of the GNU General Public License
#GPL  as published by the Free Software Foundation; either version 2
#GPL  of the License, or (at your option) any later version.
#GPL
#GPL  This program is distributed in the hope that it will be useful,
#GPL  but WITHOUT ANY WARRANTY; without even the implied warranty of
#GPL  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GPL  GNU General Public License for more details.
#GPL

=pod

=head1 ++ Sub package: forms

This subpackage contains various routines to parse and handle HTML forms.  
The goal is to parse the variable, human-readable HTML into concrete
structures useable by your program.  The forms package does do a good job
at making these structures, but I will admit: they are not exactly simple,
and thus not a cinch to work with.  But then again, representing something
as complex as a HTML form is not a simple thing either.  I think the
results are acceptable for what's trying to be done.  Anyways...

Forms are stored in perl hashes, with elements in the following format:

	$form{'element_name'}=@([ 'type', 'value', @params ])

Thus every element in the hash is an array of anonymous arrays.  The first
array value contains the element type (which is 'select', 'textarea',
'button', or an 'input' value of the form 'input-text', 'input-hidden',
'input-radio', etc).

The second value is the value, if applicable (it could be undef if no
value was specified).  Note that select elements will always have an undef
value--the actual values are in the subsequent options elements.

The third value, if defined, is an anonymous array of additional tag
parameters found in the element (like 'onchange="blah"', 'size="20"',
'maxlength="40"', 'selected', etc).

The array does contain one special element, which is stored in the hash
under a NULL character ("\0") key.  This element is of the format:

	$form{"\0"}=['name', 'method', 'action', @parameters];

The element is an anonymous array that contains strings of the form's
name, method, and action (values can be undef), and a @parameters array
similar to that found in normal elements (above).

Accessing individual values stored in the form hash becomes a test of your
perl referencing skills.  Hint: to access the 'value' of the third element
named 'choices', you would need to do:

	$form{'choices'}->[2]->[1];

The '[2]' is the third element (normal array starts with 0), and the
actual value is '[1]' (the type is '[0]', and the parameter array is
'[2]').

=cut

################################################################

=pod

=head1 - Function: LW::forms_read
  
Params: \$html_data
Return: @found_forms

This function parses the given $html_data into libwhisker form hashes.  
It returns an array of hash references to the found forms.

=cut

sub forms_read {
	my $dr=shift;
	return undef if(!ref($dr) || length($$dr)==0);

	@LW::forms_found=();
	LW::html_find_tags($dr,\&forms_parse_callback);

	if(scalar %LW::forms_current){
		my %DUP=%LW::forms_current;
		push(@LW::forms_found,\%DUP);
	}
	return @LW::forms_found;
}

################################################################

=pod

=head1 - Function: LW::forms_write
  
Params: \%form_hash
Return: $html_of_form   [undef on error]

This function will take the given %form hash and compose a generic HTML
representation of it, formatted with tabs and newlines in order to make it
neat and tidy for printing.

Note: this function does *not* escape any special characters that were
embedded in the element values.

=cut

sub forms_write {
	my $hr=shift;
	return undef if(!ref($hr) || !(scalar %$hr));
	return undef if(!defined $$hr{"\0"});
	
	my $t='<form name="'.$$hr{"\0"}->[0].'" method="';
	$t.=$$hr{"\0"}->[1].'" action="'.$$hr{"\0"}->[2].'"';
	if(defined $$hr{"\0"}->[3]){
		$t.=' '.join(' ',@{$$hr{"\0"}->[3]}); }
	$t.=">\n";

	while( my($name,$ar)=each(%$hr) ){
	  next if($name eq "\0");
	  foreach $a (@$ar){
		my $P='';
		$P=' '.join(' ', @{$$a[2]}) if(defined $$a[2]);
		$t.="\t";

		if($$a[0] eq 'textarea'){
			$t.="<textarea name=\"$name\"$P>$$a[1]";
			$t.="</textarea>\n";

		} elsif($$a[0]=~m/^input-(.+)$/){
			$t.="<input type=\"$1\" name=\"$name\" ";
			$t.="value=\"$$a[1]\"$P>\n";

		} elsif($$a[0] eq 'option'){
			$t.="\t<option value=\"$$a[1]\"$P>$$a[1]\n";

		} elsif($$a[0] eq 'select'){
			$t.="<select name=\"$name\"$P>\n";

		} elsif($$a[0] eq '/select'){
			$t.="</select$P>\n";

		} else { # button
			$t.="<button name=\"$name\" value=\"$$a[1]\">\n";
		}
	  }
	}

	$t.="</form>\n";
	return $t;
}

################################################################


=pod

=head1 - Function: LW::forms_parse_html (INTERNAL)
  
Params: $TAG, \%elements, \$html_data, $offset, $len
Return: nothing

This is an &html_find_tags callback used to parse HTML into form hashes.  
You should not call this directly, but instead use &LW::forms_read.

=cut

{ # these are private static variables for &forms_parse_html
%FORMS_ELEMENTS=(	'form'=>1,	'input'=>1,
			'textarea'=>1,	'button'=>1,
			'select'=>1,	'option'=>1,
			'/select'=>1	);
$CURRENT_SELECT=undef;
$UNKOWNS=0;

sub forms_parse_callback {
	my ($TAG, $hr, $dr, $start, $len)=(lc(shift),@_);
	my ($saveparam, $parr, $key)=(0,undef,'');

	# fastpath shortcut
	return undef if(!defined $FORMS_ELEMENTS{$TAG});
	LW::utils_lowercase_hashkeys($hr) if(scalar %$hr);

	if($TAG eq 'form'){

		if(scalar %LW::forms_current){ # save last form
			my %DUP=%LW::forms_current;
			push (@LW::forms_found, \%DUP);
			%LW::forms_current=();
		}

		$LW::forms_current{"\0"}=[$$hr{name},$$hr{method},
			$$hr{action},undef];
		delete $$hr{'name'}; delete $$hr{'method'}; delete $$hr{'action'};
		$key="\0"; $parr=\@{$LW::forms_current{"\0"}};
		$UNKNOWNS=0;

	} elsif($TAG eq 'input'){
		$$hr{type}='text' if(!defined $$hr{type});
		$$hr{name}='unknown'.$UNKNOWNS++ if(!defined $$hr{name});
		$key=$$hr{name};
	
		push( @{$LW::forms_current{$key}}, 
			(['input-'.$$hr{type},$$hr{value},undef]) );
		delete $$hr{'name'}; delete $$hr{'type'}; delete $$hr{'value'};
		$parr=\@{$LW::forms_current{$key}->[-1]};

	} elsif($TAG eq 'select'){
		$$hr{name}='unknown'.$UNKNOWNS++ if(!defined $$hr{name});
		$key=$$hr{name};
		push( @{$LW::forms_current{$key}}, (['select',undef,undef]) );
		$parr=\@{$LW::forms_current{$key}->[-1]};
		$CURRENT_SELECT=$key;
		delete $$hr{name};

	} elsif($TAG eq '/select'){
		push( @{$LW::forms_current{$CURRENT_SELECT}}, 
			(['/select',undef,undef]) );
		$CURRENT_SELECT=undef;
		return undef;

	} elsif($TAG eq 'option'){
		return undef if(!defined $CURRENT_SELECT);
		if(!defined $$hr{value}){
			my $stop=index($$dr,'<',$start+$len);
			return undef if($stop==-1); # MAJOR PUKE
			$$hr{value}=substr($$dr,$start+$len,
				($stop-$start-$len));
			$$hr{value}=~tr/\r\n//d;
		}
		push( @{$LW::forms_current{$CURRENT_SELECT}}, 
			(['option',$$hr{value},undef]) );
		delete $$hr{value};
		$parr=\@{$LW::forms_current{$CURRENT_SELECT}->[-1]};

	} elsif($TAG eq 'textarea'){
		my $stop=$start+$len;
		# find closing </textarea> tag
		do {	$stop=index($$dr,'</',$stop+2); 
			return undef if($stop==-1); # MAJOR PUKE
		} while( lc(substr($$dr,$stop+2,8)) ne 'textarea');
		$$hr{value}=substr($$dr,$start+$len,($stop-$start-$len));

		$$hr{name}='unknown'.$UNKNOWNS++ if(!defined $$hr{name});
		$key=$$hr{name};
		push( @{$LW::forms_current{$key}}, 
			(['textarea',$$hr{value},undef]) );
		$parr=\@{$LW::forms_current{$key}->[-1]};
		delete $$hr{'name'}; delete $$hr{'value'};

	} else { # button
		$$hr{name}='unknown'.$UNKNOWNS++ if(!defined $$hr{name});
		$key=$$hr{name};
		push( @{$LW::forms_current{$key}}, 
			(['button',$$hr{value},undef]) );
	}

	if(scalar %$hr){
		my @params=();
		foreach $k (keys %$hr){
			if(defined $$hr{$k}){
					push @params, "$k=\"$$hr{$k}\"";
			} else {	push @params, $k; }
		}
		$$parr[2]=\@params;
	}

	return undef;
}}
