from __future__ import unicode_literals

import unittest

from mopidy.models import Album, Artist, Playlist, Ref, SearchResult, Track
from mopidy.mpd.protocol import music_db

from tests.mpd import protocol


class QueryFromMpdSearchFormatTest(unittest.TestCase):
    def test_dates_are_extracted(self):
        result = music_db._query_from_mpd_search_parameters(
            ['Date', '1974-01-02', 'Date', '1975'], music_db._SEARCH_MAPPING)
        self.assertEqual(result['date'][0], '1974-01-02')
        self.assertEqual(result['date'][1], '1975')

    def test_empty_value_is_ignored(self):
        result = music_db._query_from_mpd_search_parameters(
            ['Date', ''], music_db._SEARCH_MAPPING)
        self.assertEqual(result, {})

    def test_whitespace_value_is_ignored(self):
        result = music_db._query_from_mpd_search_parameters(
            ['Date', '  '], music_db._SEARCH_MAPPING)
        self.assertEqual(result, {})

    # TODO Test more mappings


class QueryFromMpdListFormatTest(unittest.TestCase):
    pass  # TODO


class MusicDatabaseHandlerTest(protocol.BaseTestCase):
    def test_count(self):
        self.sendRequest('count "artist" "needle"')
        self.assertInResponse('songs: 0')
        self.assertInResponse('playtime: 0')
        self.assertInResponse('OK')

    def test_count_without_quotes(self):
        self.sendRequest('count artist "needle"')
        self.assertInResponse('songs: 0')
        self.assertInResponse('playtime: 0')
        self.assertInResponse('OK')

    def test_count_with_multiple_pairs(self):
        self.sendRequest('count "artist" "foo" "album" "bar"')
        self.assertInResponse('songs: 0')
        self.assertInResponse('playtime: 0')
        self.assertInResponse('OK')

    def test_count_correct_length(self):
        # Count the lone track
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[
                Track(uri='dummy:a', name="foo", date="2001", length=4000),
            ])
        self.sendRequest('count "title" "foo"')
        self.assertInResponse('songs: 1')
        self.assertInResponse('playtime: 4')
        self.assertInResponse('OK')

        # Count multiple tracks
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[
                Track(uri='dummy:b', date="2001", length=50000),
                Track(uri='dummy:c', date="2001", length=600000),
            ])
        self.sendRequest('count "date" "2001"')
        self.assertInResponse('songs: 2')
        self.assertInResponse('playtime: 650')
        self.assertInResponse('OK')

    def test_findadd(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[Track(uri='dummy:a', name='A')])
        self.assertEqual(self.core.tracklist.length.get(), 0)

        self.sendRequest('findadd "title" "A"')

        self.assertEqual(self.core.tracklist.length.get(), 1)
        self.assertEqual(self.core.tracklist.tracks.get()[0].uri, 'dummy:a')
        self.assertInResponse('OK')

    def test_searchadd(self):
        self.backend.library.dummy_search_result = SearchResult(
            tracks=[Track(uri='dummy:a', name='A')])
        self.assertEqual(self.core.tracklist.length.get(), 0)

        self.sendRequest('searchadd "title" "a"')

        self.assertEqual(self.core.tracklist.length.get(), 1)
        self.assertEqual(self.core.tracklist.tracks.get()[0].uri, 'dummy:a')
        self.assertInResponse('OK')

    def test_searchaddpl_appends_to_existing_playlist(self):
        playlist = self.core.playlists.create('my favs').get()
        playlist = playlist.copy(tracks=[
            Track(uri='dummy:x', name='X'),
            Track(uri='dummy:y', name='y'),
        ])
        self.core.playlists.save(playlist)
        self.backend.library.dummy_search_result = SearchResult(
            tracks=[Track(uri='dummy:a', name='A')])
        playlists = self.core.playlists.filter(name='my favs').get()
        self.assertEqual(len(playlists), 1)
        self.assertEqual(len(playlists[0].tracks), 2)

        self.sendRequest('searchaddpl "my favs" "title" "a"')

        playlists = self.core.playlists.filter(name='my favs').get()
        self.assertEqual(len(playlists), 1)
        self.assertEqual(len(playlists[0].tracks), 3)
        self.assertEqual(playlists[0].tracks[0].uri, 'dummy:x')
        self.assertEqual(playlists[0].tracks[1].uri, 'dummy:y')
        self.assertEqual(playlists[0].tracks[2].uri, 'dummy:a')
        self.assertInResponse('OK')

    def test_searchaddpl_creates_missing_playlist(self):
        self.backend.library.dummy_search_result = SearchResult(
            tracks=[Track(uri='dummy:a', name='A')])
        self.assertEqual(
            len(self.core.playlists.filter(name='my favs').get()), 0)

        self.sendRequest('searchaddpl "my favs" "title" "a"')

        playlists = self.core.playlists.filter(name='my favs').get()
        self.assertEqual(len(playlists), 1)
        self.assertEqual(playlists[0].tracks[0].uri, 'dummy:a')
        self.assertInResponse('OK')

    def test_listall_without_uri(self):
        tracks = [Track(uri='dummy:/a', name='a'),
                  Track(uri='dummy:/foo/b', name='b')]
        self.backend.library.dummy_library = tracks
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo'),
                        Ref.album(uri='dummy:/album', name='album'),
                        Ref.playlist(uri='dummy:/pl', name='pl')],
            'dummy:/foo': [Ref.track(uri='dummy:/foo/b', name='b')]}

        self.sendRequest('listall')

        self.assertInResponse('file: dummy:/a')
        self.assertInResponse('directory: /dummy/foo')
        self.assertInResponse('directory: /dummy/album')
        self.assertInResponse('directory: /dummy/pl')
        self.assertInResponse('file: dummy:/foo/b')
        self.assertInResponse('OK')

    def test_listall_with_uri(self):
        tracks = [Track(uri='dummy:/a', name='a'),
                  Track(uri='dummy:/foo/b', name='b')]
        self.backend.library.dummy_library = tracks
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')],
            'dummy:/foo': [Ref.track(uri='dummy:/foo/b', name='b')]}

        self.sendRequest('listall "/dummy/foo"')

        self.assertNotInResponse('file: dummy:/a')
        self.assertInResponse('directory: /dummy/foo')
        self.assertInResponse('file: dummy:/foo/b')
        self.assertInResponse('OK')

    def test_listall_with_unknown_uri(self):
        self.sendRequest('listall "/unknown"')

        self.assertEqualResponse('ACK [50@0] {listall} Not found')

    def test_listall_for_dir_with_and_without_leading_slash_is_the_same(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')]}

        response1 = self.sendRequest('listall "dummy"')
        response2 = self.sendRequest('listall "/dummy"')
        self.assertEqual(response1, response2)

    def test_listall_for_dir_with_and_without_trailing_slash_is_the_same(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')]}

        response1 = self.sendRequest('listall "dummy"')
        response2 = self.sendRequest('listall "dummy/"')
        self.assertEqual(response1, response2)

    def test_listall_duplicate(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.directory(uri='dummy:/a1', name='a'),
                        Ref.directory(uri='dummy:/a2', name='a')]}

        self.sendRequest('listall')
        self.assertInResponse('directory: /dummy/a')
        self.assertInResponse('directory: /dummy/a [2]')

    def test_listallinfo_without_uri(self):
        tracks = [Track(uri='dummy:/a', name='a'),
                  Track(uri='dummy:/foo/b', name='b')]
        self.backend.library.dummy_library = tracks
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo'),
                        Ref.album(uri='dummy:/album', name='album'),
                        Ref.playlist(uri='dummy:/pl', name='pl')],
            'dummy:/foo': [Ref.track(uri='dummy:/foo/b', name='b')]}

        self.sendRequest('listallinfo')

        self.assertInResponse('file: dummy:/a')
        self.assertInResponse('Title: a')
        self.assertInResponse('directory: /dummy/foo')
        self.assertInResponse('directory: /dummy/album')
        self.assertInResponse('directory: /dummy/pl')
        self.assertInResponse('file: dummy:/foo/b')
        self.assertInResponse('Title: b')
        self.assertInResponse('OK')

    def test_listallinfo_with_uri(self):
        tracks = [Track(uri='dummy:/a', name='a'),
                  Track(uri='dummy:/foo/b', name='b')]
        self.backend.library.dummy_library = tracks
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')],
            'dummy:/foo': [Ref.track(uri='dummy:/foo/b', name='b')]}

        self.sendRequest('listallinfo "/dummy/foo"')

        self.assertNotInResponse('file: dummy:/a')
        self.assertNotInResponse('Title: a')
        self.assertInResponse('directory: /dummy/foo')
        self.assertInResponse('file: dummy:/foo/b')
        self.assertInResponse('Title: b')
        self.assertInResponse('OK')

    def test_listallinfo_with_unknown_uri(self):
        self.sendRequest('listallinfo "/unknown"')

        self.assertEqualResponse('ACK [50@0] {listallinfo} Not found')

    def test_listallinfo_for_dir_with_and_without_leading_slash_is_same(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')]}

        response1 = self.sendRequest('listallinfo "dummy"')
        response2 = self.sendRequest('listallinfo "/dummy"')
        self.assertEqual(response1, response2)

    def test_listallinfo_for_dir_with_and_without_trailing_slash_is_same(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')]}

        response1 = self.sendRequest('listallinfo "dummy"')
        response2 = self.sendRequest('listallinfo "dummy/"')
        self.assertEqual(response1, response2)

    def test_listallinfo_duplicate(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.directory(uri='dummy:/a1', name='a'),
                        Ref.directory(uri='dummy:/a2', name='a')]}

        self.sendRequest('listallinfo')
        self.assertInResponse('directory: /dummy/a')
        self.assertInResponse('directory: /dummy/a [2]')

    def test_lsinfo_without_path_returns_same_as_for_root(self):
        last_modified = 1390942873222
        self.backend.playlists.playlists = [
            Playlist(name='a', uri='dummy:/a', last_modified=last_modified)]

        response1 = self.sendRequest('lsinfo')
        response2 = self.sendRequest('lsinfo "/"')
        self.assertEqual(response1, response2)

    def test_lsinfo_with_empty_path_returns_same_as_for_root(self):
        last_modified = 1390942873222
        self.backend.playlists.playlists = [
            Playlist(name='a', uri='dummy:/a', last_modified=last_modified)]

        response1 = self.sendRequest('lsinfo ""')
        response2 = self.sendRequest('lsinfo "/"')
        self.assertEqual(response1, response2)

    def test_lsinfo_for_root_includes_playlists(self):
        last_modified = 1390942873222
        self.backend.playlists.playlists = [
            Playlist(name='a', uri='dummy:/a', last_modified=last_modified)]

        self.sendRequest('lsinfo "/"')
        self.assertInResponse('playlist: a')
        # Date without milliseconds and with time zone information
        self.assertInResponse('Last-Modified: 2014-01-28T21:01:13Z')
        self.assertInResponse('OK')

    def test_lsinfo_for_root_includes_dirs_for_each_lib_with_content(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')]}

        self.sendRequest('lsinfo "/"')
        self.assertInResponse('directory: dummy')
        self.assertInResponse('OK')

    def test_lsinfo_for_dir_with_and_without_leading_slash_is_the_same(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')]}

        response1 = self.sendRequest('lsinfo "dummy"')
        response2 = self.sendRequest('lsinfo "/dummy"')
        self.assertEqual(response1, response2)

    def test_lsinfo_for_dir_with_and_without_trailing_slash_is_the_same(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')]}

        response1 = self.sendRequest('lsinfo "dummy"')
        response2 = self.sendRequest('lsinfo "dummy/"')
        self.assertEqual(response1, response2)

    def test_lsinfo_for_dir_includes_tracks(self):
        self.backend.library.dummy_library = [
            Track(uri='dummy:/a', name='a'),
        ]
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a')]}

        self.sendRequest('lsinfo "/dummy"')
        self.assertInResponse('file: dummy:/a')
        self.assertInResponse('Title: a')
        self.assertInResponse('OK')

    def test_lsinfo_for_dir_includes_subdirs(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.directory(uri='dummy:/foo', name='foo')]}

        self.sendRequest('lsinfo "/dummy"')
        self.assertInResponse('directory: dummy/foo')
        self.assertInResponse('OK')

    def test_lsinfo_for_dir_does_not_recurse(self):
        self.backend.library.dummy_library = [
            Track(uri='dummy:/a', name='a'),
        ]
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.directory(uri='dummy:/foo', name='foo')],
            'dummy:/foo': [Ref.track(uri='dummy:/a', name='a')]}

        self.sendRequest('lsinfo "/dummy"')
        self.assertNotInResponse('file: dummy:/a')
        self.assertInResponse('OK')

    def test_lsinfo_for_dir_does_not_include_self(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.directory(uri='dummy:/foo', name='foo')],
            'dummy:/foo': [Ref.track(uri='dummy:/a', name='a')]}

        self.sendRequest('lsinfo "/dummy"')
        self.assertNotInResponse('directory: dummy')
        self.assertInResponse('OK')

    def test_lsinfo_for_root_returns_browse_result_before_playlists(self):
        last_modified = 1390942873222
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.track(uri='dummy:/a', name='a'),
                        Ref.directory(uri='dummy:/foo', name='foo')]}
        self.backend.playlists.playlists = [
            Playlist(name='a', uri='dummy:/a', last_modified=last_modified)]

        response = self.sendRequest('lsinfo "/"')
        self.assertLess(response.index('directory: dummy'),
                        response.index('playlist: a'))

    def test_lsinfo_duplicate(self):
        self.backend.library.dummy_browse_result = {
            'dummy:/': [Ref.directory(uri='dummy:/a1', name='a'),
                        Ref.directory(uri='dummy:/a2', name='a')]}

        self.sendRequest('lsinfo "/dummy"')
        self.assertInResponse('directory: dummy/a')
        self.assertInResponse('directory: dummy/a [2]')

    def test_update_without_uri(self):
        self.sendRequest('update')
        self.assertInResponse('updating_db: 0')
        self.assertInResponse('OK')

    def test_update_with_uri(self):
        self.sendRequest('update "file:///dev/urandom"')
        self.assertInResponse('updating_db: 0')
        self.assertInResponse('OK')

    def test_rescan_without_uri(self):
        self.sendRequest('rescan')
        self.assertInResponse('updating_db: 0')
        self.assertInResponse('OK')

    def test_rescan_with_uri(self):
        self.sendRequest('rescan "file:///dev/urandom"')
        self.assertInResponse('updating_db: 0')
        self.assertInResponse('OK')


class MusicDatabaseFindTest(protocol.BaseTestCase):
    def test_find_includes_fake_artist_and_album_tracks(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            albums=[Album(uri='dummy:album:a', name='A', date='2001')],
            artists=[Artist(uri='dummy:artist:b', name='B')],
            tracks=[Track(uri='dummy:track:c', name='C')])

        self.sendRequest('find "any" "foo"')

        self.assertInResponse('file: dummy:artist:b')
        self.assertInResponse('Title: Artist: B')

        self.assertInResponse('file: dummy:album:a')
        self.assertInResponse('Title: Album: A')
        self.assertInResponse('Date: 2001')

        self.assertInResponse('file: dummy:track:c')
        self.assertInResponse('Title: C')

        self.assertInResponse('OK')

    def test_find_artist_does_not_include_fake_artist_tracks(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            albums=[Album(uri='dummy:album:a', name='A', date='2001')],
            artists=[Artist(uri='dummy:artist:b', name='B')],
            tracks=[Track(uri='dummy:track:c', name='C')])

        self.sendRequest('find "artist" "foo"')

        self.assertNotInResponse('file: dummy:artist:b')
        self.assertNotInResponse('Title: Artist: B')

        self.assertInResponse('file: dummy:album:a')
        self.assertInResponse('Title: Album: A')
        self.assertInResponse('Date: 2001')

        self.assertInResponse('file: dummy:track:c')
        self.assertInResponse('Title: C')

        self.assertInResponse('OK')

    def test_find_albumartist_does_not_include_fake_artist_tracks(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            albums=[Album(uri='dummy:album:a', name='A', date='2001')],
            artists=[Artist(uri='dummy:artist:b', name='B')],
            tracks=[Track(uri='dummy:track:c', name='C')])

        self.sendRequest('find "albumartist" "foo"')

        self.assertNotInResponse('file: dummy:artist:b')
        self.assertNotInResponse('Title: Artist: B')

        self.assertInResponse('file: dummy:album:a')
        self.assertInResponse('Title: Album: A')
        self.assertInResponse('Date: 2001')

        self.assertInResponse('file: dummy:track:c')
        self.assertInResponse('Title: C')

        self.assertInResponse('OK')

    def test_find_artist_and_album_does_not_include_fake_tracks(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            albums=[Album(uri='dummy:album:a', name='A', date='2001')],
            artists=[Artist(uri='dummy:artist:b', name='B')],
            tracks=[Track(uri='dummy:track:c', name='C')])

        self.sendRequest('find "artist" "foo" "album" "bar"')

        self.assertNotInResponse('file: dummy:artist:b')
        self.assertNotInResponse('Title: Artist: B')

        self.assertNotInResponse('file: dummy:album:a')
        self.assertNotInResponse('Title: Album: A')
        self.assertNotInResponse('Date: 2001')

        self.assertInResponse('file: dummy:track:c')
        self.assertInResponse('Title: C')

        self.assertInResponse('OK')

    def test_find_album(self):
        self.sendRequest('find "album" "what"')
        self.assertInResponse('OK')

    def test_find_album_without_quotes(self):
        self.sendRequest('find album "what"')
        self.assertInResponse('OK')

    def test_find_artist(self):
        self.sendRequest('find "artist" "what"')
        self.assertInResponse('OK')

    def test_find_artist_without_quotes(self):
        self.sendRequest('find artist "what"')
        self.assertInResponse('OK')

    def test_find_albumartist(self):
        self.sendRequest('find "albumartist" "what"')
        self.assertInResponse('OK')

    def test_find_albumartist_without_quotes(self):
        self.sendRequest('find albumartist "what"')
        self.assertInResponse('OK')

    def test_find_composer(self):
        self.sendRequest('find "composer" "what"')
        self.assertInResponse('OK')

    def test_find_composer_without_quotes(self):
        self.sendRequest('find composer "what"')
        self.assertInResponse('OK')

    def test_find_performer(self):
        self.sendRequest('find "performer" "what"')
        self.assertInResponse('OK')

    def test_find_performer_without_quotes(self):
        self.sendRequest('find performer "what"')
        self.assertInResponse('OK')

    def test_find_filename(self):
        self.sendRequest('find "filename" "afilename"')
        self.assertInResponse('OK')

    def test_find_filename_without_quotes(self):
        self.sendRequest('find filename "afilename"')
        self.assertInResponse('OK')

    def test_find_file(self):
        self.sendRequest('find "file" "afilename"')
        self.assertInResponse('OK')

    def test_find_file_without_quotes(self):
        self.sendRequest('find file "afilename"')
        self.assertInResponse('OK')

    def test_find_title(self):
        self.sendRequest('find "title" "what"')
        self.assertInResponse('OK')

    def test_find_title_without_quotes(self):
        self.sendRequest('find title "what"')
        self.assertInResponse('OK')

    def test_find_track_no(self):
        self.sendRequest('find "track" "10"')
        self.assertInResponse('OK')

    def test_find_track_no_without_quotes(self):
        self.sendRequest('find track "10"')
        self.assertInResponse('OK')

    def test_find_track_no_without_filter_value(self):
        self.sendRequest('find "track" ""')
        self.assertInResponse('OK')

    def test_find_genre(self):
        self.sendRequest('find "genre" "what"')
        self.assertInResponse('OK')

    def test_find_genre_without_quotes(self):
        self.sendRequest('find genre "what"')
        self.assertInResponse('OK')

    def test_find_date(self):
        self.sendRequest('find "date" "2002-01-01"')
        self.assertInResponse('OK')

    def test_find_date_without_quotes(self):
        self.sendRequest('find date "2002-01-01"')
        self.assertInResponse('OK')

    def test_find_date_with_capital_d_and_incomplete_date(self):
        self.sendRequest('find Date "2005"')
        self.assertInResponse('OK')

    def test_find_else_should_fail(self):
        self.sendRequest('find "somethingelse" "what"')
        self.assertEqualResponse('ACK [2@0] {find} incorrect arguments')

    def test_find_album_and_artist(self):
        self.sendRequest('find album "album_what" artist "artist_what"')
        self.assertInResponse('OK')

    def test_find_without_filter_value(self):
        self.sendRequest('find "album" ""')
        self.assertInResponse('OK')


class MusicDatabaseListTest(protocol.BaseTestCase):
    def test_list(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[
                Track(uri='dummy:a', name='A', artists=[
                    Artist(name='A Artist')])])

        self.sendRequest('list "artist" "artist" "foo"')

        self.assertInResponse('Artist: A Artist')
        self.assertInResponse('OK')

    def test_list_foo_returns_ack(self):
        self.sendRequest('list "foo"')
        self.assertEqualResponse('ACK [2@0] {list} incorrect arguments')

    # Artist

    def test_list_artist_with_quotes(self):
        self.sendRequest('list "artist"')
        self.assertInResponse('OK')

    def test_list_artist_without_quotes(self):
        self.sendRequest('list artist')
        self.assertInResponse('OK')

    def test_list_artist_without_quotes_and_capitalized(self):
        self.sendRequest('list Artist')
        self.assertInResponse('OK')

    def test_list_artist_with_query_of_one_token(self):
        self.sendRequest('list "artist" "anartist"')
        self.assertEqualResponse(
            'ACK [2@0] {list} should be "Album" for 3 arguments')

    def test_list_artist_with_unknown_field_in_query_returns_ack(self):
        self.sendRequest('list "artist" "foo" "bar"')
        self.assertEqualResponse('ACK [2@0] {list} not able to parse args')

    def test_list_artist_by_artist(self):
        self.sendRequest('list "artist" "artist" "anartist"')
        self.assertInResponse('OK')

    def test_list_artist_by_album(self):
        self.sendRequest('list "artist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_artist_by_full_date(self):
        self.sendRequest('list "artist" "date" "2001-01-01"')
        self.assertInResponse('OK')

    def test_list_artist_by_year(self):
        self.sendRequest('list "artist" "date" "2001"')
        self.assertInResponse('OK')

    def test_list_artist_by_genre(self):
        self.sendRequest('list "artist" "genre" "agenre"')
        self.assertInResponse('OK')

    def test_list_artist_by_artist_and_album(self):
        self.sendRequest(
            'list "artist" "artist" "anartist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_artist_without_filter_value(self):
        self.sendRequest('list "artist" "artist" ""')
        self.assertInResponse('OK')

    def test_list_artist_should_not_return_artists_without_names(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[Track(artists=[Artist(name='')])])

        self.sendRequest('list "artist"')
        self.assertNotInResponse('Artist: ')
        self.assertInResponse('OK')

    # Albumartist

    def test_list_albumartist_with_quotes(self):
        self.sendRequest('list "albumartist"')
        self.assertInResponse('OK')

    def test_list_albumartist_without_quotes(self):
        self.sendRequest('list albumartist')
        self.assertInResponse('OK')

    def test_list_albumartist_without_quotes_and_capitalized(self):
        self.sendRequest('list Albumartist')
        self.assertInResponse('OK')

    def test_list_albumartist_with_query_of_one_token(self):
        self.sendRequest('list "albumartist" "anartist"')
        self.assertEqualResponse(
            'ACK [2@0] {list} should be "Album" for 3 arguments')

    def test_list_albumartist_with_unknown_field_in_query_returns_ack(self):
        self.sendRequest('list "albumartist" "foo" "bar"')
        self.assertEqualResponse('ACK [2@0] {list} not able to parse args')

    def test_list_albumartist_by_artist(self):
        self.sendRequest('list "albumartist" "artist" "anartist"')
        self.assertInResponse('OK')

    def test_list_albumartist_by_album(self):
        self.sendRequest('list "albumartist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_albumartist_by_full_date(self):
        self.sendRequest('list "albumartist" "date" "2001-01-01"')
        self.assertInResponse('OK')

    def test_list_albumartist_by_year(self):
        self.sendRequest('list "albumartist" "date" "2001"')
        self.assertInResponse('OK')

    def test_list_albumartist_by_genre(self):
        self.sendRequest('list "albumartist" "genre" "agenre"')
        self.assertInResponse('OK')

    def test_list_albumartist_by_artist_and_album(self):
        self.sendRequest(
            'list "albumartist" "artist" "anartist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_albumartist_without_filter_value(self):
        self.sendRequest('list "albumartist" "artist" ""')
        self.assertInResponse('OK')

    def test_list_albumartist_should_not_return_artists_without_names(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[Track(album=Album(artists=[Artist(name='')]))])

        self.sendRequest('list "albumartist"')
        self.assertNotInResponse('Artist: ')
        self.assertNotInResponse('Albumartist: ')
        self.assertNotInResponse('Composer: ')
        self.assertNotInResponse('Performer: ')
        self.assertInResponse('OK')

    # Composer

    def test_list_composer_with_quotes(self):
        self.sendRequest('list "composer"')
        self.assertInResponse('OK')

    def test_list_composer_without_quotes(self):
        self.sendRequest('list composer')
        self.assertInResponse('OK')

    def test_list_composer_without_quotes_and_capitalized(self):
        self.sendRequest('list Composer')
        self.assertInResponse('OK')

    def test_list_composer_with_query_of_one_token(self):
        self.sendRequest('list "composer" "anartist"')
        self.assertEqualResponse(
            'ACK [2@0] {list} should be "Album" for 3 arguments')

    def test_list_composer_with_unknown_field_in_query_returns_ack(self):
        self.sendRequest('list "composer" "foo" "bar"')
        self.assertEqualResponse('ACK [2@0] {list} not able to parse args')

    def test_list_composer_by_artist(self):
        self.sendRequest('list "composer" "artist" "anartist"')
        self.assertInResponse('OK')

    def test_list_composer_by_album(self):
        self.sendRequest('list "composer" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_composer_by_full_date(self):
        self.sendRequest('list "composer" "date" "2001-01-01"')
        self.assertInResponse('OK')

    def test_list_composer_by_year(self):
        self.sendRequest('list "composer" "date" "2001"')
        self.assertInResponse('OK')

    def test_list_composer_by_genre(self):
        self.sendRequest('list "composer" "genre" "agenre"')
        self.assertInResponse('OK')

    def test_list_composer_by_artist_and_album(self):
        self.sendRequest(
            'list "composer" "artist" "anartist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_composer_without_filter_value(self):
        self.sendRequest('list "composer" "artist" ""')
        self.assertInResponse('OK')

    def test_list_composer_should_not_return_artists_without_names(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[Track(composers=[Artist(name='')])])

        self.sendRequest('list "composer"')
        self.assertNotInResponse('Artist: ')
        self.assertNotInResponse('Albumartist: ')
        self.assertNotInResponse('Composer: ')
        self.assertNotInResponse('Performer: ')
        self.assertInResponse('OK')

    # Performer

    def test_list_performer_with_quotes(self):
        self.sendRequest('list "performer"')
        self.assertInResponse('OK')

    def test_list_performer_without_quotes(self):
        self.sendRequest('list performer')
        self.assertInResponse('OK')

    def test_list_performer_without_quotes_and_capitalized(self):
        self.sendRequest('list Albumartist')
        self.assertInResponse('OK')

    def test_list_performer_with_query_of_one_token(self):
        self.sendRequest('list "performer" "anartist"')
        self.assertEqualResponse(
            'ACK [2@0] {list} should be "Album" for 3 arguments')

    def test_list_performer_with_unknown_field_in_query_returns_ack(self):
        self.sendRequest('list "performer" "foo" "bar"')
        self.assertEqualResponse('ACK [2@0] {list} not able to parse args')

    def test_list_performer_by_artist(self):
        self.sendRequest('list "performer" "artist" "anartist"')
        self.assertInResponse('OK')

    def test_list_performer_by_album(self):
        self.sendRequest('list "performer" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_performer_by_full_date(self):
        self.sendRequest('list "performer" "date" "2001-01-01"')
        self.assertInResponse('OK')

    def test_list_performer_by_year(self):
        self.sendRequest('list "performer" "date" "2001"')
        self.assertInResponse('OK')

    def test_list_performer_by_genre(self):
        self.sendRequest('list "performer" "genre" "agenre"')
        self.assertInResponse('OK')

    def test_list_performer_by_artist_and_album(self):
        self.sendRequest(
            'list "performer" "artist" "anartist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_performer_without_filter_value(self):
        self.sendRequest('list "performer" "artist" ""')
        self.assertInResponse('OK')

    def test_list_performer_should_not_return_artists_without_names(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[Track(performers=[Artist(name='')])])

        self.sendRequest('list "performer"')
        self.assertNotInResponse('Artist: ')
        self.assertNotInResponse('Albumartist: ')
        self.assertNotInResponse('Composer: ')
        self.assertNotInResponse('Performer: ')
        self.assertInResponse('OK')

    # Album

    def test_list_album_with_quotes(self):
        self.sendRequest('list "album"')
        self.assertInResponse('OK')

    def test_list_album_without_quotes(self):
        self.sendRequest('list album')
        self.assertInResponse('OK')

    def test_list_album_without_quotes_and_capitalized(self):
        self.sendRequest('list Album')
        self.assertInResponse('OK')

    def test_list_album_with_artist_name(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[Track(album=Album(name='foo'))])

        self.sendRequest('list "album" "anartist"')
        self.assertInResponse('Album: foo')
        self.assertInResponse('OK')

    def test_list_album_with_artist_name_without_filter_value(self):
        self.sendRequest('list "album" ""')
        self.assertInResponse('OK')

    def test_list_album_by_artist(self):
        self.sendRequest('list "album" "artist" "anartist"')
        self.assertInResponse('OK')

    def test_list_album_by_album(self):
        self.sendRequest('list "album" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_album_by_albumartist(self):
        self.sendRequest('list "album" "albumartist" "anartist"')
        self.assertInResponse('OK')

    def test_list_album_by_composer(self):
        self.sendRequest('list "album" "composer" "anartist"')
        self.assertInResponse('OK')

    def test_list_album_by_performer(self):
        self.sendRequest('list "album" "performer" "anartist"')
        self.assertInResponse('OK')

    def test_list_album_by_full_date(self):
        self.sendRequest('list "album" "date" "2001-01-01"')
        self.assertInResponse('OK')

    def test_list_album_by_year(self):
        self.sendRequest('list "album" "date" "2001"')
        self.assertInResponse('OK')

    def test_list_album_by_genre(self):
        self.sendRequest('list "album" "genre" "agenre"')
        self.assertInResponse('OK')

    def test_list_album_by_artist_and_album(self):
        self.sendRequest(
            'list "album" "artist" "anartist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_album_without_filter_value(self):
        self.sendRequest('list "album" "artist" ""')
        self.assertInResponse('OK')

    def test_list_album_should_not_return_albums_without_names(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[Track(album=Album(name=''))])

        self.sendRequest('list "album"')
        self.assertNotInResponse('Album: ')
        self.assertInResponse('OK')

    # Date

    def test_list_date_with_quotes(self):
        self.sendRequest('list "date"')
        self.assertInResponse('OK')

    def test_list_date_without_quotes(self):
        self.sendRequest('list date')
        self.assertInResponse('OK')

    def test_list_date_without_quotes_and_capitalized(self):
        self.sendRequest('list Date')
        self.assertInResponse('OK')

    def test_list_date_with_query_of_one_token(self):
        self.sendRequest('list "date" "anartist"')
        self.assertEqualResponse(
            'ACK [2@0] {list} should be "Album" for 3 arguments')

    def test_list_date_by_artist(self):
        self.sendRequest('list "date" "artist" "anartist"')
        self.assertInResponse('OK')

    def test_list_date_by_album(self):
        self.sendRequest('list "date" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_date_by_full_date(self):
        self.sendRequest('list "date" "date" "2001-01-01"')
        self.assertInResponse('OK')

    def test_list_date_by_year(self):
        self.sendRequest('list "date" "date" "2001"')
        self.assertInResponse('OK')

    def test_list_date_by_genre(self):
        self.sendRequest('list "date" "genre" "agenre"')
        self.assertInResponse('OK')

    def test_list_date_by_artist_and_album(self):
        self.sendRequest('list "date" "artist" "anartist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_date_without_filter_value(self):
        self.sendRequest('list "date" "artist" ""')
        self.assertInResponse('OK')

    def test_list_date_should_not_return_blank_dates(self):
        self.backend.library.dummy_find_exact_result = SearchResult(
            tracks=[Track(date='')])

        self.sendRequest('list "date"')
        self.assertNotInResponse('Date: ')
        self.assertInResponse('OK')

    # Genre

    def test_list_genre_with_quotes(self):
        self.sendRequest('list "genre"')
        self.assertInResponse('OK')

    def test_list_genre_without_quotes(self):
        self.sendRequest('list genre')
        self.assertInResponse('OK')

    def test_list_genre_without_quotes_and_capitalized(self):
        self.sendRequest('list Genre')
        self.assertInResponse('OK')

    def test_list_genre_with_query_of_one_token(self):
        self.sendRequest('list "genre" "anartist"')
        self.assertEqualResponse(
            'ACK [2@0] {list} should be "Album" for 3 arguments')

    def test_list_genre_by_artist(self):
        self.sendRequest('list "genre" "artist" "anartist"')
        self.assertInResponse('OK')

    def test_list_genre_by_album(self):
        self.sendRequest('list "genre" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_genre_by_full_date(self):
        self.sendRequest('list "genre" "date" "2001-01-01"')
        self.assertInResponse('OK')

    def test_list_genre_by_year(self):
        self.sendRequest('list "genre" "date" "2001"')
        self.assertInResponse('OK')

    def test_list_genre_by_genre(self):
        self.sendRequest('list "genre" "genre" "agenre"')
        self.assertInResponse('OK')

    def test_list_genre_by_artist_and_album(self):
        self.sendRequest(
            'list "genre" "artist" "anartist" "album" "analbum"')
        self.assertInResponse('OK')

    def test_list_genre_without_filter_value(self):
        self.sendRequest('list "genre" "artist" ""')
        self.assertInResponse('OK')


class MusicDatabaseSearchTest(protocol.BaseTestCase):
    def test_search(self):
        self.backend.library.dummy_search_result = SearchResult(
            albums=[Album(uri='dummy:album:a', name='A')],
            artists=[Artist(uri='dummy:artist:b', name='B')],
            tracks=[Track(uri='dummy:track:c', name='C')])

        self.sendRequest('search "any" "foo"')

        self.assertInResponse('file: dummy:album:a')
        self.assertInResponse('Title: Album: A')
        self.assertInResponse('file: dummy:artist:b')
        self.assertInResponse('Title: Artist: B')
        self.assertInResponse('file: dummy:track:c')
        self.assertInResponse('Title: C')

        self.assertInResponse('OK')

    def test_search_album(self):
        self.sendRequest('search "album" "analbum"')
        self.assertInResponse('OK')

    def test_search_album_without_quotes(self):
        self.sendRequest('search album "analbum"')
        self.assertInResponse('OK')

    def test_search_album_without_filter_value(self):
        self.sendRequest('search "album" ""')
        self.assertInResponse('OK')

    def test_search_artist(self):
        self.sendRequest('search "artist" "anartist"')
        self.assertInResponse('OK')

    def test_search_artist_without_quotes(self):
        self.sendRequest('search artist "anartist"')
        self.assertInResponse('OK')

    def test_search_artist_without_filter_value(self):
        self.sendRequest('search "artist" ""')
        self.assertInResponse('OK')

    def test_search_albumartist(self):
        self.sendRequest('search "albumartist" "analbumartist"')
        self.assertInResponse('OK')

    def test_search_albumartist_without_quotes(self):
        self.sendRequest('search albumartist "analbumartist"')
        self.assertInResponse('OK')

    def test_search_albumartist_without_filter_value(self):
        self.sendRequest('search "albumartist" ""')
        self.assertInResponse('OK')

    def test_search_composer(self):
        self.sendRequest('search "composer" "acomposer"')
        self.assertInResponse('OK')

    def test_search_composer_without_quotes(self):
        self.sendRequest('search composer "acomposer"')
        self.assertInResponse('OK')

    def test_search_composer_without_filter_value(self):
        self.sendRequest('search "composer" ""')
        self.assertInResponse('OK')

    def test_search_performer(self):
        self.sendRequest('search "performer" "aperformer"')
        self.assertInResponse('OK')

    def test_search_performer_without_quotes(self):
        self.sendRequest('search performer "aperformer"')
        self.assertInResponse('OK')

    def test_search_performer_without_filter_value(self):
        self.sendRequest('search "performer" ""')
        self.assertInResponse('OK')

    def test_search_filename(self):
        self.sendRequest('search "filename" "afilename"')
        self.assertInResponse('OK')

    def test_search_filename_without_quotes(self):
        self.sendRequest('search filename "afilename"')
        self.assertInResponse('OK')

    def test_search_filename_without_filter_value(self):
        self.sendRequest('search "filename" ""')
        self.assertInResponse('OK')

    def test_search_file(self):
        self.sendRequest('search "file" "afilename"')
        self.assertInResponse('OK')

    def test_search_file_without_quotes(self):
        self.sendRequest('search file "afilename"')
        self.assertInResponse('OK')

    def test_search_file_without_filter_value(self):
        self.sendRequest('search "file" ""')
        self.assertInResponse('OK')

    def test_search_title(self):
        self.sendRequest('search "title" "atitle"')
        self.assertInResponse('OK')

    def test_search_title_without_quotes(self):
        self.sendRequest('search title "atitle"')
        self.assertInResponse('OK')

    def test_search_title_without_filter_value(self):
        self.sendRequest('search "title" ""')
        self.assertInResponse('OK')

    def test_search_any(self):
        self.sendRequest('search "any" "anything"')
        self.assertInResponse('OK')

    def test_search_any_without_quotes(self):
        self.sendRequest('search any "anything"')
        self.assertInResponse('OK')

    def test_search_any_without_filter_value(self):
        self.sendRequest('search "any" ""')
        self.assertInResponse('OK')

    def test_search_track_no(self):
        self.sendRequest('search "track" "10"')
        self.assertInResponse('OK')

    def test_search_track_no_without_quotes(self):
        self.sendRequest('search track "10"')
        self.assertInResponse('OK')

    def test_search_track_no_without_filter_value(self):
        self.sendRequest('search "track" ""')
        self.assertInResponse('OK')

    def test_search_genre(self):
        self.sendRequest('search "genre" "agenre"')
        self.assertInResponse('OK')

    def test_search_genre_without_quotes(self):
        self.sendRequest('search genre "agenre"')
        self.assertInResponse('OK')

    def test_search_genre_without_filter_value(self):
        self.sendRequest('search "genre" ""')
        self.assertInResponse('OK')

    def test_search_date(self):
        self.sendRequest('search "date" "2002-01-01"')
        self.assertInResponse('OK')

    def test_search_date_without_quotes(self):
        self.sendRequest('search date "2002-01-01"')
        self.assertInResponse('OK')

    def test_search_date_with_capital_d_and_incomplete_date(self):
        self.sendRequest('search Date "2005"')
        self.assertInResponse('OK')

    def test_search_date_without_filter_value(self):
        self.sendRequest('search "date" ""')
        self.assertInResponse('OK')

    def test_search_comment(self):
        self.sendRequest('search "comment" "acomment"')
        self.assertInResponse('OK')

    def test_search_comment_without_quotes(self):
        self.sendRequest('search comment "acomment"')
        self.assertInResponse('OK')

    def test_search_comment_without_filter_value(self):
        self.sendRequest('search "comment" ""')
        self.assertInResponse('OK')

    def test_search_else_should_fail(self):
        self.sendRequest('search "sometype" "something"')
        self.assertEqualResponse('ACK [2@0] {search} incorrect arguments')
