# Copyright (C) 2009-2014  Lars Wirzenius
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


import errno
import gc
import hashlib
import logging
import os
import re
import stat
import sys
import time
import traceback
import tracing
import ttystatus

import obnamlib
import larch


class RepositorySettingMissingError(obnamlib.ObnamError):

    msg = ('No --repository setting. '
           'You need to specify it on the command line or '
           'a configuration file')


class BackupRootMissingError(obnamlib.ObnamError):

    msg = 'No backup roots specified'


class BackupRootDoesNotExist(obnamlib.ObnamError):

    msg = 'Backup root does not exist or is not a directory: {root}'


class BackupErrors(obnamlib.ObnamError):

    msg = 'There were errors during the backup'


class ChunkidPool(object):

    '''Data/chunkid mappings that are pending an upload to shared trees.'''

    def __init__(self):
        self.clear()

    def add(self, chunkid, token):
        if token not in self._mapping:
            self._mapping[token] = []
        self._mapping[token].append(chunkid)

    def __contains__(self, token):
        return token in self._mapping

    def get(self, token):
        return self._mapping.get(token, [])

    def clear(self):
        self._mapping = {}

    def __iter__(self):
        for token in self._mapping.keys():
            for chunkid in self._mapping[token]:
                yield chunkid, token


class BackupProgress(object):

    def __init__(self, ts):
        self.file_count = 0
        self.backed_up_count = 0
        self.uploaded_bytes = 0
        self.scanned_bytes = 0
        self.started = None
        self.errors = False

        self._ts = ts
        self._ts['current-file'] = ''
        self._ts['scanned-bytes'] = 0
        self._ts['uploaded-bytes'] = 0
        self._ts.format('%ElapsedTime() '
                        '%Counter(current-file) '
                        'files '
                        '%ByteSize(scanned-bytes) scanned: '
                        '%String(what)')

    def clear(self):
        self._ts.clear()

    def error(self, msg, exc=None):
        self.errors = True

        logging.error(msg)
        if exc:
            logging.error(repr(exc))
        self._ts.error('ERROR: %s' % msg)

    def what(self, what_what):
        if self.started is None:
            self.started = time.time()
        self._ts['what'] = what_what
        self._ts.flush()

    def update_progress(self):
        self._ts['not-shown'] = 'not shown'

    def update_progress_with_file(self, filename, metadata):
        self._ts['what'] = filename
        self._ts['current-file'] = filename
        self.file_count += 1

    def update_progress_with_scanned(self, amount):
        self.scanned_bytes += amount
        self._ts['scanned-bytes'] = self.scanned_bytes

    def update_progress_with_upload(self, amount):
        self.uploaded_bytes += amount
        self._ts['uploaded-bytes'] = self.uploaded_bytes

    def update_progress_with_removed_checkpoint(self, gen):
        self._ts['checkpoint'] = gen

    def report_stats(self, fs):
        duration = time.time() - self.started
        duration_string = obnamlib.humanise_duration(duration)

        chunk_amount, chunk_unit = obnamlib.humanise_size(
            self.uploaded_bytes)

        fs_amount, fs_unit = obnamlib.humanise_size(fs.bytes_written)

        dl_amount, dl_unit = obnamlib.humanise_size(fs.bytes_read)

        overhead_bytes = (
            fs.bytes_read + (fs.bytes_written - self.uploaded_bytes))
        overhead_bytes = max(0, overhead_bytes)
        overhead_amount, overhead_unit = obnamlib.humanise_size(
            overhead_bytes)

        speed_amount, speed_unit = obnamlib.humanise_speed(
            self.uploaded_bytes, duration)

        logging.info(
            'Backup performance statistics:')
        logging.info(
            '* files found: %s',
            self.file_count)
        logging.info(
            '* files backed up: %s',
            self.backed_up_count)
        logging.info(
            '* uploaded chunk data: %s bytes (%s %s)',
            self.uploaded_bytes, chunk_amount, chunk_unit)
        logging.info(
            '* total uploaded data (incl. metadata): %s bytes (%s %s)',
            fs.bytes_written, fs_amount, fs_unit)
        logging.info(
            '* total downloaded data (incl. metadata): %s bytes (%s %s)',
            fs.bytes_read, dl_amount, dl_unit)
        logging.info(
            '* transfer overhead: %s bytes (%s %s)',
            overhead_bytes, overhead_amount, overhead_unit)
        logging.info(
            '* duration: %s s (%s)',
            duration, duration_string)
        logging.info(
            '* average speed: %s %s',
            speed_amount, speed_unit)

        self._ts.notify(
            'Backed up %d files (of %d found), '
            'uploaded %.1f %s in %s at %.1f %s average speed' %
                (self.backed_up_count, self.file_count,
                 chunk_amount, chunk_unit,
                 duration_string, speed_amount, speed_unit))


class BackupPlugin(obnamlib.ObnamPlugin):

    def enable(self):
        self.app.add_subcommand(
            'backup', self.backup, arg_synopsis='[DIRECTORY]...')
        self.add_backup_settings()
        self.app.hooks.new('backup-finished')

    def add_backup_settings(self):

        # Backup related settings.

        backup_group = obnamlib.option_group['backup'] = 'Backing up'

        self.app.settings.string_list(
            ['root'],
            'what to backup',
            metavar='URL',
            group=backup_group)

        self.app.settings.string_list(
            ['exclude'],
            'regular expression for pathnames to '
            'exclude from backup (can be used multiple '
            'times)',
            group=backup_group)

        self.app.settings.string_list(
            ['exclude-from'],
            'read exclude patterns from FILE',
            metavar='FILE',
            group=backup_group)

        self.app.settings.boolean(
            ['exclude-caches'],
            'exclude directories (and their subdirs) '
            'that contain a CACHEDIR.TAG file (see '
            'http://www.brynosaurus.com/cachedir/spec.html for what '
            'it needs to contain, and http://liw.fi/cachedir/ for a '
            'helper tool)',
            group=backup_group)

        self.app.settings.boolean(
            ['one-file-system'],
            'exclude directories (and their subdirs) '
            'that are in a different filesystem',
            group=backup_group)

        self.app.settings.bytesize(
            ['checkpoint'],
            'make a checkpoint after a given SIZE',
            metavar='SIZE',
            default=1024**3,
            group=backup_group)

        self.app.settings.choice(
            ['deduplicate'],
            ['fatalist', 'never', 'verify'],
            'find duplicate data in backed up data '
            'and store it only once; three modes '
            'are available: never de-duplicate, '
            'verify that no hash collisions happen, '
            'or (the default) fatalistically accept '
            'the risk of collisions',
            metavar='MODE',
            group=backup_group)

        self.app.settings.boolean(
            ['leave-checkpoints'],
            'leave checkpoint generations at the end '
            'of a successful backup run',
            group=backup_group)

        self.app.settings.boolean(
            ['small-files-in-btree'],
            'put contents of small files directly into '
            'the per-client B-tree, instead of '
            'separate chunk files; do not use this '
            'as it is quite bad for performance',
            group=backup_group)

        # Performance related settings.

        perf_group = obnamlib.option_group['perf']

        self.app.settings.integer(
            ['chunkids-per-group'],
            'encode NUM chunk ids per group',
            metavar='NUM',
            default=obnamlib.DEFAULT_CHUNKIDS_PER_GROUP,
            group=perf_group)

        # Development related settings.

        devel_group = obnamlib.option_group['devel']

        self.app.settings.string_list(
            ['testing-fail-matching'],
            'development testing helper: simulate failures during backup '
            'for files that match the given regular expressions',
            metavar='REGEXP',
            group=devel_group)

    def configure_ttystatus_for_backup(self):
        self.progress = BackupProgress(self.app.ts)

    def parse_checkpoint_size(self, value):
        p = obnamlib.ByteSizeParser()
        p.set_default_unit('MiB')
        return p.parse(value)

    @property
    def pretend(self):
        return self.app.settings['pretend']

    def backup(self, args):
        '''Backup data to repository.'''
        logging.info('Backup starts')
        logging.debug(
            'Checkpoints every %s bytes' % self.app.settings['checkpoint'])

        self.app.settings.require('repository')
        self.app.settings.require('client-name')

        if not self.app.settings['repository']:
            raise RepositorySettingMissingError()

        self.configure_ttystatus_for_backup()
        self.progress.what('setting up')

        self.compile_exclusion_patterns()
        self.memory_dump_counter = 0

        self.progress.what('connecting to repository')
        self.client_name = self.app.settings['client-name']
        self.got_client_lock = False
        self.got_chunk_indexes_lock = False
        if self.pretend:
            try:
                self.repo = self.app.get_repository_object()
            except Exception as e:
                self.progress.error(
                    'Are you using --pretend without an existing '
                    'repository? That does not\n'
                    'work, sorry. You can create a small repository, '
                    'backing up just one\n'
                    'small directory, and then use --pretend with '
                    'the real data.')
                raise
        else:
            self.repo = self.app.get_repository_object(create=True)
            self.progress.what('adding client')
            self.add_client(self.client_name)
            self.progress.what('locking client')
            self.repo.lock_client(self.client_name)
            self.got_client_lock = True

            # Need to lock the shared stuff briefly, so encryption etc
            # gets initialized.
            self.progress.what(
                'initialising shared directories')
            self.repo.lock_chunk_indexes()
            self.got_chunk_indexes_lock = True
            self.repo.unlock_chunk_indexes()
            self.got_chunk_indexes_lock = False

        self.chunkid_pool = ChunkidPool()
        try:
            if not self.pretend:
                self.progress.what('starting new generation')
                self.new_generation = self.repo.create_generation(
                    self.client_name)
            self.fs = None
            roots = self.app.settings['root'] + args
            if not roots:
                raise BackupRootMissingError()
            self.backup_roots(roots)
            self.progress.what('committing changes to repository')
            if not self.pretend:
                self.progress.what(
                    'committing changes to repository: '
                    'locking shared B-trees')
                self.repo.lock_chunk_indexes()

                self.progress.what(
                    'committing changes to repository: '
                    'adding chunks to shared B-trees')
                self.add_chunks_to_shared()

                self.progress.what(
                    'committing changes to repository: '
                    'updating generation metadata')
                self.repo.set_generation_key(
                    self.new_generation,
                    obnamlib.REPO_GENERATION_FILE_COUNT,
                    self.progress.file_count)
                self.repo.set_generation_key(
                    self.new_generation,
                    obnamlib.REPO_GENERATION_TOTAL_DATA,
                    self.progress.scanned_bytes)
                self.repo.set_generation_key(
                    self.new_generation,
                    obnamlib.REPO_GENERATION_IS_CHECKPOINT,
                    False)
                
                self.progress.what(
                    'committing changes to repository: '
                    'committing client')
                self.repo.commit_client(self.client_name)

                self.progress.what(
                    'committing changes to repository: '
                    'committing shared B-trees')
                self.repo.commit_chunk_indexes()
            self.progress.what('closing connection to repository')
            self.repo.close()
            self.progress.clear()
            self.progress.report_stats(self.repo.get_fs())

            logging.info('Backup finished.')
            self.app.hooks.call('backup-finished', args, self.progress)
            self.app.dump_memory_profile('at end of backup run')
        except BaseException, e:
            logging.debug('Handling exception %s' % str(e))
            logging.debug(traceback.format_exc())
            self.unlock_when_error()
            raise

        if self.progress.errors:
            raise BackupErrors()

    def unlock_when_error(self):
        try:
            if self.got_client_lock:
                logging.info('Attempting to unlock client because of error')
                self.repo.unlock_client(self.client_name)
            if self.got_chunk_indexes_lock:
                logging.info(
                    'Attempting to unlock shared trees because of error')
                self.repo.unlock_chunk_indexes()
        except BaseException, e2:
            logging.warning(
                'Error while unlocking due to error: %s' % str(e2))
            logging.debug(traceback.format_exc())
        else:
            logging.info('Successfully unlocked')

    def add_chunks_to_shared(self):
        for chunkid, token in self.chunkid_pool:
            self.repo.put_chunk_into_indexes(chunkid, token, self.client_name)
        self.chunkid_pool.clear()

    def add_client(self, client_name):
        self.repo.lock_client_list()
        if client_name not in self.repo.get_client_names():
            tracing.trace('adding new client %s' % client_name)
            tracing.trace('client list before adding: %s' %
                            self.repo.get_client_names())
            self.repo.add_client(client_name)
            tracing.trace('client list after adding: %s' %
                            self.repo.get_client_names())
        self.repo.commit_client_list()
        self.repo = self.app.get_repository_object(repofs=self.repo.get_fs())

    def compile_exclusion_patterns(self):
        # read exclude list files from --exclude-from
        exclude_patterns = self.read_exclusion_patterns_from_files(
                                    self.app.settings['exclude-from'])
        # add patterns passed via --exclude
        exclude_patterns.extend(self.app.settings['exclude'])

        # Ignore log file, except don't exclude the words cliapp uses
        # for not logging or for logging to syslog.
        log = self.app.settings['log']
        if log and log not in ('none', 'syslog'):
            log = self.app.settings['log']
            exclude_patterns.append(log)
        for pattern in exclude_patterns:
            logging.debug('Exclude pattern: %s' % pattern)

        self.exclude_pats = []
        for x in exclude_patterns:
            if x != '':
                try:
                    self.exclude_pats.append(re.compile(x))
                except re.error, e:
                    msg = (
                        'error compiling regular expression "%s": %s' % 
                        (x, e))
                    logging.error(msg)
                    self.progress.error(msg)

    def read_exclusion_patterns_from_files(self, filenames):
        patterns = []
        for filename in filenames:
            with open(filename) as f:
                for line in f:
                    line = line.strip()
                    if line and not line.startswith('#'):
                        patterns.append(line)
        return patterns

    def backup_roots(self, roots):
        self.progress.what('connecting to to repository')
        try:
            self.fs = self.app.fsf.new(roots[0])
            self.fs.connect()
        except OSError as e:
            if e.errno == errno.ENOENT:
                raise BackupRootDoesNotExist(root=roots[0])
            raise

        absroots = []
        for root in roots:
            self.progress.what('determining absolute path for %s' % root)
            self.fs.reinit(root)
            absroots.append(self.fs.abspath('.'))

        if not self.pretend:
            self.remove_old_roots(absroots)

        self.checkpoints = []
        self.last_checkpoint = 0
        self.interval = self.app.settings['checkpoint']

        for root in roots:
            logging.info('Backing up root %s' % root)
            self.progress.what('connecting to live data %s' % root)

            try:
                self.fs.reinit(root)
            except OSError as e:
                if e.errno == errno.ENOENT:
                    raise BackupRootDoesNotExist(root=root)
                raise

            self.progress.what('scanning for files in %s' % root)
            absroot = self.fs.abspath('.')
            self.root_metadata = self.fs.lstat(absroot)

            for pathname, metadata in self.find_files(absroot):
                logging.info('Backing up %s' % pathname)
                if not self.pretend:
                    existed = self.repo.file_exists(
                        self.new_generation, pathname)
                try:
                    self.maybe_simulate_error(pathname)
                    if stat.S_ISDIR(metadata.st_mode):
                        # Directories should only be counted in the
                        # progress their metadata has changed. This
                        # covers the case when files have been deleted
                        # from them.
                        gen = self.get_current_generation()
                        if self.metadata_has_changed(gen, pathname, metadata):
                            self.progress.backed_up_count += 1

                        self.backup_dir_contents(pathname)
                        self.backup_metadata(pathname, metadata)
                    else:
                        # Non-directories' progress can be updated
                        # without further thinking.
                        self.progress.backed_up_count += 1

                        if stat.S_ISREG(metadata.st_mode):
                            assert metadata.md5 is None
                            metadata.md5 = self.backup_file_contents(
                                pathname, metadata)
                        self.backup_metadata(pathname, metadata)

                except (IOError, OSError) as e:

                    if type(e) is IOError:
                        e2 = obnamlib.ObnamIOError(
                            errno=e.errno,
                            strerror=e.strerror,
                            filename=e.filename or pathname)
                    else:
                        e2 = obnamlib.ObnamSystemError(
                            errno=e.errno,
                            strerror=e.strerror,
                            filename=e.filename or pathname)

                    msg = 'Can\'t back up %s: %s' % (pathname, str(e2))
                    self.progress.error(msg, exc=e)
                    if not existed and not self.pretend:
                        try:
                            self.repo.remove_file(
                                self.new_generation, pathname)
                        except KeyError:
                            # File removal failed, but ignore that.
                            # FIXME: This is an artifact of implementation
                            # details of the old repository class. Should
                            # be cleaned up, someday.
                            pass
                        except Exception as ee:
                            expl = (
                                ee.strerror 
                                if hasattr(ee, 'strerror') 
                                else str(ee))
                            msg = (
                                'Error removing partly backed up file %s: %s'
                                % (pathname, repr(ee)))
                            self.progress.error(msg, ee)
                    if e.errno == errno.ENOSPC:
                        raise

                if self.time_for_checkpoint():
                    self.make_checkpoint()
                    self.progress.what(pathname)

            self.backup_parents('.')

        remove_checkpoints = (not self.progress.errors and
                              not self.app.settings['leave-checkpoints']
                              and not self.pretend)
        if remove_checkpoints:
            self.progress.what('removing checkpoints')
            for gen in self.checkpoints:
                self.progress.update_progress_with_removed_checkpoint(gen)
                self.repo.remove_generation(gen)

        if self.fs:
            self.fs.close()

    def maybe_simulate_error(self, pathname):
        '''Raise an IOError if specified by --testing-fail-matching.'''

        for pattern in self.app.settings['testing-fail-matching']:
            if re.search(pattern, pathname):
                e = errno.ENOENT
                raise IOError(e, os.strerror(e), pathname)

    def time_for_checkpoint(self):
        bytes_since = (self.repo.get_fs().bytes_written - 
                       self.last_checkpoint)
        return bytes_since >= self.interval

    def make_checkpoint(self):
        logging.info('Making checkpoint')
        self.progress.what('making checkpoint')
        if not self.pretend:
            self.checkpoints.append(self.new_generation)
            self.progress.what('making checkpoint: backing up parents')
            self.backup_parents('.')
            self.progress.what('making checkpoint: locking shared B-trees')
            self.repo.lock_chunk_indexes()
            self.progress.what(
                'making checkpoint: adding chunks to shared B-trees')
            self.add_chunks_to_shared()
            self.progress.what(
                'making checkpoint: committing per-client B-tree')
            self.repo.set_generation_key(
                self.new_generation,
                obnamlib.REPO_GENERATION_IS_CHECKPOINT, 1)
            self.repo.commit_client(self.client_name)
            self.progress.what('making checkpoint: committing shared B-trees')
            self.repo.commit_chunk_indexes()
            self.last_checkpoint = self.repo.get_fs().bytes_written
            self.progress.what('making checkpoint: re-opening repository')
            self.repo = self.app.get_repository_object(
                repofs=self.repo.get_fs())
            self.progress.what('making checkpoint: locking client')
            self.repo.lock_client(self.client_name)
            self.progress.what('making checkpoint: starting a new generation')
            self.new_generation = self.repo.create_generation(
                self.client_name)
            self.app.dump_memory_profile('at end of checkpoint')
            self.progress.what('making checkpoint: continuing backup')

    def find_files(self, root):
        '''Find all files and directories that need to be backed up.

        This is a generator. It yields (pathname, metadata) pairs.

        The caller should not recurse through directories, just backup
        the directory itself (name, metadata, file list).

        '''

        for pathname, st in self.fs.scan_tree(root, ok=self.can_be_backed_up):
            tracing.trace('considering %s' % pathname)
            try:
                metadata = obnamlib.read_metadata(self.fs, pathname, st=st)
                self.progress.update_progress_with_file(pathname, metadata)
                if self.needs_backup(pathname, metadata):
                    yield pathname, metadata
                else:
                    self.progress.update_progress_with_scanned(
                        metadata.st_size)
            except GeneratorExit:
                raise
            except KeyboardInterrupt:
                logging.error('Keyboard interrupt')
                raise
            except BaseException, e:
                msg = 'Cannot back up %s: %s' % (pathname, str(e))
                self.progress.error(msg, e)

    def can_be_backed_up(self, pathname, st):
        if self.app.settings['one-file-system']:
            if st.st_dev != self.root_metadata.st_dev:
                logging.debug('Excluding (one-file-system): %s' % pathname)
                return False

        for pat in self.exclude_pats:
            if pat.search(pathname):
                logging.debug('Excluding (pattern): %s' % pathname)
                return False

        if stat.S_ISDIR(st.st_mode) and self.app.settings['exclude-caches']:
            tag_filename = 'CACHEDIR.TAG'
            tag_contents = 'Signature: 8a477f597d28d172789f06886806bc55'
            tag_path = os.path.join(pathname, 'CACHEDIR.TAG')
            if self.fs.exists(tag_path):
                # Can't use with, because Paramiko's SFTPFile does not work.
                f = self.fs.open(tag_path, 'rb')
                data = f.read(len(tag_contents))
                f.close()
                if data == tag_contents:
                    logging.debug('Excluding (cache dir): %s' % pathname)
                    return False

        return True

    def needs_backup(self, pathname, current):
        '''Does a given file need to be backed up?'''

        # Directories always require backing up so that backup_dir_contents
        # can remove stuff that no longer exists from them.
        if current.isdir():
            tracing.trace('%s is directory, so needs backup' % pathname)
            return True

        gen = self.get_current_generation()
        tracing.trace('gen=%s' % repr(gen))
        return self.metadata_has_changed(gen, pathname, current)

    def get_current_generation(self):
        '''Return the current generation.

        This handles pretend-mode correctly (in pretend-mode we don't
        have self.new_generation).

        '''

        if self.pretend:
            gens = self.repo.get_client_generation_ids(self.client_name)
            assert gens, "Can't handle --pretend without generations"
            return gens[-1]
        else:
            return self.new_generation

    def metadata_has_changed(self, gen, pathname, current):
        '''Has the metadata for pathname changed since given generation?

        Treat a file that didn't exist in the generation as changed.

        '''

        try:
            old = self.get_metadata_from_generation(gen, pathname)
        except obnamlib.ObnamError as e:
            # File does not exist in the previous generation, so it
            # does need to be backed up.
            return True

        must_be_equal = (
            'st_mtime_sec',
            'st_mtime_nsec',
            'st_mode',
            'st_nlink',
            'st_size',
            'st_uid',
            'st_gid',
            )

        for field in must_be_equal:
            current_value = getattr(current, field)
            old_value = getattr(old, field)
            if current_value != old_value:
                return True

        # Treat xattr values None (no extended attributes) and ''
        # (there are no values, but the xattr blob exists as an empty
        # string) as equal values.
        xattr_current = current.xattr or None
        xattr_old = old.xattr or None
        if xattr_current != xattr_old:
            return True

        return False

    def get_metadata_from_generation(self, gen, pathname):
        return obnamlib.Metadata(
            st_mtime_sec=self.repo.get_file_key(
                gen, pathname, obnamlib.REPO_FILE_MTIME_SEC),
            st_mtime_nsec=self.repo.get_file_key(
                gen, pathname, obnamlib.REPO_FILE_MTIME_NSEC),
            st_mode=self.repo.get_file_key(
                gen, pathname, obnamlib.REPO_FILE_MODE),
            st_nlink=self.repo.get_file_key(
                gen, pathname, obnamlib.REPO_FILE_NLINK),
            st_size=self.repo.get_file_key(
                gen, pathname, obnamlib.REPO_FILE_SIZE),
            st_uid=self.repo.get_file_key(
                gen, pathname, obnamlib.REPO_FILE_UID),
            st_gid=self.repo.get_file_key(
                gen, pathname, obnamlib.REPO_FILE_GID),
            xattr=self.repo.get_file_key(
                gen, pathname, obnamlib.REPO_FILE_XATTR_BLOB))

    def add_file_to_generation(self, filename, metadata):
        self.repo.add_file(self.new_generation, filename)

        things = [
            ('st_mtime_sec', obnamlib.REPO_FILE_MTIME_SEC),
            ('st_mtime_nsec', obnamlib.REPO_FILE_MTIME_NSEC),
            ('st_atime_sec', obnamlib.REPO_FILE_ATIME_SEC),
            ('st_atime_nsec', obnamlib.REPO_FILE_ATIME_NSEC),
            ('st_mode', obnamlib.REPO_FILE_MODE),
            ('st_nlink', obnamlib.REPO_FILE_NLINK),
            ('st_size', obnamlib.REPO_FILE_SIZE),
            ('st_uid', obnamlib.REPO_FILE_UID),
            ('st_gid', obnamlib.REPO_FILE_GID),
            ('st_blocks', obnamlib.REPO_FILE_BLOCKS),
            ('st_dev', obnamlib.REPO_FILE_DEV),
            ('st_ino', obnamlib.REPO_FILE_INO),
            ('username', obnamlib.REPO_FILE_USERNAME),
            ('groupname', obnamlib.REPO_FILE_GROUPNAME),
            ('target', obnamlib.REPO_FILE_SYMLINK_TARGET),
            ('xattr', obnamlib.REPO_FILE_XATTR_BLOB),
            ('md5', obnamlib.REPO_FILE_MD5),
            ]

        for field, key in things:
            self.repo.set_file_key(
                self.new_generation, filename, key, 
                getattr(metadata, field))

    def backup_parents(self, root):
        '''Back up parents of root, non-recursively.'''
        root = self.fs.abspath(root)
        tracing.trace('backing up parents of %s', root)

        dummy_metadata = obnamlib.Metadata(st_mode=0777 | stat.S_IFDIR)

        while True:
            parent = os.path.dirname(root)
            try:
                metadata = obnamlib.read_metadata(self.fs, root)
            except OSError, e:
                logging.warning(
                    'Failed to get metadata for %s: %s: %s' %
                        (root, e.errno or 0, e.strerror))
                logging.warning('Using fake metadata instead for %s' % root)
                metadata = dummy_metadata
            if not self.pretend:
                self.add_file_to_generation(root, metadata)

            if root == parent:
                break
            root = parent

    def backup_metadata(self, pathname, metadata):
        '''Back up metadata for a filesystem object'''

        tracing.trace('backup_metadata: %s', pathname)
        if not self.pretend:
            self.add_file_to_generation(pathname, metadata)

    def backup_file_contents(self, filename, metadata):
        '''Back up contents of a regular file.

        Return MD5 checksum of file's complete data.

        '''

        tracing.trace('backup_file_contents: %s', filename)
        if self.pretend:
            tracing.trace('pretending to upload the whole file')
            self.progress.update_progress_with_upload(metadata.st_size)
            return

        tracing.trace('setting file chunks to empty')
        if not self.pretend:
            if self.repo.file_exists(self.new_generation, filename):
                self.repo.clear_file_chunk_ids(self.new_generation, filename)
            else:
                self.repo.add_file(self.new_generation, filename)

        tracing.trace('opening file for reading')
        f = self.fs.open(filename, 'r')

        summer = hashlib.md5()

        chunk_size = int(self.app.settings['chunk-size'])
        while True:
            tracing.trace('reading some data')
            self.progress.update_progress()
            data = f.read(chunk_size)
            if not data:
                tracing.trace('end of data')
                break
            tracing.trace('got %d bytes of data' % len(data))
            self.progress.update_progress_with_scanned(len(data))
            summer.update(data)
            if not self.pretend:
                chunk_id = self.backup_file_chunk(data)
                self.repo.append_file_chunk_id(
                    self.new_generation, filename, chunk_id)
            else:
                self.self.update_progress_with_upload(len(data))

            if not self.pretend and self.time_for_checkpoint():
                logging.debug('making checkpoint in the middle of a file')
                self.make_checkpoint()
                self.progress.what(filename)

        tracing.trace('closing file')
        f.close()
        self.app.dump_memory_profile('at end of file content backup for %s' %
                                     filename)
        tracing.trace('done backing up file contents')
        return summer.digest()

    def backup_file_chunk(self, data):
        '''Back up a chunk of data by putting it into the repository.'''

        def find():
            # We ignore lookup errors here intentionally. We're reading
            # the checksum trees without a lock, so another Obnam may be
            # modifying them, which can lead to spurious NodeMissing
            # exceptions, and other errors. We don't care: we'll just
            # pretend no chunk with the checksum exists yet.
            try:
                in_tree = self.repo.find_chunk_ids_by_content(data)
            except larch.Error:
                in_tree = []
            except obnamlib.RepositoryChunkContentNotInIndexes:
                in_tree = []
            return in_tree + self.chunkid_pool.get(token)

        def get(chunkid):
            return self.repo.get_chunk_content(chunkid)

        def put():
            self.progress.update_progress_with_upload(len(data))
            return self.repo.put_chunk_content(data)

        def share(chunkid):
            self.chunkid_pool.add(chunkid, token)

        token = self.repo.prepare_chunk_for_indexes(data)

        mode = self.app.settings['deduplicate']
        if mode == 'never':
            return put()
        elif mode == 'verify':
            for chunkid in find():
                data2 = get(chunkid)
                if data == data2:
                    share(chunkid)
                    return chunkid
            else:
                chunkid = put()
                share(chunkid)
                return chunkid
        elif mode == 'fatalist':
            existing = find()
            if existing:
                chunkid = existing[0]
            else:
                chunkid = put()
            share(chunkid)
            return chunkid
        else:
            if not hasattr(self, 'bad_deduplicate_reported'):
                logging.error('unknown --deduplicate setting value')
                self.bad_deduplicate_reported = True
            chunkid = put()
            share(chunkid)
            return chunkid

    def backup_dir_contents(self, root):
        '''Back up the list of files in a directory.'''

        tracing.trace('backup_dir: %s', root)
        if self.pretend:
            return

        new_basenames = self.fs.listdir(root)
        new_pathnames = [os.path.join(root, x) for x in new_basenames]
        if self.repo.file_exists(self.new_generation, root):
            old_pathnames = self.repo.get_file_children(
                    self.new_generation, root)
        else:
            old_pathnames = []

        for old in old_pathnames:
            if old not in new_pathnames:
                self.repo.remove_file(self.new_generation, old)
            else:
                try:
                    st = self.fs.lstat(old)
                except OSError:
                    pass
                else:
                    if not self.can_be_backed_up(old, st): 
                        self.repo.remove_file(self.new_generation, old)

        # Files that are created after the previous generation will be
        # added to the directory when they are backed up, so we don't
        # need to worry about them here.

    def remove_old_roots(self, new_roots):
        '''Remove from started generation anything that is not a backup root.

        We recurse from filesystem root directory until getting to one of
        the new backup roots, or a directory or file that is not a parent
        of one of the new backup roots. We remove anything that is not a
        new backup root, or their parent.

        '''

        def is_parent(pathname):
            if not pathname.endswith(os.sep):
                pathname += os.sep
            for new_root in new_roots:
                if new_root.startswith(pathname):
                    return True
            return False

        def helper(dirname):
            if dirname in new_roots:
                tracing.trace('is a new root: %s' % dirname)
            elif is_parent(dirname):
                tracing.trace('is parent of a new root: %s' % dirname)
                if self.repo.file_exists(gen_id, dirname):
                    pathnames = [
                        os.path.join(dirname, x)
                        for x in self.repo.get_file_children(gen_id, dirname)]
                    for pathname in pathnames:
                        helper(pathname)
            else:
                tracing.trace('is extra and removed: %s' % dirname)
                self.progress.what(
                    'removing %s from new generation' % dirname)
                self.repo.remove_file(self.new_generation, dirname)
                self.progress.what(msg)

        assert not self.pretend
        msg = 'removing old backup roots from new generation'
        self.progress.what(msg)
        tracing.trace('new_roots: %s' % repr(new_roots))
        gen_id = self.new_generation
        helper('/')
